// Copyright ©2015 The Gonum Authors. All rights reserved.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

package testlapack

import (
	"testing"

	"gonum.org/v1/gonum/floats"
)

type Dlasq3er interface {
	Dlasq3(i0, n0 int, z []float64, pp int, dmin, sigma, desig, qmax float64, nFail, iter, nDiv int, ttype int, dmin1, dmin2, dn, dn1, dn2, g, tau float64) (
		i0Out, n0Out, ppOut int, dminOut, sigmaOut, desigOut, qmaxOut float64, nFailOut, iterOut, nDivOut, ttypeOut int, dmin1Out, dmin2Out, dnOut, dn1Out, dn2Out, gOut, tauOut float64)
}

type dlasq3teststruct struct {
	z                                  []float64
	i0, n0, pp                         int
	dmin, desig, sigma, qmax           float64
	nFail, iter, nDiv, ttype           int
	dmin1, dmin2, dn, dn1, dn2, g, tau float64

	zOut                                                    []float64
	i0Out, n0Out, ppOut                                     int
	dminOut, desigOut, sigmaOut, qmaxOut                    float64
	nFailOut, iterOut, nDivOut, ttypeOut                    int
	dmin1Out, dmin2Out, dnOut, dn1Out, dn2Out, gOut, tauOut float64
}

func Dlasq3Test(t *testing.T, impl Dlasq3er) {
	dTol := 1e-14
	// Tests computed from calling the netlib Dlasq
	for _, test := range []dlasq3teststruct{
		{
			i0:       1,
			n0:       21,
			z:        []float64{0.1914365246180821e+01, 0.1564384297703890e+01, 0.2493389162143899e+00, 0.3499809484769305e+00, 0.1315996513131545e+01, 0.1363862112490627e+01, 0.9898466611970759e-01, 0.2014733168553078e+00, 0.6023973979587287e+00, 0.6465544792741794e+00, 0.2210033410638781e-02, 0.5482758480425683e-01, 0.9861857233678967e-01, 0.2428190810745492e-01, 0.4756321484454819e+00, 0.7654669763997353e-01, 0.2588748143677115e+00, 0.6127784069508770e+00, 0.1078611376690004e+00, 0.1217285558623164e+00, 0.6442896492255246e+00, 0.2293835804898155e+00, 0.6203230486639705e+00, 0.5227672064047094e+00, 0.3695660678607585e+00, 0.7645233184745865e+00, 0.5378838054252265e+00, 0.2253657980501426e+00, 0.3562533181264623e+00, 0.8820486722335483e+00, 0.2222132496436145e-01, 0.1208845131814035e-01, 0.1275094303021685e+01, 0.6548746852163357e+00, 0.1647324354821218e+00, 0.6424409427697111e+00, 0.1007530576543866e+01, 0.3269551736546701e+00, 0.3453881601783118e+00, 0.8453078383713172e+00, 0.2679391719153404e+00, 0.4116714838778281e+00, 0.7328677736683723e+00, 0.2016558482158241e+00, 0.8360828138307410e+00, 0.9737579452195326e+00, 0.4813660709592822e+00, 0.5951926422795808e+00, 0.6495370513676459e+00, 0.6761876248148171e+00, 0.2325475880222648e+00, 0.4547154975121112e+00, 0.1993624802893807e+00, 0.3321819367342255e+00, 0.3782318916911257e+00, 0.9972813157741996e-01, 0.9830449403503746e+00, 0.7561080996844842e+00, 0.4429733864040367e+00, 0.6051687323570161e+00, 0.1173279550602403e+01, 0.7195724480316686e+00, 0.5035524069144587e+00, 0.8966804889747714e+00, 0.3058980395058521e+00, 0.6588832353928662e+00, 0.3014634433415453e+00, 0.1505672110274446e+00, 0.1289422237567578e+01, 0.6124645310993601e+00, 0.7583364305799440e+00, 0.9784211498097629e+00, 0.4977814779461571e+00, 0.9993813577491869e+00, 0.2841468847862598e+00, 0.2567365507769143e+00, 0.9257539794205765e+00, 0.5509268385614666e+00, 0.5231355605450990e-04, 0.6589740256453697e+00, 0.2117869221381033e-04, 0.7349224826832024e-04, 0.0000000000000000e+00, 0.0000000000000000e+00},
			pp:       0,
			dmin:     -0.0000000000000000,
			desig:    0.0000000000000000,
			qmax:     2.1637041623952107,
			ttype:    0,
			dmin1:    0.0000000000000000,
			dmin2:    0.0000000000000000,
			dn:       0.0000000000000000,
			dn1:      0.0000000000000000,
			dn2:      0.0000000000000000,
			g:        0.0000000000000000,
			tau:      0.0000000000000000,
			nFail:    0,
			iter:     2,
			sigma:    0.0000000000000000,
			nDiv:     40,
			zOut:     []float64{0.1914365246180821e+01, 0.2163704162395211e+01, 0.2493389162143899e+00, 0.1516515751224039e+00, 0.1315996513131545e+01, 0.1263329604128848e+01, 0.9898466611970759e-01, 0.4719916727467415e-01, 0.6023973979587287e+00, 0.5574082640946934e+00, 0.2210033410638781e-02, 0.3910066531356214e-03, 0.9861857233678967e-01, 0.5738597141291359e+00, 0.4756321484454819e+00, 0.2145632131068746e+00, 0.2588748143677115e+00, 0.1521727389298373e+00, 0.1078611376690004e+00, 0.4566771620366771e+00, 0.6442896492255246e+00, 0.8079355358528180e+00, 0.6203230486639705e+00, 0.2837483186776231e+00, 0.3695660678607585e+00, 0.6237015546083620e+00, 0.5378838054252265e+00, 0.3072349091217998e+00, 0.3562533181264623e+00, 0.7123973396902394e-01, 0.2222132496436145e-01, 0.3977314805803597e+00, 0.1275094303021685e+01, 0.1042095257923447e+01, 0.1647324354821218e+00, 0.1592685164190333e+00, 0.1007530576543866e+01, 0.1193650220303144e+01, 0.3453881601783118e+00, 0.7752942700755104e-01, 0.2679391719153404e+00, 0.9232775185761617e+00, 0.7328677736683723e+00, 0.6636554427529671e+00, 0.8360828138307410e+00, 0.6537934420370561e+00, 0.4813660709592822e+00, 0.4782322339990674e+00, 0.6495370513676459e+00, 0.4038524053908432e+00, 0.2325475880222648e+00, 0.1147975431483785e+00, 0.1993624802893807e+00, 0.4627968288321279e+00, 0.3782318916911257e+00, 0.8034172324482011e+00, 0.9830449403503746e+00, 0.6226010943062101e+00, 0.4429733864040367e+00, 0.8347746582554776e+00, 0.1173279550602403e+01, 0.8420572992613844e+00, 0.5035524069144587e+00, 0.1829278057427913e+00, 0.3058980395058521e+00, 0.4244336771046062e+00, 0.3014634433415453e+00, 0.9158407747236312e+00, 0.1289422237567578e+01, 0.1131917893423890e+01, 0.7583364305799440e+00, 0.3334922359541972e+00, 0.4977814779461571e+00, 0.4484361267782198e+00, 0.2841468847862598e+00, 0.5865943745895725e+00, 0.9257539794205765e+00, 0.3392119183870583e+00, 0.5231355605450990e-04, 0.3266196269153995e-08, 0.2117869221381033e-04, 0.2117542601754118e-04, 0.0000000000000000e+00, 0.3910066531356214e-03},
			i0Out:    1,
			n0Out:    21,
			ppOut:    0,
			dminOut:  2.1175426017541180e-005,
			desigOut: 0.0000000000000000,
			sigmaOut: 0.0000000000000000,
			qmaxOut:  2.1637041623952107,
			nFailOut: 0,
			iterOut:  3,
			nDivOut:  62,
			ttypeOut: -1,
			dmin1Out: 4.4311601260836921e-002,
			dmin2Out: 4.4311601260836921e-002,
			dnOut:    2.1175426017541180e-005,
			dn1Out:   0.33915960483100382,
			dn2Out:   0.16428924199195991,
			gOut:     0.0000000000000000,
			tauOut:   0.0000000000000000,
		},
		{
			i0:       1,
			n0:       21,
			z:        []float64{0.1914365246180821e+01, 0.2163704162395211e+01, 0.2493389162143899e+00, 0.1516515751224039e+00, 0.1315996513131545e+01, 0.1263329604128848e+01, 0.9898466611970759e-01, 0.4719916727467415e-01, 0.6023973979587287e+00, 0.5574082640946934e+00, 0.2210033410638781e-02, 0.3910066531356214e-03, 0.9861857233678967e-01, 0.5738597141291359e+00, 0.4756321484454819e+00, 0.2145632131068746e+00, 0.2588748143677115e+00, 0.1521727389298373e+00, 0.1078611376690004e+00, 0.4566771620366771e+00, 0.6442896492255246e+00, 0.8079355358528180e+00, 0.6203230486639705e+00, 0.2837483186776231e+00, 0.3695660678607585e+00, 0.6237015546083620e+00, 0.5378838054252265e+00, 0.3072349091217998e+00, 0.3562533181264623e+00, 0.7123973396902394e-01, 0.2222132496436145e-01, 0.3977314805803597e+00, 0.1275094303021685e+01, 0.1042095257923447e+01, 0.1647324354821218e+00, 0.1592685164190333e+00, 0.1007530576543866e+01, 0.1193650220303144e+01, 0.3453881601783118e+00, 0.7752942700755104e-01, 0.2679391719153404e+00, 0.9232775185761617e+00, 0.7328677736683723e+00, 0.6636554427529671e+00, 0.8360828138307410e+00, 0.6537934420370561e+00, 0.4813660709592822e+00, 0.4782322339990674e+00, 0.6495370513676459e+00, 0.4038524053908432e+00, 0.2325475880222648e+00, 0.1147975431483785e+00, 0.1993624802893807e+00, 0.4627968288321279e+00, 0.3782318916911257e+00, 0.8034172324482011e+00, 0.9830449403503746e+00, 0.6226010943062101e+00, 0.4429733864040367e+00, 0.8347746582554776e+00, 0.1173279550602403e+01, 0.8420572992613844e+00, 0.5035524069144587e+00, 0.1829278057427913e+00, 0.3058980395058521e+00, 0.4244336771046062e+00, 0.3014634433415453e+00, 0.9158407747236312e+00, 0.1289422237567578e+01, 0.1131917893423890e+01, 0.7583364305799440e+00, 0.3334922359541972e+00, 0.4977814779461571e+00, 0.4484361267782198e+00, 0.2841468847862598e+00, 0.5865943745895725e+00, 0.9257539794205765e+00, 0.3392119183870583e+00, 0.5231355605450990e-04, 0.3266196269153995e-08, 0.2117869221381033e-04, 0.2117542601754118e-04, 0.0000000000000000e+00, 0.3910066531356214e-03},
			pp:       1,
			dmin:     2.1175426017541180e-005,
			desig:    0.0000000000000000,
			qmax:     2.1637041623952107,
			ttype:    -1,
			dmin1:    4.4311601260836921e-002,
			dmin2:    4.4311601260836921e-002,
			dn:       2.1175426017541180e-005,
			dn1:      0.33915960483100382,
			dn2:      0.16428924199195991,
			g:        0.0000000000000000,
			tau:      0.0000000000000000,
			nFail:    0,
			iter:     3,
			sigma:    0.0000000000000000,
			nDiv:     62,
			zOut:     []float64{0.2315355737517615e+01, 0.2163704162395211e+01, 0.8274578340618610e-01, 0.1516515751224039e+00, 0.1227782987997336e+01, 0.1263329604128848e+01, 0.2142822156235013e-01, 0.4719916727467415e-01, 0.5363710491854788e+00, 0.5574082640946934e+00, 0.4183353417969536e-03, 0.3910066531356214e-03, 0.7880045918942136e+00, 0.5738597141291359e+00, 0.4143462125464707e-01, 0.2145632131068746e+00, 0.5674152797118673e+00, 0.1521727389298373e+00, 0.6502569120260687e+00, 0.4566771620366771e+00, 0.4414269425043723e+00, 0.8079355358528180e+00, 0.4009140594652070e+00, 0.2837483186776231e+00, 0.5300224042649548e+00, 0.6237015546083620e+00, 0.4129510944388858e-01, 0.3072349091217998e+00, 0.4276761051054951e+00, 0.7123973396902394e-01, 0.9691308092544145e+00, 0.3977314805803597e+00, 0.2322329650880660e+00, 0.1042095257923447e+01, 0.8186215063776209e+00, 0.1592685164190333e+00, 0.4525581409330741e+00, 0.1193650220303144e+01, 0.1581701233715052e+00, 0.7752942700755104e-01, 0.1428762837957623e+01, 0.9232775185761617e+00, 0.3036848136842134e+00, 0.6636554427529671e+00, 0.8283408623519102e+00, 0.6537934420370561e+00, 0.2331591338951825e+00, 0.4782322339990674e+00, 0.2854908146440392e+00, 0.4038524053908432e+00, 0.1860933389154074e+00, 0.1147975431483785e+00, 0.1080120722364922e+01, 0.4627968288321279e+00, 0.4631042046962229e+00, 0.8034172324482011e+00, 0.9942715478654648e+00, 0.6226010943062101e+00, 0.7069779837626068e+00, 0.8347746582554776e+00, 0.3180071212415688e+00, 0.8420572992613844e+00, 0.2441477440283845e+00, 0.1829278057427913e+00, 0.1096126707799853e+01, 0.4244336771046062e+00, 0.9457451890006905e+00, 0.9158407747236312e+00, 0.5196649403773971e+00, 0.1131917893423890e+01, 0.2877815203259632e+00, 0.3334922359541972e+00, 0.7472489810418290e+00, 0.4484361267782198e+00, 0.2662831374385604e+00, 0.5865943745895725e+00, 0.7292878421469419e-01, 0.3392119183870583e+00, 0.9483648767903632e-12, 0.3266196269153995e-08, 0.2117542506917630e-04, 0.2117542601754118e-04, 0.4183353417969536e-03, 0.3910066531356214e-03},
			i0Out:    1,
			n0Out:    21,
			ppOut:    1,
			dminOut:  2.1175425069176302e-005,
			desigOut: 0.0000000000000000,
			sigmaOut: 0.0000000000000000,
			qmaxOut:  2.1637041623952107,
			nFailOut: 0,
			iterOut:  4,
			nDivOut:  84,
			ttypeOut: -4,
			dmin1Out: 2.9944624525135358e-002,
			dmin2Out: 2.9944624525135358e-002,
			dnOut:    2.1175425069176302e-005,
			dn1Out:   7.2928780948497918e-002,
			dn2Out:   0.16065460645225654,
			gOut:     0.0000000000000000,
			tauOut:   0.0000000000000000,
		},
		{
			i0:       1,
			n0:       21,
			z:        []float64{0.2315355737517615e+01, 0.2163704162395211e+01, 0.8274578340618610e-01, 0.1516515751224039e+00, 0.1227782987997336e+01, 0.1263329604128848e+01, 0.2142822156235013e-01, 0.4719916727467415e-01, 0.5363710491854788e+00, 0.5574082640946934e+00, 0.4183353417969536e-03, 0.3910066531356214e-03, 0.7880045918942136e+00, 0.5738597141291359e+00, 0.4143462125464707e-01, 0.2145632131068746e+00, 0.5674152797118673e+00, 0.1521727389298373e+00, 0.6502569120260687e+00, 0.4566771620366771e+00, 0.4414269425043723e+00, 0.8079355358528180e+00, 0.4009140594652070e+00, 0.2837483186776231e+00, 0.5300224042649548e+00, 0.6237015546083620e+00, 0.4129510944388858e-01, 0.3072349091217998e+00, 0.4276761051054951e+00, 0.7123973396902394e-01, 0.9691308092544145e+00, 0.3977314805803597e+00, 0.2322329650880660e+00, 0.1042095257923447e+01, 0.8186215063776209e+00, 0.1592685164190333e+00, 0.4525581409330741e+00, 0.1193650220303144e+01, 0.1581701233715052e+00, 0.7752942700755104e-01, 0.1428762837957623e+01, 0.9232775185761617e+00, 0.3036848136842134e+00, 0.6636554427529671e+00, 0.8283408623519102e+00, 0.6537934420370561e+00, 0.2331591338951825e+00, 0.4782322339990674e+00, 0.2854908146440392e+00, 0.4038524053908432e+00, 0.1860933389154074e+00, 0.1147975431483785e+00, 0.1080120722364922e+01, 0.4627968288321279e+00, 0.4631042046962229e+00, 0.8034172324482011e+00, 0.9942715478654648e+00, 0.6226010943062101e+00, 0.7069779837626068e+00, 0.8347746582554776e+00, 0.3180071212415688e+00, 0.8420572992613844e+00, 0.2441477440283845e+00, 0.1829278057427913e+00, 0.1096126707799853e+01, 0.4244336771046062e+00, 0.9457451890006905e+00, 0.9158407747236312e+00, 0.5196649403773971e+00, 0.1131917893423890e+01, 0.2877815203259632e+00, 0.3334922359541972e+00, 0.7472489810418290e+00, 0.4484361267782198e+00, 0.2662831374385604e+00, 0.5865943745895725e+00, 0.7292878421469419e-01, 0.3392119183870583e+00, 0.9483648767903632e-12, 0.3266196269153995e-08, 0.2117542506917630e-04, 0.2117542601754118e-04, 0.4183353417969536e-03, 0.3910066531356214e-03},
			pp:       0,
			dmin:     2.1175425069176302e-005,
			desig:    0.0000000000000000,
			qmax:     2.1637041623952107,
			ttype:    -4,
			dmin1:    2.9944624525135358e-002,
			dmin2:    2.9944624525135358e-002,
			dn:       2.1175425069176302e-005,
			dn1:      7.2928780948497918e-002,
			dn2:      0.16065460645225654,
			g:        0.0000000000000000,
			tau:      0.0000000000000000,
			nFail:    0,
			iter:     4,
			sigma:    0.0000000000000000,
			nDiv:     84,
			zOut:     []float64{0.2315355737517615e+01, 0.2398080345610006e+01, 0.8274578340618610e-01, 0.4236466279397526e-01, 0.1227782987997336e+01, 0.1206825371451915e+01, 0.2142822156235013e-01, 0.9523728911788614e-02, 0.5363710491854788e+00, 0.5272444803016919e+00, 0.4183353417969536e-03, 0.6252320936560726e-03, 0.7880045918942136e+00, 0.8287928057414093e+00, 0.4143462125464707e-01, 0.2836732781232222e-01, 0.5674152797118673e+00, 0.1189283688611819e+01, 0.6502569120260687e+00, 0.2413561400585997e+00, 0.4414269425043723e+00, 0.6009636865971842e+00, 0.4009140594652070e+00, 0.3535878097802652e+00, 0.5300224042649548e+00, 0.2177085286147829e+00, 0.4129510944388858e-01, 0.8112190955144877e-01, 0.4276761051054951e+00, 0.1315663829494665e+01, 0.9691308092544145e+00, 0.1710650671895379e+00, 0.2322329650880660e+00, 0.8797682289623537e+00, 0.8186215063776209e+00, 0.4211038940233675e+00, 0.4525581409330741e+00, 0.1896031949674164e+00, 0.1581701233715052e+00, 0.1191897606932286e+01, 0.1428762837957623e+01, 0.5405288693957555e+00, 0.3036848136842134e+00, 0.4653859482687157e+00, 0.8283408623519102e+00, 0.5960928726645816e+00, 0.2331591338951825e+00, 0.1116684901463164e+00, 0.2854908146440392e+00, 0.3598944880993349e+00, 0.1860933389154074e+00, 0.5585061130503639e+00, 0.1080120722364922e+01, 0.9846976386969850e+00, 0.4631042046962229e+00, 0.4676068229793028e+00, 0.9942715478654648e+00, 0.1233621533334973e+01, 0.7069779837626068e+00, 0.1822471700779458e+00, 0.3180071212415688e+00, 0.3798865198782122e+00, 0.2441477440283845e+00, 0.7044652781161848e+00, 0.1096126707799853e+01, 0.1337385443370563e+01, 0.9457451890006905e+00, 0.3674861422265960e+00, 0.5196649403773971e+00, 0.4399391431629689e+00, 0.2877815203259632e+00, 0.4888049885267526e+00, 0.7472489810418290e+00, 0.5247059546398414e+00, 0.2662831374385604e+00, 0.3701064434002514e-01, 0.7292878421469419e-01, 0.3589696456182207e-01, 0.9483648767903632e-12, 0.5594353069081231e-15, 0.2117542506917630e-04, 0.1112732565966979e-09, 0.4183353417969536e-03, 0.6252320936560726e-03},
			i0Out:    1,
			n0Out:    21,
			ppOut:    0,
			dminOut:  1.1127325659669794e-010,
			desigOut: 0.0000000000000000,
			sigmaOut: 2.1175313795360271e-005,
			qmaxOut:  2.1637041623952107,
			nFailOut: 0,
			iterOut:  5,
			nDivOut:  106,
			ttypeOut: -4,
			dmin1Out: 3.1433071595911154e-002,
			dmin2Out: 3.1433071595911154e-002,
			dnOut:    1.1127325659669794e-010,
			dn1Out:   3.5896964560873705e-002,
			dn2Out:   0.25842281720128102,
			gOut:     0.0000000000000000,
			tauOut:   2.1175313795360271e-005,
		},
		{
			i0:       1,
			n0:       21,
			z:        []float64{0.2315355737517615e+01, 0.2398080345610006e+01, 0.8274578340618610e-01, 0.4236466279397526e-01, 0.1227782987997336e+01, 0.1206825371451915e+01, 0.2142822156235013e-01, 0.9523728911788614e-02, 0.5363710491854788e+00, 0.5272444803016919e+00, 0.4183353417969536e-03, 0.6252320936560726e-03, 0.7880045918942136e+00, 0.8287928057414093e+00, 0.4143462125464707e-01, 0.2836732781232222e-01, 0.5674152797118673e+00, 0.1189283688611819e+01, 0.6502569120260687e+00, 0.2413561400585997e+00, 0.4414269425043723e+00, 0.6009636865971842e+00, 0.4009140594652070e+00, 0.3535878097802652e+00, 0.5300224042649548e+00, 0.2177085286147829e+00, 0.4129510944388858e-01, 0.8112190955144877e-01, 0.4276761051054951e+00, 0.1315663829494665e+01, 0.9691308092544145e+00, 0.1710650671895379e+00, 0.2322329650880660e+00, 0.8797682289623537e+00, 0.8186215063776209e+00, 0.4211038940233675e+00, 0.4525581409330741e+00, 0.1896031949674164e+00, 0.1581701233715052e+00, 0.1191897606932286e+01, 0.1428762837957623e+01, 0.5405288693957555e+00, 0.3036848136842134e+00, 0.4653859482687157e+00, 0.8283408623519102e+00, 0.5960928726645816e+00, 0.2331591338951825e+00, 0.1116684901463164e+00, 0.2854908146440392e+00, 0.3598944880993349e+00, 0.1860933389154074e+00, 0.5585061130503639e+00, 0.1080120722364922e+01, 0.9846976386969850e+00, 0.4631042046962229e+00, 0.4676068229793028e+00, 0.9942715478654648e+00, 0.1233621533334973e+01, 0.7069779837626068e+00, 0.1822471700779458e+00, 0.3180071212415688e+00, 0.3798865198782122e+00, 0.2441477440283845e+00, 0.7044652781161848e+00, 0.1096126707799853e+01, 0.1337385443370563e+01, 0.9457451890006905e+00, 0.3674861422265960e+00, 0.5196649403773971e+00, 0.4399391431629689e+00, 0.2877815203259632e+00, 0.4888049885267526e+00, 0.7472489810418290e+00, 0.5247059546398414e+00, 0.2662831374385604e+00, 0.3701064434002514e-01, 0.7292878421469419e-01, 0.3589696456182207e-01, 0.9483648767903632e-12, 0.5594353069081231e-15, 0.2117542506917630e-04, 0.1112732565966979e-09, 0.4183353417969536e-03, 0.6252320936560726e-03},
			pp:       1,
			dmin:     1.1127325659669794e-010,
			desig:    0.0000000000000000,
			qmax:     2.1637041623952107,
			ttype:    -4,
			dmin1:    3.1433071595911154e-002,
			dmin2:    3.1433071595911154e-002,
			dn:       1.1127325659669794e-010,
			dn1:      3.5896964560873705e-002,
			dn2:      0.25842281720128102,
			g:        0.0000000000000000,
			tau:      2.1175313795360271e-005,
			nFail:    0,
			iter:     5,
			sigma:    2.1175313795360271e-005,
			nDiv:     106,
			zOut:     []float64{0.2440445008292708e+01, 0.2398080345610006e+01, 0.2094976520226600e-01, 0.4236466279397526e-01, 0.1195399335050165e+01, 0.1206825371451915e+01, 0.4200549016048655e-02, 0.9523728911788614e-02, 0.5236691632680260e+00, 0.5272444803016919e+00, 0.9895328911616120e-03, 0.6252320936560726e-03, 0.8561706005512968e+00, 0.8287928057414093e+00, 0.3940429656773515e-01, 0.2836732781232222e-01, 0.1391235531991410e+01, 0.1189283688611819e+01, 0.1042571673718422e+00, 0.2413561400585997e+00, 0.8502943288943339e+00, 0.6009636865971842e+00, 0.9053227710395735e-01, 0.3535878097802652e+00, 0.2082981609510011e+00, 0.2177085286147829e+00, 0.5123864833424303e+00, 0.8112190955144877e-01, 0.9743424132304999e+00, 0.1315663829494665e+01, 0.1544607000116935e+00, 0.1710650671895379e+00, 0.1146411422862754e+01, 0.8797682289623537e+00, 0.6964571542795012e-01, 0.4211038940233675e+00, 0.1311855086360479e+01, 0.1896031949674164e+00, 0.4911023119923957e+00, 0.1191897606932286e+01, 0.5148125055608023e+00, 0.5405288693957555e+00, 0.5388626806938843e+00, 0.4653859482687157e+00, 0.1688986820057405e+00, 0.5960928726645816e+00, 0.2379466412690434e+00, 0.1116684901463164e+00, 0.6804539597693821e+00, 0.3598944880993349e+00, 0.8082246312519304e+00, 0.5585061130503639e+00, 0.6440798303130841e+00, 0.9846976386969850e+00, 0.8956185534970393e+00, 0.4676068229793028e+00, 0.5202501498046066e+00, 0.1233621533334973e+01, 0.1330768347199243e+00, 0.1822471700779458e+00, 0.9512749631631994e+00, 0.3798865198782122e+00, 0.9903988276741268e+00, 0.7044652781161848e+00, 0.7144727578117591e+00, 0.1337385443370563e+01, 0.2262808998212762e+00, 0.3674861422265960e+00, 0.7024632317571722e+00, 0.4399391431629689e+00, 0.3651136124179467e+00, 0.4888049885267526e+00, 0.1966029864506465e+00, 0.5247059546398414e+00, 0.6757627705811050e-02, 0.3701064434002514e-01, 0.2913933674473832e-01, 0.3589696456182207e-01, 0.2136293938333395e-23, 0.5594353069081231e-15, 0.0000000000000000e+00, 0.1112732565966979e-09, 0.9895328911616120e-03, 0.6252320936560726e-03},
			i0Out:    1,
			n0Out:    21,
			ppOut:    1,
			dminOut:  0.0000000000000000,
			desigOut: -5.1698788284564230e-026,
			sigmaOut: 2.1175425068616867e-005,
			qmaxOut:  2.1637041623952107,
			nFailOut: 1,
			iterOut:  7,
			nDivOut:  150,
			ttypeOut: -15,
			dmin1Out: 2.9139336744737766e-002,
			dmin2Out: 4.9426557292086552e-002,
			dnOut:    -2.0808762284537102e-024,
			dn1Out:   2.9139336744737766e-002,
			dn2Out:   0.15959234211062134,
			gOut:     0.0000000000000000,
			tauOut:   1.1127325659669789e-010,
		},
		{
			i0:       1,
			n0:       21,
			z:        []float64{0.2440445008292708e+01, 0.2398080345610006e+01, 0.2094976520226600e-01, 0.4236466279397526e-01, 0.1195399335050165e+01, 0.1206825371451915e+01, 0.4200549016048655e-02, 0.9523728911788614e-02, 0.5236691632680260e+00, 0.5272444803016919e+00, 0.9895328911616120e-03, 0.6252320936560726e-03, 0.8561706005512968e+00, 0.8287928057414093e+00, 0.3940429656773515e-01, 0.2836732781232222e-01, 0.1391235531991410e+01, 0.1189283688611819e+01, 0.1042571673718422e+00, 0.2413561400585997e+00, 0.8502943288943339e+00, 0.6009636865971842e+00, 0.9053227710395735e-01, 0.3535878097802652e+00, 0.2082981609510011e+00, 0.2177085286147829e+00, 0.5123864833424303e+00, 0.8112190955144877e-01, 0.9743424132304999e+00, 0.1315663829494665e+01, 0.1544607000116935e+00, 0.1710650671895379e+00, 0.1146411422862754e+01, 0.8797682289623537e+00, 0.6964571542795012e-01, 0.4211038940233675e+00, 0.1311855086360479e+01, 0.1896031949674164e+00, 0.4911023119923957e+00, 0.1191897606932286e+01, 0.5148125055608023e+00, 0.5405288693957555e+00, 0.5388626806938843e+00, 0.4653859482687157e+00, 0.1688986820057405e+00, 0.5960928726645816e+00, 0.2379466412690434e+00, 0.1116684901463164e+00, 0.6804539597693821e+00, 0.3598944880993349e+00, 0.8082246312519304e+00, 0.5585061130503639e+00, 0.6440798303130841e+00, 0.9846976386969850e+00, 0.8956185534970393e+00, 0.4676068229793028e+00, 0.5202501498046066e+00, 0.1233621533334973e+01, 0.1330768347199243e+00, 0.1822471700779458e+00, 0.9512749631631994e+00, 0.3798865198782122e+00, 0.9903988276741268e+00, 0.7044652781161848e+00, 0.7144727578117591e+00, 0.1337385443370563e+01, 0.2262808998212762e+00, 0.3674861422265960e+00, 0.7024632317571722e+00, 0.4399391431629689e+00, 0.3651136124179467e+00, 0.4888049885267526e+00, 0.1966029864506465e+00, 0.5247059546398414e+00, 0.6757627705811050e-02, 0.3701064434002514e-01, 0.2913933674473832e-01, 0.3589696456182207e-01, 0.2136293938333395e-23, 0.5594353069081231e-15, 0.0000000000000000e+00, 0.1112732565966979e-09, 0.9895328911616120e-03, 0.6252320936560726e-03},
			pp:       0,
			dmin:     0.0000000000000000,
			desig:    -5.1698788284564230e-026,
			qmax:     2.1637041623952107,
			ttype:    -15,
			dmin1:    2.9139336744737766e-002,
			dmin2:    4.9426557292086552e-002,
			dn:       -2.0808762284537102e-024,
			dn1:      2.9139336744737766e-002,
			dn2:      0.15959234211062134,
			g:        0.0000000000000000,
			tau:      1.1127325659669789e-010,
			nFail:    1,
			iter:     7,
			sigma:    2.1175425068616867e-005,
			nDiv:     150,
			zOut:     []float64{0.2440445008292708e+01, 0.2461394773494974e+01, 0.2094976520226600e-01, 0.1017444891892999e-01, 0.1195399335050165e+01, 0.1189425435147283e+01, 0.4200549016048655e-02, 0.1849378635683999e-02, 0.5236691632680260e+00, 0.5228093175235037e+00, 0.9895328911616120e-03, 0.1620493249248586e-02, 0.8561706005512968e+00, 0.8939544038697832e+00, 0.3940429656773515e-01, 0.6132377362967349e-01, 0.1391235531991410e+01, 0.1434168925733579e+01, 0.1042571673718422e+00, 0.6181229879703373e-01, 0.8502943288943339e+00, 0.8790143072012576e+00, 0.9053227710395735e-01, 0.2145324219750511e-01, 0.2082981609510011e+00, 0.6992314020959263e+00, 0.5123864833424303e+00, 0.7139837844669097e+00, 0.9743424132304999e+00, 0.4148193287752837e+00, 0.1544607000116935e+00, 0.4268738185358478e+00, 0.1146411422862754e+01, 0.7891833197548568e+00, 0.6964571542795012e-01, 0.1157716892137957e+00, 0.1311855086360479e+01, 0.1687185709139079e+01, 0.4911023119923957e+00, 0.1498504938454686e+00, 0.5148125055608023e+00, 0.9038246924092180e+00, 0.5388626806938843e+00, 0.1006978425303630e+00, 0.1688986820057405e+00, 0.3061474807444209e+00, 0.2379466412690434e+00, 0.5288684194677825e+00, 0.6804539597693821e+00, 0.9598101715535300e+00, 0.8082246312519304e+00, 0.5423584775195998e+00, 0.6440798303130841e+00, 0.9973399062905237e+00, 0.8956185534970393e+00, 0.4671884516860899e+00, 0.5202501498046066e+00, 0.1861385328384410e+00, 0.1330768347199243e+00, 0.6800991665489665e+00, 0.9512749631631994e+00, 0.1261574624288360e+01, 0.9903988276741268e+00, 0.5608966509936130e+00, 0.7144727578117591e+00, 0.3798570066394223e+00, 0.2262808998212762e+00, 0.4184574968871406e+00, 0.7024632317571722e+00, 0.6491193472879784e+00, 0.3651136124179467e+00, 0.1105843276664904e+00, 0.1966029864506465e+00, 0.9277628648996712e-01, 0.6757627705811050e-02, 0.2122447413720272e-02, 0.2913933674473832e-01, 0.2701688933101806e-01, 0.2136293938333395e-23, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			i0Out:    1,
			n0Out:    21,
			ppOut:    0,
			dminOut:  0.0000000000000000,
			desigOut: -5.1698788284564230e-026,
			sigmaOut: 2.1175425068616867e-005,
			qmaxOut:  2.1637041623952107,
			nFailOut: 1,
			iterOut:  8,
			nDivOut:  172,
			ttypeOut: -1,
			dmin1Out: 2.7016889331018056e-002,
			dmin2Out: 5.3061698118516694e-002,
			dnOut:    0.0000000000000000,
			dn1Out:   2.7016889331018056e-002,
			dn2Out:   8.6018658784156071e-002,
			gOut:     0.0000000000000000,
			tauOut:   0.0000000000000000,
		},
		{
			i0:       1,
			n0:       21,
			z:        []float64{0.2440445008292708e+01, 0.2461394773494974e+01, 0.2094976520226600e-01, 0.1017444891892999e-01, 0.1195399335050165e+01, 0.1189425435147283e+01, 0.4200549016048655e-02, 0.1849378635683999e-02, 0.5236691632680260e+00, 0.5228093175235037e+00, 0.9895328911616120e-03, 0.1620493249248586e-02, 0.8561706005512968e+00, 0.8939544038697832e+00, 0.3940429656773515e-01, 0.6132377362967349e-01, 0.1391235531991410e+01, 0.1434168925733579e+01, 0.1042571673718422e+00, 0.6181229879703373e-01, 0.8502943288943339e+00, 0.8790143072012576e+00, 0.9053227710395735e-01, 0.2145324219750511e-01, 0.2082981609510011e+00, 0.6992314020959263e+00, 0.5123864833424303e+00, 0.7139837844669097e+00, 0.9743424132304999e+00, 0.4148193287752837e+00, 0.1544607000116935e+00, 0.4268738185358478e+00, 0.1146411422862754e+01, 0.7891833197548568e+00, 0.6964571542795012e-01, 0.1157716892137957e+00, 0.1311855086360479e+01, 0.1687185709139079e+01, 0.4911023119923957e+00, 0.1498504938454686e+00, 0.5148125055608023e+00, 0.9038246924092180e+00, 0.5388626806938843e+00, 0.1006978425303630e+00, 0.1688986820057405e+00, 0.3061474807444209e+00, 0.2379466412690434e+00, 0.5288684194677825e+00, 0.6804539597693821e+00, 0.9598101715535300e+00, 0.8082246312519304e+00, 0.5423584775195998e+00, 0.6440798303130841e+00, 0.9973399062905237e+00, 0.8956185534970393e+00, 0.4671884516860899e+00, 0.5202501498046066e+00, 0.1861385328384410e+00, 0.1330768347199243e+00, 0.6800991665489665e+00, 0.9512749631631994e+00, 0.1261574624288360e+01, 0.9903988276741268e+00, 0.5608966509936130e+00, 0.7144727578117591e+00, 0.3798570066394223e+00, 0.2262808998212762e+00, 0.4184574968871406e+00, 0.7024632317571722e+00, 0.6491193472879784e+00, 0.3651136124179467e+00, 0.1105843276664904e+00, 0.1966029864506465e+00, 0.9277628648996712e-01, 0.6757627705811050e-02, 0.2122447413720272e-02, 0.2913933674473832e-01, 0.2701688933101806e-01, 0.2136293938333395e-23, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			pp:       1,
			dmin:     0.0000000000000000,
			desig:    -5.1698788284564230e-026,
			qmax:     2.1637041623952107,
			ttype:    -1,
			dmin1:    2.7016889331018056e-002,
			dmin2:    5.3061698118516694e-002,
			dn:       0.0000000000000000,
			dn1:      2.7016889331018056e-002,
			dn2:      8.6018658784156071e-002,
			g:        0.0000000000000000,
			tau:      0.0000000000000000,
			nFail:    1,
			iter:     8,
			sigma:    2.1175425068616867e-005,
			nDiv:     172,
			zOut:     []float64{0.2471569222413904e+01, 0.2461394773494974e+01, 0.4896382518051712e-02, 0.1017444891892999e-01, 0.1186378431264915e+01, 0.1189425435147283e+01, 0.8149780515932184e-03, 0.1849378635683999e-02, 0.5236148327211592e+00, 0.5228093175235037e+00, 0.2766627272719901e-02, 0.1620493249248586e-02, 0.9525115502267366e+00, 0.8939544038697832e+00, 0.9233342160256496e-01, 0.6132377362967349e-01, 0.1403647802928048e+01, 0.1434168925733579e+01, 0.3870906568602875e-01, 0.6181229879703373e-01, 0.8617584837127339e+00, 0.8790143072012576e+00, 0.1740717486950262e-01, 0.2145324219750511e-01, 0.1395808011693333e+01, 0.6992314020959263e+00, 0.2121884039551361e+00, 0.7139837844669097e+00, 0.6295047433559955e+00, 0.4148193287752837e+00, 0.5351535485381410e+00, 0.4268738185358478e+00, 0.3698014604305115e+00, 0.7891833197548568e+00, 0.5281978587564573e+00, 0.1157716892137957e+00, 0.1308838344228090e+01, 0.1687185709139079e+01, 0.1034799882693896e+00, 0.1498504938454686e+00, 0.9010425466701916e+00, 0.9038246924092180e+00, 0.3421413441684364e-01, 0.1006978425303630e+00, 0.8008017657953598e+00, 0.3061474807444209e+00, 0.6338813300623194e+00, 0.5288684194677825e+00, 0.8682873190108105e+00, 0.9598101715535300e+00, 0.6229686202966810e+00, 0.5423584775195998e+00, 0.8415597376799326e+00, 0.9973399062905237e+00, 0.1033340463692495e+00, 0.4671884516860899e+00, 0.7629036530181579e+00, 0.1861385328384410e+00, 0.1124645093942705e+01, 0.6800991665489665e+00, 0.6978261813392677e+00, 0.1261574624288360e+01, 0.3053203341720497e+00, 0.5608966509936130e+00, 0.4929941693545132e+00, 0.3798570066394223e+00, 0.5509778292160957e+00, 0.4184574968871406e+00, 0.2087258457383731e+00, 0.6491193472879784e+00, 0.4915348757406203e-01, 0.1105843276664904e+00, 0.4574524632962537e-01, 0.9277628648996712e-01, 0.1253505697055357e-02, 0.2122447413720272e-02, 0.2576338363396270e-01, 0.2701688933101806e-01, 0.8149780515932184e-03, 0.0000000000000000e+00, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			i0Out:    1,
			n0Out:    20,
			ppOut:    1,
			dminOut:  2.5763383633962696e-002,
			desigOut: -5.1698788284564230e-026,
			sigmaOut: 2.1175425068616867e-005,
			qmaxOut:  2.1637041623952107,
			nFailOut: 1,
			iterOut:  9,
			nDivOut:  193,
			ttypeOut: -1,
			dmin1Out: 4.3622798915905092e-002,
			dmin2Out: 7.4536672467372611e-002,
			dnOut:    2.5763383633962696e-002,
			dn1Out:   4.3622798915905092e-002,
			dn2Out:   9.8141518071882677e-002,
			gOut:     0.0000000000000000,
			tauOut:   0.0000000000000000,
		},
		{
			i0:       1,
			n0:       20,
			z:        []float64{0.2471569222413904e+01, 0.2461394773494974e+01, 0.4896382518051712e-02, 0.1017444891892999e-01, 0.1186378431264915e+01, 0.1189425435147283e+01, 0.8149780515932184e-03, 0.1849378635683999e-02, 0.5236148327211592e+00, 0.5228093175235037e+00, 0.2766627272719901e-02, 0.1620493249248586e-02, 0.9525115502267366e+00, 0.8939544038697832e+00, 0.9233342160256496e-01, 0.6132377362967349e-01, 0.1403647802928048e+01, 0.1434168925733579e+01, 0.3870906568602875e-01, 0.6181229879703373e-01, 0.8617584837127339e+00, 0.8790143072012576e+00, 0.1740717486950262e-01, 0.2145324219750511e-01, 0.1395808011693333e+01, 0.6992314020959263e+00, 0.2121884039551361e+00, 0.7139837844669097e+00, 0.6295047433559955e+00, 0.4148193287752837e+00, 0.5351535485381410e+00, 0.4268738185358478e+00, 0.3698014604305115e+00, 0.7891833197548568e+00, 0.5281978587564573e+00, 0.1157716892137957e+00, 0.1308838344228090e+01, 0.1687185709139079e+01, 0.1034799882693896e+00, 0.1498504938454686e+00, 0.9010425466701916e+00, 0.9038246924092180e+00, 0.3421413441684364e-01, 0.1006978425303630e+00, 0.8008017657953598e+00, 0.3061474807444209e+00, 0.6338813300623194e+00, 0.5288684194677825e+00, 0.8682873190108105e+00, 0.9598101715535300e+00, 0.6229686202966810e+00, 0.5423584775195998e+00, 0.8415597376799326e+00, 0.9973399062905237e+00, 0.1033340463692495e+00, 0.4671884516860899e+00, 0.7629036530181579e+00, 0.1861385328384410e+00, 0.1124645093942705e+01, 0.6800991665489665e+00, 0.6978261813392677e+00, 0.1261574624288360e+01, 0.3053203341720497e+00, 0.5608966509936130e+00, 0.4929941693545132e+00, 0.3798570066394223e+00, 0.5509778292160957e+00, 0.4184574968871406e+00, 0.2087258457383731e+00, 0.6491193472879784e+00, 0.4915348757406203e-01, 0.1105843276664904e+00, 0.4574524632962537e-01, 0.9277628648996712e-01, 0.1253505697055357e-02, 0.2122447413720272e-02, 0.2576338363396270e-01, 0.2701688933101806e-01, 0.8149780515932184e-03, 0.1620493249248586e-02, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			pp:       0,
			dmin:     2.5763383633962696e-002,
			desig:    -5.1698788284564230e-026,
			qmax:     2.4715692224139039,
			ttype:    -1,
			dmin1:    4.3622798915905092e-002,
			dmin2:    7.4536672467372611e-002,
			dn:       2.5763383633962696e-002,
			dn1:      4.3622798915905092e-002,
			dn2:      9.8141518071882677e-002,
			g:        0.0000000000000000,
			tau:      0.0000000000000000,
			nFail:    1,
			iter:     9,
			sigma:    2.1175425068616867e-005,
			nDiv:     193,
			zOut:     []float64{0.2471569222413904e+01, 0.2471445466333236e+01, 0.4896382518051712e-02, 0.2350431231346416e-02, 0.1186378431264915e+01, 0.1179822839486443e+01, 0.8149780515932184e-03, 0.3616937915375072e-03, 0.5236148327211592e+00, 0.5209996276036221e+00, 0.2766627272719901e-02, 0.5058054349403302e-02, 0.9525115502267366e+00, 0.1034766778881179e+01, 0.9233342160256496e-01, 0.1252490967185870e+00, 0.1403647802928048e+01, 0.1312087633296770e+01, 0.3870906568602875e-01, 0.2542350442532051e-01, 0.8617584837127339e+00, 0.8487220155581966e+00, 0.1740717486950262e-01, 0.2862783537884150e-01, 0.1395808011693333e+01, 0.1574348441670908e+01, 0.2121884039551361e+00, 0.8484373804386666e-01, 0.6295047433559955e+00, 0.1074794415251550e+01, 0.5351535485381410e+00, 0.1841287608083240e+00, 0.3698014604305115e+00, 0.7088504197799252e+00, 0.5281978587564573e+00, 0.9752771411128711e+00, 0.1308838344228090e+01, 0.4320210527858890e+00, 0.1034799882693896e+00, 0.2158225196628609e+00, 0.9010425466701916e+00, 0.7144140228254550e+00, 0.3421413441684364e-01, 0.3835134583138245e-01, 0.8008017657953598e+00, 0.1391311611427577e+01, 0.6338813300623194e+00, 0.3955915526975877e+00, 0.8682873190108105e+00, 0.1090644248011184e+01, 0.6229686202966810e+00, 0.4806932321292802e+00, 0.8415597376799326e+00, 0.4591804133211825e+00, 0.1033340463692495e+00, 0.1716839812178710e+00, 0.7629036530181579e+00, 0.1710844627144272e+01, 0.1124645093942705e+01, 0.4587247601659613e+00, 0.6978261813392677e+00, 0.5394016167466366e+00, 0.3053203341720497e+00, 0.2790520826393697e+00, 0.4929941693545132e+00, 0.7598997773325197e+00, 0.5509778292160957e+00, 0.1513401067044909e+00, 0.2087258457383731e+00, 0.1015190880092246e+00, 0.4915348757406203e-01, 0.2214892234681356e-01, 0.4574524632962537e-01, 0.1982969108114764e-01, 0.1253505697055357e-02, 0.1628595626045726e-02, 0.2576338363396270e-01, 0.1911464940919745e-01, 0.8149780515932184e-03, 0.3616937915375072e-03, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			i0Out:    1,
			n0Out:    20,
			ppOut:    0,
			dminOut:  1.8576185384092288e-002,
			desigOut: 2.6427422784455342e-019,
			sigmaOut: 5.0413140237881371e-003,
			qmaxOut:  2.4715692224139039,
			nFailOut: 2,
			iterOut:  11,
			nDivOut:  235,
			ttypeOut: -15,
			dmin1Out: 1.8576185384092288e-002,
			dmin2Out: 5.2365600435162571e-002,
			dnOut:    1.9114649409197451e-002,
			dn1Out:   1.8576185384092288e-002,
			dn2Out:   5.2365600435162571e-002,
			gOut:     0.0000000000000000,
			tauOut:   5.0201385987195205e-003,
		},
		{
			i0:       1,
			n0:       20,
			z:        []float64{0.2471569222413904e+01, 0.2471445466333236e+01, 0.4896382518051712e-02, 0.2350431231346416e-02, 0.1186378431264915e+01, 0.1179822839486443e+01, 0.8149780515932184e-03, 0.3616937915375072e-03, 0.5236148327211592e+00, 0.5209996276036221e+00, 0.2766627272719901e-02, 0.5058054349403302e-02, 0.9525115502267366e+00, 0.1034766778881179e+01, 0.9233342160256496e-01, 0.1252490967185870e+00, 0.1403647802928048e+01, 0.1312087633296770e+01, 0.3870906568602875e-01, 0.2542350442532051e-01, 0.8617584837127339e+00, 0.8487220155581966e+00, 0.1740717486950262e-01, 0.2862783537884150e-01, 0.1395808011693333e+01, 0.1574348441670908e+01, 0.2121884039551361e+00, 0.8484373804386666e-01, 0.6295047433559955e+00, 0.1074794415251550e+01, 0.5351535485381410e+00, 0.1841287608083240e+00, 0.3698014604305115e+00, 0.7088504197799252e+00, 0.5281978587564573e+00, 0.9752771411128711e+00, 0.1308838344228090e+01, 0.4320210527858890e+00, 0.1034799882693896e+00, 0.2158225196628609e+00, 0.9010425466701916e+00, 0.7144140228254550e+00, 0.3421413441684364e-01, 0.3835134583138245e-01, 0.8008017657953598e+00, 0.1391311611427577e+01, 0.6338813300623194e+00, 0.3955915526975877e+00, 0.8682873190108105e+00, 0.1090644248011184e+01, 0.6229686202966810e+00, 0.4806932321292802e+00, 0.8415597376799326e+00, 0.4591804133211825e+00, 0.1033340463692495e+00, 0.1716839812178710e+00, 0.7629036530181579e+00, 0.1710844627144272e+01, 0.1124645093942705e+01, 0.4587247601659613e+00, 0.6978261813392677e+00, 0.5394016167466366e+00, 0.3053203341720497e+00, 0.2790520826393697e+00, 0.4929941693545132e+00, 0.7598997773325197e+00, 0.5509778292160957e+00, 0.1513401067044909e+00, 0.2087258457383731e+00, 0.1015190880092246e+00, 0.4915348757406203e-01, 0.2214892234681356e-01, 0.4574524632962537e-01, 0.1982969108114764e-01, 0.1253505697055357e-02, 0.1628595626045726e-02, 0.2576338363396270e-01, 0.1911464940919745e-01, 0.8149780515932184e-03, 0.3616937915375072e-03, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			pp:       1,
			dmin:     1.8576185384092288e-002,
			desig:    2.6427422784455342e-019,
			qmax:     2.4715692224139039,
			ttype:    -15,
			dmin1:    1.8576185384092288e-002,
			dmin2:    5.2365600435162571e-002,
			dn:       1.9114649409197451e-002,
			dn1:      1.8576185384092288e-002,
			dn2:      5.2365600435162571e-002,
			g:        0.0000000000000000,
			tau:      5.0201385987195205e-003,
			nFail:    2,
			iter:     11,
			sigma:    5.0413140237881371e-003,
			nDiv:     235,
			zOut:     []float64{0.2468318984233055e+01, 0.2471445466333236e+01, 0.1123474100024551e-02, 0.2350431231346416e-02, 0.1173584145846428e+01, 0.1179822839486443e+01, 0.1605699355811189e-03, 0.3616937915375072e-03, 0.5204201986859162e+00, 0.5209996276036221e+00, 0.1005707814522541e-01, 0.5058054349403302e-02, 0.1144481884123012e+01, 0.1034766778881179e+01, 0.1435914304680996e+00, 0.1252490967185870e+00, 0.1188442793922463e+01, 0.1312087633296770e+01, 0.1815610143690141e-01, 0.2542350442532051e-01, 0.8537168361686087e+00, 0.8487220155581966e+00, 0.5279290053521807e-01, 0.2862783537884150e-01, 0.1600922365848029e+01, 0.1574348441670908e+01, 0.5696064828871891e-01, 0.8484373804386666e-01, 0.1196485614439627e+01, 0.1074794415251550e+01, 0.1090859328498209e+00, 0.1841287608083240e+00, 0.1569564714711448e+01, 0.7088504197799252e+00, 0.2684440171930437e+00, 0.9752771411128711e+00, 0.3739226419241781e+00, 0.4320210527858890e+00, 0.4123490187575627e+00, 0.2158225196628609e+00, 0.3349394365677468e+00, 0.7144140228254550e+00, 0.1593084209965356e+00, 0.3835134583138245e-01, 0.1622117829797102e+01, 0.1391311611427577e+01, 0.2659792301064862e+00, 0.3955915526975877e+00, 0.1299881336702450e+01, 0.1090644248011184e+01, 0.1698038973078534e+00, 0.4806932321292802e+00, 0.4555835838996722e+00, 0.4591804133211825e+00, 0.6447216871142054e+00, 0.1716839812178710e+00, 0.1519370786864500e+01, 0.1710844627144272e+01, 0.1628548339973444e+00, 0.4587247601659613e+00, 0.6501219520571339e+00, 0.5394016167466366e+00, 0.3261720586281595e+00, 0.2790520826393697e+00, 0.5795909120773233e+00, 0.7598997773325197e+00, 0.2650819619788820e-01, 0.1513401067044909e+00, 0.9168290082662192e-01, 0.1015190880092246e+00, 0.4790492926791300e-02, 0.2214892234681356e-01, 0.1119088044887405e-01, 0.1982969108114764e-01, 0.2781732372482683e-02, 0.1628595626045726e-02, 0.1085600370518675e-01, 0.1911464940919745e-01, 0.1605699355811189e-03, 0.3616937915375072e-03, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			i0Out:    1,
			n0Out:    20,
			ppOut:    1,
			dminOut:  9.5622848228283271e-003,
			desigOut: -6.0308751014385013e-019,
			sigmaOut: 1.0518227355316156e-002,
			qmaxOut:  2.4715692224139039,
			nFailOut: 2,
			iterOut:  12,
			nDivOut:  256,
			ttypeOut: -4,
			dmin1Out: 9.5622848228283271e-003,
			dmin2Out: 6.9533978479808370e-002,
			dnOut:    1.0856003705186750e-002,
			dn1Out:   9.5622848228283271e-003,
			dn2Out:   6.9533978479808370e-002,
			gOut:     0.0000000000000000,
			tauOut:   5.4769133315280185e-003,
		},
		{
			i0:       1,
			n0:       20,
			z:        []float64{0.2468318984233055e+01, 0.2471445466333236e+01, 0.1123474100024551e-02, 0.2350431231346416e-02, 0.1173584145846428e+01, 0.1179822839486443e+01, 0.1605699355811189e-03, 0.3616937915375072e-03, 0.5204201986859162e+00, 0.5209996276036221e+00, 0.1005707814522541e-01, 0.5058054349403302e-02, 0.1144481884123012e+01, 0.1034766778881179e+01, 0.1435914304680996e+00, 0.1252490967185870e+00, 0.1188442793922463e+01, 0.1312087633296770e+01, 0.1815610143690141e-01, 0.2542350442532051e-01, 0.8537168361686087e+00, 0.8487220155581966e+00, 0.5279290053521807e-01, 0.2862783537884150e-01, 0.1600922365848029e+01, 0.1574348441670908e+01, 0.5696064828871891e-01, 0.8484373804386666e-01, 0.1196485614439627e+01, 0.1074794415251550e+01, 0.1090859328498209e+00, 0.1841287608083240e+00, 0.1569564714711448e+01, 0.7088504197799252e+00, 0.2684440171930437e+00, 0.9752771411128711e+00, 0.3739226419241781e+00, 0.4320210527858890e+00, 0.4123490187575627e+00, 0.2158225196628609e+00, 0.3349394365677468e+00, 0.7144140228254550e+00, 0.1593084209965356e+00, 0.3835134583138245e-01, 0.1622117829797102e+01, 0.1391311611427577e+01, 0.2659792301064862e+00, 0.3955915526975877e+00, 0.1299881336702450e+01, 0.1090644248011184e+01, 0.1698038973078534e+00, 0.4806932321292802e+00, 0.4555835838996722e+00, 0.4591804133211825e+00, 0.6447216871142054e+00, 0.1716839812178710e+00, 0.1519370786864500e+01, 0.1710844627144272e+01, 0.1628548339973444e+00, 0.4587247601659613e+00, 0.6501219520571339e+00, 0.5394016167466366e+00, 0.3261720586281595e+00, 0.2790520826393697e+00, 0.5795909120773233e+00, 0.7598997773325197e+00, 0.2650819619788820e-01, 0.1513401067044909e+00, 0.9168290082662192e-01, 0.1015190880092246e+00, 0.4790492926791300e-02, 0.2214892234681356e-01, 0.1119088044887405e-01, 0.1982969108114764e-01, 0.2781732372482683e-02, 0.1628595626045726e-02, 0.1085600370518675e-01, 0.1911464940919745e-01, 0.1605699355811189e-03, 0.3616937915375072e-03, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			pp:       0,
			dmin:     9.5622848228283271e-003,
			desig:    -6.0308751014385013e-019,
			qmax:     2.4715692224139039,
			ttype:    -4,
			dmin1:    9.5622848228283271e-003,
			dmin2:    6.9533978479808370e-002,
			dn:       1.0856003705186750e-002,
			dn1:      9.5622848228283271e-003,
			dn2:      6.9533978479808370e-002,
			g:        0.0000000000000000,
			tau:      5.4769133315280185e-003,
			nFail:    2,
			iter:     12,
			sigma:    1.0518227355316156e-002,
			nDiv:     256,
			zOut:     []float64{0.2468318984233055e+01, 0.2464320851971913e+01, 0.1123474100024551e-02, 0.5350323562789559e-03, 0.1173584145846428e+01, 0.1168088077064565e+01, 0.1605699355811189e-03, 0.7153898701552432e-04, 0.5204201986859162e+00, 0.5252841314829605e+00, 0.1005707814522541e-01, 0.2191222436498315e-01, 0.1144481884123012e+01, 0.1261039483864963e+01, 0.1435914304680996e+00, 0.1353250258951489e+00, 0.1188442793922463e+01, 0.1066152263103050e+01, 0.1815610143690141e-01, 0.1453842008528346e-01, 0.8537168361686087e+00, 0.8868497102573779e+00, 0.5279290053521807e-01, 0.9530062900995111e-01, 0.1600922365848029e+01, 0.1557460778765631e+01, 0.5696064828871891e-01, 0.4375878814786067e-01, 0.1196485614439627e+01, 0.1256691152780422e+01, 0.1090859328498209e+00, 0.1362446379077657e+00, 0.1569564714711448e+01, 0.1696642487635560e+01, 0.2684440171930437e+00, 0.5916231430550117e-01, 0.3739226419241781e+00, 0.7219877400150740e+00, 0.4123490187575627e+00, 0.1912940350054112e+00, 0.3349394365677468e+00, 0.2978322161977056e+00, 0.1593084209965356e+00, 0.8676597630518320e+00, 0.1622117829797102e+01, 0.1015315690490590e+01, 0.2659792301064862e+00, 0.3405260456467969e+00, 0.1299881336702450e+01, 0.1124037582002341e+01, 0.1698038973078534e+00, 0.6882320425428856e-01, 0.4555835838996722e+00, 0.1026360460398424e+01, 0.6447216871142054e+00, 0.9544125430154021e+00, 0.1519370786864500e+01, 0.7226914714852769e+00, 0.1628548339973444e+00, 0.1465016632377001e+00, 0.6501219520571339e+00, 0.8246707410864278e+00, 0.3261720586281595e+00, 0.2292385937027206e+00, 0.5795909120773233e+00, 0.3717389082113253e+00, 0.2650819619788820e-01, 0.6537783023029759e-02, 0.9168290082662192e-01, 0.8481400436921797e-01, 0.4790492926791300e-02, 0.6320870478125323e-03, 0.1119088044887405e-01, 0.8218919412378699e-02, 0.2781732372482683e-02, 0.3674266095981827e-02, 0.1085600370518675e-01, 0.2060131248039419e-02, 0.1605699355811189e-03, 0.7153898701552432e-04, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			i0Out:    1,
			n0Out:    20,
			ppOut:    0,
			dminOut:  2.0601312480394186e-003,
			desigOut: 0.0000000000000000,
			sigmaOut: 1.5639833716481661e-002,
			qmaxOut:  2.4715692224139039,
			nFailOut: 2,
			iterOut:  13,
			nDivOut:  277,
			ttypeOut: -4,
			dmin1Out: 5.4371870398960158e-003,
			dmin2Out: 8.0023511442426670e-002,
			dnOut:    2.0601312480394186e-003,
			dn1Out:   5.4371870398960158e-003,
			dn2Out:   8.0023511442426670e-002,
			gOut:     0.0000000000000000,
			tauOut:   5.1216063611655054e-003,
		},
		{
			i0:       1,
			n0:       20,
			z:        []float64{0.2468318984233055e+01, 0.2464320851971913e+01, 0.1123474100024551e-02, 0.5350323562789559e-03, 0.1173584145846428e+01, 0.1168088077064565e+01, 0.1605699355811189e-03, 0.7153898701552432e-04, 0.5204201986859162e+00, 0.5252841314829605e+00, 0.1005707814522541e-01, 0.2191222436498315e-01, 0.1144481884123012e+01, 0.1261039483864963e+01, 0.1435914304680996e+00, 0.1353250258951489e+00, 0.1188442793922463e+01, 0.1066152263103050e+01, 0.1815610143690141e-01, 0.1453842008528346e-01, 0.8537168361686087e+00, 0.8868497102573779e+00, 0.5279290053521807e-01, 0.9530062900995111e-01, 0.1600922365848029e+01, 0.1557460778765631e+01, 0.5696064828871891e-01, 0.4375878814786067e-01, 0.1196485614439627e+01, 0.1256691152780422e+01, 0.1090859328498209e+00, 0.1362446379077657e+00, 0.1569564714711448e+01, 0.1696642487635560e+01, 0.2684440171930437e+00, 0.5916231430550117e-01, 0.3739226419241781e+00, 0.7219877400150740e+00, 0.4123490187575627e+00, 0.1912940350054112e+00, 0.3349394365677468e+00, 0.2978322161977056e+00, 0.1593084209965356e+00, 0.8676597630518320e+00, 0.1622117829797102e+01, 0.1015315690490590e+01, 0.2659792301064862e+00, 0.3405260456467969e+00, 0.1299881336702450e+01, 0.1124037582002341e+01, 0.1698038973078534e+00, 0.6882320425428856e-01, 0.4555835838996722e+00, 0.1026360460398424e+01, 0.6447216871142054e+00, 0.9544125430154021e+00, 0.1519370786864500e+01, 0.7226914714852769e+00, 0.1628548339973444e+00, 0.1465016632377001e+00, 0.6501219520571339e+00, 0.8246707410864278e+00, 0.3261720586281595e+00, 0.2292385937027206e+00, 0.5795909120773233e+00, 0.3717389082113253e+00, 0.2650819619788820e-01, 0.6537783023029759e-02, 0.9168290082662192e-01, 0.8481400436921797e-01, 0.4790492926791300e-02, 0.6320870478125323e-03, 0.1119088044887405e-01, 0.8218919412378699e-02, 0.2781732372482683e-02, 0.3674266095981827e-02, 0.1085600370518675e-01, 0.2060131248039419e-02, 0.1605699355811189e-03, 0.7153898701552432e-04, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			pp:       1,
			dmin:     2.0601312480394186e-003,
			desig:    0.0000000000000000,
			qmax:     2.4715692224139039,
			ttype:    -4,
			dmin1:    5.4371870398960158e-003,
			dmin2:    8.0023511442426670e-002,
			dn:       2.0601312480394186e-003,
			dn1:      5.4371870398960158e-003,
			dn2:      8.0023511442426670e-002,
			g:        0.0000000000000000,
			tau:      5.1216063611655054e-003,
			nFail:    2,
			iter:     13,
			sigma:    1.5639833716481661e-002,
			nDiv:     277,
			zOut:     []float64{0.2463574096511276e+01, 0.2464320851971913e+01, 0.2536822079344948e-03, 0.5350323562789559e-03, 0.1166624146026729e+01, 0.1168088077064565e+01, 0.3221114082852138e-04, 0.7153898701552432e-04, 0.5458823568901986e+00, 0.5252841314829605e+00, 0.5061929508212644e-01, 0.2191222436498315e-01, 0.1344463426861069e+01, 0.1261039483864963e+01, 0.1073120173669855e+00, 0.1353250258951489e+00, 0.9720968780044319e+00, 0.1066152263103050e+01, 0.1326348631702415e-01, 0.1453842008528346e-01, 0.9676050651333883e+00, 0.8868497102573779e+00, 0.1533962535161303e+00, 0.9530062900995111e-01, 0.1446541525580445e+01, 0.1557460778765631e+01, 0.3801569533217738e-01, 0.4375878814786067e-01, 0.1353638307539094e+01, 0.1256691152780422e+01, 0.1707682473962209e+00, 0.1362446379077657e+00, 0.1583754766727924e+01, 0.1696642487635560e+01, 0.2697037855661164e-01, 0.5916231430550117e-01, 0.8850296086469572e+00, 0.7219877400150740e+00, 0.6437471225190403e-01, 0.1912940350054112e+00, 0.1099835479180717e+01, 0.2978322161977056e+00, 0.8009821360646626e+00, 0.8676597630518320e+00, 0.5535778122558079e+00, 0.1015315690490590e+01, 0.6914368034330997e+00, 0.3405260456467969e+00, 0.5001421950066134e+00, 0.1124037582002341e+01, 0.1412346654806686e+00, 0.6882320425428856e-01, 0.1838256550116241e+01, 0.1026360460398424e+01, 0.3752173792456719e+00, 0.9544125430154021e+00, 0.4926939676603885e+00, 0.7226914714852769e+00, 0.2452143584512202e+00, 0.1465016632377001e+00, 0.8074131885210117e+00, 0.8246707410864278e+00, 0.1055431169003394e+00, 0.2292385937027206e+00, 0.2714517865170992e+00, 0.3717389082113253e+00, 0.2042703660177667e-02, 0.6537783023029759e-02, 0.8212159993993635e-01, 0.8481400436921797e-01, 0.6326073178529442e-04, 0.6320870478125323e-03, 0.1054813695965874e-01, 0.8218919412378699e-02, 0.7176120699696391e-03, 0.3674266095981827e-02, 0.6073136115328898e-04, 0.2060131248039419e-02, 0.3221114082852138e-04, 0.7153898701552432e-04, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			i0Out:    1,
			n0Out:    20,
			ppOut:    1,
			dminOut:  6.0731361153288982e-005,
			desigOut: 1.7347234759768071e-018,
			sigmaOut: 1.6921621533398150e-002,
			qmaxOut:  2.4715692224139039,
			nFailOut: 2,
			iterOut:  14,
			nDivOut:  298,
			ttypeOut: -2,
			dmin1Out: 6.8738708636769136e-003,
			dmin2Out: 8.1489512892123819e-002,
			dnOut:    6.0731361153288982e-005,
			dn1Out:   6.8738708636769136e-003,
			dn2Out:   8.1489512892123819e-002,
			gOut:     0.0000000000000000,
			tauOut:   1.2817878169164906e-003,
		},
		{
			i0:       1,
			n0:       20,
			z:        []float64{0.2463574096511276e+01, 0.2464320851971913e+01, 0.2536822079344948e-03, 0.5350323562789559e-03, 0.1166624146026729e+01, 0.1168088077064565e+01, 0.3221114082852138e-04, 0.7153898701552432e-04, 0.5458823568901986e+00, 0.5252841314829605e+00, 0.5061929508212644e-01, 0.2191222436498315e-01, 0.1344463426861069e+01, 0.1261039483864963e+01, 0.1073120173669855e+00, 0.1353250258951489e+00, 0.9720968780044319e+00, 0.1066152263103050e+01, 0.1326348631702415e-01, 0.1453842008528346e-01, 0.9676050651333883e+00, 0.8868497102573779e+00, 0.1533962535161303e+00, 0.9530062900995111e-01, 0.1446541525580445e+01, 0.1557460778765631e+01, 0.3801569533217738e-01, 0.4375878814786067e-01, 0.1353638307539094e+01, 0.1256691152780422e+01, 0.1707682473962209e+00, 0.1362446379077657e+00, 0.1583754766727924e+01, 0.1696642487635560e+01, 0.2697037855661164e-01, 0.5916231430550117e-01, 0.8850296086469572e+00, 0.7219877400150740e+00, 0.6437471225190403e-01, 0.1912940350054112e+00, 0.1099835479180717e+01, 0.2978322161977056e+00, 0.8009821360646626e+00, 0.8676597630518320e+00, 0.5535778122558079e+00, 0.1015315690490590e+01, 0.6914368034330997e+00, 0.3405260456467969e+00, 0.5001421950066134e+00, 0.1124037582002341e+01, 0.1412346654806686e+00, 0.6882320425428856e-01, 0.1838256550116241e+01, 0.1026360460398424e+01, 0.3752173792456719e+00, 0.9544125430154021e+00, 0.4926939676603885e+00, 0.7226914714852769e+00, 0.2452143584512202e+00, 0.1465016632377001e+00, 0.8074131885210117e+00, 0.8246707410864278e+00, 0.1055431169003394e+00, 0.2292385937027206e+00, 0.2714517865170992e+00, 0.3717389082113253e+00, 0.2042703660177667e-02, 0.6537783023029759e-02, 0.8212159993993635e-01, 0.8481400436921797e-01, 0.6326073178529442e-04, 0.6320870478125323e-03, 0.1054813695965874e-01, 0.8218919412378699e-02, 0.7176120699696391e-03, 0.3674266095981827e-02, 0.6073136115328898e-04, 0.2060131248039419e-02, 0.3221114082852138e-04, 0.7153898701552432e-04, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			pp:       0,
			dmin:     6.0731361153288982e-005,
			desig:    1.7347234759768071e-018,
			qmax:     2.4715692224139039,
			ttype:    -2,
			dmin1:    6.8738708636769136e-003,
			dmin2:    8.1489512892123819e-002,
			dn:       6.0731361153288982e-005,
			dn1:      6.8738708636769136e-003,
			dn2:      8.1489512892123819e-002,
			g:        0.0000000000000000,
			tau:      1.2817878169164906e-003,
			nFail:    2,
			iter:     14,
			sigma:    1.6921621533398150e-002,
			nDiv:     298,
			zOut:     []float64{0.2463574096511276e+01, 0.2463770941477959e+01, 0.2536822079344948e-03, 0.1201214707955848e-03, 0.1166624146026729e+01, 0.1166479398455512e+01, 0.3221114082852138e-04, 0.1507398544447245e-04, 0.5458823568901986e+00, 0.5964297407456295e+00, 0.5061929508212644e-01, 0.1141052940222717e+00, 0.1344463426861069e+01, 0.1337613312964532e+01, 0.1073120173669855e+00, 0.7798791776646297e-01, 0.9720968780044319e+00, 0.9073156093137420e+00, 0.1326348631702415e-01, 0.1414482062243694e-01, 0.9676050651333883e+00, 0.1106799660785830e+01, 0.1533962535161303e+00, 0.2004825791345134e+00, 0.1446541525580445e+01, 0.1284017804536858e+01, 0.3801569533217738e-01, 0.4007693764646178e-01, 0.1353638307539094e+01, 0.1484272780047602e+01, 0.1707682473962209e+00, 0.1822138285193538e+00, 0.1583754766727924e+01, 0.1428454479523931e+01, 0.2697037855661164e-01, 0.1671007646458111e-01, 0.8850296086469572e+00, 0.9326374071930291e+00, 0.6437471225190403e-01, 0.7591545433480534e-01, 0.1099835479180717e+01, 0.1824845323669324e+01, 0.8009821360646626e+00, 0.2429827519008994e+00, 0.5535778122558079e+00, 0.1001975026546757e+01, 0.6914368034330997e+00, 0.3451350696526060e+00, 0.5001421950066134e+00, 0.2961849535934249e+00, 0.1412346654806686e+00, 0.8765656248686587e+00, 0.1838256550116241e+01, 0.1336851467252003e+01, 0.3752173792456719e+00, 0.1382856239786244e+00, 0.4926939676603885e+00, 0.5995658648917332e+00, 0.2452143584512202e+00, 0.3302211126778973e+00, 0.8074131885210117e+00, 0.5826783555022028e+00, 0.1055431169003394e+00, 0.4916926700063749e-01, 0.2714517865170992e+00, 0.2242683859353883e+00, 0.2042703660177667e-02, 0.7479881396448043e-03, 0.8212159993993635e-01, 0.8138003529082581e-01, 0.6326073178529442e-04, 0.8199589256196194e-05, 0.1054813695965874e-01, 0.1120071219912114e-01, 0.7176120699696391e-03, 0.3890963093641941e-05, 0.6073136115328898e-04, 0.3156808608191942e-08, 0.3221114082852138e-04, 0.1507398544447245e-04, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			i0Out:    1,
			n0Out:    20,
			ppOut:    0,
			dminOut:  3.1568086081919418e-009,
			desigOut: 4.7433845046240819e-020,
			sigmaOut: 1.6978458774649190e-002,
			qmaxOut:  2.4715692224139039,
			nFailOut: 2,
			iterOut:  15,
			nDivOut:  319,
			ttypeOut: -2,
			dmin1Out: 1.0483100129151506e-002,
			dmin2Out: 8.1316774559040517e-002,
			dnOut:    3.1568086081919418e-009,
			dn1Out:   1.0483100129151506e-002,
			dn2Out:   8.1316774559040517e-002,
			gOut:     0.0000000000000000,
			tauOut:   5.6837241251038845e-005,
		},
		{
			i0:       1,
			n0:       20,
			z:        []float64{0.2463574096511276e+01, 0.2463770941477959e+01, 0.2536822079344948e-03, 0.1201214707955848e-03, 0.1166624146026729e+01, 0.1166479398455512e+01, 0.3221114082852138e-04, 0.1507398544447245e-04, 0.5458823568901986e+00, 0.5964297407456295e+00, 0.5061929508212644e-01, 0.1141052940222717e+00, 0.1344463426861069e+01, 0.1337613312964532e+01, 0.1073120173669855e+00, 0.7798791776646297e-01, 0.9720968780044319e+00, 0.9073156093137420e+00, 0.1326348631702415e-01, 0.1414482062243694e-01, 0.9676050651333883e+00, 0.1106799660785830e+01, 0.1533962535161303e+00, 0.2004825791345134e+00, 0.1446541525580445e+01, 0.1284017804536858e+01, 0.3801569533217738e-01, 0.4007693764646178e-01, 0.1353638307539094e+01, 0.1484272780047602e+01, 0.1707682473962209e+00, 0.1822138285193538e+00, 0.1583754766727924e+01, 0.1428454479523931e+01, 0.2697037855661164e-01, 0.1671007646458111e-01, 0.8850296086469572e+00, 0.9326374071930291e+00, 0.6437471225190403e-01, 0.7591545433480534e-01, 0.1099835479180717e+01, 0.1824845323669324e+01, 0.8009821360646626e+00, 0.2429827519008994e+00, 0.5535778122558079e+00, 0.1001975026546757e+01, 0.6914368034330997e+00, 0.3451350696526060e+00, 0.5001421950066134e+00, 0.2961849535934249e+00, 0.1412346654806686e+00, 0.8765656248686587e+00, 0.1838256550116241e+01, 0.1336851467252003e+01, 0.3752173792456719e+00, 0.1382856239786244e+00, 0.4926939676603885e+00, 0.5995658648917332e+00, 0.2452143584512202e+00, 0.3302211126778973e+00, 0.8074131885210117e+00, 0.5826783555022028e+00, 0.1055431169003394e+00, 0.4916926700063749e-01, 0.2714517865170992e+00, 0.2242683859353883e+00, 0.2042703660177667e-02, 0.7479881396448043e-03, 0.8212159993993635e-01, 0.8138003529082581e-01, 0.6326073178529442e-04, 0.8199589256196194e-05, 0.1054813695965874e-01, 0.1120071219912114e-01, 0.7176120699696391e-03, 0.3890963093641941e-05, 0.6073136115328898e-04, 0.3156808608191942e-08, 0.3221114082852138e-04, 0.1507398544447245e-04, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			pp:       1,
			dmin:     3.1568086081919418e-009,
			desig:    4.7433845046240819e-020,
			qmax:     2.4715692224139039,
			ttype:    -2,
			dmin1:    1.0483100129151506e-002,
			dmin2:    8.1316774559040517e-002,
			dn:       3.1568086081919418e-009,
			dn1:      1.0483100129151506e-002,
			dn2:      8.1316774559040517e-002,
			g:        0.0000000000000000,
			tau:      5.6837241251038845e-005,
			nFail:    2,
			iter:     15,
			sigma:    1.6978458774649190e-002,
			nDiv:     319,
			zOut:     []float64{0.2463891059793043e+01, 0.2463770941477959e+01, 0.5686908130061341e-04, 0.1201214707955848e-03, 0.1166437600203943e+01, 0.1166479398455512e+01, 0.7707718980490818e-05, 0.1507398544447245e-04, 0.7105273238932086e+00, 0.5964297407456295e+00, 0.2148105431436762e+00, 0.1141052940222717e+00, 0.1200790684431606e+01, 0.1337613312964532e+01, 0.5892755169139442e-01, 0.7798791776646297e-01, 0.8625328750890724e+00, 0.9073156093137420e+00, 0.1815059242254727e-01, 0.1414482062243694e-01, 0.1289131644342084e+01, 0.1106799660785830e+01, 0.1996872873596725e+00, 0.2004825791345134e+00, 0.1124407451667935e+01, 0.1284017804536858e+01, 0.5290351604133232e-01, 0.4007693764646178e-01, 0.1613583089369911e+01, 0.1484272780047602e+01, 0.1613081850537457e+00, 0.1822138285193538e+00, 0.1283856367779054e+01, 0.1428454479523931e+01, 0.1213877407087503e-01, 0.1671007646458111e-01, 0.9964140843012472e+00, 0.9326374071930291e+00, 0.1390325207358455e+00, 0.7591545433480534e-01, 0.1928795551678665e+01, 0.1824845323669324e+01, 0.1262252233392066e+00, 0.2429827519008994e+00, 0.1220884869704444e+01, 0.1001975026546757e+01, 0.8372928285471114e-01, 0.3451350696526060e+00, 0.1089021292451660e+01, 0.2961849535934249e+00, 0.1076046951396362e+01, 0.8765656248686587e+00, 0.3990901366785531e+00, 0.1336851467252003e+01, 0.2077509116934600e+00, 0.1382856239786244e+00, 0.7220360627204584e+00, 0.5995658648917332e+00, 0.2664862668525171e+00, 0.3302211126778973e+00, 0.3653613524946110e+00, 0.5826783555022028e+00, 0.3018138637972599e-01, 0.4916926700063749e-01, 0.1948349845395949e+00, 0.2242683859353883e+00, 0.3124249032854923e-03, 0.7479881396448043e-03, 0.8107580682108434e-01, 0.8138003529082581e-01, 0.1132782305976083e-05, 0.8199589256196194e-05, 0.1120346722419663e-01, 0.1120071219912114e-01, 0.1096359327194516e-11, 0.3890963093641941e-05, 0.6968497581336674e-16, 0.3156808608191942e-08, 0.7707718980490818e-05, 0.1507398544447245e-04, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			i0Out:    1,
			n0Out:    20,
			ppOut:    1,
			dminOut:  6.9684975813366743e-017,
			desigOut: 1.5445815365207740e-018,
			sigmaOut: 1.6978461930361368e-002,
			qmaxOut:  2.4715692224139039,
			nFailOut: 2,
			iterOut:  16,
			nDivOut:  340,
			ttypeOut: -2,
			dmin1Out: 1.1199576261102989e-002,
			dmin2Out: 8.1067607231828140e-002,
			dnOut:    6.9684975813366743e-017,
			dn1Out:   1.1199576261102989e-002,
			dn2Out:   8.1067607231828140e-002,
			gOut:     0.0000000000000000,
			tauOut:   3.1557121791797713e-009,
		},
		{
			i0:       1,
			n0:       20,
			z:        []float64{0.2463891059793043e+01, 0.2463770941477959e+01, 0.5686908130061341e-04, 0.1201214707955848e-03, 0.1166437600203943e+01, 0.1166479398455512e+01, 0.7707718980490818e-05, 0.1507398544447245e-04, 0.7105273238932086e+00, 0.5964297407456295e+00, 0.2148105431436762e+00, 0.1141052940222717e+00, 0.1200790684431606e+01, 0.1337613312964532e+01, 0.5892755169139442e-01, 0.7798791776646297e-01, 0.8625328750890724e+00, 0.9073156093137420e+00, 0.1815059242254727e-01, 0.1414482062243694e-01, 0.1289131644342084e+01, 0.1106799660785830e+01, 0.1996872873596725e+00, 0.2004825791345134e+00, 0.1124407451667935e+01, 0.1284017804536858e+01, 0.5290351604133232e-01, 0.4007693764646178e-01, 0.1613583089369911e+01, 0.1484272780047602e+01, 0.1613081850537457e+00, 0.1822138285193538e+00, 0.1283856367779054e+01, 0.1428454479523931e+01, 0.1213877407087503e-01, 0.1671007646458111e-01, 0.9964140843012472e+00, 0.9326374071930291e+00, 0.1390325207358455e+00, 0.7591545433480534e-01, 0.1928795551678665e+01, 0.1824845323669324e+01, 0.1262252233392066e+00, 0.2429827519008994e+00, 0.1220884869704444e+01, 0.1001975026546757e+01, 0.8372928285471114e-01, 0.3451350696526060e+00, 0.1089021292451660e+01, 0.2961849535934249e+00, 0.1076046951396362e+01, 0.8765656248686587e+00, 0.3990901366785531e+00, 0.1336851467252003e+01, 0.2077509116934600e+00, 0.1382856239786244e+00, 0.7220360627204584e+00, 0.5995658648917332e+00, 0.2664862668525171e+00, 0.3302211126778973e+00, 0.3653613524946110e+00, 0.5826783555022028e+00, 0.3018138637972599e-01, 0.4916926700063749e-01, 0.1948349845395949e+00, 0.2242683859353883e+00, 0.3124249032854923e-03, 0.7479881396448043e-03, 0.8107580682108434e-01, 0.8138003529082581e-01, 0.1132782305976083e-05, 0.8199589256196194e-05, 0.1120346722419663e-01, 0.1120071219912114e-01, 0.1096359327194516e-11, 0.3890963093641941e-05, 0.6968497581336674e-16, 0.3156808608191942e-08, 0.7707718980490818e-05, 0.1507398544447245e-04, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			pp:       0,
			dmin:     6.9684975813366743e-017,
			desig:    1.5445815365207740e-018,
			qmax:     2.4715692224139039,
			ttype:    -2,
			dmin1:    1.1199576261102989e-002,
			dmin2:    8.1067607231828140e-002,
			dn:       6.9684975813366743e-017,
			dn1:      1.1199576261102989e-002,
			dn2:      8.1067607231828140e-002,
			g:        0.0000000000000000,
			tau:      3.1557121791797713e-009,
			nFail:    2,
			iter:     16,
			sigma:    1.6978461930361368e-002,
			nDiv:     340,
			zOut:     []float64{0.2463891059793043e+01, 0.2463947928874343e+01, 0.5686908130061341e-04, 0.2692193042748079e-04, 0.1166437600203943e+01, 0.1166418385992496e+01, 0.7707718980490818e-05, 0.4695180568393632e-05, 0.7105273238932086e+00, 0.9253331718563164e+00, 0.2148105431436762e+00, 0.2787563517334627e+00, 0.1200790684431606e+01, 0.9809618843895378e+00, 0.5892755169139442e-01, 0.5181337969514327e-01, 0.8625328750890724e+00, 0.8288700878164763e+00, 0.1815059242254727e-01, 0.2822939734392020e-01, 0.1289131644342084e+01, 0.1460589534357837e+01, 0.1996872873596725e+00, 0.1537255119449346e+00, 0.1124407451667935e+01, 0.1023585455764333e+01, 0.5290351604133232e-01, 0.8339725654733963e-01, 0.1613583089369911e+01, 0.1691494017876317e+01, 0.1613081850537457e+00, 0.1224340957564512e+00, 0.1283856367779054e+01, 0.1173561046093478e+01, 0.1213877407087503e-01, 0.1030644761994533e-01, 0.9964140843012472e+00, 0.1125140157417147e+01, 0.1390325207358455e+00, 0.2383394688796517e+00, 0.1928795551678665e+01, 0.1816681306138221e+01, 0.1262252233392066e+00, 0.8482856339700598e-01, 0.1220884869704444e+01, 0.1219785589162149e+01, 0.8372928285471114e-01, 0.7475327847832687e-01, 0.1089021292451660e+01, 0.2090314965369696e+01, 0.1076046951396362e+01, 0.2054425921547012e+00, 0.3990901366785531e+00, 0.4013984562173118e+00, 0.2077509116934600e+00, 0.3737026089221466e+00, 0.7220360627204584e+00, 0.6148197206508288e+00, 0.2664862668525171e+00, 0.1583615157552351e+00, 0.3653613524946110e+00, 0.2371812231191019e+00, 0.3018138637972599e-01, 0.2479281399828426e-01, 0.1948349845395949e+00, 0.1703545954445960e+00, 0.3124249032854923e-03, 0.1486904479375115e-03, 0.8107580682108434e-01, 0.8092824915545274e-01, 0.1132782305976083e-05, 0.1568190288260776e-06, 0.1120346722419663e-01, 0.1120331040626409e-01, 0.1096359327194516e-11, 0.6819392699821255e-26, 0.6968497581336674e-16, 0.6162975822039155e-31, 0.7707718980490818e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			i0Out:    1,
			n0Out:    20,
			ppOut:    0,
			dminOut:  6.1629758220391547e-032,
			desigOut: -1.6288286479578371e-018,
			sigmaOut: 1.6978461930361441e-002,
			qmaxOut:  2.4715692224139039,
			nFailOut: 2,
			iterOut:  17,
			nDivOut:  361,
			ttypeOut: -2,
			dmin1Out: 1.1203310405167735e-002,
			dmin2Out: 8.0927116373146771e-002,
			dnOut:    6.1629758220391547e-032,
			dn1Out:   1.1203310405167735e-002,
			dn2Out:   8.0927116373146771e-002,
			gOut:     0.0000000000000000,
			tauOut:   6.9684975806547287e-017,
		},
		{
			i0:       1,
			n0:       20,
			z:        []float64{0.2463891059793043e+01, 0.2463947928874343e+01, 0.5686908130061341e-04, 0.2692193042748079e-04, 0.1166437600203943e+01, 0.1166418385992496e+01, 0.7707718980490818e-05, 0.4695180568393632e-05, 0.7105273238932086e+00, 0.9253331718563164e+00, 0.2148105431436762e+00, 0.2787563517334627e+00, 0.1200790684431606e+01, 0.9809618843895378e+00, 0.5892755169139442e-01, 0.5181337969514327e-01, 0.8625328750890724e+00, 0.8288700878164763e+00, 0.1815059242254727e-01, 0.2822939734392020e-01, 0.1289131644342084e+01, 0.1460589534357837e+01, 0.1996872873596725e+00, 0.1537255119449346e+00, 0.1124407451667935e+01, 0.1023585455764333e+01, 0.5290351604133232e-01, 0.8339725654733963e-01, 0.1613583089369911e+01, 0.1691494017876317e+01, 0.1613081850537457e+00, 0.1224340957564512e+00, 0.1283856367779054e+01, 0.1173561046093478e+01, 0.1213877407087503e-01, 0.1030644761994533e-01, 0.9964140843012472e+00, 0.1125140157417147e+01, 0.1390325207358455e+00, 0.2383394688796517e+00, 0.1928795551678665e+01, 0.1816681306138221e+01, 0.1262252233392066e+00, 0.8482856339700598e-01, 0.1220884869704444e+01, 0.1219785589162149e+01, 0.8372928285471114e-01, 0.7475327847832687e-01, 0.1089021292451660e+01, 0.2090314965369696e+01, 0.1076046951396362e+01, 0.2054425921547012e+00, 0.3990901366785531e+00, 0.4013984562173118e+00, 0.2077509116934600e+00, 0.3737026089221466e+00, 0.7220360627204584e+00, 0.6148197206508288e+00, 0.2664862668525171e+00, 0.1583615157552351e+00, 0.3653613524946110e+00, 0.2371812231191019e+00, 0.3018138637972599e-01, 0.2479281399828426e-01, 0.1948349845395949e+00, 0.1703545954445960e+00, 0.3124249032854923e-03, 0.1486904479375115e-03, 0.8107580682108434e-01, 0.8092824915545274e-01, 0.1132782305976083e-05, 0.1568190288260776e-06, 0.1120346722419663e-01, 0.1120331040626409e-01, 0.1096359327194516e-11, 0.6819392699821255e-26, 0.6968497581336674e-16, 0.6162975822039155e-31, 0.7707718980490818e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			pp:       1,
			dmin:     6.1629758220391547e-032,
			desig:    -1.6288286479578371e-018,
			qmax:     2.4715692224139039,
			ttype:    -2,
			dmin1:    1.1203310405167735e-002,
			dmin2:    8.0927116373146771e-002,
			dn:       6.1629758220391547e-032,
			dn1:      1.1203310405167735e-002,
			dn2:      8.0927116373146771e-002,
			g:        0.0000000000000000,
			tau:      6.9684975806547287e-017,
			nFail:    2,
			iter:     17,
			sigma:    1.6978461930361441e-002,
			nDiv:     361,
			zOut:     []float64{0.2463974850804771e+01, 0.2463947928874343e+01, 0.1274454348702788e-04, 0.2692193042748079e-04, 0.1166410336629578e+01, 0.1166418385992496e+01, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.1204085798823124e+01, 0.9253331718563164e+00, 0.2271012218143261e+00, 0.2787563517334627e+00, 0.8056740422703550e+00, 0.9809618843895378e+00, 0.5330513126246473e-01, 0.5181337969514327e-01, 0.8037943538979316e+00, 0.8288700878164763e+00, 0.5129615818002433e-01, 0.2822939734392020e-01, 0.1563018888122747e+01, 0.1460589534357837e+01, 0.1006713350698832e+00, 0.1537255119449346e+00, 0.1006311377241790e+01, 0.1023585455764333e+01, 0.1401812239704283e+00, 0.8339725654733963e-01, 0.1673746889662340e+01, 0.1691494017876317e+01, 0.8584564749956700e-01, 0.1224340957564512e+00, 0.1098021846213856e+01, 0.1173561046093478e+01, 0.1056099032774466e-01, 0.1030644761994533e-01, 0.1352918635969054e+01, 0.1125140157417147e+01, 0.3200390963041470e+00, 0.2383394688796517e+00, 0.1581470773231080e+01, 0.1816681306138221e+01, 0.6542811978092533e-01, 0.8482856339700598e-01, 0.1229110747859551e+01, 0.1219785589162149e+01, 0.1271308521106110e+00, 0.7475327847832687e-01, 0.2168626705413786e+01, 0.2090314965369696e+01, 0.3802606466401751e-01, 0.2054425921547012e+00, 0.7370750004754409e+00, 0.4013984562173118e+00, 0.3117182559112661e+00, 0.3737026089221466e+00, 0.4614629804947978e+00, 0.6148197206508288e+00, 0.8139413038408401e-01, 0.1583615157552351e+00, 0.1805799067333021e+00, 0.2371812231191019e+00, 0.2338892446571373e-01, 0.2479281399828426e-01, 0.1471143614268198e+00, 0.1703545954445960e+00, 0.8179526119010886e-04, 0.1486904479375115e-03, 0.8084661071329148e-01, 0.8092824915545274e-01, 0.2173118009582292e-07, 0.1568190288260776e-06, 0.1120328867508400e-01, 0.1120331040626409e-01, 0.3751376363572422e-55, 0.6819392699821255e-26, 0.6162975822039155e-31, 0.6162975822039155e-31, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			i0Out:    1,
			n0Out:    20,
			ppOut:    1,
			dminOut:  6.1629758220391547e-032,
			desigOut: -1.6288286479578371e-018,
			sigmaOut: 1.6978461930361441e-002,
			qmaxOut:  2.4715692224139039,
			nFailOut: 2,
			iterOut:  18,
			nDivOut:  382,
			ttypeOut: -2,
			dmin1Out: 1.1203288675083998e-002,
			dmin2Out: 8.0846453894262649e-002,
			dnOut:    6.1629758220391547e-032,
			dn1Out:   1.1203288675083998e-002,
			dn2Out:   8.0846453894262649e-002,
			gOut:     0.0000000000000000,
			tauOut:   0.0000000000000000,
		},
		{
			i0:       1,
			n0:       20,
			z:        []float64{0.2463974850804771e+01, 0.2463947928874343e+01, 0.1274454348702788e-04, 0.2692193042748079e-04, 0.1166410336629578e+01, 0.1166418385992496e+01, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.1204085798823124e+01, 0.9253331718563164e+00, 0.2271012218143261e+00, 0.2787563517334627e+00, 0.8056740422703550e+00, 0.9809618843895378e+00, 0.5330513126246473e-01, 0.5181337969514327e-01, 0.8037943538979316e+00, 0.8288700878164763e+00, 0.5129615818002433e-01, 0.2822939734392020e-01, 0.1563018888122747e+01, 0.1460589534357837e+01, 0.1006713350698832e+00, 0.1537255119449346e+00, 0.1006311377241790e+01, 0.1023585455764333e+01, 0.1401812239704283e+00, 0.8339725654733963e-01, 0.1673746889662340e+01, 0.1691494017876317e+01, 0.8584564749956700e-01, 0.1224340957564512e+00, 0.1098021846213856e+01, 0.1173561046093478e+01, 0.1056099032774466e-01, 0.1030644761994533e-01, 0.1352918635969054e+01, 0.1125140157417147e+01, 0.3200390963041470e+00, 0.2383394688796517e+00, 0.1581470773231080e+01, 0.1816681306138221e+01, 0.6542811978092533e-01, 0.8482856339700598e-01, 0.1229110747859551e+01, 0.1219785589162149e+01, 0.1271308521106110e+00, 0.7475327847832687e-01, 0.2168626705413786e+01, 0.2090314965369696e+01, 0.3802606466401751e-01, 0.2054425921547012e+00, 0.7370750004754409e+00, 0.4013984562173118e+00, 0.3117182559112661e+00, 0.3737026089221466e+00, 0.4614629804947978e+00, 0.6148197206508288e+00, 0.8139413038408401e-01, 0.1583615157552351e+00, 0.1805799067333021e+00, 0.2371812231191019e+00, 0.2338892446571373e-01, 0.2479281399828426e-01, 0.1471143614268198e+00, 0.1703545954445960e+00, 0.8179526119010886e-04, 0.1486904479375115e-03, 0.8084661071329148e-01, 0.8092824915545274e-01, 0.2173118009582292e-07, 0.1568190288260776e-06, 0.1120328867508400e-01, 0.1120331040626409e-01, 0.3751376363572422e-55, 0.6819392699821255e-26, 0.6162975822039155e-31, 0.6162975822039155e-31, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			pp:       0,
			dmin:     6.1629758220391547e-032,
			desig:    -1.6288286479578371e-018,
			qmax:     2.4715692224139039,
			ttype:    -2,
			dmin1:    1.1203288675083998e-002,
			dmin2:    8.0846453894262649e-002,
			dn:       6.1629758220391547e-032,
			dn1:      1.1203288675083998e-002,
			dn2:      8.0846453894262649e-002,
			g:        0.0000000000000000,
			tau:      0.0000000000000000,
			nFail:    2,
			iter:     18,
			sigma:    1.6978461930361441e-002,
			nDiv:     382,
			zOut:     []float64{0.2463974850804771e+01, 0.2452784311062345e+01, 0.1274454348702788e-04, 0.6060609239813679e-05, 0.1166410336629578e+01, 0.1155204716501079e+01, 0.3724766654883956e-05, 0.3882375624867434e-05, 0.1204085798823124e+01, 0.1419979853975912e+01, 0.2271012218143261e+00, 0.1288536304732592e+00, 0.8056740422703550e+00, 0.7189222587736472e+00, 0.5330513126246473e-01, 0.5959804835594534e-01, 0.8037943538979316e+00, 0.7842891794360973e+00, 0.5129615818002433e-01, 0.1022287011292917e+00, 0.1563018888122747e+01, 0.1550258237777425e+01, 0.1006713350698832e+00, 0.6534828028921505e-01, 0.1006311377241790e+01, 0.1069941036637090e+01, 0.1401812239704283e+00, 0.2192904838448094e+00, 0.1673746889662340e+01, 0.1529098769031184e+01, 0.8584564749956700e-01, 0.6164441320989392e-01, 0.1098021846213856e+01, 0.1035735139045794e+01, 0.1056099032774466e-01, 0.1379518767882891e-01, 0.1352918635969054e+01, 0.1647959260308459e+01, 0.3200390963041470e+00, 0.3071268139247320e+00, 0.1581470773231080e+01, 0.1328568794801360e+01, 0.6542811978092533e-01, 0.6053010243026304e-01, 0.1229110747859551e+01, 0.1284508213253985e+01, 0.1271308521106110e+00, 0.2146341752620367e+00, 0.2168626705413786e+01, 0.1980815310529854e+01, 0.3802606466401751e-01, 0.1414976019284330e-01, 0.7370750004754409e+00, 0.1023440211907950e+01, 0.3117182559112661e+00, 0.1405518698344743e+00, 0.4614629804947978e+00, 0.3911019567584944e+00, 0.8139413038408401e-01, 0.3758136265851575e-01, 0.1805799067333021e+00, 0.1551841842545868e+00, 0.2338892446571373e-01, 0.2217266343062855e-01, 0.1471143614268198e+00, 0.1138202089714681e+00, 0.8179526119010886e-04, 0.5809925758690545e-04, 0.8084661071329148e-01, 0.6958524890097138e-01, 0.2173118009582292e-07, 0.3498739858072780e-08, 0.1120328867508400e-01, 0.8904308505808256e-09, 0.3751376363572422e-55, 0.3882375624867434e-05, 0.1697846193036144e-01, 0.6162975822039155e-31, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			i0Out:    1,
			n0Out:    19,
			ppOut:    0,
			dminOut:  8.9043085058082561e-010,
			desigOut: 1.7347234759768071e-018,
			sigmaOut: 2.8181746216274728e-002,
			qmaxOut:  2.4715692224139039,
			nFailOut: 2,
			iterOut:  19,
			nDivOut:  402,
			ttypeOut: -7,
			dmin1Out: 6.9585227169791292e-002,
			dmin2Out: 0.11373841371027797,
			dnOut:    8.9043085058082561e-010,
			dn1Out:   6.9585227169791292e-002,
			dn2Out:   0.11373841371027797,
			gOut:     0.0000000000000000,
			tauOut:   1.1203284285913290e-002,
		},
		{
			i0:       1,
			n0:       19,
			z:        []float64{0.2463974850804771e+01, 0.2452784311062345e+01, 0.1274454348702788e-04, 0.6060609239813679e-05, 0.1166410336629578e+01, 0.1155204716501079e+01, 0.3724766654883956e-05, 0.3882375624867434e-05, 0.1204085798823124e+01, 0.1419979853975912e+01, 0.2271012218143261e+00, 0.1288536304732592e+00, 0.8056740422703550e+00, 0.7189222587736472e+00, 0.5330513126246473e-01, 0.5959804835594534e-01, 0.8037943538979316e+00, 0.7842891794360973e+00, 0.5129615818002433e-01, 0.1022287011292917e+00, 0.1563018888122747e+01, 0.1550258237777425e+01, 0.1006713350698832e+00, 0.6534828028921505e-01, 0.1006311377241790e+01, 0.1069941036637090e+01, 0.1401812239704283e+00, 0.2192904838448094e+00, 0.1673746889662340e+01, 0.1529098769031184e+01, 0.8584564749956700e-01, 0.6164441320989392e-01, 0.1098021846213856e+01, 0.1035735139045794e+01, 0.1056099032774466e-01, 0.1379518767882891e-01, 0.1352918635969054e+01, 0.1647959260308459e+01, 0.3200390963041470e+00, 0.3071268139247320e+00, 0.1581470773231080e+01, 0.1328568794801360e+01, 0.6542811978092533e-01, 0.6053010243026304e-01, 0.1229110747859551e+01, 0.1284508213253985e+01, 0.1271308521106110e+00, 0.2146341752620367e+00, 0.2168626705413786e+01, 0.1980815310529854e+01, 0.3802606466401751e-01, 0.1414976019284330e-01, 0.7370750004754409e+00, 0.1023440211907950e+01, 0.3117182559112661e+00, 0.1405518698344743e+00, 0.4614629804947978e+00, 0.3911019567584944e+00, 0.8139413038408401e-01, 0.3758136265851575e-01, 0.1805799067333021e+00, 0.1551841842545868e+00, 0.2338892446571373e-01, 0.2217266343062855e-01, 0.1471143614268198e+00, 0.1138202089714681e+00, 0.8179526119010886e-04, 0.5809925758690545e-04, 0.8084661071329148e-01, 0.6958524890097138e-01, 0.2173118009582292e-07, 0.3498739858072780e-08, 0.1120328867508400e-01, 0.8904308505808256e-09, 0.3751376363572422e-55, 0.3882375624867434e-05, 0.1697846193036144e-01, 0.6162975822039155e-31, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			pp:       1,
			dmin:     8.9043085058082561e-010,
			desig:    1.7347234759768071e-018,
			qmax:     2.4715692224139039,
			ttype:    -7,
			dmin1:    6.9585227169791292e-002,
			dmin2:    0.11373841371027797,
			dn:       8.9043085058082561e-010,
			dn1:      6.9585227169791292e-002,
			dn2:      0.11373841371027797,
			g:        0.0000000000000000,
			tau:      1.1203284285913290e-002,
			nFail:    2,
			iter:     19,
			sigma:    2.8181746216274728e-002,
			nDiv:     402,
			zOut:     []float64{0.2452790370781154e+01, 0.2452784311062345e+01, 0.2854399814229969e-05, 0.6060609239813679e-05, 0.1155205743586459e+01, 0.1155204716501079e+01, 0.4772219324121025e-05, 0.3882375624867434e-05, 0.1548828711339416e+01, 0.1419979853975912e+01, 0.5981019230390531e-01, 0.1288536304732592e+00, 0.7187101139352565e+00, 0.7189222587736472e+00, 0.6503610222645050e-01, 0.5959804835594534e-01, 0.8214817774485077e+00, 0.7842891794360973e+00, 0.1929207566298143e+00, 0.1022287011292917e+00, 0.1422685760546395e+01, 0.1550258237777425e+01, 0.4914564318703864e-01, 0.6534828028921505e-01, 0.1240085876404429e+01, 0.1069941036637090e+01, 0.2703980549150243e+00, 0.2192904838448094e+00, 0.1320345126435623e+01, 0.1529098769031184e+01, 0.4835651195207321e-01, 0.6164441320989392e-01, 0.1001173813882118e+01, 0.1035735139045794e+01, 0.2270725319399537e-01, 0.1379518767882891e-01, 0.1932378820148765e+01, 0.1647959260308459e+01, 0.2111589595024383e+00, 0.3071268139247320e+00, 0.1177939936838754e+01, 0.1328568794801360e+01, 0.6600626338337756e-01, 0.6053010243026304e-01, 0.1433136124242214e+01, 0.1284508213253985e+01, 0.2966575563411978e+00, 0.2146341752620367e+00, 0.1698307513491068e+01, 0.1980815310529854e+01, 0.8526979628348903e-02, 0.1414976019284330e-01, 0.1155465101223645e+01, 0.1023440211907950e+01, 0.4757401262929915e-01, 0.1405518698344743e+00, 0.3811093058972801e+00, 0.3911019567584944e+00, 0.1530278326215801e-01, 0.3758136265851575e-01, 0.1620540635326265e+00, 0.1551841842545868e+00, 0.1557318051836492e-01, 0.2217266343062855e-01, 0.9830512682025924e-01, 0.1138202089714681e+00, 0.4112553872737890e-04, 0.5809925758690545e-04, 0.6954412597055305e-01, 0.6958524890097138e-01, 0.4479725446695418e-16, 0.3498739858072780e-08, 0.1396210563637444e-18, 0.8904308505808256e-09, 0.2854399814229969e-05, 0.3882375624867434e-05, 0.1697846193036144e-01, 0.6162975822039155e-31, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			i0Out:    1,
			n0Out:    19,
			ppOut:    1,
			dminOut:  1.3962105636374437e-019,
			desigOut: -1.5687885635568321e-018,
			sigmaOut: 2.8181747106705537e-002,
			qmaxOut:  2.4715692224139039,
			nFailOut: 2,
			iterOut:  20,
			nDivOut:  422,
			ttypeOut: -2,
			dmin1Out: 6.9544122471813200e-002,
			dmin2Out: 9.8247027562672340e-002,
			dnOut:    1.3962105636374437e-019,
			dn1Out:   6.9544122471813200e-002,
			dn2Out:   9.8247027562672340e-002,
			gOut:     0.0000000000000000,
			tauOut:   8.9043080564395014e-010,
		},
		{
			i0:       1,
			n0:       19,
			z:        []float64{0.2452790370781154e+01, 0.2452784311062345e+01, 0.2854399814229969e-05, 0.6060609239813679e-05, 0.1155205743586459e+01, 0.1155204716501079e+01, 0.4772219324121025e-05, 0.3882375624867434e-05, 0.1548828711339416e+01, 0.1419979853975912e+01, 0.5981019230390531e-01, 0.1288536304732592e+00, 0.7187101139352565e+00, 0.7189222587736472e+00, 0.6503610222645050e-01, 0.5959804835594534e-01, 0.8214817774485077e+00, 0.7842891794360973e+00, 0.1929207566298143e+00, 0.1022287011292917e+00, 0.1422685760546395e+01, 0.1550258237777425e+01, 0.4914564318703864e-01, 0.6534828028921505e-01, 0.1240085876404429e+01, 0.1069941036637090e+01, 0.2703980549150243e+00, 0.2192904838448094e+00, 0.1320345126435623e+01, 0.1529098769031184e+01, 0.4835651195207321e-01, 0.6164441320989392e-01, 0.1001173813882118e+01, 0.1035735139045794e+01, 0.2270725319399537e-01, 0.1379518767882891e-01, 0.1932378820148765e+01, 0.1647959260308459e+01, 0.2111589595024383e+00, 0.3071268139247320e+00, 0.1177939936838754e+01, 0.1328568794801360e+01, 0.6600626338337756e-01, 0.6053010243026304e-01, 0.1433136124242214e+01, 0.1284508213253985e+01, 0.2966575563411978e+00, 0.2146341752620367e+00, 0.1698307513491068e+01, 0.1980815310529854e+01, 0.8526979628348903e-02, 0.1414976019284330e-01, 0.1155465101223645e+01, 0.1023440211907950e+01, 0.4757401262929915e-01, 0.1405518698344743e+00, 0.3811093058972801e+00, 0.3911019567584944e+00, 0.1530278326215801e-01, 0.3758136265851575e-01, 0.1620540635326265e+00, 0.1551841842545868e+00, 0.1557318051836492e-01, 0.2217266343062855e-01, 0.9830512682025924e-01, 0.1138202089714681e+00, 0.4112553872737890e-04, 0.5809925758690545e-04, 0.6954412597055305e-01, 0.6958524890097138e-01, 0.4479725446695418e-16, 0.3498739858072780e-08, 0.1396210563637444e-18, 0.8904308505808256e-09, 0.2854399814229969e-05, 0.3882375624867434e-05, 0.1697846193036144e-01, 0.6162975822039155e-31, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			pp:       0,
			dmin:     1.3962105636374437e-019,
			desig:    -1.5687885635568321e-018,
			qmax:     2.4715692224139039,
			ttype:    -2,
			dmin1:    6.9544122471813200e-002,
			dmin2:    9.8247027562672340e-002,
			dn:       1.3962105636374437e-019,
			dn1:      6.9544122471813200e-002,
			dn2:      9.8247027562672340e-002,
			g:        0.0000000000000000,
			tau:      8.9043080564395014e-010,
			nFail:    2,
			iter:     20,
			sigma:    2.8181747106705537e-002,
			nDiv:     422,
			zOut:     []float64{0.2452790370781154e+01, 0.2452793225180968e+01, 0.2854399814229969e-05, 0.1344352644992036e-05, 0.1155205743586459e+01, 0.1155209171453138e+01, 0.4772219324121025e-05, 0.6398278760815103e-05, 0.1548828711339416e+01, 0.1608632505364561e+01, 0.5981019230390531e-01, 0.2672219415054498e-01, 0.7187101139352565e+00, 0.7570240220111620e+00, 0.6503610222645050e-01, 0.7057368234283014e-01, 0.8214817774485077e+00, 0.9438288517354918e+00, 0.1929207566298143e+00, 0.2908001941945216e+00, 0.1422685760546395e+01, 0.1181031209538913e+01, 0.4914564318703864e-01, 0.5160305461093759e-01, 0.1240085876404429e+01, 0.1458880876708516e+01, 0.2703980549150243e+00, 0.2447209773632920e+00, 0.1320345126435623e+01, 0.1123980661024405e+01, 0.4835651195207321e-01, 0.4307304847484577e-01, 0.1001173813882118e+01, 0.9808080186012680e+00, 0.2270725319399537e-01, 0.4473761868138884e-01, 0.1932378820148765e+01, 0.2098800160969815e+01, 0.2111589595024383e+00, 0.1185117935689049e+00, 0.1177939936838754e+01, 0.1125434406653226e+01, 0.6600626338337756e-01, 0.8405284210411720e-01, 0.1433136124242214e+01, 0.1645740838479294e+01, 0.2966575563411978e+00, 0.3061331074057168e+00, 0.1698307513491068e+01, 0.1400701385713701e+01, 0.8526979628348903e-02, 0.7034066989504621e-02, 0.1155465101223645e+01, 0.1196005046863439e+01, 0.4757401262929915e-01, 0.1515955052150448e-01, 0.3811093058972801e+00, 0.3812525386379337e+00, 0.1530278326215801e-01, 0.6504555274179689e-02, 0.1620540635326265e+00, 0.1711226887768117e+00, 0.1557318051836492e-01, 0.8946350111698947e-02, 0.9830512682025924e-01, 0.8939990224728768e-01, 0.4112553872737890e-04, 0.3199152990069927e-04, 0.6954412597055305e-01, 0.6951213444065239e-01, 0.4479725446695418e-16, 0.8997910999570377e-34, 0.1396210563637444e-18, 0.1396210563637443e-18, 0.2854399814229969e-05, 0.1344352644992036e-05, 0.1697846193036144e-01, 0.6162975822039155e-31, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			i0Out:    1,
			n0Out:    19,
			ppOut:    0,
			dminOut:  1.3962105636374430e-019,
			desigOut: -1.5687885635568321e-018,
			sigmaOut: 2.8181747106705537e-002,
			qmaxOut:  2.4715692224139039,
			nFailOut: 2,
			iterOut:  21,
			nDivOut:  442,
			ttypeOut: -2,
			dmin1Out: 6.9512134440652351e-002,
			dmin2Out: 8.9358776708560295e-002,
			dnOut:    1.3962105636374430e-019,
			dn1Out:   6.9512134440652351e-002,
			dn2Out:   8.9358776708560295e-002,
			gOut:     0.0000000000000000,
			tauOut:   0.0000000000000000,
		},
		{
			i0:       1,
			n0:       19,
			z:        []float64{0.2452790370781154e+01, 0.2452793225180968e+01, 0.2854399814229969e-05, 0.1344352644992036e-05, 0.1155205743586459e+01, 0.1155209171453138e+01, 0.4772219324121025e-05, 0.6398278760815103e-05, 0.1548828711339416e+01, 0.1608632505364561e+01, 0.5981019230390531e-01, 0.2672219415054498e-01, 0.7187101139352565e+00, 0.7570240220111620e+00, 0.6503610222645050e-01, 0.7057368234283014e-01, 0.8214817774485077e+00, 0.9438288517354918e+00, 0.1929207566298143e+00, 0.2908001941945216e+00, 0.1422685760546395e+01, 0.1181031209538913e+01, 0.4914564318703864e-01, 0.5160305461093759e-01, 0.1240085876404429e+01, 0.1458880876708516e+01, 0.2703980549150243e+00, 0.2447209773632920e+00, 0.1320345126435623e+01, 0.1123980661024405e+01, 0.4835651195207321e-01, 0.4307304847484577e-01, 0.1001173813882118e+01, 0.9808080186012680e+00, 0.2270725319399537e-01, 0.4473761868138884e-01, 0.1932378820148765e+01, 0.2098800160969815e+01, 0.2111589595024383e+00, 0.1185117935689049e+00, 0.1177939936838754e+01, 0.1125434406653226e+01, 0.6600626338337756e-01, 0.8405284210411720e-01, 0.1433136124242214e+01, 0.1645740838479294e+01, 0.2966575563411978e+00, 0.3061331074057168e+00, 0.1698307513491068e+01, 0.1400701385713701e+01, 0.8526979628348903e-02, 0.7034066989504621e-02, 0.1155465101223645e+01, 0.1196005046863439e+01, 0.4757401262929915e-01, 0.1515955052150448e-01, 0.3811093058972801e+00, 0.3812525386379337e+00, 0.1530278326215801e-01, 0.6504555274179689e-02, 0.1620540635326265e+00, 0.1711226887768117e+00, 0.1557318051836492e-01, 0.8946350111698947e-02, 0.9830512682025924e-01, 0.8939990224728768e-01, 0.4112553872737890e-04, 0.3199152990069927e-04, 0.6954412597055305e-01, 0.6951213444065239e-01, 0.4479725446695418e-16, 0.8997910999570377e-34, 0.1396210563637444e-18, 0.1396210563637443e-18, 0.2854399814229969e-05, 0.1344352644992036e-05, 0.1697846193036144e-01, 0.6162975822039155e-31, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			pp:       1,
			dmin:     1.3962105636374430e-019,
			desig:    -1.5687885635568321e-018,
			qmax:     2.4715692224139039,
			ttype:    -2,
			dmin1:    6.9512134440652351e-002,
			dmin2:    8.9358776708560295e-002,
			dn:       1.3962105636374430e-019,
			dn1:      6.9512134440652351e-002,
			dn2:      8.9358776708560295e-002,
			g:        0.0000000000000000,
			tau:      0.0000000000000000,
			nFail:    2,
			iter:     21,
			sigma:    2.8181747106705537e-002,
			nDiv:     442,
			zOut:     []float64{0.2384705997428018e+01, 0.2452793225180968e+01, 0.6512368848977837e-06, 0.1344352644992036e-05, 0.1087126346389419e+01, 0.1155209171453138e+01, 0.9467601652019932e-05, 0.6398278760815103e-05, 0.1567256659807859e+01, 0.1608632505364561e+01, 0.1290748567965171e-01, 0.2672219415054498e-01, 0.7466016465687454e+00, 0.7570240220111620e+00, 0.8921689079377818e-01, 0.7057368234283014e-01, 0.1077323583030640e+01, 0.9438288517354918e+00, 0.3187938243378626e+00, 0.2908001941945216e+00, 0.8457518677063927e+00, 0.1181031209538913e+01, 0.8901276181133644e-01, 0.5160305461093759e-01, 0.1546500520154876e+01, 0.1458880876708516e+01, 0.1778606876095877e+00, 0.2447209773632920e+00, 0.9211044497840675e+00, 0.1123980661024405e+01, 0.4586493023634136e-01, 0.4307304847484577e-01, 0.9115921349407204e+00, 0.9808080186012680e+00, 0.1030014605117353e+00, 0.4473761868138884e-01, 0.2046221921921390e+01, 0.2098800160969815e+01, 0.6518220171905387e-01, 0.1185117935689049e+00, 0.1076216474932694e+01, 0.1125434406653226e+01, 0.1285328723941422e+00, 0.8405284210411720e-01, 0.1755252501385274e+01, 0.1645740838479294e+01, 0.2442959445536252e+00, 0.3061331074057168e+00, 0.1095350936043985e+01, 0.1400701385713701e+01, 0.7680442260639305e-02, 0.7034066989504621e-02, 0.1135395583018709e+01, 0.1196005046863439e+01, 0.5090399511302622e-02, 0.1515955052150448e-01, 0.3145781222952156e+00, 0.3812525386379337e+00, 0.3538316586334169e-02, 0.6504555274179689e-02, 0.1084421501965814e+00, 0.1711226887768117e+00, 0.7375387005938499e-02, 0.8946350111698947e-02, 0.1396793466565477e-01, 0.8939990224728768e-01, 0.1592074691534444e-03, 0.3199152990069927e-04, 0.1264354865903830e-02, 0.6951213444065239e-01, 0.6512368848977837e-06, 0.8997910999570377e-34, 0.2818174710670554e-01, 0.1396210563637443e-18, 0.2854399814229969e-05, 0.1344352644992036e-05, 0.1697846193036144e-01, 0.6162975822039155e-31, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			i0Out:    1,
			n0Out:    18,
			ppOut:    1,
			dminOut:  1.2643548659038301e-003,
			desigOut: -5.0382355155104463e-018,
			sigmaOut: 9.6270319212300656e-002,
			qmaxOut:  2.4715692224139039,
			nFailOut: 2,
			iterOut:  22,
			nDivOut:  461,
			ttypeOut: -8,
			dmin1Out: 1.3935943135754067e-002,
			dmin2Out: 9.9495800084882416e-002,
			dnOut:    1.2643548659038301e-003,
			dn1Out:   1.3935943135754067e-002,
			dn2Out:   9.9495800084882416e-002,
			gOut:     0.0000000000000000,
			tauOut:   6.8088572105595116e-002,
		},
		{
			i0:       1,
			n0:       18,
			z:        []float64{0.2384705997428018e+01, 0.2452793225180968e+01, 0.6512368848977837e-06, 0.1344352644992036e-05, 0.1087126346389419e+01, 0.1155209171453138e+01, 0.9467601652019932e-05, 0.6398278760815103e-05, 0.1567256659807859e+01, 0.1608632505364561e+01, 0.1290748567965171e-01, 0.2672219415054498e-01, 0.7466016465687454e+00, 0.7570240220111620e+00, 0.8921689079377818e-01, 0.7057368234283014e-01, 0.1077323583030640e+01, 0.9438288517354918e+00, 0.3187938243378626e+00, 0.2908001941945216e+00, 0.8457518677063927e+00, 0.1181031209538913e+01, 0.8901276181133644e-01, 0.5160305461093759e-01, 0.1546500520154876e+01, 0.1458880876708516e+01, 0.1778606876095877e+00, 0.2447209773632920e+00, 0.9211044497840675e+00, 0.1123980661024405e+01, 0.4586493023634136e-01, 0.4307304847484577e-01, 0.9115921349407204e+00, 0.9808080186012680e+00, 0.1030014605117353e+00, 0.4473761868138884e-01, 0.2046221921921390e+01, 0.2098800160969815e+01, 0.6518220171905387e-01, 0.1185117935689049e+00, 0.1076216474932694e+01, 0.1125434406653226e+01, 0.1285328723941422e+00, 0.8405284210411720e-01, 0.1755252501385274e+01, 0.1645740838479294e+01, 0.2442959445536252e+00, 0.3061331074057168e+00, 0.1095350936043985e+01, 0.1400701385713701e+01, 0.7680442260639305e-02, 0.7034066989504621e-02, 0.1135395583018709e+01, 0.1196005046863439e+01, 0.5090399511302622e-02, 0.1515955052150448e-01, 0.3145781222952156e+00, 0.3812525386379337e+00, 0.3538316586334169e-02, 0.6504555274179689e-02, 0.1084421501965814e+00, 0.1711226887768117e+00, 0.7375387005938499e-02, 0.8946350111698947e-02, 0.1396793466565477e-01, 0.8939990224728768e-01, 0.1592074691534444e-03, 0.3199152990069927e-04, 0.1264354865903830e-02, 0.6951213444065239e-01, 0.6512368848977837e-06, 0.1344352644992036e-05, 0.2818174710670554e-01, 0.1396210563637443e-18, 0.2854399814229969e-05, 0.1344352644992036e-05, 0.1697846193036144e-01, 0.6162975822039155e-31, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			pp:       0,
			dmin:     1.2643548659038301e-003,
			desig:    -5.0382355155104463e-018,
			qmax:     2.3847059974280183,
			ttype:    -8,
			dmin1:    1.3935943135754067e-002,
			dmin2:    9.9495800084882416e-002,
			dn:       1.2643548659038301e-003,
			dn1:      1.3935943135754067e-002,
			dn2:      9.9495800084882416e-002,
			g:        0.0000000000000000,
			tau:      6.8088572105595116e-002,
			nFail:    2,
			iter:     22,
			sigma:    9.6270319212300656e-002,
			nDiv:     461,
			zOut:     []float64{0.2384705997428018e+01, 0.2383460331210544e+01, 0.6512368848977837e-06, 0.2970373645586866e-06, 0.1087126346389419e+01, 0.1085889199499347e+01, 0.9467601652019932e-05, 0.1366452649899944e-04, 0.1567256659807859e+01, 0.1578904163506652e+01, 0.1290748567965171e-01, 0.6103442048115082e-02, 0.7466016465687454e+00, 0.8284687778600495e+00, 0.8921689079377818e-01, 0.1160157908485996e+00, 0.1077323583030640e+01, 0.1278855299065544e+01, 0.3187938243378626e+00, 0.2108295383723412e+00, 0.8457518677063927e+00, 0.7226887736910288e+00, 0.8901276181133644e-01, 0.1904807262171572e+00, 0.1546500520154876e+01, 0.1532634164092948e+01, 0.1778606876095877e+00, 0.1068932656188067e+00, 0.9211044497840675e+00, 0.8588297969472432e+00, 0.4586493023634136e-01, 0.4868264913684867e-01, 0.9115921349407204e+00, 0.9646646288612479e+00, 0.1030014605117353e+00, 0.2184840619043245e+00, 0.2046221921921390e+01, 0.1891673744281760e+01, 0.6518220171905387e-01, 0.3708364593761750e-01, 0.1076216474932694e+01, 0.1166419383934860e+01, 0.1285328723941422e+00, 0.1934189785315259e+00, 0.1755252501385274e+01, 0.1804883149953014e+01, 0.2442959445536252e+00, 0.1482587897978486e+00, 0.1095350936043985e+01, 0.9535262710524167e+00, 0.7680442260639305e-02, 0.9145359161143373e-02, 0.1135395583018709e+01, 0.1130094305914509e+01, 0.5090399511302622e-02, 0.1416986451145964e-02, 0.3145781222952156e+00, 0.3154531349760448e+00, 0.3538316586334169e-02, 0.1216353924418735e-02, 0.1084421501965814e+00, 0.1133548658237420e+00, 0.7375387005938499e-02, 0.9088178357782586e-03, 0.1396793466565477e-01, 0.1197200684467082e-01, 0.1592074691534444e-03, 0.1681378409852773e-04, 0.1264354865903830e-02, 0.1223627446170173e-05, 0.6512368848977837e-06, 0.2970373645586866e-06, 0.2818174710670554e-01, 0.1396210563637443e-18, 0.2854399814229969e-05, 0.1344352644992036e-05, 0.1697846193036144e-01, 0.6162975822039155e-31, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			i0Out:    1,
			n0Out:    18,
			ppOut:    0,
			dminOut:  1.2236274461701734e-006,
			desigOut: -3.2526065174565133e-018,
			sigmaOut: 9.7516636666659787e-002,
			qmaxOut:  2.3847059974280183,
			nFailOut: 2,
			iterOut:  23,
			nDivOut:  480,
			ttypeOut: -2,
			dmin1Out: 1.1812799375517376e-002,
			dmin2Out: 0.10597947881780349,
			dnOut:    1.2236274461701734e-006,
			dn1Out:   1.1812799375517376e-002,
			dn2Out:   0.10597947881780349,
			gOut:     0.0000000000000000,
			tauOut:   1.2463174543591322e-003,
		},
		{
			i0:       1,
			n0:       18,
			z:        []float64{0.2384705997428018e+01, 0.2383460331210544e+01, 0.6512368848977837e-06, 0.2970373645586866e-06, 0.1087126346389419e+01, 0.1085889199499347e+01, 0.9467601652019932e-05, 0.1366452649899944e-04, 0.1567256659807859e+01, 0.1578904163506652e+01, 0.1290748567965171e-01, 0.6103442048115082e-02, 0.7466016465687454e+00, 0.8284687778600495e+00, 0.8921689079377818e-01, 0.1160157908485996e+00, 0.1077323583030640e+01, 0.1278855299065544e+01, 0.3187938243378626e+00, 0.2108295383723412e+00, 0.8457518677063927e+00, 0.7226887736910288e+00, 0.8901276181133644e-01, 0.1904807262171572e+00, 0.1546500520154876e+01, 0.1532634164092948e+01, 0.1778606876095877e+00, 0.1068932656188067e+00, 0.9211044497840675e+00, 0.8588297969472432e+00, 0.4586493023634136e-01, 0.4868264913684867e-01, 0.9115921349407204e+00, 0.9646646288612479e+00, 0.1030014605117353e+00, 0.2184840619043245e+00, 0.2046221921921390e+01, 0.1891673744281760e+01, 0.6518220171905387e-01, 0.3708364593761750e-01, 0.1076216474932694e+01, 0.1166419383934860e+01, 0.1285328723941422e+00, 0.1934189785315259e+00, 0.1755252501385274e+01, 0.1804883149953014e+01, 0.2442959445536252e+00, 0.1482587897978486e+00, 0.1095350936043985e+01, 0.9535262710524167e+00, 0.7680442260639305e-02, 0.9145359161143373e-02, 0.1135395583018709e+01, 0.1130094305914509e+01, 0.5090399511302622e-02, 0.1416986451145964e-02, 0.3145781222952156e+00, 0.3154531349760448e+00, 0.3538316586334169e-02, 0.1216353924418735e-02, 0.1084421501965814e+00, 0.1133548658237420e+00, 0.7375387005938499e-02, 0.9088178357782586e-03, 0.1396793466565477e-01, 0.1197200684467082e-01, 0.1592074691534444e-03, 0.1681378409852773e-04, 0.1264354865903830e-02, 0.1223627446170173e-05, 0.6512368848977837e-06, 0.2970373645586866e-06, 0.2818174710670554e-01, 0.1396210563637443e-18, 0.2854399814229969e-05, 0.1344352644992036e-05, 0.1697846193036144e-01, 0.6162975822039155e-31, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			pp:       1,
			dmin:     1.2236274461701734e-006,
			desig:    -3.2526065174565133e-018,
			qmax:     2.3847059974280183,
			ttype:    -2,
			dmin1:    1.1812799375517376e-002,
			dmin2:    0.10597947881780349,
			dn:       1.2236274461701734e-006,
			dn1:      1.1812799375517376e-002,
			dn2:      0.10597947881780349,
			g:        0.0000000000000000,
			tau:      1.2463174543591322e-003,
			nFail:    2,
			iter:     23,
			sigma:    9.7516636666659787e-002,
			nDiv:     480,
			zOut:     []float64{0.2383459406360114e+01, 0.2383460331210544e+01, 0.1353283656358167e-06, 0.2970373645586866e-06, 0.1085901506809686e+01, 0.1085889199499347e+01, 0.1986826396898849e-04, 0.1366452649899944e-04, 0.1584986515403004e+01, 0.1578904163506652e+01, 0.3190255011763210e-02, 0.6103442048115082e-02, 0.9412930918090910e+00, 0.8284687778600495e+00, 0.1576208411525267e+00, 0.1160157908485996e+00, 0.1332062774397564e+01, 0.1278855299065544e+01, 0.1143821023097510e+00, 0.2108295383723412e+00, 0.7987861757106404e+00, 0.7226887736910288e+00, 0.3654761154847582e+00, 0.1904807262171572e+00, 0.1274050092339202e+01, 0.1532634164092948e+01, 0.7205613198290631e-01, 0.1068932656188067e+00, 0.8354550922133908e+00, 0.8588297969472432e+00, 0.5621179414582513e-01, 0.4868264913684867e-01, 0.1126935674731953e+01, 0.9646646288612479e+00, 0.3667472533840468e+00, 0.2184840619043245e+00, 0.1562008914947536e+01, 0.1891673744281760e+01, 0.2769195683500124e-01, 0.3708364593761750e-01, 0.1332145183743590e+01, 0.1166419383934860e+01, 0.2620575140703801e+00, 0.1934189785315259e+00, 0.1691083203792688e+01, 0.1804883149953014e+01, 0.8359650824372877e-01, 0.1482587897978486e+00, 0.8790739000820365e+00, 0.9535262710524167e+00, 0.1175682535061811e-01, 0.9145359161143373e-02, 0.1119753245127243e+01, 0.1130094305914509e+01, 0.3991886785572840e-03, 0.1416986451145964e-02, 0.3162690783341116e+00, 0.3154531349760448e+00, 0.4359567385560517e-03, 0.1216353924418735e-02, 0.1138265050331695e+00, 0.1133548658237420e+00, 0.9558734450579545e-04, 0.9088178357782586e-03, 0.1189201139646886e-01, 0.1197200684467082e-01, 0.1730052806966466e-08, 0.1681378409852773e-04, 0.9598672791582074e-11, 0.1223627446170173e-05, 0.1353283656358167e-06, 0.2970373645586866e-06, 0.2818174710670554e-01, 0.1396210563637443e-18, 0.2854399814229969e-05, 0.1344352644992036e-05, 0.1697846193036144e-01, 0.6162975822039155e-31, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			i0Out:    1,
			n0Out:    18,
			ppOut:    1,
			dminOut:  9.5986727915820745e-012,
			desigOut: 6.4755668817591261e-018,
			sigmaOut: 9.7517858554454467e-002,
			qmaxOut:  2.3847059974280183,
			nFailOut: 2,
			iterOut:  24,
			nDivOut:  499,
			ttypeOut: -2,
			dmin1Out: 1.1875197612370336e-002,
			dmin2Out: 0.11291768719739126,
			dnOut:    9.5986727915820745e-012,
			dn1Out:   1.1875197612370336e-002,
			dn2Out:   0.11291768719739126,
			gOut:     0.0000000000000000,
			tauOut:   1.2218877946904154e-006,
		},
		{
			i0:       1,
			n0:       18,
			z:        []float64{0.2383459406360114e+01, 0.2383460331210544e+01, 0.1353283656358167e-06, 0.2970373645586866e-06, 0.1085901506809686e+01, 0.1085889199499347e+01, 0.1986826396898849e-04, 0.1366452649899944e-04, 0.1584986515403004e+01, 0.1578904163506652e+01, 0.3190255011763210e-02, 0.6103442048115082e-02, 0.9412930918090910e+00, 0.8284687778600495e+00, 0.1576208411525267e+00, 0.1160157908485996e+00, 0.1332062774397564e+01, 0.1278855299065544e+01, 0.1143821023097510e+00, 0.2108295383723412e+00, 0.7987861757106404e+00, 0.7226887736910288e+00, 0.3654761154847582e+00, 0.1904807262171572e+00, 0.1274050092339202e+01, 0.1532634164092948e+01, 0.7205613198290631e-01, 0.1068932656188067e+00, 0.8354550922133908e+00, 0.8588297969472432e+00, 0.5621179414582513e-01, 0.4868264913684867e-01, 0.1126935674731953e+01, 0.9646646288612479e+00, 0.3667472533840468e+00, 0.2184840619043245e+00, 0.1562008914947536e+01, 0.1891673744281760e+01, 0.2769195683500124e-01, 0.3708364593761750e-01, 0.1332145183743590e+01, 0.1166419383934860e+01, 0.2620575140703801e+00, 0.1934189785315259e+00, 0.1691083203792688e+01, 0.1804883149953014e+01, 0.8359650824372877e-01, 0.1482587897978486e+00, 0.8790739000820365e+00, 0.9535262710524167e+00, 0.1175682535061811e-01, 0.9145359161143373e-02, 0.1119753245127243e+01, 0.1130094305914509e+01, 0.3991886785572840e-03, 0.1416986451145964e-02, 0.3162690783341116e+00, 0.3154531349760448e+00, 0.4359567385560517e-03, 0.1216353924418735e-02, 0.1138265050331695e+00, 0.1133548658237420e+00, 0.9558734450579545e-04, 0.9088178357782586e-03, 0.1189201139646886e-01, 0.1197200684467082e-01, 0.1730052806966466e-08, 0.1681378409852773e-04, 0.9598672791582074e-11, 0.1223627446170173e-05, 0.1353283656358167e-06, 0.2970373645586866e-06, 0.2818174710670554e-01, 0.1396210563637443e-18, 0.2854399814229969e-05, 0.1344352644992036e-05, 0.1697846193036144e-01, 0.6162975822039155e-31, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			pp:       0,
			dmin:     9.5986727915820745e-012,
			desig:    6.4755668817591261e-018,
			qmax:     2.3847059974280183,
			ttype:    -2,
			dmin1:    1.1875197612370336e-002,
			dmin2:    0.11291768719739126,
			dn:       9.5986727915820745e-012,
			dn1:      1.1875197612370336e-002,
			dn2:      0.11291768719739126,
			g:        0.0000000000000000,
			tau:      1.2218877946904154e-006,
			nFail:    2,
			iter:     24,
			sigma:    9.7517858554454467e-002,
			nDiv:     499,
			zOut:     []float64{0.2383459406360114e+01, 0.2383459541678881e+01, 0.1353283656358167e-06, 0.6165545233233256e-07, 0.1085901506809686e+01, 0.1085921313408604e+01, 0.1986826396898849e-04, 0.2899927470478232e-04, 0.1584986515403004e+01, 0.1588147771130463e+01, 0.3190255011763210e-02, 0.1890859942802736e-02, 0.9412930918090910e+00, 0.1097023073009216e+01, 0.1576208411525267e+00, 0.1913914667196325e+00, 0.1332062774397564e+01, 0.1255053409978084e+01, 0.1143821023097510e+00, 0.7279916643176543e-01, 0.7987861757106404e+00, 0.1091463124754034e+01, 0.3654761154847582e+00, 0.4266153094141972e+00, 0.1274050092339202e+01, 0.9194909148983121e+00, 0.7205613198290631e-01, 0.6547064404326042e-01, 0.8354550922133908e+00, 0.8261962423063568e+00, 0.5621179414582513e-01, 0.7667315937770838e-01, 0.1126935674731953e+01, 0.1417009768728692e+01, 0.3667472533840468e+00, 0.4042756034295816e+00, 0.1562008914947536e+01, 0.1185425268343357e+01, 0.2769195683500124e-01, 0.3111938635974581e-01, 0.1332145183743590e+01, 0.1563083311444625e+01, 0.2620575140703801e+00, 0.2835172362390010e+00, 0.1691083203792688e+01, 0.1491162475787817e+01, 0.8359650824372877e-01, 0.4928202642453804e-01, 0.8790739000820365e+00, 0.8415486989985177e+00, 0.1175682535061811e-01, 0.1564347179719430e-01, 0.1119753245127243e+01, 0.1104508961999007e+01, 0.3991886785572840e-03, 0.1143051254380294e-03, 0.3162690783341116e+00, 0.3165907299376309e+00, 0.4359567385560517e-03, 0.1567431614478116e-03, 0.1138265050331695e+00, 0.1137653492066288e+00, 0.9558734450579545e-04, 0.9991845479738400e-05, 0.1189201139646886e-01, 0.1188202127144326e-01, 0.1730052806966466e-08, 0.1397591405272086e-17, 0.9598672791582074e-11, 0.6617412588681544e-21, 0.1353283656358167e-06, 0.6165545233233256e-07, 0.2818174710670554e-01, 0.1396210563637443e-18, 0.2854399814229969e-05, 0.1344352644992036e-05, 0.1697846193036144e-01, 0.6162975822039155e-31, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			i0Out:    1,
			n0Out:    18,
			ppOut:    0,
			dminOut:  6.6174125886815435e-022,
			desigOut: -5.0906815225379586e-018,
			sigmaOut: 9.7517858564053150e-002,
			qmaxOut:  2.3847059974280183,
			nFailOut: 2,
			iterOut:  25,
			nDivOut:  518,
			ttypeOut: -2,
			dmin1Out: 1.1882019541390456e-002,
			dmin2Out: 0.11366976186212303,
			dnOut:    6.6174125886815435e-022,
			dn1Out:   1.1882019541390456e-002,
			dn2Out:   0.11366976186212303,
			gOut:     0.0000000000000000,
			tauOut:   9.5986713933289272e-012,
		},
		{
			i0:       1,
			n0:       18,
			z:        []float64{0.2383459406360114e+01, 0.2383459541678881e+01, 0.1353283656358167e-06, 0.6165545233233256e-07, 0.1085901506809686e+01, 0.1085921313408604e+01, 0.1986826396898849e-04, 0.2899927470478232e-04, 0.1584986515403004e+01, 0.1588147771130463e+01, 0.3190255011763210e-02, 0.1890859942802736e-02, 0.9412930918090910e+00, 0.1097023073009216e+01, 0.1576208411525267e+00, 0.1913914667196325e+00, 0.1332062774397564e+01, 0.1255053409978084e+01, 0.1143821023097510e+00, 0.7279916643176543e-01, 0.7987861757106404e+00, 0.1091463124754034e+01, 0.3654761154847582e+00, 0.4266153094141972e+00, 0.1274050092339202e+01, 0.9194909148983121e+00, 0.7205613198290631e-01, 0.6547064404326042e-01, 0.8354550922133908e+00, 0.8261962423063568e+00, 0.5621179414582513e-01, 0.7667315937770838e-01, 0.1126935674731953e+01, 0.1417009768728692e+01, 0.3667472533840468e+00, 0.4042756034295816e+00, 0.1562008914947536e+01, 0.1185425268343357e+01, 0.2769195683500124e-01, 0.3111938635974581e-01, 0.1332145183743590e+01, 0.1563083311444625e+01, 0.2620575140703801e+00, 0.2835172362390010e+00, 0.1691083203792688e+01, 0.1491162475787817e+01, 0.8359650824372877e-01, 0.4928202642453804e-01, 0.8790739000820365e+00, 0.8415486989985177e+00, 0.1175682535061811e-01, 0.1564347179719430e-01, 0.1119753245127243e+01, 0.1104508961999007e+01, 0.3991886785572840e-03, 0.1143051254380294e-03, 0.3162690783341116e+00, 0.3165907299376309e+00, 0.4359567385560517e-03, 0.1567431614478116e-03, 0.1138265050331695e+00, 0.1137653492066288e+00, 0.9558734450579545e-04, 0.9991845479738400e-05, 0.1189201139646886e-01, 0.1188202127144326e-01, 0.1730052806966466e-08, 0.1397591405272086e-17, 0.9598672791582074e-11, 0.6617412588681544e-21, 0.1353283656358167e-06, 0.6165545233233256e-07, 0.2818174710670554e-01, 0.1396210563637443e-18, 0.2854399814229969e-05, 0.1344352644992036e-05, 0.1697846193036144e-01, 0.6162975822039155e-31, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			pp:       1,
			dmin:     6.6174125886815435e-022,
			desig:    -5.0906815225379586e-018,
			qmax:     2.3847059974280183,
			ttype:    -2,
			dmin1:    1.1882019541390456e-002,
			dmin2:    0.11366976186212303,
			dn:       6.6174125886815435e-022,
			dn1:      1.1882019541390456e-002,
			dn2:      0.11366976186212303,
			g:        0.0000000000000000,
			tau:      9.5986713933289272e-012,
			nFail:    2,
			iter:     25,
			sigma:    9.7517858564053150e-002,
			nDiv:     518,
			zOut:     []float64{0.2383459603334333e+01, 0.2383459541678881e+01, 0.2809066689524106e-07, 0.6165545233233256e-07, 0.1085950284592642e+01, 0.1085921313408604e+01, 0.4240998334843304e-04, 0.2899927470478232e-04, 0.1589996221089918e+01, 0.1588147771130463e+01, 0.1304604978030436e-02, 0.1890859942802736e-02, 0.1287109934750819e+01, 0.1097023073009216e+01, 0.1866247058326724e+00, 0.1913914667196325e+00, 0.1141227870577177e+01, 0.1255053409978084e+01, 0.6962466280543776e-01, 0.7279916643176543e-01, 0.1448453771362794e+01, 0.1091463124754034e+01, 0.2708190685256155e+00, 0.4266153094141972e+00, 0.7141424904159570e+00, 0.9194909148983121e+00, 0.7574342769943966e-01, 0.6547064404326042e-01, 0.8271259739846255e+00, 0.8261962423063568e+00, 0.1313543755784947e+00, 0.7667315937770838e-01, 0.1689930996579779e+01, 0.1417009768728692e+01, 0.2835846650840220e+00, 0.4042756034295816e+00, 0.9329599896190804e+00, 0.1185425268343357e+01, 0.5213749144931325e-01, 0.3111938635974581e-01, 0.1794463056234313e+01, 0.1563083311444625e+01, 0.2355970842920850e+00, 0.2835172362390010e+00, 0.1304847417920270e+01, 0.1491162475787817e+01, 0.3178396542921673e-01, 0.4928202642453804e-01, 0.8254082053664953e+00, 0.8415486989985177e+00, 0.2093310277804656e-01, 0.1564347179719430e-01, 0.1083690164346398e+01, 0.1104508961999007e+01, 0.3339325601415243e-04, 0.1143051254380294e-03, 0.3167140798430647e+00, 0.3165907299376309e+00, 0.5630296103885627e-04, 0.1567431614478116e-03, 0.1137190380910697e+00, 0.1137653492066288e+00, 0.1044005669799533e-05, 0.9991845479738400e-05, 0.1188097726577346e-01, 0.1188202127144326e-01, 0.7784240935906335e-37, 0.1397591405272086e-17, 0.6617412588681543e-21, 0.6617412588681544e-21, 0.2809066689524106e-07, 0.6165545233233256e-07, 0.2818174710670554e-01, 0.1396210563637443e-18, 0.2854399814229969e-05, 0.1344352644992036e-05, 0.1697846193036144e-01, 0.6162975822039155e-31, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			i0Out:    1,
			n0Out:    18,
			ppOut:    1,
			dminOut:  6.6174125886815426e-022,
			desigOut: -5.0906815225379586e-018,
			sigmaOut: 9.7517858564053150e-002,
			qmaxOut:  2.3847059974280183,
			nFailOut: 2,
			iterOut:  26,
			nDivOut:  537,
			ttypeOut: -2,
			dmin1Out: 1.1880977265773463e-002,
			dmin2Out: 0.11370904624558997,
			dnOut:    6.6174125886815426e-022,
			dn1Out:   1.1880977265773463e-002,
			dn2Out:   0.11370904624558997,
			gOut:     0.0000000000000000,
			tauOut:   0.0000000000000000,
		},
		{
			i0:       1,
			n0:       18,
			z:        []float64{0.2383459603334333e+01, 0.2383459541678881e+01, 0.2809066689524106e-07, 0.6165545233233256e-07, 0.1085950284592642e+01, 0.1085921313408604e+01, 0.4240998334843304e-04, 0.2899927470478232e-04, 0.1589996221089918e+01, 0.1588147771130463e+01, 0.1304604978030436e-02, 0.1890859942802736e-02, 0.1287109934750819e+01, 0.1097023073009216e+01, 0.1866247058326724e+00, 0.1913914667196325e+00, 0.1141227870577177e+01, 0.1255053409978084e+01, 0.6962466280543776e-01, 0.7279916643176543e-01, 0.1448453771362794e+01, 0.1091463124754034e+01, 0.2708190685256155e+00, 0.4266153094141972e+00, 0.7141424904159570e+00, 0.9194909148983121e+00, 0.7574342769943966e-01, 0.6547064404326042e-01, 0.8271259739846255e+00, 0.8261962423063568e+00, 0.1313543755784947e+00, 0.7667315937770838e-01, 0.1689930996579779e+01, 0.1417009768728692e+01, 0.2835846650840220e+00, 0.4042756034295816e+00, 0.9329599896190804e+00, 0.1185425268343357e+01, 0.5213749144931325e-01, 0.3111938635974581e-01, 0.1794463056234313e+01, 0.1563083311444625e+01, 0.2355970842920850e+00, 0.2835172362390010e+00, 0.1304847417920270e+01, 0.1491162475787817e+01, 0.3178396542921673e-01, 0.4928202642453804e-01, 0.8254082053664953e+00, 0.8415486989985177e+00, 0.2093310277804656e-01, 0.1564347179719430e-01, 0.1083690164346398e+01, 0.1104508961999007e+01, 0.3339325601415243e-04, 0.1143051254380294e-03, 0.3167140798430647e+00, 0.3165907299376309e+00, 0.5630296103885627e-04, 0.1567431614478116e-03, 0.1137190380910697e+00, 0.1137653492066288e+00, 0.1044005669799533e-05, 0.9991845479738400e-05, 0.1188097726577346e-01, 0.1188202127144326e-01, 0.7784240935906335e-37, 0.1397591405272086e-17, 0.6617412588681543e-21, 0.6617412588681544e-21, 0.2809066689524106e-07, 0.6165545233233256e-07, 0.2818174710670554e-01, 0.1396210563637443e-18, 0.2854399814229969e-05, 0.1344352644992036e-05, 0.1697846193036144e-01, 0.6162975822039155e-31, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			pp:       0,
			dmin:     6.6174125886815426e-022,
			desig:    -5.0906815225379586e-018,
			qmax:     2.3847059974280183,
			ttype:    -2,
			dmin1:    1.1880977265773463e-002,
			dmin2:    0.11370904624558997,
			dn:       6.6174125886815426e-022,
			dn1:      1.1880977265773463e-002,
			dn2:      0.11370904624558997,
			g:        0.0000000000000000,
			tau:      0.0000000000000000,
			nFail:    2,
			iter:     26,
			sigma:    9.7517858564053150e-002,
			nDiv:     537,
			zOut:     []float64{0.2383459603334333e+01, 0.2371578799269292e+01, 0.2809066689524106e-07, 0.1286276792433928e-07, 0.1085950284592642e+01, 0.1074111849557515e+01, 0.4240998334843304e-04, 0.6277904232066118e-04, 0.1589996221089918e+01, 0.1579357214869920e+01, 0.1304604978030436e-02, 0.1063198377376995e-02, 0.1287109934750819e+01, 0.1460790610050406e+01, 0.1866247058326724e+00, 0.1457986614708343e+00, 0.1141227870577177e+01, 0.1053173039756072e+01, 0.6962466280543776e-01, 0.9575644420574685e-01, 0.1448453771362794e+01, 0.1611635563526955e+01, 0.2708190685256155e+00, 0.1200044280642223e+00, 0.7141424904159570e+00, 0.6580006578954666e+00, 0.7574342769943966e-01, 0.9521169265880262e-01, 0.8271259739846255e+00, 0.8513878247486099e+00, 0.1313543755784947e+00, 0.2607270439790754e+00, 0.1689930996579779e+01, 0.1700907785529018e+01, 0.2835846650840220e+00, 0.1555482010511415e+00, 0.9329599896190804e+00, 0.8176684478615445e+00, 0.5213749144931325e-01, 0.1144214412274440e+00, 0.1794463056234313e+01, 0.1903757867143247e+01, 0.2355970842920850e+00, 0.1614796988702029e+00, 0.1304847417920270e+01, 0.1163270852323576e+01, 0.3178396542921673e-01, 0.2255256874351997e-01, 0.8254082053664953e+00, 0.8119079072453143e+00, 0.2093310277804656e-01, 0.2794035799797570e-01, 0.1083690164346398e+01, 0.1043902367448729e+01, 0.3339325601415243e-04, 0.1013132519024162e-04, 0.3167140798430647e+00, 0.3048794193232055e+00, 0.5630296103885627e-04, 0.2100082250625822e-04, 0.1137190380910697e+00, 0.1018182491185255e+00, 0.1044005669799533e-05, 0.1218230301111122e-06, 0.1188097726577346e-01, 0.2328703557243073e-07, 0.7784240935906335e-37, 0.1286276792433928e-07, 0.9751785856405315e-01, 0.6617412588681544e-21, 0.2809066689524106e-07, 0.6165545233233256e-07, 0.2818174710670554e-01, 0.1396210563637443e-18, 0.2854399814229969e-05, 0.1344352644992036e-05, 0.1697846193036144e-01, 0.6162975822039155e-31, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			i0Out:    1,
			n0Out:    17,
			ppOut:    0,
			dminOut:  2.3287035572430725e-008,
			desigOut: 1.7347234759768071e-018,
			sigmaOut: 0.10939869071976092,
			qmaxOut:  2.3847059974280183,
			nFailOut: 2,
			iterOut:  27,
			nDivOut:  555,
			ttypeOut: -7,
			dmin1Out: 0.10181720511285566,
			dmin2Out: 0.30482311636216664,
			dnOut:    2.3287035572430725e-008,
			dn1Out:   0.10181720511285566,
			dn2Out:   0.30482311636216664,
			gOut:     0.0000000000000000,
			tauOut:   1.1880832155707781e-002,
		},
		{
			i0:       1,
			n0:       17,
			z:        []float64{0.2383459603334333e+01, 0.2371578799269292e+01, 0.2809066689524106e-07, 0.1286276792433928e-07, 0.1085950284592642e+01, 0.1074111849557515e+01, 0.4240998334843304e-04, 0.6277904232066118e-04, 0.1589996221089918e+01, 0.1579357214869920e+01, 0.1304604978030436e-02, 0.1063198377376995e-02, 0.1287109934750819e+01, 0.1460790610050406e+01, 0.1866247058326724e+00, 0.1457986614708343e+00, 0.1141227870577177e+01, 0.1053173039756072e+01, 0.6962466280543776e-01, 0.9575644420574685e-01, 0.1448453771362794e+01, 0.1611635563526955e+01, 0.2708190685256155e+00, 0.1200044280642223e+00, 0.7141424904159570e+00, 0.6580006578954666e+00, 0.7574342769943966e-01, 0.9521169265880262e-01, 0.8271259739846255e+00, 0.8513878247486099e+00, 0.1313543755784947e+00, 0.2607270439790754e+00, 0.1689930996579779e+01, 0.1700907785529018e+01, 0.2835846650840220e+00, 0.1555482010511415e+00, 0.9329599896190804e+00, 0.8176684478615445e+00, 0.5213749144931325e-01, 0.1144214412274440e+00, 0.1794463056234313e+01, 0.1903757867143247e+01, 0.2355970842920850e+00, 0.1614796988702029e+00, 0.1304847417920270e+01, 0.1163270852323576e+01, 0.3178396542921673e-01, 0.2255256874351997e-01, 0.8254082053664953e+00, 0.8119079072453143e+00, 0.2093310277804656e-01, 0.2794035799797570e-01, 0.1083690164346398e+01, 0.1043902367448729e+01, 0.3339325601415243e-04, 0.1013132519024162e-04, 0.3167140798430647e+00, 0.3048794193232055e+00, 0.5630296103885627e-04, 0.2100082250625822e-04, 0.1137190380910697e+00, 0.1018182491185255e+00, 0.1044005669799533e-05, 0.1218230301111122e-06, 0.1188097726577346e-01, 0.2328703557243073e-07, 0.7784240935906335e-37, 0.1286276792433928e-07, 0.9751785856405315e-01, 0.6617412588681544e-21, 0.2809066689524106e-07, 0.6165545233233256e-07, 0.2818174710670554e-01, 0.1396210563637443e-18, 0.2854399814229969e-05, 0.1344352644992036e-05, 0.1697846193036144e-01, 0.6162975822039155e-31, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			pp:       1,
			dmin:     2.3287035572430725e-008,
			desig:    1.7347234759768071e-018,
			qmax:     2.3847059974280183,
			ttype:    -7,
			dmin1:    0.10181720511285566,
			dmin2:    0.30482311636216664,
			dn:       2.3287035572430725e-008,
			dn1:      0.10181720511285566,
			dn2:      0.30482311636216664,
			g:        0.0000000000000000,
			tau:      1.1880832155707781e-002,
			nFail:    2,
			iter:     27,
			sigma:    0.10939869071976092,
			nDiv:     555,
			zOut:     []float64{0.2371578788845052e+01, 0.2371578799269292e+01, 0.5825676764620370e-08, 0.1286276792433928e-07, 0.1074174599487151e+01, 0.1074111849557515e+01, 0.9230392664199863e-04, 0.6277904232066118e-04, 0.1580328086033647e+01, 0.1579357214869920e+01, 0.9827770701659698e-03, 0.1063198377376995e-02, 0.1605606471164067e+01, 0.1460790610050406e+01, 0.9563440497488795e-01, 0.1457986614708343e+00, 0.1053295055699924e+01, 0.1053173039756072e+01, 0.1465159169633776e+00, 0.9575644420574685e-01, 0.1585124051340792e+01, 0.1611635563526955e+01, 0.4981502397231050e-01, 0.1200044280642223e+00, 0.7033973032949510e+00, 0.6580006578954666e+00, 0.1152436546510613e+00, 0.9521169265880262e-01, 0.9968711907896164e+00, 0.8513878247486099e+00, 0.4448645553200346e+00, 0.2607270439790754e+00, 0.1411591407973117e+01, 0.1700907785529018e+01, 0.9010174998427344e-01, 0.1555482010511415e+00, 0.8419881158177074e+00, 0.8176684478615445e+00, 0.2587099684834221e+00, 0.1144214412274440e+00, 0.1806527574243020e+01, 0.1903757867143247e+01, 0.1039810460775319e+00, 0.1614796988702029e+00, 0.1081842351702556e+01, 0.1163270852323576e+01, 0.1692539477932339e-01, 0.2255256874351997e-01, 0.8229228471769590e+00, 0.8119079072453143e+00, 0.3544318396494814e-01, 0.2794035799797570e-01, 0.1008469291521964e+01, 0.1043902367448729e+01, 0.3062892015595061e-05, 0.1013132519024162e-04, 0.3048973339666884e+00, 0.3048794193232055e+00, 0.7013072071892081e-05, 0.2100082250625822e-04, 0.1018113345824760e+00, 0.1018182491185255e+00, 0.2786425742647189e-13, 0.1218230301111122e-06, 0.2696165428114542e-17, 0.2328703557243073e-07, 0.5825676764620370e-08, 0.1286276792433928e-07, 0.9751785856405315e-01, 0.6617412588681544e-21, 0.2809066689524106e-07, 0.6165545233233256e-07, 0.2818174710670554e-01, 0.1396210563637443e-18, 0.2854399814229969e-05, 0.1344352644992036e-05, 0.1697846193036144e-01, 0.6162975822039155e-31, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			i0Out:    1,
			n0Out:    17,
			ppOut:    1,
			dminOut:  2.6961654281145418e-018,
			desigOut: -3.8251180717391650e-018,
			sigmaOut: 0.10939871400676864,
			qmaxOut:  2.3847059974280183,
			nFailOut: 2,
			iterOut:  28,
			nDivOut:  573,
			ttypeOut: -2,
			dmin1Out: 0.10181121275944585,
			dmin2Out: 0.30487633314418217,
			dnOut:    2.6961654281145418e-018,
			dn1Out:   0.10181121275944585,
			dn2Out:   0.30487633314418217,
			gOut:     0.0000000000000000,
			tauOut:   2.3287007705477136e-008,
		},
		{
			i0:       1,
			n0:       17,
			z:        []float64{0.2371578788845052e+01, 0.2371578799269292e+01, 0.5825676764620370e-08, 0.1286276792433928e-07, 0.1074174599487151e+01, 0.1074111849557515e+01, 0.9230392664199863e-04, 0.6277904232066118e-04, 0.1580328086033647e+01, 0.1579357214869920e+01, 0.9827770701659698e-03, 0.1063198377376995e-02, 0.1605606471164067e+01, 0.1460790610050406e+01, 0.9563440497488795e-01, 0.1457986614708343e+00, 0.1053295055699924e+01, 0.1053173039756072e+01, 0.1465159169633776e+00, 0.9575644420574685e-01, 0.1585124051340792e+01, 0.1611635563526955e+01, 0.4981502397231050e-01, 0.1200044280642223e+00, 0.7033973032949510e+00, 0.6580006578954666e+00, 0.1152436546510613e+00, 0.9521169265880262e-01, 0.9968711907896164e+00, 0.8513878247486099e+00, 0.4448645553200346e+00, 0.2607270439790754e+00, 0.1411591407973117e+01, 0.1700907785529018e+01, 0.9010174998427344e-01, 0.1555482010511415e+00, 0.8419881158177074e+00, 0.8176684478615445e+00, 0.2587099684834221e+00, 0.1144214412274440e+00, 0.1806527574243020e+01, 0.1903757867143247e+01, 0.1039810460775319e+00, 0.1614796988702029e+00, 0.1081842351702556e+01, 0.1163270852323576e+01, 0.1692539477932339e-01, 0.2255256874351997e-01, 0.8229228471769590e+00, 0.8119079072453143e+00, 0.3544318396494814e-01, 0.2794035799797570e-01, 0.1008469291521964e+01, 0.1043902367448729e+01, 0.3062892015595061e-05, 0.1013132519024162e-04, 0.3048973339666884e+00, 0.3048794193232055e+00, 0.7013072071892081e-05, 0.2100082250625822e-04, 0.1018113345824760e+00, 0.1018182491185255e+00, 0.2786425742647189e-13, 0.1218230301111122e-06, 0.2696165428114542e-17, 0.2328703557243073e-07, 0.5825676764620370e-08, 0.1286276792433928e-07, 0.9751785856405315e-01, 0.6617412588681544e-21, 0.2809066689524106e-07, 0.6165545233233256e-07, 0.2818174710670554e-01, 0.1396210563637443e-18, 0.2854399814229969e-05, 0.1344352644992036e-05, 0.1697846193036144e-01, 0.6162975822039155e-31, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			pp:       0,
			dmin:     2.6961654281145418e-018,
			desig:    -3.8251180717391650e-018,
			qmax:     2.3847059974280183,
			ttype:    -2,
			dmin1:    0.10181121275944585,
			dmin2:    0.30487633314418217,
			dn:       2.6961654281145418e-018,
			dn1:      0.10181121275944585,
			dn2:      0.30487633314418217,
			g:        0.0000000000000000,
			tau:      2.3287007705477136e-008,
			nFail:    2,
			iter:     28,
			sigma:    0.10939871400676864,
			nDiv:     573,
			zOut:     []float64{0.2371578788845052e+01, 0.2371578794670729e+01, 0.5825676764620370e-08, 0.2638661645752538e-08, 0.1074174599487151e+01, 0.1074266900775131e+01, 0.9230392664199863e-04, 0.1357860766428602e-03, 0.1580328086033647e+01, 0.1581175077027170e+01, 0.9827770701659698e-03, 0.9979623676695663e-03, 0.1605606471164067e+01, 0.1700242913771285e+01, 0.9563440497488795e-01, 0.5924520849284005e-01, 0.1053295055699924e+01, 0.1140565764170461e+01, 0.1465159169633776e+00, 0.2036234219705987e+00, 0.1585124051340792e+01, 0.1431315653342504e+01, 0.4981502397231050e-01, 0.2448080089382756e-01, 0.7033973032949510e+00, 0.7941601570521848e+00, 0.1152436546510613e+00, 0.1446598374682775e+00, 0.9968711907896164e+00, 0.1297075908641373e+01, 0.4448645553200346e+00, 0.4841405038964208e+00, 0.1411591407973117e+01, 0.1017552654060970e+01, 0.9010174998427344e-01, 0.7455594793877945e-01, 0.8419881158177074e+00, 0.1026142136362350e+01, 0.2587099684834221e+00, 0.4554599945126984e+00, 0.1806527574243020e+01, 0.1455048625807853e+01, 0.1039810460775319e+00, 0.7731088667813635e-01, 0.1081842351702556e+01, 0.1021456859803743e+01, 0.1692539477932339e-01, 0.1363571444815687e-01, 0.8229228471769590e+00, 0.8447303166937503e+00, 0.3544318396494814e-01, 0.4231334180394086e-01, 0.1008469291521964e+01, 0.9661590126100381e+00, 0.3062892015595061e-05, 0.9665775484099522e-06, 0.3048973339666884e+00, 0.3049033804612119e+00, 0.7013072071892081e-05, 0.2341758973227438e-05, 0.1018113345824760e+00, 0.1018089928235306e+00, 0.2786425742647189e-13, 0.7379175991216932e-30, 0.2696165428114542e-17, 0.2696165428113804e-17, 0.5825676764620370e-08, 0.2638661645752538e-08, 0.9751785856405315e-01, 0.6617412588681544e-21, 0.2809066689524106e-07, 0.6165545233233256e-07, 0.2818174710670554e-01, 0.1396210563637443e-18, 0.2854399814229969e-05, 0.1344352644992036e-05, 0.1697846193036144e-01, 0.6162975822039155e-31, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			i0Out:    1,
			n0Out:    17,
			ppOut:    0,
			dminOut:  2.6961654281138038e-018,
			desigOut: -3.8251180717391650e-018,
			sigmaOut: 0.10939871400676864,
			qmaxOut:  2.3847059974280183,
			nFailOut: 2,
			iterOut:  29,
			nDivOut:  591,
			ttypeOut: -2,
			dmin1Out: 0.10180899282350273,
			dmin2Out: 0.30489636738914000,
			dnOut:    2.6961654281138038e-018,
			dn1Out:   0.10180899282350273,
			dn2Out:   0.30489636738914000,
			gOut:     0.0000000000000000,
			tauOut:   0.0000000000000000,
		},
		{
			i0:       1,
			n0:       17,
			z:        []float64{0.2371578788845052e+01, 0.2371578794670729e+01, 0.5825676764620370e-08, 0.2638661645752538e-08, 0.1074174599487151e+01, 0.1074266900775131e+01, 0.9230392664199863e-04, 0.1357860766428602e-03, 0.1580328086033647e+01, 0.1581175077027170e+01, 0.9827770701659698e-03, 0.9979623676695663e-03, 0.1605606471164067e+01, 0.1700242913771285e+01, 0.9563440497488795e-01, 0.5924520849284005e-01, 0.1053295055699924e+01, 0.1140565764170461e+01, 0.1465159169633776e+00, 0.2036234219705987e+00, 0.1585124051340792e+01, 0.1431315653342504e+01, 0.4981502397231050e-01, 0.2448080089382756e-01, 0.7033973032949510e+00, 0.7941601570521848e+00, 0.1152436546510613e+00, 0.1446598374682775e+00, 0.9968711907896164e+00, 0.1297075908641373e+01, 0.4448645553200346e+00, 0.4841405038964208e+00, 0.1411591407973117e+01, 0.1017552654060970e+01, 0.9010174998427344e-01, 0.7455594793877945e-01, 0.8419881158177074e+00, 0.1026142136362350e+01, 0.2587099684834221e+00, 0.4554599945126984e+00, 0.1806527574243020e+01, 0.1455048625807853e+01, 0.1039810460775319e+00, 0.7731088667813635e-01, 0.1081842351702556e+01, 0.1021456859803743e+01, 0.1692539477932339e-01, 0.1363571444815687e-01, 0.8229228471769590e+00, 0.8447303166937503e+00, 0.3544318396494814e-01, 0.4231334180394086e-01, 0.1008469291521964e+01, 0.9661590126100381e+00, 0.3062892015595061e-05, 0.9665775484099522e-06, 0.3048973339666884e+00, 0.3049033804612119e+00, 0.7013072071892081e-05, 0.2341758973227438e-05, 0.1018113345824760e+00, 0.1018089928235306e+00, 0.2786425742647189e-13, 0.7379175991216932e-30, 0.2696165428114542e-17, 0.2696165428113804e-17, 0.5825676764620370e-08, 0.2638661645752538e-08, 0.9751785856405315e-01, 0.6617412588681544e-21, 0.2809066689524106e-07, 0.6165545233233256e-07, 0.2818174710670554e-01, 0.1396210563637443e-18, 0.2854399814229969e-05, 0.1344352644992036e-05, 0.1697846193036144e-01, 0.6162975822039155e-31, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			pp:       1,
			dmin:     2.6961654281138038e-018,
			desig:    -3.8251180717391650e-018,
			qmax:     2.3847059974280183,
			ttype:    -2,
			dmin1:    0.10180899282350273,
			dmin2:    0.30489636738914000,
			dn:       2.6961654281138038e-018,
			dn1:      0.10180899282350273,
			dn2:      0.30489636738914000,
			g:        0.0000000000000000,
			tau:      0.0000000000000000,
			nFail:    2,
			iter:     29,
			sigma:    0.10939871400676864,
			nDiv:     591,
			zOut:     []float64{0.2269772292606755e+01, 0.2371578794670729e+01, 0.1248859578385863e-08, 0.2638661645752538e-08, 0.9725961809002787e+00, 0.1074266900775131e+01, 0.2207509801202943e-03, 0.1357860766428602e-03, 0.1480145783712083e+01, 0.1581175077027170e+01, 0.1146359002276934e-02, 0.9979623676695663e-03, 0.1656535258559212e+01, 0.1700242913771285e+01, 0.4079180092843103e-01, 0.5924520849284005e-01, 0.1201590880509993e+01, 0.1140565764170461e+01, 0.2425529321011353e+00, 0.2036234219705987e+00, 0.1111437017432560e+01, 0.1431315653342504e+01, 0.1749237822536809e-01, 0.2448080089382756e-01, 0.8195211115924584e+00, 0.7941601570521848e+00, 0.2289566278084960e+00, 0.1446598374682775e+00, 0.1450453280026662e+01, 0.1297075908641373e+01, 0.3396444831847066e+00, 0.4841405038964208e+00, 0.6506576141124067e+00, 0.1017552654060970e+01, 0.1175810411452473e+00, 0.7455594793877945e-01, 0.1262214585027166e+01, 0.1026142136362350e+01, 0.5250426092262996e+00, 0.4554599945126984e+00, 0.9055103985570538e+00, 0.1455048625807853e+01, 0.8721019179982008e-01, 0.7731088667813635e-01, 0.8460758777494440e+00, 0.1021456859803743e+01, 0.1361402882064932e-01, 0.1363571444815687e-01, 0.7716231249744060e+00, 0.8447303166937503e+00, 0.5298106707064092e-01, 0.4231334180394086e-01, 0.8113724074143097e+00, 0.9661590126100381e+00, 0.3632274887524200e-06, 0.9665775484099522e-06, 0.2030988542900605e+00, 0.3049033804612119e+00, 0.1173872316183806e-05, 0.2341758973227438e-05, 0.1314248578535016e-05, 0.1018089928235306e+00, 0.1248859578385863e-08, 0.7379175991216932e-30, 0.1093987140067686e+00, 0.2696165428113804e-17, 0.5825676764620370e-08, 0.2638661645752538e-08, 0.9751785856405315e-01, 0.6617412588681544e-21, 0.2809066689524106e-07, 0.6165545233233256e-07, 0.2818174710670554e-01, 0.1396210563637443e-18, 0.2854399814229969e-05, 0.1344352644992036e-05, 0.1697846193036144e-01, 0.6162975822039155e-31, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			i0Out:    1,
			n0Out:    16,
			ppOut:    1,
			dminOut:  1.3142485785350155e-006,
			desigOut: 0.0000000000000000,
			sigmaOut: 0.21120521870940451,
			qmaxOut:  2.3847059974280183,
			nFailOut: 2,
			iterOut:  30,
			nDivOut:  608,
			ttypeOut: -7,
			dmin1Out: 0.20309651253108729,
			dmin2Out: 0.57610166617362735,
			dnOut:    1.3142485785350155e-006,
			dn1Out:   0.20309651253108729,
			dn2Out:   0.81137144083676127,
			gOut:     0.0000000000000000,
			tauOut:   0.10180650470263587,
		},
		{
			i0:       1,
			n0:       16,
			z:        []float64{0.2269772292606755e+01, 0.2371578794670729e+01, 0.1248859578385863e-08, 0.2638661645752538e-08, 0.9725961809002787e+00, 0.1074266900775131e+01, 0.2207509801202943e-03, 0.1357860766428602e-03, 0.1480145783712083e+01, 0.1581175077027170e+01, 0.1146359002276934e-02, 0.9979623676695663e-03, 0.1656535258559212e+01, 0.1700242913771285e+01, 0.4079180092843103e-01, 0.5924520849284005e-01, 0.1201590880509993e+01, 0.1140565764170461e+01, 0.2425529321011353e+00, 0.2036234219705987e+00, 0.1111437017432560e+01, 0.1431315653342504e+01, 0.1749237822536809e-01, 0.2448080089382756e-01, 0.8195211115924584e+00, 0.7941601570521848e+00, 0.2289566278084960e+00, 0.1446598374682775e+00, 0.1450453280026662e+01, 0.1297075908641373e+01, 0.3396444831847066e+00, 0.4841405038964208e+00, 0.6506576141124067e+00, 0.1017552654060970e+01, 0.1175810411452473e+00, 0.7455594793877945e-01, 0.1262214585027166e+01, 0.1026142136362350e+01, 0.5250426092262996e+00, 0.4554599945126984e+00, 0.9055103985570538e+00, 0.1455048625807853e+01, 0.8721019179982008e-01, 0.7731088667813635e-01, 0.8460758777494440e+00, 0.1021456859803743e+01, 0.1361402882064932e-01, 0.1363571444815687e-01, 0.7716231249744060e+00, 0.8447303166937503e+00, 0.5298106707064092e-01, 0.4231334180394086e-01, 0.8113724074143097e+00, 0.9661590126100381e+00, 0.3632274887524200e-06, 0.9665775484099522e-06, 0.2030988542900605e+00, 0.3049033804612119e+00, 0.1173872316183806e-05, 0.2341758973227438e-05, 0.1314248578535016e-05, 0.1018089928235306e+00, 0.1248859578385863e-08, 0.2638661645752538e-08, 0.1093987140067686e+00, 0.2696165428113804e-17, 0.5825676764620370e-08, 0.2638661645752538e-08, 0.9751785856405315e-01, 0.6617412588681544e-21, 0.2809066689524106e-07, 0.6165545233233256e-07, 0.2818174710670554e-01, 0.1396210563637443e-18, 0.2854399814229969e-05, 0.1344352644992036e-05, 0.1697846193036144e-01, 0.6162975822039155e-31, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			pp:       0,
			dmin:     1.3142485785350155e-006,
			desig:    0.0000000000000000,
			qmax:     2.2697722926067549,
			ttype:    -7,
			dmin1:    0.20309651253108729,
			dmin2:    0.57610166617362735,
			dn:       1.3142485785350155e-006,
			dn1:      0.20309651253108729,
			dn2:      0.81137144083676127,
			g:        0.0000000000000000,
			tau:      0.10180650470263587,
			nFail:    2,
			iter:     30,
			sigma:    0.21120521870940451,
			nDiv:     608,
			zOut:     []float64{0.2269772292606755e+01, 0.2269770979614632e+01, 0.1248859578385863e-08, 0.5351359530665278e-09, 0.9725961809002787e+00, 0.9728156171042807e+00, 0.2207509801202943e-03, 0.3358741643642203e-03, 0.1480145783712083e+01, 0.1480954954309013e+01, 0.1146359002276934e-02, 0.1282269997958535e-02, 0.1656535258559212e+01, 0.1696043475248702e+01, 0.4079180092843103e-01, 0.2889964597634762e-01, 0.1201590880509993e+01, 0.1415242852393798e+01, 0.2425529321011353e+00, 0.1904848393814713e+00, 0.1111437017432560e+01, 0.9384432420354740e+00, 0.1749237822536809e-01, 0.1527569554079381e-01, 0.8195211115924584e+00, 0.1033200729619178e+01, 0.2289566278084960e+00, 0.3214195289148511e+00, 0.1450453280026662e+01, 0.1468676920055535e+01, 0.3396444831847066e+00, 0.1504703083827628e+00, 0.6506576141124067e+00, 0.6177670326339089e+00, 0.1175810411452473e+00, 0.2402402478867145e+00, 0.1262214585027166e+01, 0.1547015632125768e+01, 0.5250426092262996e+00, 0.3073217441808570e+00, 0.9055103985570538e+00, 0.6853975319350345e+00, 0.8721019179982008e-01, 0.1076549537133787e+00, 0.8460758777494440e+00, 0.7520336386157322e+00, 0.1361402882064932e-01, 0.1396865635082151e-01, 0.7716231249744060e+00, 0.8106342214532430e+00, 0.5298106707064092e-01, 0.5302931309687425e-01, 0.8113724074143097e+00, 0.7583421433039418e+00, 0.3632274887524200e-06, 0.9727942389020712e-07, 0.2030988542900605e+00, 0.2030986166419704e+00, 0.1173872316183806e-05, 0.7596112905317353e-11, 0.1314248578535016e-05, 0.8411884199182457e-17, 0.1248859578385863e-08, 0.5351359530665278e-09, 0.1093987140067686e+00, 0.2696165428113804e-17, 0.5825676764620370e-08, 0.2638661645752538e-08, 0.9751785856405315e-01, 0.6617412588681544e-21, 0.2809066689524106e-07, 0.6165545233233256e-07, 0.2818174710670554e-01, 0.1396210563637443e-18, 0.2854399814229969e-05, 0.1344352644992036e-05, 0.1697846193036144e-01, 0.6162975822039155e-31, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			i0Out:    1,
			n0Out:    16,
			ppOut:    0,
			dminOut:  8.4118841991824567e-018,
			desigOut: 1.0739530738237274e-017,
			sigmaOut: 0.21120653295038691,
			qmaxOut:  2.2697722926067549,
			nFailOut: 2,
			iterOut:  31,
			nDivOut:  625,
			ttypeOut: -2,
			dmin1Out: 0.20309744276965425,
			dmin2Out: 0.50018599148866161,
			dnOut:    8.4118841991824567e-018,
			dn1Out:   0.20309744276965425,
			dn2Out:   0.75834178007645303,
			gOut:     0.0000000000000000,
			tauOut:   1.3142409824136984e-006,
		},
		{
			i0:       1,
			n0:       16,
			z:        []float64{0.2269772292606755e+01, 0.2269770979614632e+01, 0.1248859578385863e-08, 0.5351359530665278e-09, 0.9725961809002787e+00, 0.9728156171042807e+00, 0.2207509801202943e-03, 0.3358741643642203e-03, 0.1480145783712083e+01, 0.1480954954309013e+01, 0.1146359002276934e-02, 0.1282269997958535e-02, 0.1656535258559212e+01, 0.1696043475248702e+01, 0.4079180092843103e-01, 0.2889964597634762e-01, 0.1201590880509993e+01, 0.1415242852393798e+01, 0.2425529321011353e+00, 0.1904848393814713e+00, 0.1111437017432560e+01, 0.9384432420354740e+00, 0.1749237822536809e-01, 0.1527569554079381e-01, 0.8195211115924584e+00, 0.1033200729619178e+01, 0.2289566278084960e+00, 0.3214195289148511e+00, 0.1450453280026662e+01, 0.1468676920055535e+01, 0.3396444831847066e+00, 0.1504703083827628e+00, 0.6506576141124067e+00, 0.6177670326339089e+00, 0.1175810411452473e+00, 0.2402402478867145e+00, 0.1262214585027166e+01, 0.1547015632125768e+01, 0.5250426092262996e+00, 0.3073217441808570e+00, 0.9055103985570538e+00, 0.6853975319350345e+00, 0.8721019179982008e-01, 0.1076549537133787e+00, 0.8460758777494440e+00, 0.7520336386157322e+00, 0.1361402882064932e-01, 0.1396865635082151e-01, 0.7716231249744060e+00, 0.8106342214532430e+00, 0.5298106707064092e-01, 0.5302931309687425e-01, 0.8113724074143097e+00, 0.7583421433039418e+00, 0.3632274887524200e-06, 0.9727942389020712e-07, 0.2030988542900605e+00, 0.2030986166419704e+00, 0.1173872316183806e-05, 0.7596112905317353e-11, 0.1314248578535016e-05, 0.8411884199182457e-17, 0.1248859578385863e-08, 0.5351359530665278e-09, 0.1093987140067686e+00, 0.2696165428113804e-17, 0.5825676764620370e-08, 0.2638661645752538e-08, 0.9751785856405315e-01, 0.6617412588681544e-21, 0.2809066689524106e-07, 0.6165545233233256e-07, 0.2818174710670554e-01, 0.1396210563637443e-18, 0.2854399814229969e-05, 0.1344352644992036e-05, 0.1697846193036144e-01, 0.6162975822039155e-31, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			pp:       1,
			dmin:     8.4118841991824567e-018,
			desig:    1.0739530738237274e-017,
			qmax:     2.2697722926067549,
			ttype:    -2,
			dmin1:    0.20309744276965425,
			dmin2:    0.50018599148866161,
			dn:       8.4118841991824567e-018,
			dn1:      0.20309744276965425,
			dn2:      0.75834178007645303,
			g:        0.0000000000000000,
			tau:      1.3142409824136984e-006,
			nFail:    2,
			iter:     31,
			sigma:    0.21120653295038691,
			nDiv:     625,
			zOut:     []float64{0.2269770980149768e+01, 0.2269770979614632e+01, 0.2293573303077261e-09, 0.5351359530665278e-09, 0.9731514910392876e+00, 0.9728156171042807e+00, 0.5111377954200868e-03, 0.3358741643642203e-03, 0.1481726086511552e+01, 0.1480954954309013e+01, 0.1467737987028945e-02, 0.1282269997958535e-02, 0.1723475383238021e+01, 0.1696043475248702e+01, 0.2373112943910766e-01, 0.2889964597634762e-01, 0.1581996562336162e+01, 0.1415242852393798e+01, 0.1129959536471923e+00, 0.1904848393814713e+00, 0.8407229839290754e+00, 0.9384432420354740e+00, 0.1877296098701645e-01, 0.1527569554079381e-01, 0.1335847297547013e+01, 0.1033200729619178e+01, 0.3533797947109679e+00, 0.3214195289148511e+00, 0.1265767433727330e+01, 0.1468676920055535e+01, 0.7343813202351114e-01, 0.1504703083827628e+00, 0.7845691484971122e+00, 0.6177670326339089e+00, 0.4737063898809231e+00, 0.2402402478867145e+00, 0.1380630986425702e+01, 0.1547015632125768e+01, 0.1525661578238559e+00, 0.3073217441808570e+00, 0.6404863278245572e+00, 0.6853975319350345e+00, 0.1264041760751794e+00, 0.1076549537133787e+00, 0.6395981188913744e+00, 0.7520336386157322e+00, 0.1770404028911661e-01, 0.1396865635082151e-01, 0.8459594942610007e+00, 0.8106342214532430e+00, 0.4753698401003136e-01, 0.5302931309687425e-01, 0.7108052565733343e+00, 0.7583421433039418e+00, 0.2779568135873871e-07, 0.9727942389020712e-07, 0.2030985888538852e+00, 0.2030986166419704e+00, 0.3146138162949754e-27, 0.7596112905317353e-11, 0.8411884198867843e-17, 0.8411884199182457e-17, 0.2293573303077261e-09, 0.5351359530665278e-09, 0.1093987140067686e+00, 0.2696165428113804e-17, 0.5825676764620370e-08, 0.2638661645752538e-08, 0.9751785856405315e-01, 0.6617412588681544e-21, 0.2809066689524106e-07, 0.6165545233233256e-07, 0.2818174710670554e-01, 0.1396210563637443e-18, 0.2854399814229969e-05, 0.1344352644992036e-05, 0.1697846193036144e-01, 0.6162975822039155e-31, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			i0Out:    1,
			n0Out:    16,
			ppOut:    1,
			dminOut:  8.4118841988678429e-018,
			desigOut: 1.0739530738237274e-017,
			sigmaOut: 0.21120653295038691,
			qmaxOut:  2.2697722926067549,
			nFailOut: 2,
			iterOut:  32,
			nDivOut:  642,
			ttypeOut: -2,
			dmin1Out: 0.20309858884628909,
			dmin2Out: 0.53283137411117854,
			dnOut:    8.4118841988678429e-018,
			dn1Out:   0.20309858884628909,
			dn2Out:   0.71080515929391042,
			gOut:     0.0000000000000000,
			tauOut:   0.0000000000000000,
		},
		{
			i0:       1,
			n0:       16,
			z:        []float64{0.2269770980149768e+01, 0.2269770979614632e+01, 0.2293573303077261e-09, 0.5351359530665278e-09, 0.9731514910392876e+00, 0.9728156171042807e+00, 0.5111377954200868e-03, 0.3358741643642203e-03, 0.1481726086511552e+01, 0.1480954954309013e+01, 0.1467737987028945e-02, 0.1282269997958535e-02, 0.1723475383238021e+01, 0.1696043475248702e+01, 0.2373112943910766e-01, 0.2889964597634762e-01, 0.1581996562336162e+01, 0.1415242852393798e+01, 0.1129959536471923e+00, 0.1904848393814713e+00, 0.8407229839290754e+00, 0.9384432420354740e+00, 0.1877296098701645e-01, 0.1527569554079381e-01, 0.1335847297547013e+01, 0.1033200729619178e+01, 0.3533797947109679e+00, 0.3214195289148511e+00, 0.1265767433727330e+01, 0.1468676920055535e+01, 0.7343813202351114e-01, 0.1504703083827628e+00, 0.7845691484971122e+00, 0.6177670326339089e+00, 0.4737063898809231e+00, 0.2402402478867145e+00, 0.1380630986425702e+01, 0.1547015632125768e+01, 0.1525661578238559e+00, 0.3073217441808570e+00, 0.6404863278245572e+00, 0.6853975319350345e+00, 0.1264041760751794e+00, 0.1076549537133787e+00, 0.6395981188913744e+00, 0.7520336386157322e+00, 0.1770404028911661e-01, 0.1396865635082151e-01, 0.8459594942610007e+00, 0.8106342214532430e+00, 0.4753698401003136e-01, 0.5302931309687425e-01, 0.7108052565733343e+00, 0.7583421433039418e+00, 0.2779568135873871e-07, 0.9727942389020712e-07, 0.2030985888538852e+00, 0.2030986166419704e+00, 0.3146138162949754e-27, 0.7596112905317353e-11, 0.8411884198867843e-17, 0.8411884199182457e-17, 0.2293573303077261e-09, 0.5351359530665278e-09, 0.1093987140067686e+00, 0.2696165428113804e-17, 0.5825676764620370e-08, 0.2638661645752538e-08, 0.9751785856405315e-01, 0.6617412588681544e-21, 0.2809066689524106e-07, 0.6165545233233256e-07, 0.2818174710670554e-01, 0.1396210563637443e-18, 0.2854399814229969e-05, 0.1344352644992036e-05, 0.1697846193036144e-01, 0.6162975822039155e-31, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			pp:       0,
			dmin:     8.4118841988678429e-018,
			desig:    1.0739530738237274e-017,
			qmax:     2.2697722926067549,
			ttype:    -2,
			dmin1:    0.20309858884628909,
			dmin2:    0.53283137411117854,
			dn:       8.4118841988678429e-018,
			dn1:      0.20309858884628909,
			dn2:      0.71080515929391042,
			g:        0.0000000000000000,
			tau:      0.0000000000000000,
			nFail:    2,
			iter:     32,
			sigma:    0.21120653295038691,
			nDiv:     642,
			zOut:     []float64{0.2269770980149768e+01, 0.2269770980379126e+01, 0.2293573303077261e-09, 0.9833566024906726e-10, 0.9731514910392876e+00, 0.9736626287363720e+00, 0.5111377954200868e-03, 0.7778528033461282e-03, 0.1481726086511552e+01, 0.1482415971695234e+01, 0.1467737987028945e-02, 0.1706410574351102e-02, 0.1723475383238021e+01, 0.1745500102102777e+01, 0.2373112943910766e-01, 0.2150819994097728e-01, 0.1581996562336162e+01, 0.1673484316042377e+01, 0.1129959536471923e+00, 0.5676676764251993e-01, 0.8407229839290754e+00, 0.8027291772735718e+00, 0.1877296098701645e-01, 0.3124068479314146e-01, 0.1335847297547013e+01, 0.1657986407464839e+01, 0.3533797947109679e+00, 0.2697830536296953e+00, 0.1265767433727330e+01, 0.1069422512121146e+01, 0.7343813202351114e-01, 0.5387701498318342e-01, 0.7845691484971122e+00, 0.1204398523394852e+01, 0.4737063898809231e+00, 0.5430210247136315e+00, 0.1380630986425702e+01, 0.9901761195359265e+00, 0.1525661578238559e+00, 0.9868601781741700e-01, 0.6404863278245572e+00, 0.6682044860823195e+00, 0.1264041760751794e+00, 0.1209927124430272e+00, 0.6395981188913744e+00, 0.5363094467374639e+00, 0.1770404028911661e-01, 0.2792585709699239e-01, 0.8459594942610007e+00, 0.8655706211740396e+00, 0.4753698401003136e-01, 0.3903729781186600e-01, 0.7108052565733343e+00, 0.6717679865571495e+00, 0.2779568135873871e-07, 0.8403591378512072e-08, 0.2030985888538852e+00, 0.2030985804502939e+00, 0.3146138162949754e-27, 0.1303059324279677e-43, 0.8411884198867843e-17, 0.8411884198867843e-17, 0.2293573303077261e-09, 0.9833566024906726e-10, 0.1093987140067686e+00, 0.2696165428113804e-17, 0.5825676764620370e-08, 0.2638661645752538e-08, 0.9751785856405315e-01, 0.6617412588681544e-21, 0.2809066689524106e-07, 0.6165545233233256e-07, 0.2818174710670554e-01, 0.1396210563637443e-18, 0.2854399814229969e-05, 0.1344352644992036e-05, 0.1697846193036144e-01, 0.6162975822039155e-31, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			i0Out:    1,
			n0Out:    16,
			ppOut:    0,
			dminOut:  8.4118841988678429e-018,
			desigOut: 1.0739530738237274e-017,
			sigmaOut: 0.21120653295038691,
			qmaxOut:  2.2697722926067549,
			nFailOut: 2,
			iterOut:  33,
			nDivOut:  659,
			ttypeOut: -2,
			dmin1Out: 0.20309858045029386,
			dmin2Out: 0.51860540644834729,
			dnOut:    8.4118841988678429e-018,
			dn1Out:   0.20309858045029386,
			dn2Out:   0.67176795876146822,
			gOut:     0.0000000000000000,
			tauOut:   0.0000000000000000,
		},
		{
			i0:       1,
			n0:       16,
			z:        []float64{0.2269770980149768e+01, 0.2269770980379126e+01, 0.2293573303077261e-09, 0.9833566024906726e-10, 0.9731514910392876e+00, 0.9736626287363720e+00, 0.5111377954200868e-03, 0.7778528033461282e-03, 0.1481726086511552e+01, 0.1482415971695234e+01, 0.1467737987028945e-02, 0.1706410574351102e-02, 0.1723475383238021e+01, 0.1745500102102777e+01, 0.2373112943910766e-01, 0.2150819994097728e-01, 0.1581996562336162e+01, 0.1673484316042377e+01, 0.1129959536471923e+00, 0.5676676764251993e-01, 0.8407229839290754e+00, 0.8027291772735718e+00, 0.1877296098701645e-01, 0.3124068479314146e-01, 0.1335847297547013e+01, 0.1657986407464839e+01, 0.3533797947109679e+00, 0.2697830536296953e+00, 0.1265767433727330e+01, 0.1069422512121146e+01, 0.7343813202351114e-01, 0.5387701498318342e-01, 0.7845691484971122e+00, 0.1204398523394852e+01, 0.4737063898809231e+00, 0.5430210247136315e+00, 0.1380630986425702e+01, 0.9901761195359265e+00, 0.1525661578238559e+00, 0.9868601781741700e-01, 0.6404863278245572e+00, 0.6682044860823195e+00, 0.1264041760751794e+00, 0.1209927124430272e+00, 0.6395981188913744e+00, 0.5363094467374639e+00, 0.1770404028911661e-01, 0.2792585709699239e-01, 0.8459594942610007e+00, 0.8655706211740396e+00, 0.4753698401003136e-01, 0.3903729781186600e-01, 0.7108052565733343e+00, 0.6717679865571495e+00, 0.2779568135873871e-07, 0.8403591378512072e-08, 0.2030985888538852e+00, 0.2030985804502939e+00, 0.3146138162949754e-27, 0.1303059324279677e-43, 0.8411884198867843e-17, 0.8411884198867843e-17, 0.2293573303077261e-09, 0.9833566024906726e-10, 0.1093987140067686e+00, 0.2696165428113804e-17, 0.5825676764620370e-08, 0.2638661645752538e-08, 0.9751785856405315e-01, 0.6617412588681544e-21, 0.2809066689524106e-07, 0.6165545233233256e-07, 0.2818174710670554e-01, 0.1396210563637443e-18, 0.2854399814229969e-05, 0.1344352644992036e-05, 0.1697846193036144e-01, 0.6162975822039155e-31, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			pp:       1,
			dmin:     8.4118841988678429e-018,
			desig:    1.0739530738237274e-017,
			qmax:     2.2697722926067549,
			ttype:    -2,
			dmin1:    0.20309858045029386,
			dmin2:    0.51860540644834729,
			dn:       8.4118841988678429e-018,
			dn1:      0.20309858045029386,
			dn2:      0.67176795876146822,
			g:        0.0000000000000000,
			tau:      0.0000000000000000,
			nFail:    2,
			iter:     33,
			sigma:    0.21120653295038691,
			nDiv:     659,
			zOut:     []float64{0.2168221690252314e+01, 0.2269770980379126e+01, 0.4415865678637858e-10, 0.9833566024906726e-10, 0.8728911912704126e+00, 0.9736626287363720e+00, 0.1321013925721922e-02, 0.7778528033461282e-03, 0.1381252078118717e+01, 0.1482415971695234e+01, 0.2156405683614187e-02, 0.1706410574351102e-02, 0.1663302606134993e+01, 0.1745500102102777e+01, 0.2163985983955575e-01, 0.2150819994097728e-01, 0.1607061933620194e+01, 0.1673484316042377e+01, 0.2835506195054301e-01, 0.5676676764251993e-01, 0.7040655098910235e+00, 0.8027291772735718e+00, 0.7356791380810462e-01, 0.3124068479314146e-01, 0.1752652257061283e+01, 0.1657986407464839e+01, 0.1646145547572216e+00, 0.2697830536296953e+00, 0.8571356821219610e+00, 0.1069422512121146e+01, 0.7570493055431493e-01, 0.5387701498318342e-01, 0.1570165327329021e+01, 0.1204398523394852e+01, 0.3424393862982660e+00, 0.5430210247136315e+00, 0.6448734608299306e+00, 0.9901761195359265e+00, 0.1022564019526126e+00, 0.9868601781741700e-01, 0.5853915063475871e+00, 0.6682044860823195e+00, 0.1108480973262629e+00, 0.1209927124430272e+00, 0.3518379162830466e+00, 0.5363094467374639e+00, 0.6870152520689508e-01, 0.2792585709699239e-01, 0.7343571035538636e+00, 0.8655706211740396e+00, 0.3571015630515456e-01, 0.3903729781186600e-01, 0.5345085484304394e+00, 0.6717679865571495e+00, 0.3193134112956561e-08, 0.8403591378512072e-08, 0.1015492870320128e+00, 0.2030985804502939e+00, 0.4415865678637858e-10, 0.1303059324279677e-43, 0.2112065329503869e+00, 0.8411884198867843e-17, 0.2293573303077261e-09, 0.9833566024906726e-10, 0.1093987140067686e+00, 0.2696165428113804e-17, 0.5825676764620370e-08, 0.2638661645752538e-08, 0.9751785856405315e-01, 0.6617412588681544e-21, 0.2809066689524106e-07, 0.6165545233233256e-07, 0.2818174710670554e-01, 0.1396210563637443e-18, 0.2854399814229969e-05, 0.1344352644992036e-05, 0.1697846193036144e-01, 0.6162975822039155e-31, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			i0Out:    1,
			n0Out:    15,
			ppOut:    1,
			dminOut:  0.10154928703201281,
			desigOut: 0.0000000000000000,
			sigmaOut: 0.31275582317553385,
			qmaxOut:  2.2697722926067549,
			nFailOut: 2,
			iterOut:  34,
			nDivOut:  675,
			ttypeOut: -9,
			dmin1Out: 0.32391205918605420,
			dmin2Out: 0.32391205918605420,
			dnOut:    0.10154928703201281,
			dn1Out:   0.53450854002684800,
			dn2Out:   0.69531980574199759,
			gOut:     0.0000000000000000,
			tauOut:   0.10154929022514693,
		},
		{
			i0:       1,
			n0:       15,
			z:        []float64{0.2168221690252314e+01, 0.2269770980379126e+01, 0.4415865678637858e-10, 0.9833566024906726e-10, 0.8728911912704126e+00, 0.9736626287363720e+00, 0.1321013925721922e-02, 0.7778528033461282e-03, 0.1381252078118717e+01, 0.1482415971695234e+01, 0.2156405683614187e-02, 0.1706410574351102e-02, 0.1663302606134993e+01, 0.1745500102102777e+01, 0.2163985983955575e-01, 0.2150819994097728e-01, 0.1607061933620194e+01, 0.1673484316042377e+01, 0.2835506195054301e-01, 0.5676676764251993e-01, 0.7040655098910235e+00, 0.8027291772735718e+00, 0.7356791380810462e-01, 0.3124068479314146e-01, 0.1752652257061283e+01, 0.1657986407464839e+01, 0.1646145547572216e+00, 0.2697830536296953e+00, 0.8571356821219610e+00, 0.1069422512121146e+01, 0.7570493055431493e-01, 0.5387701498318342e-01, 0.1570165327329021e+01, 0.1204398523394852e+01, 0.3424393862982660e+00, 0.5430210247136315e+00, 0.6448734608299306e+00, 0.9901761195359265e+00, 0.1022564019526126e+00, 0.9868601781741700e-01, 0.5853915063475871e+00, 0.6682044860823195e+00, 0.1108480973262629e+00, 0.1209927124430272e+00, 0.3518379162830466e+00, 0.5363094467374639e+00, 0.6870152520689508e-01, 0.2792585709699239e-01, 0.7343571035538636e+00, 0.8655706211740396e+00, 0.3571015630515456e-01, 0.3903729781186600e-01, 0.5345085484304394e+00, 0.6717679865571495e+00, 0.3193134112956561e-08, 0.8403591378512072e-08, 0.1015492870320128e+00, 0.2030985804502939e+00, 0.4415865678637858e-10, 0.9833566024906726e-10, 0.2112065329503869e+00, 0.8411884198867843e-17, 0.2293573303077261e-09, 0.9833566024906726e-10, 0.1093987140067686e+00, 0.2696165428113804e-17, 0.5825676764620370e-08, 0.2638661645752538e-08, 0.9751785856405315e-01, 0.6617412588681544e-21, 0.2809066689524106e-07, 0.6165545233233256e-07, 0.2818174710670554e-01, 0.1396210563637443e-18, 0.2854399814229969e-05, 0.1344352644992036e-05, 0.1697846193036144e-01, 0.6162975822039155e-31, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			pp:       0,
			dmin:     0.10154928703201281,
			desig:    0.0000000000000000,
			qmax:     2.1682216902523144,
			ttype:    -9,
			dmin1:    0.32391205918605420,
			dmin2:    0.32391205918605420,
			dn:       0.10154928703201281,
			dn1:      0.53450854002684800,
			dn2:      0.69531980574199759,
			g:        0.0000000000000000,
			tau:      0.10154929022514693,
			nFail:    2,
			iter:     34,
			sigma:    0.31275582317553385,
			nDiv:     675,
			zOut:     []float64{0.2168221690252314e+01, 0.2066680684115085e+01, 0.4415865678637858e-10, 0.1865101988102620e-10, 0.8728911912704126e+00, 0.7726711989960955e+00, 0.1321013925721922e-02, 0.2361487308570419e-02, 0.1381252078118717e+01, 0.1279505990312372e+01, 0.2156405683614187e-02, 0.2803234389363145e-02, 0.1663302606134993e+01, 0.1580598225403798e+01, 0.2163985983955575e-01, 0.2200217261925751e-01, 0.1607061933620194e+01, 0.1511873816770092e+01, 0.2835506195054301e-01, 0.1320468740761088e-01, 0.7040655098910235e+00, 0.6628877301101292e+00, 0.7356791380810462e-01, 0.1945110224949301e+00, 0.1752652257061283e+01, 0.1621214783142186e+01, 0.1646145547572216e+00, 0.8703165684534678e-01, 0.8571356821219610e+00, 0.7442679496495411e+00, 0.7570493055431493e-01, 0.1597129865933493e+00, 0.1570165327329021e+01, 0.1651350720852550e+01, 0.3424393862982660e+00, 0.1337269360034139e+00, 0.6448734608299306e+00, 0.5118619205977412e+00, 0.1022564019526126e+00, 0.1169456581236225e+00, 0.5853915063475871e+00, 0.4777529393688394e+00, 0.1108480973262629e+00, 0.8163333047984263e-01, 0.3518379162830466e+00, 0.2373651048287110e+00, 0.6870152520689508e-01, 0.2125478936639627e+00, 0.7343571035538636e+00, 0.4559783600136673e+00, 0.3571015630515456e-01, 0.4186028435717903e-01, 0.5345085484304394e+00, 0.3911072610850064e+00, 0.3193134112956561e-08, 0.8290832843879624e-09, 0.1015492870320128e+00, 0.8280021541434701e-05, 0.4415865678637858e-10, 0.1865101988102620e-10, 0.2112065329503869e+00, 0.8411884198867843e-17, 0.2293573303077261e-09, 0.9833566024906726e-10, 0.1093987140067686e+00, 0.2696165428113804e-17, 0.5825676764620370e-08, 0.2638661645752538e-08, 0.9751785856405315e-01, 0.6617412588681544e-21, 0.2809066689524106e-07, 0.6165545233233256e-07, 0.2818174710670554e-01, 0.1396210563637443e-18, 0.2854399814229969e-05, 0.1344352644992036e-05, 0.1697846193036144e-01, 0.6162975822039155e-31, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			i0Out:    1,
			n0Out:    15,
			ppOut:    0,
			dminOut:  8.2800215414347011e-006,
			desigOut: -1.3877787807814457e-017,
			sigmaOut: 0.41429682935692197,
			qmaxOut:  2.1682216902523144,
			nFailOut: 2,
			iterOut:  35,
			nDivOut:  691,
			ttypeOut: -4,
			dmin1Out: 0.16866357962181588,
			dmin2Out: 0.16866357962181588,
			dnOut:    8.2800215414347011e-006,
			dn1Out:   0.39110725789187228,
			dn2Out:   0.42026820370851276,
			gOut:     0.0000000000000000,
			tauOut:   0.10154100618138810,
		},
		{
			i0:       1,
			n0:       11,
			z:        []float64{0.1565539912114361e+01, 0.1565539887559990e+01, 0.1019378594629470e-16, 0.7719264500395369e-17, 0.1185508165627851e+01, 0.1209595044804978e+01, 0.2408690373149840e-01, 0.1773879145177052e-01, 0.8908058287637680e+00, 0.9017845275302323e+00, 0.2871751477260568e-01, 0.4376453959373496e-01, 0.1374289696466428e+01, 0.1388278114489740e+01, 0.5775298217141787e-01, 0.3977262621431848e-01, 0.9560626040960629e+00, 0.9795790109656241e+00, 0.6328905763825028e-01, 0.5927283946360358e-01, 0.9174165586530958e+00, 0.1200347187817445e+01, 0.3422034931823232e+00, 0.1225138368952072e+00, 0.4297417838091302e+00, 0.3072280530227507e+00, 0.1306631986684747e-06, 0.1577236754259593e-06, 0.3708552845063717e+00, 0.3753559465594062e+00, 0.4500844331080801e-02, 0.3315910790466669e-02, 0.2765362989488873e+00, 0.2810497954194954e+00, 0.7829431815445537e-02, 0.5406547261581845e-02, 0.1940765355151052e+00, 0.1886700618399915e+00, 0.9814083907923291e-07, 0.1277249852674422e-13, 0.2455438642568072e-07, 0.2764930279233778e-14, 0.1019378594629470e-16, 0.7719264500395369e-17, 0.8241395430971566e+00, 0.2245698748385924e-16, 0.1792267857826344e-16, 0.2548071704976161e-16, 0.7318275341991307e+00, 0.4959852501050381e-23, 0.8676862906242004e-16, 0.1414000300319855e-15, 0.5186198053161721e+00, 0.1503295986001297e-14, 0.3452895805257589e-14, 0.7981257539768321e-14, 0.4143051093784424e+00, 0.1303656847202082e-15, 0.9855809897129084e-12, 0.2614986238618434e-11, 0.2112065329503869e+00, 0.8411884198867843e-17, 0.2293573303077261e-09, 0.9833566024906726e-10, 0.1093987140067686e+00, 0.2696165428113804e-17, 0.5825676764620370e-08, 0.2638661645752538e-08, 0.9751785856405315e-01, 0.6617412588681544e-21, 0.2809066689524106e-07, 0.6165545233233256e-07, 0.2818174710670554e-01, 0.1396210563637443e-18, 0.2854399814229969e-05, 0.1344352644992036e-05, 0.1697846193036144e-01, 0.6162975822039155e-31, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			pp:       1,
			dmin:     2.7649302792337775e-015,
			desig:    4.8519526217661568e-017,
			qmax:     2.1682216902523144,
			ttype:    -2,
			dmin1:    0.18866996369915248,
			dmin2:    0.27322036360404983,
			dn:       2.7649302792337775e-015,
			dn1:      0.18866996369915248,
			dn2:      0.27322036360404983,
			g:        0.0000000000000000,
			tau:      2.4554370888251911e-008,
			nFail:    2,
			iter:     57,
			sigma:    0.91543762594185796,
			nDiv:     1001,
			zOut:     []float64{0.1565539887559988e+01, 0.1565539887559990e+01, 0.5964194309842799e-17, 0.7719264500395369e-17, 0.1227333836256746e+01, 0.1209595044804978e+01, 0.1303359134714337e-01, 0.1773879145177052e-01, 0.9325154757768211e+00, 0.9017845275302323e+00, 0.6515425651042277e-01, 0.4376453959373496e-01, 0.1362896484193633e+01, 0.1388278114489740e+01, 0.2858649229958118e-01, 0.3977262621431848e-01, 0.1010265358129644e+01, 0.9795790109656241e+00, 0.7042504782685149e-01, 0.5927283946360358e-01, 0.1252435976885798e+01, 0.1200347187817445e+01, 0.3005318297487194e-01, 0.1225138368952072e+00, 0.2771750277715515e+00, 0.3072280530227507e+00, 0.2135925446109612e-06, 0.1577236754259593e-06, 0.3786716437573254e+00, 0.3753559465594062e+00, 0.2461066374135985e-02, 0.3315910790466669e-02, 0.2839952763069384e+00, 0.2810497954194954e+00, 0.3591797791316158e-02, 0.5406547261581845e-02, 0.1850782640486854e+00, 0.1886700618399915e+00, 0.1908115364037247e-27, 0.1277249852674422e-13, 0.3470987982972452e-28, 0.2764930279233778e-14, 0.5964194309842799e-17, 0.7719264500395369e-17, 0.8241395430971566e+00, 0.2245698748385924e-16, 0.1792267857826344e-16, 0.2548071704976161e-16, 0.7318275341991307e+00, 0.4959852501050381e-23, 0.8676862906242004e-16, 0.1414000300319855e-15, 0.5186198053161721e+00, 0.1503295986001297e-14, 0.3452895805257589e-14, 0.7981257539768321e-14, 0.4143051093784424e+00, 0.1303656847202082e-15, 0.9855809897129084e-12, 0.2614986238618434e-11, 0.2112065329503869e+00, 0.8411884198867843e-17, 0.2293573303077261e-09, 0.9833566024906726e-10, 0.1093987140067686e+00, 0.2696165428113804e-17, 0.5825676764620370e-08, 0.2638661645752538e-08, 0.9751785856405315e-01, 0.6617412588681544e-21, 0.2809066689524106e-07, 0.6165545233233256e-07, 0.2818174710670554e-01, 0.1396210563637443e-18, 0.2854399814229969e-05, 0.1344352644992036e-05, 0.1697846193036144e-01, 0.6162975822039155e-31, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			i0Out:    1,
			n0Out:    11,
			ppOut:    1,
			dminOut:  3.4709879829724519e-029,
			desigOut: 3.7892243888322520e-017,
			sigmaOut: 0.91543762594186073,
			qmaxOut:  2.1682216902523144,
			nFailOut: 2,
			iterOut:  58,
			nDivOut:  1013,
			ttypeOut: -2,
			dmin1Out: 0.18507826404867261,
			dmin2Out: 0.27717487004787600,
			dnOut:    3.4709879829724519e-029,
			dn1Out:   0.18507826404867261,
			dn2Out:   0.27858872904535659,
			gOut:     0.0000000000000000,
			tauOut:   2.7649302792335523e-015,
		},
		{
			i0:       1,
			n0:       11,
			z:        []float64{0.1565539887559988e+01, 0.1565539887559990e+01, 0.5964194309842799e-17, 0.7719264500395369e-17, 0.1227333836256746e+01, 0.1209595044804978e+01, 0.1303359134714337e-01, 0.1773879145177052e-01, 0.9325154757768211e+00, 0.9017845275302323e+00, 0.6515425651042277e-01, 0.4376453959373496e-01, 0.1362896484193633e+01, 0.1388278114489740e+01, 0.2858649229958118e-01, 0.3977262621431848e-01, 0.1010265358129644e+01, 0.9795790109656241e+00, 0.7042504782685149e-01, 0.5927283946360358e-01, 0.1252435976885798e+01, 0.1200347187817445e+01, 0.3005318297487194e-01, 0.1225138368952072e+00, 0.2771750277715515e+00, 0.3072280530227507e+00, 0.2135925446109612e-06, 0.1577236754259593e-06, 0.3786716437573254e+00, 0.3753559465594062e+00, 0.2461066374135985e-02, 0.3315910790466669e-02, 0.2839952763069384e+00, 0.2810497954194954e+00, 0.3591797791316158e-02, 0.5406547261581845e-02, 0.1850782640486854e+00, 0.1886700618399915e+00, 0.1908115364037247e-27, 0.1277249852674422e-13, 0.3470987982972452e-28, 0.2764930279233778e-14, 0.5964194309842799e-17, 0.7719264500395369e-17, 0.8241395430971566e+00, 0.2245698748385924e-16, 0.1792267857826344e-16, 0.2548071704976161e-16, 0.7318275341991307e+00, 0.4959852501050381e-23, 0.8676862906242004e-16, 0.1414000300319855e-15, 0.5186198053161721e+00, 0.1503295986001297e-14, 0.3452895805257589e-14, 0.7981257539768321e-14, 0.4143051093784424e+00, 0.1303656847202082e-15, 0.9855809897129084e-12, 0.2614986238618434e-11, 0.2112065329503869e+00, 0.8411884198867843e-17, 0.2293573303077261e-09, 0.9833566024906726e-10, 0.1093987140067686e+00, 0.2696165428113804e-17, 0.5825676764620370e-08, 0.2638661645752538e-08, 0.9751785856405315e-01, 0.6617412588681544e-21, 0.2809066689524106e-07, 0.6165545233233256e-07, 0.2818174710670554e-01, 0.1396210563637443e-18, 0.2854399814229969e-05, 0.1344352644992036e-05, 0.1697846193036144e-01, 0.6162975822039155e-31, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			pp:       0,
			dmin:     3.4709879829724519e-029,
			desig:    3.7892243888322520e-017,
			qmax:     2.1682216902523144,
			ttype:    -2,
			dmin1:    0.18507826404867261,
			dmin2:    0.27717487004787600,
			dn:       3.4709879829724519e-029,
			dn1:      0.18507826404867261,
			dn2:      0.27858872904535659,
			g:        0.0000000000000000,
			tau:      2.7649302792335523e-015,
			nFail:    2,
			iter:     58,
			sigma:    0.91543762594186073,
			nDiv:     1013,
			zOut:     []float64{0.1565539887559988e+01, 0.1473000755535651e+01, 0.5964194309842799e-17, 0.4969486576955697e-17, 0.1227333836256746e+01, 0.1147828295579553e+01, 0.1303359134714337e-01, 0.1058871408116432e-01, 0.9325154757768211e+00, 0.8945418861817434e+00, 0.6515425651042277e-01, 0.9926701980086396e-01, 0.1362896484193633e+01, 0.1199676824668014e+01, 0.2858649229958118e-01, 0.2407310226126826e-01, 0.1010265358129644e+01, 0.9640781716708908e+00, 0.7042504782685149e-01, 0.9148932748822959e-01, 0.1252435976885798e+01, 0.1098460700348104e+01, 0.3005318297487194e-01, 0.7583331677723075e-02, 0.2771750277715515e+00, 0.1770527776620367e+00, 0.2135925446109612e-06, 0.4568210735249311e-06, 0.3786716437573254e+00, 0.2885931212860515e+00, 0.2461066374135985e-02, 0.2421856840585221e-02, 0.2839952763069384e+00, 0.1926260852333330e+00, 0.3591797791316158e-02, 0.3451057520197492e-02, 0.1850782640486854e+00, 0.8908807450415157e-01, 0.1908115364037247e-27, 0.4969486576955697e-17, 0.9154376259418607e+00, 0.2764930279233778e-14, 0.5964194309842799e-17, 0.7719264500395369e-17, 0.8241395430971566e+00, 0.2245698748385924e-16, 0.1792267857826344e-16, 0.2548071704976161e-16, 0.7318275341991307e+00, 0.4959852501050381e-23, 0.8676862906242004e-16, 0.1414000300319855e-15, 0.5186198053161721e+00, 0.1503295986001297e-14, 0.3452895805257589e-14, 0.7981257539768321e-14, 0.4143051093784424e+00, 0.1303656847202082e-15, 0.9855809897129084e-12, 0.2614986238618434e-11, 0.2112065329503869e+00, 0.8411884198867843e-17, 0.2293573303077261e-09, 0.9833566024906726e-10, 0.1093987140067686e+00, 0.2696165428113804e-17, 0.5825676764620370e-08, 0.2638661645752538e-08, 0.9751785856405315e-01, 0.6617412588681544e-21, 0.2809066689524106e-07, 0.6165545233233256e-07, 0.2818174710670554e-01, 0.1396210563637443e-18, 0.2854399814229969e-05, 0.1344352644992036e-05, 0.1697846193036144e-01, 0.6162975822039155e-31, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			i0Out:    1,
			n0Out:    10,
			ppOut:    0,
			dminOut:  8.9088074504151571e-002,
			desigOut: -4.1633363423443370e-017,
			sigmaOut: 1.0079767579661971,
			qmaxOut:  2.1682216902523144,
			nFailOut: 2,
			iterOut:  59,
			nDivOut:  1024,
			ttypeOut: -9,
			dmin1Out: 0.17705256406949207,
			dmin2Out: 0.17705256406949207,
			dnOut:    8.9088074504151571e-002,
			dn1Out:   0.18903428744201686,
			dn2Out:   0.28613205491191551,
			gOut:     0.0000000000000000,
			tauOut:   9.2539132024336307e-002,
		},
		{
			i0:       1,
			n0:       10,
			z:        []float64{0.1565539887559988e+01, 0.1473000755535651e+01, 0.5964194309842799e-17, 0.4969486576955697e-17, 0.1227333836256746e+01, 0.1147828295579553e+01, 0.1303359134714337e-01, 0.1058871408116432e-01, 0.9325154757768211e+00, 0.8945418861817434e+00, 0.6515425651042277e-01, 0.9926701980086396e-01, 0.1362896484193633e+01, 0.1199676824668014e+01, 0.2858649229958118e-01, 0.2407310226126826e-01, 0.1010265358129644e+01, 0.9640781716708908e+00, 0.7042504782685149e-01, 0.9148932748822959e-01, 0.1252435976885798e+01, 0.1098460700348104e+01, 0.3005318297487194e-01, 0.7583331677723075e-02, 0.2771750277715515e+00, 0.1770527776620367e+00, 0.2135925446109612e-06, 0.4568210735249311e-06, 0.3786716437573254e+00, 0.2885931212860515e+00, 0.2461066374135985e-02, 0.2421856840585221e-02, 0.2839952763069384e+00, 0.1926260852333330e+00, 0.3591797791316158e-02, 0.3451057520197492e-02, 0.1850782640486854e+00, 0.8908807450415157e-01, 0.1908115364037247e-27, 0.4969486576955697e-17, 0.9154376259418607e+00, 0.2764930279233778e-14, 0.5964194309842799e-17, 0.7719264500395369e-17, 0.8241395430971566e+00, 0.2245698748385924e-16, 0.1792267857826344e-16, 0.2548071704976161e-16, 0.7318275341991307e+00, 0.4959852501050381e-23, 0.8676862906242004e-16, 0.1414000300319855e-15, 0.5186198053161721e+00, 0.1503295986001297e-14, 0.3452895805257589e-14, 0.7981257539768321e-14, 0.4143051093784424e+00, 0.1303656847202082e-15, 0.9855809897129084e-12, 0.2614986238618434e-11, 0.2112065329503869e+00, 0.8411884198867843e-17, 0.2293573303077261e-09, 0.9833566024906726e-10, 0.1093987140067686e+00, 0.2696165428113804e-17, 0.5825676764620370e-08, 0.2638661645752538e-08, 0.9751785856405315e-01, 0.6617412588681544e-21, 0.2809066689524106e-07, 0.6165545233233256e-07, 0.2818174710670554e-01, 0.1396210563637443e-18, 0.2854399814229969e-05, 0.1344352644992036e-05, 0.1697846193036144e-01, 0.6162975822039155e-31, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			pp:       1,
			dmin:     8.9088074504151571e-002,
			desig:    -4.1633363423443370e-017,
			qmax:     2.1682216902523144,
			ttype:    -9,
			dmin1:    0.17705256406949207,
			dmin2:    0.17705256406949207,
			dn:       8.9088074504151571e-002,
			dn1:      0.18903428744201686,
			dn2:      0.28613205491191551,
			g:        0.0000000000000000,
			tau:      9.2539132024336307e-002,
			nFail:    2,
			iter:     59,
			sigma:    1.0079767579661971,
			nDiv:     1024,
			zOut:     []float64{0.1397612833703614e+01, 0.1473000755535651e+01, 0.4081328655531061e-17, 0.4969486576955697e-17, 0.1083029087828680e+01, 0.1147828295579553e+01, 0.8745885380967966e-02, 0.1058871408116432e-01, 0.9096750987696016e+00, 0.8945418861817434e+00, 0.1309130515609722e+00, 0.9926701980086396e-01, 0.1017448953536272e+01, 0.1199676824668014e+01, 0.2281033592282574e-01, 0.2407310226126826e-01, 0.9573692414042568e+00, 0.9640781716708908e+00, 0.1049724875218357e+00, 0.9148932748822959e-01, 0.9256836226719531e+00, 0.1098460700348104e+01, 0.1450441494900679e-02, 0.7583331677723075e-02, 0.1002148711561719e+00, 0.1770527776620367e+00, 0.1315527505616969e-05, 0.4568210735249311e-06, 0.2156257407670935e+00, 0.2885931212860515e+00, 0.2163530200698068e-02, 0.2421856840585221e-02, 0.1185256907207948e+00, 0.1926260852333330e+00, 0.2593936112987583e-02, 0.3451057520197492e-02, 0.1110621655912630e-01, 0.8908807450415157e-01, 0.4081328655531061e-17, 0.4969486576955697e-17, 0.9154376259418607e+00, 0.2764930279233778e-14, 0.5964194309842799e-17, 0.7719264500395369e-17, 0.8241395430971566e+00, 0.2245698748385924e-16, 0.1792267857826344e-16, 0.2548071704976161e-16, 0.7318275341991307e+00, 0.4959852501050381e-23, 0.8676862906242004e-16, 0.1414000300319855e-15, 0.5186198053161721e+00, 0.1503295986001297e-14, 0.3452895805257589e-14, 0.7981257539768321e-14, 0.4143051093784424e+00, 0.1303656847202082e-15, 0.9855809897129084e-12, 0.2614986238618434e-11, 0.2112065329503869e+00, 0.8411884198867843e-17, 0.2293573303077261e-09, 0.9833566024906726e-10, 0.1093987140067686e+00, 0.2696165428113804e-17, 0.5825676764620370e-08, 0.2638661645752538e-08, 0.9751785856405315e-01, 0.6617412588681544e-21, 0.2809066689524106e-07, 0.6165545233233256e-07, 0.2818174710670554e-01, 0.1396210563637443e-18, 0.2854399814229969e-05, 0.1344352644992036e-05, 0.1697846193036144e-01, 0.6162975822039155e-31, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			i0Out:    1,
			n0Out:    10,
			ppOut:    1,
			dminOut:  1.1106216559126303e-002,
			desigOut: -4.1633363423443370e-017,
			sigmaOut: 1.0833646797982348,
			qmaxOut:  2.1682216902523144,
			nFailOut: 2,
			iterOut:  60,
			nDivOut:  1035,
			ttypeOut: -4,
			dmin1Out: 0.10021441433509834,
			dmin2Out: 0.10021441433509834,
			dnOut:    1.1106216559126303e-002,
			dn1Out:   0.11507463320059727,
			dn2Out:   0.21320388392650824,
			gOut:     0.0000000000000000,
			tauOut:   7.5387921832037685e-002,
		},
		{
			i0:       1,
			n0:       10,
			z:        []float64{0.1397612833703614e+01, 0.1473000755535651e+01, 0.4081328655531061e-17, 0.4969486576955697e-17, 0.1083029087828680e+01, 0.1147828295579553e+01, 0.8745885380967966e-02, 0.1058871408116432e-01, 0.9096750987696016e+00, 0.8945418861817434e+00, 0.1309130515609722e+00, 0.9926701980086396e-01, 0.1017448953536272e+01, 0.1199676824668014e+01, 0.2281033592282574e-01, 0.2407310226126826e-01, 0.9573692414042568e+00, 0.9640781716708908e+00, 0.1049724875218357e+00, 0.9148932748822959e-01, 0.9256836226719531e+00, 0.1098460700348104e+01, 0.1450441494900679e-02, 0.7583331677723075e-02, 0.1002148711561719e+00, 0.1770527776620367e+00, 0.1315527505616969e-05, 0.4568210735249311e-06, 0.2156257407670935e+00, 0.2885931212860515e+00, 0.2163530200698068e-02, 0.2421856840585221e-02, 0.1185256907207948e+00, 0.1926260852333330e+00, 0.2593936112987583e-02, 0.3451057520197492e-02, 0.1110621655912630e-01, 0.8908807450415157e-01, 0.4081328655531061e-17, 0.4969486576955697e-17, 0.9154376259418607e+00, 0.2764930279233778e-14, 0.5964194309842799e-17, 0.7719264500395369e-17, 0.8241395430971566e+00, 0.2245698748385924e-16, 0.1792267857826344e-16, 0.2548071704976161e-16, 0.7318275341991307e+00, 0.4959852501050381e-23, 0.8676862906242004e-16, 0.1414000300319855e-15, 0.5186198053161721e+00, 0.1503295986001297e-14, 0.3452895805257589e-14, 0.7981257539768321e-14, 0.4143051093784424e+00, 0.1303656847202082e-15, 0.9855809897129084e-12, 0.2614986238618434e-11, 0.2112065329503869e+00, 0.8411884198867843e-17, 0.2293573303077261e-09, 0.9833566024906726e-10, 0.1093987140067686e+00, 0.2696165428113804e-17, 0.5825676764620370e-08, 0.2638661645752538e-08, 0.9751785856405315e-01, 0.6617412588681544e-21, 0.2809066689524106e-07, 0.6165545233233256e-07, 0.2818174710670554e-01, 0.1396210563637443e-18, 0.2854399814229969e-05, 0.1344352644992036e-05, 0.1697846193036144e-01, 0.6162975822039155e-31, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			pp:       0,
			dmin:     1.1106216559126303e-002,
			desig:    -4.1633363423443370e-017,
			qmax:     2.1682216902523144,
			ttype:    -4,
			dmin1:    0.10021441433509834,
			dmin2:    0.10021441433509834,
			dn:       1.1106216559126303e-002,
			dn1:      0.11507463320059727,
			dn2:      0.21320388392650824,
			g:        0.0000000000000000,
			tau:      7.5387921832037685e-002,
			nFail:    2,
			iter:     60,
			sigma:    1.0833646797982348,
			nDiv:     1035,
			zOut:     []float64{0.1397612833703614e+01, 0.1388412169967495e+01, 0.4081328655531061e-17, 0.3183635051997810e-17, 0.1083029087828680e+01, 0.1082574309473529e+01, 0.8745885380967966e-02, 0.7349069784991225e-02, 0.9096750987696016e+00, 0.1024038416809464e+01, 0.1309130515609722e+00, 0.1300706547025319e+00, 0.1017448953536272e+01, 0.9009879710204475e+00, 0.2281033592282574e-01, 0.2423774201322422e-01, 0.9573692414042568e+00, 0.1028903323176749e+01, 0.1049724875218357e+00, 0.9444163542020824e-01, 0.9256836226719531e+00, 0.8234917650105269e+00, 0.1450441494900679e-02, 0.1765115496075164e-03, 0.1002148711561719e+00, 0.9083901139795122e-01, 0.1315527505616969e-05, 0.3122684720284652e-05, 0.2156257407670935e+00, 0.2085854845469525e+00, 0.2163530200698068e-02, 0.1229394806594584e-02, 0.1185256907207948e+00, 0.1106895682910690e+00, 0.2593936112987583e-02, 0.2602667681892373e-03, 0.1110621655912630e-01, 0.1645286054818337e-02, 0.4081328655531061e-17, 0.3183635051997810e-17, 0.9154376259418607e+00, 0.2764930279233778e-14, 0.5964194309842799e-17, 0.7719264500395369e-17, 0.8241395430971566e+00, 0.2245698748385924e-16, 0.1792267857826344e-16, 0.2548071704976161e-16, 0.7318275341991307e+00, 0.4959852501050381e-23, 0.8676862906242004e-16, 0.1414000300319855e-15, 0.5186198053161721e+00, 0.1503295986001297e-14, 0.3452895805257589e-14, 0.7981257539768321e-14, 0.4143051093784424e+00, 0.1303656847202082e-15, 0.9855809897129084e-12, 0.2614986238618434e-11, 0.2112065329503869e+00, 0.8411884198867843e-17, 0.2293573303077261e-09, 0.9833566024906726e-10, 0.1093987140067686e+00, 0.2696165428113804e-17, 0.5825676764620370e-08, 0.2638661645752538e-08, 0.9751785856405315e-01, 0.6617412588681544e-21, 0.2809066689524106e-07, 0.6165545233233256e-07, 0.2818174710670554e-01, 0.1396210563637443e-18, 0.2854399814229969e-05, 0.1344352644992036e-05, 0.1697846193036144e-01, 0.6162975822039155e-31, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			i0Out:    1,
			n0Out:    10,
			ppOut:    0,
			dminOut:  1.6452860548183366e-003,
			desigOut: 7.9797279894933126e-017,
			sigmaOut: 1.0925653435343534,
			qmaxOut:  2.1682216902523144,
			nFailOut: 2,
			iterOut:  61,
			nDivOut:  1046,
			ttypeOut: -4,
			dmin1Out: 9.0837695870445614e-002,
			dmin2Out: 9.0837695870445614e-002,
			dnOut:    1.6452860548183366e-003,
			dn1Out:   0.10809563217808144,
			dn2Out:   0.20642195434625446,
			gOut:     0.0000000000000000,
			tauOut:   9.2006637361187298e-003,
		},
		{
			i0:       1,
			n0:       10,
			z:        []float64{0.1397612833703614e+01, 0.1388412169967495e+01, 0.4081328655531061e-17, 0.3183635051997810e-17, 0.1083029087828680e+01, 0.1082574309473529e+01, 0.8745885380967966e-02, 0.7349069784991225e-02, 0.9096750987696016e+00, 0.1024038416809464e+01, 0.1309130515609722e+00, 0.1300706547025319e+00, 0.1017448953536272e+01, 0.9009879710204475e+00, 0.2281033592282574e-01, 0.2423774201322422e-01, 0.9573692414042568e+00, 0.1028903323176749e+01, 0.1049724875218357e+00, 0.9444163542020824e-01, 0.9256836226719531e+00, 0.8234917650105269e+00, 0.1450441494900679e-02, 0.1765115496075164e-03, 0.1002148711561719e+00, 0.9083901139795122e-01, 0.1315527505616969e-05, 0.3122684720284652e-05, 0.2156257407670935e+00, 0.2085854845469525e+00, 0.2163530200698068e-02, 0.1229394806594584e-02, 0.1185256907207948e+00, 0.1106895682910690e+00, 0.2593936112987583e-02, 0.2602667681892373e-03, 0.1110621655912630e-01, 0.1645286054818337e-02, 0.4081328655531061e-17, 0.3183635051997810e-17, 0.9154376259418607e+00, 0.2764930279233778e-14, 0.5964194309842799e-17, 0.7719264500395369e-17, 0.8241395430971566e+00, 0.2245698748385924e-16, 0.1792267857826344e-16, 0.2548071704976161e-16, 0.7318275341991307e+00, 0.4959852501050381e-23, 0.8676862906242004e-16, 0.1414000300319855e-15, 0.5186198053161721e+00, 0.1503295986001297e-14, 0.3452895805257589e-14, 0.7981257539768321e-14, 0.4143051093784424e+00, 0.1303656847202082e-15, 0.9855809897129084e-12, 0.2614986238618434e-11, 0.2112065329503869e+00, 0.8411884198867843e-17, 0.2293573303077261e-09, 0.9833566024906726e-10, 0.1093987140067686e+00, 0.2696165428113804e-17, 0.5825676764620370e-08, 0.2638661645752538e-08, 0.9751785856405315e-01, 0.6617412588681544e-21, 0.2809066689524106e-07, 0.6165545233233256e-07, 0.2818174710670554e-01, 0.1396210563637443e-18, 0.2854399814229969e-05, 0.1344352644992036e-05, 0.1697846193036144e-01, 0.6162975822039155e-31, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			pp:       1,
			dmin:     1.6452860548183366e-003,
			desig:    7.9797279894933126e-017,
			qmax:     2.1682216902523144,
			ttype:    -4,
			dmin1:    9.0837695870445614e-002,
			dmin2:    9.0837695870445614e-002,
			dn:       1.6452860548183366e-003,
			dn1:      0.10809563217808144,
			dn2:      0.20642195434625446,
			g:        0.0000000000000000,
			tau:      9.2006637361187298e-003,
			nFail:    2,
			iter:     61,
			sigma:    1.0925653435343534,
			nDiv:     1046,
			zOut:     []float64{0.1386852748028120e+01, 0.1388412169967495e+01, 0.2485138759635906e-17, 0.3183635051997810e-17, 0.1088363957319145e+01, 0.1082574309473529e+01, 0.6914717946174950e-02, 0.7349069784991225e-02, 0.1145634931626446e+01, 0.1024038416809464e+01, 0.1022944500333619e+00, 0.1300706547025319e+00, 0.8213718410609353e+00, 0.9009879710204475e+00, 0.3036175828902931e-01, 0.2423774201322422e-01, 0.1091423778368554e+01, 0.1028903323176749e+01, 0.7125729765473905e-01, 0.9444163542020824e-01, 0.7508515569660207e+00, 0.8234917650105269e+00, 0.2135460001102832e-04, 0.1765115496075164e-03, 0.8926135754328580e-01, 0.9083901139795122e-01, 0.7297073710223142e-05, 0.3122684720284652e-05, 0.2082481603404622e+00, 0.2085854845469525e+00, 0.6534568189162364e-03, 0.1229394806594584e-02, 0.1087369563009673e+00, 0.1106895682910690e+00, 0.3938065757966957e-05, 0.2602667681892373e-03, 0.8192604968568760e-04, 0.1645286054818337e-02, 0.2485138759635906e-17, 0.3183635051997810e-17, 0.9154376259418607e+00, 0.2764930279233778e-14, 0.5964194309842799e-17, 0.7719264500395369e-17, 0.8241395430971566e+00, 0.2245698748385924e-16, 0.1792267857826344e-16, 0.2548071704976161e-16, 0.7318275341991307e+00, 0.4959852501050381e-23, 0.8676862906242004e-16, 0.1414000300319855e-15, 0.5186198053161721e+00, 0.1503295986001297e-14, 0.3452895805257589e-14, 0.7981257539768321e-14, 0.4143051093784424e+00, 0.1303656847202082e-15, 0.9855809897129084e-12, 0.2614986238618434e-11, 0.2112065329503869e+00, 0.8411884198867843e-17, 0.2293573303077261e-09, 0.9833566024906726e-10, 0.1093987140067686e+00, 0.2696165428113804e-17, 0.5825676764620370e-08, 0.2638661645752538e-08, 0.9751785856405315e-01, 0.6617412588681544e-21, 0.2809066689524106e-07, 0.6165545233233256e-07, 0.2818174710670554e-01, 0.1396210563637443e-18, 0.2854399814229969e-05, 0.1344352644992036e-05, 0.1697846193036144e-01, 0.6162975822039155e-31, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			i0Out:    1,
			n0Out:    10,
			ppOut:    1,
			dminOut:  8.1926049685687600e-005,
			desigOut: -9.9312918999672206e-017,
			sigmaOut: 1.0941247654737283,
			qmaxOut:  2.1682216902523144,
			nFailOut: 2,
			iterOut:  62,
			nDivOut:  1057,
			ttypeOut: -4,
			dmin1Out: 8.9258234858565516e-002,
			dmin2Out: 8.9258234858565516e-002,
			dnOut:    8.1926049685687600e-005,
			dn1Out:   0.10847668953277810,
			dn2Out:   0.20701876553386761,
			gOut:     0.0000000000000000,
			tauOut:   1.5594219393746818e-003,
		},
		{
			i0:       1,
			n0:       10,
			z:        []float64{0.1386852748028120e+01, 0.1388412169967495e+01, 0.2485138759635906e-17, 0.3183635051997810e-17, 0.1088363957319145e+01, 0.1082574309473529e+01, 0.6914717946174950e-02, 0.7349069784991225e-02, 0.1145634931626446e+01, 0.1024038416809464e+01, 0.1022944500333619e+00, 0.1300706547025319e+00, 0.8213718410609353e+00, 0.9009879710204475e+00, 0.3036175828902931e-01, 0.2423774201322422e-01, 0.1091423778368554e+01, 0.1028903323176749e+01, 0.7125729765473905e-01, 0.9444163542020824e-01, 0.7508515569660207e+00, 0.8234917650105269e+00, 0.2135460001102832e-04, 0.1765115496075164e-03, 0.8926135754328580e-01, 0.9083901139795122e-01, 0.7297073710223142e-05, 0.3122684720284652e-05, 0.2082481603404622e+00, 0.2085854845469525e+00, 0.6534568189162364e-03, 0.1229394806594584e-02, 0.1087369563009673e+00, 0.1106895682910690e+00, 0.3938065757966957e-05, 0.2602667681892373e-03, 0.8192604968568760e-04, 0.1645286054818337e-02, 0.2485138759635906e-17, 0.3183635051997810e-17, 0.9154376259418607e+00, 0.2764930279233778e-14, 0.5964194309842799e-17, 0.7719264500395369e-17, 0.8241395430971566e+00, 0.2245698748385924e-16, 0.1792267857826344e-16, 0.2548071704976161e-16, 0.7318275341991307e+00, 0.4959852501050381e-23, 0.8676862906242004e-16, 0.1414000300319855e-15, 0.5186198053161721e+00, 0.1503295986001297e-14, 0.3452895805257589e-14, 0.7981257539768321e-14, 0.4143051093784424e+00, 0.1303656847202082e-15, 0.9855809897129084e-12, 0.2614986238618434e-11, 0.2112065329503869e+00, 0.8411884198867843e-17, 0.2293573303077261e-09, 0.9833566024906726e-10, 0.1093987140067686e+00, 0.2696165428113804e-17, 0.5825676764620370e-08, 0.2638661645752538e-08, 0.9751785856405315e-01, 0.6617412588681544e-21, 0.2809066689524106e-07, 0.6165545233233256e-07, 0.2818174710670554e-01, 0.1396210563637443e-18, 0.2854399814229969e-05, 0.1344352644992036e-05, 0.1697846193036144e-01, 0.6162975822039155e-31, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			pp:       0,
			dmin:     8.1926049685687600e-005,
			desig:    -9.9312918999672206e-017,
			qmax:     2.1682216902523144,
			ttype:    -4,
			dmin1:    8.9258234858565516e-002,
			dmin2:    8.9258234858565516e-002,
			dn:       8.1926049685687600e-005,
			dn1:      0.10847668953277810,
			dn2:      0.20701876553386761,
			g:        0.0000000000000000,
			tau:      1.5594219393746818e-003,
			nFail:    2,
			iter:     62,
			sigma:    1.0941247654737283,
			nDiv:     1057,
			zOut:     []float64{0.1386852748028120e+01, 0.1386771331083530e+01, 0.2485138759635906e-17, 0.1950383162890474e-17, 0.1088363957319145e+01, 0.1095197258320730e+01, 0.6914717946174950e-02, 0.7233164949324962e-02, 0.1145634931626446e+01, 0.1240614799765893e+01, 0.1022944500333619e+00, 0.6772592167212049e-01, 0.8213718410609353e+00, 0.7839262607332537e+00, 0.3036175828902931e-01, 0.4227125255215914e-01, 0.1091423778368554e+01, 0.1120328406526543e+01, 0.7125729765473905e-01, 0.4775711530437247e-01, 0.7508515569660207e+00, 0.7030343793170689e+00, 0.2135460001102832e-04, 0.2711304941630149e-05, 0.8926135754328580e-01, 0.8918452636746399e-01, 0.7297073710223142e-05, 0.1703885458517271e-04, 0.2082481603404622e+00, 0.2088031613602029e+00, 0.6534568189162364e-03, 0.3402961195615630e-03, 0.1087369563009673e+00, 0.1083191813025733e+00, 0.3938065757966957e-05, 0.2978513750500819e-08, 0.8192604968568760e-04, 0.5061265815246250e-06, 0.2485138759635906e-17, 0.1950383162890474e-17, 0.9154376259418607e+00, 0.2764930279233778e-14, 0.5964194309842799e-17, 0.7719264500395369e-17, 0.8241395430971566e+00, 0.2245698748385924e-16, 0.1792267857826344e-16, 0.2548071704976161e-16, 0.7318275341991307e+00, 0.4959852501050381e-23, 0.8676862906242004e-16, 0.1414000300319855e-15, 0.5186198053161721e+00, 0.1503295986001297e-14, 0.3452895805257589e-14, 0.7981257539768321e-14, 0.4143051093784424e+00, 0.1303656847202082e-15, 0.9855809897129084e-12, 0.2614986238618434e-11, 0.2112065329503869e+00, 0.8411884198867843e-17, 0.2293573303077261e-09, 0.9833566024906726e-10, 0.1093987140067686e+00, 0.2696165428113804e-17, 0.5825676764620370e-08, 0.2638661645752538e-08, 0.9751785856405315e-01, 0.6617412588681544e-21, 0.2809066689524106e-07, 0.6165545233233256e-07, 0.2818174710670554e-01, 0.1396210563637443e-18, 0.2854399814229969e-05, 0.1344352644992036e-05, 0.1697846193036144e-01, 0.6162975822039155e-31, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			i0Out:    1,
			n0Out:    10,
			ppOut:    0,
			dminOut:  5.0612658152462498e-007,
			desigOut: -9.3553094958342964e-017,
			sigmaOut: 1.0942061824183187,
			qmaxOut:  2.1682216902523144,
			nFailOut: 2,
			iterOut:  63,
			nDivOut:  1068,
			ttypeOut: -4,
			dmin1Out: 8.9177229293753768e-002,
			dmin2Out: 8.9177229293753768e-002,
			dnOut:    5.0612658152462498e-007,
			dn1Out:   0.10831524323681536,
			dn2Out:   0.20814970454128662,
			gOut:     0.0000000000000000,
			tauOut:   8.1416944590412474e-005,
		},
		{
			i0:       1,
			n0:       10,
			z:        []float64{0.1386852748028120e+01, 0.1386771331083530e+01, 0.2485138759635906e-17, 0.1950383162890474e-17, 0.1088363957319145e+01, 0.1095197258320730e+01, 0.6914717946174950e-02, 0.7233164949324962e-02, 0.1145634931626446e+01, 0.1240614799765893e+01, 0.1022944500333619e+00, 0.6772592167212049e-01, 0.8213718410609353e+00, 0.7839262607332537e+00, 0.3036175828902931e-01, 0.4227125255215914e-01, 0.1091423778368554e+01, 0.1120328406526543e+01, 0.7125729765473905e-01, 0.4775711530437247e-01, 0.7508515569660207e+00, 0.7030343793170689e+00, 0.2135460001102832e-04, 0.2711304941630149e-05, 0.8926135754328580e-01, 0.8918452636746399e-01, 0.7297073710223142e-05, 0.1703885458517271e-04, 0.2082481603404622e+00, 0.2088031613602029e+00, 0.6534568189162364e-03, 0.3402961195615630e-03, 0.1087369563009673e+00, 0.1083191813025733e+00, 0.3938065757966957e-05, 0.2978513750500819e-08, 0.8192604968568760e-04, 0.5061265815246250e-06, 0.2485138759635906e-17, 0.1950383162890474e-17, 0.9154376259418607e+00, 0.2764930279233778e-14, 0.5964194309842799e-17, 0.7719264500395369e-17, 0.8241395430971566e+00, 0.2245698748385924e-16, 0.1792267857826344e-16, 0.2548071704976161e-16, 0.7318275341991307e+00, 0.4959852501050381e-23, 0.8676862906242004e-16, 0.1414000300319855e-15, 0.5186198053161721e+00, 0.1503295986001297e-14, 0.3452895805257589e-14, 0.7981257539768321e-14, 0.4143051093784424e+00, 0.1303656847202082e-15, 0.9855809897129084e-12, 0.2614986238618434e-11, 0.2112065329503869e+00, 0.8411884198867843e-17, 0.2293573303077261e-09, 0.9833566024906726e-10, 0.1093987140067686e+00, 0.2696165428113804e-17, 0.5825676764620370e-08, 0.2638661645752538e-08, 0.9751785856405315e-01, 0.6617412588681544e-21, 0.2809066689524106e-07, 0.6165545233233256e-07, 0.2818174710670554e-01, 0.1396210563637443e-18, 0.2854399814229969e-05, 0.1344352644992036e-05, 0.1697846193036144e-01, 0.6162975822039155e-31, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			pp:       1,
			dmin:     5.0612658152462498e-007,
			desig:    -9.3553094958342964e-017,
			qmax:     2.1682216902523144,
			ttype:    -4,
			dmin1:    8.9177229293753768e-002,
			dmin2:    8.9177229293753768e-002,
			dn:       5.0612658152462498e-007,
			dn1:      0.10831524323681536,
			dn2:      0.20814970454128662,
			g:        0.0000000000000000,
			tau:      8.1416944590412474e-005,
			nFail:    2,
			iter:     63,
			sigma:    1.0942061824183187,
			nDiv:     1068,
			zOut:     []float64{0.1386770825043033e+01, 0.1386771331083530e+01, 0.1540308069724697e-17, 0.1950383162890474e-17, 0.1102429917229558e+01, 0.1095197258320730e+01, 0.8139811288713328e-02, 0.7233164949324962e-02, 0.1300200404108803e+01, 0.1240614799765893e+01, 0.4083380405309871e-01, 0.6772592167212049e-01, 0.7853632031918177e+00, 0.7839262607332537e+00, 0.6030036143936175e-01, 0.4227125255215914e-01, 0.1107784654351057e+01, 0.1120328406526543e+01, 0.3030814137396716e-01, 0.4775711530437247e-01, 0.6727284432075471e+00, 0.7030343793170689e+00, 0.3594413905024206e-06, 0.2711304941630149e-05, 0.8920069974016231e-01, 0.8918452636746399e-01, 0.3988496406087022e-04, 0.1703885458517271e-04, 0.2091030664752072e+00, 0.2088031613602029e+00, 0.1762795624794033e-03, 0.3402961195615630e-03, 0.1081423986781113e+00, 0.1083191813025733e+00, 0.1393999949133917e-13, 0.2978513750500819e-08, 0.8607124697139263e-10, 0.5061265815246250e-06, 0.1540308069724697e-17, 0.1950383162890474e-17, 0.9154376259418607e+00, 0.2764930279233778e-14, 0.5964194309842799e-17, 0.7719264500395369e-17, 0.8241395430971566e+00, 0.2245698748385924e-16, 0.1792267857826344e-16, 0.2548071704976161e-16, 0.7318275341991307e+00, 0.4959852501050381e-23, 0.8676862906242004e-16, 0.1414000300319855e-15, 0.5186198053161721e+00, 0.1503295986001297e-14, 0.3452895805257589e-14, 0.7981257539768321e-14, 0.4143051093784424e+00, 0.1303656847202082e-15, 0.9855809897129084e-12, 0.2614986238618434e-11, 0.2112065329503869e+00, 0.8411884198867843e-17, 0.2293573303077261e-09, 0.9833566024906726e-10, 0.1093987140067686e+00, 0.2696165428113804e-17, 0.5825676764620370e-08, 0.2638661645752538e-08, 0.9751785856405315e-01, 0.6617412588681544e-21, 0.2809066689524106e-07, 0.6165545233233256e-07, 0.2818174710670554e-01, 0.1396210563637443e-18, 0.2854399814229969e-05, 0.1344352644992036e-05, 0.1697846193036144e-01, 0.6162975822039155e-31, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			i0Out:    1,
			n0Out:    10,
			ppOut:    1,
			dminOut:  8.6071246971392626e-011,
			desigOut: 7.0901951641105278e-017,
			sigmaOut: 1.0942066884588149,
			qmaxOut:  2.1682216902523144,
			nFailOut: 2,
			iterOut:  64,
			nDivOut:  1079,
			ttypeOut: -4,
			dmin1Out: 8.9183660885577137e-002,
			dmin2Out: 8.9183660885577137e-002,
			dnOut:    8.6071246971392626e-011,
			dn1Out:   0.10814239569959758,
			dn2Out:   0.20876277035564564,
			gOut:     0.0000000000000000,
			tauOut:   5.0604049633765406e-007,
		},
		{
			i0:       1,
			n0:       10,
			z:        []float64{0.1386770825043033e+01, 0.1386771331083530e+01, 0.1540308069724697e-17, 0.1950383162890474e-17, 0.1102429917229558e+01, 0.1095197258320730e+01, 0.8139811288713328e-02, 0.7233164949324962e-02, 0.1300200404108803e+01, 0.1240614799765893e+01, 0.4083380405309871e-01, 0.6772592167212049e-01, 0.7853632031918177e+00, 0.7839262607332537e+00, 0.6030036143936175e-01, 0.4227125255215914e-01, 0.1107784654351057e+01, 0.1120328406526543e+01, 0.3030814137396716e-01, 0.4775711530437247e-01, 0.6727284432075471e+00, 0.7030343793170689e+00, 0.3594413905024206e-06, 0.2711304941630149e-05, 0.8920069974016231e-01, 0.8918452636746399e-01, 0.3988496406087022e-04, 0.1703885458517271e-04, 0.2091030664752072e+00, 0.2088031613602029e+00, 0.1762795624794033e-03, 0.3402961195615630e-03, 0.1081423986781113e+00, 0.1083191813025733e+00, 0.1393999949133917e-13, 0.2978513750500819e-08, 0.8607124697139263e-10, 0.5061265815246250e-06, 0.1540308069724697e-17, 0.1950383162890474e-17, 0.9154376259418607e+00, 0.2764930279233778e-14, 0.5964194309842799e-17, 0.7719264500395369e-17, 0.8241395430971566e+00, 0.2245698748385924e-16, 0.1792267857826344e-16, 0.2548071704976161e-16, 0.7318275341991307e+00, 0.4959852501050381e-23, 0.8676862906242004e-16, 0.1414000300319855e-15, 0.5186198053161721e+00, 0.1503295986001297e-14, 0.3452895805257589e-14, 0.7981257539768321e-14, 0.4143051093784424e+00, 0.1303656847202082e-15, 0.9855809897129084e-12, 0.2614986238618434e-11, 0.2112065329503869e+00, 0.8411884198867843e-17, 0.2293573303077261e-09, 0.9833566024906726e-10, 0.1093987140067686e+00, 0.2696165428113804e-17, 0.5825676764620370e-08, 0.2638661645752538e-08, 0.9751785856405315e-01, 0.6617412588681544e-21, 0.2809066689524106e-07, 0.6165545233233256e-07, 0.2818174710670554e-01, 0.1396210563637443e-18, 0.2854399814229969e-05, 0.1344352644992036e-05, 0.1697846193036144e-01, 0.6162975822039155e-31, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			pp:       0,
			dmin:     8.6071246971392626e-011,
			desig:    7.0901951641105278e-017,
			qmax:     2.1682216902523144,
			ttype:    -4,
			dmin1:    8.9183660885577137e-002,
			dmin2:    8.9183660885577137e-002,
			dn:       8.6071246971392626e-011,
			dn1:      0.10814239569959758,
			dn2:      0.20876277035564564,
			g:        0.0000000000000000,
			tau:      5.0604049633765406e-007,
			nFail:    2,
			iter:     64,
			sigma:    1.0942066884588149,
			nDiv:     1079,
			zOut:     []float64{0.1386770825043033e+01, 0.1386770824956962e+01, 0.1540308069724697e-17, 0.1224486171222500e-17, 0.1102429917229558e+01, 0.1110569728432201e+01, 0.8139811288713328e-02, 0.9529690622753703e-02, 0.1300200404108803e+01, 0.1331504517453077e+01, 0.4083380405309871e-01, 0.2408506071837550e-01, 0.7853632031918177e+00, 0.8215785038267327e+00, 0.6030036143936175e-01, 0.8130667336500198e-01, 0.1107784654351057e+01, 0.1056786122273951e+01, 0.3030814137396716e-01, 0.1929354325655847e-01, 0.6727284432075471e+00, 0.6534352593063080e+00, 0.3594413905024206e-06, 0.4906748310831983e-07, 0.8920069974016231e-01, 0.8924053555066887e-01, 0.3988496406087022e-04, 0.9345605379795243e-04, 0.2091030664752072e+00, 0.2091858898978174e+00, 0.1762795624794033e-03, 0.9113088236382798e-04, 0.1081423986781113e+00, 0.1080512677096902e+00, 0.1393999949133917e-13, 0.1110429488179469e-22, 0.8607124697139263e-10, 0.3167883469916549e-16, 0.1540308069724697e-17, 0.1224486171222500e-17, 0.9154376259418607e+00, 0.2764930279233778e-14, 0.5964194309842799e-17, 0.7719264500395369e-17, 0.8241395430971566e+00, 0.2245698748385924e-16, 0.1792267857826344e-16, 0.2548071704976161e-16, 0.7318275341991307e+00, 0.4959852501050381e-23, 0.8676862906242004e-16, 0.1414000300319855e-15, 0.5186198053161721e+00, 0.1503295986001297e-14, 0.3452895805257589e-14, 0.7981257539768321e-14, 0.4143051093784424e+00, 0.1303656847202082e-15, 0.9855809897129084e-12, 0.2614986238618434e-11, 0.2112065329503869e+00, 0.8411884198867843e-17, 0.2293573303077261e-09, 0.9833566024906726e-10, 0.1093987140067686e+00, 0.2696165428113804e-17, 0.5825676764620370e-08, 0.2638661645752538e-08, 0.9751785856405315e-01, 0.6617412588681544e-21, 0.2809066689524106e-07, 0.6165545233233256e-07, 0.2818174710670554e-01, 0.1396210563637443e-18, 0.2854399814229969e-05, 0.1344352644992036e-05, 0.1697846193036144e-01, 0.6162975822039155e-31, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			i0Out:    1,
			n0Out:    10,
			ppOut:    0,
			dminOut:  3.1678834699165494e-017,
			desigOut: -8.6057196331979189e-017,
			sigmaOut: 1.0942066885448862,
			qmaxOut:  2.1682216902523144,
			nFailOut: 2,
			iterOut:  65,
			nDivOut:  1090,
			ttypeOut: -4,
			dmin1Out: 8.9200650586607991e-002,
			dmin2Out: 8.9200650586607991e-002,
			dnOut:    3.1678834699165494e-017,
			dn1Out:   0.10805126770967630,
			dn2Out:   0.20900961033533805,
			gOut:     0.0000000000000000,
			tauOut:   8.6071215292546838e-011,
		},
		{
			i0:       1,
			n0:       3,
			z:        []float64{0.1143865217830237e+00, 0.1154808079639322e+00, 0.2932453102768040e-16, 0.1817901582187390e-16, 0.1834406364024603e+00, 0.1845166504084682e+00, 0.1101341475930520e-06, 0.1827217490071947e-04, 0.1127820009512990e-04, 0.1105674515151256e-02, 0.1845166504084682e+00, 0.2816548643828996e+00, 0.2169993432366266e+01, 0.5067642455139780e-26, 0.3415542419024794e-17, 0.4171805735046273e-17, 0.1810817848712033e+01, 0.1972152263052530e-29, 0.6115792910959321e-18, 0.6699979816704878e-18, 0.1681677309759878e+01, 0.5241368559131172e-22, 0.4369342452764876e-18, 0.4084491703488284e-18, 0.1303743375476047e+01, 0.0000000000000000e+00, 0.2325140442247288e-18, 0.2353545363971710e-18, 0.1202161769544433e+01, 0.0000000000000000e+00, 0.2163565864913247e-18, 0.2222159192042978e-18, 0.1183377192742860e+01, 0.2033579915108999e-22, 0.2652482201353177e-18, 0.2438256017573345e-18, 0.1094206688544886e+01, 0.3167883469916549e-16, 0.9806070694382835e-18, 0.1224486171222500e-17, 0.9154376259418607e+00, 0.2764930279233778e-14, 0.5964194309842799e-17, 0.7719264500395369e-17, 0.8241395430971566e+00, 0.2245698748385924e-16, 0.1792267857826344e-16, 0.2548071704976161e-16, 0.7318275341991307e+00, 0.4959852501050381e-23, 0.8676862906242004e-16, 0.1414000300319855e-15, 0.5186198053161721e+00, 0.1503295986001297e-14, 0.3452895805257589e-14, 0.7981257539768321e-14, 0.4143051093784424e+00, 0.1303656847202082e-15, 0.9855809897129084e-12, 0.2614986238618434e-11, 0.2112065329503869e+00, 0.8411884198867843e-17, 0.2293573303077261e-09, 0.9833566024906726e-10, 0.1093987140067686e+00, 0.2696165428113804e-17, 0.5825676764620370e-08, 0.2638661645752538e-08, 0.9751785856405315e-01, 0.6617412588681544e-21, 0.2809066689524106e-07, 0.6165545233233256e-07, 0.2818174710670554e-01, 0.1396210563637443e-18, 0.2854399814229969e-05, 0.1344352644992036e-05, 0.1697846193036144e-01, 0.6162975822039155e-31, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			pp:       0,
			dmin:     1.1278200095129901e-005,
			desig:    1.1991276027689679e-016,
			qmax:     0.41760242607758735,
			ttype:    -4,
			dmin1:    0.11438652178302365,
			dmin2:    0.11438652178302365,
			dn:       1.1278200095129901e-005,
			dn1:      0.18342236422755959,
			dn2:      0.11438652178302365,
			g:        0.0000000000000000,
			tau:      1.0942861809085330e-003,
			nFail:    3,
			iter:     99,
			sigma:    2.3665909917188244,
			nDiv:     1347,
			zOut:     []float64{0.1143865217830237e+00, 0.1143752525446631e+00, 0.2932453102768040e-16, 0.4703212027287794e-16, 0.1834406364024603e+00, 0.1834294772982473e+00, 0.1101341475930520e-06, 0.6771621290952006e-11, 0.1127820009512990e-04, 0.8954962962003413e-08, 0.1845166504084682e+00, 0.1834406364024603e+00, 0.2169993432366266e+01, 0.5067642455139780e-26, 0.3415542419024794e-17, 0.4171805735046273e-17, 0.1810817848712033e+01, 0.1972152263052530e-29, 0.6115792910959321e-18, 0.6699979816704878e-18, 0.1681677309759878e+01, 0.5241368559131172e-22, 0.4369342452764876e-18, 0.4084491703488284e-18, 0.1303743375476047e+01, 0.0000000000000000e+00, 0.2325140442247288e-18, 0.2353545363971710e-18, 0.1202161769544433e+01, 0.0000000000000000e+00, 0.2163565864913247e-18, 0.2222159192042978e-18, 0.1183377192742860e+01, 0.2033579915108999e-22, 0.2652482201353177e-18, 0.2438256017573345e-18, 0.1094206688544886e+01, 0.3167883469916549e-16, 0.9806070694382835e-18, 0.1224486171222500e-17, 0.9154376259418607e+00, 0.2764930279233778e-14, 0.5964194309842799e-17, 0.7719264500395369e-17, 0.8241395430971566e+00, 0.2245698748385924e-16, 0.1792267857826344e-16, 0.2548071704976161e-16, 0.7318275341991307e+00, 0.4959852501050381e-23, 0.8676862906242004e-16, 0.1414000300319855e-15, 0.5186198053161721e+00, 0.1503295986001297e-14, 0.3452895805257589e-14, 0.7981257539768321e-14, 0.4143051093784424e+00, 0.1303656847202082e-15, 0.9855809897129084e-12, 0.2614986238618434e-11, 0.2112065329503869e+00, 0.8411884198867843e-17, 0.2293573303077261e-09, 0.9833566024906726e-10, 0.1093987140067686e+00, 0.2696165428113804e-17, 0.5825676764620370e-08, 0.2638661645752538e-08, 0.9751785856405315e-01, 0.6617412588681544e-21, 0.2809066689524106e-07, 0.6165545233233256e-07, 0.2818174710670554e-01, 0.1396210563637443e-18, 0.2854399814229969e-05, 0.1344352644992036e-05, 0.1697846193036144e-01, 0.6162975822039155e-31, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			i0Out:    1,
			n0Out:    3,
			ppOut:    0,
			dminOut:  8.9549629620034135e-009,
			desigOut: -3.3964327119002935e-017,
			sigmaOut: 2.3666022609571851,
			qmaxOut:  0.41760242607758735,
			nFailOut: 3,
			iterOut:  100,
			nDivOut:  1351,
			ttypeOut: -4,
			dmin1Out: 0.11437525254466312,
			dmin2Out: 0.11437525254466312,
			dnOut:    8.9549629620034135e-009,
			dn1Out:   0.18342936716409974,
			dn2Out:   0.11437525254466312,
			gOut:     0.0000000000000000,
			tauOut:   1.1269238360546607e-005,
		},
		{
			i0:       1,
			n0:       3,
			z:        []float64{0.1143865217830237e+00, 0.1143752525446631e+00, 0.2932453102768040e-16, 0.4703212027287794e-16, 0.1834406364024603e+00, 0.1834294772982473e+00, 0.1101341475930520e-06, 0.6771621290952006e-11, 0.1127820009512990e-04, 0.8954962962003413e-08, 0.1845166504084682e+00, 0.1834406364024603e+00, 0.2169993432366266e+01, 0.5067642455139780e-26, 0.3415542419024794e-17, 0.4171805735046273e-17, 0.1810817848712033e+01, 0.1972152263052530e-29, 0.6115792910959321e-18, 0.6699979816704878e-18, 0.1681677309759878e+01, 0.5241368559131172e-22, 0.4369342452764876e-18, 0.4084491703488284e-18, 0.1303743375476047e+01, 0.0000000000000000e+00, 0.2325140442247288e-18, 0.2353545363971710e-18, 0.1202161769544433e+01, 0.0000000000000000e+00, 0.2163565864913247e-18, 0.2222159192042978e-18, 0.1183377192742860e+01, 0.2033579915108999e-22, 0.2652482201353177e-18, 0.2438256017573345e-18, 0.1094206688544886e+01, 0.3167883469916549e-16, 0.9806070694382835e-18, 0.1224486171222500e-17, 0.9154376259418607e+00, 0.2764930279233778e-14, 0.5964194309842799e-17, 0.7719264500395369e-17, 0.8241395430971566e+00, 0.2245698748385924e-16, 0.1792267857826344e-16, 0.2548071704976161e-16, 0.7318275341991307e+00, 0.4959852501050381e-23, 0.8676862906242004e-16, 0.1414000300319855e-15, 0.5186198053161721e+00, 0.1503295986001297e-14, 0.3452895805257589e-14, 0.7981257539768321e-14, 0.4143051093784424e+00, 0.1303656847202082e-15, 0.9855809897129084e-12, 0.2614986238618434e-11, 0.2112065329503869e+00, 0.8411884198867843e-17, 0.2293573303077261e-09, 0.9833566024906726e-10, 0.1093987140067686e+00, 0.2696165428113804e-17, 0.5825676764620370e-08, 0.2638661645752538e-08, 0.9751785856405315e-01, 0.6617412588681544e-21, 0.2809066689524106e-07, 0.6165545233233256e-07, 0.2818174710670554e-01, 0.1396210563637443e-18, 0.2854399814229969e-05, 0.1344352644992036e-05, 0.1697846193036144e-01, 0.6162975822039155e-31, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			pp:       1,
			dmin:     8.9549629620034135e-009,
			desig:    -3.3964327119002935e-017,
			qmax:     0.41760242607758735,
			ttype:    -4,
			dmin1:    0.11437525254466312,
			dmin2:    0.11437525254466312,
			dn:       8.9549629620034135e-009,
			dn1:      0.18342936716409974,
			dn2:      0.11437525254466312,
			g:        0.0000000000000000,
			tau:      1.1269238360546607e-005,
			nFail:    3,
			iter:     100,
			sigma:    2.3666022609571851,
			nDiv:     1351,
			zOut:     []float64{0.1143752435897560e+00, 0.1143752525446631e+00, 0.7542783706608855e-16, 0.4703212027287794e-16, 0.1834294683501117e+00, 0.1834294772982473e+00, 0.3305882004599510e-18, 0.6771621290952006e-11, 0.5575326929115112e-13, 0.8954962962003413e-08, 0.1834294772982473e+00, 0.1834406364024603e+00, 0.2169993432366266e+01, 0.5067642455139780e-26, 0.3415542419024794e-17, 0.4171805735046273e-17, 0.1810817848712033e+01, 0.1972152263052530e-29, 0.6115792910959321e-18, 0.6699979816704878e-18, 0.1681677309759878e+01, 0.5241368559131172e-22, 0.4369342452764876e-18, 0.4084491703488284e-18, 0.1303743375476047e+01, 0.0000000000000000e+00, 0.2325140442247288e-18, 0.2353545363971710e-18, 0.1202161769544433e+01, 0.0000000000000000e+00, 0.2163565864913247e-18, 0.2222159192042978e-18, 0.1183377192742860e+01, 0.2033579915108999e-22, 0.2652482201353177e-18, 0.2438256017573345e-18, 0.1094206688544886e+01, 0.3167883469916549e-16, 0.9806070694382835e-18, 0.1224486171222500e-17, 0.9154376259418607e+00, 0.2764930279233778e-14, 0.5964194309842799e-17, 0.7719264500395369e-17, 0.8241395430971566e+00, 0.2245698748385924e-16, 0.1792267857826344e-16, 0.2548071704976161e-16, 0.7318275341991307e+00, 0.4959852501050381e-23, 0.8676862906242004e-16, 0.1414000300319855e-15, 0.5186198053161721e+00, 0.1503295986001297e-14, 0.3452895805257589e-14, 0.7981257539768321e-14, 0.4143051093784424e+00, 0.1303656847202082e-15, 0.9855809897129084e-12, 0.2614986238618434e-11, 0.2112065329503869e+00, 0.8411884198867843e-17, 0.2293573303077261e-09, 0.9833566024906726e-10, 0.1093987140067686e+00, 0.2696165428113804e-17, 0.5825676764620370e-08, 0.2638661645752538e-08, 0.9751785856405315e-01, 0.6617412588681544e-21, 0.2809066689524106e-07, 0.6165545233233256e-07, 0.2818174710670554e-01, 0.1396210563637443e-18, 0.2854399814229969e-05, 0.1344352644992036e-05, 0.1697846193036144e-01, 0.6162975822039155e-31, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			i0Out:    1,
			n0Out:    3,
			ppOut:    1,
			dminOut:  5.5753269291151117e-014,
			desigOut: 1.3632246356731358e-016,
			sigmaOut: 2.3666022699120921,
			qmaxOut:  0.41760242607758735,
			nFailOut: 3,
			iterOut:  101,
			nDivOut:  1355,
			ttypeOut: -4,
			dmin1Out: 0.11437524358975594,
			dmin2Out: 0.11437524358975594,
			dnOut:    5.5753269291151117e-014,
			dn1Out:   0.18342946834334006,
			dn2Out:   0.11437524358975594,
			gOut:     0.0000000000000000,
			tauOut:   8.9549072084035346e-009,
		},
		{
			i0:       1,
			n0:       3,
			z:        []float64{0.1143752435897560e+00, 0.1143752525446631e+00, 0.7542783706608855e-16, 0.4703212027287794e-16, 0.1834294683501117e+00, 0.1834294772982473e+00, 0.3305882004599510e-18, 0.6771621290952006e-11, 0.5575326929115112e-13, 0.8954962962003413e-08, 0.1834294772982473e+00, 0.1834406364024603e+00, 0.2169993432366266e+01, 0.5067642455139780e-26, 0.3415542419024794e-17, 0.4171805735046273e-17, 0.1810817848712033e+01, 0.1972152263052530e-29, 0.6115792910959321e-18, 0.6699979816704878e-18, 0.1681677309759878e+01, 0.5241368559131172e-22, 0.4369342452764876e-18, 0.4084491703488284e-18, 0.1303743375476047e+01, 0.0000000000000000e+00, 0.2325140442247288e-18, 0.2353545363971710e-18, 0.1202161769544433e+01, 0.0000000000000000e+00, 0.2163565864913247e-18, 0.2222159192042978e-18, 0.1183377192742860e+01, 0.2033579915108999e-22, 0.2652482201353177e-18, 0.2438256017573345e-18, 0.1094206688544886e+01, 0.3167883469916549e-16, 0.9806070694382835e-18, 0.1224486171222500e-17, 0.9154376259418607e+00, 0.2764930279233778e-14, 0.5964194309842799e-17, 0.7719264500395369e-17, 0.8241395430971566e+00, 0.2245698748385924e-16, 0.1792267857826344e-16, 0.2548071704976161e-16, 0.7318275341991307e+00, 0.4959852501050381e-23, 0.8676862906242004e-16, 0.1414000300319855e-15, 0.5186198053161721e+00, 0.1503295986001297e-14, 0.3452895805257589e-14, 0.7981257539768321e-14, 0.4143051093784424e+00, 0.1303656847202082e-15, 0.9855809897129084e-12, 0.2614986238618434e-11, 0.2112065329503869e+00, 0.8411884198867843e-17, 0.2293573303077261e-09, 0.9833566024906726e-10, 0.1093987140067686e+00, 0.2696165428113804e-17, 0.5825676764620370e-08, 0.2638661645752538e-08, 0.9751785856405315e-01, 0.6617412588681544e-21, 0.2809066689524106e-07, 0.6165545233233256e-07, 0.2818174710670554e-01, 0.1396210563637443e-18, 0.2854399814229969e-05, 0.1344352644992036e-05, 0.1697846193036144e-01, 0.6162975822039155e-31, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			pp:       0,
			dmin:     5.5753269291151117e-014,
			desig:    1.3632246356731358e-016,
			qmax:     0.41760242607758735,
			ttype:    -4,
			dmin1:    0.11437524358975594,
			dmin2:    0.11437524358975594,
			dn:       5.5753269291151117e-014,
			dn1:      0.18342946834334006,
			dn2:      0.11437524358975594,
			g:        0.0000000000000000,
			tau:      8.9549072084035346e-009,
			nFail:    3,
			iter:     101,
			sigma:    2.3666022699120921,
			nDiv:     1355,
			zOut:     []float64{0.1143752435897560e+00, 0.1143752435897003e+00, 0.7542783706608855e-16, 0.1209675067575323e-15, 0.1834294683501117e+00, 0.1834294683500558e+00, 0.3305882004599510e-18, 0.1004820715586787e-30, 0.5575326929115112e-13, 0.7669624420364386e-22, 0.1834294772982473e+00, 0.1834294683501117e+00, 0.2169993432366266e+01, 0.5067642455139780e-26, 0.3415542419024794e-17, 0.4171805735046273e-17, 0.1810817848712033e+01, 0.1972152263052530e-29, 0.6115792910959321e-18, 0.6699979816704878e-18, 0.1681677309759878e+01, 0.5241368559131172e-22, 0.4369342452764876e-18, 0.4084491703488284e-18, 0.1303743375476047e+01, 0.0000000000000000e+00, 0.2325140442247288e-18, 0.2353545363971710e-18, 0.1202161769544433e+01, 0.0000000000000000e+00, 0.2163565864913247e-18, 0.2222159192042978e-18, 0.1183377192742860e+01, 0.2033579915108999e-22, 0.2652482201353177e-18, 0.2438256017573345e-18, 0.1094206688544886e+01, 0.3167883469916549e-16, 0.9806070694382835e-18, 0.1224486171222500e-17, 0.9154376259418607e+00, 0.2764930279233778e-14, 0.5964194309842799e-17, 0.7719264500395369e-17, 0.8241395430971566e+00, 0.2245698748385924e-16, 0.1792267857826344e-16, 0.2548071704976161e-16, 0.7318275341991307e+00, 0.4959852501050381e-23, 0.8676862906242004e-16, 0.1414000300319855e-15, 0.5186198053161721e+00, 0.1503295986001297e-14, 0.3452895805257589e-14, 0.7981257539768321e-14, 0.4143051093784424e+00, 0.1303656847202082e-15, 0.9855809897129084e-12, 0.2614986238618434e-11, 0.2112065329503869e+00, 0.8411884198867843e-17, 0.2293573303077261e-09, 0.9833566024906726e-10, 0.1093987140067686e+00, 0.2696165428113804e-17, 0.5825676764620370e-08, 0.2638661645752538e-08, 0.9751785856405315e-01, 0.6617412588681544e-21, 0.2809066689524106e-07, 0.6165545233233256e-07, 0.2818174710670554e-01, 0.1396210563637443e-18, 0.2854399814229969e-05, 0.1344352644992036e-05, 0.1697846193036144e-01, 0.6162975822039155e-31, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			i0Out:    1,
			n0Out:    3,
			ppOut:    0,
			dminOut:  7.6696244203643861e-023,
			desigOut: -6.5648763085702813e-017,
			sigmaOut: 2.3666022699121481,
			qmaxOut:  0.41760242607758735,
			nFailOut: 3,
			iterOut:  102,
			nDivOut:  1359,
			ttypeOut: -4,
			dmin1Out: 0.11437524358970023,
			dmin2Out: 0.11437524358970023,
			dnOut:    7.6696244203643861e-023,
			dn1Out:   0.18342946835005580,
			dn2Out:   0.11437524358970023,
			gOut:     0.0000000000000000,
			tauOut:   5.5753269214454873e-014,
		},
		{
			i0:       1,
			n0:       3,
			z:        []float64{0.1143752435897560e+00, 0.1143752435897003e+00, 0.7542783706608855e-16, 0.1209675067575323e-15, 0.1834294683501117e+00, 0.1834294683500558e+00, 0.3305882004599510e-18, 0.1004820715586787e-30, 0.5575326929115112e-13, 0.7669624420364386e-22, 0.1834294772982473e+00, 0.1834294683501117e+00, 0.2169993432366266e+01, 0.5067642455139780e-26, 0.3415542419024794e-17, 0.4171805735046273e-17, 0.1810817848712033e+01, 0.1972152263052530e-29, 0.6115792910959321e-18, 0.6699979816704878e-18, 0.1681677309759878e+01, 0.5241368559131172e-22, 0.4369342452764876e-18, 0.4084491703488284e-18, 0.1303743375476047e+01, 0.0000000000000000e+00, 0.2325140442247288e-18, 0.2353545363971710e-18, 0.1202161769544433e+01, 0.0000000000000000e+00, 0.2163565864913247e-18, 0.2222159192042978e-18, 0.1183377192742860e+01, 0.2033579915108999e-22, 0.2652482201353177e-18, 0.2438256017573345e-18, 0.1094206688544886e+01, 0.3167883469916549e-16, 0.9806070694382835e-18, 0.1224486171222500e-17, 0.9154376259418607e+00, 0.2764930279233778e-14, 0.5964194309842799e-17, 0.7719264500395369e-17, 0.8241395430971566e+00, 0.2245698748385924e-16, 0.1792267857826344e-16, 0.2548071704976161e-16, 0.7318275341991307e+00, 0.4959852501050381e-23, 0.8676862906242004e-16, 0.1414000300319855e-15, 0.5186198053161721e+00, 0.1503295986001297e-14, 0.3452895805257589e-14, 0.7981257539768321e-14, 0.4143051093784424e+00, 0.1303656847202082e-15, 0.9855809897129084e-12, 0.2614986238618434e-11, 0.2112065329503869e+00, 0.8411884198867843e-17, 0.2293573303077261e-09, 0.9833566024906726e-10, 0.1093987140067686e+00, 0.2696165428113804e-17, 0.5825676764620370e-08, 0.2638661645752538e-08, 0.9751785856405315e-01, 0.6617412588681544e-21, 0.2809066689524106e-07, 0.6165545233233256e-07, 0.2818174710670554e-01, 0.1396210563637443e-18, 0.2854399814229969e-05, 0.1344352644992036e-05, 0.1697846193036144e-01, 0.6162975822039155e-31, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			pp:       1,
			dmin:     7.6696244203643861e-023,
			desig:    -6.5648763085702813e-017,
			qmax:     0.41760242607758735,
			ttype:    -4,
			dmin1:    0.11437524358970023,
			dmin2:    0.11437524358970023,
			dn:       7.6696244203643861e-023,
			dn1:      0.18342946835005580,
			dn2:      0.11437524358970023,
			g:        0.0000000000000000,
			tau:      5.5753269214454873e-014,
			nFail:    3,
			iter:     102,
			sigma:    2.3666022699121481,
			nDiv:     1359,
			zOut:     []float64{0.2550031738262204e+01, 0.1834294683500561e+00, 0.7542783706608855e-16, 0.1209675067575323e-15, 0.2480977513501848e+01, 0.1143752435897001e+00, 0.3305882004599510e-18, 0.1004820715586787e-30, 0.2366602269912148e+01, 0.7669624420364386e-22, 0.1834294772982473e+00, 0.1834294683501117e+00, 0.2169993432366266e+01, 0.5067642455139780e-26, 0.3415542419024794e-17, 0.4171805735046273e-17, 0.1810817848712033e+01, 0.1972152263052530e-29, 0.6115792910959321e-18, 0.6699979816704878e-18, 0.1681677309759878e+01, 0.5241368559131172e-22, 0.4369342452764876e-18, 0.4084491703488284e-18, 0.1303743375476047e+01, 0.0000000000000000e+00, 0.2325140442247288e-18, 0.2353545363971710e-18, 0.1202161769544433e+01, 0.0000000000000000e+00, 0.2163565864913247e-18, 0.2222159192042978e-18, 0.1183377192742860e+01, 0.2033579915108999e-22, 0.2652482201353177e-18, 0.2438256017573345e-18, 0.1094206688544886e+01, 0.3167883469916549e-16, 0.9806070694382835e-18, 0.1224486171222500e-17, 0.9154376259418607e+00, 0.2764930279233778e-14, 0.5964194309842799e-17, 0.7719264500395369e-17, 0.8241395430971566e+00, 0.2245698748385924e-16, 0.1792267857826344e-16, 0.2548071704976161e-16, 0.7318275341991307e+00, 0.4959852501050381e-23, 0.8676862906242004e-16, 0.1414000300319855e-15, 0.5186198053161721e+00, 0.1503295986001297e-14, 0.3452895805257589e-14, 0.7981257539768321e-14, 0.4143051093784424e+00, 0.1303656847202082e-15, 0.9855809897129084e-12, 0.2614986238618434e-11, 0.2112065329503869e+00, 0.8411884198867843e-17, 0.2293573303077261e-09, 0.9833566024906726e-10, 0.1093987140067686e+00, 0.2696165428113804e-17, 0.5825676764620370e-08, 0.2638661645752538e-08, 0.9751785856405315e-01, 0.6617412588681544e-21, 0.2809066689524106e-07, 0.6165545233233256e-07, 0.2818174710670554e-01, 0.1396210563637443e-18, 0.2854399814229969e-05, 0.1344352644992036e-05, 0.1697846193036144e-01, 0.6162975822039155e-31, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			i0Out:    1,
			n0Out:    0,
			ppOut:    1,
			dminOut:  7.6696244203643861e-023,
			desigOut: -6.5648763085702813e-017,
			sigmaOut: 2.3666022699121481,
			qmaxOut:  0.41760242607758735,
			nFailOut: 3,
			iterOut:  102,
			nDivOut:  1359,
			ttypeOut: -4,
			dmin1Out: 0.11437524358970023,
			dmin2Out: 0.11437524358970023,
			dnOut:    7.6696244203643861e-023,
			dn1Out:   0.18342946835005580,
			dn2Out:   0.11437524358970023,
			gOut:     0.0000000000000000,
			tauOut:   5.5753269214454873e-014,
		},
		{
			i0:       1,
			n0:       21,
			z:        []float64{0.1648283185136998e+01, 0.1396221235720571e+01, 0.1712714336271993e+00, 0.2520619494164272e+00, 0.1510753432847732e+01, 0.1119977945086946e+01, 0.3792800633372563e+00, 0.5620469213879850e+00, 0.1328441621586708e+01, 0.1019485448443405e+01, 0.1388998759717073e+00, 0.6882362364805590e+00, 0.4466938077796418e+00, 0.2681061628164644e+00, 0.7093206047768255e+00, 0.3174875209348847e+00, 0.6843112870203156e+00, 0.9979892152967577e+00, 0.6490582981441884e+00, 0.3956426765003833e+00, 0.5085572738629487e+00, 0.1122623886995757e+01, 0.9750235054014829e-02, 0.3499168501137979e-01, 0.1364886053450573e+00, 0.1417066070690837e+00, 0.2301225778544498e-01, 0.4532233329988395e-02, 0.1646009972289452e+01, 0.6930161671496210e+00, 0.2362515608142310e+00, 0.9760060629252760e+00, 0.5818602562677768e+00, 0.3984323866837953e+00, 0.1797665269485310e-01, 0.4196794303982125e+00, 0.5600419521166516e+00, 0.2492354636952108e-01, 0.2195137569256029e+00, 0.5530950584419837e+00, 0.4184071984843414e+00, 0.2222708575473020e+00, 0.2727864547293006e+00, 0.4156500978626423e+00, 0.6774373914466536e-01, 0.2745959086613283e+00, 0.1050967099374242e+00, 0.6593428521263771e-01, 0.2040338718098096e+00, 0.1079809097801335e+00, 0.1271971985482246e+00, 0.2011496719671002e+00, 0.4444741998443960e-01, 0.1290210252363728e+00, 0.5776327498150620e+00, 0.4262359329629137e-01, 0.3402556968467140e+00, 0.6023491555328507e+00, 0.1086565805630269e+00, 0.3155392911289253e+00, 0.9382999256694983e+00, 0.1171677238805356e+00, 0.7901625299559836e+00, 0.9297887823519896e+00, 0.6095853796269167e+00, 0.7973955560628040e+00, 0.8018125008387630e+00, 0.6023523535200964e+00, 0.2443177602187348e-01, 0.8114406374558937e+00, 0.2277830378453201e+00, 0.1480363940474286e-01, 0.6969081780841352e+00, 0.3759308106650992e+00, 0.8012406122589412e-01, 0.5487604052643561e+00, 0.9781022865072954e-01, 0.1017549972519246e+00, 0.2961248981181939e-03, 0.7617929262469909e-01, 0.8408398800007548e-04, 0.3802088861182694e-03, 0.0000000000000000e+00, 0.0000000000000000e+00},
			pp:       0,
			dmin:     -0.0000000000000000,
			desig:    0.0000000000000000,
			qmax:     1.8900334961849885,
			ttype:    0,
			dmin1:    0.0000000000000000,
			dmin2:    0.0000000000000000,
			dn:       0.0000000000000000,
			dn1:      0.0000000000000000,
			dn2:      0.0000000000000000,
			g:        0.0000000000000000,
			tau:      0.0000000000000000,
			nFail:    0,
			iter:     2,
			sigma:    0.0000000000000000,
			nDiv:     40,
			zOut:     []float64{0.1648283185136998e+01, 0.1819554618764197e+01, 0.1712714336271993e+00, 0.1422045283129674e+00, 0.1510753432847732e+01, 0.1747828967872021e+01, 0.3792800633372563e+00, 0.2882727267008810e+00, 0.1328441621586708e+01, 0.1179068770857534e+01, 0.1388998759717073e+00, 0.5262264257308433e-01, 0.4466938077796418e+00, 0.1103391769983383e+01, 0.7093206047768255e+00, 0.4399127392187891e+00, 0.6843112870203156e+00, 0.8934568459457148e+00, 0.6490582981441884e+00, 0.3694451726238032e+00, 0.5085572738629487e+00, 0.1488623362931603e+00, 0.9750235054014829e-02, 0.8939776288934402e-02, 0.1364886053450573e+00, 0.1505610868415679e+00, 0.2301225778544498e-01, 0.2515816443301624e+00, 0.1646009972289452e+01, 0.1630679888773521e+01, 0.2362515608142310e+00, 0.8429943526342391e-01, 0.5818602562677768e+00, 0.5155374736992060e+00, 0.1797665269485310e-01, 0.1952851185677853e-01, 0.5600419521166516e+00, 0.7600271971854760e+00, 0.2195137569256029e+00, 0.1208458544696003e+00, 0.4184071984843414e+00, 0.5703477987440417e+00, 0.2727864547293006e+00, 0.3240053608004366e-01, 0.6774373914466536e-01, 0.1404399130020459e+00, 0.1050967099374242e+00, 0.1526865702536626e+00, 0.2040338718098096e+00, 0.1785445001043715e+00, 0.1271971985482246e+00, 0.3166486394939177e-01, 0.4444741998443960e-01, 0.5904153058501098e+00, 0.5776327498150620e+00, 0.3328891237445398e+00, 0.3402556968467140e+00, 0.1160231536652011e+00, 0.1086565805630269e+00, 0.8787251358464724e+00, 0.9382999256694983e+00, 0.8497373197790092e+00, 0.7901625299559836e+00, 0.5668475593321608e+00, 0.6095853796269167e+00, 0.8445503211335190e+00, 0.8018125008387630e+00, 0.2319542476253924e-01, 0.2443177602187348e-01, 0.2290193891046544e+00, 0.2277830378453201e+00, 0.6931459494493321e+00, 0.6969081780841352e+00, 0.8388628986069724e-01, 0.8012406122589412e-01, 0.9342352322344821e-01, 0.9781022865072954e-01, 0.4682830325399513e-02, 0.2961248981181939e-03, 0.5317160915449039e-05, 0.8408398800007548e-04, 0.7876682708462645e-04, 0.0000000000000000e+00, 0.8939776288934402e-02},
			i0Out:    1,
			n0Out:    21,
			ppOut:    0,
			dminOut:  7.8766827084626452e-005,
			desigOut: 0.0000000000000000,
			sigmaOut: 0.0000000000000000,
			qmaxOut:  1.8900334961849885,
			nFailOut: 0,
			iterOut:  3,
			nDivOut:  62,
			ttypeOut: -1,
			dmin1Out: 1.2363512593342330e-003,
			dmin2Out: 1.2363512593342330e-003,
			dnOut:    7.8766827084626452e-005,
			dn1Out:   4.3867054272813191e-003,
			dn2Out:   3.7622286348031123e-003,
			gOut:     0.0000000000000000,
			tauOut:   0.0000000000000000,
		},
		{
			i0:       1,
			n0:       21,
			z:        []float64{0.1648283185136998e+01, 0.1819554618764197e+01, 0.1712714336271993e+00, 0.1422045283129674e+00, 0.1510753432847732e+01, 0.1747828967872021e+01, 0.3792800633372563e+00, 0.2882727267008810e+00, 0.1328441621586708e+01, 0.1179068770857534e+01, 0.1388998759717073e+00, 0.5262264257308433e-01, 0.4466938077796418e+00, 0.1103391769983383e+01, 0.7093206047768255e+00, 0.4399127392187891e+00, 0.6843112870203156e+00, 0.8934568459457148e+00, 0.6490582981441884e+00, 0.3694451726238032e+00, 0.5085572738629487e+00, 0.1488623362931603e+00, 0.9750235054014829e-02, 0.8939776288934402e-02, 0.1364886053450573e+00, 0.1505610868415679e+00, 0.2301225778544498e-01, 0.2515816443301624e+00, 0.1646009972289452e+01, 0.1630679888773521e+01, 0.2362515608142310e+00, 0.8429943526342391e-01, 0.5818602562677768e+00, 0.5155374736992060e+00, 0.1797665269485310e-01, 0.1952851185677853e-01, 0.5600419521166516e+00, 0.7600271971854760e+00, 0.2195137569256029e+00, 0.1208458544696003e+00, 0.4184071984843414e+00, 0.5703477987440417e+00, 0.2727864547293006e+00, 0.3240053608004366e-01, 0.6774373914466536e-01, 0.1404399130020459e+00, 0.1050967099374242e+00, 0.1526865702536626e+00, 0.2040338718098096e+00, 0.1785445001043715e+00, 0.1271971985482246e+00, 0.3166486394939177e-01, 0.4444741998443960e-01, 0.5904153058501098e+00, 0.5776327498150620e+00, 0.3328891237445398e+00, 0.3402556968467140e+00, 0.1160231536652011e+00, 0.1086565805630269e+00, 0.8787251358464724e+00, 0.9382999256694983e+00, 0.8497373197790092e+00, 0.7901625299559836e+00, 0.5668475593321608e+00, 0.6095853796269167e+00, 0.8445503211335190e+00, 0.8018125008387630e+00, 0.2319542476253924e-01, 0.2443177602187348e-01, 0.2290193891046544e+00, 0.2277830378453201e+00, 0.6931459494493321e+00, 0.6969081780841352e+00, 0.8388628986069724e-01, 0.8012406122589412e-01, 0.9342352322344821e-01, 0.9781022865072954e-01, 0.4682830325399513e-02, 0.2961248981181939e-03, 0.5317160915449039e-05, 0.8408398800007548e-04, 0.7876682708462645e-04, 0.0000000000000000e+00, 0.8939776288934402e-02},
			pp:       1,
			dmin:     7.8766827084626452e-005,
			desig:    0.0000000000000000,
			qmax:     1.8900334961849885,
			ttype:    -1,
			dmin1:    1.2363512593342330e-003,
			dmin2:    1.2363512593342330e-003,
			dn:       7.8766827084626452e-005,
			dn1:      4.3867054272813191e-003,
			dn2:      3.7622286348031123e-003,
			g:        0.0000000000000000,
			tau:      0.0000000000000000,
			nFail:    0,
			iter:     3,
			sigma:    0.0000000000000000,
			nDiv:     62,
			zOut:     []float64{0.1961759147077164e+01, 0.1819554618764197e+01, 0.1266970995487882e+00, 0.1422045283129674e+00, 0.1909404595024114e+01, 0.1747828967872021e+01, 0.1780101349021133e+00, 0.2882727267008810e+00, 0.1053681278528505e+01, 0.1179068770857534e+01, 0.5510526941411123e-01, 0.5262264257308433e-01, 0.1488199239788061e+01, 0.1103391769983383e+01, 0.2641064704009213e+00, 0.4399127392187891e+00, 0.9987955481685968e+00, 0.8934568459457148e+00, 0.5506279200968747e-01, 0.3694451726238032e+00, 0.1027393205724072e+00, 0.1488623362931603e+00, 0.1310094739466220e-01, 0.8939776288934402e-02, 0.3890417837770681e+00, 0.1505610868415679e+00, 0.1054511738587064e+01, 0.2515816443301624e+00, 0.6604675854498806e+00, 0.1630679888773521e+01, 0.6580113672099847e-01, 0.8429943526342391e-01, 0.4692648488349861e+00, 0.5155374736992060e+00, 0.3162862116895929e-01, 0.1952851185677853e-01, 0.8492444304861170e+00, 0.7600271971854760e+00, 0.8115939841327705e-01, 0.1208458544696003e+00, 0.5215889364108083e+00, 0.5703477987440417e+00, 0.8723974284448969e-02, 0.3240053608004366e-01, 0.2844025089712595e+00, 0.1404399130020459e+00, 0.9585480612390133e-01, 0.1526865702536626e+00, 0.1143545579298620e+00, 0.1785445001043715e+00, 0.1634864466429828e+00, 0.3166486394939177e-01, 0.7598179829516669e+00, 0.5904153058501098e+00, 0.5083170815153470e-01, 0.3328891237445398e+00, 0.9439165813601388e+00, 0.1160231536652011e+00, 0.7910503496831139e+00, 0.8787251358464724e+00, 0.6255345294280562e+00, 0.8497373197790092e+00, 0.7653155273545736e+00, 0.5668475593321608e+00, 0.1024302185414846e+00, 0.8445503211335190e+00, 0.5186166821452450e-01, 0.2319542476253924e-01, 0.8703036703394620e+00, 0.2290193891046544e+00, 0.6681052144545734e-01, 0.6931459494493321e+00, 0.1104992916386881e+00, 0.8388628986069724e-01, 0.3959179295799719e-02, 0.9342352322344821e-01, 0.7289681905152429e-03, 0.4682830325399513e-02, 0.5745324691222600e-06, 0.5317160915449039e-05, 0.7819229461550419e-04, 0.7876682708462645e-04, 0.8723974284448969e-02, 0.8939776288934402e-02},
			i0Out:    1,
			n0Out:    21,
			ppOut:    1,
			dminOut:  7.8192294615504193e-005,
			desigOut: 0.0000000000000000,
			sigmaOut: 0.0000000000000000,
			qmaxOut:  1.8900334961849885,
			nFailOut: 0,
			iterOut:  4,
			nDivOut:  84,
			ttypeOut: -4,
			dmin1Out: 7.2365102959979382e-004,
			dmin2Out: 1.7075768415239889e-002,
			dnOut:    7.8192294615504193e-005,
			dn1Out:   7.2365102959979382e-004,
			dn2Out:   1.7075768415239889e-002,
			gOut:     0.0000000000000000,
			tauOut:   0.0000000000000000,
		},
		{
			i0:       1,
			n0:       21,
			z:        []float64{0.1961759147077164e+01, 0.1819554618764197e+01, 0.1266970995487882e+00, 0.1422045283129674e+00, 0.1909404595024114e+01, 0.1747828967872021e+01, 0.1780101349021133e+00, 0.2882727267008810e+00, 0.1053681278528505e+01, 0.1179068770857534e+01, 0.5510526941411123e-01, 0.5262264257308433e-01, 0.1488199239788061e+01, 0.1103391769983383e+01, 0.2641064704009213e+00, 0.4399127392187891e+00, 0.9987955481685968e+00, 0.8934568459457148e+00, 0.5506279200968747e-01, 0.3694451726238032e+00, 0.1027393205724072e+00, 0.1488623362931603e+00, 0.1310094739466220e-01, 0.8939776288934402e-02, 0.3890417837770681e+00, 0.1505610868415679e+00, 0.1054511738587064e+01, 0.2515816443301624e+00, 0.6604675854498806e+00, 0.1630679888773521e+01, 0.6580113672099847e-01, 0.8429943526342391e-01, 0.4692648488349861e+00, 0.5155374736992060e+00, 0.3162862116895929e-01, 0.1952851185677853e-01, 0.8492444304861170e+00, 0.7600271971854760e+00, 0.8115939841327705e-01, 0.1208458544696003e+00, 0.5215889364108083e+00, 0.5703477987440417e+00, 0.8723974284448969e-02, 0.3240053608004366e-01, 0.2844025089712595e+00, 0.1404399130020459e+00, 0.9585480612390133e-01, 0.1526865702536626e+00, 0.1143545579298620e+00, 0.1785445001043715e+00, 0.1634864466429828e+00, 0.3166486394939177e-01, 0.7598179829516669e+00, 0.5904153058501098e+00, 0.5083170815153470e-01, 0.3328891237445398e+00, 0.9439165813601388e+00, 0.1160231536652011e+00, 0.7910503496831139e+00, 0.8787251358464724e+00, 0.6255345294280562e+00, 0.8497373197790092e+00, 0.7653155273545736e+00, 0.5668475593321608e+00, 0.1024302185414846e+00, 0.8445503211335190e+00, 0.5186166821452450e-01, 0.2319542476253924e-01, 0.8703036703394620e+00, 0.2290193891046544e+00, 0.6681052144545734e-01, 0.6931459494493321e+00, 0.1104992916386881e+00, 0.8388628986069724e-01, 0.3959179295799719e-02, 0.9342352322344821e-01, 0.7289681905152429e-03, 0.4682830325399513e-02, 0.5745324691222600e-06, 0.5317160915449039e-05, 0.7819229461550419e-04, 0.7876682708462645e-04, 0.8723974284448969e-02, 0.8939776288934402e-02},
			pp:       0,
			dmin:     7.8192294615504193e-005,
			desig:    0.0000000000000000,
			qmax:     1.8900334961849885,
			ttype:    -4,
			dmin1:    7.2365102959979382e-004,
			dmin2:    1.7075768415239889e-002,
			dn:       7.8192294615504193e-005,
			dn1:      7.2365102959979382e-004,
			dn2:      1.7075768415239889e-002,
			g:        0.0000000000000000,
			tau:      0.0000000000000000,
			nFail:    0,
			iter:     4,
			sigma:    0.0000000000000000,
			nDiv:     84,
			zOut:     []float64{0.1961759147077164e+01, 0.2088378163269771e+01, 0.1266970995487882e+00, 0.1158391848322702e+00, 0.1909404595024114e+01, 0.1971497461737776e+01, 0.1780101349021133e+00, 0.9513882222772962e-01, 0.1053681278528505e+01, 0.1013569642358705e+01, 0.5510526941411123e-01, 0.8090970429970110e-01, 0.1488199239788061e+01, 0.1671317922533099e+01, 0.2641064704009213e+00, 0.1578325483874163e+00, 0.9987955481685968e+00, 0.8959477084346864e+00, 0.5506279200968747e-01, 0.6314111623521673e-02, 0.1027393205724072e+00, 0.1094480729873660e+00, 0.1310094739466220e-01, 0.4656834793406785e-01, 0.3890417837770681e+00, 0.1396907091073883e+01, 0.1054511738587064e+01, 0.4985806330739840e+00, 0.6604675854498806e+00, 0.2276100057407134e+00, 0.6580113672099847e-01, 0.1356625794022653e+00, 0.4692648488349861e+00, 0.3651528072454984e+00, 0.3162862116895929e-01, 0.7355942454424362e-01, 0.8492444304861170e+00, 0.8567663209989688e+00, 0.8115939841327705e-01, 0.4940885660487250e-01, 0.5215889364108083e+00, 0.4808259707342031e+00, 0.8723974284448969e-02, 0.5160120970399038e-02, 0.2844025089712595e+00, 0.3750191107685802e+00, 0.9585480612390133e-01, 0.2922900104287094e-01, 0.1143545579298620e+00, 0.2485339201737921e+00, 0.1634864466429828e+00, 0.4998108187459615e+00, 0.7598179829516669e+00, 0.3107607890010585e+00, 0.5083170815153470e-01, 0.1543981540828483e+00, 0.9439165813601388e+00, 0.1580490693604223e+01, 0.7910503496831139e+00, 0.3130858727896049e+00, 0.6255345294280562e+00, 0.1077686100636843e+01, 0.7653155273545736e+00, 0.7274051012980143e-01, 0.1024302185414846e+00, 0.8147329327002600e-01, 0.5186166821452450e-01, 0.5539901283655780e+00, 0.8703036703394620e+00, 0.3830459800631597e+00, 0.6681052144545734e-01, 0.1927318305890360e-01, 0.1104992916386881e+00, 0.9510720451940254e-01, 0.3959179295799719e-02, 0.3034592154998885e-04, 0.7289681905152429e-03, 0.6211134452527084e-03, 0.5745324691222600e-06, 0.7232819130731082e-07, 0.7819229461550419e-04, 0.3661024252896476e-07, 0.8723974284448969e-02, 0.5160120970399038e-02},
			i0Out:    1,
			n0Out:    21,
			ppOut:    0,
			dminOut:  3.6610242528964756e-008,
			desigOut: 0.0000000000000000,
			sigmaOut: 7.8083356181667918e-005,
			qmaxOut:  1.8900334961849885,
			nFailOut: 0,
			iterOut:  5,
			nDivOut:  106,
			ttypeOut: -2,
			dmin1Out: 6.2053891278358614e-004,
			dmin2Out: 2.9611625055501498e-002,
			dnOut:    3.6610242528964756e-008,
			dn1Out:   6.2053891278358614e-004,
			dn2Out:   9.1148025223602810e-002,
			gOut:     0.0000000000000000,
			tauOut:   7.8083356181667918e-005,
		},
		{
			i0:       1,
			n0:       21,
			z:        []float64{0.1961759147077164e+01, 0.2088378163269771e+01, 0.1266970995487882e+00, 0.1158391848322702e+00, 0.1909404595024114e+01, 0.1971497461737776e+01, 0.1780101349021133e+00, 0.9513882222772962e-01, 0.1053681278528505e+01, 0.1013569642358705e+01, 0.5510526941411123e-01, 0.8090970429970110e-01, 0.1488199239788061e+01, 0.1671317922533099e+01, 0.2641064704009213e+00, 0.1578325483874163e+00, 0.9987955481685968e+00, 0.8959477084346864e+00, 0.5506279200968747e-01, 0.6314111623521673e-02, 0.1027393205724072e+00, 0.1094480729873660e+00, 0.1310094739466220e-01, 0.4656834793406785e-01, 0.3890417837770681e+00, 0.1396907091073883e+01, 0.1054511738587064e+01, 0.4985806330739840e+00, 0.6604675854498806e+00, 0.2276100057407134e+00, 0.6580113672099847e-01, 0.1356625794022653e+00, 0.4692648488349861e+00, 0.3651528072454984e+00, 0.3162862116895929e-01, 0.7355942454424362e-01, 0.8492444304861170e+00, 0.8567663209989688e+00, 0.8115939841327705e-01, 0.4940885660487250e-01, 0.5215889364108083e+00, 0.4808259707342031e+00, 0.8723974284448969e-02, 0.5160120970399038e-02, 0.2844025089712595e+00, 0.3750191107685802e+00, 0.9585480612390133e-01, 0.2922900104287094e-01, 0.1143545579298620e+00, 0.2485339201737921e+00, 0.1634864466429828e+00, 0.4998108187459615e+00, 0.7598179829516669e+00, 0.3107607890010585e+00, 0.5083170815153470e-01, 0.1543981540828483e+00, 0.9439165813601388e+00, 0.1580490693604223e+01, 0.7910503496831139e+00, 0.3130858727896049e+00, 0.6255345294280562e+00, 0.1077686100636843e+01, 0.7653155273545736e+00, 0.7274051012980143e-01, 0.1024302185414846e+00, 0.8147329327002600e-01, 0.5186166821452450e-01, 0.5539901283655780e+00, 0.8703036703394620e+00, 0.3830459800631597e+00, 0.6681052144545734e-01, 0.1927318305890360e-01, 0.1104992916386881e+00, 0.9510720451940254e-01, 0.3959179295799719e-02, 0.3034592154998885e-04, 0.7289681905152429e-03, 0.6211134452527084e-03, 0.5745324691222600e-06, 0.7232819130731082e-07, 0.7819229461550419e-04, 0.3661024252896476e-07, 0.8723974284448969e-02, 0.5160120970399038e-02},
			pp:       1,
			dmin:     3.6610242528964756e-008,
			desig:    0.0000000000000000,
			qmax:     1.8900334961849885,
			ttype:    -2,
			dmin1:    6.2053891278358614e-004,
			dmin2:    2.9611625055501498e-002,
			dn:       3.6610242528964756e-008,
			dn1:      6.2053891278358614e-004,
			dn2:      9.1148025223602810e-002,
			g:        0.0000000000000000,
			tau:      7.8083356181667918e-005,
			nFail:    0,
			iter:     5,
			sigma:    7.8083356181667918e-005,
			nDiv:     106,
			zOut:     []float64{0.2204217311496068e+01, 0.2088378163269771e+01, 0.1036089580076783e+00, 0.1158391848322702e+00, 0.1963027289351853e+01, 0.1971497461737776e+01, 0.4912301654839819e-01, 0.9513882222772962e-01, 0.1045356293504034e+01, 0.1013569642358705e+01, 0.1293586117415210e+00, 0.8090970429970110e-01, 0.1699791822573021e+01, 0.1671317922533099e+01, 0.8319236989271818e-01, 0.1578325483874163e+00, 0.8190694135595162e+00, 0.8959477084346864e+00, 0.8437225690290790e-03, 0.6314111623521673e-02, 0.1551726617464312e+00, 0.1094480729873660e+00, 0.4192211096758560e+00, 0.4656834793406785e-01, 0.1476266577866037e+01, 0.1396907091073883e+01, 0.7687090018675194e-01, 0.4985806330739840e+00, 0.2864016483502532e+00, 0.2276100057407134e+00, 0.1729653861709652e+00, 0.1356625794022653e+00, 0.2657468090128033e+00, 0.3651528072454984e+00, 0.2371551996266359e+00, 0.7355942454424362e-01, 0.6690199413712319e+00, 0.8567663209989688e+00, 0.3551024412099299e-01, 0.4940885660487250e-01, 0.4504758109776356e+00, 0.4808259707342031e+00, 0.4295777776785939e-02, 0.5160120970399038e-02, 0.3999522974286917e+00, 0.3750191107685802e+00, 0.1816316160365039e-01, 0.2922900104287094e-01, 0.7301815407101296e+00, 0.2485339201737921e+00, 0.2127164214993764e+00, 0.4998108187459615e+00, 0.2524424849785568e+00, 0.3107607890010585e+00, 0.9666552191416621e+00, 0.1543981540828483e+00, 0.9269213106461920e+00, 0.1580490693604223e+01, 0.3640096408786760e+00, 0.3130858727896049e+00, 0.7864169332819952e+00, 0.1077686100636843e+01, 0.7535963003344312e-02, 0.7274051012980143e-01, 0.6279274220262862e+00, 0.8147329327002600e-01, 0.3379430237022286e+00, 0.5539901283655780e+00, 0.6437610281386123e-01, 0.3830459800631597e+00, 0.2847358698029713e-01, 0.1927318305890360e-01, 0.6666392685468185e-01, 0.9510720451940254e-01, 0.2827355178816361e-06, 0.3034592154998885e-04, 0.6208664319525956e-03, 0.6211134452527084e-03, 0.4264931214133004e-11, 0.7232819130731082e-07, 0.4059351646211308e-14, 0.3661024252896476e-07, 0.8437225690290790e-03, 0.5160120970399038e-02},
			i0Out:    1,
			n0Out:    21,
			ppOut:    1,
			dminOut:  4.0593516462113082e-015,
			desigOut: 4.3344264097778650e-021,
			sigmaOut: 7.8119962155206313e-005,
			qmaxOut:  1.8900334961849885,
			nFailOut: 0,
			iterOut:  6,
			nDivOut:  128,
			ttypeOut: -2,
			dmin1Out: 6.2079410376128833e-004,
			dmin2Out: 4.5102919754957636e-002,
			dnOut:    4.0593516462113082e-015,
			dn1Out:   6.2079410376128833e-004,
			dn2Out:   6.6633580933131861e-002,
			gOut:     0.0000000000000000,
			tauOut:   3.6605973538398975e-008,
		},
		{
			i0:       1,
			n0:       21,
			z:        []float64{0.2204217311496068e+01, 0.2088378163269771e+01, 0.1036089580076783e+00, 0.1158391848322702e+00, 0.1963027289351853e+01, 0.1971497461737776e+01, 0.4912301654839819e-01, 0.9513882222772962e-01, 0.1045356293504034e+01, 0.1013569642358705e+01, 0.1293586117415210e+00, 0.8090970429970110e-01, 0.1699791822573021e+01, 0.1671317922533099e+01, 0.8319236989271818e-01, 0.1578325483874163e+00, 0.8190694135595162e+00, 0.8959477084346864e+00, 0.8437225690290790e-03, 0.6314111623521673e-02, 0.1551726617464312e+00, 0.1094480729873660e+00, 0.4192211096758560e+00, 0.4656834793406785e-01, 0.1476266577866037e+01, 0.1396907091073883e+01, 0.7687090018675194e-01, 0.4985806330739840e+00, 0.2864016483502532e+00, 0.2276100057407134e+00, 0.1729653861709652e+00, 0.1356625794022653e+00, 0.2657468090128033e+00, 0.3651528072454984e+00, 0.2371551996266359e+00, 0.7355942454424362e-01, 0.6690199413712319e+00, 0.8567663209989688e+00, 0.3551024412099299e-01, 0.4940885660487250e-01, 0.4504758109776356e+00, 0.4808259707342031e+00, 0.4295777776785939e-02, 0.5160120970399038e-02, 0.3999522974286917e+00, 0.3750191107685802e+00, 0.1816316160365039e-01, 0.2922900104287094e-01, 0.7301815407101296e+00, 0.2485339201737921e+00, 0.2127164214993764e+00, 0.4998108187459615e+00, 0.2524424849785568e+00, 0.3107607890010585e+00, 0.9666552191416621e+00, 0.1543981540828483e+00, 0.9269213106461920e+00, 0.1580490693604223e+01, 0.3640096408786760e+00, 0.3130858727896049e+00, 0.7864169332819952e+00, 0.1077686100636843e+01, 0.7535963003344312e-02, 0.7274051012980143e-01, 0.6279274220262862e+00, 0.8147329327002600e-01, 0.3379430237022286e+00, 0.5539901283655780e+00, 0.6437610281386123e-01, 0.3830459800631597e+00, 0.2847358698029713e-01, 0.1927318305890360e-01, 0.6666392685468185e-01, 0.9510720451940254e-01, 0.2827355178816361e-06, 0.3034592154998885e-04, 0.6208664319525956e-03, 0.6211134452527084e-03, 0.4264931214133004e-11, 0.7232819130731082e-07, 0.4059351646211308e-14, 0.3661024252896476e-07, 0.8437225690290790e-03, 0.5160120970399038e-02},
			pp:       0,
			dmin:     4.0593516462113082e-015,
			desig:    4.3344264097778650e-021,
			qmax:     1.8900334961849885,
			ttype:    -2,
			dmin1:    6.2079410376128833e-004,
			dmin2:    4.5102919754957636e-002,
			dn:       4.0593516462113082e-015,
			dn1:      6.2079410376128833e-004,
			dn2:      6.6633580933131861e-002,
			g:        0.0000000000000000,
			tau:      3.6605973538398975e-008,
			nFail:    0,
			iter:     6,
			sigma:    7.8119962155206313e-005,
			nDiv:     128,
			zOut:     []float64{0.2204217311496068e+01, 0.2307826269503742e+01, 0.1036089580076783e+00, 0.8812934261040264e-01, 0.1963027289351853e+01, 0.1924020963289845e+01, 0.4912301654839819e-01, 0.2668944646890268e-01, 0.1045356293504034e+01, 0.1148025458776649e+01, 0.1293586117415210e+00, 0.1915312145184878e+00, 0.1699791822573021e+01, 0.1591452977947247e+01, 0.8319236989271818e-01, 0.4281642408847453e-01, 0.8190694135595162e+00, 0.7770967120400667e+00, 0.8437225690290790e-03, 0.1684766835109566e-03, 0.1551726617464312e+00, 0.5742252947387723e+00, 0.4192211096758560e+00, 0.1077768810640642e+01, 0.1476266577866037e+01, 0.4753686674121431e+00, 0.7687090018675194e-01, 0.4631342794110950e-01, 0.2864016483502532e+00, 0.4130536065801048e+00, 0.1729653861709652e+00, 0.1112809541240190e+00, 0.2657468090128033e+00, 0.3916210545154161e+00, 0.2371551996266359e+00, 0.4051405202062471e+00, 0.6690199413712319e+00, 0.2993896652859737e+00, 0.3551024412099299e-01, 0.5343038813025977e-01, 0.4504758109776356e+00, 0.4013412006241577e+00, 0.4295777776785939e-02, 0.4280911574482486e-02, 0.3999522974286917e+00, 0.4138345474578556e+00, 0.1816316160365039e-01, 0.3204760309498118e-01, 0.7301815407101296e+00, 0.9108503591145208e+00, 0.2127164214993764e+00, 0.5895442813598011e-01, 0.2524424849785568e+00, 0.1160143275984235e+01, 0.9666552191416621e+00, 0.7723298847804961e+00, 0.9269213106461920e+00, 0.5186010667443678e+00, 0.3640096408786760e+00, 0.5519914319921668e+00, 0.7864169332819952e+00, 0.2419614642931688e+00, 0.7535963003344312e-02, 0.1955698951896722e-01, 0.6279274220262862e+00, 0.9463134562095434e+00, 0.3379430237022286e+00, 0.2298969194226957e-01, 0.6437610281386123e-01, 0.6985999785188474e-01, 0.2847358698029713e-01, 0.2717093011896993e-01, 0.6666392685468185e-01, 0.3949327947122574e-01, 0.2827355178816361e-06, 0.4444831994804014e-08, 0.6208664319525956e-03, 0.6208619913814727e-03, 0.4264931214133004e-11, 0.2788519153273688e-22, 0.4059351646211308e-14, 0.3786532345060857e-28, 0.8437225690290790e-03, 0.1684766835109566e-03},
			i0Out:    1,
			n0Out:    21,
			ppOut:    0,
			dminOut:  3.7865323450608567e-029,
			desigOut: -5.4012504887129591e-021,
			sigmaOut: 7.8119962159265674e-005,
			qmaxOut:  1.8900334961849885,
			nFailOut: 0,
			iterOut:  7,
			nDivOut:  150,
			ttypeOut: -2,
			dmin1Out: 6.2086198711654151e-004,
			dmin2Out: 3.9492996735707858e-002,
			dnOut:    3.7865323450608567e-029,
			dn1Out:   6.2086198711654151e-004,
			dn2Out:   3.9492996735707858e-002,
			gOut:     0.0000000000000000,
			tauOut:   4.0593516183260787e-015,
		},
		{
			i0:       1,
			n0:       21,
			z:        []float64{0.2204217311496068e+01, 0.2307826269503742e+01, 0.1036089580076783e+00, 0.8812934261040264e-01, 0.1963027289351853e+01, 0.1924020963289845e+01, 0.4912301654839819e-01, 0.2668944646890268e-01, 0.1045356293504034e+01, 0.1148025458776649e+01, 0.1293586117415210e+00, 0.1915312145184878e+00, 0.1699791822573021e+01, 0.1591452977947247e+01, 0.8319236989271818e-01, 0.4281642408847453e-01, 0.8190694135595162e+00, 0.7770967120400667e+00, 0.8437225690290790e-03, 0.1684766835109566e-03, 0.1551726617464312e+00, 0.5742252947387723e+00, 0.4192211096758560e+00, 0.1077768810640642e+01, 0.1476266577866037e+01, 0.4753686674121431e+00, 0.7687090018675194e-01, 0.4631342794110950e-01, 0.2864016483502532e+00, 0.4130536065801048e+00, 0.1729653861709652e+00, 0.1112809541240190e+00, 0.2657468090128033e+00, 0.3916210545154161e+00, 0.2371551996266359e+00, 0.4051405202062471e+00, 0.6690199413712319e+00, 0.2993896652859737e+00, 0.3551024412099299e-01, 0.5343038813025977e-01, 0.4504758109776356e+00, 0.4013412006241577e+00, 0.4295777776785939e-02, 0.4280911574482486e-02, 0.3999522974286917e+00, 0.4138345474578556e+00, 0.1816316160365039e-01, 0.3204760309498118e-01, 0.7301815407101296e+00, 0.9108503591145208e+00, 0.2127164214993764e+00, 0.5895442813598011e-01, 0.2524424849785568e+00, 0.1160143275984235e+01, 0.9666552191416621e+00, 0.7723298847804961e+00, 0.9269213106461920e+00, 0.5186010667443678e+00, 0.3640096408786760e+00, 0.5519914319921668e+00, 0.7864169332819952e+00, 0.2419614642931688e+00, 0.7535963003344312e-02, 0.1955698951896722e-01, 0.6279274220262862e+00, 0.9463134562095434e+00, 0.3379430237022286e+00, 0.2298969194226957e-01, 0.6437610281386123e-01, 0.6985999785188474e-01, 0.2847358698029713e-01, 0.2717093011896993e-01, 0.6666392685468185e-01, 0.3949327947122574e-01, 0.2827355178816361e-06, 0.4444831994804014e-08, 0.6208664319525956e-03, 0.6208619913814727e-03, 0.4264931214133004e-11, 0.2788519153273688e-22, 0.4059351646211308e-14, 0.3786532345060857e-28, 0.8437225690290790e-03, 0.1684766835109566e-03},
			pp:       1,
			dmin:     3.7865323450608567e-029,
			desig:    -5.4012504887129591e-021,
			qmax:     1.8900334961849885,
			ttype:    -2,
			dmin1:    6.2086198711654151e-004,
			dmin2:    3.9492996735707858e-002,
			dn:       3.7865323450608567e-029,
			dn1:      6.2086198711654151e-004,
			dn2:      3.9492996735707858e-002,
			g:        0.0000000000000000,
			tau:      4.0593516183260787e-015,
			nFail:    0,
			iter:     7,
			sigma:    7.8119962159265674e-005,
			nDiv:     150,
			zOut:     []float64{0.2395955612114145e+01, 0.2307826269503742e+01, 0.7077038564739886e-01, 0.8812934261040264e-01, 0.1879940024111348e+01, 0.1924020963289845e+01, 0.1629847954401656e-01, 0.2668944646890268e-01, 0.1323258193751120e+01, 0.1148025458776649e+01, 0.2303502998543534e+00, 0.1915312145184878e+00, 0.1403919102181368e+01, 0.1591452977947247e+01, 0.2369972908607684e-01, 0.4281642408847453e-01, 0.7535654596375009e+00, 0.7770967120400667e+00, 0.1283811140869274e-03, 0.1684766835109566e-03, 0.1651865724265327e+01, 0.5742252947387723e+00, 0.3101568824672333e+00, 0.1077768810640642e+01, 0.2115252128860193e+00, 0.4753686674121431e+00, 0.9043805314343908e-01, 0.4631342794110950e-01, 0.4338965075606848e+00, 0.4130536065801048e+00, 0.1004386157577793e+00, 0.1112809541240190e+00, 0.6963229589638840e+00, 0.3916210545154161e+00, 0.1741934301847783e+00, 0.4051405202062471e+00, 0.1786266232314551e+00, 0.2993896652859737e+00, 0.1200482645536405e+00, 0.5343038813025977e-01, 0.2855738476449996e+00, 0.4013412006241577e+00, 0.6203611145567293e-02, 0.4280911574482486e-02, 0.4396785394072695e+00, 0.4138345474578556e+00, 0.6639071087521134e-01, 0.3204760309498118e-01, 0.9034140763752896e+00, 0.9108503591145208e+00, 0.7570790092830114e-01, 0.5895442813598011e-01, 0.1856765259836430e+01, 0.1160143275984235e+01, 0.2157144528657351e+00, 0.7723298847804961e+00, 0.8548780458707995e+00, 0.5186010667443678e+00, 0.1562335771835849e+00, 0.5519914319921668e+00, 0.1052848766285511e+00, 0.2419614642931688e+00, 0.1757806338135458e+00, 0.1955698951896722e-01, 0.7935225143382671e+00, 0.9463134562095434e+00, 0.2023962522401480e-02, 0.2298969194226957e-01, 0.9500696544845319e-01, 0.6985999785188474e-01, 0.1129463646814219e-01, 0.2717093011896993e-01, 0.2819864744791555e-01, 0.3949327947122574e-01, 0.9786381594178541e-10, 0.4444831994804014e-08, 0.6208618935176568e-03, 0.6208619913814727e-03, 0.1700670967075909e-47, 0.2788519153273688e-22, 0.3786532345060857e-28, 0.3786532345060857e-28, 0.1283811140869274e-03, 0.1684766835109566e-03},
			i0Out:    1,
			n0Out:    21,
			ppOut:    1,
			dminOut:  3.7865323450608567e-029,
			desigOut: -5.4012504887129591e-021,
			sigmaOut: 7.8119962159265674e-005,
			qmaxOut:  1.8900334961849885,
			nFailOut: 0,
			iterOut:  8,
			nDivOut:  172,
			ttypeOut: -2,
			dmin1Out: 6.2086189351765679e-004,
			dmin2Out: 2.8198643003083550e-002,
			dnOut:    3.7865323450608567e-029,
			dn1Out:   6.2086189351765679e-004,
			dn2Out:   2.8198643003083550e-002,
			gOut:     0.0000000000000000,
			tauOut:   0.0000000000000000,
		},
		{
			i0:       1,
			n0:       21,
			z:        []float64{0.2395955612114145e+01, 0.2307826269503742e+01, 0.7077038564739886e-01, 0.8812934261040264e-01, 0.1879940024111348e+01, 0.1924020963289845e+01, 0.1629847954401656e-01, 0.2668944646890268e-01, 0.1323258193751120e+01, 0.1148025458776649e+01, 0.2303502998543534e+00, 0.1915312145184878e+00, 0.1403919102181368e+01, 0.1591452977947247e+01, 0.2369972908607684e-01, 0.4281642408847453e-01, 0.7535654596375009e+00, 0.7770967120400667e+00, 0.1283811140869274e-03, 0.1684766835109566e-03, 0.1651865724265327e+01, 0.5742252947387723e+00, 0.3101568824672333e+00, 0.1077768810640642e+01, 0.2115252128860193e+00, 0.4753686674121431e+00, 0.9043805314343908e-01, 0.4631342794110950e-01, 0.4338965075606848e+00, 0.4130536065801048e+00, 0.1004386157577793e+00, 0.1112809541240190e+00, 0.6963229589638840e+00, 0.3916210545154161e+00, 0.1741934301847783e+00, 0.4051405202062471e+00, 0.1786266232314551e+00, 0.2993896652859737e+00, 0.1200482645536405e+00, 0.5343038813025977e-01, 0.2855738476449996e+00, 0.4013412006241577e+00, 0.6203611145567293e-02, 0.4280911574482486e-02, 0.4396785394072695e+00, 0.4138345474578556e+00, 0.6639071087521134e-01, 0.3204760309498118e-01, 0.9034140763752896e+00, 0.9108503591145208e+00, 0.7570790092830114e-01, 0.5895442813598011e-01, 0.1856765259836430e+01, 0.1160143275984235e+01, 0.2157144528657351e+00, 0.7723298847804961e+00, 0.8548780458707995e+00, 0.5186010667443678e+00, 0.1562335771835849e+00, 0.5519914319921668e+00, 0.1052848766285511e+00, 0.2419614642931688e+00, 0.1757806338135458e+00, 0.1955698951896722e-01, 0.7935225143382671e+00, 0.9463134562095434e+00, 0.2023962522401480e-02, 0.2298969194226957e-01, 0.9500696544845319e-01, 0.6985999785188474e-01, 0.1129463646814219e-01, 0.2717093011896993e-01, 0.2819864744791555e-01, 0.3949327947122574e-01, 0.9786381594178541e-10, 0.4444831994804014e-08, 0.6208618935176568e-03, 0.6208619913814727e-03, 0.1700670967075909e-47, 0.2788519153273688e-22, 0.3786532345060857e-28, 0.3786532345060857e-28, 0.1283811140869274e-03, 0.1684766835109566e-03},
			pp:       0,
			dmin:     3.7865323450608567e-029,
			desig:    -5.4012504887129591e-021,
			qmax:     1.8900334961849885,
			ttype:    -2,
			dmin1:    6.2086189351765679e-004,
			dmin2:    2.8198643003083550e-002,
			dn:       3.7865323450608567e-029,
			dn1:      6.2086189351765679e-004,
			dn2:      2.8198643003083550e-002,
			g:        0.0000000000000000,
			tau:      0.0000000000000000,
			nFail:    0,
			iter:     8,
			sigma:    7.8119962159265674e-005,
			nDiv:     172,
			zOut:     []float64{0.2395955612114145e+01, 0.2466725997761544e+01, 0.7077038564739886e-01, 0.5393549207373363e-01, 0.1879940024111348e+01, 0.1842303011581631e+01, 0.1629847954401656e-01, 0.1170659574821484e-01, 0.1323258193751120e+01, 0.1541901897857258e+01, 0.2303502998543534e+00, 0.2097365510789915e+00, 0.1403919102181368e+01, 0.1217882280188454e+01, 0.2369972908607684e-01, 0.1466422291592108e-01, 0.7535654596375009e+00, 0.7390296178356669e+00, 0.1283811140869274e-03, 0.2869551596920546e-03, 0.1651865724265327e+01, 0.1961735651572868e+01, 0.3101568824672333e+00, 0.3344283443049241e-01, 0.2115252128860193e+00, 0.2685204315989660e+00, 0.9043805314343908e-01, 0.1461369444993732e+00, 0.4338965075606848e+00, 0.3881981788190908e+00, 0.1004386157577793e+00, 0.1801598202532681e+00, 0.6963229589638840e+00, 0.6903565688953941e+00, 0.1741934301847783e+00, 0.4507175802324546e-01, 0.1786266232314551e+00, 0.2536031297618502e+00, 0.1200482645536405e+00, 0.1351822623162481e+00, 0.2855738476449996e+00, 0.1565951964743188e+00, 0.6203611145567293e-02, 0.1741812487831326e-01, 0.4396785394072695e+00, 0.4886511254041676e+00, 0.6639071087521134e-01, 0.1227425859208231e+00, 0.9034140763752896e+00, 0.8563793913827676e+00, 0.7570790092830114e-01, 0.1641466407918003e+00, 0.1856765259836430e+01, 0.1908333071910365e+01, 0.2157144528657351e+00, 0.9663383852973971e-01, 0.8548780458707995e+00, 0.9144777845246447e+00, 0.1562335771835849e+00, 0.1798735100772441e-01, 0.1052848766285511e+00, 0.2630781594343725e+00, 0.1757806338135458e+00, 0.5302070335887964e+00, 0.7935225143382671e+00, 0.2653394432718723e+00, 0.2023962522401480e-02, 0.7246963929058098e-03, 0.9500696544845319e-01, 0.1055769055236896e+00, 0.1129463646814219e-01, 0.3016696409481782e-02, 0.2819864744791555e-01, 0.2518195113629758e-01, 0.9786381594178541e-10, 0.2412835834031154e-11, 0.6208618935176568e-03, 0.6208618911048210e-03, 0.1700670967075909e-47, 0.2869551596920546e-03, 0.7811996215926567e-04, 0.3786532345060857e-28, 0.1283811140869274e-03, 0.1684766835109566e-03},
			i0Out:    1,
			n0Out:    20,
			ppOut:    0,
			dminOut:  6.2086189110482101e-004,
			desigOut: -5.4012504887129591e-021,
			sigmaOut: 7.8119962159265674e-005,
			qmaxOut:  1.8900334961849885,
			nFailOut: 0,
			iterOut:  9,
			nDivOut:  193,
			ttypeOut: -7,
			dmin1Out: 2.5181951038433764e-002,
			dmin2Out: 8.7297525620826724e-002,
			dnOut:    6.2086189110482101e-004,
			dn1Out:   2.5181951038433764e-002,
			dn2Out:   9.4282269055547374e-002,
			gOut:     0.0000000000000000,
			tauOut:   0.0000000000000000,
		},
		{
			i0:       1,
			n0:       20,
			z:        []float64{0.2395955612114145e+01, 0.2466725997761544e+01, 0.7077038564739886e-01, 0.5393549207373363e-01, 0.1879940024111348e+01, 0.1842303011581631e+01, 0.1629847954401656e-01, 0.1170659574821484e-01, 0.1323258193751120e+01, 0.1541901897857258e+01, 0.2303502998543534e+00, 0.2097365510789915e+00, 0.1403919102181368e+01, 0.1217882280188454e+01, 0.2369972908607684e-01, 0.1466422291592108e-01, 0.7535654596375009e+00, 0.7390296178356669e+00, 0.1283811140869274e-03, 0.2869551596920546e-03, 0.1651865724265327e+01, 0.1961735651572868e+01, 0.3101568824672333e+00, 0.3344283443049241e-01, 0.2115252128860193e+00, 0.2685204315989660e+00, 0.9043805314343908e-01, 0.1461369444993732e+00, 0.4338965075606848e+00, 0.3881981788190908e+00, 0.1004386157577793e+00, 0.1801598202532681e+00, 0.6963229589638840e+00, 0.6903565688953941e+00, 0.1741934301847783e+00, 0.4507175802324546e-01, 0.1786266232314551e+00, 0.2536031297618502e+00, 0.1200482645536405e+00, 0.1351822623162481e+00, 0.2855738476449996e+00, 0.1565951964743188e+00, 0.6203611145567293e-02, 0.1741812487831326e-01, 0.4396785394072695e+00, 0.4886511254041676e+00, 0.6639071087521134e-01, 0.1227425859208231e+00, 0.9034140763752896e+00, 0.8563793913827676e+00, 0.7570790092830114e-01, 0.1641466407918003e+00, 0.1856765259836430e+01, 0.1908333071910365e+01, 0.2157144528657351e+00, 0.9663383852973971e-01, 0.8548780458707995e+00, 0.9144777845246447e+00, 0.1562335771835849e+00, 0.1798735100772441e-01, 0.1052848766285511e+00, 0.2630781594343725e+00, 0.1757806338135458e+00, 0.5302070335887964e+00, 0.7935225143382671e+00, 0.2653394432718723e+00, 0.2023962522401480e-02, 0.7246963929058098e-03, 0.9500696544845319e-01, 0.1055769055236896e+00, 0.1129463646814219e-01, 0.3016696409481782e-02, 0.2819864744791555e-01, 0.2518195113629758e-01, 0.9786381594178541e-10, 0.2412835834031154e-11, 0.6208618935176568e-03, 0.6208618911048210e-03, 0.1700670967075909e-47, 0.2869551596920546e-03, 0.7811996215926567e-04, 0.3786532345060857e-28, 0.1283811140869274e-03, 0.1684766835109566e-03},
			pp:       1,
			dmin:     6.2086189110482101e-004,
			desig:    -5.4012504887129591e-021,
			qmax:     1.8900334961849885,
			ttype:    -7,
			dmin1:    2.5181951038433764e-002,
			dmin2:    8.7297525620826724e-002,
			dn:       6.2086189110482101e-004,
			dn1:      2.5181951038433764e-002,
			dn2:      9.4282269055547374e-002,
			g:        0.0000000000000000,
			tau:      0.0000000000000000,
			nFail:    0,
			iter:     9,
			sigma:    7.8119962159265674e-005,
			nDiv:     193,
			zOut:     []float64{0.2520040627944239e+01, 0.2466725997761544e+01, 0.3943012599746679e-01, 0.5393549207373363e-01, 0.1813958619441340e+01, 0.1842303011581631e+01, 0.9950845630193767e-02, 0.1170659574821484e-01, 0.1741066741415017e+01, 0.1541901897857258e+01, 0.1467114516582776e+00, 0.2097365510789915e+00, 0.1085214189555058e+01, 0.1217882280188454e+01, 0.9986318978978259e-02, 0.1466422291592108e-01, 0.7287093921253419e+00, 0.7390296178356669e+00, 0.7725029665513934e-03, 0.2869551596920546e-03, 0.1993785121145770e+01, 0.1961735651572868e+01, 0.4504038193447841e-02, 0.3344283443049241e-01, 0.4095324760138526e+00, 0.2685204315989660e+00, 0.1385240464077977e+00, 0.1461369444993732e+00, 0.4292130907735224e+00, 0.3881981788190908e+00, 0.2897733504323247e+00, 0.1801598202532681e+00, 0.4450341145952761e+00, 0.6903565688953941e+00, 0.2568418582687495e-01, 0.4507175802324546e-01, 0.3624803443601846e+00, 0.2536031297618502e+00, 0.5840011260368079e-01, 0.1351822623162481e+00, 0.1149923468579125e+00, 0.1565951964743188e+00, 0.7401698075381481e-01, 0.1741812487831326e-01, 0.5367558686801371e+00, 0.4886511254041676e+00, 0.1958324578473516e+00, 0.1227425859208231e+00, 0.8240727124361776e+00, 0.8563793913827676e+00, 0.3801199318200257e+00, 0.1641466407918003e+00, 0.1624226116729040e+01, 0.1908333071910365e+01, 0.5440714051978934e-01, 0.9663383852973971e-01, 0.8774371331215411e+00, 0.9144777845246447e+00, 0.5393069221241477e-02, 0.1798735100772441e-01, 0.7872712619108886e+00, 0.2630781594343725e+00, 0.1786993199393658e+00, 0.5302070335887964e+00, 0.8674395783437358e-01, 0.2653394432718723e+00, 0.8820349511059155e-03, 0.7246963929058098e-03, 0.1070907050910267e+00, 0.1055769055236896e+00, 0.7093640994523618e-03, 0.3016696409481782e-02, 0.2385172514821930e-01, 0.2518195113629758e-01, 0.6280626703238275e-13, 0.2412835834031154e-11, 0.3257377007015450e-14, 0.6208618911048210e-03, 0.7725029665513934e-03, 0.2869551596920546e-03, 0.7811996215926567e-04, 0.3786532345060857e-28, 0.1283811140869274e-03, 0.1684766835109566e-03},
			i0Out:    1,
			n0Out:    20,
			ppOut:    1,
			dminOut:  3.2573770070154495e-015,
			desigOut: 2.1703803823424652e-020,
			sigmaOut: 6.9898185319802297e-004,
			qmaxOut:  1.8900334961849885,
			nFailOut: 0,
			iterOut:  10,
			nDivOut:  214,
			ttypeOut: -2,
			dmin1Out: 2.3851725145806461e-002,
			dmin2Out: 8.6019261441467765e-002,
			dnOut:    3.2573770070154495e-015,
			dn1Out:   2.3851725145806461e-002,
			dn2Out:   0.10407400868154487,
			gOut:     0.0000000000000000,
			tauOut:   6.2086189103875732e-004,
		},
		{
			i0:       1,
			n0:       20,
			z:        []float64{0.2520040627944239e+01, 0.2466725997761544e+01, 0.3943012599746679e-01, 0.5393549207373363e-01, 0.1813958619441340e+01, 0.1842303011581631e+01, 0.9950845630193767e-02, 0.1170659574821484e-01, 0.1741066741415017e+01, 0.1541901897857258e+01, 0.1467114516582776e+00, 0.2097365510789915e+00, 0.1085214189555058e+01, 0.1217882280188454e+01, 0.9986318978978259e-02, 0.1466422291592108e-01, 0.7287093921253419e+00, 0.7390296178356669e+00, 0.7725029665513934e-03, 0.2869551596920546e-03, 0.1993785121145770e+01, 0.1961735651572868e+01, 0.4504038193447841e-02, 0.3344283443049241e-01, 0.4095324760138526e+00, 0.2685204315989660e+00, 0.1385240464077977e+00, 0.1461369444993732e+00, 0.4292130907735224e+00, 0.3881981788190908e+00, 0.2897733504323247e+00, 0.1801598202532681e+00, 0.4450341145952761e+00, 0.6903565688953941e+00, 0.2568418582687495e-01, 0.4507175802324546e-01, 0.3624803443601846e+00, 0.2536031297618502e+00, 0.5840011260368079e-01, 0.1351822623162481e+00, 0.1149923468579125e+00, 0.1565951964743188e+00, 0.7401698075381481e-01, 0.1741812487831326e-01, 0.5367558686801371e+00, 0.4886511254041676e+00, 0.1958324578473516e+00, 0.1227425859208231e+00, 0.8240727124361776e+00, 0.8563793913827676e+00, 0.3801199318200257e+00, 0.1641466407918003e+00, 0.1624226116729040e+01, 0.1908333071910365e+01, 0.5440714051978934e-01, 0.9663383852973971e-01, 0.8774371331215411e+00, 0.9144777845246447e+00, 0.5393069221241477e-02, 0.1798735100772441e-01, 0.7872712619108886e+00, 0.2630781594343725e+00, 0.1786993199393658e+00, 0.5302070335887964e+00, 0.8674395783437358e-01, 0.2653394432718723e+00, 0.8820349511059155e-03, 0.7246963929058098e-03, 0.1070907050910267e+00, 0.1055769055236896e+00, 0.7093640994523618e-03, 0.3016696409481782e-02, 0.2385172514821930e-01, 0.2518195113629758e-01, 0.6280626703238275e-13, 0.2412835834031154e-11, 0.3257377007015450e-14, 0.6208618911048210e-03, 0.7725029665513934e-03, 0.2869551596920546e-03, 0.7811996215926567e-04, 0.3786532345060857e-28, 0.1283811140869274e-03, 0.1684766835109566e-03},
			pp:       0,
			dmin:     3.2573770070154495e-015,
			desig:    2.1703803823424652e-020,
			qmax:     1.8900334961849885,
			ttype:    -2,
			dmin1:    2.3851725145806461e-002,
			dmin2:    8.6019261441467765e-002,
			dn:       3.2573770070154495e-015,
			dn1:      2.3851725145806461e-002,
			dn2:      0.10407400868154487,
			g:        0.0000000000000000,
			tau:      6.2086189103875732e-004,
			nFail:    0,
			iter:     10,
			sigma:    6.9898185319802297e-004,
			nDiv:     214,
			zOut:     []float64{0.2520040627944239e+01, 0.2559470753941703e+01, 0.3943012599746679e-01, 0.2794508075882943e-01, 0.1813958619441340e+01, 0.1795964384312701e+01, 0.9950845630193767e-02, 0.9646675917972324e-02, 0.1741066741415017e+01, 0.1878131517155319e+01, 0.1467114516582776e+00, 0.8477220453173254e-01, 0.1085214189555058e+01, 0.1010428304002300e+01, 0.9986318978978259e-02, 0.7202019583097946e-02, 0.7287093921253419e+00, 0.7222798755087921e+00, 0.7725029665513934e-03, 0.2132421202606784e-02, 0.1993785121145770e+01, 0.1996156738136608e+01, 0.4504038193447841e-02, 0.9240506410060376e-03, 0.4095324760138526e+00, 0.5471324717806409e+00, 0.1385240464077977e+00, 0.1086689918286978e+00, 0.4292130907735224e+00, 0.6103174493771462e+00, 0.2897733504323247e+00, 0.2112982785836522e+00, 0.4450341145952761e+00, 0.2594200218384955e+00, 0.2568418582687495e-01, 0.3588779484774172e-01, 0.3624803443601846e+00, 0.3849926621161204e+00, 0.5840011260368079e-01, 0.1744336104525046e-01, 0.1149923468579125e+00, 0.1715659665664736e+00, 0.7401698075381481e-01, 0.2315671901408358e+00, 0.5367558686801371e+00, 0.5010211363866497e+00, 0.1958324578473516e+00, 0.3221025481782661e+00, 0.8240727124361776e+00, 0.8820900960779340e+00, 0.3801199318200257e+00, 0.6999293195746292e+00, 0.1624226116729040e+01, 0.9787039376741966e+00, 0.5440714051978934e-01, 0.4877761656142095e-01, 0.8774371331215411e+00, 0.8340525857813585e+00, 0.5393069221241477e-02, 0.5090576402208483e-02, 0.7872712619108886e+00, 0.9608800054480426e+00, 0.1786993199393658e+00, 0.1613217694817542e-01, 0.8674395783437358e-01, 0.7149381583730083e-01, 0.8820349511059155e-03, 0.1321201613351008e-02, 0.1070907050910267e+00, 0.1064788675771248e+00, 0.7093640994523618e-03, 0.1589006148839502e-03, 0.2385172514821930e-01, 0.2369282453339490e-01, 0.6280626703238275e-13, 0.8634837515442557e-26, 0.3257377007015450e-14, 0.9466330862652142e-28, 0.7725029665513934e-03, 0.9240506410060376e-03, 0.7811996215926567e-04, 0.3786532345060857e-28, 0.1283811140869274e-03, 0.1684766835109566e-03},
			i0Out:    1,
			n0Out:    20,
			ppOut:    0,
			dminOut:  9.4663308626521417e-029,
			desigOut: 2.1703795093889875e-020,
			sigmaOut: 6.9898185320128035e-004,
			qmaxOut:  1.8900334961849885,
			nFailOut: 0,
			iterOut:  11,
			nDivOut:  235,
			ttypeOut: -2,
			dmin1Out: 2.3692824533332088e-002,
			dmin2Out: 7.0611780886194908e-002,
			dnOut:    9.4663308626521417e-029,
			dn1Out:   2.3692824533332088e-002,
			dn2Out:   0.10576950347767239,
			gOut:     0.0000000000000000,
			tauOut:   3.2573770070067200e-015,
		},
		{
			i0:       1,
			n0:       20,
			z:        []float64{0.2520040627944239e+01, 0.2559470753941703e+01, 0.3943012599746679e-01, 0.2794508075882943e-01, 0.1813958619441340e+01, 0.1795964384312701e+01, 0.9950845630193767e-02, 0.9646675917972324e-02, 0.1741066741415017e+01, 0.1878131517155319e+01, 0.1467114516582776e+00, 0.8477220453173254e-01, 0.1085214189555058e+01, 0.1010428304002300e+01, 0.9986318978978259e-02, 0.7202019583097946e-02, 0.7287093921253419e+00, 0.7222798755087921e+00, 0.7725029665513934e-03, 0.2132421202606784e-02, 0.1993785121145770e+01, 0.1996156738136608e+01, 0.4504038193447841e-02, 0.9240506410060376e-03, 0.4095324760138526e+00, 0.5471324717806409e+00, 0.1385240464077977e+00, 0.1086689918286978e+00, 0.4292130907735224e+00, 0.6103174493771462e+00, 0.2897733504323247e+00, 0.2112982785836522e+00, 0.4450341145952761e+00, 0.2594200218384955e+00, 0.2568418582687495e-01, 0.3588779484774172e-01, 0.3624803443601846e+00, 0.3849926621161204e+00, 0.5840011260368079e-01, 0.1744336104525046e-01, 0.1149923468579125e+00, 0.1715659665664736e+00, 0.7401698075381481e-01, 0.2315671901408358e+00, 0.5367558686801371e+00, 0.5010211363866497e+00, 0.1958324578473516e+00, 0.3221025481782661e+00, 0.8240727124361776e+00, 0.8820900960779340e+00, 0.3801199318200257e+00, 0.6999293195746292e+00, 0.1624226116729040e+01, 0.9787039376741966e+00, 0.5440714051978934e-01, 0.4877761656142095e-01, 0.8774371331215411e+00, 0.8340525857813585e+00, 0.5393069221241477e-02, 0.5090576402208483e-02, 0.7872712619108886e+00, 0.9608800054480426e+00, 0.1786993199393658e+00, 0.1613217694817542e-01, 0.8674395783437358e-01, 0.7149381583730083e-01, 0.8820349511059155e-03, 0.1321201613351008e-02, 0.1070907050910267e+00, 0.1064788675771248e+00, 0.7093640994523618e-03, 0.1589006148839502e-03, 0.2385172514821930e-01, 0.2369282453339490e-01, 0.6280626703238275e-13, 0.8634837515442557e-26, 0.3257377007015450e-14, 0.9466330862652142e-28, 0.7725029665513934e-03, 0.9240506410060376e-03, 0.7811996215926567e-04, 0.3786532345060857e-28, 0.1283811140869274e-03, 0.1684766835109566e-03},
			pp:       1,
			dmin:     9.4663308626521417e-029,
			desig:    2.1703795093889875e-020,
			qmax:     1.8900334961849885,
			ttype:    -2,
			dmin1:    2.3692824533332088e-002,
			dmin2:    7.0611780886194908e-002,
			dn:       9.4663308626521417e-029,
			dn1:      2.3692824533332088e-002,
			dn2:      0.10576950347767239,
			g:        0.0000000000000000,
			tau:      3.2573770070067200e-015,
			nFail:    0,
			iter:     11,
			sigma:    6.9898185320128035e-004,
			nDiv:     235,
			zOut:     []float64{0.2587415834700532e+01, 0.2559470753941703e+01, 0.1939710234687058e-01, 0.2794508075882943e-01, 0.1786213957883803e+01, 0.1795964384312701e+01, 0.1014308840067055e-01, 0.9646675917972324e-02, 0.1952760633286381e+01, 0.1878131517155319e+01, 0.4386417535844126e-01, 0.8477220453173254e-01, 0.9737661482269571e+00, 0.1010428304002300e+01, 0.5342015449359675e-02, 0.7202019583097946e-02, 0.7190702812620393e+00, 0.7222798755087921e+00, 0.5919653562455767e-02, 0.2132421202606784e-02, 0.1991161135215158e+01, 0.1996156738136608e+01, 0.2539111990097616e-03, 0.9240506410060376e-03, 0.6555475524103290e+00, 0.5471324717806409e+00, 0.1011712753337584e+00, 0.1086689918286978e+00, 0.7204444526270400e+00, 0.6103174493771462e+00, 0.7608498315828412e-01, 0.2112982785836522e+00, 0.2192228335279531e+00, 0.2594200218384955e+00, 0.6302508481237896e-01, 0.3588779484774172e-01, 0.3394109383489919e+00, 0.3849926621161204e+00, 0.8817297145618809e-02, 0.1744336104525046e-01, 0.3943158595616905e+00, 0.1715659665664736e+00, 0.2942312715577539e+00, 0.2315671901408358e+00, 0.5288924130071619e+00, 0.5010211363866497e+00, 0.5372046576619481e+00, 0.3221025481782661e+00, 0.1044814757990615e+01, 0.8820900960779340e+00, 0.6556411803358774e+00, 0.6999293195746292e+00, 0.3718403738997403e+00, 0.9787039376741966e+00, 0.1094101127175453e+00, 0.4877761656142095e-01, 0.7297330494660218e+00, 0.8340525857813585e+00, 0.6703044469024726e-02, 0.5090576402208483e-02, 0.9703091379271934e+00, 0.9608800054480426e+00, 0.1188642714683105e-02, 0.1613217694817542e-01, 0.7162637473596872e-01, 0.7149381583730083e-01, 0.1964081696850700e-02, 0.1321201613351008e-02, 0.1046736864951580e+00, 0.1064788675771248e+00, 0.3596705640885345e-04, 0.1589006148839502e-03, 0.2365685747698604e-01, 0.2369282453339490e-01, 0.3455244592226135e-52, 0.8634837515442557e-26, 0.9466330862652142e-28, 0.9466330862652142e-28, 0.2539111990097616e-03, 0.9240506410060376e-03, 0.7811996215926567e-04, 0.3786532345060857e-28, 0.1283811140869274e-03, 0.1684766835109566e-03},
			i0Out:    1,
			n0Out:    20,
			ppOut:    1,
			dminOut:  9.4663308626521417e-029,
			desigOut: 2.1703795093889875e-020,
			sigmaOut: 6.9898185320128035e-004,
			qmaxOut:  1.8900334961849885,
			nFailOut: 0,
			iterOut:  12,
			nDivOut:  256,
			ttypeOut: -2,
			dmin1Out: 2.3656857476986041e-002,
			dmin2Out: 7.0305173122617720e-002,
			dnOut:    9.4663308626521417e-029,
			dn1Out:   2.3656857476986041e-002,
			dn2Out:   0.10451478588027406,
			gOut:     0.0000000000000000,
			tauOut:   0.0000000000000000,
		},
		{
			i0:       1,
			n0:       20,
			z:        []float64{0.2587415834700532e+01, 0.2559470753941703e+01, 0.1939710234687058e-01, 0.2794508075882943e-01, 0.1786213957883803e+01, 0.1795964384312701e+01, 0.1014308840067055e-01, 0.9646675917972324e-02, 0.1952760633286381e+01, 0.1878131517155319e+01, 0.4386417535844126e-01, 0.8477220453173254e-01, 0.9737661482269571e+00, 0.1010428304002300e+01, 0.5342015449359675e-02, 0.7202019583097946e-02, 0.7190702812620393e+00, 0.7222798755087921e+00, 0.5919653562455767e-02, 0.2132421202606784e-02, 0.1991161135215158e+01, 0.1996156738136608e+01, 0.2539111990097616e-03, 0.9240506410060376e-03, 0.6555475524103290e+00, 0.5471324717806409e+00, 0.1011712753337584e+00, 0.1086689918286978e+00, 0.7204444526270400e+00, 0.6103174493771462e+00, 0.7608498315828412e-01, 0.2112982785836522e+00, 0.2192228335279531e+00, 0.2594200218384955e+00, 0.6302508481237896e-01, 0.3588779484774172e-01, 0.3394109383489919e+00, 0.3849926621161204e+00, 0.8817297145618809e-02, 0.1744336104525046e-01, 0.3943158595616905e+00, 0.1715659665664736e+00, 0.2942312715577539e+00, 0.2315671901408358e+00, 0.5288924130071619e+00, 0.5010211363866497e+00, 0.5372046576619481e+00, 0.3221025481782661e+00, 0.1044814757990615e+01, 0.8820900960779340e+00, 0.6556411803358774e+00, 0.6999293195746292e+00, 0.3718403738997403e+00, 0.9787039376741966e+00, 0.1094101127175453e+00, 0.4877761656142095e-01, 0.7297330494660218e+00, 0.8340525857813585e+00, 0.6703044469024726e-02, 0.5090576402208483e-02, 0.9703091379271934e+00, 0.9608800054480426e+00, 0.1188642714683105e-02, 0.1613217694817542e-01, 0.7162637473596872e-01, 0.7149381583730083e-01, 0.1964081696850700e-02, 0.1321201613351008e-02, 0.1046736864951580e+00, 0.1064788675771248e+00, 0.3596705640885345e-04, 0.1589006148839502e-03, 0.2365685747698604e-01, 0.2369282453339490e-01, 0.3455244592226135e-52, 0.8634837515442557e-26, 0.9466330862652142e-28, 0.9466330862652142e-28, 0.2539111990097616e-03, 0.9240506410060376e-03, 0.7811996215926567e-04, 0.3786532345060857e-28, 0.1283811140869274e-03, 0.1684766835109566e-03},
			pp:       0,
			dmin:     9.4663308626521417e-029,
			desig:    2.1703795093889875e-020,
			qmax:     1.8900334961849885,
			ttype:    -2,
			dmin1:    2.3656857476986041e-002,
			dmin2:    7.0305173122617720e-002,
			dn:       9.4663308626521417e-029,
			dn1:      2.3656857476986041e-002,
			dn2:      0.10451478588027406,
			g:        0.0000000000000000,
			tau:      0.0000000000000000,
			nFail:    0,
			iter:     12,
			sigma:    6.9898185320128035e-004,
			nDiv:     256,
			zOut:     []float64{0.2587415834700532e+01, 0.2594984508308910e+01, 0.1939710234687058e-01, 0.1335166928493912e-01, 0.1786213957883803e+01, 0.1771176948261041e+01, 0.1014308840067055e-01, 0.1118297285215907e-01, 0.1952760633286381e+01, 0.1973613407054170e+01, 0.4386417535844126e-01, 0.2164225725832272e-01, 0.9737661482269571e+00, 0.9456374776795010e+00, 0.5342015449359675e-02, 0.4062111160297227e-02, 0.7190702812620393e+00, 0.7090993949257048e+00, 0.5919653562455767e-02, 0.1662247097070905e-01, 0.1991161135215158e+01, 0.1962964146704966e+01, 0.2539111990097616e-03, 0.8479567256479229e-04, 0.6555475524103290e+00, 0.7448056033330296e+00, 0.1011712753337584e+00, 0.9786215859981671e-01, 0.7204444526270400e+00, 0.6868388484470145e+00, 0.7608498315828412e-01, 0.2428454015756268e-01, 0.2192228335279531e+00, 0.2461349494442763e+00, 0.6302508481237896e-01, 0.8690924724014965e-01, 0.3394109383489919e+00, 0.2494905595159681e+00, 0.8817297145618809e-02, 0.1393559784278329e-01, 0.3943158595616905e+00, 0.6627831045381681e+00, 0.2942312715577539e+00, 0.2347927793131975e+00, 0.5288924130071619e+00, 0.8194758626174196e+00, 0.5372046576619481e+00, 0.6849248159595133e+00, 0.1044814757990615e+01, 0.1003702693628486e+01, 0.6556411803358774e+00, 0.2428944977310168e+00, 0.3718403738997403e+00, 0.2265275601477758e+00, 0.1094101127175453e+00, 0.3524523689025369e+00, 0.7297330494660218e+00, 0.3721552962940165e+00, 0.6703044469024726e-02, 0.1747664312451059e-01, 0.9703091379271934e+00, 0.9421927087788731e+00, 0.1188642714683105e-02, 0.9036173567869616e-04, 0.7162637473596872e-01, 0.6167166595864771e-01, 0.1964081696850700e-02, 0.3333583884775863e-02, 0.1046736864951580e+00, 0.8954764092829798e-01, 0.3596705640885345e-04, 0.9501841907954501e-05, 0.2365685747698604e-01, 0.1181892689658507e-01, 0.3455244592226135e-52, 0.8479567256479229e-04, 0.6989818532012803e-03, 0.9466330862652142e-28, 0.2539111990097616e-03, 0.9240506410060376e-03, 0.7811996215926567e-04, 0.3786532345060857e-28, 0.1283811140869274e-03, 0.1684766835109566e-03},
			i0Out:    1,
			n0Out:    19,
			ppOut:    0,
			dminOut:  1.1818926896585069e-002,
			desigOut: 3.4696444683954120e-019,
			sigmaOut: 1.2527410591694300e-002,
			qmaxOut:  1.8900334961849885,
			nFailOut: 0,
			iterOut:  13,
			nDivOut:  276,
			ttypeOut: -9,
			dmin1Out: 5.9707584261797009e-002,
			dmin2Out: 5.9707584261797009e-002,
			dnOut:    1.1818926896585069e-002,
			dn1Out:   8.9511673871889130e-002,
			dn2Out:   5.9707584261797009e-002,
			gOut:     0.0000000000000000,
			tauOut:   1.1828428738493020e-002,
		},
		{
			i0:       1,
			n0:       19,
			z:        []float64{0.2587415834700532e+01, 0.2594984508308910e+01, 0.1939710234687058e-01, 0.1335166928493912e-01, 0.1786213957883803e+01, 0.1771176948261041e+01, 0.1014308840067055e-01, 0.1118297285215907e-01, 0.1952760633286381e+01, 0.1973613407054170e+01, 0.4386417535844126e-01, 0.2164225725832272e-01, 0.9737661482269571e+00, 0.9456374776795010e+00, 0.5342015449359675e-02, 0.4062111160297227e-02, 0.7190702812620393e+00, 0.7090993949257048e+00, 0.5919653562455767e-02, 0.1662247097070905e-01, 0.1991161135215158e+01, 0.1962964146704966e+01, 0.2539111990097616e-03, 0.8479567256479229e-04, 0.6555475524103290e+00, 0.7448056033330296e+00, 0.1011712753337584e+00, 0.9786215859981671e-01, 0.7204444526270400e+00, 0.6868388484470145e+00, 0.7608498315828412e-01, 0.2428454015756268e-01, 0.2192228335279531e+00, 0.2461349494442763e+00, 0.6302508481237896e-01, 0.8690924724014965e-01, 0.3394109383489919e+00, 0.2494905595159681e+00, 0.8817297145618809e-02, 0.1393559784278329e-01, 0.3943158595616905e+00, 0.6627831045381681e+00, 0.2942312715577539e+00, 0.2347927793131975e+00, 0.5288924130071619e+00, 0.8194758626174196e+00, 0.5372046576619481e+00, 0.6849248159595133e+00, 0.1044814757990615e+01, 0.1003702693628486e+01, 0.6556411803358774e+00, 0.2428944977310168e+00, 0.3718403738997403e+00, 0.2265275601477758e+00, 0.1094101127175453e+00, 0.3524523689025369e+00, 0.7297330494660218e+00, 0.3721552962940165e+00, 0.6703044469024726e-02, 0.1747664312451059e-01, 0.9703091379271934e+00, 0.9421927087788731e+00, 0.1188642714683105e-02, 0.9036173567869616e-04, 0.7162637473596872e-01, 0.6167166595864771e-01, 0.1964081696850700e-02, 0.3333583884775863e-02, 0.1046736864951580e+00, 0.8954764092829798e-01, 0.3596705640885345e-04, 0.9501841907954501e-05, 0.2365685747698604e-01, 0.1181892689658507e-01, 0.3455244592226135e-52, 0.8479567256479229e-04, 0.6989818532012803e-03, 0.9466330862652142e-28, 0.2539111990097616e-03, 0.9240506410060376e-03, 0.7811996215926567e-04, 0.3786532345060857e-28, 0.1283811140869274e-03, 0.1684766835109566e-03},
			pp:       1,
			dmin:     1.1818926896585069e-002,
			desig:    3.4696444683954120e-019,
			qmax:     1.8900334961849885,
			ttype:    -9,
			dmin1:    5.9707584261797009e-002,
			dmin2:    5.9707584261797009e-002,
			dn:       1.1818926896585069e-002,
			dn1:      8.9511673871889130e-002,
			dn2:      5.9707584261797009e-002,
			g:        0.0000000000000000,
			tau:      1.1828428738493020e-002,
			nFail:    0,
			iter:     13,
			sigma:    1.2527410591694300e-002,
			nDiv:     276,
			zOut:     []float64{0.2596646703688871e+01, 0.2594984508308910e+01, 0.9107195378059658e-02, 0.1335166928493912e-01, 0.1761563251830163e+01, 0.1771176948261041e+01, 0.1252913577120413e-01, 0.1118297285215907e-01, 0.1971037054636311e+01, 0.1973613407054170e+01, 0.1038322923301279e-01, 0.2164225725832272e-01, 0.9276268857018078e+00, 0.9456374776795010e+00, 0.3105171497598932e-02, 0.4062111160297227e-02, 0.7109272204938374e+00, 0.7090993949257048e+00, 0.4589684232723624e-01, 0.1662247097070905e-01, 0.1905462626145317e+01, 0.1962964146704966e+01, 0.3314486004504485e-04, 0.8479567256479229e-04, 0.8309451431678238e+00, 0.7448056033330296e+00, 0.8089045693556918e-01, 0.9786215859981671e-01, 0.6185434577640304e+00, 0.6868388484470145e+00, 0.9663466631053550e-02, 0.2428454015756268e-01, 0.3116912561483949e+00, 0.2461349494442763e+00, 0.6956575230565126e-01, 0.8690924724014965e-01, 0.1821709311481225e+00, 0.2494905595159681e+00, 0.5070116699532772e-01, 0.1393559784278329e-01, 0.8351852429510604e+00, 0.6627831045381681e+00, 0.2303764547900405e+00, 0.2347927793131975e+00, 0.1262334749881915e+01, 0.8194758626174196e+00, 0.5445947541061251e+00, 0.6849248159595133e+00, 0.6903129633483998e+00, 0.1003702693628486e+01, 0.7970630839299049e-01, 0.2428944977310168e+00, 0.4875841467523446e+00, 0.2265275601477758e+00, 0.2690141110044630e+00, 0.3524523689025369e+00, 0.1089283545090865e+00, 0.3721552962940165e+00, 0.1511669372043137e+00, 0.1747664312451059e-01, 0.7794266594052606e+00, 0.9421927087788731e+00, 0.7149818024536710e-05, 0.9036173567869616e-04, 0.5330862612042146e-01, 0.6167166595864771e-01, 0.5599742376476567e-02, 0.3333583884775863e-02, 0.7226792648875177e-01, 0.8954764092829798e-01, 0.1553961492315709e-05, 0.9501841907954501e-05, 0.1278990301151681e-03, 0.1181892689658507e-01, 0.7149818024536710e-05, 0.8479567256479229e-04, 0.6989818532012803e-03, 0.9466330862652142e-28, 0.2539111990097616e-03, 0.9240506410060376e-03, 0.7811996215926567e-04, 0.3786532345060857e-28, 0.1283811140869274e-03, 0.1684766835109566e-03},
			i0Out:    1,
			n0Out:    19,
			ppOut:    1,
			dminOut:  1.2789903011516807e-004,
			desigOut: 0.0000000000000000,
			sigmaOut: 2.4216884496671885e-002,
			qmaxOut:  1.8900334961849885,
			nFailOut: 0,
			iterOut:  14,
			nDivOut:  296,
			ttypeOut: -4,
			dmin1Out: 4.9975042235645591e-002,
			dmin2Out: 4.9975042235645591e-002,
			dnOut:    1.2789903011516807e-004,
			dn1Out:   7.2258424646843816e-002,
			dn2Out:   4.9975042235645591e-002,
			gOut:     0.0000000000000000,
			tauOut:   1.1689473904977585e-002,
		},
		{
			i0:       1,
			n0:       19,
			z:        []float64{0.2596646703688871e+01, 0.2594984508308910e+01, 0.9107195378059658e-02, 0.1335166928493912e-01, 0.1761563251830163e+01, 0.1771176948261041e+01, 0.1252913577120413e-01, 0.1118297285215907e-01, 0.1971037054636311e+01, 0.1973613407054170e+01, 0.1038322923301279e-01, 0.2164225725832272e-01, 0.9276268857018078e+00, 0.9456374776795010e+00, 0.3105171497598932e-02, 0.4062111160297227e-02, 0.7109272204938374e+00, 0.7090993949257048e+00, 0.4589684232723624e-01, 0.1662247097070905e-01, 0.1905462626145317e+01, 0.1962964146704966e+01, 0.3314486004504485e-04, 0.8479567256479229e-04, 0.8309451431678238e+00, 0.7448056033330296e+00, 0.8089045693556918e-01, 0.9786215859981671e-01, 0.6185434577640304e+00, 0.6868388484470145e+00, 0.9663466631053550e-02, 0.2428454015756268e-01, 0.3116912561483949e+00, 0.2461349494442763e+00, 0.6956575230565126e-01, 0.8690924724014965e-01, 0.1821709311481225e+00, 0.2494905595159681e+00, 0.5070116699532772e-01, 0.1393559784278329e-01, 0.8351852429510604e+00, 0.6627831045381681e+00, 0.2303764547900405e+00, 0.2347927793131975e+00, 0.1262334749881915e+01, 0.8194758626174196e+00, 0.5445947541061251e+00, 0.6849248159595133e+00, 0.6903129633483998e+00, 0.1003702693628486e+01, 0.7970630839299049e-01, 0.2428944977310168e+00, 0.4875841467523446e+00, 0.2265275601477758e+00, 0.2690141110044630e+00, 0.3524523689025369e+00, 0.1089283545090865e+00, 0.3721552962940165e+00, 0.1511669372043137e+00, 0.1747664312451059e-01, 0.7794266594052606e+00, 0.9421927087788731e+00, 0.7149818024536710e-05, 0.9036173567869616e-04, 0.5330862612042146e-01, 0.6167166595864771e-01, 0.5599742376476567e-02, 0.3333583884775863e-02, 0.7226792648875177e-01, 0.8954764092829798e-01, 0.1553961492315709e-05, 0.9501841907954501e-05, 0.1278990301151681e-03, 0.1181892689658507e-01, 0.7149818024536710e-05, 0.8479567256479229e-04, 0.6989818532012803e-03, 0.9466330862652142e-28, 0.2539111990097616e-03, 0.9240506410060376e-03, 0.7811996215926567e-04, 0.3786532345060857e-28, 0.1283811140869274e-03, 0.1684766835109566e-03},
			pp:       0,
			dmin:     1.2789903011516807e-004,
			desig:    0.0000000000000000,
			qmax:     1.8900334961849885,
			ttype:    -4,
			dmin1:    4.9975042235645591e-002,
			dmin2:    4.9975042235645591e-002,
			dn:       1.2789903011516807e-004,
			dn1:      7.2258424646843816e-002,
			dn2:      4.9975042235645591e-002,
			g:        0.0000000000000000,
			tau:      1.1689473904977585e-002,
			nFail:    0,
			iter:     14,
			sigma:    2.4216884496671885e-002,
			nDiv:     296,
			zOut:     []float64{0.2596646703688871e+01, 0.2605626003858251e+01, 0.9107195378059658e-02, 0.6157023564192275e-02, 0.1761563251830163e+01, 0.1767807468828494e+01, 0.1252913577120413e-01, 0.1396950250695455e-01, 0.1971037054636311e+01, 0.1967322886153689e+01, 0.1038322923301279e-01, 0.4895872794820515e-02, 0.9276268857018078e+00, 0.9257082891959054e+00, 0.3105171497598932e-02, 0.2384715539127593e-02, 0.7109272204938374e+00, 0.7543114520732653e+00, 0.4589684232723624e-01, 0.1159397984377132e+00, 0.1905462626145317e+01, 0.1789428077358968e+01, 0.3314486004504485e-04, 0.1539126429493388e-04, 0.8309451431678238e+00, 0.9116923136304173e+00, 0.8089045693556918e-01, 0.5488064578914756e-01, 0.6185434577640304e+00, 0.5731983833972557e+00, 0.9663466631053550e-02, 0.5254756712901782e-02, 0.3116912561483949e+00, 0.3758743565324636e+00, 0.6956575230565126e-01, 0.3371567560620648e-01, 0.1821709311481225e+00, 0.1990285273285630e+00, 0.5070116699532772e-01, 0.2127577741907859e+00, 0.8351852429510604e+00, 0.8526760283416343e+00, 0.2303764547900405e+00, 0.3410582621885915e+00, 0.1262334749881915e+01, 0.1465743346590768e+01, 0.5445947541061251e+00, 0.2564847518533230e+00, 0.6903129633483998e+00, 0.5134066246793865e+00, 0.7970630839299049e-01, 0.7569737222001199e-01, 0.4875841467523446e+00, 0.6807729903281149e+00, 0.2690141110044630e+00, 0.4304410555024735e-01, 0.1089283545090865e+00, 0.2169232909544721e+00, 0.1511669372043137e+00, 0.5431576312495270e+00, 0.7794266594052606e+00, 0.2361482827650774e+00, 0.7149818024536710e-05, 0.1614015445872399e-05, 0.5330862612042146e-01, 0.5877885927277143e-01, 0.5599742376476567e-02, 0.6884818375619963e-02, 0.7226792648875177e-01, 0.6525676686594341e-01, 0.1553961492315709e-05, 0.3045663725752605e-08, 0.1278990301151681e-03, 0.7757707209639971e-09, 0.7149818024536710e-05, 0.1614015445872399e-05, 0.6989818532012803e-03, 0.9466330862652142e-28, 0.2539111990097616e-03, 0.9240506410060376e-03, 0.7811996215926567e-04, 0.3786532345060857e-28, 0.1283811140869274e-03, 0.1684766835109566e-03},
			i0Out:    1,
			n0Out:    19,
			ppOut:    0,
			dminOut:  7.7577072096399712e-010,
			desigOut: -5.4210108624275222e-020,
			sigmaOut: 2.4344779705352607e-002,
			qmaxOut:  1.8900334961849885,
			nFailOut: 1,
			iterOut:  16,
			nDivOut:  336,
			ttypeOut: -15,
			dmin1Out: 5.3179116896294863e-002,
			dmin2Out: 5.3179116896294863e-002,
			dnOut:    7.7577072096399712e-010,
			dn1Out:   6.5255212904451090e-002,
			dn2Out:   5.3179116896294863e-002,
			gOut:     0.0000000000000000,
			tauOut:   1.2789520868072135e-004,
		},
		{
			i0:       1,
			n0:       19,
			z:        []float64{0.2596646703688871e+01, 0.2605626003858251e+01, 0.9107195378059658e-02, 0.6157023564192275e-02, 0.1761563251830163e+01, 0.1767807468828494e+01, 0.1252913577120413e-01, 0.1396950250695455e-01, 0.1971037054636311e+01, 0.1967322886153689e+01, 0.1038322923301279e-01, 0.4895872794820515e-02, 0.9276268857018078e+00, 0.9257082891959054e+00, 0.3105171497598932e-02, 0.2384715539127593e-02, 0.7109272204938374e+00, 0.7543114520732653e+00, 0.4589684232723624e-01, 0.1159397984377132e+00, 0.1905462626145317e+01, 0.1789428077358968e+01, 0.3314486004504485e-04, 0.1539126429493388e-04, 0.8309451431678238e+00, 0.9116923136304173e+00, 0.8089045693556918e-01, 0.5488064578914756e-01, 0.6185434577640304e+00, 0.5731983833972557e+00, 0.9663466631053550e-02, 0.5254756712901782e-02, 0.3116912561483949e+00, 0.3758743565324636e+00, 0.6956575230565126e-01, 0.3371567560620648e-01, 0.1821709311481225e+00, 0.1990285273285630e+00, 0.5070116699532772e-01, 0.2127577741907859e+00, 0.8351852429510604e+00, 0.8526760283416343e+00, 0.2303764547900405e+00, 0.3410582621885915e+00, 0.1262334749881915e+01, 0.1465743346590768e+01, 0.5445947541061251e+00, 0.2564847518533230e+00, 0.6903129633483998e+00, 0.5134066246793865e+00, 0.7970630839299049e-01, 0.7569737222001199e-01, 0.4875841467523446e+00, 0.6807729903281149e+00, 0.2690141110044630e+00, 0.4304410555024735e-01, 0.1089283545090865e+00, 0.2169232909544721e+00, 0.1511669372043137e+00, 0.5431576312495270e+00, 0.7794266594052606e+00, 0.2361482827650774e+00, 0.7149818024536710e-05, 0.1614015445872399e-05, 0.5330862612042146e-01, 0.5877885927277143e-01, 0.5599742376476567e-02, 0.6884818375619963e-02, 0.7226792648875177e-01, 0.6525676686594341e-01, 0.1553961492315709e-05, 0.3045663725752605e-08, 0.1278990301151681e-03, 0.7757707209639971e-09, 0.7149818024536710e-05, 0.1614015445872399e-05, 0.6989818532012803e-03, 0.9466330862652142e-28, 0.2539111990097616e-03, 0.9240506410060376e-03, 0.7811996215926567e-04, 0.3786532345060857e-28, 0.1283811140869274e-03, 0.1684766835109566e-03},
			pp:       1,
			dmin:     7.7577072096399712e-010,
			desig:    -5.4210108624275222e-020,
			qmax:     1.8900334961849885,
			ttype:    -15,
			dmin1:    5.3179116896294863e-002,
			dmin2:    5.3179116896294863e-002,
			dn:       7.7577072096399712e-010,
			dn1:      6.5255212904451090e-002,
			dn2:      5.3179116896294863e-002,
			g:        0.0000000000000000,
			tau:      1.2789520868072135e-004,
			nFail:    1,
			iter:     16,
			sigma:    2.4344779705352607e-002,
			nDiv:     336,
			zOut:     []float64{0.2611783026646672e+01, 0.2605626003858251e+01, 0.4167433562238479e-02, 0.6157023564192275e-02, 0.1777609536997439e+01, 0.1767807468828494e+01, 0.1546038171944878e-01, 0.1396950250695455e-01, 0.1956758376453290e+01, 0.1967322886153689e+01, 0.2316152103168096e-02, 0.4895872794820515e-02, 0.9257768518560943e+00, 0.9257082891959054e+00, 0.1943036529261404e-02, 0.2384715539127593e-02, 0.8683082132059464e+00, 0.7543114520732653e+00, 0.2389312083572061e+00, 0.1159397984377132e+00, 0.1550512259490286e+01, 0.1789428077358968e+01, 0.9049975109102588e-05, 0.1539126429493388e-04, 0.9665639086686850e+00, 0.9116923136304173e+00, 0.3254569838994440e-01, 0.5488064578914756e-01, 0.5459074409444424e+00, 0.5731983833972557e+00, 0.3618064437406363e-02, 0.5254756712901782e-02, 0.4059719669254931e+00, 0.3758743565324636e+00, 0.1652917395900484e-01, 0.3371567560620648e-01, 0.3952571267845734e+00, 0.1990285273285630e+00, 0.4589757947481138e+00, 0.2127577741907859e+00, 0.7347584950063413e+00, 0.8526760283416343e+00, 0.6803648845168142e+00, 0.3410582621885915e+00, 0.1041863213151506e+01, 0.1465743346590768e+01, 0.1263898840735783e+00, 0.2564847518533230e+00, 0.4627141120500496e+00, 0.5134066246793865e+00, 0.1113705527974558e+00, 0.7569737222001199e-01, 0.6124465423051357e+00, 0.6807729903281149e+00, 0.1524585149425051e-01, 0.4304410555024735e-01, 0.7448350699339780e+00, 0.2169232909544721e+00, 0.1722069046798406e+00, 0.5431576312495270e+00, 0.6394299132491200e-01, 0.2361482827650774e+00, 0.1483665133446019e-05, 0.1614015445872399e-05, 0.6566219320748727e-01, 0.5877885927277143e-01, 0.6842308575232983e-02, 0.6884818375619963e-02, 0.5841446056060346e-01, 0.6525676686594341e-01, 0.4044780558898965e-16, 0.3045663725752605e-08, 0.1013559914197710e-18, 0.7757707209639971e-09, 0.1483665133446019e-05, 0.1614015445872399e-05, 0.6989818532012803e-03, 0.9466330862652142e-28, 0.2539111990097616e-03, 0.9240506410060376e-03, 0.7811996215926567e-04, 0.3786532345060857e-28, 0.1283811140869274e-03, 0.1684766835109566e-03},
			i0Out:    1,
			n0Out:    19,
			ppOut:    1,
			dminOut:  1.0135599141977102e-019,
			desigOut: 2.7104744119407903e-020,
			sigmaOut: 2.4344780481123287e-002,
			qmaxOut:  1.8900334961849885,
			nFailOut: 2,
			iterOut:  18,
			nDivOut:  376,
			ttypeOut: -15,
			dmin1Out: 5.8414457514939733e-002,
			dmin2Out: 5.8777374831867304e-002,
			dnOut:    1.0135599141977102e-019,
			dn1Out:   5.8414457514939733e-002,
			dn2Out:   5.8777374831867304e-002,
			gOut:     0.0000000000000000,
			tauOut:   7.7577068041483555e-010,
		},
		{
			i0:       1,
			n0:       19,
			z:        []float64{0.2611783026646672e+01, 0.2605626003858251e+01, 0.4167433562238479e-02, 0.6157023564192275e-02, 0.1777609536997439e+01, 0.1767807468828494e+01, 0.1546038171944878e-01, 0.1396950250695455e-01, 0.1956758376453290e+01, 0.1967322886153689e+01, 0.2316152103168096e-02, 0.4895872794820515e-02, 0.9257768518560943e+00, 0.9257082891959054e+00, 0.1943036529261404e-02, 0.2384715539127593e-02, 0.8683082132059464e+00, 0.7543114520732653e+00, 0.2389312083572061e+00, 0.1159397984377132e+00, 0.1550512259490286e+01, 0.1789428077358968e+01, 0.9049975109102588e-05, 0.1539126429493388e-04, 0.9665639086686850e+00, 0.9116923136304173e+00, 0.3254569838994440e-01, 0.5488064578914756e-01, 0.5459074409444424e+00, 0.5731983833972557e+00, 0.3618064437406363e-02, 0.5254756712901782e-02, 0.4059719669254931e+00, 0.3758743565324636e+00, 0.1652917395900484e-01, 0.3371567560620648e-01, 0.3952571267845734e+00, 0.1990285273285630e+00, 0.4589757947481138e+00, 0.2127577741907859e+00, 0.7347584950063413e+00, 0.8526760283416343e+00, 0.6803648845168142e+00, 0.3410582621885915e+00, 0.1041863213151506e+01, 0.1465743346590768e+01, 0.1263898840735783e+00, 0.2564847518533230e+00, 0.4627141120500496e+00, 0.5134066246793865e+00, 0.1113705527974558e+00, 0.7569737222001199e-01, 0.6124465423051357e+00, 0.6807729903281149e+00, 0.1524585149425051e-01, 0.4304410555024735e-01, 0.7448350699339780e+00, 0.2169232909544721e+00, 0.1722069046798406e+00, 0.5431576312495270e+00, 0.6394299132491200e-01, 0.2361482827650774e+00, 0.1483665133446019e-05, 0.1614015445872399e-05, 0.6566219320748727e-01, 0.5877885927277143e-01, 0.6842308575232983e-02, 0.6884818375619963e-02, 0.5841446056060346e-01, 0.6525676686594341e-01, 0.4044780558898965e-16, 0.3045663725752605e-08, 0.1013559914197710e-18, 0.7757707209639971e-09, 0.1483665133446019e-05, 0.1614015445872399e-05, 0.6989818532012803e-03, 0.9466330862652142e-28, 0.2539111990097616e-03, 0.9240506410060376e-03, 0.7811996215926567e-04, 0.3786532345060857e-28, 0.1283811140869274e-03, 0.1684766835109566e-03},
			pp:       0,
			dmin:     1.0135599141977102e-019,
			desig:    2.7104744119407903e-020,
			qmax:     1.8900334961849885,
			ttype:    -15,
			dmin1:    5.8414457514939733e-002,
			dmin2:    5.8777374831867304e-002,
			dn:       1.0135599141977102e-019,
			dn1:      5.8414457514939733e-002,
			dn2:      5.8777374831867304e-002,
			g:        0.0000000000000000,
			tau:      7.7577068041483555e-010,
			nFail:    2,
			iter:     18,
			sigma:    2.4344780481123287e-002,
			nDiv:     376,
			zOut:     []float64{0.2611783026646672e+01, 0.2615950460208911e+01, 0.4167433562238479e-02, 0.2831884532112553e-02, 0.1777609536997439e+01, 0.1790238034184775e+01, 0.1546038171944878e-01, 0.1689844079671380e-01, 0.1956758376453290e+01, 0.1942176087759744e+01, 0.2316152103168096e-02, 0.1104039956008399e-02, 0.9257768518560943e+00, 0.9266158484293474e+00, 0.1943036529261404e-02, 0.1820770257466081e-02, 0.8683082132059464e+00, 0.1105418651305687e+01, 0.2389312083572061e+00, 0.3351361651941490e+00, 0.1550512259490286e+01, 0.1215385144271246e+01, 0.9049975109102588e-05, 0.7197207696703830e-05, 0.9665639086686850e+00, 0.9991024098509327e+00, 0.3254569838994440e-01, 0.1778290067827487e-01, 0.5459074409444424e+00, 0.5317426047035739e+00, 0.3618064437406363e-02, 0.2762300261676148e-02, 0.4059719669254931e+00, 0.4197388406228219e+00, 0.1652917395900484e-01, 0.1556509232613395e-01, 0.3952571267845734e+00, 0.8386678292065533e+00, 0.4589757947481138e+00, 0.4021095747913879e+00, 0.7347584950063413e+00, 0.1013013804731768e+01, 0.6803648845168142e+00, 0.6997408538631263e+00, 0.1041863213151506e+01, 0.4685122433619579e+00, 0.1263898840735783e+00, 0.1248257304047288e+00, 0.4627141120500496e+00, 0.4492589344427766e+00, 0.1113705527974558e+00, 0.1518244930621437e+00, 0.6124465423051357e+00, 0.4758679007372426e+00, 0.1524585149425051e-01, 0.2386301922514691e-01, 0.7448350699339780e+00, 0.8931789553886716e+00, 0.1722069046798406e+00, 0.1232835205710967e-01, 0.6394299132491200e-01, 0.5161612293293578e-01, 0.1483665133446019e-05, 0.1887408451311279e-05, 0.6566219320748727e-01, 0.7250261437426894e-01, 0.6842308575232983e-02, 0.5512763475647510e-02, 0.5841446056060346e-01, 0.5290169708495600e-01, 0.4044780558898965e-16, 0.7749519698096867e-34, 0.1013559914197710e-18, 0.1013559914197709e-18, 0.1483665133446019e-05, 0.1887408451311279e-05, 0.6989818532012803e-03, 0.9466330862652142e-28, 0.2539111990097616e-03, 0.9240506410060376e-03, 0.7811996215926567e-04, 0.3786532345060857e-28, 0.1283811140869274e-03, 0.1684766835109566e-03},
			i0Out:    1,
			n0Out:    19,
			ppOut:    0,
			dminOut:  1.0135599141977093e-019,
			desigOut: 2.7104744119407903e-020,
			sigmaOut: 2.4344780481123287e-002,
			qmaxOut:  1.8900334961849885,
			nFailOut: 2,
			iterOut:  19,
			nDivOut:  396,
			ttypeOut: -2,
			dmin1Out: 5.1614639267802333e-002,
			dmin2Out: 5.1614639267802333e-002,
			dnOut:    1.0135599141977093e-019,
			dn1Out:   5.2901697084955956e-002,
			dn2Out:   6.5660305799035965e-002,
			gOut:     0.0000000000000000,
			tauOut:   0.0000000000000000,
		},
		{
			i0:       1,
			n0:       4,
			z:        []float64{0.6482015258643838e+00, 0.6481999396438409e+00, 0.2115037983241246e-16, 0.2912830939801045e-17, 0.8927011497345719e-01, 0.8926852875357447e-01, 0.6601400172354812e-12, 0.2332164496627946e-11, 0.3153708122363886e+00, 0.3200537757344777e+00, 0.4684549720964035e-02, 0.2356209328895529e-07, 0.1609789066298190e-05, 0.6430148075154844e-11, 0.2115037983241246e-16, 0.2912830939801045e-17, 0.1702794694134603e+01, 0.2545269924803487e-17, 0.5179704803914515e-14, 0.1264160184270297e-13, 0.1077066053646038e+01, 0.7156177943897596e-01, 0.2292199980814605e-03, 0.1899018778701386e-03, 0.1060140274732043e+01, 0.8660746506696473e-01, 0.1357005210961402e-39, 0.1331360138522907e-23, 0.9894235909971354e+00, 0.8804208964992894e-17, 0.8660746506696473e-01, 0.8679736700028205e-01, 0.9539342071687115e+00, 0.2761013168273541e-29, 0.1040577915698430e-22, 0.8107757314001177e-24, 0.8046649468928653e+00, 0.5098326619997980e-22, 0.1416795225784663e-10, 0.1216165317638265e-10, 0.7009142227469247e+00, 0.1183291357831518e-29, 0.4806926318247711e-11, 0.3824647564366386e-11, 0.5502651835254770e+00, 0.1368072596837427e-11, 0.1169503544861386e-10, 0.4068037876491279e-10, 0.4423863025187732e+00, 0.2477754901417239e-20, 0.1737857614720001e-07, 0.5295826057530262e-07, 0.3697086796938907e+00, 0.5143373102040997e-26, 0.1312431380925897e-05, 0.1614374370413396e-05, 0.1446051340026323e+00, 0.9928287808749566e-25, 0.3777515963415321e-05, 0.2773141909621761e-05, 0.1110032523123295e+00, 0.0000000000000000e+00, 0.2532463507333992e-05, 0.2934028940292093e-05, 0.7513603923341917e-01, 0.5394210206791908e-19, 0.6032617175984252e-07, 0.2575740214720034e-06, 0.6860214953971246e-01, 0.7542232825258426e-19, 0.3784397549471832e-09, 0.1131370986389306e-09, 0.2434478048112329e-01, 0.1013559914197709e-18, 0.1483665133446019e-05, 0.1887408451311279e-05, 0.6989818532012803e-03, 0.9466330862652142e-28, 0.2539111990097616e-03, 0.9240506410060376e-03, 0.7811996215926567e-04, 0.3786532345060857e-28, 0.1283811140869274e-03, 0.1684766835109566e-03},
			pp:       1,
			dmin:     6.4301480751548441e-012,
			desig:    -2.0633404957759536e-016,
			qmax:     1.6602870234255678,
			ttype:    -2,
			dmin1:    8.9268528752914328e-002,
			dmin2:    8.9268528752914328e-002,
			dn:       6.4301480751548441e-012,
			dn1:      0.31536922601351364,
			dn2:      8.9268528752914328e-002,
			g:        0.49975000000000003,
			tau:      1.5862205428611591e-006,
			nFail:    5,
			iter:     102,
			sigma:    2.0015106747272648,
			nDiv:     1395,
			zOut:     []float64{0.6481999396374125e+00, 0.6481999396438409e+00, 0.4011480356653261e-18, 0.2912830939801045e-17, 0.8926852874947827e-01, 0.8926852875357447e-01, 0.8361491594360282e-11, 0.2332164496627946e-11, 0.3200537992817811e+00, 0.3200537757344777e+00, 0.4733821287189426e-18, 0.2356209328895529e-07, 0.1787791211067534e-14, 0.6430148075154844e-11, 0.4011480356653261e-18, 0.2912830939801045e-17, 0.1702794694134603e+01, 0.2545269924803487e-17, 0.5179704803914515e-14, 0.1264160184270297e-13, 0.1077066053646038e+01, 0.7156177943897596e-01, 0.2292199980814605e-03, 0.1899018778701386e-03, 0.1060140274732043e+01, 0.8660746506696473e-01, 0.1357005210961402e-39, 0.1331360138522907e-23, 0.9894235909971354e+00, 0.8804208964992894e-17, 0.8660746506696473e-01, 0.8679736700028205e-01, 0.9539342071687115e+00, 0.2761013168273541e-29, 0.1040577915698430e-22, 0.8107757314001177e-24, 0.8046649468928653e+00, 0.5098326619997980e-22, 0.1416795225784663e-10, 0.1216165317638265e-10, 0.7009142227469247e+00, 0.1183291357831518e-29, 0.4806926318247711e-11, 0.3824647564366386e-11, 0.5502651835254770e+00, 0.1368072596837427e-11, 0.1169503544861386e-10, 0.4068037876491279e-10, 0.4423863025187732e+00, 0.2477754901417239e-20, 0.1737857614720001e-07, 0.5295826057530262e-07, 0.3697086796938907e+00, 0.5143373102040997e-26, 0.1312431380925897e-05, 0.1614374370413396e-05, 0.1446051340026323e+00, 0.9928287808749566e-25, 0.3777515963415321e-05, 0.2773141909621761e-05, 0.1110032523123295e+00, 0.0000000000000000e+00, 0.2532463507333992e-05, 0.2934028940292093e-05, 0.7513603923341917e-01, 0.5394210206791908e-19, 0.6032617175984252e-07, 0.2575740214720034e-06, 0.6860214953971246e-01, 0.7542232825258426e-19, 0.3784397549471832e-09, 0.1131370986389306e-09, 0.2434478048112329e-01, 0.1013559914197709e-18, 0.1483665133446019e-05, 0.1887408451311279e-05, 0.6989818532012803e-03, 0.9466330862652142e-28, 0.2539111990097616e-03, 0.9240506410060376e-03, 0.7811996215926567e-04, 0.3786532345060857e-28, 0.1283811140869274e-03, 0.1684766835109566e-03},
			i0Out:    1,
			n0Out:    4,
			ppOut:    1,
			dminOut:  1.7877912110675335e-015,
			desigOut: -3.7836067586147711e-017,
			sigmaOut: 2.0015106747336930,
			qmaxOut:  1.6602870234255678,
			nFailOut: 5,
			iterOut:  103,
			nDivOut:  1400,
			ttypeOut: -4,
			dmin1Out: 8.9268528747146109e-002,
			dmin2Out: 8.9268528747146109e-002,
			dnOut:    1.7877912110675335e-015,
			dn1Out:   0.32005377571968785,
			dn2Out:   8.9268528747146109e-002,
			gOut:     0.49975000000000003,
			tauOut:   6.4283598105616478e-012,
		},
		{
			i0:       1,
			n0:       4,
			z:        []float64{0.6481999396374125e+00, 0.6481999396438409e+00, 0.4011480356653261e-18, 0.2912830939801045e-17, 0.8926852874947827e-01, 0.8926852875357447e-01, 0.8361491594360282e-11, 0.2332164496627946e-11, 0.3200537992817811e+00, 0.3200537757344777e+00, 0.4733821287189426e-18, 0.2356209328895529e-07, 0.1787791211067534e-14, 0.6430148075154844e-11, 0.4011480356653261e-18, 0.2912830939801045e-17, 0.1702794694134603e+01, 0.2545269924803487e-17, 0.5179704803914515e-14, 0.1264160184270297e-13, 0.1077066053646038e+01, 0.7156177943897596e-01, 0.2292199980814605e-03, 0.1899018778701386e-03, 0.1060140274732043e+01, 0.8660746506696473e-01, 0.1357005210961402e-39, 0.1331360138522907e-23, 0.9894235909971354e+00, 0.8804208964992894e-17, 0.8660746506696473e-01, 0.8679736700028205e-01, 0.9539342071687115e+00, 0.2761013168273541e-29, 0.1040577915698430e-22, 0.8107757314001177e-24, 0.8046649468928653e+00, 0.5098326619997980e-22, 0.1416795225784663e-10, 0.1216165317638265e-10, 0.7009142227469247e+00, 0.1183291357831518e-29, 0.4806926318247711e-11, 0.3824647564366386e-11, 0.5502651835254770e+00, 0.1368072596837427e-11, 0.1169503544861386e-10, 0.4068037876491279e-10, 0.4423863025187732e+00, 0.2477754901417239e-20, 0.1737857614720001e-07, 0.5295826057530262e-07, 0.3697086796938907e+00, 0.5143373102040997e-26, 0.1312431380925897e-05, 0.1614374370413396e-05, 0.1446051340026323e+00, 0.9928287808749566e-25, 0.3777515963415321e-05, 0.2773141909621761e-05, 0.1110032523123295e+00, 0.0000000000000000e+00, 0.2532463507333992e-05, 0.2934028940292093e-05, 0.7513603923341917e-01, 0.5394210206791908e-19, 0.6032617175984252e-07, 0.2575740214720034e-06, 0.6860214953971246e-01, 0.7542232825258426e-19, 0.3784397549471832e-09, 0.1131370986389306e-09, 0.2434478048112329e-01, 0.1013559914197709e-18, 0.1483665133446019e-05, 0.1887408451311279e-05, 0.6989818532012803e-03, 0.9466330862652142e-28, 0.2539111990097616e-03, 0.9240506410060376e-03, 0.7811996215926567e-04, 0.3786532345060857e-28, 0.1283811140869274e-03, 0.1684766835109566e-03},
			pp:       0,
			dmin:     1.7877912110675335e-015,
			desig:    -3.7836067586147711e-017,
			qmax:     1.6602870234255678,
			ttype:    -4,
			dmin1:    8.9268528747146109e-002,
			dmin2:    8.9268528747146109e-002,
			dn:       1.7877912110675335e-015,
			dn1:      0.32005377571968785,
			dn2:      8.9268528747146109e-002,
			g:        0.49975000000000003,
			tau:      6.4283598105616478e-012,
			nFail:    5,
			iter:     103,
			sigma:    2.0015106747336930,
			nDiv:     1400,
			zOut:     []float64{0.6481999396374125e+00, 0.6481999396374107e+00, 0.4011480356653261e-18, 0.5524513774965514e-19, 0.8926852874947827e-01, 0.8926852875783797e-01, 0.8361491594360282e-11, 0.2997839428604580e-10, 0.3200537992817811e+00, 0.3200537992518009e+00, 0.4733821287189426e-18, 0.2644269217171004e-32, 0.1787791211067534e-14, 0.2227952244484021e-23, 0.4011480356653261e-18, 0.5524513774965514e-19, 0.1702794694134603e+01, 0.2545269924803487e-17, 0.5179704803914515e-14, 0.1264160184270297e-13, 0.1077066053646038e+01, 0.7156177943897596e-01, 0.2292199980814605e-03, 0.1899018778701386e-03, 0.1060140274732043e+01, 0.8660746506696473e-01, 0.1357005210961402e-39, 0.1331360138522907e-23, 0.9894235909971354e+00, 0.8804208964992894e-17, 0.8660746506696473e-01, 0.8679736700028205e-01, 0.9539342071687115e+00, 0.2761013168273541e-29, 0.1040577915698430e-22, 0.8107757314001177e-24, 0.8046649468928653e+00, 0.5098326619997980e-22, 0.1416795225784663e-10, 0.1216165317638265e-10, 0.7009142227469247e+00, 0.1183291357831518e-29, 0.4806926318247711e-11, 0.3824647564366386e-11, 0.5502651835254770e+00, 0.1368072596837427e-11, 0.1169503544861386e-10, 0.4068037876491279e-10, 0.4423863025187732e+00, 0.2477754901417239e-20, 0.1737857614720001e-07, 0.5295826057530262e-07, 0.3697086796938907e+00, 0.5143373102040997e-26, 0.1312431380925897e-05, 0.1614374370413396e-05, 0.1446051340026323e+00, 0.9928287808749566e-25, 0.3777515963415321e-05, 0.2773141909621761e-05, 0.1110032523123295e+00, 0.0000000000000000e+00, 0.2532463507333992e-05, 0.2934028940292093e-05, 0.7513603923341917e-01, 0.5394210206791908e-19, 0.6032617175984252e-07, 0.2575740214720034e-06, 0.6860214953971246e-01, 0.7542232825258426e-19, 0.3784397549471832e-09, 0.1131370986389306e-09, 0.2434478048112329e-01, 0.1013559914197709e-18, 0.1483665133446019e-05, 0.1887408451311279e-05, 0.6989818532012803e-03, 0.9466330862652142e-28, 0.2539111990097616e-03, 0.9240506410060376e-03, 0.7811996215926567e-04, 0.3786532345060857e-28, 0.1283811140869274e-03, 0.1684766835109566e-03},
			i0Out:    1,
			n0Out:    4,
			ppOut:    0,
			dminOut:  2.2279522444840209e-024,
			desigOut: -2.6401698146816920e-017,
			sigmaOut: 2.0015106747336948,
			qmaxOut:  1.6602870234255678,
			nFailOut: 5,
			iterOut:  104,
			nDivOut:  1405,
			ttypeOut: -4,
			dmin1Out: 8.9268528749476481e-002,
			dmin2Out: 8.9268528749476481e-002,
			dnOut:    2.2279522444840209e-024,
			dn1Out:   0.32005379925180094,
			dn2Out:   8.9268528749476481e-002,
			gOut:     0.49975000000000003,
			tauOut:   1.7877912088395813e-015,
		},
		{
			i0:       1,
			n0:       4,
			z:        []float64{0.6481999396374125e+00, 0.6481999396374107e+00, 0.4011480356653261e-18, 0.5524513774965514e-19, 0.8926852874947827e-01, 0.8926852875783797e-01, 0.8361491594360282e-11, 0.2997839428604580e-10, 0.3200537992817811e+00, 0.3200537992518009e+00, 0.4733821287189426e-18, 0.2644269217171004e-32, 0.1787791211067534e-14, 0.2227952244484021e-23, 0.4011480356653261e-18, 0.5524513774965514e-19, 0.1702794694134603e+01, 0.2545269924803487e-17, 0.5179704803914515e-14, 0.1264160184270297e-13, 0.1077066053646038e+01, 0.7156177943897596e-01, 0.2292199980814605e-03, 0.1899018778701386e-03, 0.1060140274732043e+01, 0.8660746506696473e-01, 0.1357005210961402e-39, 0.1331360138522907e-23, 0.9894235909971354e+00, 0.8804208964992894e-17, 0.8660746506696473e-01, 0.8679736700028205e-01, 0.9539342071687115e+00, 0.2761013168273541e-29, 0.1040577915698430e-22, 0.8107757314001177e-24, 0.8046649468928653e+00, 0.5098326619997980e-22, 0.1416795225784663e-10, 0.1216165317638265e-10, 0.7009142227469247e+00, 0.1183291357831518e-29, 0.4806926318247711e-11, 0.3824647564366386e-11, 0.5502651835254770e+00, 0.1368072596837427e-11, 0.1169503544861386e-10, 0.4068037876491279e-10, 0.4423863025187732e+00, 0.2477754901417239e-20, 0.1737857614720001e-07, 0.5295826057530262e-07, 0.3697086796938907e+00, 0.5143373102040997e-26, 0.1312431380925897e-05, 0.1614374370413396e-05, 0.1446051340026323e+00, 0.9928287808749566e-25, 0.3777515963415321e-05, 0.2773141909621761e-05, 0.1110032523123295e+00, 0.0000000000000000e+00, 0.2532463507333992e-05, 0.2934028940292093e-05, 0.7513603923341917e-01, 0.5394210206791908e-19, 0.6032617175984252e-07, 0.2575740214720034e-06, 0.6860214953971246e-01, 0.7542232825258426e-19, 0.3784397549471832e-09, 0.1131370986389306e-09, 0.2434478048112329e-01, 0.1013559914197709e-18, 0.1483665133446019e-05, 0.1887408451311279e-05, 0.6989818532012803e-03, 0.9466330862652142e-28, 0.2539111990097616e-03, 0.9240506410060376e-03, 0.7811996215926567e-04, 0.3786532345060857e-28, 0.1283811140869274e-03, 0.1684766835109566e-03},
			pp:       1,
			dmin:     2.2279522444840209e-024,
			desig:    -2.6401698146816920e-017,
			qmax:     1.6602870234255678,
			ttype:    -4,
			dmin1:    8.9268528749476481e-002,
			dmin2:    8.9268528749476481e-002,
			dn:       2.2279522444840209e-024,
			dn1:      0.32005379925180094,
			dn2:      8.9268528749476481e-002,
			g:        0.49975000000000003,
			tau:      1.7877912088395813e-015,
			nFail:    5,
			iter:     104,
			sigma:    2.0015106747336948,
			nDiv:     1405,
			zOut:     []float64{0.6258828074500417e+00, 0.6481999396374107e+00, 0.7879513719234823e-20, 0.5524513774965514e-19, 0.6695139660044724e-01, 0.8926852875783797e-01, 0.1433084218388560e-09, 0.2997839428604580e-10, 0.2977366669211234e+00, 0.3200537992518009e+00, 0.8926852875783797e-01, 0.2644269217171004e-32, 0.2001510674733695e+01, 0.2227952244484021e-23, 0.4011480356653261e-18, 0.5524513774965514e-19, 0.1702794694134603e+01, 0.2545269924803487e-17, 0.5179704803914515e-14, 0.1264160184270297e-13, 0.1077066053646038e+01, 0.7156177943897596e-01, 0.2292199980814605e-03, 0.1899018778701386e-03, 0.1060140274732043e+01, 0.8660746506696473e-01, 0.1357005210961402e-39, 0.1331360138522907e-23, 0.9894235909971354e+00, 0.8804208964992894e-17, 0.8660746506696473e-01, 0.8679736700028205e-01, 0.9539342071687115e+00, 0.2761013168273541e-29, 0.1040577915698430e-22, 0.8107757314001177e-24, 0.8046649468928653e+00, 0.5098326619997980e-22, 0.1416795225784663e-10, 0.1216165317638265e-10, 0.7009142227469247e+00, 0.1183291357831518e-29, 0.4806926318247711e-11, 0.3824647564366386e-11, 0.5502651835254770e+00, 0.1368072596837427e-11, 0.1169503544861386e-10, 0.4068037876491279e-10, 0.4423863025187732e+00, 0.2477754901417239e-20, 0.1737857614720001e-07, 0.5295826057530262e-07, 0.3697086796938907e+00, 0.5143373102040997e-26, 0.1312431380925897e-05, 0.1614374370413396e-05, 0.1446051340026323e+00, 0.9928287808749566e-25, 0.3777515963415321e-05, 0.2773141909621761e-05, 0.1110032523123295e+00, 0.0000000000000000e+00, 0.2532463507333992e-05, 0.2934028940292093e-05, 0.7513603923341917e-01, 0.5394210206791908e-19, 0.6032617175984252e-07, 0.2575740214720034e-06, 0.6860214953971246e-01, 0.7542232825258426e-19, 0.3784397549471832e-09, 0.1131370986389306e-09, 0.2434478048112329e-01, 0.1013559914197709e-18, 0.1483665133446019e-05, 0.1887408451311279e-05, 0.6989818532012803e-03, 0.9466330862652142e-28, 0.2539111990097616e-03, 0.9240506410060376e-03, 0.7811996215926567e-04, 0.3786532345060857e-28, 0.1283811140869274e-03, 0.1684766835109566e-03},
			i0Out:    1,
			n0Out:    3,
			ppOut:    1,
			dminOut:  6.6951396570468849e-002,
			desigOut: -1.0061396160665481e-016,
			sigmaOut: 2.0238278069210640,
			qmaxOut:  1.6602870234255678,
			nFailOut: 5,
			iterOut:  105,
			nDivOut:  1409,
			ttypeOut: -9,
			dmin1Out: 6.6951396570468849e-002,
			dmin2Out: 0.62588280745004166,
			dnOut:    0.29773666692112338,
			dn1Out:   6.6951396570468849e-002,
			dn2Out:   0.62588280745004166,
			gOut:     0.49975000000000003,
			tauOut:   2.2317132187369120e-002,
		},
		{
			i0:       1,
			n0:       3,
			z:        []float64{0.6258828074500417e+00, 0.6481999396374107e+00, 0.7879513719234823e-20, 0.5524513774965514e-19, 0.6695139660044724e-01, 0.8926852875783797e-01, 0.1433084218388560e-09, 0.2997839428604580e-10, 0.2977366669211234e+00, 0.3200537992518009e+00, 0.8926852875783797e-01, 0.2644269217171004e-32, 0.2001510674733695e+01, 0.2227952244484021e-23, 0.4011480356653261e-18, 0.5524513774965514e-19, 0.1702794694134603e+01, 0.2545269924803487e-17, 0.5179704803914515e-14, 0.1264160184270297e-13, 0.1077066053646038e+01, 0.7156177943897596e-01, 0.2292199980814605e-03, 0.1899018778701386e-03, 0.1060140274732043e+01, 0.8660746506696473e-01, 0.1357005210961402e-39, 0.1331360138522907e-23, 0.9894235909971354e+00, 0.8804208964992894e-17, 0.8660746506696473e-01, 0.8679736700028205e-01, 0.9539342071687115e+00, 0.2761013168273541e-29, 0.1040577915698430e-22, 0.8107757314001177e-24, 0.8046649468928653e+00, 0.5098326619997980e-22, 0.1416795225784663e-10, 0.1216165317638265e-10, 0.7009142227469247e+00, 0.1183291357831518e-29, 0.4806926318247711e-11, 0.3824647564366386e-11, 0.5502651835254770e+00, 0.1368072596837427e-11, 0.1169503544861386e-10, 0.4068037876491279e-10, 0.4423863025187732e+00, 0.2477754901417239e-20, 0.1737857614720001e-07, 0.5295826057530262e-07, 0.3697086796938907e+00, 0.5143373102040997e-26, 0.1312431380925897e-05, 0.1614374370413396e-05, 0.1446051340026323e+00, 0.9928287808749566e-25, 0.3777515963415321e-05, 0.2773141909621761e-05, 0.1110032523123295e+00, 0.0000000000000000e+00, 0.2532463507333992e-05, 0.2934028940292093e-05, 0.7513603923341917e-01, 0.5394210206791908e-19, 0.6032617175984252e-07, 0.2575740214720034e-06, 0.6860214953971246e-01, 0.7542232825258426e-19, 0.3784397549471832e-09, 0.1131370986389306e-09, 0.2434478048112329e-01, 0.1013559914197709e-18, 0.1483665133446019e-05, 0.1887408451311279e-05, 0.6989818532012803e-03, 0.9466330862652142e-28, 0.2539111990097616e-03, 0.9240506410060376e-03, 0.7811996215926567e-04, 0.3786532345060857e-28, 0.1283811140869274e-03, 0.1684766835109566e-03},
			pp:       0,
			dmin:     6.6951396570468849e-002,
			desig:    -1.0061396160665481e-016,
			qmax:     1.6602870234255678,
			ttype:    -9,
			dmin1:    6.6951396570468849e-002,
			dmin2:    0.62588280745004166,
			dn:       0.29773666692112338,
			dn1:      6.6951396570468849e-002,
			dn2:      0.62588280745004166,
			g:        0.49975000000000003,
			tau:      2.2317132187369120e-002,
			nFail:    5,
			iter:     105,
			sigma:    2.0238278069210640,
			nDiv:     1409,
			zOut:     []float64{0.6258828074500417e+00, 0.5589320748538995e+00, 0.7879513719234823e-20, 0.9438435755776795e-21, 0.6695139660044724e-01, 0.6641476135588615e-06, 0.1433084218388560e-09, 0.6424501268835132e-04, 0.2977366669211234e+00, 0.2307216893122929e+00, 0.8926852875783797e-01, 0.6695139660044724e-01, 0.2001510674733695e+01, 0.2227952244484021e-23, 0.4011480356653261e-18, 0.5524513774965514e-19, 0.1702794694134603e+01, 0.2545269924803487e-17, 0.5179704803914515e-14, 0.1264160184270297e-13, 0.1077066053646038e+01, 0.7156177943897596e-01, 0.2292199980814605e-03, 0.1899018778701386e-03, 0.1060140274732043e+01, 0.8660746506696473e-01, 0.1357005210961402e-39, 0.1331360138522907e-23, 0.9894235909971354e+00, 0.8804208964992894e-17, 0.8660746506696473e-01, 0.8679736700028205e-01, 0.9539342071687115e+00, 0.2761013168273541e-29, 0.1040577915698430e-22, 0.8107757314001177e-24, 0.8046649468928653e+00, 0.5098326619997980e-22, 0.1416795225784663e-10, 0.1216165317638265e-10, 0.7009142227469247e+00, 0.1183291357831518e-29, 0.4806926318247711e-11, 0.3824647564366386e-11, 0.5502651835254770e+00, 0.1368072596837427e-11, 0.1169503544861386e-10, 0.4068037876491279e-10, 0.4423863025187732e+00, 0.2477754901417239e-20, 0.1737857614720001e-07, 0.5295826057530262e-07, 0.3697086796938907e+00, 0.5143373102040997e-26, 0.1312431380925897e-05, 0.1614374370413396e-05, 0.1446051340026323e+00, 0.9928287808749566e-25, 0.3777515963415321e-05, 0.2773141909621761e-05, 0.1110032523123295e+00, 0.0000000000000000e+00, 0.2532463507333992e-05, 0.2934028940292093e-05, 0.7513603923341917e-01, 0.5394210206791908e-19, 0.6032617175984252e-07, 0.2575740214720034e-06, 0.6860214953971246e-01, 0.7542232825258426e-19, 0.3784397549471832e-09, 0.1131370986389306e-09, 0.2434478048112329e-01, 0.1013559914197709e-18, 0.1483665133446019e-05, 0.1887408451311279e-05, 0.6989818532012803e-03, 0.9466330862652142e-28, 0.2539111990097616e-03, 0.9240506410060376e-03, 0.7811996215926567e-04, 0.3786532345060857e-28, 0.1283811140869274e-03, 0.1684766835109566e-03},
			i0Out:    1,
			n0Out:    3,
			ppOut:    0,
			dminOut:  6.6400430513702258e-007,
			desigOut: 1.9428902930940239e-016,
			sigmaOut: 2.0907785395172058,
			qmaxOut:  1.6602870234255678,
			nFailOut: 5,
			iterOut:  106,
			nDivOut:  1413,
			ttypeOut: -4,
			dmin1Out: 6.6400430513702258e-007,
			dmin2Out: 0.55893207485389951,
			dnOut:    0.23072168931229292,
			dn1Out:   6.6400430513702258e-007,
			dn2Out:   0.55893207485389951,
			gOut:     0.49975000000000003,
			tauOut:   6.6950732596142107e-002,
		},
		{
			i0:       1,
			n0:       3,
			z:        []float64{0.6258828074500417e+00, 0.5589320748538995e+00, 0.7879513719234823e-20, 0.9438435755776795e-21, 0.6695139660044724e-01, 0.6641476135588615e-06, 0.1433084218388560e-09, 0.6424501268835132e-04, 0.2977366669211234e+00, 0.2307216893122929e+00, 0.8926852875783797e-01, 0.6695139660044724e-01, 0.2001510674733695e+01, 0.2227952244484021e-23, 0.4011480356653261e-18, 0.5524513774965514e-19, 0.1702794694134603e+01, 0.2545269924803487e-17, 0.5179704803914515e-14, 0.1264160184270297e-13, 0.1077066053646038e+01, 0.7156177943897596e-01, 0.2292199980814605e-03, 0.1899018778701386e-03, 0.1060140274732043e+01, 0.8660746506696473e-01, 0.1357005210961402e-39, 0.1331360138522907e-23, 0.9894235909971354e+00, 0.8804208964992894e-17, 0.8660746506696473e-01, 0.8679736700028205e-01, 0.9539342071687115e+00, 0.2761013168273541e-29, 0.1040577915698430e-22, 0.8107757314001177e-24, 0.8046649468928653e+00, 0.5098326619997980e-22, 0.1416795225784663e-10, 0.1216165317638265e-10, 0.7009142227469247e+00, 0.1183291357831518e-29, 0.4806926318247711e-11, 0.3824647564366386e-11, 0.5502651835254770e+00, 0.1368072596837427e-11, 0.1169503544861386e-10, 0.4068037876491279e-10, 0.4423863025187732e+00, 0.2477754901417239e-20, 0.1737857614720001e-07, 0.5295826057530262e-07, 0.3697086796938907e+00, 0.5143373102040997e-26, 0.1312431380925897e-05, 0.1614374370413396e-05, 0.1446051340026323e+00, 0.9928287808749566e-25, 0.3777515963415321e-05, 0.2773141909621761e-05, 0.1110032523123295e+00, 0.0000000000000000e+00, 0.2532463507333992e-05, 0.2934028940292093e-05, 0.7513603923341917e-01, 0.5394210206791908e-19, 0.6032617175984252e-07, 0.2575740214720034e-06, 0.6860214953971246e-01, 0.7542232825258426e-19, 0.3784397549471832e-09, 0.1131370986389306e-09, 0.2434478048112329e-01, 0.1013559914197709e-18, 0.1483665133446019e-05, 0.1887408451311279e-05, 0.6989818532012803e-03, 0.9466330862652142e-28, 0.2539111990097616e-03, 0.9240506410060376e-03, 0.7811996215926567e-04, 0.3786532345060857e-28, 0.1283811140869274e-03, 0.1684766835109566e-03},
			pp:       1,
			dmin:     6.6400430513702258e-007,
			desig:    1.9428902930940239e-016,
			qmax:     1.6602870234255678,
			ttype:    -4,
			dmin1:    6.6400430513702258e-007,
			dmin2:    0.55893207485389951,
			dn:       0.23072168931229292,
			dn1:      6.6400430513702258e-007,
			dn2:      0.55893207485389951,
			g:        0.49975000000000003,
			tau:      6.6950732596142107e-002,
			nFail:    5,
			iter:     106,
			sigma:    2.0907785395172058,
			nDiv:     1413,
			zOut:     []float64{0.5589315065851642e+00, 0.5589320748538995e+00, 0.1121517486324177e-26, 0.9438435755776795e-21, 0.6434089156657428e-04, 0.6641476135588615e-06, 0.2303778747300831e+00, 0.6424501268835132e-04, 0.3432463134744483e-03, 0.2307216893122929e+00, 0.6641476135588615e-06, 0.6695139660044724e-01, 0.2001510674733695e+01, 0.2227952244484021e-23, 0.4011480356653261e-18, 0.5524513774965514e-19, 0.1702794694134603e+01, 0.2545269924803487e-17, 0.5179704803914515e-14, 0.1264160184270297e-13, 0.1077066053646038e+01, 0.7156177943897596e-01, 0.2292199980814605e-03, 0.1899018778701386e-03, 0.1060140274732043e+01, 0.8660746506696473e-01, 0.1357005210961402e-39, 0.1331360138522907e-23, 0.9894235909971354e+00, 0.8804208964992894e-17, 0.8660746506696473e-01, 0.8679736700028205e-01, 0.9539342071687115e+00, 0.2761013168273541e-29, 0.1040577915698430e-22, 0.8107757314001177e-24, 0.8046649468928653e+00, 0.5098326619997980e-22, 0.1416795225784663e-10, 0.1216165317638265e-10, 0.7009142227469247e+00, 0.1183291357831518e-29, 0.4806926318247711e-11, 0.3824647564366386e-11, 0.5502651835254770e+00, 0.1368072596837427e-11, 0.1169503544861386e-10, 0.4068037876491279e-10, 0.4423863025187732e+00, 0.2477754901417239e-20, 0.1737857614720001e-07, 0.5295826057530262e-07, 0.3697086796938907e+00, 0.5143373102040997e-26, 0.1312431380925897e-05, 0.1614374370413396e-05, 0.1446051340026323e+00, 0.9928287808749566e-25, 0.3777515963415321e-05, 0.2773141909621761e-05, 0.1110032523123295e+00, 0.0000000000000000e+00, 0.2532463507333992e-05, 0.2934028940292093e-05, 0.7513603923341917e-01, 0.5394210206791908e-19, 0.6032617175984252e-07, 0.2575740214720034e-06, 0.6860214953971246e-01, 0.7542232825258426e-19, 0.3784397549471832e-09, 0.1131370986389306e-09, 0.2434478048112329e-01, 0.1013559914197709e-18, 0.1483665133446019e-05, 0.1887408451311279e-05, 0.6989818532012803e-03, 0.9466330862652142e-28, 0.2539111990097616e-03, 0.9240506410060376e-03, 0.7811996215926567e-04, 0.3786532345060857e-28, 0.1283811140869274e-03, 0.1684766835109566e-03},
			i0Out:    1,
			n0Out:    3,
			ppOut:    1,
			dminOut:  9.5878878222950548e-008,
			desigOut: -9.3233975295464906e-017,
			sigmaOut: 2.0907791077859414,
			qmaxOut:  1.6602870234255678,
			nFailOut: 6,
			iterOut:  108,
			nDivOut:  1421,
			ttypeOut: -15,
			dmin1Out: 9.5878878222950548e-008,
			dmin2Out: 0.55893150658516422,
			dnOut:    3.4324631347444829e-004,
			dn1Out:   9.5878878222950548e-008,
			dn2Out:   0.55893150658516422,
			gOut:     0.49975000000000003,
			tauOut:   5.6826873533591094e-007,
		},
		{
			i0:       1,
			n0:       3,
			z:        []float64{0.5589315065851642e+00, 0.5589320748538995e+00, 0.1121517486324177e-26, 0.9438435755776795e-21, 0.6434089156657428e-04, 0.6641476135588615e-06, 0.2303778747300831e+00, 0.6424501268835132e-04, 0.3432463134744483e-03, 0.2307216893122929e+00, 0.6641476135588615e-06, 0.6695139660044724e-01, 0.2001510674733695e+01, 0.2227952244484021e-23, 0.4011480356653261e-18, 0.5524513774965514e-19, 0.1702794694134603e+01, 0.2545269924803487e-17, 0.5179704803914515e-14, 0.1264160184270297e-13, 0.1077066053646038e+01, 0.7156177943897596e-01, 0.2292199980814605e-03, 0.1899018778701386e-03, 0.1060140274732043e+01, 0.8660746506696473e-01, 0.1357005210961402e-39, 0.1331360138522907e-23, 0.9894235909971354e+00, 0.8804208964992894e-17, 0.8660746506696473e-01, 0.8679736700028205e-01, 0.9539342071687115e+00, 0.2761013168273541e-29, 0.1040577915698430e-22, 0.8107757314001177e-24, 0.8046649468928653e+00, 0.5098326619997980e-22, 0.1416795225784663e-10, 0.1216165317638265e-10, 0.7009142227469247e+00, 0.1183291357831518e-29, 0.4806926318247711e-11, 0.3824647564366386e-11, 0.5502651835254770e+00, 0.1368072596837427e-11, 0.1169503544861386e-10, 0.4068037876491279e-10, 0.4423863025187732e+00, 0.2477754901417239e-20, 0.1737857614720001e-07, 0.5295826057530262e-07, 0.3697086796938907e+00, 0.5143373102040997e-26, 0.1312431380925897e-05, 0.1614374370413396e-05, 0.1446051340026323e+00, 0.9928287808749566e-25, 0.3777515963415321e-05, 0.2773141909621761e-05, 0.1110032523123295e+00, 0.0000000000000000e+00, 0.2532463507333992e-05, 0.2934028940292093e-05, 0.7513603923341917e-01, 0.5394210206791908e-19, 0.6032617175984252e-07, 0.2575740214720034e-06, 0.6860214953971246e-01, 0.7542232825258426e-19, 0.3784397549471832e-09, 0.1131370986389306e-09, 0.2434478048112329e-01, 0.1013559914197709e-18, 0.1483665133446019e-05, 0.1887408451311279e-05, 0.6989818532012803e-03, 0.9466330862652142e-28, 0.2539111990097616e-03, 0.9240506410060376e-03, 0.7811996215926567e-04, 0.3786532345060857e-28, 0.1283811140869274e-03, 0.1684766835109566e-03},
			pp:       0,
			dmin:     9.5878878222950548e-008,
			desig:    -9.3233975295464906e-017,
			qmax:     1.6602870234255678,
			ttype:    -15,
			dmin1:    9.5878878222950548e-008,
			dmin2:    0.55893150658516422,
			dn:       3.4324631347444829e-004,
			dn1:      9.5878878222950548e-008,
			dn2:      0.55893150658516422,
			g:        0.49975000000000003,
			tau:      5.6826873533591094e-007,
			nFail:    6,
			iter:     108,
			sigma:    2.0907791077859414,
			nDiv:     1421,
			zOut:     []float64{0.5589315065851642e+00, 0.5589314123732620e+00, 0.1121517486324177e-26, 0.1291024862446124e-30, 0.6434089156657428e-04, 0.2304421214097475e+00, 0.2303778747300831e+00, 0.3431506172718059e-03, 0.3432463134744483e-03, 0.1484300435548105e-08, 0.6641476135588615e-06, 0.6434089156657428e-04, 0.2001510674733695e+01, 0.2227952244484021e-23, 0.4011480356653261e-18, 0.5524513774965514e-19, 0.1702794694134603e+01, 0.2545269924803487e-17, 0.5179704803914515e-14, 0.1264160184270297e-13, 0.1077066053646038e+01, 0.7156177943897596e-01, 0.2292199980814605e-03, 0.1899018778701386e-03, 0.1060140274732043e+01, 0.8660746506696473e-01, 0.1357005210961402e-39, 0.1331360138522907e-23, 0.9894235909971354e+00, 0.8804208964992894e-17, 0.8660746506696473e-01, 0.8679736700028205e-01, 0.9539342071687115e+00, 0.2761013168273541e-29, 0.1040577915698430e-22, 0.8107757314001177e-24, 0.8046649468928653e+00, 0.5098326619997980e-22, 0.1416795225784663e-10, 0.1216165317638265e-10, 0.7009142227469247e+00, 0.1183291357831518e-29, 0.4806926318247711e-11, 0.3824647564366386e-11, 0.5502651835254770e+00, 0.1368072596837427e-11, 0.1169503544861386e-10, 0.4068037876491279e-10, 0.4423863025187732e+00, 0.2477754901417239e-20, 0.1737857614720001e-07, 0.5295826057530262e-07, 0.3697086796938907e+00, 0.5143373102040997e-26, 0.1312431380925897e-05, 0.1614374370413396e-05, 0.1446051340026323e+00, 0.9928287808749566e-25, 0.3777515963415321e-05, 0.2773141909621761e-05, 0.1110032523123295e+00, 0.0000000000000000e+00, 0.2532463507333992e-05, 0.2934028940292093e-05, 0.7513603923341917e-01, 0.5394210206791908e-19, 0.6032617175984252e-07, 0.2575740214720034e-06, 0.6860214953971246e-01, 0.7542232825258426e-19, 0.3784397549471832e-09, 0.1131370986389306e-09, 0.2434478048112329e-01, 0.1013559914197709e-18, 0.1483665133446019e-05, 0.1887408451311279e-05, 0.6989818532012803e-03, 0.9466330862652142e-28, 0.2539111990097616e-03, 0.9240506410060376e-03, 0.7811996215926567e-04, 0.3786532345060857e-28, 0.1283811140869274e-03, 0.1684766835109566e-03},
			i0Out:    1,
			n0Out:    3,
			ppOut:    0,
			dminOut:  1.4843004355481051e-009,
			desigOut: -1.6593817482031092e-016,
			sigmaOut: 2.0907792019978437,
			qmaxOut:  1.6602870234255678,
			nFailOut: 6,
			iterOut:  109,
			nDivOut:  1425,
			ttypeOut: -4,
			dmin1Out: 6.4246679664367451e-005,
			dmin2Out: 0.55893141237326205,
			dnOut:    1.4843004355481051e-009,
			dn1Out:   6.4246679664367451e-005,
			dn2Out:   0.55893141237326205,
			gOut:     0.49975000000000003,
			tauOut:   9.4211902206835373e-008,
		},
		{
			i0:       1,
			n0:       3,
			z:        []float64{0.5589315065851642e+00, 0.5589314123732620e+00, 0.1121517486324177e-26, 0.1291024862446124e-30, 0.6434089156657428e-04, 0.2304421214097475e+00, 0.2303778747300831e+00, 0.3431506172718059e-03, 0.3432463134744483e-03, 0.1484300435548105e-08, 0.6641476135588615e-06, 0.6434089156657428e-04, 0.2001510674733695e+01, 0.2227952244484021e-23, 0.4011480356653261e-18, 0.5524513774965514e-19, 0.1702794694134603e+01, 0.2545269924803487e-17, 0.5179704803914515e-14, 0.1264160184270297e-13, 0.1077066053646038e+01, 0.7156177943897596e-01, 0.2292199980814605e-03, 0.1899018778701386e-03, 0.1060140274732043e+01, 0.8660746506696473e-01, 0.1357005210961402e-39, 0.1331360138522907e-23, 0.9894235909971354e+00, 0.8804208964992894e-17, 0.8660746506696473e-01, 0.8679736700028205e-01, 0.9539342071687115e+00, 0.2761013168273541e-29, 0.1040577915698430e-22, 0.8107757314001177e-24, 0.8046649468928653e+00, 0.5098326619997980e-22, 0.1416795225784663e-10, 0.1216165317638265e-10, 0.7009142227469247e+00, 0.1183291357831518e-29, 0.4806926318247711e-11, 0.3824647564366386e-11, 0.5502651835254770e+00, 0.1368072596837427e-11, 0.1169503544861386e-10, 0.4068037876491279e-10, 0.4423863025187732e+00, 0.2477754901417239e-20, 0.1737857614720001e-07, 0.5295826057530262e-07, 0.3697086796938907e+00, 0.5143373102040997e-26, 0.1312431380925897e-05, 0.1614374370413396e-05, 0.1446051340026323e+00, 0.9928287808749566e-25, 0.3777515963415321e-05, 0.2773141909621761e-05, 0.1110032523123295e+00, 0.0000000000000000e+00, 0.2532463507333992e-05, 0.2934028940292093e-05, 0.7513603923341917e-01, 0.5394210206791908e-19, 0.6032617175984252e-07, 0.2575740214720034e-06, 0.6860214953971246e-01, 0.7542232825258426e-19, 0.3784397549471832e-09, 0.1131370986389306e-09, 0.2434478048112329e-01, 0.1013559914197709e-18, 0.1483665133446019e-05, 0.1887408451311279e-05, 0.6989818532012803e-03, 0.9466330862652142e-28, 0.2539111990097616e-03, 0.9240506410060376e-03, 0.7811996215926567e-04, 0.3786532345060857e-28, 0.1283811140869274e-03, 0.1684766835109566e-03},
			pp:       1,
			dmin:     1.4843004355481051e-009,
			desig:    -1.6593817482031092e-016,
			qmax:     1.6602870234255678,
			ttype:    -4,
			dmin1:    6.4246679664367451e-005,
			dmin2:    0.55893141237326205,
			dn:       1.4843004355481051e-009,
			dn1:      6.4246679664367451e-005,
			dn2:      0.55893141237326205,
			g:        0.49975000000000003,
			tau:      9.4211902206835373e-008,
			nFail:    6,
			iter:     109,
			sigma:    2.0907792019978437,
			nDiv:     1425,
			zOut:     []float64{0.2649710614371106e+01, 0.5589314123732620e+00, 0.1121517486324177e-26, 0.1291024862446124e-30, 0.2321564474027070e+01, 0.2307852720292263e+00, 0.2303778747300831e+00, 0.3431506172718059e-03, 0.2090779203479937e+01, 0.1482093454966231e-08, 0.6641476135588615e-06, 0.6434089156657428e-04, 0.2001510674733695e+01, 0.2227952244484021e-23, 0.4011480356653261e-18, 0.5524513774965514e-19, 0.1702794694134603e+01, 0.2545269924803487e-17, 0.5179704803914515e-14, 0.1264160184270297e-13, 0.1077066053646038e+01, 0.7156177943897596e-01, 0.2292199980814605e-03, 0.1899018778701386e-03, 0.1060140274732043e+01, 0.8660746506696473e-01, 0.1357005210961402e-39, 0.1331360138522907e-23, 0.9894235909971354e+00, 0.8804208964992894e-17, 0.8660746506696473e-01, 0.8679736700028205e-01, 0.9539342071687115e+00, 0.2761013168273541e-29, 0.1040577915698430e-22, 0.8107757314001177e-24, 0.8046649468928653e+00, 0.5098326619997980e-22, 0.1416795225784663e-10, 0.1216165317638265e-10, 0.7009142227469247e+00, 0.1183291357831518e-29, 0.4806926318247711e-11, 0.3824647564366386e-11, 0.5502651835254770e+00, 0.1368072596837427e-11, 0.1169503544861386e-10, 0.4068037876491279e-10, 0.4423863025187732e+00, 0.2477754901417239e-20, 0.1737857614720001e-07, 0.5295826057530262e-07, 0.3697086796938907e+00, 0.5143373102040997e-26, 0.1312431380925897e-05, 0.1614374370413396e-05, 0.1446051340026323e+00, 0.9928287808749566e-25, 0.3777515963415321e-05, 0.2773141909621761e-05, 0.1110032523123295e+00, 0.0000000000000000e+00, 0.2532463507333992e-05, 0.2934028940292093e-05, 0.7513603923341917e-01, 0.5394210206791908e-19, 0.6032617175984252e-07, 0.2575740214720034e-06, 0.6860214953971246e-01, 0.7542232825258426e-19, 0.3784397549471832e-09, 0.1131370986389306e-09, 0.2434478048112329e-01, 0.1013559914197709e-18, 0.1483665133446019e-05, 0.1887408451311279e-05, 0.6989818532012803e-03, 0.9466330862652142e-28, 0.2539111990097616e-03, 0.9240506410060376e-03, 0.7811996215926567e-04, 0.3786532345060857e-28, 0.1283811140869274e-03, 0.1684766835109566e-03},
			i0Out:    1,
			n0Out:    0,
			ppOut:    1,
			dminOut:  1.4843004355481051e-009,
			desigOut: -1.6593817482031092e-016,
			sigmaOut: 2.0907792019978437,
			qmaxOut:  1.6602870234255678,
			nFailOut: 6,
			iterOut:  109,
			nDivOut:  1425,
			ttypeOut: -4,
			dmin1Out: 6.4246679664367451e-005,
			dmin2Out: 0.55893141237326205,
			dnOut:    1.4843004355481051e-009,
			dn1Out:   6.4246679664367451e-005,
			dn2Out:   0.55893141237326205,
			gOut:     0.49975000000000003,
			tauOut:   9.4211902206835373e-008,
		},
	} {
		z := make([]float64, len(test.z))
		copy(z, test.z)
		i0 := test.i0 - 1 // zero index
		n0 := test.n0 - 1 // zero index

		i0Out, n0Out, ppOut, dminOut, sigmaOut, desigOut, qmaxOut, nFailOut, iterOut, nDivOut, ttypeOut, dmin1Out, dmin2Out, dnOut, dn1Out, dn2Out, gOut, tauOut :=
			impl.Dlasq3(i0, n0, z, test.pp, test.dmin, test.sigma, test.desig, test.qmax, test.nFail, test.iter, test.nDiv, test.ttype, test.dmin1, test.dmin2, test.dn, test.dn1, test.dn2, test.g, test.tau)

		if !floats.EqualApprox(z, test.zOut, dTol) {
			t.Error("Z mismatch")
		}
		if i0Out != test.i0Out-1 {
			t.Errorf("i0 mismatch. Want %v, got %v", test.n0Out, n0Out)
		}
		if n0Out != test.n0Out-1 {
			t.Errorf("n0 mismatch. Want %v, got %v", test.n0Out, n0Out)
		}
		if ppOut != test.ppOut {
			t.Errorf("pp mismatch. Want %v, got %v", test.ppOut, ppOut)
		}
		if !floats.EqualWithinAbsOrRel(dminOut, test.dminOut, dTol, dTol) {
			t.Errorf("dmin mismatch. Want %v, got %v", test.dminOut, dminOut)
		}
		if !floats.EqualWithinAbsOrRel(desigOut, test.desigOut, dTol, dTol) {
			t.Errorf("desig mismatch. Want %v, got %v", test.desigOut, desigOut)
		}
		if !floats.EqualWithinAbsOrRel(sigmaOut, test.sigmaOut, dTol, dTol) {
			t.Errorf("sigma mismatch. Want %v, got %v", test.sigmaOut, sigmaOut)
		}
		if !floats.EqualWithinAbsOrRel(qmaxOut, test.qmaxOut, dTol, dTol) {
			t.Errorf("qmax mismatch. Want %v, got %v", test.qmaxOut, qmaxOut)
		}
		if nFailOut != test.nFailOut {
			t.Errorf("nFail mismatch. Want %v, got %v", test.nFailOut, nFailOut)
		}
		if iterOut != test.iterOut {
			t.Errorf("iter mismatch. Want %v, got %v", test.iterOut, iterOut)
		}
		if nDivOut != test.nDivOut {
			t.Errorf("nFail mismatch. Want %v, got %v", test.nDivOut, nDivOut)
		}
		if ttypeOut != test.ttypeOut {
			t.Errorf("ttype mismatch. Want %v, got %v", test.ttypeOut, ttypeOut)
		}
		if !floats.EqualWithinAbsOrRel(dmin1Out, test.dmin1Out, dTol, dTol) {
			t.Errorf("dmin1 mismatch. Want %v, got %v", test.dmin1Out, dmin1Out)
		}
		if !floats.EqualWithinAbsOrRel(dmin2Out, test.dmin2Out, dTol, dTol) {
			t.Errorf("dmin2 mismatch. Want %v, got %v", test.dmin2Out, dmin2Out)
		}
		if !floats.EqualWithinAbsOrRel(dnOut, test.dnOut, dTol, dTol) {
			t.Errorf("dn mismatch. Want %v, got %v", test.dnOut, dnOut)
		}
		if !floats.EqualWithinAbsOrRel(dn1Out, test.dn1Out, dTol, dTol) {
			t.Errorf("dn1 mismatch. Want %v, got %v", test.dn1Out, dn1Out)
		}
		if !floats.EqualWithinAbsOrRel(dn2Out, test.dn2Out, dTol, dTol) {
			t.Errorf("dn2 mismatch. Want %v, got %v", test.dn2Out, dn2Out)
		}
		if !floats.EqualWithinAbsOrRel(gOut, test.gOut, dTol, dTol) {
			t.Errorf("g mismatch. Want %v, got %v", test.gOut, gOut)
		}
		if !floats.EqualWithinAbsOrRel(tauOut, test.tauOut, dTol, dTol) {
			t.Errorf("tau mismatch. Want %v, got %v", test.tauOut, tauOut)
		}
	}
}
