/*
 * Copyright (C) 2020 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#pragma once

#if ENABLE(LAYOUT_FORMATTING_CONTEXT)

#include "InlineRect.h"
#include "LayoutBox.h"
#include "TextFlags.h"

namespace WebCore {
namespace Layout {

struct LineRun {
    WTF_MAKE_STRUCT_FAST_ALLOCATED;
    struct Text {
        WTF_MAKE_STRUCT_FAST_ALLOCATED;
    public:
        Text(size_t position, size_t length, const String&);

        size_t start() const { return m_start; }
        size_t end() const { return start() + length(); }
        size_t length() const { return m_length; }
        String content() const { return m_contentString; }

        bool needsHyphen() const { return m_needsHyphen; }
        void setNeedsHyphen() { m_needsHyphen = true; }

        void expand(size_t delta) { m_length += delta; }
        void shrink(size_t delta) { m_length -= delta; }

    private:
        size_t m_start { 0 };
        size_t m_length { 0 };
        bool m_needsHyphen { false };
        String m_contentString;
    };

    enum class Type {
        Text,
        LineBreak,
        AtomicInlineLevelBox,
        InlineBox,
        RootInlineBox,
        GenericInlineLevelBox
    };
    struct Expansion;
    LineRun(size_t lineIndex, Type, const Box&, const InlineRect&, Expansion, std::optional<Text> = std::nullopt, bool hasContent = true, bool isLineSpanning = false);

    bool isText() const { return m_type == Type::Text; }
    bool isLineBreak() const { return m_type == Type::LineBreak; }
    bool isAtomicInlineLevelBox() const { return m_type == Type::AtomicInlineLevelBox; }
    bool isInlineBox() const { return m_type == Type::InlineBox || isRootInlineBox(); }
    bool isRootInlineBox() const { return m_type == Type::RootInlineBox; }
    Type type() const { return m_type; }

    bool hasContent() const { return m_hasContent; }
    bool isLineSpanning() const { return m_isLineSpanning; }

    const InlineRect& logicalRect() const { return m_logicalRect; }

    InlineLayoutUnit logicalTop() const { return logicalRect().top(); }
    InlineLayoutUnit logicalBottom() const { return logicalRect().bottom(); }
    InlineLayoutUnit logicalLeft() const { return logicalRect().left(); }
    InlineLayoutUnit logicalRight() const { return logicalRect().right(); }

    InlineLayoutUnit logicalWidth() const { return logicalRect().width(); }
    InlineLayoutUnit logicalHeight() const { return logicalRect().height(); }

    void moveVertically(InlineLayoutUnit offset) { m_logicalRect.moveVertically(offset); }
    std::optional<Text>& text() { return m_text; }
    const std::optional<Text>& text() const { return m_text; }

    struct Expansion {
        ExpansionBehavior behavior { DefaultExpansion };
        InlineLayoutUnit horizontalExpansion { 0 };
    };
    Expansion expansion() const { return m_expansion; }

    const Box& layoutBox() const { return *m_layoutBox; }
    size_t lineIndex() const { return m_lineIndex; }

private:
    const size_t m_lineIndex;
    const Type m_type;
    WeakPtr<const Layout::Box> m_layoutBox;
    InlineRect m_logicalRect;
    bool m_hasContent { true };
    // FIXME: This is temporary until after iterators can skip over line spanning/root inline boxes.
    bool m_isLineSpanning { false };
    Expansion m_expansion;
    std::optional<Text> m_text;
};

inline LineRun::LineRun(size_t lineIndex, Type type, const Layout::Box& layoutBox, const InlineRect& logicalRect, Expansion expansion, std::optional<Text> text, bool hasContent, bool isLineSpanning)
    : m_lineIndex(lineIndex)
    , m_type(type)
    , m_layoutBox(makeWeakPtr(layoutBox))
    , m_logicalRect(logicalRect)
    , m_hasContent(hasContent)
    , m_isLineSpanning(isLineSpanning)
    , m_expansion(expansion)
    , m_text(text)
{
}

inline LineRun::Text::Text(size_t start, size_t length, const String& contentString)
    : m_start(start)
    , m_length(length)
    , m_contentString(contentString)
{
}

}
}
#endif
