##
# Procedural model of a Alexander Schleicher ASW20 electrical system.  Includes a
# preliminary battery charge/discharge model
#
# Based on C172P electrical system.


# Initialize properties
var electrical	=	props.globals.initNode("/systems/electrical");
var serviceable	=	electrical.initNode("serviceable", 1, "BOOL");
var batt_prop	=	electrical.initNode("battery");
var output_prop	=	electrical.initNode("outputs");
var cb_prop	=	props.globals.initNode("/controls/circuit-breakers");

var volts	=	electrical.initNode("volts", 0.0, "DOUBLE");
var amps	=	electrical.initNode("amps",  0.0, "DOUBLE");

var radio	=	output_prop.initNode("comm[0]", 0.0, "DOUBLE");
var flarm	=	output_prop.initNode("flarm", 0.0, "DOUBLE");
var vario	=	output_prop.initNode("vario", 0.0, "DOUBLE");
var turn	=	output_prop.initNode("turn-coordinator", 0.0, "DOUBLE");

var turnbank_spin = props.globals.getNode("/instrumentation/turn-indicator/spin", 1);

var vario_vol = props.globals.getNode("/instrumentation/ilec-sc7/volume", 1);
var vario_aud = props.globals.getNode("/instrumentation/ilec-sc7/audio", 1);
var vario_read = props.globals.getNode("/instrumentation/ilec-sc7/te-reading-mps", 1);

var comm_ptt	=	props.globals.getNode("/instrumentation/comm[0]/ptt");
var xpdr_mode	=	props.globals.initNode("/instrumentation/transponder/inputs/knob-mode", 0, "INT");
var xpdr_brt	=	props.globals.getNode("/instrumentation/transponder/brightness", 1);

# Array of circuit breakers
var breakers = {
	master: cb_prop.getNode("master"),
	vario:	cb_prop.getNode("vario"),
	radio:	cb_prop.getNode("radio"),
	turn:	cb_prop.getNode("turn"),
};

var switches = {
	master: props.globals.getNode("/controls/electric/battery-switch"),
	turn:	props.globals.getNode("/controls/electric/turn-slip-switch"),
};

var freeze_replay	=	props.globals.getNode("/sim/freeze/replay-state");

##
# Initialize internal values
#

var vbus_volts = 0.0;

var ammeter_ave = 0.0;


##
# Battery model class.
#

var BatteryClass = {
	# Constructor
	new: func( ideal_volts, ideal_amps, amp_hours, charge_amps, n ){
		var charge_prop	= batt_prop.getNode( "charge["~n~"]" );
		var charge	= nil;
		if( getprop("/systems/electrical/battery/charge["~n~"]") != nil ){			# If the battery charge has been set from a previous FG instance
			charge = charge_prop.getDoubleValue();
		} else {
			charge = 1.0;
			charge_prop = batt_prop.initNode("charge["~n~"]", 1.0, "DOUBLE");
		}
		var obj = {
			parents: [BatteryClass],
			ideal_volts: ideal_volts,
			ideal_amps: ideal_amps,
			amp_hours: amp_hours,
			charge_amps: charge_amps,
			charge: charge,
			charge_prop: charge_prop,
			n: n 
		};
		return obj;
	},
	# Passing in positive amps means the battery will be discharged.
	# Negative amps indicates a battery charge.
	apply_load: func( amps, dt ){
		var old_charge = me.charge_prop.getDoubleValue();
		if( freeze_replay.getBoolValue() ){
			return me.amp_hours * old_charge;
		}
		var amphrs_used = amps * dt / 3600.0;
		var fraction_used = amphrs_used / me.amp_hours;
		
		var new_charge = std.max(0.0, std.min(old_charge - fraction_used, 1.0));
		
		if (new_charge < 0.1 and old_charge_percent >= 0.1)
			gui.popupTip("Warning: Low battery! Enable alternator or apply external power to recharge battery!", 10);
		me.charge = new_charge;
		me.charge_prop.setDoubleValue( new_charge );
		return me.amp_hours * new_charge;
	},
	# Return output volts based on percent charged.  Currently based on a simple
	# polynomial percent charge vs. volts function.
	get_output_volts: func() {
		var x = 1.0 - me.charge;
		var tmp = -(3.0 * x - 1.0);
		var factor = ( math.pow( tmp, 5) + 32 ) / 32;
		return me.ideal_volts * factor;
	},
	# Return output amps available.  This function is totally wrong and should be
	# fixed at some point with a more sensible function based on charge percent.
	# There is probably some physical limits to the number of instantaneous amps
	# a battery can produce (cold cranking amps?)
	get_output_amps: func() {
		var x = 1.0 - me.charge;
		var tmp = -(3.0 * x - 1.0);
		var factor = ( math.pow( tmp, 5) + 32) / 32;
		return me.ideal_amps * factor;
	},
	# Set the current charge instantly to 100 %.
	reset_to_full_charge: func() {
		me.apply_load(-(1.0 - me.charge) * me.amp_hours, 3600);
	},
	# Get current charge
	get_charge: func() {
		return me.charge;
	}
};

##	example glider battery: https://shop.segelflugbedarf24.de/Flugzeugausstattung/Akkus-Energieversorgung/Sonnenschein-Dryfit-A512-12V/6-5-Ah::731.html
##				http://www.sonnenschein.org/A500.htm	(A512-6.5S)
##				ideal volts: 12.0
##				ideal amps: 0.325 (max. 80 / 300 for 5 sec))
##				amp hours: 6.5
##				charge amps: 25

var battery = BatteryClass.new( 12.0, 0.325, 6.5, 25, 0);

var reset_circuit_breakers = func {
	# Reset circuit breakers
	setprop("/controls/circuit-breakers/master", 1);
	setprop("/controls/circuit-breakers/vario", 1);
	setprop("/controls/circuit-breakers/radio", 1);
	setprop("/controls/circuit-breakers/turn", 1);
}

var recharge_battery = func {
	# Charge battery to 100 %
	battery.reset_to_full_charge();
}
##
# This is the main electrical system update function.
#

var ElectricalSystemUpdater = {
	new : func {
		var m = {
			parents: [ElectricalSystemUpdater]
		};
		# Request that the update function be called each frame
		m.loop = updateloop.UpdateLoop.new(components: [m], update_period: 0.0, enable: 0);
		return m;
	},
	
	enable: func {
		me.loop.reset();
		me.loop.enable();
	},
	
	disable: func {
		me.loop.disable();
	},
	
	reset: func {
		# Do nothing
	},
	
	update: func (dt) {
		update_virtual_bus(dt);
	}
};

##
# Model the system of relays and connections that join the battery,master/alt switches.
#

var update_virtual_bus = func (dt) {
	var load = 0.0;
	var battery_volts = 0.0;
	var bus_volts = 0.0;
	if ( serviceable.getBoolValue() and breakers.master.getBoolValue() ) {
		battery_volts = battery.get_output_volts();
	}
	
	# determine power source
	if ( switches.master.getBoolValue() ){
		bus_volts = battery_volts;
		var power_source = battery;
	}
	
	load += electrical_bus( bus_volts );
	
	# swtich the master breaker off if load is out of limits
	if ( load > 55 ) {
		breakers.master.setBoolValue( 0 );
	}
	
	# system loads and ammeter gauge
	var ammeter = 0.0;
	if ( bus_volts > 1.0 ) {
		# ammeter gauge
		if ( power_source == "battery" ) {
			ammeter = -load;
		} else {
			ammeter = battery.charge_amps;
		}
	}
	
	# charge/discharge the battery
	battery.apply_load( load, dt );
	
	# filter ammeter needle pos
	ammeter_ave = 0.8 * ammeter_ave + 0.2 * ammeter;
	
	# outputs
	if (bus_volts > 12)
		vbus_volts = bus_volts;
	else
		vbus_volts = 0.0;
	
	amps.setDoubleValue( ammeter_ave );
	volts.setDoubleValue( bus_volts );
}

#Load sources:
#	com:		https://www.becker-avionics.com/wp-content/uploads/00_Literature/62Series_DS_en.pdf, p.2
#	vario:		http://www.ilec-gmbh.com/ilec/manuals/SC7pd.pdf
#	turn:		https://www.airteam.eu/de/p/falcon-tb02e-2-1 (not the same but similar)
#	flarm:		http://flarm.com/wp-content/uploads/man/FLARM_InstallationManual_D.pdf
#	flarm display:	https://www.air-store.eu/Display-V3-FLARM

var electrical_bus = func( bus_volts ) {
	var load = 0.0;
	
	if( bus_volts > 9 ){	# minimal working voltage
		# Radio
		if( breakers.radio.getBoolValue() ){
			radio.setDoubleValue( bus_volts );
			if( comm_ptt.getBoolValue() ){
				load += 18 / bus_volts; # transmitting: <= 2 A at 12V
			}else{
				load += 1.56 / bus_volts; # standby: <= 140 mA at 12V
			}
		} else {
			radio.setDoubleValue( 0.0 );
		}
	
		# Vario
		if( breakers.vario.getBoolValue() ){
			vario.setDoubleValue( bus_volts );
			#Energy consumption:	25mA (medium volume) 60mA (max volume) -> guess: at 12V
			#			guess: base consumption 5mA (no volume)
			load += 0.06 / bus_volts;
			if(vario_aud.getValue() == 2 or (vario_aud.getValue() == 1 and vario_read.getValue() > 0)){
				load += (vario_vol.getValue()*0.66) / bus_volts;
			}
		} else {
			vario.setDoubleValue( 0.0 );
		}
		
		
		#Turn Coordinator
		#Energy Consumption:
		#	starting ~9.9 / volts (approx)
		#	running ~7.8 / volts (approx)
		if( breakers.turn.getBoolValue() and switches.turn.getBoolValue() ) {
			turn.setDoubleValue( bus_volts );
			if( turnbank_spin.getDoubleValue() > 0.99 ){
				load += 7.8 / bus_volts;
			}else{
				load += 9.9 / bus_volts;
			}
		} else {
			turn.setDoubleValue( 0.0 );
		}
		
	} else {
		radio.setDoubleValue( 0.0 );
		vario.setDoubleValue( 0.0 );
		turn.setDoubleValue( 0.0 );
	}
	
	
	# FLARM
	if( bus_volts > 8 ){	# minimal working voltage
		flarm.setDoubleValue( bus_volts );
		load += 0.66 / bus_volts; #FLARM
		load += 0.12 / bus_volts; #FLARM display
	} else {
		flarm.setDoubleValue( 0.0 );
	}
	
	# return cumulative load
	return load;
}

##
# Initialize the electrical system
#

var system_updater = ElectricalSystemUpdater.new();

setlistener("/sim/signals/fdm-initialized", func {
	# checking if battery should be automatically recharged
	if (!getprop("/systems/electrical/save-battery-charge")) {
		battery.reset_to_full_charge();
	};
	
	system_updater.enable();
	print("Electrical system initialized");
});
