/*
 * Decompiled with CFR 0.152.
 */
package org.apache.wicket.pageStore;

import java.io.Closeable;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.OutputStream;
import java.io.RandomAccessFile;
import java.io.Serializable;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import org.apache.wicket.WicketRuntimeException;
import org.apache.wicket.pageStore.IDataStore;
import org.apache.wicket.pageStore.PageWindowManager;
import org.apache.wicket.util.file.Files;
import org.apache.wicket.util.io.IOUtils;
import org.apache.wicket.util.lang.Args;
import org.apache.wicket.util.lang.Bytes;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DiskDataStore
implements IDataStore {
    private static final Logger log = LoggerFactory.getLogger(DiskDataStore.class);
    private static final String INDEX_FILE_NAME = "DiskDataStoreIndex";
    private final String applicationName;
    private final Bytes maxSizePerPageSession;
    private final File fileStoreFolder;
    private final ConcurrentMap<String, SessionEntry> sessionEntryMap;

    public DiskDataStore(String applicationName, File fileStoreFolder, Bytes maxSizePerSession) {
        this.applicationName = applicationName;
        this.fileStoreFolder = fileStoreFolder;
        this.maxSizePerPageSession = (Bytes)Args.notNull((Object)maxSizePerSession, (String)"maxSizePerSession");
        this.sessionEntryMap = new ConcurrentHashMap<String, SessionEntry>();
        try {
            if (this.fileStoreFolder.exists() || this.fileStoreFolder.mkdirs()) {
                this.loadIndex();
            } else {
                log.warn("Cannot create file store folder for some reason.");
            }
        }
        catch (SecurityException e) {
            throw new WicketRuntimeException("SecurityException occurred while creating DiskDataStore. Consider using a non-disk based IDataStore implementation. See org.apache.wicket.Application.setPageManagerProvider(IPageManagerProvider)", e);
        }
    }

    @Override
    public void destroy() {
        log.debug("Destroying...");
        this.saveIndex();
        log.debug("Destroyed.");
    }

    @Override
    public byte[] getData(String sessionId, int id) {
        byte[] pageData = null;
        SessionEntry sessionEntry = this.getSessionEntry(sessionId, false);
        if (sessionEntry != null) {
            pageData = sessionEntry.loadPage(id);
        }
        if (log.isDebugEnabled()) {
            log.debug("Returning data{} for page with id '{}' in session with id '{}'", new Object[]{pageData != null ? "" : "(null)", id, sessionId});
        }
        return pageData;
    }

    @Override
    public boolean isReplicated() {
        return false;
    }

    @Override
    public void removeData(String sessionId, int id) {
        SessionEntry sessionEntry = this.getSessionEntry(sessionId, false);
        if (sessionEntry != null) {
            if (log.isDebugEnabled()) {
                log.debug("Removing data for page with id '{}' in session with id '{}'", (Object)id, (Object)sessionId);
            }
            sessionEntry.removePage(id);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void removeData(String sessionId) {
        SessionEntry sessionEntry = this.getSessionEntry(sessionId, false);
        if (sessionEntry != null) {
            log.debug("Removing data for pages in session with id '{}'", (Object)sessionId);
            SessionEntry sessionEntry2 = sessionEntry;
            synchronized (sessionEntry2) {
                this.sessionEntryMap.remove(sessionEntry.sessionId);
                sessionEntry.unbind();
            }
        }
    }

    @Override
    public void storeData(String sessionId, int id, byte[] data) {
        SessionEntry sessionEntry = this.getSessionEntry(sessionId, true);
        if (sessionEntry != null) {
            if (log.isDebugEnabled()) {
                log.debug("Storing data for page with id '{}' in session with id '{}'", (Object)id, (Object)sessionId);
            }
            sessionEntry.savePage(id, data);
        }
    }

    protected SessionEntry getSessionEntry(String sessionId, boolean create) {
        if (!create) {
            return (SessionEntry)this.sessionEntryMap.get(sessionId);
        }
        SessionEntry entry = new SessionEntry(this, sessionId);
        SessionEntry existing = this.sessionEntryMap.putIfAbsent(sessionId, entry);
        return existing != null ? existing : entry;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void loadIndex() {
        File storeFolder = this.getStoreFolder();
        File index = new File(storeFolder, INDEX_FILE_NAME);
        if (index.exists() && index.length() > 0L) {
            try {
                FileInputStream stream = new FileInputStream(index);
                ObjectInputStream ois = new ObjectInputStream(stream);
                try {
                    Map map = (Map)ois.readObject();
                    this.sessionEntryMap.clear();
                    this.sessionEntryMap.putAll(map);
                    for (Map.Entry entry : this.sessionEntryMap.entrySet()) {
                        SessionEntry sessionEntry = (SessionEntry)entry.getValue();
                        sessionEntry.diskDataStore = this;
                    }
                }
                finally {
                    ((InputStream)stream).close();
                    ois.close();
                }
            }
            catch (Exception e) {
                log.error("Couldn't load DiskDataStore index from file " + index + ".", (Throwable)e);
            }
        }
        Files.remove((File)index);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void saveIndex() {
        File storeFolder = this.getStoreFolder();
        if (storeFolder.exists()) {
            File index = new File(storeFolder, INDEX_FILE_NAME);
            Files.remove((File)index);
            try {
                FileOutputStream stream = new FileOutputStream(index);
                ObjectOutputStream oos = new ObjectOutputStream(stream);
                try {
                    HashMap map = new HashMap(this.sessionEntryMap.size());
                    for (Map.Entry e : this.sessionEntryMap.entrySet()) {
                        if (((SessionEntry)e.getValue()).unbound) continue;
                        map.put(e.getKey(), e.getValue());
                    }
                    oos.writeObject(map);
                }
                finally {
                    ((OutputStream)stream).close();
                    oos.close();
                }
            }
            catch (Exception e) {
                log.error("Couldn't write DiskDataStore index to file " + index + ".", (Throwable)e);
            }
        }
    }

    private String getSessionFileName(String sessionId, boolean createSessionFolder) {
        File sessionFolder = this.getSessionFolder(sessionId, createSessionFolder);
        return new File(sessionFolder, "data").getAbsolutePath();
    }

    protected File getStoreFolder() {
        return new File(this.fileStoreFolder, this.applicationName + "-filestore");
    }

    protected File getSessionFolder(String sessionId, boolean create) {
        File storeFolder = this.getStoreFolder();
        sessionId = sessionId.replace('*', '_');
        sessionId = sessionId.replace('/', '_');
        sessionId = sessionId.replace(':', '_');
        sessionId = this.createPathFrom(sessionId);
        File sessionFolder = new File(storeFolder, sessionId);
        if (create && !sessionFolder.exists()) {
            Files.mkdirs((File)sessionFolder);
        }
        return sessionFolder;
    }

    private String createPathFrom(String sessionId) {
        int hash = Math.abs(sessionId.hashCode());
        String low = String.valueOf(hash % 9973);
        String high = String.valueOf(hash / 9973 % 9973);
        StringBuilder bs = new StringBuilder(sessionId.length() + 10);
        bs.append(low);
        bs.append(File.separator);
        bs.append(high);
        bs.append(File.separator);
        bs.append(sessionId);
        return bs.toString();
    }

    @Override
    public boolean canBeAsynchronous() {
        return true;
    }

    protected static class SessionEntry
    implements Serializable {
        private static final long serialVersionUID = 1L;
        private final String sessionId;
        private transient DiskDataStore diskDataStore;
        private String fileName;
        private PageWindowManager manager;
        private boolean unbound = false;

        protected SessionEntry(DiskDataStore diskDataStore, String sessionId) {
            this.diskDataStore = diskDataStore;
            this.sessionId = sessionId;
        }

        public PageWindowManager getManager() {
            if (this.manager == null) {
                this.manager = new PageWindowManager(this.diskDataStore.maxSizePerPageSession.bytes());
            }
            return this.manager;
        }

        private String getFileName() {
            if (this.fileName == null) {
                this.fileName = this.diskDataStore.getSessionFileName(this.sessionId, true);
            }
            return this.fileName;
        }

        public String getSessionId() {
            return this.sessionId;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public synchronized void savePage(int pageId, byte[] data) {
            if (this.unbound) {
                return;
            }
            if (data != null) {
                PageWindowManager.PageWindow window = this.getManager().createPageWindow(pageId, data.length);
                FileChannel channel = this.getFileChannel(true);
                if (channel != null) {
                    try {
                        channel.write(ByteBuffer.wrap(data), window.getFilePartOffset());
                    }
                    catch (IOException e) {
                        log.error("Error writing to a channel " + channel, (Throwable)e);
                    }
                    finally {
                        IOUtils.closeQuietly((Closeable)channel);
                    }
                } else {
                    log.warn("Cannot save page with id '{}' because the data file cannot be opened.", (Object)pageId);
                }
            }
        }

        public synchronized void removePage(int pageId) {
            if (this.unbound) {
                return;
            }
            this.getManager().removePage(pageId);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public byte[] loadPage(PageWindowManager.PageWindow window) {
            byte[] result = null;
            FileChannel channel = this.getFileChannel(false);
            if (channel != null) {
                ByteBuffer buffer = ByteBuffer.allocate(window.getFilePartSize());
                try {
                    channel.read(buffer, window.getFilePartOffset());
                    if (buffer.hasArray()) {
                        result = buffer.array();
                    }
                }
                catch (IOException e) {
                    log.error("Error reading from file channel " + channel, (Throwable)e);
                }
                finally {
                    IOUtils.closeQuietly((Closeable)channel);
                }
            }
            return result;
        }

        private FileChannel getFileChannel(boolean create) {
            FileChannel channel = null;
            File file = new File(this.getFileName());
            if (create || file.exists()) {
                String mode = create ? "rw" : "r";
                try {
                    RandomAccessFile randomAccessFile = new RandomAccessFile(file, mode);
                    channel = randomAccessFile.getChannel();
                }
                catch (FileNotFoundException fnfx) {
                    log.error(fnfx.getMessage(), (Throwable)fnfx);
                }
            }
            return channel;
        }

        public synchronized byte[] loadPage(int id) {
            if (this.unbound) {
                return null;
            }
            byte[] result = null;
            PageWindowManager.PageWindow window = this.getManager().getPageWindow(id);
            if (window != null) {
                result = this.loadPage(window);
            }
            return result;
        }

        public synchronized void unbind() {
            File sessionFolder = this.diskDataStore.getSessionFolder(this.sessionId, false);
            if (sessionFolder.exists()) {
                Files.removeFolder((File)sessionFolder);
                this.cleanup(sessionFolder);
            }
            this.unbound = true;
        }

        private void cleanup(File sessionFolder) {
            File low;
            File high = sessionFolder.getParentFile();
            if (high.list().length == 0 && Files.removeFolder((File)high) && (low = high.getParentFile()).list().length == 0) {
                Files.removeFolder((File)low);
            }
        }
    }
}

