/* Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.myfaces.portlet.faces.util.map;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.portlet.PortletRequest;

import javax.portlet.faces.Bridge;
import javax.portlet.faces.BridgeUtil;

/**
 * Map of portlet request param values
 */
public class PortletRequestParameterValuesMap extends PortletAbstractMap<String[]>
{
  private final PortletRequest mPortletRequest;
  private final Map<String, String[]> mInternalAttributes;
  private Map<String,String[]> mPrivateParametersMap;

  public PortletRequestParameterValuesMap(Object request, Map<String, String[]> internal)
  {
    if (request instanceof PortletRequest)
    {
      mPortletRequest = (PortletRequest) request;
      if (internal == null)
      {
        mInternalAttributes = Collections.emptyMap();
      }
      else
      {
        mInternalAttributes = new HashMap(internal);
      }
    }
    else
    {
      throw new IllegalArgumentException(ILLEGAL_ARGUMENT);
    }
  }
  
  @Override
  public boolean equals(Object o) {
      if (o == this)
          return true;

      if (!(o instanceof Map))
          return false;
      Map<String,String[]> t = (Map<String,String[]>) o;
      if (t.size() != size())
          return false;

      try {
          Iterator<Entry<String,String[]>> i = entrySet().iterator();
          while (i.hasNext()) {
              Entry<String,String[]> e = i.next();
              String key = e.getKey();
              String[] value = e.getValue();
              if (value == null) {
                  if (!(t.get(key)==null && t.containsKey(key)))
                      return false;
              } else {
                  if (!Arrays.equals(value, t.get(key)))
                      return false;
              }
          }
      } catch(ClassCastException unused) {
          return false;
      } catch(NullPointerException unused) {
          return false;
      }

      return true;
  }

  @Override
  public String[] getAttribute(String key)
  {
    if (mPortletRequest != null)
    {
      if (mPrivateParametersMap == null)
      {
         mPrivateParametersMap =  getPrivateParameterMap();
      }
      String[] params = mPrivateParametersMap.get(key);
      if (params != null)
      {
        return params;
      }
      else
      {
        return mInternalAttributes.get(key);
      }      
    }
    else
    {
      throw new IllegalArgumentException(ILLEGAL_ARGUMENT);
    }
  }

  @Override
  public void setAttribute(String key, String[] value)
  {
    throw new UnsupportedOperationException();
  }

  @Override
  public void removeAttribute(String key)
  {
    throw new UnsupportedOperationException();
  }

  @SuppressWarnings("unchecked")
  @Override
  public Enumeration<String> getAttributeNames()
  {
    if (mPortletRequest != null)
    {
      // merged list of internal parameters & request parameters
      List<String> attrNames = new ArrayList<String>(5);
      
      if (mPrivateParametersMap == null) 
      {
        mPrivateParametersMap = getPrivateParameterMap();
      }

      attrNames.addAll(mPrivateParametersMap.keySet());
      attrNames.addAll(mInternalAttributes.keySet());

      return Collections.enumeration(attrNames);
    }
    else
    {
      throw new IllegalArgumentException(ILLEGAL_ARGUMENT);
    }
  }
  
  private Map<String,String[]> getPrivateParameterMap()
  {
    // Some portlet containers (WebSphere) claim submitted (form) data aren't private parameters 
    // during an action/resource request -- instead they are all regular params -- to workaround
    // this indicate that during an action/resource all non-public params are private params
    
    Map <String, String[]> privateParams = mPortletRequest.getPrivateParameterMap();
    if ((BridgeUtil.getPortletRequestPhase() == Bridge.PortletPhase.ACTION_PHASE || 
         BridgeUtil.getPortletRequestPhase() == Bridge.PortletPhase.RESOURCE_PHASE))
    {
      Map<String, String[]> allParams = mPortletRequest.getParameterMap();
      Map<String, String[]> publicParams = mPortletRequest.getPublicParameterMap();
    
      if (publicParams.size() == 0)
      {
        privateParams = allParams;
      }
      else if (allParams.size() != (privateParams.size() + publicParams.size()))
      {
        // Construct the non-Public param Map by excluding those which are public
        // from the overall set.  This works around issue that some portlet
        // containers think getPrivateParameters should return the private
        // render parameters -- and hence its empty during an action. (WebSphere 6.1 is one case)
        privateParams = (Map<String, String[]>) new HashMap(allParams);
        for (Map.Entry<String, String[]> entry :  publicParams.entrySet())
        {
          privateParams.remove(entry.getKey());
        }
      }
    }
    return privateParams;
  }
}
