/*
 * Copyright 2002-2008 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.springframework.samples.petclinic.web;

import static org.springframework.samples.petclinic.web.ViewUtils.getRedirectForOwner;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.samples.petclinic.Clinic;
import org.springframework.samples.petclinic.Owner;
import org.springframework.samples.petclinic.Pet;
import org.springframework.samples.petclinic.Visit;
import org.springframework.samples.petclinic.validation.VisitValidator;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.validation.BindingResult;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.SessionAttributes;
import org.springframework.web.bind.support.SessionStatus;
import org.springframework.web.servlet.ModelAndView;

/**
 * Controller that handles all {@link Visit}-related requests,
 * i.e.: all requests for /petclinic/main/visit/*
 *
 * @author Juergen Hoeller
 * @author Ken Krebs
 * @author Chris Beams
 */
@Controller
@RequestMapping
@SessionAttributes(types=Visit.class)
public class VisitController {

    private final Clinic clinic;

    @Autowired
    public VisitController(Clinic clinic) {
        this.clinic = clinic;
    }

    /**
     * Custom handler for displaying the 'Add Visit' form,
     * i.e.: all GET requests for /petclinic/main/visit/add
     * <p/>
     * Note that the return type for this handler is {@literal void}, instead
     * of a {@link ModelAndView} signifying that the logical view name should be
     * determined based on convention and derived from the incoming path info.
     * "/visit/add" -&gt; "visit/add".
     * <p/>
     * {@link Model} is supplied as a parameter to the method and populated with
     * the new {@link Visit} object to be rendered.
     *
     * @param model the ui model to be populated with a new Visit object
     * 
     * @see #add(Visit, BindingResult, SessionStatus)
     * @see WEB-INF/jsp/visit/add.jsp
     */
    @RequestMapping(method = RequestMethod.GET)
    public void add(@RequestParam int petId, Model model) {
        Pet pet = this.clinic.loadPet(petId);
        Visit visit = new Visit();
        pet.addVisit(visit);
        model.addAttribute(visit);
    }

    /**
     * Custom handler for processing the submission of the 'Add Visit' form,
     * i.e.: all POST requests for /petclinic/main/visit/add.
     * <p/>
     * Note that this handler returns a string instead of a {@link ModelAndView},
     * signifying the logical view name that should be rendered.
     *
     * @param owner the newly-bound Visit object containing values from the form
     * @param result the result of the form-to-object binding process, including
     *        any errors
     * @param status allows the controller to signify that the session-scoped
     *        visit object should may be cleaned up
     *        
     * @return the name of the view to be rendered
     * 
     * @see #add(Model)
     * @see VisitValidator
     * @see SessionStatus
     * @see BindingResult
     * @see ViewUtils#getRedirectForOwner(Owner)
     * @see WEB-INF/jsp/vet/add.jsp
     * @see WEB-INF/jsp/owner/show.jsp
     */
    @RequestMapping(method = RequestMethod.POST)
    public String add(@ModelAttribute Visit visit, BindingResult result, SessionStatus status) {
        new VisitValidator().validate(visit, result);
        if (result.hasErrors()) {
            return null; // go back to the form
        }
        else {
            this.clinic.storeVisit(visit);
            status.setComplete();
            return getRedirectForOwner(visit.getPet().getOwner());
        }
    }

}
