/*
 * Copyright 2002-2008 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.springframework.config.java.internal.model;

import static org.hamcrest.CoreMatchers.*;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertThat;

import static org.springframework.config.java.internal.util.AnnotationExtractionUtils.extractMethodAnnotation;

import org.junit.Test;

import org.springframework.config.java.annotation.AutoBean;
import org.springframework.config.java.annotation.AutowireMode;
import org.springframework.config.java.internal.model.AutoBeanMethod.AbstractAutoBeanReturnTypeError;
import org.springframework.config.java.internal.util.MethodAnnotationPrototype;
import org.springframework.config.java.model.ModelClass;

import test.common.beans.ITestBean;
import test.common.beans.TestBean;

import java.lang.reflect.Modifier;

import java.util.ArrayList;


/**
 * Unit tests for {@link AutoBeanMethod}.
 *
 * @author  Chris Beams
 */
public class AutoBeanMethodTests {

    static AutoBean DEFAULT_METADATA = extractMethodAnnotation(AutoBean.class, new MethodAnnotationPrototype() {
            @AutoBean
            public void targetMethod() { }
        }.getClass());

    static AutoBean CUSTOM_METADATA = extractMethodAnnotation(AutoBean.class, new MethodAnnotationPrototype() {
            @AutoBean(autowire = AutowireMode.BY_NAME)
            public void targetMethod() { }
        }.getClass());

    static final ModelClass DEFAULT_RETURN_TYPE = ModelClass.forClass(TestBean.class);
    static final AutoBeanMethod VALID_AUTOBEAN_METHOD =
        new AutoBeanMethod("m", DEFAULT_RETURN_TYPE, 0, DEFAULT_METADATA);

    @Test
    public void testConstruction() { new AutoBeanMethod("a", DEFAULT_RETURN_TYPE, 0, DEFAULT_METADATA); }

    @Test
    public void equality() {
        AutoBeanMethod a1 = new AutoBeanMethod("a", DEFAULT_RETURN_TYPE, 0, DEFAULT_METADATA);
        AutoBeanMethod a2 = new AutoBeanMethod("a", DEFAULT_RETURN_TYPE, 0, DEFAULT_METADATA);
        assertThat(a1, equalTo(a2));
        assertThat(a2, equalTo(a1));

        a2 = new AutoBeanMethod("b", DEFAULT_RETURN_TYPE, 0, DEFAULT_METADATA);
        assertThat(a1, not(equalTo(a2)));
        assertThat(a2, not(equalTo(a1)));

        a2 = new AutoBeanMethod("a", DEFAULT_RETURN_TYPE, 0, CUSTOM_METADATA);
        assertThat(a1, not(equalTo(a2)));
        assertThat(a2, not(equalTo(a1)));

        a2 = new AutoBeanMethod("a", DEFAULT_RETURN_TYPE, Modifier.PUBLIC, DEFAULT_METADATA);
        assertThat(a1, not(equalTo(a2)));
        assertThat(a2, not(equalTo(a1)));

        ModelClass iface = ModelClass.forClass(ITestBean.class);
        a2 = new AutoBeanMethod("a", iface, 0, DEFAULT_METADATA);
        assertThat(a1, not(equalTo(a2)));

        assertThat(a1, equalTo(a1));
        assertThat(a1, not(equalTo(null)));
        assertThat(null, not(equalTo(a1)));
        assertThat(a1, not(equalTo(new Object())));
    }

    @Test
    public void validity() {
        { // valid case
            AutoBeanMethod valid = VALID_AUTOBEAN_METHOD;
            ArrayList<UsageError> errors = new ArrayList<UsageError>();
            valid.detectUsageErrors(errors);
            assertEquals(0, errors.size());
        }

        { // invalid because return type is interface
            ModelClass rtInvalid = ModelClass.forClass(ITestBean.class);
            AutoBeanMethod valid = new AutoBeanMethod("invalid", rtInvalid, 0, DEFAULT_METADATA);
            ArrayList<UsageError> errors = new ArrayList<UsageError>();
            valid.detectUsageErrors(errors);
            assertEquals(errors.toString(), 1, errors.size());
            assertThat(errors.get(0), instanceOf(AbstractAutoBeanReturnTypeError.class));
        }
    }

    @Test
    public void testAccessors() {
        AutoBeanMethod m = new AutoBeanMethod("a", DEFAULT_RETURN_TYPE, 0, DEFAULT_METADATA);
        assertThat(m.getName(), equalTo("a"));
        assertThat(m.getMetadata(), equalTo(DEFAULT_METADATA));
        assertThat(m.getReturnType(), equalTo(DEFAULT_RETURN_TYPE));
    }
}
