/*
 * Copyright 2002-2008 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.springframework.config.java.plugin.tx;

import java.lang.annotation.Documented;
import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

import org.springframework.config.java.plugin.Plugin;


/**
 * Indicates that transaction configuration is defined by Spring's
 * {@code @Transactional} annotation on bean classes,
 * and that proxies are automatically to be created for the relevant
 * annotated beans.
 * <p/>
 * Transaction semantics such as propagation settings, the isolation
 * level, the rollback rules, etc are all defined in the annotation
 * metadata.
 * <p/>
 * Represents support equivalent to Spring XML's {@code <tx:annotation-driven/>}
 *
 * @author Chris Beams
 */
@Retention(RetentionPolicy.RUNTIME)
@Target(ElementType.TYPE)
@Documented
@Plugin(handler=AnnotationDrivenTxHandler.class)
public @interface AnnotationDrivenTx {

    /**
     * Should annotated beans be proxied using Spring's AOP framework,
     * or should they rather be weaved with an AspectJ transaction aspect?
     * <p/>
     * AspectJ weaving requires spring-aspects.jar on the classpath,
     * as well as load-time weaving (or compile-time weaving) enabled.
     * <p/>
     * Default is {@link AopMode#PROXY}, meaning AspectJ will not be required.
     */
    AopMode mode() default AopMode.PROXY;

    /**
     * The bean name of the {@code PlatformTransactionManager} that is to be used
     * to drive transactions.
     * <p/>
     * This attribute is not required, and only needs to be specified
     * explicitly if the bean name of the desired {@code PlatformTransactionManager}
     * is not {@code 'transactionManager'}.
     */
    String transactionManager() default "transactionManager";

    /**
     * Controls the ordering of the execution of the transaction advisor
     * when multiple advice executes at a specific joinpoint.
     * <p/>
     * Default value is 0, which signifies no explicit ordering.
     * If explicit ordering is desired, set this value to something other
     * than zero.
     */
    int order() default 0;

    /**
     * Are class-based (CGLIB) proxies to be created? By default (false),
     * standard Java interface-based proxies are created.
     */
    boolean proxyTargetClass() default false;

}
