/*
 * Copyright 2002-2008 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.springframework.config.java.annotation;

import org.springframework.beans.factory.config.AutowireCapableBeanFactory;


/**
 * Enumeration determining autowiring mode: that is, how JavaConfig should inject dependencies, via
 * constructor, via setter, and whether by name or by type. This enumeration is constrained to the
 * semantics of the core Spring IoC container, which means that certain things are not possible
 * e.g.: declaring a bean as candidate for both constructor and setter injection.
 *
 * <p>Implementation note: Spring core provides an <code>Autowire</code> enum, but BY_CONSTRUCTOR is
 * not available therein. JavaConfig's {@link AutoBean} annotation requires the ability to specify
 * whether injection should happen via setter (default) or via constructor, thus the creation of
 * this custom enum.</p>
 *
 * @author  Chris Beams
 * @see     org.springframework.beans.factory.annotation.Autowire
 * @see     AutowireCapableBeanFactory.AUTOWIRE_*
 */
public enum AutowireMode {

    /** Constant that indicates autowiring bean properties by name. */
    BY_NAME(AutowireCapableBeanFactory.AUTOWIRE_BY_NAME),

    /** Constant that indicates autowiring bean properties by type. */
    BY_TYPE(AutowireCapableBeanFactory.AUTOWIRE_BY_TYPE),

    /** Constant that indicates autowiring bean properties via constructor. */
    CONSTRUCTOR(AutowireCapableBeanFactory.AUTOWIRE_CONSTRUCTOR);

    private final int value;

    AutowireMode(int value) { this.value = value; }

    /**
     * Integer value associated with this enum label. Values correspond to those statically defined
     * by {@link AutowireCapableBeanFactory#AUTOWIRE_BY_TYPE}, etc.
     */
    public int value() { return this.value; }
}
