# ===========
# pysap - Python library for crafting SAP's network protocols packets
#
# SECUREAUTH LABS. Copyright (C) 2021 SecureAuth Corporation. All rights reserved.
#
# The library was designed and developed by Martin Gallo from
# the SecureAuth's Innovation Labs team.
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# ==============


class RSECCipher(object):

    MODE_ENCODE = 1
    MODE_DECODE = 2

    byte_bit = [128, 64, 32, 16, 8, 4, 2, 1]

    # This is used to rotate the values
    totrot = [1, 2, 4, 6, 8, 10, 12, 14, 15, 17, 19, 21, 23, 25, 27, 28]

    # Table for the first key generation step (transposition and compression)
    # The values in this table are the ones in the DES standard.
    pc1 = [57, 49, 41, 33, 25, 17, 9, 1, 58, 50, 42, 34, 26, 18, 10, 2,
           59, 51, 43, 35, 27, 19, 11, 3, 60, 52, 44, 36, 63, 55, 47, 39,
           31, 23, 15, 7, 62, 54, 46, 38, 30, 22, 14, 6, 61, 53, 45, 37, 29,
           21, 13, 5, 28, 20, 12, 4]

    # Table for the second key generation step (compression)
    # The values in this table are the ones in the DES standard.
    pc2 = [14, 17, 11, 24, 1, 5, 3, 28, 15, 6, 21, 10, 23, 19, 12, 4, 26, 8,
           16, 7, 27, 20, 13, 2, 41, 52, 31, 37, 47, 55, 30, 40, 51, 45, 33,
           48, 44, 49, 39, 56, 34, 53, 46, 42, 50, 36, 29, 32]

    # Table for the substitution process
    table = [
        8421888, 0, 32768, 8421890, 8421378, 33282, 2, 32768, 512, 8421888,
        8421890, 512, 8389122, 8421378, 8388608, 2, 514, 8389120, 8389120, 33280,
        33280, 8421376, 8421376, 8389122, 32770, 8388610, 8388610, 32770, 0, 514,
        33282, 8388608, 32768, 8421890, 2, 8421376, 8421888, 8388608, 8388608, 512,
        8421378, 32768, 33280, 8388610, 512, 2, 8389122, 33282, 8421890, 32770,
        8421376, 8389122, 8388610, 514, 33282, 8421888, 514, 8389120, 8389120, 0,
        32770, 33280, 0, 8421378, 1074282512, 1073758208, 16384, 540688, 524288, 16,
        1074266128, 1073758224, 1073741840, 1074282512, 1074282496, 1073741824,
        1073758208, 524288, 16, 1074266128, 540672, 524304, 1073758224, 0, 1073741824,
        16384, 540688, 1074266112, 524304, 1073741840, 0, 540672, 16400, 1074282496,
        1074266112, 16400, 0, 540688, 1074266128, 524288, 1073758224, 1074266112,
        1074282496, 16384, 1074266112, 1073758208, 16, 1074282512, 540688, 16, 16384,
        1073741824, 16400, 1074282496, 524288, 1073741840, 524304, 1073758224,
        1073741840, 524304, 540672, 0, 1073758208, 16400, 1073741824, 1074266128,
        1074282512, 540672, 260, 67174656, 0, 67174404, 67109120, 0, 65796, 67109120,
        65540, 67108868, 67108868, 65536, 67174660, 65540, 67174400, 260, 67108864, 4,
        67174656, 256, 65792, 67174400, 67174404, 65796, 67109124, 65792, 65536,
        67109124, 4, 67174660, 256, 67108864, 67174656, 67108864, 65540, 260, 65536,
        67174656, 67109120, 0, 256, 65540, 67174660, 67109120, 67108868, 256, 0,
        67174404, 67109124, 65536, 67108864, 67174660, 4, 65796, 65792, 67108868,
        67174400, 67109124, 260, 67174400, 65796, 4, 67174404, 65792, 2151682048,
        2147487808, 2147487808, 64, 4198464, 2151678016, 2151677952, 2147487744, 0,
        4198400, 4198400, 2151682112, 2147483712, 0, 4194368, 2151677952, 2147483648,
        4096, 4194304, 2151682048, 64, 4194304, 2147487744, 4160, 2151678016, 2147483648,
        4160, 4194368, 4096, 4198464, 2151682112, 2147483712, 4194368, 2151677952,
        4198400, 2151682112, 2147483712, 0, 0, 4198400, 4160, 4194368, 2151678016,
        2147483648, 2151682048, 2147487808, 2147487808, 64, 2151682112, 2147483712,
        2147483648, 4096, 2151677952, 2147487744, 4198464, 2151678016, 2147487744, 4160,
        4194304, 2151682048, 64, 4194304, 4096, 4198464, 128, 17039488, 17039360,
        553648256, 262144, 128, 536870912, 17039360, 537133184, 262144, 16777344,
        537133184, 553648256, 553910272, 262272, 536870912, 16777216, 537133056,
        537133056, 0, 536871040, 553910400, 553910400, 16777344, 553910272, 536871040, 0,
        553648128, 17039488, 16777216, 553648128, 262272, 262144, 553648256, 128,
        16777216, 536870912, 17039360, 553648256, 537133184, 16777344, 536870912,
        553910272, 17039488, 537133184, 128, 16777216, 553910272, 553910400, 262272,
        553648128, 553910400, 17039360, 0, 537133056, 553648128, 262272, 16777344,
        536871040, 262144, 0, 537133056, 17039488, 536871040, 268435464, 270532608,
        8192, 270540808, 270532608, 8, 270540808, 2097152, 268443648, 2105352, 2097152,
        268435464, 2097160, 268443648, 268435456, 8200, 0, 2097160, 268443656, 8192,
        2105344, 268443656, 8, 270532616, 270532616, 0, 2105352, 270540800, 8200, 2105344,
        270540800, 268435456, 268443648, 8, 270532616, 2105344, 270540808, 2097152, 8200,
        268435464, 2097152, 268443648, 268435456, 8200, 268435464, 270540808, 2105344,
        270532608, 2105352, 270540800, 0, 270532616, 8, 8192, 270532608, 2105352, 8192,
        2097160, 268443656, 0, 270540800, 268435456, 2097160, 268443656, 1048576,
        34603009, 33555457, 0, 1024, 33555457, 1049601, 34604032, 34604033, 1048576, 0,
        33554433, 1, 33554432, 34603009, 1025, 33555456, 1049601, 1048577, 33555456,
        33554433, 34603008, 34604032, 1048577, 34603008, 1024, 1025, 34604033, 1049600,
        1, 33554432, 1049600, 33554432, 1049600, 1048576, 33555457, 33555457, 34603009,
        34603009, 1, 1048577, 33554432, 33555456, 1048576, 34604032, 1025, 1049601,
        34604032, 1025, 33554433, 34604033, 34603008, 1049600, 0, 1, 34604033, 0,
        1049601, 34603008, 1024, 33554433, 33555456, 1024, 1048577, 134219808, 2048,
        131072, 134350880, 134217728, 134219808, 32, 134217728, 131104, 134348800,
        134350880, 133120, 134350848, 133152, 2048, 32, 134348800, 134217760, 134219776,
        2080, 133120, 131104, 134348832, 134350848, 2080, 0, 0, 134348832, 134217760,
        134219776, 133152, 131072, 133152, 131072, 134350848, 2048, 32, 134348832, 2048,
        133152, 134219776, 32, 134217760, 134348800, 134348832, 134217728, 131072,
        134219808, 0, 134350880, 131104, 134217760, 134348800, 134219776, 134219808, 0,
        134350880, 133120, 133120, 2080, 2080, 131104, 134217728, 134350848
    ]

    # Table for the initial permutation
    iperm = [
        0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0,
        1, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 1, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0,
        0, 0, 0, 0, 1, 1, 0, 0, 0, 1, 0, 0, 1, 1, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 1, 1, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 1, 1, 0, 0,
        1, 1, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 1,
        0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 1, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0,
        0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 1, 0, 0, 1, 1, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 1, 1, 0, 0, 1, 0,
        0, 0, 1, 0, 0, 0, 1, 1, 0, 0, 1, 1, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0, 0,
        0, 2, 0, 0, 0, 2, 0, 0, 2, 0, 0, 0, 0, 0, 0, 0, 2, 2, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 2, 0, 0, 2, 2, 0, 0, 0, 2, 0, 0,
        0, 0, 0, 0, 2, 0, 0, 0, 0, 2, 0, 0, 2, 0, 0, 0, 0, 0, 0, 0, 2, 2, 0, 0, 0, 2, 0, 0, 2, 2, 0, 0, 2, 0, 0, 0, 2, 0, 0, 0,
        2, 2, 0, 0, 2, 0, 0, 0, 2, 0, 0, 0, 2, 2, 0, 0, 2, 2, 0, 0, 2, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 2, 0, 0, 0, 2, 0, 0, 2, 0, 0, 0, 0, 0, 0, 0, 2, 2, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 2,
        0, 0, 2, 2, 0, 0, 0, 2, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 2, 0, 0, 2, 0, 0, 0, 0, 0, 0, 0, 2, 2, 0, 0, 0, 2, 0, 0, 2, 2,
        0, 0, 2, 0, 0, 0, 2, 0, 0, 0, 2, 2, 0, 0, 2, 0, 0, 0, 2, 0, 0, 0, 2, 2, 0, 0, 2, 2, 0, 0, 2, 2, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 4, 0, 0, 0, 4, 0, 0, 0, 4, 0, 0, 4, 0, 0, 0, 0, 0, 0, 0, 4, 4, 0, 0, 0, 0, 0, 0,
        4, 0, 0, 0, 0, 4, 0, 0, 4, 4, 0, 0, 0, 4, 0, 0, 0, 0, 0, 0, 4, 0, 0, 0, 0, 4, 0, 0, 4, 0, 0, 0, 0, 0, 0, 0, 4, 4, 0, 0,
        0, 4, 0, 0, 4, 4, 0, 0, 4, 0, 0, 0, 4, 0, 0, 0, 4, 4, 0, 0, 4, 0, 0, 0, 4, 0, 0, 0, 4, 4, 0, 0, 4, 4, 0, 0, 4, 4, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 4, 0, 0, 0, 4, 0, 0, 0, 4, 0, 0, 4, 0, 0, 0, 0, 0,
        0, 0, 4, 4, 0, 0, 0, 0, 0, 0, 4, 0, 0, 0, 0, 4, 0, 0, 4, 4, 0, 0, 0, 4, 0, 0, 0, 0, 0, 0, 4, 0, 0, 0, 0, 4, 0, 0, 4, 0,
        0, 0, 0, 0, 0, 0, 4, 4, 0, 0, 0, 4, 0, 0, 4, 4, 0, 0, 4, 0, 0, 0, 4, 0, 0, 0, 4, 4, 0, 0, 4, 0, 0, 0, 4, 0, 0, 0, 4, 4,
        0, 0, 4, 4, 0, 0, 4, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 8, 0, 0, 0, 8, 0, 0, 0, 8, 0, 0,
        8, 0, 0, 0, 0, 0, 0, 0, 8, 8, 0, 0, 0, 0, 0, 0, 8, 0, 0, 0, 0, 8, 0, 0, 8, 8, 0, 0, 0, 8, 0, 0, 0, 0, 0, 0, 8, 0, 0, 0,
        0, 8, 0, 0, 8, 0, 0, 0, 0, 0, 0, 0, 8, 8, 0, 0, 0, 8, 0, 0, 8, 8, 0, 0, 8, 0, 0, 0, 8, 0, 0, 0, 8, 8, 0, 0, 8, 0, 0, 0,
        8, 0, 0, 0, 8, 8, 0, 0, 8, 8, 0, 0, 8, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 8,
        0, 0, 0, 8, 0, 0, 0, 8, 0, 0, 8, 0, 0, 0, 0, 0, 0, 0, 8, 8, 0, 0, 0, 0, 0, 0, 8, 0, 0, 0, 0, 8, 0, 0, 8, 8, 0, 0, 0, 8,
        0, 0, 0, 0, 0, 0, 8, 0, 0, 0, 0, 8, 0, 0, 8, 0, 0, 0, 0, 0, 0, 0, 8, 8, 0, 0, 0, 8, 0, 0, 8, 8, 0, 0, 8, 0, 0, 0, 8, 0,
        0, 0, 8, 8, 0, 0, 8, 0, 0, 0, 8, 0, 0, 0, 8, 8, 0, 0, 8, 8, 0, 0, 8, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 16, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 16, 0, 0, 0, 16, 0, 0, 0, 16, 0, 0, 16, 0, 0, 0, 0, 0, 0, 0, 16, 16, 0, 0, 0, 0, 0, 0, 16, 0, 0, 0,
        0, 16, 0, 0, 16, 16, 0, 0, 0, 16, 0, 0, 0, 0, 0, 0, 16, 0, 0, 0, 0, 16, 0, 0, 16, 0, 0, 0, 0, 0, 0, 0, 16, 16, 0, 0, 0,
        16, 0, 0, 16, 16, 0, 0, 16, 0, 0, 0, 16, 0, 0, 0, 16, 16, 0, 0, 16, 0, 0, 0, 16, 0, 0, 0, 16, 16, 0, 0, 16, 16, 0, 0,
        16, 16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 16, 0, 0, 0, 16, 0, 0, 0, 16, 0, 0,
        16, 0, 0, 0, 0, 0, 0, 0, 16, 16, 0, 0, 0, 0, 0, 0, 16, 0, 0, 0, 0, 16, 0, 0, 16, 16, 0, 0, 0, 16, 0, 0, 0, 0, 0, 0, 16,
        0, 0, 0, 0, 16, 0, 0, 16, 0, 0, 0, 0, 0, 0, 0, 16, 16, 0, 0, 0, 16, 0, 0, 16, 16, 0, 0, 16, 0, 0, 0, 16, 0, 0, 0, 16,
        16, 0, 0, 16, 0, 0, 0, 16, 0, 0, 0, 16, 16, 0, 0, 16, 16, 0, 0, 16, 16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 32, 0, 0, 0, 32, 0, 0, 0, 32, 0, 0, 32, 0, 0, 0, 0, 0, 0, 0, 32, 32, 0, 0, 0, 0, 0, 0, 32, 0, 0, 0,
        0, 32, 0, 0, 32, 32, 0, 0, 0, 32, 0, 0, 0, 0, 0, 0, 32, 0, 0, 0, 0, 32, 0, 0, 32, 0, 0, 0, 0, 0, 0, 0, 32, 32, 0, 0, 0,
        32, 0, 0, 32, 32, 0, 0, 32, 0, 0, 0, 32, 0, 0, 0, 32, 32, 0, 0, 32, 0, 0, 0, 32, 0, 0, 0, 32, 32, 0, 0, 32, 32, 0, 0,
        32, 32, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32, 0, 0, 0, 32, 0, 0, 0, 32, 0, 0,
        32, 0, 0, 0, 0, 0, 0, 0, 32, 32, 0, 0, 0, 0, 0, 0, 32, 0, 0, 0, 0, 32, 0, 0, 32, 32, 0, 0, 0, 32, 0, 0, 0, 0, 0, 0, 32,
        0, 0, 0, 0, 32, 0, 0, 32, 0, 0, 0, 0, 0, 0, 0, 32, 32, 0, 0, 0, 32, 0, 0, 32, 32, 0, 0, 32, 0, 0, 0, 32, 0, 0, 0, 32,
        32, 0, 0, 32, 0, 0, 0, 32, 0, 0, 0, 32, 32, 0, 0, 32, 32, 0, 0, 32, 32, 0, 0, 0, 0, 0, 0, 0, 0, 0, 64, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 64, 0, 0, 0, 64, 0, 0, 0, 64, 0, 0, 64, 0, 0, 0, 0, 0, 0, 0, 64, 64, 0, 0, 0, 0, 0, 0, 64, 0, 0, 0,
        0, 64, 0, 0, 64, 64, 0, 0, 0, 64, 0, 0, 0, 0, 0, 0, 64, 0, 0, 0, 0, 64, 0, 0, 64, 0, 0, 0, 0, 0, 0, 0, 64, 64, 0, 0, 0,
        64, 0, 0, 64, 64, 0, 0, 64, 0, 0, 0, 64, 0, 0, 0, 64, 64, 0, 0, 64, 0, 0, 0, 64, 0, 0, 0, 64, 64, 0, 0, 64, 64, 0, 0,
        64, 64, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 64, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 64, 0, 0, 0, 64, 0, 0, 0, 64, 0, 0,
        64, 0, 0, 0, 0, 0, 0, 0, 64, 64, 0, 0, 0, 0, 0, 0, 64, 0, 0, 0, 0, 64, 0, 0, 64, 64, 0, 0, 0, 64, 0, 0, 0, 0, 0, 0, 64,
        0, 0, 0, 0, 64, 0, 0, 64, 0, 0, 0, 0, 0, 0, 0, 64, 64, 0, 0, 0, 64, 0, 0, 64, 64, 0, 0, 64, 0, 0, 0, 64, 0, 0, 0, 64,
        64, 0, 0, 64, 0, 0, 0, 64, 0, 0, 0, 64, 64, 0, 0, 64, 64, 0, 0, 64, 64, 0, 0, 0, 0, 0, 0, 0, 0, 0, 128, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 128, 0, 0, 0, 128, 0, 0, 0, 128, 0, 0, 128, 0, 0, 0, 0, 0, 0, 0, 128, 128, 0, 0, 0, 0, 0, 0, 128, 0,
        0, 0, 0, 128, 0, 0, 128, 128, 0, 0, 0, 128, 0, 0, 0, 0, 0, 0, 128, 0, 0, 0, 0, 128, 0, 0, 128, 0, 0, 0, 0, 0, 0, 0,
        128, 128, 0, 0, 0, 128, 0, 0, 128, 128, 0, 0, 128, 0, 0, 0, 128, 0, 0, 0, 128, 128, 0, 0, 128, 0, 0, 0, 128, 0, 0, 0,
        128, 128, 0, 0, 128, 128, 0, 0, 128, 128, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 128, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        128, 0, 0, 0, 128, 0, 0, 0, 128, 0, 0, 128, 0, 0, 0, 0, 0, 0, 0, 128, 128, 0, 0, 0, 0, 0, 0, 128, 0, 0, 0, 0, 128, 0,
        0, 128, 128, 0, 0, 0, 128, 0, 0, 0, 0, 0, 0, 128, 0, 0, 0, 0, 128, 0, 0, 128, 0, 0, 0, 0, 0, 0, 0, 128, 128, 0, 0, 0,
        128, 0, 0, 128, 128, 0, 0, 128, 0, 0, 0, 128, 0, 0, 0, 128, 128, 0, 0, 128, 0, 0, 0, 128, 0, 0, 0, 128, 128, 0, 0, 128,
        128, 0, 0, 128, 128
    ]

    # Table for the final permutation
    fperm = [
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 64, 0, 0, 0, 0, 0, 0, 0, 0, 64, 0, 0, 0, 0, 0, 0, 64, 64, 0, 0, 0, 0, 0, 0, 0, 0,
        64, 0, 0, 0, 0, 0, 64, 0, 64, 0, 0, 0, 0, 0, 0, 64, 64, 0, 0, 0, 0, 0, 64, 64, 64, 0, 0, 0, 0, 0, 0, 0, 0, 64, 0, 0, 0,
        0, 64, 0, 0, 64, 0, 0, 0, 0, 0, 64, 0, 64, 0, 0, 0, 0, 64, 64, 0, 64, 0, 0, 0, 0, 0, 0, 64, 64, 0, 0, 0, 0, 64, 0, 64,
        64, 0, 0, 0, 0, 0, 64, 64, 64, 0, 0, 0, 0, 64, 64, 64, 64, 0, 0, 0, 0, 0, 0, 0, 0, 64, 0, 0, 0, 0, 0, 0, 0, 0, 64, 0,
        0, 0, 0, 0, 0, 64, 64, 0, 0, 0, 0, 0, 0, 0, 0, 64, 0, 0, 0, 0, 0, 64, 0, 64, 0, 0, 0, 0, 0, 0, 64, 64, 0, 0, 0, 0, 0,
        64, 64, 64, 0, 0, 0, 0, 0, 0, 0, 0, 64, 0, 0, 0, 0, 64, 0, 0, 64, 0, 0, 0, 0, 0, 64, 0, 64, 0, 0, 0, 0, 64, 64, 0, 64,
        0, 0, 0, 0, 0, 0, 64, 64, 0, 0, 0, 0, 64, 0, 64, 64, 0, 0, 0, 0, 0, 64, 64, 64, 0, 0, 0, 0, 64, 64, 64, 64, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 16, 0, 0, 0, 0, 0, 0, 0, 0, 16, 0, 0, 0, 0, 0, 0, 16, 16, 0, 0, 0, 0, 0, 0, 0, 0,
        16, 0, 0, 0, 0, 0, 16, 0, 16, 0, 0, 0, 0, 0, 0, 16, 16, 0, 0, 0, 0, 0, 16, 16, 16, 0, 0, 0, 0, 0, 0, 0, 0, 16, 0, 0, 0,
        0, 16, 0, 0, 16, 0, 0, 0, 0, 0, 16, 0, 16, 0, 0, 0, 0, 16, 16, 0, 16, 0, 0, 0, 0, 0, 0, 16, 16, 0, 0, 0, 0, 16, 0, 16,
        16, 0, 0, 0, 0, 0, 16, 16, 16, 0, 0, 0, 0, 16, 16, 16, 16, 0, 0, 0, 0, 0, 0, 0, 0, 16, 0, 0, 0, 0, 0, 0, 0, 0, 16, 0,
        0, 0, 0, 0, 0, 16, 16, 0, 0, 0, 0, 0, 0, 0, 0, 16, 0, 0, 0, 0, 0, 16, 0, 16, 0, 0, 0, 0, 0, 0, 16, 16, 0, 0, 0, 0, 0,
        16, 16, 16, 0, 0, 0, 0, 0, 0, 0, 0, 16, 0, 0, 0, 0, 16, 0, 0, 16, 0, 0, 0, 0, 0, 16, 0, 16, 0, 0, 0, 0, 16, 16, 0, 16,
        0, 0, 0, 0, 0, 0, 16, 16, 0, 0, 0, 0, 16, 0, 16, 16, 0, 0, 0, 0, 0, 16, 16, 16, 0, 0, 0, 0, 16, 16, 16, 16, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 4, 0, 0, 0, 0, 0, 0, 0, 0, 4, 0, 0, 0, 0, 0, 0, 4, 4, 0, 0, 0, 0, 0, 0, 0, 0, 4, 0,
        0, 0, 0, 0, 4, 0, 4, 0, 0, 0, 0, 0, 0, 4, 4, 0, 0, 0, 0, 0, 4, 4, 4, 0, 0, 0, 0, 0, 0, 0, 0, 4, 0, 0, 0, 0, 4, 0, 0, 4,
        0, 0, 0, 0, 0, 4, 0, 4, 0, 0, 0, 0, 4, 4, 0, 4, 0, 0, 0, 0, 0, 0, 4, 4, 0, 0, 0, 0, 4, 0, 4, 4, 0, 0, 0, 0, 0, 4, 4, 4,
        0, 0, 0, 0, 4, 4, 4, 4, 0, 0, 0, 0, 0, 0, 0, 0, 4, 0, 0, 0, 0, 0, 0, 0, 0, 4, 0, 0, 0, 0, 0, 0, 4, 4, 0, 0, 0, 0, 0, 0,
        0, 0, 4, 0, 0, 0, 0, 0, 4, 0, 4, 0, 0, 0, 0, 0, 0, 4, 4, 0, 0, 0, 0, 0, 4, 4, 4, 0, 0, 0, 0, 0, 0, 0, 0, 4, 0, 0, 0, 0,
        4, 0, 0, 4, 0, 0, 0, 0, 0, 4, 0, 4, 0, 0, 0, 0, 4, 4, 0, 4, 0, 0, 0, 0, 0, 0, 4, 4, 0, 0, 0, 0, 4, 0, 4, 4, 0, 0, 0, 0,
        0, 4, 4, 4, 0, 0, 0, 0, 4, 4, 4, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0,
        0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 1, 1, 1, 0,
        0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 1, 1, 0, 1, 0, 0, 0, 0, 0, 0, 1, 1,
        0, 0, 0, 0, 1, 0, 1, 1, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0,
        0, 1, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0,
        1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 1, 1, 0, 1, 0, 0, 0, 0,
        0, 0, 1, 1, 0, 0, 0, 0, 1, 0, 1, 1, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 128, 0, 0, 0, 0, 0, 0, 0, 0, 128, 0, 0, 0, 0, 0, 0, 128, 128, 0, 0, 0, 0, 0, 0, 0, 0, 128, 0, 0, 0, 0, 0,
        128, 0, 128, 0, 0, 0, 0, 0, 0, 128, 128, 0, 0, 0, 0, 0, 128, 128, 128, 0, 0, 0, 0, 0, 0, 0, 0, 128, 0, 0, 0, 0, 128, 0,
        0, 128, 0, 0, 0, 0, 0, 128, 0, 128, 0, 0, 0, 0, 128, 128, 0, 128, 0, 0, 0, 0, 0, 0, 128, 128, 0, 0, 0, 0, 128, 0, 128,
        128, 0, 0, 0, 0, 0, 128, 128, 128, 0, 0, 0, 0, 128, 128, 128, 128, 0, 0, 0, 0, 0, 0, 0, 0, 128, 0, 0, 0, 0, 0, 0, 0, 0,
        128, 0, 0, 0, 0, 0, 0, 128, 128, 0, 0, 0, 0, 0, 0, 0, 0, 128, 0, 0, 0, 0, 0, 128, 0, 128, 0, 0, 0, 0, 0, 0, 128, 128,
        0, 0, 0, 0, 0, 128, 128, 128, 0, 0, 0, 0, 0, 0, 0, 0, 128, 0, 0, 0, 0, 128, 0, 0, 128, 0, 0, 0, 0, 0, 128, 0, 128, 0,
        0, 0, 0, 128, 128, 0, 128, 0, 0, 0, 0, 0, 0, 128, 128, 0, 0, 0, 0, 128, 0, 128, 128, 0, 0, 0, 0, 0, 128, 128, 128, 0,
        0, 0, 0, 128, 128, 128, 128, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 32, 0, 0, 0, 0, 0, 0, 0, 0, 32, 0, 0, 0,
        0, 0, 0, 32, 32, 0, 0, 0, 0, 0, 0, 0, 0, 32, 0, 0, 0, 0, 0, 32, 0, 32, 0, 0, 0, 0, 0, 0, 32, 32, 0, 0, 0, 0, 0, 32, 32,
        32, 0, 0, 0, 0, 0, 0, 0, 0, 32, 0, 0, 0, 0, 32, 0, 0, 32, 0, 0, 0, 0, 0, 32, 0, 32, 0, 0, 0, 0, 32, 32, 0, 32, 0, 0, 0,
        0, 0, 0, 32, 32, 0, 0, 0, 0, 32, 0, 32, 32, 0, 0, 0, 0, 0, 32, 32, 32, 0, 0, 0, 0, 32, 32, 32, 32, 0, 0, 0, 0, 0, 0, 0,
        0, 32, 0, 0, 0, 0, 0, 0, 0, 0, 32, 0, 0, 0, 0, 0, 0, 32, 32, 0, 0, 0, 0, 0, 0, 0, 0, 32, 0, 0, 0, 0, 0, 32, 0, 32, 0,
        0, 0, 0, 0, 0, 32, 32, 0, 0, 0, 0, 0, 32, 32, 32, 0, 0, 0, 0, 0, 0, 0, 0, 32, 0, 0, 0, 0, 32, 0, 0, 32, 0, 0, 0, 0, 0,
        32, 0, 32, 0, 0, 0, 0, 32, 32, 0, 32, 0, 0, 0, 0, 0, 0, 32, 32, 0, 0, 0, 0, 32, 0, 32, 32, 0, 0, 0, 0, 0, 32, 32, 32,
        0, 0, 0, 0, 32, 32, 32, 32, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 8, 0, 0, 0, 0, 0, 0, 0, 0, 8, 0, 0, 0, 0,
        0, 0, 8, 8, 0, 0, 0, 0, 0, 0, 0, 0, 8, 0, 0, 0, 0, 0, 8, 0, 8, 0, 0, 0, 0, 0, 0, 8, 8, 0, 0, 0, 0, 0, 8, 8, 8, 0, 0, 0,
        0, 0, 0, 0, 0, 8, 0, 0, 0, 0, 8, 0, 0, 8, 0, 0, 0, 0, 0, 8, 0, 8, 0, 0, 0, 0, 8, 8, 0, 8, 0, 0, 0, 0, 0, 0, 8, 8, 0, 0,
        0, 0, 8, 0, 8, 8, 0, 0, 0, 0, 0, 8, 8, 8, 0, 0, 0, 0, 8, 8, 8, 8, 0, 0, 0, 0, 0, 0, 0, 0, 8, 0, 0, 0, 0, 0, 0, 0, 0, 8,
        0, 0, 0, 0, 0, 0, 8, 8, 0, 0, 0, 0, 0, 0, 0, 0, 8, 0, 0, 0, 0, 0, 8, 0, 8, 0, 0, 0, 0, 0, 0, 8, 8, 0, 0, 0, 0, 0, 8, 8,
        8, 0, 0, 0, 0, 0, 0, 0, 0, 8, 0, 0, 0, 0, 8, 0, 0, 8, 0, 0, 0, 0, 0, 8, 0, 8, 0, 0, 0, 0, 8, 8, 0, 8, 0, 0, 0, 0, 0, 0,
        8, 8, 0, 0, 0, 0, 8, 0, 8, 8, 0, 0, 0, 0, 0, 8, 8, 8, 0, 0, 0, 0, 8, 8, 8, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 0, 0, 2, 2, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 0, 2, 0, 2, 0, 0, 0,
        0, 0, 0, 2, 2, 0, 0, 0, 0, 0, 2, 2, 2, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 2, 0, 0, 2, 0, 0, 0, 0, 0, 2, 0, 2, 0, 0,
        0, 0, 2, 2, 0, 2, 0, 0, 0, 0, 0, 0, 2, 2, 0, 0, 0, 0, 2, 0, 2, 2, 0, 0, 0, 0, 0, 2, 2, 2, 0, 0, 0, 0, 2, 2, 2, 2, 0, 0,
        0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 0, 0, 2, 2, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 0, 2, 0,
        2, 0, 0, 0, 0, 0, 0, 2, 2, 0, 0, 0, 0, 0, 2, 2, 2, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 2, 0, 0, 2, 0, 0, 0, 0, 0, 2,
        0, 2, 0, 0, 0, 0, 2, 2, 0, 2, 0, 0, 0, 0, 0, 0, 2, 2, 0, 0, 0, 0, 2, 0, 2, 2, 0, 0, 0, 0, 0, 2, 2, 2, 0, 0, 0, 0, 2, 2,
        2, 2, 0, 0, 0, 0
    ]

    def _get_bit(self, byte, bit):
        return (byte & (0xFF << (8 * bit))) >> (8 * bit)

    def _xor(self, key, pattern):
        for i in range(len(key)):
            key[i] ^= pattern[i]
        return key

    def _permute(self, iperm, key):
        outblock = [0]*8
        off = 0
        for i in key:
            for k in range(7):
                ptr = iperm[(off+(i >> 4))*8+k]
                pobyte = iperm[(off+16)*8 + 8*(i & 0xF) + k]
                outblock[k] |= pobyte | ptr
            outblock[7] |= (iperm[(off + (i >> 4))*8+7])
            outblock[7] |= iperm[(off + 16) * 8 + 8 * (i & 0xF) + 7]
            off += 32
        return outblock

    def _f(self, r, sub_key):
        r_1 = r
        r_2 = 2 * r
        if r_2 / (16 ** 8) > 0:
            r_2 = r_2 % (16 ** 8)
            r_2 |= 1
        value = self.table[2 * 64 + (((r_1 >> 19) ^ sub_key[2]) & 0x3F)] | \
                self.table[3 * 64 + (((r_1 >> 15) ^ sub_key[3]) & 0x3F)] | \
                self.table[4 * 64 + (((r_1 >> 11) ^ sub_key[4]) & 0x3F)] | \
                self.table[5 * 64 + (((r_1 >> 7) ^ sub_key[5]) & 0x3F)] | \
                self.table[6 * 64 + (((r_1 >> 3) ^ sub_key[6]) & 0x3F)] | \
                self.table[7 * 64 + ((r_2 ^ sub_key[7]) & 0x3F)] | \
                self.table[1 * 64 + (((r_1 >> 23) ^ sub_key[1]) & 0x3F)] | \
                self.table[0 * 64 + (sub_key[0] & 0x3F ^ (32 * r_1 | (r_1 >> 27)) & 0x3F)]
        return value

    def _decode_v1(self, keys, block, chain):
        if chain and block:
            chain_rest = [0]*8
            for i in range(8):
                chain_rest[i] = block[i]
        blocks = self._permute(self.iperm, block)
        init_xor = blocks[7] | ((blocks[6] | ((blocks[5] | (blocks[4] << 8)) << 8)) << 8)
        init_r = self._f(blocks[7] | ((blocks[6] | ((blocks[5] | (blocks[4] << 8)) << 8)) << 8), keys[len(keys) - 8:])
        tmp_r = (blocks[3] | ((blocks[2] | ((blocks[1] | blocks[0] << 8) << 8)) << 8)) ^ init_r
        tmp_1 = self._f(tmp_r, keys[112:120]) ^ init_xor
        tmp_r ^= self._f(tmp_1, keys[104:112])
        tmp_2 = self._f(tmp_r, keys[96:104]) ^ tmp_1
        tmp_r ^= self._f(tmp_2, keys[88:96])
        tmp_3 = self._f(tmp_r, keys[80:88]) ^ tmp_2
        tmp_r ^= self._f(tmp_3, keys[72:80])
        tmp_4 = self._f(tmp_r, keys[64:72]) ^ tmp_3
        tmp_r ^= self._f(tmp_4, keys[56:64])
        tmp_5 = self._f(tmp_r, keys[48:56]) ^ tmp_4
        tmp_r ^= self._f(tmp_5, keys[40:48])
        tmp_6 = self._f(tmp_r, keys[32:40]) ^ tmp_5
        tmp_r ^= self._f(tmp_6, keys[24:32])
        tmp_7 = self._f(tmp_r, keys[16:24]) ^ tmp_6
        tmp_r ^= self._f(tmp_7, keys[8:16])
        tmp_8 = self._f(tmp_r, keys[0:8]) ^ tmp_7
        blocks[0] = self._get_bit(tmp_8, 3)
        blocks[1] = self._get_bit(tmp_8, 2)
        blocks[2] = self._get_bit(tmp_8, 1)
        blocks[3] = self._get_bit(tmp_8, 0)
        blocks[4] = self._get_bit(tmp_r, 3)
        blocks[5] = self._get_bit(tmp_r, 2)
        blocks[6] = self._get_bit(tmp_r, 1)
        blocks[7] = self._get_bit(tmp_r, 0)
        block = self._permute(self.fperm, blocks)
        if chain and block:
            self._xor(block, chain)
            chain = chain_rest
        return block, chain

    def _encode_v1(self, keys, block, chain):
        if chain:
            block = self._xor(block, chain)
        blocks = self._permute(self.iperm, block)
        init_xor = blocks[7] | ((blocks[6] | ((blocks[5] | (blocks[4] << 8)) << 8)) << 8)
        init_r = self._f(blocks[7] | ((blocks[6] | ((blocks[5] | (blocks[4] << 8)) << 8)) << 8), keys[:8])
        tmp_r = (blocks[3] | ((blocks[2] | ((blocks[1] | blocks[0] << 8) << 8)) << 8)) ^ init_r
        tmp_1 = self._f(tmp_r, keys[8:16]) ^ init_xor
        tmp_r ^= self._f(tmp_1, keys[16:24])
        tmp_2 = self._f(tmp_r, keys[24:32]) ^ tmp_1
        tmp_r ^= self._f(tmp_2, keys[32:40])
        tmp_3 = self._f(tmp_r, keys[40:48]) ^ tmp_2
        tmp_r ^= self._f(tmp_3, keys[48:56])
        tmp_4 = self._f(tmp_r, keys[56:64]) ^ tmp_3
        tmp_r ^= self._f(tmp_4, keys[64:72])
        tmp_5 = self._f(tmp_r, keys[72:80]) ^ tmp_4
        tmp_r ^= self._f(tmp_5, keys[80:88])
        tmp_6 = self._f(tmp_r, keys[88:96]) ^ tmp_5
        tmp_r ^= self._f(tmp_6, keys[96:104])
        tmp_7 = self._f(tmp_r, keys[104:112]) ^ tmp_6
        tmp_r ^= self._f(tmp_7, keys[112:120])
        tmp_8 = self._f(tmp_r, keys[120:128]) ^ tmp_7
        blocks[0] = self._get_bit(tmp_8, 3)
        blocks[1] = self._get_bit(tmp_8, 2)
        blocks[2] = self._get_bit(tmp_8, 1)
        blocks[3] = self._get_bit(tmp_8, 0)
        blocks[4] = self._get_bit(tmp_r, 3)
        blocks[5] = self._get_bit(tmp_r, 2)
        blocks[6] = self._get_bit(tmp_r, 1)
        blocks[7] = self._get_bit(tmp_r, 0)
        block = self._permute(self.fperm, blocks)
        if chain:
            chain = block
        return block, chain

    def _generate_keys(self, key):
        keys = [0] * 128
        array_1 = [0] * 56
        array_2 = [0] * 56
        for i in range(56):
            idx_bb = (self.pc1[i] - 1) & 7
            idx_k = (self.pc1[i] - 1) >> 3
            array_1[i] = int(self.byte_bit[idx_bb] & key[idx_k] != 0)
        for j in range(16):
            value = self.totrot[j]
            for k in range(56):
                idx = value
                if k >= 28:
                    if value >= 56:
                        idx = value - 28
                else:
                    if value >= 28:
                        idx = value - 28
                array_2[k] = (array_1[idx])
                value += 1
            for n in range(48):
                if array_2[self.pc2[n] - 1]:
                    tmp = self.byte_bit[n % 6]
                    tmp = tmp >> 2
                    keys[8 * j + n / 6] |= tmp
        return keys

    def crypt(self, mode, blob, key, length):
        """Runs a round of encoding/decoding using SAP's custom TripleDES implementation as in SSFS
        and RSECTAB.

        :param mode: encode or decode mode
        :type mode: int

        :param blob: blob to encode
        :type blob: list of int

        :param key: key to use to encode
        :type key: list of int

        :return: encoded blob
        :rtype: list of int

        :raise Exception: if encoding failed
        """

        def transf_encode(blocks, blob, length):
            return blocks[len(blocks) - 8:]

        def transf_decode(blocks, blob, length):
            return blob[length - (length % 8) - 8:length - (length % 8)]

        # Pick the right functions and padding transforms according to the mode
        if mode == self.MODE_ENCODE:
            func_1 = self._encode_v1
            transf = transf_encode
        elif mode == self.MODE_DECODE:
            func_1 = self._decode_v1
            transf = transf_decode
        else:
            raise AttributeError("Invalid mode")

        blocks = []
        keys = self._generate_keys(key)
        chain = [0] * 8
        if length >= 8:
            num_block = length >> 3
            for i in range(num_block):
                inout, chain = func_1(keys, blob[i * 8:8 + i * 8], chain)
                for j in inout:
                    blocks.append(j)
        if length % 8:
            inout, chain = self._encode_v1(keys, transf(blocks, blob, length), [0] * 8)
            inout = self._xor(blob[length - (length % 8):], inout)
            for k in inout:
                blocks.append(k)
        return blocks
