//
// Firebird .NET Data Provider - Firebird managed data provider for .NET and Mono
// Copyright (C) 2002-2004  Carlos Guzman Alvarez
//
// Distributable under LGPL license.
// You may obtain a copy of the License at http://www.gnu.org/copyleft/lesser.html
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//

using System;
using System.Data;
using System.Collections;
using System.Threading;

namespace PostgreSql.Data.PgSqlClient
{
	internal class PgConnectionPool : MarshalByRefObject
	{
		#region FIELDS

		private static ConnectionPool pool = null;

		#endregion
    
		#region METHODS

		public static void Init()
		{
			if (pool ==  null)
			{
				pool = new ConnectionPool();
			}
		}
    
		public static PgDbConnection GetConnection(string connectionString, PgDbConnection connection)
		{
			Init();

			return ((PgDbConnection)pool.CheckOut(connectionString, connection));
		}

		public static void FreeConnection(PgDbConnection c) 
		{
			pool.CheckIn(c);
		}

		#endregion
	}
	
	internal class ConnectionPool
	{
		#region FIELDS

		private ArrayList	locked;
		private ArrayList	unlocked;
		private Thread		cleanUpThread;

		#endregion

		#region CONSTRUCTORS

		public ConnectionPool()		
		{			
			locked	 = ArrayList.Synchronized(new ArrayList());
			unlocked = ArrayList.Synchronized(new ArrayList());

			cleanUpThread		= new Thread(new ThreadStart(RunCleanUp));
			cleanUpThread.Name	= "CleanUp Thread";			
			cleanUpThread.Start();
			cleanUpThread.IsBackground = true;
		}

		#endregion

		#region METHODS
		
		public PgDbConnection CheckOut(string connectionString)
		{
			return this.CheckOut(connectionString, null);
		}

		public PgDbConnection CheckOut(string connectionString, PgDbConnection instance)
		{
			PgDbConnection newConnection	= null;
			long			now				= System.DateTime.Now.Ticks;

			lock (typeof(PgConnectionPool))
			{
				if (unlocked.Count > 0)
				{
					PgDbConnection[] list = new PgDbConnection[unlocked.Count];
					unlocked.CopyTo(0, list, 0, list.Length);

					foreach (PgDbConnection connection in list)
					{
						if (Validate(connection, connectionString))
						{
							if (connection.Lifetime != 0)
							{
								if ((now - connection.Created) > connection.Lifetime)
								{
									unlocked.Remove(connection);
									Expire(connection);
								}
								else
								{
									unlocked.Remove(connection);
									locked.Add(connection);
									
									return(connection);
								}
							}
							else
							{
								unlocked.Remove(connection);
								locked.Add(connection);
								
								return(connection);
							}
						}
						else
						{						
							unlocked.Remove(connection);
							Expire(connection);
						}
					}			
				}

				if (instance == null)
				{
					newConnection = Create(connectionString);
				}
				else
				{
					newConnection = instance;
					newConnection.Connect();
				}
				newConnection.Pooled	= true;
				newConnection.Created	= System.DateTime.Now.Ticks;

				locked.Add(newConnection);
			}
			
			return(newConnection);
		}	

		public void CheckIn(PgDbConnection connection)
		{			
			lock (typeof(PgConnectionPool))
			{				
				connection.Created = System.DateTime.Now.Ticks;

				locked.Remove(connection);
				unlocked.Add(connection);
			}
		}

		private void RunCleanUp()
		{		
			TimeSpan interval = new TimeSpan(0, 0, 10);

			while (true)
			{
				CleanUp(null);

				Thread.Sleep(interval);
			}
		}

		private PgDbConnection Create(string connectionString)
		{
			try 
			{
				PgDbConnection connection = new PgDbConnection(connectionString);
				connection.Connect();

				return connection;
			}
			catch (Exception ex) 
			{
				throw ex;
			}
		}
    
		private bool Validate(PgDbConnection connection, string connectionString)
		{
			try 
			{								
				return (connection.ConnectionString == connectionString && 
					connection.Verify());
			}
			catch (Exception ex)
			{
				throw ex;
			}
		}

		private void Expire(PgDbConnection connection)
		{
			try 
			{
				if (connection.Verify())
				{
					connection.Disconnect();
				}
			}
			catch (Exception)
			{
				throw new PgException("Error closing database connection.");
			}
		}
		
		private void CleanUp(object State)
		{
			long now = System.DateTime.Now.Ticks;
			
			lock (unlocked.SyncRoot)
			{
				if (unlocked.Count > 0)
				{
					PgDbConnection[] list = new PgDbConnection[unlocked.Count];
				
					unlocked.CopyTo(0, list, 0, list.Length);
					foreach (PgDbConnection connection in list)
					{
						if (connection.Lifetime != 0)
						{
							if ((now - connection.Created) >= connection.Lifetime)
							{
								unlocked.Remove(connection);
								Expire(connection);
							}
						}
					}
				}
			}
		}

		#endregion
	}
}
