(* (c) Microsoft Corporation 2005-2006.  *)

(*F#
#nowarn "62";;
F#*)

let (>>) f g x = g(f(x))
let rec list_mapi_aux f l n = 
  match l with [] -> [] | (h::t) -> let r = f n h  in r :: list_mapi_aux f t (n+1)
let list_mapi f l =  list_mapi_aux f l 0

type ident = string
type code = string * Lexing.position

type spec = 
    { header: code;
      tokens: (ident * string option) list;
      types: (ident * string) list;
      assoc: (ident * assoc) list list;
      starts: ident list;
      rules: (ident * rule list) list }
and rule = Rule of ident list * ident option * code option
and assoc = LeftAssoc | RightAssoc | NonAssoc


(*
module CompactIntSet = struct
  type t = { bits: int32 array }
  let empty : t = { bits = [| |] }

  let copy s : t = { bits = Array.copy s.bits }
  let copy_and_ensure_length len arr = 
    let oldLength = Array.length arr in 
    let narr = Array.create (max oldLength len) 0 in 
    Array.blit arr 0 narr 0 oldLength;
    narr
  let copy_and_ensure i arr = copy_and_ensure_length(i / 32 + 1) arr

  let add i s : t = 
    let arr = copy_and_ensure i s.bits in 
    let idx = i / 32 in 
    let bit = i mod 32 in 
    arr.(idx) <- arr.(idx) lor (1 lsl bit);
    { bits = arr }

end
*)

type terminal = string
type nonterminal = string
type sym = T of terminal | NT of nonterminal
type precInfo = ExplicitPrec of assoc * int | NoPrecedence
      
type prodIdx = int
type dotIdx = int
type item0 = int32  (* slam (prodIdx,dotIdx) into one integer *)

let mkItem0 (prodIdx,dotIdx) = Int32.logor (Int32.shift_left (Int32.of_int prodIdx) 16) (Int32.of_int dotIdx)

let prodIdx_of_item0 item0 = Int32.to_int (Int32.shift_right_logical item0 16)
let dotIdx_of_item0 item0 = Int32.to_int (Int32.logand item0 0xFFFFl)

let item0_compare (i1: item0) (i2:item0) = Pervasives.compare i1 i2
let item0_hash (i1:item0) = Hashtbl.hash i1
(*
let item0_compare (Item0(r1,d1)) (Item0(r2,d2)) = let c = Pervasives.compare r1 r2 in if c <> 0 then c else Pervasives.compare d1 d2
let item0_hash (Item0(r1,d1)) = Hashtbl.hash r1 + Hashtbl.hash d1 * 18923 
*)

type item1 = Item1 of item0 * terminal 
let item1_compare (Item1(item01,t1)) (Item1(item02,t2)) = let c = item0_compare item01 item02 in if c <> 0 then c else Pervasives.compare t1 t2


type syms = sym list


type action = 
  | Shift of int
  | Reduce of prodIdx
  | Accept
  | Error
    
type kernelIdx = int

type kernelItemIndex = KernelItemIdx of kernelIdx * item0
let kernelItemIndex_compare  (KernelItemIdx(n1,item01)) (KernelItemIdx(n2,item02)) = 
  let c = Pervasives.compare n1 n2 in if c <> 0 then c else item0_compare item01 item02
let kernelItemIndex_hash  (KernelItemIdx(n,item0)) = 
  Hashtbl.hash n + item0_hash item0

type gotoItemIndex = GotoKernelIdx of kernelIdx * sym

(*IF-FSHARP
let TerminalSet = Set.Make(Pervasives.compare: terminal -> terminal -> int)
let FirstSet = Set.Make(Pervasives.compare : terminal option -> terminal option -> int)
let FirstMap = Map.Make(Pervasives.compare : sym -> sym -> int)
let Productions = Map.Make(Pervasives.compare : nonterminal -> nonterminal -> int)
let Item0Set = Set.Make(item0_compare)
let Item1Set = Set.Make(item1_compare)
let Item0Sets = Set.Make(Item0Set.compare)
type kernel = item0 Tagged.Set
let kernel_compare = Item0Set.compare
ENDIF-FSHARP*)
(*IF-OCAML*)
module TerminalSet = Set.Make(struct type t = terminal let compare = Pervasives.compare end)
module FirstSet = Set.Make(struct type t = terminal option let compare = Pervasives.compare end)
module FirstMap = Map.Make(struct type t = sym let compare = Pervasives.compare end)
module Productions = Map.Make(struct type t = nonterminal let compare = Pervasives.compare end)
module Item0Set = Set.Make(struct type t = item0 let compare = item0_compare end)
module Item1Set = Set.Make(struct type t = item1 let compare = item1_compare end)
module Item0Sets = Set.Make(struct type t = Item0Set.t let compare = Item0Set.compare end)
type kernel = Item0Set.t
let kernel_compare = Item0Set.compare
(*ENDIF-OCAML*)

(*IF-FSHARP
let SpontaneousLookaheadBag = Map.Make(kernelItemIndex_compare)
let LookaheadPropagationBag = Map.Make(kernelItemIndex_compare)
let KernelItemLookaheadBag = Map.Make(kernelItemIndex_compare)
let KernelItemIdxSet = Set.Make(kernelItemIndex_compare)
let KernelIdxMap = Map.Make(kernel_compare)
ENDIF-FSHARP*)
(*IF-OCAML*)
module SpontaneousLookaheadBag = Map.Make(struct type t = kernelItemIndex let compare = kernelItemIndex_compare end)
module LookaheadPropagationBag = Map.Make(struct type t = kernelItemIndex let compare = kernelItemIndex_compare end)
module KernelItemLookaheadBag = Map.Make(struct type t = kernelItemIndex let compare = kernelItemIndex_compare end)
module KernelItemIdxSet = Set.Make(struct type t = kernelItemIndex let compare = kernelItemIndex_compare end)
module KernelIdxMap = Map.Make(struct type t = kernel let compare = kernel_compare end)
(*ENDIF-OCAML*)

let string_of_sym sym = match sym with T s -> "'" ^ s ^ "'" | NT s -> s
let outputSym os sym = Printf.fprintf os "%s" (string_of_sym sym)
let outputSyms os syms =
  Printf.fprintf os "%s" (String.concat " " (List.map string_of_sym syms))
let outputTerminalSet os tset  =
  Printf.fprintf os "%s" (String.concat ";" (TerminalSet.elements tset))
let outputAssoc os p = 
  match p with 
  | LeftAssoc -> Printf.fprintf os "left"
  | RightAssoc -> Printf.fprintf os "right"
  | NonAssoc -> Printf.fprintf os "nonassoc"
let outputPrecInfo os p = 
  match p with 
  | ExplicitPrec (assoc,n) -> Printf.fprintf os "explicit %a %d" outputAssoc assoc n
  | NoPrecedence  -> Printf.fprintf os "noprec"

let repeat (elements, mem, add) f initial iset = 
  let workList = ref (elements iset) in 
  let acc = ref initial in 
  let one item = 
    List.iter (fun i2 -> if not (mem i2 !acc) then (acc := add i2 !acc; workList := i2 :: !workList)) (f item) in
  while !workList <> [] do 
    let r = List.hd !workList in 
    workList := List.tl !workList;
    one r
  done;
  !acc

let rec mapFilter f l = 
  match l with 
    [] -> []
  | h :: t -> match f h with Some x -> x :: mapFilter f t | None -> mapFilter f t 


let addBag (mem,find,add,insert,empty) x y bag = 
  if mem x bag then add x (insert y (find x bag)) bag else add x (insert y empty) bag

let addSpontaneousLookaheadBag x y bag = addBag(SpontaneousLookaheadBag.mem,
                                                SpontaneousLookaheadBag.find,
                                                SpontaneousLookaheadBag.add,
                                                TerminalSet.add,
                                                TerminalSet.empty) x y bag
let addLookaheadPropagationBag x y bag = addBag(LookaheadPropagationBag.mem,
                                                LookaheadPropagationBag.find,
                                                LookaheadPropagationBag.add,
                                                KernelItemIdxSet.add,
                                                KernelItemIdxSet.empty) x y bag
let addKernelItemLookaheadBag x y bag = addBag(KernelItemLookaheadBag.mem,
                                               KernelItemLookaheadBag.find,
                                               KernelItemLookaheadBag.add,
                                               TerminalSet.add,
                                               TerminalSet.empty) x y bag

let memoize f = 
  let t = Hashtbl.create 10 in 
  fun x -> 
    if Hashtbl.mem t x then Hashtbl.find t x 
    else let res = f x in Hashtbl.add t x res; res 


type lspec = 
  Spec of 
     (terminal * precInfo) list * 
     nonterminal list * 
     (nonterminal * precInfo * syms * code option) list * 
     nonterminal list

let mkParserSpec (spec:spec) = 
  let explicitPrecInfo = 
    List.concat (list_mapi (fun n precSpecs -> List.map (fun (precSym, assoc) -> 
      (* Printf.eprintf "precSym = '%s'\n" precSym; *)
      precSym,ExplicitPrec (assoc, 10000 - n)) precSpecs) spec.assoc) in 
  let implicitSymPrecInfo = NoPrecedence in 
  let terminals = List.map fst spec.tokens @ ["error"]in 
  let is_terminal z = List.mem z terminals in 
  let prec_of_terminal sym implicitPrecInfo = 
     if List.mem_assoc sym explicitPrecInfo then List.assoc sym explicitPrecInfo 
     else match implicitPrecInfo with Some x -> x | None -> implicitSymPrecInfo in
     
  let mkSym s = if is_terminal s then T s else NT s in 
  let prods = 
    List.concat 
      (list_mapi 
         (fun i (nonterm,rules) -> 
           list_mapi 
             (fun j (Rule(syms,precsym,code)) -> 
               let precInfo = 
                 let precsym = List.fold_right (fun x acc -> match acc with Some _ -> acc | None -> match x with z when is_terminal z -> Some z | _ -> acc) syms precsym in 
                 let implicitPrecInfo = NoPrecedence in 
                 match precsym with 
                 | None -> implicitPrecInfo 
                 | Some sym -> if List.mem_assoc sym explicitPrecInfo then List.assoc sym explicitPrecInfo else implicitPrecInfo in 
               (nonterm, precInfo, List.map mkSym syms, code))
             rules)
         spec.rules) in 
  let insert n l = if List.mem n l then l else n::l in 
  let nonterminals = List.map fst spec.rules in
  let check_t t =  if not (is_terminal t) then failwith (Printf.sprintf "token %s is not declared" t) in 
  let check_nt nt =  if nt <> "error" && not (List.mem nt nonterminals) then failwith (Printf.sprintf "nonterminal '%s' has no productions" nt) in 
  List.iter (fun (nt,_,syms,_) -> List.iter (function NT nt -> check_nt nt | T n ->  check_t n) syms) prods;
  if spec.starts= [] then (failwith "at least one %start declaration is required\n");
  List.iter (fun (nt,_) -> check_nt nt) spec.types;
  let terminals = List.map (fun t -> (t,prec_of_terminal t None)) terminals in
  Spec (terminals, nonterminals, prods, spec.starts)


let lalrParserSpecToTables logf (Spec(terminals,nonterminals,prods,startNonTerminals)) = 
  (* Augment the grammar *)
  let fakeStartNonTerminals = List.map (fun nt -> "_start"^nt) startNonTerminals in 
  let nonterminals = fakeStartNonTerminals@nonterminals in 
  let endOfInputTerminal = "$$" in 
  let dummyLookahead = "#" in 
  let dummyPrec = NoPrecedence in
  let terminals = terminals @ [(dummyLookahead,dummyPrec); (endOfInputTerminal,dummyPrec)] in 
  let prods = List.map2 (fun a b -> a, dummyPrec,[NT b],None) fakeStartNonTerminals startNonTerminals @ prods in 
  let startNonTerminalIdx_to_prodIdx (i:int) = i in 

  (* Build indexed tables *)
  let nonterminalsWithIdxs = list_mapi (fun i n -> (i,n)) nonterminals in 
  let nonterminalIdxs = List.map fst nonterminalsWithIdxs in
  let nt_of_ntIdx = 
    let a = Array.of_list nonterminals in 
    fun i -> a.(i) in 
  let ntIdx_of_nt = 
    let a = Hashtbl.create 10 in 
    List.iter (fun (i,x) -> Hashtbl.add a x i) nonterminalsWithIdxs;
    fun i -> Hashtbl.find a i in 
  let terminalsWithIdxs = list_mapi (fun i (t,_) -> (i,t)) terminals in 
  let terminalIdxs = List.map fst terminalsWithIdxs in
  let terminal_of_terminalIdx = 
    let a = Array.of_list (List.map fst terminals) in 
    fun i -> a.(i) in 
  let terminalPrecInfo_of_terminalIdx = 
    let a = Array.of_list (List.map snd terminals) in 
    fun i -> a.(i) in 
  let terminalIdx_of_terminal = 
    let a = Hashtbl.create 10 in 
    List.iter (fun (i,x) -> Hashtbl.add a x i) terminalsWithIdxs;
    fun i -> Hashtbl.find a i in 

  (* Printf.eprintf "terminalPrecInfo(ELSE) = %a\n" outputPrecInfo (terminalPrecInfo_of_terminalIdx (terminalIdx_of_terminal "ELSE")); *)

  let errorTerminalIdx = terminalIdx_of_terminal "error" in 

  let prodsWithIdxs = list_mapi (fun i n -> (i,n)) prods in 
  let syms_of_prod = 
    let a = Array.of_list (List.map (fun (_,(_,_,syms,_)) -> Array.of_list syms) prodsWithIdxs) in 
    fun i -> a.(i) in 
  let nt_of_prod = 
    let a = Array.of_list (List.map (fun (_,(nt,_,_,_)) -> nt) prodsWithIdxs) in 
    fun i -> a.(i) in 
  let prec_of_prod = 
    let a = Array.of_list (List.map (fun (_,(_,prec,_,_)) -> prec) prodsWithIdxs) in 
    fun i -> a.(i) in 
  let sym_of_prod i n = 
    let syms = syms_of_prod i in 
    if n >= Array.length syms then None else Some (syms.(n)) in

  let cprods = List.fold_right (fun nt -> Productions.add nt (mapFilter (fun (i,(nt2,prec,syms,_)) -> if nt2=nt then Some i else None) prodsWithIdxs)) nonterminals Productions.empty in 
  let csyms = List.map (fun (t,_) -> T t) terminals @ List.map (fun nt -> NT nt) nonterminals in
 
  (* Compute the FIRST function *)
  Printf.printf  "computing first function\n"; flush stdout;

  let computedFirstTable = 
    let seeds = 
      List.map (fun (term,_) -> (T term,FirstSet.singleton (Some term))) terminals @
      List.map (fun nonTerm -> 
        (NT nonTerm, 
         List.fold_right 
           (fun prodIdx acc -> match sym_of_prod prodIdx 0 with None -> FirstSet.add None acc | Some _ -> acc) 
           (Productions.find nonTerm cprods) 
           FirstSet.empty))
        nonterminals in 
    let seed = List.fold_right (fun (x,y) acc -> FirstMap.add x y acc) seeds FirstMap.empty in 
    let add changed ss (x,y) = 
      let s = FirstMap.find x ss in 
      if FirstSet.mem y s then ss 
      else (changed := true; FirstMap.add x (FirstSet.add y s) ss) in 
    let oneRound ss = 
      let changed = ref false in 
      let first nonTermX = FirstMap.find nonTermX ss in 
      let frontier = 
        let res = ref [] in 
        List.iter 
          (fun nonTermX -> 
            List.iter
              (fun prodIdx ->
                let rhs = Array.to_list (syms_of_prod prodIdx) in 
                let rec place l =
                  match l with
                  | (yi::t) -> 
                      res := 
                         mapFilter 
                           (function None -> None | Some a -> Some (NT nonTermX,Some a)) 
                           (FirstSet.elements (first yi)) 
                         @ !res;
                      if FirstSet.mem None (first yi) then place t;
                | [] -> 
                    res := (NT nonTermX,None) :: !res in 
                place rhs)
              (Productions.find nonTermX cprods))
          nonterminals;
        !res in 
      let ss' = List.fold_left (add changed) ss frontier in
      !changed, ss' in 
    let rec loop ss = 
      let changed, ss' = oneRound ss in 
      if changed then loop ss' else ss' in 
    loop seed  in 
          
    
  let firstOfString str =
    let rec add acc l = 
      match l with 
      | [] -> FirstSet.add None acc
      | h::t -> 
          let firsth = FirstMap.find h computedFirstTable in 
          let acc = FirstSet.fold (fun x acc -> match x with None -> acc | _ -> FirstSet.add x acc) firsth acc in 
          if FirstSet.mem None firsth then add acc t else acc in 
    add FirstSet.empty str in 
  
  (* (int,int) representation of LR(0) items *)
  let prodIdx_to_item0 idx = mkItem0(idx,0)  in
  let prec_of_item0 item0 = prec_of_prod (prodIdx_of_item0 item0) in 
  let nt_of_item0 item0 = nt_of_prod (prodIdx_of_item0 item0) in 
  let lsyms_of_item0 item0 = 
    let prodIdx = prodIdx_of_item0 item0 in 
    let dotIdx = dotIdx_of_item0 item0 in 
    Array.to_list (Array.sub (syms_of_prod prodIdx) 0 dotIdx) in 
  let rsyms_of_item0 item0 = 
    let prodIdx = prodIdx_of_item0 item0 in 
    let dotIdx = dotIdx_of_item0 item0 in 
    let syms = syms_of_prod prodIdx in 
    Array.to_list (Array.sub syms dotIdx (Array.length syms - dotIdx))  in 
  let rsym_of_item0 item0 = 
    let prodIdx = prodIdx_of_item0 item0 in 
    let dotIdx = dotIdx_of_item0 item0 in 
    sym_of_prod prodIdx dotIdx in 
  let advance_of_item0 item0 = 
    let prodIdx = prodIdx_of_item0 item0 in 
    let dotIdx = dotIdx_of_item0 item0 in 
    mkItem0(prodIdx,dotIdx+1) in 

  (* Print items and other stuff *)
  let outputItem0 os item0 =
    Printf.fprintf os "    %s -> %a . %a" (nt_of_item0 item0) (* outputPrecInfo precInfo *) outputSyms (lsyms_of_item0 item0) outputSyms (rsyms_of_item0 item0)  in
      
  let outputItem1 os (Item1(item0,pretoken)) =
    Printf.fprintf os "%a, %s" outputItem0 item0 pretoken in
    
  let outputItem0Set os s = 
    Item0Set.iter (fun item -> Printf.fprintf os "%a\n" outputItem0 item) s in
  let outputItem1Set os s = 
    Item1Set.iter (fun item -> Printf.fprintf os "%a\n" outputItem1 item) s in

  let outputFirstSet os m = 
    FirstSet.iter (function None ->  Printf.fprintf os "<empty>" | Some x -> Printf.fprintf os "  term %s\n" x) m in
  let outputFirstMap os m = 
    FirstMap.iter (fun x y -> Printf.fprintf os "first '%a' = \n%a\n" outputSym x outputFirstSet y) m in
  let outputAction os m = 
    match m with 
    | Shift n -> Printf.fprintf os "  shift %d" n 
    | Reduce prodIdx ->  Printf.fprintf os "  reduce %s --> %a" (nt_of_prod prodIdx) outputSyms (Array.to_list (syms_of_prod prodIdx))
    | Error ->  Printf.fprintf os "  error"
    | Accept -> Printf.fprintf os "  accept"  in
  
  let outputActions os m = 
    Array.iteri (fun i (prec,action) -> let term = terminal_of_terminalIdx i in Printf.fprintf os "    action '%s' (%a): %a\n" term outputPrecInfo prec outputAction action) m in

  let outputActionTable os m = 
    Array.iteri (fun i n -> Printf.fprintf os "state %d:\n%a\n" i outputActions n) m in

  let outputImmediateActions os m = 
    match m with 
    | None -> Printf.fprintf os "<none>"
    | Some a -> outputAction os a in

  
  let outputGotos os m = 
    Array.iteri (fun ntIdx s -> let nonterm = nt_of_ntIdx ntIdx in match s with Some st -> Printf.fprintf os "    goto %s: %d\n" nonterm st | None -> ()) m in
  
  let outputCombined os m = 
    Array.iteri (fun i (a,b,c,d) -> Printf.fprintf os "state %d:\n  items:\n%a\n  actions:\n%a\n  immediate action: %a\n gotos:\n%a\n" i outputItem0Set a outputActions b outputImmediateActions c outputGotos d) m in
      
  
  let outputLalrTables os (prods,states, startStates,actionTable,immediateActionTable,gotoTable,endOfInputTerminalIdx,errorTerminalIdx) = 
    let combined = Array.of_list (List.map2 (fun x (y,(z,w)) -> x,y,z,w) (Array.to_list states) (List.combine (Array.to_list actionTable) (List.combine (Array.to_list immediateActionTable) (Array.to_list gotoTable)))) in 
    Printf.fprintf os "------------------------\n";
    Printf.fprintf os "states = \n%a\n" outputCombined combined;
    Printf.fprintf os "startStates = %s\n" (String.concat ";" (List.map string_of_int startStates));
    Printf.fprintf os "------------------------\n" in


  (* Closure of LR(0) nonterminals, items etc *)
  let computeClosure0_of_nt = 
    memoize
      (fun nt -> 
        let seed = (List.fold_right (prodIdx_to_item0 >> Item0Set.add) (Productions.find nt cprods) Item0Set.empty) in
        repeat 
          (Item0Set.elements, Item0Set.mem, Item0Set.add)
          (rsyms_of_item0 >> 
           (function 
             | [] -> []
             | (NT ntB) :: _ ->  List.map prodIdx_to_item0 (Productions.find ntB cprods)
             | T _ :: _ -> []))
          seed
          seed) in 

  let computeClosure0_of_rsym rsym acc = 
    match rsym with
    | Some (NT nt) -> Item0Set.union (computeClosure0_of_nt nt) acc
    | _ -> acc in

  let computeClosure0 iset = Item0Set.fold (rsym_of_item0 >> computeClosure0_of_rsym) iset iset  in

  (* Goto set of a kernel of LR(0) nonterminals, items etc *)
  (* Input is kernel, output is kernel *)
  let computeGotosOfKernel iset sym = 
    Item0Set.fold
      (fun item0 acc -> 
        match rsym_of_item0 item0 with 
        | Some sym2 when sym = sym2 -> Item0Set.add (advance_of_item0 item0) acc  
        | _ -> acc) 
      (computeClosure0 iset )
      Item0Set.empty in 
  
  (* Build the full set of LR(0) kernels *)
  Printf.printf  "building kernels\n"; flush stdout;
  let startItems = list_mapi (fun i _ -> prodIdx_to_item0 (startNonTerminalIdx_to_prodIdx i)) fakeStartNonTerminals in 
  let startKernels = List.map (Item0Set.singleton) startItems in 
  let kernels = 
    let seed = List.fold_right Item0Sets.add startKernels Item0Sets.empty in 
    Item0Sets.elements
      (repeat (Item0Sets.elements, Item0Sets.mem, Item0Sets.add)
         (fun iset -> List.filter (Item0Set.is_empty >> not) (List.map (computeGotosOfKernel iset) csyms))
         seed
         seed) in 

  (* Give an index to each LR(0) kernel, and from now on refer to them only by index *)
  (* Also develop "kernelItemIdx" to refer to individual items within a kernel *)
  let kernelsAndIdxs = list_mapi (fun i x -> (i,x)) kernels in
  let kernelIdxs = List.map fst kernelsAndIdxs in
  let kernelIdx_of_kernel = 
    let a = List.fold_right (fun (i,x) -> KernelIdxMap.add x i) kernelsAndIdxs KernelIdxMap.empty in 
    fun kernel -> 
      (* if not (KernelIdxMap.mem kernel a) then 
        (Printf.eprintf "kernelIdx_of_kernel: failed to find kernel %a\n" outputItem0Set kernel); *)
      KernelIdxMap.find kernel a in 
  let kernel_of_kernelIdx = let a = Array.of_list kernels in fun i -> a.(i) in 
  let startKernelIdxs = List.map kernelIdx_of_kernel startKernels in
  let startKernelItemIdxs = List.map2 (fun a b -> KernelItemIdx(a,b)) startKernelIdxs startItems in

  let outputKernelItemIdx os (kernelIdx,item0)  =
    Printf.fprintf os "kernel %d, item %a" kernelIdx outputItem0 item0 in

  (* An index-based cached version of the "goto" computation on LR(0) kernels *)
  let gotoKernel = 
    memoize (fun (GotoKernelIdx(kernelIdx,sym)) -> 
      let gset = computeGotosOfKernel (kernel_of_kernelIdx kernelIdx) sym in 
      if Item0Set.is_empty gset then None else Some (kernelIdx_of_kernel gset)) in 

  (* This is used to compute the closure of an LALR(1) kernel *)
  let closure1 iset = 
    repeat (Item1Set.elements, Item1Set.mem, Item1Set.add)
      (fun (Item1(item0,pretoken)) ->
        match rsyms_of_item0 item0 with 
        | [] -> []
        | (NT ntB) :: rsyms2 -> 
            let fset = firstOfString  (rsyms2 @ [T pretoken]) in 
            FirstSet.fold 
              (fun x acc -> 
                match x with 
                | None -> acc
                | Some tb -> 
                    List.fold_right 
                      (fun prodIdx acc -> Item1(prodIdx_to_item0 prodIdx,tb) :: acc) 
                      (Productions.find ntB cprods) 
                      acc)
              fset
              []
                  
        | T _ :: _ -> [])
      iset 
      iset in 

  (* Compute the "spontaneous" and "propagate" maps for each LR(0) kernelItem *)
  Printf.printf  "computing lookahead relations\n"; flush stdout;
  let spontaneous, propagate  =
    let closureOfItemWithDummy = 
      memoize (fun item0 -> closure1 (Item1Set.singleton (Item1(item0,dummyLookahead)))) in 
    let spontaneous = ref SpontaneousLookaheadBag.empty in 
    let propagate = ref LookaheadPropagationBag.empty in 
    List.iter
      (fun kernelIdx ->
        List.iter
          (fun sym ->
            match gotoKernel (GotoKernelIdx(kernelIdx,sym)) with 
            | None -> ()
            | Some gkernelIdx -> 
                Item0Set.iter
                  (fun item0 -> 
                    let jset = closureOfItemWithDummy item0 in 
                    Item1Set.iter
                      (fun (Item1(citem0, a)) -> 
                        match rsym_of_item0 citem0 with 
                        | Some rsym when rsym = sym ->
                            let gotoItem = advance_of_item0 citem0 in
                            if a <> dummyLookahead then 
                              spontaneous := addSpontaneousLookaheadBag (KernelItemIdx(gkernelIdx,gotoItem)) a !spontaneous
                            else
                              propagate := addLookaheadPropagationBag (KernelItemIdx(kernelIdx,item0)) (KernelItemIdx(gkernelIdx,gotoItem)) !propagate
                        | _ -> ())
                      jset)
                  (kernel_of_kernelIdx kernelIdx))
          csyms)
      kernelIdxs;
    !spontaneous,
    !propagate in 
 

  (* Repeatedly use the "spontaneous" and "propagate" maps to build the full set *)
  (* of lookaheads for each LR(0) kernelItem.   *)
  Printf.printf  "building lookahead table\n"; flush stdout;
  let lookaheadTable = 
    let t = ref (KernelItemLookaheadBag.empty) in 
    let add x y = t := addKernelItemLookaheadBag x y !t in 
    let mem x y = KernelItemLookaheadBag.mem x !t & TerminalSet.mem y (KernelItemLookaheadBag.find x !t) in 
    let insert changed x y = 
      if not (mem x y) then (changed := true; add x y) in 

    List.iter (fun startKernelItemIdx ->  add startKernelItemIdx endOfInputTerminal) startKernelItemIdxs;
    SpontaneousLookaheadBag.iter (fun kernelItemIdx lookaheads -> TerminalSet.iter (add kernelItemIdx) lookaheads) spontaneous;
    let rec loop () = 
      let changed = ref false in 
      List.iter 
        (fun kernelIdx -> 
          Item0Set.iter 
            (fun item0 -> 
              let kernelItemIdx =  KernelItemIdx(kernelIdx,item0) in 
              let terminalsToPropagate = if KernelItemLookaheadBag.mem kernelItemIdx !t then KernelItemLookaheadBag.find kernelItemIdx !t else TerminalSet.empty in 
              let propagateTo = if LookaheadPropagationBag.mem kernelItemIdx propagate then LookaheadPropagationBag.find kernelItemIdx propagate else KernelItemIdxSet.empty in 
              KernelItemIdxSet.iter (fun gkernelIdx -> TerminalSet.iter (insert changed gkernelIdx) terminalsToPropagate) propagateTo)
            (kernel_of_kernelIdx kernelIdx))
        kernelIdxs;
      if !changed then loop() in 
    loop();
    !t in 

  (* Now build the action tables. First a utility to merge the given action  *)
  (* into the table, taking into account precedences etc. and reporting errors. *)
  let addResolvingPrecedence arr kernelIdx termIdx (precNew, actionNew) = 
    (* Printf.eprintf "DEBUG: state %d: adding action for %s, precNew = %a, actionNew = %a\n" kernelIdx (terminal_of_terminalIdx termIdx) outputPrec precNew outputAction actionNew; *)
    (* We add in order of precedence - however the precedences may be the same, and we give warnings when rpecedence resolution is based on implicit file orderings *)
    let (precSoFar, actionSoFar) as itemSoFar = arr.(termIdx) in 
    (* Printf.eprintf "DEBUG: state %d: adding action for %s, precNew = %a, precSoFar = %a, actionSoFar = %a\n" kernelIdx (terminal_of_terminalIdx termIdx) outputPrec precNew outputPrec precSoFar outputAction actionSoFar; *)
    (* if compare_prec precSoFar precNew = -1 then failwith "addResolvingPrecedence"; *)

    let itemNew = (precNew, actionNew) in 
    let winner = 
      match itemSoFar,itemNew with 
      | (_,Shift _),(_, Shift _) -> 
         if actionSoFar <> actionNew then 
            Printf.eprintf "internal error in fsyacc: shift/shift conflict";
         itemSoFar

      | (((precShift,Shift _) as shiftItem), 
         ((precReduce,Reduce _) as reduceItem))
      | (((precReduce,Reduce _) as reduceItem), 
         ((precShift,Shift _) as shiftItem)) -> 
        begin match precReduce, precShift with 
        | (ExplicitPrec (_,p1), ExplicitPrec(assocNew,p2)) -> 
          if p1 < p2 then shiftItem
          else if p1 > p2 then reduceItem
          else
            begin match precShift with 
            | ExplicitPrec(LeftAssoc,_) ->  reduceItem
            | ExplicitPrec(RightAssoc,_) -> shiftItem
            | _ ->
               Printf.eprintf "state %d: shift/reduce error on %s\n" kernelIdx (terminal_of_terminalIdx termIdx); 
               shiftItem
            end
        | _ ->
           Printf.eprintf "state %d: shift/reduce error on %s\n" kernelIdx (terminal_of_terminalIdx termIdx); 
           shiftItem
        end
      | ((_,Reduce prodIdx1),(_, Reduce prodIdx2)) -> 
         Printf.eprintf "state %d: reduce/reduce error on %s\n" kernelIdx (terminal_of_terminalIdx termIdx); 
         if prodIdx1 < prodIdx2 then itemSoFar else itemNew
      | _ -> itemNew in 
   arr.(termIdx) <- winner in

  
  (* This build the action table for one state. *)
  let nterminals = (List.length terminals) in 
  let computeActions kernelIdx = 
    let kernel = kernel_of_kernelIdx kernelIdx in 
    let arr = Array.create nterminals (NoPrecedence,Error) in 

    let lookaheads kernelItemIdx = 
      if KernelItemLookaheadBag.mem kernelItemIdx lookaheadTable then 
        KernelItemLookaheadBag.find kernelItemIdx  lookaheadTable 
      else
        TerminalSet.empty in 

    (* Compute the LR(1) items based on lookaheads *)
    let items1 = 
      closure1
        (Item0Set.fold
           (fun item0 acc -> 
             let kernelItemIdx = KernelItemIdx(kernelIdx,item0) in 
             TerminalSet.fold 
               (fun la acc -> Item1Set.add (Item1(item0,la)) acc)
               (lookaheads kernelItemIdx)
               acc)
           kernel 
           Item1Set.empty) in 

    let items = (Item1Set.elements items1) in 

    List.iter 
      (fun (Item1(item0,lookahead)) -> 
        let nonTermA = nt_of_item0 item0 in
        match rsym_of_item0 item0 with 
        | Some (T termA) -> 
            let action =
              match gotoKernel (GotoKernelIdx(kernelIdx,T termA)) with 
              | None -> failwith "action on terminal should have found a non-empty goto state"
              | Some gkernelItemIdx -> Shift gkernelItemIdx in 
            let termIdx = terminalIdx_of_terminal termA in 
            let prec = terminalPrecInfo_of_terminalIdx termIdx in 
            addResolvingPrecedence arr kernelIdx termIdx (prec, action) 
        | None when not (List.mem nonTermA fakeStartNonTerminals) ->
            let termIdx = terminalIdx_of_terminal lookahead in 
            let prodIdx = prodIdx_of_item0 item0 in 
            let prec = prec_of_item0 item0 in 
            let action = (prec, Reduce prodIdx) in 
            addResolvingPrecedence arr kernelIdx termIdx action 
        | None when List.mem nonTermA fakeStartNonTerminals &
              lookahead = endOfInputTerminal ->
            let prec = prec_of_item0 item0 in 
            let action = (prec,Accept) in 
            let termIdx = terminalIdx_of_terminal lookahead in 
            addResolvingPrecedence arr kernelIdx termIdx action 
        | _ -> ())
      items;
    (* If there is a single item A -> B C . and no Shift or Accept actions (i.e. only Error or Reduce, so the choice of terminal *)
    (* cannot affect what we do) then we emit an immediate reduce action for the rule corresponding to that item *)
    (* Also do the same for Accept rules. *)
    let closure = (computeClosure0 kernel) in 
    let immediateAction =
      match Item0Set.elements closure with
      | [item0] ->
          begin match (rsym_of_item0 item0) with 
          | None when let reduceOrErrorAction = function Error | Reduce _ -> true | Shift _ | Accept -> false in
                      List.for_all (fun terminalIdx -> reduceOrErrorAction (snd(arr.(terminalIdx)))) terminalIdxs
              -> Some (Reduce (prodIdx_of_item0 item0))

          | None when let acceptOrErrorAction = function Error | Accept -> true | Shift _ | Reduce _ -> false in
                      List.for_all (fun terminalIdx -> acceptOrErrorAction (snd(arr.(terminalIdx)))) terminalIdxs
              -> Some Accept

          | _ -> None
          end
      | _ -> None in
    (* A -> B C . rules give rise to reductions in favour of errors *)
    Item0Set.iter 
      (fun item0 -> 
        let prec = prec_of_item0 item0 in 
        match rsym_of_item0 item0 with 
        | None ->
            List.iter 
              (fun terminalIdx -> 
                if snd(arr.(terminalIdx)) = Error then 
                  let prodIdx = prodIdx_of_item0 item0 in 
                  let action = (prec, (if List.mem (nt_of_item0 item0) fakeStartNonTerminals then Accept else Reduce prodIdx)) in 
                  addResolvingPrecedence arr kernelIdx terminalIdx action)
              terminalIdxs
        | _  -> ())
      (computeClosure0 kernel);
    arr,immediateAction in

  Printf.printf  "building action table\n"; flush stdout;
  let actionInfo = List.map computeActions kernelIdxs in 
  let actionTable = Array.of_list (List.map fst actionInfo) in
  let immediateActionTable = Array.of_list (List.map snd actionInfo) in


  (* The goto table is much simpler - it is based on LR(0) kernels alone. *)

  Printf.printf  "building goto table\n"; flush stdout;
  let gotos kernelIdx = Array.of_list (List.map (fun nt -> gotoKernel (GotoKernelIdx(kernelIdx,NT nt))) nonterminals) in 
  let gotoTable = Array.of_list (List.map gotos kernelIdxs) in

  let states = Array.of_list kernels in 
  let prods = Array.of_list (List.map (fun (nt,prec,syms,code) -> (nt, ntIdx_of_nt nt, syms,code)) prods) in 
  let tables = prods, states, startKernelIdxs, actionTable, immediateActionTable, gotoTable, terminalIdx_of_terminal endOfInputTerminal, errorTerminalIdx in 
  logf (fun logStream -> 
      Printf.printf  "writing tables to log\n"; flush stdout;
      outputLalrTables logStream tables);
  tables

  
(* Some examples for testing *)  

(*

let example1 = 
  let e = "E"  in 
  let t = "T" in 
  let plus = "+" in 
  let mul = "*" in 
  let f = "F" in 
  let lparen = "(" in 
  let rparen = ")" in 
  let id = "id" in 
  
  let terminals = [plus; mul; lparen; rparen; id] in 
  let nonterminals = [e; t; f] in 
  
  let p2 = e, (NonAssoc, ExplicitPrec 1), [NT e; T plus; NT t], None in 
  let p3 = e, (NonAssoc, ExplicitPrec 2), [NT t], None in  
  let p4 = t, (NonAssoc, ExplicitPrec 3), [NT t; T mul; NT f], None in 
  let p5 = t, (NonAssoc, ExplicitPrec 4), [NT f], None in 
  let p6 = f, (NonAssoc, ExplicitPrec  5), [T lparen; NT e; T rparen], None in 
  let p7 = f, (NonAssoc, ExplicitPrec 6), [T id], None in 

  let prods = [p2;p3;p4;p5;p6;p7] in 
  Spec(terminals,nonterminals,prods, [e])

let example2 = 
  let prods = [ "S", (NonAssoc, ExplicitPrec 1), [NT "C";NT "C"], None; 
                "C", (NonAssoc, ExplicitPrec 2), [T "c";NT "C"], None ;
                "C", (NonAssoc, ExplicitPrec 3), [T "d"] , None  ]in
  Spec(["c";"d"],["S";"C"],prods, ["S"])

let example3 = 
  let terminals = ["+"; "*"; "("; ")"; "id"] in 
  let nonterminals = ["E"; "T"; "E'"; "F"; "T'"] in 
  let prods = [ "E", (NonAssoc, ExplicitPrec 1), [ NT "T"; NT "E'" ], None;
                "E'", (NonAssoc, ExplicitPrec 2), [ T "+"; NT "T"; NT "E'"], None;
                "E'", (NonAssoc, ExplicitPrec 3), [ ], None;
                "T", (NonAssoc, ExplicitPrec 4), [ NT "F"; NT "T'" ], None;
                "T'", (NonAssoc, ExplicitPrec 5), [ T "*"; NT "F"; NT "T'"], None;
                "T'", (NonAssoc, ExplicitPrec 6), [ ], None;
                "F", (NonAssoc, ExplicitPrec 7), [ T "("; NT "E"; T ")"], None;
                "F", (NonAssoc, ExplicitPrec 8), [ T "id"], None ] in 
  Spec(terminals,nonterminals,prods, ["E"])

let example4 = 
  let terminals = ["+"; "*"; "("; ")"; "id"] in 
  let nonterminals = ["E"] in 
  let prods = [ "E", (NonAssoc, ExplicitPrec 1), [ NT "E"; T "+"; NT "E" ], None;
                "E", (NonAssoc, ExplicitPrec 2), [ NT "E"; T "*"; NT "E" ], None;
                "E", (NonAssoc, ExplicitPrec 3), [ T "("; NT "E"; T ")"], None;
                "E", (NonAssoc, ExplicitPrec 8), [ T "id"],  None ] in 
  Spec(terminals,nonterminals,prods, ["E"])

let example5 = 
  let terminals = ["+"; "*"; "("; ")"; "id"] in 
  let nonterminals = ["E"] in 
  let prods = [ "E", (NonAssoc, ExplicitPrec 1), [ NT "E"; T "+"; NT "E" ], None;
                "E", (NonAssoc, ExplicitPrec 2), [ NT "E"; T "*"; NT "E" ], None;
                "E", (NonAssoc, ExplicitPrec 3), [ T "("; NT "E"; T ")"], None;
                "E", (NonAssoc, ExplicitPrec 8), [ T "id"], None ] in 
  Spec(terminals,nonterminals,prods, ["E"])

let example6 = 
  let terminals = ["+"; "*"; "("; ")"; "id"; "-"] in 
  let nonterminals = ["E"] in 
  let prods = [ "E", (RightAssoc, ExplicitPrec 1), [ NT "E"; T "-"; NT "E" ], None;
                "E", (LeftAssoc, ExplicitPrec 1), [ NT "E"; T "+"; NT "E" ], None;
                "E", (LeftAssoc, ExplicitPrec 2), [ NT "E"; T "*"; NT "E" ], None;
                "E", (NonAssoc, ExplicitPrec 3), [ T "("; NT "E"; T ")"], None;
                "E", (NonAssoc, ExplicitPrec 8), [ T "id"], None ] in 
  Spec(terminals,nonterminals,prods, ["E"])


let example7 = 
  let prods = [ "S", (NonAssoc, ExplicitPrec 1), [NT "L";T "="; NT "R"], None; 
                "S", (NonAssoc, ExplicitPrec 2), [NT "R"], None ;
                "L", (NonAssoc, ExplicitPrec 3), [T "*"; NT "R"], None;
                "L", (NonAssoc, ExplicitPrec 3), [T "id"], None; 
                "R", (NonAssoc, ExplicitPrec 3), [NT "L"], None; ] in
  Spec(["*";"=";"id"],["S";"L";"R"],prods, ["S"])



let test ex = lalrParserSpecToTables stdout ex

(* let _ = test example2*)
(* let _ = exit 1*)
(* let _ = test example3 
let _ = test example1  
let _ = test example4
let _ = test example5
let _ = test example6 *)
*)
