(* (c) Microsoft Corporation. All rights reserved *)
(* CONTENTS-INDEX-REGEXP = FROM>^\!\* <TO *)
(*----------------------------------------------------------------------------
CONTENTS-START-LINE: HERE=3 SEP=2
CONTENTS-END-LINE:
----------------------------------------------------------------------------*)


(*-------------------------------------------------------------------------
!* The typechecker. Left-to-right constrained type checking 
 * with generalization at appropriate points.
 *------------------------------------------------------------------------- *)

(*F# 
open Microsoft.Research.AbstractIL 
open Microsoft.Research.AbstractIL.Internal 
open Microsoft.FSharp.Compiler 
module Ildiag = Microsoft.Research.AbstractIL.Diagnostics 
module Ilmorph = Microsoft.Research.AbstractIL.Morphs 
module Il = Microsoft.Research.AbstractIL.IL 
module Unsolved = Microsoft.FSharp.Compiler.FindUnsolved
module Check = Microsoft.FSharp.Compiler.PostTypecheckSemanticChecks
F#*) 
open Ildiag
open List
open Range
open Ast
open Tast
open Tastops
open Patcompile
open Nums
open Env
open Il (* Abstract IL  *)
open Lib
open Layout
open Outcome
open Infos
open Printf
open Typrelns
open Csolve
open Nameres

let generalize_inner_polymorphism = ref true
let polyrec = false
let recTyparsRigid = TyparWarnIfNotRigid
let verboseCC = false


(*-------------------------------------------------------------------------
!* Helpers that should be elsewhere
 *------------------------------------------------------------------------- *)


let isThreadOrContextStatic g attrs' = 
    fsthing_has_attrib g g.attrib_ThreadStaticAttribute attrs' ||
    fsthing_has_attrib g g.attrib_ContextStaticAttribute attrs' 

let mk_nil_pat g m ty = TPat_unionconstr(g.nil_ucref,[ty],[],m)
let mk_cons_pat g ty ph pt = TPat_unionconstr(g.cons_ucref,[ty],[ph;pt],union_ranges (range_of_pat ph) (range_of_pat pt))


let mk_let_in m nm ty e f = 
    let v,ve = mk_compgen_local m nm ty in 
    mk_let m v e (f (v,ve))

let mk_unit_delay_lambda g m e =
    let uv,ue = mk_compgen_local m "dummy" g.unit_ty in 
    mk_lambda m uv (e,type_of_expr g e) 
              


let rfinfo_instance_checks g ad m rfinfo = 
    if rfinfo_is_static rfinfo then error (Error ("field is static",m));
    rfinfo_attrib_check g rfinfo m |> commitOperationResult;        
    rfinfo_accessible_check m ad rfinfo

let il_field_instance_checks  g amap ad m finfo  =
    if il_finfo_is_static finfo then error (Error ("field is static",m));
    il_finfo_accessible_check g amap m ad finfo;
    il_finfo_attrib_check g finfo m

let minfo_checks g amap isInstance ad m minfo  =
    if minfo_is_instance minfo <> isInstance then
      error (Error (name_of_minfo minfo^" is not "^(if isInstance then "an instance" else "a static")^" method",m));

    if not (minfo_accessible g amap m  ad minfo) then 
      error (Error ("method '"^name_of_minfo minfo^"' is not accessible from this code location",m));
    minfo_attrib_check g m minfo |> commitOperationResult

(*-------------------------------------------------------------------------
!* Errors.
 *------------------------------------------------------------------------- *)

exception FunctionExpected of displayEnv * Tast.typ * range
exception NotAFunction of displayEnv * Tast.typ * range * range
exception Recursion of displayEnv * ident * Tast.typ * Tast.typ  * range
exception RecursiveUseCheckedAtRuntime of displayEnv * val_ref * range
exception LetRecEvaluatedOutOfOrder of displayEnv * val_ref * val_ref * range
exception LetRecCheckedAtRuntime of range
exception LetRecUnsound of displayEnv * val_ref list * range
exception TyconBadArgs of displayEnv * tycon_ref * int * range
exception UnionConstrWrongArguments of displayEnv * int * int * range
exception Constr_field_wrong_arguments of displayEnv * int * int * range
exception FieldsFromDifferentTypes of displayEnv * recdfield_ref * recdfield_ref * range
exception FieldGivenTwice of displayEnv * Tast.recdfield_ref * range
exception MissingFields of string list * range
exception FunctionValueUnexpected of displayEnv * Tast.typ * range
exception UnitTypeExpected of displayEnv * Tast.typ * bool * range
exception UnionPatternsBindDifferentNames of range
exception VarBoundTwice of ident
exception ValueRestriction of displayEnv * bool * val_spec * local_typar_ref * range
exception FieldNotMutable of displayEnv * Tast.recdfield_ref * range
exception ValNotMutable of displayEnv * val_ref * range
exception ValNotLocal of displayEnv * val_ref * range
exception InvalidRuntimeCoercion of displayEnv * Tast.typ * Tast.typ * range
exception IndeterminateRuntimeCoercion of displayEnv * Tast.typ * Tast.typ * range
exception IndeterminateStaticCoercion of displayEnv * Tast.typ * Tast.typ * range
exception RuntimeCoercionSourceSealed of displayEnv * Tast.typ * range
exception CoercionTargetSealed of displayEnv * Tast.typ * range
exception UpcastUnnecessary of range
exception TypeTestUnnecessary of range
exception StaticCoercionShouldUseBox of displayEnv * Tast.typ * Tast.typ * range
exception SelfRefObjCtor of bool * range
exception VirtualAugmentationOnNullValuedType of range
exception NonVirtualAugmentationOnNullValuedType of range
exception UseOfAddressOfOperator of range
exception ThreadStaticWarning of range
exception IntfImplInAugmentation of range

exception NonUniqueInferredAbstractSlot of tcGlobals * displayEnv * string * meth_info * meth_info * range
exception IndexOutOfRangeExceptionWarning of range


(*-------------------------------------------------------------------------
!* Type environments. 
 *    - Named items in scope (values)
 *    - Record of type variables that can't be generalized
 *    - Our 'location' as a concrete compilation path
 *    - mutable accumulator for the module type currently being accumulated 
 *------------------------------------------------------------------------- *)


type 
  (*F#
  [<StructuralEquality(false); StructuralComparison(false)>]
  F#*)
  tcEnv =
    { (* Named items *)
      eNameResEnv : nameResEnv; 

      (* List of items in the environment that may contain free inference *)
      (* variables (which may not be generalized). The relevant types may *)
      (* change as a result of inference equations being asserted, hence may need to *)
      (* be recomputed. *)
      eUngeneralizableItems: ((unit -> free_tyvars) * isClosedCache) list;
      
      (* Experimental : was for use by Visual Studio for pattern completions *)
      (* eLatestMatchType: Tast.typ option; *)
      
      (* Two (!) versions of the current module path *) 
      (* These are used to: *)
      (*    - Look up the appropriate point in the corresponding signature *)
      (*      see if an item is public or not *)
      (*    - Hack the fslib canonical module type to allow compiler references to these items *)
      (*    - Record the cpath for concrete modul_specs, tycon_specs and excon_specs so they can cache their generated IL representation where necessary *)
      (*    - Record the pubpath of public, concrete {val,tycon,modul,excon}_specs.  *)
      (*      This information is used mainly when building non-local references *)
      (*      to public items. *)
      (* *)
      (* Of the two, 'ePath' is the one that's barely used. It's only *)
      (* used by curr_stable_fslib_nlpath to hack the fslib module. *)
      ePath: ident list; 
      eCompPath: compilation_path; 
      eAccessPath: compilation_path; 

      (* Mutable accumulator for the current module type *) 
      eMtypeAcc: modul_typ ref; 

      eFamilyType: tycon_ref option; (* Some(tcref) indicates we can access protected members in all super types *)
      (* Information to enforce special restrictions on valid expressions *)
      (* for .NET constructors. *)
      eCtorInfo : ctorInfo option
    } 

and ctorInfo = 
    {
      (* Object model constructors have a very specific form to satisfy .NET limitations. *)
      (* For "new = \arg. { new C with ... }"; *)
      (*     ctor = 3 indicates about to type check "\arg. body", *)
      (*     ctor = 2 indicates about to type check "{ new ... }" *)
      (*     ctor = 1 indicates actually type checking the body expression *)
      (* 0 indicates everywhere else, including auxiliary expressions such e1 in "let x = e1 in { new ... }" *)
      (* REVIEW: clean up this rather odd approach ... *)
      ctorShapeCounter: int;
      
      (* Reference cell to hold results of initialized 'this' for 'type X() as x = ...' constructs *)
      (* The reference cell is used throughout the object constructor for any recursive references *)
      (* to 'this'. *)
      ctorThisRefCellVarOpt: val_ref option; 
      
      (* Are we in an object constructor, and if so have we complete construction of the *)
      (* ctor?  Used to reduce #of reactive recursion warnings *) 
      ctorPreConstruct: bool;
      ctorIsImplicit: bool  
    }
    
and isClosedCache = bool ref
  (* Flag is for: have we determined that the type of this value definitely has *)
  (* no free type variables?  Used for optimizing check for types in environment. *)
  (* Set to true if we discover the item has no further free type variables *)

let empty_tenv g =
    let cpath = CompPath (Il.ecma_mscorlib_scoref,[]) in 
    { eNameResEnv = empty_nameResEnv g;
      eUngeneralizableItems=[];
      ePath=[];
      eCompPath=cpath; (* dummy *)
      eAccessPath=cpath; (* dummy *)
      eMtypeAcc= ref (empty_mtype Namespace);
      eFamilyType=None;
      eCtorInfo=None }

let nenv_of_tenv tenv = tenv.eNameResEnv
let denv_of_tenv tenv = tenv |> nenv_of_tenv |> denv_of_nenv
let items_of_tenv tenv = tenv |> nenv_of_tenv |> items_of_nenv

(*-------------------------------------------------------------------------
!* Helpers related to determining if we're in a constructor and/or a class
 * that may be able to access "protoected" members.
 *------------------------------------------------------------------------- *)

let initialExplicitCtorInfo(ctorThisVarRefCellOpt) =
    { ctorShapeCounter=3; 
      ctorThisRefCellVarOpt = ctorThisVarRefCellOpt;
      ctorPreConstruct=true; 
      ctorIsImplicit=false} 

let initialImplicitCtorInfo() =
    { ctorShapeCounter=0; 
      ctorThisRefCellVarOpt = None; 
      ctorPreConstruct=true; 
      ctorIsImplicit=true }
      
let enter_familyRegion tcref env = { env with eFamilyType = Some tcref }
let exit_familyRegion env = 
    (* optimization to avoid reallocation *)
    match env.eFamilyType with 
    | None -> env 
    | _ -> { env with eFamilyType = None }

let within_ctorShape     env = match env.eCtorInfo with None -> false    | Some ctorInfo -> ctorInfo.ctorShapeCounter > 0
let is_ctorPreConstruct  env = match env.eCtorInfo with None -> false    | Some ctorInfo -> ctorInfo.ctorPreConstruct 
let is_ctorImplicit      env = match env.eCtorInfo with None -> false    | Some ctorInfo -> ctorInfo.ctorIsImplicit
let get_ctorShapeCounter env = match env.eCtorInfo with None -> 0        | Some ctorInfo -> ctorInfo.ctorShapeCounter
let get_recdInfo         env = match env.eCtorInfo with None -> RecdExpr | Some ctorInfo -> if ctorInfo.ctorShapeCounter = 1 then RecdExprIsObjInit else RecdExpr

let clear_ctorPreConstruct    env = {env with eCtorInfo = Option.map (fun ctorInfo -> { ctorInfo with ctorPreConstruct = false }) env.eCtorInfo }
let adjust_ctorShapeCounter f env = {env with eCtorInfo = Option.map (fun ctorInfo -> { ctorInfo with ctorShapeCounter = f ctorInfo.ctorShapeCounter }) env.eCtorInfo }
let leave_ctorShape           env = adjust_ctorShapeCounter (fun x -> 0) env

(*-------------------------------------------------------------------------
!* Add stuff to environments and register things as ungeneralizeable.
 *------------------------------------------------------------------------- *)

let free_tyvars_is_empty ftyvs = 
    Zset.is_empty ftyvs.free_loctypars &&
    Zset.is_empty ftyvs.free_loctycons 

let add_free_item_of_typ typ eUngeneralizableItems = 
    let fvs = free_in_type typ in 
    if free_tyvars_is_empty fvs then eUngeneralizableItems 
    else ((fun () -> free_in_type typ), ref false) :: eUngeneralizableItems

let rec acc_free_in_mtyp mtyp acc =
    Namemap.fold_range (type_of_val >> acc_free_in_type) mtyp.mtyp_vals
      (Namemap.fold_range (mtyp_of_modul >> acc_free_in_mtyp) (submoduls_of_mtyp mtyp) acc)
let free_in_mtyp mtyp = acc_free_in_mtyp mtyp empty_free_tyvars

let add_free_item_of_mtyp mtyp eUngeneralizableItems = 
    let fvs = free_in_mtyp mtyp in 
    if free_tyvars_is_empty fvs then eUngeneralizableItems 
    else ((fun () -> free_in_mtyp mtyp), ref false) :: eUngeneralizableItems
   
let modify_nameResEnv f env = { env with eNameResEnv = f env.eNameResEnv } 

let prim_addLocalVal v env =
    let env = modify_nameResEnv (add_vref_to_nenv (mk_local_vref v)) env in 
    {env with eUngeneralizableItems =  add_free_item_of_typ (type_of_val v) env.eUngeneralizableItems;   } 

let addLocalValMap scopem vals env =
    let env = Namemap.fold_range prim_addLocalVal vals env in 
    callEnvSink scopem (nenv_of_tenv env);
    env

let addLocalVal scopem v env = addLocalValMap scopem (Map.add (name_of_val v) v Map.empty) env

let addLocalExnc scopem exnc env =
    let env = modify_nameResEnv (add_ecref_to_nenv (mk_local_ecref exnc)) env in 
    (* Also make VisualStudio think there is an identifier in scope at the range of the identifier text of its binding location *)
    callEnvSink (range_of_tycon exnc) (nenv_of_tenv env);
    callEnvSink scopem (nenv_of_tenv env);
    env

let addLocalTycons scopem tycons env = 
     modify_nameResEnv (add_tcrefs_to_nenv (map mk_local_tcref tycons)) env 

let addLocalTyconsAndReport scopem tycons env = 
    let env = addLocalTycons scopem tycons env in
    callEnvSink scopem (nenv_of_tenv env);
    env

(*-------------------------------------------------------------------------
!* Open a structure or an IL namespace 
 *------------------------------------------------------------------------- *)

let top_modrefs_of_nonlocal_ccu ccu = top_moduls_of_ccu ccu |> Namemap.map (mk_nonlocal_ccu_top_modref ccu)
let top_tcrefs_of_nonlocal_ccu ccu  = top_tycons_of_ccu ccu |> Namemap.range |> List.map (mk_nonlocal_ccu_top_tcref ccu)

let modrefs_of_top_rooted_mtyp mtyp = submoduls_of_mtyp mtyp|> Namemap.map mk_local_modref
let tcrefs_of_top_rooted_mtyp mtyp  = tycons_of_mtyp mtyp |> Namemap.range |> List.map mk_local_tcref

let open_modul scopem env modref =
    let env = modify_nameResEnv (add_modref_contents_to_nenv modref) env in 
    callEnvSink scopem (nenv_of_tenv env);
    env

let add_top_rooted_mrefs env modrefs  = modify_nameResEnv (add_modrefs_to_nenv true modrefs) env

let add_nonlocal_ccu scopem env ccu = 
    let env = add_top_rooted_mrefs env (top_modrefs_of_nonlocal_ccu ccu) in 
    let env = modify_nameResEnv (add_tcrefs_to_nenv (top_tcrefs_of_nonlocal_ccu ccu)) env in
    callEnvSink scopem (nenv_of_tenv env);
    env

let add_local_top_rooted_mtyp scopem env mtyp = 
    let env = add_top_rooted_mrefs env (modrefs_of_top_rooted_mtyp mtyp) in 
    let env = modify_nameResEnv (add_tcrefs_to_nenv (tcrefs_of_top_rooted_mtyp mtyp)) env in
    let env = {env with eUngeneralizableItems = add_free_item_of_mtyp mtyp env.eUngeneralizableItems} in
    callEnvSink scopem (nenv_of_tenv env);
    env

let add_modul_abbrev scopem id modrefs env =
    let env = modify_nameResEnv (add_modul_abbrev_to_nenv id modrefs) env in
    callEnvSink scopem (nenv_of_tenv env);
    env

let add_local_submodul scopem env nm modul =
    let env = modify_nameResEnv (add_modref_to_nenv false (mk_local_modref modul)) env in 
    let env = {env with eUngeneralizableItems = add_free_item_of_mtyp (mtyp_of_modul modul) env.eUngeneralizableItems} in
    callEnvSink scopem (nenv_of_tenv env);
    env

let register_declared_typars typars env = 
    {env with eUngeneralizableItems =  List.fold_right (mk_typar_ty >> add_free_item_of_typ) typars env.eUngeneralizableItems }

let add_declared_typars check typars env = 
    let env = modify_nameResEnv (add_declared_typars_to_nenv check typars) env in
    register_declared_typars typars env

(*-------------------------------------------------------------------------
!* Compilation environment for typechecking a compilation unit. Contains the
 * F# and .NET modules loaded from disk, the search path, a table indicating
 * how to map F# modules to assembly names, and some nasty globals 
 * related to type inference. These are:
 *   - all the type variables generated for this compilation unit
 *   - the set of active fixups for "letrec" type inference 
 *------------------------------------------------------------------------- *)

type autoModuleResolver = (ident -> Tast.modul_ref option)

type 
  (*F#
  [<StructuralEquality(false); StructuralComparison(false)>]
  F#*)
  cenv = 
    { 

      g: Env.tcGlobals;

      (* Push an entry every time a recursive value binding is used, *)
      (* in order to be able to fix up recursive type applications as *)
      (* we infer type parameters *)
      (* REVIEW: cleanup this use of mutation? *)
      mutable recUses: (expr ref * range * bool) vspec_mmap;

      (* Environment needed to convert IL types to F# types in the importer. *)
      amap: Import.importMap; 

      (* Used to transform typars into new inference typars *)
      (* REVIEW: clean this up. See notes in nameres.ml *)
      mutable ginstf: range -> typars -> tinst;

      (* Experimental: Automatically resolve unresolved module names to assembly loads *)
      autoModuleResolver: autoModuleResolver;

      (* Holds a reference to the component being compiled. *)
      (* This field is very rarely used (mainly when fixing up forward references to fslib. *)
      topCcu: ccu; 
      
      (* Holds the current inference constraints *)
      css: constraint_solver_state;
      
      (* Are we compiling the signature of a module from fslib? *)
      compilingCanonicalFslibModuleType: bool;
      isSig: bool;
      haveSig: bool;
      
      niceNameGen: niceNameGenerator;
      
      conditionalDefines: string list;
  }

let new_cenv (g,niceNameGen,assemMap,topCcu,isSig,haveSig,resolver,conditionalDefines) =
    let amap = { Import.g=g; Import.assemMap=assemMap } in 
    { g=g;
      ginstf = (fun _ -> failwith "cenv.ginstf not initialized");
      amap= amap;
      recUses=vspec_mmap_empty(); 
      topCcu = topCcu;
      css= { css_g=g; css_amap=amap; css_cxs= Hashtbl.create 10 } ;
      niceNameGen=niceNameGen;
      autoModuleResolver=resolver;
      isSig=isSig;
      haveSig=haveSig;
      compilingCanonicalFslibModuleType=(isSig || not haveSig) && g.compilingFslib;
      conditionalDefines=conditionalDefines }


let new_anon_inference_tyvar cenv info = Csolve.new_anon_inference_tyvar info
let new_compgen_inference_tyvar cenv info = Csolve.new_compgen_inference_tyvar info
let new_inference_tyvar cenv () = Csolve.new_inference_tyvar ()
let new_error_tyvar cenv () = Csolve.new_error_tyvar ()
let new_inference_typ cenv () = Csolve.new_inference_typ ()
let new_error_typ cenv () = Csolve.new_error_typ ()
let new_inference_typs cenv l = Csolve.new_inference_typs l
let copy_and_fixup_typars cenv m rigid tpsorig = Csolve.freshen_and_fixup_typars cenv.amap m rigid [] [] tpsorig
let new_tinst cenv m tpsorig = Csolve.new_tinst cenv.amap m tpsorig
let freshen_tps cenv m tpsorig = Csolve.freshen_tps cenv.amap m tpsorig
let freshen_minfo cenv m minfo = Csolve.freshen_minfo cenv.g cenv.amap m minfo
let unifyE cenv env m ty1 ty2 = Csolve.unify (denv_of_tenv env) cenv.css m ty1 ty2



(*-------------------------------------------------------------------------
!* Generate references to the module being generated - used for
 * public items.
 *------------------------------------------------------------------------- *)

let curr_stable_fslib_nlpath cenv env = NLPath(cenv.topCcu, path_of_lid env.ePath)
let curr_cpath env = env.eCompPath
let curr_access_cpath env = env.eAccessPath


let mk_inner_env env nm istype = 
    let path = env.ePath @ [nm] in 
    (* Note: here we allocate a new module type accumulator *)
    let mtypeAcc = ref (empty_mtype istype) in
    let cpath = mk_cpath env.eCompPath nm.idText istype in 
    { env with ePath = path; 
               eCompPath = cpath;
               eAccessPath = cpath;
               eNameResEnv = { env.eNameResEnv with edenv = denv_add_open_path (path_of_lid path) (denv_of_tenv env)};
               eMtypeAcc = mtypeAcc  },mtypeAcc


let mk_inner_env_for_tcref env tcref isExtension = 
    (* Extension members don't get access to protected stuff *)
    if isExtension then env  else
    (* Regular members get access to protected stuff *)
	let env = (enter_familyRegion tcref env)  in
	(* Note: assumes no nesting *)
	let env = { env with eAccessPath = mk_cpath env.eCompPath (name_of_tcref tcref) AsNamedType } in
	env

let mk_inner_env_for_meminfo env v = 
    match member_info_of_val v with 
    | None -> env
    | Some(vspr) -> mk_inner_env_for_tcref env (apparent_parent_of_vspr_val v) (isext_of_val v)

let curr_mtyp_acc env = !(env.eMtypeAcc) 
let set_curr_mtyp_acc env x =  env.eMtypeAcc := x


(*-------------------------------------------------------------------------
!* Optimized unification routines that avoid creating new inference 
 * variables unnecessarily
 *------------------------------------------------------------------------- *)

let unify_tuple cenv denv m ty ps = 
    let ptys = 
      if is_tuple_ty ty then 
        let ptys = dest_tuple_typ ty in 
        if (length ps) = (length ptys) then ptys 
        else new_inference_typs cenv ps
      else new_inference_typs cenv ps in 
    unify denv cenv.css m ty (TType_tuple ptys);
    ptys

let unify_fun_and_undo_if_failed cenv denv m ty =
    if verbose then  dprintf0 "--> unify_fun\n";
    (*  avoid creating new inference variables unnecessarily *)
    if is_fun_ty ty then Some(dest_fun_typ ty) else
    let dty = new_inference_typ cenv () in
    let rty = new_inference_typ cenv () in
    if unify_and_undo_if_failed  denv cenv.css m ty (dty --> rty) then 
      Some(dty,rty)
    else 
      None

let unify_fun extraInfo cenv denv m ty =
    match unify_fun_and_undo_if_failed cenv denv m ty with
    | Some res -> res
    | None -> 
        match extraInfo with 
        | Some m2 -> error (NotAFunction(denv,ty,m,m2))
        | None ->    error (FunctionExpected(denv,ty,m))


let unify_unit cenv denv m ty exprOpt =
  if not (unify_and_undo_if_failed denv cenv.css m ty cenv.g.unit_ty) then begin
    let dty = new_inference_typ cenv () in 
    let rty = new_inference_typ cenv () in 
    if unify_and_undo_if_failed denv cenv.css m ty (dty --> rty) then 
        warning (FunctionValueUnexpected(denv,ty,m))
    else
        let perhapsProp = 
            type_equiv cenv.g cenv.g.bool_ty ty &&
            match exprOpt with 
            | Some(TExpr_app(TExpr_val(vf,_,_),_,_,[_;_],_)) when name_of_vref vf = opname_Equals -> true
            | _ -> false in 
        warning (UnitTypeExpected (denv,ty,perhapsProp,m)); 
  end


(*-------------------------------------------------------------------------
!* Attribute target flags
 *------------------------------------------------------------------------- *)

let attrTgtAssembly    = !!!(0x00000001)
let attrTgtModule      = !!!(0x00000002)
let attrTgtClass       = !!!(0x00000004)
let attrTgtStruct      = !!!(0x00000008)
let attrTgtEnum        = !!!(0x00000010)
let attrTgtConstructor = !!!(0x00000020)
let attrTgtMethod      = !!!(0x00000040)
let attrTgtProperty    = !!!(0x00000080)
let attrTgtField       = !!!(0x00000100)
let attrTgtEvent       = !!!(0x00000200)
let attrTgtInterface   = !!!(0x00000400)
let attrTgtParameter   = !!!(0x00000800)
let attrTgtDelegate    = !!!(0x00001000)
let attrTgtReturnValue = !!!(0x00002000)
let attrTgtAll         = !!!(0x00003FFF)

let attrTgtBinding    = attrTgtField    ||| attrTgtMethod    ||| attrTgtEvent    ||| attrTgtProperty
let attrTgtFieldDecl  = attrTgtField    ||| attrTgtProperty
let attrTgtConstrDecl = attrTgtClass    ||| attrTgtMethod    ||| attrTgtProperty
let attrTgtTyconDecl  = attrTgtClass    ||| attrTgtInterface ||| attrTgtDelegate ||| attrTgtStruct ||| attrTgtEnum
let attrTgtExnDecl    = attrTgtClass
let attrTgtModuleDecl = attrTgtClass
let attrTgtTop        = attrTgtAssembly ||| attrTgtModule    ||| attrTgtMethod


(*-------------------------------------------------------------------------
!* Typecheck constant terms.
 *------------------------------------------------------------------------- *)

let tc_const cenv ty m env c =
    let unif ty2 = unifyE cenv env m ty ty2 in 
    match c with 
    | Const_unit         -> unif cenv.g.unit_ty;       TConst_unit
    | Const_bool i       -> unif cenv.g.bool_ty;       TConst_bool i
    | Const_int8 i       -> unif cenv.g.sbyte_ty;      TConst_int8 i
    | Const_int16 i      -> unif cenv.g.int16_ty;      TConst_int16 i
    | Const_int32 i      -> unif cenv.g.int_ty;        TConst_int32 i
    | Const_int64 i      -> unif cenv.g.int64_ty;      TConst_int64 i
    | Const_nativeint i  -> unif cenv.g.nativeint_ty;  TConst_nativeint i
    | Const_uint8 i      -> unif cenv.g.byte_ty;       TConst_uint8 i
    | Const_uint16 i     -> unif cenv.g.uint16_ty;     TConst_uint16 i
    | Const_uint32 i     -> unif cenv.g.uint32_ty;     TConst_uint32 i
    | Const_uint64 i     -> unif cenv.g.uint64_ty;     TConst_uint64 i
    | Const_unativeint i -> unif cenv.g.unativeint_ty; TConst_unativeint i
    | Const_float32 f    -> unif cenv.g.float32_ty;    TConst_float32 f
    | Const_float f      -> unif cenv.g.float_ty;      TConst_float f
    | Const_char c       -> unif cenv.g.char_ty;       TConst_char c
    | Const_string (s,m) -> unif cenv.g.string_ty;     TConst_string s
    | Const_decimal s -> unifyE cenv env m ty cenv.g.decimal_ty;  TConst_decimal s
    | Const_bigint s -> unifyE cenv env m ty cenv.g.bigint_ty;  TConst_bigint s
    | Const_bignum s -> unifyE cenv env m ty cenv.g.bignum_ty;  TConst_bignum s

    | Const_bytearray _ -> error(Error("Bytearray literals may not be used here",m))

let tc_field_init m lit = 
    match lit with 
    | FieldInit_bytes s   -> TConst_string s
    | FieldInit_ref       -> error (Error("An error occurred importing a literal value for a field",m))
    | FieldInit_bool    b -> TConst_bool b
    | FieldInit_char    c -> TConst_uint16 c
    | FieldInit_int8    x -> TConst_int8 x
    | FieldInit_int16   x -> TConst_int16 x
    | FieldInit_int32   x -> TConst_int32 x
    | FieldInit_int64   x -> TConst_int64 x
    | FieldInit_uint8   x -> TConst_uint8 x
    | FieldInit_uint16  x -> TConst_uint16 x
    | FieldInit_uint32  x -> TConst_uint32 x
    | FieldInit_uint64  x -> TConst_uint64 x
    | FieldInit_float32 f -> TConst_float32 f
    | FieldInit_float64 f -> TConst_float f 


(*-------------------------------------------------------------------------
!* Arities. These serve two roles in the system: 
 *  1. syntactic arities come from the syntactic forms found in 
 *     signature files and the syntactic forms of function and member definitions.
 *  2. compiled arities representing representation choices w.r.t. internal representations of
 *     functions and members.
 *------------------------------------------------------------------------- *)

let adjustTopValSigData g ty (TopValSynData(argsData,retData) as sigMD) = 
    (*Adjust the arities that came from the parsing of the toptyp (arities) to be a topValSynData. *)
    (*This means replacing the "1" arising from a "unit -> ty" with a 0. *)
    if is_fun_ty ty && type_equiv g g.unit_ty (domain_of_fun_typ ty) && length argsData >= 1 && length(hd(argsData)) >= 1  then 
      TopValSynData(tl(hd argsData) :: tl argsData, retData)
    else 
      sigMD 

(* arity of a value, except the number of typars is not yet inferred *)
type partialArityOfVal = PartialArityInfo of topArgInfo list list * topArgInfo 

let translateTopArgSynData isArg m tc_attr (TopArgSynData(attrs,isOpt,nm)) = 
    let optAttrs = if  isOpt then [Attr(path_to_lid m ["Microsoft";"FSharp";"Core";"OptionalArgument"], mksyn_unit m,None,m)] else [] in 
    if isArg && nonNil(attrs) && isNone(nm) then 
        warning(Error("A parameter with attributes must also be given a name, e.g. '[<Attribute>] paramName : paramType'",m));

    if not(isArg) && isSome(nm) then 
        warning(Error("Return values may not have names",m));
       
    TopArgData(tc_attr (optAttrs@attrs),nm)

let translateTopValSynData m tc_attr (TopValSynData(argsData,retData)) = 
    (* Members have an arity inferred from their syntax. This "topValSynData" is not quite the same as the arities *)
    (* used in the middle and backends of the compiler ("compiledArity"). *)
    (* "0" in a topValSynData (see Ast.arity_of_pat) means a "unit" arg in a compiledArity *)
    (* Hence remove all "zeros" from arity and replace them with 1 here. *)
    (* Note we currently use the compiled form for choosing unique names, to distinguish overloads because this must match up *)
    (* between signature and implementation, and the signature just has "unit". *)
    PartialArityInfo (argsData |> map (function [] -> TopValData.unitArgData 
                                              | args -> args |> List.map (translateTopArgSynData true m (tc_attr attrTgtParameter))), 
                      retData |> translateTopArgSynData false m (tc_attr attrTgtReturnValue))

let translatePartialArity tps (PartialArityInfo (argsData,retData)) = TopValInfo(length tps,argsData,retData)


(*-------------------------------------------------------------------------
!* Members
 *------------------------------------------------------------------------- *)

let computeLogicalCompiledName (id:ident) memFlags = 
    match memFlags.memFlagsKind with 
    | MemberKindClassConstructor -> ".cctor"
    | MemberKindConstructor -> ".ctor"
    | MemberKindMember -> id.idText 
    | MemberKindPropertyGetSet ->  error(InternalError("MemberKindPropertyGetSet only expected in parse trees",id.idRange))
    | MemberKindPropertyGet ->  "get_"^id.idText
    | MemberKindPropertySet ->  "set_"^id.idText 

(* Make the unique "name" for a member. *)
(* *)
(* Note: the use oa a mangled string as the unique name for a member is one of the reasons *)
(* we end up needing things like OverloadID (which is used to make the name unique). *)
(* This means the compiler implementation gives rise to a limitation in the design - not ideal. *)
let mkMemberDataAndUniqueId g tcref isExtensionMember attrs optImplSlotTy memFlags topValSynData id  =
    (* implty = None (for classes) or Some ty (when implementing interface type ty) *)
    (* dprintf2 "mkMemberDataAndUniqueId, id = %s, optImplSlotTy = %a" *)
    let logical_il_name = computeLogicalCompiledName id memFlags in 
    let optIntfSlotTy = match optImplSlotTy with Some ty when is_interface_typ ty -> Some ty | _ -> None in 
    let qualified_il_name = Option.fold_right (tcref_of_stripped_typ >> qualified_name_of_tcref) optIntfSlotTy logical_il_name in
    let vspr = { vspr_apparent_parent=tcref; 
                 vspr_flags=memFlags; 
                 vspr_implemented=false;
                 (* If this is an interface method impl then the name we use for the method changes *)
                 vspr_il_name=qualified_il_name;
                 (* NOTE: This value is initially only set for interface implementations and those overrides *)
                 (* where we manage to pre-infer which abstract is overriden by the method. It is filled in  *)
                 (* properly when we check the allImplemented implementation checks at the end of the inference scope. *)
                 vspr_implements_slotsig=optImplSlotTy |> Option.map (fun ity -> TSlotSig(logical_il_name,ity,[],[],[],g.unit_ty)) } in 
    begin 
      let isInstance = vsprCompiledAsInstance g tcref isExtensionMember vspr attrs in
      if (memFlags.memFlagsVirtual  || memFlags.memFlagsAbstract || isSome optIntfSlotTy) then begin
        if not isInstance then
          errorR(VirtualAugmentationOnNullValuedType(id.idRange));
      end else if memFlags.memFlagsInstance then begin
        if not isExtensionMember && not isInstance then
          warning(NonVirtualAugmentationOnNullValuedType(id.idRange))
      end;
    end;
    let id = 
           let tname = name_of_tcref tcref in 
           let text = 
             match memFlags.memFlagsKind with 
             | MemberKindClassConstructor -> tname^".cctor"
             | MemberKindConstructor -> tname^".ctor"
             | MemberKindMember -> tname^"."^id.idText 
             | MemberKindPropertyGetSet ->  failwith "mkMemberDataAndUniqueId"
             | MemberKindPropertyGet ->  tname^".get_"^id.idText
             | MemberKindPropertySet ->  tname^".set_"^id.idText in 
           let text = match memFlags.memFlagsOverloadQualifier with
             | None   -> text^"."^string_of_int (fold_left (+) 0 (SynArgInfo.aritiesOfArgs topValSynData))
             | Some t -> text^"."^Bytes.unicode_bytes_as_string t in
           let text = if memFlags.memFlagsOverride then text^".override" else text in 
           let text = Option.fold_right (tcref_of_stripped_typ >> qualified_name_of_tcref) optIntfSlotTy text in
           ident(text,id.idRange) in 
    vspr,id


(*-------------------------------------------------------------------------
!* Inferred type parameters and inference flexibility
 * The declared type parameters, e.g. let f<'a> (x:'a) = x, plus an indication 
 * of whether additional polymorphism may be inferred, e.g. let f<'a,..> (x:'a) y = x 
 *------------------------------------------------------------------------- *)

type inferenceFlexibility = TIFlex of Tast.typars * bool
let infer_iflex = TIFlex ([],true) 
let no_iflex = TIFlex ([],false) 
type argAndRetAttribs = attribs list list * attribs
let noArgOrRetAttribs : argAndRetAttribs = ([],[])

(*-------------------------------------------------------------------------
 * What sort of bindings are we processing?  Processing "declaration" bindings that make up a module (such as "let x = 1 let y = 2") 
 * currently shares the same code paths (e.g. tc_let and tc_letrec) as processing expression bindings (such as "let x = 1 in ...") 
 * However the former get published to the implicitly accumulated module type, but the latter don't. 
 * Likewise there are some distinctions made when processing the "let" bindings in the implicit class constructor syntax. 
 *------------------------------------------------------------------------- *)

type declProcessingFlag = 
    | ModuleOrMemberBinding 
    | ExtensionBinding 
    | ClassLetBinding 
    | ObjectExpressionOverrideBinding
    | ExpressionBinding 

let must_have_arity = function
    | ModuleOrMemberBinding -> true
    | ExtensionBinding -> true
    | ClassLetBinding -> false
    | ObjectExpressionOverrideBinding -> false
    | ExpressionBinding -> false

let access_modifier_permitted = function
    | ModuleOrMemberBinding -> true
    | ExtensionBinding -> true
    | ClassLetBinding 
    | ObjectExpressionOverrideBinding 
    | ExpressionBinding -> false

let implicitly_static = function
    | ModuleOrMemberBinding -> true
    | ExtensionBinding -> true
    | ClassLetBinding -> false
    | ObjectExpressionOverrideBinding -> false
    | ExpressionBinding -> false

let always_generalize = function
    | ModuleOrMemberBinding -> true
    | ExtensionBinding -> true
    | ClassLetBinding -> false
    | ObjectExpressionOverrideBinding -> false
    | ExpressionBinding -> false

let can_have_attributes = function
    | ModuleOrMemberBinding -> true
    | ExtensionBinding -> true
    | ClassLetBinding -> false
    | ObjectExpressionOverrideBinding -> true
    | ExpressionBinding -> false

let can_generalize_constrained_typar = function
    | ModuleOrMemberBinding -> true
    | ExtensionBinding -> true
    | ClassLetBinding -> false  (* no since even functions are rep. as fields, and ilxgen will reject these *)
    | ObjectExpressionOverrideBinding -> true
    | ExpressionBinding -> true
    
let convert_to_linear_bindings = function
    | ModuleOrMemberBinding -> true
    | ExtensionBinding -> true
    | ClassLetBinding -> true
    | ObjectExpressionOverrideBinding -> true
    | ExpressionBinding -> false 


(*-------------------------------------------------------------------------
!* Data structures that track the gradual accumualtion of information
 * about values and members during inference.
 *------------------------------------------------------------------------- *)

(* The results of preliminary pass over patterns to extract variables being declared. *)
type prelimValScheme1 = 
    PrelimValScheme1 of 
        ident * 
        inferenceFlexibility * 
        Tast.typ * 
        Tast.val_member_info * 
        bool * 
        val_inline_info * 
        val_base_or_this * 
        argAndRetAttribs * 
        access option * 
        bool

(* The results of applying let-style generalization after type checking. *)
type prelimValScheme2 = 
    PrelimValScheme2 of 
        ident * 
        typ_scheme * 
        val_member_info * 
        bool * 
        val_inline_info * 
        val_base_or_this * 
        argAndRetAttribs * 
        access option * 
        bool *
        bool (* hasDeclaredTypars *) 
        

(* The results of applying arity inference to PrelimValScheme2 *)
type valScheme = 
    ValScheme of 
        ident * 
        typ_scheme * 
        Tast.val_arity_info * 
        val_member_info * 
        bool * 
        val_inline_info * 
        val_base_or_this * 
        access option * 
        bool * 
        bool *
        bool (* isTyFunc *) 


(* RecursiveBindingInfo - flows through initial steps of tc_letrec *)    
type recursiveBindingInfo =
    RBInfo of
      typars * 
      val_inline_info * 
      val_spec * 
      inferenceFlexibility * 
      partialArityOfVal option 
      * val_spec option * 
      val_spec option * 
      access option * 
      declProcessingFlag
        
(*-------------------------------------------------------------------------
!* Data structure that tracks the whole process of taking a syntactic binding and
 * checking it.
 *------------------------------------------------------------------------- *)

(* Translation of patterns is split into three phases. The first collects names. *)
(* The second is run after val_specs have been created for those names and inference *)
(* has been resolved. The second phase is run by applying a function returned by the *)
(* first phase. The input to the second phase is a map that gives the val_spec and type scheme *)
(* for each value bound by the pattern. *)
type tcPatPhase2Input = 
    TcPatPhase2Input of (val_spec * typ_scheme) namemap


(* The first phase of translation of binding leaves a whole goop of information. *)
(* This is a bit of a mess: much of this information is carried on a per-value basis by the *)
(* "prelimValScheme1 namemap". *)
type tbinding_info = 
    TBindingInfo of 
       val_inline_info * 
       bool *   (* immutable? *)
       attribs * 
       xmlDoc * 
       (tcPatPhase2Input -> Patcompile.pat) * 
       inferenceFlexibility * 
       prelimValScheme1 namemap * 
       expr * 
       argAndRetAttribs * 
       Tast.typ * 
       range *
       bool * (* compiler generated? *)
       tconst option (* literal value? *)

(*-------------------------------------------------------------------------
!* Helpers related to type schemes
 *------------------------------------------------------------------------- *)

let generalizedTypeForTypeScheme typeScheme = 
    let (TypeScheme(generalizedTypars,_,tau)) = typeScheme in
    try_mk_forall_ty generalizedTypars tau

let nonGenericTypeScheme ty = TypeScheme([],[],ty)

(*-------------------------------------------------------------------------
!* Helpers related to publishing values, types and members into the
 * elaborated representation.
 *------------------------------------------------------------------------- *)

let updateAccModuleType cenv env f = 
    (* Hack the fslib CCU to ensure forward stable references used by *)
    (* the compiler can be resolved ASAP. Not at all pretty but it's hard to *)
    (* find good ways to do references from the compiler into a term graph *)
    if cenv.compilingCanonicalFslibModuleType then begin
       let modul = modul_of_nlpath (curr_stable_fslib_nlpath cenv env) in 
       if verbose then  dprintf1 "updating contents of CCU-held fslib module %d in case forward references occur form the compiler to this construct\n" (stamp_of_modul modul);
       (data_of_modul modul).tycon_modul_contents <- notlazy (f true (mtyp_of_modul modul));
    end;
    set_curr_mtyp_acc env (f false (curr_mtyp_acc env))
  
let publishModuleDefn cenv env mspec = 
    updateAccModuleType cenv env (fun intoFslibCcu mty -> 
       let nm = name_of_modul mspec in 
       if intoFslibCcu then mty
       else { mty with mtyp_submoduls = Map.add nm mspec mty.mtyp_submoduls })

let publishTypeDefn cenv env tycon = 
    updateAccModuleType cenv env (fun _ mty -> { mty with mtyp_tycons = Map.add (name_of_tycon tycon) tycon mty.mtyp_tycons })

let publishValueDefnPrim cenv env vspec = 
    updateAccModuleType cenv env (fun _ mty -> { mty with mtyp_vals = Map.add (name_of_val vspec ) vspec mty.mtyp_vals } )

let checkForDuplicateVal cenv env vspec =
    let vtb = (curr_mtyp_acc env).mtyp_vals in 
    let nm = name_of_val vspec in 
    if modbind_of_val vspec && Map.mem nm vtb then 
      errorR(Duplicate("value",id_of_val vspec));
    match arity_of_val vspec with 
    | Some info when Map.mem ("get_"^nm) vtb -> 
         errorR(Error(sprintf "This module contains values '%s' and 'get_%s'. Two values in a module may not have names distinguished only by the suffix 'get_'" nm nm, range_of_val vspec))
    | _ -> ()
        
let publishValueDefn cenv env declKind vspec =
    if (declKind = ModuleOrMemberBinding) && 
       (mkind_of_mtyp (curr_mtyp_acc env) = Namespace) && 
       (isNone (member_info_of_val vspec)) then 
           errorR(Error("Namespaces may not contain values. Consider using a module to hold your value declarations",range_of_val vspec));
    if (declKind = ExtensionBinding) && 
       (mkind_of_mtyp (curr_mtyp_acc env) = Namespace) then 
           errorR(Error("Namespaces may not contain extension members. Consider using a module to hold declarations of extension members",range_of_val vspec));
    if (declKind = ModuleOrMemberBinding or declKind = ExtensionBinding) then begin
        checkForDuplicateVal cenv env vspec;
        (* Publish the value to the module type being generated. *)
        publishValueDefnPrim cenv env vspec
    end;
    match member_info_of_val vspec with 
    | Some vspr when 
         not (compgen_of_val vspec) && 
         not (declKind = ExtensionBinding) && 
         (* Static initializers don't get published to the tcaug *)
         not (vspr.vspr_flags.memFlagsKind = MemberKindClassConstructor) -> 
        
        (* Check that member doesn't clash with a field *)
        let cname = compiled_name_of_val vspec in 
             
        begin 
            (* NOTE: static and secret fields are ok here: we want to check for all name clashes *)
            match any_rfield_of_tcref_by_name (actual_parent_of_vspr_val vspec) cname with 
            | Some v -> errorR (Error(sprintf "The name of the member '%s' clashes with the name of the field" cname,range_of_val vspec));
            | None -> ()
        end;
       
        (* tcaug_adhoc cleanup: we should not need this table nor this mutation *)
        (* The values should be carried in the environment and in the generated module type. *)
        (* Opening module types should add the values to the environment. *)
        let tcaug = tcaug_of_tcref (apparent_parent_of_vspr_val vspec) in
        tcaug.tcaug_adhoc <- Namemap.add_multi vspr.vspr_il_name (mk_local_vref vspec) tcaug.tcaug_adhoc
    |  _ -> ()

let combineVisibilityAttribs vis1 vis2 m = 
   if isSome vis1 && isSome vis2 then 
        errorR(Error("Multiple visibility attributes have been specified for this identifier",m));
   if isSome vis1 then vis1 else vis2

let computeAccessAndCompPath env declKind m vis actualParent = 
    let accessPath = curr_access_cpath env in 
    if isSome vis && not (access_modifier_permitted declKind) then 
        errorR(Error("Multiple visibility attributes have been specified for this identifier. 'let' bindings in classes are always private, as are any 'let' bindings inside expressions",m)); 
    let vis = 
        match vis with 
        | None -> taccessPublic (* a module or member binding defaults to "public" *)
        | Some a when a = accessPublic -> taccessPublic
        | Some a when a = accessPrivate -> TAccess [accessPath]
        | Some a when a = accessInternal -> 
            let (CompPath(scoref,_)) = accessPath in
            TAccess [CompPath(scoref,[])]
        | _ -> 
            errorR(InternalError("Unrecognized accessibility specification",m));
            taccessPublic in

    let vis = 
        match actualParent with 
        | ParentNone -> vis 
        | Parent tcref -> 
             combineAccess vis (access_of_tycon (deref_tycon tcref)) in
    let cpath = curr_cpath env in 
    let cpath = if access_modifier_permitted declKind then Some cpath else None in 
    vis,cpath 

let mkAndPublishVal cenv env (altActualParent,inSig,declKind,vrec,(ValScheme(id,typeScheme,arity,specialRepr,mut,inlineFlag,base,vis,compgen,isIncrClass,isTyFunc)),attrs,doc,konst) =
    (* dprintf1 "mkAndPublishVal, id = %s\n" id.idText; *)
    let ty = generalizedTypeForTypeScheme typeScheme in
    let m = id.idRange in
    let isIntfImpl = 
        match specialRepr with 
        | None -> false 
        | Some vspr -> member_is_interface_impl vspr in 
   
    let isModuleBinding = (declKind = ModuleOrMemberBinding or declKind = ExtensionBinding) in
    let isExtension = (declKind=ExtensionBinding) in 
    let actualParent = 
        (* Use the parent of the member if it's available *)
        (* If it's an extension member of not a member then use the containing module. *)
        match specialRepr with 
        | Some vspr when not isExtension -> 
            if tycon_is_modul (deref_tycon (vspr.vspr_apparent_parent)) then 
                errorR(Error("vspr.vspr_apparent_parent, "^id.idText,m));

            Parent(vspr.vspr_apparent_parent)
        | _ -> altActualParent in 
             
    let vis,cpath = computeAccessAndCompPath env declKind id.idRange vis actualParent in 

    let inlineFlag = 
        if fsthing_has_attrib cenv.g cenv.g.attrib_DllImportAttribute attrs then begin 
            if inlineFlag = PseudoValue || inlineFlag = AlwaysInline then 
              errorR(Error("DLLImport stubs may not be inlined",m)); 
            NeverInline 
        end else inlineFlag in 

    let vspec = 
        new_vspec (id,ty,
                      (if ((is_byref_ty cenv.g ty) || mut) then Mutable else Immutable),
                      compgen,arity,cpath,vis,vrec,specialRepr,base,attrs,inlineFlag,doc, isModuleBinding, isExtension,isIncrClass,isTyFunc,konst,actualParent) in 
    publishValueDefn cenv env declKind vspec;

  (*  dprintf4 "register val, inSig = %b, v = %s, scopem = %a\n" inSig (name_of_val vspec) output_range (range_of_val vspec) ; *)

    (* Notify Visual Studio or another development environment that *)
    (* the value in scope at the binding point (i.e. the range of the *)
    (* identifier text for the value). If necessary use *)
    (* a faked environment containing the mapping from its display *)
    (* name to the value. 'unqualified' is true if the name at the *)
    (* binding location in the original source appears unqualified, as *)
    (* in the case of both static and instance members in signatures *)
    (* and all static members in classes. *)
    let unqualified = 
         match member_info_of_val vspec with 
         | None -> true
         | Some vspr -> inSig || not vspr.vspr_flags.memFlagsInstance in 
    (* REVIEW: this logic is to some extent part of the name resolution rules, i.e. how *)
    (* different kinds of 'values' (including members) appear in the name environment at *)
    (* different points. *)
    if unqualified && not (compgen_of_val vspec) && not (has_prefix (name_of_val vspec) "_") then begin
         let env = add_fake_named_vref_to_nenv (display_name_of_val vspec) (mk_local_vref vspec) (nenv_of_tenv env) in 
         callEnvSink (range_of_val vspec) env
      end;

    vspec

let mkAndPublishVals cenv env (altActualParent,inSig,declKind,vrec,valschemes,attrs,doc,konst) =
    Map.fold
        (fun name (ValScheme(_,typeScheme,_,_,_,_,_,_,_,_,_) as valscheme) values -> 
          Map.add name (mkAndPublishVal cenv env (altActualParent,inSig,declKind,vrec,valscheme,attrs,doc,konst), typeScheme) values)
        valschemes
        Map.empty

let mkAndPublishBaseVal cenv env baseIdOpt ty = 
    baseIdOpt |> Option.map (fun id ->
       if id.idText <> "base" then 
           warning(Error("base variables must now be called 'base'. A future revision of the F# language will require this and make 'base' a keyword",id.idRange));
       let valscheme = ValScheme(id,nonGenericTypeScheme(ty),None,None,false,NeverInline,BaseVal,None,false,false,false) in
       mkAndPublishVal cenv env (ParentNone,false,ExpressionBinding,ValNotInRecScope,valscheme,[],emptyXMLDoc,None))

let mkAndPublishCtorThisRefCellVal cenv env thisIdOpt thisTy = 
    match thisIdOpt with 
    | Some thisId -> 
        if not (is_fsobjmodel_ty thisTy) then 
            errorR(Error("Structs may only bind a 'this' parameter at member declarations",thisId.idRange));

        let thisvs = ValScheme(thisId,nonGenericTypeScheme(mk_refcell_ty cenv.g thisTy),None,None,false,NeverInline,CtorThisVal,None,false,false,false) in
        Some(mkAndPublishVal cenv env (ParentNone,false,ExpressionBinding,ValNotInRecScope,thisvs,[],emptyXMLDoc,None))
    | None -> None 


(*-------------------------------------------------------------------------
!* Type inference for recursive bindings
 *------------------------------------------------------------------------- *)

let adjustUsesOfRecValue cenv vrefTgt (ValScheme(id,typeScheme,toparity,_,_,_,_,_,_,_,_)) =
    let (TypeScheme(generalizedTypars,_,ty)) = typeScheme in
    let fty = generalizedTypeForTypeScheme typeScheme in
    let lvrefTgt = deref_val vrefTgt in 
    if nonNil(generalizedTypars) then begin
      (* Find all the uses of this recursive binding and use mutation to adjust the expressions *)
      (* at those points in order to record the inferred type parameters. *)
      let recUses = vspec_mmap_find lvrefTgt cenv.recUses in 
      recUses |> iter  (fun (fixupPoint,m,isComplete) -> 
        if not isComplete then 
            (* Keep any values for explicit type arguments *)
            let fixedUpExpr = 
                let vrefFlags,tyargs0 = 
                    match !fixupPoint with 
                    | TExpr_app(TExpr_val (_,vrefFlags,_),_,tyargs0,[],m) -> vrefFlags,tyargs0
                    | TExpr_val(_,vrefFlags,_) -> vrefFlags,[] 
                    | _ -> errorR(Error("unexpected expression at recursive inference point",m)); NormalValUse,[] in 
                
                let ityargs = generalize_typars (drop (length tyargs0) generalizedTypars) in 
                prim_mk_app (TExpr_val (vrefTgt,vrefFlags,m),fty) (tyargs0 @ ityargs) [] m in 
            fixupPoint :=   fixedUpExpr)
    end;
    cenv.recUses <- vspec_map_remove lvrefTgt cenv.recUses
     

let adjustRecType cenv v1ref (ValScheme(id,typeScheme,arity,_,_,_,_,_,_,_,_)) =
    let fty = generalizedTypeForTypeScheme typeScheme in
    let vspec = deref_local_val v1ref in 
    let v1 = data_of_val vspec in 
    (* certain properties of recursive values are only fully known after inference is complete *)
    v1.val_type <- fty;
    v1.val_arity <- arity;
    set_vrec_of_vflags v1 ValNotInRecScope
       
(* Record the generated value expression as a place where we will have to *)
(* adjust using adjustUsesOfRecValue at a letrec point. Every use of a value *)
(* under a letrec gets used at the _same_ type instantiation. *)
let recordUseOfRecValue cenv vrec vrefTgt vexp m = 
    match vrec with 
    | ValInRecScope isComplete -> 
        let fixupPoint = ref vexp in 
        cenv.recUses <- vspec_mmap_add (deref_val vrefTgt) (fixupPoint,m,isComplete) cenv.recUses;
        TExpr_link (fixupPoint)
    | ValNotInRecScope -> 
        vexp

(* Get all recursive references, for fixing up delayed recursion using laziness *)
let getAllUsesOfRecValue cenv vrefTgt = 
    vspec_mmap_find vrefTgt cenv.recUses |> map (fun (fixupPoint,m,isComplete) -> (fixupPoint,m))


(*-------------------------------------------------------------------------
!* 
 *------------------------------------------------------------------------- *)

let chooseCanonicalDeclaredTyparsAfterInference g denv declaredTypars m =

    declaredTypars |> iter (fun tp -> 
      let ty = mk_typar_ty tp in
      if not (is_typar_ty ty) then 
          error(Error("This code is less generic than required by its annotations because the explicit type variable '"^name_of_typar tp ^"' could not be generalized. It was constrained to be '"^NicePrint.pretty_string_of_typ denv ty^"'",range_of_typar tp)));
    
    let declaredTypars = normalizeDeclaredtyparsForEquiRecursiveInference declaredTypars in
    if length (gen_setify typar_ref_eq declaredTypars) <> length declaredTypars then 
        errorR(Error("One or more of the explicit class or function type variables for this binding could not be generalized, because they were constrained to other types",m));
    declaredTypars

let chooseCanonicalValSchemeAfterInference g denv valscheme m =
    let (ValScheme(id,typeScheme,arityInfo,specialRepr,mut,inlineFlag,base,vis,compgen,isIncrClass,isTyFunc)) = valscheme in
    let (TypeScheme(generalizedTypars,freeChoiceTypars,ty)) = typeScheme in
    let generalizedTypars = chooseCanonicalDeclaredTyparsAfterInference g denv generalizedTypars m in
    let typeScheme = TypeScheme(generalizedTypars,freeChoiceTypars,ty) in
    let valscheme = ValScheme(id,typeScheme,arityInfo,specialRepr,mut,inlineFlag,base,vis,compgen,isIncrClass,isTyFunc) in
    valscheme

let placeTyparsInDeclarationOrder declaredTypars generalizedTypars  m  =
    declaredTypars @ (gen_subtract typar_ref_eq generalizedTypars declaredTypars) 

let setTyparRigid g denv m tp = 
    begin match solution_of_typar tp with 
    | TType_unknown -> ()
    | ty -> 
        if compgen_of_typar tp then 
            errorR(Error("A generic type parameter has been used in a way that constrains it to always be '"^NicePrint.pretty_string_of_typ denv ty^"'",m))
        else 
            errorR(Error("This type parameter has been used in a way that constrains it to always be '"^NicePrint.pretty_string_of_typ denv ty^"'",range_of_typar tp))
    end;
    set_rigid_of_tpdata (data_of_typar tp) TyparRigid

let generalizeVal cenv denv enclosingDeclaredTypars generalizedTyparsForRecursiveBlock generalizedTyparsForThisBinding 
        (PrelimValScheme1(id,iflex,ty,specialRepr,mut,inlineFlag,base,argAttribs,vis,compgen)) = 
    let (TIFlex(declaredTypars,_)) = iflex in
    let m = id.idRange in 
    let allDeclaredTypars = enclosingDeclaredTypars@declaredTypars in
    let allDeclaredTypars = chooseCanonicalDeclaredTyparsAfterInference  cenv.g denv allDeclaredTypars m in 
    (* Trim out anything not in type of the value (as opposed to the type of the r.h.s) *)
    (* This is important when a single declaration binds *)
    (* multiple generic items, where each item does not use all the polymorphism *)
    (* of the r.h.s. , e.g. let x,y = None,[] *)
    let computeRelevantTypars thruFlag = 
        let ftps = (free_in_type_lr thruFlag ty) in 
        let generalizedTypars = gen_intersect typar_ref_eq generalizedTyparsForThisBinding ftps in 
        (* Put declared typars first *)
        let generalizedTypars = placeTyparsInDeclarationOrder allDeclaredTypars generalizedTypars  id.idRange in 
        generalizedTypars in

    let generalizedTypars = computeRelevantTypars false in 

    (* Check stability of existence and ordering of type parameters under erasure of type abbreviation *)
    let generalizedTyparsLookingThroughTypeAbbreviations = computeRelevantTypars true in 
    if not (length generalizedTypars = length generalizedTyparsLookingThroughTypeAbbreviations && List.for_all2 typar_ref_eq generalizedTypars generalizedTyparsLookingThroughTypeAbbreviations) then
        warning(Error("the type parameters inferred for this value are not stable under the erasure of type abbreviations. This is due to the use of type abbreviations which drop or reorder type parameters, e.g. \n\ttype taggedInt<'a> = int   or\n\ttype swap<'a,'b> = 'b * 'a.\nConsider declaring the type parameters for this value explicitly, e.g.\n\tlet f<'a,'b> ((x,y) : swap<'b,'a>) : swap<'a,'b> = (y,x)",m));
        
    (* if declKind && (nonNil (gen_subtract ftps generalizedTypars)) then 
      warning(Error("A type annotation is recommended for this binding",m)); *)

    (* Some recursive bindings result in free type variables, e.g. *)
    (*    let rec f (x:'a) = ()  *)
    (*    and g() = (\y. f y); () *)
    (* What is the type of y? Type inference equates it to 'a. *)
    (* But "g" is not polymorphic in 'a. Hence we get a free choice of "'a" *)
    (* in the scope of "g". Thus at each individual recursive binding we record all *)
    (* type variables for which we have a free choice, which is precisely the difference *)
    (* between the union of all sets of generalized type variables and the set generalized *)
    (* at each particular binding. *)
    (* *)
    (* We record an expression node that indicates that a free choice can be made *)
    (* for these. This expression node effectively binds the type variables. *)
    let freeChoiceTypars = gen_subtract typar_ref_eq generalizedTyparsForRecursiveBlock generalizedTypars in
    (* printf "generalizeVal: %s generalizedTypars=%s epts=%s\n" id.idText (typarsL generalizedTypars |> showL) (typarsL freeChoiceTypars |> showL); *)

    (* if nonNil freeChoiceTypars then dprintf6 "#freeChoiceTypars = %d, #generalizedTypars=%d, #generalizedTyparsForRecursiveBlock=%d, #declaredTypars=%d @ %a\n" (length freeChoiceTypars) (length generalizedTypars) (length generalizedTyparsForRecursiveBlock) (length declaredTypars) output_range m; *)

    let hasDeclaredTypars = nonNil(declaredTypars) in
    (* This is just about the only place we form a TypeScheme *)
    let tyScheme = TypeScheme(generalizedTypars, freeChoiceTypars, ty) in 
    PrelimValScheme2(id,tyScheme,specialRepr,mut,inlineFlag,base,argAttribs,vis,compgen,hasDeclaredTypars)

let generalizeVals cenv denv enclosingDeclaredTypars generalizedTyparsForRecursiveBlock generalizedTypars types = 
    Namemap.map (generalizeVal cenv denv enclosingDeclaredTypars generalizedTyparsForRecursiveBlock generalizedTypars) types

let dontGeneralizeVals types = 
    let dontGeneralizeVal (PrelimValScheme1(id,_,ty,specialRepr,mut,inlineFlag,base,argAttribs,vis,compgen)) = 
        PrelimValScheme2(id, nonGenericTypeScheme(ty), specialRepr,mut,inlineFlag,base,argAttribs,vis,compgen,false) in
    Namemap.map dontGeneralizeVal types

let inferGenericArityFromTyScheme (TypeScheme(generalizedTypars,_,_)) partialValArity =
    translatePartialArity generalizedTypars partialValArity

let computeIsTyFunc((id:ident),hasDeclaredTypars,arityInfo) = 
    hasDeclaredTypars && 
    (match arityInfo with 
     | None -> error(Error("Explicit type parameters may only be used on module or member bindings",id.idRange)) 
     | Some info -> TopValData.numCurriedArgs(info) = 0) 

let useKnownArity (PrelimValScheme2(id,typeScheme,specialRepr,mut,inlineFlag,base,_,vis,compgen,hasDeclaredTypars)) partialValArity = 
    let valArity = Option.map (inferGenericArityFromTyScheme typeScheme) partialValArity in 
    let isTyFunc = computeIsTyFunc(id,hasDeclaredTypars,valArity) in 
    ValScheme(id,typeScheme,valArity,specialRepr,mut,inlineFlag,base,vis,compgen,false,isTyFunc)
    
let dontInferArity valscheme2 = 
    useKnownArity valscheme2 None

let dontInferArities tyschemes = 
    Namemap.map dontInferArity tyschemes

let inferPartialArity declKind rhs (PrelimValScheme2(id,typeScheme,specialRepr,mut,inlineFlag,base,(argAttribs,retAttribs),vis,compgen,hasDeclaredTypars)) = 
    let (TopValInfo (_,argsi,reti)) = 
        if mut then TopValData.emptyTopValData 
        else infer_arity_of_expr (generalizedTypeForTypeScheme typeScheme) argAttribs retAttribs rhs in 

    let valArity = 
        if (must_have_arity declKind) 
        then Some (inferGenericArityFromTyScheme typeScheme (PartialArityInfo (argsi,reti))) 
        else None in    

    let isTyFunc = computeIsTyFunc(id,hasDeclaredTypars,valArity) in 
    ValScheme(id,typeScheme,valArity,specialRepr,mut,inlineFlag,base,vis,compgen,false,isTyFunc)
        
let inferPartialArityIfNotGiven declKind compiledArityOpt rhs valscheme =
    match compiledArityOpt with 
    | None -> inferPartialArity declKind rhs valscheme
    | Some a -> useKnownArity valscheme compiledArityOpt

let inferPartialArities declKind rhs tyschemes =  
    Namemap.map (inferPartialArity declKind rhs) tyschemes

let mkSimpleVals cenv env m names =
    let tyschemes  = dontGeneralizeVals names in 
    let valschemes = dontInferArities tyschemes in 
    let values     = mkAndPublishVals cenv env (ParentNone,false,ExpressionBinding,ValNotInRecScope,valschemes,[],emptyXMLDoc,None) in
    let vspecMap   = Namemap.map fst values in
    values,vspecMap
    
let mkAndPublishSimpleVals cenv env m names =
    let values,vspecMap   = mkSimpleVals cenv env m names in
    let envinner   = addLocalValMap m vspecMap env in 
    envinner,values,vspecMap

(*-------------------------------------------------------------------------
!* Helpers to freshen existing types and values, i.e. when a reference
 * to C<_> occurs then generate C<?ty> for a fresh type inference variable ?ty.
 *------------------------------------------------------------------------- *)


let  computeExtraTyparsFeasible(memFlagsOpt,declaredTypars,m) = 
    (* Properties and Constructors may only generalize the variables associated with the containing class (retrieved from the 'this' pointer) *)
    (* Also check they don't declare explicit typars. *)
    match memFlagsOpt with 
    | None -> true
    | Some(memFlags) -> 
        match memFlags.memFlagsKind with 
        (* can't infer extra polymorphism for properties *)
        | MemberKindPropertyGet 
        | MemberKindPropertySet  -> 
             if nonNil(declaredTypars) then 
                 errorR(Error("A property may not have explicit type parameters. Consider using a method instead",m));
             false
        (* can't infer extra polymorphism for class constructors *)
        | MemberKindClassConstructor ->  
             false
        (* can't infer extra polymorphism for constructors *)
        | MemberKindConstructor -> 
             if nonNil(declaredTypars) then 
                 errorR(Error("A constructor may not have explicit type parameters. Consider using a static construction method instead",m));
             false 
        | _ -> 
             (* feasible to infer extra polymorphism *)
             true                     

let computeCanInferTypars(declKind,canInferTypars,memFlagsOpt,declaredTypars,m) =  
        computeExtraTyparsFeasible(memFlagsOpt,declaredTypars,m) &&
        canInferTypars &&
        (!generalize_inner_polymorphism || always_generalize declKind) 


(*-------------------------------------------------------------------------
!* Helpers to freshen existing types and values, i.e. when a reference
 * to C<_> occurs then generate C<?ty> for a fresh type inference variable ?ty.
 *------------------------------------------------------------------------- *)
    
(* REVIEW: we should never need to pass in rigid=false here *)
(* REVIEW: this is related to how we typecheck recursive bindings in the presence *)
(* REVIEW: of explicit type parameters, including type parameters arising from classes. *)
(* REVIEW: This is under revision in any case as we will have to eventually (soon) *)
(* REVIEW: expand the power of the recursive inference to include polymorphic *)
(* REVIEW: recursion. *)
let freshenTcref cenv m rigid tcref = 
    let tpsorig = typars_of_tcref tcref in 
    let tps = copy_typars tpsorig in 
    if rigid <> TyparRigid then 
      tps |> iter (fun tp -> set_rigid_of_tpdata (data_of_typar tp) rigid);  
        
    let renaming,tinst = fixup_new_typars cenv.amap m [] [] tpsorig tps in 
    TType_app(tcref,map mk_typar_ty tpsorig),tps,renaming,TType_app(tcref,tinst)
    
let freshenPossibleForallTy cenv m rigid ty = 
    let tpsorig,tau =  try_dest_forall_typ ty in 
    if isNil tpsorig then [],[],tau
    else
        let tpsorig = normalizeDeclaredtyparsForEquiRecursiveInference tpsorig in  
        let tps,renaming,tinst = copy_and_fixup_typars cenv m rigid tpsorig in 
        tps,tinst,inst_type renaming tau

let info_of_tcref cenv m env tcref = 
    let tps,renaming,tinst = new_tinst cenv m (typars_of_tcref tcref) in 
    tps,renaming,tinst,TType_app (tcref,tinst)


(** Given a abstract method, which may be a generic method, freshen the type in preparation to apply it as a constraint to the method that implements the abstract slot *)
let freshenAbstractSlot g amap m synTyparDecls absMethInfo = 

    (* Work out if an explicit instantiation has been given. If so then the explicit type *)
    (* parameters will be made rigid and checked for generalization. If not then auto-generalize *)
    (* by making the copy of the type parameters on the virtual being overriden rigid. *)

    let typarsFromAbsSlotAreRigid = 
        
        match synTyparDecls with 
        | ValTyparDecls(synTypars,infer,_) -> 
            if nonNil synTypars && infer then errorR(Error("You must explicitly declare either all or no type parameters when overriding a generic abstract method",m));
            isNil synTypars in 
            
    let argtys,rty,fmtps,_ = sig_of_minfo g amap m absMethInfo in 
    (* dprintf2 "nm = %s, #fmtps = %d\n" (name_of_minfo absMethInfo) (length fmtps); *) 
    
    (* If the virual method is a generic method then copy its type parameters *)
    let typarsFromAbsSlot,typarInstFromAbsSlot,_ = 
        let ttps = (formal_tctps_of_minfo g absMethInfo) in 
        let ttinst = tinst_of_stripped_typ (typ_of_minfo absMethInfo) in 
        let rigid = (if typarsFromAbsSlotAreRigid then TyparRigid else TyparWarnIfNotRigid) in 
        Csolve.freshen_and_fixup_typars amap m rigid ttps ttinst fmtps in

   (*  dprintf1 "#typarsFromAbsSlot = %d\n" (length typarsFromAbsSlot); *)

    (* Work out the required type of the member *)
    let argTysFromAbsSlot = map (inst_type typarInstFromAbsSlot) argtys in 
    let retTyFromAbsSlot = inst_type typarInstFromAbsSlot rty  in 
    typarsFromAbsSlotAreRigid,typarsFromAbsSlot,argTysFromAbsSlot, retTyFromAbsSlot

(*-------------------------------------------------------------------------
!* Helpers related to accessibility
 *------------------------------------------------------------------------- *)

let accessRightsOfEnv env = AccessibleFrom (curr_access_cpath env,env.eFamilyType)


let check_rfield_mutation g m denv rfinfo ftinst = 
    if not (rfield_of_rfinfo rfinfo).rfield_mutable then error (FieldNotMutable(denv,rfref_of_rfinfo rfinfo,m));
    if nonNil(ftinst) then error (Error("Fields which are type functions may not be mutated",m))


(*-------------------------------------------------------------------------
!* Helpers to typecheck expressions and patterns
 *------------------------------------------------------------------------- *)

let gen_build_fldmap cenv env ty flds m = 
    if isNil flds then invalid_arg "gen_build_fldmap";
    
    let frefSets = 
        flds |> map (fun (fld,fldExpr) -> 
            let frefSet = tc_field_id cenv.ginstf cenv.g cenv.amap env.eNameResEnv ty fld in
            frefSet, fldExpr) in 
    let relevantTypeSets = frefSets |> map (fst >> chooseList (fun (tcref,isRecdField) -> if isRecdField then Some(tcref_of_rfref tcref) else None)) in
    
    let tcref = 
        match fold_left (gen_intersect cenv.g.tcref_eq) (List.hd relevantTypeSets) (List.tl relevantTypeSets) with
        | [tcref] -> tcref
        | _ -> 
            (* OK, there isn't a unique type dictated by the intersection for the field refs. *)
            (* We're going to get an error of some kind below. *)
            (* Just choose one field ref and let the error come later *)
            let fref1 = fst (List.hd (fst (List.hd frefSets))) in 
            tcref_of_rfref fref1 in
    
    let fldsmap,rfldsList = 
        List.fold_left 
            (fun (fs,rfldsList) (frefs,fldExpr) -> 
                match frefs |> List.filter (fun (fref2,isRecdField) -> cenv.g.tcref_eq tcref (tcref_of_rfref fref2)) with
                | [(fref2,isRecdField)] -> 
                    if not isRecdField then errorR(DeprecatedClassFieldInference(m));
                    rfref_accessible_check m (accessRightsOfEnv env) fref2;
                    fs_attrib_check cenv.g (pattribs_of_rfref fref2) m |> commitOperationResult;        
                    if  Map.mem (name_of_rfref fref2) fs then 
                        errorR (Error("The field "^name_of_rfref fref2^" appears twice in this record expression or pattern",m));
                    if  not (cenv.g.tcref_eq tcref (tcref_of_rfref fref2)) then 
                        let fref1 = fst (List.hd (fst (List.hd frefSets))) in 
                        errorR (FieldsFromDifferentTypes((denv_of_tenv env),fref1,fref2,m));
                        (fs,rfldsList)
                    else (Map.add (name_of_rfref fref2) fldExpr fs,
                          (name_of_rfref fref2,fldExpr)::rfldsList)
                | _ -> error(Error("This record contains fields from inconsistent types",m)))
            (Map.empty,[])
            frefSets in 
    tcref,fldsmap,List.rev rfldsList

let rec gen_constr_unify (mk_constrf,mk_exnconstrf) m cenv env ty item =
    let family = accessRightsOfEnv env in 
    match item with 
    | Item_ecref ecref -> 
        fs_attrib_check cenv.g (attribs_of_tcref ecref) m |> commitOperationResult;
        unifyE cenv env m ty cenv.g.exn_ty;
        tcref_accessible_check m family ecref;
        let mkf = mk_exnconstrf(ecref) in
        mkf,typs_of_ecref_rfields ecref
    | Item_ucref ucref ->   
        fs_attrib_check cenv.g (attribs_of_ucref ucref) m |> commitOperationResult;
        ucref_accessible_check m family ucref;
        let _,inst,tinst,_ = info_of_tcref cenv m env (tcref_of_ucref ucref) in 
        let gtyp2 = rty_of_uctyp ucref tinst in 
        unifyE cenv env m ty gtyp2;
        let mkf = mk_constrf(ucref,tinst) in 
        mkf,typs_of_ucref_rfields inst ucref 
    | _ -> invalid_arg "gen_constr_unify"

let expr_constr_unify m cenv env ty c = 
  gen_constr_unify ((fun (a,b) args -> mk_constr(a,b,args,m)),
                    (fun a args -> mk_exnconstr (a,args,m))) m cenv env ty c
      
let pat_constr_unify m cenv env ty c = 
  gen_constr_unify ((fun (a,b) args -> TPat_unionconstr(a,b,args,m)),
                    (fun a args -> TPat_exnconstr(a,args,m))) m cenv env ty c

let gen_constr_check env nargtys nargs m =
  if nargs <> nargtys then error (UnionConstrWrongArguments((denv_of_tenv env),nargtys,nargs,m))

let tc_constr_field cenv env ty1 m c n funcs =
    let mkf,argtys = 
      match tc_pat_lid AllIdsOK None cenv.ginstf cenv.g cenv.amap m env.eNameResEnv defaultTypeNameResInfo c with
      | (Item_ucref _ | Item_ecref _) as item ->
        gen_constr_unify funcs m cenv env ty1 item
      | _ -> error(Error("unknown data constructor",m)) in 
    if n >= length argtys then 
      error (Constr_field_wrong_arguments((denv_of_tenv env),length argtys,n,m));
    let ty2 = List.nth argtys n in 
    mkf,ty2

(*-------------------------------------------------------------------------
!* Environment of explicit type parameters, e.g. 'a in "(x : 'a)"
 *------------------------------------------------------------------------- *)

type syntacticUnscopedTyparEnv = UnscopedTyparEnv of local_typar_ref namemap

let emptyTpenv : syntacticUnscopedTyparEnv = UnscopedTyparEnv(Map.empty )

let addUnscopedTypar n p (UnscopedTyparEnv tab) = UnscopedTyparEnv (Map.add n p tab)

let tryFindUnscopedTypar n (UnscopedTyparEnv tab) = Map.tryfind n tab

let hideUnscopedTypars typars (UnscopedTyparEnv tab) = 
    UnscopedTyparEnv (List.fold_left (fun acc tp -> Map.remove (name_of_typar tp) acc) tab typars)

(*-------------------------------------------------------------------------
!* Helpers for generalizing type variables
 *------------------------------------------------------------------------- *)

let computeUngeneralizableTypars env = 
    let acc_loctypars_in_free_item acc (f,closed)  =
       if !closed then acc else
       let ftyvs = (f()).free_loctypars in 
       if Zset.is_empty ftyvs then closed := true;
       union_free_loctypars ftyvs acc in

    List.fold_left acc_loctypars_in_free_item empty_free_loctypars env.eUngeneralizableItems 

let computeUnabstractableTycons env = 
    let acc_loctycons_in_free_item acc (f,closed)  =
       let fvs = (f()).free_loctycons in 
       union_free_loctycons fvs acc in

    List.fold_left acc_loctycons_in_free_item empty_free_loctycons env.eUngeneralizableItems 

let rec isGeneralizableValue g t = 
    match t with 
    | TExpr_lambda _ | TExpr_tlambda _ | TExpr_const _ | TExpr_val _ -> true
    | TExpr_op(op,_,args,_) ->
        begin match op with 
        | TOp_tuple  -> true
        | TOp_uconstr uc -> not (ucref_alloc_observable uc)
        | TOp_recd(ctorInfo,tcref) -> 
            begin match ctorInfo with 
            | RecdExpr ->  not (tcref_alloc_observable tcref)
            | RecdExprIsObjInit -> false
            end
        | TOp_array ->  isNil args
        | TOp_exnconstr ec -> not (ecref_alloc_observable ec)
        | _ -> false
        end
        && for_all (isGeneralizableValue g) args

    | TExpr_letrec(binds,body,_,_)  ->
        for_all (rhs_of_bind >> isGeneralizableValue g) binds &&
        isGeneralizableValue g body
    | TExpr_let(bind,body,_,_) -> 
        (rhs_of_bind >> isGeneralizableValue g) bind &&
        isGeneralizableValue g body


    (* Applications of type functions are _not_ normally generalizable unless explicitly marked so *)
    | TExpr_app(TExpr_val (vref,_,_),_,_,[],_) 
         when is_tyfunc_of_vref vref -> fsthing_has_attrib g g.attrib_GeneralizableValueAttribute (attribs_of_vref vref)
         
    | TExpr_app(e1,_,_,[],_) -> isGeneralizableValue g e1
    | TExpr_tchoose(_,b,_) -> isGeneralizableValue g b
(*
    | TExpr_quote _ -> true
*)
    | TExpr_obj (_,ty,_,_,_,_,_,_) -> is_interface_typ ty || is_delegate_typ ty
    | _ -> false  

(* Recursively knock out typars we can't generalize. *)
(* For non-generalized type variables be careful to iteratively knock out *)
(* both the typars and any typars free in the constraints of the typars*)
(* into the set that are considered free in the environment. *)
type generalizeConstrainedTyparOptions = 
    | CanGeneralizeConstrainedTypars 
    | DoNotGeneralizeConstrainedTypars

let canGeneralizeConstrainedTyparsForDecl declKind = 
    if can_generalize_constrained_typar declKind 
    then CanGeneralizeConstrainedTypars 
    else DoNotGeneralizeConstrainedTypars
    
let rec trimUngeneralizableTypars genConstrainedTyparFlag inlineFlag generalizedTypars freeInEnv = 
    (* Do not generalize type variables with a static requirement unless function is marked 'inline' *)
    let generalizedTypars,ungeneralizableTypars1 =  
        if inlineFlag = PseudoValue then generalizedTypars,[]
        else partition (fun tp -> static_req_of_tpref tp = NoStaticReq) generalizedTypars in 

    (* Do not generalize type variables which would escape their scope *)
    (* because they are free in the environment *)
    let generalizedTypars,ungeneralizableTypars2 = 
        partition (fun x -> not (Zset.mem x freeInEnv)) generalizedTypars in

    (* Some situations, e.g. implicit class constructions that represent functions as fields, *)
    (* do not allow generalisation over constrained typars. (since they can not be represented as fields *)
    let generalizedTypars,ungeneralizableTypars3 = 
        partition (fun tp -> genConstrainedTyparFlag = CanGeneralizeConstrainedTypars || 
                             isNil (constraints_of_typar tp)) generalizedTypars in

    if isNil ungeneralizableTypars1 && isNil ungeneralizableTypars2 && isNil ungeneralizableTypars3 then
        generalizedTypars, freeInEnv
    else 
        let freeInEnv = (union_free_loctypars (acc_free_tprefs ungeneralizableTypars1 (acc_free_tprefs ungeneralizableTypars2 (acc_free_tprefs ungeneralizableTypars3 empty_free_tyvars))).free_loctypars freeInEnv) in 
        trimUngeneralizableTypars genConstrainedTyparFlag inlineFlag generalizedTypars freeInEnv

let verboseA = false

let canonicalizePartialInferenceProblem (cenv,denv,m) tps =
    (* Canonicalize constraints prior to generalization *)
    let csenv = (mk_csenv cenv.css m denv) in 
    tryD (fun () -> Csolve.canonicalizeRelevantMemberConstraints csenv NoTrace tps)
         (fun res -> errorD (ErrorFromAddingConstraint(denv,res,m))) 
    |> raiseOperationResult

let computeGeneralizedTypars(cenv,denv,m,immut,freeInEnv,canInferTypars,genConstrainedTyparFlag,inlineFlag,exprOpt,allDeclaredTypars,maxInferredTypars) =

    if verboseA then dprintf3 "%a: computeGeneralizedTypars (-2), freeInEnv = %s\n" output_range m (String.concat "," (List.map name_of_typar (Zset.elements freeInEnv)));
    if verboseA then dprintf3 "%a: computeGeneralizedTypars (-1), maxInferredTypars = %s\n" output_range m (String.concat "," (maxInferredTypars  |> List.map name_of_typar));
    let allDeclaredTypars = chooseCanonicalDeclaredTyparsAfterInference cenv.g  denv allDeclaredTypars m in 

    if verboseA then dprintf1 "computeGeneralizedTypars (0), allDeclaredTypars = %s\n" (String.concat "," (allDeclaredTypars  |> List.map name_of_typar));

    let typarsToAttemptToGeneralize = 
        if canInferTypars && immut && (match exprOpt with None -> true | Some e -> isGeneralizableValue cenv.g e) 
        then (gen_union_favour_left typar_ref_eq  allDeclaredTypars maxInferredTypars)
        else allDeclaredTypars in 

    if verboseA then dprintf1 "computeGeneralizedTypars (1), typarsToAttemptToGeneralize = %s\n" (String.concat "," (typarsToAttemptToGeneralize  |> List.map name_of_typar));

    let generalizedTypars,freeInEnv = 
        trimUngeneralizableTypars genConstrainedTyparFlag inlineFlag typarsToAttemptToGeneralize freeInEnv in 

    if verboseA then dprintf1 "computeGeneralizedTypars (1), generalizedTypars = %s\n" (String.concat "," (generalizedTypars  |> List.map name_of_typar));
    
    allDeclaredTypars |> iter (fun tp -> 
          if Zset.mem_of freeInEnv tp then
            let ty =  mk_typar_ty tp in
            error(Error("This code is not sufficiently generic. The type variable "^NicePrint.pretty_string_of_typ denv ty^" could not be generalized because it would escape its scope",m)));

    if verboseA then dprintf2 "computeGeneralizedTypars: %s, generalizedTypars = %s\n" (string_of_range m) (Layout.showL(typarsL generalizedTypars));
    
    (* Generalization turns inference type variables into rigid, quantified type variables. *)
    generalizedTypars |> List.iter (setTyparRigid cenv.g denv m);
    
    let csenv = (mk_csenv cenv.css m denv) in 
    eliminateConstraintsForGeneralizedTypars csenv NoTrace generalizedTypars;
    
    generalizedTypars



let computeInlineFlag memFlagsOption pseudo mut = 
    (* Mutable values may never be inlined *)
    (* Constructors may never be inlined *)
    (* Calls to virtual/abstract slots may never be inlined *)
    if mut || 
       (match memFlagsOption with 
        | None -> false
        | Some x -> (x.memFlagsKind = MemberKindConstructor) || x.memFlagsAbstract) 
    then NeverInline 
    else if pseudo then PseudoValue 
    else OptionalInline


(*-------------------------------------------------------------------------
!* Binding normalization.
 *
 * Determine what sort of beast is being bound (normal value, instance
 * member, normal function, static member etc.) and make some
 * name-resolution-sensitive adjustments to the syntax tree.
 *
 * One part of this "normalization" ensures: 
 *        "let Pat_lid(f) = e" when f not a datatype constructor --> let Pat_var(f) = e" 
 *        "let Pat_lid(f) pat = e" when f not a datatype constructor --> let Pat_var(f) = \pat. e" 
 *        "let (Pat_lid(f) : ty) = e" when f not a datatype constructor --> let (Pat_var(f) : ty) = e" 
 *        "let (Pat_lid(f) : ty) pat = e" when f not a datatype constructor --> let (Pat_var(f) : ty) = \pat. e" 
 * 
 * This is because the first lambda in a function definition "let F x = e" 
 * now looks like a constructor application, i.e. let (F x) = e ... 
 *  also let A.F x = e ... 
 *  also let f x = e ... 
 *
 * The other parts turn property definitions into method definitions.
 *------------------------------------------------------------------------- *)
 
let push_one_pat_rhs isMember p (BindingExpr(spatsL,rtyOpt,e)) = 
    let spats,e = push_one_pat isMember p e in 
    BindingExpr(spats::spatsL, rtyOpt,e)

type normalizedBindingPatternInfo = 
    NormalizedBindingPat of synpat * bindingExpr * memberInfo option * valTyparDecls (* pat,rhsExpr,memberInfo,typars *)

(* Note: Binding kind has disappeared from 'Binding', valTyparDecls has been added *)
type normBinding = 
  | NormBinding of 
      access option *
      bool *  (* pesudo/mustinline value? *)
      bool *  (* mutable *)
      attributes * 
      xmlDoc *
      valTyparDecls * 
      memberInfo option * 
      synpat * 
      bindingExpr *
      range 


type isObjExprBinding = 
    | ObjExprBinding 
    | ValOrMemberBinding

(* Push a bunch of pats at once. They may contain patterns, e.g. let f (A x) (B y) = ... *)
(* In this case the sematnics is let f a b = let A x = a in let B y = b *)
let push_many_pats_rhs isMember ps (BindingExpr(spatsL,rtyOpt,e)) = 
    let spatsL2,e = push_many_pats isMember ps e in 
    BindingExpr(spatsL2@spatsL, rtyOpt,e)


let mk_normalized_static_or_val_binding isObjExprBinding id vis typars args rhsExpr memInfo : normalizedBindingPatternInfo = 
    NormalizedBindingPat(mksyn_pat_var vis id, push_many_pats_rhs ((isObjExprBinding = ObjExprBinding) or isSome memInfo) args rhsExpr,memInfo,typars)

let mk_normalized_instance_member_binding thisn memberName vis m typars args rhsExpr memInfo = 
    NormalizedBindingPat(Pat_instance_member(thisn, memberName,vis,m), push_many_pats_rhs true args rhsExpr,Some memInfo,typars)

let normStaticMemberBinding ((memFlags,topValSynData,thisIdOpt) as memberInfo) id vis typars args m rhsExpr = 
    let narity = SynArgInfo.numCurriedArgs topValSynData in 
    if memFlags.memFlagsInstance then 
        (* instance method without adhoc "this" argument *)
        error(Error("This instance member needs a parameter to represent the object being invoked. Make the member static or use the notation 'member x.Member(args) = ...'",m));
    match args, memFlags.memFlagsKind  with 
    | _,MemberKindPropertyGetSet    -> error(Error("Unexpected source-level property specification in syntax tree",m))
    | [],MemberKindClassConstructor -> error(Error("A static initializer requires an argument",m))
    | [],MemberKindConstructor     -> error(Error("An object constructor requires an argument",m))
    | [_],MemberKindClassConstructor  
    | [_],MemberKindConstructor  -> mk_normalized_static_or_val_binding ValOrMemberBinding id vis typars args rhsExpr (Some(memberInfo))
    (* Instance property declared using 'static member P = expr: transformed to a method taking a "unit" argument *)
    (* static property: these transformed into methods taking one "unit" argument *)
    | [],MemberKindMember when narity=0 -> 
        let memberInfo = ({memFlags with memFlagsKind = MemberKindPropertyGet},SynArgInfo.incorporateEmptyTupledArg topValSynData,thisIdOpt) in 
        NormalizedBindingPat(mksyn_pat_var vis id,
                             push_one_pat_rhs true (Pat_const(Const_unit,m)) rhsExpr,
                             Some(memberInfo),
                             typars)
    | _ -> mk_normalized_static_or_val_binding ValOrMemberBinding id vis typars args rhsExpr (Some(memberInfo))

let normInstanceMemberBinding ((memFlags,topValSynData,thisIdOpt) as memberInfo) thisn memberName vis typars args m rhsExpr = 
    let narity = SynArgInfo.numCurriedArgs topValSynData in     
    if not memFlags.memFlagsInstance then 
      (* static method with adhoc "this" argument *)
      error(Error("This static member should not have a 'this' parameter. Consider using the notation 'member Member(args) = ...'",m));
    match args, memFlags.memFlagsKind  with 
    | _,MemberKindClassConstructor  -> error(Error("An explicit static initializer should use the syntax 'static new(args) = expr'",m))
    | _,MemberKindConstructor  -> error(Error("An explicit object constructor should use the syntax 'new(args) = expr'",m))
    | _,MemberKindPropertyGetSet  -> error(Error("Unexpected source-level property specification",m))
    | [],(MemberKindPropertyGet | MemberKindPropertySet)  -> mk_normalized_instance_member_binding thisn memberName vis m typars [] rhsExpr memberInfo 

    (* Instance property declared using 'x.Member': transformed to methods taking a "this" and a "unit" argument *)
    (* We push across the 'this' arg in mk_rec_binds *)
    | [],MemberKindMember when narity=1 -> 
        NormalizedBindingPat
            (Pat_instance_member(thisn, memberName,vis,m), 
             push_one_pat_rhs true (Pat_const(Const_unit,m)) rhsExpr,
             (* Update the member info to record that this is a MemberKindPropertyGet *)
             Some({memFlags with memFlagsKind = MemberKindPropertyGet},SynArgInfo.incorporateEmptyTupledArg topValSynData,thisIdOpt),
             typars)

    | _ -> mk_normalized_instance_member_binding thisn memberName vis m typars args rhsExpr memberInfo

let normBindingPat isObjExprBinding ginstf g amap (env:tcEnv) memberInfo pat rhsExpr =
    let rec normPattern pat = 
        (* One major problem with versions of F# prior to 1.9.x is that data constructors easily 'pollute' the namespace *)
        (* of available items, to the point that you can't even define a function with the same name as an existing data constructor. *)
        (* OCaml doesn't have this flaw because it doesn't allow uppercase function names (which is in itself a flaw of a different kind) *)
        (* To give ourselves future wiggle room here I've added a warning that people parenthesize patterns at 'let' bindings that involve *)
        (* data cosntructor tags, thus         *)
        (*      let C x = ...                 *)
        (* will give a warning if C is a data tag in scope and users will have to change to  *)
        (*      let (C x) = ...               *)
        (* if it's truly a pattern. *)
        (* Note that (C x) will come through here as a Pat_paren. *)
        match pat with 
        | Pat_lid (lid,tyargs,args,vis,m) ->
            let typars = (match tyargs with None -> inferredTyparDecls | Some typars -> typars) in 
            begin match memberInfo with 
            | None ->                
                (* Thus only permit the interpretation of the identifier as a pattern data constructor if all of the following conditions hold *)
                let ambiguousDataTagHasArgs = 
                    (if isNone tyargs then Some(nonNil args) else None) in
                    
                begin match tc_pat_lid AllIdsOK ambiguousDataTagHasArgs ginstf g amap m env.eNameResEnv defaultTypeNameResInfo lid with
                | Item_newdef id                    -> mk_normalized_static_or_val_binding isObjExprBinding id vis typars args rhsExpr None
                | res -> 
                    begin match res,lid with 
                    | (Item_ucref _ | Item_ecref _ | Item_apelem _),[id] when id.idText <> opname_Cons && id.idText <> opname_Nil -> 
                          warning(OCamlCompatibility("The identifier '"^id.idText^"' is a pattern constructor, perhaps brought into scope by an 'open' declaration. Please put parentheses around this pattern as a future revision of the language will require this",m))
                    | _ -> ()
                    end;
                    NormalizedBindingPat(pat,rhsExpr,memberInfo,typars)
                end
                
            | Some memInfo ->                
                begin match lid with 
                (* x.Member in member binding patterns. *)
                | [thisn;memberName] -> normInstanceMemberBinding memInfo thisn memberName vis typars args m rhsExpr 
                | [memberName]       -> normStaticMemberBinding memInfo memberName vis typars args m rhsExpr 
                | _                  -> NormalizedBindingPat(pat,rhsExpr,memberInfo,typars)
                end
            end
        (* Object constructors are normalized in tc_letrec *)
        (* Here we are normalizing member definitions with simple (not long) ids, *)
        (* e.g. "static member x = 3" and "member x = 3" (instance with missing "this." comes through here. It is trapped and generates a warning) *)
        | Pat_as (Pat_wild _, id, false, vis, m) 
            when 
               (match memberInfo with 
                | None -> false 
                | Some(memFlags,_,_) -> 
                     not (memFlags.memFlagsKind = MemberKindConstructor) &&
                     not (memFlags.memFlagsKind = MemberKindClassConstructor)) ->            
            normStaticMemberBinding (the memberInfo) id vis inferredTyparDecls [] m rhsExpr 
        | Pat_typed(pat',x,y) ->             
            let (NormalizedBindingPat(pat'',e'',memberInfo,typars)) = normPattern pat' in 
            NormalizedBindingPat(Pat_typed(pat'',x,y), e'',memberInfo,typars)
        | Pat_attrib(pat',x,y) ->             
            let (NormalizedBindingPat(pat'',e'',memberInfo,typars)) = normPattern pat' in 
            NormalizedBindingPat(Pat_attrib(pat'',x,y), e'',memberInfo,typars)
        | _ ->
            NormalizedBindingPat(pat,rhsExpr,memberInfo,inferredTyparDecls)  in 
    normPattern pat

let normBinding isObjExprBinding cenv (env:tcEnv) b = 
    let ginstf = cenv.ginstf  in
    let g = cenv.g  in
    let amap = cenv.amap in
    match b with 
    | Binding (vis,_,pseudo,mut,attrs,doc,memberInfo,p,rhsExpr,bindm) ->
        let (NormalizedBindingPat(pat',rhsExpr',memberInfo',typars')) = 
            normBindingPat isObjExprBinding ginstf g amap env memberInfo p rhsExpr in 
        (* Ildiag.dprintf3 "#typars = %d @ %a\n" (let (ValTyparDecls(a,_,_)) = typars' in length a) output_range bindm; *)
        NormBinding(vis,pseudo,mut,attrs,doc,typars',memberInfo',pat',rhsExpr',bindm)

(*-------------------------------------------------------------------------
 * Helpers to adjust the 'this' pointer before making a call.
 *------------------------------------------------------------------------- *)

type boxopts = 
    | CoerceThisBeforeCall of Tast.typ * Tast.typ 
    | DontCoerceThisBeforeCall

(* Compute whether we insert a 'coerce' on the 'this' pointer for an object model call *)
(* For example, when calling an interface method on a struct, or a method on a constrained *)
(* variable type. *)
let computeBoxOpts cenv m (objArgs,mtyp) =
    match objArgs with 
    | [h] when not (type_definitely_subsumes_type_no_coercion 0 cenv.g cenv.amap m mtyp (type_of_expr cenv.g h)) ->
        let hty = type_of_expr cenv.g h in 
        CoerceThisBeforeCall (hty, mtyp)
    | _ -> 
        DontCoerceThisBeforeCall

(* Adjust the 'this' pointer before making a call *)
(* Take the address of a struct, and coerce to an interface/base/constraint type if necessary *)
let takeObjAddr cenv minfo mut m objArgs f =
    let boxopts = computeBoxOpts cenv m (objArgs,typ_of_minfo minfo)  in
    let valu = minfo_is_struct minfo in 
    let wrap,objArgs = 
        match objArgs with
        | [h] -> 
            let hty = type_of_expr cenv.g h in 
            let wrap,h' = mk_expra_of_expr cenv.g valu mut h m  in 
            let h' = 
              match boxopts with 
              | CoerceThisBeforeCall(srcty,tgtty) -> mk_coerce(h',tgtty,m,srcty)
              | DontCoerceThisBeforeCall -> h' in 
            wrap,[h'] 
        | _ -> (fun x -> x), objArgs in
    let e,ety = f objArgs in 
    wrap e,ety

let freshenThisTy cenv m rigid tcref = 
    let tcrefObjTy,enclosingDeclaredTypars,renaming,objTy = freshenTcref cenv m rigid tcref in 
    (* Struct members have a byref 'this' type *)
    let thisTy = if is_struct_tcref tcref then mk_byref_typ cenv.g objTy else objTy in 
    tcrefObjTy,enclosingDeclaredTypars,renaming,objTy,thisTy

(*-------------------------------------------------------------------------
!* tc_val. "Use" a value, normally at a fresh type instance (unless optInst is
 * given). optInst is set when an explicit type instantiation is given, e.g. 
 *      Seq.empty<string>
 * In this case the vrefFlags inside optInst are just NormalValUse.
 *
 * optInst is is also set when building the final call for a reference to an
 * F# object model member, in which case the optInst is the type instantiation
 * inferred by member overload resolution, and vrefFlags indicate if the
 * member is being used in a special way, i.e. may be one of:
 *    | CtorValUsedAsSuperInit    "inherit Panel()"
 *    | CtorValUsedAsSelfInit     "new() = new OwnType(3)"
 *    | VSlotDirectCall           "base.OnClick(eventArgs)"
 *------------------------------------------------------------------------- *)
 

let tc_val cenv env vref optInst m =
    let v = deref_val vref in 
    let vrec = vrec_of_val v in 
    vref_accessible_check m (accessRightsOfEnv env) vref;
    fs_attrib_check cenv.g (attribs_of_val v) m |> commitOperationResult;
    let vty = (type_of_val v) in 
    (* byref types get dereferenced *)
    if is_byref_ty cenv.g vty then 
        mk_lval_get m vref ,dest_byref_ty cenv.g vty
    else 
      (* Literal values go to constants *)
      match literal_const_of_val v with 
      | Some c -> TExpr_const(c,m,vty),vty
      | None -> 
        (* References to 'this' in classes get checked *)
        if base_of_val v = CtorThisVal then 
            let exprForVal = expr_for_vref m vref in 
            if is_ctorPreConstruct env then 
                warning(SelfRefObjCtor(is_ctorImplicit env, m));

            let ty = (dest_refcell_ty cenv.g vty) in 
            mk_nonnull_poke cenv.g m (mk_refcell_get cenv.g m ty exprForVal), ty
        else 
          (* Instantiate the value *)
          let exprForVal,tinst,tau = 
             (* Have we got an explicit instantiation? *)
              match optInst with 
              | None -> 
                  if fsthing_has_attrib cenv.g cenv.g.attrib_RequiresExplicitTypeArgumentsAttribute (attribs_of_val v) then
                       errorR(Error("The generic function '"^display_name_of_val v^"' must be given explicit type argument(s)",m));
              
                  let exprForVal = expr_for_vref m vref in 
                  begin match vrec with 
                  | ValInRecScope(false) -> 
                    let _,tau =  try_dest_forall_typ vty in 
                    exprForVal,[],tau
                  | ValInRecScope(true) 
                  | ValNotInRecScope ->
                    let tps,tinst,tau = freshenPossibleForallTy cenv m TyparFlexible vty  in 
                    exprForVal,tinst,tau
                  end

             (* If we have got an explicit instantiation then use that *)
              | Some(vrefFlags,tinst) -> 
                  let exprForVal = TExpr_val (vref,vrefFlags,m) in 
                  let tau2 = 
                      match vrec with 
                      | ValInRecScope(false) -> 
                          let tpsorig,tau =  try_dest_forall_typ vty in 
                          (* dprintf2 "tc_val: #tpsorig = %d, #tps = %d\n" (length tpsorig) (length tinst); *)
                          if length tpsorig <> length tinst then error(Error(sprintf "This value expects %d type parameter(s) but is given %d" (length tpsorig) (length tinst),m));
                          let tau2 = inst_type (mk_typar_inst tpsorig tinst) tau in 
                          iter2
                            (fun tp ty -> 
                              try unifyE cenv env m (mk_typar_ty tp) ty
                              with _ -> error (Recursion((denv_of_tenv env),(id_of_val v),tau2,tau,m))) 
                            tpsorig 
                            tinst;
                          tau2  
                      | ValInRecScope(true) 
                      | ValNotInRecScope ->
                          let tps,_,tau2 = freshenPossibleForallTy cenv m TyparFlexible vty  in 
                          if length tps <> length tinst then error(Error(sprintf "This value expects %d type parameter(s) but is given %d" (length tps) (length tinst),m));
                          iter2 (fun tp ty -> unifyE cenv env m (mk_typar_ty tp) ty) tps tinst;
                          tau2 in 
                  exprForVal,tinst,tau2   in 
          let exprForVal = mk_tyapp m (exprForVal,vty) tinst in 
          let exprForVal =  recordUseOfRecValue cenv vrec vref exprForVal m in 
          exprForVal,tau

(*-------------------------------------------------------------------------
!* Checks and warnings coercions
 *------------------------------------------------------------------------- *)

(*  Checks, warnings and constraint assertions for downcasts *)
let tc_runtime_type_test cenv denv m tgty srcty =
    if type_definitely_subsumes_type_no_coercion 0 cenv.g cenv.amap m tgty srcty then 
      warning(TypeTestUnnecessary(m));

    if is_typar_ty srcty then 
      error(IndeterminateRuntimeCoercion(denv,srcty,tgty,m));

    if is_sealed_typ cenv.g srcty then 
      error(RuntimeCoercionSourceSealed(denv,srcty,m));

    if is_sealed_typ cenv.g tgty or 
       is_typar_ty tgty or 
       not (is_interface_typ srcty) then 
      type_must_subsume_type denv cenv.css m NoTrace srcty tgty

(*  Checks, warnings and constraint assertions for upcasts *)
let tc_static_upcast cenv denv m tgty srcty =
    if is_typar_ty tgty then 
       error(IndeterminateStaticCoercion(denv,srcty,tgty,m)); 

    if is_sealed_typ cenv.g tgty then 
      warning(CoercionTargetSealed(denv,tgty,m));

    if type_equiv cenv.g srcty tgty then 
      warning(UpcastUnnecessary(m)); 

    type_must_subsume_type denv cenv.css m NoTrace tgty srcty


(*-------------------------------------------------------------------------
!* Helpers dealing with building calls to object model members
 *------------------------------------------------------------------------- *)

(* Is this a 'base' call (in the sense of C#) *)
let isBaseCall objArgs = 
    match objArgs with 
    | [TExpr_val(v,_,m)] when base_of_vref v  = BaseVal -> true
    | _ -> false

(* Build an expression that calls a given method info. *)
(* This is called after overload resolution, and also to call other *)
(* methods such as 'setters' for properties. *)
(*   isProp: is it a property get? *)
(*   minst: the instantiation to apply for a generic method *)
(*   objArgs: the 'this' argument, if any *)
(*   args: the arguments, if any *)
let buildMethodCall cenv env mut m isProp minfo vFlags minst objArgs args =
    let direct = isBaseCall objArgs in 
    let vFlags = if (direct && vFlags = NormalValUse) then VSlotDirectCall else vFlags in

    let conditionalCallDefine = 
        minfo_bind_attrib cenv.g cenv.g.attrib_ConditionalAttribute minfo 
                     (function ([CustomElem_string (Some(msg)) ],_) -> Some(msg) | _ -> None) 
                     (function (Attrib(_,[ TExpr_const (TConst_string(msg),_,_) ],_)) -> Some(Bytes.unicode_bytes_as_string msg) | _ -> None) in

    match conditionalCallDefine with 
    | Some(d) when not (List.mem d cenv.conditionalDefines) -> 

        (* Methods marked with 'Conditional' must return 'unit' *)
        unifyE cenv env m cenv.g.unit_ty (ret_typ_of_minfo cenv.g cenv.amap m minfo minst);
        mk_unit cenv.g m, cenv.g.unit_ty

    | _ -> 

        takeObjAddr cenv minfo mut m objArgs (fun objArgs -> 
            let allArgs = (objArgs @ args) in 
            match minfo with 
            
            (* Build a call to a .NET method *)
            | ILMeth(ilminfo) -> 
                mk_il_minfo_call cenv.g cenv.amap m isProp ilminfo vFlags minst direct allArgs

            (* Build a call to an F# method *)
            | FSMeth(typ,vref) -> 

                (* Go see if this is a use of a recursive definition... Note we know the value instantiation *)
                (* we want to use so we pass that in in order not to create a new one. *)
                let vexp, vexpty = tc_val cenv env vref (Some(vFlags,tinst_of_stripped_typ typ @ minst)) m in

                if verbose then dprintf2 "--> Build Method Call to %s, typ = %s\n" (name_of_minfo minfo) (showL (typeL typ));
                if verbose then dprintf2 "--> Build Method Call to %s, vexpty = %s\n" (name_of_minfo minfo) (showL (typeL vexpty));

                mk_fs_minfo_app cenv.g m vref vexp vexpty allArgs

            (* Build a 'call' to a struct default constructor *)
            | DefaultStructCtor typ -> 
                mk_ilzero (m,typ), typ)



(* Build the 'test and dispose' part of a 'use' statement *)
let buildDisposeCleanup cenv env m  v = 
    let disv,dise = Tastops.mk_compgen_local m (cenv.niceNameGen.nngApply "$dispose_var" m) cenv.g.system_IDisposable_typ in 
    let disposeMethod = 
        match try_find_minfo cenv.g cenv.amap m "Dispose" cenv.g.system_IDisposable_typ with 
        | [x] ->  x 
        | _ -> error(InternalError("Couldn't find Dispose on IDisposable, or it was overloaded",m))  in
    let disposeE,_ = buildMethodCall cenv env PossiblyMutates   m false disposeMethod NormalValUse [] [dise] [] in
    mk_isinst_cond cenv.g m cenv.g.system_IDisposable_typ (v,expr_for_val (range_of_val v) v) disv disposeE (mk_unit cenv.g m) 

let buildIlFieldGet g amap m objExpr finfo = 
    let (ILFieldInfo(tinfo,fdef)) = finfo in
    let fref = fref_of_il_finfo finfo in 
    let isValueType = il_finfo_is_struct finfo in 
    let valu = if isValueType then AsValue else AsObject in 
    let tinst = tinst_of_il_tinfo tinfo in
    let fieldType = vtyp_of_il_finfo amap m  finfo in 

    let wrap,objExpr = mk_expra_of_expr g isValueType NeverMutates objExpr m  in
      (* The empty instantiation on the AbstractIL fspec is OK, since we make the correct fspec in Ilxgen.gen_asm *)
      (* This ensures we always get the type instantiation right when doing this from *)
      (* polymorphic code, after inlining etc. *) 
    let fspec = mk_fspec(fref,mk_named_typ valu (tref_of_il_tinfo tinfo) []) in 
    (* Add an I_nop if this is an initonly field to make sure we never recognize it as an lvalue. See mk_expra_of_expr. *)
    wrap (mk_asm (([ mk_normal_ldfld fspec ] @ (if initonly_of_il_finfo finfo then [ I_arith AI_nop ] else [])), tinst,[objExpr],[fieldType],m)) 

let buildIlFieldSet g m objExpr finfo argExpr = 
    let (ILFieldInfo(tinfo,fdef)) = finfo in
    let fref = fref_of_il_finfo finfo in 
    let isValueType = il_finfo_is_struct finfo in 
    let valu = if isValueType then AsValue else AsObject in 
    let tinst = tinst_of_il_tinfo tinfo in
      (* The empty instantiation on the AbstractIL fspec is OK, since we make the correct fspec in Ilxgen.gen_asm *)
      (* This ensures we always get the type instantiation right when doing this from *)
      (* polymorphic code, after inlining etc. *) 
    let fspec = mk_fspec(fref,mk_named_typ valu (tref_of_il_tinfo tinfo) []) in 
    if initonly_of_il_finfo finfo then error (Error ("this field is readonly",m));
    let wrap,objExpr = mk_expra_of_expr g isValueType DefinitelyMutates objExpr m  in
    (mk_asm ([ mk_normal_stfld fspec ], tinst,[objExpr; argExpr],[],m)) 
    
let buildRecdFieldSet g m denv objExpr rfinfo ftinst argExpr = 
    let tgty = (enclosing_vtyp_of_rfinfo rfinfo) in 
    let valu = is_struct_typ tgty in 
    let objExpr = if valu then objExpr else mk_coerce(objExpr,tgty,m,type_of_expr g objExpr) in
    check_rfield_mutation g m denv rfinfo ftinst;
    mk_recd_field_set g (objExpr,rfref_of_rfinfo rfinfo,tinst_of_rfinfo rfinfo,argExpr,m) 
    
    
(*-------------------------------------------------------------------------
!* Helpers dealing with named and optional args at callsites
 *------------------------------------------------------------------------- *)

let destNamedArg e = 
    match e with 
    | Expr_app (Expr_app(Expr_lid_get(false,[{ idText = nm; idRange=_}], _), 
                         Expr_lid_get(isOpt,[{ idText = a; idRange=_}], _), _),b,_) when nm = opname_Equals ->
        Some(isOpt,a,b)
    | _ -> None 
let isNamedArg e = (destNamedArg e <> None) 

let getMethodArgs cenv env arg =
    if verbose then  dprintf0 "--> tc_get_method_args\n";
    let args = 
        match arg with 
        | Expr_const (Const_unit,m) -> []
        | Expr_paren(Expr_tuple (args,m),_) | Expr_tuple (args,m) -> args
        | Expr_paren(arg,_) | arg -> [arg] in 
    let unnamedCallerArgs,namedCallerArgs = list_take_until isNamedArg args in 
    let namedCallerArgs = 
        namedCallerArgs |> chooseList (fun e -> 
          if not (isNamedArg e) then 
              error(Error("Anonymous arguments may not occur after named arguments",range_of_synexpr e)); 
          destNamedArg e) in 
    if verbose then  dprintf0 "in getMethodArgs\n";
    unnamedCallerArgs, namedCallerArgs


(*-------------------------------------------------------------------------
!* Helpers dealing with adhoc conversions (functions to delegates)
 *------------------------------------------------------------------------- *)

let coerce_fun_as_delegate eventInfoOpt cenv delty (minfo,del_argtys,del_rty) (f,fty) m =
    if verbose then dprintf0 "--> coerce_fun_as_delegate\n";
    let slotsig = slotsig_of_minfo cenv.g cenv.amap m minfo in
    let del_argvs,expr = 
        let arity_info = TopValInfo(0,replicate (length del_argtys) TopValData.unnamedTopArg, TopValData.unnamedRetVal) in

        (* Try to pull apart an explicit lambda and use it directly *)
        (* Don't do this in the case where we're adjusting the arguments of a function used to build a .NET-compatible event handler *)
        match (if isSome eventInfoOpt then None 
               else try_dest_top_lambda_upto cenv.g cenv.amap arity_info (f, fty)) with 
        | None -> 
        
            if exists (is_byref_ty cenv.g) del_argtys then
                    error(Error("This function value is being used to construct a delegate type whose signature includes a byref argument. You must use an explicit lambda expresion taking "^string_of_int(length del_argtys)^" arguments",m)); 

            let del_argvs = list_mapi (fun i argty -> fst (mk_compgen_local m "cg" argty)) del_argtys in
            let expr = 
                let args = 
                    match eventInfoOpt with 
                    | Some einfo -> 
                      begin match del_argvs with 
                      | [] -> error(event_err einfo m)
                      | h :: _ when not (type_equiv cenv.g cenv.g.obj_ty (type_of_val h)) -> error(event_err einfo m)
                      | _ :: t -> [mk_tupled_vars cenv.g m t] 
                      end
                    | None -> if isNil del_argtys then [mk_unit cenv.g m] else List.map (expr_for_val m) del_argvs in
                mk_appl((f,fty),[],args,m) in
            del_argvs,expr
            
        | Some _ -> 
           if isNil del_argtys then [], mk_appl((f,fty),[],[mk_unit cenv.g m],m) 
           else 
               let _,_,vsl,body,_ = iterated_adjust_arity_of_lambda_body cenv.g cenv.amap nng arity_info f in
               List.concat vsl, body in
            
(*
        | Some (_,_,vsl,body,_) -> 
           if isNil del_argtys then [], mk_appl((f,fty),[],[mk_unit cenv.g m],m) 
           else List.concat vsl, body in
*)

    let meth = TMethod(slotsig,[],del_argvs,expr,m) in
    TExpr_obj (new_uniq(),delty,None,mk_obj_ctor_call cenv.g m,[meth],[],m,new_cache())


(*-------------------------------------------------------------------------
!* Helpers dealing with pattern match compilation
 *------------------------------------------------------------------------- *)

let compilePatternForMatch cenv env exprm matchm warnOnUnused onfail (v,generalizedTypars) clauses rty =
  let dtree,targets = compilePattern cenv.niceNameGen cenv.g (denv_of_tenv env) cenv.amap exprm matchm warnOnUnused onfail (v,generalizedTypars) clauses rty in 
  mk_and_optimize_match exprm matchm rty dtree targets

let compilePatternForMatchClauses cenv env exprm matchm warnOnUnused onfail dty rty tclauses = 
  (* avoid creating a dummy in the common cases where we are about to bind a name for the expression *)
  (* REVIEW: avoid code duplication with code further below, i.e.all callers should call compilePatternForMatch *)
  match tclauses with 
  | [TClause(TPat_as (pat1,PBind (v,TypeScheme(generalizedTypars,_,_)),m1),None,TTarget(vs,e),m2) as clause] ->
      let expr = compilePatternForMatch cenv env exprm matchm warnOnUnused onfail (v,generalizedTypars) [TClause(pat1,None,TTarget(gen_remove vspec_eq v vs,e),m2)] rty in 
      v,expr
  | _ -> 
      let idv,idve = Tastops.mk_compgen_local exprm "matchval" dty in 
      let expr = compilePatternForMatch cenv env exprm matchm warnOnUnused onfail (idv,[]) tclauses rty in 
      idv,expr


(*-------------------------------------------------------------------------
!* Helpers dealing with comprehensions
 *------------------------------------------------------------------------- *)

    
(* An intermediate data structure used during type checking *)
(* of comprehensions *)
type tcompExpr = 
    | TComp_zero of Tast.typ * range
    | TComp_sum of (Tast.expr -> Tast.expr) * Tast.typ * tcompExpr * tcompExpr * range
    | TComp_for of (Patcompile.pat * val_spec list) * Tast.typ * Tast.typ * expr * Tast.typ * tcompExpr
    | TComp_while of (Tast.expr -> Tast.expr) * Tast.typ * expr * tcompExpr
    | TComp_cond of expr * Tast.typ * tcompExpr * tcompExpr * range
    | TComp_result of expr  * Tast.typ 
    | TComp_result_comp of expr  * Tast.typ * Tast.typ
    | TComp_match_and_bind of bool * expr * Tast.typ * Tast.typ * Tast.typ * (Patcompile.pat * val_spec list) * tcompExpr  
    | TComp_match of Tast.typ * expr * (tcompExpr * (Tast.expr -> tclause)) list
    | TComp_try_finally of (Tast.expr -> Tast.expr) * Tast.typ * tcompExpr * Tast.expr
    
    (*| TComp_concat of tcompExpr  list * Tast.typ * range *)


(* Get the fragmentary expressions resulting from turning *)
(* an expression into an enumerable value, e.g. at 'for' loops *)

(* TODO: do this in a 'lowering' phase later in compilation *)
(* to ensure the quotation contains a higher level view of the expression *)
let tc_ienumerator_exprs_for_arb_typed_expr cenv env m exprty expr =
    let exprToSearchForGetEnumeratorAndItem,tyToSearchForGetEnumeratorAndItem = 
        let argty = new_inference_typ cenv () in 
        let exprty_as_seq = mk_seq_ty cenv.g argty in
        if (type_must_subsume_type_and_undo_if_failed (denv_of_tenv env) cenv.css m exprty_as_seq exprty) then 
           mk_coerce(expr,exprty_as_seq,range_of_expr expr,exprty),exprty_as_seq
        else
           expr,exprty in 

    let find_minfo g amap m nm ty = 
        match try_find_minfo g amap m nm ty with 
        | [] -> error(Error("The type '"^NicePrint.pretty_string_of_typ (denv_of_tenv env) ty^"' is not a type whose values can be enumerated with this syntax, i.e. is not compatible with either seq<_>, IEnumerable<_> or IEnumerable and does not have a GetEnumerator method",m));
        | res :: _ -> res  
        (* We can't apply this condition because IEnumerable<_> itself has multiple GetEnumerator etc. methods *)
        (* | _ -> error(Error("The type '"^NicePrint.pretty_string_of_typ (denv_of_tenv env) ty^"' has an overloaded '"^nm^" method and may not be enumerated using an enumeration loop",m)) *)
        in
      
    let getEnumerator_minfo    = find_minfo cenv.g cenv.amap m "GetEnumerator" tyToSearchForGetEnumeratorAndItem in 
    let retTypeOfGetEnumerator = ret_typ_of_minfo cenv.g cenv.amap m getEnumerator_minfo [] in 

    let moveNext_minfo         = find_minfo cenv.g cenv.amap m "MoveNext" retTypeOfGetEnumerator in 
    let get_Current_minfo      = find_minfo cenv.g cenv.amap m "get_Current" retTypeOfGetEnumerator in
    let argty                  = ret_typ_of_minfo cenv.g cenv.amap m get_Current_minfo [] in 
    
    (* Like C#, we deploy some essentially adhoc (well, Framework-dependent) methods to determine a better *)
    (* type for .NET version 1.x abstractions that don't support the correct generic interface. *)
    let argty = 
        if type_equiv cenv.g argty cenv.g.obj_ty then begin

            (* Look for an 'Item' property, or a set of these with consistent return types *)
            let allEquivReturnTypes minfo others = 
                let rty = ret_typ_of_minfo cenv.g cenv.amap m minfo [] in 
                others |> List.for_all (fun other -> type_equiv cenv.g (ret_typ_of_minfo cenv.g cenv.amap m other []) rty) in
            
            match try_find_minfo cenv.g cenv.amap m "get_Item" tyToSearchForGetEnumeratorAndItem with
            | (minfo :: others) when allEquivReturnTypes minfo others -> 
                ret_typ_of_minfo cenv.g cenv.amap m minfo []
            
            | _ -> 
            
            (* Some types such as XmlNodeList have only an Item method  *)
            match try_find_minfo cenv.g cenv.amap m "Item" tyToSearchForGetEnumeratorAndItem with
            | (minfo :: others) when allEquivReturnTypes minfo others -> 
                ret_typ_of_minfo cenv.g cenv.amap m minfo []
            
            | _ -> argty
        end else argty in 
    if verbose then  dprintf3 "argty = %s @ %a\n" (NicePrint.pretty_string_of_typ (empty_denv cenv.g) argty) output_range m;
    let ienumeratorv,ienumeratore = Tastops.mk_mut_compgen_local m "_enumerator" retTypeOfGetEnumerator in
    let getEnumE  ,getEnumTy  = buildMethodCall cenv env PossiblyMutates   m false getEnumerator_minfo NormalValUse [] [exprToSearchForGetEnumeratorAndItem] [] in
    let guarde  ,guardty      = buildMethodCall cenv env DefinitelyMutates m false moveNext_minfo      NormalValUse [] [ienumeratore] [] in
    let currente,currentty    = buildMethodCall cenv env DefinitelyMutates m false get_Current_minfo   NormalValUse [] [ienumeratore] [] in
    let better_currente  = mk_coerce(currente,argty,range_of_expr currente,currentty) in 
    ienumeratorv, ienumeratore,retTypeOfGetEnumerator,argty,getEnumE,getEnumTy, guarde,guardty, better_currente
    
    
let tc_seq_of_arb_typed_expr cenv ty env expr =
    let m = (range_of_expr expr) in 
    let argty = new_inference_typ cenv () in 
    if (type_must_subsume_type_and_undo_if_failed (denv_of_tenv env) cenv.css m ( mk_seq_ty cenv.g argty) ty) then 
        expr,argty
    else          
        let enumv,enume = mk_compgen_local m "enum" ty in 
        let ienumeratorv, _,retTypeOfGetEnumerator,argty,getEnumE,getEnumTy,guarde,guardty,better_currente = tc_ienumerator_exprs_for_arb_typed_expr cenv env m ty enume in
        if is_struct_typ getEnumTy then errorR(Error("This expression has a method called GetEnumerator, but its return type is a value type. Methods returning struct enumerators cannot be used in this expression form",m));
        let getEnumE = mk_unit_delay_lambda cenv.g m getEnumE in
        let expr = 
           mk_let m enumv expr (mk_call_seq_of_functions cenv.g m retTypeOfGetEnumerator argty getEnumE 
                                             (mk_lambda m ienumeratorv (guarde,guardty)) 
                                             (mk_lambda m ienumeratorv (better_currente,argty))) in
        expr,argty           

let mk_seq_empty cenv env m genTy =
    (* We must discover the 'zero' of the monadic algebra being generated in order to compile failing matches. *)
    let genResultTy = new_inference_typ cenv () in 
    unifyE cenv env m genTy (mk_seq_ty cenv.g genResultTy);
    mk_call_seq_empty cenv.g m genResultTy 

let mk_seq_map_concat cenv env m enumElemTy genTy lam enumExpr =
    let genResultTy = new_inference_typ cenv () in 
    unifyE cenv env m genTy (mk_seq_ty cenv.g genResultTy);
    mk_call_seq_map_concat cenv.g m enumElemTy genResultTy lam enumExpr 

let mk_seq_using cenv env m resourceTy genTy lam enumExpr =
    type_must_subsume_type (denv_of_tenv env) cenv.css m NoTrace cenv.g.system_IDisposable_typ resourceTy;
    let genResultTy = new_inference_typ cenv () in 
    unifyE cenv  env m genTy (mk_seq_ty cenv.g genResultTy);
    mk_call_seq_using cenv.g m resourceTy genResultTy lam enumExpr 

let mk_seq_delay cenv env m genTy lam =
    let genResultTy = new_inference_typ cenv () in 
    unifyE cenv env m genTy (mk_seq_ty cenv.g genResultTy);
    mk_call_seq_delay cenv.g m genResultTy (mk_unit_delay_lambda cenv.g m lam) 


let mk_seq_append cenv env m genTy e1 e2 =
    let genResultTy = new_inference_typ cenv () in 
    unifyE cenv env m genTy (mk_seq_ty cenv.g genResultTy);
    mk_call_seq_append cenv.g m genResultTy e1 e2 

let mk_seq_generated cenv env m genTy e1 e2 =
    let genResultTy = new_inference_typ cenv () in 
    unifyE cenv env m genTy (mk_seq_ty cenv.g genResultTy);
    mk_call_seq_generated cenv.g m genResultTy e1 e2 

let mk_seq_finally cenv env m genTy e1 e2 =
    let genResultTy = new_inference_typ cenv () in 
    unifyE cenv env m genTy (mk_seq_ty cenv.g genResultTy);
    mk_call_seq_finally cenv.g m genResultTy e1 e2 

let mk_monadic_match_clauses (pat',vspecs) innerExpr = 
    [ TClause(pat',None,TTarget(vspecs, innerExpr),range_of_pat pat') ]

let mk_seq_match_clauses cenv env (pat',vspecs) genTy innerExpr m = 
    [TClause(pat',None,TTarget(vspecs, innerExpr),range_of_pat pat'); 
     (* add a clause that returns an empty IEnumerable if all other clauses fail *)
     TClause(TPat_wild(range_of_expr innerExpr),None,
             TTarget([],mk_seq_empty cenv env m genTy),m) ] 

let conv_tcomp_match_clauses cenv env inputExprMark (pat',vspecs) innerExpr bindPatTy genInnerTy wholeExprMark = 
    let patMark = (range_of_pat pat') in 
    let tclauses = mk_seq_match_clauses cenv env (pat',vspecs) genInnerTy innerExpr wholeExprMark in
    compilePatternForMatchClauses cenv env inputExprMark patMark false IncompleteWarnOnLast bindPatTy genInnerTy tclauses 

(* REVIEW: order reversal may confuse quotation *)
(* REVIEW: range marks could be better here *)
let rec conv_tcomp cenv env tcomp = 
    let denv = denv_of_tenv env in 
    let ad = accessRightsOfEnv env in
    match tcomp with 

    (* peephole optimization: "for x in e -> x" == "e" 
    | TComp_for((TPat_as (TPat_wild _,PBind (v,TypeScheme(generalizedTypars,_,_)),_), [_]),enumElemTy,genOuterTy,enumExpr,_,None,TComp_result (TExpr_val(vf,_,_),_,None)) 
        when cenv.g.vref_eq vf  (mk_local_vref v) -> 
        let enumExprMark = (range_of_expr enumExpr) in 
        (* 'e' might be a list or even a value type *)
        mk_coerce(enumExpr,genOuterTy,enumExprMark, type_of_expr cenv.g enumExpr)
*)
    (* peephole optimization: "for x in e1 -> e2" == "e1 |> map (fun x -> e2)" *)
    | TComp_for((TPat_as (TPat_wild _,PBind (v,TypeScheme(generalizedTypars,_,_)),_), [_]),enumElemTy,genOuterTy,enumExpr,_,TComp_result (yexpr,genEnumElemTy)) -> 
        let enumExprMark = (range_of_expr enumExpr) in 
        let lam = mk_lambda enumExprMark v (yexpr,genEnumElemTy) in
        let m = union_ranges enumExprMark (range_of_expr yexpr)   in 
        mk_call_seq_map cenv.g m enumElemTy genEnumElemTy lam enumExpr 
    | TComp_for((pat',vspecs),enumElemTy,genOuterTy,enumExpr,genInnerTy,innerComp)  ->
        let innerExpr = conv_tcomp cenv env innerComp in 
        let enumExprMark = (range_of_expr enumExpr) in 
        let wholeExprMark = union_ranges enumExprMark (range_of_expr innerExpr)   in 
        let matchv,matchExpr = conv_tcomp_match_clauses cenv env enumExprMark (pat',vspecs) innerExpr enumElemTy genInnerTy wholeExprMark in 
        let lam = mk_lambda enumExprMark matchv (matchExpr,type_of_expr cenv.g matchExpr) in
        mk_seq_map_concat cenv env wholeExprMark enumElemTy genOuterTy lam enumExpr 
        
    | TComp_while(mkDelayedExpr,genOuterTy,guardExpr,innerComp)  ->
    
        let guardExprMark = (range_of_expr guardExpr) in 
        let guardExpr = mk_unit_delay_lambda cenv.g guardExprMark guardExpr in 
        let innerExpr = mkDelayedExpr (conv_tcomp cenv env innerComp) in 
        mk_seq_generated cenv env guardExprMark genOuterTy guardExpr innerExpr

    | TComp_try_finally(mkDelayedExpr,genOuterTy,innerComp,unwindExpr)  ->
    
        let unwindExprMark = (range_of_expr unwindExpr) in 
        let unwindExpr = mk_unit_delay_lambda cenv.g unwindExprMark unwindExpr in 
        let innerExpr = mkDelayedExpr (conv_tcomp cenv env innerComp) in 
        let innerExprMark = (range_of_expr innerExpr) in
        
        mk_seq_finally cenv env innerExprMark genOuterTy innerExpr unwindExpr 

        
    | TComp_result (resultExpr,genResultTy)  ->
        mk_call_seq_singleton cenv.g (range_of_expr resultExpr) genResultTy resultExpr 
    | TComp_result_comp(resultExpr,genOuterTy,ty') ->
        mk_coerce(resultExpr,genOuterTy,range_of_expr resultExpr,ty')
(*
    | TComp_do (stmt,innerComp)  ->
        let innerExpr = conv_tcomp cenv denv innerComp in 
        let m = union_ranges (range_of_expr stmt) (range_of_expr innerExpr)   in 
        mk_seq m stmt innerExpr 
*)
    | TComp_match_and_bind (isUse, inputExpr,bindPatTy,genOuterTy,genInnerTy,(pat',vspecs),innerComp)  ->
        let inputExprTy = type_of_expr cenv.g inputExpr in 
        let innerExpr = conv_tcomp cenv env innerComp in 
        let innerExprMark = range_of_expr innerExpr   in 
        let inputExprMark = (range_of_expr inputExpr) in 
        let wholeExprMark = union_ranges inputExprMark innerExprMark in 
        let inputExprMark = (range_of_expr inputExpr) in 
        let matchv,matchExpr = conv_tcomp_match_clauses cenv env inputExprMark (pat',vspecs) innerExpr bindPatTy genInnerTy wholeExprMark in 
        let consumeExpr = mk_lambda wholeExprMark matchv (matchExpr,genInnerTy) in
        (* 'use' binding in sequence expression *)
        if isUse then 
            mk_seq_using cenv env wholeExprMark bindPatTy genOuterTy inputExpr consumeExpr 
        else 
            mk_let wholeExprMark matchv inputExpr matchExpr 

    | TComp_cond(guarde,genOuterTy,thenComp,elseComp,m) ->
        let thenExpr = conv_tcomp cenv env thenComp in 
        let elseExpr = conv_tcomp cenv env elseComp in 
        mk_cond m genOuterTy guarde thenExpr elseExpr

    | TComp_match(genOuterTy,inputExpr,clauses) ->
        let inputExprTy = type_of_expr cenv.g inputExpr in 
        let inputExprMark = (range_of_expr inputExpr) in 
        let tclauses = clauses |> map (fun (innerComp,f) -> f (conv_tcomp cenv env innerComp)) in
        let matchv,matchExpr = compilePatternForMatchClauses cenv env inputExprMark inputExprMark true Incomplete inputExprTy genOuterTy tclauses  in
        mk_let inputExprMark matchv inputExpr matchExpr 

    | TComp_zero(genOuterTy,m) ->
        mk_seq_empty cenv env m genOuterTy                 
        
    | TComp_sum(mkDelayedExpr,genOuterTy,innerComp1,innerComp2,m) ->
        let innerExpr1 = conv_tcomp cenv env innerComp1 in 
        let innerExpr2 = conv_tcomp cenv env innerComp2 in 
        let innerExpr2 = mkDelayedExpr innerExpr2 in 
        mk_seq_append cenv env m genOuterTy innerExpr1 innerExpr2

(*-------------------------------------------------------------------------
!* Post-transform initialization graphs using the 'lazy' interpretation.
 * See Don's ML workshop paper.
 *------------------------------------------------------------------------- *)

type initializationGraphAnalysisState = 
    | Top
    | InnerTop
    | DefinitelyStrict
    | MaybeLazy
    | DefinitelyLazy

let eliminateInitializationGraphs g mustHaveArity denv fixups_bindsWithoutLaziness bindsm =
    (* BEGIN INITIALIZATION GRAPHS *)
    (* Check for safety and determine if we need to insert lazy thunks *)
    let fixupsl,bindsWithoutLaziness = split fixups_bindsWithoutLaziness in
    let fixupsl : (expr ref * range) list list = fixupsl in
    let rvs = map (fun (v,_) -> mk_local_vref v) bindsWithoutLaziness in 
    let outOfOrder = ref false in 
    let runtimeChecks = ref false in 
    let directRecursiveData = ref false in 
    let reportedEager = ref false in 
    let definiteDependencies = ref [] in 
    let check availIfInOrder boundv expr = 
        let strict = function
            | MaybeLazy -> MaybeLazy
            | DefinitelyLazy -> DefinitelyLazy
            | Top | DefinitelyStrict | InnerTop -> DefinitelyStrict in 
        let lzy = function 
            | Top | InnerTop | DefinitelyLazy -> DefinitelyLazy 
            | MaybeLazy | DefinitelyStrict -> MaybeLazy in
        let fixable = function 
            | Top | InnerTop -> InnerTop
            | DefinitelyStrict -> DefinitelyStrict
            | MaybeLazy -> MaybeLazy
            | DefinitelyLazy -> DefinitelyLazy in 

        let rec check_expr st e = 
            if verbose then  dprintf2 "--> check_expr@%a\n" output_range (range_of_expr e);
            match strip_expr e with 
              (* Expressions with some lazy parts *)
            | TExpr_lambda (_,_,_,b,_,_,_) | TExpr_tlambda (_,_,b,_,_,_) -> checkDelayed st b
            | TExpr_obj (_,ty,_,e,overrides,iimpls,_,_) -> 
                (* NOTE: we can't fixup recursive references inside delegates since the closure delegee of a delegate is not accessible *)
                (* from outside. Object expressions implementing interfaces can, on the other hand, be fixed up. See FSharp 1.0 bug 1469 *)
                if is_interface_typ ty (* || is_delegate_typ ty *) then begin
                  iter (fun (TMethod(_,_,_,e,m)) ->  checkDelayed st e) overrides;
                  iter (snd >> iter (fun (TMethod(_,_,_,e,m)) ->  checkDelayed st e)) iimpls;
                end else begin
                  check_expr (strict st) e;
                  iter (fun (TMethod(_,_,_,e,m)) ->  check_expr (lzy (strict st)) e) overrides;
                  iter (snd >> iter (fun (TMethod(_,_,_,e,m)) ->  check_expr (lzy (strict st)) e)) iimpls;
                end
                
              (* Expressions where fixups may be needed *)
            | TExpr_val (v,_,m) -> check_val st v m

              (* Expressions where subparts may be fixable *)
            | TExpr_op((TOp_tuple | TOp_uconstr _ | TOp_recd _),_,args,_) -> 
                     (* REVIEW: ONLY FIXABLE WHEN (item_ref_in_this_assembly accessible_ccus tycon) *)
                iter (check_expr (fixable st)) args

              (* Composite expressions *)
            | TExpr_const _ -> ()
            | TExpr_letrec (binds,e,_,_)  ->
                iter (check_bind (strict st)) binds; 
                check_expr (strict st) e
            | TExpr_let (bind,e,_,_) ->  
                check_bind (strict st) bind; 
                check_expr (strict st) e
            | TExpr_match (_,pt,targets,_,_,_) -> 
                check_dtree (strict st) pt; 
                Array.iter (check_target (strict st)) targets 
            | TExpr_app(e1,_,_,args,_) -> 
                check_expr (strict st) e1;  
                iter (check_expr (strict st)) args 
          (* Binary expressions *)
            | TExpr_seq (e1,e2,_,_)
            | TExpr_static_optimization (_,e1,e2,_) ->
                 check_expr (strict st) e1;  check_expr (strict st) e2
          (* n-ary expressions *)
            | TExpr_op(op,_,args,m)  -> check_op st op m;  List.iter (check_expr (strict st)) args
          (* misc *)
            | TExpr_link(eref) -> check_expr st !eref
            | TExpr_tchoose (_,b,_)  -> check_expr st b
            | TExpr_hole(m,_) -> error(Error("unexpected TExpr_hole",m))
            | TExpr_quote _  -> ()

        and check_method st (TMethod(_,_,_,e,m)) =  check_expr (lzy (strict st)) e
        and check_bind st (TBind(v,e)) = check_expr st e 
        and check_dtree st = function
            | TDSwitch(e1,csl,dflt,_) -> check_expr st e1; iter (fun (TCase(_,d)) -> check_dtree st d) csl; Option.iter (check_dtree st) dflt
            | TDSuccess (es,n) -> iter (check_expr st) es
            | TDBind(bind,e) -> check_bind st bind; check_dtree st e
        and check_target st (TTarget(vs,e)) = check_expr st e

        and check_op st op m = 
            match op with 
            | TOp_lval_op (kind,lvr) -> check_val (strict st) lvr m
            | _ -> ()
          
        and check_val st v m = 
            match st with 
            | MaybeLazy -> 
                if gen_mem g.vref_eq v rvs then begin 
                  warning (RecursiveUseCheckedAtRuntime (denv,v,m)); 
                  if not !reportedEager then 
                    (warning (LetRecCheckedAtRuntime m); reportedEager := true);
                  runtimeChecks := true;
                end 
            | Top | DefinitelyStrict ->
                if gen_mem g.vref_eq v rvs then begin
                  if not (gen_mem g.vref_eq v availIfInOrder) then begin
                    warning (LetRecEvaluatedOutOfOrder (denv,boundv,v,m)); 
                    outOfOrder := true;
                    if not !reportedEager then 
                      (warning (LetRecCheckedAtRuntime m); reportedEager := true);
                  end;
                  definiteDependencies := (boundv,v) :: !definiteDependencies
                end
            | InnerTop -> 
                if gen_mem g.vref_eq v rvs then 
                  directRecursiveData := true
            | DefinitelyLazy -> () 
        and checkDelayed st b = 
            match st with 
            | MaybeLazy | DefinitelyStrict -> check_expr MaybeLazy b
            | DefinitelyLazy | Top | InnerTop -> () 
          
        in 
        check_expr Top expr
    in 

    ignore
      (List.fold_left 
         (fun availIfInOrder (v,e) -> 
           check availIfInOrder (mk_local_vref v) e; 
           (mk_local_vref v::availIfInOrder))
         [] bindsWithoutLaziness);
    
     (* ddg = definiteDependencyGraph *)
    let ddgNodes = map (fun (v,_) -> mk_local_vref v) bindsWithoutLaziness in 
    let ddg = mk_graph (stamp_of_vref, int_ord) ddgNodes (map (fun (v1,v2) -> (stamp_of_vref v1, stamp_of_vref v2)) !definiteDependencies) in 
    iter_cycles (fun path -> error (LetRecUnsound (denv,path,range_of_vref (hd path)))) ddg;

    let requiresLazyBindings = !runtimeChecks || !outOfOrder in 
    if !directRecursiveData && requiresLazyBindings then 
        error(Error("This recursive binding uses an invalid mixture of recursive forms",bindsm));

    let bindsBefore, bindsAfter = 
      if requiresLazyBindings then 
        let bindsBeforeL, bindsAfterL = 
          split
            (map2
               (fun fixupPoints (v,e) -> 
                 match strip_expr e with
                 | TExpr_lambda _ | TExpr_tlambda _ -> 
                     [mk_bind v e],[] 
                 | _ -> 
                     if verbose then  dprintf1 "value '%s' will use lazy thunks and runtime checks\n" (name_of_val v);
                     let ty = (type_of_val v) in 
                     let m = (range_of_val v) in 
                     let vty = (mk_lazy_ty g ty) in
                     let vlazy,velazy = Tastops.mk_compgen_local m  ("_"^(name_of_val v)) vty  in
                     let vrhs = (mk_lazy_delayed g m ty e) in 
                     if mustHaveArity then (data_of_val vlazy).val_arity <- Some(infer_arity_of_expr vty [] [] vrhs);
                     fixupPoints |> iter (fun (fp,m) -> fp := mk_lazy_force g (range_of_expr !fp) ty vlazy (mk_lazystatus_undefined g (range_of_expr !fp) ty));
                     [mk_bind vlazy vrhs],
                     [mk_bind v (mk_lazy_force g m ty vlazy (mk_lazystatus_undefined g m ty))])
               fixupsl
               bindsWithoutLaziness) in 
        List.concat bindsBeforeL, List.concat bindsAfterL
      else
        map (fun (v,e) -> mk_bind v e) bindsWithoutLaziness,[] in 
    bindsBefore @ bindsAfter

(*-------------------------------------------------------------------------
 * Check the shape of an object constructor and rewrite calls 
 *------------------------------------------------------------------------- *)

let checkAndRewriteObjectCtor g env ctorLambaExpr =

     let m = range_of_expr ctorLambaExpr in 
     let tps,vsl,body,rty = dest_top_lambda (ctorLambaExpr,type_of_expr g ctorLambaExpr) in 

    (* Rewrite legitimate self-construction calls to CtorValUsedAsSelfInit *)
    let error(expr) = 
        errorR(Error(sprintf "This is not a valid object construction expression. Object constructors can only be implemented by a limited range of expressions",range_of_expr expr));
        expr in

    (* Build an assignment into the mutable reference cell that holds recursive references to 'this' *)
    let rewriteContruction recdExpr = 
       match env.eCtorInfo with 
       | None -> recdExpr
       | Some ctorInfo -> 
           match ctorInfo.ctorThisRefCellVarOpt with 
           | None -> recdExpr
           | Some vref -> 
               let ty = type_of_expr g recdExpr in 
               TExpr_seq(recdExpr,(mk_refcell_set g  m ty (expr_for_vref m vref) (mk_ldarg0 m ty)),ThenDoSeq,m) in

    let rec checkAndRewrite expr = 
        match expr with 
        (* <ctor-body> = { fields } *)
        (* The constructor ends in an object initialization expression - good *)
        | TExpr_op(TOp_recd(RecdExprIsObjInit,_),_,_,_) -> rewriteContruction expr

        (* <ctor-body> = "a; <ctor-body>" *)
        | TExpr_seq(a,body,NormalSeq,b)  -> TExpr_seq(a,checkAndRewrite body,NormalSeq,b) 

        (* <ctor-body> = "<ctor-body> then <expr>" *)
        | TExpr_seq(body,a,ThenDoSeq,b) -> TExpr_seq(checkAndRewrite body,a,ThenDoSeq,b)

        (* <ctor-body> = "let pat = expr in <ctor-body>" *)
        | TExpr_let(bind,body,m,_)  -> mk_let_bind m bind (checkAndRewrite body)

        (* The constructor is a sequence "let pat = expr in <ctor-body>" *)
        | TExpr_match(a,b,targets,c,d,e)  -> TExpr_match(a,b, (targets |> Array.map (fun (TTarget(vs,body)) -> TTarget(vs, checkAndRewrite body))),c,d,e)

        (* <ctor-body> = "let rec binds in <ctor-body>" *)
        | TExpr_letrec(a,body,_,b) -> TExpr_letrec (a,checkAndRewrite body ,m,new_cache())

        (* <ctor-body> = "new C(...)" *)
        | TExpr_app(f,b,c,d,m) -> 
            (* The application had better be an application of a ctor *)
            let f = checkAndRewriteCtorUsage f in
            let expr = TExpr_app(f,b,c,d,m) in
            rewriteContruction expr 

        | _ -> 
            error(expr)

    and checkAndRewriteCtorUsage expr = 
         match expr with 
         | TExpr_link eref -> 
               let e = checkAndRewriteCtorUsage !eref in
               eref := e;
               expr
               
         (* Type applications are ok, e.g. 
              type C<'a>(x:int) = 
                  new() = C<'a>(3) *)                 
         | TExpr_app(f,fty,tyargs,[],m) -> 
             let f = checkAndRewriteCtorUsage f in
             TExpr_app(f,fty,tyargs,[],m)

         (* Self-calls are OK and get rewritten. *)
         | TExpr_val(vref,NormalValUse,a) ->
           let isCtor = 
               match member_info_of_vref vref with 
              | None -> false
              | Some(vspr) -> (vspr.vspr_flags.memFlagsKind = MemberKindConstructor) in

           if not isCtor then error(expr) else
           TExpr_val(vref,CtorValUsedAsSelfInit,a)
        | _ -> 
            error(expr) in
    
    let body = checkAndRewrite body in 
    mk_multi_lambdas m tps vsl (body, rty) 
    

(*-------------------------------------------------------------------------
(* Post-typechecking normalizations to enforce semantic constraints*)
(* Lazy and, Lazy or, -int32, -int64 *)
 *------------------------------------------------------------------------- *)
let build_app cenv expr exprty arg m = 
    match expr,arg with        
    | TExpr_app(TExpr_val(vf,_,_),_,_,[],_), TExpr_const(TConst_int32(n),m,ty) 
         when cenv.g.vref_eq vf cenv.g.unary_neg_vref -> 
           TExpr_const(TConst_int32(Int32.neg n),m,ty)
    | TExpr_val(vf,_,_), TExpr_const(TConst_int64(n),m,ty)
         when cenv.g.vref_eq vf cenv.g.unary_neg_vref -> 
           TExpr_const(TConst_int64(Int64.neg n),m,ty)
    | TExpr_app(TExpr_val(vf,_,_),_,_,[x0],_) , _ 
         when cenv.g.vref_eq vf cenv.g.and_vref 
           || cenv.g.vref_eq vf cenv.g.and2_vref  -> 
        mk_lazy_and cenv.g m x0 arg
    | TExpr_app(TExpr_val(vf,_,_),_,_,[x0],_), _ 
         when cenv.g.vref_eq vf cenv.g.or_vref
           || cenv.g.vref_eq vf cenv.g.or2_vref -> 
        mk_lazy_or cenv.g m x0 arg 
    | TExpr_app(TExpr_val(vf,_,_),_,_,[],_), _ when cenv.g.vref_eq vf cenv.g.addrof_vref || 
                                                    cenv.g.vref_eq vf cenv.g.addrof2_vref -> 
        if cenv.g.vref_eq vf cenv.g.addrof2_vref then warning(UseOfAddressOfOperator(m));
        let wrap,e1a' = mk_expra_of_expr cenv.g true DefinitelyMutates arg m  in
        wrap(e1a')
    | _ -> 
        if verbose then  dprintf2 "--> tc_delayed, calling mk_appl @%a\n" output_range m; 
        mk_appl((expr,exprty),[],[arg],m) 
            

(*-------------------------------------------------------------------------
!* Additional data structures used by type checking
 *------------------------------------------------------------------------- *)

type delayedItem = 
  | TyApp of (Ast.typ list * Range.range)
  | App of (Ast.synexpr * Range.range)
  | Lvalue_get of (Ast.ident list * Range.range)
  | Lvalue_set of (Ast.synexpr * Range.range)

let mk_Lvalue_set(e,m) = 
    (* We have lid <- e. Wrap 'e' in another pair of parentheses to ensure it's never interpreted as *)
    (* a named argument, e.g. for "el.Checked <- (el = el2)" *)
    Lvalue_set(Expr_paren(e,m),m)

type newslotsOK = NewSlotsOK | NoNewSlots

type overridesOK = OverridesOK | WarnOnOverrides

type implictlyBoundTyparsAllowed = NewTyparsOK | NoNewTypars

type checkConstraints = CheckCxs | NoCheckCxs

type typeCheckPass = SecondPass | FirstPass 

(** Provides information about the context for a value or member definition *)
type containerInfo = ContainerInfo of (** The nearest containing module. Used as the 'actual' parent for extension members and values *)
                                      (*parent:*)parent_ref * 
                                       (** The logical apparent parent of a value/member, either a module, type or exception *)
                                      ((*tcref:*)tycon_ref * 
                                       (*optIntfSlotTy:*)(Tast.typ * slotImplSet) option * 
                                       (*optBaseVal:*)val_spec option) option 

(** Indicates a declaration is contained in an expression *)
let exprContainerInfo = ContainerInfo(ParentNone,None)
(** Indicates a declaration is contained in the given module *)
let moduleContainerInfo modref = ContainerInfo(Parent(modref),Some(modref,None,None))
(** Indicates a declaration is contained in the given type definition in the given module *)
let tyconContainerInfo (parent,tcref) = ContainerInfo(parent,Some(tcref,None,None))

type recBindingDefn = RecBindingDefn of containerInfo * newslotsOK * declProcessingFlag * binding

type normRecBindingDefn = NormRecBindingDefn of containerInfo * newslotsOK * declProcessingFlag * normBinding

type tyconBindingDefn = TyconBindingDefn of containerInfo * newslotsOK * declProcessingFlag * classMemberDefn * range

type tyconBindingDefns = TyconBindingDefns of tycon_ref * tyconBindingDefn list

type tyconMemberData = TyconMemberData of declProcessingFlag * tycon_ref * val_spec option * classMemberDefn list * range * newslotsOK


type valSpecResult = ValSpecResult of parent_ref * val_meminfo option * ident * typars * typars * Tast.typ * partialArityOfVal * declProcessingFlag 

(*-------------------------------------------------------------------------
!* Specifications of contraints on type parameters may occur at
 * various points.
 *------------------------------------------------------------------------- *)

let rec tc_typar_constraint ridx cenv newOk checkCxs env tpenv c = 
    match c with 
    | WhereTyparEqualsType(tp,ty,m) ->
       let ty',tpenv = tc_type cenv newOk checkCxs env tpenv ty in 
       let tp',tpenv = tc_typar cenv env newOk tpenv tp in
       if (newOk = NoNewTypars) then errorR(Error("invalid constraint",m));
       unifyE cenv env m (mk_typar_ty tp') ty';
       tpenv
    | WhereTyparDefaultsToType(tp,ty,m) ->
       let ty',tpenv = tc_type cenv newOk checkCxs env tpenv ty in 
       let tp',tpenv = tc_typar cenv env newOk tpenv tp in
       let csenv = (mk_csenv cenv.css m (denv_of_tenv env)) in 
       recordConstraint 0 csenv m NoTrace tp' (TTyparDefaultsToType(ridx,ty',m)) |> commitOperationResult;
       tpenv
    | WhereTyparSubtypeOfType(tp,ty,m) ->
       let ty',tpenv = tc_type cenv newOk checkCxs env tpenv ty in 
       let tp',tpenv = tc_typar cenv env newOk tpenv tp in
       if (newOk = NoNewTypars) && is_sealed_typ cenv.g ty' then 
         errorR(Error("invaid constraint: the type used for the constraint is sealed, which means the constraint could only be satisfied by at most one solution",m));
       type_must_subsume_type (denv_of_tenv env) cenv.css m NoTrace  ty' (mk_typar_ty tp') ;
       tpenv
    | WhereTyparSupportsNull(tp,m) ->
       let tp',tpenv = tc_typar cenv env newOk tpenv tp in
       type_must_support_null (denv_of_tenv env) cenv.css m NoTrace (mk_typar_ty tp') ;
       tpenv
    | WhereTyparIsReferenceType(tp,m) ->
       let tp',tpenv = tc_typar cenv env newOk tpenv tp in
       type_is_reference_type (denv_of_tenv env) cenv.css m NoTrace (mk_typar_ty tp') ;
       tpenv
    | WhereTyparIsValueType(tp,m) ->
       let tp',tpenv = tc_typar cenv env newOk tpenv tp in
       type_is_value_type (denv_of_tenv env) cenv.css m NoTrace (mk_typar_ty tp') ;
       tpenv
    | WhereTyparIsEnum(tp,tyargs,m) ->
       let tp',tpenv = tc_typar cenv env newOk tpenv tp in
       begin match tyargs with 
       | [underlying] -> 
           let underlying',tpenv = tc_type cenv newOk checkCxs env tpenv underlying in 
           type_is_enum (denv_of_tenv env) cenv.css m NoTrace (mk_typar_ty tp') underlying';
       | _ -> errorR(Error("an 'enum' constraint must be of the form 'enum<type>'",m));
       end;
       tpenv
    | WhereTyparIsDelegate(tp,tyargs,m) ->
       let tp',tpenv = tc_typar cenv env newOk tpenv tp in
       begin match tyargs with 
       | [a;b] -> 
           let a',tpenv = tc_type cenv newOk checkCxs env tpenv a in 
           let b',tpenv = tc_type cenv newOk checkCxs env tpenv b in 
           type_is_delegate (denv_of_tenv env) cenv.css m NoTrace (mk_typar_ty tp') a' b';
       | _ -> errorR(Error("an 'enum' constraint must be of the form 'enum<type>'",m));
       end;
       tpenv
    | WhereTyparSupportsMember(tps,memSpfn,m) ->
        let tps',tpenv = map_acc_list (tc_typar cenv env newOk) tpenv tps in
        let traitInfo,tpenv = tc_pseudo_member_spec cenv newOk checkCxs env tps tpenv memSpfn m in 
        match traitInfo with 
        | TTrait([ty],"new",memFlags,[_],rty) when (memFlags.memFlagsKind=MemberKindConstructor) ->
            type_must_support_default_ctor (denv_of_tenv env) cenv.css m NoTrace ty ;
            tpenv
        | _ ->  
            type_must_support_trait (denv_of_tenv env) cenv.css m NoTrace traitInfo;
            tpenv
      
and tc_pseudo_member_spec cenv newOk checkCxs env tps tpenv memSpfn m = 
    let tps',tpenv = map_acc_list (tc_typar cenv env newOk) tpenv tps in
    let tys = List.map mk_typar_ty tps' in 
    match memSpfn with 
    | ClassMemberSpfn_binding (valSpfn,memFlags,m) ->
        let members,tpenv = tc_val_spec cenv env ModuleOrMemberBinding (exprContainerInfo) (Some memFlags) (Some (List.hd tys)) tpenv valSpfn [] in
        begin match members with 
        | [ValSpecResult(_,_,id,_,_,ty',partialValArity,_)] -> 
            (* REVIEW: Test pseudo constraints cannot refer to polymorphic methods, *)
            (* REVIEW: since when using pseudo constraints for a wider variety of purposes restriction would bite *)
            let tps,_ = try_dest_forall_typ ty' in 
            let compiledArity = translatePartialArity tps partialValArity in
            let tps,argtysl,rty,_ = dest_top_type compiledArity ty' in 
            let argtys = concat argtysl in 
            let argtys = map fst argtys in
            TTrait(tys,text_of_id id,memFlags,argtys,rty),tpenv
        | _ -> error(InternalError("This constraint is invalid",m))
        end
    | _ -> error(InternalError("This constraint is invalid",m))


(*-------------------------------------------------------------------------
!* Contraints may include member specifications
 *------------------------------------------------------------------------- *)
 
and tc_val_spec cenv env declKind containerInfo memFlagsOpt thisTyOpt tpenv (ValSpfn(attrs,id,ValTyparDecls(synTypars,_,synTyparConstraints), ty, topValSigData, _,mutableFlag,doc, vis,literalExprOpt,m)) attrs' =
    (*printf "tc_val_spec: id=%s\n" id.idText;*)
    let declaredTypars = tc_typar_decls cenv env synTypars in
    let (ContainerInfo(altActualParent,tcrefContainerInfo)) = containerInfo in
    let enclosingDeclaredTypars,vsprInfo,thisTyOpt,declKind = 
        match tcrefContainerInfo with 
        | Some(tcref,_,_) -> 
            let _,enclosingDeclaredTypars,_,_,thisTy = freshenThisTy cenv m TyparRigid tcref in
            (* An implemented interface type is in terms of the type's type parameters. *)
            (* We need a signature in terms of the values' type parameters. *)
            (* let optIntfSlotTy = Option.map (inst_type renaming) optIntfSlotTy in  *)
            enclosingDeclaredTypars,Some(tcref),Some(thisTy),declKind
        | None -> 
            [],None,thisTyOpt, ModuleOrMemberBinding in 
    let allDeclaredTypars = (enclosingDeclaredTypars@declaredTypars) in            
    let envinner = add_declared_typars NoCheckForDuplicateTypars allDeclaredTypars env in 
    let newOk = NewTyparsOK in 
    let checkCxs = CheckCxs in 
    let tpenv = tc_typar_constraints cenv newOk checkCxs envinner tpenv synTyparConstraints in 
    (* process the type, including any constraints *)
    let ty',tpenv = tc_type cenv newOk checkCxs envinner tpenv ty in (* REVIEW: check ramifications of allowing NewTyparsOK here *)
    (* no more constraints allowed on declared typars *)
    allDeclaredTypars |> List.iter (setTyparRigid cenv.g (denv_of_tenv env) m);
    
    let res = 
        match memFlagsOpt,thisTyOpt with 
        | Some(memFlags),Some(thisTy) -> 
            let generateOneMember(memFlags) = 
                let ty',topValSigData = 
                    match memFlags.memFlagsKind with 
                    | MemberKindClassConstructor
                    | MemberKindConstructor
                    | MemberKindMember -> 
                        ty',topValSigData
                    | MemberKindPropertyGet ->  
                        if SynArgInfo.hasNoArgs topValSigData then 
                          (cenv.g.unit_ty --> ty'), (SynArgInfo.incorporateUnitArg topValSigData)
                        else
                          ty',topValSigData
                    | MemberKindPropertySet ->  
                        (* let tps,tau = try_dest_forall_typ ty' in  *)
                        let dtys,rty = strip_fun_typ_upto (SynArgInfo.numCurriedArgs topValSigData) ty' in
                        (mk_iterated_fun_ty dtys (rty --> cenv.g.unit_ty)), (SynArgInfo.incorporateSetterArg topValSigData)
                    | MemberKindPropertyGetSet -> error(Error("unexpected MemberKindPropertyGetSet from signature parsing",m)) in 
                  
                let topValSynData = adjustTopValSigData cenv.g ty' topValSigData in
                (* Curried members (e.g. "this.f x1 x2 x3 = ...") are compiled and called as *)
                (* members returning functions, (e.g. "this.f x1 = (fun x2 x3 -> ...)") *)
                (* That is, before adding the 'this' pointer the arities of members are limited to [], [_], and also *)
                (* [_;_] in the indexed property setter case. *)
                (* See matching code in ast.ml for definitions of members. *)

                let topValSynData = 
                    let (TopValSynData(args,ret)) = topValSynData in 
                    match args with 
                    | [] | [_] -> topValSynData
                    | [h1;h2] when memFlags.memFlagsKind=MemberKindPropertySet -> topValSynData
                    | h1 :: _ -> TopValSynData([h1],ret) in

                let ty',topValSynData = 
                    if memFlags.memFlagsInstance then 
                      (thisTy --> ty'), (SynArgInfo.incorporateSelfArg topValSynData)
                    else
                      ty',topValSynData in 

                let (PartialArityInfo(argsData,retData)) as compiledArity = 
                    translateTopValSynData m (tc_attributes cenv env) topValSynData in

                (* Fold in the optional arugment information *)
                (* Resort to using the syntactic arugment information since that is what tells us *)
                (* what is optional and what is not. *)
                let ty' = 

                    try 
                        if SynArgInfo.hasOptionalArgs topValSynData then 
                            let argtysl,rty = dest_top_tau_type argsData ty' in
                            let argtysl = 
                                (combine (map (map fst) argtysl) (SynArgInfo.getArgInfos topValSynData)) 
                                |> map (fun (argtys,argInfos) ->
                                     (combine argtys argInfos)
                                     |> map (fun (argty,argInfo) ->
                                         if SynArgInfo.isOptionalArg argInfo then mk_option_ty cenv.g argty
                                         else argty)) in
                            mk_iterated_fun_ty (map (mk_tupled_ty cenv.g) argtysl) rty
                        else ty' 
                    with err -> 
                        warning(Error("unexpected problem while building optional argument types in signature:"^Printexc.to_string err, m));
                        ty' in
                        

                let vsprOpt,id = 
                    match vsprInfo with 
                    | Some(tcref) -> 
                      let isExtensionBinding = (declKind = ExtensionBinding) in 
                      let vspr,id = mkMemberDataAndUniqueId cenv.g tcref isExtensionBinding attrs' None memFlags topValSynData id in
                      Some(vspr),id
                    | None -> None,id in

                
                ValSpecResult(altActualParent,vsprOpt,id,enclosingDeclaredTypars,declaredTypars,ty',compiledArity,declKind) in 
                
            begin match memFlags.memFlagsKind with 
            | MemberKindClassConstructor
            | MemberKindConstructor
            | MemberKindMember 
            | MemberKindPropertyGet 
            | MemberKindPropertySet ->
                [ generateOneMember(memFlags) ]
            | MemberKindPropertyGetSet ->
                [ generateOneMember({memFlags with memFlagsKind=MemberKindPropertyGet});
                  generateOneMember({memFlags with memFlagsKind=MemberKindPropertySet}); ]
            end
        | _ ->
            let topValSynData = adjustTopValSigData cenv.g ty' topValSigData in
            let compiledArity = translateTopValSynData m (tc_attributes cenv env) topValSynData in
            [ ValSpecResult(altActualParent,None,id,enclosingDeclaredTypars,declaredTypars,ty',compiledArity,declKind) ] 
    in res,tpenv


(*-------------------------------------------------------------------------
!* Bind types 
 *------------------------------------------------------------------------- *)

and tc_typar cenv env newOk tpenv (Typar(id,_,_) as tp) =
  let key = id.idText in 
  match Map.tryfind key env.eNameResEnv.eTypars with
  | Some res -> res, tpenv
  | None -> 
  match tryFindUnscopedTypar key tpenv with
  | Some res -> res, tpenv
  | None -> 
    if newOk = NoNewTypars then error (UndefinedName(0,"type parameter",id,["<unimplemented>"]));
    (* OK, this is an implicit declaration of a type parameter *)
    let tp' = tc_typar_decl cenv env (TyparDecl([],tp)) in
    tp',addUnscopedTypar key tp' tpenv

and tc_typar_decl cenv env (TyparDecl(attrs,(Typar(id,_,_) as tp))) =
   let attrs' = tc_attributes cenv env attrTgtParameter  attrs in 
   new_typar(TyparWarnIfNotRigid,tp,false,attrs')

and tc_typar_decls cenv env synTypars = map (tc_typar_decl cenv env) synTypars

and tc_type cenv newOk checkCxs env (tpenv:syntacticUnscopedTyparEnv) ty =
    if verbose then  dprintf2 "--> tc_t ype@%a\n" output_range (range_of_syntype ty); 
    match ty with 
    | Type_app (tc,args,m) -> 
        let tcref = forceRaise(tc_tycon_id OpenQualified cenv.g cenv.amap m env.eNameResEnv tc (length args)) in 
        if cenv.g.tcref_eq tcref cenv.g.system_IndexOutOfRangeException_tcref then 
            warning(IndexOutOfRangeExceptionWarning(m)); 
        tc_type_app cenv newOk checkCxs env tpenv m tcref [] args 
    | Type_proj_then_app (ltyp,lid,args,m) -> 
        let ltyp,tpenv = tc_type cenv newOk checkCxs env tpenv ltyp in 
        if not (is_stripped_tyapp_typ ltyp) then error(Error("This type has no nested types",m));
        let tcref,tinst = dest_stripped_tyapp_typ ltyp in 
        let tcref = forceRaise(tc_tycon_id_in_tcref cenv.g cenv.amap (ResolveTypeNamesToTypeRefs,Some(length args)) 0 m tcref lid ) in 
        tc_type_app cenv newOk checkCxs env tpenv m tcref tinst args 
    | Type_tuple(args,m) ->
        let args',tpenv = tc_types cenv newOk checkCxs env tpenv args in 
        TType_tuple(args'),tpenv
    | Type_fun(dty,rty,m) -> 
        let dty',tpenv = tc_type cenv newOk checkCxs env tpenv dty in 
        let rty',tpenv = tc_type cenv newOk checkCxs env tpenv rty in 
        (dty' --> rty'), tpenv
    | Type_forall (td,rty,m) -> 
        (* note: no constraints allowed here yet *)
        let tp' = tc_typar_decl cenv env td in
        setTyparRigid cenv.g (denv_of_tenv env) m tp';
        let rty',tpenv = tc_type cenv newOk checkCxs (add_declared_typars NoCheckForDuplicateTypars [tp'] env) tpenv rty in 
        ([tp'] +-> rty'), tpenv
    | Type_arr (n,dty,m) -> 
        let ty',tpenv = tc_type cenv newOk checkCxs env tpenv dty in 
        (* When using erasure all type variables used within IL array types or operations must be pseudo *)
        if cenv.g.typeCheckerConfiguredToAssumeErasureOfGenerics then 
          begin 
            try solveTypStaticReq (mk_csenv cenv.css m (denv_of_tenv env)) NoTrace CompleteStaticReq ty' |> raiseOperationResult
            with e -> errorR (ErrorFromAddingConstraint((denv_of_tenv env),e,m))
          end;
        mk_il_arr_ty cenv.g n ty', tpenv
    | Type_var (tp,m) -> 
        let tp',tpenv = tc_typar cenv env newOk tpenv tp in 
        TType_var tp',tpenv
    | Type_anon m ->           (* _ *)
        let tp = tc_anon_type cenv TyparFlexible newOk m in
        TType_var tp, tpenv
    | Type_with_global_constraints(ty,wcs,m) ->
        let cty,tpenv = tc_type cenv newOk checkCxs env tpenv ty in
        let tpenv = tc_typar_constraints cenv newOk checkCxs env tpenv wcs in
        cty,tpenv
    | Type_anon_constraint(ty,m) ->  (* #typ *)
        let tp' = tc_anon_type cenv TyparWarnIfNotRigid newOk m in
        let ty',tpenv = tc_type cenv newOk checkCxs env tpenv ty in
        type_must_subsume_type (denv_of_tenv env) cenv.css m NoTrace  ty' (mk_typar_ty tp') ;
        TType_var tp', tpenv

and tc_anon_type cenv rigid newOk m =
    if newOk = NoNewTypars then errorR (Error("anonymous type variables are not permitted in this declaration",m));
    new_anon_inference_tyvar cenv (m,rigid,NoStaticReq,false)
 
and tc_types cenv newOk checkCxs env tpenv args =
   map_acc_list (tc_type cenv newOk checkCxs env) tpenv args 

and tc_typar_constraints cenv newOk checkCxs env tpenv wcs =
    (* Mark up default constraints with a priority in reverse order: last gets 0, second last gets 1 etc. See comment on TTyparDefaultsToType *)
    let _,tpenv = List.fold_left (fun (ridx,tpenv) tc -> ridx - 1, tc_typar_constraint ridx cenv newOk checkCxs env tpenv tc) (List.length wcs - 1, tpenv) wcs in
    tpenv

and tc_type_app cenv newOk checkCxs env tpenv m tcref pathTypeArgs args =
    tcref_accessible_check m (accessRightsOfEnv env) tcref;
    tcref_attrib_check cenv.g tcref m |> commitOperationResult;
    let tps,inst,tinst,gtyp = info_of_tcref cenv m env tcref in 
    (* If we're not checking constraints, i.e. when we first assert the super/interfaces of a type definition, then just *)
    (* clear the constaint lists of the freshly generated type variables. A little ugly but fairly localized. *)
    if checkCxs = NoCheckCxs then tps |> iter (fun tp -> (data_of_typar tp).typar_constraints <- []);
    let args',tpenv = tc_types cenv newOk checkCxs env tpenv args in 
    let args' = pathTypeArgs @ args' in 
    if length tinst <> length args' then error (TyconBadArgs((denv_of_tenv env),tcref,length args',m));
    iter2 (unifyE cenv env m) tinst args';
    TType_app(tcref,args'),tpenv
    
and tc_type_and_recover cenv newOk checkCxs env tpenv ty   =
    try tc_type cenv newOk checkCxs env tpenv ty 
    with e -> 
        errorRecoveryPoint(e); 
        (if newOk = NewTyparsOK then new_error_typ cenv () else cenv.g.obj_ty),tpenv 

and tc_nested_type_app cenv newOk checkCxs env tpenv m typ tyargs =
    if not (is_stripped_tyapp_typ typ) then error(Error("This type has no nested types",m));
    match typ with 
    | TType_app(tcref,tinst) -> 
        let pathTypeArgs,_ = chop_at (max (length tinst - length (typars_of_tcref tcref)) 0) tinst in 
        tc_type_app cenv newOk checkCxs env tpenv m tcref pathTypeArgs tyargs 
    | _ -> error(InternalError("tc_nested_type_app: expected type application",m))


(*-------------------------------------------------------------------------
!* Bind elements of lambdas. Not at all clear why we don't use tc_pat.
 *------------------------------------------------------------------------- *)

and tc_simple_pat optArgsOK cenv ty env (tpenv,names) p = 
  match p with 
  | SPat_as (id,isMemberThis,isOpt,m) -> 
      if isOpt && not optArgsOK then errorR(Error("optional arguments are only permitted on type members (2)",m));
      if isOpt then begin
          let tyarg = new_inference_typ cenv () in 
          unifyE cenv env m ty (mk_option_ty cenv.g tyarg);
      end;
              
      let _,names = tc_pat_bind_name cenv env id ty isMemberThis None (OptionalInline,infer_iflex,noArgOrRetAttribs,false,None,false) names in
      id.idText, 
      (tpenv,names)
  | SPat_typed (p,cty,m) ->
      let cty',tpenv = tc_type_and_recover cenv NewTyparsOK CheckCxs env tpenv cty in 
      begin match p with 
      (* Optional arguments on members *)
      | SPat_as(_,_,true,_) -> unifyE cenv env m ty (mk_option_ty cenv.g cty');
      | _ -> unifyE cenv env m ty cty';
      end;
      tc_simple_pat optArgsOK cenv ty env (tpenv,names) p
  | SPat_attrib (p,_,m) ->
      tc_simple_pat optArgsOK cenv ty env (tpenv,names) p

and tc_simple_pats cenv optArgsOK  ty env (tpenv,names) p = 
  match p with 
  | SPats ([],m) -> 
      let id = ident("_unit",m) in 
      unifyE cenv env m ty cenv.g.unit_ty;
      let _,names = tc_pat_bind_name cenv env id ty false None (OptionalInline,infer_iflex,noArgOrRetAttribs,false,None,false) names in
      [id.idText],(tpenv,names)
  | SPats ([p],m) -> 
      let v,(tpenv,names) = tc_simple_pat optArgsOK cenv ty env (tpenv,names) p in 
      [v],(tpenv,names)
  | SPats (ps,m) -> 
      let ptys = unify_tuple cenv (denv_of_tenv env) m ty ps in 
      let ps',(tpenv,names) = map_acc_list (fun tpenv (ty,e) -> tc_simple_pat optArgsOK cenv ty env tpenv e) (tpenv,names) (combine ptys ps) in 
      ps',(tpenv,names)
  | SPats_typed (p,cty,m) ->
      let cty',tpenv = tc_type_and_recover cenv NewTyparsOK CheckCxs env tpenv cty in 
      begin match p with 
      (* Solitary optional arguments on members *)
      | SPats([SPat_as(_,_,true,_)],_) -> unifyE cenv env m ty (mk_option_ty cenv.g cty');
      | _ -> unifyE cenv env m ty cty';
      end;
      tc_simple_pats cenv optArgsOK  ty env (tpenv,names) p


(*-------------------------------------------------------------------------
!* Bind patterns. Patterns are type-checked in three phases: 
 * 1. tc_pat builds a map from simple variable names to inferred types for 
 *    those variables. It also returns a function to perform the second phase.
 * 2. The second phase assumes the caller has built the actual value_spec's 
 *    for the values being defined, and has decided if the types of these 
 *    variables are to be generalized. The caller hands this information to
 *    the second-phase function in terms of a map from names to actual
 *    value specifications. 
 *------------------------------------------------------------------------- *)

and tc_pat_bind_name cenv env id idty isMemberThis vis1 (inlineFlag,declaredTypars,argAttribs,mut,vis2,compgen) names = 
    (*
    (* Values with active-pattern name have the corresponding active-pattern type *)
    (* This reduces the need for type annotations in the recursive case. *)
    (* REVIEW: is this really needed? Any recursive uses of the active pattern will assert the required type! *)
    (* NO, it's not needed, and doesn't work for parameterized patterns. *)
    begin match apinfo_of_vname id.idText with 
    | None -> ()
    | Some apinfo -> 
        let restys = new_inference_typs cenv (names_of_apinfo apinfo) in 
        let dty = new_inference_typ cenv () in 
        let act_pat_ty = mk_apinfo_typ cenv.g apinfo dty restys  in
        unifyE cenv env id.idRange act_pat_ty idty;
    end;
    *)
    let vis = if isSome vis1 then vis1 else vis2 in 
    if Map.mem id.idText names then errorR (VarBoundTwice id);
    let base = if isMemberThis then MemberThisVal else NormalVal in 
    let names = Map.add id.idText (PrelimValScheme1(id,declaredTypars,idty,None,mut,inlineFlag,base,argAttribs,vis,compgen)) names in 
    (fun (TcPatPhase2Input values) -> 
        let (vspec,typeScheme) = 
            try Map.find id.idText values 
            with Not_found -> error(Error("Name "^id.idText^" not bound in pattern context",id.idRange)) in 
        PBind(vspec,typeScheme)),
    names

and tc_pat warnOnUpper cenv env vFlags (tpenv,names) ty pat = 
    if verbose then  dprintf2 "--> tc_pat@%a\n" output_range (range_of_synpat pat);
    match pat with 
    | Pat_const (c,m) -> 
        let c' = tc_const cenv ty m env c in
        (fun (_:tcPatPhase2Input) -> TPat_const(c',m)),(tpenv,names)
        
    | Pat_wild m ->
        (fun _ -> TPat_wild m), (tpenv,names)

    | Pat_isinst(cty,m) 
    | Pat_as (Pat_isinst(cty,m),_,_,_,_) -> 
        let srcty = ty in 
        let tgty,tpenv = tc_type_and_recover cenv NewTyparsOK CheckCxs env tpenv cty in 
        tc_runtime_type_test cenv (denv_of_tenv env) m tgty srcty;
        begin match pat with 
        | Pat_isinst(cty,m) ->
            (fun _ -> TPat_isinst (srcty,tgty,None,m)),(tpenv,names)
        | Pat_as (Pat_isinst(cty,_),id,isMemberThis,vis,m) -> 
            let bindf,names = tc_pat_bind_name cenv env id tgty isMemberThis vis vFlags names in
            (fun values -> TPat_isinst (srcty,tgty,Some(bindf values),m)),
            (tpenv,names)
        | _ -> failwith "tc_pat"
        end

    | Pat_opt_var (_,m) -> error(Error("optional arguments are only permitted on type members",m))
    | Pat_as (p,id,isMemberThis,vis,m) -> 
        let bindf,names = tc_pat_bind_name cenv env id ty isMemberThis vis vFlags names in
        let pat',acc = tc_pat warnOnUpper cenv env vFlags (tpenv,names) ty p in 
        (fun values -> TPat_as (pat' values,bindf values,m)), 
        acc
    | Pat_typed (p,cty,m) ->
        let cty',tpenv = tc_type_and_recover cenv NewTyparsOK CheckCxs env tpenv cty in 
        unifyE cenv env m ty cty';
        tc_pat warnOnUpper cenv env vFlags (tpenv,names) ty p
    | Pat_attrib (p,attrs,m) ->
        let attrs' = tc_attributes cenv env attrTgtBinding  attrs in 
        tc_pat warnOnUpper cenv env vFlags (tpenv,names) ty p
    | Pat_disj (pat1,pat2,m) ->
        let pat1',(tpenv,names1) = tc_pat warnOnUpper cenv env vFlags (tpenv,names) ty pat1 in 
        let pat2',(tpenv,names2) = tc_pat warnOnUpper cenv env vFlags (tpenv,names) ty pat2 in 
        if not (Zset.equal (Namemap.domain names1) (Namemap.domain names2)) then
          errorR (UnionPatternsBindDifferentNames m);
        names1 |> Map.iter (fun u (PrelimValScheme1(id1,_,ty1,_,_,_,_,_,_,_)) -> 
          match Map.tryfind id1.idText names2 with 
          | None -> () 
          | Some (PrelimValScheme1(id2,_,ty2,_,_,_,_,_,_,_)) -> 
              unifyE cenv env m ty1 ty2);
        (fun values -> TPat_disjs ([pat1' values;pat2' values],m)), (tpenv,names1)

    | Pat_conjs (pats,m) ->
        let pats',acc = tc_pats warnOnUpper cenv env vFlags (tpenv,names) (map (fun pat -> ty) pats) pats in 
        (fun values -> TPat_conjs(map (fun f -> f values) pats',m)), acc

    | Pat_lid (lid,tyargs,args,vis,m) ->
        if isSome tyargs then errorR(Error("Type arguments may not be specified here",m));
        let warnOnUpper = if isNil args then warnOnUpper else AllIdsOK in
        begin match tc_pat_lid warnOnUpper None cenv.ginstf cenv.g cenv.amap m env.eNameResEnv defaultTypeNameResInfo lid with
        | Item_newdef id -> 
          begin match args with 
          | [] -> tc_pat warnOnUpper cenv env vFlags (tpenv,names) ty (mksyn_pat_var vis id)
          | _ -> error (UndefinedName(0,"pattern discriminator",id,[]))
          end

        | Item_apelem(APElemRef(apinfo,vref,idx) as apref) -> 
            (* TOTAL/PARTIAL ACTIVE PATTERNS *)
            let vexp, vexpty = tc_val cenv env vref None m in 

            let exprargs,patarg = match args with | [x] -> [],x | [] -> [],Pat_const(Const_unit,m) | _ -> frontAndBack args in
            if nonNil exprargs && length (names_of_apinfo apinfo) <> 1 then error(Error("Only active patterns returning exactly one result may accept arguments.",m));
            (* if nonNil exprargs then warning(Experimental("The syntax for parameterized active patterns is under review and may change in a future release",m)); *)

            (* Note we parse arguments to parameterized pattern labels as patterns, not expressions. *)
            (* This means the range of syntactic expression forms that can be used here is limited. *)
            let rec convSynPatToSynExpr x = 
                match x with
                | Pat_const (c,m) -> Expr_const(c,m)
                | Pat_as (Pat_wild _,id,_,None,m) -> Expr_lid_get(false,[id],m)
                | Pat_typed (p,cty,m) -> Expr_typed (convSynPatToSynExpr p,cty,m)
                | Pat_lid (lid,tyargs,args,None,m) -> List.fold_left (fun f x -> Expr_app(f,convSynPatToSynExpr x,m)) (Expr_lid_get(false,lid,m)) args
                | Pat_tuple (args,m) -> Expr_tuple(map convSynPatToSynExpr args,m)
                | Pat_paren (p,m) -> convSynPatToSynExpr p
                | Pat_array_or_list (isArray,args,m) -> Expr_array_or_list(isArray,map convSynPatToSynExpr args,m)
                | Pat_expr (e,m) -> e
                | Pat_null m -> Expr_null(m)
                | _ -> error(Error("Invalid argument to parameterized pattern label",range_of_synpat x)) in
            let exprargs = map convSynPatToSynExpr exprargs in 

            let restys = new_inference_typs cenv (names_of_apinfo apinfo) in 
            let act_pat_ty = mk_apinfo_typ cenv.g apinfo ty restys  in

            let pexprty = new_inference_typ cenv () in 
            let pexp, tpenv = propagate_then_tc_delayed cenv act_pat_ty env tpenv (range_of_expr vexp) vexp vexpty (map (fun e -> App(e,range_of_synexpr e)) exprargs) in

            if idx >= (length restys) then error(Error("intenal error: Invalid index into active pattern array",m));
            let argty = List.nth restys idx  in
                
            let arg',(tpenv,names) = tc_pat warnOnUpper cenv env vFlags (tpenv,names) argty patarg in 
            (* If there are any expression args then we've lost identity. *)
            let identityVrefOpt = (if nonNil exprargs then None else Some vref) in
            (fun values -> TPat_query((pexp, restys, identityVrefOpt,idx,apinfo), arg' values, m)), 
            (tpenv,names)

        | (Item_ucref _ | Item_ecref _) as item ->
            (* DATA MATCH CONSTRUTORS *)
            let mkf,argtys = pat_constr_unify m cenv env ty item in 
            let nargtys = length argtys in 
            let args = 
              match args with 
              | []-> []
              (* note: the next will always be parenthesized *)
              | [(Pat_tuple (args,m)) | Pat_paren(Pat_tuple (args,m),_)] when nargtys > 1 -> args

              (* note: like OCaml we allow both 'C _' and 'C (_)' regardless of number of argument of the pattern *)
              | [(Pat_wild m as e) | Pat_paren(Pat_wild m as e,_)] -> Array.to_list (Array.create nargtys e)
              | [arg] -> [arg] 
              | _ when nargtys = 0 -> error(Error("This data constructor does not take arguments",m)) 
              | _ when nargtys = 1 -> error(Error("This data constructor takes one argument",m)) 
              | _ -> error(Error("This data constructor expects "^string_of_int nargtys^" arguments in tupled form",m)) in
            gen_constr_check env nargtys (length args) m;
            let args',(tpenv,names) = 
              tc_pats warnOnUpper cenv env vFlags (tpenv,names) argtys args in 
            (fun values -> mkf(map (fun f -> f values) args')), (tpenv,names)
        | Item_il_field finfo ->
            (* LITERAL .NET FIELDS *)
            il_finfo_accessible_check cenv.g cenv.amap m (accessRightsOfEnv env) finfo;
            if not (il_finfo_is_static finfo) then errorR (Error ("field "^name_of_il_finfo finfo^" is not static",m));
            il_finfo_attrib_check cenv.g finfo m;
            begin match il_finfo_literal_value finfo with 
            | None -> error (Error("this field is not a literal and cannot be used in a pattern", m));
            | Some lit -> 
                unifyE cenv env m ty (vtyp_of_il_finfo cenv.amap m finfo);
                let c' = tc_field_init m lit in 
                (fun _ -> TPat_const (c',m)),(tpenv,names)             
            end
            
        | Item_recdfield rfinfo ->
            (* LITERAL F# FIELDS *)
            rfinfo_accessible_check m (accessRightsOfEnv env) rfinfo;
            if not (rfinfo_is_static rfinfo) then errorR (Error ("field "^name_of_rfinfo rfinfo^" is not static",m));
            rfinfo_attrib_check cenv.g rfinfo m  |> commitOperationResult;        
            begin match rfinfo_literal_value rfinfo with 
            | None -> error (Error("this field is not a literal and cannot be used in a pattern", m));
            | Some lit -> 
                unifyE cenv env m ty (vtyp_of_rfinfo rfinfo);
                (fun _ -> TPat_const (lit,m)),(tpenv,names)             
            end

        | Item_val vref ->
            begin match literal_const_of_val (deref_val vref) with 
            | None -> error (Error("this value is not a literal and cannot be used in a pattern", m));
            | Some lit -> 
                let (vexp,vexpty) = tc_val cenv env vref None m in
                vref_accessible_check m (accessRightsOfEnv env) vref;
                fs_attrib_check cenv.g (attribs_of_vref vref) m |> commitOperationResult;
                unifyE cenv env m ty vexpty;
                (fun _ -> TPat_const (lit,m)),(tpenv,names)             
            end

        |  _ -> error (Error("this is not a variable, constant, active recognizer or literal",m))
        end

    | Pat_expr(exp,m) -> error (Error("this is not a valid pattern",m))
          
    | Pat_tuple (args,m) ->
        let argtys = new_inference_typs cenv args in 
        unifyE cenv env m ty (TType_tuple argtys);
        let args',acc = tc_pats warnOnUpper cenv env vFlags (tpenv,names) argtys args in 
        (fun values -> TPat_tuple(map (fun f -> f values) args',argtys,m)), acc

    | Pat_paren (p,m) ->
        tc_pat warnOnUpper cenv env vFlags (tpenv,names) ty p
    | Pat_array_or_list (isArray,args,m) ->
        let argty = new_inference_typ cenv () in 
        unifyE cenv env m ty (if isArray then Tastops.mk_array_ty cenv.g argty else Tastops.mk_list_ty cenv.g argty);
        let args',acc = tc_pats warnOnUpper cenv env vFlags (tpenv,names) (map (fun _ -> argty) args) args in 
        (fun values -> 
            let args' = map (fun f -> f values) args' in 
            if isArray then TPat_array(args', argty, m)
            else list_fold_right (mk_cons_pat cenv.g argty) args' (mk_nil_pat cenv.g m argty)), acc

    | Pat_recd (flds,m) ->
        let tcref,fldsmap,fldsList (* REVIEW: use this *) = gen_build_fldmap cenv env ty flds m in
        (* REVIEW: use fldsList to type check pattern in code order not field defn order *) 
        let _,inst,tinst,gtyp = info_of_tcref cenv m env tcref in 
        unifyE cenv env m ty gtyp;
        let fields = instance_rfields_of_tcref tcref in 
        let ftys = map (fun fsp -> freshenPossibleForallTy cenv m TyparFlexible (typ_of_rfield inst fsp),fsp) fields in
        let fldsmap',acc = 
          map_acc_list 
            (fun s ((_,ftinst,ty),fsp) -> 
              if Map.mem fsp.rfield_id.idText  fldsmap then 
                let f,s = tc_pat warnOnUpper cenv env vFlags s ty (Map.find fsp.rfield_id.idText fldsmap) in 
                (ftinst,f),s
              else ([],(fun values -> TPat_wild m)),s)
            (tpenv,names)
            ftys in
        (fun values -> TPat_recd (tcref,tinst,map (fun (ftinst,f) -> (ftinst,f values)) fldsmap',m)), 
        acc

    | Pat_range (c1,c2,m) -> 
        warning(OCamlCompatibility("Character range matches are included for compatibility with OCaml only and are under revision for inclusion in F#. Consider using a 'when' pattern guard instead",m));
        unifyE cenv env m ty (cenv.g.char_ty);
        (fun values -> TPat_range(c1,c2,m)),(tpenv,names)
    | Pat_null m -> 
        type_must_support_null (denv_of_tenv env) cenv.css m NoTrace ty;
        (fun _ -> TPat_null m),(tpenv,names)
    | Pat_instance_member (_,_,_,m) -> 
        errorR(Error("illegal pattern",range_of_synpat pat));
        (fun _ -> TPat_wild m), (tpenv,names)

and tc_pats warnOnUpper cenv env vFlags s argtys args = 
    assert (length args  = length argtys);
    map_acc_list (fun s (ty,pat) -> tc_pat warnOnUpper cenv env vFlags s ty pat) s (combine argtys args)


(*-------------------------------------------------------------------------
!* tc_expr
 *
 * During checking of expressions of the form (x(y)).z(w1,w2) 
 * keep a stack of things on the right. This lets us recognize .NET 
 * method applications. 
 *------------------------------------------------------------------------- *)

and solveTypAsError cenv denv m ty =
    let ty2 = new_error_typ cenv () in 
    assert(from_error_of_typar (dest_typar_typ ty2));
    solveTypEqualsTypKeepAbbrevs 0 (mk_csenv cenv.css m denv) m NoTrace ty ty2 |> ignore

and tc_expr_of_unknown_type cenv env tpenv expr =
    let exprty = new_inference_typ cenv () in
    let expr',tpenv = tc_expr cenv exprty env tpenv expr in
    expr',exprty,tpenv

and tc_expr cenv ty (env:tcEnv) tpenv expr =
  let m = range_of_synexpr expr in 

  if verbose then  dprintf2 "--> tc_expr@%a\n" output_range (range_of_synexpr expr);

  (* Start an error recovery handler *)
  (* Note the try/catch can lead to tail-recursion problems for iterated constructs, e.g. let... in... *)
  (* So be careful! *)
  try 
    (* Count our way through the expression shape that makes up an object constructor *)
    (* See notes at definition of "ctor" re. object model constructors. *)
    let env = 
        if get_ctorShapeCounter env > 0 then adjust_ctorShapeCounter (fun x -> x - 1) env 
        else env in 

    let tm,tpenv = tc_expr_then cenv ty env tpenv expr [] in 

    if verbose then  dprintf2 "<-- tc_expr@%a\n" output_range (range_of_synexpr expr);
    tm,tpenv
  with e -> 
    if verbose then  dprintf2 "!!! tc_expr@%a\n" output_range (range_of_synexpr expr);

    (* Error recovery - return some rubbish expression, but replace/annotate *)
    (* the type of the current expression with a type variable that indicates an error *)
    errorRecoveryPoint e; 
    solveTypAsError cenv (denv_of_tenv env) m ty;
    if verbose then  dprintf2 "<-- tc_expr@%a\n" output_range (range_of_synexpr expr);
    mk_throw m ty (mk_one cenv.g m), tpenv


(* This is used to typecheck legitimate 'main body of constructor' expressions *)
and tc_ctor_expr ctorThisVarRefCellOpt cenv ty env tpenv expr =
  let env = {env with eCtorInfo = Some(initialExplicitCtorInfo(ctorThisVarRefCellOpt)) } in 
  let expr,tpenv = tc_expr cenv ty env tpenv expr in 
  let expr = checkAndRewriteObjectCtor cenv.g env expr in
  expr,tpenv

(* This is used to typecheck all ordinary expressions including constituent *)
(* parts of ctor. *)
and tc_body_expr cenv ty env tpenv expr =
  let env = if within_ctorShape env then adjust_ctorShapeCounter (fun x -> x + 1) env else env in 
  tc_expr cenv ty env tpenv expr

(* This is used to typecheck legitimate 'non-main body of object constructor' expressions *)
and tc_nonbody_expr cenv ty env tpenv expr =
  let env = if within_ctorShape env then leave_ctorShape env else env in 
  tc_expr cenv ty env tpenv expr

(* This is used to typecheck legitimate 'non-main body of object constructor' expressions *)
and tc_nonbody_stmt cenv env tpenv expr =
  let env = if within_ctorShape env then leave_ctorShape env else env in 
  tc_stmt cenv env tpenv expr

and tc_stmt cenv env tpenv expr =
  let expr',ty,tpenv = tc_expr_of_unknown_type cenv env tpenv expr in 
  let m = range_of_synexpr expr in 
  unify_unit cenv (denv_of_tenv env) m ty (Some expr');
  expr',tpenv


and tc_expr_then cenv ty env tpenv expr delayed =
    match expr with 

    | Expr_lid_get (isOpt,lid,m) ->
        if isOpt then errorR(Error("syntax error - unexpected '?' symbol",m));
        tc_lid_then cenv ty env tpenv m lid delayed

    | Expr_app (f,x,m) ->
        tc_expr_then cenv ty env tpenv f ((App (x,m)):: delayed)

    | Expr_tyapp (f,x,m) ->
        tc_expr_then cenv ty env tpenv f ((TyApp (x,m)):: delayed)

    | Expr_lvalue_get (e1,lid,m) ->
        tc_expr_then cenv ty env tpenv e1 ((Lvalue_get (lid,m))::delayed)
           
    | Expr_lbrack_get (e1,idx,m) 
    | Expr_lbrack_set (e1,idx,_,m) ->
        tc_indexer_then cenv env ty m tpenv expr e1 idx delayed
    
    | _  ->
        match delayed with 
        | [] -> tc_expr_undelayed cenv ty env tpenv expr
        | _ -> 
            let expr',exprty,tpenv = tc_expr_undelayed_notype cenv env tpenv expr in
            propagate_then_tc_delayed cenv ty env tpenv (range_of_expr expr') expr' exprty delayed 

and tc_exprs cenv env m tpenv argtys args = 
    if (length args  <> length argtys) then error(Error(sprintf "expected %d expressions, got %d" (length argtys) (length args),m));
    map_acc_list (fun tpenv (ty,e) -> tc_expr cenv ty env tpenv e) tpenv (combine argtys args) 


(*-------------------------------------------------------------------------
!* tc_expr_undelayed
 *------------------------------------------------------------------------- *)

and tc_expr_undelayed_notype cenv env tpenv expr =
    let exprty = new_inference_typ cenv () in
    let expr',tpenv = tc_expr_undelayed cenv exprty env tpenv expr in
    expr',exprty,tpenv

and tc_expr_undelayed cenv ty env tpenv expr =
    let m = range_of_synexpr expr in 
    (* dprintf3 "tc_expr_undelayed: %a: isSome(env.eFamilyType) = %b\n" output_range m (isSome env.eFamilyType); *)

    if verbose then  dprintf2 "--> tc_expr_undelayed@%a\n" output_range (range_of_synexpr expr); 
    match expr with 
    | Expr_paren (expr2,m2) -> 
        (* We invoke callQualSink for every construct which is atomic in the syntax, i.e. where a '.' immediately following the *)
        (* construct is a dot-lookup for the result of the construct. *)
        callQualSink m (nenv_of_tenv env,Item_typs("<expr>",[ty]), denv_of_tenv env);
        tc_expr cenv ty env tpenv expr2

    | Expr_lbrack_get _ | Expr_lbrack_set _
    | Expr_tyapp _ | Expr_lid_get _ | Expr_app _ | Expr_lvalue_get _ -> error(Error("tc_expr_undelayed: delayed", range_of_synexpr expr))

    | Expr_const (Const_string (s,m),_) -> 
        callQualSink m (nenv_of_tenv env,Item_typs("<expr>",[ty]), denv_of_tenv env);
        tc_const_string_expr cenv ty env m tpenv s

    | Expr_const (c,m) -> 
        callQualSink m (nenv_of_tenv env,Item_typs("<expr>",[ty]), denv_of_tenv env);
        tc_const_expr cenv ty env m tpenv c

    | Expr_lambda (isMember,spats,bodyExpr,m) ->
        let dty,rty = unify_fun None cenv (denv_of_tenv env) m ty in 
        let vs, (tpenv,names) = tc_simple_pats cenv isMember dty env (tpenv,Map.empty) spats in
        let envinner,_,vspecMap = mkAndPublishSimpleVals cenv env m names in
        let envinner = if isMember then envinner else exit_familyRegion envinner in
        let bodyExpr,tpenv = tc_expr cenv rty envinner tpenv bodyExpr in 
        mk_multi_lambda m (map (fun nm -> Namemap.find nm vspecMap) vs) (bodyExpr,rty),tpenv 

    | Expr_match (x,matches,isExnMatch,m) ->
        if verbose then  dprint_endline "tc Expr_match";
        let x',dty,tpenv = tc_expr_of_unknown_type cenv env tpenv x in 
        (* let env = record_latest_match_typ m dty env in // hack for intellisense to record the match type in scope *)
        let exprm = range_of_expr x' in 
        let v,e, tpenv = tc_and_patcompile_match_clauses exprm m (if isExnMatch then Throw else Incomplete) cenv dty ty env tpenv matches in
        mk_let exprm v x' e,tpenv 

    | Expr_assert (x,m) ->
        tc_assert_expr cenv ty env m tpenv x

    | Expr_typed (e,cty,m) ->
        let tgty,tpenv = tc_type_and_recover cenv NewTyparsOK CheckCxs env tpenv cty in 
        unifyE cenv env m ty tgty;
        let e',tpenv = tc_expr cenv ty env tpenv e  in 
        e',tpenv

    | Expr_isinst (e,tgty,m) ->
        let e',srcty,tpenv = tc_expr_of_unknown_type cenv env tpenv e  in 
        unifyE cenv env m ty cenv.g.bool_ty;
        let tgty,tpenv = tc_type cenv NewTyparsOK CheckCxs env tpenv tgty in 
        tc_runtime_type_test cenv (denv_of_tenv env) m tgty srcty;        
        let e' = mk_call_istype cenv.g m tgty  e' in
        e', tpenv
    
    (* Expr_addrof is noted in the syntax ast in order to recognize it as concrete type information *)
    (* during type checking, in particular prior to resolving overloads. This helps distinguish *)
    (* its use at method calls from the use of the conflicting 'ref' mechanism for passing byref parameters *)
    | Expr_addrof(byref,e,m) -> 
        tc_expr cenv ty env tpenv (mksyn_prefix m (if byref then "~&" else "~&&") e) 
        
    | Expr_upcast (e,_,m) | Expr_arb_upcast (e,m) -> 
        let e',srcty,tpenv = tc_expr_of_unknown_type cenv env tpenv e  in 
        let tgty,tpenv = 
          match expr with
          | Expr_upcast (e,tgty,m) -> 
              let tgty,tpenv = tc_type cenv NewTyparsOK CheckCxs env tpenv tgty in 
              unifyE cenv env m tgty ty;
              tgty,tpenv
          | Expr_arb_upcast (e,m) -> ty,tpenv 
          | _ -> failwith "upcast" in 
        tc_static_upcast cenv (denv_of_tenv env) m tgty srcty;
        mk_coerce(e',tgty,m,srcty),tpenv

    | Expr_downcast(e,_,m) | Expr_arb_downcast (e,m) ->
        let e',srcty,tpenv = tc_expr_of_unknown_type cenv env tpenv e  in 
        let tgty,tpenv = 
          match expr with
          | Expr_downcast (e,tgty,m) -> 
              let tgty,tpenv = tc_type cenv NewTyparsOK CheckCxs env tpenv tgty in 
              unifyE cenv env m tgty ty;
              tgty,tpenv
          | Expr_arb_downcast (e,m) -> ty,tpenv 
          | _ -> failwith "downcast" in 
        tc_runtime_type_test cenv (denv_of_tenv env) m tgty srcty;
        (* tc_runtime_type_test ensures tgty is a nominal type. Hence we can insert a check here *)
        (* based on the nullness semantics of the nominal type. *)
        let e' = mk_call_unbox cenv.g m tgty  e' in
        e',tpenv

    | Expr_null (m) ->
        type_must_support_null (denv_of_tenv env) cenv.css m NoTrace ty;
        mk_null m ty,tpenv

    | Expr_ifnull (e1,e2,m) ->
        let e1',tpenv = tc_expr cenv ty env tpenv e1  in 
        let e2',tpenv = tc_expr cenv ty env tpenv e2  in 
        type_must_support_null (denv_of_tenv env) cenv.css m NoTrace ty;
        mk_let_in m "nullTestVariable" ty e1' (fun (v,ve) -> 
        mk_nonnull_cond cenv.g m ty ve ve e2'),tpenv

    | Expr_tuple (args,m) -> 
        let argtys = unify_tuple cenv (denv_of_tenv env) m ty args in 
        let args',tpenv = tc_exprs cenv env m tpenv argtys args in 
        mk_tupled cenv.g m args' argtys, tpenv

    | Expr_array_or_list (isArray,args,m) -> 
        callQualSink m (nenv_of_tenv env,Item_typs("<expr>",[ty]), denv_of_tenv env);
        let argty = new_inference_typ cenv () in 
        unifyE cenv env m ty (if isArray then Tastops.mk_array_ty cenv.g argty else Tastops.mk_list_ty cenv.g argty);
        let args',tpenv = map_acc_list (tc_expr cenv argty env) tpenv args in 
        let expr = 
            if isArray then TExpr_op(TOp_array, [argty],args',m)
            else list_fold_right (mk_cons cenv.g argty) args' (mk_nil cenv.g m argty) in
        expr,tpenv

    | Expr_new (superInit,objTy,arg,m) -> 
        if verbose then  dprintf2 "--> Expr_new@%a\n" output_range m;
        let objTy',tpenv = tc_type cenv NewTyparsOK CheckCxs env tpenv objTy in
        unifyE cenv env m ty objTy';
        tc_new cenv env tpenv objTy' superInit arg m

    | Expr_impl(objTy,argopt,binds,iimpls,m) ->
        callQualSink m (nenv_of_tenv env,Item_typs("<expr>",[ty]), denv_of_tenv env);
        tc_object_expr cenv ty env tpenv (objTy,argopt,binds,iimpls,m)
            
    | Expr_recd (inherits,opt_old, flds, m) -> 
        callQualSink m (nenv_of_tenv env,Item_typs("<expr>",[ty]), denv_of_tenv env);
        tc_recd_expr cenv ty env tpenv (inherits,opt_old,flds,m)

    | Expr_while (e1,e2,m) ->
        unifyE cenv env m ty cenv.g.unit_ty;
        let e1',tpenv = tc_expr cenv (cenv.g.bool_ty) env tpenv e1 in 
        let e2',tpenv = tc_stmt cenv env tpenv e2 in 
        mk_while cenv.g (e1',e2',m),tpenv

    | Expr_for (id,start,dir,finish,body,m) ->
        unifyE cenv env m ty cenv.g.unit_ty;
        let start' ,tpenv = tc_expr cenv (cenv.g.int_ty) env tpenv start in 
        let finish',tpenv = tc_expr cenv (cenv.g.int_ty) env tpenv finish in
        let idv,ide         = Tastops.mk_local id.idRange              id.idText                               cenv.g.int_ty in
        let envinner = addLocalVal m idv env in 
        let body',tpenv = tc_stmt cenv envinner tpenv body in
        mk_fast_for_loop  cenv.g m idv start' dir finish' body', tpenv
        
    | Expr_foreach (pat,enumExpr,body,m) ->
        tc_foreach_expr cenv ty env tpenv (pat,enumExpr,body,m)

    | Expr_comprehension (comp,m) ->
        let env = exit_familyRegion env in
        tc_comprehension cenv env ty m None tpenv comp
        
    | Expr_list_of_seq (comp,m) 
    | Expr_array_of_seq (comp,m) ->
        callQualSink m (nenv_of_tenv env,Item_typs("<expr>",[ty]), denv_of_tenv env);
        let isList = match expr with Expr_list_of_seq _ -> true | _ -> false in 
        let genCollElemTy = new_inference_typ cenv () in 
        let genCollTy =  (if isList then mk_list_ty else Tastops.mk_array_ty) cenv.g genCollElemTy in 
        unifyE cenv env m ty genCollTy;
        let exprty = new_inference_typ cenv () in 
        let genEnumTy =  mk_seq_ty cenv.g genCollElemTy in 
        type_must_subsume_type (denv_of_tenv env) cenv.css m NoTrace genEnumTy exprty; 
        let expr,tpenv = tc_expr cenv exprty env tpenv comp in 
        (if isList then mk_call_seq_to_list else mk_call_seq_to_array) 
           cenv.g m genCollElemTy (mk_coerce(expr,genEnumTy,range_of_expr expr,exprty)),tpenv

    | Expr_let (isRec,isUse,binds,body,m) ->
        tc_iterated_let_exprs cenv env ty (fun x -> x) tpenv (isRec,isUse,binds,body,m) 

    | Expr_try_catch (e1,m1,handler_clauses,m2,m) ->
        let e1',tpenv = tc_expr cenv ty env tpenv e1 in 
        (* Compile the pattern twice, once as a filter with all succeeding targets returning "1", and once as a proper catch block. *)
        let filter_clauses = handler_clauses |> map (function (Clause(pat,opt_when,e,m)) -> Clause(pat,opt_when,(Expr_const(Const_int32 1l,m)),m)) in 
        let tfilter_clauses, tpenv = tc_match_clauses cenv cenv.g.exn_ty cenv.g.int_ty env tpenv filter_clauses in
        let thandler_clauses, tpenv = tc_match_clauses cenv cenv.g.exn_ty ty env tpenv handler_clauses in
        let v1,filter_expr = compilePatternForMatchClauses cenv env m2 m2 true FailFilter cenv.g.exn_ty cenv.g.int_ty tfilter_clauses in 
        let v2,handler_expr = compilePatternForMatchClauses cenv env m2 m2 true Rethrow cenv.g.exn_ty ty thandler_clauses in 
        mk_try_catch cenv.g (e1',v1,filter_expr,v2,handler_expr,m,ty),tpenv

    | Expr_try_finally (e1,e2,m2) ->
        let e1',tpenv = tc_expr cenv ty env tpenv e1 in 
        let e2',tpenv = tc_stmt cenv env tpenv e2 in
        mk_try_finally cenv.g (e1',e2',m2,ty),tpenv

    | Expr_seq (dir,e1,e2,m) ->
        if dir then 
          let e1',tpenv = tc_nonbody_stmt cenv env tpenv e1 in 
          let e2',tpenv = tc_body_expr cenv ty env tpenv e2 in 
          TExpr_seq(e1',e2',NormalSeq,m),tpenv
        else 
          (* Constructors using "new (...) = <ctor-expr> then <expr>" *)
          let e1',tpenv = tc_body_expr cenv ty env tpenv e1 in 
          if (get_ctorShapeCounter env) <> 1 then 
              errorR(Error("the expression form 'expr then expr' may only be used as part of an explicit object constructor",m));
          let e2',tpenv = tc_nonbody_stmt cenv (clear_ctorPreConstruct env) tpenv e2 in 
          TExpr_seq(e1',e2',ThenDoSeq,m),tpenv

    | Expr_cond (e1,e2,e3opt,m) ->
        let e1',tpenv = tc_nonbody_expr cenv cenv.g.bool_ty env tpenv e1 in 
        (if e3opt = None then unifyE cenv env m ty cenv.g.unit_ty);
        let e2',tpenv = tc_body_expr cenv ty env tpenv e2 in 
        let e3',tpenv = 
          match e3opt with 
          | None -> mk_unit cenv.g m,tpenv
          | Some e3 -> tc_body_expr cenv ty env tpenv e3 in 
        mk_cond m ty e1' e2' e3', tpenv

    (* This is for internal use in the libraries only *)
    | Expr_static_optimization (constraints,e2,e3,m) ->
        let constraints',tpenv = map_acc_list (tc_static_optimization_constraint cenv env) tpenv constraints in
        (* Do not force the types of the two expressions to be equal *)
        (* REVIEW: check the types are the same after applying the constraints *)
        let e2',_, tpenv = tc_expr_of_unknown_type cenv env tpenv e2 in 
        let e3',tpenv = tc_expr cenv ty env tpenv e3 in 
        TExpr_static_optimization(constraints',e2',e3',m), tpenv

    | Expr_lvalue_set (e1,f,e2,m) ->
        tc_expr_then cenv ty env tpenv e1 [Lvalue_get(f,m); mk_Lvalue_set(e2,m)]

    | Expr_lvalue_indexed_set (e1,f,e2,e3,m) ->
        tc_expr_then cenv ty env tpenv e1 [Lvalue_get(f,m); App(e2,m); mk_Lvalue_set(e3,m)]

    | Expr_lid_set (lid,e2,m) -> 
        tc_lid_then cenv ty env tpenv m lid [ mk_Lvalue_set(e2,m) ]
    
    (* Type.Items(e1) <- e2 *)
    | Expr_lid_indexed_set (lid,e1,e2,m) ->
        tc_lid_then cenv ty env tpenv m lid [ App(e1,m); mk_Lvalue_set(e2,m) ]

    | Expr_trait_call(tps,memSpfn,args,m) ->
        let (TTrait(_,_,_,argtys,rty) as traitInfo),tpenv = tc_pseudo_member_spec cenv NewTyparsOK CheckCxs env tps  tpenv memSpfn m in 
        let args',tpenv = tc_exprs cenv env m tpenv argtys args in 
        type_must_support_trait (denv_of_tenv env) cenv.css m NoTrace traitInfo;
        unifyE cenv env m ty rty;      
        TExpr_op(TOp_trait_call(traitInfo), [], args', m), tpenv
          
    | Expr_typeof(sty,m) ->
        warning(Deprecated("Use 'typeof<_>' instead",m));
        let sty',tpenv = tc_type cenv NewTyparsOK CheckCxs env tpenv sty in 
        unifyE cenv env m ty cenv.g.system_Type_typ;      
        mk_call_typeof cenv.g m sty', tpenv

    | Expr_constr_field_get (e1,c,n,m) ->
        let e1',ty1,tpenv = tc_expr_of_unknown_type cenv env tpenv e1 in
        let mkf,ty2 = tc_constr_field cenv env ty1 m c n 
                          ((fun (a,b) n -> mk_uconstr_field_get(e1',a,b,n,m)),
                           (fun a n -> mk_exnconstr_field_get(e1',a,n,m))) in 
        unifyE cenv env m ty ty2;
        mkf n,tpenv

    | Expr_constr_field_set (e1,c,n,e2,m) ->
        unifyE cenv env m ty cenv.g.unit_ty;
        let e1',ty1,tpenv = tc_expr_of_unknown_type cenv env tpenv e1 in
        let mkf,ty2 = tc_constr_field cenv  env ty1 m c n
                          ((fun (a,b) n e2' -> 
                             if not (ucref_rfield_mutable cenv.g a n) then errorR(Error("this field is not mutable",m));
                             mk_uconstr_field_set(e1',a,b,n,e2',m)),
                           (fun a n e2' -> 
                             if not (ecref_rfield_mutable a n) then errorR(Error("this field is not mutable",m));
                             mk_exnconstr_field_set(e1',a,n,e2',m))) in
        let e2',tpenv = tc_expr cenv ty2 env tpenv e2 in
        mkf n e2',tpenv

    | Expr_asm (s,tyargs,args,rtys,m) ->
        let argtys = new_inference_typs cenv args in 
        let tyargs',tpenv = tc_types cenv NewTyparsOK CheckCxs env tpenv tyargs in 
        let args',tpenv = tc_exprs cenv env m tpenv argtys args in 
        let rtys',tpenv = tc_types cenv NewTyparsOK CheckCxs env tpenv rtys in 
        if length rtys' > 1 then error(Error("Only zero or one pushed items are permitted",m));
        unifyE cenv env m ty (if length rtys' = 1 then hd rtys' else cenv.g.unit_ty);
        mk_asm(Array.to_list s,tyargs',args',rtys',m),tpenv

    | Expr_quote (oper,raw,ast,m) ->
        callQualSink m (nenv_of_tenv env,Item_typs("<expr>",[ty]), denv_of_tenv env);
        tc_quotation_expr cenv ty env tpenv (oper,raw,ast,m)


    | Expr_hole ((tcInfoRef,spliceExprOpt),m) ->
        callQualSink m (nenv_of_tenv env,Item_typs("<expr>",[ty]), denv_of_tenv env);
        tc_quotation_hole cenv ty env tpenv (tcInfoRef,spliceExprOpt,m)

(* Check expr.[idx] *)
(* This is a little over complicated for my liking. Basically we want to intepret e1.[idx] as e1.Item(idx). *)
(* However it's not so simple as all that. First "Item" can have a different name according to an attribute in *)
(* .NET metadata. Next, we want to give good warning messages for F#'s "old" way of doing things for OCaml *)
(* compatibility. This means we manually typecheck 'e1' and look to see if it has a nominal type. We then *)
(* do the right thing in each case. *)
and tc_indexer_then cenv env ty m tpenv expr e1 idx delayed = 
    
    let e1',e1ty,tpenv = tc_expr_of_unknown_type cenv env tpenv e1 in
    
    (* Find the first type in the effective hierarchy that either has a DefaultMember attribute OR *)
    (* has a member called 'Item' *)
    let propName = 
        fold_primary_hierarchy_of_typ (fun typ acc -> 
            match acc with
            | None ->
                let isNominal = is_stripped_tyapp_typ typ in
                if isNominal then 
                    let tcref = tcref_of_stripped_typ typ in
                    tcref_bind_attrib cenv.g cenv.g.attrib_DefaultMemberAttribute tcref 
                             (function ([CustomElem_string (Some(msg)) ],_) -> Some msg | _ -> None)
                             (function (Attrib(_,[ TExpr_const (TConst_string(bytes),_,_) ],_))  -> Some(Bytes.unicode_bytes_as_string bytes) | _ -> None)
                 else
                    begin match all_pinfos_of_typ_in_scope (nenv_of_tenv env).eExtensionMembers (Some("Item"), DontIncludePrivate) IgnoreOverrides cenv.g cenv.amap m typ with
                    | [] -> None
                    | _ -> Some("Item")
                    end
             | _ -> acc)
          cenv.g cenv.amap m 
          e1ty
          None in

    let isNominal = is_stripped_tyapp_typ e1ty in
    
    
    (* NOTE: It looks like we only need to make "array" different on this codepath *)
    (* NOTE: Array lookups should map to ArrayGet nodes in quotation trees. This is not currently happening *)
    (*       because they map to one of two (.[]) operators (one for legacy non-nominal lookups and one for nominal lookups *)
    (* NOTE: String lookups should map to direct calls to the Chars indexer property *)
    
    (* REVIEW: we have to adjust this so the string and array go through the nominal codepath *)
    let isArrayOrString = isArrayTypeWithIndexer cenv.g e1ty or type_equiv cenv.g cenv.g.string_ty e1ty in

    if (isNominal || isSome propName) && not isArrayOrString then 

        let nm = 
            match propName with 
            | None -> "Item"
            | Some(nm) -> nm in 
        let delayed = 
            match expr with 
            | Expr_lbrack_get _ -> Lvalue_get([ident(nm,m)],m) :: App(Expr_paren(idx,m),m) :: delayed
            | Expr_lbrack_set(_,idx,e3,_) -> Lvalue_get([ident(nm,m)],m) :: App(Expr_paren(idx,m),m) :: mk_Lvalue_set(e3,m) :: delayed
            | _ -> error(InternalError("unreachable",m)) in
        propagate_then_tc_delayed cenv ty env tpenv m e1' e1ty delayed
    else
        (* OK, build an old fashioned constrained lookup *)
        let fnm,idxs = 
            match expr with 
            | Expr_lbrack_get(_,Expr_tuple ([_;_],_),_) -> lbrack_get2, [idx]
            | Expr_lbrack_get(_,Expr_tuple ([_;_;_],_),_) -> lbrack_get3, [idx]
            | Expr_lbrack_get _ -> lbrack_get, [idx]
            | Expr_lbrack_set(_,Expr_tuple ([_;_],_),e3,_) -> lbrack_set2, [idx;e3]
            | Expr_lbrack_set(_,Expr_tuple ([_;_;_],_),e3,_) -> lbrack_set3, [idx;e3]
            | Expr_lbrack_set(_,_,e3,_) -> lbrack_set, [idx;e3]
            | _ -> error(InternalError("unreachable",m)) in
        let path = ["Microsoft";"FSharp";"Core"] @ (if isArrayOrString then ["LanguagePrimitives";"IntrinsicFunctions"] else ["Operators"]) in 
        let operPath = (mksyn_lid_get m path (compileOpName fnm)) in
        let f,fty,tpenv = tc_expr_of_unknown_type cenv env tpenv operPath in 
        let dty,rty = unify_fun (Some m) cenv (denv_of_tenv env) m fty in 
        unifyE cenv env m dty e1ty; 
        let f' = build_app cenv f fty e1' m in
        let delayed = map (fun arg -> App(arg,m)) idxs @ delayed in
        propagate_then_tc_delayed cenv ty env tpenv m f' rty delayed



(* Check a 'new Type(args)' expression, also an 'inherits' declaration in an implicit or explicit class *)
and tc_new cenv env tpenv objTy superInit arg m =
    (* Handle the case 'new 'a()' *)
    if (is_typar_ty objTy) then begin 
      if superInit then error(Error("cannot inherit from a variable type",m));
      type_must_support_default_ctor (denv_of_tenv env) cenv.css m NoTrace objTy;
      begin match arg with 
      | Expr_const (Const_unit,_) -> ()
      | _ -> errorR(Error("Calls to object constructors on type parameters can not be given arguments",m))
      end;
      mk_call_create_instance cenv.g m objTy ,tpenv
    end else begin 
      if not (is_stripped_tyapp_typ objTy) then error(Error(sprintf "'%s' may only be used with named types" (if superInit then "inherit" else "new"),m));
      let item,rest = forceRaise (tc_tdef_ctor (denv_of_tenv env) cenv.g cenv.amap m objTy) in 
      tc_ctor_call false cenv env tpenv objTy objTy item superInit arg m (delay_rest rest m [])
    end 

(* Check an 'inherits' declaration in an implicit or explicit class *)
and tc_ctor_call isNaked cenv env tpenv typ objTy item superInit arg m delayed =
    let family = accessRightsOfEnv env in 
    let isSuperInit = (if superInit then CtorValUsedAsSuperInit else NormalValUse) in 

    if is_interface_typ objTy then 
      error(Error((if superInit then  "'inherit' may not be used on interface types. Consider implementing the interface by using 'interface ... with ... end' instead"
                   else            "'new' may not be used on interface types. Consider using an object expression '{ new ... with ... }' instead"),m));
    let tycon = (deref_tycon (tcref_of_stripped_typ objTy)) in 
    if not superInit && is_partially_implemented_tycon tycon then 
      error(Error("Instances of this type cannot be created since it has been marked 'abstract' or not all methods have been given implementations. Consider using an object expression '{ new ... with ... }' instead",m));
    match item with 
    | Item_ctor_group(methodName,minfos) ->
        if verbose then  dprintf2 "--> Item_ctor_group@%a\n" output_range m;
        let meths = map (fun minfo -> minfo,None) minfos in 
        if isNaked && type_feasibly_subsumes_type 0 cenv.g cenv.amap m cenv.g.system_IDisposable_typ NoCoerce objTy then
          warning(Error("It is recommended that objects that support the IDisposable interface are created using 'new Type(args)' rather than 'Type(args)' to indicate that resources may be owned by the generated value",m));
        tc_method_args_and_apps_then cenv env typ tpenv None [] m methodName family PossiblyMutates false meths isSuperInit [arg] delayed
    | Item_delegate_ctor typ ->
        tc_delegate_ctor_then cenv objTy env tpenv m typ arg delayed
    | _ -> error(Error(sprintf "'%s' may only be used to construct object types" (if superInit then "inherit" else "new"),m))

(* Check a 'new Type()' expression on a struct *)
and tc_default_struct_ctor_opt defaultStructCtorInfo cenv typ objTy env tpenv m arg delayed = 
    match defaultStructCtorInfo with 
    | None -> None
    | Some exprty -> 
          match arg with 
          | Expr_const (Const_unit,_) -> 
              if verbose then  dprintf2 "--> Expr_new (arg Expr_const)@%a\n" output_range m;
              let expr = mk_ilzero (m,exprty) in 
              unifyE cenv env m exprty objTy;
              Some(propagate_then_tc_delayed cenv typ env tpenv m expr exprty delayed)
          |  _ -> 
              None

(*-------------------------------------------------------------------------
!* tc_recd_construction
 *------------------------------------------------------------------------- *)
  
(* Check a record consutrction expression *)
and tc_recd_construction cenv ty env tpenv optOrigExpr objTy fldsList m =
    let tcref = tcref_of_stripped_typ objTy in 
    let tycon = deref_tycon tcref in 
    let tinst = Tastops.tinst_of_stripped_typ objTy in 
    unifyE cenv env m ty objTy;

    (* Types with implicit constructors can't use record or object syntax: all constructions must go through the implicit constructor *)
    if (tycon |> tcaug_of_tycon).tcaug_adhoc |> Namemap.exists_in_range_multi is_implicit_ctor_of_vref then 
        errorR(Error(sprintf "Constructors for the type '%s' must directly or indirectly call its implicit object constructor. Consider using a call to a constructor instead of a record expression" (display_name_of_tycon tycon),m));
                
    let fspecs = instance_rfields_of_tycon tycon in
    (* Type check the supplied bindings *)                        
    let fldsList,tpenv = 
        map_acc_list 
            (fun tpenv (fname,fexpr) -> 
              let fspec = try  List.find (fun fspec -> name_of_rfield fspec = fname) fspecs
                          with Not_found -> error (Error("The field '" ^ fname ^ "' has been been given a value, but is not present in the type '"^NicePrint.pretty_string_of_typ (denv_of_tenv env) objTy^"'",m)) in
              let (declaredTypars,ftinst,fty) = freshenPossibleForallTy cenv m TyparRigid (actual_typ_of_rfield tycon tinst fspec) in 
              let fieldExpr,tpenv = tc_expr cenv fty env tpenv fexpr in
              (* Polymorphic fields require generializeable expressions. *)
              if nonNil(declaredTypars) then (
              
                  (* Canonicalize constraints prior to generalization *)
                  let denv = denv_of_tenv env in 
                  canonicalizePartialInferenceProblem (cenv,denv,m) declaredTypars;

                  let freeInEnv = computeUngeneralizableTypars env in 
                  let generalizedTypars = computeGeneralizedTypars(cenv,denv,m,true,freeInEnv,false,CanGeneralizeConstrainedTypars,OptionalInline,Some(fieldExpr),declaredTypars,[]) in 
                  (fname,mk_tlambda m declaredTypars (fieldExpr,fty)), tpenv
              ) else  (
                  (fname,fieldExpr),tpenv)
              )
            tpenv
            fldsList in
    (* Add rebindings for unbound field when an "old value" is available *)
    let oldFldsList = 
      match optOrigExpr with
      | None -> []
      | Some (_,_,oldve') -> 
             (* When we have an "old" value, append bindings for the unbound fields. *)
             (* Effect order - mutable fields may get modified by other bindings... *)
             let fieldNameUnbound nom = for_all (fun (name,exp) -> name <> nom) fldsList in
             fspecs 
             |> filter (name_of_rfield >> fieldNameUnbound)
             |> map (fun fspec ->
                 (* CODE REVIEW: check next line is ok to be repeated *)
                 let (_,ftinst,fty) = freshenPossibleForallTy cenv m TyparRigid (actual_typ_of_rfield tycon tinst fspec) in 
                 name_of_rfield fspec, mk_recd_field_get cenv.g (oldve',rfref_of_rfield tcref fspec,tinst,ftinst,m)) in

    let fldsList = fldsList @ oldFldsList in

    (* From now on only interested in fspecs that truly need values. *)
    let fspecs = fspecs |> List.filter (zero_init_of_rfield >> not) in
    
    (* Check all fields are bound *)
    fspecs |> List.iter (fun fspec ->
      if not (exists (fun (fname,fexpr) -> fname = name_of_rfield fspec) fldsList) then
        error(Error("no assignment given for field '"^fspec.rfield_id.idText^"'",m)));

    (* Other checks (overlap with above check now clear) *)
    if isNone optOrigExpr then begin
      let ns1 = nameset_of_list (map fst fldsList) in 
      let ns2 = nameset_of_list (map (fun x -> x.rfield_id.idText) fspecs) in 
      if  not (Zset.subset ns2 ns1) then 
        error (MissingFields(Zset.elements (Zset.diff ns2 ns1),m));
      if  not (Zset.subset ns1 ns2) then 
        error (Error("Extraneous fields have been given values",m));
    end;
    (* Build record *)
    let rfrefs = map (fst >> mk_rfref tcref) fldsList in

    (* Check accessibility: this is also done in gen_build_fldmap, but also need to check *)
    (* for fields in { new R with a=1 and b=2 } constructions and { r with a=1 }  copy-and-update expressions *)
    rfrefs |> List.iter (fun rfref -> 
        rfref_accessible_check m (accessRightsOfEnv env) rfref;
        fs_attrib_check cenv.g (pattribs_of_rfref rfref) m |> commitOperationResult);        

    let args   = map snd fldsList in
    
    let expr = mk_recd (get_recdInfo env, tcref, tinst, rfrefs, args, m) in 

    let expr = 
      match optOrigExpr with 
      | None ->
          (* '{ recd fields }'. *)
          expr
          
      | Some (old',oldv',_) -> 
          (* '{ recd with fields }'. *)
          (* Assign the first object to a tmp and then construct *)
          mk_let m oldv' old' expr in

    expr, tpenv

(*-------------------------------------------------------------------------
!* tc_object_expr
 *------------------------------------------------------------------------- *)


and tc_object_expr cenv ty env tpenv (objTy,argopt,binds,iimpls,m) = 
    let objTy',tpenv = tc_type cenv NewTyparsOK CheckCxs env tpenv objTy in
    if not (is_stripped_tyapp_typ objTy') then error(Error("'new' must be used with a named type",m));
    if not (is_recd_ty objTy') && not (is_interface_typ objTy') && is_sealed_typ cenv.g objTy' then errorR(Error("Cannot create an extension of a sealed type",m));
    (* Object expression members can access protected members of the implemented type *)
    let env = enter_familyRegion (tcref_of_stripped_typ objTy') env in 
    
    if (* record construction *)
       (is_recd_ty objTy') || 
       (* object construction *)
       (is_fsobjmodel_ty objTy' && not (is_interface_typ objTy') && argopt = None) then begin 
      if argopt <> None then error(Error("No arguments may be given when constructing a record value",m));
      if iimpls <> [] then error(Error("Interface implementations may not be given on construction expressions",m));
      if is_fsobjmodel_ty objTy' && get_ctorShapeCounter env <> 1 then error(Error("Object construction expressions may only be used to implement constructors in class types",m));
      let fldsList = 
        List.map
          (fun b -> 
            match normBinding ObjExprBinding cenv env b with 
            | NormBinding (_,_,_,[],_,_,_,Pat_as(Pat_wild _, id,_,_,_),BindingExpr(_,_,e),_) -> id.idText,e
            | _ -> error(Error("Only simple bindings of the form 'id = expr' can be used in construction expressions",m)))
          binds in
      
       tc_recd_construction cenv ty env tpenv None objTy' fldsList m
    end else
      let item,rest = forceRaise (tc_tdef_ctor (denv_of_tenv env) cenv.g cenv.amap m objTy') in 
      if rest <> [] then error(InternalError("unexpected rest from tc_tdef_ctor",m));
      if is_fsobjmodel_ty objTy' && get_ctorShapeCounter env = 1 then error(Error("Objects must be initialized by an object construction expression that calls an inherited object constructor and assigns a value to each field",m));
      unifyE cenv env m ty objTy';
      let ctor_call,baseIdOpt,tpenv =
        begin match item,argopt with 
        | Item_ctor_group(methodName,minfos),Some (arg,baseIdOpt) -> 
            let meths = map (fun minfo -> minfo,None) minfos in 
            let ad = accessRightsOfEnv env in 
            let expr,tpenv = tc_method_args_and_apps_then cenv env ty tpenv None [] m methodName ad PossiblyMutates false meths CtorValUsedAsSuperInit [arg] [] in
            expr,
            baseIdOpt,
            tpenv
        | Item_fake_intf_ctor ityp,None -> 
            unifyE cenv env m ty ityp;
            let expr = mk_obj_ctor_call cenv.g m in 
            expr,None,tpenv
        | Item_fake_intf_ctor _,Some _ -> 
            error(Error("Constructor expressions for interfaces do not take arguments",m));
        | Item_ctor_group _,None -> 
            error(Error("This object constructor requires arguments",m));
        | _ -> error(Error("'new' may only be used with object constructors",m))
        end in 

      let baseVarOpt = mkAndPublishBaseVal cenv env baseIdOpt ty in
      let env = Option.fold_right (addLocalVal m) baseVarOpt env in 
      
    (* Work out the type of any interfaces to implement *)
      let iimpls',tpenv = 
        map_acc_list 
          (fun tpenv (InterfaceImpl(ity,overrides,m)) -> 
            let ity',tpenv = tc_type cenv NewTyparsOK CheckCxs env tpenv ity in 
            if not (is_interface_typ ity') then
              error(Error("Expected an interface type",m));
            (m,ity',overrides),tpenv)
          tpenv 
          iimpls in
      
      let impls = (m,ty,binds) :: iimpls' in 
      
      
      (* 1. collect all the relevant abstract slots for each type we have to implement *)
      let implTySets = getImplSets (denv_of_tenv env) cenv.g cenv.amap (map (fun (m,ty,_) -> ty,m) impls) in

      let allImpls = map2 (fun (m,ty,binds) implTySet -> m, ty,binds,implTySet) impls implTySets in 
      
      let overridesAndVirts,tpenv = 
          map_acc_list 
              (fun tpenv (m,implty,binds, SlotImplSet(virts,availPriorImplSlots,_,_) ) ->
                  
                  (* 2. collect all name/arity of all overrides *)
                  let virtNameAndArityPairs = map (fun virt -> ((name_of_minfo virt,arity_of_minfo cenv.g virt),virt)) virts in
                  let bindNameAndArityPairs = 
                      let rec lookBinding b sofar = 
                        match normBinding ObjExprBinding cenv env b with 
                        | NormBinding (_,_,_,attrs,typars,_,memberInfo,p,e,bindm) -> lookPat memberInfo p e sofar 
                      and lookPat memberInfo p e sofar =
                        match p,memberInfo with 
                        | Pat_typed(pat',_,_),_ -> lookPat memberInfo pat' e sofar 
                        | Pat_paren(pat',_),_ -> lookPat memberInfo pat' e sofar 
                        | Pat_attrib(pat',_,_),_ -> lookPat memberInfo pat' e sofar 
                        | Pat_as (Pat_wild _, id,_,None,_),None -> 
                             (* let e = push_one_pat_rhs (mksyn_this_pat_var (ident "_this" id.idRange)) e in  *)
                             lookRhsExpr id e sofar
                        | Pat_instance_member(thisn, memberName,None,m),Some(memFlags,_,_) ->
                             let logicalMethName = ident (computeLogicalCompiledName memberName memFlags,memberName.idRange) in
                             lookRhsExpr logicalMethName e sofar

                        | _ -> errorR(Error("Only overrides of abstract and virtual members can be specified in object expressions",m)); sofar
                      and lookRhsExpr id (BindingExpr(pushedPats,_,_)) sofar =
                        let nargs = 
                          match pushedPats with 
                          | SPats([],_) :: _ -> 0
                          | SPats (args,_)  :: _-> length args
                          | _ -> 1 in
                        (id.idText,nargs)::sofar in
                      fold_right lookBinding binds [] in 

                  (* 3. infer must-have types by name/arity *)
                  let preAssignedVirtsPerBinding = 
                      bindNameAndArityPairs |> map (fun bkey -> 
                          (* first attempt to fix bug 635. But more work is required in checkOverridesAreAllUsedOnce and checkAbstractMembersAreImplemented *)
                          (* to impedence match the expected number of arguments. *)
                          (* if length (filter (orderOn fst (=) bkey) bindNameAndArityPairs) = 1 
                          then filter (fst >> orderOn fst (=) bkey) virtNameAndArityPairs
                          else *) 
                          if length (filter ((=) bkey) bindNameAndArityPairs) = 1 
                          then filter (fst >> (=) bkey) virtNameAndArityPairs
                          else [])  in 
                  
                  let bindtys = 

                      let rec loop  l1 l2 l3 = 
                          match l1,l2,l3 with 
                          | bind::t1,(bindnm,_)::t2,absSlots::t3 -> 
                              let res = 
                                  let fail() = false,[],new_inference_typ cenv () in 
                                  begin match absSlots with 
                                  | [] -> 
                                      let absSlotsByName = filter (fst >> fst >> (=) bindnm) virtNameAndArityPairs in 
                                      begin match absSlotsByName with 
                                      | []              -> errorR(Error("The member "^bindnm^" does not correspond to any abstract or virtual method available to override or implement",m));
                                      | [(_,absSlot)]     -> errorR(Error("The member "^bindnm^" does not accept the correct number of arguments, "^string_of_int (arity_of_minfo cenv.g absSlot)^" arguments are expected",m));
                                      | (_,absSlot) :: _  -> errorR(Error("The member "^bindnm^" does not accept the correct number of arguments. One overload accepts "^string_of_int (arity_of_minfo cenv.g absSlot)^" arguments",m));
                                      end;
                                      fail()
                                  | [(_,absSlot)] -> 
                                      (* dprintf2 "nm = %s, #fmtps = %d\n" (name_of_minfo absSlot) (length fmtps); *) 
                                      
                                      let synTyparDecls = 
                                          match normBinding ObjExprBinding cenv env bind with 
                                          | NormBinding (_,_,_,_,_,synTyparDecls,_,_,_,_) -> synTyparDecls in 

                                      let typarsFromAbsSlotAreRigid,typarsFromAbsSlot,argTysFromAbsSlot, retTyFromAbsSlot
                                         = freshenAbstractSlot cenv.g cenv.amap m synTyparDecls absSlot in

                                      (* Work out the required type of the member *)
                                      let memberTyFromAbsSlot = implty --> (mk_meth_ty cenv.g argTysFromAbsSlot retTyFromAbsSlot)  in 
                                      
                                      typarsFromAbsSlotAreRigid,typarsFromAbsSlot,memberTyFromAbsSlot
                                      
                                  | (_,absSlot1) :: (_,absSlot2) :: _ -> 
                                      warning(NonUniqueInferredAbstractSlot(cenv.g,denv_of_tenv env, bindnm, absSlot1, absSlot2,m));
                                      fail()
                                  end in 
                              res :: loop t1 t2 t3 
                              
                          | _ -> [] in 
                      loop binds bindNameAndArityPairs preAssignedVirtsPerBinding in 
                  
                  (* 4. typecheck/typeinfer overrides using this information *)
                  let overrides,tpenv = 
                      (combine bindtys binds) |> map_acc_list (fun tpenv ((typarsFromAbsSlotAreRigid,typarsFromAbsSlot,memberTyFromAbsSlot),bind) -> 

                          (* 4a1. normalize the binding (note: needlessly repeating what we've done above) *)
                          let (NormBinding(vis,pseudo,mut,attrs,doc,synTyparDecls,memberInfo,p,(BindingExpr(pushedPats,pushedRtyOpt,e) as rhs),bindm)) = 
                              (normBinding ObjExprBinding cenv env bind) in

                          (* 4a2. adjust the binding, especially in the "member" case, a subset of the logic of mk_rec_value *)
                          let bind = 
                              let rhs,logicalMethName = 
                                  let rec pat p = 
                                      match p,memberInfo with  
                                      | Pat_as (Pat_wild _, id,_,_,_),None -> 
                                             let rhs = push_one_pat_rhs true (mksyn_this_pat_var (ident ("_this",id.idRange))) rhs  in 
                                             let logicalMethName = id in
                                             rhs,logicalMethName
                                      | Pat_instance_member(thisn, memberName,_,_),Some(memFlags,_,_) -> 
                                             checkMemberFlags cenv.g None  NewSlotsOK OverridesOK memFlags bindm;
                                             let rhs = push_one_pat_rhs true (mksyn_this_pat_var thisn) rhs in 
                                             let logicalMethName = ident (computeLogicalCompiledName memberName memFlags,memberName.idRange) in
                                             rhs,logicalMethName 
                                      | _ -> 
                                          error(InternalError("unexpect member binding",m)) in
                                  pat p in
                              NormBinding (vis,pseudo,mut,attrs,doc,synTyparDecls,memberInfo,mksyn_pat_var vis logicalMethName,rhs,bindm)  in 
                          
                            (* 4b. typecheck the binding *)
                            
                            let (TBindingInfo(inlineFlag,immut,_,_,_,TIFlex(declaredTypars,_),nameToPrelimValSchemeMap,rhs,_,_,m,_,_),tpenv) = 
                                let flex = tc_nonrec_binding_typar_decls cenv env tpenv bind in
                                tc_norm_binding ObjectExpressionOverrideBinding cenv env tpenv memberTyFromAbsSlot None ([],flex) bind in

                            (* 4c. generalize the binding - only relevant when implementing a generic virtual method *)
                            
                            match Namemap.range nameToPrelimValSchemeMap with 
                            | [PrelimValScheme1(id,_,_,_,_,_,_,_,_,_)] -> 
                                let denv = denv_of_tenv env in

                                let declaredTypars = (if typarsFromAbsSlotAreRigid then typarsFromAbsSlot else declaredTypars) in 
                                (* Canonicalize constraints prior to generalization *)
                                canonicalizePartialInferenceProblem (cenv,denv,m) declaredTypars;

                                let freeInEnv = computeUngeneralizableTypars env in 

                                (* dprintf4 "id = %s, typarsFromAbsSlotAreRigid = %b, #declaredTypars = %d, #freeInEnv = %d\n" id.idText typarsFromAbsSlotAreRigid (length declaredTypars) (length (Zset.elements freeInEnv)); *) 

                                let generalizedTypars = computeGeneralizedTypars(cenv,denv,m,immut,freeInEnv,false,CanGeneralizeConstrainedTypars,inlineFlag,Some(rhs),declaredTypars,[]) in
                                let declaredTypars = chooseCanonicalDeclaredTyparsAfterInference cenv.g  (denv_of_tenv env) declaredTypars m in 

                                (* dprintf1 "ungen = %b\n" (declaredTypars |> List.exists (Zset.mem_of freeInEnv)); *) 

                                let generalizedTypars = placeTyparsInDeclarationOrder declaredTypars generalizedTypars  id.idRange in 
                                (* dprintf1 "#generalizedTypars = %d\n" (length generalizedTypars); *) 
                                (id,(generalizedTypars +-> memberTyFromAbsSlot),rhs),tpenv
                            | _ -> error(Error("A simple method name is required here",m)))
                        tpenv in 
                (m,implty,virts,availPriorImplSlots,overrides),tpenv)
            tpenv
            allImpls in 
      
      overridesAndVirts |> iter (fun (m,implty,virts,availPriorImplSlots,overrides) -> 
          let ovspecs = map (dest_objexpr_override cenv.g cenv.amap >> fst) overrides in 
          checkOverridesAreAllUsedOnce (denv_of_tenv env) cenv.g cenv.amap (m,implty,virts,ovspecs);
          checkAbstractMembersAreImplemented true (denv_of_tenv env) cenv.g cenv.amap (m,implty,virts,availPriorImplSlots,ovspecs) |> ignore);
      
    (* 6c. create the specs of overrides *)
      let all_iimpls = 
        overridesAndVirts |> map (fun (m,implty,virts,_,overrides) -> 
            let overrides' = overrides |> map (fun ((v,ty,e) as overrideMeth) -> 
                  let (Override(id,(mtps,_),argtys,rty) as ovinfo),(_,thisv,vs,body) = dest_objexpr_override cenv.g cenv.amap overrideMeth in 
                  let overridden = 
                      match tryfind (fun virt -> is_exact_match cenv.g cenv.amap m virt ovinfo) virts with 
                      | Some x -> x
                      | None -> error(Error("At least one override did not correctly implement its corresponding abstract member",range_of_syntype objTy)) in
                  let method_vs,body' = adjust_arity_of_lambda_body cenv.g cenv.niceNameGen (length vs) vs body in   
                  TMethod(slotsig_of_minfo cenv.g cenv.amap m overridden, mtps, thisv::method_vs,body',id.idRange)) in
            (implty,overrides'))
           in 
      let (ty,overrides') = hd all_iimpls in 
      let iimpls' = tl all_iimpls in 
      
    (* 7. Build the implementation *)
      TExpr_obj(new_uniq(), ty, baseVarOpt, ctor_call, overrides',iimpls',m,new_cache()),
      tpenv



(*-------------------------------------------------------------------------
!* tc_const_string_expr
 *------------------------------------------------------------------------- *)

(* Check a constant string expression. It might be a 'printf' format string *)
and tc_const_string_expr cenv ty env m tpenv s  =

    if (unify_and_undo_if_failed (denv_of_tenv env) cenv.css m ty cenv.g.string_ty) then 
      mk_string cenv.g m s,tpenv
    else 
      let aty = new_inference_typ cenv () in 
      let bty = new_inference_typ cenv () in 
      let cty = new_inference_typ cenv () in
      let dty = new_inference_typ cenv () in
      let ety = new_inference_typ cenv () in
      let ty' = mk_format_ty cenv.g aty bty cty dty ety in
      if (type_must_subsume_type_and_undo_if_failed (denv_of_tenv env) cenv.css m ty ty') then begin
        (* Parse the format string to work out the phantom types *)
        let aty',ety' = (try Formats.parse_format cenv.amap m cenv.g s bty cty dty with Failure s -> error (Error(s,m))) in
        unifyE cenv env m aty aty';
        unifyE cenv env m ety ety';
        mk_call_new_format cenv.g m aty bty cty dty ety (mk_string cenv.g m s),tpenv
      end else begin
        unifyE cenv env m ty cenv.g.string_ty;
        mk_string cenv.g m s,tpenv
      end

(*-------------------------------------------------------------------------
!* tc_const_expr
 *------------------------------------------------------------------------- *)

(* Check a constant expression. *)
and tc_const_expr cenv ty env m tpenv c  =
    match c with 

    (* NOTE: these aren't "really" constants *)
    | Const_bytearray (bytes,m) -> 
       unifyE cenv env m ty (mk_bytearray_ty cenv.g); TExpr_op(TOp_bytes bytes,[],[],m),tpenv

    | _ -> 
        let c' = tc_const cenv ty m env c in 
        TExpr_const (c',m,ty),tpenv


(*-------------------------------------------------------------------------
!* tc_assert_expr
 *------------------------------------------------------------------------- *)

(* Check an 'assert(x)' expression. *)
and tc_assert_expr cenv ty env m tpenv x  =
    let callDiagnosticsExpr = Expr_app(mksyn_lid_get m ["System";"Diagnostics";"Debug"] "Assert", 
                                       (* wrap an extra parentheses so 'assert(x=1) isn't considered a named argument to a method call *)
                                       Expr_paren(x,m), m) in

    let expr2 = 
        let rec synexpr_is_false e = 
          match e with  
          | Expr_const(Const_bool false,_) -> true
          | Expr_paren(e,_) -> synexpr_is_false e
          | _ -> false in 

        if synexpr_is_false x then 
            Expr_seq(true,callDiagnosticsExpr,
                     Expr_app(mksyn_mod_item m "Operators" "raise", 
                              Expr_app(mksyn_item m "AssertionFailure",
                              (* REVIEW: use unicode names for files and get rid of this use of Bytes.string_as_unicode_bytes *)
                                       Expr_tuple([ Expr_const(Const_string (Bytes.string_as_unicode_bytes (file_of_range m),m), m); 
                                                    Expr_const(Const_int32 (Int32.of_int (m |> start_of_range |> line_of_pos)), m); 
                                                    Expr_const(Const_int32 (Int32.of_int (m |> start_of_range |> col_of_pos)), m) ], m), m),m),m)
        else callDiagnosticsExpr in
    tc_expr cenv ty env tpenv expr2



(*-------------------------------------------------------------------------
!* tc_recd_expr
 *------------------------------------------------------------------------- *)

and tc_recd_expr cenv ty env tpenv (inherits,opt_old,flds,m) =

    let req_ctor = (get_ctorShapeCounter env = 1) (* Hacky way to get special expression forms for constructors *) in 
    let have_ctor = (isSome inherits) in 

    let optOrigExpr,tpenv = 
      match opt_old with 
      | None -> None, tpenv 
      | Some e -> 
          if isSome inherits then error(Error("Invalid record construction",m));
          let e',tpenv = tc_expr cenv ty env tpenv e in 
          let v',ve' = Tastops.mk_compgen_local m "old" ty in 
          Some (e',v',ve'), tpenv in

    let fldsList = 
        match flds with 
        | [] -> []
        | _ -> 
            let tcref,fldsmap,fldsList = gen_build_fldmap cenv env ty flds m in 
            let _,_,_,gtyp = info_of_tcref cenv m env tcref in 
            unifyE cenv env m ty gtyp;      
            fldsList in

    if isSome optOrigExpr && not (is_recd_ty ty) then 
        errorR(Error("The expression form { expr with ... } may only be used with record types. To build object types use { new Type(...) with ... }",m));

    if req_ctor || have_ctor then (
        if not (is_fsobjmodel_ty ty) then errorR(Error("The inherited type is not an object model type",m));
        if not req_ctor then errorR(Error("Object construction expressions (i.e. record expressions with inheritance specifications) may only be used to implement constructors in object model types. Use 'new ObjectType(args)' to construct instances of object model types outside of constructors",m));
    ) else (
        if isNil flds then error(Error("'{ }' is not a valid expression. Records must include at least one field. Empty sequences are specified by using Seq.empty or an empty list '[]'",m));
        if is_fsobjmodel_ty ty then errorR(Error("This type is not a record type. Values of class and struct types must be created using calls to object constructors",m))
        else if not (is_recd_ty ty) then errorR(Error("This type is not a record type",m));
    );

    let super',tpenv = 
        match inherits, super_of_typ cenv.g cenv.amap m ty with 
        | Some (superTyp,arg,m), Some(realSuperTyp) ->
            (* Constructor expression, with an explicit 'inherits' clause. Check the inherits clause. *)
            let e,tpenv = tc_expr cenv realSuperTyp  env tpenv (Expr_new(true,superTyp,arg,m)) in 
            Some(e),tpenv
        | None, Some(realSuperTyp) when req_ctor -> 
            (* Constructor expression, No 'inherits' clause, hence look for a default constructor *)
            let e,tpenv = tc_new cenv env tpenv realSuperTyp true (Expr_const (Const_unit,m)) m in 
            Some(e),tpenv
        | None,_ -> 
            None,tpenv
        | _, None -> 
            errorR(InternalError("unexpected failure in getting super type",m));
            None,tpenv in 

    let expr,tpenv = 
        tc_recd_construction cenv ty env tpenv optOrigExpr  ty fldsList m in 

    let expr = 
        match super' with 
        | _ when is_struct_typ ty -> expr
        | Some(e) -> mk_seq m e expr
        | None -> expr in 
    expr,tpenv


(*-------------------------------------------------------------------------
!* tc_foreach_expr 
 *------------------------------------------------------------------------- *)
 
and tc_foreach_expr cenv ty env tpenv (pat,enumExpr,body,m)  =
    unifyE cenv env m ty cenv.g.unit_ty;
    let enumExpr',iety,tpenv = 
        let env = exit_familyRegion env in
        tc_expr_of_unknown_type cenv env tpenv enumExpr in 
    let ienumeratorv, ienumeratore,_,enumElemTy,getEnumE,getEnumTy,guarde,_,currente = tc_ienumerator_exprs_for_arb_typed_expr cenv env (range_of_expr enumExpr') iety enumExpr' in
    let pat',_,vspecs,envinner,tpenv = tc_match_pattern cenv enumElemTy env tpenv (pat,None) in 
    let idv,ide,pat'' =      
        (* nice: don't introduce awful temporary for r.h.s. in the 99% case where we know what we're binding it to *)
        match pat' with
        | TPat_as (pat1,PBind(v,TypeScheme([],_,_)),m1) -> 
              v,expr_for_val (range_of_val v) v, pat1
        | _ -> 
              let tmp,tmpe = Tastops.mk_compgen_local m (cenv.niceNameGen.nngApply "$for_var" m) enumElemTy in 
              tmp,tmpe,pat' in 

(*      let idv,ide = Tastops.mk_local id.idRange id.idText enumElemTy in *)
(*      let envinner = addLocalVal m idv env in  *)
    let body',tpenv = tc_stmt cenv envinner tpenv body in
    let body' = 
        let valsDefinedByMatching = gen_remove vspec_eq idv vspecs in
        compilePatternForMatch cenv env m m false Incomplete (idv,[]) 
            [TClause(pat'',None,TTarget(valsDefinedByMatching,body'),m);
             TClause(TPat_wild(range_of_expr body'),None,TTarget([],mk_unit cenv.g m),m) ] ty in 
    let for_loop =  
        match enumExpr' with 
        (* optimize 'for i in n .. m do' *)
        | TExpr_app(TExpr_val(vf,_,_),_,[tyarg],[start';finish'],_) 
             when cenv.g.vref_eq vf cenv.g.range_op_vref && type_equiv cenv.g tyarg cenv.g.int_ty -> 
               mk_fast_for_loop  cenv.g m idv start' true finish' body'
        | _ -> 

            let cleanupE = buildDisposeCleanup cenv env m ienumeratorv in
            mk_let (range_of_expr getEnumE)  ienumeratorv getEnumE (
            mk_try_finally cenv.g (mk_while cenv.g (guarde,mk_let (range_of_expr body') idv currente body',m),cleanupE,m,cenv.g.unit_ty)) in
    for_loop, tpenv

(*-------------------------------------------------------------------------
!* tc_quotation_expr
 *------------------------------------------------------------------------- *)

and tc_quotation_expr cenv ty env tpenv (oper,raw,ast,m) =
    let env = exit_familyRegion env in
    let oper,operty,tpenv = tc_expr_of_unknown_type cenv env tpenv oper in 
    if verbose then dprintf1 "checking operator '%s' is a function\n" ((DebugPrint.showExpr oper));
    let operdty,operrty = unify_fun None cenv (denv_of_tenv env) m operty in 
    let holes = holes_of_synexpr ast [] in 


    if verbose then dprintf1 "checking domain type '%s' of operator\n" ((DebugPrint.showType operdty));
    if List.exists (snd >> isSome) holes && List.exists (snd >> isNone) holes then error(Error("This quotation has both holes filled by arguments and holes filled by splices. This is not permitted",m));
    let nholes = length holes in
    let astty = new_inference_typ cenv () in 
    let holetys = map (fun _ -> new_inference_typ cenv ()) holes  in 
    let eholetys = 
        if raw then (
            let eholetys = map (fun _ -> mk_raw_expr_ty cenv.g) holetys in 
            let rawrty = mk_raw_expr_ty cenv.g in 
            let rawfty = mk_iterated_fun_ty eholetys rawrty in 
            let rawtupty = mk_tupled_ty cenv.g eholetys in 
            unifyE cenv env m operdty (mk_raw_expr_template_ty cenv.g rawtupty rawfty);      
            eholetys
        ) else (
            let eholetys = map (mk_expr_ty cenv.g) holetys in 
            let rty = mk_expr_ty cenv.g astty in 
            let fty = mk_iterated_fun_ty eholetys rty in 
            let tupty = mk_tupled_ty cenv.g eholetys in 
            unifyE cenv env m operdty (mk_expr_template_ty cenv.g astty tupty fty rty);      
            eholetys
        ) in

    let tcInfos = map2 (fun hty ehty -> (hty,ehty, ref None) ) holetys eholetys in 
    iter2 (fun (tcInfoOptRef,_) info  -> tcInfoOptRef :=  Some (Obj.repr (info : Tast.typ * Tast.typ * Tast.expr option ref))) holes tcInfos;

    (* Push the known type information *)
    if verbose then dprintf1 "applying types,  operrty = %s\n" ((DebugPrint.showType operrty));
    let rec apply opty eholetys holes = 
       match eholetys,holes with
       | (ehty::eholetys'),((_,exprSpliceOpt)::holes') -> 
           begin match exprSpliceOpt with 
           | None -> apply opty eholetys' holes' 
           | Some _ -> 
               if verbose then dprintf1 "applying splice type,  ehty = %s\n" ((DebugPrint.showType ehty));
               let dty,rty = unify_fun None cenv (denv_of_tenv env) m opty in 
               unifyE cenv env m dty ehty;
               apply rty eholetys' holes' 
           end;
       | _ -> 
           (* OK, were' done with the args, now assert the return type *)
           unifyE cenv env m ty opty in 

    apply operrty eholetys holes;

    let ast',tpenv = tc_expr cenv astty env tpenv ast   in
            (* Fill in the remaining phantom types *)
            (*
            let holetys2 = map (mk_expr_ty cenv.g) (List.rev (foldExpr { exprFolder0 with exprIntercept = (fun _ z e -> match e with TExpr_hole (m,ty) -> Some(ty :: z) | _ -> None) } [] ast')) in
            let fty2 = mk_iterated_fun_ty holetys2 rty in 
            unifyE cenv env m fty fty2;      
            let tupty2 = mk_tupled_ty cenv.g holetys2 in 
            unifyE cenv env m tupty tupty2;       
            ast',tpenv *)
    if verbose then dprintf1 "making quotation,  operdty = %s\n" ((DebugPrint.showType operdty));
    let expr = TExpr_quote(raw,ast',m,operdty) in 
    if verbose then dprintf1 "mk_appl oper ast, operty = %s\n" ((DebugPrint.showType operty));
    let expr = mk_appl((oper,operty),[],[expr],m) in
    if verbose then dprintf0 "building result\n";
    (* Apply any spliced arguments *)
    let expr = 
        fold_left (fun expr (hty,ehty,exprOptionRef) -> 
                     match !exprOptionRef with 
                     | None -> expr
                     | Some e -> mk_appl((expr,type_of_expr cenv.g expr), [],[e],m))
             expr
             tcInfos in
    if verbose then dprintf1 "done, ty = %s\n" ((DebugPrint.showType ty));
    (* We serialize the quoted expression to bytes in Ilxgen after type inference *)
    (* etc. is complete. Here we wrap the deserialization call around the bytes. *)
    expr,tpenv

(*-------------------------------------------------------------------------
!* tc_quotation_hole
 *------------------------------------------------------------------------- *)

and tc_quotation_hole cenv ty env tpenv (tcInfoRef,spliceExprOpt,m) =
    let tpenv = 
        match !tcInfoRef with 
        | None -> tpenv
        | Some tcInfo -> 
            let (holeTy, eholeTy, holeExprRef) = (Obj.obj tcInfo : Tast.typ * Tast.typ * Tast.expr option ref) in 
            unifyE cenv env m ty holeTy; 
            match spliceExprOpt with 
            | None -> 
                tpenv
            | Some expr -> 
                let expr,tpenv = tc_expr cenv eholeTy env tpenv expr in 
                holeExprRef := Some expr;
                tpenv in 
    TExpr_hole(m,ty),tpenv


(*-------------------------------------------------------------------------
!* tc_comprehension
 *------------------------------------------------------------------------- *)

and tc_comprehension cenv env ty m interpValOpt tpenv comp = 

    let findNonOverloadedMethod m interpExprTy methName =
        match try_find_minfo cenv.g cenv.amap m methName interpExprTy with 
        | [] -> 
            error(Error("No method called '"^methName^"' was found on the type '"^NicePrint.pretty_string_of_typ (denv_of_tenv env) interpExprTy^"'",m));
  
        | [methInfo] -> 
            methInfo

        | _  -> 
            error(Error("The method called '"^methName^"' was found on the type '"^NicePrint.pretty_string_of_typ (denv_of_tenv env) interpExprTy^"' but is overloaded. Members implementing computation expressions may not be overloaded",m)) in
            

    let findMethod m env interpExprTy methName genOuterTy =
        let methInfo = findNonOverloadedMethod m interpExprTy methName in
        let methInfoInst = freshen_minfo cenv m methInfo in
        unifyE cenv env m (ret_typ_of_minfo cenv.g cenv.amap m methInfo methInfoInst) genOuterTy;
        (methInfo, methInfoInst), params_of_minfo cenv.g cenv.amap m methInfo methInfoInst in
            


    match interpValOpt with 
    | Some (interpExpr,interpExprTy) -> 

        let mkDelayedExpr coreExpr = 
            let m = range_of_expr coreExpr in 
            let ty = type_of_expr cenv.g coreExpr in 
            let ad = accessRightsOfEnv env in
            let resultMethInfo = findNonOverloadedMethod m interpExprTy "Delay" in
            let resultMethInfoInst = freshen_minfo cenv m resultMethInfo in
            begin match params_of_minfo cenv.g cenv.amap m resultMethInfo resultMethInfoInst with 
            | [(_,pty)] -> unifyE cenv env m pty (cenv.g.unit_ty -->  ty)
            | _ -> error(Error("A 'Delay' method was found on the type '"^NicePrint.pretty_string_of_typ (denv_of_tenv env) interpExprTy^"' but it had the wrong signature. The method should take one argument of tye (unit -> M) where M is the type of the computation",m));
            end;
            let resultE,resultTy  = buildMethodCall cenv env PossiblyMutates m false resultMethInfo NormalValUse resultMethInfoInst [interpExpr] [mk_unit_delay_lambda cenv.g m coreExpr] in
            resultE in 

        let interpVarName = "_builder"^string_of_int(new_unit_uniq()) in 
        let interpVarRange = range_of_expr interpExpr in 
        let interpVar = mksyn_item interpVarRange interpVarName in 

        let mksynCall nm m args = 
            let args = 
                match args with 
                | [] -> Expr_const(Const_unit,m)
                | [arg] -> Expr_paren(Expr_paren(arg,m),m)
                | args -> Expr_paren(Expr_tuple(args,m),m) in
                
            Expr_app (Expr_lvalue_get(interpVar,[mksyn_id m nm], m), args,m) in


        let rec trans comp =
            match comp with 
            | Comp_for (seqExprOnly,pat,pseudoEnumExpr,innerComp) -> 
                let m = (range_of_synexpr pseudoEnumExpr) in
                if seqExprOnly then warning (Error("'for v in E ->' and 'for v in E' clauses should only be used inside compact sequence expressions. Consider using 'for v in E do ...' in computation expressions",m));
                mksynCall "For" m [pseudoEnumExpr; mksyn_match_lambda false false m [Clause(pat,None, trans innerComp,m)] ]

            | Comp_while (guardExpr,innerComp) -> 
                let m = (range_of_synexpr guardExpr) in 
                mksynCall "While" m [mksyn_delay m guardExpr; mksynCall "Delay" m [mksyn_delay m (trans innerComp)]]

            | Comp_try_finally (innerComp,unwindExpr) -> 
                let m = (range_of_synexpr unwindExpr) in 
                mksynCall "TryFinally" m [mksynCall "Delay" m [mksyn_delay m (trans innerComp)]; mksyn_delay m unwindExpr] 

            | Comp_zero -> 
                (* REVIEW: Bad range!!! *)
                mksynCall "Zero" (range_of_expr interpExpr)  []

            | Comp_sum(innerComp1, innerComp2,m1, m2) -> 
                mksynCall "Combine" m1 
                    [trans innerComp1; 
                     mksynCall "Delay" m2 [mksyn_delay m2 (trans innerComp2)]] 

            | Comp_cond(seqExprOnly,guardExpr,thenComp,elseComp) -> 
                if seqExprOnly then warning (Error("'when' clauses should only be used inside compact sequence expressions. Consider using 'if guardExpr then ...' in computation expressions",m));
                Expr_cond(guardExpr, trans thenComp, Some(trans elseComp), range_of_synexpr guardExpr)

            | Comp_bind(isUse,binderOpt,patOpt,synInputExpr,innerComp) -> 
                let methName = 
                    match binderOpt with 
                    | None -> (if isUse then "Using" else "Let") 
                    | Some binder -> "Bind"^(if binder = "let" then "" else if binder = "use" then "Using" else String.capitalize binder) in 
                let pat = match patOpt with None -> Pat_const(Const_unit,range_of_synexpr synInputExpr) | Some p -> p in 

                let m = range_of_synexpr synInputExpr in
                let consumeExpr = mksyn_match_lambda false false m [Clause(pat,None, trans innerComp,m)] in 

                mksynCall methName m [synInputExpr; consumeExpr ]

            | Comp_match(expr,clauses) -> 
                let clauses = clauses |> map (fun (CompClause(pat,cond,innerComp,patm)) -> Clause(pat,cond,trans innerComp,patm)) in
                Expr_match(expr, clauses, false,range_of_synexpr expr)

            | Comp_try_with(innerComp,clauses) -> 
                (* REVIEW: Bad range!!! *)
                let m = range_of_expr interpExpr in
                let clauses = clauses |> map (fun (CompClause(pat,cond,innerComp,patm)) -> Clause(pat,cond,trans innerComp,patm)) in
                let consumeExpr = mksyn_match_lambda false true m clauses  in 
                mksynCall "TryWith" m [mksynCall "Delay" m [mksyn_delay m (trans innerComp)]; consumeExpr]

            | Comp_yieldm((isTrueYield,isTrueReturn),yieldExpr) -> 
                let m = range_of_synexpr yieldExpr in 
                if not isTrueReturn then warning(Error("We recommend you use 'return!' instead of '->>' within computation expressions, as future extensions to the syntax may require this change, or may allow 'return!' to be dropped altogether",m));
                yieldExpr

            | Comp_yield((isTrueYield,isTrueReturn),yieldExpr) -> 
                let m = range_of_synexpr yieldExpr in 
                if not isTrueReturn then warning(Error("We recommend you use 'return' instead of '->' within computation expressions, as future extensions to the syntax may require this change",m));
                mksynCall "Return" m [yieldExpr] in 

         let coreSynExpr = trans comp in 

         let lambdaExpr = Expr_lambda (false,SPats ([mksyn_spat_var false (mksyn_id interpVarRange interpVarName)],interpVarRange), coreSynExpr, interpVarRange) in
         let lambdaExpr ,tpenv= tc_expr cenv (interpExprTy --> ty) env tpenv lambdaExpr in
         let coreExpr = mk_appl((lambdaExpr,type_of_expr cenv.g lambdaExpr),[],[interpExpr],interpVarRange) in
         let delayedExpr = mkDelayedExpr coreExpr in 
         delayedExpr,tpenv

    | None -> 

        let mkDelayedExpr coreExpr = 
            let m = range_of_expr coreExpr in 
            let ty = type_of_expr cenv.g coreExpr in 
            mk_seq_delay cenv env m ty coreExpr in

        let rec tc_comp env genOuterTy tpenv comp =
            match comp with 
            | Comp_for (seqExprOnly,pat,pseudoEnumExpr,innerComp) -> 
                let m = range_of_synexpr pseudoEnumExpr in
                (* This expression is not checked with the knowledge it is an IEnumerable, since we permit other enumerable types with GetEnumerator/MoveNext methods, as does C# *)
                let pseudoEnumExpr,arb_ty,tpenv = tc_expr_of_unknown_type cenv env tpenv pseudoEnumExpr in
                let enumExpr,enumElemTy = tc_seq_of_arb_typed_expr cenv arb_ty env pseudoEnumExpr in
                let pat',_,vspecs,envinner,tpenv = tc_match_pattern cenv enumElemTy env tpenv (pat,None) in 
                let innerComp,tpenv = tc_comp envinner genOuterTy tpenv innerComp in
                
                TComp_for((pat',vspecs),enumElemTy,genOuterTy,enumExpr,genOuterTy,innerComp),tpenv

            | Comp_while (guardExpr,innerComp) -> 
                let m = range_of_synexpr guardExpr in
                let guardExpr,tpenv = tc_expr cenv cenv.g.bool_ty env tpenv guardExpr in
                let innerComp,tpenv = tc_comp env genOuterTy tpenv innerComp in
                TComp_while(mkDelayedExpr,genOuterTy,guardExpr,innerComp),tpenv

            | Comp_try_finally (innerComp,unwindExpr) -> 
                let m = range_of_synexpr unwindExpr in
                let innerComp,tpenv = tc_comp env genOuterTy tpenv innerComp in
                let unwindExpr,tpenv = tc_expr cenv cenv.g.unit_ty env tpenv unwindExpr in
                TComp_try_finally(mkDelayedExpr,genOuterTy,innerComp,unwindExpr),tpenv

            | Comp_zero -> 
                TComp_zero(genOuterTy,m),tpenv 

            | Comp_sum(innerComp1, innerComp2,m1, m2) -> 
                let innerComp1,tpenv = tc_comp env genOuterTy tpenv innerComp1 in
                let innerComp2,tpenv = tc_comp env genOuterTy tpenv innerComp2 in
                TComp_sum(mkDelayedExpr,genOuterTy,innerComp1,innerComp2,m),tpenv 

            | Comp_cond(seqExprOnly,guardExpr,thenComp,elseComp) -> 
                let guardExpr',tpenv = tc_expr cenv cenv.g.bool_ty env tpenv guardExpr in
                let thenComp,tpenv = tc_comp env genOuterTy tpenv thenComp in
                let elseComp,tpenv = tc_comp env genOuterTy tpenv elseComp in
                TComp_cond(guardExpr',genOuterTy,thenComp,elseComp,m),tpenv 

            | Comp_bind(isUse,binderOpt,patOpt,synInputExpr,innerComp) -> 
                let m = range_of_synexpr synInputExpr in 
                (* this differs from Comp_matchm in the order of type checking *)
                let bindPatTy = new_inference_typ cenv () in 
                let inputExprTy = new_inference_typ cenv () in 
                let pat = match patOpt with None -> Pat_const(Const_unit,range_of_synexpr synInputExpr) | Some p -> p in 
                let pat',_,vspecs,envinner,tpenv = tc_match_pattern cenv bindPatTy env tpenv (pat,None) in 
                unifyE cenv env m inputExprTy bindPatTy;
                let inputExpr,tpenv = tc_expr cenv inputExprTy env tpenv synInputExpr in
                let innerComp,tpenv = tc_comp envinner genOuterTy tpenv innerComp in
                (* REVIEW: order reversal will confuse quotation hole filling? *)
                TComp_match_and_bind(isUse,inputExpr,bindPatTy,genOuterTy,genOuterTy,(pat',vspecs),innerComp),tpenv 

            | Comp_match(expr,clauses) -> 
                let expr',matchty,tpenv = tc_expr_of_unknown_type cenv env tpenv expr in
                let clauses,tpenv = 
                    map_acc_list 
                        (fun tepnv (CompClause(pat,cond,innerComp,patm)) ->
                              let pat',cond',vspecs,envinner,tpenv = tc_match_pattern cenv matchty env tpenv (pat,cond) in 
                              let innerComp,tpenv = tc_comp envinner genOuterTy tpenv innerComp in
                              (innerComp,(fun expr -> TClause(pat',cond',TTarget(vspecs, expr),range_of_pat pat'))),tpenv)
                        tpenv
                        clauses in 
                TComp_match(genOuterTy,expr',clauses),tpenv

            | Comp_try_with(innerComp,clauses) -> 
                error(Error("'try'/'with' may not be used with sequence expressions",m))

            | Comp_yieldm((isTrueYield,isTrueReturn),yieldExpr) -> 
                let m = range_of_synexpr yieldExpr in 
                let yieldExpr',genExprTy,tpenv = tc_expr_of_unknown_type cenv env tpenv yieldExpr in

                if not isTrueYield then warning(Error("We recommend you use 'yield!' instead of '->>' sequence expressions containing constructs other than for/when/->, as future extensions to the syntax may require this change",m)) ;

                type_must_subsume_type (denv_of_tenv env) cenv.css m  NoTrace genOuterTy genExprTy;
                TComp_result_comp(yieldExpr',genOuterTy,genExprTy),tpenv 

            | Comp_yield((isTrueYield,isTrueReturn),yieldExpr) -> 
                let m = range_of_synexpr yieldExpr in 
                let genResultTy = new_inference_typ cenv () in 
                if not isTrueYield then warning(Error("We recommend you use 'yield' instead of '->' within sequence expressions containing constructs other than for/when/->, as future extensions to the syntax may require this change",m));
                unifyE cenv env m genOuterTy (mk_seq_ty cenv.g genResultTy);

                let yieldExpr',tpenv = tc_expr cenv genResultTy env tpenv yieldExpr in
                TComp_result(yieldExpr',genResultTy),tpenv  in

        let genEnumElemTy = new_inference_typ cenv () in 
        unifyE cenv env m ty (mk_seq_ty cenv.g genEnumElemTy);

        let tcomp,tpenv = tc_comp env ty tpenv comp in 
        let coreExpr = conv_tcomp cenv env tcomp in 
        let delayedExpr = mkDelayedExpr coreExpr in 
        delayedExpr,tpenv

(*-------------------------------------------------------------------------
!* Typecheck "expr ... " constructs where "..." is a sequence of applications,
 * type applications and dot-notation projections. First extract known
 * type information from the "..." part to use during type checking.
 *
 * 'ty' is the type expected for the entire chain of expr + lookups.
 * 'exprty' is the type of the expression on the left of the lookup chain.
 *
 * Unsophisticated applications can propagate information from the expected overall type 'ty' 
 * through to the leading function type 'exprty'. This is because the application 
 * unambiguously implies a function type 
 *------------------------------------------------------------------------- *)

and propagate_then_tc_delayed cenv ty env tpenv m expr exprty delayed = 
    if verbose then dprintf1 "--> propagate_then_tc_delayed, #delayed = %d\n" (length delayed); 

    let rec propagate delayed' m exprty = 
      match delayed' with 
      | [] -> 
          (* Avoid unifying twice: we're about to unify in tc_delayed *)
          if nonNil delayed then 
              unifyE cenv env m ty exprty
      | Lvalue_set _ :: _
      | Lvalue_get _ :: _ -> ()
      | TyApp (x,m) :: delayed'' ->
          (* Note this case should not occur: would eventually give an "unexpected type application" error in tc_delayed *)
          propagate delayed'' m exprty 

      | App (arg,m) :: delayed'' ->
          let denv = denv_of_tenv env in
          match unify_fun_and_undo_if_failed cenv denv m exprty with
          | Some (_,rty) -> 
              propagate delayed'' m rty 
          | None -> 
              let m2 = range_of_synexpr arg in
              match arg with 
              | Expr_comprehension (comp,_) -> ()
              | _ -> error (NotAFunction(denv,ty,m,m2))  in
              
    propagate delayed m exprty;
    tc_delayed cenv ty env tpenv m expr exprty delayed

(*-------------------------------------------------------------------------
!* Typecheck "expr ... " constructs where "..." is a sequence of applications,
 * type applications and dot-notation projections.
 *------------------------------------------------------------------------- *)

and tc_delayed cenv ty env tpenv m expr exprty delayed = 
    if verbose then dprintf2 "--> tc_delayed@%a\n" output_range m; 
    (* OK, we've typechecked the thing on the left of the delayed lookup chain. *)
    (* We can now record for posterity the type of this expression and the location of the expression. *)
    callQualSink m (nenv_of_tenv env,Item_typs("<expr>",[exprty]), denv_of_tenv env);

    match delayed with 
    | [] -> unifyE cenv env m ty exprty; expr,tpenv
    (* expr.m(args) where x.m is a .NET method or index property *)
    (* expr.m<tyargs>(args) where x.m is a .NET method or index property *)
    (* expr.m where x.m is a .NET method or index property *)
    | Lvalue_get (lid,m) :: delayed' ->
         tc_lvalue_then cenv ty env tpenv expr exprty lid delayed' m
    (* f x *)
    | App (arg,m) :: delayed' ->
        tc_app_then cenv ty env tpenv m expr exprty arg delayed'
    (* f<tyargs> *)
   | TyApp (x,m) :: delayed' ->
        error(Error("unexpected type application",m))
    | Lvalue_set _ :: delayed' ->      
        error(Error("Invalid assignment (1)",m))


and delay_rest rest m delayed = 
    match rest with 
    | [] -> delayed 
    | _ -> (Lvalue_get (rest,m) :: delayed)


(*-------------------------------------------------------------------------
!* tc_app_then: Typecheck "expr x" + projections
 *------------------------------------------------------------------------- *)

and tc_app_then cenv ty env tpenv m expr exprty arg delayed' = 
    if verbose then  dprint_endline "tc Expr_app";
    let denv = denv_of_tenv env in
    match unify_fun_and_undo_if_failed cenv denv m exprty with
    | Some (_,rty) -> 
        let dty,rty = unify_fun (Some (range_of_synexpr arg)) cenv (denv_of_tenv env) m exprty in 
        if verbose then  dprintf3 "tc_app_then %a: dty = %s\n" output_range m (NicePrint.pretty_string_of_typ (empty_denv cenv.g) dty); 
        let arg',tpenv = tc_expr cenv dty env tpenv arg in 
        if verbose then  dprintf3 "tc_app_then %a: dty = %s\n" output_range m (NicePrint.pretty_string_of_typ (empty_denv cenv.g) dty); 
        let expr' = build_app cenv expr exprty arg' m in
        tc_delayed cenv ty env tpenv m expr' rty delayed'
    | None -> 
        let m2 = range_of_synexpr arg in
        match arg with 
        | Expr_comprehension (comp,_) -> 
            let expr',tpenv = tc_comprehension cenv env ty m (Some(expr,exprty)) tpenv comp in
            tc_delayed cenv ty env tpenv m expr' (type_of_expr cenv.g expr') delayed'
        | _ -> 
            error (NotAFunction(denv,ty,m,m2)) 

(*-------------------------------------------------------------------------
!* tc_lid_then : Typecheck "A.B.C<D>.E.F ... " constructs
 *------------------------------------------------------------------------- *)

and tc_lid_then cenv ty env tpenv m lid delayed =
    if verbose then dprintf3 "--> tc_lid_then, m = %a, lid = %s\n" output_range m (text_of_lid lid); 

    let typeNameResInfo = 
        (* Given 'MyOverloadedType<int>.MySubType...' use arity of #given type arguments to help *)
        (* resolve type name lookup of 'MyOverloadedType' *)
        (* Also determine if type names should resolve to Item_typs or Item_ctor_group *)
        match delayed with 
        | (TyApp (tyargs,_)) :: Lvalue_get _ :: _ -> 
            if verbose then dprintf3 "--> tc_lid_then (ResolveTypeNamesToTypeRefs), m = %a, lid = %s\n" output_range m (text_of_lid lid); 
            (ResolveTypeNamesToTypeRefs, Some(length tyargs))

        | (TyApp (tyargs,_)) :: App _ :: _ -> 
            (ResolveTypeNamesToCtors, Some(length tyargs))

        | _ -> defaultTypeNameResInfo in

    let itemAndRest = tc_expr_lid_and_compute_range cenv.ginstf cenv.g cenv.amap m env.eNameResEnv typeNameResInfo lid in
    tc_item_then cenv ty env tpenv itemAndRest delayed

(*-------------------------------------------------------------------------
!* Typecheck "item+projections" 
 *------------------------------------------------------------------------- *)

and tc_item_then cenv ty env tpenv (item,m,rest) delayed =
    if verbose then dprintf2 "--> tc_item_then, m = %a\n" output_range m; 
    let delayed = delay_rest rest m delayed in 
    let family = accessRightsOfEnv env in 
    match item with
    (* x where x is a data constructor or active pattern result tag. *)
    | (Item_ucref _ | Item_ecref _ | Item_apres _) as item -> 
        (* uconstrAppTy is the type of the union constructor applied to its (optional) argument *)
        let uconstrAppTy = new_inference_typ cenv () in 
        let mkConstrApp,argtys = 
          match item with 
          | Item_apres(apinfo,n) -> 
              let aparity = length (names_of_apinfo apinfo) in 
              begin match aparity with 
              | 0 | 1 -> 
                  let mkConstrApp = function [arg] -> arg | _ -> error(InternalError("gen_constr_unify",m)) in
                  mkConstrApp, [uconstrAppTy]
              | _ ->
                  let ucref = mk_choices_ucref cenv.g aparity n in
                  expr_constr_unify m cenv env uconstrAppTy (Item_ucref ucref)
              end
          | _ -> expr_constr_unify m cenv env uconstrAppTy item in 
        let nargtys = length argtys in 
        begin match delayed with 
        (* This is where the constructor is applied to an argument *)
        | ((App (arg,m))::delayed') ->
            if isNil(delayed') then 
                unifyE cenv env m ty uconstrAppTy; 
                  
            let args = 
              match arg with 
              | Expr_paren(Expr_tuple(args,m),_)
              | Expr_tuple(args,m)     when nargtys > 1 -> args
              | Expr_paren(arg,_)
              | arg -> [arg] in 
            let nargs = length args in
            gen_constr_check env nargtys nargs m;
            let args',tpenv = tc_exprs cenv env m tpenv argtys args in 
            propagate_then_tc_delayed cenv ty env tpenv m (mkConstrApp args') uconstrAppTy delayed'
        | TyApp (x,m) :: delayed' ->
            error(Error("unexpected type application",m))
        | _ -> 
            (* Work out how many syntactic arguments we really expect. Also return a function that builds the overall *)
            (* expression, but don't apply this function until after we've checked that the number of arguments is OK *)
            (* (or else we would be building an invalid expression) *)
            
            (* Unit-taking active pattern result can be applied to no args *)
            let nargs,mkExpr = 
                (* This is where the constructor is an active pattern result applied to no argument *)
                (* Unit-taking active pattern result can be applied to no args *)
                if (nargtys = 1 && match item with Item_apres _ -> true | _ -> false) then (
                    unifyE cenv env m (List.hd argtys) cenv.g.unit_ty;
                    1,(fun () -> mkConstrApp [mk_unit cenv.g m])
                ) 
                (* This is where the constructor expects no arguments and is applied to no argument *)
                else if nargtys = 0 then 0,(fun () -> mkConstrApp []) 
                else 
                   (* This is where the constructor expects arguments but is not applied to arguments, hence build a lambda *)
                    nargtys, 
                    (fun () -> 
                        let vs,args = argtys |> list_mapi (fun i ty -> mk_compgen_local m ("$arg_"^string_of_int i) ty) |> split in 
                        let constrApp = mkConstrApp args in 
                        let lam = mk_multi_lambda m vs (constrApp, type_of_expr cenv.g constrApp) in
                        lam) in
            gen_constr_check env nargtys nargs m;
            let expr = mkExpr() in
            let exprTy = type_of_expr cenv.g expr in 
            propagate_then_tc_delayed cenv ty env tpenv m expr exprTy delayed 
        end
    | Item_typs(_,(typ::_)) -> 
    
        begin match delayed with 
        | ((TyApp(tyargs,tyappm))::(Lvalue_get (lid,_))::delayed') ->

            (* If Item_typs is returned then the typ will be of the form TType_app(tcref,genericTyargs) where tyargs *)
            (* is a fresh instantiation for tcref. tc_nested_type_app will chop off precisely #genericTyargs args *)
            (* and replace them by 'tyargs' *)
            let typ,tpenv = tc_nested_type_app cenv NewTyparsOK CheckCxs env tpenv tyappm typ tyargs in
            tc_item_then cenv ty env tpenv (tc_expr_dot_lid_and_compute_range cenv.ginstf cenv.g cenv.amap m env.eNameResEnv typ lid IgnoreOverrides) delayed'
        | _ -> 
            (* In this case the type is not generic, and indeed we should never have returned Item_typs. *)
            (* That's because ResolveTypeNamesToCtors should have been set at the original *)
            (* call to tc_expr_lid_and_compute_range *)
            error(Error("Invalid use of a type name",m));
        end
    | Item_meth_group (methodName,minfos) -> 
        (* .NET Static method calls Type.Foo(arg1,...,argn) *)
        (* REVIEW: for ctors check that the type is not abstract *)
        (* REVIEW: in theory passing false here is not right: these may actually mutate a value type object - perhaps at least emit a (very) otpional warning? *) 
        let meths = map (fun minfo -> minfo,None) minfos in 
        begin match delayed with 
        | ((App (arg,m))::delayed') ->
            tc_method_args_and_apps_then cenv env ty tpenv None [] m methodName family NeverMutates false meths NormalValUse [arg] delayed'
        | ((TyApp(tys,tyappm))::(App (arg,m))::delayed') ->
            let tyargs,tpenv = tc_types cenv NewTyparsOK CheckCxs env tpenv tys in 
            tc_method_args_and_apps_then cenv env ty tpenv (Some tyargs) [] m methodName family NeverMutates false meths NormalValUse [arg] delayed'
        | _ -> 
            tc_method_args_and_apps_then cenv env ty tpenv None [] m methodName family NeverMutates false meths NormalValUse [] delayed
        end
    | (Item_ctor_group (_,minfos) as item) ->
        let objTy = 
            match minfos with 
            | (minfo :: _) -> typ_of_minfo minfo 
            | [] -> error(Error("This type has no accessible object constructors",m)) in 
        begin match delayed with 
        | ((App (arg,argm))::delayed') ->
(*
            if nonNil(typars_of_tcref (tcref_of_stripped_typ objTy)) then 
                warning(Error("This expression refers to a object constructor for a generic type, but doesn't give explicit type arguments. This langauge feature is under review. For now, object constructors for generic types should be given explicit type arguments, e.g. 'Dictionary<int,int>()' or 'Dictionary<_,_>()'",m));
*)
            (* REVIEW: the langauge currently allows 'KeyValuePair(k,v)' *)
            (* REVIEW: this should perhaps be reconsidered. *)
            (* Here is the code to enforce the check: *)
            (*   This checks that the type really doesn't expect any type arguments. *)
            (*   let objTy,tpenv = tc_nested_type_app cenv NewTyparsOK CheckCxs env tpenv m objTy [] in *)
            (*   This should be a nop but is included to keep symmetry with the next case *)
            (*   minfos |> List.iter (fun minfo -> unifyE cenv env m (typ_of_minfo minfo) objTy); *)
            tc_ctor_call true cenv env tpenv ty objTy item false arg argm delayed'
        | ((TyApp(tyargs,tyappm))::(App (arg,m))::delayed') ->
            let objTy,tpenv = tc_nested_type_app cenv NewTyparsOK CheckCxs env tpenv tyappm objTy tyargs in
            minfos |> List.iter (fun minfo -> unifyE cenv env tyappm (typ_of_minfo minfo) objTy);
            tc_ctor_call true cenv env tpenv ty objTy item false arg m delayed'
        | _ -> 
            let text = List.map (string_of_minfo cenv.g cenv.amap m (denv_of_tenv env)) minfos in
            error(Error("Invalid use of a type name and/or object constructor. If necessary use 'new' and apply the constructor to its arguments, e.g. 'new Type(args)'"^
                         (if nonNil minfos then ". Overloads are:  \n\t"^String.concat "\n\t" text else ""),m))
        end
    | Item_fake_intf_ctor _ ->
        error(Error("Invalid use of an interface type",m))
    | Item_delegate_ctor typ ->
        begin match delayed with 
        | ((App (arg,m))::delayed') ->
            tc_delegate_ctor_then cenv ty env tpenv m typ arg delayed'
        | ((TyApp(tyargs,tyappm))::(App (arg,m))::delayed') ->
            let typ,tpenv = tc_nested_type_app cenv NewTyparsOK CheckCxs env tpenv tyappm typ tyargs in
            tc_delegate_ctor_then cenv ty env tpenv m typ arg delayed'
        | _ -> error(Error("Invalid use of a delegate constructor. Use the syntax new Type(args) or just Type(args)",m))
        end
    | Item_val vref -> 
        if verbose then dprintf2 "--> tc_item_then, value, m = %a\n" output_range m; 

        begin match delayed with 
        (* Mutable value set: 'v <- e' *)
        | Lvalue_set(e2,m) :: delayed' ->
            if nonNil(delayed') then error(Error("Invalid assignment (2)",m));
            unifyE cenv env m ty (cenv.g.unit_ty);
            let vty = (type_of_vref vref) in 
            let vexp,tepnv = 
                if is_byref_ty cenv.g vty then 
                  let e2',tpenv = tc_expr cenv (dest_byref_ty cenv.g vty) env tpenv e2 in 
                  mk_lval_set m vref e2', tpenv
                else 
                  let e2',tpenv = tc_expr cenv vty env tpenv e2 in 
                  if mutability_of_vref vref <> Mutable then error (ValNotMutable((denv_of_tenv env),vref,m));
                  mk_val_set m vref e2', tpenv in
                
            propagate_then_tc_delayed cenv ty env tpenv m vexp (type_of_expr cenv.g vexp) delayed'

        (* Value instantiation: v<tyargs> ... *)
        | (TyApp(tys,tyappm)::delayed') ->
            let tyargs,tpenv = tc_types cenv NewTyparsOK CheckCxs env tpenv tys in 
            (* Note: we know this is a NormalValUse because: *)
            (*   - it isn't a CtorValUsedAsSuperInit *)
            (*   - it isn't a CtorValUsedAsSelfInit *)
            (*   - it isn't a VSlotDirectCall (uses of base values do not take type arguments *)
            let (vexp,vexpty) = tc_val cenv env vref (Some (NormalValUse, tyargs)) m in
            propagate_then_tc_delayed cenv ty env tpenv m vexp vexpty delayed'

        (* Value get *)
        | _ ->  
            let (vexp,vexpty) = tc_val cenv env vref None m in             
            propagate_then_tc_delayed cenv ty env tpenv m vexp vexpty delayed
        end
        
    | Item_property (nm,pinfos) ->
        if isNil pinfos then error (Error ("unexpected error: empty property list",m));
        let pinfo = List.hd pinfos in 
        let tyargsOpt,args,delayed,tpenv = 
            if pinfo_is_indexer cenv.g pinfo 
            then get_member_app_args delayed cenv env tpenv m 
            else None,[mksyn_unit m],delayed,tpenv in 
        if not (pinfo_is_static pinfo) then error (Error ("property '"^nm^" is not static",m));
        begin match delayed with 
        | Lvalue_set(e2,m) :: delayed' ->
            if nonNil(delayed') then error(Error("Invalid assignment (3)",m));
            (* Static Property Set (possibly indexer) *)
            unifyE cenv env m ty (cenv.g.unit_ty);
            let meths = chooseList (fun pinfo -> if pinfo_has_setter pinfo then Some(setter_minfo_of_pinfo pinfo,Some pinfo) else None) pinfos in 
            tc_method_args_and_apps_then cenv env ty tpenv tyargsOpt [] m nm family DefinitelyMutates false meths NormalValUse (args@[e2]) delayed'
        | _ -> 
            (* Static Property Get (possibly indexer) *)
            let meths = chooseList (fun pinfo -> if pinfo_has_getter pinfo then Some(getter_minfo_of_pinfo pinfo,Some pinfo) else None) pinfos in 
            if isNil(meths) then error (Error ("property '"^nm^"' is not readable",m));
            tc_method_args_and_apps_then cenv env ty tpenv tyargsOpt [] m nm family PossiblyMutates false meths NormalValUse args delayed
        end

    | Item_il_field finfo -> 
        begin match delayed with 
        | Lvalue_set(e2,m) :: delayed' ->
            error(Error("Static .NET fields may not be set using F#",m))
        | _ -> 
           (* Get static IL field *)
            il_finfo_accessible_check cenv.g cenv.amap m family finfo;
            if not (il_finfo_is_static finfo) then error (Error ("field "^name_of_il_finfo finfo^" is not static",m));
            il_finfo_attrib_check cenv.g finfo m;
            let fref = fref_of_il_finfo finfo in 
            let exprty = vtyp_of_il_finfo cenv.amap m  finfo in 
            let expr = 
              match il_finfo_literal_value finfo with 
              | Some lit -> 
                  TExpr_const(tc_field_init m lit,m,exprty) 
              | None -> 
                let isValueType = il_finfo_is_struct finfo in 
                let valu = if isValueType then AsValue else AsObject in 
                (* The empty instantiation on the fspec is OK, since we make the correct fspec in Ilxgen.gen_asm *)
                (* This ensures we always get the type instantiation right when doing this from *)
                (* polymorphic code, after inlining etc. *) 
                (* REVIEW: stop generating ABSIL instructions here. This causes a number of minor problems (e.g. with quotation reflection) *)
                let fspec = mk_fspec(fref,mk_named_typ valu (tref_of_fref fref) []) in 
                (* Add an I_nop if this is an initonly field to make sure we never recognize it as an lvalue. See mk_expra_of_expr. *)
                mk_asm ([ mk_normal_ldsfld fspec ] @ (if initonly_of_il_finfo finfo then [ I_arith AI_nop ] else []), 
                        tinst_of_il_finfo finfo,[],[exprty],m) in
            propagate_then_tc_delayed cenv ty env tpenv m expr exprty delayed
        end
    | Item_recdfield rfinfo -> 
        (* Get static F# field or literal *)
        rfinfo_accessible_check m family rfinfo;
        if not (rfinfo_is_static rfinfo) then error (Error ("field "^name_of_rfinfo rfinfo^" is not static",m));
        rfinfo_attrib_check cenv.g rfinfo m |> commitOperationResult;        
        let fref = rfref_of_rfinfo rfinfo in 
        let fieldTy = vtyp_of_rfinfo rfinfo in 
        begin match delayed with 
        | Lvalue_set(e2,m) :: delayed' ->
            if nonNil(delayed') then error(Error("Invalid assignment (5)",m));
        
            (* Set static F# field *)
            check_rfield_mutation cenv.g m (denv_of_tenv env) rfinfo [];
            unifyE cenv env m ty cenv.g.unit_ty;
            let fieldTy = vtyp_of_rfinfo rfinfo in 
            let e2',tpenv = tc_expr cenv fieldTy env tpenv e2 in 
            let expr = mk_static_rfield_set (rfref_of_rfinfo rfinfo,tinst_of_rfinfo rfinfo,e2',m) in
            expr,tpenv
            
        | _  ->
            let exprty = fieldTy in 
            let expr = 
              match rfinfo_literal_value rfinfo with 
              (* Get literal F# field *)
              | Some lit -> TExpr_const(lit,m,exprty)
              (* Get static F# field *)
              | None -> mk_static_rfield_get (fref,tinst_of_rfinfo rfinfo,m)  in
            propagate_then_tc_delayed cenv ty env tpenv m expr exprty delayed
        end
    | Item_il_event (ILEventInfo(tinfo,edef) as einfo) -> 
        let nm = name_of_il_einfo einfo in
        if not (il_einfo_is_static einfo) then error (Error ("event '"^nm^"' is not static",m));
        error (Error ("Static .NET events are currently manipulated using the explicit 'add_"^nm^"' and 'remove_"^nm^"' methods",m));
     
    | _ -> error(Error("This lookup may not be used here", m))


(*-------------------------------------------------------------------------
!* Typecheck "expr.A.B.C ... " constructs
 *------------------------------------------------------------------------- *)

and get_member_app_args delayed cenv env tpenv m =
    match delayed with 
    | App (arg,m) :: delayed' -> None,[arg],delayed',tpenv
    | TyApp(tyargs,_) :: App (arg,m) :: delayed' ->
        let tyargs,tpenv = tc_types cenv NewTyparsOK CheckCxs env tpenv tyargs in 
        Some(tyargs),[arg],delayed',tpenv
    | delayed' ->
        None,[],delayed',tpenv


and tc_lvalue_then cenv ty env tpenv e1' e1ty' lid delayed m =
    let objArgs = [e1'] in 

    (* 'base' calls use a different resolution strategy when finding methods. *)
    (* They look for 'base' calls use a different resolution strategy when finding methods. *)
    let findFlag = 
        let baseCall = isBaseCall objArgs in
        (if baseCall then PreferOverrides else IgnoreOverrides) in 
        
    if verbose then dprintf2 "--> tc_lvalue_then%a\n" output_range m; 

    (* Canonicalize inference problem prior to '.' lookup on variable types *)
    if is_typar_ty e1ty' then 
        canonicalizePartialInferenceProblem (cenv,(denv_of_tenv env),m) (free_in_type_lr false e1ty');
    
    let item,m,rest = tc_expr_dot_lid_and_compute_range cenv.ginstf cenv.g cenv.amap m env.eNameResEnv e1ty' lid findFlag in
    let delayed = delay_rest rest m delayed in 
    let family = accessRightsOfEnv env in 

    match item with
    | Item_meth_group (methodName,minfos) -> 
        let tyargsOpt,args,delayed,tpenv = get_member_app_args delayed cenv env tpenv m in
        let meths = map (fun minfo -> minfo,None) minfos in 
        (* We pass PossiblyMutates here because these may actually mutate a value type object *) 
        (* To get better warnings we special case some of the few known mutate-a-struct method names *) 
        let mutates = (if methodName = "MoveNext" || methodName = "GetNextArg" then DefinitelyMutates else PossiblyMutates) in 
        tc_method_args_and_apps_then cenv env ty tpenv tyargsOpt objArgs m methodName family mutates false meths NormalValUse args delayed

    | Item_property (nm,pinfos) ->
        (* Instance property *)
        if isNil pinfos then error (Error ("unexpected error: empty property list",m));
        let pinfo = List.hd pinfos in 
        if pinfo_is_static pinfo then error (Error ("property '"^nm^" is static",m));
        let tyargsOpt,args,delayed,tpenv = 
            if pinfo_is_indexer cenv.g pinfo 
            then get_member_app_args delayed cenv env tpenv m 
            else None,[mksyn_unit m],delayed,tpenv in 
        begin match delayed with 
        | Lvalue_set(e2,m) :: delayed' ->
            if nonNil(delayed') then error(Error("Invalid assignment (2)",m));
        
            (* Instance property setter *)
            unifyE cenv env m ty (cenv.g.unit_ty);
            let meths = chooseList (fun pinfo -> if pinfo_has_setter pinfo then Some(setter_minfo_of_pinfo pinfo,Some pinfo) else None) pinfos in 
            if isNil meths then error (Error ("Property '"^nm^"' may not be set",m));
            tc_method_args_and_apps_then cenv env ty tpenv tyargsOpt objArgs m nm family DefinitelyMutates false meths NormalValUse (args @ [e2]) []
        | _ ->                   
            let meths = chooseList (fun pinfo -> if pinfo_has_getter pinfo then Some(getter_minfo_of_pinfo pinfo,Some pinfo) else None) pinfos in 
            if isNil meths then error (Error ("Property '"^nm^" is not readable",m));
            tc_method_args_and_apps_then cenv env ty tpenv tyargsOpt objArgs m nm family PossiblyMutates true meths NormalValUse args delayed
        end
        
    | Item_recdfield rfinfo ->
        (* Get or set instance F# field or literal *)
        rfinfo_instance_checks cenv.g family m rfinfo;
        let tgty = (enclosing_vtyp_of_rfinfo rfinfo) in 
        let valu = is_struct_typ tgty in 
        type_must_subsume_type (denv_of_tenv env) cenv.css m NoTrace tgty e1ty'; 
        let e1' = if valu then e1' else mk_coerce(e1',tgty,m,e1ty') in
        let _,ftinst,e1ty'' = freshenPossibleForallTy cenv m TyparFlexible (vtyp_of_rfinfo rfinfo) in
        begin match delayed with 
        | Lvalue_set(e2,m) :: delayed' ->
            (* Mutable value set: 'v <- e' *)
            if nonNil(delayed') then error(Error("Invalid assignment (2)",m));
            check_rfield_mutation cenv.g m (denv_of_tenv env) rfinfo ftinst;
            unifyE cenv env m ty (cenv.g.unit_ty);
            let e2',tpenv = tc_expr cenv e1ty'' env tpenv e2 in 
            buildRecdFieldSet cenv.g m (denv_of_tenv env) e1' rfinfo ftinst e2',tpenv

        | _ ->

           (* Instance F# Record or Class field *)
            let e1'' = mk_recd_field_get cenv.g (e1',rfref_of_rfinfo rfinfo,tinst_of_rfinfo rfinfo,ftinst,m) in 
            propagate_then_tc_delayed cenv ty env tpenv m e1'' e1ty'' delayed 
        end
        
    | Item_il_field (ILFieldInfo(tinfo,fdef) as finfo) -> 
       (* Get or set instance IL field *)
        il_field_instance_checks  cenv.g cenv.amap family m finfo;
        let fref = fref_of_il_finfo finfo in 
        let exprty = vtyp_of_il_finfo cenv.amap m  finfo in 
        let isValueType = il_finfo_is_struct finfo in 
        let valu = if isValueType then AsValue else AsObject in 
        let tinst = tinst_of_il_tinfo tinfo in
        
        begin match delayed with 
        (* Set instance IL field *)
        | Lvalue_set(e2,m) :: delayed' ->
            (* The empty instantiation on the fspec is OK, since we make the correct fspec in Ilxgen.gen_asm *)
            (* This ensures we always get the type instantiation right when doing this from *)
            (* polymorphic code, after inlining etc. *) 
            unifyE cenv env m ty (cenv.g.unit_ty);
            let e2',tpenv = tc_expr cenv exprty env tpenv e2 in 
            let expr = buildIlFieldSet cenv.g m e1' finfo e2' in
            expr,tpenv
        | _ ->        
            let expr = buildIlFieldGet cenv.g cenv.amap m e1' finfo  in
            propagate_then_tc_delayed cenv ty env tpenv m expr exprty delayed
        end

    | Item_il_event (ILEventInfo(tinfo,edef) as einfo) -> 
       (* Instance IL event (fake up event-as-value) *)
        let nm = name_of_il_einfo einfo in
        if il_einfo_is_static einfo then error (Error ("event '"^nm^" is static",m));
        if edef.eventType = None then error (Error ("the event '"^nm^" must be accessed using the explicit add_"^nm^" and remove_"^nm^" methods for the event",m));
        let del_ty = del_typ_of_il_einfo cenv.amap m einfo in
        let minfo,del_argtys,del_rty,_ = sig_of_fun_as_delegate cenv.g cenv.amap del_ty m in
        (* This checks for and drops the 'object' sender *)
        let args_ty = args_ty_of_il_einfo cenv.g cenv.amap m einfo in 
        if not (slotsig_has_void_rty cenv.g (slotsig_of_minfo cenv.g cenv.amap m minfo)) then errorR (event_err einfo m);
        let devent_ty = mk_fslib_IDelegateEvent_ty cenv.g del_ty args_ty in
        let event_ty = mk_fslib_IEvent_ty cenv.g args_ty in
        let ctor_call = mk_obj_ctor_call cenv.g m in 
        let valu = il_tinfo_is_struct (tinfo_of_il_einfo einfo) in 

        (* Bind the object target expression to make sure we only run its sdie effects once, and to make *)
        (* sure if it's a mutable reference then we dereference it - see FSharp 1.0 bug 942 *)
        let expr = 
            mk_let_in m "eventTarget" e1ty' e1' (fun (v,ve) -> 
                let mk_event_override nm minfo = 
                   let thisv,thise = mk_compgen_local m "this" devent_ty in
                   let dv,de = mk_compgen_local m "d" del_ty in
                   let callExpr,_ = buildMethodCall cenv env PossiblyMutates m false (ILMeth minfo) NormalValUse [] [ve] [de] in
                   TMethod(mk_slotsig(nm,devent_ty,[vara;varb], [],[mk_slotparam vara_ty], cenv.g.unit_ty),[], [thisv;dv],callExpr,m) in
           
                let overrides =
                  [ mk_event_override "AddHandler" (add_minfo_of_il_einfo einfo);
                    mk_event_override "RemoveHandler" (remove_minfo_of_il_einfo einfo) ] in
                let iimpls = 
                  [(event_ty,[
                      (let fvty = (args_ty  --> cenv.g.unit_ty) in 
                       let fv,fe = mk_compgen_local m "d" fvty in
                       let thisv,thise = mk_compgen_local m "this" event_ty in
                       let de = coerce_fun_as_delegate (Some einfo) cenv del_ty (minfo,del_argtys,del_rty) (fe,fvty) m in
                       let minfo = add_minfo_of_il_einfo einfo in
                       let callExpr,_ = buildMethodCall cenv env PossiblyMutates m false (ILMeth minfo) NormalValUse [] [ve] [de] in
                       TMethod(mk_slotsig("Add",event_ty,[vara], [],
                               [mk_slotparam (vara_ty  --> cenv.g.unit_ty)], cenv.g.unit_ty),
                               [], [thisv;fv],callExpr,m));
                   ])] in 
                TExpr_obj(new_uniq(), devent_ty, None, ctor_call, overrides,iimpls,m,new_cache())) in
        let exprty = devent_ty in 
        propagate_then_tc_delayed cenv ty env tpenv m expr exprty delayed
     

     
    | (Item_fake_intf_ctor _ | Item_delegate_ctor _) -> error (Error ("Constructors must be applied to arguments and cannot be used as first-class values. If necessary use an anonymous function '(fun arg1 ... argN -> new Type(arg1,...,argN))'", m))
    | _ -> error (Error ("The syntax 'expr.id' may only be used with record labels, properties and fields", m))

(*-------------------------------------------------------------------------
!* Typecheck method/member calls and uses of members as first-class values.
 *------------------------------------------------------------------------- *)

and tc_method_args_and_apps_then 
       cenv env
       ty  (* The type of the overall expression including "delayed". Note this may be a use of a member as a first-class function value, when this would be a function type. *)
       tpenv 
       userTypeArgs (* The return type of the overall expression including "delayed" *)
       objArgs (* The 'obj' arguments in obj.M(...) and obj.M, if any *)
       m 
       methodName (* string, name of the method *) 
       ad         (* accessibility rights of the caller *) 
       mut        (* what do we know/assume about whether this method will mutate or not? *) 
       isProp     (* is this a property call? Used for better error messages and passed to buildMethodCall *) 
       meths       (* the set of methods we may be calling *)
       isSuperInit (* is this a special invocation, e.g. a super-class constructor call. Passed through to buildMethodCall *) 
       args        (* the _syntactic_ method arguments, not yet type checked. *)
       delayed     (* further lookups and applications that follow this *)
     =

    (* Work out if we know anything about the return type of the overall expression. If there are any delayed *)
    (* lookups then we don't know anything. *)
    let exprTyOpt = (if isNil delayed then Some(ty) else None) in

    (* Call the helper below to do the real checking *)
    let (expr,attributeAssignedNamedItems),tpenv = 
        tc_method_args_and_apps false cenv env tpenv userTypeArgs objArgs m methodName ad mut isProp meths isSuperInit args exprTyOpt in 

    (* Give errors if some things couldn't be assigned *)
    if nonNil attributeAssignedNamedItems then (
        let (CallerNamedArg(_,CallerArg(_,m,_,_))) = List.hd attributeAssignedNamedItems in 
        errorR(Error("The named argument 'id = expr' did not match any argument or mutable property",m));
    );


    (* Resolve the "delayed" lookups *)
    let exprty = (type_of_expr cenv.g expr) in 

    if verbose then dprintf1 "<-- tc_method_args_and_apps, exprty = %s\n" (showL (typeL exprty));
    if verbose then dprintf1 "<-- tc_method_args_and_apps, ty = %s\n" (showL (typeL ty));

    propagate_then_tc_delayed cenv ty env tpenv m expr exprty delayed 

and new_inference_typ_for_method_arg_synexpr cenv x =
    match x with 
    | Expr_paren(a,_) -> new_inference_typ_for_method_arg_synexpr cenv a
    | Expr_addrof(true,a,_) -> mk_byref_typ cenv.g (new_inference_typ_for_method_arg_synexpr cenv a)
    | Expr_lambda(_,_,a,_) -> (new_inference_typ cenv () --> new_inference_typ_for_method_arg_synexpr cenv a)
    | _ -> new_inference_typ cenv ()

(* Method calls, property lookups, attribute constructions etc. get checked through here *)
and tc_method_args_and_apps 
        checkingAttributeCall 
        cenv 
        env 
        tpenv 
        tyargsOpt 
        objArgs 
        m 
        methodName 
        ad 
        mut 
        isProp 
        calledMethGroup 
        isSuperInit 
        args 
        exprTyOpt 
    =
    if verbose then dprintf2 "--> tc_method_args_and_apps@%a\n" output_range m;
    (* Nb. args is always of length <= 1 except for indexed setters, when it is 2  *)
    let m = List.fold_left (fun m arg -> union_ranges m (range_of_synexpr arg)) m args in
    let denv = denv_of_tenv env in
    
    (* Split the syntactic arguments (if any) into named and unnamed parameters *)
    let unnamedCallerArgsOpt, namedCallerArgs = 
      match args with 
      | [] -> None,[]
      | _ -> 
          let a,b = split (map (getMethodArgs cenv env) args) in 
          let unnamedCallerArgs = a |> concat |> map (fun x -> x, new_inference_typ_for_method_arg_synexpr cenv x,range_of_synexpr x) in
          let namedCallerArgs = b |> concat |> map (fun (isOpt,nm,x) -> nm,isOpt,x,new_inference_typ_for_method_arg_synexpr cenv x,range_of_synexpr x) in
          (Some unnamedCallerArgs, namedCallerArgs) in 
    
    (* STEP 1. unifyUniqueOverloading. This happens BEFORE we type check the arguments. *)
    (* Extract what we know about the caller arguments, either type-directed if *)
    (* no arguments are given or else based on the syntax of the arguments. *)
    let uniquelyResolved,preArgumentTypeCheckingCalledMethGroup = 
        let dummyExpr = mk_unit cenv.g m in
      
        (* Build the CallerArg values for the caller's arguments. *)
        (* Fake up some arguments if this is the use of a method as a first class function *)
        let unnamedCallerArgs = 
            let candidates = 
                calledMethGroup 
                |> map fst
                |> filter (minfo_accessible cenv.g cenv.amap m ad) in

            match unnamedCallerArgsOpt,candidates,exprTyOpt with 
            (* error: no arguments, no overall type available *)
            | None,_,None -> 
                let text = 
                    calledMethGroup
                    |> map fst
                    |> List.map (string_of_minfo cenv.g cenv.amap m denv) in
                if isProp 
                then error(Error("An indexer property requires arguments. For the 'Item' indexer property you can use the syntax 'expr.[idx]'",m))
                else error(Error("Invalid use of a method. Apply the method to its arguments, e.g. 'Type.Method(args)'. Overloads are:  \n\t"^String.concat "\n\t" text,m))

            (* "single named item" rule. This is where we have a single accessible method *)
            (*      memeber x.M(arg1,...,argN) *)
            (* being used in a first-class way, i.e. *)
            (*      x.M  *)
            (* Because there is only one accessible method info available based on the name of the item *)
            (* being accessed we know the number of arguments the first class use of this *)
            (* method will take. Optional and out args are _not_ included, which means they will be resolved *)
            (* to their default values (for optionals) and be part of the return tuple (for out args). *)
            | None,[calledMInfo],Some exprTy -> 
                let methodArgAttribs = param_attrs_of_minfo cenv.g cenv.amap m calledMInfo in 
                let argTys = 
                    methodArgAttribs 
                    |> filter (fun (isParamArrayArg,isOutArg,optArgInfo) -> not isOutArg && (optArgInfo = NotOptional))
                    |> new_inference_typs cenv in
                let dty,rty = unify_fun None cenv denv m exprTy in 
                unifyE cenv env m  dty (mk_tupled_ty cenv.g argTys);
                argTys |> map (fun ty -> CallerArg(ty,m,false,dummyExpr))  
                
            (* "type directed" rule for first-class uses of ambiguous methods. *)
            (* By context we know a type for the input argument. If it's a tuple *)
            (* this gives us the number of arguments expected. Indeed even if it's a variable *)
            (* type we assume the number of arguments is just "1". *)
            | None,_,Some exprTy ->
                let dty,rty = unify_fun None cenv denv m exprTy in 
                let argTys = if is_unit_typ cenv.g dty then [] else  try_dest_tuple_typ dty in 
                argTys |> map (fun ty -> CallerArg(ty,m,false,dummyExpr)) 

            | Some unnamedCallerArgs,_,_ -> 
                unnamedCallerArgs |> map (fun (x,xty,xm) -> CallerArg(xty,xm,false,dummyExpr)) in
        
        let namedCallerArgs = 
            namedCallerArgs |> map (fun (nm,isOpt,x,xty,xm) -> CallerNamedArg(nm,CallerArg(xty,xm,isOpt,dummyExpr)))  in

        let callerArgCounts = (length unnamedCallerArgs, length namedCallerArgs) in

        let mk_CalledMeth (minfo,pinfoOpt) = 
            let minst = freshen_minfo cenv m minfo in
            let userTypeArgs = option_otherwise tyargsOpt minst in 
            if verbose then dprintf4 "--> tc_method_args_and_apps@%a, #minst = %d, #userTypeArgs = %d\n" output_range m (length minst) (length userTypeArgs);
            mk_calledMeth checkingAttributeCall (freshen_minfo cenv) cenv.g cenv.amap m minfo minst userTypeArgs pinfoOpt unnamedCallerArgs namedCallerArgs in 

        let preArgumentTypeCheckingCalledMethGroup = map mk_CalledMeth calledMethGroup in
        let csenv = (mk_csenv cenv.css m denv) in
        let uniquelyResolved = unifyUniqueOverloading csenv callerArgCounts methodName ad preArgumentTypeCheckingCalledMethGroup |> commitOperationResult in
        uniquelyResolved,preArgumentTypeCheckingCalledMethGroup in 
        
    (* STEP 2. Type check arguments *)
    if verbose then dprintf2 "--- tc_method_args_and_apps (step 2)@%a\n" output_range m;

    let unnamedCallerArgs,namedCallerArgs,lambdaVars,rtyOpt,tpenv =  
    
        (* STEP 2a. First extract what we know about the caller arguments, either type-directed if *)
        (* no arguments are given or else based on the syntax of the arguments. *)
        let unnamedCallerArgs,lambdaVars,rtyOpt,tpenv = 
            match unnamedCallerArgsOpt,exprTyOpt with 
            | None,None -> failwith "unreachable"
            | None,Some exprTy ->
                let dty,rty = unify_fun None cenv denv m exprTy in 
                let argTys = if is_unit_typ cenv.g dty then [] else  try_dest_tuple_typ dty in 
                let lambdaVarsAndExprs = argTys |> list_mapi (fun i ty -> mk_compgen_local m ("$arg_"^string_of_int i) ty) in 
                let unnamedCallerArgs = lambdaVarsAndExprs |> map (fun (v,e) -> CallerArg(type_of_expr cenv.g e,range_of_expr e,false,e)) in
                unnamedCallerArgs,Some(map fst lambdaVarsAndExprs), Some(rty),tpenv
            | Some unnamedCallerArgs,_ ->
                let unnamedCallerArgs = map (fun (x,xty,xm) -> CallerArg(xty,xm,false,x)) unnamedCallerArgs in
                let unnamedCallerArgs,tpenv =  tc_method_args cenv env tpenv unnamedCallerArgs in 
                unnamedCallerArgs,None,exprTyOpt,tpenv in 

        let namedCallerArgs = namedCallerArgs |> map (fun (nm,isOpt,x,xty,xm) -> CallerNamedArg(nm,CallerArg(xty,xm,isOpt,x)))  in
        let namedCallerArgs,tpenv =  tc_method_named_args cenv env tpenv namedCallerArgs in 
        unnamedCallerArgs,namedCallerArgs,lambdaVars,rtyOpt,tpenv in

    let preArgumentTypeCheckingCalledMethGroup = 
       preArgumentTypeCheckingCalledMethGroup |> List.map (fun (CalledMeth(minfo,minst,_,_,_,_,_,_,pinfoOpt,_,_,_,_)) -> (minfo, minst, pinfoOpt)) in
    
    (* STEP 3. Resolve overloading *)

    if verbose then dprintf2 "--- tc_method_args_and_apps (step 3)@%a\n" output_range m;

    let (CalledMeth(minfo,minst,_,unnamedCalledArgs,unnamedCallerArgs,mrty,assignedNamedArgs,assignedNamedProps,pinfoOpt,unassignedNamedItems, attributeAssignedNamedItems,unnamedCalledOptArgs,unnamedCalledOutArgs)) = 
        let mk_CalledMeth2 (minfo,minst,pinfoOpt) = 
            let userTypeArgs = option_otherwise tyargsOpt minst in 
            
            if verbose then dprintf1 "--> minfo.Type = %s" (showL (typeL (typ_of_minfo minfo)));
            
            mk_calledMeth checkingAttributeCall (freshen_minfo cenv) cenv.g cenv.amap m minfo minst userTypeArgs pinfoOpt unnamedCallerArgs namedCallerArgs in 
          
        let postArgumentTypeCheckingCalledMethGroup = map mk_CalledMeth2 preArgumentTypeCheckingCalledMethGroup in

        let rtyOpt2 = rtyOpt |> Option.map (fun rty -> (rty, (* dummy : this is unused *) mk_unit cenv.g m)) in 

        let callerArgCounts = (length unnamedCallerArgs, length namedCallerArgs) in
        let csenv = (mk_csenv cenv.css m denv) in
        
        (* Commit unassociated constraints prior to member overload resolution where there is ambiguity *)
        (* about the possible target of the call. *)
        if not uniquelyResolved then 
            canonicalizePartialInferenceProblem (cenv,denv,m)
                 ((match rtyOpt with None -> [] | Some rty -> free_in_type_lr false rty) @
                  (unnamedCallerArgs |> mapConcat (fun (CallerArg(xty,_,_,_)) -> free_in_type_lr false xty)));

        if verbose then dprintf2 "--> tc_method_args_and_apps (resolve overloading) @%a\n" output_range m;
        let result, errors = 
            resolveOverloading csenv NoTrace methodName callerArgCounts ad postArgumentTypeCheckingCalledMethGroup rtyOpt2  in 
        
        (* Raise the errors from the constraint solving *)
        raiseOperationResult errors;
        match result with 
        | None -> error(InternalError("at least one error should be returned by failed method overloading",m))
        | Some res ->  res in

    (* STEP 4. Check the attributes on the corresponding event/property, if any *)

    if verbose then dprintf2 "--- tc_method_args_and_apps (step 4)@%a\n" output_range m;

    Option.iter (fun pinfo -> pinfo_attrib_check cenv.g pinfo m |> commitOperationResult) pinfoOpt;

    let isInstance = nonNil objArgs in 
    minfo_checks cenv.g cenv.amap isInstance ad m minfo;


    (* STEP 5. Build the argument list. Adjust for byref arguments and coercions - refs will have been passed for these - get the address of the contents of the ref. *)

    if verbose then dprintf2 "--- tc_method_args_and_apps (step 5)@%a\n" output_range m;

    (* For unapplied 'e.M' we first evaluate 'e' outside the lambda, i.e. 'let v = e in (fun arg -> v.M(arg))' *)
    let objArgPreBinder,objArgs = 
        match isInstance,objArgs,lambdaVars with 
        | true,[objArg],Some _   -> 
            let objArgTy = type_of_expr cenv.g objArg in
            let v,ve = mk_compgen_local m "objectArg" objArgTy in 
            (fun body -> mk_let m v objArg body), [ve]

        | _ -> 
            (fun e -> e),objArgs in

  
    let allArgs,outArgExprs,outArgTmpBinds = 
        let normalUnnamedArgs = map2 (fun called caller -> AssignedCalledArg(called,caller)) unnamedCalledArgs unnamedCallerArgs in 

        let optArgs = 
          unnamedCalledOptArgs 
            |> map (fun (CalledArg(_,_,optArgInfo,_,_,calledArgTy) as calledArg) -> 
                  let expr = 
                      match optArgInfo with 
                      | NotOptional -> 
                          error(InternalError("unexpected NotOptional",m))
                      | CallerSide None when type_equiv cenv.g calledArgTy cenv.g.obj_ty -> 
                          (* Add an I_nop if this is an initonly field to make sure we never recognize it as an lvalue. See mk_expra_of_expr. *)
                          mk_asm ([ mk_normal_ldsfld (fspec_Missing_Value cenv.g.ilg); I_arith AI_nop ],[],[],[calledArgTy],m)
                      | CallerSide None -> 
                          mk_ilzero(m,calledArgTy)
                      | CallerSide (Some fieldInit) -> 
                          TExpr_const(tc_field_init m fieldInit,m,calledArgTy)  
                      | CalleeSide -> 
                          mk_constr(mk_none_ucref cenv.g,[calledArgTy],[],m) in

                  let callerArg = CallerArg(calledArgTy,m,false,expr) in
                  AssignedCalledArg(calledArg,callerArg)) in 


        let wrapOptionalArg (AssignedCalledArg((CalledArg(_,_,optArgInfo,_,_,_) as calledArg) ,CallerArg(callerArgTy,m,isOptCallerArg,expr)) as assignedArg) = 
            match optArgInfo with 
            | NotOptional -> 
                if isOptCallerArg then errorR(Error("The corresponding formal argument is not optional",m));
                assignedArg

            | _ -> 
                let expr = 
                    match optArgInfo with 
                    | CallerSide _ -> 
                        if isOptCallerArg then 
                            mk_uconstr_field_get(expr,mk_some_ucref cenv.g,[dest_option_ty cenv.g callerArgTy],0,m) 
                        else 
                            expr
                    | CalleeSide -> 
                        if isOptCallerArg then 
                            expr 
                        else
                            mk_constr(mk_some_ucref cenv.g,[callerArgTy],[expr],m) 
                    | _ -> failwith "Unreachable" in
                AssignedCalledArg(calledArg,CallerArg((type_of_expr cenv.g expr),m,isOptCallerArg,expr)) in

        let outArgsAndExprs,outArgTmpBinds = 
            unnamedCalledOutArgs 
              |> map (fun (CalledArg(_,_,_,_,_,calledArgTy) as calledArg) -> 
                let outArgTy = dest_byref_ty cenv.g calledArgTy in
                let outv,outvref,outArgExpr = Tastops.mk_mut_local m "$outarg" outArgTy in (* mutable! *)
                let expr = mk_ilzero(m,outArgTy) in 
                let callerArg = CallerArg(calledArgTy,m,false,mk_val_addr m outvref) in
                (AssignedCalledArg(calledArg,callerArg), outArgExpr), TBind(outv, expr)) 
              |> split in 
        let outArgs, outArgExprs = split outArgsAndExprs in 

        let allArgs =
            map wrapOptionalArg normalUnnamedArgs @ 
            map wrapOptionalArg assignedNamedArgs @ 
            optArgs @ 
            outArgs in 
        
        let allArgs = List.sort (orderOn (fun (AssignedCalledArg(CalledArg(i,_,_,_,_,_),_)) -> i) int_ord) allArgs in 
        allArgs,outArgExprs,outArgTmpBinds in
  
    let coerce (AssignedCalledArg(CalledArg(_,_,optArgInfo,isOutArg,_,calledArgTy),CallerArg(callerArgTy,m,isOptCallerArg,e))) = 
    
       if is_byref_ty cenv.g calledArgTy && is_refcell_ty cenv.g callerArgTy then 
           TExpr_op(TOp_get_ref_lval,[dest_refcell_ty cenv.g callerArgTy],[e],m) 

       else if is_delegate_typ calledArgTy && is_fun_ty callerArgTy then 
           let minfo,del_argtys,del_rty,_ = sig_of_fun_as_delegate cenv.g cenv.amap calledArgTy m in
           coerce_fun_as_delegate None cenv calledArgTy (minfo,del_argtys,del_rty)  (e,callerArgTy) m

       (* note: out args do not need to be coerced *)
       else if isOutArg then 
           e
       (* note: this is (perhaps premature) optimization, since this casts are not reported in quotations *)
       else if type_definitely_subsumes_type_no_coercion 0 cenv.g cenv.amap m calledArgTy callerArgTy then 
           e
       else 
           mk_coerce(e,calledArgTy,m,(type_of_expr cenv.g e)) in
    let args'' = List.map coerce  allArgs in

    (* Make the call expression *)

    if verbose then dprintf2 "--- tc_method_args_and_apps (making call)@%a\n" output_range m;
    let expr,exprty = buildMethodCall cenv env mut m isProp minfo isSuperInit minst objArgs args'' in 

    (* Bind "out" parameters as part of the result tuple *)

    let expr,exprty = 
        if isNil outArgTmpBinds then expr,exprty
        else 
            let outArgTys = (map (type_of_expr cenv.g) outArgExprs) in 
            let expr = if is_unit_typ cenv.g exprty then mk_seq m expr  (mk_tupled cenv.g  m outArgExprs outArgTys)
                       else  mk_tupled cenv.g  m (expr :: outArgExprs) (exprty :: outArgTys) in
            let expr = mk_lets_bind m outArgTmpBinds expr in 
            expr, type_of_expr cenv.g expr in

    (* Handle post-hoc property assignments *)

    if verbose then dprintf2 "--- tc_method_args_and_apps (set properties)@%a\n" output_range m;
    let expr = 
        if isNil assignedNamedProps then expr else 
        (* this holds the result of the call *)
        let objv,_,objExpr = Tastops.mk_mut_local m "$retval" exprty in (* mutable in case it's a struct *)
        (* this expression  mutates the properties on the result of the call *)
        let propSetExpr = 
            fold_left 
                (fun acc (AssignedItemSetter(item,CallerArg(callerArgTy,m,isOptCallerArg,argExpr))) ->
                    if isOptCallerArg then error(Error("invalid optional assignment to a property or field",m));
                    let action = 
                        match item with 
                        | AssignedPropSetter(pminfo,pminst) -> 
                            minfo_checks cenv.g cenv.amap true ad m pminfo;
                            let calledArgTy = List.hd (argtys_of_minfo cenv.g cenv.amap m pminfo pminst) in
                            let argExpr = mk_coerce(argExpr,calledArgTy,m,callerArgTy) in 
                            buildMethodCall cenv env DefinitelyMutates m true pminfo NormalValUse pminst [objExpr] [argExpr] |> fst 

                        | AssignedIlFieldSetter(finfo) ->
                            (* Get or set instance IL field *)
                            il_field_instance_checks  cenv.g cenv.amap ad m finfo;
                            buildIlFieldSet cenv.g m objExpr finfo argExpr 
                        
                        | AssignedRecdFieldSetter(rfinfo) ->
                            rfinfo_instance_checks cenv.g ad m rfinfo; 
                            let _,ftinst,_ = freshenPossibleForallTy cenv m TyparFlexible (vtyp_of_rfinfo rfinfo) in
                            buildRecdFieldSet cenv.g m denv objExpr rfinfo ftinst argExpr  in

                    mk_seq m acc action)
                (mk_unit cenv.g m) 
                assignedNamedProps in 
(*
        if not (minfo_is_ctor minfo) then
            warning(Error("This method call uses named arguments to set properties on the returned object. This construct is under revision and its use is likely to be restricted to object constructors. Consider setting the properties of the returned object explicitly",m));
*)
        (* now put them together *)
        let expr = mk_let m objv expr  (mk_seq m propSetExpr objExpr) in 
        expr in

    (* Build the lambda expression if any *)
    if verbose then dprintf2 "--- tc_method_args_and_apps (build lambda)@%a\n" output_range m;
    let expr = 
        match lambdaVars with 
        | None -> expr
        | Some [] -> mk_unit_delay_lambda cenv.g m expr 
        | Some vs -> mk_multi_lambda m vs (expr, type_of_expr cenv.g expr) in 

    (* Apply the objArgPreBinder, if any - see above *)
    let callExpr = objArgPreBinder expr in 
    
    if verbose then dprintf2 "--- tc_method_args_and_apps (done)@%a\n" output_range m;
    (callExpr,attributeAssignedNamedItems),tpenv
            

and tc_method_args cenv env tpenv args =  map_acc_list (tc_method_arg cenv env) tpenv args
and tc_method_arg  cenv env tpenv (CallerArg(ty,m,isOpt,e)) = let e',tpenv = tc_expr cenv ty env tpenv e in CallerArg(ty,m,isOpt,e'),tpenv
and tc_method_named_args cenv env tpenv args =  map_acc_list (tc_method_named_arg cenv env) tpenv args
and tc_method_named_arg  cenv env tpenv (CallerNamedArg(nm,arg)) = let arg',tpenv = tc_method_arg cenv env tpenv arg in CallerNamedArg(nm,arg'),tpenv


(*-------------------------------------------------------------------------
!* Typecheck "new Delegate(fun x y z -> ...)" constructs
 *------------------------------------------------------------------------- *)

and tc_delegate_ctor_then cenv ty env tpenv m delty arg delayed =
    unifyE cenv env m ty delty;
    let minfo,del_argtys,del_rty,fty = sig_of_fun_as_delegate cenv.g cenv.amap delty m in
    let args = getMethodArgs cenv env arg in 
    match args with 
    | [farg],[] -> 
        let m = range_of_synexpr arg in 
        let fargs',tpenv =  tc_method_args cenv env tpenv [CallerArg(fty,m,false,farg)] in 
        let (CallerArg(_,_,_,farg')) = List.hd fargs' in 
        let expr = coerce_fun_as_delegate None cenv delty (minfo,del_argtys,del_rty) (farg',fty) m  in
        propagate_then_tc_delayed cenv ty env tpenv m expr delty delayed 
    | _ ->  error(Error("A delegate constructor must be passed a single function value",m))


(*-------------------------------------------------------------------------
!* Avoid stack overflow on really larger "let" and "letrec" lists
 *------------------------------------------------------------------------- *)


and bind_letrec binds m e = if isNil binds then e else TExpr_letrec (binds,e,m,new_cache()) 

(* It is helpful for the bootstrapped compiler to process a sequence of iterated lets *)
(* "let ... in let ... in ..." in a tail recursive way *)
and tc_iterated_let_exprs cenv env ty builder tpenv (isRec,isUse,binds,body,m) =
    if verbose then  dprintf2 "--> tc_iterated_let_exprs@%a\n" output_range m;
    if isRec then 
      let binds = map (fun x -> RecBindingDefn(exprContainerInfo,NoNewSlots,ExpressionBinding,x)) binds in 
      if isUse then errorR(Error("A binding may not be marked both 'use' and 'rec'",m));
      let binds,envinner,tpenv = 
        tc_letrec WarnOnOverrides cenv env tpenv (new_inference_typs cenv binds) (binds,m,m) in 
      let body',tpenv = tc_body_expr cenv ty envinner tpenv body  in 
      let body' = bind_letrec binds m body' in 
      fst (builder (body',ty)),tpenv
    else 
      let mkf,envinner,tpenv =
        tc_let cenv isUse env exprContainerInfo ExpressionBinding tpenv (binds,m,range_of_synexpr body) in 
      let builder' x = builder (mkf x) in
      match body with 
      | Expr_let (isRec',isUse',binds',body',m') ->
          tc_iterated_let_exprs cenv envinner ty builder' tpenv (isRec',isUse',binds',body',m')
      | _ -> 
          let body',tpenv = tc_body_expr cenv ty envinner tpenv body  in 
          fst (builder' (body',ty)),tpenv


(*-------------------------------------------------------------------------
!* Typecheck and compile pattern-matching constructs
 *------------------------------------------------------------------------- *)

and tc_and_patcompile_match_clauses exprm matchm onfail cenv dty rty env tpenv clauses =
    if verbose then  dprintf2 "--> tc_and_patcompile_match_clauses@%a\n" output_range matchm;
    let tclauses, tpenv = tc_match_clauses cenv dty rty env tpenv clauses in
    let v,expr = compilePatternForMatchClauses cenv env exprm matchm true onfail dty rty tclauses in 
    v,expr,tpenv

and tc_match_pattern cenv dty env tpenv (pat,opt_when) =
    let m = range_of_synpat pat in 
    if verbose then  dprintf2 "--> tc_match_pattern@%a\n" output_range m;
    let patf',(tpenv,names) = tc_pat WarnOnUpperCase cenv env (OptionalInline,infer_iflex,noArgOrRetAttribs,false,None,false) (tpenv,Map.empty) dty pat in 
    let envinner,values,vspecMap = mkAndPublishSimpleVals cenv env m names in
    let opt_when',tpenv = map_acc_option (tc_expr cenv cenv.g.bool_ty envinner) tpenv opt_when in 
    patf' (TcPatPhase2Input values),opt_when',Namemap.range vspecMap,envinner,tpenv

and tc_match_clauses cenv dty rty env tpenv clauses =
    map_acc_list (tc_match_clause cenv dty rty env) tpenv clauses 

and tc_match_clause cenv dty rty env tpenv (Clause(pat,opt_when,e,patm)) =
    let pat',opt_when',vspecs,envinner,tpenv = tc_match_pattern cenv dty env tpenv (pat,opt_when) in 
    (* todo: consider making this tail-recursive *)
    let e',tpenv = tc_body_expr cenv rty envinner tpenv e in 
    TClause(pat',opt_when',TTarget(vspecs, e'),patm),tpenv

and tc_static_optimization_constraint cenv env tpenv c = 
    match c with 
    | WhenTyparTyconEqualsTycon(tp,ty,m) ->
       if not cenv.g.compilingFslib then 
            warning(LibraryUseOnly(m));
       let ty',tpenv = tc_type cenv NewTyparsOK CheckCxs env tpenv ty in 
       let tp',tpenv = tc_typar cenv env NewTyparsOK tpenv tp in
       TTyconEqualsTycon(mk_typar_ty tp', ty'),tpenv


(*-------------------------------------------------------------------------
!* tc_norm_binding
 *------------------------------------------------------------------------- *)

and tc_norm_binding declKind cenv env tpenv ty ctorThisVarRefCellOpt (enclosingDeclaredTypars,(TIFlex(declaredTypars,infer) as flex)) bind =
    let envinner = add_declared_typars NoCheckForDuplicateTypars (enclosingDeclaredTypars@declaredTypars) env in
    match bind with 

    | NormBinding(vis,pseudo,mut,attrs,doc,_,memberInfo,p,BindingExpr(spatsL,rtyOpt,expr),bindm) ->
        
        (* Check the attributes of the binding *)
        let tc_attrs tgt attrs = 
            if not (can_have_attributes declKind) && nonNil(attrs) then errorR(Error("Attributes are not permitted on 'let' bindings in expressions",bindm));
            tc_attributes cenv envinner tgt attrs in
    
        let attrs' = tc_attrs attrTgtBinding  attrs in 
        let inlineFlag = computeInlineFlag (Option.map (fun (memFlags,_,_) -> memFlags) memberInfo) pseudo mut in 
        let argAttribs = 
            spatsL |> map (SynArgInfo.argdata_of_spats >> map (SynArgInfo.attribsOfArgData >> tc_attrs attrTgtParameter)) in
        let retAttribs = 
            match rtyOpt with Some (_,_,retAttrs) -> tc_attrs attrTgtReturnValue retAttrs | None -> []  in
        let argAndRetAttribs = argAttribs, retAttribs in 

        let isThreadStatic = isThreadOrContextStatic cenv.g attrs' in
        if isThreadStatic && not (implicitly_static declKind) then errorR(Error("Thread static variables must be defined as a static binding in a module",bindm));
        if isThreadStatic && not mut then errorR(Error("Thread static variables must be marked 'mutable'",bindm));
        if isThreadStatic then warning(ThreadStaticWarning(bindm));

        if fsthing_has_attrib cenv.g cenv.g.attrib_ConditionalAttribute attrs' && isNone(memberInfo) then 
            errorR(Error("The 'ConditionalAttribute' attribute may only be used on members",bindm));

        if fsthing_has_attrib cenv.g cenv.g.attrib_EntryPointAttribute attrs' then begin
            if isSome(memberInfo) then 
                errorR(Error("The 'EntryPointAttribute' attribute may only be used on function definitions in modules",bindm))
            else 
                unifyE cenv env bindm ty (mk_il_arr_ty cenv.g 1 cenv.g.string_ty --> cenv.g.int_ty)
        end;

        if mut && pseudo then errorR(Error("Mutable values may not be marked 'inline'",bindm));
        if mut && nonNil declaredTypars then errorR(Error("Mutable values may not have generic parameters",bindm));
        let flex = if mut then no_iflex else flex in 
        if mut && nonNil spatsL then errorR(Error("Mutable function values should be written 'let mutable f = (fun args -> ...)'",bindm));
        let pseudo = 
            if pseudo && isNil(spatsL) && isNil(declaredTypars) then (
                warning(Deprecated("Only functions may be marked 'inline'",bindm));
                false
            ) else (
                pseudo 
            ) in

        let compgen = false in
        
        (* Check the pattern of the l.h.s. of the binding *)
        let tcPatPhase2,(tpenv,nameToPrelimValSchemeMap) = 
            tc_pat AllIdsOK cenv envinner (inlineFlag,flex,argAndRetAttribs,mut,vis,compgen) (tpenv,Namemap.empty) ty p in 
        
        (* Add active pattern result names to the environment *)
        let envinner = 
            match Namemap.range nameToPrelimValSchemeMap with 
            | [PrelimValScheme1(id,_,idty,_,_,_,_,_,_,_) ] ->
                begin match apinfo_of_vname id.idText with 
                | None -> envinner
                | Some apinfo -> modify_nameResEnv (add_apctxt_to_nenv cenv.g apinfo) envinner 
                end
            | _ -> envinner in 
        
        (* Now tc the r.h.s. *)
        (* If binding a ctor then set the somewhat hacky counter that permits us to write ctor expressions on the r.h.s. *)
        let isCtor = (match memberInfo with Some(memFlags,_,_) -> memFlags.memFlagsKind = MemberKindConstructor | _ -> false) in
    (*printf "tc_norm_binding: isCtor=%b\n" isCtor;--------------*)
        let tc = if isCtor then tc_ctor_expr ctorThisVarRefCellOpt else tc_nonbody_expr in 

        let expr',tpenv = tc cenv ty envinner tpenv expr in 

        let hasLiteralAttr,konst = tc_literal cenv ty env tpenv (attrs',expr) in
        if hasLiteralAttr && isThreadStatic then 
            errorR(Error("A literal value may not be given the [<ThreadStatic>] or [<ContextStatic>] attributes",bindm));
        if hasLiteralAttr && mut then 
            errorR(Error("A literal value may not be marked 'mutable'",bindm));
        if hasLiteralAttr && pseudo then 
            errorR(Error("A literal value may not be marked 'inline'",bindm));
        if hasLiteralAttr && nonNil declaredTypars then 
            errorR(Error("Literal values may not have generic parameters",bindm));

        TBindingInfo(inlineFlag,true,attrs',doc,tcPatPhase2,flex,nameToPrelimValSchemeMap,expr',argAndRetAttribs,ty,bindm,compgen,konst),tpenv


and tc_literal cenv ty' env tpenv (attrs',e) = 
    let hasLiteralAttr = fsthing_has_attrib cenv.g cenv.g.attrib_LiteralAttribute attrs' in
    if not hasLiteralAttr then  hasLiteralAttr,None else 
        let expr',tpenv = tc_expr cenv ty' env tpenv e in 
        let rec eval e = 
            match strip_expr e with 
            | TExpr_const(c,_,_) -> c
            | _ -> 
                errorR(Error("This is not a valid constant expression",range_of_expr e));
                TConst_unit in
        hasLiteralAttr,Some(eval expr') 
    
and tc_binding_typar_decls rigid cenv env m tpenv (ValTyparDecls(synTypars,infer,synTyparConstraints)) = 
    let declaredTypars = tc_typar_decls cenv env synTypars in
    let envinner = add_declared_typars CheckForDuplicateTypars declaredTypars env in 
    let tpenv = tc_typar_constraints cenv NoNewTypars CheckCxs envinner tpenv synTyparConstraints in 
    if rigid then declaredTypars |> List.iter (setTyparRigid cenv.g (denv_of_tenv env) m);
    TIFlex(declaredTypars,infer) 

and tc_nonrec_binding_typar_decls cenv env tpenv bind = 
    match bind with 
    | NormBinding(_,_,_,_,_,synTyparDecls,_,_,_,m) -> tc_binding_typar_decls true cenv env m tpenv synTyparDecls

and tc_nonrec_binding declKind cenv env tpenv ty b =
    let b = normBinding ValOrMemberBinding cenv env b in 
    let flex = tc_nonrec_binding_typar_decls cenv env tpenv b in
    tc_norm_binding declKind cenv env tpenv ty None ([],flex) b 

and tc_rec_binding declKind cenv env tpenv ty ctorThisVarRefCellOpt flex b =
    let b = normBinding ValOrMemberBinding cenv env b in 
    tc_norm_binding declKind cenv env tpenv ty ctorThisVarRefCellOpt flex b

(*-------------------------------------------------------------------------
!* tc_attribute*
 *------------------------------------------------------------------------*)

and tc_attribute cenv (env:tcEnv) attrTgt (Attr(tycon,arg,targetIndicator,m))  =
    let (typath,tyid) = frontAndBack tycon in 
    let tpenv = emptyTpenv in 
    let ty,tpenv =  
        let try1 n = 
            let tyid = mksyn_id tyid.idRange n in  
            let tycon = (typath @ [tyid]) in 
            match tc_tycon_id OpenQualified cenv.g cenv.amap m env.eNameResEnv tycon 0 with
            | Raze err -> raze(err)
            | _ ->  success(tc_type_and_recover cenv NoNewTypars CheckCxs env tpenv (Type_app(tycon,[],m)) )in
        forceRaise ((try1 (tyid.idText^"Attribute")) |> Outcome.otherwise (fun () -> (try1 tyid.idText))) in 
  (*   if not (ty <: System.Attribute) then error (Error("A custom attribute must be a subclass of System.Attribute",m)); *)

    let family = accessRightsOfEnv env in 

    if not (tcref_of_typ_accessible family ty) then  errorR(Error("This type is not accessible from this code location",m));

    let tcref = tcref_of_stripped_typ ty in 
    (* REVIEW: take notice of allowMultiple, inherited *)
    let validOn,allowMultiple,inherited = 
      let validOnDefault = Int32.of_int 0x7fff in 
      let allowMultipleDefault = false in 
      let inheritedDefault = true in 
      if is_il_tcref tcref then 
        let _,_,tdef = dest_il_tcref  tcref in 
        let (AttribInfo(tref,_)) = cenv.g.attrib_AttributeUsageAttribute in
        
        match ilthing_decode_il_attrib cenv.g tref tdef.tdCustomAttrs with 
        | Some ([CustomElem_int32 validOn ],named) -> 
            let allowMultiple = 
              (choose (function ("AllowMultiple",_,_,CustomElem_bool res) -> Some res | _ -> None) named) 
              +? (fun () -> allowMultipleDefault) in 
            let inherited = 
              (choose (function ("Inherited",_,_,CustomElem_bool res) -> Some res | _ -> None) named) 
              +? (fun () -> inheritedDefault) in 
            (validOn, allowMultiple,inherited)
        | Some ([CustomElem_int32 validOn; CustomElem_bool allowMultiple; CustomElem_bool inherited ],_) -> 
            (validOn, allowMultiple,inherited)
        | _ -> 
            (validOnDefault, allowMultipleDefault,inheritedDefault)
      else
        match (fsthing_tryfind_attrib cenv.g cenv.g.attrib_AttributeUsageAttribute (attribs_of_tcref tcref)) with
        | Some(Attrib(_,[ TExpr_const (TConst_int32(validOn),_,_) ],_)) ->
            (validOn, allowMultipleDefault,inheritedDefault)
        | Some(Attrib(_,[ TExpr_const (TConst_int32(validOn),_,_);
                          TExpr_const (TConst_bool(allowMultiple),_,_);
                          TExpr_const (TConst_bool(inherited),_,_)],_)) ->
            (validOn, allowMultiple,inherited)
        | _ -> 
            (validOnDefault, allowMultipleDefault,inheritedDefault) in 
    let possibleTgts = validOn &&& attrTgt in 
    let possibleTgts = match targetIndicator with
                       | Some id when id.idText = "assembly" -> attrTgtAssembly
                       | Some id when id.idText = "module" -> attrTgtModule
                       | Some id when id.idText = "return" -> attrTgtReturnValue
                       | Some id when id.idText = "field" -> attrTgtField
                       | Some id when id.idText = "property" -> attrTgtProperty
                       | Some id when id.idText = "type"    -> attrTgtTyconDecl
                       | Some id when id.idText = "constructor"    -> attrTgtConstructor
                       | Some id when id.idText = "event"    -> attrTgtEvent
                       | _ -> possibleTgts in
    if (possibleTgts &&& attrTgt) = 0l then 
      if (possibleTgts = attrTgtAssembly or possibleTgts = attrTgtModule) then 
        warning(Error("This attribute is not valid for use on this language element. Assembly attributes should be attached to a 'do ()' declaration, if necessary within an F# module",m))
      else
        warning(Error("This attribute is not valid for use on this language element",m));

    let item,rest = forceRaise (tc_tdef_ctor (denv_of_tenv env) cenv.g cenv.amap m ty) in 
    let attrib = 
      match item with 
      | Item_ctor_group(methodName,minfos) ->
          let meths = List.map (fun minfo -> minfo,None) minfos in 

          let (expr,namedCallerArgs),tpenv = 
            tc_method_args_and_apps true cenv env tpenv None [] m methodName family PossiblyMutates false meths NormalValUse [arg] None in 

          unifyE cenv env m ty (type_of_expr cenv.g expr);

          let namedAttribArgMap = 
            namedCallerArgs |> List.map (fun (CallerNamedArg(nm,CallerArg(argtyv,m,isOpt,expr))) ->
              if isOpt then error(Error("Optional arguments may not be used in custom attributes",m));
              let m = range_of_expr expr in 
              let nm,isProp,argty = 
                match forceRaise (tc_lid_in_typ (nenv_of_tenv env) 0 cenv.ginstf cenv.g cenv.amap m [mksyn_id m nm] IgnoreOverrides defaultTypeNameResInfo ty |> fst |> at_most_one_result m) with   
                | Item_property (_,[pinfo]),[] -> 
                    if not (pinfo_has_setter pinfo) then 
                      errorR(Error("This property may not be set",m));
                    nm,true,vtyp_of_pinfo cenv.g cenv.amap m pinfo
                | Item_il_field finfo,[] -> 
                    il_finfo_accessible_check cenv.g cenv.amap m family finfo;
                    il_finfo_attrib_check cenv.g finfo m;
                    nm,false,vtyp_of_il_finfo cenv.amap m finfo
                | Item_recdfield rfinfo,[] when not (rfinfo_is_static rfinfo) -> 
                    rfinfo_attrib_check cenv.g rfinfo m  |> commitOperationResult;        
                    rfinfo_accessible_check m family rfinfo;
                    (* This uses the F# backend name mangling of fields.... *)
                    let nm =  gen_field_name (deref_tycon (tcref_of_rfinfo rfinfo)) (rfield_of_rfinfo rfinfo) in
                    nm,false,vtyp_of_rfinfo rfinfo
                |  _ -> 
                    errorR(Error("This property or field was not found on this custom attribute type",m)); 
                    nm,false,cenv.g.unit_ty in 

              type_must_subsume_type (denv_of_tenv env) cenv.css m NoTrace argty argtyv;

              (nm,argty,isProp,expr)) in 

          begin match expr with 
          | TExpr_op(TOp_ilcall((virt,protect,valu,_,_,_,_,_,mref),[],[],rtys),[],args,m) -> 
              if valu then error (Error("A custom attribute must be a reference type",m));
              if List.length args <> List.length (args_of_mref mref) then error (Error("The number of args for a custom attribute does not match the expected number of args for the attribute constructor",m));
              Attrib(ILAttrib(mref),args,namedAttribArgMap)

          | TExpr_app(TExpr_val(vref,_,_),_,_,args,_) -> 
              let try_dest_unit_or_tuple = function TExpr_const(TConst_unit,_,_) -> [] | expr -> try_dest_tuple expr in 
              Attrib(FSAttrib(vref),concat (map (try_dest_unit_or_tuple) args),namedAttribArgMap)

          | _ -> error (Error("A custom attribute must invoke an object constructor",m))
          end

      | _ -> error(Error("attribute expressions must be calls to object constructors",m)) in 
    (possibleTgts, attrib)

and tc_attributes_with_possible_targets cenv env attrTgt attrs = 
    mapConcat (fun attr -> try [tc_attribute cenv env attrTgt attr] with e -> errorRecoveryPoint e; []) attrs        
and tc_attributes cenv env attrTgt attrs = 
    tc_attributes_with_possible_targets cenv env attrTgt attrs |> map snd

(*-------------------------------------------------------------------------
!* tc_let
 *------------------------------------------------------------------------*)

and tc_let cenv isUse env containerInfo declKind tpenv (binds,bindsm,scopem) =
    let binds',tpenv = map_acc_list (fun tpenv b -> tc_nonrec_binding declKind cenv env tpenv (new_inference_typ cenv ()) b) tpenv binds in 
    let (ContainerInfo(altActualParent,tcrefContainerInfo)) = containerInfo in
    
    (* Canonicalize constraints prior to generalization *)
    let denv = denv_of_tenv env in
    canonicalizePartialInferenceProblem (cenv,denv,bindsm) 
        (binds' |> mapConcat (fun tbinfo -> 
            let (TBindingInfo(inlineFlag,immut,attrs,doc,tcPatPhase2,flex,nameToPrelimValSchemeMap,rhs,_,tauTy,m,_,_)) = tbinfo in
            let (TIFlex(declaredTypars,canInferTypars)) = flex in
            let maxInferredTypars = (free_in_type_lr false tauTy) in
            declaredTypars @ maxInferredTypars));

    let freeInEnv = computeUngeneralizableTypars env in 

    let (mkf,env,tpenv) = 
      fold_left
        (fun (mkf_sofar,env,tpenv) tbinfo -> 
            let (TBindingInfo(inlineFlag,immut,attrs,doc,tcPatPhase2,flex,nameToPrelimValSchemeMap,rhs,_,tauTy,m,_,konst)) = tbinfo in
            let enclosingDeclaredTypars  = [] in
            let (TIFlex(declaredTypars,canInferTypars)) = flex in
            let allDeclaredTypars  =  enclosingDeclaredTypars @ declaredTypars in
            let generalizedTypars,prelimValSchemes2 = 
                let canInferTypars = computeCanInferTypars(declKind,canInferTypars,None,declaredTypars,m) in
                (*if declaredTypars <> [] then List.iter (fun tp -> dprintf3 "%a: declared typar %s\n" output_range m (name_of_typar tp)) declaredTypars;*)
                (* dprintf3 "%a: tauTy = %s\n" output_range m (NicePrint.pretty_string_of_typ (empty_denv cenv.g) tauTy); *)
                let maxInferredTypars = (free_in_type_lr false tauTy) in
                let generalizedTypars = computeGeneralizedTypars(cenv,denv, m, immut, freeInEnv, canInferTypars, canGeneralizeConstrainedTyparsForDecl(declKind), inlineFlag, Some rhs, allDeclaredTypars, maxInferredTypars) in 
                (* if generalizedTypars <> [] then List.iter (fun tp -> dprintf3 "%a: generalized typar %s\n" output_range m (name_of_typar tp)) generalizedTypars; *)
                let prelimValSchemes2 = generalizeVals cenv denv enclosingDeclaredTypars  [] generalizedTypars nameToPrelimValSchemeMap in
                generalizedTypars,prelimValSchemes2 in

            (* REVIEW: this scopes generalized type variables. Ensure this is handled properly *)
            (* on all other paths. *)
            let tpenv = hideUnscopedTypars generalizedTypars tpenv in
            let valschemes = inferPartialArities declKind rhs prelimValSchemes2 in
            let values = mkAndPublishVals cenv env (altActualParent,false,declKind,ValNotInRecScope,valschemes,attrs,doc,konst) in 
            let pat' = tcPatPhase2 (TcPatPhase2Input values) in 
            let prelimRecValues = Namemap.map fst values in 
            
            (* Now bind the r.h.s. to the l.h.s. *)
            let rhse = mk_tlambda m generalizedTypars (rhs,tauTy) in
            (* For some reason the code below passes the type of the expression along too - but why not use type_of_expr? *)
            let mk_typed f (tm,tmty) = (f tm, tmty) in

(*
            match pat' with 
              (* nice: don't introduce temporary or 'let' for 'match against wild' or 'match against unit' *)

            | (TPat_wild _ | TPat_const (TConst_unit,_)) ->
                (mk_typed (mk_seq m rhse) << mkf_sofar,env,tpenv)
                
            | _ -> 
*)
              (* nice: don't introduce awful temporary for r.h.s. in the 99% case where we know what we're binding it to *)
                let tmp,tmpe,pat'' = 
                    match pat' with 
                      (* nice: don't introduce awful temporary for r.h.s. in the 99% case where we know what we're binding it to *)
                    | TPat_as (pat1,PBind(v,TypeScheme(generalizedTypars',_,_)),m1) 
                        when length generalizedTypars' = length generalizedTypars && 
                             for_all2 tpspec_eq generalizedTypars generalizedTypars' -> 
                          (* if generalizedTypars <> [] then dprintf3 "%a: ty = %s\n" output_range m (NicePrint.pretty_string_of_typ (empty_denv cenv.g) (type_of_val v));*)
                          v,expr_for_val (range_of_val v) v, pat1
                    | _ when mustinline(inlineFlag)  -> error(Error("invalid inline specification",m))
                    | _ -> 
                        let tmp,tmpe = Tastops.mk_compgen_local m (cenv.niceNameGen.nngApply "patrhs" m) (generalizedTypars +-> tauTy) in 
                        (* This assignement forces representation as declKind value, to maintain the invariant from the *)
                        (* type checker that anything related to binding declKind-level values is marked with an *)
                        (* arity. *)
                        if isUse then 
                            errorR(Error("'use' bindings must be of the form 'use <var> = <expr>'",m));
                        if (must_have_arity declKind) then (data_of_val tmp).val_arity <- Some(infer_arity_of_expr_bind tmp rhse);
                        tmp,tmpe,pat' in 
                let mk_rhs_bind (tm,tmty) = mk_let_typed m tmp rhse (tm,tmty) in
                let allValsDefinedByPattern = (Namemap.range prelimRecValues) in
                let mk_pat_bind (tm,tmty) =
                    let valsDefinedByMatching = gen_remove vspec_eq tmp allValsDefinedByPattern in
                    let matchx = compilePatternForMatch cenv env m m true Incomplete (tmp,generalizedTypars) [TClause(pat'',None,TTarget(valsDefinedByMatching,tm),m)] tmty in
                    let matchx = if (convert_to_linear_bindings declKind) then linearise_top_match cenv.g matchx else matchx in
                    matchx,tmty in 
                let mk_cleanup (tm,tmty) =
                    if isUse then 
                        fold_right 
                            (fun v (tm,tmty) ->
                                type_must_subsume_type denv cenv.css (range_of_val v) NoTrace cenv.g.system_IDisposable_typ (type_of_val v);
                                let cleanupE = buildDisposeCleanup cenv env m v in
                                mk_try_finally cenv.g (tm,cleanupE,m,tmty),tmty)
                            allValsDefinedByPattern
                            (tm,tmty)
                    else (tm,tmty)  in
                    
                (mk_rhs_bind << mk_pat_bind << mk_cleanup << mkf_sofar,
                 addLocalValMap scopem prelimRecValues env,
                 tpenv))
        ((fun x -> x), env, tpenv)
        binds' in
    (mkf,env,tpenv)

and tc_let_bindings cenv env containerInfo declKind tpenv (binds,bindsm,scopem) =
    (* Return binds corresponding to the linearised let-bindings.
     * This reveals the bound items, e.g. when the lets occur in incremental object defns.
     * RECAP:
     *   The LHS of let-bindings are patterns.
     *   These patterns could fail, e.g. "let Some x = ...".
     *   So letbindings could contain a fork at a match construct, with one branch being the match failure.
     *   If bindings are linearised, then this fork is pushed to the RHS.
     *   In this case, the letbindings type check to a sequence of let bindings.
     *)
    assert(convert_to_linear_bindings declKind);
    let isUse = false in (* 'use' bindings not valid in classes *)
    let mkf,env,tpenv = tc_let cenv false env containerInfo declKind tpenv (binds,bindsm,scopem) in
    let unite = mk_unit cenv.g bindsm in  
    let expr,ty = mkf (unite,cenv.g.unit_ty) in
    let rec strip_lets = function
      | TExpr_let (bind,body,_,_)      -> bind :: strip_lets body
      | TExpr_const (TConst_unit,_,_) -> []
      | _ -> failwith "tc_let_bindings: let sequence is non linear. Maybe a LHS pattern was not linearised?" in
    let binds = strip_lets expr in
    binds,env,tpenv

and checkMemberFlags g optIntfSlotTy newslotsOK overridesOK memFlags m = 
    if newslotsOK = NoNewSlots && (memFlags.memFlagsVirtual || memFlags.memFlagsAbstract) then 
      errorR(Error("Abstract members are not permitted in an augmentation - they must be defined as part of the type itself",m));
    if overridesOK = WarnOnOverrides && (memFlags.memFlagsOverride) && isNone optIntfSlotTy then 
      warning(OverrideInAugmentation(m))
    
(*-------------------------------------------------------------------------
!* applyTypesFromArgumentPatterns
 *------------------------------------------------------------------------*)

(* This applies the pre-assumed knowledge available to type inference prior to looking at *)
(* the _body_ of the binding. For example, in a letrec we may assume this knowledge *)
(* for each binding in the letrec prior to any type inference. This might, for example, *)
(* tell us the type of the arguments to a recursive function. *)
and applyTypesFromArgumentPatterns cenv env optArgsOK ty m tpenv (BindingExpr(pushedPats,pushedRtyOpt,e)) =  
  match pushedPats with
  | [] ->
      begin match pushedRtyOpt with 
      | None -> ()
      | Some (rty,m,_) -> 
          let rty',_ = tc_type_and_recover cenv NewTyparsOK CheckCxs env tpenv rty in 
          unifyE cenv env m ty rty'
      end
  | p :: t -> 
      let dty,rty = unify_fun None cenv (denv_of_tenv env) m ty in 
      ignore (tc_simple_pats cenv optArgsOK dty env (tpenv,Map.empty) p);
      applyTypesFromArgumentPatterns cenv env optArgsOK rty m tpenv (BindingExpr(t,pushedRtyOpt,e))

(*-------------------------------------------------------------------------
!* computeIsComplete
 *------------------------------------------------------------------------*)

(** Do the type annotations give the full and complete generic type? If so, enable generic recursion *)
and computeIsComplete enclosingDeclaredTypars declaredTypars ty = 
    Zset.is_empty (List.fold_left (fun acc v -> Zset.remove v acc) (free_in_type ty).free_loctypars (enclosingDeclaredTypars@declaredTypars)) 

(*-------------------------------------------------------------------------
!* applyAbstractSlotInference
 *------------------------------------------------------------------------*)

(** Determine if a uniquely-identified-abstract-slot exists for an override member (or interface member implementation) based on the information available *)
(** at the syntactic definition of the member (i.e. prior to type inference). If so, we know the expected signature of the override, and the full slotsig *)
(** it implements. Apply the inferred slotsig. *)
and applyAbstractSlotInference cenv denv envinner (rty,bindm,synTyparDecls,declaredTypars,memberName,tcrefObjTy,renaming,objTy,optIntfSlotTy,topValSynData,memFlags) = 

    let typToSearchForAbstractMembers = 
        match optIntfSlotTy with 
        | Some (ty,abstractSlots) -> 
            (* The interface type is in terms of the type's type parameters. *)
            (* We need a signature in terms of the values' type parameters. *)
            ty,Some(abstractSlots) 
        | None -> 
            tcrefObjTy,None in 

    (* Determine if a uniquely-identified-override exists based on the information *)
    (* at the member signature. If so, we know the type of this member, and the full slotsig *)
    (* it implements. Apply the inferred slotsig. *)
    if memFlags.memFlagsOverride then 
         
        match memFlags.memFlagsKind with 
        | MemberKindMember -> 
             let virts,virtsArityMatch = 
                 abstract_minfos_for_syn_method memberName cenv.g cenv.amap bindm typToSearchForAbstractMembers topValSynData in 

             let uniqueAbstractMethOpt = 
                 match virts with 
                 | [] -> 
                     errorR(Error("No abstract or interface member was found that corresponds to this override",memberName.idRange));
                     None

                 | _ -> 
                     match virtsArityMatch with 
                     | [minfo] -> Some minfo
                     | [] -> 

                         errorR(Error("This override takes a different number of arguments to the corresponding abstract member",memberName.idRange));
                         None

                     | minfo2 :: minfo3 :: _ -> 

                         (* We hit this case when it is ambiguous which abstract method is being implemented. *)
                         (* REVIEW: consider removing this warning. It triggers even when compiling the F# library *)
                         warning(NonUniqueInferredAbstractSlot(cenv.g,denv, memberName.idText, minfo2, minfo3,memberName.idRange));

                         None in 
               
             
             (* If we determined a unique member then utilize the type information from the slotsig *)
             (* REVIEW: this is not complete when overriding generic abstract methods (rare) *)
             (* REVIEW: propagate types from abstract slot to override when overriding generic abstract methods *)
             let declaredTypars = 
                 match uniqueAbstractMethOpt with 
                 | None -> declaredTypars 
                 | Some uniqueAbstractMeth -> 

                     let uniqueAbstractMeth = inst_minfo cenv.amap bindm renaming uniqueAbstractMeth in
                     
                     let typarsFromAbsSlotAreRigid,typarsFromAbsSlot,argTysFromAbsSlot, retTyFromAbsSlot = 
                         freshenAbstractSlot cenv.g cenv.amap bindm synTyparDecls uniqueAbstractMeth in

                     let declaredTypars = (if typarsFromAbsSlotAreRigid then typarsFromAbsSlot else declaredTypars) in 

                     let absty = mk_meth_ty cenv.g argTysFromAbsSlot retTyFromAbsSlot in 

                     unifyE cenv envinner bindm rty absty;
                     declaredTypars in

             (* OK, detect 'default' members in the same type and mark them as implemented. *)
             (* REVIEW: consider moving this to the all-implemented analysis at the end of the *)
             (* type-checking scope, since we have inferred full signature types at that point. *)
             (* HOWEVER: we need to know which members have default implementations in order to be *)
             (* able to typecheck obejct expressions and sub-classes. *)
             begin match filter (fun virt -> cenv.g.tcref_eq (tcref_of_stripped_typ objTy) (tcref_of_stripped_typ (typ_of_minfo virt))) virtsArityMatch with 
             | [] -> ()
             | [FSMeth(_,vref)] -> 
                 let virt_vspr = the (member_info_of_vref vref) in 
                 if virt_vspr.vspr_implemented then errorR(Error("This method already has a default implementation",memberName.idRange));
                 virt_vspr.vspr_implemented <- true

             | _ ->  
                 errorR(Error("The method implemented by this default is ambiguous",memberName.idRange));

             end;

             (* What's the type containing the abstract slot we're implementing? Used later on in mkMemberDataAndUniqueId. *)
             (* This type must be in terms of the enclosing type's formal type parameters, hence the application of revRenaming *)

             let optInferredImplSlotTy = 
                 match optIntfSlotTy with 
                 | Some (x,_) -> Some x 
                 | None -> Option.map typ_of_minfo uniqueAbstractMethOpt in

             optInferredImplSlotTy,declaredTypars

       | MemberKindPropertyGet 
       | MemberKindPropertySet as k ->
           let virts = abstract_pinfos_for_syn_property memberName cenv.g cenv.amap bindm typToSearchForAbstractMembers k topValSynData in 

           (* Only consider those abstract slots where the get/set flags match the value we're defining *)
           let virts = 
               virts 
               |> filter (fun pinfo -> (pinfo_has_getter pinfo && k=MemberKindPropertyGet) ||
                                       (pinfo_has_setter pinfo && k=MemberKindPropertySet)) in 
                                       
           (* Find the unique abstract slot if it exists *)
           let uniqueAbstractPropOpt = 
               match virts with 
               | [] -> 
                   errorR(Error("No abstract property was found that corresponds to this override",memberName.idRange)); 
                   None
               | [uniqueAbstractProp] -> Some(uniqueAbstractProp)
               | _ -> 
                   (* We hit this case when it is ambiguous which abstract property is being implemented. *)
                   (* REVIEW: Consider disallowing overrides whose corresponding abstract slot cannot be *)
                   (* determined based on syntactic (rather than inferred) information alone. *)
                   None in

           (* If we determined a unique member then utilize the type information from the slotsig *)
           uniqueAbstractPropOpt |> Option.iter (fun uniqueAbstractProp -> 

               let kIsGet = (k = MemberKindPropertyGet) in 

               if not ((if kIsGet then pinfo_has_getter else pinfo_has_setter) uniqueAbstractProp) then 
                   error(Error("This property overrides or implements an abstract property but the abstract property doesn't have a corresponding "^(if kIsGet then "getter" else "setter"),memberName.idRange));

               let uniqueAbstractMeth = (if kIsGet then getter_minfo_of_pinfo else setter_minfo_of_pinfo) uniqueAbstractProp in

               let uniqueAbstractMeth = inst_minfo cenv.amap bindm renaming uniqueAbstractMeth in

               let typarsFromAbsSlotAreRigid,typarsFromAbsSlot,argTysFromAbsSlot, retTyFromAbsSlot = 
                    freshenAbstractSlot cenv.g cenv.amap bindm synTyparDecls uniqueAbstractMeth in

               if nonNil(typarsFromAbsSlot) then 
                   errorR(InternalError("unexpected generic property",memberName.idRange));

               let absty = 
                   if (memFlags.memFlagsKind = MemberKindPropertyGet) 
                   then mk_meth_ty cenv.g argTysFromAbsSlot retTyFromAbsSlot 
                   else 
                     match argTysFromAbsSlot with 
                     | [] -> warning(Error("strange signature for set member",memberName.idRange)); retTyFromAbsSlot --> cenv.g.unit_ty
                     | [h] -> h --> cenv.g.unit_ty
                     | _ ->
                       let a,b = frontAndBack argTysFromAbsSlot in 
                       mk_tupled_ty cenv.g a --> (b --> cenv.g.unit_ty) in 

               unifyE cenv envinner bindm rty absty);
           

           (* OK, detect 'default' members in the type being defined and mark them as implemented. *)
           (* REVIEW: consider moving this to the all-implemented analysis at the end of the *)
           (* type-checking scope, since we have inferred full signature types at that point. *)
           (* HOWEVER: we need to know which members have default implementations in order to be *)
           (* able to typecheck object expressions and sub-classes. Hence it MUST be here. *)
           begin 
               let assertImplements(vref) =
                   let virt_vspr = the (member_info_of_vref vref) in 
                   if virt_vspr.vspr_implemented then errorR(Error("This property already has a default implementation",memberName.idRange));
                   virt_vspr.vspr_implemented <- true in 
               let relevant = 
                   virts 
                   |> filter (fun virt -> cenv.g.tcref_eq (tcref_of_stripped_typ objTy) (tcref_of_stripped_typ (typ_of_pinfo virt))) in
               match relevant  with 
               | [] -> ()
               | [FSProp(_,Some vref,_)] when k=MemberKindPropertyGet -> assertImplements(vref)
               | [FSProp(_,_,Some vref)] when k=MemberKindPropertySet -> assertImplements(vref)
               | _ ->  errorR(Error("The property implemented by this default is ambiguous",memberName.idRange));
           end;
           
           (* What's the type containing the abstract slot we're implementing? Used later on in mkMemberDataAndUniqueId. *)
           (* This type must be in terms of the enclosing type's formal type parameters, hence the application of revRenaming *)
           
           let optInferredImplSlotTy = 
               match optIntfSlotTy with 
               | Some (x,_) -> Some x 
               | None -> Option.map (typ_of_pinfo) uniqueAbstractPropOpt in

           optInferredImplSlotTy,declaredTypars

       | _ -> 
           match optIntfSlotTy with 
           | Some (x,_) -> Some x, declaredTypars 
           | None -> None, declaredTypars

    else
    
       (* Check for use of an instance member that obscures an abstract member. *)
       (* REVIEW: consider moving this to the all-implemented analysis at the end of the *)
       (* type-checking scope, since we have inferred full signature types at that point. *)
       begin
           if memFlags.memFlagsInstance then begin 
             match memFlags.memFlagsKind with 
             | MemberKindMember -> 
                 let virts,virtsArityMatch = abstract_minfos_for_syn_method memberName cenv.g cenv.amap bindm typToSearchForAbstractMembers topValSynData in 
                 begin match virtsArityMatch with 
                 | uniqueAbstractMeth2 :: _ -> warning(Error("This new member hides the abstract member '"^string_of_minfo cenv.g cenv.amap bindm denv uniqueAbstractMeth2^"'. Rename the member or use 'override' instead",memberName.idRange))
                 | [] ->  ()
                 end
             | MemberKindPropertyGet 
             | MemberKindPropertySet as k ->
                 let virts = abstract_pinfos_for_syn_property memberName cenv.g cenv.amap bindm typToSearchForAbstractMembers k topValSynData in 
                 begin match virts with 
                 | uniqueAbstractProp :: _ -> 
                     let uniqueAbstractMeth = (if k = MemberKindPropertyGet then getter_minfo_of_pinfo else setter_minfo_of_pinfo) uniqueAbstractProp in
                     warning(Error("This new member hides the abstract member '"^string_of_minfo cenv.g cenv.amap bindm denv uniqueAbstractMeth^"'. Rename the member or use 'override' instead",memberName.idRange))
                 | [] -> ()
                 end;
             | _ -> ()
           end;  (* end of lack of "override" warning analysis *)
           None, declaredTypars 
       end

(*-------------------------------------------------------------------------
!* tc_letrec - mk_rec_value(s)
 * 
 * This is a major routine that generates the val_spec for a recursive binding 
 * prior to the analysis of the definition of the binding. This includes
 * members of all flavours (including properties, implicit class constructors
 * and overrides). At this point we perform override inference, to infer
 * which method we are overriding, in order to add constraints to the
 * implementation of the method.
 *------------------------------------------------------------------------*)

and checkForNonAbstractInterface declKind tcref memFlags m =
    if is_interface_tcref tcref then begin
        if memFlags.memFlagsKind = MemberKindClassConstructor then 
            error(Error("Interfaces may not contain definitions of static initializers",m))
        else if memFlags.memFlagsKind = MemberKindConstructor then 
            error(Error("Interfaces may not contain definitions of object constructors",m))
        else if memFlags.memFlagsOverride or memFlags.memFlagsVirtual then 
            error(Error("Interfaces may not contain definitions of member overrides",m))
        else if not (declKind=ExtensionBinding || memFlags.memFlagsAbstract ) then
            error(Error("Interfaces may not contain definitions of concrete members. You may need to define a constructor on your type, or use implicit class construction, to indicate that the type is a concrete implementation class",m))
    end

and mk_rec_value overridesOK cenv env (tpenv,nameValueMap) (ty,NormRecBindingDefn(containerInfo,newslotsOK,declKind,binding)) =
    let denv = denv_of_tenv env in 
    match binding with 
    | NormBinding(vis1,pseudo,mut,attrs,doc,synTyparDecls,memberInfo,p,(BindingExpr(pushedPats,pushedRtyOpt,e) as rhs),bindm) ->

        let inlineFlag = computeInlineFlag (Option.map (fun (memFlags,_,_) -> memFlags) memberInfo) pseudo mut in 
        if mut then errorR(Error("only record fields and simple 'let' bindings may be marked mutable",bindm));
        let flex = tc_binding_typar_decls false cenv env bindm tpenv synTyparDecls in 
        let (TIFlex(declaredTypars,infer)) = flex in
        let (ContainerInfo(altActualParent,tcrefContainerInfo)) = containerInfo in
        
        (* dprintf2 "mk_rec_value: declaredTypars = %s, ty = %s\n" (Layout.showL  (typarsL declaredTypars)) (Layout.showL  (typeL ty));  *)
        
        let envinner = add_declared_typars CheckForDuplicateTypars declaredTypars env in 
        let attrs' = tc_attributes cenv envinner attrTgtBinding attrs in 
        let hasLiteralAttr, konst = tc_literal cenv ty env tpenv (attrs',e) in

        let optArgsOK = isSome(memberInfo) in
        let rec lookPat p = 
            match p with  
            | Pat_typed(pat',cty,_) -> 
                let cty',tpenv = tc_type_and_recover cenv NewTyparsOK CheckCxs envinner tpenv cty in 
                unifyE cenv envinner bindm ty cty';
                lookPat pat' 
            | Pat_attrib(pat',attribs,_) -> 
                lookPat pat' 

            (* for 'let rec x = ... and do ... and y = ...' *)
            | Pat_const (Const_unit, m) -> 
                 let id = ident (cenv.niceNameGen.nngApply "_doval" m,m) in
                 lookPat (Pat_as (Pat_wild m, id,false,None,m))
                 
            | Pat_as (Pat_wild _, id,_,vis2,m) -> 
                (* Check if we're defining a member, in which case generate the internal unique *)
                (* name for the member and the information about which type it is agumenting *)
                let vis = combineVisibilityAttribs vis1 vis2 id.idRange in
                let vspr,uniquen,enclosingDeclaredTypars,envinner,ctorThisVarRefCellOpt,baseVarOpt = 
                  
                    match tcrefContainerInfo,memberInfo with 
                    | Some(tcref,optIntfSlotTy,baseVarOpt),
                      Some(memFlags,topValSynData,thisIdOpt) -> 
                      
                        checkMemberFlags cenv.g optIntfSlotTy newslotsOK overridesOK memFlags bindm;
                        begin 
                          checkForNonAbstractInterface declKind tcref memFlags id.idRange;
                              
                          if tycon_is_exnc (deref_tycon tcref) && 
                             (memFlags.memFlagsKind = MemberKindConstructor) then 
                              error(Error("Constructors may not be specified in exception augmentations",id.idRange));                  
                        end;

                        let _,enclosingDeclaredTypars,_,objTy,thisTy = freshenThisTy cenv bindm recTyparsRigid tcref in 
                        let envinner = add_declared_typars CheckForDuplicateTypars enclosingDeclaredTypars envinner in 
                        let envinner = mk_inner_env_for_tcref envinner tcref (declKind = ExtensionBinding) in 
     
                        (* dprintf2 "mk_rec_value, : enclosingDeclaredTypars = %s, ty = %s\n" (Layout.showL  (typarsL enclosingDeclaredTypars)) (Layout.showL  (typeL ty));   *)
            
                        let ctorThisVarRefCellOpt,baseVarOpt = 
                          match memFlags.memFlagsKind with 
                          | MemberKindConstructor  ->
                              (* A fairly adhoc place to put this check *)
                              if is_struct_tcref tcref && (match memberInfo with Some(_,TopValSynData([[]],_),_) -> true | _ -> false) then
                                  errorR(Error("Structs may not have an object constructor with no arguments. This is a restriction imposed on all .NET languages as structs automatically support a default constructor",bindm));

                              if not (is_fsobjmodel_tcref tcref) then 
                                  errorR(Error("Constructors may not be defined for this type",id.idRange));

                              let ctorThisVarRefCellOpt = mkAndPublishCtorThisRefCellVal cenv envinner thisIdOpt thisTy in
                                
                              (* baseVarOpt is the 'base' variable associated with the inherited portion of a class *)
                              (* It is declared once on the 'inherits' clause, but a fresh binding is made for *)
                              (* each member that may use it. *)
                              let baseVarOpt = 
                                  match super_of_typ cenv.g cenv.amap m objTy with 
                                  | Some(superTy) -> mkAndPublishBaseVal cenv envinner (Option.map id_of_val baseVarOpt) superTy 
                                  | None -> None in

                              let dty = new_inference_typ cenv () in

                              (* This is the type we pretend a struct has, because its implementation must ultimately appear to return a value of the given type *)
                              (* This is somewhat awkward later in codegen etc. *)
                              unifyE cenv envinner bindm ty (dty --> objTy);

                              ctorThisVarRefCellOpt,baseVarOpt
                              
                          | _ -> 
                              None,None in
                          
                        let vspr,uniquen = 
                            let optIntfSlotTy = Option.map fst optIntfSlotTy in 
                            let isExtensionBinding = (declKind = ExtensionBinding) in 
                            mkMemberDataAndUniqueId cenv.g tcref  isExtensionBinding attrs' optIntfSlotTy memFlags topValSynData id in

                        Some(vspr),uniquen,enclosingDeclaredTypars,envinner,ctorThisVarRefCellOpt,baseVarOpt
                    | _ -> 
                        (* non-member bindings. How easy. *)
                        None,id,[],envinner,None,None in 

                let topValSynDataOpt = 
                    match memberInfo with 
                    | Some(_,topValSynData,_) -> Some topValSynData
                    | _ -> None in 

                (* dprintf2 "applyTypesFromArgumentPatterns cenv envinner (before): declaredTypars = %s, ty = %s\n" (Layout.showL  (typarsL declaredTypars)) (Layout.showL  (typeL ty));   *)

                (* For some constructs (members) we know the precise arities syntactically *)
                (* This also applies any types given by the user *)
                (* dprintf1 "applyTypesFromArgumentPatterns cenv envinner (before): ty = %s\n" (Layout.showL  (typeL ty));  *)
                applyTypesFromArgumentPatterns cenv envinner optArgsOK ty bindm tpenv rhs;

                (* Do the type annotations give the full and complete generic type? *)
                let isComplete =  computeIsComplete enclosingDeclaredTypars declaredTypars ty in
                    
                (* dprintf2 "applyTypesFromArgumentPatterns cenv envinner (after): declaredTypars = %s, ty = %s\n" (Layout.showL  (typarsL declaredTypars)) (Layout.showL  (typeL ty));   *)

                (* dprintf3 "applyTypesFromArgumentPatterns (%s): ty = %s, isComplete = %b\n" id.idText (Layout.showL  (typeL ty)) isComplete;  *)
                (* Option.iter (apply_arity ty) topValSynData; *)
                (* "0" in topValSynData means a "unit" arg in compiledArity *)
                let partialValArityOpt = Option.map (translateTopValSynData m (tc_attributes cenv env)) topValSynDataOpt in

                (* NOTE: The type scheme here is not 'complete'!!!! The type is more or less *)
                (* just a type variable at this point. *)
                (* NOTE: toparity, type and typars get fixed-up after inference *)
                let prelimTyscheme = TypeScheme(enclosingDeclaredTypars@declaredTypars,[],ty) in
                let prelimValScheme = ValScheme(uniquen,prelimTyscheme,Option.map (inferGenericArityFromTyScheme prelimTyscheme) partialValArityOpt,vspr,false,inlineFlag,NormalVal,vis,false,false,false) in
                let vspec = mkAndPublishVal cenv envinner (altActualParent,false,declKind,ValInRecScope(isComplete),prelimValScheme,attrs',doc ,konst) in
                
                (* Nore return the results *)
                let bind = NormBinding (vis1,pseudo,mut,attrs,doc,synTyparDecls,memberInfo,mksyn_pat_var vis2 uniquen,rhs,bindm) in 
                let rbinfo = RBInfo(enclosingDeclaredTypars,inlineFlag,vspec,flex,partialValArityOpt,baseVarOpt,ctorThisVarRefCellOpt,vis,declKind) in
                let nameValueMap = Map.add uniquen.idText (vspec,prelimTyscheme) nameValueMap in 

                (* Done - add the declared name to the map and return the bundle for use by tc_letrec *)
                (rbinfo,bind),(tpenv,nameValueMap)
                
            | Pat_instance_member(thisn, memberName,vis2,m) ->
               let vis = combineVisibilityAttribs vis1 vis2 bindm in
               begin match tcrefContainerInfo,memberInfo with 
               | Some(tcref,optIntfSlotTy,baseVarOpt),
                 Some(memFlags,topValSynData,_) -> 
                   (* Normal instance members. *)
                 
                   checkMemberFlags cenv.g optIntfSlotTy newslotsOK overridesOK memFlags bindm;
                 
                   (* Syntactically push the "this" variable across to be a lambda on the right *)
                   let rhs = push_one_pat_rhs true (mksyn_this_pat_var thisn) rhs in 
                 
                   (* The type being augmented tells us the type of 'this' *)
                   let tcrefObjTy,enclosingDeclaredTypars,renaming,objTy,thisTy = freshenThisTy cenv bindm recTyparsRigid tcref in 

                   let envinner = add_declared_typars CheckForDuplicateTypars enclosingDeclaredTypars envinner in 

                   (* If private, the member's accessibility is related to 'tcref' *)
                   let envinner = mk_inner_env_for_tcref envinner tcref (declKind = ExtensionBinding) in 

                   let baseVarOpt = if is_fsobjmodel_tcref tcref then baseVarOpt else None in 

                   (* Apply the known type of 'this' *)
                   let rty = new_inference_typ cenv () in
                   unifyE cenv envinner bindm ty (thisTy --> rty);

                   checkForNonAbstractInterface declKind tcref memFlags memberName.idRange; 
                   
                   (* Determine if a uniquely-identified-override exists based on the information *)
                   (* at the member signature. If so, we know the type of this member, and the full slotsig *)
                   (* it implements. Apply the inferred slotsig. *)
                   let optInferredImplSlotTy, declaredTypars = applyAbstractSlotInference cenv denv envinner (rty,bindm,synTyparDecls,declaredTypars,memberName,tcrefObjTy,renaming,objTy,optIntfSlotTy,topValSynData,memFlags) in
                   
                   (* Update the TIFlex to reflect the declaredTypars inferred from the abstract slot *)
                   let flex = TIFlex(declaredTypars,infer) in

                   (* baseVarOpt is the 'base' variable associated with the inherited portion of a class *)
                   (* It is declared once on the 'inherits' clause, but a fresh binding is made for *)
                   (* each member that may use it. *)
                   let baseVarOpt = 
                       match super_of_typ cenv.g cenv.amap m objTy with 
                       | Some(superTy) -> mkAndPublishBaseVal cenv envinner (Option.map id_of_val baseVarOpt) superTy 
                       | None -> None in

                   let vspr,uniquen = 
                       let optIntfSlotTy = Option.map snd optIntfSlotTy in 
                       let isExtensionBinding = (declKind = ExtensionBinding) in 
                       mkMemberDataAndUniqueId cenv.g tcref isExtensionBinding attrs' optInferredImplSlotTy memFlags topValSynData memberName in

                   (* For members we know the precise arities syntactically *)
                   (* dprintf1 "applyTypesFromArgumentPatterns cenv envinner (2, before): ty = %s\n" (Layout.showL  (typeL ty));  *)
                   applyTypesFromArgumentPatterns cenv envinner optArgsOK ty bindm tpenv rhs;
                   (* dprintf1 "applyTypesFromArgumentPatterns cenv envinner (2, after): ty = %s\n" (Layout.showL  (typeL ty));  *)

                   (* Do the type annotations give the full and complete generic type? *)
                   (* If so, generic recursion can be used when using this type. *)
                   let isComplete =  computeIsComplete enclosingDeclaredTypars declaredTypars ty in

                   (* dprintf3 "applyTypesFromArgumentPatterns (%s): ty = %s, isComplete = %b\n" memberName.idText (Layout.showL  (typeL ty)) isComplete;  *) 

                  (* NOTE: The type scheme here is normally not 'complete'!!!! The type is more or less *)
                  (* just a type variable at this point. *)
                  (* NOTE: toparity, type and typars get fixed-up after inference *)
                   let prelimTyscheme = TypeScheme(enclosingDeclaredTypars@declaredTypars,[],ty) in
                   let topValSynDataOpt = 
                       match memberInfo with 
                       | Some(_,topValSynData,_) -> Some topValSynData
                       | _ -> if (must_have_arity declKind) then Some(SynArgInfo.emptyTopValData) else None in 

                   (* Option.iter (apply_arity ty) topValSynData; *)
                   let partialValArityOpt = Option.map (translateTopValSynData m (tc_attributes cenv env)) topValSynDataOpt in
                   let prelimValScheme = ValScheme(uniquen,prelimTyscheme,Option.map (inferGenericArityFromTyScheme prelimTyscheme) partialValArityOpt,Some(vspr),false,inlineFlag,NormalVal,vis,false,false,false) in
                 
                   (* Create the value *)
                   let vspec = mkAndPublishVal cenv envinner (altActualParent,false,declKind,ValInRecScope(isComplete),prelimValScheme,attrs',doc,konst) in
                 
                   (* Done - add the declared name to the map and return the bundle for use by tc_letrec *)
                   let ctorThisVarRefCellOpt = None in
                   let bind = NormBinding (vis1,pseudo,mut,attrs,doc,synTyparDecls,memberInfo,mksyn_pat_var vis2 uniquen,rhs,bindm) in 
                   let rbinfo = RBInfo(enclosingDeclaredTypars,inlineFlag,vspec,flex,partialValArityOpt,baseVarOpt,ctorThisVarRefCellOpt,vis,declKind) in 
                   let nameValueMap = Map.add uniquen.idText (vspec,prelimTyscheme) nameValueMap in 
                   (rbinfo,bind),(tpenv,nameValueMap)
               | _ -> error(Error("recursive bindings that include member specifications can only occur as a direct augmentation of a type",bindm)) 
               end

            | _ -> error(Error("only simple variable patterns can be bound in 'let rec' constructs",bindm)) in 
        (* OK, analyze the pseudo-pattern using the above function *)
        lookPat p 
        (* REVIEW: no more constraints allowed on declared typars *)
        (* REVIEW: this is related to changing recTyparsRigid to 'true' *)
        (* allDeclaredTypars |> List.iter setTyparRigid; *)

and mk_rec_values  overridesOK cenv env tpenv bindtys binds = 
    map_acc_list (mk_rec_value  overridesOK cenv env) (tpenv,Map.empty) (List.combine bindtys binds)


(*-------------------------------------------------------------------------
 * tc_letrec_tc_info_binding
 *------------------------------------------------------------------------*)

and tc_letrec_tc_info_binding cenv env scopem prelimRecValues tpenv (bindty,(rbinfo,(bind:normBinding))) = 
    (* REVIEW: push prelimRecValues into env before calling *)
    let tau = bindty in
    let (RBInfo(enclosingDeclaredTypars,inlineFlag,v,flex,partialValArityOpt,baseVarOpt,ctorThisVarRefCellOpt,_,declKind)) = rbinfo in
    let (TIFlex(declaredTypars,infer)) = flex in
    let allDeclaredTypars = enclosingDeclaredTypars @ declaredTypars in
    assert(inlineFlag_of_val v = inlineFlag);
    let m = (range_of_val v) in 
    (* dprintf1 "tc_letrec (before): tau = %s\n" (Layout.showL  (typeL tau));  *)

    let envrec = addLocalValMap scopem prelimRecValues env  in 
    let envrec = Option.fold_right (addLocalVal scopem) baseVarOpt envrec in 
    let envrec = Option.fold_right (addLocalVal scopem) ctorThisVarRefCellOpt envrec in 

    (* Members can access protected members of parents of the type *)
    (* Members can access private members in the type *)
    let envrec = mk_inner_env_for_meminfo envrec v in 

    let tbinding',tpenv = 
        tc_norm_binding declKind cenv envrec tpenv tau (Option.map mk_local_vref ctorThisVarRefCellOpt) (enclosingDeclaredTypars,flex) bind in 
        
        
    (* dprintf2 "tc_letrec (%s, after): tau = %s\n" (name_of_val v) (Layout.showL  (typeL tau));    *)
    (try unifyE cenv env m (allDeclaredTypars +-> tau) (type_of_val v) with e -> error (Recursion((denv_of_tenv env),(id_of_val v),tau,(type_of_val v),(range_of_val v))));

    (* dprintf2 "tc_letrec (%s, after unify): type_of_val v = %s\n" (name_of_val v) (Layout.showL  (typeL (type_of_val v)));   *)
    
    (rbinfo,tbinding'),tpenv

(*-------------------------------------------------------------------------
 * tc_letrec_computeGeneralizedTyparsForBinding
 *------------------------------------------------------------------------*)

(** Compute the type variables which may be generalized *)
and tc_letrec_computeGeneralizedTyparsForBinding cenv denv freeInEnv (rbinfo,tbinding) =
    let (RBInfo(enclosingDeclaredTypars,_,vspec,flex,_,_,_,_,declKind)) = rbinfo in
    let m = range_of_val vspec in
    let (TBindingInfo(inlineFlag,immut,_,_,_,_,_,expr,_,_,m,_,_)) = tbinding in
    let (TIFlex(declaredTypars,canInferTypars)) = flex in 
    let allDeclaredTypars = enclosingDeclaredTypars @ declaredTypars in

    let memFlagsOpt = vspec |> member_info_of_val  |> Option.map (fun memInfo -> memInfo.vspr_flags) in 
    let canInferTypars = computeCanInferTypars(declKind,canInferTypars,memFlagsOpt,declaredTypars,m) in

    let _,tau = try_dest_forall_typ (type_of_val vspec) in 
    let maxInferredTypars = (free_in_type_lr false tau) in
     
    computeGeneralizedTypars(cenv,denv,m,immut,freeInEnv,canInferTypars,canGeneralizeConstrainedTyparsForDecl(declKind),inlineFlag, Some(expr), allDeclaredTypars, maxInferredTypars)

(** Compute the type variables which may have member constraints that need to be canonicalized prior to generalization *)
and tc_letrec_computeSupportForBinding (rbinfo,tbinding) =
    let (RBInfo(enclosingDeclaredTypars,_,vspec,flex,_,_,_,_,declKind)) = rbinfo in
    let (TIFlex(declaredTypars,canInferTypars)) = flex in 
    let allDeclaredTypars = enclosingDeclaredTypars @ declaredTypars in
    let _,tau = try_dest_forall_typ (type_of_val vspec) in 
    let maxInferredTypars = (free_in_type_lr false tau) in
    allDeclaredTypars @ maxInferredTypars

(*-------------------------------------------------------------------------
 * tc_letrec_generalizeBinding
 *------------------------------------------------------------------------*)

and tc_letrec_generalizeBinding cenv denv generalizedTyparsForRecursiveBlock freeInEnv generalizedTypars (rbinfo,tbind) =
    (* Generalise generalizedTypars from tbinding.
     * Any tp in generalizedTyparsForRecursiveBlock \ generalizedTypars has free choice, see comment in generalizeVal *)
    let (RBInfo(enclosingDeclaredTypars,_,vspec,flex,partialValArityOpt,_,_,vis,declKind)) = rbinfo in
    let (TBindingInfo(inlineFlag,immut,attrs,_,_,_,_,expr,argAttribs,_,m,compgen,_)) = tbind in
     
    let tps,tau = try_dest_forall_typ (type_of_val vspec) in 

    let pvalscheme1 = PrelimValScheme1((id_of_val vspec),flex,tau,None,false,inlineFlag,NormalVal,argAttribs,vis,compgen) in 
    let pvalscheme2 = generalizeVal cenv denv enclosingDeclaredTypars generalizedTyparsForRecursiveBlock generalizedTypars pvalscheme1 in
  (*dprintf4 "tc_letrec (%s, before adjust): tau = %s, #generalizedTyparsForThisBinding = %d\n" (name_of_val vspec) (Layout.showL  (typeL tau)) (length generalizedTyparsForThisBinding); *)
    let valscheme = inferPartialArityIfNotGiven declKind partialValArityOpt expr pvalscheme2 in 
    adjustRecType cenv vspec valscheme;
  (*dprintf2 "tc_letrec (%s, after adjust): ty = %s\n" (name_of_val vspec) (Layout.showL  (typeL (type_of_val vspec)));  *)
    valscheme,(vspec,expr)  (* NOTE: (vspec,'e) could be a TBind(vspec,'e) : Tast.bind *)


and tc_letrec_compute_ctorThisVarRefCellBind cenv ctorThisVarRefCellOpt =
    ctorThisVarRefCellOpt |> Option.map (fun ctorv -> 
        let m = range_of_val ctorv in 
        let ty = dest_refcell_ty cenv.g (type_of_val ctorv) in 
        TBind(ctorv,mk_refcell cenv.g m ty (mk_null m ty)))

and tc_letrec_bind_ctorv cenv (x,(vspec,expr)) rbinfo = 
    let (RBInfo(_,_,_,_,_,baseVarOpt,ctorThisVarRefCellOpt,_,_)) = rbinfo in
    let expr = 
        match tc_letrec_compute_ctorThisVarRefCellBind cenv ctorThisVarRefCellOpt with 
        | None -> expr
        | Some bind -> 
            let m = range_of_expr expr in 
            let tps,vsl,body,rty = dest_top_lambda (expr,type_of_val vspec) in 
            mk_multi_lambdas m tps vsl (mk_let_bind m bind body, rty) in 
    let expr = 
        match baseVarOpt with 
        | None -> expr
        | _ -> 
            let m = range_of_expr expr in 
            let tps,vsl,body,rty = dest_top_lambda (expr,type_of_val vspec) in 
            mk_basev_multi_lambdas m tps baseVarOpt vsl (body, rty) in 
              
    x,(vspec,expr)

and tc_letrec_fixup_vxbind cenv denv (valscheme,vxbind) =
    let vspec,expr = vxbind in      
    (* Check coherence of generalization of variables for vspr members in generic classes *)
    begin match member_info_of_val vspec with 
    | Some(vspr) -> 
       begin match partition_val_typars vspec with
       | Some(parentTypars,memberParentTypars,_,_,tinst) -> 
          ignore(subtyp_forall_typars cenv.g denv (range_of_val vspec) tyeq_env_empty memberParentTypars parentTypars)
       | None -> 
          errorR(Error("this member is not sufficiently generic",range_of_val vspec))
        end
    | None -> ()
    end;
    (* Fixup recursive calls and insert code into ctor bodies... *)
    let (ValScheme(_,typeScheme,_,_,_,_,_,_,_,_,_)) = valscheme in
    let m = (range_of_val vspec) in 
    let fixupPoints = getAllUsesOfRecValue cenv (mk_lref vspec) in
    adjustUsesOfRecValue cenv (mk_local_vref vspec) valscheme;
    (* dprintf3 "tc_letrec (%s, after gen): #fixupPoints = %d, ty = %s\n" (name_of_val vspec) (length fixupPoints) (Layout.showL  (typeL (type_of_val vspec))); *)
    (* Insert the thunk-initializer for 'new() as vspec' *)
    let expr = mk_poly_bind_rhs m typeScheme expr in      
    fixupPoints,(vspec,expr)
    
(*-------------------------------------------------------------------------
!* tc_letrec
 *------------------------------------------------------------------------*)

and unionGeneralizedTypars typarSets = fold_right (gen_union_favour_right typar_ref_eq) typarSets [] 
    

and tc_letrec  overridesOK cenv env tpenv bindtys (binds,bindsm,scopem) =
    (* create prelimRecValues for the recursive items (includes type info from LHS of bindings) *)
    let binds = binds |> map (fun (RecBindingDefn(a,b,c,bind)) -> NormRecBindingDefn(a,b,c,normBinding ValOrMemberBinding cenv env bind)) in 
    let rbinfosAndBinds,(tpenv,nameToPrelimValueMap) = mk_rec_values  overridesOK cenv env tpenv bindtys binds in
    let prelimRecValues  = Namemap.map fst nameToPrelimValueMap in
    (* REVIEW: prelimRecValues - added to env for checking bindings, and for resultant env following bindings *)

    (* typecheck bindings *)  
    let rbinfo_tbinds,tpenv = map_acc_list (tc_letrec_tc_info_binding cenv env scopem prelimRecValues) tpenv (combine bindtys rbinfosAndBinds) in

    (* decide tps to generalize per binding *)  
    let denv = (denv_of_tenv env) in 

    
    begin         
        let supportForBindings = rbinfo_tbinds |> mapConcat tc_letrec_computeSupportForBinding  in
        canonicalizePartialInferenceProblem (cenv,denv,bindsm) supportForBindings; 
    end;
     

    let freeInEnv = computeUngeneralizableTypars env in 
    let generalizedTyparsL = List.map (tc_letrec_computeGeneralizedTyparsForBinding cenv denv freeInEnv) rbinfo_tbinds in  
    let generalizedTyparsForRecursiveBlock = unionGeneralizedTypars generalizedTyparsL in

    if verboseCC then List.iter (fun tp -> printf "tc_letrec: generalizedTyparsForRecursiveBlock = %s\n" (typarL tp |> Layout.showL)) generalizedTyparsForRecursiveBlock;
    
    (* generalize - gives valscheme and projects (vspec,x) *)
    let vxbinds = List.map2 (tc_letrec_generalizeBinding cenv (denv_of_tenv env) generalizedTyparsForRecursiveBlock freeInEnv) generalizedTyparsL rbinfo_tbinds in      
    let tpenv = hideUnscopedTypars (concat generalizedTyparsL) tpenv in

    let vxbinds = List.map2 (tc_letrec_bind_ctorv cenv) vxbinds (map fst rbinfo_tbinds) in
    (* Now that we know what we've generalized we can adjust the recursive references *)
    let vxbinds = List.map (tc_letrec_fixup_vxbind cenv (denv_of_tenv env)) vxbinds in
    
    (* Now eliminate any initialization graphs *)
    let binds = 
        let bindsWithoutLaziness = vxbinds in 
        let mustHaveArity = 
            match rbinfosAndBinds with 
            | [] -> false
            | ((rbinfo,_) :: _) -> 
                let (RBInfo(_,_,vspec,flex,partialValArityOpt,_,_,vis,declKind)) = rbinfo in
                must_have_arity declKind in
            
        eliminateInitializationGraphs cenv.g mustHaveArity (denv_of_tenv env) bindsWithoutLaziness bindsm in
    
    (* Post letrec env *)
    let envbody = addLocalValMap scopem prelimRecValues env  in 
    binds,envbody,tpenv

(*-------------------------------------------------------------------------
!* Bind specifications of values
 *------------------------------------------------------------------------- *)

let tc_and_publish_val_spec cenv env containerInfo declKind memFlagsOpt tpenv ((ValSpfn(attrs,id,ValTyparDecls(_,canInferTypars,_), _, _, pseudo, mutableFlag,doc, vis,literalExprOpt,m)) as vspfn) = 
  let attrs' = tc_attributes cenv env attrTgtBinding attrs in 
  let valinfos,tpenv = tc_val_spec cenv env declKind containerInfo memFlagsOpt None tpenv vspfn attrs' in 
  let denv = denv_of_tenv env in 
  
  map_acc_list 
      (fun tpenv (ValSpecResult(altActualParent,vsprOpt,(id:ident),enclosingDeclaredTypars,declaredTypars,ty',arity,declKind)) -> 
          let inlineFlag = computeInlineFlag (vsprOpt |> Option.map (fun vspr -> vspr.vspr_flags)) pseudo mutableFlag in 
          
          (* dprintf3 "%a: ty' = %s\n" output_range m (NicePrint.pretty_string_of_typ (empty_denv cenv.g) ty'); *)
          let freeInType = (free_in_type_lr false ty') in
          let allDeclaredTypars = enclosingDeclaredTypars @ declaredTypars in
          let flex = TIFlex(declaredTypars,canInferTypars) in 
          
          let canInferTypars = computeCanInferTypars(declKind,canInferTypars,memFlagsOpt,declaredTypars,m) in
          
          let generalizedTypars = computeGeneralizedTypars(cenv,denv,id.idRange,canInferTypars,empty_free_loctypars,canInferTypars,CanGeneralizeConstrainedTypars,inlineFlag,None,allDeclaredTypars,freeInType) in
          
          (* dprintf6 "nm = %s, #freeInType = %d, #enclosingDeclaredTypars = %d, #declaredTypars = %d, #generalizedTypars = %d, ty' = %s\n" id.idText  (length freeInType) (length enclosingDeclaredTypars) (length declaredTypars) (length generalizedTypars)  (NicePrint.pretty_string_of_typ (empty_denv cenv.g) ty');  *)
          (* dprintf1 "#uniqfreeInType = %d\n" (length (gen_setify typar_ref_eq freeInType)); *)
          let valscheme1 = PrelimValScheme1(id,flex,ty',vsprOpt,mutableFlag,inlineFlag,NormalVal,noArgOrRetAttribs,vis,false) in
          

          let valscheme2 = generalizeVal cenv denv enclosingDeclaredTypars  [] generalizedTypars valscheme1 in

          let tpenv = hideUnscopedTypars generalizedTypars tpenv in

          let valscheme = useKnownArity valscheme2 (Some arity) in 

          let konst = 
              match literalExprOpt with 
              | None -> 
                  let hasLiteralAttr = fsthing_has_attrib cenv.g cenv.g.attrib_LiteralAttribute attrs' in
                  if hasLiteralAttr then 
                      errorR(Error("A declaration may only be the [<Literal>] attribute if a constant value is also given, e.g. 'val x : int = 1'",m));
                  None

              
              | Some(e) -> 
                  let hasLiteralAttr,konst = tc_literal cenv ty' env tpenv (attrs',e) in
                  if not hasLiteralAttr then 
                      errorR(Error("A declaration may only be given a value in a signature if the declaration has the [<Literal>] attribute",range_of_synexpr e));
                  konst in

          let vspec = mkAndPublishVal cenv env (altActualParent,true,declKind,ValNotInRecScope,valscheme,attrs',doc,konst) in
          (*      if name_of_val idv = "ListenerSet.Fire" then dprintf1 "typ = %s\n" (NicePrint.pretty_string_of_typ (empty_denv cenv.g) (type_of_val idv)); *)
          assert(inlineFlag_of_val vspec = inlineFlag);
          vspec,tpenv)
      tpenv
      valinfos


(*-------------------------------------------------------------------------
!* Bind elements of data definitions for exceptions and types (fields, etc.)
 *------------------------------------------------------------------------- *)

let mk_rfield_spec cenv env parent declKind  (stat,konst,ty',attrs1,attrs2,id,mut,xmldoc,vis,m) =
    let vis,_ = computeAccessAndCompPath env declKind m vis parent in 
    new_rfield stat konst id ty' mut attrs1 attrs2 xmldoc vis false

let tc_field_decl cenv env parent  declKind isIncrClass tpenv (stat,attrs,id,ty,mut,xmldoc,vis,m) =
    let attrs' = tc_attributes_with_possible_targets cenv env attrTgtFieldDecl attrs in
    let attrs1,attrs2 = List.partition (fun (pt,_) -> (pt &&& attrTgtProperty) <> 0l) attrs' in 
    let attrs1 = (map snd attrs1)  in
    let attrs2 = (map snd attrs2) in
    let ty',_ = tc_type_and_recover cenv NoNewTypars CheckCxs env tpenv ty in 
    let zeroInit = fsthing_has_attrib cenv.g cenv.g.attrib_DefaultValueAttribute attrs2 in
    
    (* Check if it's marked unsafe *)
    let zeroInitUnsafe = fsthing_bool_attrib cenv.g cenv.g.attrib_DefaultValueAttribute attrs2 in
    if zeroInit && zeroInitUnsafe <> Some(false) then 
       if not (typSatisfiesNullConstraint cenv.g ty') && not (is_struct_typ ty') then 
           errorR(Error("The type of this field must either have 'null' as a proper value or be a value type. You can use 'DefaultValue(false)' to disable this check",m));

    if isIncrClass && (not zeroInit || not mut) then errorR(Error("Unintialized 'val' fields in implicit construction types must be mutable and marked with the '[<DefaultValue>]' attribute. Consider using a 'let' binding instead of a 'val' field",m));
    if stat && (not zeroInit || not mut) then errorR(Error("Static 'val' fields in types must be mutable and marked with the '[<DefaultValue>]' attribute. They are initialized to the 'null' or 'zero' value for their type. Consider also using a 'let' binding in a module",m));
    let konst = if zeroInit then Some(TConst_default) else None in    
    mk_rfield_spec cenv env parent  declKind  (stat,konst,ty',attrs1,attrs2,id,mut,xmldoc,vis,m)

let tc_anon_field_decl cenv env parent declKind tpenv nm (Field(attribs,stat,id,ty,mut,xmldoc,vis,m)) =
    let id = (match id with None -> mksyn_id m nm | Some id -> id) in
    tc_field_decl cenv env parent declKind false tpenv (stat,attribs,id,ty,mut,xmldoc,vis,m) 

let tc_named_field_decl cenv env parent declKind isIncrClass tpenv (Field(attribs,stat,id,ty,mut,xmldoc,vis,m)) =
    match id with 
    | None -> error (Error("this field requires a name",m))
    | Some(id) -> tc_field_decl cenv env parent declKind isIncrClass  tpenv (stat,attribs,id,ty,mut,xmldoc,vis,m) 

let check_duplicates (idf : _ -> ident) k elems = 
    elems |> list_iteri (fun i uc1 -> 
        elems |> list_iteri (fun j uc2 -> 
            if j > i &&  (idf uc1).idText = (idf uc2).idText then 
                errorR (Duplicate(k,idf uc1))));
    elems

let tc_named_field_decls cenv env parent declKind isIncrClass tpenv fields =
    fields |> map (tc_named_field_decl cenv env parent declKind isIncrClass tpenv) |> check_duplicates id_of_rfield "field" 


(*-------------------------------------------------------------------------
!* Bind other elements of type definitions (constructors etc.)
 *------------------------------------------------------------------------- *)

exception LowerCaseConstructor of range

let is_lower_string s = 
    String.length s >= 1 && (Char.uppercase s.[0] <> s.[0]) 
    
let tc_constr_decl cenv env parent declKind thisTy tpenv (UnionConstr (attrs,id,args,xmldoc,vis,m)) =
    let attrs' = tc_attributes cenv env attrTgtConstrDecl attrs in 
    let vis,cpath = computeAccessAndCompPath env declKind m vis parent in 
    let ilname =  
        if id.idText = opname_Cons then "Cons" 
        else if id.idText = opname_Nil then "Nil"
        else id.idText in 
    
    if is_lower_string ilname && ilname <> opname_Cons && ilname <> opname_Nil then errorR(LowerCaseConstructor(m));
    let mk_name i = (ilname^string_of_int (i+1)) in 
    let args',rty = 
        match args with
        | ConstrFields flds -> 
            list_mapi (fun i fdef -> tc_anon_field_decl cenv env parent declKind tpenv (mk_name i) fdef) flds,
            thisTy
        | ConstrFullType (ty,arity) -> 
            let ty',tpenv = tc_type_and_recover cenv NoNewTypars CheckCxs env tpenv ty in
            let argtysl,rty = dest_top_tau_type (arity |> translateTopValSynData m (tc_attributes cenv env) |> translatePartialArity [] |> TopValData.getArgInfos) ty' in 
            if List.length argtysl > 1 then errorR(Error("Explicit type declarations for constructors must be of the form 'ty1 * ... * tyN -> resTy'. Parentheses may be required around 'resTy'",m));   
            argtysl |> List.concat |> list_mapi (fun i (argty,TopArgData(_,nmOpt)) ->
                let id = (match nmOpt with Some id -> id | None -> mksyn_id m (mk_name i)) in
                mk_rfield_spec cenv env parent declKind  (false,None,argty,[],[],id,false,emptyXMLDoc,None,m)),
            rty in 
    new_uconstr id ilname args' rty attrs' xmldoc vis

let tc_constr_decls cenv env parent declKind thisTy tpenv constrs =
    let constrs' = map (tc_constr_decl cenv env parent declKind thisTy tpenv) constrs in                                   
    check_duplicates id_of_uconstr "data constructor" constrs'

let tc_enum_decl cenv env parent declKind thisTy field_ty (EnumConstr (attrs,id,v,xmldoc,m)) =
    let attrs' = tc_attributes cenv env attrTgtConstrDecl attrs in 
    let v = tc_const cenv field_ty m env v in 
    let vis,cpath = computeAccessAndCompPath env declKind m None parent in 
    new_rfield true (Some v) id thisTy false [] [] emptyXMLDoc vis false
  
let tc_enum_decls cenv env parent declKind thisTy constrs =
    let field_ty = new_inference_typ cenv () in 
    let constrs' = constrs |> map (tc_enum_decl cenv env parent declKind thisTy field_ty)  |> check_duplicates id_of_rfield "enum element" in 
    field_ty,constrs'


(*-------------------------------------------------------------------------
!* Bind elements of classes
 *------------------------------------------------------------------------- *)

let publish_interface cenv denv tcref m compgen ty' = 
    if not (is_interface_typ ty') then errorR(Error("The type '"^NicePrint.pretty_string_of_typ denv ty'^"' is not an interface type",m));
    let tcaug = tcaug_of_tcref tcref in
    if tcaug_has_interface cenv.g tcaug ty'  then 
        errorR(Error("duplicate specification of an interface",m));
    tcaug.tcaug_implements <- (ty',compgen,m) :: tcaug.tcaug_implements

let tc_and_publish_member_spec newslotsOK overridesOK cenv env containerInfo declKind tcref tpenv memb = 
    match memb with 
    | ClassMemberSpfn_field(fdecl,m) -> error(Error("A field/val declaration is not permitted here",m))
    | ClassMemberSpfn_inherit(typ,m) -> error(Error("A inheritance declaration is not permitted here",m))
    | ClassMemberSpfn_tycon(_,m) -> error(Error("Types may not contain nested type definitions",m))
    | ClassMemberSpfn_binding(valSpfn,memFlags,m) -> 
        tc_and_publish_val_spec cenv env containerInfo declKind (Some(memFlags)) tpenv valSpfn
    | ClassMemberSpfn_interface(ty,m) -> [],tpenv
(* For specifications these now get done in tc_tycon_cores:
      let ty',tpenv = tc_type cenv false env tpenv ty in 
      publish_interface (denv_of_tenv env) tcref m ty';
      [],tpenv
*)

  
let tc_class_spec newslotsOK overridesOK cenv env containerInfo declKind tcref tpenv (augSpfn: classSpfn)  =
    let members,tpenv = map_acc_list (tc_and_publish_member_spec newslotsOK overridesOK cenv env containerInfo declKind tcref) tpenv augSpfn in
    List.concat members,tpenv


(*-------------------------------------------------------------------------
!* Bind 'open' declarations
 *------------------------------------------------------------------------- *)

let tc_namespace_lid_and_permit_auto_resolve resolver env lid =
    match tc_namespace_lid OpenQualified env lid  with 
    | Success res -> Success res
    | Raze err -> 
       begin match lid with 
       | mn :: rest -> 
         begin match resolver mn with
         | Some modref -> one_result (tc_namespace_in_namespace 1 modref (mtyp_of_modref modref) rest)
         | None -> raze err
         end
       | _ -> raze err
       end

let tc_open_namespace_unchecked resolver scopem env lid = 
    let mvvs = forceRaise (tc_namespace_lid_and_permit_auto_resolve  resolver env.eNameResEnv lid) in
    let env = fold_right (fun (depth,modref,mty) acc -> open_modul scopem acc modref) mvvs env in
    env
    

let tc_open_namespace resolver m scopem env (lid : ident list) = 
    let remap dests  = 
        lid |> mapConcat (fun id -> if id.idText = "MLLib" then map (fun dest -> ident(dest,id.idRange)) dests else [id]) in
        
    match path_of_lid lid with
    | "Microsoft" ::  "FSharp" ::  "MLLib" :: [] 
    |                              "MLLib" :: [] -> 
        warning(Error("The modules previously under Microsoft.FSharp.MLLib have been moved to Microsoft.FSharp.Core, Microsoft.FSharp.Control, Microsoft.FSharp.Collections, Microsoft.FSharp.Text and Microsoft.FSharp.Compatibility.OCaml. Please adjust this path accordingly. Ignoring this declaration",m));
        env
    | "Microsoft" ::  "FSharp" ::  "MLLib" :: (("LazyList" | "List" | "Array" | "Array2" | "Array3" | "Map" | "Set" | "ReadonlyArray") as moduleName) :: [] 
    |                              "MLLib" :: (("LazyList" | "List" | "Array" | "Array2" | "Array3" | "Map" | "Set" | "ReadonlyArray") as moduleName) :: [] ->
        warning(Error("This module has been renamed to Microsoft.FSharp.Collections."^moduleName^". Opening that module instead. Please adjust this reference",m));
        tc_open_namespace_unchecked resolver scopem env (remap ["Collections"])
    | "Microsoft" ::  "FSharp" ::  "MLLib" :: (("Int32" | "UInt32" | "Int16" | "UInt16" | "Option" | "Byte" | "SByte" | "UInt64"  | "Int64"  | "Float"  | "Float32" | "UInt8" | "Int8" | "Char") as moduleName) :: []
    |                              "MLLib" :: (("Int32" | "UInt32" | "Int16" | "UInt16" | "Option" | "Byte" | "SByte" | "UInt64"  | "Int64"  | "Float"  | "Float32" | "UInt8" | "Int8" | "Char") as moduleName) :: [] ->
        warning(Error("This module has been renamed to Microsoft.FSharp.Core."^moduleName^". Opening that module instead. Please adjust this reference",m));
        tc_open_namespace_unchecked resolver scopem env (remap ["Core"])
    | "Microsoft" ::  "FSharp" ::  "MLLib" :: (("IEvent" | "Lazy") as moduleName) :: []
    |                              "MLLib" :: (("IEvent" | "Lazy") as moduleName) :: [] ->
        warning(Error("This module has been renamed to Microsoft.FSharp.Control."^moduleName^". Opening that module instead. Please adjust this reference",m));
        tc_open_namespace_unchecked resolver scopem env (remap ["Control"])
    | "Microsoft" ::  "FSharp" ::  "MLLib" :: (("Printf") as moduleName) :: []
    |                              "MLLib" :: (("Printf") as moduleName) :: [] ->
        warning(Error("This module has been renamed to Microsoft.FSharp.Text"^moduleName^". Opening that module instead. Please adjust this reference",m));
        tc_open_namespace_unchecked resolver scopem env (remap ["Text"])
    | "Microsoft" ::  "FSharp" ::  "MLLib" :: (("Big_int" | "Buffer" | "Bytearray" | "Filename" | "Hashtbl" | "Lexing" | "Num" | "Obj" | "Parsing" | "Pervasives" | "Printexc" | "Sys") as moduleName) :: [] 
    |                              "MLLib" :: (("Big_int" | "Buffer" | "Bytearray" | "Filename" | "Hashtbl" | "Lexing" | "Num" | "Obj" | "Parsing" | "Pervasives" | "Printexc" | "Sys") as moduleName) :: [] ->
        warning(Error("This module has been renamed to Microsoft.FSharp.Compatibility.OCaml."^moduleName^". Opening that module instead. Please adjust this reference",m));
        tc_open_namespace_unchecked resolver scopem env (remap ["Compatibility";"OCaml"])
    | _ -> 
        tc_open_namespace_unchecked resolver scopem env lid



(*-------------------------------------------------------------------------
!* Partial class defns - make incrClassInstanceCtorVal and this_v
 *------------------------------------------------------------------------- *)

type incrClassCtorLhs = (* Typechecked info for implicit constructor and it's arguments *)
    {incrClassTcref                            : tycon_ref;
     incrClassInstanceCtorDeclaredTypars       : typars;     
     incrClassRevTypeInst                      : typar_inst; (* rename: ctor_to_formal_tinst *)
     incrClassStaticCtorVal                    : val_spec;
     incrClassStaticCtorValScheme              : valScheme;
     incrClassStaticCtorArgs                   : val_spec list;
     incrClassInstanceCtorVal                  : val_spec;
     incrClassInstanceCtorValScheme            : valScheme;
     incrClassInstanceCtorArgs                 : val_spec list;
     incrClassInstanceCtorThisVarRefCellOpt    : val_spec option;
     incrClassInstanceCtorBaseValOpt           : val_spec option;
     incrClassInstanceCtorThisVar              : val_spec;
    }

let tc_simple_pats_of_unknown_type cenv optArgsOK env tpenv spats =
    let argty = new_inference_typ cenv () in
    tc_simple_pats cenv optArgsOK argty env (tpenv,Namemap.empty) spats

let tc_implict_ctor_lhs cenv env tpenv tcref spats thisIdOpt baseVarOpt m =

    (* make fresh version of the class type for type checking the members and lets *)    
    let tcrefObjTy,ctorDeclaredTypars,renaming,objTy,thisTy = freshenThisTy cenv m TyparRigid tcref in
    (* Note: tcrefObjTy contains the original "formal" typars, thisTy is the "fresh" one... f<>fresh. *)
    let incrClassRevTypeInst = List.combine ctorDeclaredTypars (typars_of_tcref tcref |> List.map mk_typar_ty) in

    let baseVarOpt = 
        match super_of_typ cenv.g cenv.amap m objTy with 
        | Some(superTy) -> mkAndPublishBaseVal cenv env (Option.map id_of_val baseVarOpt) superTy
        | None -> None in

    (* Use class type directly for type checking the members and lets *)
    (* The (instance) members get fresh tyvar if their "this" requires it.
     * Those typar are not "rigid" so they can be equated to each other through their mutual recursion.
     * The types typar are made rigid in the "cores" pass.
     *)
    (* These typars should not be instantiated.
     * Post TC they should still be unique typars.
     * Setting rigid ensures they do not get substituted for another typar either.
     *)
    iter (setTyparRigid cenv.g (denv_of_tenv env) m) ctorDeclaredTypars; 

    if verboseCC then dprintf1 "ctorDeclaredTypars: %s\n" (showL (typarsL ctorDeclaredTypars));    

    (* Add class typars to env *)
    let env = add_declared_typars CheckForDuplicateTypars ctorDeclaredTypars env in

    (* Type check arguments by processing them as 'simple' patterns *)
    (* TODO: if we allow richer patterns here this is where we'd process those patterns *)
    let ctorArgNames,(tpenv,names) = tc_simple_pats_of_unknown_type cenv true env tpenv (SPats (spats,m)) in
    
    (* Create the values with the given names *)
    let _,vspecs = mkSimpleVals cenv env m names in
    
    (* Put them in order *)
    let ctorArgs = List.map (fun v -> Namemap.find v vspecs) ctorArgNames in          
    let ctorThisVarRefCellOpt = mkAndPublishCtorThisRefCellVal cenv env thisIdOpt thisTy in
    
    (* NOTE: the type scheme here is not complete!!! The ctorTy is more or less *)
    (* just a type variable. The type and typars get fixed-up after inference *)
    let ctorValScheme,ctorVal = 
        let argty = mk_tupled_ty cenv.g (List.map type_of_val ctorArgs) in
        let ctorTy = mk_fun_ty argty objTy in  (* initial type has known information *)
        let memFlagsOverloadQualifier = None in  (* REVIEW: no OverloadID can currently be specified for the implicit constructor *)
        let attribs = [] in                      (* REVIEW: no attributes can currently be specified for the implicit constructor *)
        let memFlags      = ctorMemFlags memFlagsOverloadQualifier in
                              
        let synArgInfos   = List.map (SynArgInfo.argdata_of_spat []) spats in
        let topValSynData = TopValSynData([synArgInfos],SynArgInfo.unnamedRetVal) in
        let id            = ident ("new",m) in

        checkForNonAbstractInterface ModuleOrMemberBinding tcref memFlags id.idRange;
        let vspr,uniquen  = mkMemberDataAndUniqueId cenv.g tcref false attribs None memFlags topValSynData id in
        let partialValArityOpt = Option.map (translateTopValSynData m (tc_attributes cenv env)) (Some topValSynData) in
        let prelimTyschemeG = TypeScheme(ctorDeclaredTypars,[],ctorTy) in
        let isComplete = computeIsComplete ctorDeclaredTypars [] ctorTy in
        let ctorValScheme = ValScheme(uniquen,prelimTyschemeG,Option.map (inferGenericArityFromTyScheme prelimTyschemeG) partialValArityOpt,Some(vspr),false,NeverInline,NormalVal,None,false,true,false) in          
        let ctorVal = mkAndPublishVal cenv env (ParentNone,false,ModuleOrMemberBinding,ValInRecScope(isComplete),ctorValScheme,[(* REVIEW: no attributes*)],emptyXMLDoc,None)  in
        ctorValScheme,ctorVal in

    let cctorArgs = [ fst(mk_compgen_local m "dummy" cenv.g.unit_ty) ] in 
 
    let cctorVal,cctorValScheme = 
        let cctorTy = mk_fun_ty cenv.g.unit_ty cenv.g.unit_ty in  
        let topValSynData = TopValSynData([[]],SynArgInfo.unnamedRetVal) in
        let id            = ident ("cctor",m) in
        checkForNonAbstractInterface ModuleOrMemberBinding tcref cctorMemFlags id.idRange;
        let vspr,uniquen  = mkMemberDataAndUniqueId cenv.g tcref false [(*no attributes*)] None cctorMemFlags topValSynData id in
        let partialValArityOpt = Option.map (translateTopValSynData m (tc_attributes cenv env)) (Some topValSynData) in
        let prelimTyschemeG = TypeScheme(ctorDeclaredTypars,[],cctorTy) in
        let cctorValScheme = ValScheme(uniquen,prelimTyschemeG,Option.map (inferGenericArityFromTyScheme prelimTyschemeG) partialValArityOpt,Some(vspr),false,NeverInline,NormalVal,None,false,true,false) in
         
        let cctorVal = mkAndPublishVal cenv env (ParentNone,false,ModuleOrMemberBinding,ValNotInRecScope,cctorValScheme,[(* no attributes*)],emptyXMLDoc,None)  in
        cctorVal,cctorValScheme in

    let thisVar = 
        (* --- Create this for use inside constructor *)
        let thisId  = ident ("this",m) in
        let thisValScheme  = ValScheme(thisId,nonGenericTypeScheme(thisTy),None,None,false,NeverInline,NormalVal,None,false,false,false) in
        let thisVar    = mkAndPublishVal cenv env (ParentNone,false,ClassLetBinding,ValNotInRecScope,thisValScheme,[],emptyXMLDoc,None) in
        if verboseCC then dprintf1 "mk_thisVar: v = %s\n" (showL (vspecTyL thisVar));
        thisVar in

    {incrClassTcref                         = tcref;
     incrClassInstanceCtorDeclaredTypars    = ctorDeclaredTypars;
     incrClassRevTypeInst                   = incrClassRevTypeInst;
     incrClassStaticCtorArgs              = cctorArgs;
     incrClassStaticCtorVal                 = cctorVal;
     incrClassStaticCtorValScheme         = cctorValScheme;
     incrClassInstanceCtorArgs              = ctorArgs;
     incrClassInstanceCtorVal               = ctorVal;
     incrClassInstanceCtorValScheme         = ctorValScheme;
     incrClassInstanceCtorBaseValOpt                    = baseVarOpt;
     incrClassInstanceCtorThisVarRefCellOpt = ctorThisVarRefCellOpt;
     incrClassInstanceCtorThisVar           = thisVar;
    }


(*-------------------------------------------------------------------------
!* Partial class defns - local val mapping to fields
 *------------------------------------------------------------------------- *)
  
(* Create the field for a "let" binding in a type definition.
 * The "v" is the local typed w.r.t. tyvars of the implicit ctor.
 * The formal_tinst does the formal-typars/implicit-ctor-typars subst.
 * Field specifications added to a tcref must be in terms of the tcrefs formal typars.
 *)
let mkIncrClassField cpath (formal_tinst:typar_inst) (v,(isStatic,rfref)) =
    let name = name_of_rfref rfref in
    let id  = ident (name,range_of_val v) in
    let ty  = type_of_val v |> inst_type formal_tinst in
    let mut = (mutability_of_val v = Mutable) in

    let taccess = TAccess [cpath] in
    new_rfield isStatic None id ty mut [(*no property attributes*)] [(*no field attributes *)] emptyXMLDoc taccess (*secret:*)true

(* Mutate a type definition by adding fields *)
(* Used as part of processing "let" bindings in a type definition. *)
let publishIncrClassFields rfspecs tcref =   
    let tycon_rfields = mk_rfields_table (rfspecs @ all_rfields_of_tcref tcref) in
    let obspec = tycon_objmodel_data_of_tcref tcref in
    let obspec = {obspec with fsobjmodel_rfields = tycon_rfields} in
    (data_of_tycon (deref_tycon tcref)).tycon_repr <- Some (TFsObjModelRepr obspec)  (* mutation *)


(*-------------------------------------------------------------------------
!* Partial class defns - local val representation and occurance fixups
 *------------------------------------------------------------------------- *)

type incrClassReprInfo = 
    { (* vals mapped to these field references *)
      incrClassSecretFields  : (val_spec,((*isStatic:*)bool * recdfield_ref)) Zmap.t; 
      (* vals represented as fields from this point on *)
      incrClassValsInFields  : val_spec Zset.t; }

let lookupIncrClassRepr g localRep v = 
    (* mutable unit fields are not stored, just run rhs for effects *)
    if is_unit_typ g (type_of_val v) then None else 
    match Zmap.tryfind v localRep.incrClassSecretFields with 
    | None -> error(InternalError("lookupIncrClassRepr: failed to find representation for value",range_of_val v))
    | res -> res

let emptyIncrClassRepr = 
    { incrClassSecretFields = Zmap.empty val_spec_order; 
      incrClassValsInFields = Zset.empty val_spec_order}

let extendIncrClassRepr  isStatic incrClassCtorLhs g lrep  bind = 
    let v = var_of_bind bind in 
    let fieldName = nng.nngApply ("_" ^ name_of_val v) (range_of_val v) in
    let rfref = RFRef(incrClassCtorLhs.incrClassTcref,fieldName) in
    {lrep with incrClassSecretFields = Zmap.add (var_of_bind bind) (isStatic,rfref) lrep.incrClassSecretFields}  

let reprValIsNowStoredInField  g lrep bind = 
    let v = var_of_bind bind in 
    {lrep with incrClassValsInFields = Zset.add v lrep.incrClassValsInFields}

(* fixup: constructs *)
let mk_let_field_lookup g localRep thisVarOpt tinst localv m =
    match lookupIncrClassRepr g localRep localv, thisVarOpt with 
    | Some (false,rfref),Some(thisv) -> 
        let thise = expr_for_val m thisv in
        mk_recd_field_get_via_expra(thise,rfref,tinst,m)
    | Some (true,rfref),_ -> 
        mk_static_rfield_get(rfref,tinst,m)
    | _ -> 
        mk_unit g m (* unit fields are not stored *)
  
let mk_let_field_assign g localRep thisVarOpt tinst localv expr m =
    match lookupIncrClassRepr g localRep localv, thisVarOpt with 
    | Some (false,rfref),Some(thisv) -> 
        let thise = expr_for_val m thisv in
        mk_recd_field_set_via_expra(thise,rfref,tinst,expr,range_of_expr expr)
    | Some (true,rfref),_ -> 
        mk_static_rfield_set(rfref,tinst,expr,range_of_expr expr)
    | _ -> 
        expr (* run for side effects *)
  
let mk_let_field_get_addr g localRep thisVarOpt tinst localv m =
    match lookupIncrClassRepr g localRep localv,thisVarOpt with 
    | Some (false,rfref),Some(thisv) -> 
        let thise = expr_for_val m thisv in
        mk_recd_field_get_addr_via_expra(thise,rfref,tinst,m)
    | Some (true,rfref),_ -> 
        mk_static_rfield_get_addr(rfref,tinst,m)
    | _  ->
        error(InternalError("Local mutable of type unit has was been given storage, yet later it's address has been required",m))

(* Given localRep saying how locals have been represented, e.g. as fields.
 * Given an expr under a given thisv context.
 * Fix up the references to the locals, e.g. v -> this.fieldv.
 *)
let fixupHiddenFieldRefsForIncrClassExpr cenv localRep thisVarOpt thisTyInst expr = 
    (* fixup: intercept and expr rw *)
    let fixupExprNode e =
        match e with
        | TExpr_val (v,flags,m)                         when Zset.mem (deref_val v) localRep.incrClassValsInFields -> Some (mk_let_field_lookup   cenv.g localRep thisVarOpt thisTyInst (deref_val v)     m)
        | TExpr_op(TOp_lval_op (LSet,v)    ,[],[arg],m) when Zset.mem (deref_val v) localRep.incrClassValsInFields -> Some (mk_let_field_assign   cenv.g localRep thisVarOpt thisTyInst (deref_val v) arg m)
        | TExpr_op(TOp_lval_op (LGetAddr,v),[],[]   ,m) when Zset.mem (deref_val v) localRep.incrClassValsInFields -> Some (mk_let_field_get_addr cenv.g localRep thisVarOpt thisTyInst (deref_val v)     m)
        | other -> None in
    Tastops.rewrite_expr { pre_intercept= None; 
                           post_transform = fixupExprNode;
                           underQuotations=true } expr 


(*-------------------------------------------------------------------------
!* Partial class defns 
 *------------------------------------------------------------------------- *)

type tyconBindingsPassA =
  | PassAIncrClassCtor     of incrClassCtorLhs
  | PassAInherit           of Ast.typ * synexpr * val_spec option * range
  | PassAIncrClassBindings of binding list * (* isStatic:*) bool * (*recursive:*) bool
  | PassAMember            of Tast.typ * recursiveBindingInfo * normBinding
  | PassAOpen              of module_path * range
  (* The last 'field' has been initialized, only 'do' comes after *)
  | PassAIncrClassCtorComplete 

type incrClassBindingGroup = 
  | IncrClassBindingGroup of Tast.bind list * (*isStatic:*) bool* (*recursive:*) bool

type tyconBindingsPassB =
  | PassBIncrClassCtor     of incrClassCtorLhs * Tast.bind option
  | PassBInherit           of expr * val_spec option
  | PassBIncrClassBindings of incrClassBindingGroup list
  | PassBMember            of recursiveBindingInfo * tbinding_info * local_typar_ref list
  (* The last 'field' has been initialized, only 'do' comes after *)
  | PassBIncrClassCtorComplete of free_loctypars 
  | PassBOpen              of module_path * range

type tyconBindingsPassC =
  | PassCIncrClassCtor     of incrClassCtorLhs * Tast.bind option
  | PassCInherit           of expr * val_spec option
  | PassCIncrClassBindings of incrClassBindingGroup list
  | PassCMember            of (Tast.expr ref * Range.range) list * (Tast.val_spec * Tast.expr)
  | PassCOpen              of module_path * range
  (* the last 'field' has been initialized, only 'do' comes after *)
  | PassCIncrClassCtorComplete     

(* Given localDecs being mutual bindings (recursive or not). For now, they all get field representation *)
(* apart from unit-typed fields. Generate their initialisation expressions. *)
let mkCtorForIncrClassConstruction cenv denv tpenv incrClassCtorLhs inheritsExpr (decs : tyconBindingsPassC list) = 
    if verboseCC then dprintf0 "---- mkCtorForIncrClassConstruction\n";
    let thisv      = incrClassCtorLhs.incrClassInstanceCtorThisVar  in

    let thisTy    = type_of_val thisv in
      
    (* REVIEW: sloppy mark! *)
    let m = range_of_val thisv in
    let ctorDeclaredTypars = incrClassCtorLhs.incrClassInstanceCtorDeclaredTypars in
    if verboseCC then dprintf1 "ctorDeclaredTypars original : %s\n" (showL (typarsL ctorDeclaredTypars));
    let ctorDeclaredTypars = chooseCanonicalDeclaredTyparsAfterInference cenv.g denv ctorDeclaredTypars m in (* ?? *)  
    if verboseCC then dprintf1 "ctorDeclaredTypars canonical: %s\n" (showL (typarsL ctorDeclaredTypars));

    let thisTyInst = List.map mk_typar_ty ctorDeclaredTypars in

    (* Work out the implicit construction side effects of single bit of a 'let' or 'let rec' bidning in the implicit class construction sequence *)
    let transBind reps (TBind (v,rhs)) =
        (* move to checkMembersForm?? *)
        if mustinline (inlineFlag_of_val v) then
            error(Error("Local class bindings may not be marked inline. Consider lifting the definition out of the class or else do not mark it as inline",range_of_val v));
        let rhs = fixupHiddenFieldRefsForIncrClassExpr cenv reps (Some(thisv)) thisTyInst rhs in
        mk_let_field_assign cenv.g reps (Some(thisv)) thisTyInst v rhs (range_of_expr rhs) in

    (* Work out the implicit construction side effects of a 'let', 'let rec' or 'do' binding in the implicit class construction sequence *)
    let transTrueDec reps (IncrClassBindingGroup(binds,isStatic,isRec)) =
        let actions,reps = 
            if isRec then
                (* recursive is broken, can collapse to sequential bindings *)
                let reps     = List.fold_left (extendIncrClassRepr  isStatic incrClassCtorLhs cenv.g) reps binds in (* extend *)
                let reps     = List.fold_left (reprValIsNowStoredInField         cenv.g) reps binds in (* inscope before  *)
                let actions  = List.map (transBind reps) binds in                    (* since can occur in RHS of own defns *)
                actions,reps
            else (
                if debug then dprintf2 "transDec: %d bindings, isRec=%b\n" (List.length binds) isRec;
                let reps     = List.fold_left (extendIncrClassRepr isStatic incrClassCtorLhs cenv.g) reps binds in  (* extend *)               
                let actions  = List.map (transBind reps) binds in
                let reps     = List.fold_left (reprValIsNowStoredInField cenv.g) reps binds in (* inscope after *)               
                actions,reps
            ) in
        if isStatic then (actions,[]),reps else ([],actions),reps in

    (* Work out the implicit construction side effects of each declaration in the implicit class construction sequence *)
    let transDec reps dec = 
        match dec with 
        (* Construction is done so we can set the ref cell *)
        | PassCIncrClassCtorComplete ->  
            let fixups = 
                match incrClassCtorLhs.incrClassInstanceCtorThisVarRefCellOpt with 
                | None ->  []
                | Some ctorv -> 
                    [ fixupHiddenFieldRefsForIncrClassExpr cenv reps (Some(thisv)) thisTyInst
                        (mk_refcell_set cenv.g m (type_of_val incrClassCtorLhs.incrClassInstanceCtorThisVar) (expr_for_val m ctorv) (expr_for_val m incrClassCtorLhs.incrClassInstanceCtorThisVar)) ] in
            ([],fixups),reps
            
        | PassCIncrClassBindings binds -> 
            let initActions,reps = map_acc_list transTrueDec reps binds  in
            let cctorInitActions, ctorInitActions = split initActions in
            (concat cctorInitActions, concat ctorInitActions), reps 
            
        | _ -> 
            ([],[]),reps in 

    let initActions,reps = map_acc_list transDec emptyIncrClassRepr decs in
    let cctorInitActions, ctorInitActions = split initActions in
    let cctorInitActions = concat cctorInitActions in
    let ctorInitActions = concat ctorInitActions in

    let ctorBody =
        let ctorInitAction = mk_seqs cenv.g m ctorInitActions in
        let m = range_of_val thisv in
        let ldarg0   = mk_ldarg0 m thisTy in
        let ctorBody = mk_seq m inheritsExpr (mk_let m thisv ldarg0 ctorInitAction) in
        let ctorBody = mk_basev_multi_lambdas m [] incrClassCtorLhs.incrClassInstanceCtorBaseValOpt [incrClassCtorLhs.incrClassInstanceCtorArgs] (ctorBody,cenv.g.unit_ty) in
        ctorBody in

    let cctorBodyOpt =
        (* Omit the .cctor if it's empty *)
        if isNil(cctorInitActions) 
        then None 
        else
            let cctorInitAction = mk_seqs cenv.g m cctorInitActions in
            let m = range_of_val thisv in
            let cctorBody = mk_basev_multi_lambdas m [] None [incrClassCtorLhs.incrClassStaticCtorArgs] (cctorInitAction,cenv.g.unit_ty) in
            Some(cctorBody) in
    
    ctorBody,cctorBodyOpt,reps


(*-------------------------------------------------------------------------
!* Partial class defns - "this" from instance member binding
 *------------------------------------------------------------------------- *)

let vx_instance_member_thisv (v,x) =
  (* Skip over LAM tps. Choose 'a. *)
  if member_val_is_instance v then
    let rec firstArg e =
      match e with
        | TExpr_tlambda (_,tps,b,_,rty,_) -> firstArg b
        | TExpr_tchoose (_,b,_) -> firstArg b
        | TExpr_lambda  (_,_,[v],b,_,_,_) -> Some v
        | _ -> failwith "vx_instance_member_thisv: instance member did not have expected internal form"
    in
    firstArg x
  else
    None


(*-------------------------------------------------------------------------
!* Implicit type construction - bindings
 *------------------------------------------------------------------------- *)

let map_acc_list_aux initial f list = map_acc_list f initial list

let tc_tycon_bindings overridesOK cenv env tpenv bindsm scopem (bindsl : tyconBindingDefns list) =
    
    (* PLAN: multiple passes.
     * - create val_specs for recursive items given names and args
     * - type check AST to TAST collecting (sufficient) type constraints
     * - determine typars to generalize over
     * - generalize definitions (fixing up recursive instances)
     * - build ctor binding
     * Yields set of recursive bindings for the ctors and members of the types.
     *)
    let denv = denv_of_tenv env in
    let envInitial = env in
    let env = () in
    
    let defnsAs, (tpenv,prelimRecValues) =
        if verboseCC then dprintf0 "---------- passA: --------------------\n";
        (* PassA: create member prelimRecValues for "recursive" items, i.e. ctor val and member vals *)
        (* PassA: also processes their arg patterns - collecting type assertions *)

        bindsl |> map_acc_list_aux (tpenv,Namemap.empty) (fun (tpenv,prelimRecValues) (TyconBindingDefns(tcref, binds)) -> 

            (* Class members can access protected members of the implemented type *)
            (* Class members can access private members in the type *)
            let envForTycon = mk_inner_env_for_tcref envInitial tcref false in
            let cpath = curr_access_cpath envForTycon in 
            let defnAs,(_,envForTycon,tpenv,prelimRecValues) = 
                binds  |> map_acc_list_aux (None,envForTycon,tpenv,prelimRecValues) (fun (incrClassCtorLhsOpt,env,tpenv,prelimRecValues) (TyconBindingDefn(containerInfo,newslotsOK,declKind,classMemberDef,m)) ->

                    if is_abbrev_tcref tcref then error(Error("Type abbreviations may not have members",m));
                    if is_enum_tcref tcref then error(Error("Enumerations may not have members",m));

                    match classMemberDef, containerInfo with
                    
                      | ClassMemberDefn_implicit_ctor (spats,thisIdOpt, m), ContainerInfo(_,Some(tcref,_,baseVarOpt)) ->
                          (* note; ContainerInfo is certain to be an option *)
                          (* PassA: make incrClassCtorLhs - ctorv, thisv etc, type depends on argty(s) *)
                          let incrClassCtorLhs = tc_implict_ctor_lhs cenv env tpenv tcref spats thisIdOpt baseVarOpt m in
                          (* PassA: Add ctorDeclaredTypars from incrClassCtorLhs - or from tcref *)
                          let env = add_declared_typars CheckForDuplicateTypars incrClassCtorLhs.incrClassInstanceCtorDeclaredTypars env in
                          PassAIncrClassCtor incrClassCtorLhs,
                          (Some(incrClassCtorLhs),env,tpenv,prelimRecValues)
                          
                      | ClassMemberDefn_implicit_inherit (typ,arg,baseIdOpt,m),_ ->
                          (* PassA: inherit typ(arg) as base - pass through *)
                          let baseVarOpt = incrClassCtorLhsOpt |> Option.bind (fun x -> x.incrClassInstanceCtorBaseValOpt) in
                          PassAInherit (typ,arg,baseVarOpt,m),   (* pick up baseVarOpt! *)
                          (incrClassCtorLhsOpt,env,tpenv,prelimRecValues)
                          
                      | ClassMemberDefn_let_bindings (letBinds,isStatic,isRec,m),_ ->

                          if is_struct_tcref tcref then 
                              errorR(Error("Structs may not contain 'let' bindings. Consider adding additional arguments to the primary constructor for the type",m));

                          (* PassA: let-bindings - pass through *)
                          PassAIncrClassBindings (letBinds,isStatic,isRec),
                          (incrClassCtorLhsOpt,env,tpenv,prelimRecValues)     
                          
                      | ClassMemberDefn_member_binding (bind,m),_ ->
                          if verboseCC then dprintf0 "PassA: member\n";
                          (* PassA: member binding - create prelim valspec (for recursive reference) and recursiveBindingInfo *)
                          let bindty = new_inference_typ cenv () in
                          let bind   = normBinding ValOrMemberBinding cenv env bind in
                          let rbind = NormRecBindingDefn(containerInfo,newslotsOK,declKind,bind) in 
                          let (rbinfo,bind),(tpenv,prelimRecValues) = mk_rec_value overridesOK cenv env (tpenv,prelimRecValues) (bindty,rbind) in
                          PassAMember (bindty,rbinfo,bind),
                          (incrClassCtorLhsOpt,env,tpenv,prelimRecValues)
                    
                      | ClassMemberDefn_open (mp,m),_ ->
                          PassAOpen (mp,m),
                          (incrClassCtorLhsOpt,env,tpenv,prelimRecValues)
                    
                      | _ -> 
                          error(InternalError("unexpected definition",m))) in 

            let defnAs = 
                (* Insert PassAIncrClassCtorComplete at the point where local construction is known to have been finished *)
                (* todo: this is quadratic *)
                let rec insertDone defns = 
                    if defns |> List.for_all (function 
                      | PassAOpen _ | PassAIncrClassCtor _ | PassAInherit _ | PassAIncrClassCtorComplete -> false
                      | PassAIncrClassBindings (binds,isStatic,isRec) -> 
                          (* Detect 'let _ =' and 'let () =' bindings, which are 'do' bindings. *)
                          binds |> List.for_all (function (Binding (_,DoBinding,_,_,_,_,_,_,_,_)) -> true | _ -> false)
                      | PassAMember _ -> true)
                    then PassAIncrClassCtorComplete :: defns 
                    else List.hd defns :: insertDone (List.tl defns) in
                insertDone defnAs in

            (envForTycon,defnAs),(tpenv,prelimRecValues)) in


    let prelimRecValues = Namemap.map fst prelimRecValues in
    (* PassB: type check pass, convert from ast to tast and collects type assertions *)
    if verboseCC then dprintf0 "---------- passB: --------------------\n";        


    let defnsBs,tpenv = 

        defnsAs |> map_acc_list_aux tpenv (fun tpenv (envForTycon, defnAs) -> 
            (* add prelimRecValues to env (breaks recursion) and vrec=true *)
            let envinner = addLocalValMap scopem prelimRecValues envForTycon in
            if verboseCC then Namemap.iteri (fun name v -> dprintf2 "prelim %s : %s\n" name ((DebugPrint.showType (type_of_val v)))) prelimRecValues;
            
            (* Set up the environment so use-before-definition warnings are given, at least *)
            (* until we reach a PassAIncrClassCtorComplete. *)
            let envinner = { envinner with eCtorInfo = Some (initialImplicitCtorInfo()) } in
                
            let defnBs,(tpenv,_,_) = 
                defnAs |> map_acc_list_aux (tpenv,envinner,envinner) (fun (tpenv,envInstance,envStatic) defnA -> 
                    match defnA with
                    | PassAIncrClassCtor incrClassCtorLhs ->
                        if verboseCC then dprintf0 "PassB: ctor\n";
                        (* PassB: enrich envInstance with implicit ctor args *)
                        let envInstance = match incrClassCtorLhs.incrClassInstanceCtorThisVarRefCellOpt with Some ctorv -> addLocalVal scopem ctorv envInstance | None -> envInstance in
                        let envInstance = List.fold_right prim_addLocalVal incrClassCtorLhs.incrClassInstanceCtorArgs envInstance in (* Note: previously, would have called callEnvSink *)
                        let thisVarRefCellBindOpt = tc_letrec_compute_ctorThisVarRefCellBind cenv incrClassCtorLhs.incrClassInstanceCtorThisVarRefCellOpt in 
                        PassBIncrClassCtor (incrClassCtorLhs,thisVarRefCellBindOpt),
                        (tpenv,envInstance,envStatic)
                        
                    | PassAInherit (sty,arg,baseVarOpt,m) ->
                        (* PassB: build new object expr for the inherit-call *)
                        let ty,tpenv = tc_type cenv NoNewTypars CheckCxs envInstance tpenv sty in
                        let inheritsExpr,tpenv = tc_new cenv envInstance tpenv ty true arg m in
                        let envInstance = 
                            match baseVarOpt with 
                            | Some basev -> addLocalVal scopem basev envInstance 
                            | None -> envInstance in
                        PassBInherit (inheritsExpr,baseVarOpt),
                        (tpenv,envInstance,envStatic)
                        
                    | PassAIncrClassBindings (binds,isStatic,isRec) ->
                        if verboseCC then dprintf0 "PassB: bindings\n";
                        let envForMember = if isStatic then envStatic else envInstance in
                        (* PassB: let bindings *)
                        let binds,bindRs,env,tpenv = 
                            if isRec then
                                (* type check local recursive binding *)
                                let binds = binds |> map (fun bind -> RecBindingDefn(exprContainerInfo,NoNewSlots,ClassLetBinding,bind)) in
                                let binds,env,tpenv = tc_letrec WarnOnOverrides cenv envForMember tpenv (new_inference_typs cenv binds) (binds,scopem(*bindsm*),scopem) in
                                let bindRs = [IncrClassBindingGroup(binds,isStatic,true)] in
                                binds,bindRs,env,tpenv 
                            else
                                (* type check local binding *)
                                let binds,env,tpenv = tc_let_bindings cenv envForMember exprContainerInfo ClassLetBinding tpenv (binds,bindsm,scopem) in
                                let bindRs = map (fun bind -> IncrClassBindingGroup([bind],isStatic,false)) binds in
                                binds,bindRs,env,tpenv in
                        (* Also add static entries to the envInstance if necessary *)
                        let envInstance = (if isStatic then List.fold_right (var_of_bind >> addLocalVal scopem) binds envInstance else env) in
                        let envStatic = (if isStatic then env else envStatic) in
                        PassBIncrClassBindings bindRs,
                        (tpenv,envInstance,envStatic)
                          
                    | PassAIncrClassCtorComplete -> 
                        (* Lift the restriction that results in use-before-initialization warnings *)
                        PassBIncrClassCtorComplete empty_free_loctypars, (* <-- PATCHED: the ungeneralisable typars are computed in PassB2 below *)
                        (tpenv, clear_ctorPreConstruct envInstance,envStatic)
                        
                    | PassAOpen(mp,m) -> 
                        (* Lift the restriction that results in use-before-initialization warnings *)
                        let envInstance = tc_open_namespace cenv.autoModuleResolver m scopem envInstance mp in 
                        let envStatic = tc_open_namespace cenv.autoModuleResolver m scopem envStatic mp in 
                        PassBOpen(mp,m),
                        (tpenv, envInstance,envStatic  )

                    (* Note: this path doesn't add anything the environment, because the member is already available off via its type *)
                    | PassAMember (bindty,rbinfo,bind) ->
                        if verboseCC then dprintf0 "PassB: member\n";
                        (* PassB: Typecheck member binding, generalize them later, when all type constraints are known *)
                        (* static members are checked under envinner.
                         * envinner contains class typars and the (ungeneralized) members on the class(es).
                         * envinner has no instance-variables (local let-bindings or ctor args). *)
                        let (RBInfo(_,_,v,_,_,_,_,_,_)) = rbinfo in 

                        let envForMember = if member_val_is_instance v then envInstance else envStatic in

                        let (rbinfo,tbind),tpenv = tc_letrec_tc_info_binding cenv envForMember scopem prelimRecValues tpenv (bindty,(rbinfo,bind)) in
                         
                        if verboseCC then dprintf1 "PassBMember: vspec = %s\n" (vspecTyL v |> showL);
                        
                        let uncomputedGeneralizedTypars= [] in (* <-- PATCHED: generalized typars are computed in PassB2 below *)
                        
                        PassBMember (rbinfo,tbind,uncomputedGeneralizedTypars),
                        (tpenv,envInstance,envStatic)) in
            defnBs,tpenv) in

    defnsBs |> iter (fun defnBs -> 
        (* Take ctor (and inherit if present) and build ctor binding rhs expr.
         * Also yields fixup function to fixup references to local state.
         * That must be applied to the expr in the member definitions - and - doing so makes some type assertions on the member this typars.
         * Q: can the member this typars ever be anything other than free typars???
         *------
         * Is it enough to make all the "this" arguments type equal?
         * If they refer to a local state variable, then this assertion will be made.
         * If they dont, is there harm in making that assertion??
         * Surely, those typars must after typechecking still be a unique set of typar?
         *)
        (* XXX match failure XXX *)
        match defnBs with 
        | PassBIncrClassCtor (incrClassCtorLhs,thisVarRefCellBindOpt) :: otherDefnBs ->
            (* Assert the types of 'this' *)
            otherDefnBs |> List.iter (function
                | PassBMember (rbinfo,tbind,_) ->
                    let (RBInfo(_,_,vspec,_,_,_,_,_,_)) = rbinfo in
                    let (TBindingInfo(_,_,_,_,_,_,_,expr,_,_,m,_,_)) = tbind in
                    (match vx_instance_member_thisv (vspec,expr) with
                       | None -> ()
                       | Some vx_thisv ->
                           let incrClassInstanceCtorThisVar = incrClassCtorLhs.incrClassInstanceCtorThisVar in
                           if verboseCC then dprintf2 "vx_thisv.Type = %s, incrClassInstanceCtorThisVar.Type = %s \n" (vspecTyL vx_thisv |> showL) (showL (typeL (type_of_val incrClassInstanceCtorThisVar)));
                           unifyE cenv envInitial m (type_of_val vx_thisv) (type_of_val incrClassInstanceCtorThisVar))
                | _ -> ())
        | _ -> ());
      
    (* This is enough to proceed with generalisation. *)

    (* Pass B2: This computes the ungeneralisable typars for the member block(s).
     * These typars are:
     * - those typars free in envInitial
     * - those typars free in the let-bindings inscope over the members,
     *   in particular, these dangling typars are allowed to remain,
     *   to be resolved later in the file, but they must not be generalized.
     *   CONSIDER: ungeneralized typars could generate a warning.
     * Implementation:
     *   For each type,
     *     We add the defns to an environment based on envInitial (not envinner).
     *     Note, envinner = envInitial + prelimRecValues, and the prelimRecValues are the members.
     *   The free typars of each type's env determines ungeneralisable typars.
     *   The union of the these are ungeneralisable over the members letrec block.
     *)
    let defnsBs = 
        defnsBs |> map (fun defnBs ->
            let defnBs,_ = 
                defnBs |> map_acc_list_aux envInitial (fun env defnB ->
                  match defnB with
                    | PassBIncrClassCtor (incrClassCtorLhs,_) ->
                        if verboseCC then dprintf0 "PassB2: ctor\n";
                        let env = match incrClassCtorLhs.incrClassInstanceCtorThisVarRefCellOpt with Some ctorv -> addLocalVal scopem ctorv env | None -> env in
                        let env = List.fold_right prim_addLocalVal incrClassCtorLhs.incrClassInstanceCtorArgs env in (* Note: previously, would have called callEnvSink *)
                        defnB,env
                    | PassBInherit (inheritsExpr,baseVarOpt) ->
                        let env = match baseVarOpt with Some basev -> addLocalVal scopem basev env | None -> env in
                        defnB,env
                    | PassBIncrClassBindings bindRs ->
                        let collectBind env bind = prim_addLocalVal (var_of_bind bind) env in
                        let collectBindRs env (IncrClassBindingGroup(binds,_,_)) = List.fold_left collectBind env binds in
                        let env = List.fold_left collectBindRs env bindRs in
                        defnB,env
                    | PassBIncrClassCtorComplete ignoredSinceItIsComputedHere ->
                        PassBIncrClassCtorComplete (computeUngeneralizableTypars env),  (* <--- PATCHING: this pass computes this typar set *)
                        env
                    | PassBMember (bindty,rbinfo,bind) ->
                        defnB,env
                    | PassBOpen (_,m) -> error(Error("'open' declarations may not be used in classes",m))) in
            defnBs) in

    begin         
        let supportForBindings = 
            defnsBs |> mapConcat (fun defnBs -> 
                defnBs |> mapConcat (fun defnB -> 
                    match defnB with
                      | PassBOpen _ | PassBIncrClassCtor  _ | PassBInherit _  | PassBIncrClassBindings _ | PassBIncrClassCtorComplete _ -> []
                      | PassBMember (rbinfo ,tbind,ignoredSinceItIsComputedHere) -> tc_letrec_computeSupportForBinding (rbinfo,tbind))) in 
        canonicalizePartialInferenceProblem (cenv,denv,bindsm) supportForBindings; 
    end;

    let freeInEnv = 
        let freeInEnvBase = computeUngeneralizableTypars envInitial in
        let freeInEnvForMembers = fold1 Zset.union (defnsBs |> concat |> chooseList (function PassBIncrClassCtorComplete ungeneralizableTypars -> Some ungeneralizableTypars | _ -> None)) in
        (* The type variables on the type and associated with the constructor are generalizable in the members. *)
        (* UnGen = freeInEnv(Ctor) + (freeInEnv(Members) - freeInEnv(ctorTyVars)) *)
        
        let freeInEnvForMembersMinusCtorVars = 
            List.fold_right (fun defnBs acc -> 
                match defnBs with 
                | PassBIncrClassCtor (incrClassCtorLhs,thisVarRefCellBindOpt) :: otherDefnBs ->
                    List.fold_right Zset.remove incrClassCtorLhs.incrClassInstanceCtorDeclaredTypars acc 
                | _ -> acc)
                defnsBs
                freeInEnvForMembers in
        let ungeneralizableTypars  = Zset.union freeInEnvBase freeInEnvForMembersMinusCtorVars in
        ungeneralizableTypars in

    (* PassB3: fill in generalizedTypars now all equations/constraints are known *)
    if verboseCC then dprintf0 "---------- passB3: --------------------\n";    

    let defnsBs = 
        defnsBs |> map (fun defnBs -> 
            defnBs |> map (fun defnB -> 
                match defnB with
                  | PassBOpen _ | PassBIncrClassCtor  _ | PassBInherit _  | PassBIncrClassBindings _ | PassBIncrClassCtorComplete _ -> defnB
                  | PassBMember (rbinfo ,tbind,ignoredSinceItIsComputedHere) ->
                      let generalizedTypars = tc_letrec_computeGeneralizedTyparsForBinding cenv denv freeInEnv (rbinfo,tbind) in
                      if verboseCC then dprintf1 "PassB3: member generalizedTypars = %s\n" (showL (typarsL generalizedTypars));
                      PassBMember (rbinfo,tbind,generalizedTypars))) in (* <--- PATCHING: filling in generalizedTypars *)

    (* Compute the entire set of generalized typars, including those on ctors *)
    let generalizedTyparsForRecursiveBlock =
        defnsBs 
        |> map (chooseList (function
                           | PassBMember(_,_,generalizedTypars) -> Some generalizedTypars
                           | PassBIncrClassCtor (incrClassCtorLhs,_) -> Some (incrClassCtorLhs.incrClassInstanceCtorDeclaredTypars)
                           | _ -> None) 
                >> unionGeneralizedTypars)
        |> unionGeneralizedTypars in

    if verboseCC then dprintf1 "PassB3: generalizedTyparsForRecursiveBlock = %s\n" (typarsL generalizedTyparsForRecursiveBlock |> Layout.showL);

    (* PassC: generalize - both ctor and members *)
    if verboseCC then dprintf0 "---------- passC: --------------------\n";

    let defnsCs,tpenv = 
        defnsBs |> map_acc_list_aux tpenv (fun tpenv defnBs -> 
            defnBs |> map_acc_list_aux tpenv (fun tpenv defnB -> 

                (* PassC: Generalise implicit ctor val *)
                match defnB with
                | PassBIncrClassCtor (incrClassCtorLhs,thisVarRefCellBindOpt) ->
                    let valscheme = incrClassCtorLhs.incrClassInstanceCtorValScheme in
                    let valscheme = chooseCanonicalValSchemeAfterInference cenv.g denv valscheme scopem in
                    adjustRecType cenv incrClassCtorLhs.incrClassInstanceCtorVal valscheme;
                    PassCIncrClassCtor (incrClassCtorLhs,thisVarRefCellBindOpt),tpenv

                | PassBInherit (inheritsExpr,basevOpt) -> 
                    PassCInherit (inheritsExpr,basevOpt),tpenv

                | PassBIncrClassBindings bindRs             -> 
                    PassCIncrClassBindings bindRs,tpenv

                | PassBIncrClassCtorComplete _ -> 
                    PassCIncrClassCtorComplete, tpenv

                | PassBOpen(mp,m) -> 
                    PassCOpen(mp,m), tpenv

                | PassBMember (rbinfo,tbind,generalizedTypars)  ->
                    if verboseCC then dprintf0 "PassC: member\n";
                    (* PassC: Generalise member bindings *)
                    let vxbind = tc_letrec_generalizeBinding cenv denv generalizedTyparsForRecursiveBlock freeInEnv generalizedTypars (rbinfo,tbind) in
                    let tpenv = hideUnscopedTypars generalizedTypars tpenv in
                    let vxbind = tc_letrec_bind_ctorv cenv vxbind rbinfo in
                    let fixups,vxbind = tc_letrec_fixup_vxbind cenv (denv_of_tenv envInitial) vxbind in
                    PassCMember (fixups,vxbind),
                    tpenv)) in
                    
    if verboseCC then dprintf0 "---------- make inherit expression --------------------\n";

    (* --- Extract local vals from let-bindings *)
    let fixups_vxbinds =
        defnsCs |> mapConcat (fun defnCs -> 
            match defnCs with 
            | PassCIncrClassCtor (incrClassCtorLhs,thisVarRefCellBindOpt) :: defnCs -> 

                (* This is the type definition we're processing *)
                let tcref = incrClassCtorLhs.incrClassTcref in

                (* Assumes inhert call immediately follows implicit ctor. Checked by checkMembersForm *)
                let inheritsExpr,baseVarOpt,defnCs = 
                    match defnCs with
                    | PassCInherit (inheritsExpr,baseVarOpt) :: defnCs -> inheritsExpr,baseVarOpt,defnCs
                    | defnCs ->
                        if is_struct_tcref tcref then 
                            mk_unit cenv.g bindsm, None, defnCs
                        else
                            let inheritsExpr,tpenv = tc_new cenv envInitial tpenv cenv.g.obj_ty true (Expr_const(Const_unit,bindsm)) bindsm in
                            inheritsExpr,None,defnCs in
                   
                (* Compute the cpath used when creating the hidden fields *)
                let cpath = 
                    let envForTycon = mk_inner_env_for_tcref envInitial tcref false in 
                    curr_access_cpath envForTycon in 

                let localDecs  = 
                    defnCs |> filter (function PassCIncrClassBindings _ | PassCIncrClassCtorComplete -> true | _ -> false) in

                (* Extend localDecs with "let ctorv = ref null" if there is a ctorv *)
                let localDecs  = 
                    match thisVarRefCellBindOpt with 
                    | None -> localDecs 
                    | Some bind -> PassCIncrClassBindings [IncrClassBindingGroup([bind],false,false)] :: localDecs in
                    
                (* Extend localDecs with "let arg = arg" bindings for each ctor arg, which is current mechanism for storing them as fields *)
                let localDecs  = 
                    let binds = incrClassCtorLhs.incrClassInstanceCtorArgs |> List.map (fun v -> mk_bind v (expr_for_val (range_of_val v) v)) in 
                    let ctorArgBinds = IncrClassBindingGroup(binds,false,false) in
                    PassCIncrClassBindings [ctorArgBinds] :: localDecs in

                (* Carve out the initialization sequence and decide on the localRep *)
                let ctorBodyLambdaExpr,cctorBodyLambdaExprOpt,localReps = 
                    mkCtorForIncrClassConstruction cenv (denv_of_tenv envInitial) tpenv incrClassCtorLhs inheritsExpr localDecs in

                (* Generate the (value,expr) pairs for the implicit object constructor and implicit static initializer *)
                let ctor_vxbinds = 
                    [ let ctor_vxbind = (incrClassCtorLhs.incrClassInstanceCtorVal,ctorBodyLambdaExpr) in
                      tc_letrec_fixup_vxbind cenv (denv_of_tenv envInitial) (incrClassCtorLhs.incrClassInstanceCtorValScheme ,ctor_vxbind) ]
                    @ 
                    ( match cctorBodyLambdaExprOpt with 
                      | None -> []
                      | Some(cctorBodyLambdaExpr) -> 
                         [ let cctor_vxbind = (incrClassCtorLhs.incrClassStaticCtorVal,cctorBodyLambdaExpr) in
                           tc_letrec_fixup_vxbind cenv (denv_of_tenv envInitial) (incrClassCtorLhs.incrClassStaticCtorValScheme ,cctor_vxbind) ] )  in

                (* Publish the fields of the representation to the type *)
                begin 
                    let rfspecs   = 
                        localReps.incrClassSecretFields
                        |> Zmap.to_list 
                        |> List.filter (fst >> lookupIncrClassRepr cenv.g localReps >> isSome)
                        |> List.map (mkIncrClassField cpath incrClassCtorLhs.incrClassRevTypeInst) in

                    if verboseCC then dprintf0 "add fields to tcref\n";
                    publishIncrClassFields rfspecs tcref; (* mutation *)    
                end;
                
                (* --- Members *)
                if verboseCC then dprintf0 "---- fixup members\n";      
                let fixups_vxbinds = 
                    let vxbinds = defnCs |> chooseList (function PassCMember(fixups,vxbind) -> Some (fixups,vxbind) | _ -> None)  in
                    let applySubToVXBind (v,x) =
                        (* Work out the 'this' variable and type instantiation for field fixups. *)
                        (* We use the instantiation from the instance member if any. Note: It is likely this is not strictly needed *)
                        (* since we unify the types of the 'this' variables with those of the ctor declared typars. *)
                        let thisVarOpt = vx_instance_member_thisv (v,x) in
                        let thisTyInst = 
                            match thisVarOpt with 
                            | None -> 
                                map mk_typar_ty incrClassCtorLhs.incrClassInstanceCtorDeclaredTypars 
                            | Some thisv -> 
                                let thisTy = type_of_val thisv in 
                                let objTy = (if is_byref_ty cenv.g thisTy then dest_byref_ty cenv.g thisTy else thisTy) in
                                tinst_of_stripped_typ objTy in
                        
                        v,fixupHiddenFieldRefsForIncrClassExpr cenv localReps thisVarOpt thisTyInst x in
                    
                    let vxbinds = List.map (map2'2 applySubToVXBind) vxbinds in  
                    vxbinds in
                    
                ctor_vxbinds @ fixups_vxbinds 
            | _ -> 
                let fixups_vxbinds = defnCs |> chooseList (function PassCMember(fixups,vxbind) -> Some (fixups,vxbind) | _ -> None)  in
                fixups_vxbinds) in   

    if verboseCC then dprintf0 "---- init graphs\n";          
    (* INITIALIZATION GRAPHS *)
    let binds = eliminateInitializationGraphs cenv.g (must_have_arity ModuleOrMemberBinding) (denv_of_tenv envInitial) fixups_vxbinds bindsm in

    (* Post letrec env *)
    let envbody = addLocalValMap scopem prelimRecValues envInitial  in
    if verboseCC then dprintf0 "tc_tycon_bindings: done\n";
    binds,envbody,tpenv


(*-------------------------------------------------------------------------
!* The member portions of class defns
 *------------------------------------------------------------------------- *)
    
let tc_tycon_member_defns overridesOK cenv env parent bindsm scopem membersl = 
    let interfacesFromTypeDefn (TyconMemberData(_,tcref,_,members,m,_)) =
        members |> mapConcat (function 
            | ClassMemberDefn_interface(ity,defnOpt,m) -> 
                  if is_abbrev_tcref tcref then error(Error("Type abbreviations may not have interface declarations",m));
                  if is_enum_tcref tcref then error(Error("Enumerations may not have interface declarations",m));

                  begin match defnOpt with 
                  | Some(defn) -> 
                      let tcaug = tcaug_of_tcref tcref in 
                      let ity' = 
                          let envinner = add_declared_typars CheckForDuplicateTypars (typars_of_tcref tcref) env in 
                          tc_type_and_recover cenv NoNewTypars CheckCxs envinner emptyTpenv ity |> fst in 
                      if not (is_interface_typ ity') then errorR(Error("This type is not an interface type",range_of_syntype ity));
                      
                      if not (tcaug_has_interface cenv.g tcaug ity') then 
                          error(Error("All implemented interfaces should be declared on the initial declaration of the type",range_of_syntype ity));
                      if (type_equiv cenv.g ity' cenv.g.mk_IComparable_ty && isSome(tcaug.tcaug_compare)) || 
                          (type_equiv cenv.g ity' cenv.g.mk_IStructuralHash_ty && isSome(tcaug.tcaug_structural_hash)) then 
                          errorR(Error("A default implementation of this interface has already been added because the explicit implementation of the interface was not specified at the definition of the type",range_of_syntype ity));
                      if overridesOK = WarnOnOverrides then  
                          warning(IntfImplInAugmentation(range_of_syntype ity));
                      [ (ity',defn,m) ]
                  | _-> []
                  end
                  
            | _ -> [])  in

    let interfaceMembersFromTypeDefn (TyconMemberData(declKind,tcref,baseVarOpt,_,m,newslotsOK)) (ity',defn,m) implTySet =
        let tcaug = tcaug_of_tcref tcref in 
        let containerInfo = ContainerInfo(parent,Some(tcref,Some(ity',implTySet),baseVarOpt)) in 
        defn  |> chooseList (fun mem ->
                match mem with
                | ClassMemberDefn_member_binding(b,m) -> 
                    Some(TyconBindingDefn(containerInfo,newslotsOK,declKind,mem,m))
                | ClassMemberDefn_let_bindings(_,_,_,m)    (* <-- possible design suggestion: relax this *)
                | ClassMemberDefn_implicit_ctor(_,_,m)
                | ClassMemberDefn_implicit_inherit(_,_,_,m)
                | ClassMemberDefn_interface(_,_,m) 
                | ClassMemberDefn_slotsig(_,_,m)
                | ClassMemberDefn_inherit(_,_,m)
                | ClassMemberDefn_field(_,m)
                | ClassMemberDefn_open (_,m)
                | ClassMemberDefn_tycon(_,_,m) -> errorR(Error("This member is not permitted in an interface implementation",m)); None) in

    let tpenv = emptyTpenv in
    if verboseCC then dprintf0 "tc_tycon_member_defns:\n";
    if verboseCC then membersl |> iter (fun (TyconMemberData(declKind,tcref,baseVarOpt,members,m,newslotsOK)) -> dprintf1 "members: %d\n" (List.length members));
    try
      (* Some preliminary checks *)
      membersl |> iter (fun (TyconMemberData(declKind,tcref,baseVarOpt,(members : classMemberDefn list),m,newslotsOK)) -> 
             let tcaug = tcaug_of_tcref tcref in 
             if tcaug.tcaug_closed && declKind <> ExtensionBinding then 
               error(Error("Augmentations of types are only permitted in the same compilation unit as the definition of the type",m));
             members |> iter (function 
                    | ClassMemberDefn_member_binding _ -> ()
                    | ClassMemberDefn_interface _ -> () 
                    | ClassMemberDefn_open _ 
                    | ClassMemberDefn_let_bindings _  (* accept local definitions *)
                    | ClassMemberDefn_implicit_ctor _ (* accept implicit ctor pattern, should be first! *)
                    | ClassMemberDefn_implicit_inherit _ when newslotsOK = NewSlotsOK -> () (* accept implicit ctor pattern, should be first! *)
                    (* The follow should have been removed by splitting, they belong to "core" (they are "shape" of type, not implementation) *)
                    | ClassMemberDefn_open (_,m) 
                    | ClassMemberDefn_let_bindings(_,_,_,m) 
                    | ClassMemberDefn_implicit_ctor(_,_,m)
                    | ClassMemberDefn_implicit_inherit(_,_,_,m) 
                    | ClassMemberDefn_slotsig(_,_,m)
                    | ClassMemberDefn_inherit(_,_,m)
                    | ClassMemberDefn_field(_,m)
                    | ClassMemberDefn_tycon(_,_,m) -> error(Error("This declaration element is not permitted in an augmentation",m))));

      let tyconBindingsOfTypeDefn (TyconMemberData(declKind,tcref,baseVarOpt,members,m,newslotsOK)) =
          let containerInfo = ContainerInfo(parent,Some(tcref,None,baseVarOpt)) in
          members 
          |> chooseList (fun memb ->
              match memb with 
              | ClassMemberDefn_implicit_ctor(_,_,m)
              | ClassMemberDefn_implicit_inherit(_,_,_,m) 
              | ClassMemberDefn_let_bindings(_,_,_,m) 
              | ClassMemberDefn_member_binding(_,m) 
              | ClassMemberDefn_open (_,m) 
                  -> Some(TyconBindingDefn(containerInfo,newslotsOK,declKind,memb,m))

              (* Interfaces exist in the member list - handled above in interfaceMembersFromTypeDefn *)
              | ClassMemberDefn_interface _  -> None

              (* The following should have been split out already in split_tyconDefn *)
              | ClassMemberDefn_slotsig (_,_,m) 
              | ClassMemberDefn_field (_,m)             
              | ClassMemberDefn_inherit (_,_,m)    -> error(InternalError("Unexpected declaration element",m))
              | ClassMemberDefn_tycon (_,_,m)      -> error(Error("Types may not contain nested type definitions",m))) in
          
      let binds  = 
          membersl |> map (fun (TyconMemberData(_,tcref,_,_,_,_) as tyconMemberData) -> 
              let obinds = tyconBindingsOfTypeDefn tyconMemberData in
              let ibinds  = 
                      let intfTypes = interfacesFromTypeDefn tyconMemberData in 
                      let implTySets = getImplSets (denv_of_tenv env) cenv.g cenv.amap (map (fun (ity,_,m) -> (ity,m)) intfTypes) in
                      concat (map2 (interfaceMembersFromTypeDefn tyconMemberData) intfTypes implTySets) in
              if verboseCC then dprintf1 "obinds: %d\n" (List.length obinds);
              if verboseCC then dprintf1 "ibinds: %d\n" (List.length ibinds);
              TyconBindingDefns(tcref, obinds @ ibinds)) in
      
      let results = tc_tycon_bindings overridesOK cenv env tpenv bindsm scopem binds in
      let binds,envbody,tpenv = results in
      binds,envbody

    with e -> errorRecoveryPoint e; [], env

(*-------------------------------------------------------------------------
!* Bind exception definitions
 *------------------------------------------------------------------------- *)

let add_compare_bindings cenv env tycon =
    if Augment.is_augmented_with_compare cenv.g tycon then 
        let tcref = mk_local_tcref tycon in 
        let tcaug = tcaug_of_tycon tycon in 
        let m = range_of_tycon tycon in 
        let hasExplicitIComparable     = tcaug_has_interface cenv.g tcaug cenv.g.mk_IComparable_ty in 
        let cvspec = Augment.mk_vspecs_for_compare_augmentation cenv.g tcref in 

        if not hasExplicitIComparable then begin
         publish_interface cenv (denv_of_tenv env) tcref m true cenv.g.mk_IComparable_ty;
         set_tcaug_compare tcaug (mk_local_vref cvspec);
         publishValueDefn cenv env ModuleOrMemberBinding cvspec;
        end;
        Augment.mk_bindings_for_compare_augmentation cenv.g tcref tycon
    else
        []

let add_hash_bindings cenv env tycon =
    if Augment.is_augmented_with_hash cenv.g tycon then 
        let tcref = mk_local_tcref tycon in 
        let tcaug = tcaug_of_tycon tycon in 
        let m = range_of_tycon tycon in 
        let hasExplicitIStructuralHash = tcaug_has_interface cenv.g tcaug cenv.g.mk_IStructuralHash_ty in 
        let hvspec = Augment.mk_vspecs_for_hash_augmentation cenv.g tcref in 

        if not hasExplicitIStructuralHash then begin
         publish_interface cenv (denv_of_tenv env) tcref m true cenv.g.mk_IStructuralHash_ty;
         set_tcaug_hash tcaug (mk_local_vref hvspec);
         publishValueDefn cenv env ModuleOrMemberBinding hvspec;
        end;
        Augment.mk_bindings_for_hash_augmentation cenv.g tcref tycon
    else
        []

let add_hash_compare_bindings cenv env tycon =
    add_hash_bindings cenv env tycon @ add_compare_bindings cenv env tycon 



(* We can only add the Equals override after we've done the augmentation becuase we have to wait until tcaug_has_override can give correct results *)
let add_equals_bindings_late cenv env tycon =
    if Augment.is_augmented_with_equals cenv.g tycon then (
        let tcref = mk_local_tcref tycon in 
        let tcaug = tcaug_of_tycon tycon in 
        (* Note: tcaug_has_override only gives correct results after we've done the type augmentation *)
        let hasExplicitObjectEqualsOverride = tcaug_has_override cenv.g tcaug "Equals" [cenv.g.obj_ty] in 

        (* Note: only provide the equals method if IComparable not implemented explicitly *)
        (* Prior to F# 1.9.3.14 you only had to implement 'System.IComparable' to customize structural comparison AND equality on F# types *)
        (* Post 1.9.3.14 this is insufficient: you must override Equals as well. For compat we currently *)
        (* give a warning in this situation (see ilxgen.ml) and use the IComparable implementation instead *)
        if not (Augment.is_augmented_with_compare cenv.g tycon  && isNone tcaug.tcaug_compare) &&
           not hasExplicitObjectEqualsOverride then (

            let vspec1,vspec2 = Augment.mk_vspecs_for_equals_augmentation cenv.g tcref in 
            set_tcaug_equals tcaug (mk_local_vref vspec1, mk_local_vref vspec2);
            publishValueDefn cenv env ModuleOrMemberBinding vspec1;
            publishValueDefn cenv env ModuleOrMemberBinding vspec2;
            Augment.mk_bindings_for_equals_augmentation cenv.g tcref tycon
        ) else []
    ) else []

(*-------------------------------------------------------------------------
!* Bind exception definitions
 *------------------------------------------------------------------------- *)

let checkForDuplicateConcreteType cenv env nm m  = 
    let curr = curr_mtyp_acc env in
    if Map.mem nm curr.mtyp_tycons then 
        (* Use 'error' instead of 'errorR' here to avoid cascading errors - see bug 1177 in FSharp 1.0 *)
        error (Duplicate("type, exception or module",ident(nm,m)))

let checkForDuplicateModule cenv env nm m  = 
    let curr = curr_mtyp_acc env in
    if mtyp_has_submodul nm curr then 
        errorR (Duplicate("type or module",ident(nm,m)))

let tc_exn_core cenv env parent declKind tpenv (ExconCore(attrs,UnionConstr(_,id,args,_,_,_),repr,doc,vis,m), scopem) =
    let attrs' = tc_attributes cenv env attrTgtExnDecl attrs in 
    let args = match args with (ConstrFields args) -> args | _ -> error(Error("Explicit type specifications may not be used for exception constructors",m)) in 
    
    let args' = list_mapi (fun i fdef -> tc_anon_field_decl cenv env parent declKind tpenv ("data"^string_of_int i) fdef) args in 
    if is_lower_string id.idText then error(LowerCaseConstructor(m));
    let vis,cpath = computeAccessAndCompPath env declKind m vis parent in 
    let exnc = 
      match repr with 
      | Some lid ->
          (* REVIEW: permit type arguments in this lid. *)
          begin match tc_expr_lid cenv.ginstf cenv.g cenv.amap m env.eNameResEnv defaultTypeNameResInfo lid with
          | Item_ecref exnc, [] -> 
              tcref_accessible_check m (accessRightsOfEnv env) exnc;
              if List.length args' <> 0 then 
                errorR (Error("Exception abbreviations should not have argument lists",m));
              new_exnc cpath id vis (TExnAbbrevRepr exnc) attrs' doc
          | Item_ctor_group(_,meths) , [] -> 
              (* REVIEW: check this really is an exception type *)
              begin match args' with 
              | [] -> ()
              | _ -> error (Error("Abbreviations for .NET exceptions may not take arguments",m));
              end;
              begin
                let candidates = 
                    meths |> filter (fun minfo -> 
                        arity_of_minfo cenv.g minfo = length args' &&
                        (* & (length args' <> 1 or  typ_of_param (hd md.mdParams) = Il.typ_String) *) 
                        generic_arity_of_minfo cenv.g minfo = 0)  in
                match candidates with 
                | [minfo] -> 
                    let err() = 
                        Error("Exception abbreviations must refer to existing exceptions or F# types deriving from System.Exception",m) in
                    if not (type_definitely_subsumes_type_no_coercion 0 cenv.g cenv.amap m cenv.g.exn_ty (typ_of_minfo minfo)) then 
                      errorR(err());
                    let tref = 
                        match minfo with 
                        | ILMeth minfo -> tref_of_il_minfo minfo
                        | FSMeth _ -> 
                            begin match il_repr_of_tcref (tcref_of_stripped_typ (typ_of_minfo minfo )) with 
                            | TyrepNamed (tref,_) -> tref
                            | _ -> 
                                error (err()) 
                            end 
                        | _ -> error (err())  in
                    new_exnc  cpath id vis (TExnAsmRepr tref) attrs' doc
                | _ -> 
                    error (Error("Abbreviations for .NET exception types must have a matching object constructor",m))
              end
          | _ ->
              error (Error("not an exception",m))
          end
      | None -> 
         new_exnc cpath id vis (TExnFresh (mk_rfields_table args')) attrs' doc in 

    let tcaug = tcaug_of_tycon exnc in
    tcaug.tcaug_super <- Some cenv.g.exn_ty;

    checkForDuplicateConcreteType cenv env (id.idText ^ "Exception") id.idRange;
    checkForDuplicateConcreteType cenv env id.idText id.idRange;
    publishTypeDefn cenv env exnc;
    (* let env = addLocalExnc scopem exnc (addLocalTycons scopem [exnc] env) in  *)
    (* Augment the exception constructor with comparison and hash methods if needed *)
    let binds = 
      match exn_repr_of_tycon exnc with 
      | TExnAbbrevRepr _ | TExnNone | TExnAsmRepr _ -> []
      | TExnFresh _ -> add_hash_compare_bindings cenv env exnc in 
    binds,
    exnc,
    addLocalExnc scopem exnc (addLocalTycons scopem [exnc] env)

let tc_exn_def cenv env parent declKind tpenv (ExconDefn(core,aug,m),scopem) = 
  let binds1,exnc,env = tc_exn_core cenv env parent declKind tpenv (core,scopem) in 
  let binds2,env = tc_tycon_member_defns OverridesOK cenv env parent m scopem [TyconMemberData(declKind,(mk_local_ecref exnc),None,aug,m,NoNewSlots)] in 
  (* Augment types with references to values that implement the pre-baked semantics of the type *)
  let binds3 = add_equals_bindings_late cenv env exnc in
  binds1 @ binds2 @ binds3,exnc,env

let tc_exn_spec cenv env parent declKind tpenv (ExconSpfn(core,aug,m),scopem) = 
  let binds,exnc,env = tc_exn_core cenv env parent declKind tpenv (core,scopem) in 
  let ecref = mk_local_ecref exnc in
  let vals,_ = tc_class_spec NoNewSlots OverridesOK cenv env (ContainerInfo(parent,Some(ecref,None,None))) ModuleOrMemberBinding ecref tpenv aug in
  binds,vals,ecref,env


(*-------------------------------------------------------------------------
!* Bind type definitions - first code that is common to defs and specs
 * (i.e. defining the tycons and dealing with abbreviations, simple
 * representations such as records etc.)
 *------------------------------------------------------------------------- *)

let compute_isModule kind parms constraints im = 
    (if nonNil parms || nonNil constraints then error(Error("unexpected constraints or parameters on module specification",im)));
    (match kind with TMK_Module -> true | TMK_Namespace -> false | TMK_Constraint | TMK_Tycon -> error(Error("unexpected constraint or type definition",im)))
    
let assert_isTycon kind im = 
    (match kind with TMK_Tycon -> () | TMK_Constraint | TMK_Module | TMK_Namespace -> error(Error("unexpected constraint, module or namespace specification",im))) 
let assert_isConstraint kind im = 
    (match kind with TMK_Constraint -> () | TMK_Tycon | TMK_Module | TMK_Namespace -> error(Error("unexpected type, module or namespace specification",im))) 

let computeTyconName (longPath: ident list) typars = 
    if List.length longPath <> 1 then error(Error("invalid type extension",(List.hd longPath).idRange));
    let id = List.hd longPath in 
    mksyn_id id.idRange (if isNil typars then id.idText else id.idText^"`"^string_of_int (length typars)) 

let computeModuleName (longPath: ident list) = 
    if List.length longPath <> 1 then error(Error("invalid module extension",(List.hd longPath).idRange));
    List.hd longPath 

let computeTyconDeclKind newslotsOK cenv env m typars longPath = 
    let tcref = forceRaise(tc_tycon_id OpenQualified cenv.g cenv.amap m env.eNameResEnv longPath (length typars)) in    
    let newslotsOK  = (if newslotsOK = NewSlotsOK && is_fsobjmodel_tcref tcref then NewSlotsOK else NoNewSlots) in (* NewslotsOK only on fsobjs *)
    let isLocal     = item_ref_in_this_assembly (cenv.compilingCanonicalFslibModuleType) tcref in
    let isClosed    = (tcaug_of_tcref tcref).tcaug_closed in    
    let isInterface = is_fsobjmodel_interface_tycon (deref_tycon tcref) in    
    let isFirstDef  = (newslotsOK = NewSlotsOK) in (* not partial def, but first def *)
    (* Member definitions are intrinsic (added directly to the type) if:
     * a) For interfaces, only if it is in the original defn.
     *    Augmentations to interfaces via partial type defns will always be extensions, e.g. extension members on interfaces.
     * b) For other types, if the type is local and not yet closed (same file) then permit intrinsic additions. *)
    let declKind = 
        if (isInterface && isFirstDef) || (isLocal && not isClosed && not isInterface) 
        then ModuleOrMemberBinding 
        else ExtensionBinding in
        
    declKind, tcref

let inferTyconKind g (kind,attrs', (* possibleInterfaces,definiteInterfaces, slotsigs, *)  fields,isConcrete,m) =
    let hasClassAttr = fsthing_has_attrib g g.attrib_ClassAttribute attrs' in
    let hasInterfaceAttr = fsthing_has_attrib g g.attrib_InterfaceAttribute attrs' in
    let hasStructAttr = fsthing_has_attrib g g.attrib_StructAttribute attrs' in
    if (hasClassAttr && hasInterfaceAttr) or (hasClassAttr && hasStructAttr) or (hasStructAttr && hasInterfaceAttr) then 
       error(Error("The attributes of this type specify multiple kinds for the type",m));
    
    match kind with 
    | TyconUnspecified ->
        if hasClassAttr then TyconClass        
        else if hasInterfaceAttr then TyconInterface
        else if hasStructAttr then TyconStruct
        else if isConcrete or nonNil(fields) (* or isNil(slotsigs) *)  then TyconClass
        else (* (nonNil(slotsigs) or nonNil(definiteInterfaces) or nonNil(possibleInterfaces)) && isNil(fields) && not(isConcrete) && (possibleInterfaces |> for_all is_interface_typ) then *) TyconInterface
        (* else TyconClass *)
    | k -> 
        if hasClassAttr && (k <> TyconClass) || hasInterfaceAttr && (k <> TyconInterface) || hasStructAttr && (k <> TyconStruct) then 
            error(Error("The kind of the type specified by its attributes does not match the kind implied by its definition",m));
        k
    
let tc_tycon_cores cenv env parent declKind tpenv (tdefs,m,scopem) =
    (* First define the type constructors and the abbreviations, if any. *)
    if verbose then  dprintf2 "--> tc_tycon_cores@%a\n" output_range m;
    let tycons = 
        tdefs |> map (fun (tinfo,srepr,_) ->
            let (ComponentInfo(_,kind,typars, cs,id,doc,preferPostfix, vis,_)) = tinfo in 
            let id = computeTyconName id typars in 
            (* Augmentations of type definitions are allowed within the same file as long as no new type representation or abbreviation is given *)
            checkForDuplicateConcreteType cenv env id.idText id.idRange;
            checkForDuplicateModule cenv env id.idText id.idRange;
            let vis,cpath = computeAccessAndCompPath env declKind id.idRange vis parent in 
            let repr_vis = vis in 
            let typars' = tc_typar_decls cenv env typars in
            (* REVIEW: nested values, types and modules *)
            let lmtyp = (notlazy (empty_mtype AsNamedType)) in 
            let tycon = new_tycon cpath (id.idText,id.idRange) vis repr_vis typars' doc preferPostfix lmtyp in
            tycon) in 

    if verbose then  dprintf2 "--> tc_tycon_cores (publish)@%a\n" output_range m;
    (* Publish the preliminary tycons *)
    let new_tdescs = List.combine tdefs tycons in
    tycons |> iter (publishTypeDefn cenv env);

    if verbose then  dprintf2 "--> tc_tycon_cores (add to env)@%a\n" output_range m;
    
    (* Add them to the environment, though this does not add the fields and *)
    (* constructors. We re-add them to the original environment later on. *)
    (* Don't report them to Visual Studio yet as we don't know if they are well-formed (e.g. free of abbreviation cycles - see bug 952) *)
    let envinner = addLocalTycons scopem tycons env in 

    if verbose then  dprintf2 "--> tc_tycon_cores (publish interfaces)@%a\n" output_range m;

    (* PART 2a ----------------------------------------------*)
    (* Establish the kind of each type constructor *)
    (* Here we run inferTyconKind and record partial information about the kind of the type constructor. *)
    (* This means tycon_objmodel_kind is set, which means is_sealed_typ, is_interface_typ etc. give accurate results. *)
    new_tdescs |> iter (fun ((tinfo,srepr,_), tycon) -> 
        let (ComponentInfo(attrs,_,typars, cs,id, _, _,_,_)) = tinfo in 
        let m = range_of_tycon tycon in 
        let repr = 
            match srepr with 
            | TyconCore_repr_hidden m -> None
            | TyconCore_abbrev(eq,_) -> None
            | TyconCore_funion (constrs,m) -> Some (mk_TFiniteUnionRepr [])
            | TyconCore_asm (s,m) -> Some (TAsmRepr s)
            | TyconCore_recd (fields,m) -> Some (TRecdRepr (mk_rfields_table  []) )
            | TyconCore_general (kind,inherits,slotsigs,fields,isConcrete,isIncrClass,_) ->
                let attrs' = tc_attributes cenv envinner attrTgtTyconDecl attrs in 
                let kind = inferTyconKind cenv.g (kind,attrs',fields,isConcrete,m) in
                let kind = 
                  match kind with
                  | TyconClass               -> TTyconClass
                  | TyconInterface           -> TTyconInterface
                  | TyconDelegate (ty,arity) -> TTyconDelegate (mk_slotsig("Invoke",cenv.g.unit_ty,[],[],[], cenv.g.unit_ty))
                  | TyconStruct              -> TTyconStruct 
                  | TyconUnspecified -> error(InternalError("should have inferred tycon kind",m)) in 

                Some(TFsObjModelRepr { tycon_objmodel_kind=kind; 
                                       fsobjmodel_vslots=[];
                                       fsobjmodel_rfields=mk_rfields_table [] })
            | TyconCore_enum (decls,m) -> 
                let kind = TTyconEnum in 
                Some(TFsObjModelRepr { tycon_objmodel_kind=kind; 
                                       fsobjmodel_vslots=[];
                                       fsobjmodel_rfields= mk_rfields_table [] }) in

        (data_of_tycon tycon).tycon_repr <- repr);


    (* PART 2b ----------------------------------------------*)
    
    let establishSuperTypesAndInterfaceTypes pass = 
        let checkCxs = if (pass = SecondPass) then CheckCxs else NoCheckCxs in 
        let firstPass = (pass = FirstPass) in 
  
        (* Publish the immediately declared interfaces. *)
        let implementsL = 
            new_tdescs |> map (fun ((tinfo,srepr,explicitImplements), tycon) -> 
                let (ComponentInfo(attrs,_,typars, cs,id, _, _,_,_)) = tinfo in 
                let tcref = mk_local_tcref tycon in
                let envinner = add_declared_typars CheckForDuplicateTypars (typars_of_tycon tycon) envinner in 
                let envinner = mk_inner_env_for_tcref envinner tcref false in 
                
                let implements',_ = map_acc_list (map_acc_fst (tc_type_and_recover cenv NoNewTypars checkCxs envinner)) tpenv explicitImplements in 

                (* Review: should skip checking constraints while checking attributes on first pass, though it's hard to imagine when that would matter *)
                let attrs' = tc_attributes cenv envinner attrTgtTyconDecl attrs in 

                if firstPass then 
                    (data_of_tycon tycon).tycon_attribs <- attrs';

                let implements',inherits' = 
                    match srepr with 
                    | TyconCore_general (kind,inherits,slotsigs,fields,isConcrete,isIncrClass,m) ->
                        let kind = inferTyconKind cenv.g (kind,attrs',(* (map fst inherits'),explicitImplements, slotsigs, *) fields,isConcrete,m) in

                        let inherits = map (fun (ty,m,baseIdOpt) -> (ty,m)) inherits in 
                        let inherits' = fst(map_acc_list (map_acc_fst (tc_type_and_recover cenv NoNewTypars checkCxs envinner)) tpenv inherits) in 
                        let implements',inherits' = (if kind = TyconInterface then (implements' @ inherits'),[] else implements',inherits') in
                        implements',inherits' 
                    | TyconCore_enum _ | TyconCore_repr_hidden _ | TyconCore_abbrev _ 
                    
                    | TyconCore_funion _ | TyconCore_asm _ | TyconCore_recd _ -> 
                        (* REVIEW: we could do the IComparable/IStructuralHash interface analysis here. *)
                        (* This would let the type satisfy more recursive IComparable/IStructuralHash constraints *)
                        implements',[] in


                (* Publish interfaces, but only on the first pass, to avoid a duplicate interface check *)
                if firstPass then 
                    iter (fun (ty,m) -> publish_interface cenv (denv_of_tenv env) tcref m false ty) implements';

                attrs',inherits') in

        (* Publish the attributes and supertype  *)
        (combine implementsL new_tdescs) 
        |> iter (fun ((attrs',inherits'),((tinfo,srepr,_), tycon)) -> 
          try 
              if verbose then  dprintf3 "--> tc_tycon_cores (representations of %s)@%a\n" (name_of_tycon tycon) output_range m;
              let (ComponentInfo(attrs,_,typars, cs,id, _, _,_,_)) = tinfo in 
              let id = computeTyconName id typars in 
              let m = id.idRange in 
              let this_tcref = mk_local_tcref tycon in
              let envinner = add_declared_typars CheckForDuplicateTypars (typars_of_tycon tycon) envinner in 
              let envinner = mk_inner_env_for_tcref envinner this_tcref false in 
              let super = 
                  match srepr with 
                  | TyconCore_repr_hidden _ -> None
                  | TyconCore_abbrev(eq,_) -> None
                  | TyconCore_funion (constrs,_) -> None
                  | TyconCore_asm (s,_) -> None
                  | TyconCore_recd (fields,_) -> None
                  | TyconCore_general (kind,_,slotsigs,fields,isConcrete,isIncrClass,_) ->
                      let kind = inferTyconKind cenv.g (kind,attrs',(* (map fst inherits'),implements', slotsigs, *) fields,isConcrete,m) in
                                           
                      begin match inherits' with 
                      | [] -> 
                          begin match kind with 
                          | TyconStruct -> Some(mk_system_mono_typ cenv.g.sysCcu "ValueType")
                          | TyconDelegate _ -> Some(mk_system_mono_typ cenv.g.sysCcu "MulticastDelegate" )
                          | TyconClass | TyconInterface -> None
                          | TyconUnspecified -> error(InternalError("should have inferred tycon kind",m)) 
                          end
                      | [(ty,m)] -> 
                          if not firstPass && kind <> TyconClass then 
                            errorR (Error("Structs, interfaces, enums and delegates may not inherit from other types",m)); 
                          Some ty 
                      | _ -> error(Error("Types may not inherit from multiple concrete types",m))
                      end
                  | TyconCore_enum (decls,_) -> 
                      Some(mk_system_mono_typ cenv.g.sysCcu "Enum")  in

              let tcaug = tcaug_of_tycon tycon in
              tcaug.tcaug_super <- super
              
           with e -> errorRecoveryPoint e) in 
    

    establishSuperTypesAndInterfaceTypes FirstPass;

    (* PART 3 ----------------------------------------------*)


    (* Publish the explicit constraints. *)
    if verbose then  dprintf2 "--> tc_tycon_cores (do explicit constraints)@%a\n" output_range m;
    new_tdescs |> iter (fun ((tinfo,srepr,_), tycon) -> 
        let (ComponentInfo(_,kind,_, wcs,_,_,_, _,im)) = tinfo in 
        assert_isTycon kind im;
        let envinner = add_declared_typars CheckForDuplicateTypars (typars_of_tycon tycon) envinner in 
        let this_tcref = mk_local_tcref tycon in
        let envinner = mk_inner_env_for_tcref envinner this_tcref false in 
        try tc_typar_constraints cenv NoNewTypars CheckCxs envinner tpenv  wcs |> ignore
        with e -> errorRecoveryPoint e);

    (* no more constraints allowed on declared typars *)
    tycons |> iter (typars_of_tycon >> iter (setTyparRigid cenv.g (denv_of_tenv env) m));
    

    (* OK, now recheck the super/interface types checking constraints *)
    establishSuperTypesAndInterfaceTypes SecondPass;


    (* PART 4 ----------------------------------------------*)
    (* Now do the representations *)

    if verbose then  dprintf2 "--> tc_tycon_cores (do representations)@%a\n" output_range m;
    (* Now do the representations *)
    let basevopts = 
      new_tdescs |> map (fun ((tinfo,srepr,_), tycon) -> 
      try 
        if verbose then  dprintf3 "--> tc_tycon_cores (representations of %s)@%a\n" (name_of_tycon tycon) output_range m;
        let (ComponentInfo(attrs,_,typars, cs,id, _, _,_,_)) = tinfo in 
        let id = computeTyconName id typars in 
        let m = range_of_tycon tycon in 
        let this_tcref = mk_local_tcref tycon in
        let thisTy = snd (generalize_tcref this_tcref) in 
        let attrs' = tc_attributes cenv envinner attrTgtTyconDecl attrs in 


        let hasAbstractAttr = fsthing_has_attrib cenv.g cenv.g.attrib_AbstractClassAttribute attrs' in
        let hasSealedAttr = fsthing_bool_attrib cenv.g cenv.g.attrib_SealedAttribute attrs' in

        if hasAbstractAttr then begin
            (tcaug_of_tycon tycon).tcaug_abstract <- true;
        end;

        (data_of_tycon tycon).tycon_attribs <- attrs';

        let envinner = add_declared_typars CheckForDuplicateTypars (typars_of_tycon tycon) envinner in 
        let envinner = mk_inner_env_for_tcref envinner this_tcref false in 
        let ty_eq',repr',baseVarOpt = 
          begin match srepr with 
          | TyconCore_repr_hidden _ -> 
              if isSome hasSealedAttr then errorR (Error("Types whose representations are hidden by signatures are always sealed",m));
              if hasAbstractAttr then errorR (Error("Only classes may be given the 'AbstractClass' attribute",m));
              None,None,None
          | TyconCore_abbrev(eq,_) ->
              if isSome hasSealedAttr then errorR (Error("Abbreviated types may not be given the 'Sealed' attribute",m));
              if hasAbstractAttr then errorR (Error("Only classes may be given the 'AbstractClass' attribute",m));
              begin match eq with
              (* This gross case deals with "type x = A" *)
              (* In F# this only defines a new type if A is not in scope *)
              (* as a type constructor, or if the form type A = A is used. *)
              (* "type x = | A" can always be used instead. *)
              | Type_app ([tcn],[],_) when 
                isNil (lookupTypeNameInEnvNoArity tcn.idText envinner.eNameResEnv) ||
                id.idText = tcn.idText -> 
                  None, Some (mk_TFiniteUnionRepr [ new_uconstr tcn tcn.idText [] thisTy [] emptyXMLDoc (access_of_tycon tycon) ]),None
              | _ -> 
                  let ty_eq',_ = tc_type_and_recover cenv NoNewTypars CheckCxs envinner tpenv eq in 

                  if verbose then  dprintf2 "--> tc_tycon_cores (TyconCore_abbrev)@%a\n" output_range m;
                  Some ty_eq',None,None
              end
          | TyconCore_funion (constrs,_) -> 
              if isSome hasSealedAttr then errorR (Error("Discriminated union types are always sealed",m));
              if hasAbstractAttr then errorR (Error("Only classes may be given the 'AbstractClass' attribute",m));
              None,Some (mk_TFiniteUnionRepr (tc_constr_decls cenv envinner parent declKind thisTy tpenv constrs) ),None
          | TyconCore_asm (s,_) -> 
              if isSome hasSealedAttr then errorR (Error("Assembly code types are always sealed",m));
              if hasAbstractAttr then errorR (Error("Only classes may be given the 'AbstractClass' attribute",m));
              None,Some (TAsmRepr s),None
          | TyconCore_recd (fields,_) -> 
              let fields' = tc_named_field_decls cenv envinner parent declKind false tpenv fields in 
              if isSome hasSealedAttr then errorR (Error("Record types are always sealed",m));
              if hasAbstractAttr then errorR (Error("Only classes may be given the 'AbstractClass' attribute",m));
              None,Some (TRecdRepr (mk_rfields_table fields'))  ,None
          | TyconCore_general (kind,inherits,slotsigs,fields,isConcrete,isIncrClass,_) ->
              let fields' = tc_named_field_decls cenv envinner parent declKind isIncrClass tpenv fields in 
              let vslotsl = 
                slotsigs |> map (fun (valSpfn,memFlags) -> 
                  checkMemberFlags cenv.g None NewSlotsOK OverridesOK memFlags m;
                  fst (tc_and_publish_val_spec cenv envinner (tyconContainerInfo(parent,this_tcref)) ModuleOrMemberBinding (Some(memFlags)) tpenv valSpfn)) in

              let super' = (tcaug_of_tycon tycon).tcaug_super in
              let kind = inferTyconKind cenv.g (kind,attrs',(* Option.to_list super',implements', slotsigs, *) fields,isConcrete,m) in

              (* Note: for a mutually recursive set we can't check this condition until "is_sealed_typ" and "is_class_typ" give reliable results. *)
              super' |> Option.iter (fun ty -> 
                  let m = match inherits with | [] -> m | ((_,m,_) :: _) -> m in 
                  if is_sealed_typ cenv.g ty then errorR(Error("Cannot inherit a sealed type",m));
                  if not (is_class_typ ty) then errorR(Error("Cannot inherit a non-class type",m)));

              if nonNil slotsigs && (match kind with 
                                     | TyconStruct -> true 
                                     | TyconDelegate _ | TyconClass | TyconInterface -> false
                                     | TyconUnspecified -> error(InternalError("should have inferred tycon kind",m))) then 
                errorR (Error("Struct types may not not contain abstract members",m)); 
             
              begin match kind,fields' with 
                | (TyconInterface | TyconDelegate _),(rf :: _) -> errorR (Error("Interface types may not contain fields",range_of_rfield rf))
                | (TyconInterface | TyconDelegate _ | TyconClass | TyconStruct),_ -> ()
                | TyconUnspecified,_ -> error(InternalError("should have inferred tycon kind",m)) 
              end ;
              begin match kind with 
                | (TyconStruct | TyconDelegate _) when isSome hasSealedAttr -> errorR (Error("Struct and delegate types are always sealed",m))
                | (TyconInterface) when isSome hasSealedAttr -> errorR (Error("Interface types may never be sealed",m))
                | _ -> ()
              end ;
              begin match kind with 
                | (TyconStruct | TyconDelegate _ | TyconInterface) when hasAbstractAttr -> 
                      errorR (Error("Only classes may be given the 'AbstractClass' attribute",m));
                | TyconClass -> ()
                | _ -> ()
              end ;
              let kind = 
                match kind with
                | TyconClass -> TTyconClass
                | TyconInterface -> TTyconInterface
                | TyconDelegate (ty,arity) -> 
                    let ty',_ = tc_type_and_recover cenv NoNewTypars CheckCxs envinner tpenv ty in 
                    let argtysl,rty = dest_top_tau_type (arity |> translateTopValSynData m (tc_attributes cenv env)  |> translatePartialArity [] |> TopValData.getArgInfos) ty' in 
                    if length argtysl <> 1 then error(Error("Delegate specifications must be of the form 'typ -> typ'",m));
                    let fparams = map mk_slotparam (map fst (concat argtysl)) in 
                    TTyconDelegate (mk_slotsig("Invoke",thisTy,typars_of_tcref this_tcref,[],fparams, rty))
                | TyconStruct -> 
                    TTyconStruct 
                | TyconUnspecified -> error(InternalError("should have inferred tycon kind",m)) in 

              let baseIdOpt = 
                  match srepr with 
                  | TyconCore_repr_hidden _ -> None
                  | TyconCore_abbrev _ -> None
                  | TyconCore_funion _ -> None
                  | TyconCore_asm _ -> None
                  | TyconCore_recd _ -> None
                  | TyconCore_enum _ -> None
                  | TyconCore_general (_,inherits,slotsigs,fields,isConcrete,isIncrClass,m) ->
                      match inherits with 
                      | [] -> None
                      | ((ty,m,baseIdOpt) :: _) -> baseIdOpt in
                  
              let baseVarOpt = mkAndPublishBaseVal cenv env baseIdOpt (super_of_tycon cenv.g tycon) in 
              None, Some(TFsObjModelRepr { tycon_objmodel_kind=kind; 
                                           fsobjmodel_vslots=map mk_local_vref (concat vslotsl);
                                           fsobjmodel_rfields=mk_rfields_table fields'}), baseVarOpt
          | TyconCore_enum (decls,m) -> 
              let field_ty,fields' = tc_enum_decls cenv env parent declKind thisTy decls in 
              let kind = TTyconEnum in 
              let vfld = new_rfield false None (ident("value__",m))  field_ty false [] [] emptyXMLDoc taccessPublic true in 
              
              if not (gen_mem (type_equiv cenv.g) field_ty [ cenv.g.int32_ty; cenv.g.int16_ty; cenv.g.sbyte_ty; cenv.g.int64_ty; cenv.g.char_ty; cenv.g.float32_ty; cenv.g.float_ty; cenv.g.bool_ty; cenv.g.uint32_ty; cenv.g.uint16_ty; cenv.g.byte_ty; cenv.g.uint64_ty ]) then 
                  errorR(Error("Literal enumerations must have type int, uint, int16, uint16, int64, uint64, byte, sbyte, char, float or float32",m));
                  
              None, Some(TFsObjModelRepr { tycon_objmodel_kind=kind; 
                                           fsobjmodel_vslots=[];
                                           fsobjmodel_rfields= mk_rfields_table (vfld :: fields') }), None
          end in 
        
        if verbose then  dprintf2 "--> tc_tycon_cores (adjust representations)@%a\n" output_range m;
        (data_of_tycon tycon).tycon_abbrev <- ty_eq';
        (data_of_tycon tycon).tycon_repr <- repr';
        baseVarOpt
       with e -> errorRecoveryPoint e; None) in 


    (* PART 5 ----------------------------------------------*)
    (* check for cyclic abbreviations *)
    if verbose then  dprintf2 "--> tc_tycon_cores (check for cycles)@%a\n" output_range m;
    let edges tycon =

        let insert tcref acc = 
            let tycon' = deref_tycon tcref in 
            if gen_mem (=!=) tycon' tycons  then (tycon,tycon') ::acc else acc in

        let rec acc_in_inline_tyrep ty acc  = 
            match strip_tpeqns ty with 
            | TType_app (tc ,tinst) when is_struct_tycon tycon  ->
                let tycon' = (deref_tycon tc) in 
                if gen_mem (=!=) tycon' tycons  then 
                    (tycon,tycon') ::acc 
                else if is_abbrev_tcref tc  then
                    acc_in_inline_tyrep (reduce_tcref_abbrev tc tinst) acc
                else
                    List.fold_right (actual_typ_of_rfield tycon' tinst >> acc_in_inline_tyrep) (instance_rfields_of_tycon tycon') acc
            | _ -> acc in

        let rec acc_in_abbrev_type ty acc  = 
            match strip_tpeqns ty with 
            | TType_tuple l -> 
                acc_in_abbrev_types l acc
            
            (* REVIEW: this algorithm is not yet correct for all cases *)
            | TType_app (tc,tinst) -> 
                let tycon' = (deref_tycon tc) in 
                let acc = acc_in_abbrev_types tinst  acc in
                (* Record immediate recursive references *)
                if gen_mem (=!=) tycon' tycons  then 
                    (tycon,tycon') ::acc 
                (* Expand the representation of abbreviations *)
                else if is_abbrev_tcref tc  then
                    acc_in_abbrev_type (reduce_tcref_abbrev tc tinst) acc
                (* Otherwise H<inst> - explore the instantiation. *)
                else 
                    acc

            | TType_fun (d,r) -> 
                acc_in_abbrev_type d (acc_in_abbrev_type r acc)
            
            | TType_var r -> acc
            
            | TType_forall (tps,r) -> acc_in_abbrev_type r acc
            
            | TType_modul_bindings | TType_unknown -> failwith "acc_in_abbrev_type: naked unknown"

        and acc_in_abbrev_types tys acc = 
            fold_right acc_in_abbrev_type tys acc in
            
        let acc = [] in
        let acc = 
            match abbrev_of_tycon tycon with 
            | None -> acc
            | Some ty -> 
                if not cenv.isSig && not cenv.haveSig && (access_of_tycon tycon <> taccessPublic || repr_access_of_tycon tycon <> taccessPublic) then 
                   errorR(Error("Type abbreviations must be public. If you want to use a private type abbreviation you must use an explicit signature",range_of_tycon tycon));
                acc_in_abbrev_type ty acc in

        let acc = 
            if is_struct_tycon tycon then
                let this_tcref = mk_local_tcref tycon in
                let tinst,ty = generalize_tcref this_tcref in 
                List.fold_right (actual_typ_of_rfield tycon tinst >> acc_in_inline_tyrep) (instance_rfields_of_tycon tycon) acc
            else
                acc in

        let acc =
            (* Note: only the nominal type counts *)
            let super = super_of_tycon cenv.g tycon in
            insert (tcref_of_stripped_typ super) acc in
        let acc =
            (* Note: only the nominal type counts *)
            let interfaces = implements_of_tycon cenv.g tycon in
            List.fold_right (tcref_of_stripped_typ  >> insert) interfaces acc in
        acc in            

    let graph = mk_graph (stamp_of_tycon,int_ord) tycons (map (fun (tc1,tc2) -> (stamp_of_tycon tc1, stamp_of_tycon tc2)) (mapConcat edges tycons)) in 
    iter_cycles (fun path -> error(Error("This type definition involves an immediate cyclic reference through an abbreviation, struct field or inheritance relation",range_of_tycon (hd path)))) graph;
    
    (* PART 6 ----------------------------------------------*)

    if verbose then  dprintf2 "--> tc_tycon_cores (check explicit constructors)@%a\n" output_range m;
    (* Check explicit constructor type specifications are valid *)
    new_tdescs |> iter (fun ((tinfo,srepr,implements), tycon) -> 
        let this_tcref = mk_local_tcref tycon in
        let thisTyInst,thisTy = generalize_tcref this_tcref in 
        uconstrs_array_of_tycon tycon |> Array.iter (fun uc -> 
            let ty1 = rty_of_uctyp (ucref_of_uconstr this_tcref uc) thisTyInst in 
            if not (type_equiv cenv.g ty1 thisTy) then error(Error("Return types of constructors must be identical to the constructed type up to abbreviations",m))
        ));

    (* PART 7 ----------------------------------------------*)

    if verbose then  dprintf2 "--> tc_tycon_cores (augment)@%a\n" output_range m;
    (* Augment types with references to values that implement the pre-baked semantics of the type *)
    let binds = tycons |> mapConcat (add_hash_compare_bindings cenv env) in 

    (* PART 8 ----------------------------------------------*)

    if verbose then  dprintf2 "--> tc_tycon_cores@%a (add to environment, second time)" output_range m;
    (* Add the tycons again - this will add the constructors and fields. *)
    let env = addLocalTyconsAndReport scopem tycons env in 
    if verbose then  dprintf2 "<-- tc_tycon_cores@%a" output_range m;
    (binds,tycons,env,basevopts)


(*-------------------------------------------------------------------------
!* Bind type definitions
 *------------------------------------------------------------------------- *)

let is_binding          = function ClassMemberDefn_member_binding _          -> true | _ -> false
let is_implicit_ctor    = function ClassMemberDefn_implicit_ctor _    -> true | _ -> false
let is_implicit_inherit = function ClassMemberDefn_implicit_inherit _ -> true | _ -> false
let is_bindings         = function ClassMemberDefn_let_bindings _         -> true | _ -> false
let is_slotsig          = function ClassMemberDefn_slotsig _          -> true | _ -> false
let is_interface        = function ClassMemberDefn_interface _        -> true | _ -> false
let is_inherit          = function ClassMemberDefn_inherit _          -> true | _ -> false
let is_field            = function ClassMemberDefn_field _            -> true | _ -> false
let is_tycon            = function ClassMemberDefn_tycon _            -> true | _ -> false

(* French logical ops work over predicates *)
let ou p q x = p x || q x
let ouL ps = List.fold_left ou (fun _ -> false) ps
let non p x = not (p x)

(* Accepted forms:
 *
 * Implicit Construction:
 *   implicit_ctor
 *   optional implicit_inherit
 *   multiple bindings
 *   multiple member-binding(includes-overrides) or abstract-slot-declaration or interface-bindings
 *
 * Classic construction:
 *   multiple (binding or slotsig or field or interface or inherit).
 *   i.e. not local-bindings, implicit ctor or implicit inherit (or tycon?).
 *   atMostOne inherit.
 *)
let checkMembersForm ds = 
    match ds with
    | d::ds when is_implicit_ctor d ->
        (* Implicit construction *)
        let ds = match ds with
                 | d::ds when is_implicit_inherit d -> ds  (* skip inherit call if it comes next *)
                 | ds -> ds in
        let localbindings ,ds = list_take_until (non is_bindings) ds in
        let memberbindings,ds = list_take_until (non (ouL [is_binding;is_slotsig;is_interface])) ds in
        begin match ds with
         | ClassMemberDefn_member_binding (_,m)              :: _ -> errorR(InternalError("list_take_until is wrong, have binding",m))
         | ClassMemberDefn_slotsig (_,_,m)            :: _ -> errorR(InternalError("list_take_until is wrong, have slotsig",m))
         | ClassMemberDefn_interface (_,_,m)          :: _ -> errorR(InternalError("list_take_until is wrong, have interface",m))
         | ClassMemberDefn_implicit_ctor (_,_,m)      :: _ -> errorR(InternalError("implicit class construction with two implicit constructions",m))
         | ClassMemberDefn_implicit_inherit (_,_,_,m) :: _ -> errorR(Error("Type definitions using implicit construction may only have one 'inherit' specification and it must be the first declaration",m))
         | ClassMemberDefn_let_bindings (_,_,_,m)     :: _ -> errorR(Error("Type definitions using implicit construction must have local let/do-bindings preceding member and interface definitions",m))
         | ClassMemberDefn_inherit (_,_,m)            :: _ -> errorR(Error("This 'inherit' declaration specifies the inherited type but no arguments. Consider supplying arguments, e.g. 'inherit BaseType(args)'",m))
(*         | ClassMemberDefn_field (_,m)              :: _ -> errorR(Error("Class definitions using both an implicit construction sequence and field specifications are not yet implemented",m)) *)
         | ClassMemberDefn_tycon (_,_,m)              :: _ -> errorR(Error("Types may not contain nested type definitions",m))
         | _ -> ()
        end
    | ds ->
        (* Classic class construction *)
        let memberbindings,ds = list_take_until (non (ouL [is_binding;is_slotsig;is_interface;is_inherit;is_field;is_tycon])) ds in
        begin match ds with
         | ClassMemberDefn_member_binding (_,m)              :: _ -> errorR(InternalError("checkMembersForm: list_take_until is wrong",m))
         | ClassMemberDefn_implicit_ctor (_,_,m)      :: _ -> errorR(InternalError("checkMembersForm: implicit ctor line should be first",m))
         | ClassMemberDefn_implicit_inherit (_,_,_,m) :: _ -> errorR(Error("This 'inherit' construction call is not part of an implicit construction sequence. Only the inherited type should be specified at this point. Calls to the inherited constructor should be placed inside the object intialization expression of your object constructor. Alternatively use an implicit construction sequence by modifying the type declaration to include arguments, e.g. 'type X(args) = ...'",m))
         | ClassMemberDefn_let_bindings (_,_,_,m)           :: _ -> errorR(Error("'let' and 'do' bindings are not permitted in class definitions unless an implicit construction sequence is used. You can use an implicit construction sequence by modifying the type declaration to include arguments, e.g. 'type X(args) = ...'",m))
         | ClassMemberDefn_slotsig (_,_,m)            :: _ 
         | ClassMemberDefn_interface (_,_,m)          :: _ 
         | ClassMemberDefn_inherit (_,_,m)            :: _ 
         | ClassMemberDefn_field (_,m)                :: _ 
         | ClassMemberDefn_tycon (_,_,m)              :: _ -> errorR(InternalError("checkMembersForm: list_take_until is wrong",m))
         | _ -> ()
        end
                 

(* Parallel split_tyconSpfn/split_tyconDefn] *)    
(* Separates the definition into core (shape) and body.
 * core = tinfo,simpleRepr,interfaceTypes
 *        where simpleRepr can contain inherit type, declared fields and virtual slots.
 * body = members
 *        where members contain methods/overrides, also implicit ctor, inheritCall and local definitions.
 *------
 * The tinfos arg are the enclosing types when processing nested types...
 * The tinfos arg is not currently used... just stacked up.
 *)
let rec split_tyconDefn tinfos (TyconDefn(tinfo,trepr,withclass,m)) =
    let implements1 = chooseList (function ClassMemberDefn_interface (f,_,m) -> Some(f,m) | _ -> None) withclass in 
    match trepr with
    | TyconDefnRepr_class(kind,cspec,m) ->
        checkMembersForm cspec;
        let fields      = cspec |> chooseList (function ClassMemberDefn_field (f,_) -> Some(f) | _ -> None) in 
        let implements2 = cspec |> chooseList (function ClassMemberDefn_interface (ty,_,m) -> Some(ty,m) | _ -> None) in 
        let inherits    = cspec |> chooseList (function ClassMemberDefn_inherit          (ty,id,m)     -> Some(ty,m,id)
                                                      | ClassMemberDefn_implicit_inherit (ty,arg,id,m) -> Some(ty,m,id)
                                                      | _ -> None) in
        let tycons      = cspec |> chooseList (function ClassMemberDefn_tycon (x,_,_) -> Some(x) | _ -> None) in 
        let slotsigs    = cspec |> chooseList (function ClassMemberDefn_slotsig (x,y,_) -> Some(x,y) | _ -> None) in 
        let members     = cspec |> filter (function | ClassMemberDefn_interface _
                                                    | ClassMemberDefn_member_binding _ 
                                                    | ClassMemberDefn_let_bindings _
                                                    | ClassMemberDefn_implicit_ctor _
                                                    | ClassMemberDefn_open _
                                                    | ClassMemberDefn_implicit_inherit _ -> true
                                                    | ClassMemberDefn_tycon  (_,_,m)  -> error(Error("Types may not contain nested type definitions",m)); false
                                                    | ClassMemberDefn_field _   -> false (* covered above *)
                                                    | ClassMemberDefn_inherit _ -> false (* covered above *)
                                                    | ClassMemberDefn_slotsig _ -> false (* covered above *)
                                                    ) in
        let a,b = split_tyconDefns (tinfos @ [tinfo]) tycons in

        let isConcrete = 
            members |> exists (function 
                | ClassMemberDefn_member_binding(Binding(_,_,_,_,_,_,Some(memFlags,_,_),_,_,_),_) -> not memFlags.memFlagsAbstract 
                | ClassMemberDefn_interface (_,defOpt,_) -> isSome defOpt
                | ClassMemberDefn_let_bindings _ -> true
                | ClassMemberDefn_implicit_ctor _ -> true
                | ClassMemberDefn_implicit_inherit _ -> true
                | _ -> false) in 

        let isIncrClass = 
            members |> exists (function 
                | ClassMemberDefn_implicit_ctor _ -> true
                | _ -> false) in 
                
        let core = (tinfo, TyconCore_general(kind,inherits,slotsigs,fields,isConcrete,isIncrClass,m), implements2@implements1) in
        core :: a,
        members :: b
    | TyconDefnRepr_simple(r,_) -> [(tinfo,r,implements1)],[ [] ]

and split_tyconDefns tinfos tycons = 
    let a,b = split (map (split_tyconDefn tinfos) tycons) in
    List.concat a, List.concat b 

let prepare_tycon_member_defns newslotsOK cenv env  (tinfo,baseVarOpt,members,m) =
    let (ComponentInfo(_,kind,typars, _,longPath, _, _, _,im)) = tinfo in 
    assert_isTycon kind im;
    let declKind,tcref = computeTyconDeclKind newslotsOK cenv env m typars longPath in

    let newslotsOK = (if newslotsOK = NewSlotsOK && is_fsobjmodel_tcref tcref then NewSlotsOK else NoNewSlots) in 
    if nonNil(members) && is_abbrev_tcref tcref then errorR(Error("Type abbreviations may not have augmentations",m));
    if verboseCC then dprintf1 "prepare_tycon_member_defns: #members=%d\n" (List.length members);
    TyconMemberData(declKind,tcref,baseVarOpt,members,m,newslotsOK)

(*-------------------------------------------------------------------------
!* Bind type definitions - main
 *------------------------------------------------------------------------- *)

let tc_tycon_defs cenv env parent declKind tpenv ((tdefs: tyconDefn list),m,scopem) =
    let cores,membersl = split_tyconDefns [] tdefs in 
    let binds,tycons,env,basevopts = tc_tycon_cores cenv env parent declKind tpenv (cores,m,scopem) in 
    let augments = 
        map (fun (TyconDefn(tinfo,_,extraMembers,m),(baseVarOpt,members)) -> 
               prepare_tycon_member_defns NewSlotsOK cenv env (tinfo,baseVarOpt,members@extraMembers,m))
          (combine tdefs (combine basevopts membersl)) in 

    let val_exprfs,env = tc_tycon_member_defns OverridesOK cenv env parent m scopem augments in

    (* Augment types with references to values that implement the pre-baked semantics of the type *)
    let binds3 = tycons |> mapConcat (add_equals_bindings_late cenv env) in

    (binds @ val_exprfs @ binds3),tycons,env  


(*-------------------------------------------------------------------------
!* Bind type specifications
 *------------------------------------------------------------------------- *)

(* Parallel split_tycon[Spfn/Defn] *)
let rec split_tyconSpfn tinfos (TyconSpfn(tinfo,trepr,withclass,m)) = 

    let isExtension = 
        let (ComponentInfo(_,_,typars, _,longPath, _, _, _,m)) = tinfo in 
        List.length longPath > 1 in

    let implements1 = chooseList (function ClassMemberSpfn_interface (f,m) -> Some(f,m) | _ -> None) withclass in 
    match trepr with
    | TyconSpfnRepr_class(kind,cspec,m) -> 
        let fields      = cspec |> chooseList (function ClassMemberSpfn_field (f,_) -> Some(f) | _ -> None) in 
        let implements2 = cspec |> chooseList (function ClassMemberSpfn_interface (ty,m) -> Some(ty,m) | _ -> None) in 
        let inherits    = cspec |> chooseList (function ClassMemberSpfn_inherit (ty,_) -> Some(ty,m,None) | _ -> None) in 
        let tycons      = cspec |> chooseList (function ClassMemberSpfn_tycon (x,_) -> Some(x) | _ -> None) in 
        let slotsigs    = cspec |> chooseList (function ClassMemberSpfn_binding (v,fl,_) when fl.memFlagsVirtual || fl.memFlagsAbstract -> Some(v,fl) | _ -> None) in 
        let members     = cspec |> filter (function   | ClassMemberSpfn_interface _ -> true
                                                      | ClassMemberSpfn_binding (_,memFlags,_) when not memFlags.memFlagsAbstract -> true
                                                      | ClassMemberSpfn_tycon  (_,m) -> error(Error("Types may not contain nested type definitions",m)); false
                                                      | _ -> false) in 
        let isConcrete = 
            members |> exists (function 
                | ClassMemberSpfn_binding (_,memFlags,_) -> not memFlags.memFlagsAbstract
                | _ -> false) in 

        let a,b = split_tyconSpfns (tinfos @ [tinfo]) tycons in
        (if isExtension then [] else [ (tinfo, TyconCore_general(kind,inherits,slotsigs,fields,isConcrete,false,m),implements2@implements1) ]) @ a,
        [ members ] @ b
    | TyconSpfnRepr_simple(r,_) -> 
        (if isExtension then [] else [(tinfo,r,implements1)]),[ [] ] 

and split_tyconSpfns tinfos tycons = 
    let a,b = split (map (split_tyconSpfn tinfos) tycons) in
    List.concat a, List.concat b 

let tc_tycon_class_spfn newslotsOK overridesOK cenv env parent tpenv tycons taugments =
    map_acc_list 
      (fun tpenv (tinfo,members) -> 
        let (ComponentInfo(_,_,typars, _,longPath, _, _, _,m)) = tinfo in 
        let declKind,tcref = computeTyconDeclKind newslotsOK cenv env m typars longPath in

        (* TOOD: EXTENSION MEMBERS: REVIEW TREATMENT OF TYPARS *)
        let envinner = add_declared_typars CheckForDuplicateTypars (typars_of_tcref tcref) env in 
        let envinner = mk_inner_env_for_tcref envinner tcref (declKind = ExtensionBinding) in 

        tc_class_spec newslotsOK overridesOK cenv envinner (tyconContainerInfo(parent,tcref)) declKind tcref tpenv members)
      tpenv 
      (List.combine tycons taugments) 

let tc_tycon_specs cenv env parent declKind tpenv ((tspecs:tyconSpfn list),m,scopem) =
    let tinfos = tspecs |> map (fun (TyconSpfn(tinfo,_,_,_)) -> tinfo) in
    let cores,membersl = split_tyconSpfns [] tspecs in 
    let binds,tycons,env,_ = tc_tycon_cores cenv env parent declKind tpenv (cores,m,scopem) in 
    (* REVIEW: these should just all be done together *)
    (* Now do the rigid class augmentations  *)
    let vals1,tpenv = tc_tycon_class_spfn NewSlotsOK OverridesOK cenv env parent tpenv tinfos membersl in 
    (* Now do the optional partial class augmentations *)
    let vals2,tpenv = tc_tycon_class_spfn NoNewSlots OverridesOK cenv env parent tpenv tinfos (map (fun (TyconSpfn(_,_,members,_)) -> members) tspecs) in 
    env

(*-------------------------------------------------------------------------
!* Bind module types
 *------------------------------------------------------------------------- *)

let rec tc_signature_spec cenv parentModule endm (env:tcEnv) e =
    let parent = Parent(parentModule) in
    let containerInfo = moduleContainerInfo(parentModule) in 
    try 
        match e with 
        | Spec_exn (edef,m) ->
            let scopem = union_ranges (end_range_of_range m) endm in 
            let _,_,_,env = tc_exn_spec cenv env parent ModuleOrMemberBinding emptyTpenv (edef,scopem) in 
            (* todo: incrementaly addd to a module type *)
            env

        | Spec_tycon (tspecs,m) -> 
            let scopem = union_ranges m endm in 
            let env = tc_tycon_specs cenv env parent ModuleOrMemberBinding emptyTpenv (tspecs,m,scopem) in
            (* todo: incrementaly add to a module type, rather than using mutation *)
            env 

        | Spec_open (mp,m) -> 
            let scopem = union_ranges (end_range_of_range m) endm in 
            tc_open_namespace cenv.autoModuleResolver m scopem env mp

        | Spec_val (vspec,m) -> 
            let idvs,_ = tc_and_publish_val_spec cenv env containerInfo ModuleOrMemberBinding None emptyTpenv vspec in
            let scopem = union_ranges m endm in 
            list_fold_right (addLocalVal scopem) idvs env

        | Spec_module(ComponentInfo(attribs,kind,parms, constraints,longPath,xml,preferPostfix,vis,im),mdefs,m) ->
            let id = computeModuleName longPath in 
            let isModule =  compute_isModule kind parms constraints im in
            let vis,_ = computeAccessAndCompPath env ModuleOrMemberBinding im vis parent in 
            let mspec,_ = tc_signature_module cenv env (id,isModule,mdefs,xml,attribs,vis,m) in
            let scopem = union_ranges m endm in 
            publishModuleDefn cenv env mspec; 
            add_local_submodul scopem env (text_of_id id) mspec
            
        | Spec_module_abbrev (id,p,m) -> 
            let mvvs = forceRaise (tc_namespace_lid OpenQualified env.eNameResEnv p) in
            let scopem = union_ranges m endm in 
            add_modul_abbrev scopem id (map p23 mvvs) env

        | Spec_hash _ -> env

    with e -> errorRecoveryPoint e; env

and tc_signature_specs cenv parentModule endm env defs = 
    fold_left (tc_signature_spec cenv parentModule endm) env defs

and compute_istype g isModule attribs = 
    if not isModule then Namespace 
    else if moduleNameIsMangled g attribs then AsMangledNamedType "Module"
    else AsNamedType

and tc_signature_module cenv env (id,isModule,defs,xml,attribs,vis,m) =
    if verbose then  dprintf0 "tc_signature_module...\n";
    let attribs = tc_attributes cenv env attrTgtModuleDecl attribs in 
    let istype = compute_istype cenv.g isModule attribs in
    if isModule then checkForDuplicateConcreteType cenv env (adjust_module_name istype id.idText) id.idRange;
    if isModule then checkForDuplicateModule cenv env id.idText id.idRange;

    (* Now typecheck the signature, accumulating and then recording the submodule description. *)
    let mspec = new_mspec  (Some(curr_cpath env)) vis id xml attribs (notlazy (empty_mtype istype))  in
    let innerParent = mk_local_modref mspec in
    
    let mtyp,envAtEnd = tc_signature_specs_core cenv innerParent env (id,istype,defs,m,xml) in
    (data_of_tycon mspec).tycon_modul_contents <- notlazy mtyp; 
    
    mspec, envAtEnd

and tc_signature_specs_core cenv parentModule env (id,istype,defs,m,xml) =

    let endm = end_range_of_range m in (* use end of range for errors *)

    if verbose then  dprintf2 "--> tc_signature_specs_core, endm = %a\n" output_range endm;
    (* Create the module type that will hold the results of type checking.... *)
    let envinner,mtypeAcc = mk_inner_env env id istype in 

    (* Ensure the modul_of_nlpath call in updateAccModuleType succeeds *)
    if cenv.compilingCanonicalFslibModuleType then 
        ensure_fslib_has_submodul_at cenv.topCcu envinner.ePath (curr_cpath envinner) xml;

    (* Now typecheck the signature, using mutation to fill in the submodule description. *)
    let envAtEnd = tc_signature_specs cenv parentModule endm envinner defs in 
    
    (* mtypeAcc has now accumulated the module type *)
    !mtypeAcc, envAtEnd

(*-------------------------------------------------------------------------
!* Bind definitions within modules
 *------------------------------------------------------------------------- *)

let mk_binds m binds (e,ty) = (bind_letrec binds m e),ty

let rec tc_def cenv parentModule scopem env e : ((modul_def list -> modul_def list) * _) * tcEnv =
    let tpenv = emptyTpenv in 
    let parent = Parent(parentModule) in
    let containerInfo = moduleContainerInfo(parentModule) in 
    try 
      if verbose then  dprintf2 "--> tc_def@%a\n" output_range (range_of_syndecl e);
      match e with 

      | Def_module_abbrev (id,p,m) -> 
          let mvvs = forceRaise (tc_namespace_lid OpenQualified env.eNameResEnv p) in
          ((fun e -> e), []), add_modul_abbrev scopem id (map p23 mvvs) env

      | Def_exn (edef,m) -> 
          let binds,decl,env = tc_exn_def cenv env parent ModuleOrMemberBinding tpenv (edef,scopem) in
          ((fun e -> TMDefRec([decl],binds,m) :: e),[]), env

      | Def_tycon (tdefs,m) -> 
          let scopem = union_ranges m scopem in 
          let binds,tycons,env' = tc_tycon_defs cenv env parent ModuleOrMemberBinding tpenv (tdefs,m,scopem) in 
          (* check the non-escaping condition as we build the expression on the way back up *)
          let exprfWithEscapeCheck e = 
              let freeInEnv = computeUnabstractableTycons env in 
              tycons |> List.iter(fun tycon -> 
                  let nm = display_name_of_tycon tycon in 
                  if Zset.mem tycon freeInEnv then warning(Error(sprintf "The type '%s' is used in an invalid way. This is because a value prior to '%s' has an inferred type involving '%s', which is an invalid forward reference" nm nm nm, range_of_tycon tycon)));
              TMDefRec(tycons,binds,m) :: e in
          (exprfWithEscapeCheck,[]),env'

      | Def_partial_tycon (tcinfo,members,m) -> 
          let scopem = union_ranges m scopem in 
          let augment = prepare_tycon_member_defns NoNewSlots cenv env (tcinfo,None,members,m) in 
          let binds,env = tc_tycon_member_defns WarnOnOverrides cenv env parent m scopem [augment] in
          (* REVIEW: record a TDecl_partial_tycon instead of mutating the type constructor *)
          ((fun e -> TMDefRec([],binds,m) :: e),[]),env 

      | Def_open (mp,m) -> 
          ((fun e -> e),[]), tc_open_namespace cenv.autoModuleResolver m scopem env mp

      | Def_let (letrec, binds, m) -> 
          if letrec then 
            let scopem = union_ranges m scopem in 
            let binds = binds |> map (fun bind -> RecBindingDefn(containerInfo,NoNewSlots,ModuleOrMemberBinding,bind)) in 
            let binds,env,_ = tc_letrec  WarnOnOverrides cenv env tpenv (new_inference_typs cenv binds) (binds,m, scopem) in 
            ((fun e -> TMDefRec([],binds,m) :: e),[]),env
          else 
            let binds,env,_ = tc_let_bindings cenv env containerInfo ModuleOrMemberBinding tpenv (binds,m,scopem) in 
            ((fun e -> List.fold_right (fun b acc -> TMDefLet(b,m) :: acc) binds e),[]),env 

      | Def_attributes (attrs,_) -> 
          let attrs' = tc_attributes_with_possible_targets cenv env attrTgtTop attrs in 
          ((fun e -> e), attrs'), env

      | Def_hash (i,m) -> 
         ((fun e -> e), []), env

      | Def_module(ComponentInfo(attribs,kind,parms, constraints,longPath,xml,preferPostfix,vis,im),mdefs,explicitSigOpt,m) ->
          let id = computeModuleName longPath in 
          let isModule =  compute_isModule kind parms constraints im in

          let modAttrs = tc_attributes cenv env attrTgtModuleDecl attribs in 
          let istype = compute_istype cenv.g isModule modAttrs in
          if verbose then  dprintf0 "Def_module\n";
          if isModule then checkForDuplicateConcreteType cenv env (adjust_module_name istype id.idText) im;
          if isModule then checkForDuplicateModule cenv env id.idText id.idRange;
          let vis,_ = computeAccessAndCompPath env ModuleOrMemberBinding id.idRange vis parent in 
             
          let (attrs, mspecPriorToOuterOrExplicitSig,TMBind(modNameInfo,mexpr)),_ =
              tc_defs cenv env (id,isModule,mdefs,xml,modAttrs,vis,m) in

          let mspec,mdef = 
              match explicitSigOpt with 
              | None -> mspecPriorToOuterOrExplicitSig,TMDefModul(TMBind(modNameInfo,mexpr))
              | Some (Sign_explicit mspecs) -> 
              
                  (* Take the implementation and build a new module with a different module type *)
                  let mspecExplicitSig = 
                      mspecPriorToOuterOrExplicitSig |> new_mspec_modified (fun _ -> 
                          (* Check the elements of the signature *)
                          let mtyp,_ = tc_signature_specs_core cenv parentModule env (id,istype,mspecs,m,emptyXMLDoc) in
                          mtyp) in

                  let explicitSig = mtyp_of_modul mspecExplicitSig in 
              
                  (* Check the module signature *)
                  begin
                      (* Compute the alpha-conversion mapping between type contructors in signature and implementation *)
                      let aenv = 
                          let remapInfo ,hidingInfo = mk_mtyp_to_mtyp_remapping (mtyp_of_modul mspecPriorToOuterOrExplicitSig) explicitSig in
                          { tyeq_env_empty with ae_tcrefs = tcref_map_of_list remapInfo.mrpiTycons } in
                          
                      subtyp_modul cenv.g cenv.amap (denv_of_tenv env) aenv (mk_local_modref mspecPriorToOuterOrExplicitSig) explicitSig |> ignore;
                  end;

                  mspecExplicitSig,TMDefModul(TMBind(modNameInfo,TMAbstract(TMTyped(explicitSig,mexpr,m))))

              |  Some (Sign_named _) -> error(Error("Modules may not use named module signature definitions",im)) in         

          publishModuleDefn cenv env mspec; 
          let env = add_local_submodul scopem env (text_of_id id) mspec in
          ((fun e -> mdef :: e),attrs), env
(*
      | Def_named_compsig(ComponentInfo(attribs,kind,parms, constraints,id,xml,preferPostfix,vis,im),mspecs,m) ->
          assert_isConstraint kind im;
          let isModule = false in
          let vis,_ = computeAccessAndCompPath env declKind id.idRange vis in 
          let mspec,_ = tc_signature_module cenv env (id,isModule,mspecs,xml,attribs,vis,m) in
          publishTypeDefn cenv env mspec; 
          let env = add_local_submodul scopem env (text_of_id id) mspec in
          ((fun e -> e), []), env
      | Def_partial_inline_compsig(mspec,m) -> 
          error(Error("This construct is not yet implemented",m))
*)
      
    with e -> errorRecoveryPoint e; ((fun e -> e), []),env

and tc_defs_core cenv parent endm (defsSoFar,env) moreDefs =
    match moreDefs with 
    | (h1 :: t) ->
        (* lookahead one to find out the scope of the next declaration *)
        let scopem = 
            if isNil t then union_ranges (range_of_syndecl h1) endm
            else union_ranges (range_of_syndecl (List.hd t)) endm in 
        let h1',env' = tc_def cenv parent scopem env h1 in
        (* tail recursive *)
        tc_defs_core  cenv parent endm ( (h1' :: defsSoFar), env') t
    | [] -> List.rev defsSoFar,env
  
and tc_defs cenv env (id,isModule,defs,xml,modAttrs,vis,m) =
    if verbose then  dprintf0 "tc_defs...\n";
    let endm = end_range_of_range m in 
    let istype = compute_istype cenv.g isModule modAttrs in
    let envinner,mtypeAcc = mk_inner_env env id istype in 
    (* Ensure the modul_of_nlpath call in updateAccModuleType succeeds *)
    if cenv.compilingCanonicalFslibModuleType then 
        ensure_fslib_has_submodul_at cenv.topCcu envinner.ePath (curr_cpath envinner) xml;

    let cpath = (curr_cpath envinner) in

    (* Create the new module type to hold the accumulated results of the type of the module *)
    (* Also record this in the environment as the accumulator *)
    let mspec = new_mspec (Some(curr_cpath env)) vis id xml modAttrs (notlazy (empty_mtype istype)) in 
    let innerParent = mk_local_modref mspec in

    (* Now typecheck. *)
    let defs',envAtEnd = tc_defs_core cenv innerParent endm ([],envinner) defs  in 
    (* Get the inferred type of the decls. It's precisely the one we created before checking *)
    (* and mutated as we went. Record it in the mspec. *)
    (data_of_tycon mspec).tycon_modul_contents <- notlazy !mtypeAcc ; 

    (* Apply the functions for each declaration to build the overall expression-builder *)
    let mexpr = TMDefs(List.fold_right (fun (f,_) x -> f x) defs' [])  in

    if verbose then  dprintf2 "tc_defs %s, created %d\n" (name_of_modul mspec) (stamp_of_modul mspec);

    (* Collect up the attributes that are global to the file *)
    let topAttrs = List.fold_right (fun (_,y) x -> y@x) defs' [] in
    
    (* Build a dummy inaccessible tycon to act as the 'static type' for the module *)
    let tycon = new_tycon (Some cpath) (id.idText,id.idRange) vis taccessPublic [] emptyXMLDoc false  (notlazy (empty_mtype istype)) in
    (data_of_tycon tycon).tycon_attribs <- modAttrs;
    
    (topAttrs,mspec,TMBind(tycon,mexpr)), envAtEnd



(*--------------------------------------------------------------------------
!* Typecheck a module, close the inference scope and 
 * check the module meets its signature
 *-------------------------------------------------------------------------- *)

(* Set up the initial environment *)
let locate_tenv ccu env enclosingNamespacePath =
    let cpath = cpath_of_ccu ccu in
    let env = {env with ePath = []; eCompPath = cpath; eAccessPath=cpath } in 
    let env = List.fold_left (fun env id -> mk_inner_env env id Namespace |> fst) env enclosingNamespacePath in
    env

let buildTopRootedModul enclosingNamespacePath mspec = List.fold_right wrap_modul_in_namespace  enclosingNamespacePath mspec 
        
let buildTopRootedModulBind enclosingNamespacePath mbind = List.fold_right wrap_mbind_in_namespace  enclosingNamespacePath mbind 

(*--------------------------------------------------------------------------
!* typecheckOneImplFile - Typecheck all the namespace fragments in a file.
 *-------------------------------------------------------------------------- *)

let init_tenv g scopem ccus =
    fold_left (add_nonlocal_ccu scopem) (empty_tenv g) ccus

type conditionalDefines = 
    string list


type topAttribs =
    { mainMethodAttrs: attrib list;
      netModuleAttrs: attrib list;
      assemblyAttrs : attrib list  }

let emptyTopAttrs =
    { mainMethodAttrs=[];
      netModuleAttrs=[];
      assemblyAttrs =[]  }

let combineTopAttrs topAttrs1 topAttrs2 =
    { mainMethodAttrs = topAttrs1.mainMethodAttrs @ topAttrs2.mainMethodAttrs;
      netModuleAttrs  = topAttrs1.netModuleAttrs @ topAttrs2.netModuleAttrs;
      assemblyAttrs   = topAttrs1.assemblyAttrs @ topAttrs2.assemblyAttrs } 


let tc_namespace_fragment cenv
                 (topAttrs,env,envAtEnd,fragTypesPriorToSig,implFileBinds) 
                 (ModuleImpl(lid,isModule,defs,xml,attribs,vis,m)) =
               
    if !progress then dprint_endline ("Typecheck implementation "^text_of_lid lid);
    let endm = end_range_of_range m in 

    let enclosingNamespacePath,moduleName = frontAndBack lid in 
    let envinner = locate_tenv cenv.topCcu env enclosingNamespacePath in 

    let vis,_ = computeAccessAndCompPath envinner ModuleOrMemberBinding endm vis ParentNone in 

    if verbose then  dprint_endline ("checking "^text_of_lid lid);
    let (topAttrsNew,mspec,mbind), envAtEnd  = 
        let modAttrs = tc_attributes cenv envinner attrTgtModuleDecl attribs in 
        tc_defs cenv envinner (moduleName,isModule,defs,xml,modAttrs,vis,m) in 

    let modulRef = mk_local_modref mspec in 
    let implFileSpecPriorToSig = buildTopRootedModul enclosingNamespacePath mspec in 
    let mbindTopRooted = buildTopRootedModulBind enclosingNamespacePath mbind in
    let topAttrsNew = 
        let il_main_attrs,others = List.partition (fun (possTargets,_) -> possTargets &&& attrTgtMethod <> 0l) topAttrsNew in 
        let il_assem_attrs,others = List.partition (fun (possTargets,_) -> possTargets &&& attrTgtAssembly <> 0l) others in 
        let il_module_attrs,others = List.partition (fun (possTargets,_) -> possTargets &&& attrTgtModule <> 0l) others in 
        { mainMethodAttrs = map snd il_main_attrs;
          netModuleAttrs  = map snd il_module_attrs;
          assemblyAttrs   = map snd il_assem_attrs} in
          
    let env = add_local_top_rooted_mtyp m env (wrap_modul_as_mtyp_in_namespace implFileSpecPriorToSig) in 
    
    let topAttrs = combineTopAttrs topAttrs topAttrsNew in
    let fragTypesPriorToSig = fragTypesPriorToSig@[implFileSpecPriorToSig] in 
    let implFileBinds = implFileBinds @ [TMDefModul(mbindTopRooted)] in
    (topAttrs,env,envAtEnd,fragTypesPriorToSig,implFileBinds)


let rec iterTyconsOfModuleTyp f mty = 
    mty.mtyp_tycons |> Map.iter (fun _ tycon -> f tycon);
    mty |> submoduls_of_mtyp |> Namemap.iter (fun v -> 
        iterTyconsOfModuleTyp f (mtyp_of_modul v))



let typecheckOneImplFile 
       (* resolver: autoModuleResolver *)        
       (* checkForNoErrors: A function to help us stop reporting cascading errors *)        
       (g,niceNameGen,amap,topCcu,resolver,checkForNoErrors,conditionalDefines) 
       env 
       topRootedSigOpt  
       (ImplFile(fileName,qualNameOfFile,implFileFrags)) =

    let cenv = new_cenv (g,niceNameGen,amap,topCcu,false,isSome(topRootedSigOpt),resolver,conditionalDefines) in 
    cenv.ginstf <- (freshen_tps cenv);

    let topAttrs,env,envAtEnd,fragTypesPriorToSig,implFileBinds = 
        List.fold_left (tc_namespace_fragment cenv) (emptyTopAttrs,env,env,[],[]) implFileFrags in 

    (* Note: we currently give errors w.r.t. the display environment that includes ALL 'opens' from ALL the namespace fragments *)
    let denvAtEnd = denv_of_tenv envAtEnd in
    let fragDefn = TMDefs(implFileBinds) in 

    (* Note: unless we can do better, any errors are given w.r.t. 'range_of_qualNameOfFile', i.e. the range of the leading module/namespace declaration *)
    let m = range_of_qualNameOfFile qualNameOfFile in 

    (* Combine the fragments *)
    let implFileTypePriorToSig = combine_mtyps m (map wrap_modul_as_mtyp_in_namespace fragTypesPriorToSig) in
    let implFileSpecPriorToSig = wrap_mtyp_as_mspec (id_of_qualNameOfFile qualNameOfFile) (cpath_of_ccu topCcu) implFileTypePriorToSig in 

    (* Defaults get applied before the module signature is checked and before the implementation conditions on virtuals/overrides. *)
    (* Defaults get applied in priority order. Defaults listed last get priority 0 (lowest), 2nd last priority 1 etc. *)

    begin
        try
            let unsolved = Unsolved.unsolved_typars_of_mdef g cenv.amap denvAtEnd fragDefn in

            if verboseCC then dprintf0 "calling canonicalizePartialInferenceProblem\n" ;
            canonicalizePartialInferenceProblem (cenv,denvAtEnd,m) unsolved;

            if verboseCC then dprintf0 "applying defaults..." ;
    
            let applyDefaults priority =
                 if verboseCC then dprintf1 "assigning defaults to pseudo variables at priority %d...\n" priority;
                 unsolved |> List.iter (fun tp -> 
                    if not (tpref_is_solved tp) then 
                        (* Apply the first default. If we're defaulting one type variable to another then *)
                        (* the defaults will be propagated to the new type variable. *)
                        (constraints_of_typar tp) |> List.iter (fun tpc -> 
                            match tpc with 
                            | TTyparDefaultsToType(priority2,ty2,m) when priority2 = priority -> 
                                let ty1 = (mk_typar_ty tp) in
                                if (tpref_is_solved tp) || (type_equiv cenv.g ty1 ty2) then (
                                    if verbose then dprintf5 "skipping solved/equal default '%s' for variable '%s' near %a at priority %d\n" ((DebugPrint.showType ty2)) ((DebugPrint.showType ty1)) output_range m priority2;
                                ) else (
                                    if verbose then dprintf5 "assigning default '%s' for variable '%s' near %a at priority %d\n" ((DebugPrint.showType ty2)) ((DebugPrint.showType ty1)) output_range m priority2;
                                    let csenv = (mk_csenv cenv.css m denvAtEnd) in 
                                    tryD (fun () -> Csolve.solveTyparEqualsTyp 0 csenv m NoTrace ty1 ty2)
                                         (fun e -> solveTypAsError cenv denvAtEnd m ty1;
                                                   errorD(ErrorFromApplyingDefault(g,denvAtEnd,tp,ty2,e,m)))
                                    |> raiseOperationResult;
                                )
                            | _ -> ())) in
                    
            for priority = 10 downto 0 do
                applyDefaults priority
            done;

            (* OK, now apply defaults for any unsolved HeadTypeStaticReq or CompleteStaticReq  *)
            unsolved |> List.iter (fun tp ->     
                if not (tpref_is_solved tp) then 
                    if (static_req_of_tpref tp <> NoStaticReq) then
                        Csolve.choose_typar_solution_and_solve cenv.css (denv_of_tenv envAtEnd) tp);
        with e -> errorRecoveryPoint e
    end;

    (* Check completion of all classes defined across this file. *)
    (* REVIEW: this is not a great technique if inner signatures are permitted to hide *)
    (* virtual dispatch slots. *)


    begin 
        if verboseCC then dprintf0 "checking all virtual promises implemented...\n";
        try implFileTypePriorToSig |> iterTyconsOfModuleTyp (checkAllImplemented true cenv.g cenv.amap denvAtEnd);
        with e -> errorRecoveryPoint(e) 
    end;

    (* Check the value restriction *)

    if (checkForNoErrors()) then  (
        (* Check generalization. *)
        if verboseCC then dprintf0 "checking generalization of exported things...\n";

        let rec check mty =
            mty.mtyp_vals |> Map.iter (fun _ v -> 
                let ftyvs = (free_in_val v).free_loctypars |> Zset.elements in 
                if not (compgen_of_val v) && not (exists from_error_of_typar ftyvs) then 
                  match ftyvs with 
                  | tp :: _ -> error (ValueRestriction(envAtEnd.eNameResEnv.edenv,false,v, tp,(range_of_val v)))
                  | _ -> ());
            Map.iter (fun _ v -> check (mtyp_of_modul v)) (submoduls_of_mtyp mty) in
        try check implFileTypePriorToSig with e -> errorRecoveryPoint(e)
    );

    (* Solve unsolved internal type variables *)
    
    if (checkForNoErrors()) then  (
        if verboseCC then dprintf0 "solving non-default unresolved typars...\n";

        let unsolved = Unsolved.unsolved_typars_of_mdef g cenv.amap (denv_of_tenv envAtEnd) fragDefn in

        if verboseCC then dprintf1 "#unsolved = %d\n" (length unsolved);

        unsolved |> List.iter (fun tp -> 
                if (rigid_of_tpref tp <> TyparRigid) && not (tpref_is_solved tp) then 
                    Csolve.choose_typar_solution_and_solve cenv.css (denv_of_tenv envAtEnd) tp);
    );

    (* Check the module matches the signature *)
    let implFileExprAfterSig = 
        if verboseCC then dprintf0 "checking implementation meets signature...\n";
        match topRootedSigOpt with 
        | None -> 
            (* Deep copy the inferred type of the module *)
            let implFileTypePriorToSigCopied = 
                if verbose then dprintf1 "Compilation unit type before copy:\n%s\n" (Layout.showL (Layout.squashTo 192 (mtypeL implFileTypePriorToSig)));
                let res = copy_mtyp g false implFileTypePriorToSig in 
                if verbose then dprintf1 "Compilation unit type after copy:\n%s\n" (Layout.showL (Layout.squashTo 192 (mtypeL implFileTypePriorToSig)));
                res in

            TMTyped(implFileTypePriorToSigCopied,fragDefn,m)
            
        | Some sigFileType -> 

            if verbose then dprintf1 "Compilation unit constrained type:\n%s\n" (Layout.showL (Layout.squashTo 192 (mtypeL sigFileType)));
 
            (* We want to show imperative type variables in any types in error messages at this late point *)
            let denv = { envAtEnd.eNameResEnv.edenv with showImperativeTyparAnnotations=true; } in          
            begin 
                try 
                
                    (* As typechecked the signature and implementation use different tycons etc. *)
                    (* Here we (a) check there are enough names, (b) match them up to build a renaming and   *)
                    (* (c) check subsumption up to this renaming. *)
                    if not (check_names_modul denv (mk_local_ref implFileSpecPriorToSig) sigFileType) then 
                        raise ReportedError;

                    let remapInfo ,hidingInfo = mk_mtyp_to_mtyp_remapping implFileTypePriorToSig sigFileType in
                     
                    (*
                    dprintf1 "implFileSpecPriorToSig = \n-----------\n%s\n\n" (showL (mspecL implFileSpecPriorToSig));
                    dprintf1 "modulImplTopRootedPriorToSigRemapped = \n-----------\n%s\n\n" (showL (mspecL modulImplTopRootedPriorToSigRemapped));
                    dprintf1 "sigFileType = \n-----------\n%s\n\n" (showL (mspecL sigFileType));
                    *)
                    
                    let aenv = { tyeq_env_empty with ae_tcrefs = tcref_map_of_list remapInfo.mrpiTycons } in
                    
                    if not (subtyp_modul cenv.g cenv.amap denv aenv (mk_local_modref implFileSpecPriorToSig) sigFileType) then  (
                        (* we can just raise 'ReportedError' since subtyp_modul raises its own error *)
                        raise ReportedError;
                    )
                with e -> errorRecoveryPoint e;
            end;
            if verbose then dprintf0 "adjusting signature...\n";
            
            TMTyped(sigFileType,TMDefs(implFileBinds),m)  (* TMTyped(sigFileType, mexprTopRootedPriorToSig, m) *) in 

    let implFile = TImplFile(qualNameOfFile,implFileExprAfterSig) in
    
    begin 
        if verboseCC then dprintf0 "post-inference checks...\n";
        
        try Check.check_top_impl (Check.mk_cenv g cenv.amap cenv.topCcu (denv_of_tenv envAtEnd)) implFile;
        with e -> errorRecoveryPoint e
    end;

    if verbose then dprintf1 "<-- typecheckOneImplFile, nm = %s\n" (name_of_modul implFileSpecPriorToSig);
    topAttrs,implFile,envAtEnd


(*-------------------------------------------------------------------------
!* Check signature files
 *------------------------------------------------------------------------- *)
   
let tc_sig_file_fragment 
        (g,niceNameGen,amap,topCcu,resolver,conditionalDefines) 
        env 
        (ModuleSpec(lid,isModule,defs,xml,attribs,vis,m)) = 

    if verbose then  dprint_endline ("Typecheck interface "^text_of_lid lid);
    let cenv = new_cenv (g,niceNameGen,amap,topCcu,true,false,resolver,conditionalDefines) in
    cenv.ginstf <- (freshen_tps cenv);

    let enclosingNamespacePath,id = frontAndBack lid in 
    let env = locate_tenv cenv.topCcu env enclosingNamespacePath in 
    let vis,_ = computeAccessAndCompPath env ModuleOrMemberBinding id.idRange vis ParentNone in 
    let spec = (id,isModule,defs,xml,attribs,vis,m) in
      
    let mspec,envAtEnd = tc_signature_module cenv env spec in 
    
    (* Record the declKind-level definition of the module type, partly to be able to check if all signatures *)
    (* have corresponding implementations.... *)
    let sigFileFragType = wrap_modul_as_mtyp_in_namespace (buildTopRootedModul enclosingNamespacePath mspec) in 

    begin 
        if verboseCC then dprintf0 "checking all virtual promises implemented...\n";
        try sigFileFragType |> iterTyconsOfModuleTyp (checkAllImplemented false cenv.g cenv.amap (denv_of_tenv envAtEnd));
        with e -> errorRecoveryPoint(e) 
    end;

    sigFileFragType,envAtEnd

let typecheckOneSigFile  
       (g,niceNameGen,amap,topCcu,resolver,conditionalDefines) 
       tcEnv 
       (SigFile(fileName,qualNameOfFile, sigFileFrags)) = 
       
    let (tcEnvAtEnd,tcEnv,sigFileFragTypes) = 
        List.fold_left 
           (fun (tcEnvAtEnd,tcEnv,sigFileFragTypes) (ModuleSpec(lid,_,_,_,_,_,m) as mspec) ->
              let smodulTypeTopRooted,tcEnvAtEnd = tc_sig_file_fragment (g,niceNameGen,amap,topCcu,resolver,conditionalDefines) tcEnv mspec in
              let tcEnv = add_local_top_rooted_mtyp m tcEnv smodulTypeTopRooted in 
              let sigFileFragTypes = sigFileFragTypes@[smodulTypeTopRooted] in 
              tcEnvAtEnd, tcEnv,sigFileFragTypes)
           (tcEnv,tcEnv,[]) 
           sigFileFrags in
    let m = range_of_qualNameOfFile qualNameOfFile in 
    let sigFileType = combine_mtyps m sigFileFragTypes in
    tcEnvAtEnd,tcEnv,sigFileType

