(*==========================================================================
 * (c) Microsoft Corporation 2005-2007.  
 *
 *=========================================================================*)

module Microsoft.FSharp.Compatibility.OCaml.Lexing

open Microsoft.FSharp.Core
open Microsoft.FSharp.Compatibility
open Microsoft.FSharp.Compatibility.OCaml
open Microsoft.FSharp.Compatibility.OCaml.Pervasives
open System.IO
open Microsoft.FSharp.Tools.FsLex

type Position = 
  { pos_fname : string;
    pos_lnum : int;
    pos_bol : int;
    pos_cnum : int; }
  with 
     member x.FileName = x.pos_fname
     member x.Line = x.pos_lnum
     member x.Char = x.pos_cnum
     member x.AbsoluteOffset = x.pos_cnum
     member x.StartOfLine = x.pos_bol
     member x.StartOfLineAbsoluteOffset = x.pos_bol
     member x.Column = x.pos_cnum - x.pos_bol
     member pos.NextLine = 
          {pos with pos_lnum = pos.Line+1; pos_bol = pos.AbsoluteOffset }
     member pos.AsNewLinePos() = pos.NextLine
  end
and position = Position  
let dummy_pos = {pos_fname=""; pos_lnum= 0; pos_bol= 0; pos_cnum=0 }

type lexbuf =  LexBuffer<position,byte>

let new_lexbuf (lb:lexbuf) = 
  lb.MatchEvent.Add(fun () -> 
      let curr = lb.EndPos in 
      lb.StartPos <- curr;
      lb.EndPos <- { curr with pos_cnum=curr.pos_cnum + lb.LexemeLength });
  lb

let from_function f = new_lexbuf (LexBuffer.FromByteFunction(dummy_pos,f))
let from_channel is  = from_function (fun s max -> Pervasives.input is s 0 max)

let from_text_reader (enc: #System.Text.Encoding) (tr: #TextReader) =
  from_function (fun bytebuf len ->
    /// Don't read too many characters!
    let lenc = (len * 99) / enc.GetMaxByteCount(100) in 
    let charbuf : char[] = CompatArray.zero_create lenc in
    let nRead = tr.Read(charbuf,0,lenc) in
    if nRead = 0 then 0 
                 else enc.GetBytes(charbuf,0,nRead,bytebuf,0))
let from_bytearray s  = new_lexbuf (LexBuffer.FromBytes(dummy_pos,s))
let from_string s  = from_bytearray (Bytearray.string_to_ascii s)

// deprecated - need to decode the characters, and accessing the base stream directly is evil.
let from_stream_reader (sr: StreamReader)  = from_function (fun s max -> sr.BaseStream.Read(s,0,max))
let from_binary_reader (sr: #BinaryReader)  = from_function (fun s max -> sr.Read(s,0,max))

let lexeme_char (lb:lexbuf) n =  Char.chr (Byte.to_int (lb.LexemeChar n))
let lexeme_start_p (lb:lexbuf) = lb.StartPos
let lexeme_end_p (lb:lexbuf) = lb.EndPos
let lexeme_start (lb:lexbuf) = (lexeme_start_p lb).pos_cnum
let lexeme_end (lb:lexbuf) = (lexeme_end_p lb).pos_cnum
let lexeme (lb:lexbuf) = System.Text.Encoding.ASCII.GetString(Compatibility.CompatArray.of_array lb.Lexeme)
let lexeme_utf8 (lb:lexbuf) = System.Text.Encoding.UTF8.GetString(Compatibility.CompatArray.of_array lb.Lexeme)
let lexeme_bytes (lb:lexbuf) = lb.Lexeme
let flush_input (lb: lexbuf) = lb.DiscardInput ()

let lexbuf_curr_p lb = lexeme_end_p lb
let lexbuf_set_curr_p (lb:lexbuf) (p : position) = lb.EndPos  <- p
let lexbuf_set_start_p (lb:lexbuf) (p : position) = lb.StartPos <- p

