(* (c) Microsoft Corporation. All rights reserved *)

(*----------------------------------------------------------------------------
!* LexFilter - process the token stream prior to parsing.
 * Imlements the offside rule and a copule of other lexical transformations.
 *--------------------------------------------------------------------------*)

(*F# 
open Microsoft.FSharp.Compiler 
open Microsoft.Research.AbstractIL
open Microsoft.Research.AbstractIL.Internal 
module Ildiag = Microsoft.Research.AbstractIL.Diagnostics 
F#*)

open List
open Ast
open Lib
open Lexing
open Lexhelp
open Ildiag

let debug = (*IF-OCAML*) false (*ENDIF-OCAML*) (*F# false F#*)  

type column = int
type hardWhiteLexFilterContext = 
  (* position is position of keyword *)
  | CtxtLetDecl of bool * position  (* bool indicates 'LET' is an offside let that's part of a CtxtSeqBlock where the 'in' is optional *)
  | CtxtIf of position  
  | CtxtTry of position  
  | CtxtFun of position  
  | CtxtFunction of position  
  | CtxtWithAsLet of position  (* 'with' when used in an object expression *)
  | CtxtWithAsAugment of position   (* 'with' as used in a type augmentation *)
  | CtxtMatch of position  
  | CtxtFor of position  
  | CtxtWhile of position  
  | CtxtWhen of position   
  | CtxtVanilla of position
  | CtxtThen of position  
  | CtxtElse of position 
  | CtxtDo of position 
  | CtxtInterfaceHead of position 
  | CtxtType of position 
  
  | CtxtNamespaceHead of position 
  | CtxtModuleHead of position 
  | CtxtMemberHead of position 
  | CtxtMemberBody of position 
  | CtxtModuleBody of position 
  | CtxtException of position 
  | CtxtParen of Pars.token * position (* bool indicates whether a subsequent 'DOT' is low-precedence *)

  (* Stack mark used to change the '.' in 'f(x).M' to be low-precedence  *)

  (* position is position of following token *)
  | CtxtSeqBlock of firstInSequence * position * addBlockEnd   (* first bool is true if first in sequence *)
  | CtxtMatchClauses of bool * position   (* first bool indicates "was this 'with' followed immediately by a '|'"? *)
and addBlockEnd = AddBlockEnd | NoAddBlockEnd | AddOneSidedBlockEnd
and firstInSequence = FirstInSeqBlock | NotFirstInSeqBlock

let col_of_pos p = (p.pos_cnum - p.pos_bol)
let line_of_pos p = p.pos_lnum 
let add_to_col p n = {p with pos_cnum = p.pos_cnum + n}

let output_pos os p = Printf.fprintf os "(%d:%d)" (line_of_pos p) (col_of_pos p)
let string_of_pos p = Printf.sprintf    "(%d:%d)" (line_of_pos p) (col_of_pos p)

let string_of_ctxt c = 
  match c with 
  | CtxtNamespaceHead _ -> "nshead"
  | CtxtModuleHead _ -> "modhead"
  | CtxtException _ -> "exception"
  | CtxtModuleBody _ -> "modbody"
  | CtxtLetDecl(b,p) -> Printf.sprintf "let(%b,%s)" b (string_of_pos p)
  | CtxtWithAsLet p -> Printf.sprintf "withlet(%s)" (string_of_pos p)
  | CtxtWithAsAugment _ -> "withaug"
  | CtxtDo _ -> "do"
  | CtxtInterfaceHead _ -> "interface-decl"
  | CtxtType _ -> "type"
  | CtxtParen _ -> "paren"
  | CtxtMemberHead _ -> "member-head"
  | CtxtMemberBody _ -> "body"
  | CtxtSeqBlock (b,p,addBlockEnd) -> Printf.sprintf "seqblock(%s,%s)" (match b with FirstInSeqBlock -> "first" | NotFirstInSeqBlock -> "subsequent") (string_of_pos p)
  | CtxtMatchClauses _ -> "withblock"

  | CtxtIf _ -> "if"
  | CtxtMatch _ -> "match"
  | CtxtFor _ -> "for"
  | CtxtWhile p -> Printf.sprintf "while(%s)" (string_of_pos p)
  | CtxtWhen _ -> "when" 
  | CtxtTry _ -> "try"
  | CtxtFun _ -> "fun"
  | CtxtFunction _ -> "function"

  | CtxtThen _ -> "then"
  | CtxtElse p -> Printf.sprintf "else(%s)" (string_of_pos p)
  | CtxtVanilla (p) -> Printf.sprintf "vanilla(%s)" (string_of_pos p)

let string_of_stack s = String.concat ";" (List.map string_of_ctxt s)


open Pars

let isInfix token = 
  match token with 
  | COMMA | BAR_BAR | AMP_AMP | AMP | OR
  | INFIX_BAR_OP _ 
  | INFIX_AMP_OP _  
  | INFIX_COMPARE_OP _ 
  | DOLLAR 
  (* LESS |GREATER *) 
  | INFIX_AT_HAT_OP _
  | PLUS_MINUS_OP _ 
  | MINUS  
  | STAR 
  | INFIX_STAR_DIV_MOD_OP _
  | INFIX_STAR_STAR_OP _ 
  | QMARK_QMARK -> true
  | _ -> false

let pos_of_ctxt c = 
  match c with 
  | CtxtNamespaceHead p | CtxtModuleHead p | CtxtException p | CtxtModuleBody p
  | CtxtLetDecl (_,p) | CtxtDo p | CtxtInterfaceHead p | CtxtType p | CtxtParen(_,p) | CtxtMemberHead p | CtxtMemberBody p
  | CtxtWithAsLet(p)
  | CtxtWithAsAugment(p)
  | CtxtMatchClauses (_,p) | CtxtIf p | CtxtMatch p | CtxtFor p | CtxtWhile p | CtxtWhen p | CtxtFunction p | CtxtFun p | CtxtTry p | CtxtThen p | CtxtElse (p) | CtxtVanilla p
  | CtxtSeqBlock (_,p,_) -> p

let col_of_ctxt ctxt = col_of_pos (pos_of_ctxt ctxt) 
let line_of_ctxt ctxt = line_of_pos (pos_of_ctxt ctxt) 

let rec isIfBlockContinuator token =
  match token with 
  | THEN | ELSE | ELIF -> true  (* These tokens may align with the "if" without closing the "if", e.g.
                                    if  ...
                                    then  ...
                                    elif ...
                                    else ... *) 
  | END | RPAREN -> true  (* REVIEW: review these two *)
  | ORIGHT_BLOCK_END | OBLOCKEND | ODECLEND -> true (* REVIEW: review these.  They arise during reprocessing of the inserted tokens when we hit a DONE *)
  | ODUMMY(token) -> isIfBlockContinuator(token)
  | _ -> false
let rec isTryBlockContinuator token =
  match token with 
  | FINALLY | WITH -> true  
                (* These tokens may align with the "try" without closing the construct, e.g.
                           try ...
                           with ... *)
  | ORIGHT_BLOCK_END | OBLOCKEND | ODECLEND -> true (* REVIEW: review these.  They arise during reprocessing of the inserted tokens when we hit a DONE *)
  | ODUMMY(token) -> isTryBlockContinuator(token)
  | _ -> false

let rec isThenBlockContinuator token =
  match token with 
  | ORIGHT_BLOCK_END | OBLOCKEND | ODECLEND -> true (* REVIEW: review these.  They arise during reprocessing of the inserted tokens when we hit a DONE *)
  | ODUMMY(token) -> isThenBlockContinuator(token)
  | _ -> false

let rec isDoContinuator token =
  match token with 
  | DONE -> true (* These tokens may align with the "for" without closing the construct, e.g.
                           for ... 
                              do 
                                 ... 
                              done *)
  | ORIGHT_BLOCK_END | OBLOCKEND | ODECLEND -> true (* REVIEW: review these.  They arise during reprocessing of the inserted tokens when we hit a DONE *)
  | ODUMMY(token) -> isDoContinuator(token)
  | _ -> false

let rec isInterfaceContinuator token =
  match token with 
  | END -> true (* These tokens may align with the token "interface" without closing the construct, e.g.
                           interface ... with 
                             ...
                           end *)
  | ORIGHT_BLOCK_END | OBLOCKEND | ODECLEND -> true (* REVIEW: review these.  They arise during reprocessing of the inserted tokens when we hit a DONE *)
  | ODUMMY(token) -> isInterfaceContinuator(token)
  | _ -> false

let rec isTypeContinuator token =
  match token with 
  | WITH | BAR | AND | END -> true (* These tokens may align with the token "type" without closing the construct, e.g.
                           type X = 
                           | A
                           | B
                           and type Y = c 
                           
                           type Complex = struct
                             val im : float
                           end with
                             static member M() = 1
                           end *)
                           
  | ORIGHT_BLOCK_END | OBLOCKEND | ODECLEND -> true (* REVIEW: review these.  They arise during reprocessing of the inserted tokens when we hit a DONE *)
  | ODUMMY(token) -> isTypeContinuator(token)
  | _ -> false

let rec isForLoopContinuator token =
  match token with 
  | DONE -> true (* These tokens may align with the "for" without closing the construct, e.g.
                           for ... do 
                              ... 
                           done *)
  | ORIGHT_BLOCK_END | OBLOCKEND | ODECLEND -> true(* REVIEW: review these.  They arise during reprocessing of the inserted tokens when we hit a DONE *)
  | ODUMMY(token) -> isForLoopContinuator(token)
  | _ -> false

let rec isWhileBlockContinuator token =
  match token with 
  | DONE -> true (* These tokens may align with the "while" without closing the construct, e.g.
                           while ... do 
                              ... 
                           done *)
  | ORIGHT_BLOCK_END | OBLOCKEND | ODECLEND -> true (* REVIEW: review these.  They arise during reprocessing of the inserted tokens when we hit a DONE *)
  | ODUMMY(token) -> isWhileBlockContinuator(token)
  | _ -> false

let rec isLetContinuator token =
  match token with 
  | AND -> true  (* These tokens may align with the "let" without closing the construct, e.g.
                           let ...
                           and ... *)
  | ORIGHT_BLOCK_END | OBLOCKEND | ODECLEND -> true (* REVIEW: review these.  They arise during reprocessing of the inserted tokens when we hit a DONE *)
  | ODUMMY(token) -> isLetContinuator(token)
  | _ -> false

let rec isTypeSeqBlockElementContinuator token = 
  match token with 
  | BAR -> true
        (* A sequence of items separated by '|' counts as one sequence block element, e.g.
           type x = 
             | A                 <-- These together count as one element
             | B                 <-- These together count as one element
             member x.M1
             member x.M2 *)
  | OBLOCKBEGIN | ORIGHT_BLOCK_END | OBLOCKEND | ODECLEND -> true (* REVIEW: review these.  They arise during reprocessing of the inserted tokens when we hit a DONE *)
  | ODUMMY(token) -> isTypeSeqBlockElementContinuator token 
  | _ -> false

(* Work out when a token doesn't terminate a single item in a sequence definition *)
let rec isSeqBlockElementContinuator token =
  isInfix token || 
        (* Infix tokens may align with the first column of a sequence block without closing a sequence element and starting a new one *)
        (* e.g. 
          let f x
              h x 
              |> y                              <------- NOTE: Not a new element in the sequence
        *) 
  (* end tokens *)
  match token with 
  | END | AND | WITH | THEN | RPAREN | RBRACE | RBRACK | BAR_RBRACK | RQUOTE _ -> true 
        (* The above tokens may align with the first column of a sequence block without closing a sequence element and starting a new one *)
        (* e.g. 
          new MenuItem("&Open...", 
                       new EventHandler(fun _ _ -> 
                           ...
                       ),                              <------- NOTE RPAREN HERE
                       Shortcut.CtrlO)
        *) 
  | ORIGHT_BLOCK_END | OBLOCKEND | ODECLEND -> true (* REVIEW: review these.  They arise during reprocessing of the inserted tokens when we hit a DONE *)
  | ODUMMY(token) -> isSeqBlockElementContinuator token 
  | _ -> false
let rec isWithAugmentBlockContinuator token = 
  match token with 
  | END -> true    (* These tokens may align with "with" of an augmentation block without closing the construct, e.g.
                           interface Foo
                              with 
                                 member ...
                              end *)
  | ODUMMY(token) -> isWithAugmentBlockContinuator(token)
  | _ -> false

let isLongIdentifier token = (match token with IDENT _ | DOT -> true | _ -> false)

let isAtomicExprEndToken token = 
    match token with
    | (IDENT _ 
       | INT8 _ | INT16 _ | INT32 _ | INT64 _ | NATIVEINT _ 
       | UINT8 _ | UINT16 _ | UINT32 _ | UINT64 _ | UNATIVEINT _
       | DECIMAL _ | BIGNUM _ | BIGINT _  | STRING _ | BYTEARRAY _  | CHAR _ 
       | IEEE32 _ | IEEE64 _ 
       | RPAREN | RBRACK | RBRACE | BAR_RBRACK | END 
       | NULL | FALSE | TRUE | UNDERSCORE) -> true
    | _ -> false
    
(*----------------------------------------------------------------------------
!* give a 'begin' token, does an 'end' token match?
 *--------------------------------------------------------------------------*)
let parenTokensBalance t1 t2 = 
    match t1,t2 with 
    | (LPAREN,RPAREN) 
    | (LBRACE,RBRACE) 
    | (LBRACK,RBRACK) 
    | (INTERFACE,END) 
    | (CLASS,END) 
    | (SIG,END) 
    | (STRUCT,END) 
    | (LBRACK_BAR,BAR_RBRACK)
    | (BEGIN,END) -> true 
    | (LQUOTE q1,RQUOTE q2) when q1 = q2 -> true 
    | _ -> false
    
type t = 
  { offsideStack : hardWhiteLexFilterContext list ref;
    lightSyntaxStatus: lightSyntaxStatus;
    lexbuf : Lexing.lexbuf;
    lexer: Lexing.lexbuf -> Pars.token }
type doneMarker = Done of token


(*----------------------------------------------------------------------------
!* build a hardWhiteLexFilter
 *--------------------------------------------------------------------------*)

let create lightSyntaxStatus lexer (lexbuf: Lexing.lexbuf)  = 

    delayInsertedToWorkaroundKnownNgenBug "Delay1" |< fun () ->
    
    (*----------------------------------------------------------------------------
    !* Part I. Building a new lex stream from an old
     *
     * A lexbuf is a stateful object that can be enticed to emit tokens by calling
     * 'lexer' functions designed to work with the lexbuf.  Here we fake a new stream
     * coming out of an existing lexbuf.  Ideally lexbufs would be abstract interfaces
     * and we could just build a new abstract interface that wraps an existing one.
     * However that is not how either OCaml or F# lexbufs work.
     * 
     * Part of the fakery we perform involves buffering a lookahead token which 
     * we eventually pass on to the client.  However, this client also looks at
     * other aspects of the 'state' of lexbuf directly, e.g. OCaml lexbufs
     * have 
     *    (start-pos, end-pos)
     * states, and F# lexbufs have a triple
     *    (start-pos, end-pos, eof-reached)
     *
     * You may ask why the F# parser reads this lexbuf state directly.  Well, the
     * pars.mly code itself it doesn't, but the parser engines (Parsing, prim-parsing.fs) 
     * certainly do for both F# and OCaml. e.g. when these parsers read a token 
     * from the lexstream they also read the position information and keep this in 
     * a related stack. 
     *
     * Anyway, this explains the functions getLexbufState(), setLexbufState() etc.
     *--------------------------------------------------------------------------*)

    (* make sure we don't report 'eof' when inserting a token, and set the positions to the *)
    (* last reported token position *)
    let lexbufStateForInsertedDummyTokens (lastTokenStartPos,lastTokenEndPos) =
         (*IF-OCAML*) 
              (lastTokenStartPos,lastTokenEndPos)
         (*ENDIF-OCAML*)
         (*F# (lastTokenStartPos,lastTokenEndPos,false) F#*) in

    let getLexbufState() = 
         (*IF-OCAML*) 
              (lexeme_start_p lexbuf, lexeme_end_p lexbuf) 
         (*ENDIF-OCAML*)
         (*F# let p = (lexbuf.StartPos, lexbuf.EndPos, lexbuf.IsPastEndOfStream)   in 
              if debug then dprintf2 "GET lex state: %a\n" output_any p;
              p 
         F#*)  in

    let setLexbufState p =
         (*IF-OCAML*) 
              let (p1,p2) = p in 
              lexbuf.Lexing.lex_start_p <- p1;  
              lexbuf.Lexing.lex_curr_p <- p2;
         (*ENDIF-OCAML*)
         (*F# 
              if debug then dprintf2 "SET lex state to; %a\n" output_any p; 
              let (p1,p2,eof) = p in  
              lexbuf.StartPos <- p1;  
              lexbuf.EndPos <- p2; 
              lexbuf.IsPastEndOfStream <- eof  
           F#*) in

    let startPosOfLexbufState = 
         (*IF-OCAML*) 
              (fun (s1,e1)      -> s1)
         (*ENDIF-OCAML*)
         (*F# 
              (fun (s1,e1,eof1) -> s1)
         F#*)  in

    let endPosOfLexbufState = 
         (*IF-OCAML*) 
              (fun (s1,e1)      -> e1)
         (*ENDIF-OCAML*)
         (*F# 
              (fun (s1,e1,eof1) -> e1)
         F#*)  in

    let startPosOfTokenTup (token,tokenLexbufState,_) = 
          match token with
          (* EOF token is processed as if they were on column -1 *)
          (* This forces the closure of all contexts. REVIEW: use isSemiSemi *)
          | Pars.EOF _ 
          (* | Pars.SEMICOLON_SEMICOLON *) -> 
              let p = startPosOfLexbufState tokenLexbufState in 
              { p with pos_cnum = p.pos_bol-1 }
          | _ ->  (startPosOfLexbufState tokenLexbufState)  in

    let tokenOfTokenTup (token,_,_) = token in
    let endPosOfTokenTup (_,tokenLexbufState,_) = endPosOfLexbufState tokenLexbufState in

    (*----------------------------------------------------------------------------
    !* Part II. The state of the new lex stream object.
     *--------------------------------------------------------------------------*)

    (* Ok, we're going to the wrapped lexbuf.  Set the lexstate back so that the lexbuf *)
    (* appears consistent and correct for the wrapped lexer function. *)
    let runWrappedLexerInConsistentLexbufState =
        let savedLexbufState = ref None in  
        fun () -> 
            !savedLexbufState |> Option.iter setLexbufState;
            let lastTokenRange = 
                let state = (match !savedLexbufState with Some x -> x | None -> getLexbufState()) in
                startPosOfLexbufState state, endPosOfLexbufState state in
            let token = lexer lexbuf in
            (* Now we've got the token, remember the lexbuf state, associating it with the token *)
            (* and remembering it as the last observed lexbuf state for the wrapped lexer function. *)
            let tokenLexbufState = getLexbufState() in 
            savedLexbufState := Some tokenLexbufState;
            token,tokenLexbufState,lastTokenRange in

    (*----------------------------------------------------------------------------
    !* Fetch a raw token, either from the old lexer or from our delayedStack
     *--------------------------------------------------------------------------*)

    let delayedStack = ref [] in

    let delayToken tokenTup = 
      delayedStack := tokenTup :: !delayedStack in

    let popNextTokenTup() = 
      match !delayedStack with 
      | (_,tokenLexbufState,_) as tokenTup :: rest -> 
          delayedStack := rest; 
          if debug then dprintf2 "popNextTokenTup: delayed token, tokenStartPos = %a\n" output_pos (startPosOfTokenTup tokenTup); 
          tokenTup
      | [] -> 
          if debug then dprintf0 "popNextTokenTup: no delayed tokens, running lexer...\n";
          runWrappedLexerInConsistentLexbufState()  in
    

    (*----------------------------------------------------------------------------
    !* Part III. Initial configuration of state.
     *
     * We read a token.  In F# Interactive the parser thread will be correctly blocking
     * here.
     *--------------------------------------------------------------------------*)

    let initialized = ref false in    
    let offsideStack = ref [] in  
    let prevWasAtomicEnd = ref false in  
    
    let peekInitial() =
        let initialLookaheadTokenTup  = popNextTokenTup() in
        if debug then dprintf2 "first token: initialLookaheadTokenLexbufState = %a\n" output_pos (startPosOfTokenTup initialLookaheadTokenTup); 
        delayToken initialLookaheadTokenTup; 
        initialized := true;
        offsideStack := (CtxtSeqBlock(FirstInSeqBlock,startPosOfTokenTup initialLookaheadTokenTup,NoAddBlockEnd)) :: !offsideStack in

    let warn s msg = 
        warning(Lexhelp.IndentationProblem(msg,mksyn_range (startPosOfTokenTup s) (endPosOfTokenTup s))) in

    (*----------------------------------------------------------------------------
    !* Part IV. Helper functions for pushing contexts and giving good warnings
     * if a context is undented.  
     *
     * Undentation rules
     *--------------------------------------------------------------------------*)

    let pushCtxt tokenTup newCtxt =
        let rec unindentationLimit strict stack = 
            match newCtxt,stack with 
              | _, [] -> (pos_of_ctxt newCtxt, -1) 
              (* | _, (CtxtSeqBlock _ :: (CtxtModuleBody _ | CtxtMatchClauses _ | CtxtThen _ | CtxtElse _ | CtxtDo _ | CtxtParen _ | CtxtMemberBody _) :: _) -> () *) 

              (* ignore SeqBlock because something more interesting is coming *)
              (* | CtxtSeqBlock _ :: rest -> unindentationLimit strict rest*)
              (* ignore Vanilla because a SeqBlock is always coming *)
              | _, (CtxtVanilla _ :: rest) -> unindentationLimit strict rest

              | _, (CtxtSeqBlock _ :: rest) when not strict -> unindentationLimit strict rest
              | _, (CtxtParen _ :: rest) when not strict -> unindentationLimit strict rest



              (* 'begin match' limited by minimum of two  *)
              (* '(match' limited by minimum of two  *)
              | _,(((CtxtMatch _) as ctxt1) :: CtxtSeqBlock _ :: (CtxtParen ((BEGIN | LPAREN),_) as ctxt2) :: rest)
                        -> if col_of_ctxt ctxt1 <= col_of_ctxt ctxt2 
                           then (pos_of_ctxt ctxt1,col_of_ctxt ctxt1) 
                           else (pos_of_ctxt ctxt2,col_of_ctxt ctxt2) 

               (* 'let ... = function' limited by 'let', precisely  *)
               (* This covers the common form *)
               (*                          *)
               (*     let f x = function   *)
               (*     | Case1 -> ...       *)
               (*     | Case2 -> ...       *)
              | (CtxtMatchClauses _), (CtxtFunction _ :: CtxtSeqBlock _ :: (CtxtLetDecl  _ as limitCtxt) :: rest)
                        -> (pos_of_ctxt limitCtxt,col_of_ctxt limitCtxt)

              (* Otherwise 'function ...' places no limit until we hit a CtxtLetDecl etc...  (Recursive) *)
              | (CtxtMatchClauses _), (CtxtFunction _ :: rest)
                        -> unindentationLimit false rest

              (* 'try ... with'  limited by 'try'  *)
              | _,(CtxtMatchClauses _ :: (CtxtTry _ as limitCtxt) :: rest)
                        -> (pos_of_ctxt limitCtxt,col_of_ctxt limitCtxt)

              (* 'fun ->' places no limit until we hit a CtxtLetDecl etc...  (Recursive) *)
              | _,(CtxtFun _ :: rest)
                        -> unindentationLimit false rest

              (* 'f ...{' places no limit until we hit a CtxtLetDecl etc... *)
              | _,(CtxtParen (LBRACE,_) :: CtxtVanilla _ :: CtxtSeqBlock _ :: rest)
              | _,(CtxtSeqBlock _ :: CtxtParen(LBRACE,_) :: CtxtVanilla _ :: CtxtSeqBlock _ :: rest)
                        -> unindentationLimit false rest


               (* MAJOR PERMITTED UNDENTATION This is allowing:
                    if x then y else
                    let x = 3 + 4
                    x + x  
                  This is a serious thing to allow, but is required since there is no "return" in this language.
                  Without it there is no way of escaping special cases in large bits of code without indenting the main case.
                  *)
              | CtxtSeqBlock _, (CtxtElse _  :: (CtxtIf _ as limitCtxt) :: rest) 
                        -> (pos_of_ctxt limitCtxt,col_of_ctxt limitCtxt)

              (* Permitted inner-construct precise block alighnment: 
                           interface ...
                           with ... 
                           end 
                           
                           type ...
                           with ... 
                           end *)
              | CtxtWithAsAugment _,((CtxtInterfaceHead _ | CtxtMemberHead _ | CtxtException _ | CtxtType _) as limitCtxt  :: rest)
                        -> (pos_of_ctxt limitCtxt,col_of_ctxt limitCtxt) 

              (* Permit unindentation via parentheses (or begin/end) following a 'then', 'else' or 'do':
                        if nr > 0 then (  
                              nr <- nr - 1;
                              acc <- combine nr acc (int_of_byte (CompatArray.get x i));
                              i <- i - 1
                        ) else (
                              i <- -1
                        );
              *)

              (* PERMITTED UNDENTATION: Inner construct (then,with,else,do) that dangle, places no limit until we hit the corresponding leading construct CtxtIf, CtxtFor, CtxtWhile, CtxtVanilla etc... *)
              (*    e.g.   if ... then ...
                              expr
                           else
                              expr
                    rather than forcing 
                           if ... 
                           then expr
                           else expr
                           
                           
                           
                  Also  ...... with
                           ...           <-- this is before the "with"
                        end


               *)

              | _,((CtxtWithAsAugment _ | CtxtThen _ | CtxtElse _ | CtxtDo _ )  :: rest)
                        -> unindentationLimit false rest


              (* '... (function ->' places no limit until we hit a CtxtLetDecl etc....  (Recursive) *)
              | _,(CtxtFunction _ :: rest)
                        -> unindentationLimit false rest

              (* 'module ... : sig'    limited by 'module' *)
              (* 'module ... : struct' limited by 'module' *)
              (* 'module ... : begin'  limited by 'module' *)
              (* 'if ... then ('       limited by 'if' *)
              (* 'if ... then {'       limited by 'if' *)
              (* 'if ... then ['       limited by 'if' *)
              (* 'if ... then [|'       limited by 'if' *)
              (* 'if ... else ('       limited by 'if' *)
              (* 'if ... else {'       limited by 'if' *)
              (* 'if ... else ['       limited by 'if' *)
              (* 'if ... else [|'       limited by 'if' *)
              (* 'f ... ('       limited by 'f' *)
              (* 'f ... {'       limited by 'f' *)
              (* 'f ... ['       limited by 'f' *)
              (* 'f ... [|'       limited by 'f' *)
              (* 'type C = class ... '       limited by 'type' *)
              (* 'type C = interface ... '       limited by 'type' *)
              (* 'type C = struct ... '       limited by 'type' *)
              | _,(CtxtParen ((SIG | STRUCT | BEGIN),_) :: CtxtSeqBlock _  :: (CtxtModuleBody _ as limitCtxt) ::  _)
              | _,(CtxtParen ((BEGIN | LPAREN | LBRACK | LBRACE | LBRACK_BAR)      ,_) :: CtxtSeqBlock _ :: CtxtThen _ :: (CtxtIf _         as limitCtxt) ::  _)
              | _,(CtxtParen ((BEGIN | LPAREN | LBRACK | LBRACE | LBRACK_BAR)      ,_) :: CtxtSeqBlock _ :: CtxtElse _ :: (CtxtIf _         as limitCtxt) ::  _)
              | _,(CtxtParen ((BEGIN | LPAREN | LBRACK (* | LBRACE *) | LBRACK_BAR)      ,_) :: CtxtVanilla _ :: (CtxtSeqBlock _         as limitCtxt) ::  _)
              | _,(CtxtParen ((CLASS | STRUCT | INTERFACE),_) :: CtxtSeqBlock _ :: (CtxtType _ as limitCtxt) ::  _)
                        -> (pos_of_ctxt limitCtxt,col_of_ctxt limitCtxt + 1) 

              | _,(CtxtSeqBlock _ :: CtxtParen((BEGIN | LPAREN | LBRACK (* | LBRACE *) | LBRACK_BAR),_) :: CtxtVanilla _ :: (CtxtSeqBlock _ as limitCtxt) :: _)
              | (CtxtSeqBlock _),(CtxtParen ((BEGIN | LPAREN | LBRACE | LBRACK | LBRACK_BAR)      ,_) :: CtxtSeqBlock _ :: ((CtxtType _ | CtxtLetDecl _ | CtxtMemberBody _ | CtxtWithAsLet _) as limitCtxt) ::  _)
                        -> (pos_of_ctxt limitCtxt,col_of_ctxt limitCtxt + 1) 

              (* Permitted inner-construct (e.g. "then" block and "else" block in overall "if-then-else" block ) block alighnment: 
                           if ... 
                           then expr
                           elif expr  
                           else expr  *)
              | (CtxtIf   _ | CtxtElse _ | CtxtThen _), (CtxtIf _ as limitCtxt) :: rest  
                        -> (pos_of_ctxt limitCtxt,col_of_ctxt limitCtxt)
              (* Permitted inner-construct precise block alighnment: 
                           while  ... 
                           do expr
                           done   *)
              | (CtxtDo _), ((CtxtFor  _ | CtxtWhile _) as limitCtxt) :: rest  
                        -> (pos_of_ctxt limitCtxt,col_of_ctxt limitCtxt)


              (* These contexts all require indentation by at least one space *)
              | _,((CtxtInterfaceHead _ | CtxtNamespaceHead _ | CtxtModuleHead _ | CtxtException _ | CtxtModuleBody _ | CtxtIf _ | CtxtWithAsLet _ | CtxtLetDecl _ | CtxtMemberHead _ | CtxtMemberBody _) as limitCtxt :: _) 
                        -> (pos_of_ctxt limitCtxt,col_of_ctxt limitCtxt + 1) 

              (* These contexts can have their contents exactly aligning *)
              | _,((CtxtParen _ | CtxtFor _ | CtxtWhen _ | CtxtWhile _ | CtxtType _ | CtxtMatch _  | CtxtTry _ | CtxtMatchClauses _ | CtxtSeqBlock _) as limitCtxt :: _)
                        -> (pos_of_ctxt limitCtxt,col_of_ctxt limitCtxt) 
          in 


          begin match newCtxt with 
          (* Dont bother to check pushes of Vanilla blocks since we've always already pushed a SeqBlock at this position *)
          | CtxtVanilla _ -> ()
          | _ -> 
              let p1,c1 = unindentationLimit true !offsideStack in 
              let c2 = col_of_pos (pos_of_ctxt newCtxt) in 
              if c2 < c1 then 
                  warn tokenTup 
                         (if debug then (Printf.sprintf "possible incorrect indentation: this token is offside of context at position %s, newCtxt = %s, stack = %s, newCtxtPos = %s, c1 = %d, c2 = %d" (string_of_pos p1) (string_of_ctxt newCtxt) (string_of_stack !offsideStack) (string_of_pos (pos_of_ctxt newCtxt)) c1 c2)  
                          else          (Printf.sprintf "possible incorrect indentation: this token is offside of context started at position %s. Try indenting this token further or using standard formatting conventions" (string_of_pos p1))    )
          end;
          let newOffsideStack = newCtxt :: !offsideStack in
          if debug then dprintf1 "--> pushing, stack = %s\n" (string_of_stack newOffsideStack);
          offsideStack := newOffsideStack in

    let popCtxt() = 
        match !offsideStack with 
        |  [] -> ()
        | h :: rest -> 
             if debug then dprintf2 "<-- popping Context(%s), stack = %s\n" (string_of_ctxt h) (string_of_stack rest);
             offsideStack := rest in

    let replaceCtxt p ctxt = popCtxt(); pushCtxt p ctxt in


    (*----------------------------------------------------------------------------
    !* Peek ahead at a token, either from the old lexer or from our delayedStack
     *--------------------------------------------------------------------------*)

    let peekNextTokenTup() = 
        let tokenTup = popNextTokenTup() in 
        delayToken tokenTup; 
        tokenTup in
    
    let peekNextToken() = 
        tokenOfTokenTup (peekNextTokenTup()) in
    
     (*----------------------------------------------------------------------------
     !* Adjacency precedence rule
      *--------------------------------------------------------------------------*)

    let isAdjacent leftTokenTup rightTokenTup =
        let lparenStartPos = startPosOfTokenTup rightTokenTup in
        let tokenEndPos = endPosOfTokenTup leftTokenTup in
        (tokenEndPos = lparenStartPos) in
    
    let nextTokenIsAdjacentLParenOrLBrack tokenTup =
        match peekNextTokenTup() with 
        | ((LPAREN | LBRACK),_,_) as lparenTokenTup-> 
            isAdjacent tokenTup lparenTokenTup
        | _ -> false in 

    let nextTokenIsAdjacent firstTokenTup =
        let nextTokenTup = peekNextTokenTup() in
        isAdjacent firstTokenTup nextTokenTup in

    let infixComparOpIsTyparsClose txt = string_foralli (fun i c -> c = '>' or (c = '.' && i = String.length txt - 1)) txt in
    
    let peekAdjacentTypars indentation tokenTup =
        match peekNextTokenTup() with 
(*
  IMPLEMENTATION FOR DESIGN CHANGE 1600 IF REQUIRED
        | (INFIX_COMPARE_OP "<>",_,_) -> true
*)
        | (LESS,_,_) as lparenTokenTup-> 
            let tokenEndPos = endPosOfTokenTup tokenTup in
            if isAdjacent tokenTup lparenTokenTup then 
                let stack = ref [] in 
                let rec scanAhead nParen = 
                   let lookaheadTokenTup = popNextTokenTup() in 
                   let lookaheadToken = tokenOfTokenTup lookaheadTokenTup in 
                   stack := lookaheadTokenTup :: !stack;
                   let lookaheadTokenStartPos = startPosOfTokenTup lookaheadTokenTup in
                   match lookaheadToken with 
                   | Pars.EOF _ | SEMICOLON_SEMICOLON -> false 
                   | _ when indentation && lookaheadTokenStartPos < tokenEndPos -> false
                   | (RPAREN | RBRACK) ->
                       let nParen = nParen - 1 in 
                       if nParen > 0 then scanAhead nParen else false
                   | GREATER | GREATER_DOT -> 
                       let nParen = nParen - 1 in 
                       if nParen > 0 then scanAhead nParen else true
                   | INFIX_COMPARE_OP txt when infixComparOpIsTyparsClose txt -> 
                       let nParen = nParen - String.length txt + (if txt.[String.length txt-1] = '.' then 1 else 0) in 
                       if nParen > 0 then scanAhead nParen else true
                   | (LPAREN | LESS | LBRACK) -> 
                       scanAhead (nParen+1)
                   | DOLLAR | DOT | UNDERSCORE | IDENT _ | COMMA | STAR | QUOTE  -> scanAhead nParen
                   | _ -> if nParen > 1 then scanAhead nParen else false in 
                let res = scanAhead 0 in 
                !stack |> List.iter (function 
                    | (GREATER_DOT,pos,tokenPrevEndPos) when res -> 
                        delayToken (DOT,pos,tokenPrevEndPos); delayToken (GREATER,pos,tokenPrevEndPos)
                    | (INFIX_COMPARE_OP txt,pos,tokenPrevEndPos) when infixComparOpIsTyparsClose txt -> 
                        for i = String.length txt - 1 downto 0 do
                            let c = txt.[i] in 
                            delayToken ((if c = '.' then DOT else GREATER),pos,tokenPrevEndPos)
                        done
                    | t -> delayToken t);
                res
            else 
                false
        | _ -> false in 

     (*----------------------------------------------------------------------------
     !* End actions
      *--------------------------------------------------------------------------*)

    let returnToken tokenLexbufState tok = 
        setLexbufState(tokenLexbufState);
        prevWasAtomicEnd  := isAtomicExprEndToken(tok);
        Done tok in
              
     (*----------------------------------------------------------------------------
     !* Parse and transform the stream of tokens coming from popNextTokenTup, pushing
      * contexts where needed, popping them where things are offside, balancing
      * parentheses and other constructs.
      *--------------------------------------------------------------------------*)

              
    let rec hwTokenFetch (useBlockRule) =
            let token,tokenLexbufState,tokenPrevEndPos as tokenTup = popNextTokenTup() in 
            let tokenStartPos = (startPosOfTokenTup tokenTup) in
            if debug then dprintf4 "fetch, tokenStartPos = %a, OBLOCKBEGIN=%b, BAR=%b\n" output_pos tokenStartPos (token=OBLOCKBEGIN) (token=BAR); 
            let tokenStartCol = col_of_pos tokenStartPos in 

            let sameLine() = (line_of_pos (startPosOfTokenTup (peekNextTokenTup())) = line_of_pos tokenStartPos) in
            let reprocess() = 
              delayToken tokenTup; 
              hwTokenFetch(useBlockRule) in

            let reprocessWithoutBlockRule() = 
              delayToken tokenTup; 
              hwTokenFetch(false) in
            
            let insertTokenFromPrevPosToCurrentPos(tok) = 
              delayToken tokenTup; 
              (*F# if debug then dprintf2 "inserting %a\n" output_any tok; F#*)
              (* returnToken (lexbufStateForInsertedDummyTokens tokenPrevEndPos) tok in *)
              (* returnToken (lexbufStateForInsertedDummyTokens (fst tokenPrevEndPos, endPosOfTokenTup tokenTup)) tok in *)
              returnToken (lexbufStateForInsertedDummyTokens (startPosOfTokenTup tokenTup, endPosOfTokenTup tokenTup)) tok in

            let insertToken(tok) = 
              delayToken tokenTup; 
              (*F# if debug then dprintf2 "inserting %a\n" output_any tok; F#*)
              (* returnToken (lexbufStateForInsertedDummyTokens tokenPrevEndPos) tok in *)
              returnToken (lexbufStateForInsertedDummyTokens (startPosOfTokenTup tokenTup, endPosOfTokenTup tokenTup)) tok in

            let isSemiSemi = match token with SEMICOLON_SEMICOLON -> true | _ -> false in 

            (* if token = RARROW then Ildiag.dprintf2 "pushing RARROW at %s, !offsideStack = %s\n" (string_of_pos tokenStartPos) (string_of_stack !offsideStack);  *)
              
            let tokenReplaced = rulesForBothSoftWhiteAndHardWhite(tokenTup) in
            if tokenReplaced then hwTokenFetch(useBlockRule) else 

            match token,!offsideStack with 

            (* Balancing rule. Every 'in' terminates all surrounding blocks up to a CtxtLetDecl, and will be swallowed by *)
            (* terminating the corresponding CtxtLetDecl in the rule below. *)
            (* Balancing rule. Every 'done' terminates all surrounding blocks up to a CtxtDo, and will be swallowed by *)
            (* terminating the corresponding CtxtDo in the rule below. *)
            |  (END | SEMICOLON_SEMICOLON | ELSE | ELIF |  DONE |  IN | RPAREN | RBRACE | RBRACK | BAR_RBRACK | WITH | FINALLY | RQUOTE _),  stack
                
                when 
                  ((match stack with [] -> false | _ :: _ -> true) &&
                   match token,stack with 
                   | END, (CtxtWithAsAugment(_)  :: _)
                   | (ELSE | ELIF), (CtxtIf _ :: _)
                   | DONE         , (CtxtDo _ :: _)
                   | IN           , ((CtxtFor _ (* for x in ienum ... *) | CtxtLetDecl _) :: _)
                   (* WITH balances except in the following contexts.... Phew - an overused keyword! *)
                   | WITH         , (  ((CtxtMatch _ | CtxtException _ | CtxtMemberHead _ | CtxtInterfaceHead _ | CtxtTry _ | CtxtType _)  :: _)
                                         (* This is the nasty record/object-expression case *)
                                         | (CtxtSeqBlock _ :: CtxtParen(LBRACE,_)  :: _) )
                   | FINALLY      , (CtxtTry _  :: _) -> false
                   | t2           , (CtxtParen(t1,_) :: _) -> not (parenTokensBalance t1  t2)
                   | _ -> true)
                
                -> 
                let ctxt = List.hd !offsideStack in
                if debug then dprintf4 "IN/ELSE/ELIF/DONE/RPAREN/RBRACE/END at %a terminates context at position %a\n" output_pos tokenStartPos output_pos (pos_of_ctxt ctxt);
                popCtxt();
                (match ctxt with 
                 | CtxtFun _
                 | CtxtMatchClauses _ 
                 | CtxtWithAsLet _       
                         -> (if debug then dprintf0 "--> inserting OEND\n");     
                            insertToken OEND

                 | CtxtWithAsAugment _       
                         -> (if debug then dprintf0 "--> closing WithAsAugment that didn't have an END, inserting ODECLEND\n");     
                            insertToken ODECLEND
                 
                 | CtxtDo _        
                 | CtxtLetDecl (true,_) -> 
                             (if debug then dprintf0 "--> inserting ODECLEND\n"); 
                             insertToken ODECLEND 
                             
                 | CtxtSeqBlock(_,_,AddBlockEnd) ->  
                             (if debug then dprintf0 "--> inserting OBLOCKEND\n"); 
                             insertToken OBLOCKEND 

                 | CtxtSeqBlock(_,_,AddOneSidedBlockEnd) ->  
                             (if debug then dprintf0 "--> inserting ORIGHT_BLOCK_END\n"); 
                             insertToken ORIGHT_BLOCK_END 

                 
                 | _                -> reprocess())

            (* reset on ';;' rule. A ';;' terminates ALL entries *)
            |  SEMICOLON_SEMICOLON, []  -> 
                if debug then dprintf0 ";; scheduling a reset\n";
                delayToken(ORESET,tokenLexbufState,tokenPrevEndPos);
                returnToken tokenLexbufState SEMICOLON_SEMICOLON
            |  ORESET, []  -> 
                if debug then dprintf0 "performing a reset after a ;; has been swallowed\n";
                (* NOTE: The parser thread of F# Interactive will often be blocked on this call, e.g. after an entry has been *)
                (* processed and we're waiting for the first token of the next entry. *)
                peekInitial();
                hwTokenFetch(true) 


            (* Balancing rule. Encountering an 'in' balances with a 'let'. i.e. even a non-offside 'in' closes a 'let' *)
            (* The 'IN' token is thrown away and becomes an ODECLEND *)
            |  IN, (CtxtLetDecl (blockLet,offsidePos) :: _) -> 
                if debug then dprintf3 "IN at %a (becomes %s)\n" output_pos tokenStartPos (if blockLet then "ODECLEND" else "IN");
                if tokenStartCol < col_of_pos offsidePos then warn tokenTup "the indentation of this 'in' token is incorrect with respect to the corresponding 'let'";
                popCtxt();
                delayToken(ODUMMY(token),tokenLexbufState,tokenPrevEndPos); (* make sure we queue a dummy token at this position to check if any other pop rules apply*)
                returnToken tokenLexbufState (if blockLet then ODECLEND else token)

            (* Balancing rule. Encountering a 'done' balances with a 'do'. i.e. even a non-offside 'done' closes a 'do' *)
            (* The 'DONE' token is thrown away and becomes an ODECLEND *)
            |  DONE, (CtxtDo offsidePos :: _) -> 
                if debug then dprintf4 "DONE at %a terminates CtxtDo(offsidePos=%a)\n" output_pos tokenStartPos output_pos offsidePos;
                popCtxt();
                (* reprocess as the DONE may close a DO context *)
                delayToken(ODECLEND,tokenLexbufState,tokenPrevEndPos); 
                hwTokenFetch(useBlockRule)

            (* Balancing rule. Encountering a ')' or '}' balances with a '(' or '{', even if not offside *)
            |  ((END | RPAREN | RBRACE | RBRACK | BAR_RBRACK | RQUOTE _) as t2), (CtxtParen (t1,_) :: _) 
                    when parenTokensBalance t1 t2  ->
                if debug then dprintf2 "RPAREN/RBRACE/RBRACK/BAR_RBRACK/RQUOTE/END at %a terminates CtxtParen()\n" output_pos tokenStartPos;
                popCtxt();
                delayToken(ODUMMY(token),tokenLexbufState,tokenPrevEndPos); (* make sure we queue a dummy token at this position to check if any closing rules apply*)
                returnToken tokenLexbufState token

            (* Balancing rule. Encountering a 'end' can balance with a 'with' but only when not offside *)
            |  END, (CtxtWithAsAugment(offsidePos) :: _) 
                       when not (tokenStartCol + 1 <= col_of_pos offsidePos) -> 
                if debug then dprintf2 "END at %a terminates CtxtWithAsAugment()\n" output_pos tokenStartPos;
                popCtxt();
                delayToken(ODUMMY(token),tokenLexbufState,tokenPrevEndPos); (* make sure we queue a dummy token at this position to check if any closing rules apply*)
                returnToken tokenLexbufState OEND

            (*  Balancing rule. CtxtNamespaceHead ~~~> CtxtSeqBlock *)
            (*  Applied when a token other then a long identifier is seen *)
            | _, (CtxtNamespaceHead _ :: _) 
                when not (isLongIdentifier token) -> 
                 if debug then dprintf0 "CtxtNamespaceHead: EQUALS, pushing CtxtSeqBlock\n";
                 popCtxt();
                 pushCtxtSeqBlockAt(tokenTup,false,NoAddBlockEnd);
                 reprocess()

            (*  Balancing rule. CtxtModuleHead ~~~> CtxtSeqBlock *)
            (*  Applied when a ':' or '=' token is seen *)
            (*  Otherwise it's a 'head' module declaration, so ignore it *)
            | _, (CtxtModuleHead offsidePos :: _) 
                when not (isLongIdentifier token) && not (match token with PUBLIC | PRIVATE | INTERNAL -> true | _ -> false) -> 
                 if (match token with COLON | EQUALS -> true | _ -> false) then begin
                   if debug then dprintf0 "CtxtModuleHead: COLON/EQUALS, pushing CtxtModuleBody and CtxtSeqBlock\n";
                   popCtxt();
                   pushCtxt tokenTup (CtxtModuleBody offsidePos);
                   pushCtxtSeqBlock(true,AddBlockEnd);
                   returnToken tokenLexbufState token
                 end else begin
                   popCtxt();
                   reprocessWithoutBlockRule()
                 end

            (*  Offside rule for SeqBlock.  
                f x
                g x
              ...
            *)
            | _, (CtxtSeqBlock(_,offsidePos,addBlockEnd) :: rest) when 
                            
                    isSemiSemi or 
                        let grace = 
                            match token, rest with 
                             (* When in a type context allow a grace of 2 column positions for '|' tokens, permits 
                                 type x = 
                                     A of string    <-- note missing '|' here - bad style, and perhaps should be disallowed
                                   | B of int *)
                                      
                            | BAR, (CtxtType _ :: _) -> 2

                             (* This ensures we close a type context seq block when the '|' marks
                                of a type definition are aligned with the 'type' token. This lack of
                                indentation is fundamentally perverse and should probably not be allowed,
                                but occurs in Foundations of F# and in quite a lot of user code.
                                
                                 type x = 
                                 | A 
                                 | B 
                                 
                                 <TOKEN>    <-- close the type context sequence block here *)

                            | _, (CtxtType posType :: _) when col_of_pos offsidePos = col_of_pos posType && not (isTypeSeqBlockElementContinuator token) -> -1


                            | _ -> 
                               (* Allow a grace of 3 column positions for infix tokens, permits 
                                   let x =           
                                         expr + expr 
                                       + expr + expr 
                                  And   
                                     let x =           
                                           expr  
                                        |> f expr 
                                        |> f expr  
                                  Note you need a semicolon in the following situation:

                                   let x =           
                                         stmt
                                        -expr     <-- not allowed, as prefix token is here considered infix

                                  i.e.

                                   let x =           
                                         stmt;
                                         -expr     
                            *)
                                (if isInfix token then 3 else 0) in
                        (tokenStartCol + grace < col_of_pos offsidePos) -> 
               if debug then dprintf3 "offside token at column %d indicates end of CtxtSeqBlock started at %a!\n" tokenStartCol output_pos offsidePos;
               popCtxt();
               if debug then (match addBlockEnd with AddBlockEnd -> dprintf0 "end of CtxtSeqBlock, insert OBLOCKEND \n" | _ -> ()) ;
               (match addBlockEnd with 
                | AddBlockEnd -> insertToken(OBLOCKEND) 
                | AddOneSidedBlockEnd -> insertToken(ORIGHT_BLOCK_END) 
                | NoAddBlockEnd -> reprocess() )

            (*  Offside rule for SeqBlock.
                  fff
                     eeeee
                <tok>
            *)
            | _, (CtxtVanilla(offsidePos) :: _) when isSemiSemi or tokenStartCol <= col_of_pos offsidePos -> 
               if debug then dprintf3 "offside token at column %d indicates end of CtxtVanilla started at %a!\n" tokenStartCol output_pos offsidePos;
               popCtxt();
               reprocess()

            (*  Offside rule for SeqBlock - special case
                [< ... >]
                decl
            *)

            | _, (CtxtSeqBlock(NotFirstInSeqBlock,offsidePos,addBlockEnd) :: _) 
                     when (match token with GREATER_RBRACK -> true | _ -> false) -> 
               (* attribute-end tokens mean CtxtSeqBlock rule is NOT applied to the next token, *)
               replaceCtxt tokenTup (CtxtSeqBlock (FirstInSeqBlock,offsidePos,addBlockEnd));
               reprocessWithoutBlockRule()

            (*  Offside rule for SeqBlock - avoiding inserting OBLOCKSEP on first item in block
            *)

            | _, (CtxtSeqBlock (FirstInSeqBlock,offsidePos,addBlockEnd) :: _) when useBlockRule -> 
               (* This is the first token in a block, or a token immediately *)
               (* following an infix operator (see above). *)
               (* Return the token, but only after processing any additional rules *)
               (* applicable for this token.  Don't apply the CtxtSeqBlock rule for *)
               (* this token, but do apply it on subsequent tokens. *)
               if debug then dprintf0 "repull for CtxtSeqBlockStart\n" ;
               replaceCtxt tokenTup (CtxtSeqBlock (NotFirstInSeqBlock,offsidePos,addBlockEnd));
               reprocessWithoutBlockRule()

            (*  Offside rule for SeqBlock - inserting OBLOCKSEP on subsequent items in a block when they are precisely aligned

               let f1 () = 
                  expr
                  ...
               ~~> insert OBLOCKSEP
           
               let f1 () = 
                  let x = expr
                  ...
               ~~> insert OBLOCKSEP
             
               let f1 () = 
                  let x1 = expr
                  let x2 = expr
                  let x3 = expr
                  ...
               ~~> insert OBLOCKSEP
            *)
            | _, (CtxtSeqBlock (NotFirstInSeqBlock,offsidePos,addBlockEnd) :: rest) 
                   when  useBlockRule 
                      && not (let isTypeCtxt = (match rest with 
                                               | (CtxtType _ :: _) -> true
                                               | _ -> false) in
                              if isTypeCtxt then isTypeSeqBlockElementContinuator token
                              else isSeqBlockElementContinuator  token)
                      && (tokenStartCol = col_of_pos offsidePos) 
                      && (line_of_pos tokenStartPos <> line_of_pos offsidePos) -> 
                 if debug then dprintf3 "offside at column %d matches start of block(%a)! delaying token, returning OBLOCKSEP\n" tokenStartCol output_pos offsidePos;
                 replaceCtxt tokenTup (CtxtSeqBlock (FirstInSeqBlock,offsidePos,addBlockEnd));
                 (* no change to offside stack: another statement block starts *)
                 insertTokenFromPrevPosToCurrentPos OBLOCKSEP

            (*  Offside rule for CtxtLetDecl *)
            (* let .... = 
                  ...
               <and>
            *)
            (* let .... = 
                  ...
               <in>
            *)
            (*   let .... =
                     ...
                <*>
            *)
            | _, (CtxtLetDecl (true,offsidePos) :: _) when 
                          isSemiSemi or (if isLetContinuator token then tokenStartCol + 1 else tokenStartCol) <= col_of_pos offsidePos -> 
               if debug then dprintf3 "token at column %d is offside from LET(offsidePos=%a)! delaying token, returning ODECLEND\n" tokenStartCol output_pos offsidePos;
               popCtxt();
               insertToken ODECLEND

            | _, (CtxtDo offsidePos :: _) 
                   when isSemiSemi or (if isDoContinuator token then tokenStartCol + 1 else tokenStartCol) <= col_of_pos offsidePos -> 
               if debug then dprintf3 "token at column %d is offside from DO(offsidePos=%a)! delaying token, returning ODECLEND\n" tokenStartCol output_pos offsidePos;
               popCtxt();
               insertToken ODECLEND

            (* class
                  interface AAA
                ...
               ...
               
            *)

            | _, (CtxtInterfaceHead offsidePos :: _) 
                   when isSemiSemi or (if isInterfaceContinuator token then tokenStartCol + 1 else tokenStartCol) <= col_of_pos offsidePos -> 
               if debug then dprintf3 "token at column %d is offside from INTERFACE(offsidePos=%a)! pop and reprocess\n" tokenStartCol output_pos offsidePos;
               popCtxt();
               reprocess()

            | _, (CtxtType offsidePos :: _) 
                   when isSemiSemi or 
                      ((if isTypeContinuator token then tokenStartCol + 1 else tokenStartCol) <= col_of_pos offsidePos) -> 
               if debug then dprintf3 "token at column %d is offside from TYPE(offsidePos=%a)! pop and reprocess\n" tokenStartCol output_pos offsidePos;
               popCtxt();
               reprocess()

            (* module M = ...
               end
            *)
            (*  module M = ...
               ...
            *)
            | _, ((CtxtModuleBody offsidePos) :: _) when isSemiSemi or tokenStartCol <= col_of_pos offsidePos -> 
               if debug then dprintf3 "token at column %d is offside from MODULE with offsidePos %a! delaying token\n" tokenStartCol output_pos offsidePos;
               popCtxt();
               reprocess()

            | _, ((CtxtException offsidePos) :: _) when isSemiSemi or tokenStartCol <= col_of_pos offsidePos -> 
               if debug then dprintf3 "token at column %d is offside from EXCEPTION with offsidePos %a! delaying token\n" tokenStartCol output_pos offsidePos;
               popCtxt();
               reprocess()

            (* Pop CtxtMemberBody when offside.  Insert an ODECLEND to indicate the end of the member *)
            | _, ((CtxtMemberBody(offsidePos)) :: _) when isSemiSemi or tokenStartCol <= col_of_pos offsidePos -> 
               if debug then dprintf3 "token at column %d is offside from MEMBER/OVERRIDE head with offsidePos %a!\n" tokenStartCol output_pos offsidePos;
               popCtxt();
               insertToken ODECLEND
               (* hwTokenFetch(useBlockRule) *)

            (* Pop CtxtMemberHead when offside *)
            | _, ((CtxtMemberHead(offsidePos)) :: _) when isSemiSemi or tokenStartCol <= col_of_pos offsidePos -> 
               if debug then dprintf3 "token at column %d is offside from MEMBER/OVERRIDE head with offsidePos %a!\n" tokenStartCol output_pos offsidePos;
               popCtxt();
               reprocess()

            | _, (CtxtIf offsidePos :: _) 
                       when isSemiSemi or (if isIfBlockContinuator token then  tokenStartCol + 1 else tokenStartCol) <= col_of_pos offsidePos -> 
               if debug then dprintf0 "offside from CtxtIf\n";
               popCtxt();
               reprocess()
                
            | _, (CtxtWithAsLet offsidePos :: _) 
                       when isSemiSemi or (if isLetContinuator token then  tokenStartCol + 1 else tokenStartCol) <= col_of_pos offsidePos -> 
               if debug then dprintf0 "offside from CtxtWithAsLet\n";
               popCtxt();
               insertToken OEND
                
            | _, (CtxtWithAsAugment(offsidePos) :: _) 
                       when isSemiSemi or (if isWithAugmentBlockContinuator token then tokenStartCol + 1  else tokenStartCol) <= col_of_pos offsidePos -> 
               if debug then dprintf1 "offside from CtxtWithAsAugment, isWithAugmentBlockContinuator = %b\n" (isWithAugmentBlockContinuator token);
               popCtxt();
               insertToken ODECLEND 
                
            | _, (CtxtMatch offsidePos :: _) 
                       when isSemiSemi or tokenStartCol <= col_of_pos offsidePos -> 
               if debug then dprintf0 "offside from CtxtMatch\n";
               popCtxt();
               reprocess()
                
            | _, (CtxtFor offsidePos :: _) 
                       when isSemiSemi or (if isForLoopContinuator token then  tokenStartCol + 1 else tokenStartCol) <= col_of_pos offsidePos -> 
               if debug then dprintf0 "offside from CtxtFor\n";
               popCtxt();
               reprocess()
                
            | _, (CtxtWhile offsidePos :: _) 
                       when isSemiSemi or (if isWhileBlockContinuator token then  tokenStartCol + 1 else tokenStartCol) <= col_of_pos offsidePos -> 
               if debug then dprintf0 "offside from CtxtWhile\n";
               popCtxt();
               reprocess()
                
            | _, (CtxtWhen offsidePos :: _) 
                       when isSemiSemi or tokenStartCol <= col_of_pos offsidePos -> 
               if debug then dprintf0 "offside from CtxtWhen\n";
               popCtxt();
               reprocess()
                
            | _, (CtxtFun offsidePos :: _) 
                       when isSemiSemi or tokenStartCol <= col_of_pos offsidePos -> 
               if debug then dprintf0 "offside from CtxtFun\n";
               popCtxt();
               insertToken OEND
                
             | _, (CtxtFunction offsidePos :: _) 
                       when isSemiSemi or tokenStartCol <= col_of_pos offsidePos -> 
               popCtxt();
               reprocess()
                
           | _, (CtxtTry offsidePos :: _) 
                       when isSemiSemi or (if isTryBlockContinuator token then  tokenStartCol + 1 else tokenStartCol) <= col_of_pos offsidePos -> 
               if debug then dprintf0 "offside from CtxtTry\n";
               popCtxt();
               reprocess()
                
            (*  then 
                   ...
                else  
            *)
            (*  then 
                   ...
            *)
            | _, (CtxtThen offsidePos :: _) when isSemiSemi or  (if isThenBlockContinuator token then  tokenStartCol + 1 else tokenStartCol)<= col_of_pos offsidePos -> 
               if debug then dprintf0 "offside from CtxtThen\n";
               popCtxt();
               reprocess()
                
            (*  else ...
               ....
            *)
            | _, (CtxtElse (offsidePos) :: _) when isSemiSemi or tokenStartCol <= col_of_pos offsidePos -> 
               if debug then dprintf0 "offside from CtxtElse\n";
               popCtxt();
               reprocess()

            | _, (CtxtMatchClauses (leadingBar,offsidePos) :: _) 
                       (* leadingBar=false represents a bit of a hack to permit match patterns without an initial '|' *)
                       when isSemiSemi or 
                            (match token with 
                             (* BAR occurs in pattern matching 'with' blocks *)
                             | BAR -> tokenStartCol + (if leadingBar then 1 else 2)  < col_of_pos offsidePos
                             | END -> tokenStartCol + (if leadingBar then -1 else 1) < col_of_pos offsidePos
                             | _   -> tokenStartCol + (if leadingBar then -1 else 1) < col_of_pos offsidePos) -> 
                if debug then dprintf3 "offside from WITH, tokenStartCol = %d, offsidePos = %a, delaying token, returning OEND\n" tokenStartCol output_pos offsidePos;
                popCtxt();
                insertToken OEND
                

            (*  namespace ... ~~~> CtxtNamespaceHead *)
            |  NAMESPACE,(_ :: _) -> 
                if debug then dprintf0 "NAMESPACE: entering CtxtNamespaceHead, awaiting end of long identifier to push CtxtSeqBlock\n" ;
                pushCtxt tokenTup (CtxtNamespaceHead tokenStartPos);
                returnToken tokenLexbufState token
                
            (*  module ... ~~~> CtxtModuleHead *)
            |  MODULE,(_ :: _) -> 
                if debug then dprintf2 "MODULE: entering CtxtModuleHead, awaiting EQUALS to go to CtxtSeqBlock (%a)\n" output_pos tokenStartPos;
                pushCtxt tokenTup (CtxtModuleHead tokenStartPos);
                returnToken tokenLexbufState token
                
            (*  exception ... ~~~> CtxtException *)
            |  EXCEPTION,(_ :: _) -> 
                if debug then dprintf2 "EXCEPTION: entering CtxtException(%a)\n" output_pos tokenStartPos;
                pushCtxt tokenTup (CtxtException tokenStartPos);
                returnToken tokenLexbufState token
                
            (*  let ... ~~~> CtxtLetDecl *)
            (*     -- this rule only applies to *)
            (*              - 'let' 'right-on' a SeqBlock line *)
            (*              - 'static let' *)
            | LET(isUse), (ctxt :: _) -> 
                let blockLet = match ctxt with CtxtSeqBlock _ |  CtxtMemberHead _ -> true | _ -> false in
                if debug then dprintf3 "LET: entering CtxtLetDecl(blockLet=%b), awaiting EQUALS to go to CtxtSeqBlock (%a)\n" blockLet output_pos tokenStartPos;
                pushCtxt tokenTup (CtxtLetDecl(blockLet,tokenStartPos));
                returnToken tokenLexbufState (if blockLet then OLET(isUse) else token)
                
            (*  let!  ... ~~~> CtxtLetDecl *)
            | BINDER b, (ctxt :: _) -> 
                let blockLet = match ctxt with CtxtSeqBlock _ -> true | _ -> false in
                if debug then dprintf3 "LET: entering CtxtLetDecl(blockLet=%b), awaiting EQUALS to go to CtxtSeqBlock (%a)\n" blockLet output_pos tokenStartPos;
                pushCtxt tokenTup (CtxtLetDecl(blockLet,tokenStartPos));
                returnToken tokenLexbufState (if blockLet then OBINDER b else token)
                
            (*  static member ... ~~~> CtxtMemberHead *)
            (*  static ... ~~~> CtxtMemberHead *)
            (*  member ... ~~~> CtxtMemberHead *)
            (*  override ... ~~~> CtxtMemberHead *)
            (*  default ... ~~~> CtxtMemberHead *)
            |  (STATIC | ABSTRACT | MEMBER | OVERRIDE | DEFAULT),(ctxt :: _) when (match ctxt with CtxtMemberHead _ -> false | _ -> true) -> 
                if debug then dprintf2 "STATIC/MEMBER/OVERRIDE/DEFAULT: entering CtxtMemberHead, awaiting EQUALS to go to CtxtSeqBlock (%a)\n" output_pos tokenStartPos;
                pushCtxt tokenTup (CtxtMemberHead(tokenStartPos));
                returnToken tokenLexbufState token
                
            (*  new( ~~~> CtxtMemberHead *)
            | NEW,_  when (match peekNextToken() with LPAREN -> true | _ -> false)   -> 
                if debug then dprintf2 "NEW: entering CtxtMemberHead, awaiting EQUALS to go to CtxtSeqBlock (%a)\n" output_pos tokenStartPos;
                pushCtxt tokenTup (CtxtMemberHead(tokenStartPos));
                returnToken tokenLexbufState token
                                     
            (*  'let ... = ' ~~~> CtxtSeqBlock *)
            | EQUALS, (CtxtLetDecl _ :: _) ->  
                if debug then dprintf0 "CtxtLetDecl: EQUALS, pushing CtxtSeqBlock\n";
                pushCtxtSeqBlock(true,AddBlockEnd);
                returnToken tokenLexbufState token

            | EQUALS, (CtxtType _ :: _) ->  
                if debug then dprintf0 "CtxType: EQUALS, pushing CtxtSeqBlock\n";
                pushCtxtSeqBlock(true,AddBlockEnd);
                returnToken tokenLexbufState token

            | LAZY, _ ->  
                if debug then dprintf0 "LAZY, pushing CtxtSeqBlock\n";
                pushCtxtSeqBlock(false,NoAddBlockEnd);
                returnToken tokenLexbufState token

            (*  'with ... = ' ~~~> CtxtSeqBlock *)
            (* We don't insert begin/end block tokens here since we can't properly distinguish single-line *)
            (* OCaml-style record update expressions such as "{ t with gbuckets=Array.copy t.gbuckets; gcount=t.gcount }" *)
            (* These have a syntactically odd status because of the use of ";" to terminate expressions, so each *)
            (* "=" binding is not properly balanced by "in" or "and" tokens in the single line syntax (unlike other bindings) *)
            (* REVIEW: However we should be able to insert an OBLOCKEND in the offside case, e.g. 
                     { t with field1 = f
                               x
                              field2 = f 
                               y }
                 correctly reports an error because the "x" is offside from the CtxtSeqBlock started by the first equation. *)
            | EQUALS, ((CtxtWithAsLet _) :: _) ->  
                if debug then dprintf0 "CtxtLetDecl/CtxtWithAsLet: EQUALS, pushing CtxtSeqBlock\n";
                pushCtxtSeqBlock(false,NoAddBlockEnd);
                returnToken tokenLexbufState token

            (*  'new(... =' ~~~> CtxtMemberBody, CtxtSeqBlock *)
            (*  'member ... =' ~~~> CtxtMemberBody, CtxtSeqBlock *)
            (*  'static member ... =' ~~~> CtxtMemberBody, CtxtSeqBlock *)
            (*  'default ... =' ~~~> CtxtMemberBody, CtxtSeqBlock *)
            (*  'override ... =' ~~~> CtxtMemberBody, CtxtSeqBlock *)
            | EQUALS, ((CtxtMemberHead(offsidePos)) :: _) ->  
                if debug then dprintf0 "CtxtMemberHead: EQUALS, pushing CtxtSeqBlock\n";
                replaceCtxt tokenTup (CtxtMemberBody (offsidePos));
                pushCtxtSeqBlock(true,AddBlockEnd);
                returnToken tokenLexbufState token

            (* '(' tokens are balanced with ')' tokens and also introduce a CtxtSeqBlock *)
            | (BEGIN | LPAREN | SIG | LBRACE | LBRACK | LBRACK_BAR | LQUOTE _), _ ->                      
                if debug then dprintf2 "LPAREN etc., pushes CtxtParen, pushing CtxtSeqBlock, tokenStartPos = %a\n" output_pos tokenStartPos;
                pushCtxt tokenTup (CtxtParen (token,tokenStartPos));
                pushCtxtSeqBlock(false,NoAddBlockEnd);
                returnToken tokenLexbufState token

            (* '(' tokens are balanced with ')' tokens and also introduce a CtxtSeqBlock *)
            | STRUCT, ctxts                       
                   when match ctxts with 
                        | CtxtSeqBlock _ :: (CtxtModuleBody _ | CtxtType _) :: _ -> 
                                (* type ... = struct ... end *)
                                (* module ... = struct ... end *)
                            true 
                             
                        | _ -> false (* type X<'a when 'a : struct> *) ->
                if debug then dprintf2 "LPAREN etc., pushes CtxtParen, pushing CtxtSeqBlock, tokenStartPos = %a\n" output_pos tokenStartPos;
                pushCtxt tokenTup (CtxtParen (token,tokenStartPos));
                pushCtxtSeqBlock(false,NoAddBlockEnd);
                returnToken tokenLexbufState token

            | (RARROW | RARROW2), ctxts 
                   (* Only treat '->' as a sequence block in certain circumstances *)
                   (* Only treat '->' as a sequence block in certain circumstances *)
                   when let rec check = function (CtxtWhile _ | CtxtFor _ | CtxtWhen _ | CtxtMatchClauses _ | CtxtFun _) :: _ -> true (* comprehension/match *)
                                               | (CtxtSeqBlock _ :: CtxtParen ((LBRACK | LBRACE | LBRACK_BAR), _) :: _) -> true  (* comprehension *)
                                               | (CtxtSeqBlock _ :: (CtxtDo _ | CtxtWhile _ | CtxtFor _ | CtxtWhen _ | CtxtMatchClauses _  | CtxtTry _ | CtxtThen _ | CtxtElse _) :: _) -> true (* comprehension *)
                                               (* | (((* CtxtWhen _ | *) CtxtFor _) :: rest) -> check rest (* comprehension *) *)
                                               | _ -> false in
                        check ctxts ->
                if debug then dprintf2 "RARROW/RARROW2, pushing CtxtSeqBlock, tokenStartPos = %a\n" output_pos tokenStartPos;
                pushCtxtSeqBlock(false,AddOneSidedBlockEnd);
                returnToken tokenLexbufState token

(*
            | LARROW, ctxts  ->
                if debug then dprintf2 "LARROW, pushing CtxtSeqBlock, tokenStartPos = %a\n" output_pos tokenStartPos;
                pushCtxtSeqBlock(true,AddBlockEnd);
                returnToken tokenLexbufState token
*)

            (*  do  ~~> CtxtDo;CtxtSeqBlock  (unconditionally) *)
            | (DO | DO_BANG), _ -> 
                if debug then dprintf2 "DO: pushing CtxtSeqBlock, tokenStartPos = %a\n" output_pos tokenStartPos;
                pushCtxt tokenTup (CtxtDo (tokenStartPos));
                pushCtxtSeqBlock(true,AddBlockEnd);
                returnToken tokenLexbufState (match token with DO -> ODO | DO_BANG -> ODO_BANG | _ -> failwith "unreachable")

            (* The r.h.s. of an infix token begins a new block *)
            | _,_ when isInfix token && not (sameLine()) -> 
                if debug then dprintf2 "(Infix etc.), pushing CtxtSeqBlock, tokenStartPos = %a\n" output_pos tokenStartPos;
                pushCtxtSeqBlock(false,NoAddBlockEnd);
                returnToken tokenLexbufState token

            | WITH, ((CtxtTry _ | CtxtMatch _) :: _)  -> 
                let lookaheadTokenTup = peekNextTokenTup() in 
                let lookaheadTokenStartPos = startPosOfTokenTup lookaheadTokenTup in 
                let leadingBar = match (peekNextToken()) with BAR -> true | _ -> false in
                if debug then dprintf4 "WITH, pushing CtxtMatchClauses, lookaheadTokenStartPos = %a, tokenStartPos = %a\n" output_pos lookaheadTokenStartPos output_pos tokenStartPos;
                pushCtxt lookaheadTokenTup (CtxtMatchClauses(leadingBar,lookaheadTokenStartPos));
                returnToken tokenLexbufState OWITH 

            | FINALLY, (CtxtTry _ :: _)  -> 
                let leadingBar = match (peekNextToken()) with BAR -> true | _ -> false in
                if debug then dprintf2 "FINALLY, pushing pushCtxtSeqBlock, tokenStartPos = %a\n" output_pos tokenStartPos;
                pushCtxtSeqBlock(true,AddBlockEnd);
                returnToken tokenLexbufState token

            | WITH, (((CtxtException _ | CtxtType _ | CtxtMemberHead _ | CtxtInterfaceHead _) as limCtxt) :: _) 
            | WITH, ((CtxtSeqBlock _) as limCtxt :: CtxtParen(LBRACE,_) :: _)  -> 
                let (lookaheadToken,_,_) as lookaheadTokenTup = peekNextTokenTup() in 
                let lookaheadTokenStartPos = startPosOfTokenTup lookaheadTokenTup in 
                begin match lookaheadToken with 

                | IDENT _ 
                (* The next clause detects the access annotations after the 'with' in:
                      member  x.PublicGetSetProperty 
                                   with public get i = "Ralf"
                                   and  private set i v = ()  
                   *)
                | PUBLIC | PRIVATE | INTERNAL -> 

                    let offsidePos = 
                       if col_of_pos lookaheadTokenStartPos > col_of_pos (endPosOfTokenTup tokenTup) then
                            (* This detects:
                                  { new Foo 
                                    with M() = 1
                                    and  N() = 2 } 
                               and treats the inner bindings as if they were member bindings. 
                               It also happens to detect
                                  { foo with m = 1;
                                             n = 2 }
                               So we're careful to set the offside column to be the minimum required *)
                          tokenStartPos
                        else
                            (* This detects:
                                  { foo with 
                                      m = 1;
                                      n = 2 }
                               So we're careful to set the offside column to be the minimum required *)
                          pos_of_ctxt limCtxt in
                    if debug then dprintf4 "WITH, pushing CtxtWithAsLet, tokenStartPos = %a, lookaheadTokenStartPos = %a\n" output_pos tokenStartPos output_pos lookaheadTokenStartPos;
                    pushCtxt tokenTup (CtxtWithAsLet(offsidePos));
                    returnToken tokenLexbufState OWITH 
                | _ -> 
                    if debug then dprintf3 "WITH, pushing CtxtWithAsAugment and CtxtSeqBlock, tokenStartPos = %a, limCtxt = %s\n" output_pos tokenStartPos (string_of_ctxt limCtxt);

                    (* In these situations
                          interface I with 
                              ...
                          end
                          exception ... with 
                              ...
                          end
                          type ... with 
                              ...
                          end
                          member x.P 
                             with get() = ...
                             and  set() = ...
                          member x.P with 
                              get() = ...
                       The limit is "interface"/"exception"/"type" *)
                    let offsidePos = pos_of_ctxt limCtxt in
                       
                    pushCtxt tokenTup (CtxtWithAsAugment(offsidePos));
                    pushCtxtSeqBlock(true,AddBlockEnd);
                    returnToken tokenLexbufState token 
                end;

            | WITH, stack  -> 
                dprintf1 "WITH, hd stack = %s\n" (string_of_ctxt (List.hd stack));
                if debug then dprintf1 "WITH --> NO MATCH, pushing CtxtWithAsAugment (type augmentation), stack = %s" (string_of_stack stack);
                pushCtxt tokenTup (CtxtWithAsAugment(tokenStartPos));
                pushCtxtSeqBlock(true,AddBlockEnd);
                returnToken tokenLexbufState token 

            | FUNCTION, _  -> 
                let lookaheadTokenTup = peekNextTokenTup() in 
                let lookaheadTokenStartPos = startPosOfTokenTup lookaheadTokenTup in 
                let leadingBar = match (peekNextToken()) with BAR -> true | _ -> false in
                pushCtxt tokenTup (CtxtFunction(tokenStartPos));
                pushCtxt lookaheadTokenTup (CtxtMatchClauses(leadingBar,lookaheadTokenStartPos));
                returnToken tokenLexbufState OFUNCTION

            | THEN,_  -> 
                if debug then dprintf2 "THEN, replacing THEN with OTHEN, pushing CtxtSeqBlock;CtxtThen(%a)\n" output_pos tokenStartPos;
                pushCtxt tokenTup (CtxtThen(tokenStartPos));
                pushCtxtSeqBlock(true,AddBlockEnd);
                returnToken tokenLexbufState OTHEN 

            | ELSE, _   -> 
                let lookaheadTokenTup = peekNextTokenTup() in 
                let lookaheadTokenStartPos = startPosOfTokenTup lookaheadTokenTup in 
                begin match peekNextToken() with 
                | IF when sameLine() ->
                  (* EXCEPTION-TO-THE-RULE: We convert ELSE IF to ELIF since it then opens the block at the right point, *)
                  (* In particular the case
                        if e1 then e2
                        else if e3 then e4
                        else if e5 then e6 *)
                  let _ = popNextTokenTup() in 
                  if debug then dprintf2 "ELSE IF: replacing ELSE IF with ELIF, pushing CtxtIf, CtxtVanilla(%a)\n" output_pos tokenStartPos;
                  pushCtxt tokenTup (CtxtIf(tokenStartPos));
                  returnToken tokenLexbufState ELIF
                  
                | _ -> 
                  if debug then dprintf2 "ELSE: replacing ELSE with OELSE, pushing CtxtSeqBlock, CtxtElse(%a)\n" output_pos lookaheadTokenStartPos;
                  pushCtxt tokenTup (CtxtElse(tokenStartPos));
                  pushCtxtSeqBlock(true,AddBlockEnd);
                  returnToken tokenLexbufState OELSE
                end

            | (ELIF | IF), _   -> 
                if debug then dprintf2 "IF, pushing CtxtIf(%a)\n" output_pos tokenStartPos;
                pushCtxt tokenTup (CtxtIf (tokenStartPos));
                returnToken tokenLexbufState token

            | MATCH, _   -> 
                if debug then dprintf2 "MATCH, pushing CtxtMatch(%a)\n" output_pos tokenStartPos;
                pushCtxt tokenTup (CtxtMatch (tokenStartPos));
                returnToken tokenLexbufState token

            | FOR, _   -> 
                if debug then dprintf2 "FOR, pushing CtxtFor(%a)\n" output_pos tokenStartPos;
                pushCtxt tokenTup (CtxtFor (tokenStartPos));
                returnToken tokenLexbufState token

            | WHILE, _   -> 
                if debug then dprintf2 "WHILE, pushing CtxtWhile(%a)\n" output_pos tokenStartPos;
                pushCtxt tokenTup (CtxtWhile (tokenStartPos));
                returnToken tokenLexbufState token

            | WHEN, ((CtxtSeqBlock _) :: _)  -> 
                if debug then dprintf2 "WHEN, pushing CtxtWhen(%a)\n" output_pos tokenStartPos;
                pushCtxt tokenTup (CtxtWhen (tokenStartPos));
                returnToken tokenLexbufState token

            | FUN, _   -> 
                if debug then dprintf2 "FUN, pushing CtxtFun(%a)\n" output_pos tokenStartPos;
                pushCtxt tokenTup (CtxtFun (tokenStartPos));
                returnToken tokenLexbufState OFUN

            | INTERFACE, _  -> 
                let (lookaheadToken,_,_) as lookaheadTokenTup = peekNextTokenTup() in 
                let lookaheadTokenStartPos = startPosOfTokenTup lookaheadTokenTup in 
                begin match lookaheadToken with 
                (* type I = interface .... end *)
                | DEFAULT | OVERRIDE | INTERFACE | NEW | TYPE | STATIC | END | MEMBER | ABSTRACT  | INHERIT | LBRACK_LESS -> 
                    if debug then dprintf4 "INTERFACE, pushing CtxtParen, tokenStartPos = %a, lookaheadTokenStartPos = %a\n" output_pos tokenStartPos output_pos lookaheadTokenStartPos;
                    pushCtxt tokenTup (CtxtParen (token,tokenStartPos));
                    pushCtxtSeqBlock(true,AddBlockEnd);
                    returnToken tokenLexbufState token
                (* type C with interface .... with *)
                (* type C = interface .... with *)
                | _ -> 
                    if debug then dprintf4 "INTERFACE, pushing CtxtInterfaceHead, tokenStartPos = %a, lookaheadTokenStartPos = %a\n" output_pos tokenStartPos output_pos lookaheadTokenStartPos;
                    pushCtxt tokenTup (CtxtInterfaceHead(tokenStartPos));
                    returnToken tokenLexbufState OINTERFACE_MEMBER
                end;

            | CLASS, _   -> 
                if debug then dprintf2 "CLASS, pushing CtxtParen(%a)\n" output_pos tokenStartPos;
                pushCtxt tokenTup (CtxtParen (token,tokenStartPos));
                pushCtxtSeqBlock(true,AddBlockEnd);
                returnToken tokenLexbufState token

            | TYPE, _   -> 
                if debug then dprintf2 "TYPE, pushing CtxtType(%a)\n" output_pos tokenStartPos;
                pushCtxt tokenTup (CtxtType(tokenStartPos));
                returnToken tokenLexbufState token

            | TRY, _   -> 
                if debug then dprintf2 "Try, pushing CtxtTry(%a)\n" output_pos tokenStartPos;
                pushCtxt tokenTup (CtxtTry (tokenStartPos));
                (* REVIEW: we would like to push both a begin/end block pair here, but we can only do that *)
                (* if we are able to balance the WITH with the TRY.  We can't do that because of the numeraous ways *)
                (* WITH is used in the grammar (see what happens when we hit a WITH below. *)
                (* This hits us especially in the single line case: "try make ef1 t with _ -> make ef2 t". *)
                
                pushCtxtSeqBlock(false,AddOneSidedBlockEnd);
                returnToken tokenLexbufState token

            |  OBLOCKBEGIN,_ -> 
              (* if debug then dprintf3 "returning token (%s), pos = %a\n" (match token with END -> "end" | _ -> "?") output_pos tokenStartPos;*)
              returnToken tokenLexbufState token  
                
            |  ODUMMY(_),_ -> 
              if debug then dprintf0 "skipping dummy token as no offside rules apply\n";
              hwTokenFetch (useBlockRule) 
                
            (* ordinary tokens start a vanilla block *)
            |  _,CtxtSeqBlock _ :: _ -> 
                pushCtxt tokenTup (CtxtVanilla(tokenStartPos));
                if debug then dprintf2 "pushing CtxtVanilla at tokenStartPos = %a\n" output_pos tokenStartPos;
                returnToken tokenLexbufState token  
                
            |  _ -> 
              (* if debug then dprintf3 "returning token (%s), pos = %a\n" (match token with END -> "end" | _ -> "?") output_pos tokenStartPos;*)
              returnToken tokenLexbufState token  

     and rulesForBothSoftWhiteAndHardWhite(tokenTup) = 
          let (token,tokenLexbufState,tokenPrevEndPos) = tokenTup in 
          match token with 
          (* Insert HIGH_PRECEDENCE_APP if needed *)
          |  IDENT _ when nextTokenIsAdjacentLParenOrLBrack tokenTup ->
              let (_,dotTokenLexbufState,dotTokenPrevEndPos) as dotTokenTup = peekNextTokenTup() in 
              if debug then dprintf2 "inserting HIGH_PRECEDENCE_APP at dotTokenPos = %a\n" output_pos (startPosOfTokenTup dotTokenTup);
              delayToken(HIGH_PRECEDENCE_APP,dotTokenLexbufState,dotTokenPrevEndPos);
              delayToken(tokenTup);
              true
              
          (* Insert HIGH_PRECEDENCE_TYAPP if needed *)
          |  IDENT _ when peekAdjacentTypars false tokenTup ->
              let (_,dotTokenLexbufState,tokenPrevEndPos) as dotTokenTup = peekNextTokenTup() in 
              if debug then dprintf2 "softwhite inserting HIGH_PRECEDENCE_TYAPP at dotTokenPos = %a\n" output_pos (startPosOfTokenTup dotTokenTup);
              delayToken(HIGH_PRECEDENCE_TYAPP,dotTokenLexbufState,tokenPrevEndPos);
              begin match token with 
(*  IMPLEMENTATION FOR DESIGN CHANGE 1600 IF REQUIRED
              | INFIX_COMPARE_OP "<>" ->
                  delayToken(LESS,tokenLexbufState,tokenPrevEndPos);
                  delayToken(GREATER,tokenLexbufState,tokenPrevEndPos);
                  delayToken(tokenTup);
*)
              | _ -> 
                  delayToken(tokenTup);
              end;
              true

          (* Split this token to allow "1..2" for range specification *)
          |  INT32_DOT_DOT i ->
              delayToken(DOT_DOT,tokenLexbufState,tokenPrevEndPos);
              delayToken(INT32(i),tokenLexbufState,tokenPrevEndPos);
              true

          |  MINUS | PLUS_MINUS_OP _
                when (match token with | PLUS_MINUS_OP s -> (s = "+") | _ -> true) &&
                     nextTokenIsAdjacent tokenTup && 
                     not (!prevWasAtomicEnd && (snd(tokenPrevEndPos) = startPosOfTokenTup tokenTup)) ->

              let (nextToken,nextTokenLexbufState,nextTokenPrevEndPos) = popNextTokenTup() in 
              begin 
(*  ADD THIS TO ENABLE NEGATIVE DESIGN CHANGE
                match nextToken with 
                | INT8(v) -> Some(returnToken nextTokenLexbufState (INT8(v |> Nums.i8_to_i32 |> Int32.neg |> Nums.i32_to_i8)))
                | INT16(v) -> Some(returnToken nextTokenLexbufState (INT16(v |> Nums.i16_to_i32 |> Int32.neg |> Nums.i32_to_i16)))
                | INT32(v) -> Some(returnToken nextTokenLexbufState (INT32(v |> Int32.neg)))
                | INT64(v) -> Some(returnToken nextTokenLexbufState (INT64(v |> Int64.neg)))
                | NATIVEINT(v) -> Some(returnToken nextTokenLexbufState (NATIVEINT(v |> Int64.neg)))
                | IEEE32(v) -> Some(returnToken nextTokenLexbufState (IEEE32(v |> Nums.ieee32_to_float |> (fun x -> -. x) |> Nums.float_to_ieee32)))
                | IEEE64(v) -> Some(returnToken nextTokenLexbufState (IEEE64(v |> Nums.ieee64_to_float |> (fun x -> -. x) |> Nums.float_to_ieee64)))
                | BIGINT(v) -> Some(returnToken nextTokenLexbufState (BIGINT(Bytes.append (Bytes.string_as_unicode_bytes "-") v)))
                | DECIMAL(v) -> Some(returnToken nextTokenLexbufState (DECIMAL(Bytes.append (Bytes.string_as_unicode_bytes "-") v)))
                | BIGNUM(v) -> Some(returnToken nextTokenLexbufState (BIGNUM(Bytes.append (Bytes.string_as_unicode_bytes "-") v)))
                | _ -> 
*)
                  delayToken (nextToken,nextTokenLexbufState,nextTokenPrevEndPos); 
                  delayToken(ADJACENT_PREFIX_PLUS_MINUS_OP (match token with PLUS_MINUS_OP s -> s | MINUS -> "-" | _ -> failwith "unreachable" ),
                             tokenLexbufState,tokenPrevEndPos);
                  true
                  
              end

          | _ -> 
              false
  
     and pushCtxtSeqBlock(addBlockBegin,addBlockEnd) = pushCtxtSeqBlockAt (peekNextTokenTup(),addBlockBegin,addBlockEnd) 
     and pushCtxtSeqBlockAt(((token,tokenLexbufState,tokenPrevEndPos) as p),addBlockBegin,addBlockEnd) = 
         if addBlockBegin then (
           if debug then dprintf0 "--> insert OBLOCKBEGIN \n" ;
           delayToken(OBLOCKBEGIN,tokenLexbufState,tokenPrevEndPos)
         );
         pushCtxt p (CtxtSeqBlock(FirstInSeqBlock, startPosOfTokenTup p,addBlockEnd))  in

    let rec swTokenFetch() = 
          let (token,tokenLexbufState,_ as tokenTup) = popNextTokenTup() in 
          let tokenReplaced = rulesForBothSoftWhiteAndHardWhite(tokenTup) in
          if tokenReplaced then swTokenFetch() 
          else returnToken tokenLexbufState token in 

    (*----------------------------------------------------------------------------
    !* Part VI. The new lexer function.  In light 
     *--------------------------------------------------------------------------*)

    let lexer = fun _ -> 
        if not !initialized then 
            peekInitial();

        let (Done token) = 
            if usingLightSyntax(lightSyntaxStatus)
            then hwTokenFetch(true)  
            else swTokenFetch() in
        token  in 

    { offsideStack = offsideStack;
      lightSyntaxStatus = lightSyntaxStatus;
      lexbuf = lexbuf;
      lexer = lexer }

let get_lexer x = x.lexer
let get_lexbuf x = x.lexbuf
  
