(* (c) Microsoft Corporation. All rights reserved *)

(* Breakable block layout implementation.
   This is a fresh implementation of pre-existing ideas.
 *)

(*--------------------------------------------------------------------------
INDEX: opens
--------------------------------------------------------------------------*)

open Printf
open List  

let spaces n = String.make n ' '

(*--------------------------------------------------------------------------
INDEX: layout
--------------------------------------------------------------------------*)

type joint =
   (******
    * a joint, between 2 layouts, is either:
    * - unbreakable, or
    * - breakable, and if broken the second block has a given indentation.
    ******)
 | Unbreakable
 | Breakable of int
 | Broken of int

(*F# open Microsoft.FSharp.Text.StructuredFormat F#*)

(* This type must be identical to the one in the F# library *)
type layout =
(*F# Microsoft.FSharp.Text.StructuredFormat.layout F#*)
(*IF-OCAML*)
 | Leaf  of bool * Obj.t * bool
 | Node  of bool * layout * bool * layout * bool * joint
 | Attr of string * (string * string) list * layout
(*ENDIF-OCAML*)

(*--------------------------------------------------------------------------
INDEX: support
--------------------------------------------------------------------------*)

let rec juxtLeft = function
    Leaf (jl,text,jr)         -> jl
  | Node (jl,l,jm,r,jr,joint) -> jl
  | Attr (tag,attrs,l)        -> juxtLeft l

let rec juxtRight = function
    Leaf (jl,text,jr)         -> jr
  | Node (jl,l,jm,r,jr,joint) -> jr
  | Attr (tag,attrs,l)        -> juxtRight l

(* NOTE:
 * emptyL might be better represented as a constructor,
 * so then (Sep"") would have true meaning
 *)
let emptyL = Leaf (true,Obj.repr "",true)
let isEmptyL = function Leaf(true,tag,true) when Obj.obj tag = "" -> true | _ -> false
      
let mkNode l r joint =
   if isEmptyL l then r else
   if isEmptyL r then l else
   let jl = juxtLeft  l in
   let jm = juxtRight l || juxtLeft r in
   let jr = juxtRight r in
   Node(jl,l,jm,r,jr,joint)


(*--------------------------------------------------------------------------
INDEX: constructors
--------------------------------------------------------------------------*)

let wordL  (str:string) = Leaf (false,Obj.repr str,false)
let sepL   (str:string) = Leaf (true ,Obj.repr str,true)   
let rightL (str:string) = Leaf (true ,Obj.repr str,false)   
let leftL  (str:string) = Leaf (false,Obj.repr str,true)

let aboveL  l r = mkNode l r (Broken 0)
let joinN i l r = mkNode l r (Breakable i)                                      
let join  = joinN 0
let join1 = joinN 1
let join2 = joinN 2
let join3 = joinN 3

let tagAttrL str attrs ly = Attr (str,attrs,ly)
let tagL str ly = tagAttrL str [] ly
let linkL str ly = tagAttrL "html:a" [("href",str)] ly

(*--------------------------------------------------------------------------
INDEX: constructors derived
--------------------------------------------------------------------------*)

let apply2 f l r = if isEmptyL l then r else
                   if isEmptyL r then l else f l r

let ($$)    l r = mkNode l r (Unbreakable)
let (++)    l r = mkNode l r (Breakable 0)
let (--)    l r = mkNode l r (Breakable 1)
let (---)   l r = mkNode l r (Breakable 2)
let (----)  l r = mkNode l r (Breakable 3)
let (-----) l r = mkNode l r (Breakable 4)    
let (@@)    l r = apply2 (fun l r -> mkNode l r (Broken 0)) l r
let (@@-)   l r = apply2 (fun l r -> mkNode l r (Broken 1)) l r
let (@@--)  l r = apply2 (fun l r -> mkNode l r (Broken 2)) l r

let tagListL tagger = function
    []    -> emptyL
  | [x]   -> x
  | x::xs ->
      let rec process' prefixL = function
	  []    -> prefixL
	| y::ys -> process' ((tagger prefixL) ++ y) ys
      in  process' x xs
    
let commaListL x = tagListL (fun prefixL -> prefixL $$ rightL ",") x
let semiListL x  = tagListL (fun prefixL -> prefixL $$ rightL ";") x
let spaceListL x = tagListL (fun prefixL -> prefixL) x
let sepListL x y = tagListL (fun prefixL -> prefixL $$ x) y

let bracketL l = leftL "(" $$ l $$ rightL ")"
let tupleL xs = bracketL (sepListL (sepL ",") xs)
let aboveListL = function
    []    -> emptyL
  | [x]   -> x
  | x::ys -> List.fold_left (fun pre y -> pre @@ y) x ys

let optionL xL = function
    None   -> wordL "None"
  | Some x -> wordL "Some" -- (xL x)

let listL xL xs = leftL "[" $$ sepListL (sepL ";") (map xL xs) $$ rightL "]"


(*--------------------------------------------------------------------------
INDEX: breaks v2
--------------------------------------------------------------------------*)
	
(* A very quick reimplementation of break stack.
 * FOR NOW:
 * - rely on single threading in usage below.
 * LATER:
 * - add monad to enforce single threading... or
 * - add ref tags to check linear thread (faster)
 *------------
 * State is:
 *)

type breaks = int *     (* pos of next free slot *)
              int *     (* pos of next possible "outer" break - OR - outer=next if none possible *)
              int array (* stack of savings, -ve means it has been broken *)

(* next  is next slot to push into - aka size of current occupied stack.
 * outer counts up from 0, and is next slot to break if break forced.
 * - if all breaks forced, then outer=next.
 * - popping under these conditions needs to reduce outer and next.
 *)
      
let chunkN = 400      
let breaks0 () = 0,0,Array.create chunkN 0
let pushBreak saving (next,outer,stack) =
  let stack = if next = Array.length stack then
                Array.append stack (Array.create chunkN 0) (* expand if full *)
              else
                stack
  in
  stack.(next) <- saving;
  next+1,outer,stack
let popBreak (next,outer,stack) =
  if next=0 then raise (Failure "popBreak: underflow");
  let topBroke = stack.(next-1)<0 in
  let outer = if outer=next then outer-1 else outer in  (* if all broken, unwind *)
  let next  = next - 1 in
  (next,outer,stack),topBroke
let forceBreak (next,outer,stack) =
  if outer=next then
    (* all broken *)
    None
  else
    let saving = stack.(outer) in
    stack.(outer) <- -stack.(outer);    
    let outer = outer+1 in
    Some ((next,outer,stack),saving)


(*--------------------------------------------------------------------------
INDEX: fitting
--------------------------------------------------------------------------*)
	
let squashTo maxWidth layout =
   let rec fit breaks (pos,layout) =
     (******
      * breaks = break context, can force to get indentation savings.
      * pos    = current position in line
      * layout = to fit
      *------
      * returns:
      * breaks
      * layout - with breaks put in to fit it.
      * pos    - current pos in line = rightmost position of last line of block.
      * offset - width of last line of block
      * NOTE: offset <= pos -- depending on tabbing of last block
      ******)
     (*Printf.printf "\n\nCalling pos=%d layout=[%s]\n" pos (showL layout);*)
     let breaks,layout,pos,offset =
     match layout with
      | Attr (tag,attrs,l) ->
	 let breaks,layout,pos,offset = fit breaks (pos,l) in
	 let layout = Attr (tag,attrs,layout) in
	 breaks,layout,pos,offset
      | Leaf (jl,text,jr) ->
	 let textWidth = String.length (Obj.obj text) in
         let rec fitLeaf breaks pos =
	   if pos + textWidth <= maxWidth then
	     breaks,layout,pos + textWidth,textWidth (* great, it fits *)
	   else
	     match forceBreak breaks with
	       None                 -> ((*Printf.printf "\nNo breaks   , pos=%d\n" (pos+textWidth);*)
		                        breaks,layout,pos + textWidth,textWidth (* tough, no more breaks *))
	     | Some (breaks,saving) -> ((*Printf.printf "\nForce break, saving=%d\n" saving;*)
		                        let pos = pos - saving in
                                        fitLeaf breaks pos)
	 in
	 fitLeaf breaks pos
     | Node (jl,l,jm,r,jr,joint) ->
	 let mid = if jm then 0 else 1 in
	 match joint with
	   Unbreakable    ->
             let breaks,l,pos,offsetl = fit breaks (pos,l) in    (* fit left *)
             let pos = pos + mid in                              (* fit space if juxt says so *)
             let breaks,r,pos,offsetr = fit breaks (pos,r) in    (* fit right *)
             breaks,Node (jl,l,jm,r,jr,Unbreakable),pos,offsetl + mid + offsetr
         | Broken indent ->
             let breaks,l,pos,offsetl = fit breaks (pos,l) in    (* fit left *)
             let pos = pos - offsetl + indent in                 (* broken so - offset left + indent *)
             let breaks,r,pos,offsetr = fit breaks (pos,r) in    (* fit right *)
             breaks,Node (jl,l,jm,r,jr,Broken indent),pos,indent + offsetr
         | Breakable indent ->
             let breaks,l,pos,offsetl = fit breaks (pos,l) in    (* fit left *)
             (* have a break possibility, with saving *)
             let saving = offsetl + mid - indent in
	     let pos = pos + mid in
             if saving>0 then
               let breaks = pushBreak saving breaks in
               let breaks,r,pos,offsetr = fit breaks (pos,r) in
               let breaks,broken = popBreak breaks in
               if broken then
                 breaks,Node (jl,l,jm,r,jr,Broken indent)   ,pos,indent + offsetr
               else
                 breaks,Node (jl,l,jm,r,jr,Breakable indent),pos,offsetl + mid + offsetr
             else
               (* actually no saving so no break *)
               let breaks,r,pos,offsetr = fit breaks (pos,r) in
               breaks,Node (jl,l,jm,r,jr,Breakable indent)  ,pos,offsetl + mid + offsetr
     in
     (*Printf.printf "\nDone:     pos=%d offset=%d" pos offset;*)
     breaks,layout,pos,offset
   in
   let breaks = breaks0 () in
   let pos = 0 in
   let breaks,layout,pos,offset = fit breaks (pos,layout) in
   layout

(*--------------------------------------------------------------------------
INDEX: render
--------------------------------------------------------------------------*)

type ('a,'b) render =
    (* exists 'b.
       -- could use object type to get "exists 'b" on private state,
       -- but there is no common object type between F# and OCAML.
    *)
    { start        : unit -> 'b;
      addText      : 'b -> string -> 'b;
      addBreak     : 'b -> int -> 'b;
      addTag       : 'b -> string * (string * string) list * bool -> 'b;
      finish       : 'b -> 'a
    }
      
let renderL rr layout =
  let rec addL z pos i = function
      (* pos is tab level *)
      Leaf (jl,text,jr)                 -> rr.addText z (Obj.obj text),i + String.length (Obj.obj text)
    | Node (jl,l,jm,r,jr,Broken indent) -> let z,i = addL z pos i l in
                                           let z,i = rr.addBreak z (pos+indent),(pos+indent) in
                                           let z,i = addL z (pos+indent) i r in
                                           z,i
    | Node (jl,l,jm,r,jr,_)             -> let z,i = addL z pos i l in
                                           let z,i = if jm then z,i else rr.addText z " ",i+1 in
					   let pos = i in
					   let z,i = addL z pos i r in
					   z,i
    | Attr (tag,attrs,l)                -> let z   = rr.addTag z (tag,attrs,true) in
                                           let z,i = addL z pos i l in
                                           let z   = rr.addTag z (tag,attrs,false) in
                                           z,i
  in
  let pos = 0 in
  let z,i = rr.start(),0 in
  let z,i = addL z pos i layout in
  rr.finish z

(*--------------------------------------------------------------------------
INDEX: string render 
--------------------------------------------------------------------------*)
    
(* string render *)
let start () = ([] : string list) (* reverse collector *)
let push x rstrs = x::rstrs
let z0 = [],0
let addText  rstrs text  = push text rstrs
let addBreak rstrs n     = (* \n then spaces... *)
                           let rstrs = push "\n"   rstrs in
                           let rstrs = push (spaces n) rstrs in
                           rstrs
let extract rstrs = String.concat "" (List.rev rstrs)

let stringR =
  { start        = start;
    addText      = addText;
    addBreak     = addBreak;
    addTag       = (fun z _ -> z);
    finish       = extract}

(*--------------------------------------------------------------------------
INDEX: channel render
--------------------------------------------------------------------------*)

let channelR chan =
  { start        = (fun () -> ());
    addText      = (fun () s -> output_string chan s);
    addBreak     = (fun () n -> output_string chan "\n"; output_string chan (spaces n));
    addTag       = (fun () (tag,attrs,start) -> ());
    finish       = (fun () -> ()) }

(*--------------------------------------------------------------------------
INDEX: buffer render
--------------------------------------------------------------------------*)

let bufferR os =
  { start        = (fun () -> ());
    addText      = (fun () s -> bprintf os "%s" s);
    addBreak     = (fun () n -> bprintf os "\n"; bprintf os "%s" (spaces n));
    addTag       = (fun () (tag,attrs,start) -> ());
    finish       = (fun () -> ()) }

(*--------------------------------------------------------------------------
INDEX: html render - wraps HTML encoding (TODO) and hyperlinks
--------------------------------------------------------------------------*)
    
let htmlR stringR =
  { stringR with
    addText      = stringR.addText;  (* REVIEW: escape HTML chars *)
    addBreak     = (fun z n ->
		      (*let z = stringR.addText z "<br>" in  - not with <pre>*)
		      let z = stringR.addBreak z n in
		      z);
    addTag       = (fun z (tag,attrs,start) -> 
                     match tag,attrs with 
                     | "html:a",[("href",link)] ->
                        if start
		        then stringR.addText z (sprintf "<a href='%s'>" link)
		        else stringR.addText z (sprintf "</a>")
		     | _ -> z)
  }


(*--------------------------------------------------------------------------
INDEX: indent render - wraps fixed indentation
--------------------------------------------------------------------------*)
    
let indentR indent baseR =
  { baseR with
    start    = (fun ()  -> let z = baseR.start() in
                           let z = baseR.addText z (spaces indent) in
			   z);
    addBreak = (fun z n -> baseR.addBreak z (n+indent));
  }
    

(*--------------------------------------------------------------------------
INDEX: showL, outL are most common
--------------------------------------------------------------------------*)

let showL      layout = renderL stringR         layout
let outL chan  layout = renderL (channelR chan) layout
let bufferL os layout = renderL (bufferR os)    layout
    

