(* (c) Microsoft Corporation. All rights reserved *)

(*-------------------------------------------------------------------------
!* tinfos, minfos, finfos, pinfos - summaries of information for references
 * related to .NET constructs.
 *------------------------------------------------------------------------- *)

(*F# 
open Microsoft.Research.AbstractIL 
open Microsoft.Research.AbstractIL.Internal 
open Microsoft.FSharp.Compiler 
module Ildiag = Microsoft.Research.AbstractIL.Diagnostics 
module Ilmorph = Microsoft.Research.AbstractIL.Morphs 
module Il = Microsoft.Research.AbstractIL.IL 
F#*) 
open Ildiag
open List
open Range
open Ast
open Tast
open Tastops
open Nums
open Env
open Il (* Abstract IL  *)
open Lib

(*-------------------------------------------------------------------------
!* From IL types to F# types
 *------------------------------------------------------------------------- *)

(* importInst gives the context for interpreting type variables *)
let import_typ scoref amap m importInst ilty = 
    ilty |> rescope_typ scoref |>  Import.typ_of_il_typ amap m importInst 

(*-------------------------------------------------------------------------
!* Fold the hierarchy. 
 *  REVIEW: this code generalizes the iteration used below for member lookup.
 *------------------------------------------------------------------------- *)

let is_fsobjmodel_or_exn_typ typ = 
    is_fsobjmodel_ty typ  or (is_stripped_tyapp_typ typ && tycon_is_exnc (deref_tycon (tcref_of_stripped_typ typ)))
    
let super_of_typ g amap m typ = 
    if is_il_named_typ typ then 
        let tcref,tinst = dest_stripped_tyapp_typ typ in 
        let scoref,_,tdef = dest_il_tcref  tcref in 
        begin match tdef.tdExtends with 
        | None -> None
        | Some ilty -> Some (import_typ scoref amap m tinst ilty)
        end
    else if is_fsobjmodel_or_exn_typ typ then 
        let tcref,tinst = dest_stripped_tyapp_typ typ in 
        Some (inst_type (mk_inst_for_stripped_typ typ) (super_of_tycon g (deref_tycon tcref)))
    else if is_any_array_typ g typ then
        Some(g.system_Array_typ)
    else if is_ref_typ g typ && not (type_equiv g typ g.obj_ty) then 
        Some(g.obj_ty)
    else None

let mk_System_Collections_Generic_IList_ty g ty = TType_app(g.tcref_System_Collections_Generic_IList,[ty])


let implements_of_typ g amap m typ = 
    let itys = 
        if is_stripped_tyapp_typ typ then
            let tcref,tinst = dest_stripped_tyapp_typ typ in 
            if is_il_tcref tcref then 
                let scoref,_,tdef = dest_il_tcref  tcref in 
                map (import_typ scoref amap m tinst) tdef.tdImplements
            else  
                let inst = mk_inst_for_stripped_typ typ in
                map (fun (x,_,_) -> inst_type inst x) (tcaug_of_tcref tcref).tcaug_implements
        else [] in
        
    let itys =
        if g.typeCheckerConfiguredToAssumeV20Library && 
           is_il_arr1_typ g typ then 
            mk_System_Collections_Generic_IList_ty g (dest_il_arr1_typ g typ) :: itys
        else 
            itys in
    itys
        
    
let rec fold_hierarchy_of_typA ndeep followInterfaces f g amap m typ (visited,acc) =
    if gen_mem (type_equiv g) typ visited then visited,acc else
    let state = typ::visited, acc in 
    if verbose then  dprintf2 "--> fold_hierarchy_of_typA, ndeep = %d, typ = %s...\n" ndeep ((DebugPrint.showType typ));
    if ndeep > 100 then (errorR(Error("recursive class hierarchy (detected in fold_hierarchy_of_typA), typ = "^(DebugPrint.showType typ),m)); (visited,acc)) else
    let visited,acc = 
        if is_interface_typ  typ then 
            fold_right 
               (fold_hierarchy_of_typA (ndeep+1) followInterfaces f g amap m) 
               (implements_of_typ g amap m typ) 
                  (fold_hierarchy_of_typA ndeep followInterfaces f g amap m g.obj_ty state)
        else if is_typar_ty typ then 
            let tp = dest_typar_typ typ in 
            fold_right 
                (fun x vacc -> 
                  match x with 
                  | TTyparMayResolveMemberConstraint _
                  | TTyparDefaultsToType _
                  | TTyparIsEnum _
                  | TTyparIsDelegate _
                  | TTyparSupportsNull _
                  | TTyparIsNotNullableValueType _ 
                  | TTyparIsReferenceType _ 
                  | TTyparSimpleChoice _ 
                  | TTyparRequiresDefaultConstructor _ -> vacc
                  | TTyparCoercesToType(TTyparSubtypeConstraintFromFS cty,_) -> 
                          fold_hierarchy_of_typA (ndeep + 1)  followInterfaces f g amap m cty vacc
                  | TTyparCoercesToType(TTyparSubtypeConstraintFromIL _,_) -> 
                      warning(Error("fold_hierarchy_of_typA: unexpected TTyparSubtypeConstraintFromIL",m));
                      vacc) 
                (constraints_of_tpref tp) 
                state
        else 
            let state = 
                if followInterfaces then 
                    fold_right 
                      (fold_hierarchy_of_typA (ndeep+1) followInterfaces f g amap m) 
                      (implements_of_typ g amap m typ) 
                      state 
                else 
                    state in 
            let state = 
                Option.fold_right 
                  (fold_hierarchy_of_typA (ndeep+1) followInterfaces f g amap m) 
                  (super_of_typ g amap m typ) 
                  state in 
            state in  
                                
    (visited,f typ acc) 

let fold_primary_hierarchy_of_typ f g amap m typ acc = fold_hierarchy_of_typA 0 false f g amap m typ ([],acc) |> snd
let fold_entire_hierarchy_of_typ f g amap m typ acc = fold_hierarchy_of_typA 0 true f g amap m typ ([],acc) |> snd
let iter_entire_hierarchy_of_typ f g amap m typ = fold_hierarchy_of_typA 0 true (fun ty () -> f ty) g amap m typ ([],())  |> snd
let exists_in_entire_hierarchy_of_typ f g amap m typ = fold_hierarchy_of_typA 0 true (fun ty acc -> acc || f ty ) g amap m typ ([],false) |> snd
let first_in_entire_hierarchy_of_typ f g amap m typ = fold_hierarchy_of_typA 0 true (fun ty acc -> match acc with None -> if f ty then Some(ty) else None | Some _ -> acc) g amap m typ ([],None) |> snd

let super_types_of_typ g amap m ty = fold_hierarchy_of_typA 0 true (gen_insert (type_equiv g)) g amap m ty ([],[]) |> snd


(*-------------------------------------------------------------------------
!* Basic infos
 *------------------------------------------------------------------------- *)

type il_type_info = 
    | ILTypeInfo of tycon_ref * Il.type_ref * tinst * Il.type_def

type type_info = 
    | ILType of il_type_info
    | FSType of Tast.typ

type il_meth_info =
    | ILMethInfo of il_type_info * Il.method_def * typars (* typars are the uninstantiated generic method args *) 
type meth_info = 
    | FSMeth of Tast.typ * val_ref  
    | ILMeth of il_meth_info
    | DefaultStructCtor of Tast.typ
type il_field_info = 
    | ILFieldInfo of il_type_info * Il.field_def (* .NET IL fields *)
type recdfield_info = 
    | RecdFieldInfo of tinst * Tast.recdfield_ref (* F# fields *)

type il_property_info = 
    | ILPropInfo of il_type_info * Il.property_def 
type prop_info = 
    | FSProp of Tast.typ * val_ref option * val_ref option
    | ILProp of il_property_info

type il_event_info = 
    | ILEventInfo of il_type_info * Il.event_def



(*-------------------------------------------------------------------------
!* Copy constraints.  If the constraint comes from a type parameter associated
 * with a type constructor then we are simply renaming type variables.  If it comes
 * from a generic method in a generic class (e.g. typ.M<_>) then we may be both substituting the
 * instantiation assocaited with 'typ' as well as copying the type parameters associated with 
 * M.
 *
 * Type parameters may come from IL code and may have IL constraints.  These are
 * converted to F# constraints. The substitution importInst is applied while 
  * reading the constraint.
 *
 * Note importInst and tprefInst specify equivalent substitutions: tprefInst is applied to F#
 * constraints and importInst is used while reading off an IL constraint.
 *------------------------------------------------------------------------- *)

let copy_or_import_typar_constraints amap m tprefInst importInst tporig =
    (constraints_of_typar tporig) 
    |>  map (fun tpc -> 
           match tpc with 
           | TTyparCoercesToType(TTyparSubtypeConstraintFromFS ty,_) -> 
               TTyparCoercesToType (TTyparSubtypeConstraintFromFS(inst_type tprefInst ty),m)
           | TTyparCoercesToType(TTyparSubtypeConstraintFromIL (scoref,ilty),_) -> 
               TTyparCoercesToType(TTyparSubtypeConstraintFromFS(import_typ scoref amap m importInst ilty),m)
           | TTyparDefaultsToType(priority,ty,_) -> 
               TTyparDefaultsToType (priority,inst_type tprefInst ty,m)
           | TTyparSupportsNull _ -> 
               TTyparSupportsNull m
           | TTyparIsEnum(uty,_) -> 
               TTyparIsEnum (inst_type tprefInst uty,m)
           | TTyparIsDelegate(aty, bty,_) -> 
               TTyparIsDelegate (inst_type tprefInst aty,inst_type tprefInst bty,m)
           | TTyparIsNotNullableValueType _ -> 
               TTyparIsNotNullableValueType m
           | TTyparIsReferenceType _ -> 
               TTyparIsReferenceType m
           | TTyparSimpleChoice (tys,_) -> 
               TTyparSimpleChoice (map (inst_type tprefInst) tys,m)
           | TTyparRequiresDefaultConstructor _ -> 
               TTyparRequiresDefaultConstructor m
           | TTyparMayResolveMemberConstraint(traitInfo,_) -> 
               TTyparMayResolveMemberConstraint (inst_trait tprefInst traitInfo,m))

(* The constraints for each typar copied from another typar can only be fixed up once *)
(* we have generated all the new constraints, e.g. f<A :> List<B>, B :> List<A>> ... *)
let fixup_new_typars amap m ftctps tinst tpsorig tps =
    let renaming,tptys = (mk_typar_to_typar_renaming tpsorig tps) in 
    let tprefInst = (mk_typar_inst ftctps tinst) @ renaming in 
    iter2 (fun tporig tp -> fixup_typar_constraints tp (copy_or_import_typar_constraints amap  m tprefInst (tinst@tptys) tporig)) tpsorig tps;
    renaming,tptys


(*-------------------------------------------------------------------------
!* tinfos
 *------------------------------------------------------------------------- *)

let tcref_of_il_tinfo (ILTypeInfo(tcref,_,_,_)) = tcref
let tref_of_il_tinfo  (ILTypeInfo(_,tref,_,_))  = tref
let tinst_of_il_tinfo (ILTypeInfo(_,_,tinst,_)) = tinst
let tdef_of_il_tinfo  (ILTypeInfo(_,_,_,tdef))  = tdef
let typ_of_il_tinfo   (ILTypeInfo(tcref,_,tinst,_)) = TType_app(tcref,tinst)
let scoref_of_il_tinfo = tref_of_il_tinfo >> scoref_of_tref 
let name_of_il_tinfo = tref_of_il_tinfo >> tname_of_tref 
let il_tinfo_is_interface x =  ((tdef_of_il_tinfo x).tdKind = TypeDef_interface)
let il_tinfo_is_struct x = is_value_tdef (tdef_of_il_tinfo x)

let inst_il_tinfo inst (ILTypeInfo(tcref,tref,tinst,tdef)) = ILTypeInfo(tcref,tref,inst_types inst tinst,tdef)

let formal_tctps_of_il_tinfo x = typars_of_tcref (tcref_of_il_tinfo x)
let import_typ_from_metadata amap m (ILTypeInfo(_,tref,tinst,_)) minst ilty = 
    import_typ (scoref_of_tref tref) amap m (tinst@minst) ilty

let tdef_of_il_tcref tcref = 
    let _,_,tdef = dest_il_tcref  tcref in
    tdef 
    
let tdef_of_il_typ ty = tdef_of_il_tcref  (tcref_of_stripped_typ ty)

let tinfo_of_il_typ ty = 
    if is_il_named_typ ty then 
        let tcref,tinst = dest_stripped_tyapp_typ ty in 
        let scoref,enc,tdef = dest_il_tcref  tcref in 
        ILTypeInfo(tcref,mk_nested_tref (scoref,(List.map name_of_tdef enc),name_of_tdef tdef),tinst,tdef)
    else failwith "tinfo_of_il_typ"

(*-------------------------------------------------------------------------
!* Build method infos.  
 *------------------------------------------------------------------------- *)

let mk_il_minfo amap m tinfo md = 
    let ilmtps = Import.new_il_typars m (scoref_of_il_tinfo tinfo) (name_of_il_tinfo tinfo)  md.mdGenericParams in 
    let tinst = tinst_of_il_tinfo tinfo in 
    (* QUERY: do we really need to copy the generic method typars here? *)
    (* NOTE: Part of this is because we don't turn generic method typars into *)
    (* "real" F# typars until late (we don't do it in the import phase since it *)
    (* turns out hard to import the types on the constraints at the same time). *)
    (* It could be worth revisiting this as piping g/amap/m down to this point *)
    (* is a bit of a pain. *)
    (* NOTE: note that these are not inference typars: they are rigid. *)
    let mtps = copy_typars ilmtps in 
    let _ = fixup_new_typars amap m (formal_tctps_of_il_tinfo tinfo) tinst ilmtps mtps in 
    ILMeth (ILMethInfo(tinfo,md,mtps))

(*-------------------------------------------------------------------------
!* il_minfo, il_pinfo
 *------------------------------------------------------------------------- *)



let tinfo_of_il_minfo (ILMethInfo(tinfo,_,_)) = tinfo
let mdef_of_il_minfo  (ILMethInfo(_,md,_)) = md
let arity_of_il_minfo (ILMethInfo(_,md,_)) = length md.mdParams
let scoref_of_il_minfo  = tinfo_of_il_minfo >> scoref_of_il_tinfo
let tref_of_il_minfo    = tinfo_of_il_minfo >> tref_of_il_tinfo
let tinst_of_il_minfo   = tinfo_of_il_minfo >> tinst_of_il_tinfo
let name_of_il_minfo x = (mdef_of_il_minfo x).mdName
let kind_of_il_minfo x = (mdef_of_il_minfo x).mdKind
let generic_arity_of_il_minfo x = length (mdef_of_il_minfo x).mdGenericParams

let mdef_is_ctor md = md.mdName = ".ctor" 
let mdef_is_cctor md = md.mdName = ".cctor" 

let mdef_is_protected md = 
    not (mdef_is_ctor md) &&
    not (mdef_is_cctor md) &&
    (md.mdAccess = MemAccess_family) &&
    not (is_static_callconv md.mdCallconv) 

let il_minfo_is_ctor       = mdef_of_il_minfo >> mdef_is_ctor 
let il_minfo_is_cctor      = mdef_of_il_minfo >> mdef_is_cctor
let il_minfo_is_protected  = mdef_of_il_minfo >> mdef_is_protected
let il_minfo_is_virt x     = mdef_is_virt (mdef_of_il_minfo x)
let il_minfo_is_abstract x = match (mdef_of_il_minfo x).mdKind with MethodKind_virtual vinfo -> vinfo.virtAbstract | _ -> false
let il_minfo_is_static x   = is_static_callconv (mdef_of_il_minfo x).mdCallconv
let il_minfo_is_newslot x   = match (mdef_of_il_minfo x).mdKind with MethodKind_virtual vinfo -> vinfo.virtNewslot | _ -> false
let il_minfo_is_instance x = not (il_minfo_is_ctor x ||  (is_static_callconv (mdef_of_il_minfo x).mdCallconv) )
let il_minfo_is_struct x   = il_tinfo_is_struct (tinfo_of_il_minfo x)

let argtys_of_il_minfo amap m (ILMethInfo (tinfo,mdef,_)) minst =
    mdef.mdParams |> map (fun p -> import_typ_from_metadata amap m tinfo minst (typ_of_param p)) 

let params_of_il_minfo amap m (ILMethInfo (tinfo,mdef,_)) minst =
    mdef.mdParams |> map (fun p -> p.paramName, import_typ_from_metadata amap m tinfo minst  (typ_of_param p)) 

let ret_typ_of_il_minfo amap m (ILMethInfo(tinfo,mdef,_)) minst =
    import_typ_from_metadata amap m tinfo minst (typ_of_return mdef.mdReturn) 

let mref_of_il_minfo minfo = 
    let mref = mk_mref_to_mdef (tref_of_il_minfo minfo,mdef_of_il_minfo minfo) in
    rescope_mref (scoref_of_il_minfo minfo) mref 

let typ_of_il_minfo x = typ_of_il_tinfo (tinfo_of_il_minfo x)

let inst_il_minfo amap m inst (ILMethInfo(tinfo,md,mtps)) = 
    mk_il_minfo amap m (inst_il_tinfo inst tinfo) md

let il_minfo_is_DllImport g minfo = 
    let (AttribInfo(tref,_)) = g.attrib_DllImportAttribute in
    (mdef_of_il_minfo minfo).mdCustomAttrs |> ilthing_decode_il_attrib g tref |> isSome

(** Build an expression node that is a call to a .NET method. *)
let mk_il_minfo_call g amap m isProp minfo vFlags minst direct args = 
    let isStatic = not (il_minfo_is_ctor minfo || il_minfo_is_instance minfo) in 
    let valu = il_minfo_is_struct minfo in 
    let ctor = (match (kind_of_il_minfo minfo) with MethodKind_ctor -> true | _ -> false) in
    let virt = 
        not valu  && not direct &&
        (match (kind_of_il_minfo minfo) with 
        | MethodKind_nonvirtual | MethodKind_ctor | MethodKind_static -> false
        | MethodKind_cctor -> error (InternalError ((name_of_il_minfo minfo)^": cannot call a class constructor",m));
        | MethodKind_virtual _ -> true) in 
    let protect = il_minfo_is_protected minfo in
    let mref = mref_of_il_minfo minfo in 
    let newobj = ctor && (vFlags = NormalValUse) in
    let exprty = if ctor then typ_of_il_minfo minfo else ret_typ_of_il_minfo amap m minfo minst in 
    let tinst = tinst_of_il_minfo minfo in 
    let rty = (if not ctor && (ret_of_mref mref = Il.Type_void) then [] else [exprty]) in 
    let isDllImport = il_minfo_is_DllImport g minfo in
    TExpr_op(TOp_ilcall((virt,protect,valu,newobj,vFlags,isProp,isDllImport,None,mref),tinst,minst, rty),[],args,m),
    exprty
  
let mk_obj_ctor_call g m =
    let mref = formal_mref_of_mspec(mk_nongeneric_ctor_mspec(g.ilg.tref_Object,AsObject,[])) in 
    TExpr_op(TOp_ilcall((false,false,false,false,CtorValUsedAsSuperInit,false,false,None,mref),[],[],[g.obj_ty]),[],[],m)

(*-------------------------------------------------------------------------
!* il_pinfo
 *------------------------------------------------------------------------- *)

let tinfo_of_il_pinfo (ILPropInfo(tinfo,_)) = tinfo
let pdef_of_il_pinfo  (ILPropInfo(_,pd)) = pd
let scoref_of_il_pinfo = tinfo_of_il_pinfo >> scoref_of_il_tinfo
let name_of_il_pinfo  (ILPropInfo(_,pdef)) = pdef.propName

let getter_minfo_of_il_pinfo (ILPropInfo(tinfo,pdef)) =
    let mdef = mdef_for_semantic_mref (tdef_of_il_tinfo tinfo) (the pdef.propGet) in 
    ILMethInfo(tinfo,mdef,[]) 

let setter_minfo_of_il_pinfo (ILPropInfo(tinfo,pdef)) =
    let mdef = mdef_for_semantic_mref (tdef_of_il_tinfo tinfo) (the pdef.propSet) in 
    ILMethInfo(tinfo,mdef,[]) 

let il_pinfo_has_getter (ILPropInfo(_,pdef)) = isSome pdef.propGet 
let il_pinfo_has_setter (ILPropInfo(tinfo,pdef)) = isSome pdef.propSet 
let il_pinfo_is_static  (ILPropInfo(tinfo,pdef)) = (pdef.propCallconv = CC_static) 

let pdef_accessibility tdef pd =   
    match pd.propGet with 
    | Some mref -> (mdef_for_semantic_mref tdef mref).mdAccess 
    | None -> 
        match pd.propSet with 
          None -> MemAccess_public
        | Some mref -> (mdef_for_semantic_mref tdef mref).mdAccess

let il_pinfo_is_protected pinfo =  
    (il_pinfo_has_getter pinfo && il_minfo_is_protected (getter_minfo_of_il_pinfo pinfo)) or
    (il_pinfo_has_setter pinfo && il_minfo_is_protected (setter_minfo_of_il_pinfo pinfo)) 

let il_pinfo_is_virt pinfo = 
    (il_pinfo_has_getter pinfo && il_minfo_is_virt (getter_minfo_of_il_pinfo pinfo)) or
    (il_pinfo_has_setter pinfo && il_minfo_is_virt (setter_minfo_of_il_pinfo pinfo)) 

let il_pinfo_is_newslot pinfo = 
    (il_pinfo_has_getter pinfo && il_minfo_is_newslot (getter_minfo_of_il_pinfo pinfo)) or
    (il_pinfo_has_setter pinfo && il_minfo_is_newslot (setter_minfo_of_il_pinfo pinfo)) 

let il_pinfo_is_abstract pinfo = 
    (il_pinfo_has_getter pinfo && il_minfo_is_abstract (getter_minfo_of_il_pinfo pinfo)) or
    (il_pinfo_has_setter pinfo && il_minfo_is_abstract (setter_minfo_of_il_pinfo pinfo)) 

let param_typs_of_il_pinfo amap m (ILPropInfo (tinfo,pdef)) =
    map (fun ty -> None, import_typ_from_metadata amap m tinfo [] ty) pdef.propArgs

let vtyp_of_il_pinfo amap m (ILPropInfo(tinfo,pdef)) =
    import_typ_from_metadata amap m tinfo [] pdef.propType

(*-------------------------------------------------------------------------
!* il_einfo
 *------------------------------------------------------------------------- *)

let tinfo_of_il_einfo (ILEventInfo(tinfo,_)) = tinfo
let scoref_of_il_einfo = tinfo_of_il_einfo >> scoref_of_il_tinfo
let tref_of_il_einfo   = tinfo_of_il_einfo >> tref_of_il_tinfo
let name_of_il_einfo  (ILEventInfo(_,edef)) = edef.eventName

let add_minfo_of_il_einfo (ILEventInfo(tinfo,edef)) =
    let mdef = mdef_for_semantic_mref (tdef_of_il_tinfo tinfo) edef.eventAddOn in 
    ILMethInfo(tinfo,mdef,[]) 

let remove_minfo_of_il_einfo (ILEventInfo(tinfo,edef)) =
    let mdef = mdef_for_semantic_mref (tdef_of_il_tinfo tinfo) edef.eventRemoveOn in 
    ILMethInfo(tinfo,mdef,[]) 

let il_einfo_has_fire (ILEventInfo(tinfo,edef)) = edef.eventFire <> None
let il_einfo_is_static    x = x |> add_minfo_of_il_einfo |> il_minfo_is_static
let il_einfo_is_protected x = x |> add_minfo_of_il_einfo |> il_minfo_is_protected
let il_einfo_is_virt      x = x |> add_minfo_of_il_einfo |> il_minfo_is_virt 

let edef_accessibility tdef ed = (mdef_for_semantic_mref tdef ed.eventAddOn).mdAccess 
let del_typ_of_il_einfo amap m (ILEventInfo(tinfo,edef)) =
  import_typ_from_metadata amap m tinfo [] (the edef.eventType)


(*-------------------------------------------------------------------------
!* minfo, pinfo
 *------------------------------------------------------------------------- *)

(** Get the enclosing ("parent") type of the method info. *)
let typ_of_minfo = function 
  | ILMeth(x) -> typ_of_il_minfo x
  | FSMeth(typ,_) -> typ
  | DefaultStructCtor typ -> typ

(** Apply a type instantiation to a method info, i.e. apply the instantiation to the enclosing type. *)
let inst_minfo amap m inst = function 
  | ILMeth(x) -> inst_il_minfo amap m inst x
  | FSMeth(typ,vref) -> FSMeth(inst_type inst typ,vref)
  | DefaultStructCtor typ -> DefaultStructCtor(inst_type inst typ)

let logical_name_of_vspr_vref vref = 
    let membInfo = (the (member_info_of_vref vref)) in 
    match membInfo.vspr_implements_slotsig with 
    | Some slotsig -> name_of_slotsig slotsig
    | _ -> membInfo.vspr_il_name 

let property_name_of_vspr_vref vref = 
    get_property_name (the (member_info_of_vref vref)) 

let name_of_minfo = function 
    | ILMeth(x) -> name_of_il_minfo x
    | FSMeth(_,vref) -> logical_name_of_vspr_vref vref
    | DefaultStructCtor _ -> ".ctor"

(* Do two minfos have the same underlying definitions? *)
(* Used to merge operator overloads collected from left and right of an operator constraint *)
let minfos_use_identical_definitions g x1 x2 = 
    match x1,x2 with 
    | ILMeth(x1), ILMeth(x2) -> (mdef_of_il_minfo x1 =!=  mdef_of_il_minfo x2)
    | FSMeth(ty1,vref1), FSMeth(ty2,vref2)  -> g.vref_eq vref1 vref2 
    | DefaultStructCtor ty1, DefaultStructCtor ty2 -> g.tcref_eq (tcref_of_stripped_typ ty1) (tcref_of_stripped_typ ty2) 
    | _ -> false

(* declaredTypars are not YET always rigid. For example, in a let rec binding *)
(* two declaredTypars may be inferred to be equal, because of a lack of polymorphic recursion. *)
(*    e.g. let rec f<'a> (x:'a) = g x and g<'b> (y:'b) = f y *)
(* Here 'a='b is inferred and one of the "declared" type parameters is *)
(* eliminated by unification. *)
(* Hence we check that all declaredTypars are, at most, unified to another type *)
(* variable, and collect up the "canonical" declaredTypars by doing a *)
(* mk_typar_ty >> dest_typar_typ. We also check that there are no equations *)
(* amongst the canonical typars. *)

let dest_fs_minfo g (typ,vref) = 
    (* if vrec_of_vref vref then rty else *)
    let tps,_,rty,_ = dest_member_vref_typ g vref in 
    
    let parentTyargs = tinst_of_stripped_typ typ in 
    let memberParentTypars,memberMethodTypars = chop_at (length parentTyargs) tps in 

    if verbose then dprintf1 "  dest_fs_minfo, memberParentTypars = %s\n" (Layout.showL (Layout.sepListL (Layout.rightL ";") (map typarL memberParentTypars)));
    if verbose then dprintf1 "  dest_fs_minfo, memberMethodTypars = %s\n" (Layout.showL (Layout.sepListL (Layout.rightL ";") (map typarL memberMethodTypars)));
    if verbose then dprintf1 "  dest_fs_minfo, rty = %s\n" (Layout.showL (typeL rty));
    memberParentTypars,memberMethodTypars,rty,parentTyargs

let formal_mtps_of_minfo g minfo = 
    match minfo with 
    | ILMeth(ILMethInfo(_,_,mtps)) -> mtps
    | FSMeth(typ,vref) ->  
       let _,mtps,_,_ = dest_fs_minfo g (typ,vref) in
       mtps 
    | DefaultStructCtor _ -> []
       
let formal_minst_of_minfo g minfo = generalize_typars (formal_mtps_of_minfo g minfo)
let mk_fs_minfo_tinst ttps mtps tinst minst = (mk_typar_inst ttps tinst @ mk_typar_inst mtps minst) 

let ret_typ_of_minfo g amap m minfo minst = 
    match minfo with 
    | ILMeth(ilminfo) -> ret_typ_of_il_minfo amap m ilminfo minst
    | FSMeth(typ,vref) -> 
       let ttps,mtps,rty,tinst = dest_fs_minfo g (typ,vref) in
       inst_type (mk_fs_minfo_tinst ttps mtps tinst minst) rty
    | DefaultStructCtor _ -> g.unit_ty
       

let param_of_arginfo (ty,TopArgData(_,nm)) = (Option.map text_of_id nm,ty)

let params_of_vspr_vref g vref = map param_of_arginfo (arginfos_of_member_vref g vref)


let argtys_of_minfo g amap m minfo minst = 
    match minfo with 
    | ILMeth(ilminfo) -> argtys_of_il_minfo amap m ilminfo minst
    | FSMeth(typ,vref) -> 
        let ttps,mtps,rty,tinst = dest_fs_minfo g (typ,vref) in
        let paramTypes = params_of_vspr_vref g vref in
        map (snd >> inst_type (mk_fs_minfo_tinst ttps mtps tinst minst)) paramTypes
    | DefaultStructCtor _ -> []

let params_of_minfo g amap m minfo minst = 
    match minfo with 
    | ILMeth(ilminfo) -> params_of_il_minfo amap m ilminfo minst
    | FSMeth(typ,vref) -> 
        let ttps,mtps,rty,tinst = dest_fs_minfo g (typ,vref) in
        let paramTypes = params_of_vspr_vref g vref in
        map (map_snd (inst_type (mk_fs_minfo_tinst ttps mtps tinst minst))) paramTypes
    | DefaultStructCtor _ -> []

let empty_custom_attrs = mk_custom_attrs []

(* The caller-side value for the optional arg, is any *)
type optionalArgFieldInitValue = field_init option
type optionalArgInfo = 
    (* isOptArg *)
    | NotOptional
    (* an F# callee-side optional arg *)
    | CalleeSide
    (* a caller-side .NET optional or default arg *)
    | CallerSide of optionalArgFieldInitValue 
    

let param_attrs_of_minfo g amap m  = function 
    | ILMeth(x) -> 
        (mdef_of_il_minfo x).mdParams 
        |> map (fun p -> 
             let isParamArrayArg = ilthing_has_il_attrib (mk_tref (g.ilg.mscorlib_scoref,"System.ParamArrayAttribute")) p.paramCustomAttrs in
             let isOutArg = (p.paramOut && not p.paramIn) in
             (* Note: we get default argument values frmo VB and other .NET language metadata *)
             let optArgInfo = if p.paramOptional then CallerSide p.paramDefault else NotOptional in
             (isParamArrayArg, isOutArg, optArgInfo))
    | FSMeth(_,vref) -> 
        vref 
        |> arginfos_of_member_vref g 
        |> map (fun (ty,TopArgData(attrs,nm)) -> 
            let isParamArrayArg = false in (* Design Suggestion 1429: can't declare 'params' args in F#  (even an explicit ParamArrayAttribute gets ignored by the compiler) *)
            let isOutArg = false in        (* Design Suggestion 1427: Can't specify "out" args in F# *)
            let isOptArg = fsthing_has_attrib g g.attrib_OptionalArgumentAttribute attrs in
            (* Note: can't specify caller-side default arguments in F#, by design (default is specified on the callee-side) *)
            let optArgInfo = if isOptArg then CalleeSide else NotOptional in        
            (isParamArrayArg,isOutArg,optArgInfo)) 
    | DefaultStructCtor _ -> []


let tps_of_tptys m minst =
    map (fun ty -> if is_typar_ty ty then dest_typar_typ ty else (error(InternalError("expected a typar type in the instantiation of an IL generic method slot",m))))  minst 

(* REVIEW: should attributes always be empty here? *)
let mk_slotparam ty = TSlotParam(None, ty, false,false,false,[]) 

let mk_slotsig (nm,typ,ctps,mtps,paraml,rty) = copy_slotsig (TSlotSig(nm,typ,ctps,mtps,paraml, rty))


(* slotsigs must contain the formal types for the arguments and return type *)
(* a _formal_ 'void' return type is represented as a 'unit' type. *)
(* slotsigs are independent of instantiation: if an instantiation *)
(* happens to make the return type 'unit' (i.e. it was originally a variable type *)
(* then that does not correspond to a slotsig compiled as a 'void' return type. *)
(* REVIEW: should we copy down attributes to slot params? *) 
let slotsig_of_il_minfo amap m  (ILMethInfo(tinfo,mdef,filmtps)) =
    let tcref = tcref_of_stripped_typ (typ_of_il_tinfo tinfo) in 
    let filtctps = (typars_of_tcref tcref) in 
    let ftctps = copy_typars filtctps in 
    let _,ftctptys = fixup_new_typars amap m [] [] filtctps ftctps in 
    let ftinfo = tinfo_of_il_typ (TType_app(tcref,ftctptys)) in 
    let fmtps = copy_typars filmtps in 
    let _,fmtptys = fixup_new_typars amap m ftctps ftctptys filmtps fmtps in 
    let frty = import_typ_from_metadata amap m ftinfo fmtptys (typ_of_return mdef.mdReturn)  in 
    let fparams = 
      map (fun p -> TSlotParam(p.paramName, import_typ_from_metadata amap m ftinfo fmtptys (typ_of_param p),
                               p.paramIn, p.paramOut, p.paramOptional,
                               [])) mdef.mdParams in 
    mk_slotsig(mdef.mdName,typ_of_il_tinfo tinfo,ftctps, fmtps,fparams, frty)

let slotsig_of_minfo g amap m minfo = 
    match minfo with 
    | ILMeth(x) -> slotsig_of_il_minfo amap m x
    | FSMeth(typ,vref) -> 
        begin match vrec_of_vref vref with 
        | ValInRecScope(false) -> error(Error("Invalid recursive reference to an abstract slot",m));
        | _ -> ()
        end;
        let tps,_,rty,_ = dest_member_vref_typ g vref in 
        let ctps = typars_of_tcref (tcref_of_stripped_typ typ) in 
        let ctpsorig,fmtps = chop_at (length ctps) tps in 
        let crenaming,_ = mk_typar_to_typar_renaming ctpsorig ctps in 
        let fargtys = params_of_vspr_vref g vref in 
        let fparams = map (snd >> inst_type crenaming >> mk_slotparam ) fargtys in 
        let frty = inst_type crenaming rty in 
        mk_slotsig(name_of_minfo minfo,typ_of_minfo minfo,ctps,fmtps,fparams, frty)
    | DefaultStructCtor _ -> error(InternalError("no slotsig for DefaultStructCtor",m))

let arity_of_minfo g = function 
    | ILMeth(x) -> arity_of_il_minfo x
    | FSMeth(_,vref) -> length (params_of_vspr_vref  g vref)
    | DefaultStructCtor _ -> 0

let minfo_is_nullary g minfo = (arity_of_minfo g minfo = 0)
let generic_arity_of_minfo g = function 
    | ILMeth(x) -> generic_arity_of_il_minfo x
    | FSMeth(typ,vref) -> 
        let _,mtps,_,_ = dest_fs_minfo g (typ,vref) in
        length mtps
    | DefaultStructCtor _ -> 0

let minfo_is_instance = function 
    | ILMeth(x) -> il_minfo_is_instance x
    | FSMeth(_,vref) -> member_vref_is_instance vref
    | DefaultStructCtor _ -> false

let minfo_is_protected = function 
    | ILMeth(x) -> il_minfo_is_protected x
    | FSMeth _ -> false
    | DefaultStructCtor _ -> false

let minfo_is_struct x = x |> typ_of_minfo |> is_struct_typ

let vref_is_virt vref = 
    let flags = (the (member_info_of_vref vref)).vspr_flags in 
    flags.memFlagsVirtual || flags.memFlagsAbstract || flags.memFlagsOverride

let minfo_is_virt = function 
    | ILMeth(x) -> il_minfo_is_virt x
    | FSMeth(_,vref) -> vref_is_virt vref
    | DefaultStructCtor _ -> false

let vspr_is_abstract membInfo = 
    membInfo.vspr_flags.memFlagsAbstract && not membInfo.vspr_implemented

let vspr_vref_is_definite_override vref = 
    let membInfo = the (member_info_of_vref vref) in 
    let flags = membInfo.vspr_flags in 
    flags.memFlagsOverride || isSome membInfo.vspr_implements_slotsig

let vspr_vref_is_abstract vref =  vspr_is_abstract (the (member_info_of_vref vref))

let minfo_is_abstract = function 
    | ILMeth(x) -> il_minfo_is_abstract x
    | FSMeth(_,vref) as x -> 
        is_interface_typ (typ_of_minfo x)  || 
        vspr_vref_is_abstract vref
    | DefaultStructCtor _ -> false

let minfo_is_newslot x = 
    is_interface_typ (typ_of_minfo x)  || 
    (minfo_is_virt x && 
      (match x with 
       | ILMeth(x) -> il_minfo_is_newslot x
       | FSMeth(_,vref) -> vspr_vref_is_abstract vref
       | DefaultStructCtor _ -> false))

let minfo_is_definite_override = function 
    | ILMeth(x) -> false
    | FSMeth(_,vref) -> vspr_vref_is_definite_override vref
    | DefaultStructCtor _ -> false

let minfo_is_ctor = function 
    | ILMeth(x) -> il_minfo_is_ctor x
    | FSMeth(_,vref) ->
        let flags = (the (member_info_of_vref vref)).vspr_flags in 
        (flags.memFlagsKind = MemberKindConstructor)
    | DefaultStructCtor _ -> true

let minfo_is_cctor = function 
    | ILMeth(x) -> il_minfo_is_cctor x
    | FSMeth _ -> false
    | DefaultStructCtor _ -> false

let pinfo_is_virt = function 
    | ILProp(x) -> il_pinfo_is_virt x
    | FSProp(typ,Some vref,_) | FSProp(typ,_, Some vref) -> vref_is_virt vref
    | FSProp(typ,None,None) -> failwith "unreachable"

let pinfo_is_newslot = function 
    | ILProp(x) -> il_pinfo_is_newslot x
    | FSProp(typ,Some vref,_) | FSProp(typ,_, Some vref) -> vspr_vref_is_abstract vref
    | FSProp(typ,None,None) -> failwith "unreachable"

let pinfo_is_abstract = function 
    | ILProp(x) -> il_pinfo_is_abstract x
    | FSProp(typ,Some vref,_) | FSProp(typ,_, Some vref) -> vspr_vref_is_abstract vref
    | FSProp(typ,None,None) -> failwith "unreachable"

let pinfo_is_protected = function 
    | ILProp(x) -> il_pinfo_is_protected x
    | FSProp _ -> false

let name_of_pinfo = function 
    | ILProp(x) -> name_of_il_pinfo x
    | FSProp(typ,Some vref,_) | FSProp(typ,_, Some vref) -> 
        property_name_of_vspr_vref vref
    | FSProp(typ,None,None) -> failwith "unreachable"
  
let typ_of_pinfo = function 
    | ILProp(x) -> typ_of_il_tinfo (tinfo_of_il_pinfo x)
    | FSProp(typ,_,_) -> typ

let pinfo_is_struct = typ_of_pinfo >> is_struct_typ 

let getter_minfo_of_pinfo = function 
    | ILProp(x) -> ILMeth(getter_minfo_of_il_pinfo x)
    | FSProp(typ,Some vref,_) -> FSMeth(typ,vref) 
    | FSProp(_,_,_) -> failwith "getter_minfo_of_pinfo"

let setter_minfo_of_pinfo = function 
    | ILProp(x) -> ILMeth(setter_minfo_of_il_pinfo x)
    | FSProp(typ,_,Some vref) -> FSMeth(typ,vref)
    | FSProp(_,_,_) -> failwith "setter_minfo_of_pinfo"

let pinfo_has_getter = function 
    | ILProp(x) -> il_pinfo_has_getter x
    | FSProp(_,x,_) -> x <> None

let pinfo_has_setter = function 
    | ILProp(x) -> il_pinfo_has_setter x
    | FSProp(_,_,x) -> x <> None

let pinfo_is_static = function 
    | ILProp(x) -> il_pinfo_is_static x
    | FSProp(_,Some vref,_) | FSProp(_,_, Some vref) -> not (member_vref_is_instance vref)
    | FSProp(_,None,None) -> failwith "pinfo_is_static: unreachable"

let pinfo_is_definite_fsharp_override = function
    | ILProp _ -> false
    | FSProp(_,Some vref,_) | FSProp(_,_, Some vref) -> vspr_vref_is_definite_override vref
    | FSProp(_,None,None) -> failwith "pinfo_is_definite_fsharp_override: unreachable"

let pinfo_is_indexer g = function 
    | ILProp(ILPropInfo(tinfo,pdef)) -> pdef.propArgs <> []
    | FSProp(typ,Some vref,_) as pinfo ->
        (* A getter has signature  { OptionalObjectType } -> Unit -> PropertyType *)
        (* A getter indexer has signature  { OptionalObjectType } -> TupledIndexerArguments -> PropertyType *)
        length (params_of_vspr_vref g vref) >= 1
    | FSProp(typ,_, Some vref) -> 
        (* A setter has signature  { OptionalObjectType } -> PropertyType -> Void *)
        (* A setter indexer has signature  { OptionalObjectType } -> TupledIndexerArguments -> PropertyType -> Void *)
        let _,arginfos,_,_ = dest_member_vref_typ g vref in 
        length arginfos >= 2 
    | FSProp(typ,None,None) -> failwith "pinfo_is_indexer: unreachable"

let param_typs_of_pinfo g amap m = function
    | ILProp (x) -> param_typs_of_il_pinfo amap m x
    | FSProp (typ,Some vref,_) | FSProp (typ,_,Some vref) -> 
        arginfos_of_propery_val g (deref_val vref) |> List.map param_of_arginfo 
    | FSProp(typ,None,None) -> failwith "param_typs_of_pinfo: unreachable"

let vtyp_of_pinfo g amap m = function
    | ILProp (x) -> vtyp_of_il_pinfo amap m x
    | FSProp (typ,Some vref,_) | FSProp (typ,_,Some vref) -> vtyp_of_property_val g (deref_val vref)
    | FSProp(typ,None,None) -> failwith "vtyp_typs_of_pinfo: unreachable"

let xmldoc_of_pinfo = function 
    | ILProp(x) -> emptyXMLDoc
    | FSProp(typ,Some vref,_) | FSProp(typ,_, Some vref) -> xmldoc_of_vref vref
    | FSProp(typ,None,None) -> failwith "unreachable"

let xmldoc_of_minfo = function 
    | ILMeth(x) -> emptyXMLDoc
    | FSMeth(_,vref) -> xmldoc_of_vref vref
    | DefaultStructCtor _ -> emptyXMLDoc

let arb_vref_of_pinfo = function 
    | ILProp(x) -> None
    | FSProp(typ,Some vref,_) | FSProp(typ,_, Some vref) -> Some(vref)
    | FSProp(typ,None,None) -> failwith "unreachable"

let arb_vref_of_minfo = function 
    | ILMeth(x) -> None
    | FSMeth(_,vref) -> Some(vref)
    | DefaultStructCtor typ -> None


(*-------------------------------------------------------------------------
!* finfo
 *------------------------------------------------------------------------- *)

let tinfo_of_il_finfo (ILFieldInfo(tinfo,_)) = tinfo
let fdef_of_il_finfo (ILFieldInfo(_,pd)) = pd
let scoref_of_il_finfo = tinfo_of_il_finfo >> scoref_of_il_tinfo
let tdef_of_il_finfo = tinfo_of_il_finfo >> tdef_of_il_tinfo
let tref_of_il_finfo = tinfo_of_il_finfo >> tref_of_il_tinfo
let tinst_of_il_finfo = tinfo_of_il_finfo >> tinst_of_il_tinfo
let name_of_il_finfo x = (fdef_of_il_finfo x).fdName
let initonly_of_il_finfo x = (fdef_of_il_finfo x).fdInitOnly
let il_finfo_is_struct = tinfo_of_il_finfo >> il_tinfo_is_struct
let il_finfo_is_static finfo = (fdef_of_il_finfo finfo).fdStatic
let il_finfo_literal_value finfo = 
    let fdef = fdef_of_il_finfo finfo in 
    if fdef.fdLiteral then fdef.fdInit else None
   
let vtyp_of_il_finfo amap m (ILFieldInfo (tinfo,fdef)) =
    import_typ_from_metadata amap m tinfo [] fdef.fdType

let fref_of_il_finfo finfo =
    let fref = mk_fref_in_tref(tref_of_il_finfo finfo,name_of_il_finfo finfo,(fdef_of_il_finfo finfo).fdType) in 
    rescope_fref (scoref_of_il_finfo finfo) fref 


(*-------------------------------------------------------------------------
!* rfinfo
 *------------------------------------------------------------------------- *)

let tinst_of_rfinfo (RecdFieldInfo(tinst,_)) = tinst
let rfref_of_rfinfo (RecdFieldInfo(_,rfref)) = rfref
let rfinfo_is_static (RecdFieldInfo(_,rfref)) = static_of_rfref rfref 
let rfinfo_literal_value (RecdFieldInfo(_,rfref)) = literal_value_of_rfref rfref 
let rfield_of_rfinfo (RecdFieldInfo(_,rfref)) = rfield_of_rfref rfref
let tcref_of_rfinfo (RecdFieldInfo(_,rfref)) = tcref_of_rfref rfref
let rfinfo_is_struct x = x |> tcref_of_rfinfo |> deref_tycon |> is_struct_tycon
let name_of_rfinfo x = (rfield_of_rfinfo x).rfield_id.idText
let vtyp_of_rfinfo (RecdFieldInfo (tinst,fref)) = actual_rtyp_of_rfref fref tinst
let enclosing_vtyp_of_rfinfo (RecdFieldInfo (tinst,fref)) = TType_app (tcref_of_rfref fref,tinst)
let pattribs_of_rfinfo x = x |> rfref_of_rfinfo |> pattribs_of_rfref 

(*-------------------------------------------------------------------------
!* Printing
 *------------------------------------------------------------------------- *)

open Printf

let boutput_minfo g amap m denv os minfo =
    let fminst = formal_minst_of_minfo g minfo in 
    match minfo with 
    | DefaultStructCtor typ -> 
        bprintf os "%a()" (NicePrint.output_tcref denv) (tcref_of_stripped_typ (typ_of_minfo minfo));
    | FSMeth(_,vref) -> 
        NicePrint.output_qualified_val_spec denv os (deref_val vref)
    | ILMeth(x) -> 
        let rty = ret_typ_of_minfo g amap m minfo fminst in 
        bprintf os "%a" (NicePrint.output_tcref denv) (tcref_of_stripped_typ (typ_of_minfo minfo));
        if name_of_minfo minfo = ".ctor" then  
          bprintf os "("
        else
          bprintf os ".%a(" (NicePrint.output_typars denv (name_of_minfo minfo)) (formal_mtps_of_minfo g minfo);
        let first = ref true in
        iter
          (fun (pn,pty) -> 
            if not !first then bprintf os ", "; first := false;
            bprintf os "%a" (NicePrint.output_typ denv) pty;
            match pn with 
            | None -> ()
            | Some n -> bprintf os " %s" n)
          (params_of_minfo g amap m minfo fminst);
        bprintf os ") : %a"  (NicePrint.output_typ denv) rty

let string_of_minfo g amap m denv d = bufs (fun buf -> boutput_minfo g amap m denv buf d)


(*-------------------------------------------------------------------------
!* Basic accessibility logic
 *------------------------------------------------------------------------- *)

(* What keys do we have to access other constructs? *)
type accessorDomain = 
    | AccessibleFrom of 
        compilation_path * (* we have the keys to access any members private to the given path *)
        tycon_ref option   (* we have the keys to access any protected members of the super types of 'tycon_ref' *)
    | AccessibleFromEverywhere
    | AccessibleFromSomewhere

let il_tyaccess_accessible access =
    access = TypeAccess_public || access= TypeAccess_nested MemAccess_public

let taccess_accessible ad taccess = 
    match ad with 
    | AccessibleFromEverywhere -> can_access_from_everywhere taccess
    | AccessibleFromSomewhere -> can_access_from_somewhere taccess
    | AccessibleFrom (cpath,tcrefViewedFromOption) -> 
        (* TODO: protected access in F# code *)
        can_access_from taccess cpath


let memaccess_accessible g amap m (ILTypeInfo(tcrefOfViewedItem,_,_,_)) ad access = 
    match ad with 
    | AccessibleFromEverywhere -> 
          access = MemAccess_public
    | AccessibleFromSomewhere -> 
         (access = MemAccess_public || 
          access = MemAccess_family  || 
          access = MemAccess_famorassem) 
    | AccessibleFrom (cpath,tcrefViewedFromOption) ->
         (access = MemAccess_public) ||
          ((access = MemAccess_family  || 
            access = MemAccess_famorassem) &&
           match tcrefViewedFromOption with 
           | None -> false
           | Some tcrefViewedFrom ->
              exists_in_entire_hierarchy_of_typ (fun typ -> is_stripped_tyapp_typ typ && g.tcref_eq (tcref_of_stripped_typ typ) tcrefOfViewedItem) g amap m (generalize_tcref tcrefViewedFrom |> snd))

(*-------------------------------------------------------------------------
!* Accessibility of infos
 *------------------------------------------------------------------------- *)

let tdef_accessible tdef =
    il_tyaccess_accessible tdef.tdAccess 


let tcref_accessible ad tcref = 
    if is_il_tcref tcref then 
        (let scoref,enc,tdef = dest_il_tcref tcref in  
         List.for_all tdef_accessible enc && 
         tdef_accessible tdef)
    else  
         tcref |> deref_tycon |> access_of_tycon |> taccess_accessible ad

let tcref_accessible_check m ad tcref =
    if not (tcref_accessible ad tcref) then  
        errorR(Error("The type '"^display_name_of_tcref tcref^"' is not accessible from this code location",m))

let tcref_repr_accessible_check m ad tcref =
    tcref_accessible_check m ad tcref;
    if not (taccess_accessible ad (repr_access_of_tycon (deref_tycon tcref))) then
        errorR (Error ("The data constructors, fields and other representation details of the type '"^display_name_of_tcref tcref^"' is not accessible from this code location",m))
        
let tcref_of_typ_accessible ad ty = 
    not (is_stripped_tyapp_typ ty) or
    tcref_accessible ad (fst (dest_stripped_tyapp_typ ty))

let rec typ_accessible ad ty = 
    not (is_stripped_tyapp_typ ty) or
    let tcref,tinst = dest_stripped_tyapp_typ ty in 
    tcref_accessible ad tcref && tinst_accessible ad tinst
and tinst_accessible ad tinst = 
    match tinst with [] -> true | _ -> List.for_all (typ_accessible ad) tinst

let il_tinfo_accessible ad (ILTypeInfo(_,_,tinst,tdef)) = 
    tdef_accessible tdef
                   
let il_mem_accessible g amap m ad tinfo access = 
    il_tinfo_accessible ad tinfo && memaccess_accessible g amap m tinfo ad access

let il_finfo_accessible g amap m ad (ILFieldInfo (tinfo,fd)) =
    il_mem_accessible g amap m ad tinfo fd.fdAccess

let il_einfo_accessible g amap m ad (ILEventInfo (tinfo,edef)) =
    il_mem_accessible g amap m ad tinfo (edef_accessibility (tdef_of_il_tinfo tinfo) edef)

let il_minfo_accessible g amap m ad (ILMethInfo (tinfo,mdef,_)) =
    il_mem_accessible g amap m ad tinfo mdef.mdAccess 

let il_pinfo_accessible g amap m ad (ILPropInfo(tinfo,pdef)) =
    il_mem_accessible g amap m ad tinfo (pdef_accessibility (tdef_of_il_tinfo tinfo) pdef)

let vref_accessible m ad vref = 
    vref |> deref_val |> access_of_val |> taccess_accessible ad

let vref_accessible_check  m ad vref = 
    if not (vref_accessible m ad vref) then 
        errorR (Error ("The value '"^name_of_vref vref^"' is not accessible from this code location",m))
    
let ucref_accessible_check m ad ucref =
    tcref_repr_accessible_check m ad (tcref_of_ucref ucref);
    if not (taccess_accessible ad (access_of_uconstr (uconstr_of_ucref ucref))) then 
        errorR (Error ("The data constructor '"^name_of_ucref ucref^"' is not accessible from this code location",m))

let rfref_accessible_check m ad rfref =
    tcref_repr_accessible_check m ad (tcref_of_rfref rfref);
    if not (taccess_accessible ad (access_of_rfield (rfield_of_rfref rfref))) then 
        errorR (Error ("The record field '"^name_of_rfref rfref^"' is not accessible from this code location",m))

let rfinfo_accessible_check m ad rfinfo = 
    rfref_accessible_check m ad (rfref_of_rfinfo rfinfo)

let il_finfo_accessible_check g amap m ad finfo =
    if not (il_finfo_accessible g amap m ad finfo) then 
        errorR (Error ("field '"^name_of_il_finfo finfo^"' is not accessible from this code location",m))

let minfo_accessible g amap m ad = function 
    | ILMeth (x) -> il_minfo_accessible g amap m ad x
    | FSMeth (_,vref) -> vref_accessible m ad vref
    | DefaultStructCtor typ -> typ_accessible ad typ

let pinfo_accessible g amap m ad = function 
    | ILProp (x) -> il_pinfo_accessible g amap m ad x
    | FSProp (_,Some vref,_) 
    | FSProp (_,_,Some vref) -> vref_accessible m ad vref
    | _ -> false

(*-------------------------------------------------------------------------
!* Check custom attributes
 *------------------------------------------------------------------------- *)

exception Obsolete of string * range
exception OverrideInAugmentation of range

let minfo_bind_attribs minfo f1 f2 = 
    match minfo with 
    | ILMeth (x) -> f1 (mdef_of_il_minfo x).mdCustomAttrs 
    | FSMeth (_,vref) -> f2 (attribs_of_vref vref)
    | DefaultStructCtor typ -> f2 []

let minfo_bind_attrib g (AttribInfo(atref,_) as attribSpec) minfo f1 f2 = 
    minfo_bind_attribs minfo 
        (fun ilAttribs -> ilthing_decode_il_attrib g atref ilAttribs |> Option.bind f1)
        (fun fsAttribs -> fsthing_tryfind_attrib g attribSpec fsAttribs |> Option.bind f2)


let il_attrib_check g cattrs m = 
    let (AttribInfo(tref,_)) = g.attrib_SystemObsolete in
    match ilthing_decode_il_attrib g tref cattrs with 
    | Some ([CustomElem_string (Some(msg)) ],_) -> 
         warnD(Obsolete(msg,m))
    | Some ([CustomElem_string (Some(msg)); CustomElem_bool isError ],_) -> 
        (if isError then errorD else warnD) (Obsolete(msg,m))
    | Some ([CustomElem_string None ],_) -> 
         warnD(Obsolete("",m))
    | Some _ -> 
        warnD(Obsolete("",m))
    | None -> 
        completeD
  
let fs_attrib_check g attribs m = 
    if isNil attribs then completeD
    else begin
        (match fsthing_tryfind_attrib g g.attrib_SystemObsolete attribs with
        | Some(Attrib(_,[ TExpr_const (TConst_string(bytes),_,_) ],_)) ->
            warnD(Obsolete(Bytes.unicode_bytes_as_string bytes,m))
        | Some(Attrib(_,[ TExpr_const (TConst_string(bytes),_,_); TExpr_const (TConst_bool(isError),_,_) ],_)) -> 
            (if isError then errorD else warnD) (Obsolete(Bytes.unicode_bytes_as_string bytes,m))
        | Some _ -> 
            warnD(Obsolete("", m))
        | None -> 
            completeD
        ) ++ (fun () -> 
        
        match fsthing_tryfind_attrib g g.attrib_OCamlCompatibilityAttribute attribs with
        | Some(Attrib(_,[ TExpr_const (TConst_string(bytes),_,_) ],_)) -> 
            warnD(OCamlCompatibility(Bytes.unicode_bytes_as_string bytes,m))
        | Some _ -> 
            warnD(OCamlCompatibility("This construct is for OCaml-compatibility", m))
        | None -> 
            completeD
        ) ++ (fun () -> 
        
        match fsthing_tryfind_attrib g g.attrib_ExperimentalAttribute attribs with
        | Some(Attrib(_,[ TExpr_const (TConst_string(bytes),_,_) ],_)) -> 
            warnD(Experimental(Bytes.unicode_bytes_as_string bytes,m))
        | Some _ -> 
            warnD(Experimental("This construct is experimental", m))
        | _ ->  
            completeD
        )
    end
  
let pinfo_attrib_check g pinfo m = 
    match pinfo with
    | ILProp (ILPropInfo(tinfo,pdef)) -> il_attrib_check g pdef.propCustomAttrs m
    | FSProp (typ,Some vref,_) 
    | FSProp (typ,_,Some vref) -> fs_attrib_check g (attribs_of_vref vref) m
    | FSProp(typ,None,None) -> failwith "pinfo_attrib_check: unreachable"
  
let il_finfo_attrib_check g finfo m = 
    il_attrib_check g (fdef_of_il_finfo finfo).fdCustomAttrs m |> commitOperationResult

let minfo_attrib_check g m minfo = 
    match minfo_bind_attribs minfo 
              (fun ilAttribs -> Some(il_attrib_check g ilAttribs m)) 
              (fun fsAttribs -> Some(fs_attrib_check g fsAttribs m)) with
    | Some res -> res
    | None -> completeD (* no attribute = no errors *)

 
let tcref_attrib_check g tcref m = 
    if is_il_tcref tcref then 
        let tdef = tdef_of_il_tcref tcref in 
        il_attrib_check g tdef.tdCustomAttrs m
    else fs_attrib_check g (attribs_of_tcref tcref) m


(*-------------------------------------------------------------------------
!* Build calls to F# methods
 *------------------------------------------------------------------------- *)

let mk_fs_minfo_app g m vref vexp vexprty (args: expr list) =
    (* Consume the arguments in chunks and build applications.  This copes with various F# calling signatures *)
    (* all of which ultimately become 'methods' *)
    (* QUERY: this looks overly complex considering that we are doing a fundamentally simple *)
    (* thing here. *)
    let arities =  TopValData.aritiesOfArgs (arity2_of_val (deref_val vref)) in 
    
    let args3,(leftover,rty) = 
      map_acc_list 
        (fun (args,fty) arity -> 
          match arity,args with 
          | (0|1),[] when type_equiv g (domain_of_fun_typ fty) g.unit_ty -> mk_unit g m, (args, range_of_fun_typ fty)
          | 0,_ -> error(Error("Unexpected zero arity",m))
          | 1,arg :: argst -> arg, (argst, range_of_fun_typ fty)
          | 1,[] -> error(Error("expected additional arguments here",m))
          | n,args -> 
              if length args < n then error(Error("internal error in getting arguments, n = "^string_of_int n^", #args = "^string_of_int (length args),m));
              let tupargs,argst = chop_at n args in 
              let tuptys = map (type_of_expr g) tupargs in 
              (mk_tupled g m tupargs tuptys),
              (argst, range_of_fun_typ fty) )
        (args,vexprty)
        arities in 
    if leftover <> [] then error(Error("unexpected "^string_of_int (length leftover)^" remaining arguments in method application",m));
    mk_appl((vexp,vexprty),[],args3,m), 
    rty
    
let mk_fs_minfo_call g m (typ,vref) vFlags minst args =
    let vexp = TExpr_val (vref,vFlags,m) in 
    let vexpty = (type_of_vref vref) in 
    let tpsorig,tau =  try_dest_forall_typ vexpty in 
    let vtinst = tinst_of_stripped_typ typ @ minst in
    if length tpsorig <> length vtinst then error(Error("mk_fs_minfo_call: unexpected length mismatch",m));
    let expr = mk_tyapp m (vexp,vexpty) vtinst in 
    let exprty = inst_type (mk_typar_inst tpsorig vtinst) tau in 
    (* REVIEW: this is passing in the instantiated type.  This is not correct - it should be the formal type? *)
    mk_fs_minfo_app g m vref expr exprty args
    
(*-------------------------------------------------------------------------
!* Sets of methods up the hierarchy, ignoring duplicates by name and sig.
 * Used to collect sets of virtual methods, protected methods, protected
 * properties etc. 
 *  REVIEW: this code generalizes the iteration used below for member lookup.
 *  REVIEW: this doesn't take into account newslot decls.
 *------------------------------------------------------------------------- *)

let member_is_interface_impl membInfo = 
    match membInfo.vspr_implements_slotsig with 
    | None -> false
    | Some slotsig -> is_interface_typ (enclosing_typ_of_slotsig slotsig) 

let select_from_augmentation_vrefs f tcref = 
    let aug = tcaug_of_tcref tcref in 
    (* tcaug_adhoc cleanup: this should select from all accessible/available vrefs *)
    (* that are part of any augmentation of this type. *)
    let vrefs = Namemap.range_multi aug.tcaug_adhoc  in 
    
    chooseList (fun vref ->
        match member_info_of_vref vref with 
        (* Workaround the fact that values providing override and interface implementations are somewhat needlessly being published in inferred module types *)
        (* These cannot be selected directly via the "." notation. *)
        (* However, it certainly is useful to be able to publish these values, as we can in theory *)
        (* optimize code to make direct calls to these methods. *)
        | Some membInfo when not (member_is_interface_impl membInfo) -> f membInfo vref
    (*    | Some membInfo -> f membInfo vref *)
        | _ ->  None) vrefs 

let checkFilter optFilter nm = match optFilter with None -> true | Some n2 -> nm = n2

type privateFilter = DontIncludePrivate | IncludePrivate

let select_vref_minfo g optFilter typ membInfo vref =
    if checkFilter optFilter membInfo.vspr_il_name then 
        (* if not polyrec && vrec_of_vref vref then 
          Some(FSMeth(typ,vref))
        else 
        *)
        let tps,_,_,_ = dest_member_vref_typ g vref in 
        let ntps = length tps in 
        let tinst = tinst_of_stripped_typ typ in 
        let ntinst = length tinst in 
        (* POLYREC: if ntinst >= ntps then warning(Error(Printf.sprintf "internal error: typar count mismatch, ntinst = %d, ntps = %d" ntinst ntps,m)); *)
        Some(FSMeth(typ,vref))
    else None

let immediate_intrinsic_minfos_of_typ (optFilter,includePrivate) g amap m typ =
     let minfos =
         if is_il_named_typ typ then 
             let tinfo = tinfo_of_il_typ typ in 
             let mdefs = (match optFilter with None -> dest_mdefs | Some(nm) -> find_mdefs_by_name nm)  (tdef_of_il_tinfo tinfo).tdMethodDefs in
             map (mk_il_minfo amap m tinfo) mdefs
         else if not (is_stripped_tyapp_typ typ) then []
         else select_from_augmentation_vrefs (select_vref_minfo g optFilter typ) (tcref_of_stripped_typ typ) in
     let minfos = if includePrivate = IncludePrivate then minfos else minfos |> filter (minfo_accessible g amap m AccessibleFromSomewhere) in
     minfos

let raw_intrinsic_minfos_of_typ (optFilter,includePrivate) g amap m typ =
    fold_primary_hierarchy_of_typ (fun typ acc -> 
         let minfos = immediate_intrinsic_minfos_of_typ (optFilter,includePrivate) g amap m typ in
         minfos @ acc)
      g amap m 
      typ
      []


(* QUERY: surely there's a much better way to do this! *)
(* Use these gross tables to join up getters and setters which are not associated in the F# data structure *)
let gather_vref_pinfos typ optFilter vsprVrefIterator = 
    let setters = Hashtbl.create 10 in 
    let getters = Hashtbl.create 10 in 
    (* 'pinfoIterator' is an iteration function, which pushes each pinfo getter/setter vref we find into a hash table *)
    vsprVrefIterator (fun membInfo vref -> 
        if membInfo.vspr_flags.memFlagsKind = MemberKindPropertyGet then begin
          let nm = (property_name_of_vspr_vref vref) in 
          if checkFilter optFilter nm then 
              Hashtbl.add getters nm vref;
        end else if membInfo.vspr_flags.memFlagsKind = MemberKindPropertySet then begin
          let nm = (property_name_of_vspr_vref vref) in 
          if checkFilter optFilter nm then 
              Hashtbl.add setters nm vref;
        end);
    let keys = 
      Hashtbl.fold (fun x _ acc -> Zset.add x acc) getters  
        (Hashtbl.fold (fun x _ acc -> Zset.add x acc) setters 
          (Zset.empty string_ord) ) in 
    let res = ref []  in
    Zset.iter (fun key -> 
      let getter = if Hashtbl.mem getters key then Some(Hashtbl.find getters key) else None in 
      let setter = if Hashtbl.mem setters key then Some(Hashtbl.find setters key) else None in 
      res := FSProp(typ,getter,setter)  :: !res) keys;
    !res 

let raw_intrinsic_pinfos_of_typ (optFilter,includePrivate) g amap m typ =
    fold_primary_hierarchy_of_typ (fun typ acc -> 
         let pinfos =
           if is_il_named_typ typ then 
               let tinfo = tinfo_of_il_typ typ in 
               let pdefs = (match optFilter with None -> dest_pdefs | Some(nm) -> find_pdefs nm)  (tdef_of_il_tinfo tinfo).tdProperties in 
               map (fun pd -> ILProp(ILPropInfo(tinfo,pd))) pdefs
           else if not (is_stripped_tyapp_typ typ) then []
           else
             gather_vref_pinfos typ optFilter (fun gather ->
                 let _ = 
                   select_from_augmentation_vrefs 
                     (fun membInfo vref -> gather membInfo vref; None)
                     (tcref_of_stripped_typ typ) in
                 ()) in
               
         let pinfos = if includePrivate = IncludePrivate then pinfos else pinfos |> filter (pinfo_accessible g amap m AccessibleFromSomewhere) in
         pinfos @ acc)
      g amap m
      typ
      []

let il_finfos_of_typ (optFilter,includePrivate) g amap m typ =
    fold_primary_hierarchy_of_typ 
      (fun typ acc -> 
         let infos =
             if is_il_named_typ typ then 
                 let tinfo = tinfo_of_il_typ typ in 
                 let fdefs = (match optFilter with None -> dest_fdefs | Some(nm) -> find_fdefs nm)  (tdef_of_il_tinfo tinfo).tdFieldDefs in 
                 map (fun pd -> ILFieldInfo(tinfo,pd)) fdefs
             else if not (is_stripped_tyapp_typ typ) then []
             else [] in
         let infos = if includePrivate = IncludePrivate then infos else infos |> filter (il_finfo_accessible g amap m  AccessibleFromSomewhere) in
         infos @ acc)          
      g amap m
      typ
      []

let mk_rfinfo typ tcref fspec = RecdFieldInfo(tinst_of_stripped_typ typ,rfref_of_rfield tcref fspec)

let (++++) x y = match x with None -> y | _ -> x

let rfinfo_of_typ_by_name g amap m typ nm =
    fold_primary_hierarchy_of_typ 
      (fun typ acc -> 
          (if is_stripped_tyapp_typ typ then 
             let tcref = tcref_of_stripped_typ typ in 
             (* Note;secret fields are not allowed in lookups here, as we're only looking *)
             (* up user-visible fields in name resolution. *)
             match any_rfield_of_tcref_by_name tcref nm with
             | Some rfield when not (secret_of_rfield rfield) -> Some (mk_rfinfo typ tcref rfield)
             | _ -> None
           else None) ++++ acc)          
      g amap m
      typ
      None


let il_einfos_of_typ (optFilter,includePrivate) g amap m typ =
    fold_primary_hierarchy_of_typ 
      (fun typ acc -> 
         let infos =
           if is_il_named_typ typ then 
             let tinfo = tinfo_of_il_typ typ in 
             let edefs = (match optFilter with None -> dest_edefs | Some(nm) -> find_edefs nm)  (tdef_of_il_tinfo tinfo).tdEvents in 
               map (fun pd -> ILEventInfo(tinfo,pd)) edefs 
           else if not (is_stripped_tyapp_typ typ) then []
           else [] in
         let infos = if includePrivate = IncludePrivate then infos else infos |> filter (il_einfo_accessible g amap m AccessibleFromSomewhere) in
         infos @ acc)          
      g amap m
      typ
      []

(*-------------------------------------------------------------------------
!* Constructor infos
 *------------------------------------------------------------------------- *)


let cinfos_of_il_typ amap m typ = 
    let tdef = tdef_of_il_typ  typ in 
    tdef 
    |> Il.methods_of_tdef 
    |> Il.dest_mdefs 
    |> filter (fun md -> match md.mdKind with MethodKind_ctor -> true | _ -> false) 
    |> map (mk_il_minfo amap m (tinfo_of_il_typ typ)) 
    
let intrinsic_cinfos_of_typ amap m ty = 
    if verbose then   dprintf0 "--> intrinsic_cinfos_of_typ\n"; 
    if is_stripped_tyapp_typ ty then
        if is_il_named_typ ty then 
            cinfos_of_il_typ amap m ty
        else 
            let tcref = tcref_of_stripped_typ ty in
            let nm = ".ctor" in
            let aug = tcaug_of_tcref tcref  in 
            (* tcaug_adhoc cleanup: this should select from all accessible/available vrefs *)
            (* that are part of any augmentation of this type. That's assuming that constructors can *)
            (* be in augmentations. *)
            let vrefs = Namemap.find_multi nm aug.tcaug_adhoc in 
            let vrefs = chooseList(fun vref -> 
                match member_info_of_vref vref with 
                | Some membInfo when (membInfo.vspr_flags.memFlagsKind = MemberKindConstructor) -> Some(vref) 
                | _ -> None) vrefs in
            map (fun x -> FSMeth(ty,x)) vrefs  
    else []
    

(*-------------------------------------------------------------------------
!* Method signatures
 *------------------------------------------------------------------------- *)

let argtys_of_pinfo g amap m pinfo = map snd (param_typs_of_pinfo g amap m pinfo) 

let sig_of_minfo g amap m minfo = 
    let fmtps = formal_mtps_of_minfo g minfo in 
    let fminst = generalize_typars fmtps in 
    let vargtys = argtys_of_minfo g amap m minfo fminst in 
    let vrty = ret_typ_of_minfo g amap m minfo fminst in 

    (* The formal method typars returned are completely formal - they don't take into account the instantiation *)
    (* of the enclosinng type. For example, they may have constraints involving the _formal_ type parameters *)
    (* of the enclosing type. This instaniations can be used to interpret those type parameters *)
    let fmtpinst = 
        let tinst = tinst_of_stripped_typ (typ_of_minfo minfo) in
        let ttps  =           
            match minfo with 
            | FSMeth(typ,vref) -> 
                let ttps,mtps,rty,tinst = dest_fs_minfo g (typ,vref) in
                ttps
            | _ -> 
                typars_of_tcref (tcref_of_stripped_typ (typ_of_minfo minfo)) in
        mk_typar_inst ttps tinst in 
            
    vargtys,vrty,fmtps,fmtpinst

let sig_of_pinfo g amap m pinfo = 
    let vargtys = argtys_of_pinfo g amap m pinfo in 
    let vrty = vtyp_of_pinfo g amap m pinfo in 
    vargtys,vrty 

(** Used to hide/filter members from super classes based on signature *)
let minfos_equiv_by_name_and_partial_sig g amap m minfo minfo2 = 
    (name_of_minfo minfo = name_of_minfo minfo2) &&
    (generic_arity_of_minfo g minfo = generic_arity_of_minfo g minfo2) &&
    let fmtps = formal_mtps_of_minfo g minfo in 
    let fminst = generalize_typars fmtps in 
    let fmtps2 = formal_mtps_of_minfo g minfo2 in 
    let fminst2 = generalize_typars fmtps2 in 
    let argtys = argtys_of_minfo g amap m minfo fminst in 
    let argtys2 = argtys_of_minfo g amap m minfo2 fminst2 in 
    (length argtys = length argtys2) &&
    for_all2 (type_aequiv g (mk_tyeq_env fmtps fmtps2)) argtys argtys2 

(** Used to hide/filter members from super classes based on signature *)
let minfos_equiv_by_name_and_sig g amap m minfo minfo2 = 
    minfos_equiv_by_name_and_partial_sig g amap m minfo minfo2 &&
    let argtys,rty,fmtps,_ = sig_of_minfo g amap m minfo in 
    let argtys2,rty2,fmtps2,_ = sig_of_minfo g amap m minfo2 in 
    type_aequiv g (mk_tyeq_env fmtps fmtps2) rty rty2 

(** Used to hide/filter members from super classes based on signature *)
let pinfos_equiv_by_name_and_partial_sig g amap m pinfo pinfo2 = 
    name_of_pinfo pinfo = name_of_pinfo pinfo2 &&
    let argtys = argtys_of_pinfo g amap m pinfo in 
    let argtys2 = argtys_of_pinfo g amap m pinfo2 in 
    length argtys = length argtys2 &&
    for_all2 (type_equiv g) argtys argtys2 
  
(** Used to hide/filter members from super classes based on signature *)
let pinfos_equiv_by_name_and_sig g amap m pinfo pinfo2 = 
    pinfos_equiv_by_name_and_partial_sig g amap m pinfo pinfo2 &&
    let rty = vtyp_of_pinfo g amap m pinfo in 
    let rty2 = vtyp_of_pinfo g amap m pinfo2 in 
    type_equiv g rty rty2 
  
(* nb. Prefer items toward the top of the hierarchy if the items are virtual *)
(* but not when resolving base calls (OO goop!  ack!). Also get overrides instead *)
(* of abstract slots when measuring whether a class/interface implements all its *)
(* required slots. *)

type findMemberFlag = 
  | IgnoreOverrides 
  | PreferOverrides

(* The input list is sorted from most-derived to least-derived type, so any System.Object methods *)
(* are at the end of the list. Return a filtered list where prior/subsequent members matching by name and *)
(* that are in the same equivalence class have been removed. We keep a name-indexed table to *)
(* be more efficient when we check to see if we've already seen a particular named method. *)

let addIndexedList equiv nmf item (res,resIndexed) = 
    (* Have we already seen an item with the same name and that is in the same equivalence class?*)
    (* If so, ignore this one. *)
    let nm = nmf item in
    if exists (equiv item) (Namemap.find_multi nm resIndexed) then res,resIndexed
    else item :: res,Namemap.add_multi nm item resIndexed  

let emptyIndexedList = [],Namemap.empty_multi

let gen_exclude_prior equiv nmf = 
    let rec loop items = 
        match items with
        | [] -> emptyIndexedList
        | item :: rest -> addIndexedList equiv nmf item (loop rest) in
    fun items -> (loop items |> fst)

let gen_exclude_subsequent equiv nmf = 
    let rec loop items acc = 
        match items with
        | [] -> List.rev (fst acc)
        | item :: rest ->  loop rest (addIndexedList equiv nmf item acc) in
    fun items -> loop items emptyIndexedList


(** Used to find all method declarations *)
let gen_filter_ignore_overrides (isvirt,isNewSlot,isDefiniteOverride,equivSigs,nmf) items = 
    gen_exclude_prior (fun x y -> isvirt x && isvirt y && not (isNewSlot x) && (isDefiniteOverride x || equivSigs x y)) nmf items

(** Used to find all method implementations *)
let gen_filter_prefer_overrides (isvirt,equiv,nmf) items = 
    gen_exclude_subsequent (fun x y -> isvirt x && isvirt y && equiv x y) nmf items 

let gen_filter_overrides findFlag (isvirt,isNewSlot,isDefiniteOverride,equiv,nmf) rest = 
    match findFlag with 
    | PreferOverrides -> gen_filter_prefer_overrides (isvirt,equiv,nmf) rest 
    | IgnoreOverrides ->  gen_filter_ignore_overrides (isvirt,isNewSlot,isDefiniteOverride,equiv,nmf) rest
    
let filter_overrides_of_minfos findFlag g amap m minfos = 
    gen_filter_overrides findFlag (minfo_is_virt,minfo_is_newslot,minfo_is_definite_override,minfos_equiv_by_name_and_sig g amap m,name_of_minfo) minfos

let filter_overrides_of_pinfos findFlag g amap m props = 
    gen_filter_overrides findFlag (pinfo_is_virt,pinfo_is_newslot,pinfo_is_definite_fsharp_override,pinfos_equiv_by_name_and_sig g amap m, name_of_pinfo) props

let exclude_hidden_of_minfos g amap m minfos = 
    gen_exclude_subsequent 
        (fun m1 m2 -> 
             (* only hide those truly from super classes *)
             not (g.tcref_eq (tcref_of_stripped_typ (typ_of_minfo m1)) (tcref_of_stripped_typ (typ_of_minfo m2))) &&
             minfos_equiv_by_name_and_partial_sig g amap m m1 m2)
        name_of_minfo 
        minfos

let exclude_hidden_of_pinfos g amap m props = 
    gen_exclude_subsequent (pinfos_equiv_by_name_and_partial_sig g amap m) name_of_pinfo props


let intrinsic_minfos_of_typ optFilter findFlag g amap m typ = 
    typ |>
    raw_intrinsic_minfos_of_typ optFilter g amap m |> 
    filter_overrides_of_minfos findFlag g amap m
  
let intrinsic_pinfos_of_typ optFilter findFlag g amap m typ = 
    typ |>
    raw_intrinsic_pinfos_of_typ optFilter g amap m |> 
    filter_overrides_of_pinfos findFlag g amap m

(* Try to detect the existence of a method on a type *)
(* Used for *)
(*     -- getting the GetEnumerator, get_Current, MoveNext methods for enumerable types *)
(*     -- getting the Dispose method when resolving the 'use' construct *)
(*     -- getting the various methods used to desugar the computation expression syntax *)
let try_find_minfo g amap m nm ty = 
    intrinsic_minfos_of_typ (Some(nm),DontIncludePrivate) IgnoreOverrides g amap m ty 


(* Make a call to a method info. Used by the optimizer only to build *)
(* calls to the type-directed resolutions of overloaded operators *)
let mk_minfo_call g amap m minfo minst args =
    let vFlags = NormalValUse in (* correct unless if we allow wild trait constraints like "T has a ctor and can be used as a parent class" *)
    match minfo with 
    | ILMeth(ilminfo) -> 
        let direct = not (minfo_is_virt minfo) in 
        let isProp = false in (* not necessarily correct, but this is only used post-creflect where this flag is irrelevant *)
        mk_il_minfo_call g amap m isProp ilminfo vFlags minst direct args |> fst
    | FSMeth(typ,vref) -> 
        mk_fs_minfo_call g m (typ,vref) vFlags minst args |> fst
    | DefaultStructCtor typ -> 
       mk_ilzero (m,typ)

(*-------------------------------------------------------------------------
!* Sig 
 *------------------------------------------------------------------------- *)

(* Given a delegate type work out the minfo, argument types, return type *)
(* and F# function type by looking at the Invoke signature of the delegate. *)
let sig_of_fun_as_delegate g amap delty m =
    let minfo = 
        match intrinsic_minfos_of_typ (Some "Invoke",DontIncludePrivate) IgnoreOverrides g amap m delty with 
        | [h] -> h
        | [] -> error(Error("No Invoke methods found for delegate type",m))
        | h :: _ -> warning(Error("More than one Invoke method found for delegate type",m)); h in 
    
    let minst = [] in  (* a delegate's Invoke method is never generic *)
    let basic_del_argtys = argtys_of_minfo g amap m minfo minst in 
    let del_argtys = if isNil basic_del_argtys then [g.unit_ty] else basic_del_argtys in 
    let del_rty = ret_typ_of_minfo g amap m minfo minst in
    (* REVIEW: this accessibility check is not correct - it is too restrictive when accessing *)
    (* protected delegate types or internal delegate types defined in F# code *)
    if not (tcref_of_typ_accessible AccessibleFromEverywhere delty) then 
        error (Error ("The type '"^name_of_tcref (tcref_of_stripped_typ delty)^"' is not accessible from this code location",m));
        
    minfo_attrib_check g m minfo |> commitOperationResult;
    let fty = mk_iterated_fun_ty del_argtys del_rty in 
    minfo,basic_del_argtys,del_rty,fty

let try_dest_standard_delegate_ty g amap m del_ty =
  let minfo,del_argtys,del_rty,_ = sig_of_fun_as_delegate g amap del_ty m in
  match del_argtys with 
  | senderTy :: argTys when (type_equiv g g.obj_ty senderTy)  -> Some(mk_tupled_ty g argTys,del_rty)
  | _ -> None


(* Create an error object to raise should an event not have the shape expected by the .nET idiom described further below *)
let event_err einfo m = 
  let nm = name_of_il_einfo einfo in
  Error ("The event '"^nm^" has a non-standard delegate type and must be accessed using the explicit add_"^nm^" and remove_"^nm^" methods for the event",m)

(* We take advantage of the following idiom to simplify away the bogus "object" parameter of the 
   of the "Add" methods associated with events.  If you want to access it you
   can use AddHandler instead.
   
   The .NET Framework guidelines indicate that the delegate type used for
   an event should take two parameters, an "object source" parameter
   indicating the source of the event, and an "e" parameter that
   encapsulates any additional information about the event. The type of
   the "e" parameter should derive from the EventArgs class. For events
   that do not use any additional information, the .NET Framework has
   already defined an appropriate delegate type: EventHandler.
   (from http://msdn.microsoft.com/library/default.asp?url=/library/en-us/csref/html/vcwlkEventsTutorial.asp) 
 *)
let is_standard_il_einfo g amap m einfo =
  match try_dest_standard_delegate_ty g amap m (del_typ_of_il_einfo amap m einfo) with
  | Some _ -> true
  | None -> false

(* Get the argument types accepted by an event *)
let args_ty_of_il_einfo g amap m einfo  =
  match try_dest_standard_delegate_ty g amap m (del_typ_of_il_einfo amap m einfo) with
  | Some(argtys,_) -> argtys
  | None -> error(event_err einfo m)

(* Get the type of the event when looked at as if it is a property *)
(* Used when displaying the property in Intellisense *)
let prop_typ_of_il_einfo g amap m einfo =  
  let del_ty = del_typ_of_il_einfo amap m einfo in
  let args_ty = args_ty_of_il_einfo g amap m einfo  in 
  mk_fslib_IDelegateEvent_ty g del_ty args_ty

let rfinfo_attrib_check g rfinfo m = 
    fs_attrib_check g (pattribs_of_rfinfo rfinfo) m
