(* (c) Microsoft Corporation. All rights reserved *)
(*F# 
open Microsoft.Research.AbstractIL 
open Microsoft.Research.AbstractIL.Internal 
open Microsoft.FSharp.Compiler 
module Il = Microsoft.Research.AbstractIL.IL 
module Ildiag = Microsoft.Research.AbstractIL.Diagnostics 
F#*) 

open List
open Tast
open Tastops
open Il
open Env
open Range
open Ast
open Lib

type assemMap = (Range.range -> string -> ccu)


(*-------------------------------------------------------------------------
 * Load an IL assembly into the compiler's internal data structures
 * Careful use is made of laziness here to ensure we don't read the entire IL
 * assembly on startup.
 *-------------------------------------------------------------------------- *)

let ungenericize n = 
  if String.contains n '`' && 
    begin
      let m = String.rindex n '`' in 
      let res = ref (m < String.length n - 1) in 
      for i = m + 1 to String.length n - 1 do
        res := !res && String.get n i >= '0' && String.get n i <= '9';
      done;
      !res
    end then 
    String.sub n 0 (String.rindex n '`')
  else n

let new_il_typar m scoref tdname  gp = 
  let tp = mk_rigid_typar gp.Il.gpName m in
  let constraints = map (fun ty -> TTyparCoercesToType(TTyparSubtypeConstraintFromIL (scoref,ty),m)) gp.gpConstraints in
  let constraints = if gp.gpReferenceTypeConstraint then (TTyparIsReferenceType(m)::constraints) else constraints in
  let constraints = if gp.gpNotNullableValueTypeConstraint then (TTyparIsNotNullableValueType(m)::constraints) else constraints in
  let constraints = if gp.gpDefaultConstructorConstraint then (TTyparRequiresDefaultConstructor(m)::constraints) else constraints in
  fixup_typar_constraints tp constraints;
  tp

let new_il_typars m scoref tdname gps = map (new_il_typar m scoref tdname) gps
let new_il_typars_for_tdef m scoref tdname tdef =  new_il_typars m scoref tdname (Il.gparams_of_tdef tdef)

let multiset_discriminate_and_map nodef tipf l = 
    let partial_keys = Hashtbl.create 10 in
    let buckets = Hashtbl.create 10 in
    let now,later = l |> List.partition (fun (k,v) -> isNil k) in 
    let nowRes = list_map (snd >> tipf) now in
    later |> List.iter (fun (k,v) -> 
       match k with 
      | [] -> failwith "multiset_discriminate_and_map: no more keys!" 
      | h::t -> Hashtbl.replace partial_keys h h; Hashtbl.add buckets h (t,v));
    let laterRes = Hashtbl.fold (fun n _ acc -> nodef n (Hashtbl.find_all buckets n)::acc) partial_keys [] in
    nowRes,laterRes
 

let rec conv_il_tdef m filterByAttr scoref cpath enc nm ltdef  =
    (* Add modules for the nested types, if any *)
    let tdef = Lazy.force ltdef in
    let nested = nested_of_tdef tdef in
    let tdname = name_of_tdef tdef in
    let lmtyp = 
      let ltdefs = dest_lazy_tdefs nested in
      if isNil ltdefs then notlazy (empty_mtype AsNamedType)
      else 
        let cpath = (mk_cpath cpath nm AsNamedType) in 
        lazy (mtyp_of_il_tdefs m filterByAttr scoref cpath (enc@[tdef]) ltdefs)  in 
    (* Add the type itself. *)
    mk_il_type_def_tycon (Some cpath) (nm,m) (new_il_typars_for_tdef m scoref nm tdef) (scoref,enc,tdef) lmtyp 
       

and mtyp_of_il_tdef_list m filterByAttr scoref cpath enc tdefs =
  (* Split into the ones with namespaces and without *)
  let items = tdefs |> List.map (fun (ns,n,x,y) -> (ns,(n,x,y))) in 
  let rec add cpath items = 
      (* This is a non-trivial function.  *)
      (* Add the ones with namespaces in buckets *)
      (* That is, multi-set discriminate based in the first element of the namespace list (e.g. "System") *)
      (* and for each bag resulting from the discrimination fold-in a lazy computation to add the types under that bag *)
      let tycons,namespaceModules = 
         multiset_discriminate_and_map 
            (* nodef - called for each bag, 'n' is the head element of the namespace used
               as a key in the discrimination, tgs is the remaining descriptors.  We create a sub-module for 'n' *)
            (fun n tgs ->
                let modty = lazy (add (mk_cpath cpath n Namespace) tgs) in
                let mspec = new_mspec (Some cpath) taccessPublic (mksyn_id m n) emptyXMLDoc [] modty in 
                mspec)

            (* tipf - called if there are no namespace items left to discriminate on. *)
            (fun (n,attrs,ltdef) -> 
               if (filterByAttr && List.exists is_CompilationMappingAttr (dest_custom_attrs attrs)) 
                   (* Don't add this condition: we need private types since processing private members *)
                   (* may resolve references to private types. We could filter out all private members etc. *)
                   (* but then error messages wouldn't be so good. *)
                   (* or (Lazy.force ltdef).tdAccess = TypeAccess_private *)
               then None
               else Some(conv_il_tdef m filterByAttr scoref cpath enc n ltdef))
            items in 
      let tycons = chooseList (fun x -> x) tycons in 
      (new_mtype (if isNil enc then Namespace else AsNamedType) (tycons@namespaceModules) [] ) in 
    
  add cpath items
 

and mtyp_of_il_tdefs m filterByAttr scoref cpath enc ltdefs =
  mtyp_of_il_tdef_list m filterByAttr scoref cpath enc ltdefs 

let mtyp_of_mainmod_tdefs m filterByAttr scoref modul = 
  mtyp_of_il_tdefs m filterByAttr scoref (CompPath(scoref,[])) [] (dest_lazy_tdefs modul.Il.modulTypeDefs)

let add_auxmod_tdef m filterByAttr aux_mod_loader scoref ce = 
  match 
    try 
       let scoref2,modul = aux_mod_loader scoref ce.exportedTypeScope in 
       Some (scoref2,find_tdef ce.exportedTypeName modul.modulTypeDefs) 
    with _ -> None
  with 
  | None -> 
      error(Error("A reference to the DLL '"^qualified_name_of_scoref ce.exportedTypeScope ^"' is required and must come before the reference to the DLL '" ^ qualified_name_of_scoref scoref ^ "'. The imported type "^ce.exportedTypeName^" has been moved to the first DLL and could not be resolved"  ,m))
  | Some (scoref2,td) -> 
      let ns,n = split_type_name td.tdName in 
      mtyp_of_il_tdef_list m filterByAttr scoref (CompPath(scoref2,[])) [] [(ns,n,td.tdCustomAttrs,notlazy td)] 

let mtyp_of_il_auxmod_tdefs m filterByAttr aux_mod_loader scoref ces = 
  list_map (add_auxmod_tdef m filterByAttr aux_mod_loader scoref) (dest_exported_types ces) 

let mtyp_of_il_mainmod m filterByAttr aux_mod_loader aref mainmod = 
  let scoref = ScopeRef_assembly aref in 
  let auxmods = mtyp_of_il_auxmod_tdefs m filterByAttr aux_mod_loader scoref (manifest_of_mainmod mainmod).manifestExportedTypes in 
  let mainmod = mtyp_of_mainmod_tdefs m filterByAttr scoref mainmod in 
  list_fold_right (combine_msigtyps m) (mainmod :: auxmods) (empty_mtype Namespace)
  

(*-------------------------------------------------------------------------
 * Import an IL type ref as an F# type constructor.
 *------------------------------------------------------------------------- *)

type importMap = { g: tcGlobals; 
                   assemMap: assemMap }

let tycon_of_il_tref env m tref ninst = 
    let tname = tname_of_tref tref in 
    let encl = enclosing_tnames_of_tref tref in 
    let path,nsnb = (match encl with [] -> split_type_name tname | h :: t -> split_namespace h @ t, tname) in 
    let ccu =  
        match scoref_of_tref tref with 
        | ScopeRef_local    -> error(InternalError("tycon_of_il_tref: unexpected local scope",m))
        | ScopeRef_module _ -> error(InternalError("tycon_of_il_tref: reference found to a type in an auxiliary module",m))
        | ScopeRef_assembly assref -> env.assemMap m assref.assemRefName  in
    (* This tcref may contain module names suffixed by "Module". This is not a valid F# tcref, *)
    (* but the resolution logic in deref_path_in_modul accounts for this (we could move that logic here. *)
    let fake_tcref = mk_nonlocal_ref (NLPath(ccu,path)) nsnb in 
    let tycon = 
        try 
            deref_tycon fake_tcref
        with _ ->
            error (Error("A reference to the type '"^nested_tname_of_tref tref^"' supposedly in assembly "^(name_of_ccu ccu)^" was found, but the type could not be found in that assembly",m)) in         
    (* Check the number of type arguments *)
    let ntypars = List.length(typars_of_tycon tycon) in
    if ninst = 0 && ntypars <> ninst then 
        error (Error ("An assembly uses the non-generic type '"^String.concat "." (path@[nsnb])^"' with\ntype parameters.  If this is an F# type then this may be because the type cannot be re-imported into F# as an F# type.\nThis happens when type parameters are lost when referring to F# code from\nC# and other languages (a problem that will go away once the .NET CLI\nsupports generics).  For now ensure that\nassemblies written in .NET languages which reference F# code are either\nnot re-imported into F# or do not use generic F# types in their\npublic interfaces.",m));
    if ntypars <> ninst then 
        error (Error ("An imported assembly uses the type '"^String.concat "." (path@[nsnb])^"' with an incorrect number of type parameters",m));
    match pubpath_of_tycon tycon with 
    | None -> error (Error ("An imported assembly uses the type '"^String.concat "." (path@[nsnb])^"' but that type is not public",m));
    | Some pubpath -> rescope_tycon_pubpath ccu pubpath tycon

(*-------------------------------------------------------------------------
 * Import an IL type as an F# type
 *------------------------------------------------------------------------- *)

let rec typ_of_il_typ env m tinst typ = 
    match typ with
    | Type_void -> env.g.unit_ty (* Typechecking code does the job of making the "void" into a "unit" value, whatever the repr. of "unit" is. *)
    | Type_array(bounds,ty) -> 
        mk_il_arr_ty env.g (Int32.to_int (rank_of_array_shape bounds)) (typ_of_il_typ env m tinst ty)
    | Type_boxed  tspec | Type_value tspec ->
        let ninst = (List.length (inst_of_tspec tspec)) in
        let tycon = tycon_of_il_tref env m (tref_of_tspec tspec) ninst in 
        if List.length(typars_of_tycon (deref_tycon tycon)) <> ninst then 
          error (Error ("The type '"^tname_of_tspec tspec^"' is used with an incorrect number of type parameters in referenced code",m));
        let inst = map (typ_of_il_typ env m tinst) (inst_of_tspec tspec) in
        (* Prefer the F# abbreviation for some built-in types, e.g. *)
        (* 'string' rather than 'System.String', since users don't *)
        (* on the whole realise that these are defined in terms of their .NET equivalents *)
        (* Also on import we decompile uses of FastFunc and Tuple. *)
        begin match env.g.better_tcref_map tycon inst with 
        | Some res -> res
        | None -> TType_app (tycon,inst) 
        end

    | Type_byref ty -> mk_byref_typ env.g (typ_of_il_typ env m tinst ty)
    | Type_ptr ty  -> mk_nativeptr_ty env.g (typ_of_il_typ env m tinst ty)
    | Type_fptr _ -> env.g.nativeint_ty (* failwith "cannot import this kind of type (ptr, fptr)" *)
    | Type_modified(_,_,ty) -> typ_of_il_typ env m tinst ty
    | Type_tyvar u16 -> 
        begin 
          try List.nth tinst (Nums.u16_to_int u16) 
          with _ -> 
	    error(Error("internal error or badly formed meatdata: not enough type parameters were in scope while importing",m))
        end
    | Type_other(e1) -> failwith "typ_of_il_typ: unexpected ILX type"
