// (c) Microsoft Corporation 2005-2007.

#light

/// Immutable sets implemented via binary trees
namespace Microsoft.FSharp.Collections

open System
open Microsoft.FSharp.Core
open Microsoft.FSharp.Collections
#if CLI_AT_MOST_1_1
open Microsoft.FSharp.Compatibility
#else
open System.Collections.Generic
#endif

[<CompilationRepresentation(CompilationRepresentationFlags.ModuleSuffix)>]
module Set = 

    /// For use when not opening the Set module, e.g. Set.t
    type 'a t = Set<'a>  

    ///The empty set.
    [<GeneralizableValueAttribute>]
    val empty<'a> : Set<'a>

    ///The set containing the given one element.
    val singleton: 'a -> Set<'a>

    /// Return a new set with an element added to the set.  No exception is raised if
    ///the set already contains the given element.
    val add: 'a -> Set<'a> -> Set<'a>

    ///Evaluates to "true" if the given element is in the given set
    val mem: 'a -> Set<'a> -> bool

    ///Return the number of elements in the set.  Same as "size"
    val cardinal: Set<'a> -> int

    ///Return the number of elements in the set
    val size: Set<'a> -> int

    ///Choose an arbitrary element from a set
    val choose: Set<'a> -> 'a

    ///Return a new set with the elements of the second set
    ///removed from the first.
    val diff: Set<'a> -> Set<'a> -> Set<'a>

    ///The elements of the set as a list.
    val elements: Set<'a> -> 'a list

    ///Test if two sets are equal
    val equal: Set<'a> -> Set<'a> -> bool

    ///Compare two sets. Places sets into a total order. 
    val compare: Set<'a> -> Set<'a> -> int

    ///Test if any element of the collection satisfies the given predicate.
    ///If the elements are "i0...iN" 
    ///then computes "p i0 or ... or p iN".
    val exists: ('a -> bool) -> Set<'a> -> bool

    ///Return a new collection containing only the elements of the collection
    ///for which the given predicate returns "true"
    val filter: ('a -> bool) -> Set<'a> -> Set<'a>

    ///Return a new collection containing the results of applying the
    ///given function to each element of the input set
    val map: ('a -> 'b) -> Set<'a> -> Set<'b>

    ///Apply the given accumulating function to all the elements of the set
    val fold: ('a -> 'b -> 'b) -> Set<'a> -> 'b -> 'b

    ///Test if all elements of the collection satisfy the given predicate.
    ///If the elements are "i0...iN" and "j0...jN"
    ///then computes "p i0 &amp;&amp; ... &amp;&amp; p iN".
    val for_all: ('a -> bool) -> Set<'a> -> bool

    ///Compute the intersection of the two sets.
    val inter: Set<'a> -> Set<'a> -> Set<'a>

    ///Compute the union of the two sets.
    val union: Set<'a> -> Set<'a> -> Set<'a>

    ///Return "true" if the set is empty
    val is_empty: Set<'a> -> bool

    ///Apply the given function to each element of the set, in order according
    ///to the comparison function
    val iter: ('a -> unit) -> Set<'a> -> unit

    ///Returns the lowest element in the set according to the ordering being used for the set
    val min_elt: Set<'a> -> 'a

    ///Returns the highest element in the set according to the ordering being used for the set
    val max_elt: Set<'a> -> 'a

    ///Split the set into two sets containing the elements for which the given predicate
    ///returns true and false respectively
    val partition: ('a -> bool) -> Set<'a> -> (Set<'a> * Set<'a>)

    ///Return a new set with the given element removed.  No exception is raised in 
    ///the set doesn't contain the given element.
    val remove: 'a -> Set<'a> -> Set<'a>

    ///Evaluates to "true" if all elements of the first set are in the second
    val subset: Set<'a> -> Set<'a> -> bool

    /// Build a set that contains the same elements as the given list
    val of_list: 'a list -> Set<'a>

    /// Build a list that contains the elements of the set in order
    val to_list: Set<'a> -> 'a list

    /// Build a set that contains the same elements as the given array
    val of_array: 'a array -> Set<'a>

    /// Build an array that contains the elements of the set in order
    val to_array: Set<'a> -> 'a array

    /// Return a view of the collection as an enumerable object
    [<System.Obsolete("Consider using to_seq instead")>]
    val to_IEnumerable: Set<'a> -> seq<'a>

    /// Build a new collection from the given enumerable object
    [<System.Obsolete("Consider using of_seq instead")>]
    val of_IEnumerable: #seq<'a> -> Set<'a>

    /// Return a view of the collection as an enumerable object
    val to_seq: Set<'a> -> seq<'a>

    /// Build a new collection from the given enumerable object
    val of_seq: #seq<'a> -> Set<'a>


#if CLI_AT_LEAST_2_0
    /// Return a view of the collection as a .NET collection. Results in a ReadOnly collection
    val to_ICollection: Set<'a> -> ICollection<'a>

    /// Build a new collection from any type that supports the .NET ICollection interface
    /// Unspecified behaviour if the underlying collection is mutated.
    [<Obsolete("Consider using of_IEnumerable instead")>]
    val of_ICollection: #ICollection<'a> -> Set<'a>
#endif

    [<Obsolete("Consider using Collections.Set<_,_> or just Set<_>")>]
    type Set<'a> = Microsoft.FSharp.Collections.Set<'a>

