
// (c) Microsoft Corporation 2005-2007. 

#light

namespace Microsoft.FSharp.Core

    //-------------------------------------------------------------------------
    // Basic type abbreviations

    open System
    type obj = System.Object
    type exn = System.Exception
    type nativeint = System.IntPtr
    type unativeint = System.UIntPtr
    type string = System.String
    type float32 = System.Single
    type float = System.Double
    type single = System.Single
    type double = System.Double
    type sbyte = System.SByte
    type byte = System.Byte
    type int8 = System.SByte
    type uint8 = System.Byte
    type int16 = System.Int16
    type uint16 = System.UInt16
    type int32 = System.Int32
    type uint32 = System.UInt32
    type int64 = System.Int64
    type uint64 = System.UInt64
    type char = System.Char
    type bool = System.Boolean
    type decimal = System.Decimal
    type int = int32

    type 'a ``[]`` = (# "!0[]" #)
    type 'a ``[,]`` = (# "!0[,]" #)
    type 'a ``[,,]`` = (# "!0[,,]" #)
    type 'a ``[,,,]`` = (# "!0[,,,]" #)

    #if CLI_AT_LEAST_2_0
    type 'a array = 'a[]
    #else
    type 'a array = (# "!0[]!" #) // ILX erasable array type
    #endif

    type 'a byref = (# "!0&" #)

    type 'a nativeptr = (# "native int" #)
    type 'a ilsigptr = (# "!0*" #)

    //-------------------------------------------------------------------------
    // IStructuralHash

    type IStructuralHash =
        abstract GetStructuralHashCode : nodesRemaining: int byref -> int

    //-------------------------------------------------------------------------
    // Unit

    and Unit =
        { dummy:int }
        override x.GetHashCode() = 0
        interface System.IComparable with 
            member x.CompareTo(obj:obj) = 0
        interface IStructuralHash with 
            member x.GetStructuralHashCode(nodesRemaining:int byref) = 0
    and unit = Unit

    //-------------------------------------------------------------------------
    // .NET enumerations

    type SourceLevelConstruct = 
       | Unspecified = 0
       | SumType = 1
       | RecordType = 2
       | ObjectType = 3
       | Field = 4
       | Exception = 5
       | Closure = 6
       | Module = 7
       | Alternative = 8
       | Value = 9

    type CompilationRepresentationFlags = 
       | Unspecified = 0
       | Static = 1
       | Instance = 2      
       | ModuleSuffix = 4  // append 'Module' to the end of a non-unique module
       | PermitNull = 8  // Note if you change this then change CompilationRepresentationFlags_PermitNull further below


    [<System.AttributeUsage(System.AttributeTargets.Class,AllowMultiple=false)>]
    type SealedAttribute() =
        inherit System.Attribute()
      
    [<System.AttributeUsage(System.AttributeTargets.Class,AllowMultiple=false)>]
    type AbstractClassAttribute() =
        inherit System.Attribute()
      
    [<System.AttributeUsage (System.AttributeTargets.Class,AllowMultiple=false)>]  
    type DefaultAugmentationAttribute(value:bool) = 
        inherit System.Attribute()
        member x.Value = value

    [<System.AttributeUsage (System.AttributeTargets.Class,AllowMultiple=false)>]  
    type AutoSerializableAttribute(value:bool) = 
        inherit System.Attribute()
        member x.Value = value

    [<System.AttributeUsage (System.AttributeTargets.Field,AllowMultiple=false)>]  
    type DefaultValueAttribute(check:bool) = 
        inherit System.Attribute()
        member x.Check = check
        new() = new DefaultValueAttribute(true)

    [<System.AttributeUsage (System.AttributeTargets.Method,AllowMultiple=false)>]  
    type EntryPointAttribute() = 
        inherit System.Attribute()

    [<System.AttributeUsage (System.AttributeTargets.Class,AllowMultiple=false)>]  
    type ReferenceEqualityAttribute(value:bool) = 
        inherit System.Attribute()
        member x.Value = value
        new() = ReferenceEqualityAttribute(true)

    [<System.AttributeUsage (System.AttributeTargets.Class,AllowMultiple=false)>]  
    type StructuralComparisonAttribute(value:bool) = 
        inherit System.Attribute()
        member x.Value = value
        new() = StructuralComparisonAttribute(true)

    [<System.AttributeUsage (System.AttributeTargets.Class,AllowMultiple=false)>]  
    type StructuralEqualityAttribute(value:bool) = 
        inherit System.Attribute()
        member x.Value = value
        new() = StructuralEqualityAttribute(true)

    [<System.AttributeUsage (System.AttributeTargets.Method,AllowMultiple=false)>]  
    type OverloadIDAttribute(uniqueName:string)  =
        inherit System.Attribute()
        member x.UniqueName = uniqueName

    [<System.AttributeUsage (System.AttributeTargets.All,AllowMultiple=false)>]  
    type ReflectedDefinitionAttribute() =
        inherit System.Attribute()

    [<System.AttributeUsage (System.AttributeTargets.All,AllowMultiple=false)>]  
    type StructAttribute() =
        inherit System.Attribute()

    [<System.AttributeUsage (System.AttributeTargets.All,AllowMultiple=false)>]  
    type InterfaceAttribute() =
        inherit System.Attribute()

    [<System.AttributeUsage (System.AttributeTargets.All,AllowMultiple=false)>]  
    type ClassAttribute() =
        inherit System.Attribute()

    [<System.AttributeUsage(System.AttributeTargets.All,AllowMultiple=false)>]
    type LiteralAttribute() =
        inherit System.Attribute()

    [<System.AttributeUsage(System.AttributeTargets.All,AllowMultiple=false)>]
    type FSharpInterfaceDataVersionAttribute(major:int,minor:int,release:int)  =
        inherit System.Attribute()
        member x.Major = major
        member x.Minor = minor
        member x.Release = release

    [<System.AttributeUsage(System.AttributeTargets.All,AllowMultiple=false)>]
    type CompilationMappingAttribute(sourceConstruct:SourceLevelConstruct,
                                     variantNumber:int,
                                     sequenceNumber:int)  =
        inherit System.Attribute()
        member x.SourceConstruct = sourceConstruct
        member x.SequenceNumber = sequenceNumber
        member x.VariantNumber = variantNumber
        new(sourceConstruct) = CompilationMappingAttribute(sourceConstruct,0,0)
        new(sourceConstruct,sequenceNumber) = CompilationMappingAttribute(sourceConstruct,0,sequenceNumber)

    //-------------------------------------------------------------------------
    [<System.AttributeUsage(System.AttributeTargets.All,AllowMultiple=false)>]
    type CompilationRepresentationAttribute (flags : CompilationRepresentationFlags) =
        inherit System.Attribute()
        member x.Flags = flags

    [<System.AttributeUsage(System.AttributeTargets.All,AllowMultiple=false)>]
    type ExperimentalAttribute(message:string) =
        inherit System.Attribute()
        member x.Message = message

    [<System.AttributeUsage(System.AttributeTargets.All,AllowMultiple=false)>]
    type OCamlCompatibilityAttribute(message:string) =
        inherit System.Attribute()
        member x.Message = message
        new() = OCamlCompatibilityAttribute(null)

    [<System.AttributeUsage(System.AttributeTargets.All,AllowMultiple=false)>]
    type UnverifiableAttribute() =
        inherit System.Attribute()

    [<System.AttributeUsage(System.AttributeTargets.All,AllowMultiple=false)>]
    type OptionalArgumentAttribute() =
        inherit System.Attribute()
      
    [<System.AttributeUsage(System.AttributeTargets.All,AllowMultiple=false)>]
    type GeneralizableValueAttribute() =
        inherit System.Attribute()
      
    [<System.AttributeUsage(System.AttributeTargets.All,AllowMultiple=false)>]
    type RequiresExplicitTypeArgumentsAttribute() =
        inherit System.Attribute()
      
    //-------------------------------------------------------------------------
    // The main aim here is to bootsrap the definition of structural hashing 
    // and comparison.  Calls to these form part of the auto-generated 
    // code for each new datatype.

    module LanguagePrimitives =  
        //-------------------------------------------------------------------------
        // Bootsrap Structural Hashing and Comparison.  Calls to these form part of the
        // auto-generated code for each new datatype.

        // This little prelude is the definitions required for this module.  The definitions
        // should not be used externally.
        //let inline not (b:bool) = (# " ldc.i4 1 xor" b : bool #)
        
        module Basics = 
            let inline unbox_prim<'a>(x:obj) = (# "unbox.any !0" type ('a) x : 'a #)
            let inline box (x:'a) = (# "box !0" type ('a) x : obj #)
            let inline not (b:bool) = (# " ldc.i4 1 xor" b : bool #)
            let inline (=) (x:int) (y:int)  = (# "ceq" x y : bool  #) 
            let inline (<>) (x:int) (y:int)  = not(# "ceq" x y : bool #) 
            let inline (>=) (x:int) (y:int)  = not(# "clt" x y : bool #)
            let inline (>=.) (x:int64) (y:int64)  = not(# "clt" x y : bool #)

            let inline (+) (x:int) (y:int) = (# "add" x y : int #)
            let inline (+.) (x:int64) (y:int64) = (# "add" x y : int64 #)
            let inline (^) (x:string) (y:string) = System.String.Concat(x,y)
            let inline (<<<) (x:int) (y:int) = (# "shl" x y : int #)
            let inline ( * ) (x:int) (y:int) = (# "mul" x y : int #)
            let inline (-) (x:int) (y:int) = (# "sub" x y : int #)
            let inline (>) (x:int) (y:int) = (# "cgt" x y : bool #)
            let inline (<) (x:int) (y:int) = (# "clt" x y : bool #)
            let inline rethrow() = unbox_prim(# "rethrow ldnull" : System.Object #)
            let inline int_of_byte (b:byte) = (# "" b : int #)
            let inline raise (e:> System.Exception) = (# "throw" (e :> System.Exception) : 'b #)

            let inline length (x: 'a[]) = (# "ldlen" x : int #)
            let inline zero_create (n:int) = (# "newarr !0" type ('a) n : 'a[] #)
            let inline get (arr: 'a[]) (n:int) =  (# "ldelem.any !0" type ('a) arr n : 'a #)
            #if CLI_AT_LEAST_2_0
            let set (arr: 'a[]) (n:int) (x:'a) =  (# "stelem.any !0" type ('a) arr n x #)
            #else
            // The above do not work for value types when compiling without generics, since 
            // ILX is not able to erase the use of stelem.any
            let inline set (arr: 'a[]) (n:int) (x:'a) =  (# "stobj !0" type ('a) (# "ldelema !0" type ('a) arr n : System.UIntPtr #) x #)
            #endif

            let inline obj_eq (xobj:obj) (yobj:obj) = (# "ceq" xobj yobj : bool #)
            let inline int64_equals (x:int64) (y:int64) = (# "ceq" x y : bool #) 
            let inline int32_equals (x:int32) (y:int32) = (# "ceq" x y : bool #) 
            let inline int_compare (x:int) (y:int) = if (# "clt" x y : bool #) then (0-1) else (# "cgt" x y : int #)
            let inline int64_compare (x:int64) (y:int64) = if (# "clt" x y : bool #) then (0-1) else (# "cgt" x y : int #)
            let inline byte_compare (x:byte) (y:byte) = if (# "clt" x y : bool #) then (0-1) else (# "cgt" x y : int #)
            let inline byte_equals (x:byte) (y:byte) = (# "ceq" x y : bool #) 
            let inline int64 (x:int) = (# "conv.i8" x  : int64 #)

            #if CLI_AT_MOST_1_1
            let inline typeof< $a > : System.Type =
                let tok = (# "ldtoken !0" type($a) : System.RuntimeTypeHandle #)
                System.Type.GetTypeFromHandle(tok) 
            let inline sizeof< $a > =
                (# "sizeof !0" type($a) : int #) 
            #else
            let inline typeof<'a> =
                let tok = (# "ldtoken !0" type('a) : System.RuntimeTypeHandle #)
                System.Type.GetTypeFromHandle(tok) 
            let inline sizeof<'a>  =
                (# "sizeof !0" type('a) : int #) 
            #endif

            let inline unsafe_default<'a> : 'a = (# "ilzero !0" type ('a) : 'a #)
            let inline isinst_prim<'a>(x:obj) = (# "isinst !0" type ('a) x : obj #)
            let inline castclass_prim<'a>(x:obj) = (# "castclass !0" type ('a) x : 'a #)
            let inline notnull_prim<'a when 'a : not struct>(x:'a) = (# "ldnull cgt.un" x : bool #)
            let inline iscast_prim<'a when 'a : not struct>(x:obj) = (# "isinst !0" type ('a) x : 'a #)


        open Basics

        let inline DefaultValue<'a when 'a : null> = unsafe_default<'a>


        module IntrinsicOperators =        
            //-------------------------------------------------------------------------
            // Lazy and/or.  Laziness added by the F# compiler.
            
            let (&) x y = if x then y else false
            let (&&) x y = if x then y else false
            let (or) x y = if x then true else y
            let (||) x y = if x then true else y
            
            //-------------------------------------------------------------------------
            // Address-of
            // Note, "raise<'a> : exn -> 'a" is manually inlined below.
            // Byref usage checks prohibit type instantiations involving byrefs.

            let inline (~&)  (x : 'a) : 'a byref     = let e = new System.ArgumentException("first class uses of address-of operators are not permitted") in
                                                       (# "throw" (e :> System.Exception) : 'a byref #)
            let inline (~&&) (x : 'a) : 'a nativeptr = let e = new System.ArgumentException("first class uses of address-of operators are not permitted") in
                                                       (# "throw" (e :> System.Exception) : 'a nativeptr #)     
          
        
            //-------------------------------------------------------------------------
            // Unary negation - recognised as a special value by the compiler
            // in order to normalize out unary negation of integer constants.
            
            let inline (~-) (x: ^a) : ^a = 
                (^a : (static member (~-) : ^a -> ^a) (x))
                 when ^a : int32     = (# "neg" x  : int32 #)
                 when ^a : float     = (# "neg" x  : float #)
                 when ^a : float32   = (# "neg" x  : float32 #)
                 when ^a : int64     = (# "neg" x  : int64 #)
                 when ^a : int16     = (# "neg" x  : int16 #)
                 when ^a : nativeint = (# "neg" x  : nativeint #)
                 when ^a : sbyte     = (# "neg" x  : sbyte #)

        open IntrinsicOperators
        module IntrinsicFunctions =        
            
            //-------------------------------------------------------------------------
            // Unboxing, type casts, type tests

            type TypeNullnessSemantics = int
            // .NET reference types
            let TypeNullnessSemantics_NullIsExtraValue = 1
            // F# types with [<PermitNull>]
            let TypeNullnessSemantics_NullTrueValue = 2
            // F# record, union, tuple, function types
            let TypeNullnessSemantics_NullNotLiked = 3
            // structs
            let TypeNullnessSemantics_NullNever = 4
            
            // duplicated from above since we're using integers in this section
            let CompilationRepresentationFlags_PermitNull = 8

#if CLI_AT_MOST_1_1
            let inline UnboxGeneric< 'a >(x:obj) = unbox_prim< 'a >(x)
            let inline UnboxFast< 'a >(x:obj) = unbox_prim< 'a >(x)
            let inline TypeTestGeneric< $a >(x:obj) = notnull_prim(isinst_prim< $a >(x)) 
            let inline TypeTestFast< $a >(x:obj) = notnull_prim(isinst_prim< $a >(x)) 
#else
            type TypeInfo<'a>() = 
               // Compute an on-demand per-instantiation static field
               static let info = 
                         let ty = typeof<'a>
                         if ty.IsValueType 
                         then TypeNullnessSemantics_NullNever else
                         let mappingAttrs = ty.GetCustomAttributes(typeof<CompilationMappingAttribute>,false)
                         if mappingAttrs.Length = 0 
                         then TypeNullnessSemantics_NullIsExtraValue
                         elif ty.Equals(typeof<unit>) then 
                             TypeNullnessSemantics_NullTrueValue
                         else
                             let reprAttrs = ty.GetCustomAttributes(typeof<CompilationRepresentationAttribute>,false)
                             if reprAttrs.Length = 0 
                             then TypeNullnessSemantics_NullNotLiked 
                             else
                                 let reprAttr = get reprAttrs 0
                                 let reprAttr = (# "unbox.any !0" type (CompilationRepresentationAttribute) reprAttr : CompilationRepresentationAttribute #)
                                 if (# "and" reprAttr.Flags CompilationRepresentationFlags_PermitNull : int #) = 0
                                 then TypeNullnessSemantics_NullNotLiked
                                 else TypeNullnessSemantics_NullTrueValue

               // Publish the results of that compuation
               static member TypeInfo = info
                         

            // Note: cheap nullness test for generic value:
            //  IL_0000:  ldarg.1
            //  IL_0001:  box        !TKey
            //  IL_0006:  brtrue.s   IL_000e

            // worst case: nothing known about source or destination
            let UnboxGeneric<'a>(x:obj) = 
                if notnull_prim(x) or TypeInfo<'a>.TypeInfo <> TypeNullnessSemantics_NullNotLiked then 
                    unbox_prim<'a>(x)
                else
                    //System.Console.WriteLine("UnboxGeneric, x = {0}, 'a = {1}", x, typeof<'a>)
                    raise (System.NullReferenceException()) 

            // better: source is NOT TypeNullnessSemantics_NullNotLiked 
            let inline UnboxFast<'a>(x:obj) = 
                // assert not(TypeInfo<'a>.TypeInfo = TypeNullnessSemantics_NullNotLiked)
                unbox_prim<'a>(x)

            // worst case: nothing known about source or destination
            let TypeTestGeneric<'a>(x:obj) = 
                if notnull_prim(isinst_prim<'a>(x)) then true
                elif notnull_prim(x) then false
                else (TypeInfo<'a>.TypeInfo = TypeNullnessSemantics_NullTrueValue)

            // quick entry: source is NOT TypeNullnessSemantics_NullTrueValue 
            let inline TypeTestFast<'a>(x:obj) = 
                //assert not(TypeInfo<'a>.TypeInfo = TypeNullnessSemantics_NullTrueValue)
                notnull_prim(isinst_prim<'a>(x)) 

#endif
    (* 
            let IsTypeAndUnboxGeneric<'a> (x:obj) (res: 'a byref) : bool = 
                if notnull_prim(isinst_prim<'a>(x)) then (res <- unbox_prim<'a>(x); true)
                elif notnull_prim(x) or not (TypeInfo<'a>.TypeInfo = TypeNullnessSemantics_NullTrueValue) then 
                    false
                else 
                    (res <- unsafe_default<'a>; true)
    *)

            let inline ParseDecimal(s:string) =  System.Decimal.Parse(s,System.Globalization.CultureInfo.InvariantCulture)
            let inline ArrayGet(arr: 'a array) (n:int) =  (# "ldelem.erasable !0" type ('a) arr n : 'a #)  
#if CLI_AT_LEAST_2_0
            let inline CreateInstance<'a>() = 
                 (System.Activator.CreateInstance() : 'a)
#else
            let inline CreateInstance< $a >() = 
                 (unbox_prim (System.Activator.CreateInstance(typeof< $a >)) : $a)
#endif    

            let inline compatarray_get (arr: 'a[]) (n:int) =  (# "ldelem.any !0" type ('a) arr n : 'a #)
#if CLI_AT_LEAST_2_0
            let inline compatarray_set (arr: 'a[]) (n:int) (x:'a) =  (# "stelem.any !0" type ('a) arr n x #)
#else
            // The above do not work for value types when compiling without generics, since 
            // ILX is not able to erase the use of stelem.any
            let inline compatarray_set (arr: 'a[]) (n:int) (x:'a) =  (# "stobj !0" type ('a) (# "ldelema !0" type ('a) arr n : System.UIntPtr #) x #)
#endif

            let inline array_get (arr: 'a array) (n:int) =  (# "ldelem.erasable !0" type ('a) arr n : 'a #)  
            let inline array_set (arr: 'a array) (n:int) (x:'a) =  (# "stelem.erasable !0" type ('a) arr n x #)  
            let inline array2_get (arr: 'a[,])  (n1:int) (n2:int)                 = (# "ldelem.multi 2 !0" type ('a) arr n1 n2 : 'a #)  
            let inline array3_get (arr: 'a[,,]) (n1:int) (n2:int) (n3:int)        = (# "ldelem.multi 3 !0" type ('a) arr n1 n2 n3 : 'a #)  
            let inline array2_set (arr: 'a[,])  (n1:int) (n2:int) (x:'a)          = (# "stelem.multi 2 !0" type ('a) arr n1 n2 x #)  
            let inline array3_set (arr: 'a[,,]) (n1:int) (n2:int) (n3:int) (x:'a) = (# "stelem.multi 3 !0" type ('a) arr n1 n2 n3 x #)  

            let inline retype (x:'a) : 'b = (# "" x : 'b #)


#if CLI_AT_LEAST_2_0
            let inline compatarray_sub arr (start:int) (len:int) =
                let res = zero_create len   
                for i = 0 to len - 1 do 
                    compatarray_set res i (compatarray_get arr (start + i))
                res
#else
            let inline array_zero_create (n:int) = (# "newarr.erasable !0" type ('a) n : 'a array #)
            let inline array_sub (arr:'a array) (start:int) (len:int) =
                let res = array_zero_create len   
                for i = 0 to len - 1 do 
                    (array_set res i (array_get arr (start + i) : 'a))
                res
#endif

            let inline (.[]) (s: ^src) (x: 'a) : 'b =    
               (^src : (member get_Item : 'a -> 'b)(s,x))
               when ^src : string  = (retype ((retype s : string).Chars((retype x : int))) : 'b)
#if CLI_AT_LEAST_2_0
               when ^src : _ []    = (compatarray_get (retype s : 'b[])     (retype x : int) : 'b)
        #else
               when ^src : _ array = (array_get       (retype s : 'b array) (retype x : int) : 'b)
        #endif

            let inline (.[,]) (s: ^src) ((x1: 'a1),(x2:'a2)) : 'b =    
               (^src : (member get_Item : 'a1 * 'a2 -> 'b)(s,x1,x2))
        #if CLI_AT_LEAST_2_0
               when ^src : _ [,]    = (array2_get (retype s : 'b[,]) (retype x1 : int) (retype x2 : int) : 'b)
        #endif

            let inline (.[,,]) (s: ^src) ((x1: 'a1),(x2:'a2),(x3:'a3)) : 'b =    
               (^src : (member get_Item : 'a1 * 'a2 * 'a3 -> 'b)(s,x1,x2,x3))
        #if CLI_AT_LEAST_2_0
               when ^src : _ [,,]    = (array3_get (retype s : 'b[,,]) (retype x1 : int) (retype x2 : int) (retype x3 : int) : 'b)
        #endif

            let inline (.[]<-) (s: ^src) (x: 'a) (y: 'b) : unit =    
               (^src : (member set_Item : 'a * 'b -> unit)(s,x,y))
        #if CLI_AT_LEAST_2_0
               when ^src : _ []    = (compatarray_set (retype s : 'b[])     (retype x : int) y)  
        #else
               when ^src : _ array = (array_set   (retype s : 'b array) (retype x : int) y)  
        #endif
          
            let inline (.[,]<-) (s: ^src)  ((x1: 'a1),(x2:'a2)) (y: 'b) : unit =    
               (^src : (member set_Item : 'a1 * 'a2 * 'b -> unit)(s,x1,x2,y))
        #if CLI_AT_LEAST_2_0
               when ^src : _ [,]    = (array2_set (retype s : 'b[,])     (retype x1 : int) (retype x2 : int) y)  
        #endif
          

            let inline (.[,,]<-) (s: ^src)  ((x1: 'a1),(x2:'a2),(x3:'a3)) (y: 'b) : unit =    
               (^src : (member set_Item : 'a1 * 'a2 * 'a3 * 'b -> unit)(s,x1,x2,x3,y))
        #if CLI_AT_LEAST_2_0
               when ^src : _ [,,]    = (array3_set (retype s : 'b[,,])     (retype x1 : int) (retype x2 : int) (retype x3 : int) y)  
        #endif
          

        module HashCompare = 
        
            let PhysicalEqualityObj (xobj:obj) (yobj:obj) = 
                  obj_eq xobj yobj 
                   || (match xobj with 
                        | null -> false
                        | :? System.ValueType -> xobj.Equals(yobj)
                        | _ -> false)

            // NOTE: compiler/optimizer is aware of this function and optimizes calls to it in many situations
            // where it is known that PhysicalEqualityObj is identical to reference comparison
            let PhysicalEqualityIntrinsic (x:'a) (y:'a) = 
                  PhysicalEqualityObj (box x) (box y)

            let rec inline PhysicalEqualityFast (x:'a) (y:'a) = 
                  PhysicalEqualityIntrinsic x y
                  when 'a : bool   = (# "ceq" x y : bool #)
                  when 'a : int    = (# "ceq" x y : bool #)
                  when 'a : sbyte  = (# "ceq" x y : bool #)
                  when 'a : int16  = (# "ceq" x y : bool #)
                  when 'a : int32  = (# "ceq" x y : bool #)
                  when 'a : int64  = (# "ceq" x y : bool #)
                  when 'a : byte   = (# "ceq" x y : bool #)
                  when 'a : uint16 = (# "ceq" x y : bool #)
                  when 'a : uint32 = (# "ceq" x y : bool #)
                  when 'a : uint64 = (# "ceq" x y : bool #)
                  when 'a : float  = (# "ceq" x y : bool #)
                  when 'a : char   = (# "ceq" x y : bool #)

            let PhysicalHashObj (xobj: obj) = 
                  match xobj with
                  | :? System.ValueType as v -> v.GetHashCode()
                  | _ -> 
#if CLI_AT_MOST_1_1
            // NOTE: a direct call to GetHashCode, which in V1.x. is permitted and 
            // returns the object hash
                   (# "call instance int32 [mscorlib]System.Object::GetHashCode()" xobj : int32 #)
#else   
            // System.Runtime.CompilerServices.RuntimeHelpers.GetHashCode is only available post- v1.0.  
                   System.Runtime.CompilerServices.RuntimeHelpers.GetHashCode(xobj)
#endif
          
            let PhysicalHashIntrinsic (x: 'a) = PhysicalHashObj (box x)
            
            let inline PhysicalHashFast (x: 'a) = 
                  PhysicalHashIntrinsic x
                  // NOTE: only specialziations on value types are allowed, and 
                  // any type specializations on value types must return identical values to 
                  // the mscorlib implementations of the ::GetHashCode methods on the given types. 
                  when 'a : bool   = (# "" x : int #)
                  when 'a : int    = (# "" x : int #)
                  when 'a : int32  = (# "" x : int #)
                  when 'a : byte   = (# "" x : int #)
                  when 'a : uint32 = (# "" x : int #)



            let inline HashCombine nr x y = (x <<< 1) + y + 631 * nr


            // The depth-limited structural hash.  Note we can hash _any_ terms apart 
            // from those containing closures.  Non-F# types are hashed using GetHashCode.
            //
            // Implementation: Note that some representations chosen by F# are legitimately allowed to be null, e.g. the empty list. 
            let rec inline GenericHashParamFast<'a> (x:'a) (nr: int byref) = 
                   GenericHashParamIntrinsic x (&nr)
                  // NOTE: any type specializations must return identical values to 
                  // the mscorlib implementations of the ::GetHashCode methods on the given types. 
                   when 'a : bool =   nr <- nr - 1; (# "" x : int #)
                   when 'a : int =    nr <- nr - 1; (# "" x : int #)
                   when 'a : int32 =  nr <- nr - 1; (# "" x : int #)
                   when 'a : byte =   nr <- nr - 1; (# "" x : int #)
                   when 'a : uint32 = nr <- nr - 1; (# "" x : int #)
                   when 'a : string = 
                    match (# "" x : obj #) with 
                    | null -> 0 
                    | _ -> nr <- nr - 1; (# "call instance int32 [mscorlib]System.String:: GetHashCode()" x : int #)

            // Core implementation of structural hashing.  
            // NOTE: compiler/optimizer is aware of this function and optimizes calls to it in many situations
            // where it is known that GenericHashParamObj dispatches via IStructuralHash
            and GenericHashParamIntrinsic<'a> (x: 'a) (nr: int byref) = GenericHashParamObj (box x) (&nr)

            // Core implementation of structural hashing.  Search for the IStructuralHash interface.
            // Arrays and strings are structurally hashed through a separate technique.
            // Objects which do not support this interface get hashed using a virtual
            // call to the standard Object::GetStructuralHashCode().
            and GenericHashParamObj (x: obj) (nr: int byref) =
                  match x with 
                  | null -> 0 
                  | :? IStructuralHash as a -> a.GetStructuralHashCode(&nr)
                  | :? System.Array as a -> 
                      match a with 
                      | :? (obj[]) as oa -> GenericHashObjArray oa (&nr)
                      | :? (byte[]) as ba -> GenericHashByteArray ba (&nr) 
                      | _ -> GenericHashArbArray a (&nr) 
                  | _ -> 
                      nr <- nr - 1;
                      x.GetHashCode()

            // optimized case - byte arrays are often used for strings 
            and GenericHashByteArray (x: byte[]) (nr: int byref) =
                  let len = length x 
                  let mutable i = len - 1 
                  let mutable acc = 0   
                  while (i >= 0) do 
                    if nr > 0 then 
                        nr <- nr - 1;
                        acc <- HashCombine nr acc (int_of_byte (get x i));
                        i <- i - 1
                    else 
                        i <- -1                      
                  done;
                  acc

            // special case - arrays do not by default have a decent structural hashing function
            and GenericHashArbArray (x: System.Array) (nr: int byref) =
                  match x.Rank  with 
                  | 1 -> 
                    let b = x.GetLowerBound(0) 
                    let len = x.Length 
                    let mutable i = b + len - 1 
                    let mutable acc = 0                  
                    while (i >= b) do 
                        if nr > 0 then 
                          // NOTE: GenericHash* call decreases nr 
                          acc <- HashCombine nr acc (GenericHashParamObj (x.GetValue(i)) (&nr));
                          i <- i - 1
                        else 
                          i <- b-1
                    done;
                    acc
                  | n -> 
                     nr <- nr - 1; 
                     HashCombine nr (x.GetLength(0)) (x.GetLength(1)) 

                

            // optimized case - hash on things compatible with obj[]
            and GenericHashObjArray (x: obj[]) (nr: int byref) =
                  let len = x.Length 
                  let mutable i = len - 1 
                  let mutable acc = 0   
                  while (i >= 0) do 
                    if nr > 0 then 
                        // NOTE: GenericHash* call decreases nr 
                        acc <- HashCombine nr acc (GenericHashParamObj (x.GetValue(i)) (&nr));
                        i <- i - 1
                    else 
                        i <- -1
                  done;
                  acc


            let defaultHashNodes = 18 // WARNING: CHANGE THIS IN ilxgen.ml  TOO!! 

            /// Core entry into structural hashing.  Hash to a given depth limit.
            let inline GenericHashFast (x:'a) = 
                let mutable nNodesRemaining = defaultHashNodes 
                GenericHashParamFast x (&nNodesRemaining)
                when 'a : bool   = (# "" x : int #)
                when 'a : int    = (# "" x : int #)
                when 'a : int32  = (# "" x : int #)
                when 'a : byte   = (# "" x : int #)
                when 'a : uint32 = (# "" x : int #)

            //-------------------------------------------------------------------------
            // The structural term ordering.  Note we can compare _any_ terms apart 
            // from those containing closures and those containing non-F# types which 
            // do not support IComparable. 
            //
            // Note that some representations chosen by F# are legitimately allowed to be null, e.g. the empty list. 
            // However, because null values don't support the polymorphic virtual comparison operation CompareTo 
            // the test for nullness must be made on the caller side (see poly_compare above) 
            // This in turn means that F# itself must commit to placing any values represented by "null" 
            // in the ordering of values, i.e. anything represented by "null" becomes smaller than any 
            // other F# value.  This in turn means that for consistency the check in the generated 
            // comparison functions must make sure that any value is _bigger_ than null as well. 
            //
            // - Terms-with-F#-static-types are those whose type is an F# record, disc. union or tuple, or
            //   any of the env types (int, char etc.).  Two terms with the same F#-static type 
            //   are guaranteed to be comparable.
            //
            // - Certain non-F# CLS types can count as F#-static types if they have certain properties, 
            //   e.g. support a well-behaved System.IComparable and T::GetHashCode.
            //
            // - An array type T[] is an F#-static type if T is an F#-static-type.  Unlike .NET arrays, 
            //   F# array types do support comparison between similarly typed terms.
            //
            // - Terms-with-non-F#-static-types are not guaranteed to be comparable.  For example, two
            //   terms of type "obj" are not guaranteed to be comparable.
            //
            // - "null" is always comparable to every value of a boxed-F#-static-type, and is always less than 
            //   all such values.
            //
            //------------------------------------------------------------------------- 

            /// Core implementation of structural comparison on arbitrary values.
            let rec inline GenericComparisonFast<'a> (x:'a) (y:'a) = 
                 GenericComparisonIntrinsic x y
                 when 'a : bool   = if (# "clt" x y : bool #) then (-1) else (# "cgt" x y : int #)
                 when 'a : sbyte  = if (# "clt" x y : bool #) then (-1) else (# "cgt" x y : int #)
                 when 'a : int16  = if (# "clt" x y : bool #) then (-1) else (# "cgt" x y : int #)
                 when 'a : int32  = if (# "clt" x y : bool #) then (-1) else (# "cgt" x y : int #)
                 when 'a : int    = if (# "clt" x y : bool #) then (-1) else (# "cgt" x y : int #)
                 when 'a : int64  = if (# "clt" x y : bool #) then (-1) else (# "cgt" x y : int #)
                 when 'a : nativeint  = if (# "clt" x y : bool #) then (-1) else (# "cgt" x y : int #)
                 when 'a : byte   = if (# "clt.un" x y : bool #) then (-1) else (# "cgt.un" x y : int #)
                 when 'a : uint16 = if (# "clt.un" x y : bool #) then (-1) else (# "cgt.un" x y : int #)
                 when 'a : uint32 = if (# "clt.un" x y : bool #) then (-1) else (# "cgt.un" x y : int #)
                 when 'a : uint64 = if (# "clt.un" x y : bool #) then (-1) else (# "cgt.un" x y : int #)
                 when 'a : unativeint = if (# "clt.un" x y : bool #) then (-1) else (# "cgt.un" x y : int #)
                 // We have to be careful to give consistent results for NaN.  Note that we can give correct NaN treatment for
                 // each particular relational operator "<", ">=" etc, where "NaN RELOP NaN" is always false.
                 // But for arbitrary comparison we
                 // need to the Double.CompareTo semantics, which differs from relational operator semantics,
                 // in particular because "compare NaN NaN == 0" (there's no other choice as to 
                 // what it could be!).  
                 // Note: dummy "" instruction ensures that the comparison instruction doesn't get turned into the corresponding branch instruction, particularly when negated
                 when 'a : float  = (# "" x : float #).CompareTo((# "" y : float #)) 
                 when 'a : float32 = (# "" x : float32 #).CompareTo((# "" y : float #)) 
                 when 'a : char   = if (# "clt.un" x y : bool #) then (-1) else (# "cgt.un" x y : int #)
                 when 'a : string = 
                     // NOTE: we don't have to null check here because System.String.CompareOrdinal
                     // gives reliable results on null values.
                     System.String.CompareOrdinal((# "" x : string #) ,(# "" y : string #))

            // NOTE: compiler/optimizer is aware of this function and optimizes calls to it in many situations
            // where it is known how GenericComparisonObj will dispatch via IComparable.
            and GenericComparisonIntrinsic<'a> (x:'a) (y:'a) = 
                GenericComparisonObj (box x) (box y)

            and FailGenericComparison (obj: obj)  = 
                raise (new System.ArgumentException("Failure during generic comparison: the type "^obj.GetType().ToString()^" does not implement the System.IComparable interface. This error may be arise from the use of a function such as 'compare', 'max' or 'min' or a data structure such as 'Set' or 'Map' whose keys contain instances of this type"))
            
            /// Core implementation of structural comparison on boxed values.
            and GenericComparisonObj (xobj:obj) (yobj:obj) = 
                (*if obj_eq xobj yobj then 0 else *)
                  match xobj,yobj with 
                   | null,null -> 0
                   | null,_ -> -1
                   | _,null -> 1
                   // Use Ordinal comparison for strings
                   | (:? string as x),(:? string as y) -> System.String.CompareOrdinal(x, y)
                   // Check for IComparable
                   | (:? System.IComparable as x),_ -> x.CompareTo(yobj)
                   // Permit structural comparison on arrays
                   | (:? System.Array as arr1),_ -> 
                     begin match arr1,yobj with 
                     // Fast path
                     | (:? (obj[]) as arr1) , (:? (obj[]) as arr2)      -> GenericComparisonObjArray arr1 arr2
                     // Fast path
                     | (:? (byte[]) as arr1), (:? (byte[]) as arr2)     -> GenericComparisonByteArray arr1 arr2
                     | _                    , (:? System.Array as arr2) -> GenericComparisonArbArray arr1 arr2
                     | _ -> FailGenericComparison xobj
                     end
                   // These don't implement IComparable, but we need coherence, e.g. compare [0n] [1n] = -(compare [1n] [0n])
                   | (:? nativeint as x),(:? nativeint as y) -> if (# "clt" x y : bool #) then (-1) else (# "cgt" x y : int #)
                   | (:? unativeint as x),(:? unativeint as y) -> if (# "clt.un" x y : bool #) then (-1) else (# "cgt.un" x y : int #)
                   | _,(:? System.IComparable as yc) -> -(yc.CompareTo(xobj))
                   | _ -> FailGenericComparison xobj

            /// specialcase: Core implementation of structural comparison on arbitrary arrays.
            and GenericComparisonArbArray (x:System.Array) (y:System.Array) =
#if CLI_AT_MOST_1_1
                if x.Rank = 1 && y.Rank = 1 then 
                    let lenx = x.Length
                    let leny = y.Length 
                    let c = int_compare lenx leny 
                    if c <> 0 then c else
                    let rec check i =
                       if i >= lenx then 0 else 
                       let c = GenericComparisonObj (x.GetValue(i)) (y.GetValue(i)) 
                       if c <> 0 then c else check (i + 1)
                    check 0
                elif x.Rank = 2 && y.Rank = 2 then 
                    let lenx0 = x.GetLength(0)
                    let leny0 = y.GetLength(0)
                    let c = int_compare lenx0 leny0 
                    if c <> 0 then c else
                    let lenx1 = x.GetLength(1)
                    let leny1 = y.GetLength(1)
                    let c = int_compare lenx1 leny1 
                    if c <> 0 then c else
                    let basex0 = (x.GetLowerBound(0))
                    let basex1 = (x.GetLowerBound(1))
                    let basey0 = (y.GetLowerBound(0))
                    let basey1 = (y.GetLowerBound(1))
                    let c = int_compare basex0 basey0
                    if c <> 0 then c else
                    let c = int_compare basex1 basey1
                    if c <> 0 then c else
                    let rec check0 i =
                       let rec check1 j = 
                           if j >= lenx1 then 0 else
                           let c = GenericComparisonObj (x.GetValue(i + basex0,j + basex1)) (y.GetValue(i + basey0,j + basey1)) 
                           if c <> 0 then c else check1 (j + 1)
                       if i >= lenx0 then 0 else 
                       let c = check1 0
                       if c <> 0 then c else
                       check0 (i + 1)
                    check0 0
                else
                    let c = int_compare x.Rank y.Rank
                    if c <> 0 then c else
                    let ndims = x.Rank
                    // check lengths 
                    let rec precheck k = 
                        if k >= ndims then 0 else
                        let c = int_compare (x.GetLength(k)) (y.GetLength(k))
                        if c <> 0 then c else
                        let c = int_compare (x.GetLowerBound(k)) (y.GetLowerBound(k))
                        if c <> 0 then c else
                        precheck (k+1)
                    let c = precheck 0 
                    if c <> 0 then c else
                    let idxs : int[] = zero_create ndims 
                    let rec checkN k base i lim =
                       if i >= lim then 0 else
                       set idxs k (base + i)
                       let c = 
                           if k = ndims - 1
                           then GenericComparisonObj (x.GetValue(idxs)) (y.GetValue(idxs)) 
                           else check (k+1) 
                       if c <> 0 then c else 
                       checkN k base (i + 1) lim
                    and check k =
                       if k >= ndims then 0 else
                       let base = x.GetLowerBound(k)
                       checkN k base 0 (x.GetLength(k))
                    check 0
#else
                if x.Rank = 1 && y.Rank = 1 then 
                    let lenx = x.LongLength
                    let leny = y.LongLength 
                    let c = int64_compare lenx leny 
                    if c <> 0 then c else
                    let rec check i =
                       if i >=. lenx then 0 else 
                       let c = GenericComparisonObj (x.GetValue(i)) (y.GetValue(i)) 
                       if c <> 0 then c else check (i +. 1L)
                    check 0L
                elif x.Rank = 2 && y.Rank = 2 then 
                    let lenx0 = x.GetLongLength(0)
                    let leny0 = y.GetLongLength(0)
                    let c = int64_compare lenx0 leny0 
                    if c <> 0 then c else
                    let lenx1 = x.GetLongLength(1)
                    let leny1 = y.GetLongLength(1)
                    let c = int64_compare lenx1 leny1 
                    if c <> 0 then c else
                    let basex0 = int64 (x.GetLowerBound(0))
                    let basex1 = int64 (x.GetLowerBound(1))
                    let basey0 = int64 (y.GetLowerBound(0))
                    let basey1 = int64 (y.GetLowerBound(1))
                    let c = int64_compare basex0 basey0
                    if c <> 0 then c else
                    let c = int64_compare basex1 basey1
                    if c <> 0 then c else
                    let rec check0 i =
                       let rec check1 j = 
                           if j >=. lenx1 then 0 else
                           let c = GenericComparisonObj (x.GetValue(i +. basex0,j +. basex1)) (y.GetValue(i +. basey0,j +. basey1)) 
                           if c <> 0 then c else check1 (j +. 1L)
                       if i >=. lenx0 then 0 else 
                       let c = check1 0L
                       if c <> 0 then c else
                       check0 (i +. 1L)
                    check0 0L
                else
                    let c = int_compare x.Rank y.Rank
                    if c <> 0 then c else
                    let ndims = x.Rank
                    // check lengths 
                    let rec precheck k = 
                        if k >= ndims then 0 else
                        let c = int64_compare (x.GetLongLength(k)) (y.GetLongLength(k))
                        if c <> 0 then c else
                        let c = int_compare (x.GetLowerBound(k)) (y.GetLowerBound(k))
                        if c <> 0 then c else
                        precheck (k+1)
                    let c = precheck 0 
                    if c <> 0 then c else
                    let idxs : int64[] = zero_create ndims 
                    let rec checkN k base i lim =
                       if i >=. lim then 0 else
                       set idxs k (base +. i)
                       let c = 
                           if k = ndims - 1
                           then GenericComparisonObj (x.GetValue(idxs)) (y.GetValue(idxs)) 
                           else check (k+1) 
                       if c <> 0 then c else 
                       checkN k base (i +. 1L) lim
                    and check k =
                       if k >= ndims then 0 else
                       let base = x.GetLowerBound(k)
                       checkN k (int64 base) 0L (x.GetLongLength(k))
                    check 0
#endif
                
              
            /// optimized case: Core implementation of structural comparison on object arrays.
            and GenericComparisonObjArray (x:obj[]) (y:obj[]) =
                let lenx = x.Length 
                let leny = y.Length 
                let c = int_compare lenx leny 
                if c <> 0 then c 
                else
                    let mutable i = 0
                    let mutable res = 0  
                    while i < lenx do 
                      let c = GenericComparisonObj (get x i) (get y i) 
                      if c <> 0 then (res <- c; i <- lenx) 
                      else i <- i + 1
                    done;
                    res

            /// optimized case: Core implementation of structural comparison on byte arrays.
            and GenericComparisonByteArray (x:byte[]) (y:byte[]) =
                let lenx = x.Length 
                let leny = y.Length 
                let c = int_compare lenx leny 
                if c <> 0 then c 
                else
                    let mutable i = 0 
                    let mutable res = 0 
                    while i < lenx do 
                      let c = byte_compare (get x i) (get y i) 
                      if c <> 0 then (res <- c; i <- lenx) 
                      else i <- i + 1
                    done;
                    res


            let rec GenericEqualityIntrinsic (x : 'a) (y : 'a) = 
                GenericEqualityObj (box x) (box y)
                
            and GenericEqualityObj xobj yobj = 
                (*if obj_eq xobj yobj then true else  *)
                  match xobj,yobj with 
                   | null,null -> true
                   | null,_ -> false
                   | _,null -> false
                   | (:? string as xs),(:? string as ys) -> System.String.Equals(xs,ys)
                   // Permit structural equality on arrays
                   | (:? System.Array as arr1),_ -> 
                     begin match arr1,yobj with 
                     // Fast path
                     | (:? (obj[]) as arr1) , (:? (obj[]) as arr2)      -> GenericEqualityObjArray arr1 arr2
                     // Fast path
                     | (:? (byte[]) as arr1), (:? (byte[]) as arr2)     -> GenericEqualityByteArray arr1 arr2
                     | _                    , (:? System.Array as arr2) -> GenericEqualityArbArray arr1 arr2
                     | _ -> xobj.Equals(yobj)
                     end
                   // Ensure NaN semantics on recursive calls
                   | (:? float as f1), (:? float as f2) -> (# "ceq" f1 f2 : bool #)
                   | (:? float32 as f1), (:? float32 as f2) -> (# "ceq" f1 f2 : bool #)
                   | _ -> xobj.Equals(yobj)

            /// specialcase: Core implementation of structural equality on arbitrary arrays.
            and GenericEqualityArbArray (x:System.Array) (y:System.Array) =
#if CLI_AT_MOST_1_1
                if x.Rank = 1 && y.Rank = 1 then 
                    // check lengths 
                    let lenx = x.Length
                    let leny = y.Length 
                    (int32_equals lenx leny) &&
                    // check contents
                    let rec check i = (i >= lenx) || (GenericEqualityObj (x.GetValue(i)) (y.GetValue(i)) && check (i + 1))
                    check 0          
                elif x.Rank = 2 && y.Rank = 2 then 
                    // check lengths 
                    let lenx0 = x.GetLength(0)
                    let leny0 = y.GetLength(0)
                    (int32_equals lenx0 leny0) && 
                    let lenx1 = x.GetLength(1)
                    let leny1 = y.GetLength(1)
                    (int32_equals lenx1 leny1) && 
                    let basex0 = (x.GetLowerBound(0))
                    let basex1 = (x.GetLowerBound(1))
                    let basey0 = (y.GetLowerBound(0))
                    let basey1 = (y.GetLowerBound(1))
                    (int32_equals basex0 basey0) && 
                    (int32_equals basex1 basey1) && 
                    // check contents
                    let rec check0 i =
                       let rec check1 j = (j >= lenx1) || (GenericEqualityObj (x.GetValue(basex0 + i,basex1 + j)) (y.GetValue(basey0 + i,basey1 + j)) && check1 (j + 1))
                       (i >= lenx0) || (check1 0 && check0 (i + 1))
                    check0 0
                else 
                    (x.Rank = y.Rank) && 
                    let ndims = x.Rank
                    // check lengths 
                    let rec precheck k = 
                        (k >= ndims) || 
                        (int32_equals (x.GetLength(k)) (y.GetLength(k)) && 
                         int32_equals (x.GetLowerBound(k)) (y.GetLowerBound(k)) && 
                         precheck (k+1))
                    precheck 0 &&
                    let idxs : int[] = zero_create ndims 
                    // check contents
                    let rec checkN k base i lim =
                       (i >= lim) ||
                       (set idxs k (base + i);
                        (if k = ndims - 1
                         then GenericEqualityObj (x.GetValue(idxs)) (y.GetValue(idxs)) 
                         else check (k+1)) && 
                        checkN k base (i + 1) lim)
                    and check k = 
                       (k >= ndims) || 
                       (let base = x.GetLowerBound(k)
                        checkN k base 0 (x.GetLength(k)))
                           
                    check 0
#else
                if x.Rank = 1 && y.Rank = 1 then 
                    // check lengths 
                    let lenx = x.LongLength
                    let leny = y.LongLength 
                    (int64_equals lenx leny) &&
                    // check contents
                    let rec check i = (i >=. lenx) || (GenericEqualityObj (x.GetValue(i)) (y.GetValue(i)) && check (i +. 1L))
                    check 0L                    
                elif x.Rank = 2 && y.Rank = 2 then 
                    // check lengths 
                    let lenx0 = x.GetLongLength(0)
                    let leny0 = y.GetLongLength(0)
                    (int64_equals lenx0 leny0) && 
                    let lenx1 = x.GetLongLength(1)
                    let leny1 = y.GetLongLength(1)
                    (int64_equals lenx1 leny1) && 
                    let basex0 = int64 (x.GetLowerBound(0))
                    let basex1 = int64 (x.GetLowerBound(1))
                    let basey0 = int64 (y.GetLowerBound(0))
                    let basey1 = int64 (y.GetLowerBound(1))
                    (int64_equals basex0 basey0) && 
                    (int64_equals basex1 basey1) && 
                    // check contents
                    let rec check0 i =
                       let rec check1 j = (j >=. lenx1) || (GenericEqualityObj (x.GetValue(basex0 +. i,basex1 +. j)) (y.GetValue(basey0 +. i,basey1 +. j)) && check1 (j +. 1L))
                       (i >=. lenx0) || (check1 0L && check0 (i +. 1L))
                    check0 0L
                else 
                    (x.Rank = y.Rank) && 
                    let ndims = x.Rank
                    // check lengths 
                    let rec precheck k = 
                        (k >= ndims) || 
                        (int64_equals (x.GetLongLength(k)) (y.GetLongLength(k)) && 
                         int32_equals (x.GetLowerBound(k)) (y.GetLowerBound(k)) && 
                         precheck (k+1))
                    precheck 0 &&
                    let idxs : int64[] = zero_create ndims 
                    // check contents
                    let rec checkN k base i lim =
                       (i >=. lim) ||
                       (set idxs k (base +. i);
                        (if k = ndims - 1
                         then GenericEqualityObj (x.GetValue(idxs)) (y.GetValue(idxs)) 
                         else check (k+1)) && 
                        checkN k base (i +. 1L) lim)
                    and check k = 
                       (k >= ndims) || 
                       (let base = x.GetLowerBound(k)
                        checkN k (int64 base) 0L (x.GetLongLength(k)))
                           
                    check 0
#endif

              
            /// optimized case: Core implementation of structural equality on object arrays.
            and GenericEqualityObjArray (x:obj[]) (y:obj[]) =
                let lenx = x.Length 
                let leny = y.Length 
                let c = (lenx = leny )
                if not c then c 
                else
                    let mutable i = 0
                    let mutable res = true
                    while i < lenx do 
                      let c = GenericEqualityObj (get x i) (get y i) 
                      if not c then (res <- false; i <- lenx) 
                      else i <- i + 1
                    done;
                    res

            /// optimized case: Core implementation of structural equality on byte arrays.
            and GenericEqualityByteArray (x:byte[]) (y:byte[]) =
                let lenx = x.Length 
                let leny = y.Length 
                let c = (lenx = leny)
                if not c then c 
                else
                    let mutable i = 0 
                    let mutable res = true
                    while i < lenx do 
                      let c = byte_equals (get x i) (get y i) 
                      if not c then (res <- false; i <- lenx) 
                      else i <- i + 1
                    done;
                    res

            let inline GenericEqualityFast (x : 'a) (y : 'a) = 
                  GenericEqualityIntrinsic x y
                  when 'a : bool    = (# "ceq" x y : bool #)
                  when 'a : int     = (# "ceq" x y : bool #)
                  when 'a : sbyte   = (# "ceq" x y : bool #)
                  when 'a : int16   = (# "ceq" x y : bool #)
                  when 'a : int32   = (# "ceq" x y : bool #)
                  when 'a : int64   = (# "ceq" x y : bool #)
                  when 'a : byte    = (# "ceq" x y : bool #)
                  when 'a : uint16  = (# "ceq" x y : bool #)
                  when 'a : uint32  = (# "ceq" x y : bool #)
                  when 'a : uint64  = (# "ceq" x y : bool #)
                  when 'a : float   = (# "ceq" x y : bool #)
                  when 'a : float32 = (# "ceq" x y : bool #)
                  when 'a : char    = (# "ceq" x y : bool #)
                  when 'a : string  = System.String.Equals((# "" x : string #),(# "" y : string #))



            // Check for 'a = float or 'a = float32 (in case we didn't resolve this statically).
            // In these cases we must use the right "clt" to cope with NaN.  Note "<" etc. take
            // on a different meaning for floats nested inside data structures than when immediately 
            // applied to type float.  This gives us the IEEE semantics when "<" etc. are used on
            // floats, and a sound structural semantics when used on structured data, though 
            // comparisons on structural data containing floats do not get optimized.
            let GenericLessThanIntrinsic (x:'a) (y:'a) = 
                let xobj = box x 
                let yobj = box y 
                match xobj,yobj with 
                | _,null -> false
                | null,_ -> true
                | (:? float   as xs),(:? float   as ys) -> (# "" (# "clt" xs ys : bool #) : bool #)
                | (:? float32 as xs),(:? float32 as ys) -> (# "" (# "clt" xs ys : bool #) : bool #)
                | _ -> 
                     // OK, descend into structural comparison
                     (# "clt" (GenericComparisonObj xobj yobj) 0 : bool #)

            let GenericGreaterThanIntrinsic (x:'a) (y:'a) = 
                let xobj = box x 
                let yobj = box y 
                match xobj,yobj with 
                | null,_ -> false
                | _,null -> true
                // catch the NaN case for polymorphic use of the operator
                | (:? float   as xs),(:? float   as ys) -> (# "" (# "cgt" xs ys : bool #) : bool #)
                | (:? float32 as xs),(:? float32 as ys) -> (# "" (# "cgt" xs ys : bool #) : bool #)
                | _ -> 
                     // OK, descend into structural comparison
                     (# "cgt" (GenericComparisonObj xobj yobj) 0 : bool #)
               
            let GenericGreaterOrEqualIntrinsic (x:'a) (y:'a) = 
                let xobj = box x 
                let yobj = box y 
                match xobj,yobj with 
                | _,null -> true
                | null,_ -> false
                // catch the NaN case for polymorphic use of the operator
                | (:? float   as xs),(:? float   as ys) -> not (# "" (# "clt.un" xs ys : bool #) : bool #)
                | (:? float32 as xs),(:? float32 as ys) -> not (# "" (# "clt.un" xs ys : bool #) : bool #)
                | _ -> 
                     // OK, descend into structural comparison
                     (# "cgt" (GenericComparisonObj xobj yobj) (-1) : bool #)
               
            let GenericLessOrEqualIntrinsic (x:'a) (y:'a) = 
                let xobj = box x 
                let yobj = box y 
                match xobj,yobj with 
                | null,_ -> true
                | _,null -> false
                // catch the NaN case for polymorphic use of the operator
                | (:? float   as xs),(:? float   as ys) -> not (# "" (# "cgt.un" xs ys : bool #) : bool #)
                | (:? float32 as xs),(:? float32 as ys) -> not (# "" (# "cgt.un" xs ys : bool #) : bool #)
                | _ -> 
                     // OK, descend into structural comparison
                     (# "clt" (GenericComparisonObj xobj yobj) 1 : bool #)

            let inline GenericInequalityFast (x:'a) (y:'a) = (not(GenericEqualityFast x y) : bool)
              
            let inline GenericLessThanFast (x:'a) (y:'a) = 
                GenericLessThanIntrinsic x y
                when 'a : bool   = (# "clt" x y : bool #)
                when 'a : int    = (# "clt" x y : bool #)
                when 'a : sbyte  = (# "clt" x y : bool #)
                when 'a : int16  = (# "clt" x y : bool #)
                when 'a : int32  = (# "clt" x y : bool #)
                when 'a : int64  = (# "clt" x y : bool #)
                when 'a : byte   = (# "clt.un" x y : bool #)
                when 'a : uint16 = (# "clt.un" x y : bool #)
                when 'a : uint32 = (# "clt.un" x y : bool #)
                when 'a : uint64 = (# "clt.un" x y : bool #)
                when 'a : unativeint = (# "clt.un" x y : bool #)
                // Careful with NaN: dummy "" instruction ensures that the comparison instruction 
                // doesn't get turned into the corresponding negated branch instruction
                when 'a : nativeint  = (# "clt" x y : bool #)
                when 'a : float  = (# "" (# "clt" x y : bool #) : bool #)
                when 'a : float32= (# "" (# "clt" x y : bool #) : bool #)
                when 'a : char   = (# "clt" x y : bool #)
              
            let inline GenericGreaterThanFast (x:'a) (y:'a) = 
                GenericGreaterThanIntrinsic x y
                when 'a : bool       = (# "cgt" x y : bool #)
                when 'a : int        = (# "cgt" x y : bool #)
                when 'a : sbyte      = (# "cgt" x y : bool #)
                when 'a : int16      = (# "cgt" x y : bool #)
                when 'a : int32      = (# "cgt" x y : bool #)
                when 'a : int64      = (# "cgt" x y : bool #)
                when 'a : nativeint  = (# "cgt" x y : bool #)
                when 'a : byte       = (# "cgt.un" x y : bool #)
                when 'a : uint16     = (# "cgt.un" x y : bool #)
                when 'a : uint32     = (# "cgt.un" x y : bool #)
                when 'a : uint64     = (# "cgt.un" x y : bool #)
                when 'a : unativeint = (# "cgt.un" x y : bool #)
                // Careful with NaN: dummy "" instruction ensures that the comparison instruction 
                // doesn't get turned into the corresponding negated branch instruction
                when 'a : float      = (# "" (# "cgt" x y : bool #) : bool #)
                when 'a : float32    = (# "" (# "cgt" x y : bool #) : bool #)
                when 'a : char       = (# "cgt" x y : bool #)

            let inline GenericLessOrEqualFast (x:'a) (y:'a) = 
                GenericLessOrEqualIntrinsic x y
                when 'a : bool       = not (# "cgt" x y : bool #)
                when 'a : int        = not (# "cgt" x y : bool #)
                when 'a : sbyte      = not (# "cgt" x y : bool #)
                when 'a : int16      = not (# "cgt" x y : bool #)
                when 'a : int32      = not (# "cgt" x y : bool #)
                when 'a : int64      = not (# "cgt" x y : bool #)
                when 'a : nativeint  = not (# "cgt" x y : bool #)
                when 'a : byte       = not (# "cgt.un" x y : bool #)
                when 'a : uint16     = not (# "cgt.un" x y : bool #)
                when 'a : uint32     = not (# "cgt.un" x y : bool #)
                when 'a : uint64     = not (# "cgt.un" x y : bool #)
                when 'a : unativeint = not (# "cgt.un" x y : bool #)
                // Careful with NaN: dummy "" instruction ensures that the comparison instruction 
                // doesn't get turned into the corresponding negated branch instruction
                when 'a : float      = not(# "" (# "cgt.un" x y : bool #) : bool #)
                when 'a : float32    = not(# "" (# "cgt.un" x y : bool #) : bool #)
                when 'a : char       = not(# "cgt" x y : bool #)

            let inline GenericGreaterOrEqualFast (x:'a) (y:'a) = 
                GenericGreaterOrEqualIntrinsic x y
                when 'a : bool       = not (# "clt" x y : bool #)
                when 'a : int        = not (# "clt" x y : bool #)
                when 'a : sbyte      = not (# "clt" x y : bool #)
                when 'a : int16      = not (# "clt" x y : bool #)
                when 'a : int32      = not (# "clt" x y : bool #)
                when 'a : int64      = not (# "clt" x y : bool #)
                when 'a : nativeint  = not (# "clt" x y : bool #)
                when 'a : byte       = not (# "clt.un" x y : bool #)
                when 'a : uint16     = not (# "clt.un" x y : bool #)
                when 'a : uint32     = not (# "clt.un" x y : bool #)
                when 'a : uint64     = not (# "clt.un" x y : bool #)
                when 'a : unativeint = not (# "clt.un" x y : bool #)
                // Careful with NaN: dummy "" instruction ensures that the comparison instruction 
                // doesn't get turned into the corresponding negated branch instruction
                when 'a : float      = not (# "" (# "clt.un" x y : bool #) : bool #)
                when 'a : float32    = not (# "" (# "clt.un" x y : bool #) : bool #)
                when 'a : char       = not (# "clt" x y : bool #)

            let GenericMinimumFast x y = if GenericLessThanFast x y then x else y
            let GenericMaximumFast x y = if GenericLessThanFast x y then y else x


        let inline GenericEquality x y   = HashCompare.GenericEqualityFast x y
        let inline GenericComparison x y = HashCompare.GenericComparisonFast x y
        let inline GenericInequality x y = HashCompare.GenericInequalityFast x y
        let inline GenericLessThan x y = HashCompare.GenericLessThanFast x y
        let inline GenericGreaterThan x y = HashCompare.GenericGreaterThanFast x y
        let inline GenericLessOrEqual x y = HashCompare.GenericLessOrEqualFast x y
        let inline GenericGreaterOrEqual x y = HashCompare.GenericGreaterOrEqualFast x y
        let inline GenericMinimum x y = if HashCompare.GenericLessThanFast x y then x else y
        let inline GenericMaximum x y = if HashCompare.GenericLessThanFast x y then y else x

        let inline PhysicalEquality x y     = HashCompare.PhysicalEqualityFast x y
        let inline PhysicalInequality x y  = not (PhysicalEquality x y)
        let inline PhysicalHash x         = HashCompare.PhysicalHashFast x

        let inline GenericHashParam x nr         = HashCompare.GenericHashParamFast x nr
        let inline GenericHash x         = HashCompare.GenericHashFast x

        let inline StructuralEquality x y   = HashCompare.GenericEqualityFast x y
        let inline StructuralComparison x y = HashCompare.GenericComparisonFast x y
        let inline StructuralInequality x y = HashCompare.GenericInequalityFast x y
        let inline StructuralLessThan x y = HashCompare.GenericLessThanFast x y
        let inline StructuralGreaterThan x y = HashCompare.GenericGreaterThanFast x y
        let inline StructuralLessOrEqual x y = HashCompare.GenericLessOrEqualFast x y
        let inline StructuralGreaterOrEqual x y = HashCompare.GenericGreaterOrEqualFast x y
        let inline StructuralMinimum x y = if HashCompare.GenericLessThanFast x y then x else y
        let inline StructuralMaximum x y = if HashCompare.GenericLessThanFast x y then y else x
        let inline StructuralHashParam x nr         = HashCompare.GenericHashParamFast x nr
        let inline StructuralHash x         = HashCompare.GenericHashFast x

        let inline EnumOfValue (u : 'u) : 'e when 'e : enum<'u> = (# "" u : 'e)
        let inline EnumToValue (e : 'e) : 'u when 'e : enum<'u> = (# "" e : 'u)

    open LanguagePrimitives
    open LanguagePrimitives.Basics
    open LanguagePrimitives.IntrinsicOperators
    open LanguagePrimitives.IntrinsicFunctions


#if CLI_AT_LEAST_2_0
    [<System.Diagnostics.DebuggerDisplay("({Item1},{Item2})")>]
#endif
    type Tuple<'a,'b>                = { Item1: 'a; Item2: 'b }

#if CLI_AT_LEAST_2_0
    [<System.Diagnostics.DebuggerDisplay("({Item1},{Item2},{Item3})")>]
#endif
    type Tuple<'a,'b,'c>             = { Item1: 'a; Item2: 'b; Item3: 'c }

#if CLI_AT_LEAST_2_0
    [<System.Diagnostics.DebuggerDisplay("({Item1},{Item2},{Item3},{Item4})")>]
#endif
    type Tuple<'a,'b,'c,'d>          = { Item1: 'a; Item2: 'b; Item3: 'c ; Item4: 'd }

#if CLI_AT_LEAST_2_0
    [<System.Diagnostics.DebuggerDisplay("({Item1},{Item2},{Item3},{Item4},{Item5})")>]
#endif
    type Tuple<'a,'b,'c,'d,'e>       = { Item1: 'a; Item2: 'b; Item3: 'c ; Item4: 'd ; Item5: 'e  }

#if CLI_AT_LEAST_2_0
    [<System.Diagnostics.DebuggerDisplay("({Item1},{Item2},{Item3},{Item4},{Item5},{Item6})")>]
#endif
    type Tuple<'a,'b,'c,'d,'e,'f>    = { Item1: 'a; Item2: 'b; Item3: 'c ; Item4: 'd ; Item5: 'e ; Item6: 'f   }

#if CLI_AT_LEAST_2_0
    [<System.Diagnostics.DebuggerDisplay("({Item1},{Item2},{Item3},{Item4},{Item5},{Item6},{Item7})")>]
#endif
    type Tuple<'a,'b,'c,'d,'e,'f,'g> = { Item1: 'a; Item2: 'b; Item3: 'c ; Item4: 'd ; Item5: 'e ; Item6: 'f ; Item7: 'g  }

    [<DefaultAugmentation(false)>]
    type Choice<'a,'b> = 
      | Choice2_1 of 'a 
      | Choice2_2 of 'b
    
    [<DefaultAugmentation(false)>]
    type Choice<'a,'b,'c> = 
      | Choice3_1 of 'a 
      | Choice3_2 of 'b
      | Choice3_3 of 'c
    
    [<DefaultAugmentation(false)>]
    type Choice<'a,'b,'c,'d> = 
      | Choice4_1 of 'a 
      | Choice4_2 of 'b
      | Choice4_3 of 'c
      | Choice4_4 of 'd
    
    [<DefaultAugmentation(false)>]
    type Choice<'a,'b,'c,'d,'e> = 
      | Choice5_1 of 'a 
      | Choice5_2 of 'b
      | Choice5_3 of 'c
      | Choice5_4 of 'd
      | Choice5_5 of 'e
    
    [<DefaultAugmentation(false)>]
    type Choice<'a,'b,'c,'d,'e,'f> = 
      | Choice6_1 of 'a 
      | Choice6_2 of 'b
      | Choice6_3 of 'c
      | Choice6_4 of 'd
      | Choice6_5 of 'e
      | Choice6_6 of 'f
    
    [<DefaultAugmentation(false)>]
    type Choice<'a,'b,'c,'d,'e,'f,'g> = 
      | Choice7_1 of 'a 
      | Choice7_2 of 'b
      | Choice7_3 of 'c
      | Choice7_4 of 'd
      | Choice7_5 of 'e
      | Choice7_6 of 'f
      | Choice7_7 of 'g
          
    //-------------------------------------------------------------------------
    // F#-specific Exceptions

    exception Failure of string
    exception InvalidArgument of string
    exception AssertionFailure of string * int * int
    exception MatchFailure of string * int * int

    //-------------------------------------------------------------------------
    // Function Values

    [<AbstractClass>]
    type TypeFunc() = 
        abstract Specialize<'T> : unit -> obj

    [<AbstractClass>]
    type FastFunc<'T,'U>() = 
        abstract Invoke : 'T -> 'U

    module OptimizedClosures = 

          [<AbstractClass>]
          type FastFunc2 <'T,'U,'V>() = 
              inherit FastFunc<'T,('U -> 'V)>()
              abstract Invoke : 'T * 'U -> 'V
              override f.Invoke(t) = (fun u -> f.Invoke(t,u))
              static member Adapt(f : 'T -> 'U -> 'V) = 
                  match box f with 
                  | :? FastFunc2 <'T,'U,'V> as f -> f
                  | _ -> { new FastFunc2<'T,'U,'V>() with 
                              member x.Invoke(t,u) = f t u }

          [<AbstractClass>]
          type FastFunc3<'T,'U,'V,'W>() = 
              inherit FastFunc<'T,('U -> 'V -> 'W)>()
              abstract Invoke : 'T * 'U * 'V -> 'W
              override f.Invoke(t) = (fun u v -> f.Invoke(t,u,v))
              static member Adapt(f : 'T -> 'U -> 'V -> 'W) = 
                  match box f with 
                  | :? FastFunc3<'T,'U,'V,'W> as f -> f
                  | :? FastFunc2<'T,'U,FastFunc<'V,'W>> as f ->
                         { new FastFunc3<'T,'U,'V,'W>() with 
                              member x.Invoke(t,u,v) = f.Invoke(t,u).Invoke(v) }
                  | _ -> { new FastFunc3<'T,'U,'V,'W>() with 
                              member x.Invoke(t,u,v) = f t u v }

          [<AbstractClass>]
          type FastFunc4 <'T,'U,'V,'W,'X>() = 
              inherit FastFunc<'T,('U -> 'V -> 'W -> 'X)>()
              abstract Invoke : 'T * 'U * 'V * 'W -> 'X
              static member Adapt(f : 'T -> 'U -> 'V -> 'W -> 'X) = 
                  match box f with 
                  | :? FastFunc4<'T,'U,'V,'W,'X> as f -> f
                  | :? FastFunc3<'T,'U,'V,FastFunc<'W,'X>> as f ->
                         { new FastFunc4<'T,'U,'V,'W,'X>() with 
                              member x.Invoke(t,u,v,w) = f.Invoke(t,u,v).Invoke(w) }
                  | :? FastFunc2<'T,'U,FastFunc<'V,FastFunc<'W,'X>>> as f ->
                         { new FastFunc4<'T,'U,'V,'W,'X>() with 
                              member x.Invoke(t,u,v,w) = f.Invoke(t,u).Invoke(v).Invoke(w) }
                  | _ -> { new FastFunc4<'T,'U,'V,'W,'X>() with 
                              member x.Invoke(t,u,v,w) = f t u v w }
              override f.Invoke(t) = (fun u v w -> f.Invoke(t,u,v,w))

          /// The .NET type used to represent F# function values that accept
          /// five iterated (curried) arguments without intervening execution.  This type should not
          /// typically used directly from either F# code or from other .NET languages.
          [<AbstractClass>]
          type FastFunc5<'T,'U,'V,'W,'X,'Y>() =
              inherit FastFunc<'T,('U -> 'V -> 'W -> 'X -> 'Y)>()
              abstract Invoke : 'T * 'U * 'V * 'W * 'X -> 'Y
              override f.Invoke(t) = (fun u v w x -> f.Invoke(t,u,v,w,x))
          
          let inline invokeFast2((f1 : FastFunc<_,_>), t,u) =
              match f1 with
              | :? FastFunc2<_,_,_> as f2 -> f2.Invoke(t,u)
              | _                         -> (f1.Invoke(t)) u     
          
          let inline invokeFast3((f1 : FastFunc<_,_>), t,u,v) =
               match f1 with
               | :? FastFunc3<_,_,_,_> as f3 -> f3.Invoke(t,u,v)
               | :? FastFunc2<_,_,_>   as f2 -> (f2.Invoke(t,u)) v
               | _                           -> (f1.Invoke(t)) u v
               
          let inline invokeFast4((f1 : FastFunc<_,_>), t,u,v,w) =
               match f1 with
               | :? FastFunc4<_,_,_,_,_> as f4 -> f4.Invoke(t,u,v,w)
               | :? FastFunc3<_,_,_,_>   as f3 -> (f3.Invoke(t,u,v)) w
               | :? FastFunc2<_,_,_>     as f2 -> (f2.Invoke(t,u)) v w
               | _                             -> (f1.Invoke(t)) u v w

          let inline invokeFast5((f1 : FastFunc<_,_>), t,u,v,w,x) =
               match f1 with
               | :? FastFunc5<_,_,_,_,_,_> as f5 -> f5.Invoke(t,u,v,w,x)
               | :? FastFunc4<_,_,_,_,_>   as f4 -> (f4.Invoke(t,u,v,w)) x
               | :? FastFunc3<_,_,_,_>     as f3 -> (f3.Invoke(t,u,v)) w x
               | :? FastFunc2<_,_,_>       as f2 -> (f2.Invoke(t,u)) v w x
               | _                                     -> (f1.Invoke(t)) u v w x

        (*  An alternative implementation, where arity-N functions also support
            entry points for 2...N-1 argument applications.  Creates some more type instantiations
            for little gain.  
          /// The .NET type used to represent F# function values that accept
          /// three iterated (curried) arguments without intervening execution.  This type should not
          /// typically used directly from either F# code or from other .NET languages.
          type FastFunc3<'T,'U,'V,'W> = 
              inherit FastFunc2<'T,'U,('V -> 'W)>()
              abstract Invoke : 'T * 'U * 'V -> 'W
              override f.Invoke(t,u) = (fun v -> f.Invoke(t,u,v))

          /// The .NET type used to represent F# function values that accept
          /// four iterated (curried) arguments without intervening execution.  This type should not
          /// typically used directly from either F# code or from other .NET languages.
          type FastFunc4<'T,'U,'V,'W,'X> = 
              inherit FastFunc3<'T,'U,'V,('W -> 'X)>()
              abstract Invoke : 'T * 'U * 'V * 'W -> 'X
              override f.Invoke(t,u,v) = (fun w -> f.Invoke(t,u,v,w))

          /// The .NET type used to represent F# function values that accept
          /// five iterated (curried) arguments without intervening execution.  This type should not
          /// typically used directly from either F# code or from other .NET languages.
          type FastFunc5<'T,'U,'V,'W,'X,'Y> = 
            inherit FastFunc4<'T,'U,'V,'W,('X -> 'Y)>()
            abstract Invoke : 'T * 'U * 'V * 'W * 'X -> 'Y
            override f.Invoke(t,u,v,w) = (fun x -> f.Invoke(t,u,v,w,x))
          
        *)


    type FastFunc<'T,'U> with
#if CLI_AT_LEAST_2_0
        [<OverloadID("FromConverter")>]
        static member op_Implicit(f : System.Converter<_,_>) : ('T -> 'U) =  (fun t -> f.Invoke(t))
        [<OverloadID("ToConverter")>]
        static member op_Implicit( f : ('T -> 'U) ) =  new System.Converter<'T,'U>(f)
#endif
        static member InvokeFast2(f,(t:'T),(u:'U))       = OptimizedClosures.invokeFast2(f,t,u) 
        static member InvokeFast3(f,(t:'T),(u:'U),v)     = OptimizedClosures.invokeFast3(f,t,u,v)
        static member InvokeFast4(f,(t:'T),(u:'U),v,w)   = OptimizedClosures.invokeFast4(f,t,u,v,w)
        static member InvokeFast5(f,(t:'T),(u:'U),v,w,x) = OptimizedClosures.invokeFast5(f,t,u,v,w,x)


    //-------------------------------------------------------------------------
    // Refs
    //-------------------------------------------------------------------------

    #if CLI_AT_LEAST_2_0
    [<System.Diagnostics.DebuggerDisplay("{contents}")>]
    #endif
    type Ref<'a> = 
        { mutable contents: 'a }
        member x.Value 
            with get() = x.contents
            and  set v = x.contents <- v
#if CLI_AT_MOST_1_1
#else
        member inline x.Pin (f : 'a nativeptr -> 'b) : 'b = 
            let inline of_nativeint (x:nativeint) = (# "" x : 'a nativeptr    #)
            let gch = System.Runtime.InteropServices.GCHandle.Alloc(x,System.Runtime.InteropServices.GCHandleType.Pinned) 
            let addr = of_nativeint (gch.AddrOfPinnedObject()) 
            try  f addr
            finally
              gch.Free()

        member inline x.PinHandle () : 'a nativeptr * _ = 
            let inline of_nativeint (x:nativeint) = (# "" x : 'a nativeptr    #)
            let gch = System.Runtime.InteropServices.GCHandle.Alloc(x,System.Runtime.InteropServices.GCHandleType.Pinned) 
            let addr = of_nativeint (gch.AddrOfPinnedObject()) 
            addr , gch
#endif

    and 'a ref = Ref<'a> 

    //-------------------------------------------------------------------------
    // Options
    //-------------------------------------------------------------------------

    [<DefaultAugmentation(false)
    #if CLI_AT_LEAST_2_0
      ;System.Diagnostics.DebuggerDisplay("Some{Item}")
    #endif
      ; CompilationRepresentation(CompilationRepresentationFlags.PermitNull)
    >]
    type Option<'a> = 
        | None :       'a option
        | Some : 'a -> 'a option 
        /// The module 'Option' contains additional values and functions related to this type.
        [<CompilationRepresentation(CompilationRepresentationFlags.Instance)>]
        member x.Item = match x with Some x -> x | None -> raise (new System.InvalidOperationException("Option.Item"))

        [<CompilationRepresentation(CompilationRepresentationFlags.Instance)>]
        member x.Value = match x with Some x -> x | None -> raise (new System.InvalidOperationException("Option.Value"))

        /// The module 'Option' contains additional values and functions related to this type.
        member x.IsNone = match x with None -> true | _ -> false

        /// The module 'Option' contains additional values and functions related to this type.
        member x.IsSome = match x with Some _ -> true | _ -> false

        /// The module 'Option' contains additional values and functions related to this type.
        static member None : 'a option = None

        /// The module 'Option' contains additional values and functions related to this type.
        static member Some(x) : 'a option = Some(x)

    and 'a option = 'a Option 


    type ReifiedType<'a> = { result: System.Type }



#if CLI_AT_MOST_1_1
//-------------------------------------------------------------------------
// Compatibility collections
//-------------------------------------------------------------------------

namespace Microsoft.FSharp.Compatibility

    open Microsoft.FSharp.Core

    type IEnumerator<'a> =
        interface
            inherit System.Collections.IEnumerator
            inherit System.IDisposable
            abstract Current : 'a
        end

    type IEnumerable<'a> =
        interface
            inherit System.Collections.IEnumerable
            abstract GetEnumerator : unit -> IEnumerator<'a>
        end

    type KeyValuePair<'Key,'Value> =
        struct
            val Key: 'Key
            val Value: 'Value
            new (k,v) = {Key=k; Value=v }
        end

    type KeyNotFoundException(message:string) =
        inherit System.Exception(message)
        new () = KeyNotFoundException("key not found in dictionary")

    type IEqualityComparer<'key> = 
        interface 
            abstract GetHashCode : 'key -> int
            abstract Equals : 'key * 'key -> bool
        end

    type IComparer<'key> = 
        interface 
            abstract Compare: 'key * 'key -> int
        end

#endif

//-------------------------------------------------------------------------
// Operators
//-------------------------------------------------------------------------


namespace Microsoft.FSharp.Core

    open System
    open Microsoft.FSharp.Core
    open Microsoft.FSharp.Core.LanguagePrimitives
    open Microsoft.FSharp.Core.LanguagePrimitives.IntrinsicOperators
    open Microsoft.FSharp.Core.LanguagePrimitives.IntrinsicFunctions
    open Microsoft.FSharp.Core.LanguagePrimitives.Basics
    #if CLI_AT_LEAST_2_0
    open System.Collections.Generic
    #else
    open Microsoft.FSharp.Compatibility
    #endif

    type IRange<'a> = 
        interface
            inherit IEnumerable<'a>
            abstract Start  : 'a
            abstract Finish : 'a option
            abstract Step : 'a
        end

    type seq<'a> = IEnumerable<'a>


    module Operators = 

        let seq x = x (* temp addition for comprehension/workflow transitions *)
        let inline unbox (x:obj) = UnboxGeneric(x)
        let inline box   (x:'a)  = (# "box !0"       type ('a) x : obj #)
        let inline raise (e: #exn) = (# "throw" (e :> exn) : 'a #)

        let inline not (b:bool) = (# " ldc.i4 0 ceq" b : bool #)
           
        let inline obj_eq (xobj:obj) (yobj:obj) = (# "ceq" xobj yobj : bool #)

        let inline (+) (x: ^a) (y: ^b) : ^c = 
             ((^a or ^b): (static member (+) : ^a * ^b -> ^c) (x,y))
             when ^a : int32       and ^b : int32      = (# "add" x y : int32 #)
             when ^a : float       and ^b : float      = (# "add" x y : float #)
             when ^a : float32     and ^b : float32    = (# "add" x y : float32 #)
             when ^a : int64       and ^b : int64      = (# "add" x y : int64 #)
             when ^a : uint64      and ^b : uint64     = (# "add" x y : uint64 #)
             when ^a : uint32      and ^b : uint32     = (# "add" x y : uint32 #)
             when ^a : nativeint   and ^b : nativeint  = (# "add" x y : nativeint #)
             when ^a : unativeint  and ^b : unativeint = (# "add" x y : unativeint #)
             when ^a : int16       and ^b : int16      = (# "conv.i2" (# "add" x y : int32 #) : int16 #)
             when ^a : uint16      and ^b : uint16     = (# "conv.u2" (# "add" x y : uint32 #) : uint16 #)
             when ^a : sbyte       and ^b : sbyte      = (# "conv.i1" (# "add" x y : int32 #) : sbyte #)
             when ^a : byte        and ^b : byte       = (# "conv.u1" (# "add" x y : uint32 #) : byte #)
             when ^a : string      and ^b : string     = (# "" (System.String.Concat((# "" x : string #),(# "" y : string #))) : ^a #)

        let inline (-) (x: ^a) (y: ^b) : ^c = 
             ((^a or ^b): (static member (-) : ^a * ^b -> ^c) (x,y))
             when ^a : int32      and ^b : int32      = (# "sub" x y : int32 #)
             when ^a : float      and ^b : float      = (# "sub" x y : float #)
             when ^a : float32    and ^b : float32    = (# "sub" x y : float32 #)
             when ^a : int64      and ^b : int64      = (# "sub" x y : int64 #)
             when ^a : uint64     and ^b : uint64     = (# "sub" x y : uint64 #)
             when ^a : uint32     and ^b : uint32     = (# "sub" x y : uint32 #)
             when ^a : nativeint  and ^b : nativeint  = (# "sub" x y : nativeint #)
             when ^a : unativeint and ^b : unativeint = (# "sub" x y : unativeint #)
             when ^a : int16       and ^b : int16      = (# "conv.i2" (# "sub" x y : int32 #) : int16 #)
             when ^a : uint16      and ^b : uint16     = (# "conv.u2" (# "sub" x y : uint32 #) : uint16 #)
             when ^a : sbyte       and ^b : sbyte      = (# "conv.i1" (# "sub" x y : int32 #) : sbyte #)
             when ^a : byte        and ^b : byte       = (# "conv.u1" (# "sub" x y : uint32 #) : byte #)


        let inline ( * ) (x: ^a) (y: ^b) : ^c = 
           ((^a or ^b): (static member ( * ) : ^a * ^b -> ^c) (x,y))
           when ^a : int32      and ^b : int32      = (# "mul" x y : int32 #)
           when ^a : float      and ^b : float      = (# "mul" x y : float #)
           when ^a : float32    and ^b : float32    = (# "mul" x y : float32 #)
           when ^a : int64      and ^b : int64      = (# "mul" x y : int64 #)
           when ^a : uint64     and ^b : uint64     = (# "mul" x y : uint64 #)
           when ^a : uint32     and ^b : uint32     = (# "mul" x y : uint32 #)
           when ^a : nativeint  and ^b : nativeint  = (# "mul" x y : nativeint #)
           when ^a : unativeint and ^b : unativeint = (# "mul" x y : unativeint #)
           when ^a : int16       and ^b : int16      = (# "conv.i2" (# "mul" x y : int32 #) : int16 #)
           when ^a : uint16      and ^b : uint16     = (# "conv.u2" (# "mul" x y : uint32 #) : uint16 #)
           when ^a : sbyte       and ^b : sbyte      = (# "conv.i1" (# "mul" x y : int32 #) : sbyte #)
           when ^a : byte        and ^b : byte       = (# "conv.u1" (# "mul" x y : uint32 #) : byte #)

        let inline ( / ) (x: ^a) (y: ^b) : ^c = 
           ((^a or ^b): (static member (/) : ^a * ^b -> ^c) (x,y))
           when ^a : int32       and ^b : int32      = (# "div" x y : int32 #)
           when ^a : float       and ^b : float      = (# "div" x y : float #)
           when ^a : float32     and ^b : float32    = (# "div" x y : float32 #)
           when ^a : int64       and ^b : int64      = (# "div" x y : int64 #)
           when ^a : uint64      and ^b : uint64     = (# "div.un" x y : uint64 #)
           when ^a : uint32      and ^b : uint32     = (# "div.un" x y : uint32 #)
           when ^a : nativeint   and ^b : nativeint  = (# "div" x y : nativeint #)
           when ^a : unativeint  and ^b : unativeint = (# "div.un" x y : unativeint #)
           when ^a : int16       and ^b : int16      = (# "conv.i2" (# "div" x y : int32 #) : int16 #)
           when ^a : uint16      and ^b : uint16     = (# "conv.u2" (# "div.un" x y : uint32 #) : uint16 #)
           when ^a : sbyte       and ^b : sbyte      = (# "conv.i1" (# "div" x y : int32 #) : sbyte #)
           when ^a : byte        and ^b : byte       = (# "conv.u1" (# "div.un" x y : uint32 #) : byte #)

        let inline ( % ) (x: ^a) (y: ^b) : ^c = 
           ((^a or ^b): (static member (%) : ^a * ^b -> ^c) (x,y))
           when ^a : int32       and ^b : int32      = (# "rem" x y : int32 #)
           when ^a : float       and ^b : float      = (# "rem" x y : float #)
           when ^a : float32     and ^b : float32    = (# "rem" x y : float32 #)
           when ^a : int64       and ^b : int64      = (# "rem" x y : int64 #)
           when ^a : uint64      and ^b : uint64     = (# "rem.un" x y : uint64 #)
           when ^a : uint32      and ^b : uint32     = (# "rem.un" x y : uint32 #)
           when ^a : nativeint   and ^b : nativeint  = (# "rem" x y : nativeint #)
           when ^a : unativeint  and ^b : unativeint = (# "rem.un" x y : unativeint #)
           when ^a : int16       and ^b : int16      = (# "conv.i2" (# "rem"    x y : int32  #) : int16  #)
           when ^a : uint16      and ^b : uint16     = (# "conv.u2" (# "rem.un" x y : uint32 #) : uint16 #)
           when ^a : sbyte       and ^b : sbyte      = (# "conv.i1" (# "rem"    x y : int32  #) : sbyte  #)
           when ^a : byte        and ^b : byte       = (# "conv.u1" (# "rem.un" x y : uint32 #) : byte   #)

        let inline (~+) (x: ^a) : ^a =
           (^a: (static member (~+) : ^a -> ^a) (x))
           when ^a : int32      = x
           when ^a : float      = x
           when ^a : float32    = x
           when ^a : int64      = x
           when ^a : uint64     = x
           when ^a : uint32     = x
           when ^a : int16      = x
           when ^a : uint16     = x
           when ^a : nativeint  = x
           when ^a : unativeint = x
           when ^a : sbyte      = x
           when ^a : byte       = x
           when ^a : decimal    = x

        let inline mask (n:int) (m:int) = (# "and" n m : int #)
        let inline (<<<) (x: ^a) (n:int) : ^a = 
           (^a: (static member (<<<) : ^a * int -> ^a) (x,n))
           when ^a : int32      = (# "shl" x (mask n 31) : int #)
           when ^a : uint32     = (# "shl" x (mask n 31) : uint32 #)
           when ^a : int64      = (# "shl" x (mask n 63) : int64 #)
           when ^a : uint64     = (# "shl" x (mask n 63) : uint64 #)
           when ^a : nativeint  = (# "shl" x n : nativeint #)
           when ^a : unativeint = (# "shl" x n : unativeint #)
           when ^a : int16      = (# "conv.i2" (# "shl" x (mask n 15) : int32  #) : int16 #)
           when ^a : uint16     = (# "conv.u2" (# "shl" x (mask n 15) : uint32 #) : uint16 #)
           when ^a : sbyte      = (# "conv.i1" (# "shl" x (mask n 7 ) : int32  #) : sbyte #)
           when ^a : byte       = (# "conv.u1" (# "shl" x (mask n 7 ) : uint32 #) : byte #)

        let inline (>>>) (x: ^a) (n:int) : ^a = 
           (^a: (static member (>>>) : ^a * int -> ^a) (x,n))
           when ^a : int32      = (# "shr"    x (mask n 31) : int32 #)
           when ^a : uint32     = (# "shr.un" x (mask n 31) : uint32 #)
           when ^a : int64      = (# "shr"    x (mask n 63) : int64 #)
           when ^a : uint64     = (# "shr.un" x (mask n 63) : uint64 #)
           when ^a : nativeint  = (# "shr"    x n : nativeint #)
           when ^a : unativeint = (# "shr.un" x n : unativeint #)
           when ^a : int16      = (# "conv.i2" (# "shr"    x (mask n 15) : int32  #) : int16 #)
           when ^a : uint16     = (# "conv.u2" (# "shr.un" x (mask n 15) : uint32 #) : uint16 #)
           when ^a : sbyte      = (# "conv.i1" (# "shr"    x (mask n 7 ) : int32  #) : sbyte #)
           when ^a : byte       = (# "conv.u1" (# "shr.un" x (mask n 7 ) : uint32 #) : byte #)

        let inline op_OverloadedLogicalRightShift (x: ^a) (n:int) : ^a = 
           (^a: (static member (>>>) : ^a * int -> ^a) (x,n))
           when ^a : int32      = (# "shr.un" x (mask n 31) : int32 #)
           when ^a : uint32     = (# "shr.un" x (mask n 31) : uint32 #)
           when ^a : int64      = (# "shr.un" x (mask n 63) : int64 #)
           when ^a : uint64     = (# "shr.un" x (mask n 63) : uint64 #)
           when ^a : nativeint  = (# "shr.un" x n : nativeint #)
           when ^a : unativeint = (# "shr.un" x n : unativeint #)
           when ^a : int16      = (# "conv.i2" (# "shr.un" x (mask n 15) : int32  #) : int16 #)
           when ^a : uint16     = (# "conv.u2" (# "shr.un" x (mask n 15) : uint32 #) : uint16 #)
           when ^a : sbyte      = (# "conv.i1" (# "shr.un" x (mask n 7 ) : int32  #) : sbyte #)
           when ^a : byte       = (# "conv.u1" (# "shr.un" x (mask n 7 ) : uint32 #) : byte #)

        let inline (&&&) (x: ^a) (y: ^a) : ^a = 
           (^a: (static member (&&&) : ^a * ^a -> ^a) (x,y))
           when ^a : int32      = (# "and" x y : int32 #)
           when ^a : int64      = (# "and" x y : int64 #)
           when ^a : uint64     = (# "and" x y : uint64 #)
           when ^a : uint32     = (# "and" x y : uint32 #)
           when ^a : int16      = (# "and" x y : int16 #)
           when ^a : uint16     = (# "and" x y : uint16 #)
           when ^a : nativeint  = (# "and" x y : nativeint #)
           when ^a : unativeint = (# "and" x y : unativeint #)
           when ^a : sbyte      = (# "and" x y : sbyte #)
           when ^a : byte       = (# "and" x y : byte #)

        let inline (|||) (x: ^a) (y: ^a) : ^a = 
           (^a: (static member (|||) : ^a * ^a -> ^a) (x,y))
           when ^a : int32      = (# "or" x y : int32 #)
           when ^a : int64      = (# "or" x y : int64 #)
           when ^a : uint64     = (# "or" x y : uint64 #)
           when ^a : uint32     = (# "or" x y : uint32 #)
           when ^a : int16      = (# "or" x y : int16 #)
           when ^a : uint16     = (# "or" x y : uint16 #)
           when ^a : nativeint  = (# "or" x y : nativeint #)
           when ^a : unativeint = (# "or" x y : unativeint #)
           when ^a : sbyte      = (# "or" x y : sbyte #)
           when ^a : byte       = (# "or" x y : byte #)

        let inline (^^^) (x: ^a) (y: ^a) : ^a = 
           (^a: (static member (^^^) : ^a * ^a -> ^a) (x,y))
           when ^a : int32      = (# "xor" x y : int32 #)
           when ^a : int64      = (# "xor" x y : int64 #)
           when ^a : uint64     = (# "xor" x y : uint64 #)
           when ^a : uint32     = (# "xor" x y : uint32 #)
           when ^a : int16      = (# "xor" x y : int16 #)
           when ^a : uint16     = (# "xor" x y : uint16 #)
           when ^a : nativeint  = (# "xor" x y : nativeint #)
           when ^a : unativeint = (# "xor" x y : unativeint #)
           when ^a : sbyte      = (# "xor" x y : sbyte #)
           when ^a : byte       = (# "xor" x y : byte #)

        let inline (~~~) (x: ^a) : ^a = 
           (^a: (static member (~~~) : ^a -> ^a) (x))
           when ^a : int32      = (# "not" x : int32 #)
           when ^a : int64      = (# "not" x : int64 #)
           when ^a : uint64     = (# "not" x : uint64 #)
           when ^a : uint32     = (# "not" x : uint32 #)
           when ^a : nativeint  = (# "not" x : nativeint #)
           when ^a : unativeint = (# "not" x : unativeint #)
           when ^a : int16      = (# "conv.i2" (# "not" x : int32  #) : int16 #)
           when ^a : uint16     = (# "conv.u2" (# "not" x : uint32 #) : uint16 #)
           when ^a : sbyte      = (# "conv.i1" (# "not" x : int32  #) : sbyte #)
           when ^a : byte       = (# "conv.u1" (# "not" x : uint32 #) : byte #)

        let inline castToString (x:'a) = (# "" x : string #)  // internal
        
        // Conversion functions named by target type:
        //   byte, sbyte, uint16, int16, uint32, int32, uint64, int64
        let inline byte (x: ^a) = 
            (^a : (static member ToByte: ^a -> byte) (x))
             when ^a : string     = (System.Byte.Parse(castToString x,System.Globalization.CultureInfo.InvariantCulture))
             when ^a : float      = (# "conv.u1" x  : byte #)
             when ^a : float32    = (# "conv.u1" x  : byte #)
             when ^a : int64      = (# "conv.u1" x  : byte #)
             when ^a : int32      = (# "conv.u1" x  : byte #)
             when ^a : int16      = (# "conv.u1" x  : byte #)
             when ^a : nativeint  = (# "conv.u1" x  : byte #)
             when ^a : sbyte      = (# "conv.u1" x  : byte #)
             when ^a : uint64     = (# "conv.u1" x  : byte #)
             when ^a : uint32     = (# "conv.u1" x  : byte #)
             when ^a : uint16     = (# "conv.u1" x  : byte #)
             when ^a : char       = (# "conv.u1" x  : byte #)
             when ^a : unativeint = (# "conv.u1" x  : byte #)
             when ^a : byte       = (# "conv.u1" x  : byte #)

        let inline sbyte (x: ^a) = 
            (^a : (static member ToSByte: ^a -> sbyte) (x))
             when ^a : string     = (System.SByte.Parse(castToString x,System.Globalization.CultureInfo.InvariantCulture))
             when ^a : float     = (# "conv.i1" x  : sbyte #)
             when ^a : float32   = (# "conv.i1" x  : sbyte #)
             when ^a : int64     = (# "conv.i1" x  : sbyte #)
             when ^a : int32     = (# "conv.i1" x  : sbyte #)
             when ^a : int16     = (# "conv.i1" x  : sbyte #)
             when ^a : nativeint = (# "conv.i1" x  : sbyte #)
             when ^a : sbyte     = (# "conv.i1" x  : sbyte #)
             when ^a : uint64     = (# "conv.i1" x  : sbyte #)
             when ^a : uint32     = (# "conv.i1" x  : sbyte #)
             when ^a : uint16     = (# "conv.i1" x  : sbyte #)
             when ^a : char       = (# "conv.i1" x  : sbyte #)
             when ^a : unativeint = (# "conv.i1" x  : sbyte #)
             when ^a : byte     = (# "conv.i1" x  : sbyte #)

        let inline uint16 (x: ^a) = 
            (^a : (static member ToUInt16: ^a -> uint16) (x))
             when ^a : string     = (System.UInt16.Parse(castToString x,System.Globalization.CultureInfo.InvariantCulture))
             when ^a : float     = (# "conv.u2" x  : uint16 #)
             when ^a : float32   = (# "conv.u2" x  : uint16 #)
             when ^a : int64     = (# "conv.u2" x  : uint16 #)
             when ^a : int32     = (# "conv.u2" x  : uint16 #)
             when ^a : int16     = (# "conv.u2" x  : uint16 #)
             when ^a : nativeint = (# "conv.u2" x  : uint16 #)
             when ^a : sbyte     = (# "conv.u2" x  : uint16 #)
             when ^a : uint64     = (# "conv.u2" x  : uint16 #)
             when ^a : uint32     = (# "conv.u2" x  : uint16 #)
             when ^a : uint16     = (# "conv.u2" x  : uint16 #)
             when ^a : char       = (# "conv.u2" x  : uint16 #)
             when ^a : unativeint = (# "conv.u2" x  : uint16 #)
             when ^a : byte     = (# "conv.u2" x  : uint16 #)

        let inline int16 (x: ^a) = 
            (^a : (static member ToInt16: ^a -> int16) (x))
             when ^a : string     = (System.Int16.Parse(castToString x,System.Globalization.CultureInfo.InvariantCulture))
             when ^a : float     = (# "conv.i2" x  : int16 #)
             when ^a : float32   = (# "conv.i2" x  : int16 #)
             when ^a : int64     = (# "conv.i2" x  : int16 #)
             when ^a : int32     = (# "conv.i2" x  : int16 #)
             when ^a : int16     = (# "conv.i2" x  : int16 #)
             when ^a : nativeint = (# "conv.i2" x  : int16 #)
             when ^a : sbyte     = (# "conv.i2" x  : int16 #)
             when ^a : uint64     = (# "conv.i2" x  : int16 #)
             when ^a : uint32     = (# "conv.i2" x  : int16 #)
             when ^a : uint16     = (# "conv.i2" x  : int16 #)
             when ^a : char       = (# "conv.i2" x  : int16 #)
             when ^a : unativeint = (# "conv.i2" x  : int16 #)
             when ^a : byte     = (# "conv.i2" x  : int16 #)

        let inline uint32 (x: ^a) = 
            (^a : (static member ToUInt32: ^a -> uint32) (x))
             when ^a : string     = (System.UInt32.Parse(castToString x,System.Globalization.CultureInfo.InvariantCulture))
             when ^a : float     = (# "conv.u4" x  : uint32 #)
             when ^a : float32   = (# "conv.u4" x  : uint32 #)
             when ^a : int64     = (# "conv.u4" x  : uint32 #)
             when ^a : int32     = (# "conv.u4" x  : uint32 #)
             when ^a : int16     = (# "conv.u4" x  : uint32 #)
             when ^a : nativeint = (# "conv.u4" x  : uint32 #)
             when ^a : sbyte     = (# "conv.u4" x  : uint32 #)
             when ^a : uint64     = (# "conv.u4" x  : uint32 #)
             when ^a : uint32     = (# "conv.u4" x  : uint32 #)
             when ^a : uint16     = (# "conv.u4" x  : uint32 #)
             when ^a : char       = (# "conv.u4" x  : uint32 #)
             when ^a : unativeint = (# "conv.u4" x  : uint32 #)
             when ^a : byte     = (# "conv.u4" x  : uint32 #)

        let inline int32 (x: ^a) = 
            (^a : (static member ToInt32: ^a -> int32) (x))
             when ^a : string     = (System.Int32.Parse(castToString x,System.Globalization.CultureInfo.InvariantCulture))
             when ^a : float     = (# "conv.i4" x  : int32 #)
             when ^a : float32   = (# "conv.i4" x  : int32 #)
             when ^a : int64     = (# "conv.i4" x  : int32 #)
             when ^a : int32     = (# "conv.i4" x  : int32 #)
             when ^a : int16     = (# "conv.i4" x  : int32 #)
             when ^a : nativeint = (# "conv.i4" x  : int32 #)
             when ^a : sbyte     = (# "conv.i4" x  : int32 #)
             when ^a : uint64     = (# "conv.i4" x  : int32 #)
             when ^a : uint32     = (# "conv.i4" x  : int32 #)
             when ^a : uint16     = (# "conv.i4" x  : int32 #)
             when ^a : char       = (# "conv.i4" x  : int32 #)
             when ^a : unativeint = (# "conv.i4" x  : int32 #)
             when ^a : byte     = (# "conv.i4" x  : int32 #)

        let inline int    x = int32  x         
        let inline enum< ^a when ^a : enum<int32> > (x:int32) : ^a = (# "" x : ^a #)
        let inline uint64 (x: ^a) = 
            (^a : (static member ToUInt64: ^a -> uint64) (x))
             when ^a : string     = (System.UInt64.Parse(castToString x,System.Globalization.CultureInfo.InvariantCulture))
             when ^a : float     = (# "conv.u8" x  : uint64 #)
             when ^a : float32   = (# "conv.u8" x  : uint64 #)
             when ^a : int64     = (# "conv.u8" x  : uint64 #)
             when ^a : int32     = (# "conv.u8" x  : uint64 #)
             when ^a : int16     = (# "conv.u8" x  : uint64 #)
             when ^a : nativeint = (# "conv.u8" x  : uint64 #)
             when ^a : sbyte     = (# "conv.u8" x  : uint64 #)
             when ^a : uint64     = (# "conv.u8" x  : uint64 #)
             when ^a : uint32     = (# "conv.u8" x  : uint64 #)
             when ^a : uint16     = (# "conv.u8" x  : uint64 #)
             when ^a : char       = (# "conv.u8" x  : uint64 #)
             when ^a : unativeint = (# "conv.u8" x  : uint64 #)
             when ^a : byte     = (# "conv.u8" x  : uint64 #)

        let inline int64 (x: ^a) = 
            (^a : (static member ToInt64: ^a -> int64) (x))
             when ^a : string     = (System.Int64.Parse(castToString x,System.Globalization.CultureInfo.InvariantCulture))
             when ^a : float     = (# "conv.i8" x  : int64 #)
             when ^a : float32   = (# "conv.i8" x  : int64 #)
             when ^a : int64     = (# "conv.i8" x  : int64 #)
             when ^a : int32     = (# "conv.i8" x  : int64 #)
             when ^a : int16     = (# "conv.i8" x  : int64 #)
             when ^a : nativeint = (# "conv.i8" x  : int64 #)
             when ^a : sbyte     = (# "conv.i8" x  : int64 #)
             when ^a : uint64     = (# "conv.i8" x  : int64 #)
             when ^a : uint32     = (# "conv.i8" x  : int64 #)
             when ^a : uint16     = (# "conv.i8" x  : int64 #)
             when ^a : char       = (# "conv.i8" x  : int64 #)
             when ^a : unativeint = (# "conv.i8" x  : int64 #)
             when ^a : byte     = (# "conv.i8" x  : int64 #)

        let inline float32 (x: ^a) = 
            (^a : (static member ToSingle : ^a -> float32) (x))
             when ^a : string     = (System.Single.Parse(castToString x,System.Globalization.CultureInfo.InvariantCulture))
             when ^a : float     = (# "conv.r4" x  : float32 #)
             when ^a : float32   = (# "" x  : float32 #)
             when ^a : int64     = (# "conv.r4" x  : float32 #)
             when ^a : int32     = (# "conv.r4" x  : float32 #)
             when ^a : int16     = (# "conv.r4" x  : float32 #)
             when ^a : nativeint = (# "conv.r4" x  : float32 #)
             when ^a : sbyte     = (# "conv.r4" x  : float32 #)
             when ^a : uint64     = (# "conv.r.un conv.r4" x  : float32 #)
             when ^a : uint32     = (# "conv.r.un conv.r4" x  : float32 #)
             when ^a : uint16     = (# "conv.r.un conv.r4" x  : float32 #)
             when ^a : char       = (# "conv.r.un conv.r4" x  : float32 #)
             when ^a : unativeint = (# "conv.r.un conv.r4" x  : float32 #)
             when ^a : byte     = (# "conv.r.un conv.r4" x  : float32 #)

        let inline float (x: ^a) = 
            (^a : (static member ToDouble : ^a -> float) (x))
             when ^a : string     = (System.Double.Parse(castToString x,System.Globalization.CultureInfo.InvariantCulture))
             when ^a : float     = (# "" x  : float #)
             when ^a : float32   = (# "conv.r8" x  : float #)
             when ^a : int64     = (# "conv.r8" x  : float #)
             when ^a : int32     = (# "conv.r8" x  : float #)
             when ^a : int16     = (# "conv.r8" x  : float #)
             when ^a : nativeint = (# "conv.r8" x  : float #)
             when ^a : sbyte     = (# "conv.r8" x  : float #)
             when ^a : uint64     = (# "conv.r.un conv.r8" x  : float #)
             when ^a : uint32     = (# "conv.r.un conv.r8" x  : float #)
             when ^a : uint16     = (# "conv.r.un conv.r8" x  : float #)
             when ^a : char       = (# "conv.r.un conv.r8" x  : float #)
             when ^a : unativeint = (# "conv.r.un conv.r8" x  : float #)
             when ^a : byte       = (# "conv.r.un conv.r8" x  : float #)
             when ^a : decimal    = (System.Convert.ToDouble((# "" x : decimal #))) 

        let inline decimal (x: ^a) = 
            (^a : (static member ToDecimal : ^a -> decimal) (x))
             when ^a : string     = (System.Decimal.Parse(castToString x,System.Globalization.CultureInfo.InvariantCulture))
             when ^a : float      = (System.Convert.ToDecimal((# "" x : float #))) 
             when ^a : float32    = (System.Convert.ToDecimal((# "" x : float32 #))) 
             when ^a : int64      = (System.Convert.ToDecimal((# "" x : int64 #))) 
             when ^a : int32      = (System.Convert.ToDecimal((# "" x : int32 #))) 
             when ^a : int16      = (System.Convert.ToDecimal((# "" x : int16 #))) 
             when ^a : nativeint  = (System.Convert.ToDecimal(int64 (# "" x : nativeint #))) 
             when ^a : sbyte      = (System.Convert.ToDecimal((# "" x : sbyte #))) 
             when ^a : uint64     = (System.Convert.ToDecimal((# "" x : uint64 #))) 
             when ^a : uint32     = (System.Convert.ToDecimal((# "" x : uint32 #))) 
             when ^a : uint16     = (System.Convert.ToDecimal((# "" x : uint16 #))) 
             when ^a : unativeint = (System.Convert.ToDecimal(uint64 (# "" x : unativeint #))) 
             when ^a : byte       = (System.Convert.ToDecimal((# "" x : byte #))) 
             when ^a : decimal    = (# "" x : decimal #)

        // Recall type names.
        // Framework names:     sbyte, byte,  int16, uint16, int32, uint32, int64, uint64, single,  double.
        // C# names:            sbyte, byte,  short, ushort, int,   uint,   long,  ulong,  single,  double.
        // F# names:            sbyte, byte,  int16, uint16, int,   uint32, int64, uint64, float32, float.
        // Systematic names:    int8,  uint8, int16, uint16, int32, uint32, int64, uint64, float32, float64.

        // The names float/float32 are being migrated to double/single
        let inline single x = float32 x
        let inline double x = float x

        let inline unativeint (x: ^a) = 
            (^a : (static member ToUIntPtr: ^a -> unativeint) (x))
             when ^a : float     = (# "conv.u" x  : unativeint #)
             when ^a : float32   = (# "conv.u" x  : unativeint #)
             when ^a : int64     = (# "conv.u" x  : unativeint #)
             when ^a : int32     = (# "conv.u" x  : unativeint #)
             when ^a : int16     = (# "conv.u" x  : unativeint #)
             when ^a : nativeint = (# "conv.u" x  : unativeint #)
             when ^a : sbyte     = (# "conv.u" x  : unativeint #)
             when ^a : uint64     = (# "conv.u" x  : unativeint #)
             when ^a : uint32     = (# "conv.u" x  : unativeint #)
             when ^a : uint16     = (# "conv.u" x  : unativeint #)
             when ^a : char       = (# "conv.u" x  : unativeint #)
             when ^a : unativeint = (# "conv.u" x  : unativeint #)
             when ^a : byte     = (# "conv.u" x  : unativeint #)

        let inline nativeint (x: ^a) = 
            (^a : (static member ToIntPtr: ^a -> nativeint) (x))
             when ^a : float      = (# "conv.i" x  : nativeint #)
             when ^a : float32    = (# "conv.i" x  : nativeint #)
             when ^a : int64      = (# "conv.i" x  : nativeint #)
             when ^a : int32      = (# "conv.i" x  : nativeint #)
             when ^a : int16      = (# "conv.i" x  : nativeint #)
             when ^a : nativeint  = (# "conv.i" x  : nativeint #)
             when ^a : sbyte      = (# "conv.i" x  : nativeint #)
             when ^a : uint64     = (# "conv.i" x  : nativeint #)
             when ^a : uint32     = (# "conv.i" x  : nativeint #)
             when ^a : uint16     = (# "conv.i" x  : nativeint #)
             when ^a : char       = (# "conv.i" x  : nativeint #)
             when ^a : unativeint = (# "conv.i" x  : nativeint #)
             when ^a : byte       = (# "conv.i" x  : nativeint #)


        let inline char (x: ^a) = 
            (^a : (static member ToChar: ^a -> char) (x))
             when ^a : string     = (System.Char.Parse(castToString x))
             when ^a : float      = (# "conv.u2" x  : char #)
             when ^a : float32    = (# "conv.u2" x  : char #)
             when ^a : int64      = (# "conv.u2" x  : char #)
             when ^a : int32      = (# "conv.u2" x  : char #)
             when ^a : int16      = (# "conv.u2" x  : char #)
             when ^a : nativeint  = (# "conv.u2" x  : char #)
             when ^a : sbyte      = (# "conv.u2" x  : char #)
             when ^a : uint64     = (# "conv.u2" x  : char #)
             when ^a : uint32     = (# "conv.u2" x  : char #)
             when ^a : uint16     = (# "conv.u2" x  : char #)
             when ^a : char       = (# "conv.u2" x  : char #)
             when ^a : unativeint = (# "conv.u2" x  : char #)
             when ^a : byte       = (# "conv.u2" x  : char #)

        let inline (<) x y = GenericLessThan x y
        let inline (>) x y = GenericGreaterThan x y
        let inline (>=) x y = GenericGreaterOrEqual x y
        let inline (<=) x y = GenericLessOrEqual x y
        let inline (=) x y = GenericEquality x y
        let inline (<>) x y = GenericInequality x y
        let inline compare (x:'a) (y:'a) = GenericComparison x y
        let inline max x y = GenericMaximum x y
        let inline min x y = GenericMinimum x y

        let inline ``assert`` (b) = System.Diagnostics.Debug.Assert(b)
        
        let failwith s = raise (Failure(s))
        let invalid_arg s = raise (InvalidArgument(s))
        let invalid_op s = raise (System.InvalidOperationException(s))
        let not_found() = raise (new KeyNotFoundException())
        let inline rethrow() = unbox<_>(# "rethrow ldnull" : System.Object #)
        
        let fst (a,b) = a
        let snd (a,b) = b
        let ignore x = ()
        let ref x = { contents = x }
        let (:=) x y = x.contents <- y
        let (!) x = x.contents
        let inline (|>) x f = f x
        let inline (<|) f x = f x
        let inline (>>) f g x = g(f x)
        let inline (<<) f g x = f(g x)

        let defaultArg x y = match x with None -> y | Some v -> v
        
        let inline (.())     arr n            = array_get arr n
        let inline (.(,))    arr (n1,n2)      = array2_get arr n1 n2
        let inline (.(,,))   arr (n1,n2,n3)   = array3_get arr n1 n2 n3
        let inline (.()<-)   arr n          x = array_set arr n    x
        let inline (.(,)<-)  arr (n1,n2)    x = array2_set arr n1 n2    x
        let inline (.(,,)<-) arr (n1,n2,n3) x = array3_set arr n1 n2 n3 x

        let inline (.[]) (s: ^src) (x: 'a) : 'b                                      =    LanguagePrimitives.IntrinsicFunctions.(.[]) s x
        let inline (.[,]) (s: ^src) ((x1: 'a1),(x2:'a2)) : 'b                        =    LanguagePrimitives.IntrinsicFunctions.(.[,]) s (x1,x2)
        let inline (.[,,]) (s: ^src) ((x1: 'a1),(x2:'a2),(x3:'a3)) : 'b              =    LanguagePrimitives.IntrinsicFunctions.(.[,,]) s (x1,x2,x3)
        let inline (.[]<-) (s: ^src) (x: 'a) (y: 'b) : unit                          =    LanguagePrimitives.IntrinsicFunctions.(.[]<-) s x y
        let inline (.[,]<-) (s: ^src)  ((x1: 'a1),(x2:'a2)) (y: 'b) : unit           =    LanguagePrimitives.IntrinsicFunctions.(.[,]<-) s (x1,x2) y
        let inline (.[,,]<-) (s: ^src)  ((x1: 'a1),(x2:'a2),(x3:'a3)) (y: 'b) : unit =    LanguagePrimitives.IntrinsicFunctions.(.[,,]<-) s (x1,x2,x3) y

        let inline (.[..]) (src: ^src) (start: 'idx option) (finish: 'idx option) : ^res when ^src : (member GetSlice : 'idx option * 'idx option -> ^res)  and ^res : (member get_Item : 'idx -> 'elem) =    
           (^src : (member GetSlice : 'idx option * 'idx option -> ^res)(src,start,finish))
           when ^src : string  = 
               let str = (retype src : string) in
               let start  = (match (retype start : int option) with None -> 0 | Some n -> n) in
               let finish = (match (retype finish : int option) with None -> str.Length - 1 | Some n -> n) in
               (retype (str.Substring(start, finish-start+1))  : ^res)
        #if CLI_AT_LEAST_2_0
           when ^src : _ []    = 
               let arr = (retype src : 'elem[]) in
               let start  = (match (retype start : int option) with None -> 0 | Some n -> n) in
               let finish = (match (retype finish : int option) with None -> arr.Length - 1 | Some n -> n) in
               (retype (compatarray_sub arr start (finish - start + 1)) : ^res)
        #else
           when ^src : _ array = 
               let arr = (retype src : 'elem array) in
               let start  = (match (retype start : int option) with None -> 0 | Some n -> n) in
               let finish = (match (retype finish : int option) with None -> arr.Length - 1 | Some n -> n) in
               (retype (array_sub       arr start (finish - start + 1)) : ^res)
        #endif

        let inline (.[..,..]) (s: ^src) (x: 'idx option) (y: 'idx option) (x2: 'idx2 option) (y2: 'idx2 option) : ^res =    
           (^src : (member GetSlice2D : 'idx option * 'idx option * 'idx2 option * 'idx2 option -> ^res)(s,x,y,x2,y2))
        
        module StandardRanges = 
            let inline notStarted() = raise (new System.InvalidOperationException("Enumeration has not started. Call MoveNext."))
            let inline alreadyFinished() = raise (new System.InvalidOperationException("Enumeration already finished."))
            
            open System.Collections
            
            // Notes on "inline" with range ienumerable generation.
            // "inline" is used to ensure explicit lambda arguments can be reduced by the optimiser.
            // However, it is not used to ensure that primitive ops like add,sub etc. are direct calls.

            type mode = NotStarted | Running | Finished // Another example motivating local type
            let inline makeEnumerator (z:'z) (canStart:'z->bool) (canStep:'z->bool) (doReset:'z->unit) (current:'z->'a) =
                // Generate enumerator from mutable state "z".
                // Marked "inline" to ensure argument functions are reduced (by optimiser).
                let mode = ref NotStarted
                let getCurrent() = 
                    match !mode with
                    | NotStarted -> notStarted()
                    | Running    -> current z
                    | Finished   -> alreadyFinished()
                {   new IEnumerator<'b> with
                        member obj.Current = getCurrent()
                    interface System.Collections.IEnumerator with 
                        member obj.Current = box (getCurrent())
                        member obj.MoveNext() = 
                            match !mode with
                            | NotStarted -> if canStart z then (mode := Running; true) else (mode := Finished; false)
                            | Running    -> if canStep  z then true                    else (mode := Finished; false)
                            | Finished   -> false
                        member obj.Reset() = 
                            mode := NotStarted
                            doReset z                            
                    interface System.IDisposable with 
                        member obj.Dispose() = ()
                }

            let makeEmptyEnumerator() =
                makeEnumerator () (fun () -> false) (fun () -> false) (fun () -> ())
                                  (fun () -> failwith "this makeEmptyEnumerator getter should be dead code")

            let makeSingletonEnumerator v =
                let doReset  () = ()
                let current  () = v
                let canStart () = true
                let canStep  () = false
                makeEnumerator () canStart canStep doReset current

            let inline makeProperIntegralRangeEnumerator (<<) add (x0,dx,x1) f = 
                // NOTE: The ordering << is an argument.
                // << will be < or > depending on direction.
                // Assumes x0 << x1 and zero << dx (i.e. it's a "proper" range).
                //--------
                // NOTE: "inline" so << becomes direct operation (should be IL comparision operation)
                let z = ref x0
                let doReset  z = z := x0
                let current  z = f !z
                let canStart z = true
                let canStep  z = let x  = !z
                                 let x' = add !z dx
                                 if    x' << x             then false           // x+dx has wrapped around
                                 elif  x' = x              then false           // x+dx has not moved (unexpected, dx<>0)
                                 elif  x << x' && x' << x1 then (z := x'; true) // x+dx has moved towards end point
                                 elif  x' = x1             then (z := x'; true) // x+dx has reached end point
                                 else (* x1 << x' then *)       false           // x+dx has passed end point
                makeEnumerator z canStart canStep doReset current

            let inline integralRangeEnumerator (zero,add) x0 dx x1 f =
                // Generates sequence z_i where z_i = f (x0 + i.dx) while x0 + i.dx is in region (x0,x1)
                // NOTE: could/should drop "inline", trade indirect add against code size.
                if dx = zero then invalid_arg "step of a range cannot be zero";
                if x0 = x1 then
                    makeSingletonEnumerator (f x0)
                elif x0 < x1 then
                    if dx < zero then
                        makeEmptyEnumerator() // interval increasing, step decreasing 
                    else
                        makeProperIntegralRangeEnumerator (<) add (x0,dx,x1) f // generate proper increasing sequence
                else
                    if dx > zero then
                        makeEmptyEnumerator() // interval decreasing, step increasing
                    else                       
                        makeProperIntegralRangeEnumerator (>) add (x0,dx,x1) f // generate proper decreasing sequence
    
            let id x = x
            let inline integralRange zero (add:'a -> 'a -> 'a) n step m =
                // NOTE: could/should drop "inline", trade indirect add against code size.
                let gen() = integralRangeEnumerator (zero,add) n step m id
                { new IEnumerable<'a> with 
                      member x.GetEnumerator() = gen() 
                  interface System.Collections.IEnumerable with 
                      member x.GetEnumerator() = (gen() :> System.Collections.IEnumerator) }
                :> seq<_>

            let inline isNaN x = x <> x // NaN is the only value that does not equal itself.
            let inline makeProperFloatingRangeEnumerator (<<) add (x0,dx,x1) = 
                // NOTE: The ordering << is an argument.
                // << will be < or > depending on direction.
                // Assumes x0 << x1 and zero << dx.
                // NOTE: "inline" ensures << is comparision op
                let z = ref x0
                let doReset  z = z := x0
                let current  z = !z
                let canStart z = true
                let canStep  z = let x  = !z
                                 let x' = add !z dx
                                 // NOTE: The following code is similar to the integer case, but there are differences.
                                 // * With floats, there is a NaN case to consider.
                                 // * With floats, there should not be any wrapping around arithmetic.
                                 // * With floats, x + dx == x is possible when dx>0.                                 
                                 if   x' = x                then false              // no movement, loss of precision
                                 elif x << x' && x' << x1   then (z := x'; true)    // advanced towards end point
                                 elif x' = x1               then (z := x'; true)    // hit end point
                                 elif x1 << x'              then false              // passed beyond end point
                                                                                    // [includes x' infinite, x1 finite case]
                                 elif isNaN x'              then false              // x' has become NaN, which is possible...
                                                                                    // e.g. -infinity + infinity = NaN
                                 elif x' << x               then failwith           // x + dx should not move against <<
                                                                   "Broken invariant in F# floating point range"
                                 else                            false              // Are there any cases left? [i think not]
                makeEnumerator z canStart canStep doReset current

            let floatingRangeEnumerator (zero,add) x0 dx x1 =
                if dx = zero then invalid_arg "step of a range cannot be zero";
                if isNaN x0  then invalid_arg "start of a range cannot be NaN";
                if isNaN dx  then invalid_arg "step of a range cannot be NaN";
                if isNaN x1  then invalid_arg "end of a range cannot be NaN";
                if x0 = x1 then
                    makeSingletonEnumerator x0
                elif x0 < x1 then
                    if dx < zero then
                        makeEmptyEnumerator() // interval increasing, step decreasing 
                    else
                        makeProperFloatingRangeEnumerator (<) add (x0,dx,x1) // generate proper increasing sequence
                else
                    if dx > zero then
                        makeEmptyEnumerator() // interval decreasing, step increasing
                    else                       
                        makeProperFloatingRangeEnumerator (>) add (x0,dx,x1) // generate proper decreasing sequence

            // When is a System.Double an System.Int32?
            let minIntR = -2147483648.0
            let maxIntR =  2147483647.0
            let asInt x = if minIntR <= x && x <= maxIntR && System.Math.Floor x = x then Some (int x) else None

            // When a floating range looks like an exact number of steps, generate using {x0+i.dx} for i from an integer range.
            let optimisedFloatingRangeEnumerator (zero    : 'a,
                                                  add     : 'a -> 'a -> 'a,
                                                  sub     : 'a -> 'a -> 'a,
                                                  mul     : 'a -> 'a -> 'a,
                                                  div     : 'a -> 'a -> 'a,
                                                  ofInt   : int -> 'a,
                                                  toFloat : 'a -> float
                                                 )
                                                 x0 dx x1 =                                                 
                match asInt (toFloat (div (sub x1 x0) dx)) with
                | Some n -> 
                    integralRangeEnumerator (0,(+))    0  1  n  (fun i -> add x0 (mul (ofInt i) dx))
                | None ->
                    floatingRangeEnumerator (zero,add) x0 dx x1

            let floatingRange (zero,add,sub,mul,div,ofInt,toFloat) n step m =
                let gen() = optimisedFloatingRangeEnumerator (zero,add,sub,mul,div,ofInt,toFloat) n step m 
                { new IEnumerable<'a> with 
                      member x.GetEnumerator() = gen() 
                  interface System.Collections.IEnumerable with 
                      member x.GetEnumerator() = (gen() :> System.Collections.IEnumerator) }
                :> seq<_>

            let floatOps   = (0.0 ,(+),(-),( * ),(/),float  ,float)
            let float32Ops = (0.0f,(+),(-),( * ),(/),float32,float)
            
            let int        n step m : seq<int>        = integralRange 0    (+) n step m
            let int64      n step m : seq<int64>      = integralRange 0L   (+) n step m
            let uint64     n step m : seq<uint64>     = integralRange 0UL  (+) n step m
            let uint32     n step m : seq<uint32>     = integralRange 0ul  (+) n step m
            let nativeint  n step m : seq<nativeint>  = integralRange 0n   (+) n step m
            let unativeint n step m : seq<unativeint> = integralRange 0un  (+) n step m
            let int16      n step m : seq<int16>      = integralRange 0s   (+) n step m
            let uint16     n step m : seq<uint16>     = integralRange 0us  (+) n step m
            let sbyte      n step m : seq<sbyte>      = integralRange 0y   (+) n step m
            let byte       n step m : seq<byte>       = integralRange 0uy  (+) n step m
            let float      n step m : seq<float>      = floatingRange floatOps   n step m
            let float32    n step m : seq<float32>    = floatingRange float32Ops n step m

            let generate   zero add n step m : seq<'a> = integralRange zero add  n step m

            let char (n:char) (m:char) = integralRange (retype 0us : char) (fun (x:char) (y:char) -> retype ((retype x : uint16) + (retype y : uint16)) ) n (retype 1us) m

        let inline (..) n m = 
           (^a: (static member (..) : ^a * ^a -> seq< ^a >) (n,m))
           when ^a : int32       = StandardRanges.int        (retype n) 1    (retype m)
           when ^a : float       = StandardRanges.float      (retype n) 1.0  (retype m)
           when ^a : float32     = StandardRanges.float32    (retype n) 1.0f (retype m)
           when ^a : int64       = StandardRanges.int64      (retype n) 1L   (retype m)
           when ^a : uint64      = StandardRanges.uint64     (retype n) 1UL  (retype m)
           when ^a : uint32      = StandardRanges.uint32     (retype n) 1ul  (retype m)
           when ^a : nativeint   = StandardRanges.nativeint  (retype n) 1n   (retype m)
           when ^a : unativeint  = StandardRanges.unativeint (retype n) 1un  (retype m)
           when ^a : int16       = StandardRanges.int16      (retype n) 1s   (retype m)
           when ^a : uint16      = StandardRanges.uint16     (retype n) 1us  (retype m)
           when ^a : sbyte       = StandardRanges.sbyte      (retype n) 1y   (retype m)
           when ^a : byte        = StandardRanges.byte       (retype n) 1uy  (retype m)
           when ^a : char        = StandardRanges.char       (retype n) (retype m)

        let inline (.. ..) n step m = 
           (^a: (static member (.. ..) : ^a * ^a * ^a -> seq< ^a >) (n,step,m))
           when ^a : int32       = StandardRanges.int        (retype n) (retype step) (retype m)
           when ^a : float       = StandardRanges.float      (retype n) (retype step) (retype m)
           when ^a : float32     = StandardRanges.float32    (retype n) (retype step) (retype m)
           when ^a : int64       = StandardRanges.int64      (retype n) (retype step) (retype m)
           when ^a : uint64      = StandardRanges.uint64     (retype n) (retype step) (retype m)
           when ^a : uint32      = StandardRanges.uint32     (retype n) (retype step) (retype m)
           when ^a : nativeint   = StandardRanges.nativeint  (retype n) (retype step) (retype m)
           when ^a : unativeint  = StandardRanges.unativeint (retype n) (retype step) (retype m)
           when ^a : int16       = StandardRanges.int16      (retype n) (retype step) (retype m)
           when ^a : uint16      = StandardRanges.uint16     (retype n) (retype step) (retype m)
           when ^a : sbyte       = StandardRanges.sbyte      (retype n) (retype step) (retype m)
           when ^a : byte        = StandardRanges.byte       (retype n) (retype step) (retype m)
        
        
        #if CLI_AT_LEAST_2_0
        [<assembly: System.Runtime.CompilerServices.DefaultDependency(System.Runtime.CompilerServices.LoadHint.Always)>] 
        [<assembly: System.Security.Permissions.SecurityPermission(System.Security.Permissions.SecurityAction.RequestMinimum, Execution=true)>]
        [<assembly: System.Runtime.InteropServices.ComVisible(false)>]
        [<assembly: System.CLSCompliant(true)>]
        [<assembly: System.Security.Permissions.PermissionSet(System.Security.Permissions.SecurityAction.RequestOptional,Name="Nothing") >]
        do ()
        #else
        #endif

        let inline lock (lockobj : 'a when 'a : not struct) f  = 
            System.Threading.Monitor.Enter(lockobj);
            try f()
            finally
                System.Threading.Monitor.Exit(lockobj)


        let using (ie :> System.IDisposable) f = 
            try f(ie)
            finally match (box ie) with null -> () | _ -> ie.Dispose()

        #if CLI_AT_MOST_1_1
        let inline typeof< $a > = LanguagePrimitives.Basics.typeof< $ a >
        let inline sizeof< $a > = LanguagePrimitives.Basics.sizeof< $ a >
        #else
        let inline typeof<'a> = LanguagePrimitives.Basics.typeof<'a>
        let inline typedefof<'a> = let ty = typeof<'a> in if ty.IsGenericType then ty.GetGenericTypeDefinition() else ty
        let inline sizeof<'a> = LanguagePrimitives.Basics.sizeof<'a>
        #endif

        let inline ( $* ) x (y:^a)      : ^a = (^a : (static member ( $* ) : 'b * ^a -> ^a) (x,y))
        let inline ( *$ ) (x:^a) y      : ^a = (^a: (static member ( *$ )  : ^a * 'b -> ^a) (x,y))
        let inline ($*$ ) (x:^a) (y:^b) : 'c = (^a: (static member ( $*$ ) : ^a * ^b -> 'c) (x,y))
        let inline ( %* ) (x:^a)  y     : ^a = (^a: (static member ( %* )  : ^a * 'b -> ^a) (x,y))
        let inline (%*%)  (x:^a) (y:^b) : 'c = (^a: (static member ( %*% ) : ^a * ^b -> 'c) (x,y))
        let inline ( *% )  x (y:^a)     : ^a = (^a: (static member ( *% )  : 'b * ^a -> ^a) (x,y))
        let inline ( .* ) (x:^a) (y:'b) : ^a = (^a: (static member ( .* )  : ^a * 'b -> ^a) (x,y))
        let inline ( .^ ) (x:^a) (y:'b) : ^a = (^a: (static member ( .^ )  : ^a * 'b -> ^a) (x,y))
        let inline ( .+ ) (x:^a) (y:'b) : ^a = (^a: (static member ( .+ )  : ^a * 'b -> ^a) (x,y))
        let inline ( .% ) (x:^a) (y:'b) : ^a = (^a: (static member ( .% )  : ^a * 'b -> ^a) (x,y))
        let inline ( ./ ) (x:^a) (y:'b) : ^a = (^a: (static member ( ./ )  : ^a * 'b -> ^a) (x,y))

        let inline ( *$= ) (x:^a) y      : unit = (^a: (static member ( *$= ): ^a * 'b -> unit) (x,y))
        let inline ( %*= ) (x:^a) y      : unit = (^a: (static member ( %*= ): ^a * 'b -> unit) (x,y))
        let inline ( *%= ) x (y:^a)      : unit = (^a: (static member ( *%= ): 'b * ^a -> unit) (x,y))
        let inline ( .*= ) (x:^a) (y:'b) : unit = (^a: (static member ( .*= ): ^a * 'b -> unit) (x,y))
        let inline ( .^= ) (x:^a) (y:'b) : unit = (^a: (static member ( .^= ): ^a * 'b -> unit) (x,y))
        let inline ( .+= ) (x:^a) (y:'b) : unit = (^a: (static member ( .+= ): ^a * 'b -> unit) (x,y))
        let inline ( .%= ) (x:^a) (y:'b) : unit = (^a: (static member ( .%= ): ^a * 'b -> unit) (x,y))
        let inline ( ./= ) (x:^a) (y:'b) : unit = (^a: (static member ( ./= ): ^a * 'b -> unit) (x,y))

        let inline ( *= ) (x:^a) y : unit = (^a: (static member ( *= ): ^a * 'b -> unit) (x,y))
        let inline ( += ) (x:^a) y : unit = (^a: (static member ( += ): ^a * 'b -> unit) (x,y))
        let inline ( /= ) (x:^a) y : unit = (^a: (static member ( /= ): ^a * 'b -> unit) (x,y))
        let inline ( -= ) (x:^a) y : unit = (^a: (static member ( -= ): ^a * 'b -> unit) (x,y))

        let inline hash  (x: 'a) = LanguagePrimitives.GenericHash x
        let inline hashq (x: 'a) = LanguagePrimitives.PhysicalHash x

        let inline abs (x: ^a) : ^a = 
             (^a: (static member Abs : ^a -> ^a) (x))
             when ^a : int32       = let x : int32     = retype x in if x >= 0 then x else -x
             when ^a : float       = let x : float     = retype x in if x >= 0.0 then x else -x
             when ^a : float32     = let x : float32   = retype x in if x >= 0.0f then x else -x
             when ^a : int64       = let x : int64     = retype x in if x >= 0L then x else -x
             when ^a : nativeint   = let x : nativeint = retype x in if x >= 0n then x else -x
             when ^a : int16       = let x : int16     = retype x in if x >= 0s then x else -x
             when ^a : sbyte       = let x : sbyte     = retype x in if x >= 0y then x else -x

        let inline to_float (x:float32) = (# "conv.r8" x : float #)
        let inline of_float (x:float) = (# "conv.r4" x : float32 #)
        let inline  acos(x: ^a) : ^a = 
             (^a: (static member Acos : ^a -> ^a) (x))
             when ^a : float       = System.Math.Acos(retype x)
             when ^a : float32     = System.Math.Acos(to_float (retype x)) |> of_float


        let inline  asin(x: ^a) : ^a = 
             (^a: (static member Asin : ^a -> ^a) (x))
             when ^a : float       = System.Math.Asin(retype x)
             when ^a : float32     = System.Math.Asin(to_float (retype x)) |> of_float

        let inline  atan(x: ^a) : ^a = 
             (^a: (static member Atan : ^a -> ^a) (x))
             when ^a : float       = System.Math.Atan(retype x)
             when ^a : float32     = System.Math.Atan(to_float (retype x)) |> of_float

        let inline  atan2(x: ^a) (y: ^a) : ^a = 
             (^a: (static member Atan2 : ^a * ^a -> ^a) (x,y))
             when ^a : float       = System.Math.Atan2(retype x, retype y)
             when ^a : float32     = System.Math.Atan2(to_float (retype x), to_float(retype y)) |> of_float

        let inline  ceil(x: ^a) : ^a = 
             (^a: (static member Ceiling : ^a -> ^a) (x))
             when ^a : float       = System.Math.Ceiling(retype x : float)
             when ^a : float32     = System.Math.Ceiling(to_float (retype x)) |> of_float

        let inline  exp(x: ^a) : ^a = 
             (^a: (static member Exp : ^a -> ^a) (x))
             when ^a : float       = System.Math.Exp(retype x)
             when ^a : float32     = System.Math.Exp(to_float (retype x)) |> of_float

        let inline floor (x: ^a) : ^a = 
             (^a: (static member Floor : ^a -> ^a) (x))
             when ^a : float       = System.Math.Floor(retype x : float)
             when ^a : float32     = System.Math.Floor(to_float (retype x)) |> of_float

        let inline round (x: ^a) : ^a = 
             (^a: (static member Round : ^a -> ^a) (x))
             when ^a : float       = System.Math.Round(retype x : float)
             when ^a : float32     = System.Math.Round(to_float (retype x)) |> of_float

        let inline sign (x: ^a) : int = 
             (^a: (static member Sign : ^a -> int) (x))
             when ^a : int32       = System.Math.Sign(retype x : int32)
             when ^a : int64       = System.Math.Sign(retype x : int64)
             when ^a : nativeint   = if (retype x : nativeint) < 0n then -1 else if (retype x : nativeint) > 0n then 1 else 0
             when ^a : int16       = System.Math.Sign(retype x : int16)
             when ^a : int64       = System.Math.Sign(retype x : int64)
             when ^a : sbyte       = System.Math.Sign(retype x : sbyte)
             when ^a : float       = System.Math.Sign(retype x : float)
             when ^a : float32     = System.Math.Sign(to_float (retype x)) 
             when ^a : decimal     = System.Math.Sign(retype x : decimal) 

        let inline  log(x: ^a) : ^a = 
             (^a: (static member Log : ^a -> ^a) (x))
             when ^a : float       = System.Math.Log(retype x)
             when ^a : float32     = System.Math.Log(to_float (retype x)) |> of_float

        let inline  log10(x: ^a) : ^a = 
             (^a: (static member Log10 : ^a -> ^a) (x))
             when ^a : float       = System.Math.Log10(retype x)
             when ^a : float32     = System.Math.Log10(to_float (retype x)) |> of_float

        let inline  sqrt(x: ^a) : ^a = 
             (^a: (static member Sqrt : ^a -> ^a) (x))
             when ^a : float       = System.Math.Sqrt(retype x)
             when ^a : float32     = System.Math.Sqrt(to_float (retype x)) |> of_float

        let inline  cos(x: ^a) : ^a = 
             (^a: (static member Cos : ^a -> ^a) (x))
             when ^a : float       = System.Math.Cos(retype x)
             when ^a : float32     = System.Math.Cos(to_float (retype x)) |> of_float


        let inline  cosh(x: ^a) : ^a = 
             (^a: (static member Cosh : ^a -> ^a) (x))
             when ^a : float       = System.Math.Cosh(retype x)
             when ^a : float32     = System.Math.Cosh(to_float (retype x)) |> of_float

        let inline  sin(x: ^a) : ^a = 
             (^a: (static member Sin : ^a -> ^a) (x))
             when ^a : float       = System.Math.Sin(retype x)
             when ^a : float32     = System.Math.Sin(to_float (retype x)) |> of_float

        let inline  sinh(x: ^a) : ^a = 
             (^a: (static member Sinh : ^a -> ^a) (x))
             when ^a : float       = System.Math.Sinh(retype x)
             when ^a : float32     = System.Math.Sinh(to_float (retype x)) |> of_float

        let inline  tan(x: ^a) : ^a = 
             (^a: (static member Tan : ^a -> ^a) (x))
             when ^a : float       = System.Math.Tan(retype x)
             when ^a : float32     = System.Math.Tan(to_float (retype x)) |> of_float


        let inline  tanh(x: ^a) : ^a = 
             (^a: (static member Tanh : ^a -> ^a) (x))
             when ^a : float       = System.Math.Tanh(retype x)
             when ^a : float32     = System.Math.Tanh(to_float (retype x)) |> of_float

        let inline  ( ** ) (x: ^a) (y: ^a) : ^a = 
             (^a: (static member Pow : ^a * ^a -> ^a) (x,y))
             when ^a : float       = System.Math.Pow((retype x : float), (retype y: float))
             when ^a : float32     = System.Math.Pow(to_float (retype x), to_float(retype y)) |> of_float



        /// See OutChannelImpl
        type InChannelImpl(r : unit -> System.IO.TextReader) = 
            inherit System.IO.TextReader()

            override x.Dispose(deep:bool) = (r()).Close()
            override x.Peek() = (r()).Peek()
            override x.Read() = (r()).Read()
            override x.Read((buffer:char[]),(index:int),(count:int)) = (r()).Read(buffer,index,count)

        type OutChannelImpl(r: unit -> System.IO.TextWriter) =
            inherit System.IO.TextWriter()
            override x.Dispose(deep:bool) = (r()).Close()
            override x.Encoding = (r()).Encoding
            override x.FormatProvider = (r()).FormatProvider
            override x.NewLine = (r()).NewLine
            [<OverloadID("Write_string")>]
            override x.Write(s:string) = (r()).Write(s)
            [<OverloadID("Write_char")>]
            override x.Write(c:char) = (r()).Write(c)
            [<OverloadID("Write_char_array")>]
            override x.Write(c:char[]) = (r()).Write(c)
            override x.Write((c:char[]),(index:int),(count:int)) = (r()).Write(c,index,count)
            
        let stdin = new InChannelImpl (fun () -> System.Console.In) :> System.IO.TextReader
        let stdout =  new OutChannelImpl (fun() -> System.Console.Out) :> System.IO.TextWriter
        let stderr =  new OutChannelImpl (fun() -> System.Console.Error) :> System.IO.TextWriter


        let inline defaultof<'a> = unsafe_default<'a>


        module Unchecked = 
            let inline defaultof<'a> = unsafe_default<'a>

        module Checked = 
        
            let inline (+) (x: ^a) (y: ^b) : ^c = 
                 ((^a or ^b): (static member (+) : ^a * ^b -> ^c) (x,y))
                 when ^a : int32       and ^b : int32      = (# "add.ovf" x y : int32 #)
                 when ^a : float       and ^b : float      = (# "add" x y : float #)
                 when ^a : float32     and ^b : float32    = (# "add" x y : float32 #)
                 when ^a : int64       and ^b : int64      = (# "add.ovf" x y : int64 #)
                 when ^a : uint64      and ^b : uint64     = (# "add.ovf.un" x y : uint64 #)
                 when ^a : uint32      and ^b : uint32     = (# "add.ovf.un" x y : uint32 #)
                 when ^a : nativeint   and ^b : nativeint  = (# "add.ovf" x y : nativeint #)
                 when ^a : unativeint  and ^b : unativeint = (# "add.ovf.un" x y : unativeint #)
                 when ^a : int16       and ^b : int16      = (# "conv.ovf.i2" (# "add.ovf" x y : int32 #) : int16 #)
                 when ^a : uint16      and ^b : uint16     = (# "conv.ovf.u2" (# "add.ovf" x y : uint32 #) : uint16 #)
                 when ^a : sbyte       and ^b : sbyte      = (# "conv.ovf.i1" (# "add.ovf" x y : int32 #) : sbyte #)
                 when ^a : byte        and ^b : byte       = (# "conv.ovf.u1" (# "add.ovf" x y : uint32 #) : byte #)
                 when ^a : string      and ^b : string     = (# "" (System.String.Concat((# "" x : string #),(# "" y : string #))) : ^a #)

            let inline (-) (x: ^a) (y: ^b) : ^c = 
                 ((^a or ^b): (static member (-) : ^a * ^b -> ^c) (x,y))
                 when ^a : int32      and ^b : int32      = (# "sub.ovf" x y : int32 #)
                 when ^a : float      and ^b : float      = (# "sub" x y : float #)
                 when ^a : float32    and ^b : float32    = (# "sub" x y : float32 #)
                 when ^a : int64      and ^b : int64      = (# "sub.ovf" x y : int64 #)
                 when ^a : uint64     and ^b : uint64     = (# "sub.ovf.un" x y : uint64 #)
                 when ^a : uint32     and ^b : uint32     = (# "sub.ovf.un" x y : uint32 #)
                 when ^a : nativeint  and ^b : nativeint  = (# "sub.ovf" x y : nativeint #)
                 when ^a : unativeint and ^b : unativeint = (# "sub.ovf.un" x y : unativeint #)
                 when ^a : int16       and ^b : int16      = (# "conv.ovf.i2" (# "sub.ovf" x y : int32 #) : int16 #)
                 when ^a : uint16      and ^b : uint16     = (# "conv.ovf.u2" (# "sub.ovf" x y : uint32 #) : uint16 #)
                 when ^a : sbyte       and ^b : sbyte      = (# "conv.ovf.i1" (# "sub.ovf" x y : int32 #) : sbyte #)
                 when ^a : byte        and ^b : byte       = (# "conv.ovf.u1" (# "sub.ovf" x y : uint32 #) : byte #)


            let inline ( * ) (x: ^a) (y: ^b) : ^c = 
               ((^a or ^b): (static member ( * ) : ^a * ^b -> ^c) (x,y))
               when ^a : int32      and ^b : int32      = (# "mul.ovf" x y : int32 #)
               when ^a : float      and ^b : float      = (# "mul" x y : float #)
               when ^a : float32    and ^b : float32    = (# "mul" x y : float32 #)
               when ^a : int64      and ^b : int64      = (# "mul.ovf" x y : int64 #)
               when ^a : uint64     and ^b : uint64     = (# "mul.ovf.un" x y : uint64 #)
               when ^a : uint32     and ^b : uint32     = (# "mul.ovf.un" x y : uint32 #)
               when ^a : nativeint  and ^b : nativeint  = (# "mul.ovf" x y : nativeint #)
               when ^a : unativeint and ^b : unativeint = (# "mul.ovf.un" x y : unativeint #)
               when ^a : int16       and ^b : int16      = (# "conv.ovf.i2" (# "mul.ovf" x y : int32 #) : int16 #)
               when ^a : uint16      and ^b : uint16     = (# "conv.ovf.u2" (# "mul.ovf" x y : uint32 #) : uint16 #)
               when ^a : sbyte       and ^b : sbyte      = (# "conv.ovf.i1" (# "mul.ovf" x y : int32 #) : sbyte #)
               when ^a : byte        and ^b : byte       = (# "conv.ovf.u1" (# "mul.ovf" x y : uint32 #) : byte #)


            let inline byte (x: ^a) = 
                (^a : (static member ToByte: ^a -> byte) (x))
                 when ^a : float     = (# "conv.ovf.u1" x  : byte #)
                 when ^a : float32   = (# "conv.ovf.u1" x  : byte #)
                 when ^a : int64     = (# "conv.ovf.u1" x  : byte #)
                 when ^a : int32     = (# "conv.ovf.u1" x  : byte #)
                 when ^a : int16     = (# "conv.ovf.u1" x  : byte #)
                 when ^a : nativeint = (# "conv.ovf.u1" x  : byte #)
                 when ^a : sbyte     = (# "conv.ovf.u1" x  : byte #)
                 when ^a : uint64     = (# "conv.ovf.u1.un" x  : byte #)
                 when ^a : uint32     = (# "conv.ovf.u1.un" x  : byte #)
                 when ^a : uint16     = (# "conv.ovf.u1.un" x  : byte #)
                 when ^a : char       = (# "conv.ovf.u1.un" x  : byte #)
                 when ^a : unativeint = (# "conv.ovf.u1.un" x  : byte #)
                 when ^a : byte     = (# "conv.ovf.u1.un" x  : byte #)

            let inline sbyte (x: ^a) = 
                (^a : (static member ToSByte: ^a -> sbyte) (x))
                 when ^a : float     = (# "conv.ovf.i1" x  : sbyte #)
                 when ^a : float32   = (# "conv.ovf.i1" x  : sbyte #)
                 when ^a : int64     = (# "conv.ovf.i1" x  : sbyte #)
                 when ^a : int32     = (# "conv.ovf.i1" x  : sbyte #)
                 when ^a : int16     = (# "conv.ovf.i1" x  : sbyte #)
                 when ^a : nativeint = (# "conv.ovf.i1" x  : sbyte #)
                 when ^a : sbyte     = (# "conv.ovf.i1" x  : sbyte #)
                 when ^a : uint64     = (# "conv.ovf.i1.un" x  : sbyte #)
                 when ^a : uint32     = (# "conv.ovf.i1.un" x  : sbyte #)
                 when ^a : uint16     = (# "conv.ovf.i1.un" x  : sbyte #)
                 when ^a : char       = (# "conv.ovf.i1.un" x  : sbyte #)
                 when ^a : unativeint = (# "conv.ovf.i1.un" x  : sbyte #)
                 when ^a : byte     = (# "conv.ovf.i1.un" x  : sbyte #)

            let inline uint16 (x: ^a) = 
                (^a : (static member ToUInt16: ^a -> uint16) (x))
                 when ^a : float      = (# "conv.ovf.u2" x  : uint16 #)
                 when ^a : float32    = (# "conv.ovf.u2" x  : uint16 #)
                 when ^a : int64      = (# "conv.ovf.u2" x  : uint16 #)
                 when ^a : int32      = (# "conv.ovf.u2" x  : uint16 #)
                 when ^a : int16      = (# "conv.ovf.u2" x  : uint16 #)
                 when ^a : nativeint  = (# "conv.ovf.u2" x  : uint16 #)
                 when ^a : sbyte      = (# "conv.ovf.u2" x  : uint16 #)
                 when ^a : uint64     = (# "conv.ovf.u2.un" x  : uint16 #)
                 when ^a : uint32     = (# "conv.ovf.u2.un" x  : uint16 #)
                 when ^a : uint16     = (# "conv.ovf.u2.un" x  : uint16 #)
                 when ^a : char       = (# "conv.ovf.u2.un" x  : uint16 #)
                 when ^a : unativeint = (# "conv.ovf.u2.un" x  : uint16 #)
                 when ^a : byte       = (# "conv.ovf.u2.un" x  : uint16 #)

            let inline char (x: ^a) = 
                (^a : (static member ToChar: ^a -> char) (x))
                 when ^a : string     = (System.Char.Parse(castToString x))
                 when ^a : float      = (# "conv.ovf.u2" x  : char #)
                 when ^a : float32    = (# "conv.ovf.u2" x  : char #)
                 when ^a : int64      = (# "conv.ovf.u2" x  : char #)
                 when ^a : int32      = (# "conv.ovf.u2" x  : char #)
                 when ^a : int16      = (# "conv.ovf.u2" x  : char #)
                 when ^a : nativeint  = (# "conv.ovf.u2" x  : char #)
                 when ^a : sbyte      = (# "conv.ovf.u2" x  : char #)
                 when ^a : uint64     = (# "conv.ovf.u2.un" x  : char #)
                 when ^a : uint32     = (# "conv.ovf.u2.un" x  : char #)
                 when ^a : uint16     = (# "conv.ovf.u2.un" x  : char #)
                 when ^a : char       = (# "conv.ovf.u2.un" x  : char #)
                 when ^a : unativeint = (# "conv.ovf.u2.un" x  : char #)
                 when ^a : byte       = (# "conv.ovf.u2.un" x  : char #)


            let inline int16 (x: ^a) = 
                (^a : (static member ToInt16: ^a -> int16) (x))
                 when ^a : float     = (# "conv.ovf.i2" x  : int16 #)
                 when ^a : float32   = (# "conv.ovf.i2" x  : int16 #)
                 when ^a : int64     = (# "conv.ovf.i2" x  : int16 #)
                 when ^a : int32     = (# "conv.ovf.i2" x  : int16 #)
                 when ^a : int16     = (# "conv.ovf.i2" x  : int16 #)
                 when ^a : nativeint = (# "conv.ovf.i2" x  : int16 #)
                 when ^a : sbyte     = (# "conv.ovf.i2" x  : int16 #)
                 when ^a : uint64     = (# "conv.ovf.i2.un" x  : int16 #)
                 when ^a : uint32     = (# "conv.ovf.i2.un" x  : int16 #)
                 when ^a : uint16     = (# "conv.ovf.i2.un" x  : int16 #)
                 when ^a : char       = (# "conv.ovf.i2.un" x  : int16 #)
                 when ^a : unativeint = (# "conv.ovf.i2.un" x  : int16 #)
                 when ^a : byte     = (# "conv.ovf.i2.un" x  : int16 #)

            let inline uint32 (x: ^a) = 
                (^a : (static member ToUInt32: ^a -> uint32) (x))
                 when ^a : float     = (# "conv.ovf.u4" x  : uint32 #)
                 when ^a : float32   = (# "conv.ovf.u4" x  : uint32 #)
                 when ^a : int64     = (# "conv.ovf.u4" x  : uint32 #)
                 when ^a : int32     = (# "conv.ovf.u4" x  : uint32 #)
                 when ^a : int16     = (# "conv.ovf.u4" x  : uint32 #)
                 when ^a : nativeint = (# "conv.ovf.u4" x  : uint32 #)
                 when ^a : sbyte     = (# "conv.ovf.u4" x  : uint32 #)
                 when ^a : uint64     = (# "conv.ovf.u4.un" x  : uint32 #)
                 when ^a : uint32     = (# "conv.ovf.u4.un" x  : uint32 #)
                 when ^a : uint16     = (# "conv.ovf.u4.un" x  : uint32 #)
                 when ^a : char       = (# "conv.ovf.u4.un" x  : uint32 #)
                 when ^a : unativeint = (# "conv.ovf.u4.un" x  : uint32 #)
                 when ^a : byte     = (# "conv.ovf.u4.un" x  : uint32 #)

            let inline int32 (x: ^a) = 
                (^a : (static member ToInt32: ^a -> int32) (x))
                 when ^a : float     = (# "conv.ovf.i4" x  : int32 #)
                 when ^a : float32   = (# "conv.ovf.i4" x  : int32 #)
                 when ^a : int64     = (# "conv.ovf.i4" x  : int32 #)
                 when ^a : int32     = (# "conv.ovf.i4" x  : int32 #)
                 when ^a : int16     = (# "conv.ovf.i4" x  : int32 #)
                 when ^a : nativeint = (# "conv.ovf.i4" x  : int32 #)
                 when ^a : sbyte     = (# "conv.ovf.i4" x  : int32 #)
                 when ^a : uint64     = (# "conv.ovf.i4.un" x  : int32 #)
                 when ^a : uint32     = (# "conv.ovf.i4.un" x  : int32 #)
                 when ^a : uint16     = (# "conv.ovf.i4.un" x  : int32 #)
                 when ^a : char       = (# "conv.ovf.i4.un" x  : int32 #)
                 when ^a : unativeint = (# "conv.ovf.i4.un" x  : int32 #)
                 when ^a : byte     = (# "conv.ovf.i4.un" x  : int32 #)

            let inline int x = int32 x
            let inline uint64 (x: ^a) = 
                (^a : (static member ToUInt64: ^a -> uint64) (x))
                 when ^a : float     = (# "conv.ovf.u8" x  : uint64 #)
                 when ^a : float32   = (# "conv.ovf.u8" x  : uint64 #)
                 when ^a : int64     = (# "conv.ovf.u8" x  : uint64 #)
                 when ^a : int32     = (# "conv.ovf.u8" x  : uint64 #)
                 when ^a : int16     = (# "conv.ovf.u8" x  : uint64 #)
                 when ^a : nativeint = (# "conv.ovf.u8" x  : uint64 #)
                 when ^a : sbyte     = (# "conv.ovf.u8" x  : uint64 #)
                 when ^a : uint64     = (# "conv.ovf.u8.un" x  : uint64 #)
                 when ^a : uint32     = (# "conv.ovf.u8.un" x  : uint64 #)
                 when ^a : uint16     = (# "conv.ovf.u8.un" x  : uint64 #)
                 when ^a : char       = (# "conv.ovf.u8.un" x  : uint64 #)
                 when ^a : unativeint = (# "conv.ovf.u8.un" x  : uint64 #)
                 when ^a : byte     = (# "conv.ovf.u8.un" x  : uint64 #)

            let inline int64 (x: ^a) = 
                (^a : (static member ToInt64: ^a -> int64) (x))
                 when ^a : float     = (# "conv.ovf.i8" x  : int64 #)
                 when ^a : float32   = (# "conv.ovf.i8" x  : int64 #)
                 when ^a : int64     = (# "conv.ovf.i8" x  : int64 #)
                 when ^a : int32     = (# "conv.ovf.i8" x  : int64 #)
                 when ^a : int16     = (# "conv.ovf.i8" x  : int64 #)
                 when ^a : nativeint = (# "conv.ovf.i8" x  : int64 #)
                 when ^a : sbyte     = (# "conv.ovf.i8" x  : int64 #)
                 when ^a : uint64     = (# "conv.ovf.i8.un" x  : int64 #)
                 when ^a : uint32     = (# "conv.ovf.i8.un" x  : int64 #)
                 when ^a : uint16     = (# "conv.ovf.i8.un" x  : int64 #)
                 when ^a : char       = (# "conv.ovf.i8.un" x  : int64 #)
                 when ^a : unativeint = (# "conv.ovf.i8.un" x  : int64 #)
                 when ^a : byte     = (# "conv.ovf.i8.un" x  : int64 #)

            let inline unativeint (x: ^a) = 
                (^a : (static member ToUIntPtr: ^a -> unativeint) (x))
                 when ^a : float     = (# "conv.ovf.u" x  : unativeint #)
                 when ^a : float32   = (# "conv.ovf.u" x  : unativeint #)
                 when ^a : int64     = (# "conv.ovf.u" x  : unativeint #)
                 when ^a : int32     = (# "conv.ovf.u" x  : unativeint #)
                 when ^a : int16     = (# "conv.ovf.u" x  : unativeint #)
                 when ^a : nativeint = (# "conv.ovf.u" x  : unativeint #)
                 when ^a : sbyte     = (# "conv.ovf.u" x  : unativeint #)
                 when ^a : uint64     = (# "conv.ovf.u.un" x  : unativeint #)
                 when ^a : uint32     = (# "conv.ovf.u.un" x  : unativeint #)
                 when ^a : uint16     = (# "conv.ovf.u.un" x  : unativeint #)
                 when ^a : char       = (# "conv.ovf.u.un" x  : unativeint #)
                 when ^a : unativeint = (# "conv.ovf.u.un" x  : unativeint #)
                 when ^a : byte     = (# "conv.ovf.u.un" x  : unativeint #)

            let inline nativeint (x: ^a) = 
                (^a : (static member ToIntPtr: ^a -> nativeint) (x))
                 when ^a : float     = (# "conv.ovf.i" x  : nativeint #)
                 when ^a : float32   = (# "conv.ovf.i" x  : nativeint #)
                 when ^a : int64     = (# "conv.ovf.i" x  : nativeint #)
                 when ^a : int32     = (# "conv.ovf.i" x  : nativeint #)
                 when ^a : int16     = (# "conv.ovf.i" x  : nativeint #)
                 when ^a : nativeint = (# "conv.ovf.i" x  : nativeint #)
                 when ^a : sbyte     = (# "conv.ovf.i" x  : nativeint #)
                 when ^a : uint64     = (# "conv.ovf.i.un" x  : nativeint #)
                 when ^a : uint32     = (# "conv.ovf.i.un" x  : nativeint #)
                 when ^a : uint16     = (# "conv.ovf.i.un" x  : nativeint #)
                 when ^a : char       = (# "conv.ovf.i.un" x  : nativeint #)
                 when ^a : unativeint = (# "conv.ovf.i.un" x  : nativeint #)
                 when ^a : byte     = (# "conv.ovf.i.un" x  : nativeint #)

        

//============================================================================
//============================================================================
namespace Microsoft.FSharp.Collections

    //-------------------------------------------------------------------------
    // Lists
    //-------------------------------------------------------------------------

#if CLI_AT_LEAST_2_0
    open System.Collections.Generic
#else
    open Microsoft.FSharp.Compatibility
#endif
    open Microsoft.FSharp.Core
    open Microsoft.FSharp.Core.Operators
    open Microsoft.FSharp.Core.LanguagePrimitives.IntrinsicOperators
    open Microsoft.FSharp.Core.LanguagePrimitives.Basics
    [<DefaultAugmentation(false)
#if CLI_AT_LEAST_2_0
     ;System.Diagnostics.DebuggerTypeProxy(typedefof<ListDebugView<_>>)
#endif
    >]

    type 'a List = 
       | ([])  :                               'a list
       | (::)  : _Head: 'a * _Tail: 'a list -> 'a list
#if CLI_AT_MOST_1_1
       interface Microsoft.FSharp.Compatibility.IEnumerable<'a>
#else
       interface System.Collections.Generic.IEnumerable<'a>
#endif
       interface System.Collections.IEnumerable
        
     and 'a list = 'a List

    //-------------------------------------------------------------------------
    // List (debug view)
    //-------------------------------------------------------------------------

#if CLI_AT_LEAST_2_0
    and ListDebugView<'a>(l:list<'a>) =
       [<System.Diagnostics.DebuggerBrowsable(System.Diagnostics.DebuggerBrowsableState.RootHidden)>]
       member x.Items =
           let rec count l n = match l with [] -> n | _::t -> if n > 100 then n else count t (n+1) 
           let n = count l 0 
           let items = zero_create n 
           let rec copy items l i = 
             match l with
             | [] -> () 
             | h::t -> if i < n then (items.[i] <- h; copy items t (i+1)) 
           copy items l 0;
           items
#endif
    
      
    //-------------------------------------------------------------------------
    // List (augmentation)
    //-------------------------------------------------------------------------

    module PrivateListHelpers = 

        let notStarted() = raise (new System.InvalidOperationException("Enumeration has not started. Call MoveNext."))
        let alreadyFinished() = raise (new System.InvalidOperationException("Enumeration already finished."))
        let nonempty x = match x with [] -> false | _ -> true

        type ListEnumerator<'a> (s: 'a list) = 
              let mutable curr = s 
              let mutable started = false 

              member x.GetCurrent() = 
                  if started then 
                      match curr with 
                      | [] -> alreadyFinished()
                      | h::t -> h
                  else 
                      notStarted()

              interface IEnumerator<'a> with 
                  member x.Current = x.GetCurrent()

              interface System.Collections.IEnumerator with 
                  member x.MoveNext() = 
                       if started then 
                         match curr with 
                         | h :: t -> (curr <- t; nonempty curr)
                         | _ -> false
                       else 
                         (started <- true; nonempty curr)
                  member x.Current = box (x.GetCurrent())
                  member x.Reset() = started <- false; curr <- s

              interface System.IDisposable with 
                  member x.Dispose() = () 

        let mkListEnumerator s = (new ListEnumerator<'a>(s) :> IEnumerator<'a>)

        let length x = 
            let rec lacc acc l = match l with [] -> acc | h::t -> lacc (acc+1) t 
            lacc 0 x

        let rec iter f x = match x with [] -> () | (h::t) -> f h; iter f t

        // optimized mutation-based implementation. This code is only valid in fslib, where mutation of private
        // tail cons cells is permitted in carefully written library code.
        let map f x = 
          match x with
          | [] -> []
          | (h::t) -> 
              let res = [f h] 
              let mutable curr = t 
              let mutable cell = res 
              while nonempty curr do
                  let nw = [f curr.(::).0] 
                  cell.(::).1 <- nw;
                  curr <- curr.(::).1;
                  cell <- nw
              res

        let mapi f x = 
          let f = OptimizedClosures.FastFunc2<_,_,_>.Adapt(f)
          match x with
          | [] -> []
          | (h::t) -> 
              let res = [f.Invoke(0,h)] 
              let mutable idx = 1 
              let mutable curr = t 
              let mutable cell = res 
              while nonempty curr do
                  let nw = [f.Invoke(idx,curr.(::).0)] 
                  cell.(::).1 <- nw;
                  curr <- curr.(::).1;
                  idx <- idx + 1;
                  cell <- nw
              res

        let rec for_all f l1 = 
            match l1 with 
            | [] -> true
            | (h1::t1) -> f h1 && for_all f t1

        let rec exists f l1 = 
            match l1 with 
            | [] -> false
            | (h1::t1) -> f h1 || exists f t1

        // optimized mutation-based implementation. This code is only valid in fslib, where mutation of private
        // tail cons cells is permitted in carefully written library code.
        let rev l = 
             let mutable res = [] 
             let mutable curr = l 
             while nonempty curr do
                 res <- (curr.(::).0) :: res; 
                 curr <- curr.(::).1;
             res

        // optimized mutation-based implementation. This code is only valid in fslib, where mutation of private
        // tail cons cells is permitted in carefully written library code.
        let set_fresh_cons_tail cons t = 
            cons.(::).1 <- t

        // optimized mutation-based implementation. This code is only valid in fslib, where mutation of private
        // tail cons cells is permitted in carefully written library code.
        let rec filter_to_fresh_cons_tail cons f l = 
            match l with 
            | [] -> ()
            | h::t -> 
                if f h then 
                    let cons2 = [h] 
                    set_fresh_cons_tail cons cons2;
                    filter_to_fresh_cons_tail cons2 f t
                else 
                    filter_to_fresh_cons_tail cons f t
          
        let rec filter f l = 
            match l with 
            | [] -> []
            | [h] -> if f h then l else []
            | h::t -> if f h then (let cons = [h]in filter_to_fresh_cons_tail cons f t; cons) else filter f t

        let iteri f x = 
            let f = OptimizedClosures.FastFunc2<_,_,_>.Adapt(f)
            let rec loop n x = match x with [] -> () | (h::t) -> f.Invoke(n,h); loop (n+1) t
            loop 0 x

    
    type 'a List with
        member l.Length = PrivateListHelpers.length l
        
        member l.Head   = match l with a :: b -> a | [] -> not_found()
        member l.Tail   = match l with a :: b -> b | [] -> not_found()
        member l.IsCons = match l with [] -> false | _ -> true
        member l.IsNil  = match l with [] -> true | _ -> false
        member l.IsEmpty  = match l with [] -> true | _ -> false
        static member Nil       : 'a list = []
        static member Empty     : 'a list = []
        static member Cons(h,t) : 'a list = h::t

        member l.Map(f)            = PrivateListHelpers.map f l
        member l.MapIndexed(f)     = PrivateListHelpers.mapi f l        
        member l.Iterate(f)        = PrivateListHelpers.iter f l
        member l.IterateIndexed(f) = PrivateListHelpers.iteri f l
        member l.ForAll(p)         = PrivateListHelpers.for_all p l        
        member l.Filter(p)         = PrivateListHelpers.filter p l        
        member l.Exists(p)         = PrivateListHelpers.exists p l        
        member l.Reverse()         = PrivateListHelpers.rev l
               
        interface IEnumerable<'a> with
            member l.GetEnumerator() = PrivateListHelpers.mkListEnumerator l

        interface System.Collections.IEnumerable with
            member l.GetEnumerator() = (PrivateListHelpers.mkListEnumerator l :> System.Collections.IEnumerator)

//============================================================================
//============================================================================

namespace Microsoft.FSharp.Control

    //-------------------------------------------------------------------------
    // Lazy
    //-------------------------------------------------------------------------

    open Microsoft.FSharp.Core
    open Microsoft.FSharp.Core
    open Microsoft.FSharp.Core.Operators
    
    module LazyStatus = 
    
        //-------------------------------------------------------------------------
        // Undefined

         [<DefaultAugmentation(false)>]
         type LazyStatus<'a> =
         | Delayed of (unit -> 'a)
         | Value of 'a
         | Exception of System.Exception

    open LazyStatus

    exception Undefined
                    
    [<DefaultAugmentation(false)>]
    type Lazy<'a> = { mutable status : LazyStatus<'a> }
    and 'a ``lazy`` = Lazy<'a>

    //-------------------------------------------------------------------------
    // Lazy (augmentation)

    type Lazy<'a> with
        static member Create(f) : Lazy<'a> = {status=(LazyStatus.Delayed f)}
        member x.IsForced = match x.status with LazyStatus.Value _ -> true | _ -> false
        member x.IsDelayed = match x.status with LazyStatus.Delayed _ -> true | _ -> false
        member x.IsException = match x.status with LazyStatus.Exception _ -> true | _ -> false
        member x.UnsynchronizedForce() = 
            match x.status with 
            | LazyStatus.Delayed f -> 
                x.status <- Exception Undefined; 
                begin 
                  try 
                      let res = f () 
                      x.status <- LazyStatus.Value res
                      res 
                  with e -> 
                      x.status <- LazyStatus.Exception(e)
                      rethrow()
                end
            | LazyStatus.Value x -> x
            | LazyStatus.Exception e -> raise e
        member x.Value = x.SynchronizedForce()
        member x.Force() = x.SynchronizedForce()
        member x.SynchronizedForce() = 
            match x.status with 
            | LazyStatus.Value x -> x
            | _ -> 
                  System.Threading.Monitor.Enter(x);
                  try 
                      x.UnsynchronizedForce()
                  finally
                    System.Threading.Monitor.Exit(x)


//============================================================================
//============================================================================


namespace Microsoft.FSharp.Control

    open Microsoft.FSharp.Core

    type IEvent<'a> = 
        abstract Add: ('a -> unit) -> unit

    type IPrimitiveDelegateEvent<'del> (* when 'del :> System.MulticastDelegate  *) =
        abstract AddHandler: 'del -> unit
        abstract RemoveHandler: 'del -> unit 

    type IDelegateEvent<'del,'args> (* when 'del :> System.MulticastDelegate  *) =
        inherit IEvent<'args>
        inherit IPrimitiveDelegateEvent<'del>

    type Handler<'a> =  delegate of obj * 'a -> unit 

    type IHandlerEvent<'a> = IDelegateEvent<Handler<'a>, 'a>
    type ICompatEvent<'a> = IHandlerEvent<'a>

