(*F# 
/// Language-independent byte arrays
module Microsoft.Research.AbstractIL.Internal.Bytes 
open Microsoft.Research.AbstractIL 
open Microsoft.Research.AbstractIL.Internal 
F#*)


let (???) = Int32.to_int 
let (!!!) = Int32.of_int 
let (&&&) = Int32.logand 
let (lsr) = Int32.shift_right_logical
let b0 n = ??? (n &&& 0xFFl)
let b1 n = ??? ((n lsr 8) &&& 0xFFl)
let b2 n = ??? ((n lsr 16) &&& 0xFFl)
let b3 n = ??? ((n lsr 24) &&& 0xFFl)

(*F# 

type bytes = byte[] 

let length b = Bytearray.length b
let get ( b) n = Byte.to_int (Bytearray.get b n)  
let make f n = Bytearray.init n (fun i -> Byte.of_int (f i))  
let zero_create n = Bytearray.zero_create n      

let really_input is n =
    let buff = Bytearray.create n in 
     really_input is buff 0 n;
     buff

let maybe_input is n =
   let buff = Bytearray.create n  in 
   let x = input is buff 0 n in 
   Bytearray.sub buff 0 x

let output os b =
  Pervasives.output os b 0 (Bytearray.length b) 
  
let sub ( b) s l = Bytearray.sub b s l   
let set bb n b = Bytearray.set bb n (Byte.of_int b) 
let blit a b c d e = Bytearray.blit a b c d e 
let string_as_unicode_bytes (s:string) = System.Text.Encoding.Unicode.GetBytes s 
let utf8_bytes_as_string (b:bytes) = System.Text.Encoding.UTF8.GetString b 
let unicode_bytes_as_string (b:bytes) = System.Text.Encoding.Unicode.GetString b 
let compare (b1:bytes) (b2:bytes) = Pervasives.compare b1 b2
F#*)

(*IF-OCAML*) 

type bytes = string

let length b =  String.length b     
let get b n = Char.code (String.get b n)      
let make f n = 
   let s = String.create n in 
   for i = 0 to n - 1 do String.set s i (Char.chr (f(i))) done;
   s                   

let zero_create n = 
  let s = String.create n in 
  for i = 0 to n - 1 do String.set s i '\000' done;
  s                                             

let really_input is n =
    let buff =  String.create  n in 
    really_input is buff 0 n;
     buff

let maybe_input is n =
    let buff =  String.create n    in 
     let x = input is buff 0 n in 
     String.sub buff 0 x

let output os ( b) = Pervasives.output os b 0  (String.length b)  
let sub ( b) s l = String.sub b s l   
let set bb n b = String.set bb n (Char.chr b)       
let blit a b c d e = String.blit a b c d e 

let string_as_unicode_bytes (s:string) = 
  let len = String.length s in
  make (fun n -> if n mod 2 = 1 then 0x0 else (Char.code s.[n/2]))  (len * 2)

let utf8_bytes_as_string (b:bytes) =  (b:string)

let unicode_bytes_as_string (b:bytes) = 
  (* This is a hacky approximation that is only right when the string only contains ascii *)
  let ascii = make (fun i -> get b (i*2)) (length b / 2) in 
  ascii

let compare (b1:bytes) (b2:bytes) = Pervasives.compare b1 b2

(*ENDIF-OCAML*)

let to_intarray (b:bytes) =  Array.init (length b) (get b)
let of_intarray arr = make (fun i -> arr.(i)) (Array.length arr)

let string_as_utf8_bytes (s:string) = 
 (*F#  System.Text.Encoding.UTF8.GetBytes s  F#*)
 (*IF-OCAML*) s (*ENDIF-OCAML*)

let append (b1: bytes) (b2:bytes) = 
  (*F# Bytearray.append b1 b2 F#*) 
  (*IF-OCAML*) b1 ^ b2 (*ENDIF-OCAML*)

let string_as_utf8_bytes_null_terminated (s:string) = 
  append (string_as_utf8_bytes s) (of_intarray [| 0x0 |]) 

let string_as_unicode_bytes_null_terminated (s:string) = 
  append (string_as_unicode_bytes s) (of_intarray [| 0x0 |]) 


module Bytestream = struct
    type t = { bytes: bytes; mutable pos: int; max: int }

    let of_bytes b n len = 
      if n < 0 or (n+len) > length b then failwith "Bytestream.of_bytes";
      { bytes = b; pos = n; max = n+len }

    let read_byte b  = 
      if b.pos >= b.max then failwith "Bytestream.of_bytes.read_byte: end of stream";
      let res = get b.bytes b.pos in
      b.pos <- b.pos + 1;
      res 
      
    let read_bytes b n  = 
      if b.pos + n > b.max then failwith "Bytestream.read_bytes: end of stream";
      let res = sub b.bytes b.pos n in
      b.pos <- b.pos + n;
      res 

    let read_unicode_bytes_as_string (b:t) n = 
      (*F# let res = System.Text.Encoding.Unicode.GetString(b.bytes,b.pos,n) in  b.pos <- b.pos + n; res   F#*)
      (*IF-OCAML*) unicode_bytes_as_string (read_bytes b n) (*ENDIF-OCAML*)

    let read_utf8_bytes_as_string (b:t) n = 
      (*F# let res = System.Text.Encoding.UTF8.GetString(b.bytes,b.pos,n) in  b.pos <- b.pos + n; res F#*)
      (*IF-OCAML*) utf8_bytes_as_string (read_bytes b n) (*ENDIF-OCAML*)

end  

module Bytebuf = struct
    type t = 
        { mutable bbArray: bytes; 
          mutable bbCurrent: int }

    let create sz = 
      { bbArray=zero_create sz; 
        bbCurrent = 0; }
        
    let ensure_bytebuf buf new_size = 
      let old_buf_size = length buf.bbArray in 
      if new_size > old_buf_size then begin
        let old = buf.bbArray in 
        buf.bbArray <- zero_create (max new_size (old_buf_size * 2));
        blit old 0 buf.bbArray 0 buf.bbCurrent;
      end

    let close buf = sub buf.bbArray 0 buf.bbCurrent

    let emit_int_as_byte buf i = 
      let new_size = buf.bbCurrent + 1 in 
      ensure_bytebuf buf new_size;
      set buf.bbArray buf.bbCurrent i;
      buf.bbCurrent <- new_size 

    let emit_bytes buf i = 
      let n = length i in 
      let new_size = buf.bbCurrent + n in 
      ensure_bytebuf buf new_size;
      blit i 0 buf.bbArray buf.bbCurrent n;
      buf.bbCurrent <- new_size 
    let emit_i32_as_u16 buf n = 
      let new_size = buf.bbCurrent + 2 in 
      ensure_bytebuf buf new_size;
      set buf.bbArray buf.bbCurrent (b0 n);
      set buf.bbArray (buf.bbCurrent + 1) (b1 n);
      buf.bbCurrent <- new_size 
    
    let emit_i32 buf n = 
      let new_size = buf.bbCurrent + 4 in 
      ensure_bytebuf buf new_size;
      set buf.bbArray buf.bbCurrent (b0 n);
      set buf.bbArray (buf.bbCurrent + 1) (b1 n);
      set buf.bbArray (buf.bbCurrent + 2) (b2 n);
      set buf.bbArray (buf.bbCurrent + 3) (b3 n);
      buf.bbCurrent <- new_size 


    let emit_intarray_as_bytes buf arr = 
      let n = Array.length arr in 
      let new_size = buf.bbCurrent + n in 
      ensure_bytebuf buf new_size;
      let bbarr = buf.bbArray in
      let bbbase = buf.bbCurrent in
      for i= 0 to n - 1 do set bbarr (bbbase + i) arr.(i) done;
      buf.bbCurrent <- new_size 

    let length bb = bb.bbCurrent

end


