//-----------------------------------------------------------------------------
// A sample script for use with fsi.exe or for compiling with fsc.exe 
// that shows the use of the Microsoft.FSharp.Reflection API
// for walking F# terms and analyzing them statistically.
//
// Copyright (c) Microsoft Corporation 2005-2006.
// This sample code is provided "as is" without warranty of any kind. 
// We disclaim all warranties, either express or implied, including the 
// warranties of merchantability and fitness for a particular purpose. 
//-----------------------------------------------------------------------------

#light

open Compatibility
open Printf
open System
open System.Reflection
open System.Collections.Generic
open System.Runtime.InteropServices
open Microsoft.FSharp.Reflection
open Microsoft.FSharp.Collections

//-----------------------------------------------------------------------------
// Statistics
//-----------------------------------------------------------------------------

type IMeasure<'a,'b> = 
    abstract Sample : 'a -> unit 
    abstract GetStatistic : unit -> 'b
  
let mkAverager(toFloat) = 
    let count = ref 0
    let total = ref 0.0 
    { new IMeasure<_,float>
      with Sample(x) = incr count; total := !total + toFloat x
      and  GetStatistic() = (!total / float(!count)) }
    
let mkTotalizer(zero,add) = 
    let total = ref zero 
    { new IMeasure<_,_>
      with Sample(x) = total := add !total x
      and  GetStatistic() = !total }

let mkMaximizer(start,compare) = 
    let highest = ref start 
    { new IMeasure<_,_>
      with Sample(x) = if compare !highest x < 0 then highest := x
      and  GetStatistic() = !highest }

//-----------------------------------------------------------------------------
// Collect the object graph.  Treat properties called "Details" as special:
// asssume they project a representation of the contents of the type.
//-----------------------------------------------------------------------------

type ObjGraph = 
   { objOccs: HashMultiMap<obj,(int ref)>;
     absObjs: HashSet<obj >;
     mutable count: int }

let freshObjGraph () = 
   { objOccs=HashMultiMap.Create(HashIdentity.Reference,100);
     absObjs=HashSet.Create(HashIdentity.Reference,100);
     count=0 }


let getProperty (obj:obj) name =
  let ty = obj.GetType() 
  ty.InvokeMember(name, Enum.combine [ BindingFlags.GetProperty; 
                                       BindingFlags.Instance; 
                                       BindingFlags.Public; 
                                       BindingFlags.NonPublic ], null, obj, 
                                           CompatArray.of_array [| |])


let rec walkObj (objGraph: ObjGraph) (x:obj) = 
    match x with 
    | null -> 
        walkValInfo objGraph (Value.GetInfo(x))
    | _ -> 
        if objGraph.objOccs.Contains(x) then  
            incr objGraph.objOccs.[x]
        else 
            objGraph.objOccs.Add(x,ref 1);
            objGraph.count <- objGraph.count + 1;
            walkValInfo objGraph  (Value.GetInfo(x))

and walkValInfo objGraph repr = 
    match repr with 
    | TupleValue(vals) -> List.iter (walkObj objGraph) vals; 
    | RecordValue(recd) 
    | ConstructorValue(_,_,recd) 
    | ExceptionValue(_,recd) -> List.iter (snd >> walkObj objGraph) recd
    | FunctionClosureValue(_,obj) -> objGraph.absObjs.Add(obj)
    | ObjectValue(obj) ->
         match (try Some(getProperty obj "Details") with _ -> None) with
         | Some(obj2) -> walkObj objGraph obj2
         | None -> objGraph.absObjs.Add(obj)
    | UnitValue -> ()


//-----------------------------------------------------------------------------
// Collect the object graph and analyze it for some simple statistics
//-----------------------------------------------------------------------------

let walkValue obj = 
    let objGraph = (freshObjGraph()) 
    walkObj objGraph (box obj);
    printf "#objs = %d\n" objGraph.count;
    //objGraph.objOccs.Iterate (fun o count -> printf "%d had %d references\n" (hashq o) !count);

    // Analyze the graph for total objects of various types and the average number of 
    // times each object of each type is referenced
    begin 
        let table = HashMultiMap.Create(HashIdentity.Reference,100) 
        objGraph.objOccs.Iterate(fun o count -> 
            let ty = o.GetType() 
            if not (table.Contains(ty)) then 
                table.Add(ty,(mkAverager(Int32.to_float), mkTotalizer(0,(+))));
            let avS,totS = table.[ty] 
            avS.Sample(!count);
            totS.Sample(1));

        // Format the results 
        let nameFieldTitle = "Name" 
        let maxNameLengthS = mkMaximizer(nameFieldTitle.Length,compare) 
        table.Iterate(fun ty (avS,totS) -> maxNameLengthS.Sample((ty.ToString()).Length));
        let maxNameLength = maxNameLengthS.GetStatistic() 
        let nameFieldWidth = maxNameLength + 2 
        let pad (n:string) = String.make (max 0 (nameFieldWidth-n.Length)) ' ' 
        printf "%s%s   Total  %%Total  AvRefsTo\n" nameFieldTitle (pad nameFieldTitle);
        table.Iterate(fun ty (avS,totS) -> 
            let n = ty.ToString()
            let tot = (totS.GetStatistic()) 
            let ptot = (float(tot)/float(objGraph.count))*100.0 
            let av = (avS.GetStatistic()) 
            printf "%s%s%8d%8.2g%9.2g\n" n (pad n) tot ptot av) ;
    end;
    begin 
        let t = HashMultiMap.Create(HashIdentity.Reference,100) 
        objGraph.absObjs.Iterate(fun o -> 
            let ty = o.GetType() 
            if not (t.Contains(ty)) then 
                t.Add(ty,mkTotalizer(0,(+)));
            (t.[ty]).Sample(1));
        t.Iterate(fun ty sampler -> 
            printf "A total of %d instances of type %s were abstract\n" 
                (sampler.GetStatistic()) 
                (ty.ToString()) );
    end;
    objGraph.absObjs.Iterate (fun o -> printf "%d of type %s was abstract\n" (hashq o) ((o.GetType()).ToString()));
    ()
          
  
  
//-----------------------------------------------------------------------------
// Some samples
//-----------------------------------------------------------------------------


// This one reveals if strings are interned or not
walkValue ["a";"b";"b"]

// This one shows if identical lists are from different expressions  are
// shared or not (they aren't)
walkValue (["a";"b";"b"],["a";"b";"b"])

// The walker detects the sharing in the following:
walkValue (let l = ["a";"b";"b"] in l,l)

// This one shows what happens when you hit an abstract type
walkValue (Math.BigNum.of_string "10227232")

