// Copyright (c) Microsoft Corporation 2005-2007.
// This sample code is provided "as is" without warranty of any kind. 
// We disclaim all warranties, either express or implied, including the 
// warranties of merchantability and fitness for a particular purpose. 
//

#light

open System
open System.Collections
open System.Collections.Generic
open System.Linq
open System.Xml.Linq
open Sample.Support
open System.Xml
open System.Text.RegularExpressions
open System.IO
open Microsoft.FSharp.Linq.SequenceOps
open Microsoft.FSharp.Xml.Linq.SequenceOps

let dataPath = Path.Combine(Environment.GetFolderPath(Environment.SpecialFolder.ProgramFiles), @"LINQ Preview\Data\")

[<Category("Load");
  Title("Load document from file");
  Description("Load an XML document from a file")>]
let XLinq1() =
    let doc = XDocument.Load(dataPath + "bib.xml") 
    Console.WriteLine(doc)

[<Category("Load");
  Title("Load document from string");
  Description("Load document from string")>]
let XLinq2() =
    let xml = "<book price='100' isbn='1002310'>
                <title>XClarity Samples</title>
                <author>Matt</author>
               </book>" 
    let doc = XDocument.Parse(xml) 
    Console.WriteLine(doc)
  
[<Category("Load");
  Title("Load document from XmlReader");
  Description("Load an XML document from XmlReader")>]
let XLinq3() =
    let reader = XmlReader.Create(dataPath + "bib.xml") 
    let doc = XDocument.Load(reader) 
    Console.WriteLine(doc)


(*
[<Category("Load");
  Title("Element from XmlReader - 1");
  Description("Construct XElement from XmlReader positioned on an element")>]
let XLinq4() =
    let reader = XmlReader.Create(dataPath + "nw_customers.xml") 
    ignore(reader.Read());//move to root
    ignore(reader.Read()); // move to fist customer
    let c = XElement(reader) 
    Console.WriteLine(c)

[<Category("Load");
  Title("Element from XmlReader - 2");
  Description("Read XElement content from XmlReader not positioned on an Element")>]
let XLinq5() =
    let reader = XmlReader.Create(dataPath + "config.xml") 
    ///the file has comments and whitespace at the start
    ignore(reader.Read());
    ignore(reader.Read());
    let config = XElement(xname "appSettings",xargs ["This content will be replaced"]) 
    config.ReadContentFrom(reader);
    Console.WriteLine(config)
*)


[<Category("Construction");
  Title("Construct an XElement from string");
  Description("Construct an XElement from string")>]
let XLinq6() =
    let xml = "<purchaseOrder price='100'>
                   <item price='50'>Motor</item>
                   <item price='50'>Cable</item>
               </purchaseOrder>"
    let po = XElement.Parse(xml) 
    Console.WriteLine(po)
  

[<Category("Construction");
  Title("Computed element name");
  Description("Computed element name")>]
let XLinq8() =
    let customers = XDocument.Load(dataPath + "nw_customers.xml") 
    let name = 
        customers.Elements(xname"Root")
        |> elements (xname("Customers"))
        |> hd 
        |> attribute("CustomerID") 
        |> attribute_to_string 
    let result = XElement(xname name, xargs ["Element with a computed name"] ) 
    Console.WriteLine(result)
  
  (*
  
[<Category("Construction");
  Title("Document creation");
  Description("Create a simple config file")>]
let XLinq9() =

            XDocument myDocument =
            new XDocument(
              XElement("configuration",
                XElement("system.web",
                  XElement("memebership",
                    XElement("providers",
                      XElement("add",
                        new XAttribute("name",
                                       "WebAdminMembershipProvider"),
                        new XAttribute("type",
                                       "System.Web.Administration.WebAdminMembershipProvider")))),
                  XElement("httpModules",
                    XElement("add",
                      new XAttribute("name",
                                      "WebAdminModule"),
                      new XAttribute("type",
                                      "System.Web.Administration.WebAdminModule"))),
                  XElement("authentication",
                    new XAttribute("mode", "Windows")),
                  XElement("authorization",
                    XElement("deny",
                      new XAttribute("users", "?"))),
                  XElement("identity",
                    new XAttribute("impersonate", "true")),
                  XElement("trust",
                    new XAttribute("level", "full")),
                  XElement("pages",
                    new XAttribute("validationRequest", "true")))));

            Console.WriteLine(myDocument);

        }

        // <xsd:schema xmlns:xsd="http://www.w3.org/2001/XMLSchema" 
        //             xmlns:sql="urn:schemas-microsoft-com:mapping-schema">

        // <xsd:element name="root" sql:is-constant="1">
        // <xsd:complexType>
        // <xsd:sequence>
        // <xsd:element name="Customers" minOccurs="100" maxOccurs="unbounded">
        //     <xsd:complexType>
        //     <xsd:sequence>
        //         <xsd:element name="CompanyName" type="xsd:string" /> 
        //         <xsd:element name="ContactName" type="xsd:string" /> 
        //         <xsd:element name="ContactTitle" type="xsd:string" /> 
        //         <xsd:element name="Phone" type="xsd:string" /> 
        //         <xsd:element name="Fax" type="xsd:string"/> 
        //         <xsd:element ref="FullAddress" maxOccurs="3"/>
        //         <xsd:element name="Date" type="xsd:date"/>
        //     </xsd:sequence>
        //     <xsd:attribute name="CustomerID" type="xsd:integer" /> 
        //     </xsd:complexType>
        // </xsd:element>
        // </xsd:sequence>
        // </xsd:complexType>
        // </xsd:element>
        // <xsd:element name="FullAddress" sql:relation="Customers" sql:relationship="CustAdd" sql:key-fields="CustomerID" >
        //     <xsd:complexType>
        //     <xsd:sequence>
        //         <xsd:element name="Address" type="xsd:string" /> 
        //         <xsd:element name="City" type="xsd:string" /> 
        //         <xsd:element name="Region" type="xsd:string" /> 
        //         <xsd:element name="PostalCode" type="xsd:string" /> 
        //         <xsd:element name="Country" type="xsd:string" /> 
        //     </xsd:sequence>
        //     </xsd:complexType>
        // </xsd:element>       
        //</xsd:schema>

[<Category("Construction");
  Title("Create an XmlSchema");
  Description("Create an XmlSchema")>]
let XLinq10() =

            string XSD = "{http://www.w3.org/2001/XMLSchema}";
            string XMLNS = "{http://www.w3.org/2000/xmlns/}";
            string SQL = "{urn:schemas-microsoft-com:mapping-schema}";
            XElement result =
            XElement(XSD + "schema",
              XElement(XSD + "element",
                new XAttribute(XMLNS + "xsd", "http://www.w3.org/2001/XMLSchema"),
                new XAttribute(XMLNS + "sql", "urn:schemas-microsoft-com:mapping-schema"),
                new XAttribute(SQL + "is-constant", "1"),
                new XAttribute("name", "root"),
                XElement(XSD + "complexType",
                  XElement(XSD + "sequence",
                    XElement(XSD + "element",
                      new XAttribute("name", "Customers"),
                      new XAttribute("minOccurs", "100"),
                      new XAttribute("maxOccurs", "unbounded"),
                      XElement(XSD + "complexType",
                        XElement(XSD + "sequence",
                          XElement(XSD + "element",
                            new XAttribute("name", "CompanyName"),
                            new XAttribute("type", "xsd:string")),
                          XElement(XSD + "element",
                            new XAttribute("name", "ContactName"),
                            new XAttribute("type", "xsd:string")),
                          XElement(XSD + "element",
                            new XAttribute("name", "ContactTitle"),
                            new XAttribute("type", "xsd:string")),
                          XElement(XSD + "element",
                            new XAttribute("name", "Phone"),
                            new XAttribute("type", "xsd:string")),
                          XElement(XSD + "element",
                            new XAttribute("name", "Fax"),
                            new XAttribute("type", "xsd:string")),
                          XElement(XSD + "element",
                            new XAttribute("ref", "FullAddress"),
                            new XAttribute("maxOccurs", "3")),
                          XElement(XSD + "element",
                            new XAttribute("name", "Date"),
                            new XAttribute("type", "xsd:date")))))))),
              XElement(XSD + "element",
                new XAttribute(XSD + "name", "FullAddress"),
                new XAttribute(SQL + "relation", "Customers"),
                new XAttribute(SQL + "relationship", "CustAdd"),
                new XAttribute(SQL + "key-fields", "CustomerID"),
                XElement(XSD + "complexType",
                  XElement(XSD + "sequence",
                    XElement(XSD + "element",
                      new XAttribute("name", "Address"),
                      new XAttribute("type", "xsd:string")),
                    XElement(XSD + "element",
                      new XAttribute("name", "City"),
                      new XAttribute("type", "xsd:string")),
                    XElement(XSD + "element",
                      new XAttribute("name", "Region"),
                      new XAttribute("type", "xsd:string")),
                    XElement(XSD + "element",
                      new XAttribute("name", "PostalCode"),
                      new XAttribute("type", "xsd:string")),
                    XElement(XSD + "element",
                      new XAttribute("name", "Country"),
                      new XAttribute("type", "xsd:string"))))));
            Console.WriteLine(result);

        }

[<Category("Construction");
  Title("Construct a PI");
  Description("Create an XML document with an XSLT PI")>]
let XLinq11() =

            XDocument result = new XDocument(
                                new XProcessingInstruction("xml-stylesheet",
                                                           "type='text/xsl' href='diff.xsl'"),
                                XElement("foo"));
            Console.WriteLine(result);
        }

[<Category("Construction");
  Title("XML comment construction");
  Description("XML comment construction")>]
let XLinq12() =

            XDocument result =
              new XDocument(
                new XComment("My phone book"),
                XElement("phoneBook",
                  new XComment("My friends"),
                  XElement("Contact",
                    new XAttribute("name", "Ralph"),
                    XElement("home phone", "425-234-4567"),
                    XElement("cell phone", "206-345-75656")),
                  XElement("Contact",
                    new XAttribute("name", "Dave"),
                    XElement("home phone", "516-756-9454"),
                    XElement("cell phone", "516-762-1546")),
                  new XComment("My family"),
                  XElement("Contact",
                    new XAttribute("name", "Julia"),
                    XElement("home phone", "425-578-1053"),
                    XElement("cell phone", "")),
                  new XComment("My team"),
                  XElement("Contact",
                    new XAttribute("name", "Robert"),
                    XElement("home phone", "345-565-1653"),
                    XElement("cell phone", "321-456-2567"))));

            Console.WriteLine(result);
        }

[<Category("Construction");
  Title("Create a CData section");
  Description("Create a CData section")>]
let XLinq13() =

            XElement e = XElement("Dump",
                          new XCData("<dump>this is some xml</dump>"));
            Console.WriteLine(e);
        }

[<Category("Construction");
  Title("Create a sequence of nodes");
  Description("Create a sequence of customer elements")>]
let XLinq14() =

            var cSequence =new[] {
                    XElement("cutomer",
                                 new XAttribute("id","x"),"new customer"),
                    XElement("cutomer",
                                 new XAttribute("id","y"),"new customer"),
                    XElement("cutomer",
                                 new XAttribute("id","z"),"new customer")};
            foreach (var c cSequence)
                Console.WriteLine(c);

        }

[<Category("Write");
  Title("Write an XElement to XmlWriter");
  Description("Write an XElement to XmlWriter open the WriteTo method")>]
let XLinq15() =

            XElement po1 = XElement("PurchaseOrder",
                            XElement("Item", "Motor",
                              new XAttribute("price", "100")));

            XElement po2 = XElement("PurchaseOrder",
                            XElement("Item", "Cable",
                              new XAttribute("price", "10")));

            XElement po3 = XElement("PurchaseOrder",
                            XElement("Item", "Switch",
                              new XAttribute("price", "10")));

            StringWriter sw = new StringWriter();
            XmlWriterSettings settings = new XmlWriterSettings();
            settings.Indent = true;
            XmlWriter w = XmlWriter.Create(sw, settings);
            w.WriteStartElement("PurchaseOrders");

            po1.WriteTo(w);
            po2.WriteTo(w);
            po3.WriteTo(w);

            w.WriteEndElement();
            w.Close();
            Console.WriteLine(sw.ToString());
        }


[<Category("Write");
  Title("Write the content of an XDocument to XmlWriter");
  Description("Write the content of XDocument to XmlWriter open the WriteTo method")>]
let XLinq16() =

            XDocument doc1 = new XDocument(
              XElement("PurchaseOrders",
                XElement("PurchaseOrder",
                  XElement("Item", "Motor",
                    new XAttribute("price", "100"))),
                XElement("PurchaseOrder",
                  XElement("Item", "Cable",
                    new XAttribute("price", "10")))));
            XDocument doc2 = new XDocument(
              XElement("PurchaseOrders",
                XElement("PurchaseOrder",
                  XElement("Item", "Switch",
                    new XAttribute("price", "10")))));

            StringWriter sw = new StringWriter();
            XmlWriter w = XmlWriter.Create(sw);
            w.WriteStartDocument();
            w.WriteStartElement("AllPurchaseOrders");
            doc1.WriteTo(w);
            doc2.WriteTo(w);
            w.WriteEndElement();
            w.WriteEndDocument();
            w.Close();
            Console.WriteLine(sw.ToString());
        }


[<Category("Write");
  Title("Save XDocument");
  Description("Save XDocument open XmlWriter/TextWriter/File")>]
let XLinq17() =

            XDocument doc = new XDocument(
                XElement("PurchaseOrders",
                    XElement("PurchaseOrder",
                      XElement("Item",
                        "Motor",
                        new XAttribute("price", "100"))),
                    XElement("PurchaseOrder",
                      XElement("Item",
                        "Switch",
                        new XAttribute("price", "10"))),
                    XElement("PurchaseOrder",
                      XElement("Item",
                        "Cable",
                        new XAttribute("price", "10")))));

            StringWriter sw = new StringWriter();
            //save to XmlWriter
            XmlWriterSettings settings = new XmlWriterSettings();
            settings.Indent = true;
            XmlWriter w = XmlWriter.Create(sw, settings);
            doc.Save(w);
            w.Close();
            Console.WriteLine(sw.ToString());

            //save to file
            doc.Save("out.xml");

        }

[<Category("Query");
  Title("Query for child elements");
  Description("Select all the customers the xml document")>]
let XLinq18() =

            XDocument doc = XDocument.Load(dataPath + "nw_customers.xml");
            foreach (XElement result doc.Elements("Root")
                                           .Elements("Customers"))
                Console.WriteLine(result);

        }

[<Category("Query");
  Title("Query for all child elements");
  Description("Select all the child elements of the first customer")>]
let XLinq19() =

            XDocument doc = XDocument.Load(dataPath + "nw_customers.xml");
            var query = doc.Element("Root")
                           .Element("Customers")
                           .Elements();
            foreach (XElement result query)
                Console.WriteLine(result);
        }

[<Category("Query");
  Title("Query for first child element - 1");
  Description("Select the first customer the document")>]
let XLinq20() =

            XDocument doc = XDocument.Load(dataPath + "nw_customers.xml");
            var result = doc.Element("Root")
                            .Element("Customers");
            Console.WriteLine(result);
        }

[<Category("Query");
  Title("Query for first child element - 2");
  Description("Query for one child element on a sequence of elements")>]
let XLinq21() =

            XDocument doc = XDocument.Load(dataPath + "nw_customers.xml");
            var result = doc.Elements()
                            .Elements("Customers")
                            .First()
                            .Element("CompanyName");
            Console.WriteLine(result);
        }

[<Category("Query");
  Title("Query for attributes");
  Description("Selects all the CustomerIDs the xml document")>]
let XLinq22() =

            XDocument doc = XDocument.Load(dataPath + "nw_customers.xml");
            var query = doc.Element("Root")
                           .Elements("Customers")
                           .Attributes("CustomerID");
            foreach (XAttribute result query)
                Console.WriteLine(result.Name + " = " + result.Value);

        }

[<Category("Query");
  Title("Cast an attribute to a number");
  Description("Find orders with price > 100")>]
let XLinq23() =

            string xml = "<order >" +
                            "<item price='150'>Motor</item>" +
                            "<item price='50'>Cable</item>" +
                            "<item price='50'>Modem</item>" +
                            "<item price='250'>Monitor</item>" +
                            "<item price='10'>Mouse</item>" +
                        "</order>";
            XElement order = XElement.Parse(xml);
            var query = 
                from 
                    i order.Elements("item")
                where 
                    (int)i.Attribute("price") > 100
                select i;
            foreach(var result query)			
            Console.WriteLine("Expensive Item {0} costs {1}",
                              (string)result, 
                              (string)result.Attribute("price"));
        }

[<Category("Query");
  Title("Get the root element of a document");
  Description("Get the root element of a document")>]
let XLinq24() =

            XElement root = XDocument.Load(dataPath + "config.xml")
                                     .Root;
            Console.WriteLine("Name of root element is {0}", root.Name);
        }

[<Category("Query");
  Title("Filter query results open where");
  Description("Filter query results open where")>]
let XLinq25() =

            XDocument doc = XDocument.Load(dataPath + "nw_customers.xml");
            var query = 
                    from
                        c doc.Element("Root")
                                .Elements("Customers")
                    where
                        c.Element("FullAddress")
                         .Element("Country")
                         .Value == "Germany"
                    select c;
                        
            foreach (XElement result query)
                Console.WriteLine(result);    			
        }

[<Category("Query");
  Title("Select all descendants of an element");
  Description("Select all ContactName elements the document")>]
let XLinq26() =

            XDocument doc = XDocument.Load(dataPath + "nw_customers.xml");
            var query = doc.Descendants("ContactName");
            foreach (XElement result query)
                Console.WriteLine(result);
        }

[<Category("Query");
  Title("Select all descendants of a given type");
  Description("Select all text the document")>]
let XLinq27() =

            XDocument doc = XDocument.Load(dataPath + "nw_customers.xml");
            var query = doc.Descendants<string>();
            foreach (string result query)
                Console.WriteLine(result);
        }

[<Category("Query");
  Title("Select all ancestors");
  Description("Check if two nodes belong to the same document")>]
let XLinq28() =

            XDocument doc = XDocument.Load(dataPath + "nw_customers.xml");
            XElement element1 = doc.Element("Root"); 
            XElement element2 = doc.Descendants("Customers")
                                .ElementAt(3);
            var query = from 
                            a element1.SelfAndAncestors(), 
                            b element2.SelfAndAncestors() 
                        where 
                            a == b
                        select a;
            Console.WriteLine(query.Any());    			
        }

[<Category("Query");
  Title("Query for parent");
  Description("Query for parent of an Element")>]
let XLinq29() =

            XElement item = XElement("item-01",
                                         "Computer");
            XElement order = XElement("order", item);
            XElement p = item.Parent;
            Console.WriteLine(p.Name);
        }

[<Category("Query");
  Title("Join over two sequences");
  Description("Add customer company info to orders of the first customer")>]
let XLinq30() =

            XDocument customers = XDocument.Load(dataPath + "nw_customers.xml");
            XDocument orders = XDocument.Load(dataPath + "nw_orders.xml");
            
            var query = 
                from 
                    customer customers.Descendants("Customers").Take(1), 
                    order orders.Descendants("Orders")
                where 
                    (string)customer.Attribute("CusomterID") == (string)customer.Element("CustomerID")
                select 
                    XElement("Order",
                                order.Content(),
                                customer.Element("CompanyName"));
            
            foreach(var result query)
                Console.WriteLine(result);
            
        }

[<Category("Query");
  Title("Query content of a type");
  Description("Query content of a given type of an existing element")>]
let XLinq31() =

            XElement elem =
              XElement("customer",
                           XElement("name",
                                        "jack"),
                           "some text",
                           new XComment("new customer"),
                           new XAttribute("id",
                                          "abc"));

            //string content               
            foreach (string s elem.Content<string>())
                Console.WriteLine("String content: {0}", s);

            //element content
            foreach (XElement e elem.Content<XElement>())
                Console.WriteLine("Element content: {0}", e);

            //comment content
            foreach (XComment c elem.Content<XComment>())
                Console.WriteLine("Comment content: {0}", c);
        }

[<Category("Query");
  Title("Query for CData");
  Description("Query for CData content")>]
let XLinq32() =

            XElement e =
            XElement("Dump",
                            "some text",
                            new XCData("<dump>this is some xml</dump>"),
                            "some more text",
                            new XCData("<dump>this is some more xml</dump>"));

            var cdata = e.Content<XCData>();

            foreach (var c cdata)
                Console.WriteLine(c);

        }

[<Category("Query");
  Title("Positional predicate");
  Description("Query the 3rd customer the document")>]
let XLinq33() =

            XDocument doc = XDocument.Load(dataPath + "nw_customers.xml");
            var c = doc.Descendants("Customers")
                       .ElementAt(3);
            Console.WriteLine(c);
        }

[<Category("Query");
  Title("Union two sequences of nodes");
  Description("Union books authored by two authors: Anders and Peter")>]
let XLinq34() =

            XDocument doc = XDocument.Load(dataPath + "bib.xml");
            var b1 = doc.Descendants("book")
                        .Where(b => b.Elements("author")
                                     .Elements("first")
                                     .Any(f => (string)f == "Anders"));
            var b2 = doc.Descendants("book")
                        .Where(b => b.Elements("author")
                                     .Elements("first")
                                     .Any(f => (string)f == "Peter"));
            var books = b1.Union(b2);                         
            foreach(var b books)
                Console.WriteLine(b);    			
        }

[<Category("Query");
  Title("Intersect two sequences of nodes");
  Description("Intersect books that are common for both authors")>]
let XLinq35() =

            XDocument doc = XDocument.Load(dataPath + "bib.xml");
            var b1 = doc.Descendants("book")
                        .Where(b => b.Elements("author")
                                        .Elements("first")
                                        .Any(f => (string)f == "Anders"));
            var b2 = doc.Descendants("book")
                        .Where(b => b.Elements("author")
                                        .Elements("first")
                                        .Any(f => (string)f == "Peter"));
            var books = b1.Intersect(b2);                         
            foreach(var b books)
                Console.WriteLine(b);    			
        }

[<Category("Query");
  Title("All nodes sequence 1 except the nodes sequence 2");
  Description("Find books that are authored by Peter and did not have Anders as co-author")>]
let XLinq36() =

            XDocument doc = XDocument.Load(dataPath + "bib.xml");			    
            var b1 = doc.Descendants("book")
                        .Where(b => b.Elements("author")
                                     .Elements("first")
                                     .Any(f => (string)f == "Anders"));
            var b2 = doc.Descendants("book")
                        .Where(b => b.Elements("author")
                                     .Elements("first")
                                     .Any(f => (string)f == "Peter"));
            var books = b2.Except(b1);                         
            foreach(var b books)
                Console.WriteLine(b);    			
        }

[<Category("Query");
  Title("Reverse the order of nodes a sequence");
  Description("Display the path to a node")>]
        [LinkedMethod("PrintPath");
let XLinq37() =

            XDocument doc = XDocument.Load(dataPath + "bib.xml");
            XElement e = doc.Descendants("last")
                            .First();
            PrintPath(e);
        }

        static void PrintPath(XElement e)

            var nodes = e.SelfAndAncestors()
                            .Reverse();
            foreach (var n nodes)
                Console.Write(n.Name + (n == e ? "" : "->"));
        }


[<Category("Query");
  Title("Equality of sequences");
  Description("Check if 2 sequences of nodes are equal. " +
                     "Did anders and peter co-author all of their the books?")>]
let XLinq38() =
  
            XDocument doc = XDocument.Load(dataPath + "bib.xml");			    
            var b1 = doc.Descendants("book")
                        .Where(b => b.Elements("author")
                                        .Elements("first")
                                        .Any(f => (string)f == "Anders"));
            var b2 = doc.Descendants("book")
                        .Where(b => b.Elements("author")
                                        .Elements("first")
                                        .Any(f => (string)f == "Peter"));
            bool result = b2.EqualAll(b1);                         
            Console.WriteLine(result);    						    
        }


[<Category("Query");
  Title("TakeWhile operator");
  Description("List books until total price is less that $150")>]
let XLinq39() =

            XDocument doc = XDocument.Load(dataPath + "bib.xml");
            double sum = 0;
            var query = doc.Descendants("book")
                           .TakeWhile(c => (sum += (double)c.Element("price")) <= 150);
            foreach(var result query)
                Console.WriteLine(result);    		
        }

[<Category("Query");
  Title("Create a list of numbers");
  Description("Create 5 new customers with different IDs")>]
let XLinq40() =

            var query = from 
                            i Sequence.Range(1,5)
                        select 
                            XElement("Customer", 
                                         new XAttribute("id",i),
                                         "New customer");
            foreach(var result query)
            Console.WriteLine(result);    		
        }

[<Category("Query");
  Title("Repeat operator");
  Description("Initialize new orders with items")>]
let XLinq41() =

            var orders = XElement[] {
                XElement("order", new XAttribute("itemCount",5)),
                XElement("order", new XAttribute("itemCount",2)),
                XElement("order", new XAttribute("itemCount",3))};

            //add empty items
            foreach (var order orders)
                order.Add(Sequence.Repeat(XElement("item", "New item"),
                                          (int)order.Attribute("itemCount")));

            foreach (var o orders)
                Console.WriteLine(o);

        }

[<Category("Query");
  Title("Any operator");
  Description("Check if there are any customers Argentina")>]
let XLinq42() =

            XDocument doc = XDocument.Load(dataPath + "nw_Customers.xml");
            if(doc.Descendants("Country").Any(c => (string)c == "Argentina"))
                Console.WriteLine("There are cusotmers Argentina");
            else
                Console.WriteLine("There are no cusotmers Argentina");    		
        }

[<Category("Query");
  Title("All operator");
  Description("Check if all books have at least one author")>]
let XLinq43() =

            XDocument doc = XDocument.Load(dataPath + "bib.xml");
            bool query = doc.Descendants("book")
                            .All(b => b.Descendants("author").Count()>0);
            if(query)
                Console.WriteLine("All books have authors");
            else
                Console.WriteLine("Some books dont have authors");
        }

[<Category("Query");
  Title("Count operartor");
  Description("Find the number of orders for a customer")>]
let XLinq44() =

            XDocument doc = XDocument.Load(dataPath + "nw_Orders.xml");
            var query = doc.Descendants("Orders")
                           .Where(o => (string) o.Element("CustomerID") == "VINET");
            Console.WriteLine("Customer has {0} orders",query.Count());    		
        }

[<Category("Query");
  Title("Fold operator");
  Description("Find tax on an order")>]
        [LinkedMethod("Tax");
let XLinq45() =

            string xml = "<order >" +
                            "<item price='150'>Motor</item>" +
                            "<item price='50'>Cable</item>" +
                         "</order>";

            XElement order = XElement.Parse(xml);
            double tax = order.Elements("item")
                              .Fold((double)0, Tax);

            Console.WriteLine("The total tax on the order @10% is ${0}", tax);

        }
        static double Tax(double seed, XElement item)

            return seed + (double)item.Attribute("price") * .01;
        }

[<Category("Query");
  Title("Distinct operator");
  Description("Find all the countries where there is a customer")>]
let XLinq46() =

            XDocument doc = XDocument.Load(dataPath + "nw_Customers.xml");
            var countries = doc.Descendants("Country")
                               .Distinct()
                               .OrderBy(c=> (string)c);
            foreach(var c countries)
                Console.WriteLine(c);    		
        }

[<Category("Query");
  Title("Concat operator");
  Description("List all books by Anders and Peter with co-authored books repeated")>]
let XLinq47() =

            XDocument doc = XDocument.Load(dataPath + "bib.xml");
            var b1 = doc.Descendants("book")
                        .Where(b => b.Elements("author")
                                     .Elements("first")
                                     .Any(f => (string)f == "Anders"));
            var b2 = doc.Descendants("book")
                        .Where(b => b.Elements("author")
                                     .Elements("first")
                                     .Any(f => (string)f == "Peter"));
            var books = b1.Concat(b2);                         
            foreach(var b books)
                Console.WriteLine(b);    			
            
        }
[<Category("Query");
  Title("Take operator");
  Description("Query the first two customers")>]
let XLinq48() =

            XDocument doc = XDocument.Load(dataPath + "nw_Customers.xml");
            var customers = doc.Descendants("Customers").Take(2);
            foreach (var c customers)
                Console.WriteLine(c);
        }

[<Category("Query");
  Title("Skip operator");
  Description("Skip the first 3 books")>]
let XLinq49() =

            XDocument doc = XDocument.Load(dataPath + "bib.xml");
            var books = doc.Descendants("book").Skip(3);
            foreach (var b books)
                Console.WriteLine(b);
        }

[<Category("Query");
  Title("Skip nodes based on a condition");
  Description("Print items that dont fit budget")>]
let XLinq50() =

            string xml = "<order >"+
                            "<item price='150'>Motor</item>"+
                            "<item price='50'>Cable</item>"+
                            "<item price='50'>Modem</item>"+
                            "<item price='250'>Monitor</item>"+
                            "<item price='10'>Mouse</item>"+
                        "</order>" ;
            XElement order = XElement.Parse(xml);			
            int sum=0;
            var items = order.Descendants("item")
                             .SkipWhile(i=> (sum+=(int)i.Attribute("price")) < 300);
            foreach(var i items)
                Console.WriteLine("{0} does not fit you budget",(string)i);    		
        }

[<Category("Query");
  Title("SelectMany operator");
  Description("Get all books authored by Anders and Peter")>]
        [LinkedMethod("GetBooks");
let XLinq51() =

            string[] authors = {"Anders","Peter"};
            var books = authors.SelectMany(a=> GetBooks(a));
            foreach(var b books)
                Console.WriteLine(b);
            
        }
        static IEnumerable<XElement> GetBooks(string author)

            XDocument doc = XDocument.Load(dataPath + "bib.xml");
            var query = doc.Descendants("book")
                            .Where(b => b.Elements("author")
                                         .Elements("first")
                                         .Any(f => (string)f == author));
                            
            return query;
        }

[<Category("Query");
  Title("Container document");
  Description("Find the container document of an element")>]
let XLinq52() =

            XElement c = XDocument.Load(dataPath + "bib.xml").Descendants("book").First();
            XDocument container = c.Document;
            Console.WriteLine(container);
        }

[<Category("Grouping");
  Title("Group orders by customer");
  Description("Group orders by customer")>]
let XLinq53() =

            XDocument doc = XDocument.Load(dataPath + "nw_orders.xml");  
            var query = 
                from 
                    o doc.Descendants("Orders")
                group o by 
                    (string)o.Element("CustomerID") into oGroup
                select
                    oGroup.Group;
            foreach(var result query.SelectMany(g=>g))
                Console.WriteLine(result);
        }

[<Category("Grouping");
  Title("Group customers by country and city");
  Description("Create a directory of customers grouped by country and city")>]
let XLinq54() =

            XDocument customers = XDocument.Load(dataPath + "nw_customers.xml");
            XElement directory = 
             XElement("directory",
                from 
                    customer customers.Descendants("Customers"),
                    country customer.Elements("FullAddress").Elements("Country")
                group customer by 
                    country into countryGroup
                select 
                    XElement("Country",
                                 new XAttribute("name",
                                                countryGroup.Key.Value), 
                                 new XAttribute("numberOfCustomers",
                                                countryGroup.Group.Count()),
                                 from 
                                    customer countryGroup.Group, 
                                    city customer.Descendants("City")
                                 group customer by 
                                    (string)city into cityGroup
                                 select 
                                    XElement("City", 
                                                 new XAttribute("name",
                                                                cityGroup.Key), 
                                                 new XAttribute("numberOfCustomers",
                                                                cityGroup.Group.Count()),
                                                 cityGroup.Group.Elements("ContactName"))));
            Console.WriteLine(directory);
            
        }

[<Category("Grouping");
  Title("Group orders by customer");
  Description("Group orders by customer and return all customers (+ orders) for customers who have more than 25 orders ")>]
let XLinq55() =

            XDocument customers = XDocument.Load(dataPath + "nw_customers.xml");
            XDocument orders = XDocument.Load(dataPath + "nw_orders.xml");
            XElement custOrder = XElement("CustomerOrders", 
                from 
                    order orders.Descendants("Orders")
                group order by 
                    order.Element("CustomerID") into cust_orders
                where 
                    cust_orders.Group.Count() > 25
                select 
                    XElement("Customer", 
                        new XAttribute("CustomerID",cust_orders.Key.Value),
                        from 
                            customer customers.Descendants("Customers")
                        where 
                            (string)customer.Attribute("CustomerID") == (string)cust_orders.Key.Value
                        select 
                            customer.Content(),
                        cust_orders.Group));
            Console.WriteLine(custOrder);
            
        }

[<Category("Sort");
  Title("Sort customers by name");
  Description("Sort customers by name ascending order")>]
let XLinq56() =

            XDocument doc = XDocument.Load(dataPath + "nw_customers.xml");
            var query =                              
                from 
                    customer doc.Descendants("Customers")
                orderBy 
                    (string)customer.Elements("ContactName").First()
                select 
                    customer.Element("ContactName");
            XElement result = 
                XElement("Sorted customers",query);
            Console.WriteLine(result);    		
        }

*)

(*
<Category("Sort");
  Title("Sort orders by date");
  Description("Sort orders by date ascending order")>]
let XLinq57() =
  let doc = XDocument.Load(dataPath + "nw_orders.xml") 
  let query =     
    { for order in (doc.Descendants(xname "Orders") |> orderBy (fun order -> order.Element(xname "OrderDate") |> element_to_DateTime))
      let date = order.Element(xname "OrderDate") |> element_to_DateTime 
      let cid = order.Element(xname "CustomerID") 
      -> XElement(xname "Order",  
                      xargs [new XAttribute(xname "date",date); 
                             new XAttribute(xname "custid",cid) ]) } 
  let result = XElement(xname "SortedOrders",xargs (to_list query))
  Console.WriteLine(result)
*)
  
  (*

[<Category("Sort");
  Title("Descending order");
  Description("Sort customers by name descending order")>]
let XLinq58() =

            XDocument doc = XDocument.Load(dataPath + "nw_Customers.xml");
            var query =                 
                from 
                    customer doc.Descendants("Customers")
                orderBy 
                    (string)customer.Elements("ContactName").First() descending
                select 
                    customer.Element("ContactName");
            foreach(var result query)
                Console.WriteLine(result);
            
        }

[<Category("Sort");
  Title("Multiple sort keys");
  Description("Sort customers by country and city")>]
let XLinq59() =

            XDocument doc = XDocument.Load(dataPath + "nw_Customers.xml");
            var query =
                from 
                    customer doc.Descendants("Customers")
                orderBy 
                    (string)customer.Descendants("Country").First(),
                    (string)customer.Descendants("City").First()
                select 
                    customer;
            foreach(var result query)
                Console.WriteLine(result);
            
        }
[<Category("DML");
  Title("Add an element as the last child");
  Description("Add an element as the last child")>]
let XLinq60() =

            XDocument doc = XDocument.Load(dataPath + "config.xml");
            XElement config = doc.Element("config");
            config.Add(XElement("logFolder", "c:\\log"));
            Console.WriteLine(config);

        }

[<Category("DML");
  Title("Add an element as the first child");
  Description("Add an element as the first child")>]
let XLinq61() =

            XDocument doc = XDocument.Load(dataPath + "config.xml");
            XElement config = doc.Element("config");
            config.AddFirst(XElement("logFolder", "c:\\log"));
            Console.WriteLine(config);

        }

[<Category("DML");
  Title("Add multiple elemens as children");
  Description("Add multiple elemens as children")>]
let XLinq62() =

            XDocument doc = XDocument.Load(dataPath + "config.xml");
            XElement[] first = { 
                XElement("logFolder", "c:\\log"), 
                XElement("resultsFolders", "c:\\results")};
            XElement[] last = { 
                XElement("mode", "client"), 
                XElement("commPort", "2")};
            XElement config = doc.Element("config");
            config.AddFirst(first);
            config.Add(last);
            Console.WriteLine(config);
        }


[<Category("DML");
  Title("Add an attribute to an element");
  Description("Add an attribute to an element")>]
let XLinq63() =

            XElement elem = XElement("customer",
                                         "this is an XElement",
                                         new XAttribute("id", "abc"));
            Console.WriteLine("Original element {0}", elem);

            elem.Add(new XAttribute("name", "Jack"));
            Console.WriteLine("Updated element {0}", elem);
        }


[<Category("DML");
  Title("Add content to an existing element");
  Description("Add attributes and elements")>]
let XLinq64() =

            XElement elem = XElement("customer",
                                         "this is an XElement",
                                         new XAttribute("id", "abc"));
            Console.WriteLine("Original element {0}", elem);

            object[] additionalContent = { 
                XElement("phone", "555-555-5555"), 
                new XComment("new customer"), 
                new XAttribute("name", "Jack")};

            elem.Add(additionalContent);
            Console.WriteLine("Updated element {0}", elem);
        }

[<Category("DML");
  Title("Replace content of a container (element or document");
  Description("Replace content to an existing element")>]
let XLinq65() =

            XElement elem = XElement("customer",
                                         "this is an XElement",
                                         new XAttribute("id", "abc"));
            Console.WriteLine("Original element {0}", elem);

            elem.ReplaceContent("this is a coustomer element");
            Console.WriteLine("Updated element {0}", elem);

            object[] newContent = { 
                "this is a customer element", 
                XElement("phone", "555-555-5555"), 
                new XComment("new customer"), 
                new XAttribute("name", "Jack") };

            elem.ReplaceContent(newContent);
            Console.WriteLine("Updated element {0}", elem);

        }

[<Category("DML");
  Title("Remove content of an element");
  Description("Remove content of an element")>]
let XLinq66() =

            XElement elem = XElement("customer",
                                         "this is an XElement",
                                         new XAttribute("id", "abc"));
            Console.WriteLine("Original element {0}", elem);

            elem.RemoveContent();
            Console.WriteLine("Updated element {0}", elem);

        }

[<Category("DML");
  Title("Remove all content");
  Description("Remove all content and attributes of an element")>]
let XLinq67() =

            XElement elem = XElement("customer",
                                         XElement("name", "jack"),
                                         "this is an XElement",
                                         new XComment("new customer"),
                                         new XAttribute("id", "abc"));
            Console.WriteLine("Original element {0}", elem);

            elem.RemoveAll();
            Console.WriteLine("Stripped element {0}", elem);
        }

[<Category("DML");
  Title("Remove all attributes");
  Description("Remove all attributes of an element")>]
let XLinq68() =

            XElement elem = XElement("customer",
                                         new XAttribute("name", "jack"),
                                         "this is an XElement",
                                         new XComment("new customer"),
                                         new XAttribute("id", "abc"));
            Console.WriteLine("Original element {0}", elem);

            elem.RemoveAttributes();
            Console.WriteLine("Stripped element {0}", elem);

        }

[<Category("DML");
  Title("Remove an attribute of an element");
  Description("Remove an attribute of an element")>]
let XLinq69() =

            XElement elem = XElement("customer",
                                         new XAttribute("name", "jack"),
                                         "this is an XElement",
                                         new XComment("new customer"),
                                         new XAttribute("id", "abc"));
            Console.WriteLine("Original element {0}", elem);

            elem.SetAttribute("id", null);
            Console.WriteLine("Updated element {0}", elem);

        }

[<Category("DML");
  Title("Update an attribute");
  Description("Update the value of an attribute")>]
let XLinq70() =

            XElement elem = XElement("customer",
                                         new XAttribute("name", "jack"),
                                         "this is an XElement",
                                         new XComment("new customer"),
                                         new XAttribute("id", "abc"));
            Console.WriteLine("Original element {0}", elem);

            elem.SetAttribute("name", "David");
            Console.WriteLine("Updated attribute {0}", elem);

        }


[<Category("DML");
  Title("Delete an element by name");
  Description("Remove a child element by name")>]
let XLinq71() =

            XElement elem = XElement("customer",
                                         XElement("name", "jack"),
                                         "this is an XElement",
                                         new XComment("new customer"),
                                         new XAttribute("id", "abc"));
            Console.WriteLine("Original element {0}", elem);

            elem.SetElement("name", null);
            Console.WriteLine("Updated element {0}", elem);
        }

[<Category("DML");
  Title("Update a child element by name");
  Description("Update a child element by name")>]
let XLinq72() =

            XElement elem = XElement("customer",
                                         XElement("name", "jack"),
                                         "this is an XElement",
                                         new XComment("new customer"),
                                         new XAttribute("id", "abc"));
            Console.WriteLine("Original element {0}", elem);

            elem.SetElement("name", "David");
            Console.WriteLine("Updated element {0}", elem);
        }

[<Category("DML");
  Title("Remove a list of elements");
  Description("Remove a list of elements")>]
let XLinq73() =

            XDocument doc = XDocument.Load(dataPath + "nw_customers.xml");
            var elems = doc.Descendants("Customers");
            Console.WriteLine("Before count {0}", elems.Count());

            elems.Take(15).Remove();
            Console.WriteLine("After count {0}", elems.Count());
        }

[<Category("DML");
  Title("Remove a list of attributes");
  Description("Remove a list of attributes")>]
let XLinq74() =

            XDocument doc = XDocument.Load(dataPath + "nw_customers.xml");
            var attrs = doc.Descendants("Customers")
                           .Attributes();
            Console.WriteLine("Before count {0}", attrs.Count());

            attrs.Take(15).Remove();
            Console.WriteLine("After count {0}", attrs.Count());
        }


[<Category("DML");
  Title("Add an un-parented element to an element");
  Description("Add an un-parented element to an element")>]
let XLinq75() =

            XElement e = XElement("foo",
                                      "this is an element");
            Console.WriteLine("Parent : " + 
                (e.Parent==null?"null":e.Parent.Value));

            XElement p = XElement("bar",e); //add to document
            Console.WriteLine("Parent : " + 
                (e.Parent == null ? "null":e.Parent.Name));
        }

[<Category("DML");
  Title("Add an parented element to a document");
  Description("Adding a parented element to another container clones it")>]
let XLinq76() =

            XElement e = XElement("foo",
                                      "this is an element");
            XElement p1 = XElement("p1", e);
            Console.WriteLine("Parent : " + e.Parent.Name);

            XElement p2 = XElement("p2", e);
            Console.WriteLine("Parent : " + e.Parent.Name);
        }

[<Category("Transform");
  Title("Create a table of customers");
  Description("Generate html with a list of customers that is numbered")>]
let XLinq77() =
	
            XDocument doc =XDocument.Load(dataPath + "nw_customers.xml");

            var header =new []{                 
                    XElement("th","#"),
                    XElement("th",
                                 "customer id"),
                    XElement("th",
                                 "contact name")};
            int index = 0;
            var rows = 				    
                from 
                    customer doc.Descendants("Customers")
                select 
                    XElement("tr",
                                 XElement("td", 
                                              ++index),
                                 XElement("td", 
                                              (string)customer.Attribute("CustomerID")),
                                 XElement("td", 
                                              (string)customer.Element("ContactName")));

            XElement html = XElement("html",
                                         XElement("table", 
                                                      header,
                                                      rows));
            Console.Write(html);    		
        }

[<Category("Transform");
  Title("Create html tables of books");
  Description("Generate a html tables of books by authors")>]
        [LinkedMethod("GetBooksTable");
let XLinq78() =

            XDocument doc = XDocument.Load(dataPath+"bib.xml");
            var content = 						
                from 
                    b doc.Descendants("book"),
                    a b.Elements("author")
                group 
                    b by (string)a.Element("first") + " "+ (string)a.Element("last") into authorGroup
                select XElement("p","Author: "+authorGroup.Key,
                                            GetBooksTable(authorGroup.Group));

            XElement result = 
                XElement("html",
                             content);
            Console.WriteLine(result);    		
        }

        static XElement GetBooksTable(IEnumerable<XElement> books)

            var header = XElement[]{
                            XElement("th","Title"), 
                            XElement("th", "Year")};
            var rows   = 
                from 
                    b books
                select 
                    XElement("tr",
                                 XElement("td",
                                              (string)b.Element("title")), 
                                 XElement("td",
                                              (string)b.Attribute("year")));			

            return XElement("table",
                                header,
                                rows);
                
        }
[<Category("Language Integration");
  Title("Find all orders for customers a List");
  Description("Find all orders for customers a List");
        [LinkedMethod("SomeMethodToGetCustomers")>]
        [LinkedClass("Customer")>]
let XLinq79() =
	
            XDocument doc =XDocument.Load(dataPath+"nw_orders.xml");
            List<Customer> customers = SomeMethodToGetCustomers();
            var result = 
                from 
                    customer customers, 
                    order doc.Descendants("Orders")
                where 
                    customer.id == (string)order.Element("CustomerID")
                select 
                    new {custid = (string)customer.id,
                            orderdate = (string)order.Element("OrderDate")};
            foreach(var tuple result)
                Console.WriteLine("Customer id = {0}, Order Date = {1}",
                                    tuple.custid, tuple.orderdate);
            
        }
        class Customer

            public Customer(string id) { this.id = id; }
            public string id;
        }
        static List<Customer> SomeMethodToGetCustomers()

            List<Customer> customers = new List<Customer>();
            customers.Add(new Customer("VINET"));
            customers.Add(new Customer("TOMSP"));
            return customers;
        }

[<Category("Language Integration");
  Title("Find sum of items a shopping cart");
  Description("Find sum of items a shopping cart");
        [LinkedMethod("GetShoppingCart");
        [LinkedClass("Item")>]
let XLinq80()
	
            XDocument doc = XDocument.Load(dataPath+"inventory.xml");
            List<Item> cart = GetShoppingCart();
            var subtotal =	
                from 
                    item cart, 
                    inventory doc.Descendants("item")								  
                where 
                    item.id == (string)inventory.Attribute("id")
                select  
                    (double)item.quantity * (double)inventory.Element("price");
            Console.WriteLine("Total payment = {0}",subtotal.Sum());    		
        }
        class Item

            public Item(string id, int quantity)
    
                this.id = id;
                this.quantity = quantity;
            }
            public int quantity;
            public string id;
        }
        static List<Item> GetShoppingCart()

            List<Item> items = new List<Item>();
            items.Add(new Item("1", 10));
            items.Add(new Item("5", 5));
            return items;
        }

[<Category("Language Integration");
  Title("Consume a config file");
  Description("Load and use a config file");
        [LinkedMethod("Initialize")>]
let XLinq81() =

            XElement config = XDocument.Load(dataPath + "config.xml").Element("config");
            Initialize((string)config.Element("rootFolder"),
                       (int)config.Element("iterations"),
                       (double)config.Element("maxMemory"),
                       (string)config.Element("tempFolder"));

        }
        static void Initialize(string root, int iter, double mem, string temp)

            Console.WriteLine("Application initialized to root folder: " +
                              "{0}, iterations: {1}, max memory {2}, temp folder: {3}",
                              root, iter, mem, temp);
        }

[<Category("Language Integration");
  Title("Convert a Sequence of nodes to Array");
  Description("Convert a Sequence of nodes to Array")>]
let XLinq82() =

            XDocument doc = XDocument.Load(dataPath + "nw_Customers.xml");
            XElement[] custArray = doc.Descendants("Customers").ToArray() =;
            foreach (var c custArray)
                Console.WriteLine(c);
        }

[<Category("Language Integration");
  Title("Convert a Sequence of nodes to List");
  Description("Convert a Sequence of nodes to List")>]
let XLinq83() =

            XDocument doc = XDocument.Load(dataPath + "nw_Customers.xml");
            List<XElement> clist = doc.Descendants("Customers").ToList();
            foreach (var c clist)
                Console.WriteLine(c);
        }

[<Category("Language Integration");
  Title("Create a dictionary of customers");
  Description("Create a dictionary of customers")>]
let XLinq84() =

            XDocument doc = XDocument.Load(dataPath + "nw_Customers.xml");
            var dictionary = doc.Descendants("Customers")
                                .ToDictionary(c => (string)c.Attribute("CustomerID"));
            Console.WriteLine(dictionary["ALFKI"]);    		
        }


[<Category("Language Integration");
  Title("Using anonnymous types ");
  Description("Number all the countries and list them")>]
let XLinq85() =

            XDocument doc = XDocument.Load(dataPath + "nw_Customers.xml");    
            var countries = doc.Descendants("Country")
                                .Distinct()
                                .Select((c, index) => new {i=index, name = (string)c});
            foreach(var c countries)
                Console.WriteLine(c.i +" "+ c.name);
        }

[<Category("XName");
  Title("Create elements and attributes a namespace");
  Description("Create elements and attributes a namespace")>]
let XLinq86() =

            string ns = "{http://myNamespace}";
            XElement result = XElement(ns + "foo",
                                           new XAttribute(ns + "bar", "attribute"));
            Console.WriteLine(result);
        }

[<Category("XName");
  Title("Query for elements a namespace");
  Description("Find xsd:element with name=FullAddress")>]
let XLinq87() =

            XDocument doc = XDocument.Load(dataPath + "nw_customers.xsd");
            string XSD = "{http://www.w3.org/2001/XMLSchema}";
            XElement result = doc.Descendants(XSD + "element")
                                 .Where(e => (string)e.Attribute("name") == "FullAddress")
                                 .First();
            Console.WriteLine(result);    			
        }

[<Category("XName");
  Title("Create a namespace prefix decleration");
  Description("Create a namespace prefix decleration")>]
let XLinq88() =

            string myNS = "http://myNamespace";
            string xmlns = "{http://www.w3.org/2000/xmlns/}";
            XElement result = XElement("myElement",
                                           new XAttribute(xmlns + "myPrefix", myNS));
            Console.WriteLine(result);
        }

[<Category("XName");
  Title("Local-name and namespace");
  Description("Get the local-name and namespace of an element")>]
let XLinq89() =

            string ns = "{http://myNamespace}";
            XElement e = XElement(ns + "foo");

            Console.WriteLine("Local name of element: {0}", e.Name.LocalName);
            Console.WriteLine("Namespace of element : {0}", e.Name.NamespaceName);

        }

[<Category("Misc");
  Title("Get the outer XML of a node");
  Description("Get the outer XML of a node")>]
let XLinq90() =

            string xml = "<order >" +
                            "<item price='150'>Motor</item>" +
                            "<item price='50'>Cable</item>" +
                            "<item price='50'>Modem</item>" +
                            "<item price='250'>Monitor</item>" +
                            "<item price='10'>Mouse</item>" +
                         "</order>";
            XElement order = XElement.Parse(xml);
            Console.WriteLine(order.Xml);
        }

[<Category("Misc");
  Title("Get the inner text of a node");
  Description("Get the inner text of a node")>]
let XLinq91() =

            string xml = "<order >" +
                            "<item price='150'>Motor</item>" +
                            "<item price='50'>Cable</item>" +
                            "<item price='50'>Modem</item>" +
                            "<item price='250'>Monitor</item>" +
                            "<item price='10'>Mouse</item>" +
                         "</order>";
            XElement order = XElement.Parse(xml);
            Console.WriteLine(order.Value);

        }

[<Category("Misc");
  Title("Check if an element has attributes");
  Description("Check if an element has attributes")>]
let XLinq92() =

            XDocument doc = XDocument.Load(dataPath + "nw_customers.xml");
            XElement e = doc.Element("Root")
                            .Element("Customers");
            Console.WriteLine("Customers has attributes? {0}", e.HasAttributes);

        }

[<Category("Misc");
  Title("Check if an element has element children");
  Description("Check if an element has element children")>]
let XLinq93() =

            XDocument doc = XDocument.Load(dataPath + "nw_customers.xml");
            XElement e = doc.Element("Root")
                            .Element("Customers");
            Console.WriteLine("Customers has elements? {0}", e.HasElements);

        }

[<Category("Misc");
  Title("Check if an element is empty");
  Description("Check if an element is empty")>]
let XLinq94() =

            XDocument doc = XDocument.Load(dataPath + "nw_customers.xml");
            XElement e = doc.Element("Root")
                            .Element("Customers");
            Console.WriteLine("Customers element is empty? {0}", e.IsEmpty);

        }

[<Category("Misc");
  Title("Get the name of an element");
  Description("Get the name of an element")>]
let XLinq95() =

            XDocument doc = XDocument.Load(dataPath + "nw_customers.xml");
            XElement e = doc.Elements()
                            .First();
            Console.WriteLine("Name of element {0}", e.Name);
        }

[<Category("Misc");
  Title("Get the name of an attribute");
  Description("Get the name of an attribute")>]
let XLinq96() =

            XDocument doc = XDocument.Load(dataPath + "nw_customers.xml");
            XAttribute a = doc.Element("Root")
                              .Element("Customers")
                              .Attributes().First();
            Console.WriteLine("Name of attribute {0}", a.Name);
        }

[<Category("Misc");
  Title("Get the XML decleration");
  Description("Get the XML decleration")>]
let XLinq97() =

            XDocument doc = XDocument.Load(dataPath + "config.xml");
            Console.WriteLine("Version {0}", doc.Declaration.Version);
            Console.WriteLine("Encoding {0}", doc.Declaration.Encoding);
            Console.WriteLine("Standalone {0}", doc.Declaration.Standalone);
        }


[<Category("Misc");
  Title("Find the type of the node");
  Description("Find the type of the node")>]
let XLinq98() =

            XNode o = XElement("foo");
            Console.WriteLine(o.NodeType);
        }

[<Category("Misc");
  Title("Verify phone numbers");
  Description("Verify that the phone numbers of the format xxx-xxx-xxxx");
        [LinkedMethod("CheckPhone")>]
let XLinq99() =
	
            XDocument doc = XDocument.Load(dataPath + "nw_customers.xml");
            var query = 
                from customer doc.Descendants("Customers")
            select 
                XElement("customer", 
                             customer.Attribute("CustomerID"),
                             customer.Descendants("Phone").First(),
                             CheckPhone((string)customer.Descendants("Phone").First()));
            foreach(var result query)
                Console.WriteLine(result);
            
        }
        static XElement CheckPhone(string phone)

            Regex regex = new Regex("([0-9]{3}-)|('('[0-9]{3}')')[0-9]{3}-[0-9]{4}");
            return XElement("isValidPhone", regex.IsMatch(phone));
        }

[<Category("Misc");
  Title("Quick validation");
  Description("Valiate file structure");
        [LinkedMethod("VerifyCustomer")>]
let XLinq100() =

            XDocument doc = XDocument.Load(dataPath + "nw_customers.xml");
            foreach (XElement customer doc.Descendants("Customers"))
    
                string err = VerifyCustomer(customer);
                if (err != "")
                    Console.WriteLine("Cusotmer {0} is invalid. Missing {1}", 
                                        (string)customer.Attribute("CustomerID"), err);
            }

        }
        static string VerifyCustomer(XElement c)

            if (c.Element("CompanyName") == null)
                return "CompanyName";
            if (c.Element("ContactName") == null)
                return "ContactName";
            if (c.Element("ContactTitle") == null)
                return "ContactTitle";
            if (c.Element("Phone") == null)
                return "Phone";
            if (c.Element("Fax") == null)
                return "Fax";
            if (c.Element("FullAddress") == null)
                return "FullAddress";
            return "";
        }


[<Category("Misc");
  Title("Aggregate functions");
  Description("Calculate sum, average, min, max of freight of all orders")>]
let XLinq101() =

            XDocument doc = XDocument.Load(dataPath + "nw_orders.xml");
            var query = 
                from 
                    order doc.Descendants("Orders")
                where 
                    (string)order.Element("CustomerID") == "VINET" &&
                    order.Elements("ShipInfo").Elements("Freight").Any()
                select 
                    (double) order.Element("ShipInfo").Element("Freight");
            
            double sum = query.Sum();
            double average = query.Average();
            double min = query.Min();
            double max = query.Max();
            
            Console.WriteLine("Sum: {0}, Average: {1}, Min: {2}, Max: {3}",sum, average, min, max);
            
        }
    }
}
*)
