// Copyright (c) Microsoft Corporation 2005-2007.
// This sample code is provided "as is" without warranty of any kind. 
// We disclaim all warranties, either express or implied, including the 
// warranties of merchantability and fitness for a particular purpose. 
//

#light

module Sample.Support

open System

type sample = 
    { Run: (unit -> unit);
      Category: string;
      Name: string;
      Title: string;
      Description: string;
      File: string;
      Code: string;
      StartIndex: int }

[<System.AttributeUsage(AttributeTargets.Method, AllowMultiple = false)>]
type TitleAttribute(title:string) = 
    inherit Attribute()
    member x.Title = title

[<System.AttributeUsage(AttributeTargets.Method, AllowMultiple = false)>]
type SupportAttribute(v:string) =
    inherit Attribute()
    member x.SampleName = v

[<System.AttributeUsage(AttributeTargets.Method, AllowMultiple = false)>]
type PrefixAttribute(prefix:string) = 
    inherit Attribute()
    member x.Prefix = prefix

[<System.AttributeUsage(AttributeTargets.Method, AllowMultiple = false)>]
type CategoryAttribute(category:string) = 
    inherit Attribute()
    member x.Category = category

[<System.AttributeUsage(AttributeTargets.Method, AllowMultiple = false)>]
type DescriptionAttribute(description: string) = 
    inherit Attribute()
    member x.Description = description


[<System.AttributeUsage(AttributeTargets.Method, AllowMultiple = false)>]
type LinkedMethodAttribute(methodName:string) = 
    inherit Attribute()
    member x.MethodName = methodName


[<System.AttributeUsage(AttributeTargets.Method, AllowMultiple = false)>]
type LinkedClassAttribute(className:string) = 
    inherit Attribute()
    member x.ClassName = className


type ThisAssem = { dummy: int }

let onEachSampleHook = ref (fun () -> ())
let onEachSample f = let oldHook = (!onEachSampleHook) in onEachSampleHook := (fun () -> f(); oldHook ())

let getSamples () =
  let assem = (typeof<ThisAssem>).Assembly 
  let appdir = AppDomain.CurrentDomain.BaseDirectory 
  Reflection.Assembly.GetFSharpModules(assem)
  |> List.map(fun fsharpModule -> 
      let typ = fsharpModule.CompiledHandle 
      let file = System.IO.Path.Combine(appdir, typ.Name ^ ".fs") 
      let samples = 
          fsharpModule.ConcreteValues
          // We only want the methods with a TitleAttribute, which should always be static
          |> List.filter (fun v -> (v.CompiledHandle.GetCustomAttributes(typeof<TitleAttribute>,false)).Length <> 0) 
          // Prepare an entry for each one...
          |> List.map (fun v -> 
               let m = v.CompiledHandle 
               // We only add title attributes to true functions
               let m = (m :?> System.Reflection.MethodInfo) 
               let name = m.Name 
               let category = 
                 let arr = (m.GetCustomAttributes(typeof<CategoryAttribute>,false)) 
                 if arr.Length = 0 then "<no category>" else (arr.[0] :?> CategoryAttribute).Category 
               let title = 
                 let arr = (m.GetCustomAttributes(typeof<TitleAttribute>,false)) 
                 if arr.Length = 0 then "<no title>" else (arr.[0] :?> TitleAttribute).Title 

               let desc = 
                   let arr = (m.GetCustomAttributes(typeof<DescriptionAttribute>,false)) 
                   if arr.Length = 0 then "<no description>" else (arr.[0] :?> DescriptionAttribute).Description 

               let code,blockStart = 
                   try 
                       let allCode = use sr = new System.IO.StreamReader(file) in sr.ReadToEnd() 
                       let funcStart = allCode.IndexOf("let "+name) 
                       printf "name = %s, funcNameStart = %d, #allCode = %d\n" name funcStart allCode.Length;
                       let codeBlock (blockStart:int) = 
                           let cut x = if x = -1 then allCode.Length else x 
                           let blockEnd = 
                               min (cut (allCode.IndexOf("[<",blockStart )))
                                   (cut (allCode.IndexOf("(*",blockStart ))) 
                           allCode.Substring(blockStart, blockEnd - blockStart)

                       let supportCode = 
                           let supportAttribute = allCode.LastIndexOf("Support(\"" + name + "\")" ,funcStart) 
                           if supportAttribute = -1 then "" 
                           else codeBlock(allCode.IndexOf("let",supportAttribute)) 

                       let code = codeBlock(funcStart) 
                       supportCode + code,funcStart
                   with e -> e.ToString(),0
               { Run = (fun () -> (!onEachSampleHook) (); ignore(m.Invoke((null:obj), [|  |] )));
                 Category = category;
                 Name=name;
                 Title=title;
                 Description=desc;
                 StartIndex=blockStart;
                 Code= code;
                 File=file }) 
          |> List.sort (fun m1 m2 -> compare m1.StartIndex m2.StartIndex)
       
      typ.Name,samples)
  |> List.filter (fun (_,s) -> s <> [])

