
//--------------------------------------------------------------------------
// Copyright (c) Microsoft Corporation 2005-2006.
// This sample code is provided "as is" without warranty of any kind. 
// We disclaim all warranties, either express or implied, including the 
// warranties of merchantability and fitness for a particular purpose. 
//
// This sample shows common patterns for explicit management of resource 
// ownership  F#.  Two types of resources are considered:
//   1. Managed resources, i.e. objects that implement/provide 
//      the IDisposable interface.  The type typically implements the
//      interface directly, although some abstractions may 
//      provide an auxiliary object that implements the interface.
//
//   2. Unmanaged resources, for example integers of type 'nativeint' that
//      represent operating system handles or pointers into raw memory.
//      In the examples below we assume Alloc and Dealloc are functions
//      that directly create and release some unmanaged resources.  In reality these
//      would be accessed via a call into a P/Invoke or COM-interop
//      layer (such methods can be defined using Managed C++ or C#)
//

#light

open System
open System.IO

//--------------------------------------------------------------------------
// Here is an example F# type that owns a single managed resource
// that need to be disposed.  The example is contrived because it
// does nothing but wrap an existing resource.

type WrapOneStream(file) = 
    let myManagedResource = new IO.FileStream(file, FileMode.Open, FileAccess.Read) 

    let mutable disposed = false
    let cleanup () = 
        if not disposed then 
            disposed <- true
            myManagedResource.Close()

    member x.ReadByte() = myManagedResource.ReadByte()
    
    interface IDisposable with
        override x.Dispose() = cleanup()


let Test1() = 
    File.WriteAllText("test.txt", "Some Text")
    use oneWrappedStream = new WrapOneStream("test.txt")
    printfn "byte = %A" (oneWrappedStream.ReadByte())
    // the resource now gets disposed as it drops out of scope.    

Test1()
    
//--------------------------------------------------------------------------
// Here is an example F# type that owns several managed resources
// resources that need to be disposed.

type LineChooser(fileName1, fileName2) = 
    let file1 = File.OpenText(fileName1)
    let file2 = File.OpenText(fileName2)
    let rnd = new System.Random()
    
    let mutable disposed = false
    let cleanup() = 
        if not disposed then 
            disposed <- true; 
            file1.Dispose();
            file2.Dispose();
        
    interface System.IDisposable with 
        member x.Dispose() = cleanup()

    member obj.Close() = cleanup()
    member obj.GetLine() = 
        if not file1.EndOfStream && rnd.Next() % 2 = 0 then file1.ReadLine()
        elif not file2.EndOfStream && rnd.Next() % 2 = 0 then file2.ReadLine()
        elif not file1.EndOfStream then file1.ReadLine()
        elif not file2.EndOfStream then file2.ReadLine()
        else raise (new EndOfStreamException())


// now test it...
File.WriteAllLines("test1.txt", [| "Daisy, Daisy"; "Give me your hand oh do" |])
File.WriteAllLines("test2.txt", [| "I'm a little teapot"; "Short and stout" |])
let chooser = new LineChooser("test1.txt", "test2.txt")

chooser.GetLine()
// val it : string = "Daisy, Daisy"
chooser.GetLine()
// val it : string = "I'm a little teapot"
(chooser :> IDisposable).Dispose()
// The next line will raise an exception!
//chooser.GetLine()
      
//--------------------------------------------------------------------------
// Here is an example F# type that owns some unmanaged resources
// that need to be disposed. Here we mimic an external resource by 
// vis a data structure that generates fresh, reclaimable integer tickets. 
// The idea is that customers will each be given an integer ticket, but that 
// this will be kept internal to the customer, and that customers will return 
// their ticket to the pool when they themselves leave (i.e. are disposed). 

open System.Collections.Generic
type TicketGenerator() = 
    let mutable free = []
    let mutable max = 0
    member h.Alloc() = match free with [] -> max <- max + 1; max | h::t -> free <- t; h
    member h.Dealloc(n:int) = free <- n :: free


let ticketGenerator = new TicketGenerator()

let id1 = ticketGenerator.Alloc()
let id2 = ticketGenerator.Alloc()
ticketGenerator.Dealloc(id1)
let id3 = ticketGenerator.Alloc()

type Customer(name:string) = 
    let myTicket = ticketGenerator.Alloc() 
    let mutable disposed = false
    let cleanup() =  
         if not disposed then 
             disposed <- true;
             ticketGenerator.Dealloc(myTicket)

    member x.Ticket = myTicket
    member x.Name = name

    interface IDisposable with
       member x.Dispose() = cleanup(); GC.SuppressFinalize(x)

    override x.Finalize() = cleanup()


let bill = new Customer("Bill")
printfn "bill.Ticket = %d" bill.Ticket
// Note that joe and jane get the same ticket!
begin 
    use joe = new Customer("Joe")
    printfn "joe.Ticket = %d" joe.Ticket
end;
begin 
    use jane = new Customer("Jane")
    printfn "jane.Ticket = %d" jane.Ticket
end


//--------------------------------------------------------------------------
// Here is an example F# type that owns both unmanaged resources
// and managed resourcese.

type OwnsManyResources(file) = 
    let myManagedResource = new IO.FileStream(file, FileMode.Open, FileAccess.Read) 
    let myUnmanagedResource = ticketGenerator.Alloc() 
    let mutable disposed = false
    let cleanup(deep) = 
        if not disposed then 
            if deep then myManagedResource.Dispose();
            ticketGenerator.Dealloc(myUnmanagedResource)
    
    // Here is the standard machinery to implement disposability.
    // This tends to be replicated  each type that must support the
    // disposal paradigm.
    interface IDisposable with
        member x.Dispose() = cleanup(true); GC.SuppressFinalize(x)

    override x.Finalize() = cleanup(false)

    
//--------------------------------------------------------------------------
// Here is an example F# object expression that extends a type
// which may own both unmanaged and managed resources.  We assume the object 
// expression also owns additional managed resources.

let newExtendedForm(file : string) =
    let myManagedResource = new IO.FileStream(file, FileMode.Open, FileAccess.Read) 

    // The object expression does not have a Finalize method
    // or a Dispose method with parameters because it inherits
    // them from the base class.

    { new Windows.Forms.Form() as base
      with Dispose(deep) =
          if deep then myManagedResource.Close();
          base.Dispose(deep); }

  
  
