// Copyright (c) Microsoft Corporation 2005-2006.
// This sample code is provided "as is" without warranty of any kind. 
// We disclaim all warranties, either express or implied, including the 
// warranties of merchantability and fitness for a particular purpose. 

// Game of life: the algorithm (imperative implementation)

// NOTE: This sample uses 'light' syntax.  This means whitespace
// is signficant.

#light


module Game

open List

type point = int * int
type points = point list

let prev sz i = if i = 0 then sz-1 else i-1 
let next sz i = if i = sz-1 then 0 else i+1
let neighbours sz (i,j) =
    [ (prev sz i,prev sz j); (prev sz i,j); (prev sz i,next sz j);
      (i,        prev sz j);                (i,        next sz j);
      (next sz i,prev sz j); (next sz i,j); (next sz i,next sz j)]

type cell =
    { x: int
      y: int
      mutable neighbours:  cell list
      mutable count: int  
      mutable alive: bool }

let dummy_cell = 
    { x=0
      y=0 
      neighbours= [ ]
      count=0
      alive=false } 

type grid = 
    { size: int; 
      arr: cell[,]; 
      mutable living: point list }
  with 
      // Defining the 'Item' property lets us use grid.[i,j] syntax
      member grid.Item 
          with get(p)   = let (x,y) = p in grid.arr.[x,y]
          and  set(p) v = let (x,y) = p in grid.arr.[x,y] <- v

      member grid.Neighbours(p) = 
         neighbours grid.size p
         |> map (fun p -> grid.[p])  
  end

let empty = 
    { size=0; 
      arr = Array2.create 0 0 dummy_cell; 
      living=[] }

let mk_grid size = 
    // First make all the cells 
    let grid = 
        { size = size
          arr = Array2.init size size (fun i j -> { x=i;y=j; neighbours= [ ]; count=0; alive=false; })
          living=[] } 
    // Now record the neighbours of each cell using direct references
    // to the neighbouring cell objects stored in the array.  This is
    // an imperative implementation of the Game of Life
    for i = 0 to size - 1 do
        for j = 0 to size - 1 do 
            let p = (i,j)
            grid.[p].neighbours <- grid.Neighbours(p)
    grid

let union l1 l2 = 
    List.fold_left (fun l x -> if List.mem x l then l else x::l) l1 l2

let augment grid living =
    // Set each cell the list to "live" 
    living |> iter (fun p -> grid.[p].alive <- true)
    grid.living <- union grid.living living

let (--) f x = f x

let step grid =
    let living = grid.living 
    // For each living cell or neighbour of a living cell, set the count to 0 
    living |> iter (fun p -> 
        grid.[p].count <- 0
        grid.[p].neighbours  |> iter (fun nb -> nb.count <- 0))
        
    // Now compute the number of neighbours for these cells 
    living |> iter (fun p -> 
        grid.[p].neighbours |> iter (fun nb -> nb.count <- nb.count + 1))
        
    // Now compute which cells live/die 
    let born = ref [] 
    let died = ref [] 
    let alive = ref [] 
    living |> iter (fun p -> 
        let c1 = grid.[p] 
      
        if c1.count = 2 || c1.count = 3 then 
            alive := (c1.x,c1.y):: !alive
        else 
            c1.alive <- false
            died := (c1.x,c1.y) :: !died
          
        c1.neighbours |> iter (fun nb -> 
            if (not nb.alive) && (nb.count = 3) then
                nb.alive <- true
                born := (nb.x,nb.y) :: !born
                alive := (nb.x,nb.y):: !alive
        )
    )
       
    grid.living <- !alive
    !born, !died

let alive grid = grid.living


let random_gen size = 
    let gen = new System.Random()
    let density = 0.05 + (gen.NextDouble() * 0.1)
    let acc = ref []
    for i = 0 to size-1 do
        for j = 0 to size-1 do
            if gen.NextDouble() < density then acc:=(i,j)::(!acc)
    !acc

let newRandomGame size = 
    let grid = mk_grid size 
    let born = random_gen size 
    augment grid born
    grid, born
