// (c) Microsoft Corporation 2005-2007. 

(*==========================================================================
 * ROArrays: immutable (read-only) arrays
 *=========================================================================*)

#light

/// This namespace contains constructs for compatibility with earlier versions of F#
/// but which are no longer part of the core library.
namespace Microsoft.FSharp.Compatibility.FSharp

open System
open Microsoft.FSharp.Core
open Microsoft.FSharp.Core.Operators
open Microsoft.FSharp.Collections
open Microsoft.FSharp.Collections
#if CLI_AT_LEAST_2_0
open System.Collections.Generic
#else
open Microsoft.FSharp.Compatibility
#endif

/// A type of read-only arrays
type ReadonlyArray<'a>
type 'a roarray = ReadonlyArray<'a>

[<CompilationRepresentation(CompilationRepresentationFlags.ModuleSuffix)>]
module ReadonlyArray : 

    [<System.Obsolete("Consider using the ReadonlyArray<_> type instead")>]
    type 'a t = ReadonlyArray<'a> // For use when not opening module, e.g. ReadonlyArray.t

    ///Get the length of a read-only array
    val length: ReadonlyArray<'a> -> int

    ///Get an element from a read-only array
    val get: ReadonlyArray<'a> -> int -> 'a

    /// Create a read-only array by index
    /// "init n f" creates the roarray "[| f 0; ...; f (n-1) |]".
    val init: int -> (int -> 'a) -> ReadonlyArray<'a>

    /// "append a1 a2" is equivalent to "[|a1.(0);...;a1.(n-1);a2.(0);...;a2.(m-1)|]"
    /// where "n" is the length of "a1" and "m" is the length of "a2".
    val append: ReadonlyArray<'a> -> ReadonlyArray<'a> -> ReadonlyArray<'a>

    /// "concat" is similar to [append] but conjoins a list of read-only array.  Only
    /// one new roarray is allocated.
    val concat: ReadonlyArray<'a> list -> ReadonlyArray<'a>

    /// "sub a n m" is equivalent to "[| a.(n); ...; a.(n+m) |]".
    val sub: ReadonlyArray<'a> -> int -> int -> ReadonlyArray<'a>

    ///Build a list from the given collection
    val to_list: ReadonlyArray<'a> -> 'a list

    ///Build a collection from the given list
    val of_list: 'a list -> ReadonlyArray<'a>

    ///Apply a function to each element of the collection, threading an 'accumulator' argument
    ///through the computation. If the elements are "i0...iN" then computes "f (... (f s i0)...) iN"
    val fold_left: ('a -> 'b -> 'a) -> 'a -> ReadonlyArray<'b> -> 'a

    ///Apply a function to each element of the collection, threading an 'accumulator' argument
    ///through the computation. If the elements are "i0...iN" then computes "f i0 (...(f iN s))".
    val fold_right: ('a -> 'b -> 'b) -> ReadonlyArray<'a> -> 'b -> 'b

    ///Apply the given function to each element of the collection. 
    val iter: ('a -> unit) -> ReadonlyArray<'a> -> unit

    ///Build a new collection whose elements are the results of applying the given function
    ///to each of the elements of the collection.
    val map: ('a -> 'b) -> ReadonlyArray<'a> -> ReadonlyArray<'b>

    ///Apply the given function to each element of the collection. The integer passed to the
    ///function indicates the index of element.
    val iteri: (int -> 'a -> unit) -> ReadonlyArray<'a> -> unit

    ///Build a new collection whose elements are the results of applying the given function
    ///to each of the elements of the collection. The integer index passed to the
    ///function indicates the index of element being transformed.
    val mapi: (int -> 'a -> 'b) -> ReadonlyArray<'a> -> ReadonlyArray<'b>


    ///Return a view of the collection as an enumerable object
    [<System.Obsolete("Consider using to_seq instead")>]
    val to_IEnumerable: ReadonlyArray<'a> -> IEnumerable<'a>

    ///Build a new collection from the given enumerable object
    [<System.Obsolete("Consider using of_seq instead")>]
    val of_IEnumerable: #IEnumerable<'a> -> ReadonlyArray<'a>

    ///Return a view of the collection as an enumerable object
    val to_seq: ReadonlyArray<'a> -> seq<'a>

    ///Build a new collection from the given enumerable object
    val of_seq: #seq<'a> -> ReadonlyArray<'a>

    #if CLI_AT_LEAST_2_0
    ///Return a view of the collection as a .NET collection 
    val to_ICollection: ReadonlyArray<'a> -> ICollection<'a>

    ///Build a new collection from any type that supports the .NET ICollection interface
    [<Obsolete("Use of_seq instead")>]
    val of_ICollection: #ICollection<'a> -> ReadonlyArray<'a>

    val to_ResizeArray: ReadonlyArray<'a> -> ResizeArray<'a>

    val of_ResizeArray: ResizeArray<'a> -> ReadonlyArray<'a>

    [<Obsolete("This function has been renamed to_ResizeArray")>]
    val to_List: ReadonlyArray<'a> -> System.Collections.Generic.List<'a>
    [<Obsolete("This function has been renamed of_ResizeArray")>]
    val of_List: System.Collections.Generic.List<'a> -> ReadonlyArray<'a>
    #endif
