// (c) Microsoft Corporation 2005-2007. 
#light

//-------------------------------------------------------------------------
// Reflection on F# values. Analyze an object to see if it the representation
// of an F# value.
//-------------------------------------------------------------------------

module Microsoft.FSharp.Reflection 

open System
open System.Reflection
open Microsoft.FSharp.Compatibility
open Microsoft.FSharp.Core
open Microsoft.FSharp.Core.Operators
open Microsoft.FSharp.Core.LanguagePrimitives.IntrinsicOperators
open Microsoft.FSharp.Collections
open Microsoft.FSharp.Collections.List

let debug = false


let emptyArray arr = Array.length arr = 0
let nonEmptyArray arr = Array.length arr > 0
let inline encodeA arr = CompatArray.of_array arr
let inline decodeA arr = CompatArray.to_array arr

let isNamedType(ty:Type) = not (ty.IsArray || ty.IsByRef || ty.IsPointer)

let equivHeadTypes (ty1:Type) (ty2:Type) = 
    isNamedType(ty1) &&
  #if CLI_AT_LEAST_2_0  
    if ty1.IsGenericType then 
      (ty1.GetGenericTypeDefinition()).Equals(ty2.GetGenericTypeDefinition())
    else 
  #endif
      ty1.Equals(ty2)

let isTuple2Type ty = equivHeadTypes ty (type int * int)
let isTuple3Type ty = equivHeadTypes ty (type int * int * int)
let isTuple4Type ty = equivHeadTypes ty (type int * int * int * int)
let isTuple5Type ty = equivHeadTypes ty (type int * int * int * int * int)
let isTuple6Type ty = equivHeadTypes ty (type int * int * int * int * int * int)
let isTuple7Type ty = equivHeadTypes ty (type int * int * int * int * int * int * int)

let isTupleType ty = 
       isTuple2Type ty 
    || isTuple3Type ty 
    || isTuple4Type ty 
    || isTuple5Type ty 
    || isTuple6Type ty 
    || isTuple7Type ty 

let instancePropertyFlags = BindingFlags.GetProperty ||| BindingFlags.Instance ||| BindingFlags.Public 
let staticPropertyFlags = BindingFlags.GetProperty |||  BindingFlags.Static ||| BindingFlags.Public 
let staticInvokeMethodFlags = BindingFlags.InvokeMethod ||| BindingFlags.Static ||| BindingFlags.Public 
let staticFlags = BindingFlags.Static ||| BindingFlags.Public 

let emptyObjArray : obj[] = encodeA [| |]
let getProperty (obj:obj) name =
    let ty = obj.GetType() in 
    ty.InvokeMember(name, instancePropertyFlags, null, obj, emptyObjArray)

let getStaticProperty (obj:obj) name =
    let ty = obj.GetType() in 
    ty.InvokeMember(name, staticPropertyFlags, null, null, encodeA [| obj |])

let callStaticMethod (ty:Type) name args =
    ty.InvokeMember(name, staticInvokeMethodFlags, null, null, encodeA args)

let typeofCompilationMappingAttribute = (type CompilationMappingAttribute)
let typeofException = (type System.Exception)

let memberHasSourceLanguageConstruct (info: #MemberInfo) = 
  let attrs = info.GetCustomAttributes (typeofCompilationMappingAttribute, false) |> decodeA in 
  nonEmptyArray attrs 
let compilationMappingAttributeOfMemberInfo (info: #MemberInfo) = 
  let attrs = info.GetCustomAttributes (typeofCompilationMappingAttribute, false) in 
  let attr = (CompatArray.get (attrs:obj[]) 0 :?> CompilationMappingAttribute) in 
  attr

let sourceLanguageConstructOfMember (ty: #MemberInfo) = (compilationMappingAttributeOfMemberInfo ty).SourceConstruct
let sequenceNumberOfMember (ty: #MemberInfo) = (compilationMappingAttributeOfMemberInfo ty).SequenceNumber
let variantNumberOfMember (ty: #MemberInfo) = (compilationMappingAttributeOfMemberInfo ty).VariantNumber
let typeHasSourceLanguageConstruct (ty:Type) = 
  let attrs = ty.GetCustomAttributes (typeofCompilationMappingAttribute, false) |> decodeA in 
  nonEmptyArray attrs 
let sourceLanguageConstructOfType (ty:Type) = 
  let attrs = ty.GetCustomAttributes (typeofCompilationMappingAttribute, false) in 
  let attr = (CompatArray.get (attrs:obj[]) 0 :?> CompilationMappingAttribute) in 
  attr.SourceConstruct
  
let isSourceLanguageConstruct (ty:Type) = 
  let attrs = ty.GetCustomAttributes (typeofCompilationMappingAttribute, false) in 
  let attr = (CompatArray.get (attrs:obj[]) 0 :?> CompilationMappingAttribute) in 
  attr.SourceConstruct

#if CLI_AT_MOST_1_1
type TypeInfo =
  | TupleType of int
  | FunctionType of unit * unit
  | RecordType of (string * Type) list
  | SumType of (string * (string * Type) list) list
  | DelegateType of Type list * Type
  | UnitType
  | ObjectType of Type
let getTupleTypeInfo (ty:Type) =
  match ty.Name with
    | "Tuple2" -> 2
    | "Tuple3" -> 3
    | "Tuple4" -> 4
    | "Tuple5" -> 5
    | "Tuple6" -> 6
    | "Tuple7" -> 7
    | _ -> failwith "getTupleTypeInfo: expected type name TupleN where N=2..7"
let getFunctionTypeInfo (ty:Type) = (),()
#else
type TypeInfo =
  | TupleType of Type list
  | FunctionType of Type * Type
  | RecordType of (string * Type) list
  | SumType of (string * (string * Type) list) list
  | DelegateType of Type list * Type
  | UnitType
  | ObjectType of Type
let getTupleTypeInfo    (ty:Type) = Array.to_list (ty.GetGenericArguments() |> decodeA)
let getFunctionTypeInfo (ty:Type) =
  begin match Array.to_list(ty.GetGenericArguments() |> decodeA) with 
      [d;r] -> d,r
    | _ -> failwith "TypeInfoOfType: function type has no arguments"
  end
#endif  

let IsOptionType ty = equivHeadTypes ty (type int option)
let isFunctionType ty = equivHeadTypes ty (type int -> int)
let IsUnitType ty = equivHeadTypes ty (type unit)
let IsListType ty = equivHeadTypes ty (type int list)

let isSumType (ty:Type) = 
  IsOptionType ty || IsListType ty || 
  let res = 
    typeHasSourceLanguageConstruct(ty) &&
    sourceLanguageConstructOfType(ty) = SourceLevelConstruct.SumType in 
  if debug then  System.Console.WriteLine("isSumType {0} = {1}",ty,res); 
  res

let isRecordType (ty:Type) = 
  let res = 
    typeHasSourceLanguageConstruct(ty) &&
    sourceLanguageConstructOfType(ty) = SourceLevelConstruct.RecordType  in 
  if debug then System.Console.WriteLine("isRecordType {0} = {1}",ty,res); 
  res

let isFieldProperty (prop : PropertyInfo) =
        memberHasSourceLanguageConstruct(prop) &&
        sourceLanguageConstructOfMember(prop) = SourceLevelConstruct.Field

let sortFreshArray f arr = Array.sort f arr; arr


let fieldPropsOfRecordType(ty:Type) =
  ty.GetProperties(instancePropertyFlags) 
  |> decodeA
  |> Array.filter isFieldProperty
  |> sortFreshArray (fun p1 p2 -> compare (sequenceNumberOfMember p1) (sequenceNumberOfMember p2))

let isRecordRepr ty = isRecordType(ty)
let fieldPropsOfRecordRepr ty = fieldPropsOfRecordType(ty)

let rec isClosureRepr ty = 
    isFunctionType ty || 
    (match ty.BaseType with null -> false | bty -> isClosureRepr bty) 

let isFSharpObjectType ty =
    (typeHasSourceLanguageConstruct(ty) &&
     sourceLanguageConstructOfType(ty) = SourceLevelConstruct.ObjectType)

let discriminatorNameFromSumConstructor (m:MethodInfo) =
    let nm = m.Name in
    if (m.GetParameters()).Length = 0 then
        if nm.Substring(0,4) = "get_" then nm.Substring(4) else nm
    else
        nm

let discriminatorsOfSumType(ty:Type) =
    if IsOptionType ty then [| (0, "None"); (1, "Some") |]
    elif IsListType ty then [| (0, "Nil"); (1, "Cons") |]
    else
      ty.GetMethods(staticFlags) 
      |> decodeA
      |> Array.filter (fun meth -> 
          memberHasSourceLanguageConstruct(meth) &&
          sourceLanguageConstructOfMember(meth) = SourceLevelConstruct.Alternative) 
      |> Array.map (fun p -> sequenceNumberOfMember p, discriminatorNameFromSumConstructor p)
      |> sortFreshArray (fun (i,_) (j,_) -> compare i j)

let getPropertyInfo (ty: Type) propName = ty.GetProperty(propName,instancePropertyFlags) 
let getPropertyInfos ty l = l |> Array.of_list |> Array.map (getPropertyInfo ty) 

  
let getSumTypeTagFields (ty:Type) = 
    let sfields = ty.GetFields(BindingFlags.GetField ||| BindingFlags.Static ||| BindingFlags.Public) in 
    let sfields = Array.to_list (sfields |> decodeA) in 
    let tagfields = List.filter (fun (f:FieldInfo) -> f.Name.Length > 4 &&  f.Name.Substring(0,4) = "tag_") sfields in 
    tagfields
    

let fieldsPropsOfConstructor((ty:Type), (tag:int)) =
    if IsOptionType ty then 
        match tag with 
        | 0 (* None *) -> getPropertyInfos ty []
        | 1 (* Some *) -> getPropertyInfos ty ["Item"] 
        | _ -> failwith "getSumRecordReader"
    elif IsListType ty then 
        match tag with 
        | 0 (* Nil *)  -> getPropertyInfos ty []
        | 1 (* Cons *) -> getPropertyInfos ty ["Head"; "Tail"]
        | _ -> failwith "getSumRecordReader"
    else
        ty.GetProperties(instancePropertyFlags) 
        |> decodeA
        |> Array.filter isFieldProperty
        |> Array.filter (fun prop -> variantNumberOfMember prop = tag)
        |> sortFreshArray (fun p1 p2 -> compare (sequenceNumberOfMember p1) (sequenceNumberOfMember p2))
            
       
// Check the base type - if it is also an F# type then
// for the moment we know it is a Discriminated Union
let isConstructorRepr (ty:Type) = 
  let res = 
    let rec get (ty:Type) = isSumType ty || match ty.BaseType with null -> false | b -> get b in 
    get ty in 
  if debug then    System.Console.WriteLine("isConstructorRepr {0} = {1}",ty,res); 
  res
let sumTypeOfConstructorRepr (ty:Type) = 
    let rec get (ty:Type) = if isSumType ty then ty else match ty.BaseType with null -> ty | b -> get b in 
    get ty 
       
// Check the base type - if it is also an F# type then
// for the moment we know it is a Discriminated Union
let isExceptionRepr (ty:Type) = 
  let res = 
    typeHasSourceLanguageConstruct(ty) &&
    let k = sourceLanguageConstructOfType(ty) in 
    k = SourceLevelConstruct.Exception  in
  if debug then  System.Console.WriteLine("isExceptionRepr {0} = {1}",ty,res); 
  res

let recdDescOfProps props = 
   props |> Array.to_list |> List.map (fun (p:PropertyInfo) -> p.Name, p.PropertyType) 

let isDelegateType (ty:Type) = 
  if (ty.IsSubclassOf(type Delegate)) then
    match ty.GetMethod("Invoke") with
    | null -> false
    | _ -> true
  else
    false

let getDelegateTypeInfo (ty:Type) = 
  let mi = ty.GetMethod("Invoke")
  (mi.GetParameters() |> CompatArray.to_array |> Array.to_list |> List.map (fun p -> p.ParameterType), mi.ReturnType)

module Type = 
    let IsUnitType ty = IsUnitType ty
    let IsListType ty = IsListType ty
    let IsOptionType ty = IsOptionType ty

    let MaxTupleSize = 7


    let GetTypeOfReprType (ty:Type) = 
      if isExceptionRepr(ty) then ty.BaseType
      elif isConstructorRepr(ty) then sumTypeOfConstructorRepr(ty)
      elif isClosureRepr(ty) then 
        let rec get (ty:Type) = if isFunctionType ty then ty else match ty.BaseType with null -> ty | b -> get b in 
        get ty 
      else ty

    let GetTypeInfoOfType (ty:Type) = 
        let ty = GetTypeOfReprType ty in 
        if isTupleType ty then 
          TupleType (getTupleTypeInfo ty)
        elif isFunctionType ty then
          let d,r = getFunctionTypeInfo ty in
          FunctionType (d,r)
        elif isSumType ty then 
          let cnames = discriminatorsOfSumType(ty) |> Array.to_list in 
          if cnames = [] then ObjectType(ty) 
          else  SumType(cnames |> map (fun (tag,cname) -> cname,recdDescOfProps(fieldsPropsOfConstructor(ty,tag))))
        elif isRecordType ty then 
          RecordType(recdDescOfProps(fieldPropsOfRecordType(ty)))
        elif isDelegateType ty then
          let d,r = getDelegateTypeInfo ty in
          DelegateType (d,r)
        else
          ObjectType(ty)

    let GetInfo ty = GetTypeInfoOfType ty

let GetTypeInfoOfType (ty:Type) = Type.GetTypeInfoOfType ty

type 'a typeinfo = { result: TypeInfo }

#if CLI_AT_LEAST_2_0
let typeinfoof () : typeinfo<'a> =
#else
let inline typeinfoof () : typeinfo< $a > =
#endif
    let tok = (# "ldtoken !0" type('a) : System.RuntimeTypeHandle #) in
    let res = System.Type.GetTypeFromHandle(tok) in 
    { result = GetTypeInfoOfType(res) }

#if CLI_AT_LEAST_2_0
let GetTypeInfoOfValue (v:'a) = 
#else
let inline GetTypeInfoOfValue (v: 'a) = 
#endif
  GetTypeInfoOfType (type 'a)

type ValueInfo =
  | TupleValue of obj list
  | FunctionClosureValue of System.Type * obj
  | RecordValue of (string * obj) list
  | ConstructorValue of System.Type * string * (string * obj) list
  | ExceptionValue of System.Type * (string * obj) list
  | UnitValue
  | ObjectValue of obj

let getRecd obj props = 
    props |> map (fun (prop: PropertyInfo) -> (prop.Name, getProperty obj prop.Name)) 

let getRecordReader(ty:Type) = 
    let props = fieldPropsOfRecordType(ty) in 
    (fun (obj:obj) -> props |> Array.map (fun prop -> prop.GetValue(obj,null)) |> encodeA)

let getRecordConstructor(ty:Type) = 
    let props = fieldPropsOfRecordType(ty) in 
    let ctor = ty.GetConstructor(props |> Array.map (fun p -> p.PropertyType) |> encodeA)
    (fun (args:obj[]) -> ctor.Invoke(args))

let getPropertyReader (ty: Type) propName =
    match ty.GetProperty(propName,instancePropertyFlags) with
    | null -> None
    | prop -> Some(fun (obj:obj) -> prop.GetValue(obj,null))

let getRecordFieldReader ty propName = 
    match getPropertyReader ty propName with 
    | None -> failwith "getRecordFieldReader"
    | Some reader -> reader

let getTupleReader (ty:Type) = getRecordReader ty
let getTupleConstructor (ty:Type) = getRecordConstructor ty

let getSumRecordReader (ty:Type) (tag:int) = 
    let props = fieldsPropsOfConstructor(ty,tag) in 
    (fun (obj:obj) -> props |> Array.map (fun prop -> prop.GetValue(obj,null)) |> encodeA)

let getSumTagReader (ty:Type) = 
    if IsOptionType ty then (function null -> 0 | _ -> 1)
    else
      let tagfields = getSumTypeTagFields ty
      let tagreader = 
        match getPropertyReader ty "Tag" with
        | Some reader -> reader
        | None -> (fun obj -> callStaticMethod ty "GetTag" [|obj|]) in 
      (fun obj -> (tagreader obj :?> int))
    
let getSumTypeTagNameMap (ty:Type) = 
    ty |> getSumTypeTagFields 
       |> List.map (fun tagfield -> 
              let constrname = tagfield.Name.Substring(4,tagfield.Name.Length - 4) in
              (tagfield.GetValue(null) :?> int),constrname)

let getSumTagConverter (ty:Type) = 
    let tagfieldmap = ty |> getSumTypeTagNameMap |> Map.of_list 
    (fun tag -> tagfieldmap.[tag])

let swap (x,y) = (y,x)

let getSumTagConverters (ty:Type) = 
    let tagfields = ty |> getSumTypeTagNameMap 
    let tagfieldmap1 = tagfields |> Map.of_list 
    let tagfieldmap2 = tagfields |> List.map swap |> Map.of_list 
    tagfields.Length, (fun tag -> tagfieldmap1.[tag]), (fun tag -> tagfieldmap2.[tag])

let getSumConstructor (ty:Type) (tag:int) = 
    let props = fieldsPropsOfConstructor(ty,tag) 
    let constrname = getSumTagConverter ty tag 
    let methname = if emptyArray props then "get_"+constrname else constrname 
    // NOTE: if the user has specified additional static methods with this name then this will not 
    // behave correctly.  REVIEW: look for the appropriate CompilationMappingAttribute 
    let meth =  ty.GetMethod(methname, staticFlags) 
    (fun args -> meth.Invoke(null,args))


// Analyze an object to see if it the representation
// of an F# value.
let GetValueInfoOfObject(obj : obj) = 
  match obj with 
  | null -> ObjectValue(obj)
  | _ -> 
    let reprty = obj.GetType() in 

    // First a bunch of special rules for tuples
    // Because of the way F# currently compiles tuple values 
    // of size > 7 we can only reliably reflect on sizes up
    // to 7.

    if isTupleType reprty then 
        TupleValue (getTupleReader reprty obj |> CompatArray.to_list)
    elif isClosureRepr reprty then 
        FunctionClosureValue(reprty,obj)
        
    // It must be exception, abstract, record or union.
    // Either way we assume the only properties defined on
    // the type are the actual fields of the type.  Again,
    // we should be reading attributes here that indicate the
    // true structure of the type, e.g. the order of the fields.   
    elif isConstructorRepr reprty then 
        let ty = sumTypeOfConstructorRepr(reprty) in
        let tag = getSumTagReader ty obj in
        // Now some rules for List and Option types.  These use the DefaultAugmentationAttribute(false) hence
        // we can't search properties using fieldsPropsOfConstructor. Other F# types may do the same, but we don't cover those here. 
        if IsOptionType ty then 
            ConstructorValue(reprty,"Some",[("Item", getProperty obj "Item")])
        elif IsListType ty then 
            let constrname = getSumTagConverter ty tag in
            match constrname with 
            | "Cons" -> 
                ConstructorValue(reprty,"Cons",[("Head", getProperty obj "Head");  
                                                ("Tail", getProperty obj "Tail")])
            | _ -> ConstructorValue(reprty,"Nil",[])
        else                            
            let constrname = getSumTagConverter ty tag in
            let props = fieldsPropsOfConstructor(ty,tag) |> Array.to_list in 
            let recd = getRecd obj props in         
            ConstructorValue(ty, constrname, recd)

    elif isExceptionRepr reprty then 
        let props = fieldPropsOfRecordRepr(reprty) |> Array.to_list in 
        let recd = getRecd obj props in 
        ExceptionValue(reprty, recd)
    elif isRecordRepr reprty then 
        let props = fieldPropsOfRecordRepr(reprty) |> Array.to_list in 
        let recd = getRecd obj props in 
        RecordValue(recd)
          
    else
        ObjectValue(obj)
 
// This one is like the above but can make use of additional
// statically-known type information to aid in the
// analysis of null values. 
let 
#if CLI_AT_MOST_1_1
 inline
#endif
 GetValueInfo(x : 'a) = 
    let obj = (box x) in 
    match obj with 
    | null -> 
       let typ = (type 'a) in 
       if IsOptionType typ then  ConstructorValue(typ,"None", [])
       elif IsUnitType typ then  UnitValue
       else ObjectValue(obj)
    | _ -> 
      GetValueInfoOfObject(obj) 


type ValueDefinition= { Handle: System.Reflection.MemberInfo }
    with 
      member x.CompiledHandle = x.Handle
    end
  
type TypeDefinition = { Handle: System.Type }
    with 
      member x.CompiledHandle = x.Handle
    end

type ModuleDefinition = { Handle: System.Type }
    with
      member x.CompiledHandle = x.Handle
    end
    
let isFSharpType typ = isSumType typ || isRecordType typ || isFSharpObjectType typ


let getModulesOfTypes (l : System.Type[]) = 
  l |> CompatArray.to_list
    |> List.filter (fun ty -> typeHasSourceLanguageConstruct(ty) && sourceLanguageConstructOfType(ty) = SourceLevelConstruct.Module)
    |> List.map (fun ty -> { new ModuleDefinition with Handle = ty })
    
let getTypeDefinitionsOfTypes t =
  t |> CompatArray.to_list
    |> List.filter isFSharpType
    |> List.map (fun ty -> { new TypeDefinition with Handle = ty })

let getAssemblyModules (a : System.Reflection.Assembly) = 
    a.GetTypes()  |> getModulesOfTypes

let getAssemblyTypes  (a : System.Reflection.Assembly) = 
    a.GetTypes() |> getTypeDefinitionsOfTypes


module Value = 
#if CLI_AT_LEAST_2_0
    let GetInfo v = GetValueInfo v
    let GetTypeInfo v = GetTypeInfoOfValue v
    let GetType (v:'a) = 
#else
    let inline GetInfo v = GetValueInfo v
    let inline GetTypeInfo v = GetTypeInfoOfValue v
    let inline GetType (v:'a) = 
#endif
            match box v with 
            | null -> (type 'a) 
            | x -> Type.GetTypeOfReprType(x.GetType())
    let GetRecordFieldReader(ty,fld) = getRecordFieldReader ty fld
    let GetRecordReader(ty) = getRecordReader ty 
    let GetRecordConstructor(ty) = getRecordConstructor ty
    //let GetSumFieldReader(ty,tag,fld) = getSumTypeRecordFieldReader
    let GetSumTagReader(ty) = getSumTagReader ty
    let GetSumTagConverters(ty) = getSumTagConverters ty
    let GetSumConstructor(ty,tag) = getSumConstructor ty tag
    let GetSumRecordReader(ty,tag) = getSumRecordReader ty tag
    let GetTupleReader(ty) = getTupleReader ty
    let GetTupleConstructor(ty) = getTupleConstructor ty

type ModuleDefinition 
    with
        member x.TypeDefinitions = x.Handle.GetNestedTypes() |> getTypeDefinitionsOfTypes
        member x.ModuleDefinitions = x.Handle.GetNestedTypes() |> getModulesOfTypes
        member x.ConcreteValues = 
            x.Handle.GetMembers(BindingFlags.Static ||| BindingFlags.Public) 
            |> CompatArray.to_list
            |> List.map (fun x -> { new ValueDefinition with Handle = x })

    end

type TypeDefinition 
    with
        member x.TypeInfo = GetTypeInfoOfType x.Handle
    end

module Assembly = 
    let GetFSharpModules a = getAssemblyModules a
    let GetFSharpTypeDefinitions a = getAssemblyTypes a
            
let GetAssemblyModules a = getAssemblyModules a
let GetAssemblyTypes a = getAssemblyTypes a
//let GetTypeInfoOfType ty = getTypeInfoOfType ty
//let GetValueInfo v = getValueInfo v
//let GetTypeInfoOfValue v = GetTypeInfoOfValue v
//let GetValueInfoOfObject obj = GetValueInfoOfObject obj
//let IsTupleType ty = isTupleType ty
    
[<Obsolete("This data constructor has been renamed to ObjectType")>]
let ExternalType o = ObjectType o

[<Obsolete("This data constructor has been renamed to ObjectType")>]
let AbstractType o = ObjectType o

[<Obsolete("This data recognizer has been renamed to ObjectType")>]
let (|ExternalType|_|) ty =
  match ty with 
  | ObjectType o -> Some o
  | _ -> None
  
[<Obsolete("This data recognizer has been renamed to ObjectType")>]
let (|AbstractType|_|) ty = 
  match ty with 
  | ObjectType o -> Some o
  | _ -> None

[<Obsolete("This data constructor has been renamed to ObjectValue")>]
let ExternalValue o = ObjectValue o

[<Obsolete("This data constructor has been renamed to ObjectValue")>]
let AbstractValue o = ObjectValue o

[<Obsolete("This data recognizer has been renamed to ObjectValue")>]
let (|ExternalValue|_|) v =
  match v with 
  | ObjectValue o -> Some o 
  | _ -> None

[<Obsolete("This data recognizer has been renamed to ObjectValue")>]
let (|AbstractValue|_|) v =
  match v with 
  | ObjectValue o -> Some o 
  | _ -> None
