﻿// (c) Microsoft Corporation 2005-2007. 

#light

/// Types and functions related to expression and type quotations, used
/// when accessing meta-programming libraries such as LINQ.  Expressions
/// can be quoted using the notation such as &lt;@ @&gt, once the
/// [[Quotations.Typed]] namespace has been opened.
namespace Microsoft.FSharp.Quotations

open Microsoft.FSharp.Core
open Microsoft.FSharp.Collections
//open Microsoft.FSharp
open Microsoft.FSharp.Text.StructuredFormat
open System
open System.Reflection

#if CLI_AT_MOST_1_1
#else


(*
[<Sealed>]
/// Information at the binding site of a variable
type Variable =
    interface Microsoft.FSharp.Text.StructuredFormat.IFormattable 
    member Type : Type
    member Name : string
    new : string * Type * ?isMutable : bool -> Variable
    static member GetGlobal : string * Type -> Variable
    static member GetPositionMarker : int * Type -> Variable
    static member CreateFresh : string * Type -> Variable

    [<Obsolete("Use the .Name property instead")>]
    member Text : string
    [<Obsolete("Use the .CreateFresh method instead",true)>]
    static member Create : Variable -> Variable
    [<Obsolete("Use the .CreateFresh method instead")>]
    static member Fresh : string * Type -> Variable


[<Obsolete("This API has been simplified and you should now replace references to Quotations.ExprVarName by Quotations.Variable")>]
type ExprVarName  = Variable
    
[<Obsolete("This API has been simplified and you should now replace references to Quotations.ExprVar by Quotations.Variable")>]
type ExprVar = Variable

*)


/// Specificaton of a reference to a top-level definition, including
/// the namespace/module path to an F# value
type TopDefnData with
    member Assembly : Assembly
    member Path : string list * string
    member TypeScheme : (Type list -> Type)
    
    [<Obsolete("Avoid constructing TopDefnData values directly in your code. Instead use the active pattern 'AnyTopDefnUse' to match a quotation containing a use of the relevant function")>] 
    static member Create : Assembly * (string list * string) * (Type list -> Type) -> TopDefnData


/// Names of variables.  See [[FreshExprVarName]], [[ExprVar.Create]] and [[ExprVar.Fresh]] to generate new names.
type ExprVarName  with 
    /// A textual representation of the name of a variable
    member Text : string
    member Type : Type
    interface Microsoft.FSharp.Text.StructuredFormat.IFormattable
    interface IComparable
    
/// Information at the binding site of a variable
type ExprVar with 
    interface Microsoft.FSharp.Text.StructuredFormat.IFormattable 
    member Type : Type
    member Name : ExprVarName
    member Text : string
    static member Create : ExprVarName -> ExprVar
    static member Fresh : string * Type -> ExprVar

/// Expression contants.  
type ExprConstInfo with 
    interface Microsoft.FSharp.Text.StructuredFormat.IFormattable 

/// Quoted expressions annotated with System.Type values. 
type Expr with 
    interface Microsoft.FSharp.Text.StructuredFormat.IFormattable 
    
    /// Substitute through the given expression using the given functions
    /// to map variables to new values.  The functions must give consistent results
    /// at each application.  Variable renaming may occur on the target expression
    /// if variable capture occurs.
    member Substitute : (ExprVarName -> Expr option) -> Expr 

    /// Get the free expression variables of an expression as a list
    member GetFreeVariables : unit -> seq<ExprVarName>

    /// Returns type of an expression
    member Type : Type

/// Type-carrying quoted expressions.  Expressions are generated either
/// by quotations and quotation templates in source text or
/// programatically by using runtime-type-annotated expressions [[Expr]] and using
/// [[of_raw]] to annotate them.
type Expr<'a> with
    member Raw: Expr 
    interface Microsoft.FSharp.Text.StructuredFormat.IFormattable
    /// Substitution and instantiation.
    /// Replace expression variables with parameters using the
    /// given substitution maps.  The functions return expressions that
    /// do not carry static type annotations (Expr),
    /// hence there is an implicit requirement that these functions return
    /// correctly typed expressions.
    member Substitute : (ExprVarName -> Expr option) -> Expr<'a>

/// Type-carrying quoted raw expression templates.
type Template<'rawtupty,'rawfty> with
    member Raw: Expr 
    interface Microsoft.FSharp.Text.StructuredFormat.IFormattable

/// Templates for type-carrying quoted expressions.  
type Template <'a,'tupty,'fty,'rty> with
    member Raw: Expr 
    interface Microsoft.FSharp.Text.StructuredFormat.IFormattable


/// Quoted expressions and operations related to them.  These
/// expressions trees are dynamically annotated with type information which is
/// dynamically checked for consistency.
module Raw =

    val MkNamedType : Type * Type list -> Type
    
    /// Build variable expressions     
    val MkVar    : ExprVarName -> Expr
    
    /// Build expression applications
    val MkApp    : Expr * Expr -> Expr
    
    /// Build lambda (abstraction) expressions 
    val MkLambda : ExprVar * Expr -> Expr
    
    /// Build  holes in quotation templates
    val MkHole   : Type -> Expr 
    
    /// Build quoted expressions (for inner quotations)
    val MkQuote  : Expr  -> Expr

    //-------------------------------
    // These are ways of making expressions of the form (App(...App(Const,arg1),...),argN)

    /// Build expressions that are constants referring to top-level public definitions,
    /// where the constants may generic and applied to a number of type arguments.
    val MkAnyTopDefnUse : TopDefnData * Type list -> Expr
    
    /// Build 'if ... then ... else' expressions    
    val MkCond : Expr * Expr * Expr -> Expr 
    
    /// Build recursives expressions associated with 'let rec' constructs
    val MkLetRec  : (ExprVar * Expr) list * Expr -> Expr 
    
    /// Build expressions associated with 'let' constructs
    val MkLet : (ExprVar * Expr) * Expr -> Expr
    
    /// Build record-construction expressions 
    val MkRecd : Type * Expr list -> Expr 
    
    /// Build record-access expressions 
    val MkRecdGet :  Type * string * Expr -> Expr 
    
    /// Build record-access expressions 
    val MkRecdSet : Type * string * Expr * Expr -> Expr
    
    /// Build discriminated-union construction expressions 
    val MkSum :  Type * string * Expr list -> Expr 
    
    /// Build discriminated-union access expressions, resulting primarily
    /// from the compilation of pattern matches
    val MkSumFieldGet : Type * string * int * Expr  -> Expr 
    
    /// Build discriminated-union discrimination test expressions, resulting 
    /// primarily from the compilation of pattern matches
    val MkSumTagTest :  Type * string * Expr -> Expr 
    
    /// Build tuple-expressions
    val MkTuple : Type * Expr list -> Expr 
    
    /// Build tuple access expressions
    val MkTupleGet :  Type * int * Expr -> Expr 
    
    /// Build equality tests, resulting primarily from the 
    /// compilation of pattern matching 
    //[<Obsolete("This type of node no longer occurs in expressions. Equality tests arising from pattern matching are now represented as calls to the '=' operator.")>]
    val MkEquality : Expr * Expr -> Expr 
    
    /// Build coercion expressions, resulting from implicit coercions at object-model calls and explicit coercions elsewhere.
    val MkCoerce : Type * Expr -> Expr 
    
    /// Build array construction expressions
    val MkNewArray : Type * Expr list -> Expr
    
    /// Build 'unit' constant expressions
    val MkUnit : unit -> Expr 
    
    /// Build constant boolean expressions
    val MkBool : bool -> Expr 
    
    /// Build constant string expressions
    val MkString : string -> Expr 
    
    /// Build constant 32-bit floating point number expressions
    val MkSingle : float32 -> Expr 
    
    /// Build constant 64-bit floating point number expressions
    val MkDouble : float -> Expr 
    
    /// Build constant unicode character expressions
    val MkChar :  char  -> Expr 
    
    /// Build constant signed 8-bit integers
    val MkSByte : sbyte -> Expr 
    val MkByte : byte -> Expr 
    val MkInt16 : int16 -> Expr 
    val MkUInt16 : uint16 -> Expr 
    val MkInt32 : int32 -> Expr 
    val MkUInt32 : uint32 -> Expr 
    val MkInt64 : int64 -> Expr 
    val MkUInt64 : uint64 -> Expr 
    
    /// Build address-access expressions.
    val MkGetAddr : Expr -> Expr 
    
    /// Build sequence expressions "a;b"
    val MkSeq : Expr * Expr -> Expr 
    
    /// Build for-loop expressions, where the body of the for-loop is encoded as a lambda.
    val MkEncodedForLoop : Expr * Expr * Expr -> Expr 
    
    /// Build while-loop expressions, where the body of the loop is encoded as a lambda.
    val MkEncodedWhileLoop : Expr * Expr -> Expr  
    
    /// Build calls to construct delegate values in the .NET obejct model
    val MkNewDelegate : Type * Expr -> Expr 
    
    /// Build calls to property accessors or field-fetches in the .NET object model
    val MkPropGet :  PropertyInfo * Expr -> Expr 
    
    /// Build .NET IL field-fetches in the .NET object model
    val MkFieldGet :  FieldInfo * Expr -> Expr 
    
    /// Build new-object calls to constructors in the .NET obejct model
    val MkCtorCall :  ConstructorInfo * Expr list -> Expr 
    
    /// Build calls to the .NET obejct model, excluding property accesses etc.
    val MkMethodCall : MethodInfo * Expr list -> Expr 
    
    /// Build expressions made up of arbitrary reified (lifted) values
    val MkLiftedValue : obj * Type -> Expr

    /// Build expressions corresponding to uses of a particular known top level generic value.
    /// e.g. MkGenericTopDefnUse (&lt;@ List.map @&gt;) [(type int); (type string)]
    val MkGenericTopDefnUse : Expr -> (Type list -> Expr)

    /// Build expressions corresponding to uses and applications of a particular known top level generic definition.
    /// e.g. MkGenericTopDefnApp (&lt;@ List.map @&gt;) ([(type int); (type string)], [&lt;@ 1;2 @&gt;]
    val MkGenericTopDefnApp : Expr -> (Type list * Expr list -> Expr)

    /// Build an iterated series of lambdas abstractions
    val MkLambdas       : (ExprVar list * Expr) -> Expr
    /// Build an iterated series of function applications f x1 ... xN
    val MkApps          : (Expr * Expr list) -> Expr

    /// Recognize variable expressions     
    val (|Var|_|)           : Expr -> ExprVarName option
    /// Recognize expression applications
    val (|App|_|)           : Expr -> (Expr * Expr) option
    /// Recognize lambda (abstraction) expressions 
    val (|Lambda|_|)        : Expr -> (ExprVar * Expr) option
    /// Recognize expressions representing holes in quotation templates
    val (|Hole|_|)          : Expr -> Type option
    /// Recognize quoted expressions (for inner quotations)
    val (|Quote|_|)         : Expr -> Expr option
    /// Recognize 'if ... then ... else' expressions    
    val (|Cond|_|)          : Expr -> (Expr * Expr * Expr) option

    /// Recognize unapplied occurrences of module-bound values or functions 
    /// (i.e. those defined using 'let' or 'let rec')
    val (|AnyTopDefnUse|_|)    : Expr -> (TopDefnData * Type list)  option

    /// Recognize recursives expressions associated with 'let rec' constructs
    val (|LetRec|_|)        : Expr -> ((ExprVar * Expr) list * Expr) option
    /// Recognize expressions associated with 'let' constructs
    val (|Let|_|)           : Expr -> ((ExprVar * Expr) * Expr) option
    /// Recognize record-construction expressions 
    val (|Recd|_|)          : Expr -> (Type * Expr list) option 
    /// Recognize record-access expressions 
    val (|RecdGet|_|)       : Expr -> (Type * string * Expr) option 
    /// Recognize record-access expressions 
    val (|RecdSet|_|)       : Expr -> (Type * string * Expr * Expr) option
    /// Recognize discriminated-union construction expressions 
    val (|Sum|_|)           : Expr -> (Type * string * Expr list) option 
    /// Recognize discriminated-union access expressions, resulting primarily
    /// from the compilation of pattern matches
    val (|SumFieldGet|_|)   : Expr -> (Type * string * int * Expr) option 
    /// Recognize discriminated-union discrimination test expressions, resulting 
    /// primarily from the compilation of pattern matches
    val (|SumTagTest|_|)    : Expr -> (Type * string * Expr) option 
    /// Recognize tuple-expressions
    val (|Tuple|_|)         : Expr -> (Type * Expr list) option 
    /// Recognize tuple access expressions
    val (|TupleGet|_|)      : Expr -> (Type * int * Expr) option 
    /// Recognize equality tests, resulting primarily from the 
    /// compilation of pattern matching 
    //[<Obsolete("This type of node no longer occurs in expressions. Equality tests arising from pattern matching are now represented as calls to the '=' operator.")>]
    val (|Equality|_|)      : Expr -> (Expr * Expr) option 
    /// Recognize coercion expressions, resulting from implicit coercions at object-model calls and explicit coercions elsewhere.
    val (|Coerce|_|)        : Expr -> (Type * Expr) option 
    /// Recognize array construction expressions
    val (|NewArray|_|)      : Expr -> (Type * Expr list) option
    /// Recognize 'unit' constant expressions
    val (|Unit|_|)          : Expr -> unit option 
    /// Recognize constant boolean expressions
    val (|Bool|_|)          : Expr -> bool option 
    /// Recognize constant string expressions
    val (|String|_|)        : Expr -> string option 
    /// Recognize constant 32-bit floating point number expressions
    val (|Single|_|)        : Expr -> float32 option 
    /// Recognize constant 64-bit floating point number expressions
    val (|Double|_|)        : Expr -> float option 
    /// Recognize constant unicode character expressions
    val (|Char|_|)          : Expr -> char  option 
    /// Recognize constant signed 8-bit integers
    val (|SByte|_|)         : Expr -> sbyte option 
    val (|Byte|_|)          : Expr -> byte option 
    val (|Int16|_|)         : Expr -> int16 option 
    val (|UInt16|_|)        : Expr -> uint16 option 
    val (|Int32|_|)         : Expr -> int32 option 
    val (|UInt32|_|)        : Expr -> uint32 option 
    val (|Int64|_|)         : Expr -> int64 option 
    val (|UInt64|_|)        : Expr -> uint64 option 
    /// Recognize address-access expressions.
    val (|GetAddr|_|)       : Expr -> Expr option 
    /// Recognize sequence expressions "a;b"
    val (|Seq|_|)           : Expr -> (Expr * Expr) option 
    /// Recognize for-loop expressions, where the body of the for-loop is encoded as a lambda.
    val (|EncodedForLoop|_|)       : Expr -> (Expr * Expr * Expr) option 
    /// Recognize while-loop expressions, where the body of the loop is encoded as a lambda.
    val (|EncodedWhileLoop|_|)     : Expr -> (Expr * Expr) option  
    /// Recognize calls to construct delegate values in the .NET obejct model
    val (|NewDelegate|_|)   : Expr -> (Type * Expr) option 
    /// Recognize calls to property accessors or field-fetches in the .NET object model
    val (|PropGet|_|)   : Expr -> (PropertyInfo * Expr) option 
    /// Recognize .NET IL field-fetches in the .NET object model
    val (|FieldGet|_|)      : Expr -> (FieldInfo * Expr) option 
    /// Recognize new-object calls to constructors in the .NET obejct model
    val (|CtorCall|_|)      : Expr -> (ConstructorInfo * Expr list) option 
    /// Recognize calls to the .NET obejct model, excluding property accesses etc.
    val (|MethodCall|_|)    : Expr -> (MethodInfo * Expr list) option 
    /// Recognize expressions made up of arbitrary reified (lifted) values
    val (|LiftedValue|_|)   : Expr -> (obj * Type) option
    /// Recognize an iterated series of lambdas abstractions
    val (|Lambdas|)       : Expr -> (ExprVar list * Expr)
    /// Recognize an iterated series of function applications f x1 ... xN
    val (|Apps|)          : Expr -> (Expr * Expr list) 
    /// Recognize the encoded form of 'a &amp;&amp; b'
    val (|LazyAnd|_|)     : Expr -> (Expr * Expr) option 
    /// Recognize the encoded form of 'a || b'
    val (|LazyOr|_|)      : Expr -> (Expr * Expr) option 

    /// Recognize expressions corresponding to uses of a particular known 
    /// module-bound value or function (i.e. one defined using 'let' or 'let rec').
    ///
    /// e.g. GenericTopDefnUse (&lt;@ List.map @&gt;)  will recognize 
    /// (&lt;@ List.map : (string -&gt; int) -%gt; string list -&gt; int list l@&gt;) as
    /// an application of the map function to types [(type int); (type string)].
    val (|GenericTopDefnUse|_|) : Expr -> (Expr -> Type list option)

    /// Recognize expressions corresponding to uses and applications of a particular known 
    /// module-bound value or function (i.e. one defined using 'let' or 'let rec').
    ///
    /// e.g. GenericTopDefnApp (&lt;@ List.map @&gt;) will recognize (&lt;@ List.map string_of_int [1;2] @&gt;) as
    /// an application of the map function to types [(type int); (type string)] and arguments
    /// (&lt;@ string_of_int @&gt;) ad [&lt;@ 1;2 @&gt;].
    val (|GenericTopDefnApp|_|) : Expr -> (Expr -> (Type list * Expr list) option)

    /// Recognize expressions corresponding to uses and applications of a particular known 
    /// module-bound value or function (i.e. one defined using 'let' or 'let rec').
    ///
    /// e.g. TopDefnApp (&lt;@ string_of_int @&gt;) will recognize (&lt;@ string_of_int 3 @&gt;) as
    /// an application. This function may be used with top level generic definitions, but the types
    /// arguments to the generic definition will not be returned. If the type arguments are relevant then
    /// use GenericTopDefnApp instead.
    val (|TopDefnApp|_|) : Expr -> (Expr -> Expr list option)

    module BindingStructure =
        /// A complete decomposition viewing the expression tree as a binding structure
        val (|Var|Lambda|ConstApp|Hole|Quote|) : Expr -> Choice<ExprVarName,                  // Var
                                                                (ExprVar * Expr),             // Lambda
                                                                (ExprConstInfo * Expr list),  // Const
                                                                Type,                         // Hole
                                                                Expr>                         // Quote
        /// Build constant expressions
        val MkConstApp  : ExprConstInfo * Expr list -> Expr

        module Unchecked =
          
          // unfortunately sometimes we need this...
          [<Obsolete("Checked versions should be used!")>]
          val MkConstApp  : ExprConstInfo * Expr list -> Expr

    /// Files compiled with --quotation-data persist definitions to 
    /// the emitted assembly ('cross stage persistence of level 0 definitions').
    /// They are saved as a resource in the emitted module.  This is the base name
    /// for the resources.
    val PickledDefinitionsResourceNameBase : string
    
    /// Attempt to resolve a module-bound value or function (i.e. those defined using 'let' or 'let rec')
    /// to an implementation.  If the value or function is 
    /// generic then appropriate type arguments must also be specified.
    val ResolveTopDefinition: TopDefnData * Type list -> Expr option


    /// Attempt to resolve a module-bound value or function (i.e. those defined using 'let' or 'let rec')
    /// to an implementation.  If the value or function is 
    /// generic then appropriate type arguments must also be specified.
    val (|ResolvedTopDefn|_|): TopDefnData * Type list -> Expr option

    /// Attempt to resolve a method definition to an implementation.  
    val ResolveMethodDefn: #MethodBase -> Expr option

    /// Attempt to resolve a method definition to an implementation.  
    val (|ResolvedMethodDefn|_|): #MethodBase -> Expr option

    /// Attempt to resolve a module-bound value or function (i.e. those defined using 'let' or 'let rec')
    /// to an implementation.  If the value or function is 
    /// generic then appropriate type arguments must also be specified.
    val (|ResolvedTopDefnUse|_|): Expr -> (TopDefnData * Expr) option

    /// Permit interactive environments such as F# Interactive
    /// to explicitly register new pickled resources that represent persisted 
    /// top level definitions. The string indicates a unique name for the resources
    /// being added.
    val ExplicitlyRegisterTopDefs: Assembly -> string -> byte[] -> unit

    /// Perform macro expansion (deep call-by-value reduction) 
    /// on an expression.  Lambda applications are beta-reduced 
    /// after reducing the function term and the argument term.
    /// Top-level definitions are expanded according to any top definition data
    /// that is available via ResolveTopDefinition.  
    ///
    /// The interior bodies of lambda expressions are expanded.  Reduction rules
    /// for unions, records, arithmetic etc. are not applied.
    ///
    /// A cutoff that halts the rewriting for expressions
    /// that pass the given test (usually a test for inclusion within a particular
    /// 'target' langauge).
    ///
    /// This function is mainly included for demonstration purposes and should
    /// not be used in production quotation processing.
    [<Obsolete("This function is included in this API for demonstration purposes and will be removed in a future release")>]
    val DeepMacroExpandUntil : (Expr -> bool) -> Expr -> Expr

    /// This function is called automatically when quotation syntax (&lt;@@ @@&gt;) and related raw-expression
    /// quotations are used. The bytes are a pickled binary representation of an unlinke form of the qutoed expression,
    /// and the System.Type argument is any type in the assembly where the quoted
    /// expression occurs, i.e. it helps scope the interpretation of the cross-assembly
    /// references in the bytes.
    val Unpickle : System.Type ->  byte[]  -> Template<'a,'b>

    /// ASCII syntax for raw expression quotation.  If there are holes
    /// in the expression then 'b is a function type that is used produced to generate new terms.
    /// For example, if there are no holes then 'b is always just 'Expr'.
    /// If there is one hole, then 'b is Expr -&gt; Expr.
    val (<@@ @@>)  : Template<_,'b> -> 'b 
    /// ASCII syntax for raw expression template quotation.
    val (<@@. .@@>)  : Template<'a,'b> -> Template<'a,'b>

    // Special prefix operator for splicing expressions into quotation holes
    val (§§) : 'a -> Expr
    val (~%%) : 'a -> Expr

    /// Match an existing raw expression against the given expression template
    val MkTemplate: Template<'rawqty,'rawfty> -> ('rawqty -> Expr)
    /// Match an existing raw expression against the given raw expression template
    val (|Template|_|): Template<'rawqty,'rawfty> -> (Expr -> 'rawqty option)

    /// For internal use only.
    type RawArgSaver<'b> = 
        new : unit -> 'b RawArgSaver 
        member Make : obj -> obj

    /// Contains construction functions that don't perform type-checking
    module Unchecked =

        /// Build variable expressions     
        val MkVar    : ExprVarName -> Expr
        
        /// Build expression applications
        val MkApp    : Expr * Expr -> Expr
        
        /// Build lambda (abstraction) expressions 
        val MkLambda : ExprVar * Expr -> Expr
        
        /// Build  holes in quotation templates
        val MkHole   : Type -> Expr 
        
        /// Build quoted expressions (for inner quotations)
        val MkQuote  : Expr  -> Expr

        /// Build 'if ... then ... else' expressions    
        val MkCond : Expr * Expr * Expr -> Expr 
        
        /// Build recursives expressions associated with 'let rec' constructs
        val MkLetRec  : (ExprVar * Expr) list * Expr -> Expr 
        
        /// Build expressions associated with 'let' constructs
        val MkLet : (ExprVar * Expr) * Expr -> Expr
        
        /// Build record-construction expressions 
        val MkRecd : Type * Expr list -> Expr 
        
        /// Build record-access expressions 
        val MkRecdGet :  Type * string * Expr -> Expr 
        
        /// Build record-access expressions 
        val MkRecdSet : Type * string * Expr * Expr -> Expr
        
        /// Build discriminated-union construction expressions 
        val MkSum :  Type * string * Expr list -> Expr 
        
        /// Build discriminated-union access expressions, resulting primarily
        /// from the compilation of pattern matches
        val MkSumFieldGet : Type * string * int * Expr  -> Expr 
        
        /// Build discriminated-union discrimination test expressions, resulting 
        /// primarily from the compilation of pattern matches
        val MkSumTagTest :  Type * string * Expr -> Expr 
        
        /// Build tuple-expressions
        val MkTuple : Type * Expr list -> Expr 
        
        /// Build tuple access expressions
        val MkTupleGet :  Type * int * Expr -> Expr 
        
        /// Build equality tests, resulting primarily from the 
        /// compilation of pattern matching 
        //[<Obsolete("This type of node no longer occurs in expressions. Equality tests arising from pattern matching are now represented as calls to the '=' operator.")>]
        val MkEquality : Expr * Expr -> Expr 
        
        /// Build coercion expressions, resulting from implicit coercions at object-model calls and explicit coercions elsewhere.
        val MkCoerce : Type * Expr -> Expr 
        
        /// Build array construction expressions
        val MkNewArray : Type * Expr list -> Expr
        
        /// Build 'unit' constant expressions
        val MkUnit : unit -> Expr 
        
        /// Build constant boolean expressions
        val MkBool : bool -> Expr 
        
        /// Build constant string expressions
        val MkString : string -> Expr 
        
        /// Build constant 32-bit floating point number expressions
        val MkSingle : float32 -> Expr 
        
        /// Build constant 64-bit floating point number expressions
        val MkDouble : float -> Expr 
        
        /// Build constant unicode character expressions
        val MkChar :  char  -> Expr 
        
        /// Build constant signed 8-bit integers
        val MkSByte : sbyte -> Expr 
        val MkByte : byte -> Expr 
        val MkInt16 : int16 -> Expr 
        val MkUInt16 : uint16 -> Expr 
        val MkInt32 : int32 -> Expr 
        val MkUInt32 : uint32 -> Expr 
        val MkInt64 : int64 -> Expr 
        val MkUInt64 : uint64 -> Expr 
        
        /// Build address-access expressions.
        val MkGetAddr : Expr -> Expr 
        
        /// Build sequence expressions "a;b"
        val MkSeq : Expr * Expr -> Expr 
        
        /// Build for-loop expressions, where the body of the for-loop is encoded as a lambda.
        val MkEncodedForLoop : Expr * Expr * Expr -> Expr 
        
        /// Build while-loop expressions, where the body of the loop is encoded as a lambda.
        val MkEncodedWhileLoop : Expr * Expr -> Expr  
        
        /// Build calls to construct delegate values in the .NET obejct model
        val MkNewDelegate : Type * Expr -> Expr 
        
        /// Build calls to property accessors or field-fetches in the .NET object model
        val MkPropGet :  PropertyInfo * Expr -> Expr 
        
        /// Build .NET IL field-fetches in the .NET object model
        val MkFieldGet :  FieldInfo * Expr -> Expr 
        
        /// Build new-object calls to constructors in the .NET obejct model
        val MkCtorCall :  ConstructorInfo * Expr list -> Expr 
        
        /// Build calls to the .NET obejct model, excluding property accesses etc.
        val MkMethodCall : MethodInfo * Expr list -> Expr 
        
        /// Build expressions made up of arbitrary reified (lifted) values
        val MkLiftedValue : obj * Type -> Expr

        /// Build an iterated series of lambdas abstractions
        val MkLambdas       : (ExprVar list * Expr) -> Expr
        /// Build an iterated series of function applications f x1 ... xN
        val MkApps          : (Expr * Expr list) -> Expr
        
        /// Build expressions corresponding to uses and applications of a particular a module-bound value or function (i.e. one defined using 'let' or 'let rec').
        /// e.g. MkGenericTopDefnApp (&lt;@ List.map @&gt;) ([(type int); (type int)], [&lt;@ (fun x -&gt; x + 1) @&gt; &lt;@ [1;2] @&gt;]
        val MkGenericTopDefnApp : Expr -> (Type list * Expr list -> Expr)


/// Type-carrying quoted expressions and operations related to them.
module Typed =

    /// Reify any runtime value as an expression of family LiftedValue.
    val lift : 'a -> Expr<'a> 

    /// Return the untyped expression underlying this type-annotated expression
    val to_raw : Expr<'a> -> Expr 

    /// Apply a an operation to the underlying raw term
    val map_raw : (Expr -> Expr) -> Expr<'a> -> Expr<'a>

    /// This function is mainly included for demonstration purposes and should
    /// not be used in production quotation processing.
    val DeepMacroExpandUntil : (Expr -> bool) -> Expr<'a> -> Expr<'a>

    /// Return a new typed expression given an underlying runtime-typed expression.
    /// A type annotation is usually required to use this function, and 
    /// using an incorrect type annotation may result in a later runtime exception.
    val of_raw : Expr -> Expr<'a> 

    /// Quote an expression.  If there are holes
    /// in the expression then a function is produced to generate new terms
    /// via the function 'fill'.
    val ( « » ) : Template<'ety,_,'fty,Expr<'ety>> -> 'fty 
    /// ASCII syntax for typed expression quotation.  If there are holes
    /// in the expression then a function is produced to generate new terms
    /// by filling in the holes with the arguments given to the function.
    val (<@ @>)  : Template<'ety,_,'fty,Expr<'ety>> -> 'fty 
    /// ASCII syntax for typed expression template quotation.
    val (<@. .@>)  : ('a,'b,'c,Expr<'a>) Template -> ('a,'b,'c,Expr<'a>) Template 

    /// This function is called automatically when quotation syntax (&lt;@ @&gt;) and related typed-expression
    /// quotations are used. The bytes are a pickled binary representation of an unlinke form of the qutoed expression,
    /// and the System.Type argument is any type in the assembly where the quoted
    /// expression occurs, i.e. it helps scope the interpretation of the cross-assembly
    /// references in the bytes.
    val Unpickle : System.Type ->  byte[] -> Template<'a,'b,'c,'d> 

    /// Fill the holes in a typed expression template
    val MkTemplate: Template<'ety,_,'fty,Expr<'ety>> -> 'fty 

    /// Match an existing typed expression against the given typed expression template
    val (|Template|_|): Template<'ety,'qty,_,Expr<'ety>> -> (Expr<'ety> -> 'qty option)

    /// For internal use only.
    type ArgSaver<'a,'b> = 
        new : unit -> ('a,'b) ArgSaver 
        member Make : obj -> obj

    // Special prefix operator for splicing expressions into quotation holes
    val (§) : 'a -> Expr<'a>
    val (~%) : 'a -> Expr<'a>


#endif
