﻿// (c) Microsoft Corporation 2005-2007.

#light

// -------------------------------------------------------------------- 
//
// -------------------------------------------------------------------- 

namespace Microsoft.FSharp.Quotations
#nowarn "57"

open System
open System.IO
open System.Reflection
open Microsoft.FSharp
open Microsoft.FSharp.Core
open Microsoft.FSharp.Core.LanguagePrimitives.IntrinsicOperators
open Microsoft.FSharp.Core.Operators
open Microsoft.FSharp.Collections
open Microsoft.FSharp.Collections.List
open Microsoft.FSharp.Primitives.Basics
open Microsoft.FSharp.Compatibility
open Microsoft.FSharp.Text
open Microsoft.FSharp.Text.Printf
open Microsoft.FSharp.Text.StructuredFormat

#if CLI_AT_MOST_1_1
#else

type IFormattable = Microsoft.FSharp.Text.StructuredFormat.IFormattable

module Preliminaries = 

    let string_of_int (i:int) = i.ToString(System.Globalization.CultureInfo.InvariantCulture)

    type Bytestream = 
    
        { bytes: byte[]; mutable pos: int; max: int }
        
        static member of_bytes (b:byte[]) n len = 
            if n < 0 || (n+len) > b.Length then failwith "stream_of_bytes";
            { bytes = b; pos = n; max = n+len }
          
        member b.ReadByte() = 
            if b.pos >= b.max then failwith "end of stream";
            let res = int32 (b.bytes.[b.pos])
            b.pos <- b.pos + 1;
            res 
        
        member b.ReadBytes n  = 
            if b.pos + n > b.max then failwith "Bytestream.ReadBytes: end of stream";
            let res = Bytearray.sub b.bytes b.pos n
            b.pos <- b.pos + n;
            res 

        member b.ReadUtf8BytesAsString n = 
            let res = System.Text.Encoding.UTF8.GetString(b.bytes,b.pos,n)
            b.pos <- b.pos + n;
            res


open Preliminaries

//--------------------------------------------------------------------------
// RAW quotations - basic data types
//--------------------------------------------------------------------------

/// Represents specifications of a subset of F# expressions 
type 
  [<StructuralEquality(true); StructuralComparison(false)>]
  Expr =
    | CombExpr   of ExprConstInfo * Expr list
    | VarExpr    of ExprVarName
    | QuoteExpr  of Expr 
    | LambdaExpr of ExprVar * Expr 
    | HoleExpr   of Type 
    interface IFormattable 
  
and 
  [<StructuralEquality(false); StructuralComparison(false)>]
  ExprVarName = 
    { vText: string; vType: Type } 
    member x.Text = x.vText
    member x.Type = x.vType
    interface IFormattable 
    interface System.IComparable with 
        member x.CompareTo(obj:obj) = 
            match obj with 
            | :? ExprVarName as obj -> compare x.Text obj.Text
            | _ -> invalid_arg "argument obj is not an ExprVarName"
    override x.Equals(obj:obj) = 
            match obj with 
            | :? ExprVarName as obj -> x.Text = obj.Text
            | _ -> invalid_arg "argument obj is not an ExprVarName"
    override x.GetHashCode() = 
            hash x.Text

and 
  [<StructuralEquality(true); StructuralComparison(false)>]
  ExprVar = 
    { vName: ExprVarName; }
    member x.Name = x.vName
    member x.Type = x.Name.Type
    member x.Text = x.Name.Text
    static member Create(nm) = { vName=nm; }
    interface IFormattable 
     
and 
  [<StructuralEquality(true); StructuralComparison(false)>]
  ExprConstInfo = 
    | AppOp
    | CondOp  
    | TopDefOp of TopDefnData * Type list
    | LetRecOp  
    | LetRecCombOp  
    | LetOp  
    | RecdMkOp      of Type
    | RecdGetOp     of Type * string
    | RecdSetOp     of Type * string
    | SumMkOp       of Type * string
    | SumFieldGetOp of Type * string * int
    | SumTagTestOp  of Type * string
    | TupleMkOp     of Type
    | TupleGetOp    of Type * int
    | EqualityOp 
    | UnitOp   
    | BoolOp   of bool   
    | StringOp of string 
    | SingleOp of float32 
    | DoubleOp of float  
    | CharOp   of char   
    | SByteOp  of sbyte
    | ByteOp   of byte   
    | Int16Op  of int16  
    | UInt16Op of uint16 
    | Int32Op  of int32  
    | UInt32Op of uint32 
    | Int64Op  of int64  
    | UInt64Op of uint64 
    | PropGetOp of PropertyInfo
    | FieldGetOp of FieldInfo
    | CtorCallOp of ConstructorInfo 
    | MethodCallOp of MethodInfo 
    | CoerceOp of Type
    | ArrayMkOp of Type
    | DelegateOp of Type
    | GetAddrOp 
    | SeqOp 
    | EncodedForLoopOp 
    | EncodedWhileLoopOp 
    // Arbitrary spliced values - not serialized
    | LiftedValueOp of obj * Type
    interface IFormattable 
    
and 
  [<StructuralEquality(false); StructuralComparison(false)>]
  TopDefnData = 
    { vAssembly: System.Reflection.Assembly; 
      vPath: string list * string;
      vTypeScheme : (Type list -> Type) } 
    member this.Assembly = this.vAssembly
    member this.Path = this.vPath
    member this.TypeScheme = this.vTypeScheme
    
    static member Create (asm,pth,tys) = { vAssembly=asm; vPath=pth; vTypeScheme=tys; }
        
    override x.GetHashCode() = hash (x.Path, x.Assembly) 
    
    override x.Equals(obj:obj) = 
        match obj with 
        | :? TopDefnData as yt -> (x.Path = yt.Path) && (x.Assembly = yt.Assembly)
        | _ -> invalid_arg("Object is not a TopDefnData")
  
and 'a tyenv = (int -> Type) -> 'a

and 
  [<StructuralEquality(true); StructuralComparison(false)>]
  Template<'rawtupty,'rawfty> = 
    | RawTemplate of Expr
    member x.Raw = match x with RawTemplate e -> e
    interface IFormattable with 
        member x.GetLayout(env) = env.GetLayout(x.Raw)

//--------------------------------------------------------------------------
// RAW quotations 
//--------------------------------------------------------------------------

module Raw = begin

    let funTyC = (type (obj -> obj)).GetGenericTypeDefinition()  
    let templateTyC = System.Reflection.Assembly.GetExecutingAssembly().GetType("Microsoft.FSharp.Quotations.Template`4");
    let exprTyC = System.Reflection.Assembly.GetExecutingAssembly().GetType("Microsoft.FSharp.Quotations.Expr`1");
    let checkVoid a = if a = (type System.Void) then (type unit) else a
    let mkFunTy a b = 
        let (a, b) = checkVoid a, checkVoid b
        funTyC.MakeGenericType([| a;b |])

    let mkArrayTy (t:Type) = t.MakeArrayType();
    let mkTemplateTy (t:Type) = 
        let et = exprTyC.MakeGenericType([| t |])
        templateTyC.MakeGenericType([| t; (type unit); et; et |]);

    let qZeroOrMoreRLinear q inp =
        let rec queryAcc rvs e = 
            match q e with 
            | Some(v,body) -> queryAcc (v::rvs) body 
            | None -> (List.rev rvs,e) 
        queryAcc [] inp 

    let qZeroOrMoreLLinear q inp =
        let rec queryAcc e rvs = 
            match q e with 
            | Some(body,v) -> queryAcc body (v::rvs) 
            | None -> (e,rvs) 
        queryAcc inp []

    let mkRLinear mk (vs,body) = List.fold_right (fun v acc -> mk(v,acc)) vs body 
    let mkLLinear mk (body,vs) = List.fold_left (fun acc v -> mk(acc,v)) body vs 

    open Microsoft.FSharp.Text.StructuredFormat.LayoutOps

    //--------------------------------------------------------------------------
    // Active patterns for decomposing quotations
    //--------------------------------------------------------------------------

    let (|Comb0|_|) = function CombExpr(k,[])  -> Some(k) | _ -> None
    let (|Comb1|_|) = function CombExpr(k,[x]) -> Some(k,x) | _ -> None
    let (|Comb2|_|) = function CombExpr(k,[x1;x2]) -> Some(k,x1,x2) | _ -> None
    let (|Comb3|_|) = function CombExpr(k,[x1;x2;x3]) -> Some(k,x1,x2,x3) | _ -> None
    let (|CombN|_|) = function CombExpr(k,xN) -> Some(k,xN) | _ -> None
    
    let (|Var|_|)           = function VarExpr v        -> Some v     | _ -> None 
    let (|Hole|_|)          = function HoleExpr v       -> Some v     | _ -> None 
    let (|App|_|)           = function Comb2(AppOp,a,b) -> Some (a,b) | _ -> None 
    let (|Lambda|_|)        = function LambdaExpr(a,b)  -> Some (a,b) | _ -> None 
    let (|Quote|_|)         = function QuoteExpr(a)     -> Some (a)   | _ -> None 
    let (|Cond|_|)          = function Comb3(CondOp,e1,e2,e3) -> Some(e1,e2,e3) | _ -> None

    let (|AnyTopDefnUse|_|) = function Comb0(TopDefOp(td,tyargs)) -> Some(td,tyargs) | _ -> None
    let (|Tuple|_|)         = function CombN(TupleMkOp(ty),e) -> Some(ty,e) | _ -> None

    let (|Recd|_|)          = function CombN(RecdMkOp(x),es) -> Some(x,es) | _ -> None
    let (|RecdGet|_|)       = function Comb1(RecdGetOp(x,y),e) -> Some(x,y,e) | _ -> None
    let (|RecdSet|_|)       = function Comb2(RecdSetOp(x,y),e1,e2) -> Some(x,y,e1,e2) | _ -> None
    let (|Sum|_|)           = function CombN(SumMkOp(x,y),es) -> Some(x,y,es) | _ -> None
    let (|SumFieldGet|_|)   = function Comb1(SumFieldGetOp(x,y,z),e) -> Some(x,y,z,e) | _ -> None
    let (|SumTagTest|_|)    = function Comb1(SumTagTestOp(x,y),e) -> Some(x,y,e) | _ -> None
    let (|TupleGet|_|)      = function Comb1(TupleGetOp(ty,n),e) -> Some(ty,n,e) | _ -> None
    let (|Equality|_|)      = function Comb2(EqualityOp,e1,e2) -> Some(e1,e2) | _ -> None
    let (|Coerce|_|)        = function Comb1(CoerceOp ty,e1) -> Some(ty,e1) | _ -> None
    let (|NewArray|_|)      = function CombN(ArrayMkOp ty,es) -> Some(ty,es) | _ -> None
    let (|Bool|_|)          = function Comb0(BoolOp x) -> Some(x) | _ -> None
    let (|String|_|)        = function Comb0(StringOp x) -> Some(x) | _ -> None
    let (|Single|_|)        = function Comb0(SingleOp x) -> Some(x) | _ -> None
    let (|Double|_|)        = function Comb0(DoubleOp x) -> Some(x) | _ -> None
    let (|Char|_|)          = function Comb0(CharOp x) -> Some(x) | _ -> None
    let (|SByte|_|)         = function Comb0(SByteOp x) -> Some(x) | _ -> None
    let (|Byte|_|)          = function Comb0(ByteOp x) -> Some(x) | _ -> None
    let (|Int16|_|)         = function Comb0(Int16Op x) -> Some(x) | _ -> None
    let (|UInt16|_|)        = function Comb0(UInt16Op x) -> Some(x) | _ -> None
    let (|Int32|_|)         = function Comb0(Int32Op x) -> Some(x) | _ -> None
    let (|UInt32|_|)        = function Comb0(UInt32Op x) -> Some(x) | _ -> None
    let (|Int64|_|)         = function Comb0(Int64Op x) -> Some(x) | _ -> None
    let (|UInt64|_|)        = function Comb0(UInt64Op x) -> Some(x) | _ -> None

    let (|GetAddr|_|)       = function Comb1(GetAddrOp,e) -> Some(e) | _ -> None
    let (|Seq|_|)           = function Comb2(SeqOp,e1,e2) -> Some(e1,e2) | _ -> None
    let (|EncodedForLoop|_|)       = function Comb3(EncodedForLoopOp,e1,e2,e3) -> Some(e1,e2,e3) | _ -> None
    let (|EncodedWhileLoop|_|)     = function Comb2(EncodedWhileLoopOp,e1,e2) -> Some(e1,e2) | _ -> None
    let (|NewDelegate|_|)   = function Comb1(DelegateOp ty,e) -> Some(ty,e) | _ -> None
    let (|PropGet|_|)       = function Comb1(PropGetOp ty,e) -> Some(ty,e) | _ -> None
    let (|FieldGet|_|)      = function Comb1(FieldGetOp ty,e) -> Some(ty,e) | _ -> None
    let (|CtorCall|_|)      = function CombN(CtorCallOp ty,e) -> Some(ty,e) | _ -> None
    let (|MethodCall|_|)    = function CombN(MethodCallOp ty,e) -> Some(ty,e) | _ -> None
    let (|LiftedValue|_|)   = function Comb0(LiftedValueOp (v,ty)) -> Some(v,ty) | _ -> None
    let (|Unit|_|)          = function Comb0(UnitOp) -> Some() | _ -> None
    let (|LetRaw|_|)        = function Comb2(LetOp,e1,e2) -> Some(e1,e2) | _ -> None
    let (|LetRecRaw|_|)     = function Comb2(LetRecOp,e1,e2) -> Some(e1,e2) | _ -> None
    let (|LetRecCombRaw|_|) = function CombN(LetRecCombOp,es) -> Some(es) | _ -> None
    let (|Lambdas|) e = qZeroOrMoreRLinear (|Lambda|_|) e
    let (|Apps|) e = qZeroOrMoreLLinear (|App|_|) e

    let (|Let|_|) = function LetRaw(e,Lambda(v,body)) -> Some((v,e),body) | _ -> None

    let (|LetRec|_|) e = 
        match e with 
        | LetRecRaw(Lambdas(vs1,LetRecCombRaw(es)),Lambdas(vs2,body)) -> Some(List.combine vs1 es,body)
        | _ -> None
    
    //--------------------------------------------------------------------------
    // Getting the type of Raw quotations
    //--------------------------------------------------------------------------

    open Microsoft.FSharp.Reflection
    
    let swap (a,b) = (b,a)
    let getSome  f expr = match (f expr) with | Some o -> o | _ -> failwith "internal error";
    
    /// Returns type of lambda applciation - something like "(fun a -> ..) b"
    let rec typeOfAppliedLambda (f, v) =
        let fty = ((typeOf f):Type) 
        match fty.GetGenericArguments() with 
        | [|a; b|] -> b
        | _ -> failwith "ill formed expression: AppOp or LetOp"          

    /// Returns type of "Comb" node in the quotation
    and typeOfConst c expr = 
        match c with 
        | AppOp               -> expr |> getSome (|App|_|) |> typeOfAppliedLambda 
        | LetOp               -> expr |> getSome (|LetRaw|_|) |> swap |> typeOfAppliedLambda
        | CondOp              -> let (_,t,_) = getSome (|Cond|_|) expr in (typeOf t)
        | TopDefOp (td,tl)    -> td.TypeScheme tl
        | LetRecOp            -> 
            let (ves, body) = getSome (|LetRec|_|) expr
            (typeOf body)
        | LetRecCombOp        -> failwith "typeOfConst: LetRecCombOp"
        | RecdMkOp ty         -> ty
        | RecdGetOp (ty,str)  -> 
            match Type.GetInfo(ty) with
            | RecordType mems -> let (_, ty) = (mems |> List.find (fun (n, ty) -> n = str)) in ty
            | _ -> failwith "ill formed expression: RecdGetOp"
        | RecdSetOp (ty,n)        -> (type Unit)
        | SumMkOp (ty,str)        -> ty
        | SumFieldGetOp (t,str,i) -> 
            match Type.GetInfo(t) with
            | SumType(mems) -> 
                let (_, itlst) = (mems |> List.find (fun (n, _) -> n = str)) 
                let (_, ty) = List.nth itlst i 
                ty
            | _ -> failwith "ill formed expression: SumFieldGetOp"
        | SumTagTestOp (ty,str)   -> (type Boolean)
        | LiftedValueOp (o, ty)   -> ty
        | TupleGetOp (ty,i) -> 
            match Type.GetInfo(ty) with
            | TupleType mems -> List.nth mems i
            | _ -> failwith "ill formed expression: TupleGetOp"
        | TupleMkOp ty      -> ty
        | EqualityOp        -> (type Boolean)
        | BoolOp   a        -> (type Boolean)
        | StringOp a        -> (type String)
        | SingleOp a        -> (type Single)
        | DoubleOp a        -> (type Double)
        | CharOp   a        -> (type Char)
        | SByteOp  a        -> (type SByte)
        | ByteOp   a        -> (type Byte)
        | Int16Op  a        -> (type Int16)
        | UInt16Op a        -> (type UInt16)
        | Int32Op  a        -> (type Int32)
        | UInt32Op a        -> (type UInt32)
        | Int64Op  a        -> (type Int64)
        | UInt64Op a        -> (type UInt64)
        | UnitOp            -> (type Unit)
        | PropGetOp prop    -> prop.PropertyType
        | FieldGetOp fld    -> fld.FieldType
        | CtorCallOp ctor   -> ctor.DeclaringType
        | MethodCallOp mi   -> mi.ReturnType
        | GetAddrOp         -> failwith "typeOfConst: GetAddrOp"
        | CoerceOp ty       -> ty
        | SeqOp             -> let (a,b) = getSome (|Seq|_|) expr in (typeOf b) // warning if 'a' <> unit
        | EncodedForLoopOp  -> (type Unit)
        | ArrayMkOp ty      -> mkArrayTy ty
        | DelegateOp ty     -> ty
        | EncodedWhileLoopOp-> (type Unit)
    
    /// Returns type of the Raw quotation or fails if the quotation is ill formed
    /// if 'verify' is true, verifies all branches, otherwise ignores some of them when not needed
    and typeOf e = 
        match e with 
        | VarExpr    v        -> v.Type
        | LambdaExpr (v,b)    -> mkFunTy (v.Type) (typeOf b)
        | QuoteExpr  v        -> mkTemplateTy (typeOf v)
        | HoleExpr   ty       -> ty
        | CombExpr   (c,args) -> typeOfConst c e

    //--------------------------------------------------------------------------
    // Constructors for building Raw quotations
    //--------------------------------------------------------------------------
      
    let mkFEE op = CombExpr(op,[])
    let mkFEN op l = CombExpr(op,l)
    let mkFE1 op x = CombExpr(op,[x])  
    let mkFE2 op (x,y) = CombExpr(op,[x;y])  
    let mkFE3 op (x,y,z) = CombExpr(op,[x;y;z])  
    let mkOp v () = v

    // internal
    let MkLetRaw v = mkFE2 LetOp v
    let MkLetRecRaw v = mkFE2 LetRecOp v
    let MkLetRecCombRaw v = mkFEN LetRecCombOp v
    let MkAnyTopDefnUse (td,tyargs) = mkFEE (TopDefOp (td,tyargs)) 

    let (|GenericTopDefnUse|_|) tm = 
        match tm with 
        | AnyTopDefnUse(td1,_) -> 
            (function (AnyTopDefnUse(td2,args)) when td1 = td2 -> Some(args) | _ -> None)
        | _ ->  failwith "(|GenericTopDefnUse|) - not a TopDef"

    let MkGenericTopDefnUse tm = 
        match tm with 
        | AnyTopDefnUse(td,tyargs) -> 
            (function tyargs -> MkAnyTopDefnUse(td,tyargs)) 
        | _ ->  failwith "MkGenericTopDefnUse - not a TopDef"

    module Unchecked = 
        let MkVar v       = VarExpr v 
        let MkHole v      = HoleExpr v 
        let MkApp(a,b)    = mkFE2 AppOp (a,b) 
        let MkLambda(a,b) = LambdaExpr (a,b) 
        let MkQuote(a)    = QuoteExpr (a) 
              
        let MkUnit     () = mkFEE UnitOp
            
        let MkTuple    (ty,args) = mkFEN (TupleMkOp ty) args 
        let MkRecd     (x,y) = mkFEN (RecdMkOp x) y
        let MkRecdGet  (x,y,z) = mkFE1 (RecdGetOp (x,y)) z
        let MkRecdSet  (x,y,z1,z2) = mkFE2 (RecdSetOp (x,y)) (z1,z2)
        let MkSum    (x,y,z) = mkFEN (SumMkOp (x,y)) z
        let MkSumFieldGet (x,y,z1,z2) = mkFE1 (SumFieldGetOp (x,y,z1)) z2
        let MkSumTagTest  (x,y,z) = mkFE1 (SumTagTestOp (x,y)) z
        let MkTupleGet   (ty,n,x) = mkFE1 (TupleGetOp (ty,n)) x

        let MkBool        v = mkFEE (BoolOp v)
        let MkString      v = mkFEE (StringOp v)
        let MkSingle      v = mkFEE (SingleOp v)
        let MkDouble      v = mkFEE (DoubleOp v)
        let MkChar        v = mkFEE (CharOp v)
        let MkSByte       v = mkFEE (SByteOp v)
        let MkByte        v = mkFEE (ByteOp v)
        let MkInt16       v = mkFEE (Int16Op v)
        let MkUInt16      v = mkFEE (UInt16Op v)
        let MkInt32       v = mkFEE (Int32Op v)
        let MkUInt32      v = mkFEE (UInt32Op v)
        let MkInt64       v = mkFEE (Int64Op v)
        let MkUInt64      n = mkFEE (UInt64Op n) 
        let MkGetAddr     v = mkFE1 (GetAddrOp) v
        let MkSeq         v = mkFE2 (SeqOp) v
        let MkEncodedForLoop     v = mkFE3 (EncodedForLoopOp) v
        let MkEncodedWhileLoop   v = mkFE2 (EncodedWhileLoopOp) v
        let MkNewDelegate (ty,e) = mkFE1 (DelegateOp ty) e
        let MkPropGet (v,xs) = mkFE1 (PropGetOp v) xs
        
        let MkFieldGet    (v,xs) = mkFE1 (FieldGetOp v) xs
        let MkCtorCall    (v,xs) = mkFEN (CtorCallOp v) xs
        let MkMethodCall  (v,xs) = mkFEN (MethodCallOp v) xs
        let MkLiftedValue (v,ty) = mkFEE (LiftedValueOp(v,ty))
        
        let MkCond     x = mkFE3 CondOp x 
        let MkEquality x = mkFE2 (EqualityOp) x 
        let MkCoerce   (ty,x) = mkFE1 (CoerceOp ty) x
        let MkNewArray  (ty,x) = mkFEN (ArrayMkOp ty) x

        let MkLet ((v,e),b) = MkLetRaw (e,MkLambda (v,b))
        
        let MkLambdas(vs,es) = mkRLinear MkLambda (vs,es)
        let MkApps(f,es) = mkLLinear MkApp (f,es)
        
        let MkLetRec (ves:(ExprVar*Expr) list,body) =
            let vs,es = List.split ves 
            let vtys = vs |> List.map (fun v -> v.Type) 
            MkLetRecRaw(MkLambdas (vs, MkLetRecCombRaw es),
                        MkLambdas (vs, body))
                        
        let MkGenericTopDefnApp x (tyargs,args) = MkApps(MkGenericTopDefnUse x tyargs, args)
        let (|GenericTopDefnApp|_|) x = function Apps(GenericTopDefnUse(x)(tyargs),args) -> Some(tyargs,args)  | _ -> None
    
    //--------------------------------------------------------------------------
    // Type-checked constructors for building Raw quotations
    //--------------------------------------------------------------------------
  
    // t2 is inherited from t1 / t2 implements interface t1 or t2 == t1
    let assignableFrom (t1:Type) (t2:Type) =  
        t1.IsAssignableFrom(t2)
      
    let checkTypes (expectedType: Type) (receivedType : Type)  msg1 msg2 =
        if (expectedType <> receivedType) then 
          failwithf "type mismatch when building '%s': %s. Expected '%A', got type '%A'" msg1 msg2 expectedType receivedType

    let checkTypesWeak (expectedType: Type) (receivedType : Type)  msg1 msg2 = 
        if (not (assignableFrom expectedType receivedType)) then 
          failwithf "type mismatch when building '%s': %s. Expected '%A', got type '%A'" msg1 msg2 expectedType receivedType
  
    let genericError msg1 msg2 =
        failwithf "error when building '%s': %s." msg1 msg2 
      
    // Checks lambda application for correctnes
    let checkAppliedLambda (f, v) =
        let fty = typeOf f
        let ftyG = fty.GetGenericTypeDefinition() 
        checkTypes funTyC ftyG "app" "expected function type in function application or let binding"
        let vty = (typeOf v)
        match fty.GetGenericArguments() with 
          | [|a; b|] -> checkTypes vty a "app" "function argument type doesn't match"
          | _ -> genericError "app" "invalid function type"  
  
    // Returns tuple members for specified tuple type
    let getTupleType ty =
        match Type.GetInfo(ty) with
        | TupleType mems -> mems
        | _ -> genericError "tuple" "non-tuple type used where tuple expected"
  
    // Returns record members
    let getRecordType ty =
        match Type.GetInfo(ty) with
        | RecordType mems -> mems
        | _ -> genericError "record" "non-record type used where record expected"      

    // Returns record member specified by name
    let getRecordMember ty n =    
        let mems = getRecordType ty
        match (Seq.tryfind (fst >> ((=) n)) mems) with
        | Some (m) -> m
        | _ -> genericError "record" "couldn't find record member with specified name"

    // Returns option (by name) of a Sum type
    let getSumOption ty n expr =       
      match expr with 
        | Some e -> checkTypes ty (typeOf e) "sum" "expression is not a union type"      
        | _ -> ();
      match Type.GetInfo(ty) with
        | SumType(mems) -> 
            match (Seq.tryfind (fst >> ((=) n)) mems) with 
              | Some (_,e) -> e
              | None -> genericError "sum" "union doesn't contain required field"
        | _ -> genericError "sum" "type is not a union type"
  
    let checkBind(v:ExprVar,e) = 
        let ety = typeOf e
        checkTypes v.Type ety "let" "the variable type doesn't match the type of the rhs of a let binding"
  
    // [Correct by definition]
    let MkVar v       = VarExpr v 
    let MkHole v      = HoleExpr v 
    let MkQuote(a)    = QuoteExpr (a) 
          
    let MkUnit       () = mkFEE UnitOp
    let MkBool        v = mkFEE (BoolOp v)
    let MkString      v = mkFEE (StringOp v)
    let MkSingle      v = mkFEE (SingleOp v)
    let MkDouble      v = mkFEE (DoubleOp v)
    let MkChar        v = mkFEE (CharOp v)
    let MkSByte       v = mkFEE (SByteOp v)
    let MkByte        v = mkFEE (ByteOp v)
    let MkInt16       v = mkFEE (Int16Op v)
    let MkUInt16      v = mkFEE (UInt16Op v)
    let MkInt32       v = mkFEE (Int32Op v)
    let MkUInt32      v = mkFEE (UInt32Op v)
    let MkInt64       v = mkFEE (Int64Op v)
    let MkUInt64      n = mkFEE (UInt64Op n) 
    let MkGetAddr     v = mkFE1 (GetAddrOp) v
    let MkSeq         v = mkFE2 (SeqOp) v // "a; b" - warning when a.Type <> unit
    
    let MkCoerce      (ty,x) = mkFE1 (CoerceOp ty) x
    let MkNull        (ty)   = mkFEE (LiftedValueOp(null,ty))
    let MkLiftedValue (v,ty) = mkFEE (LiftedValueOp(v,ty))
    
    let MkLambda(var,body) = LambdaExpr(var,body)       
    // [/Correct by definition]
    
    
    // [Type checked]
    let MkApp v = checkAppliedLambda v; mkFE2 AppOp v 

    // Tuples
    let MkTuple    (ty,args:Expr list) = 
        let mems = getTupleType ty
        if (args.Length <> mems.Length) then genericError "tupe" "incompatible tuple length"
        List.iter2 (fun mt a -> checkTypes mt (typeOf a) "tuple" "Mismatching type of argument and tuple element." ) mems args
        mkFEN (TupleMkOp ty) args 
    
    let MkTupleGet (ty,n,x) = 
        checkTypes ty (typeOf x) "tupleGet" "expression doesn't match the tuple type"  
        let mems = getTupleType ty
        if (mems.Length <= n) then genericError "tupleGet" "tuple access out of range" else mkFE1 (TupleGetOp (ty,n)) x
    
    // Records
    let MkRecd (ty,args:list<Expr>) = 
        let mems = getRecordType ty
        if (args.Length <> mems.Length) then genericError "recd" "incompatible record length"
        List.iter2 (fun (_, mt) a -> checkTypes mt (typeOf a) "recd" "incorrect argument type for a record") mems args
        mkFEN (RecdMkOp ty) args
      
    let MkRecdGet (ty,n,expr) =
        checkTypes ty (typeOf expr) "recdGet" "expression doesn't match the record type"  
        (getRecordMember ty n) |> ignore
        mkFE1 (RecdGetOp (ty,n)) expr
      
    let MkRecdSet  (ty,n,expr,v) = 
        checkTypes ty (typeOf expr) "recdSet" "expression doesn't match the record type"  
        let (_, mty) = getRecordMember ty n
        checkTypes mty (typeOf v) "recdSet" "invalid type in record member assignment"
        mkFE2 (RecdSetOp (ty,n)) (expr,v)
      
    // Discriminated unions        
    let MkSum (ty,n,args) = 
        let sargs = getSumOption ty n None;
        if ((List.length args) <> sargs.Length) then genericError "sum" "union type requires different numer of arguments"
        List.iter2 (fun (_, mt) a  -> checkTypes mt (typeOf a) "sum" "incorrect argument type for a union") sargs args
        mkFEN (SumMkOp (ty,n)) args
        
    let MkSumFieldGet (ty,n,idx,expr) = 
        let sargs = getSumOption ty n (Some expr);
        if (sargs.Length <= idx) then genericError "sumFieldGet" "union access out of range" 
        mkFE1 (SumFieldGetOp (ty,n,idx)) expr
      
    let MkSumTagTest (ty,n,expr) = 
        getSumOption ty n (Some expr) |> ignore
        mkFE1 (SumTagTestOp (ty,n)) expr

    // Conditional etc..
    let MkCond (e,t,f) = 
        checkTypes (typeOf t) (typeOf f) "cond" "types of true and false branches differ"
        checkTypes (typeof<Boolean>) (typeOf e) "cond" "condition expression must be of type bool"
        mkFE3 CondOp (e,t,f)               
        
    let MkEquality (l,r) = 
        checkTypes (typeOf l) (typeOf r) "equality" "argument types differ" 
        mkFE2 (EqualityOp) (l,r)
    
    let MkNewArray (ty,args) = 
        List.iter (fun a -> checkTypes ty (typeOf a) "newArray" "initializer doesn't match array type") args
        mkFEN (ArrayMkOp ty) args
        
    let MkFieldGet    (fi:FieldInfo,inst) =
        let intyp = (typeOf inst)
        if (not (assignableFrom fi.DeclaringType intyp)) then
          genericError "fieldGet" "incorrect instance type when reading field"
        mkFE1 (FieldGetOp fi) inst
      
    let MkCtorCall (ci:ConstructorInfo,args:list<Expr>) =
        let ap = ci.GetParameters()
        if (ap.Length <> args.Length) then failwith "type mismatch: Incorrect number of arguments for a constructor"
        List.iter2 ( fun (p:ParameterInfo) a -> checkTypes p.ParameterType (typeOf a) "ctorCall" "invalid argument to a constructor call" ) (ap |> Array.to_list) args
        mkFEN (CtorCallOp ci) args

    let MkPropGet (pi:PropertyInfo,arg) = 
        if (not pi.CanRead) then genericError "propGet" "reading set-only property"
        if (pi.GetGetMethod().GetParameters().Length > 0) then genericError "propGet" "can't use MkPropGet for indexers, use method call instead"
        if (assignableFrom pi.DeclaringType (typeOf arg)) then
          mkFE1 (PropGetOp pi) arg
        else
          genericError "propGet" "incorrect instance type when reading property"
      
    let MkMethodCall (mi:MethodInfo,allargs) =
        let ap = mi.GetParameters()
        let args = 
          if (mi.IsStatic) then allargs else
            match allargs with
              | inst::other when (assignableFrom mi.DeclaringType (typeOf inst)) -> other
              | _ -> genericError "methodCall" "invalid instance argument in a method call"
        if (ap.Length <> args.Length) then failwith "type mismatch: Incorrect number of arguments for a method"
        List.iter2 ( fun (p:ParameterInfo) a -> checkTypesWeak p.ParameterType (typeOf a) "methodCall" "invalid parameter for a method") (ap |> Array.to_list) args
                                                // todo: shouldn't this be "strong" type check? sometimes?
        mkFEN (MethodCallOp mi) allargs      
    
    let MkEncodedForLoop (ef,et,body) = 
        checkTypes (type int) (typeOf ef) "for" "lower bound variable must be an integer"
        checkTypes (type int) (typeOf et) "for" "upper bound variable must be an integer"
        match body with 
          | Lambda(v, _) -> checkTypes (type int) (v.Type) "for" "body of the for loop must be lambda taking integer as an argument"
          | _ -> genericError "for" "Body of the for loop must be lambda"
        mkFE3 (EncodedForLoopOp) (ef, et, body)
      
    let MkEncodedWhileLoop (e,b) = 
        checkTypes (type bool) (typeOf e) "while" "condition must return boolean"
        mkFE2 (EncodedWhileLoopOp) (e,b)
    
    let MkNewDelegate (ty,e) = 
        let dlfun =
            match Type.GetInfo(ty) with
            | DelegateType (res, ar) -> List.fold_right mkFunTy res ar 
            | _ -> genericError "newDelegate" "Expecting delegate type"
        checkTypes (dlfun) (typeOf e) "newDelegate" "Function type doesn't match delegate type."
        mkFE1 (DelegateOp ty) e
    
    let MkLet (((v,e) as bind),b) = 
        checkBind bind;
        MkLetRaw (e,MkLambda (v,b))

    // [Expressed using correct]
    let MkLambdas(vs,es) = mkRLinear MkLambda (vs,es)
    let MkApps(f,es) = mkLLinear MkApp (f,es)
    // [/Expressed using correct]
    
    let MkLetRec (ves:(ExprVar*Expr) list,body) = 
        List.iter checkBind ves;
        let vs,es = List.split ves 
        let vtys = vs |> List.map (fun v -> v.Type) 
        MkLetRecRaw(MkLambdas (vs, MkLetRecCombRaw es),
                    MkLambdas (vs, body))

    //--------------------------------------------------------------------------
    // Further expression families
    //--------------------------------------------------------------------------
    
    /// Reverse the compilation of And and Or
    let (|LazyAnd|_|) x = 
        match x with 
        | Cond(x,y,Bool(false)) -> Some(x,y)
        | _ -> None
        
    let (|LazyOr|_|) x = 
        match x with 
        | Cond(x,Bool(true),y) -> Some(x,y)
        | _ -> None

    let (|BetaReducible|_|) x = 
      match x with 
      | Let((v,e),b) -> Some((v,e),b)
      | App(Lambda(v,b),e) -> Some((v,e),b)
      | _ -> None

    let MkGenericTopDefnApp x (tyargs,args) = MkApps(MkGenericTopDefnUse x tyargs, args)
    let (|GenericTopDefnApp|_|) x = function Apps(GenericTopDefnUse(x)(tyargs),args) -> Some(tyargs,args)  | _ -> None
    let (|TopDefnApp|_|) x = function GenericTopDefnApp (x) (tyargs,args) -> Some(args)  | _ -> None

    //--------------------------------------------------------------------------
    // Full decomposition
    //--------------------------------------------------------------------------
    
    /// Full decomposition
    module BindingStructure = 
        let (|Var|Lambda|ConstApp|Hole|Quote|) e = 
          match e with 
          | VarExpr v       -> Var(v)
          | LambdaExpr(v,b) -> Lambda(v,b)
          | CombExpr(op,args) -> ConstApp(op,args)
          | HoleExpr(a)     -> Hole(a)
          | QuoteExpr(a)    -> Quote(a)

        let gt1A n args = match args with [a1] -> (a1) | _ -> genericError n "invalid number of arguments; expected 1 arguments"
        let gt2A n args = match args with [a1;a2] -> (a1,a2) | _ -> genericError n "invalid number of arguments; expected 2 arguments"
        let gt3A n args = match args with [a1;a2;a3] -> (a1,a2,a3) | _ -> genericError n "invalid number of arguments; expected 3 arguments"

        let mk1C1A n ct a args  = let (a1) = gt1A n args in ct(a,a1)
        let mk2C1A n ct a b args  = let (a1) = gt1A n args in ct(a,b,a1)
        let mk2C2A n ct a b args  = let (a1,a2) = gt2A n args in ct(a,b,a1,a2)      
        let mk3C1A n ct a b c args  = let (a1) = gt1A n args in ct(a,b,c,a1)
        let mk1A n ct args        = let (a1) = gt1A n args in ct(a1)
        let mk2A n ct args        = let (a1,a2) = gt2A n args in ct(a1,a2)
        let mk3A n ct args        = let (a1,a2,a3) = gt3A n args in ct(a1,a2,a3)
          
        let MkConstApp(a,args) =  
            match a with 
            | AppOp             -> mk2A "app" MkApp args
            | CondOp            -> mk3A "cond" MkCond args
            | TopDefOp (td,tl)  -> MkAnyTopDefnUse(td,tl)           // TODO: is the number of type arguments correct?
            | LetRecOp          -> mk2A "letRec" MkLetRecRaw args   // TODO: correct / unchecked?
            | LetRecCombOp      -> MkLetRecCombRaw(args)            // TODO: correct / unchecked?
            | LetOp             -> mk2A "letRaw" MkLetRaw args      // TODO: correct / unchecked?
            | RecdMkOp (ty)     -> MkRecd(ty, args)
            | RecdGetOp (ty,s)  -> mk2C1A "recdGet" MkRecdGet ty s args
            | RecdSetOp (ty,s)  -> mk2C2A "recdSet" MkRecdSet ty s args
            | SumMkOp (ty,s)    -> MkSum(ty, s, args)
            | SumFieldGetOp (ty,s,i)  -> mk3C1A "sumFieldGet" MkSumFieldGet ty s i args
            | SumTagTestOp (ty,s)     -> mk2C1A "sumTagTest" MkSumTagTest ty s args
            | TupleMkOp (ty)    -> MkTuple(ty, args)
            | TupleGetOp (ty,i) -> mk2C1A "tupleGet" MkTupleGet ty i args
            | EqualityOp        -> mk2A "equality" MkEquality args
            | UnitOp            -> MkUnit()
            | BoolOp (v)        -> MkBool v
            | StringOp (v)      -> MkString v
            | SingleOp (v)      -> MkSingle v
            | DoubleOp (v)      -> MkDouble v
            | CharOp (v)        -> MkChar v
            | SByteOp (v)       -> MkSByte v
            | ByteOp (v)        -> MkByte v
            | Int16Op (v)       -> MkInt16 v
            | UInt16Op (v)      -> MkUInt16 v
            | Int32Op (v)       -> MkInt32 v
            | UInt32Op (v)      -> MkUInt32 v
            | Int64Op (v)       -> MkInt64 v
            | UInt64Op (v)      -> MkUInt64 v
            (* Object Model Goop! *)
            | PropGetOp (mi)    -> mk1C1A "propGet" MkPropGet mi args
            | FieldGetOp (mi)   -> mk1C1A "fieldGet" MkFieldGet mi args
            | CtorCallOp (mi)   -> MkCtorCall(mi,args)
            | MethodCallOp (mi) -> MkMethodCall(mi,args)
            | CoerceOp (ty)     -> mk1C1A "coerce" MkCoerce ty args
            | ArrayMkOp (ty)    -> MkNewArray(ty,args)
            | DelegateOp (ty)   -> mk1C1A "newDelegate" MkNewDelegate ty args
            (* Unclean/imperative! *)
            | GetAddrOp         -> mk1A "getAddr" MkGetAddr args
            | SeqOp             -> mk2A "seq" MkSeq args
            | EncodedForLoopOp      -> mk3A "for" MkEncodedForLoop args
            | EncodedWhileLoopOp    -> mk2A "while" MkEncodedWhileLoop args
            // Arbitrary spliced values - not serialized
            | LiftedValueOp (v,ty)  -> MkLiftedValue(v,ty)
            
        module Unchecked = 
            open Unchecked
            
            let MkConstApp(a,args) =  
              match a with 
              | AppOp             -> mk2A "app" MkApp args
              | CondOp            -> mk3A "cond" MkCond args
              | TopDefOp (td,tl)  -> MkAnyTopDefnUse(td,tl)           // TODO: is the number of type arguments correct?
              | LetRecOp          -> mk2A "letRec" MkLetRecRaw args   // TODO: correct / unchecked?
              | LetRecCombOp      -> MkLetRecCombRaw(args)            // TODO: correct / unchecked?
              | LetOp             -> mk2A "letRaw" MkLetRaw args      // TODO: correct / unchecked?
              | RecdMkOp (ty)     -> MkRecd(ty, args)
              | RecdGetOp (ty,s)  -> mk2C1A "recdGet" MkRecdGet ty s args
              | RecdSetOp (ty,s)  -> mk2C2A "recdSet" MkRecdSet ty s args
              | SumMkOp (ty,s)    -> MkSum(ty, s, args)
              | SumFieldGetOp (ty,s,i)  -> mk3C1A "sumFieldGet" MkSumFieldGet ty s i args
              | SumTagTestOp (ty,s)     -> mk2C1A "sumTagTest" MkSumTagTest ty s args
              | TupleMkOp (ty)    -> MkTuple(ty, args)
              | TupleGetOp (ty,i) -> mk2C1A "tupleGet" MkTupleGet ty i args
              | EqualityOp        -> mk2A "equality" MkEquality args
              | UnitOp            -> MkUnit()
              | BoolOp (v)        -> MkBool v
              | StringOp (v)      -> MkString v
              | SingleOp (v)      -> MkSingle v
              | DoubleOp (v)      -> MkDouble v
              | CharOp (v)        -> MkChar v
              | SByteOp (v)       -> MkSByte v
              | ByteOp (v)        -> MkByte v
              | Int16Op (v)       -> MkInt16 v
              | UInt16Op (v)      -> MkUInt16 v
              | Int32Op (v)       -> MkInt32 v
              | UInt32Op (v)      -> MkUInt32 v
              | Int64Op (v)       -> MkInt64 v
              | UInt64Op (v)      -> MkUInt64 v
              (* Object Model Goop! *)
              | PropGetOp (mi)    -> mk1C1A "propGet" MkPropGet mi args
              | FieldGetOp (mi)   -> mk1C1A "fieldGet" MkFieldGet mi args
              | CtorCallOp (mi)   -> MkCtorCall(mi,args)
              | MethodCallOp (mi) -> MkMethodCall(mi,args)
              | CoerceOp (ty)     -> mk1C1A "coerce" MkCoerce ty args
              | ArrayMkOp (ty)    -> MkNewArray(ty,args)
              | DelegateOp (ty)   -> mk1C1A "newDelegate" MkNewDelegate ty args
              (* Unclean/imperative! *)
              | GetAddrOp         -> mk1A "getAddr" MkGetAddr args
              | SeqOp             -> mk2A "seq" MkSeq args
              | EncodedForLoopOp      -> mk3A "for" MkEncodedForLoop args
              | EncodedWhileLoopOp    -> mk2A "while" MkEncodedWhileLoop args
              // Arbitrary spliced values - not serialized
              | LiftedValueOp (v,ty)  -> MkLiftedValue(v,ty)
    
    //--------------------------------------------------------------------------
    // Pickle/unpickle expression and type specifications
    //--------------------------------------------------------------------------

    let PickledDefinitionsResourceNameBase = "PickledDefinitions"

    let freshExprVarName = 
      let i = ref 0 
      fun n ty -> 
        lock i (fun () -> i := !i + 1; {vText=(n+"."+string_of_int (!i)); vType=ty}) 

    //-------------------------------------------------------------------------
    // General Method Binder

    let typeEquals     (s:Type) (t:Type) = s.Equals(t)
    let typesEqual ss tt =
      (length ss = length tt) && List.for_all2 typeEquals ss tt

    let staticOrInstance = BindingFlags.Instance ||| BindingFlags.Static ||| BindingFlags.Public ||| BindingFlags.NonPublic ||| BindingFlags.DeclaredOnly
    let instanceBindingFlags = BindingFlags.Instance ||| BindingFlags.Public ||| BindingFlags.NonPublic ||| BindingFlags.DeclaredOnly

    let bindMethodBySearch convType (parentT: Type) (nm,marity,argtys,rty) =
        let methInfos = parentT.GetMethods(staticOrInstance) |> Array.to_list 
          (* First, filter on name, if unique, then binding "done" *)
        let tyargTs = if parentT .IsGenericType then parentT .GetGenericArguments()      else [| |] 
        let selectName (methInfo:MethodInfo) = (methInfo.Name = nm) 
        let methInfos = filter selectName methInfos 
        match methInfos with 
        | [methInfo] -> 
            methInfo
        | _ ->
          (* Second, type match. Note type erased (non-generic) F# code would not type match but they have unique names *)
            let select (methInfo:MethodInfo) =
                let tyargTIs = if parentT.IsGenericType then parentT.GetGenericArguments() else [| |] 
                // mref implied Types 
                let mtyargTIs = if methInfo.IsGenericMethod then methInfo.GetGenericArguments() else [| |] 
                if Array.length mtyargTIs  <> marity then false (* method generic arity mismatch *) else
                let argTs,resT = 
                    let emEnv = (Array.to_list tyargTs) @ (Array.to_list mtyargTIs) 
                    let argTs = map (convType emEnv) argtys 
                    let resT  = convType emEnv rty 
                    argTs,resT 
                
                // methInfo implied Types 
                let haveArgTs = 
                    let parameters = Array.to_list (methInfo.GetParameters()) 
                    parameters |> map (fun param -> param.ParameterType) 
                let haveResT  = methInfo.ReturnType 
                // check for match 
                if length argTs <> length haveArgTs then false (* method argument length mismatch *) else
                let res = typesEqual (resT::argTs) (haveResT::haveArgTs) 
                res
            match List.tryfind select methInfos with
            | None          -> failwith "convMethodRef: could not bind to method"
            | Some methInfo -> methInfo (* return MethodInfo for (generic) type's (generic) method *)

    let bindMethod convType (parentT: Type)  ((nm,marity,argtys,rty) as mref) =
      if parentT = null then invalid_arg "parentT";
      if marity = 0 then 
          let tyargTs = if parentT.IsGenericType then parentT.GetGenericArguments() else [| |] 
          let argTs,resT = 
              let emEnv = Array.to_list tyargTs 
              let argTs = Array.of_list (map (convType emEnv) argtys) 
              let resT  = convType emEnv rty 
              argTs,resT 
          let methInfo = 
              try 
                 match parentT.GetMethod(nm,staticOrInstance,null,argTs,null) with 
                 | null -> None
                 | res -> Some(res)
               with :? AmbiguousMatchException -> None 
          match methInfo with 
          | Some methInfo when (typeEquals resT methInfo.ReturnType) -> methInfo
          | _ -> bindMethodBySearch convType parentT mref
      else 
          bindMethodBySearch convType parentT mref

    let MkNamedType (tc:Type,tyargs)  =
        match  tyargs with 
        | [] -> tc
        | _ -> tc.MakeGenericType(Array.of_list tyargs)

    let nonNull x y = match box y with null -> raise (new NullReferenceException(x)) | _ -> y

    let bindProp ((tc,propName),tyargs) =
        let typ = MkNamedType(tc,tyargs)
        typ.GetProperty(propName, staticOrInstance) |> nonNull ("bindProp: failed to bind property '"+ propName+"'") 

    let bindField ((tc,fldName),tyargs) =
        let typ = MkNamedType(tc,tyargs)
        typ.GetField(fldName,staticOrInstance) |> nonNull ("bindField: failed to bind field '"+ fldName+"'") 

    let bindCtor ((tc,argTypes : (Type list tyenv)),tyargs) =
        let typ = MkNamedType(tc,tyargs)
        let argtyps = argTypes (List.nth tyargs)
        typ.GetConstructor(instanceBindingFlags,null,Array.of_list argtyps,null) |> nonNull ("bindCtor: failed to bind constructor") 

    let splitAt n xs =
        if n<0 then failwith "splitAt: -ve" else
        let rec split l = 
            match l with 
            | 0,xs    -> [],xs
            | n,x::xs -> let front,back = split (n-1,xs)
                         x::front,back
            | n,[]    -> failwith "splitAt: not enough elts list"
        split (n,xs)

    let bindMeth ((tc:Type,(argTypes : Type tyenv list),retType,methName),tyargs) =
        let ntyargs = (tc.GetGenericArguments()).Length 
        let enclTypeArgs,methTypeArgs = splitAt ntyargs tyargs
        let nmtyargs = methTypeArgs.Length
        let ngmeth = bindMethod (fun env (ty : Type tyenv) -> ty (List.nth env)) tc (methName,nmtyargs,argTypes,retType) 
        if (ngmeth.GetGenericArguments()).Length = 0 then ngmeth(* non generic *) 
        else ngmeth.MakeGenericMethod(Array.of_list methTypeArgs) 

    //--------------------------------------------------------------------------
    // Unpickling
    //--------------------------------------------------------------------------

    module SimpleUnpickle = 

        type instate = 
          { is: Bytestream; 
            istrings: string array;
            localAssembly: System.Reflection.Assembly  }

        let u_byte_as_int st = st.is.ReadByte() 

        type 'a unpickler = instate -> 'a

        let u_bool st = let b = u_byte_as_int st in (b = 1) 
        let u_void (is: instate) = ()
        let u_unit (is: instate) = ()
        let prim_u_int32 st = 
            let b0 =  (u_byte_as_int st)
            let b1 =  (u_byte_as_int st)
            let b2 =  (u_byte_as_int st)
            let b3 =  (u_byte_as_int st)
            b0 ||| (b1 <<< 8) ||| (b2 <<< 16) ||| (b3 <<< 24)

        let u_int32 st = 
            let b0 = u_byte_as_int st 
            if b0 <= 0x7F then b0 
            elif b0 <= 0xbf then 
                let b0 = b0 &&& 0x7f 
                let b1 = (u_byte_as_int st) 
                (b0 <<< 8) ||| b1
            else  
                prim_u_int32 st

        let u_bytes st = 
            let n = u_int32 st 
            st.is.ReadBytes(n)

        let prim_u_string st = 
            let len =  (u_int32 st) 
            st.is.ReadUtf8BytesAsString(len)

        let u_int    st = u_int32 st
        let u_sbyte  st = sbyte (u_int32 st)
        let u_byte   st = byte (u_byte_as_int st)
        let u_int16  st = int16 (u_int32 st)
        let u_uint16 st = uint16 (u_int32 st)
        let u_uint32 st = uint32 (u_int32 st)
        let u_int64  st = 
            let b1 = int64 (u_int32 st) &&& 0xFFFFFFFFL 
            let b2 = int64 (u_int32 st) 
            b1 ||| (b2 <<< 32)
        let u_uint64  st = uint64 (u_int64 st)
        let u_double st = System.BitConverter.Int64BitsToDouble (u_int64 st)
        let u_float32 st = System.BitConverter.ToSingle(System.BitConverter.GetBytes(u_int32 st),0)
        let u_char st = Char.chr (int32 (u_uint16 st))
        let inline u_tup2 p1 p2 st = let a = p1 st in let b = p2 st in (a,b)
        let inline u_tup3 p1 p2 p3 st =
            let a = p1 st in let b = p2 st in let c = p3 st in (a,b,c)
        let inline u_tup4 p1 p2 p3 p4 st =
            let a = p1 st in let b = p2 st in let c = p3 st in let d = p4 st in (a,b,c,d)
        let u_uniq tbl st = 
            let n = u_int st 
            if n < 0 || n >= Array.length tbl then failwith ("u_uniq: out of range, n = "+string_of_int n+ ", sizeof(tab) = " + string_of_int (Array.length tbl)); 
            tbl.(n)
        let u_string st = u_uniq st.istrings st

        let rec u_list_aux f acc st = 
            let tag = u_byte_as_int st 
            match tag with
            | 0 -> List.rev acc
            | 1 -> let a = f st in u_list_aux f (a::acc) st 
            | n -> failwith ("u_list: found number " + string_of_int n)
        let u_list f st = u_list_aux f [] st
         
        let unpickle_obj localAssembly u phase2bytes =
            let phase2data = 
                let st2 = 
                   { is = Bytestream.of_bytes phase2bytes 0 phase2bytes.Length; 
                     istrings = [| |];
                     localAssembly=localAssembly }
                u_tup2 (u_list prim_u_string) u_bytes st2 
            let stringTab,phase1bytes = phase2data 
            let st1 = 
               { is = Bytestream.of_bytes phase1bytes 0 phase1bytes.Length; 
                 istrings = Array.of_list stringTab;
                   localAssembly=localAssembly } 
            let res = u st1 
            res 


    open SimpleUnpickle

    let decodeFunTy =
        let tc = (type (int -> int)).GetGenericTypeDefinition() 
        function 
        | [d;r] -> tc.MakeGenericType([| d; r |])
        | _ -> invalid_arg "decodeFunTy"
(*
    let decodeTupleTy args = 
        let n = length args in 
        let repr = 
          match n with 
          | 0 | 1 -> failwith "dtype_to_Type_open: tupletyc"
          | 2 -> (type (int * int))
          | 3 -> (type (int * int * int))
          | 4 -> (type (int * int * int * int))
          | 5 -> (type (int * int * int * int * int))
          | 6 -> (type (int * int * int * int * int * int))
          | 7 -> (type (int * int * int * int * int * int * int))
          | _ -> failwith "dtype_to_Type_open: tuple too long" in 
        repr.GetGenericTypeDefinition().MakeGenericType(Array.of_list args)
*)

    let decodeArrayTy n (tys: Type list) = 
        match tys with
        | [ty] -> if (n = 1) then ty.MakeArrayType() else ty.MakeArrayType(n)  
                  // (type int).MakeArrayType(1) returns "Int[*]" but we need "Int[]"
        | _ -> invalid_arg "decodeArrayTy"
        

    let mkNamedTycon (tcName,ass:Assembly) =
        match ass.GetType(tcName) with 
        | null as r -> 
            // For some reason we get 'null' returned here even when a type with the right name exists... Hence search the slow way...
            match (ass.GetTypes() |> Array.tryfind (fun a -> a.FullName = tcName)) with 
            | Some ty -> ty
            | None -> failwithf "mkNamedTycon: failed to bind type '%s' in assembly '%O', r = '%O'. Available types are:\n%A" tcName ass r (ass.GetTypes() |> Array.map (fun a -> a.FullName))
        | ty -> ty

    let makeGenericType (a:Type) (tyargs:Type list) = 
        if (a.IsGenericTypeDefinition && tyargs.Length = 0) then a.MakeGenericType(tyargs |> List.to_array) else a
        
    let decodeNamedTy tc tsR = MkNamedType(tc,tsR)

    let mscorlib = (type System.Int32).Assembly
    let u_assref st = 
        let a = u_string st 
        if a = "" then mscorlib
        elif a = "." then st.localAssembly 
        else 
            match System.Reflection.Assembly.Load(a) with 
            | null -> failwithf "failed to bind assembly '%s' while processing quotation data" a
            | ass -> ass
        
    let u_namedTyconstSpec st = let a,b = u_tup2 u_string u_assref st in mkNamedTycon (a,b)
    let u_tyvarSpec st = let tvName = u_string st in ()
    let u_tyconstSpec st = 
      let tag = u_byte_as_int st 
      match tag with 
      | 1 -> u_unit             st |> (fun () -> decodeFunTy) 
      | 2 -> u_namedTyconstSpec st |> (fun x  -> decodeNamedTy x) 
      | 3 -> u_int              st |> (fun x  -> decodeArrayTy x) 
      | _ -> failwith "u_tyconstSpec" 

    let appL fs env = List.map (fun f -> f env) fs
    
    let rec u_dtype st : (int -> Type) -> Type = 
      let tag = u_byte_as_int st 
      match tag with 
      | 0 -> u_int                              st |> (fun x env     -> env(x)) 
      | 1 -> u_tup2 u_tyconstSpec (u_list u_dtype) st |> (fun (a,b) env -> a (appL b env))
      | _ -> failwith "u_dtype" 

    let tyenvClosed  (n:int) : Type = failwith "unexpected type variable in closed type"
    let u_dtypes st = let a = u_list u_dtype st in appL a 

    let rec u_expr st = 
        let tag = u_byte_as_int st 
        match tag with 
        | 0 -> u_tup3 u_constSpec u_dtypes (u_list u_expr) st 
                |> (fun (a,b,args) env -> let tyargs = (b env) 
                                          CombExpr(a tyargs, List.map (fun e -> e env) args )) 
        | 1 -> u_exprVarName st 
                |> (fun x env -> VarExpr (x env)) 
        | 2 -> u_tup2 u_varSpec u_expr st
                |> (fun (a,b) env -> LambdaExpr(a env,b env)) 
        | 3 -> u_dtype st
                |> (fun (a)   env -> HoleExpr(a env)) 
        | 4 -> u_expr st
                |> (fun (a)   env -> QuoteExpr(a env)) 
        | _ -> failwith "u_expr"
    and u_exprVarName st = let s,b = u_tup2 u_string u_dtype st in (fun env -> {vText=s; vType=b env})
    and u_varSpec st = let a = u_exprVarName st in (fun env -> {vName=a env;})
    and u_recdFieldSpec st = u_tup2 u_namedTyconstSpec u_string st 
    and u_uconstrSpec st = u_tup2 u_namedTyconstSpec u_string st 
    and u_constSpec st = 
        let tag = u_byte_as_int st 
        match tag with 
        | 0 -> u_void              st |> (fun () tyargs -> CondOp)
        | 1 -> u_topDefData        st |> (fun x tyargs -> TopDefOp (x,tyargs))
        | 2 -> u_void              st |> (fun () tyargs -> LetRecOp)
        | 3 -> u_namedTyconstSpec  st |> (fun x tyargs -> RecdMkOp (MkNamedType(x,tyargs)))
        | 4 -> u_recdFieldSpec     st |> (fun (x,y) tyargs -> RecdGetOp (MkNamedType(x,tyargs),y))
        | 5 -> u_uconstrSpec       st |> (fun (x,y) tyargs -> SumMkOp (MkNamedType(x,tyargs),y))
        | 6 -> u_tup2 u_uconstrSpec u_int st |> (fun ((a,b),c) tyargs -> SumFieldGetOp ((makeGenericType a tyargs), b, c) )
        | 7 -> u_uconstrSpec       st |> (fun (x,y) tyargs -> SumTagTestOp (MkNamedType(x,tyargs),y))
        | 8 -> u_void          st |> (fun () tyargs -> TupleMkOp (List.hd tyargs))
        | 9 -> u_int           st |> (fun x tyargs -> TupleGetOp (List.hd tyargs,x))
        | 10 -> u_void         st |> (fun () tyargs -> EqualityOp)
        | 11 -> u_bool         st |> (fun x tyargs -> BoolOp x)
        | 12 -> u_string       st |> (fun x tyargs -> StringOp x)
        | 13 -> u_float32      st |> (fun x tyargs -> SingleOp x)
        | 14 -> u_double      st |> (fun a tyargs -> DoubleOp a)
        | 15 -> u_char      st |> (fun a tyargs -> CharOp a)
        | 16 -> u_sbyte         st |> (fun a tyargs -> SByteOp a)
        | 17 -> u_byte        st |> (fun a tyargs -> ByteOp a)
        | 18 -> u_int16        st |> (fun a tyargs -> Int16Op a)
        | 19 -> u_uint16       st |> (fun a tyargs -> UInt16Op a)
        | 20 -> u_int32        st |> (fun a tyargs -> Int32Op a)
        | 21 -> u_uint32       st |> (fun a tyargs -> UInt32Op a)
        | 22 -> u_int64        st |> (fun a tyargs -> Int64Op a)
        | 23 -> u_uint64       st |> (fun a tyargs -> UInt64Op a)
        | 24 -> u_void         st |> (fun a tyargs -> UnitOp)
        | 25 -> u_tup2 u_namedTyconstSpec u_string st |> (fun p tyargs -> PropGetOp (bindProp(p,tyargs)))
        | 26 -> u_tup2 u_namedTyconstSpec u_dtypes st |> (fun p tyargs  -> CtorCallOp (bindCtor(p,tyargs)))
        | 27 -> u_void         st |> (fun a tyargs -> GetAddrOp)
        | 28 -> u_void         st |> (fun a tyargs -> CoerceOp (List.hd tyargs))
        | 29 -> u_void         st |> (fun a tyargs -> SeqOp)
        | 30 -> u_void         st |> (fun a tyargs -> EncodedForLoopOp)
        | 31 -> u_tup4 u_namedTyconstSpec (u_list u_dtype) u_dtype u_string st |> (fun p tyargs -> MethodCallOp (bindMeth(p,tyargs)))
        | 32 -> u_void           st |> (fun a tyargs -> ArrayMkOp (List.hd tyargs))
        | 33 -> u_void           st |> (fun a tyargs -> DelegateOp (List.hd tyargs))
        | 34 -> u_void           st |> (fun a tyargs -> EncodedWhileLoopOp)
        | 35 -> u_void           st |> (fun () tyargs -> LetOp)
        | 36 -> u_recdFieldSpec  st |> (fun (x,y) tyargs -> RecdSetOp (x,y))
        | 37 -> u_tup2 u_namedTyconstSpec u_string st |> (fun p tyargs -> FieldGetOp( bindField(p,tyargs)))
        | 38 -> u_void           st |> (fun () tyargs -> LetRecCombOp)
        | 39 -> u_void           st |> (fun () tyargs -> AppOp)
        | 40 -> u_void           st |> (fun () tyargs -> LiftedValueOp(null,(List.hd tyargs)))
        | _ -> failwith "u_constSpec" 
    and u_topDefPath st = u_tup2 (u_list u_string) u_string st
    and u_topDefData st = 
        let (a,b,c) = u_tup3 u_assref u_topDefPath u_dtype st 
        {vAssembly=a; vPath=b; vTypeScheme=(fun tyargs -> c (List.nth  tyargs)) } 
    let unpickle_raw_expr (localType : System.Type) = unpickle_obj localType.Assembly u_expr 
    let u_defn = u_tup4 u_topDefPath (u_list u_tyvarSpec) u_dtype u_expr
    let u_defns = u_list u_defn
    let unpickleDefns (localAssembly : System.Reflection.Assembly) = unpickle_obj localAssembly u_defns

    //--------------------------------------------------------------------------
    // General utilities that will eventually be folded into 
    // Microsoft.FSharp.Quotations.Typed
    //--------------------------------------------------------------------------
    
    /// Collect the specifications of all the 
    /// types of all the holes in a template, left to right
    let rec holesInRawExprAcc e acc = 
        match e with 
        | VarExpr _ | QuoteExpr _ -> acc
        | CombExpr (_, ag) -> List.fold_right (fun a acc -> holesInRawExprAcc a acc) ag acc
        | LambdaExpr (v,b) -> holesInRawExprAcc b acc
        | HoleExpr   ty -> ty :: acc

    /// Fill the holes in an Expr 
    let rec fillHolesInRawExpr l e = 
        match e with 
        | VarExpr _ | QuoteExpr _ -> e,l
        | LambdaExpr (v,b) -> 
           let b2,l = fillHolesInRawExpr l b 
           LambdaExpr(v, b2),l
        | CombExpr   (op,args) ->
           let nl, args2 = args |> List.fold_left (fun (l, nargs) arg -> 
              let (arg2, l2) = fillHolesInRawExpr l arg in (l2, arg2::nargs)) (l, [])
           CombExpr(op, args2 |> List.rev),nl 
        | HoleExpr   ty -> 
           match l with 
           | h :: t -> 
              checkTypes (typeOf h) ty "fill" "type of the argument doesn't match the hole type"
              h,t 
           | [] -> failwith "fillHolesInRawExpr"


    // TODO: remove this in favour of sets
    let rec try_assoc x l = 
        match l with 
        | [] -> None
        | ((h,r)::t) -> if x = h then Some(r) else try_assoc x t

    // TODO: remove this in favour of sets
    let gen_mem e v l = List.exists (e v) l
    let gen_insert e v l = if gen_mem e v l then l else v::l
    let gen_union e l1 l2 = 
        match l1 with 
        | [] -> l2 
        | _ -> 
        match l2 with 
        | [] -> l1 
        | _ -> 
        List.fold_left (fun l v -> gen_insert e v l) l1 l2
    let gen_remove e v l = List.filter (fun v2 -> not (e v v2)) l

    let rec freeInExprAcc e acc = 
        match e with 
        | HoleExpr   _ | QuoteExpr  _ -> acc
        | CombExpr (_, ag) -> ag |> List.fold_left (fun acc a -> freeInExprAcc a acc) acc
        | VarExpr    v -> if List.exists (fun v2 -> v = v2) acc then acc else v::acc
        | LambdaExpr (v,b) -> gen_union (=) (gen_remove (=) v.vName (freeInExpr b)) acc
    and freeInExpr e = freeInExprAcc e []

    // utility for folding
    let fold_while f st (ie:#seq<'a>)  = 
        use e = ie.GetEnumerator()
        let mutable res = Some st
        while (Option.is_some res && e.MoveNext()) do
          res <-  f (match res with Some a -> a | _ -> failwith "internal error") e.Current;
        done;
        res      
    
    /// Match against an expression template
    let rec queryAcc aenv e1 e2 acc = 
        match e1,e2 with 
        | CombExpr (c1, a1), CombExpr (c2, a2) ->
            if (c1 <> c2) then None else
            let args = Seq.zip a1 a2 
            args |> fold_while (fun acc (a1, a2) -> (queryAcc aenv a1 a2 acc)) acc
        | CombExpr _, _ -> None 
        | VarExpr    v1,VarExpr v2 -> 
            let v1 = match try_assoc v1 aenv with Some x -> x | None -> v1 
            if v1 = v2 then Some(acc) else None
        | VarExpr    v1,_ -> None
        | QuoteExpr    v1,QuoteExpr v2 -> if v1 = v2 then Some(acc) else None
        | QuoteExpr    v1,_ -> None
        | LambdaExpr (v1,b1), LambdaExpr(v2,b2) -> 
            if v1.vName.vType = v2.vName.vType then 
              queryAcc ((v1.vName,v2.vName)::aenv) b1 b2 [] |> Option.bind (fun res -> 
                if List.exists (fun (_,fvs) -> gen_mem (=) v2.vName fvs) res then None (* escaping variable *)
                else Some(res@acc))
            else None 
        | LambdaExpr (v1,b1),_ -> None 
        | HoleExpr   _, t2 -> Some((t2,freeInExpr t2)::acc)

    let mk_tpsubst tyargs =
        let arr = Array.of_list tyargs 
        let n = Array.length arr 
        fun idx -> 
          if idx < n then arr.(idx)
          else failwith "type argument out of range"

    exception Clash of ExprVarName

    /// Replace type variables and expression variables with parameters using the
    /// given substitution functions/maps.  
    let rec substituteB bvs tmsubst e = 
        match e with 
        | CombExpr (c, args) -> 
            let substargs = args |> List.map (fun arg -> substituteB bvs tmsubst arg) 
            CombExpr(c, substargs)
        | VarExpr    v -> 
            match tmsubst v with 
            | None -> e 
            | Some e2 -> 
                let fvs = freeInExpr e2 
                match List.filter (fun v -> gen_mem (=) v fvs) bvs with
                | [] -> e2
                | bv :: _ -> raise (Clash(bv)) // REVIEW: multiple clashes will resolve inefficiently
        | QuoteExpr  q -> e
        | LambdaExpr (v,b) -> 
             let vname = v.vName 
             try LambdaExpr(v,substituteB (vname ::bvs) tmsubst b)
             with Clash(bv) when bv = vname ->
                 let v2name = freshExprVarName vname.Text vname.Type
                 let v2exp = VarExpr v2name 
                 let v2 = {v with vName=v2name}
                 LambdaExpr(v2,
                            substituteB bvs (fun v -> if v = vname then Some(v2exp) else tmsubst v) b)
        | HoleExpr   ty -> HoleExpr(ty)


    let substitute tmsubst e = substituteB [] tmsubst e 

    let readToEnd (s : Stream) = 
        let n = int s.Length 
        let res = Array.zero_create n 
        let i = ref 0 
        while (!i < n) do 
          i := !i + s.Read(res,!i,(n - !i)) 
        done;
        res 

    let decodedTopResources = HashSet<Assembly * string>.Create(HashIdentity.Structural,10)

    type topEnvTableKey = (string list * string) * string
    type topEnvTableEntry = unit list * tyenv<Type> * tyenv<Expr>

    let topEnvTable = Microsoft.FSharp.Collections.HashMultiMap<topEnvTableKey,topEnvTableEntry>.Create(HashIdentity.Structural, 10)

    let ExplicitlyRegisterTopDefs (assem : Assembly) rn (bytes:byte[]) =
        lock topEnvTable (fun () -> 
            let env = unpickleDefns assem  bytes 
            let assemFullName = assem.FullName
            env |> List.iter (fun (nm,tps,tauty,e) -> 
                let key = (nm, assemFullName) 
                //printfn "Adding %A, hc = %d, hc2 = %d" key (key.GetHashCode()) (assem.GetHashCode());
                topEnvTable.Add(key,(tps,tauty,e)));
            //System.Console.WriteLine("Added {0} resource {1}", assem.FullName, rn);
            decodedTopResources.Add((assem,rn)))

    let ResolveTopDefinition (td,tyargs) =
        let data = 
            let assem = td.vAssembly 
            let assemFullName = td.vAssembly.FullName
            let key = (td.vPath, assemFullName)
            //printfn "Looking for %A, hc = %d, hc2 = %d" key (key.GetHashCode()) (assem.GetHashCode());
            begin match topEnvTable.TryFind(key) with 
            | Some res ->  Some res
            | None ->
                //System.Console.WriteLine("Loading {0}", td.Assembly);
                let qdataResources = 
                    // dynamic assemblies don't support the GetManifestResourceNames 
                    match assem with 
                    | :? System.Reflection.Emit.AssemblyBuilder -> []
                    | _ -> 
                        (try assem.GetManifestResourceNames()  
                         // This raises NotSupportedException for dynamic assemblies
                         with :? NotSupportedException -> [| |])
                        |> Array.to_list 
                        |> List.filter (fun rn -> 
                              //System.Console.WriteLine("Considering resource {0}", rn);
                              rn.StartsWith(PickledDefinitionsResourceNameBase)) 
                        |> List.filter (fun rn -> not (decodedTopResources.Contains((td.vAssembly,rn)))) 
                        |> List.map (fun rn -> rn,unpickleDefns assem (readToEnd (assem.GetManifestResourceStream(rn)))) 
                    
                // ok, add to the table
                lock topEnvTable (fun () -> 
                     // check another thread didn't get in first
                     if not (topEnvTable.Contains(key)) then
                         qdataResources 
                         |> List.iter (fun (rn,defns) ->
                             defns |> List.iter (fun (nm,tps,tauty,e) -> 
                                let key = (nm, assemFullName) 
                                topEnvTable.Add(key,(tps,tauty,e)));
                             decodedTopResources.Add((td.vAssembly,rn)))
                );

                // we know it's in the table now
                topEnvTable.TryFind(key)
            end;
        match data with 
        | Some (tps,tauty,e) -> 
            if (length tps <> length tyargs) then 
                invalid_arg (sprintf "the definition '%A' expects %d type arguments but %d were provided" td.Path (length tps) (length tyargs));
            Some(e (mk_tpsubst tyargs))
        | None -> None

    let (|ResolvedTopDefn|_|) inp = ResolveTopDefinition inp
    let (|ResolvedTopDefnUse|_|) = function (AnyTopDefnUse(ResolvedTopDefn(expr) & (data,_))) -> Some(data,expr) | _ -> None

    // Utility pattern 
    let (|MethodCtor|MethodOrdinary|) = fun (m:#MethodBase) -> 
        if (m.IsConstructor) then MethodCtor 
        else MethodOrdinary    
    
    // Find quotations for specified member in the class
    //
    // Note: this re-implements the mangling scheme used in the F# compiler.
    // However, this is not otherwise revealed externally from this library module.
    let ResolveMethodDefn (methodInfo: #MethodBase) =
        let parentType = methodInfo.DeclaringType in
        let nsNames = 
            match parentType.Namespace with 
            | null -> [] 
            | ns -> ns.Split([| '.' |]) |> Array.to_list in
            
        let modNames = 
            let rec getModNames acc (t:System.Type) = 
                match t.DeclaringType with 
                | null -> acc 
                | t2 -> getModNames (t2.Name ::acc) t2 in
            getModNames [] parentType in
            
        let path = nsNames @ modNames in

        // Note: this re-implements the mangling scheme used in the F# compiler - see mkMemberDataAndUniqueId
        let (metaName,alt) = 
            let instAdd = if (Enum.test methodInfo.Attributes MethodAttributes.Static) then 0 else 1 in
            let overrideFlag = if ((Enum.test methodInfo.Attributes MethodAttributes.HideBySig) && (Enum.test methodInfo.Attributes MethodAttributes.Virtual)) then ".override" else ""
            let argCountText = 
              let attrs = methodInfo.GetCustomAttributes(typeof<OverloadIDAttribute>, true) in
              if attrs.Length > 0 then 
                  (attrs.[0] :?> OverloadIDAttribute).UniqueName 
              else 
                match methodInfo with
                | MethodCtor -> sprintf "%d" (methodInfo.GetParameters().Length)
                | _ -> sprintf "%d" (methodInfo.GetParameters().Length + instAdd)  in

            match methodInfo with
            | MethodCtor -> (sprintf "%s.ctor.%s" parentType.Name argCountText, "")
            | MethodOrdinary -> (sprintf "%s.%s.%s%s" parentType.Name methodInfo.Name argCountText overrideFlag, // tupled / not tupled
                                 sprintf "%s.%s.2%s" parentType.Name methodInfo.Name overrideFlag) in    
        
        let modDef = TopDefnData.Create(parentType.Assembly, (path, metaName), (fun _ -> assert(false)) ) in
        let modDef2 = TopDefnData.Create(parentType.Assembly, (path, alt), (fun _ -> assert(false)) )in
        let tyargs = Seq.concat [(if parentType.IsGenericType then parentType.GetGenericArguments() else [| |]);
                                 (if methodInfo.IsGenericMethod then methodInfo.GetGenericArguments() else [| |])] |> Seq.to_list in
        //printfn "path = %A" path;                                 
        //printfn "metaName = %A" metaName;                                 
        //printfn "alt = %A" alt;                                 
        match ResolveTopDefinition (modDef, tyargs) with
        | Some _ as res -> res
        | None -> 
            //printfn "no luck with metaName";                                 
            ResolveTopDefinition (modDef2, tyargs) 
                

    let (|ResolvedMethodDefn|_|) inp = ResolveMethodDefn inp


    let rec DeepMacroExpandUntil cutoff (e: Expr) : Expr = 
        if cutoff e then e else
        match e with  
        | CombExpr(f,((_ :: _) as x)) -> 
            let x = x |> List.map (DeepMacroExpandUntil cutoff)            
            let t = CombExpr(f,x)
            match t with  
            | BetaReducible((v,e),b) -> DeepMacroExpandUntil cutoff (substitute (fun v2 -> if v.vName = v2 then Some(e) else None) b)
            | _ -> t
        | ResolvedTopDefnUse(_,e2) -> 
            DeepMacroExpandUntil cutoff e2
        | Lambda(v,b) -> MkLambda (v,DeepMacroExpandUntil cutoff b)
        | _ ->   e


    /// A helper class to implement the raw polytypic fill opeation
    type RawArgSaver<'b>() = 
        member this.Make(acc:obj) : obj = 
          let f (r:Expr list) (e: Expr) : 'b = (unbox acc : Expr list -> 'b) (e::r) 
          box f

    let rawSaverTyC = (type RawArgSaver<obj>).GetGenericTypeDefinition()  
    let cast a = unbox(box a)
    let construct (args : obj array) : 'dty =
        let dty = (type 'dty) 
        (cast (System.Activator.CreateInstance(dty,args,(null:obj[]))) : 'dty)

    let Unpickle localAssembly bytes = RawTemplate (unpickle_raw_expr localAssembly bytes tyenvClosed)

    /// Untyped (raw) expression splicing
    let fillRawTemplate (RawTemplate e : Template<'rawtupty,'rawfty>) : 'rawfty = 
        let holetyRs = holesInRawExprAcc e [] 
        match holetyRs with 
        | [] -> 
            // This cast is needed to prove that 'rawfty = Expr
            (cast e : 'rawfty)
        | _ ->
            let holetys = holetyRs 
            // This is the last thing we do (the accumulated args argument
            // is applied by ArgSaver above).  This uses 
            // the collected arguments to fill in the
            // holes in the untyped term, and then return the result.
            let acc = 
                box (fun savedArgs ->
                    let args = List.rev savedArgs 
                    // Fill in the arguments of the untyped term
                    let filled,leftover = fillHolesInRawExpr args e 
                    filled) 
            let accT = (type Expr) 
            let res,accT = 
               List.fold_left
                   (fun (acc,accT) _ ->
                       // Build an objet which closes over the ploymorphic-recursive type
                       let saverTy = rawSaverTyC.MakeGenericType( [| accT |]) 
                       let saver =  System.Activator.CreateInstance(saverTy) 
                       // Invoke the helper to build a function from the domain to the
                       // range.  The function saves its argument into savedArgSink.
                       let acc = saverTy.InvokeMember("Make",(BindingFlags.Public ||| BindingFlags.InvokeMethod ||| BindingFlags.Instance),(null:Binder),saver,[| acc|])
                       acc,(mkFunTy (type Expr) accT))
                   (acc,accT) 
                   (List.rev holetys) 
            (unbox res : Expr list -> 'rawfty) [] 
       
    /// Untyped (raw) expression splicing, taking a unit/tuple as argument
    let fillq (RawTemplate e : Template<'rawtupty,'rawfty>) (args : 'rawtupty) : Expr = 
        let holetyRs = holesInRawExprAcc e [] 
        match holetyRs with 
        | [] -> e
        | [_] -> fst (fillHolesInRawExpr [(cast args : Expr)] e)
        | _ -> 
            match Reflection.Value.GetInfo args with 
            |  Reflection.TupleValue(args) -> fst (fillHolesInRawExpr (List.map cast args) e)
            | _ -> failwith "fillq: expected a tuple value"
       
    let query (RawTemplate e : Template<'rawtupty,'rawfty>) (e2 : Expr) : 'rawtupty option = 
        match queryAcc [] e e2 [] with 
        | None -> None
        | Some([]) -> Some (cast () : 'rawtupty)
        | Some([((res : Expr),_)]) -> 
            // 'rawtupty is a solitary expression type, relying on the existence of a constructor
            // on the expression type
            Some (cast res : 'rawtupty)
        | Some(resl) ->
            let resl = List.map fst (List.rev resl) 
            // 'rawtupty is a tuple type of expression types.  First create the individual strongly
            // typed expressions.
            let dtys = (type 'rawtupty).GetGenericArguments() 
            let resl = Array.of_list resl 
            let exprObjs = Array.map box resl 
            // Now create the tuple type (again relying on the existence of an appropriate constructor on the tuple type)
            // REVIEW: it may be better to expose a tuple construction
            // primitve in Microsoft.FSharp.Basics.Reflection
            Some (construct exprObjs : 'rawtupty)
             
    let (<@@. .@@>) (t : ('a,'b) Template)   = t
    let (<@@   @@>) t   = fillRawTemplate t

    let MkTemplate t = fillq t
    let (|Template|_|) t x = query t x
    let (§§) (x:'a) = MkLiftedValue (box x,(type 'a))
    let (~%%) (x:'a) = MkLiftedValue (box x,(type 'a))

    let bracketIfL b l = (if b then bracketL l else l)
    let rec exprL env prec x = 
        match x with 
        | Let((v,e),b) -> (wordL "let" ++ exprVarL v ++ wordL "=" ++ exprL env 4 e) --- exprL env 4 b
        | PropGet(prop,e) -> (exprL env 2 e ++ wordL "." ++ wordL prop.Name) 
        | MethodCall(meth,es)  -> 
            if meth.IsStatic || es.IsNil then (wordL meth.DeclaringType.FullName ++ wordL "." ++ wordL meth.Name ++ bracketL (commaListL (map (exprL env 2) es)))
            else (exprL env 2 es.Head ++ wordL meth.Name ++ bracketL (commaListL (map (exprL env 2) es.Tail)))
        | FieldGet(fld,e) -> 
            (exprL env 2 e ++ wordL "." ++ wordL fld.Name) 
        | CombExpr(c,args) -> 
            let a = args |> List.fold_left (fun st a -> st --- (exprL env 3 a)) (exprConstrL env prec c) 
            bracketL a
        | VarExpr i -> exprVarNameL i
        | QuoteExpr i -> quoteExprL env i
        | LambdaExpr (v,e) ->  bracketIfL (prec <= 3) ((wordL "fun" ++ exprVarL v ++ wordL "->") --- exprL env 4 e)
        | HoleExpr i -> wordL "_"
    and exprVarL (x:ExprVar) = exprVarNameL x.Name
    and exprVarNameL (x:ExprVarName) = wordL x.Text
    and quoteExprL env x = leftL "<@ " ++ exprL env 4 x ++ rightL " @>"
    and exprConstrL (env : IEnvironment) prec x = 
        match x with 
        | AppOp -> wordL "App"
        | CondOp -> wordL "Cond" 
        | TopDefOp (d,_) -> wordL (String.concat "." (fst d.Path @ [snd d.Path ]))
        | LetRecOp  -> wordL "Rec"
        | LetOp  -> wordL "Let"
        | RecdMkOp _  -> wordL "Recd"
        | RecdGetOp (x,y) -> wordL ("#"+ y)
        | RecdSetOp (x,y) -> wordL ("(#"+y+"<-)")
        | SumMkOp _ -> wordL "Sum"
        | SumFieldGetOp _ -> wordL "MkSumFieldGet"
        | SumTagTestOp _ -> wordL "MkSumTagTest"
        | TupleMkOp _ -> wordL "MkTuple"
        | TupleGetOp (_,x) -> wordL ("#"+string_of_int x)
        | EqualityOp -> wordL "Equality"
        | UnitOp   -> wordL "()"
        | BoolOp   s -> bracketIfL (prec <= 3) (wordL "Bool" ++ env.GetLayout(box s))
        | SingleOp s -> bracketIfL (prec <= 3) (wordL "Single" ++ env.GetLayout(box s))
        | DoubleOp s -> bracketIfL (prec <= 3) (wordL "Double" ++ env.GetLayout(box s))
        | CharOp   s -> bracketIfL (prec <= 3) (wordL "Char" ++ env.GetLayout(box s))
        | SByteOp  s -> bracketIfL (prec <= 3) (wordL "SByte" ++ env.GetLayout(box s))
        | ByteOp  s  -> bracketIfL (prec <= 3) (wordL "Byte" ++ env.GetLayout(box s))
        | Int16Op s  -> bracketIfL (prec <= 3) (wordL "Int16" ++ env.GetLayout(box s))
        | UInt16Op s -> bracketIfL (prec <= 3) (wordL "UInt16" ++ env.GetLayout(box s))
        | PropGetOp x -> bracketIfL (prec <= 3) (wordL "Prop" ++ wordL (x.ToString()))
        | FieldGetOp x -> bracketIfL (prec <= 3) (wordL "Field" ++ wordL (x.ToString()))
        | CtorCallOp s -> bracketIfL (prec <= 3) (wordL "new" ++ wordL (s.ToString()))
        | MethodCallOp s -> bracketIfL (prec <= 3) (wordL "call" ++ wordL (s.ToString()))
        | StringOp s -> bracketIfL (prec <= 3) (wordL "String" ++ env.GetLayout(box s))
        | Int32Op s -> bracketIfL (prec <= 3) (wordL "Int32" ++ env.GetLayout(box s))
        | UInt32Op s -> bracketIfL (prec <= 3) (wordL "UInt32" ++ env.GetLayout(box s))
        | Int64Op s -> bracketIfL (prec <= 3) (wordL "Int64" ++ env.GetLayout(box s))
        | UInt64Op s -> bracketIfL (prec <= 3) (wordL "UInt64" ++ env.GetLayout(box s))
        | CoerceOp ty -> wordL "Coerce"
        | ArrayMkOp _ -> wordL "MkNewArray"
        | DelegateOp _-> wordL "Delegate"
        | GetAddrOp -> wordL "GetAddr"
        | SeqOp -> wordL "Seq"
        | EncodedForLoopOp -> wordL "For"
        | EncodedWhileLoopOp -> wordL "While"
        | LiftedValueOp (v,ty) -> objL v
        | LetRecCombOp -> wordL "LetRecUnwind"
              

end (* end of Raw module *)

type Expr with 
    member x.Type = Raw.typeOf x 

type Template<'a,'tupty,'fty,'rty> = 
    | TypedTemplate of Expr
    member x.Raw = match x with TypedTemplate e -> e
    interface IFormattable with 
        member x.GetLayout(env) = env.GetLayout(x.Raw)

type Expr<'a> = 
    | TypedExpr of Expr
    member x.Raw = match x with TypedExpr e -> e
    interface IFormattable with 
        member x.GetLayout(env) = env.GetLayout(x.Raw)


module Typed = begin

    open Raw

    let Unpickle localAssembly bytes = TypedTemplate (Raw.unpickle_raw_expr localAssembly bytes tyenvClosed)

    let to_raw (TypedExpr e) = e
    let of_raw e = (TypedExpr e) 
    let map_raw f (inp : Expr<'a>) : Expr<'a> = inp |> to_raw |> f |> of_raw

    // A helper class to implement the polytypic fill opeation
    // This is fairly obscure stuff. We want to build a function type of type "ExprList -> Expr<'a> -> 'b"
    // where we dynamically instantiate 'a and 'b using reflection System.Type values.
    type ArgSaver<'a,'b>() = 
        member this.Make(acc:obj) : obj = 
            let f (r:Expr list) (TypedExpr e:Expr<'a>) : 'b = 
               (unbox acc : Expr list -> 'b) (e::r) 
            box f

    let saverTyC = (type ArgSaver<obj,obj>).GetGenericTypeDefinition()  
    let exprTyC = (type (obj Expr)).GetGenericTypeDefinition()  

    let debug = false

    let cast a = unbox(box a)

    /// Typesafe expression splicing
    let fillTypedTemplate (TypedTemplate e : Template<'ety,'tupty,'fty,'ety Expr>) : 'fty = 
        let holetyRs = holesInRawExprAcc e [] 
        match holetyRs with 
        | [] -> 
          // This cast is needed to prove that 'fty matches 'ety
          (cast (of_raw e : 'ety Expr) : 'fty)
        | _ ->
          let holetys = holetyRs 
          // This is the last thing we do (the accumulated args argument
          // is applied by ArgSaver above).  This uses 
          // the collected arguments to fill in the
          // holes in the untyped term, and then return the result.
          let acc = 
              box (fun savedArgs -> 
                  let args = List.rev savedArgs 
                  // Fill in the arguments of the untyped term
                  let filled,leftover = Raw.fillHolesInRawExpr args e 
                  (of_raw filled : 'ety Expr)) 
          let accT = (type 'ety Expr) 
          let res,accT = 
             List.fold_left
               (fun (acc,accT) (holeT: System.Type) ->
                 // domT = 'holeT Expr for some 'holeT.  Find out what holeT is....
                 let domT = exprTyC.MakeGenericType( [| holeT |]) 

                 // Build an objet which closes over the domain/range types
                 // and acts as a helper to build a function from the domain to the
                 // range.
                 let saverTy = saverTyC.MakeGenericType( [| holeT; accT |]) 
                 let saver =  System.Activator.CreateInstance(saverTy) 

                 // Invoke the helper to build a function from the domain to the
                 // range.  The function saves its argument into savedArgSink.
                 let acc = saverTy.InvokeMember("Make",(BindingFlags.Public ||| BindingFlags.InvokeMethod ||| BindingFlags.Instance),(null:Binder),saver,[| acc|])
                 acc,(mkFunTy domT accT))
               (acc,accT) 
               (List.rev holetys) 

          (unbox res : Expr list -> 'fty) [] 
       
    let construct (args : obj array) : 'dty =
      let dty = (type 'dty) 
      (cast (System.Activator.CreateInstance(dty,args,(null:obj[]))) : 'dty)

    let query (TypedTemplate e : Template<'ety,'tupty,'fty,'ety Expr>) (e2 : 'ety Expr) : 'tupty option = 
      match Raw.queryAcc [] e (to_raw e2) [] with 
      | None -> None
      | Some([]) -> Some (cast () : 'tupty)
      | Some([((res : Expr),_)]) -> 
          // 'tupty is a solitary expression type, relying on the existence of a constructor
          // on the expression type
          Some (construct [| box res |] : 'tupty)
      | Some(resl) ->
          let resl = List.map fst (List.rev resl) 
          // 'tupty is a tuple type of expression types.  First create the individual strongly
          // typed expressions.
          let dtys = (type 'tupty).GetGenericArguments() 
          let resl = Array.of_list resl 
          let exprObjs = 
            dtys |> Array.mapi (fun i tupty -> 
                let res = Array.get resl i  
                System.Activator.CreateInstance(tupty,[|box res|],(null : obj[]))) 
          // Now create the tuple type (again relying on the existence of an appropriate constructor on the tuple type)
          // REVIEW: better to expose primitves in Microsoft.FSharp.Reflection
          Some (construct exprObjs : 'tupty)
         //Some (obj :?> 'tupty)
             
    let DeepMacroExpandUntil cutoff e = map_raw (Raw.DeepMacroExpandUntil cutoff) e
    let substitute tmenv e = map_raw (Raw.substitute tmenv) e
     
    let (« ») t     = fillTypedTemplate t
    let (<@ @>) t   = fillTypedTemplate t
    let (<@. .@>) (t : ('a,'b,'c,Expr<'a>) Template)   = t

    let lift (x : 'a) : Expr<'a> = 
        TypedExpr (MkLiftedValue (box x,(type 'a)))

    let (§) x = lift x
    let (~%) x = lift x

    let MkTemplate x = fillTypedTemplate x
    let (|Template|_|) x = query x
end  (* End of module Typed *)
open Raw

type ExprConstInfo with 
    interface IFormattable with 
      member x.GetLayout(env) = exprConstrL env 0 x 
  
type Expr with 
    interface IFormattable with 
        member x.GetLayout(env) = quoteExprL env x 
    member x.Substitute f = Raw.substitute f x
    member x.GetFreeVariables ()  = (Raw.freeInExpr x :> seq<_>)

type Expr<'a> with 
    member x.Substitute f = Typed.substitute f x

type ExprVar with 
    static member Fresh(nm,ty:Type) = ExprVar.Create(freshExprVarName nm ty)
    interface IFormattable with 
        member x.GetLayout(env) = exprVarL x 

type ExprVarName with 
    interface IFormattable with 
        member x.GetLayout(env) = exprVarNameL x 

#endif
