// (c) Microsoft Corporation 2005-2007. 

#light

/// C-like operations on native pointers.  Use of any of these operators will
/// result in the generation of unverifiable code.

namespace Microsoft.FSharp.NativeInterop

open Microsoft.FSharp.Core
open Microsoft.FSharp.Collections
open Microsoft.FSharp.Collections
open Microsoft.FSharp.Math

/// This type wraps a pointer to a blob of unmanaged memory assumed to contain
/// a C-style one-dimensional array of items compatible with the (presumably blittable) 
/// type 'a.  The blob of memory must be allocated and managed externally, 
/// e.g. by a computation routine written in C.
///
/// All operations on this type are marked inlined
/// because the code used to implement the operations is not verifiable.  
///
/// Any code that uses these operations will be unverifiable and may 
/// cause memory corruption if not used with extreme care.
type NativeArray<'a> 
  with 
    /// Creates a C-style one dimensional array from a native pointer and the length of the array
    /// Nothing is actually copied.
    static member FromPtr : startAddress: 'a nativeptr * length: int -> NativeArray<'a>
    /// Pointer to the C-style one-dimensional array
    member Ptr: 'a nativeptr
    /// WARNING: use of this function may lead to unverifiable or invalid code
    [<Unverifiable>]
    member inline Item : int -> 'a with get,set
    /// Length of the C-style one-dimensional array
    member Length : int
  end

/// This type wraps a pointer to a blob of unmanaged memory assumed to contain
/// a C-style row major two-dimensional matrix of items compatible with the (presumably blittable) 
/// type 'a. The blob of memory must be allocated and managed externally, 
/// e.g. by a computation routine written in C.
///
/// All operations on this type are marked inlined
/// because the code used to implement the operations is not verifiable.  
///
/// Any code that uses these operations will be unverifiable and may 
/// cause memory corruption if not used with extreme care.

type NativeArray2<'a> 
  with 
    /// Creates a C-style row major two-dimensional array from a native pointer, the number of rows and the number of columns.  
    /// Nothing is actually copied.
    static member FromPtr : 'a nativeptr * nrows:int * ncols:int -> NativeArray2<'a>
    /// Pointer to the C-style row major two-dimensional array 
    member Ptr: 'a nativeptr
    /// Number of rows of the native array
    member NumRows : int
    /// Number of columns of the native array
    member NumCols : int
    /// WARNING: use of this function may lead to unverifiable or invalid code
    [<Unverifiable>]
    member inline Item : int * int -> 'a with get,set
    /// View a CMatrix as a FortranMatrix.  Doesn't actually allocate
    /// a new matirx - just gives a different label to the same bits, and swaps the
    /// row/column count information associated with the bits.
    member NativeTranspose : FortranMatrix<'a>
  end

/// See NativeArray2
and CMatrix<'a> = NativeArray2<'a> 

/// This type wraps a pointer to a blob of unmanaged memory assumed to contain
/// a Fortran-style column major two-dimensional matrix of items compatible with the (presumably blittable) 
/// type 'a. The blob of memory must be allocated and managed externally, 
/// e.g. by a computation routine written in C.
///
/// All operations on this type are marked inlined
/// because the code used to implement the operations is not verifiable.  
///
/// Any code that uses these operations will be unverifiable and may 
/// cause memory corruption if not used with extreme care.
and FortranMatrix<'a> 
  with 
    static member FromPtr : 'a nativeptr * nrows:int * ncols:int -> FortranMatrix<'a>

    member Ptr: 'a nativeptr

    member NumRows : int
    member NumCols : int

    /// WARNING: use of this function may lead to unverifiable or invalid code
    [<Unverifiable>]
    member inline Item : int * int -> 'a with get,set
    
    /// View a FortranMatrix as a CMatrix.  Doesn't actually allocate
    /// a new matirx - just gives a different label to the same bits, and swaps the
    /// row/column count information associated with the bits.
    member NativeTranspose : CMatrix<'a>
  end
  
module NativePtr =

    /// WARNING: use of this function may lead to unverifiable or invalid code
    [<Unverifiable>]
    val inline of_nativeint : nativeint -> 'a nativeptr

    [<System.Obsolete("This value is now the identity function")>]
    val inline of_ilsigptr: 'a nativeptr -> 'a nativeptr

    /// WARNING: use of this function may lead to unverifiable or invalid code
    [<Unverifiable>]
    val inline to_nativeint : 'a nativeptr -> nativeint

    /// WARNING: use of this function may lead to unverifiable or invalid code
    [<System.Obsolete("This value is now the identity function")>]
    val inline to_ilsigptr: 'a nativeptr -> 'a nativeptr

    /// WARNING: use of this function may lead to unverifiable or invalid code
    [<Unverifiable>]
    val inline to_NativeArray : 'a nativeptr -> int -> NativeArray<'a>

    /// WARNING: use of this function may lead to unverifiable or invalid code
    [<Unverifiable>]
    val inline of_NativeArray : NativeArray<'a> -> 'a nativeptr 

    /// WARNING: use of this function may lead to unverifiable or invalid code
    [<Unverifiable>]
    val inline add : 'a nativeptr -> int -> 'a nativeptr

    /// WARNING: use of this function may lead to unverifiable or invalid code
    [<Unverifiable>]
    val inline get : 'a nativeptr -> int -> 'a

    /// WARNING: use of this function may lead to unverifiable or invalid code
    [<Unverifiable>]
    val inline set : 'a nativeptr -> int -> 'a -> unit


#if CLI_AT_MOST_1_1
#else
module Ref =
    /// Pin the given ref for the duration of a single call to the given function.  A native pointer to
    /// the contents of the ref is passed to the given function.  Cleanup the GCHandle associated with the 
    /// pin when the function completes, even if an exception is raised.
    ///
    /// This function should only be used if 'a is a simple blittable type
    /// such as "int" that does not contain any further heap references.
    ///
    /// WARNING: use of this function may lead to unverifiable or invalid code
    [<Unverifiable>]
    val inline pin : 'a ref -> ('a nativeptr -> 'b) -> 'b

/// Represents a pinned handle to a structure with an underlying 1D array, i.e. an underlying NativeArray.
/// Used when interfacing with native code math libraries such as LAPACK.
type PinnedArray<'a>
    with
        (* interface System.IDisposable *)
        member Ptr : 'a nativeptr 
        member Length : int 
        member NativeArray : NativeArray<'a>
        static member FromNative : NativeArray<'a> * System.Runtime.InteropServices.GCHandle -> PinnedArray<'a>
        static member inline of_vector : Vector<'a> -> PinnedArray<'a>
        static member inline of_rowvec : RowVector<'a> -> PinnedArray<'a>
        static member inline of_array : 'a[] -> PinnedArray<'a>
        member Free : unit -> unit
    end  

/// Represents a pinned handle to a structure with an underlying 2D array, i.e. an underlying NativeArray2.
/// Used when interfacing with native code math libraries such as LAPACK.
type PinnedArray2<'a> 
    with
        (* interface System.IDisposable *)
        static member FromNative : NativeArray2<'a> * System.Runtime.InteropServices.GCHandle -> PinnedArray2<'a> 
        member Ptr : 'a nativeptr 
        member NumRows : int 
        member NumCols : int 
        member NativeArray : NativeArray2<'a>
        static member inline of_matrix : Matrix<'a> -> PinnedArray2<'a>
        static member inline of_array2 : 'a[,] -> PinnedArray2<'a>
        member Free : unit -> unit
    end  
#endif

