// (c) Microsoft Corporation 2005-2007.  

#light

namespace Microsoft.FSharp.Collections

open Microsoft.FSharp.Core
open Microsoft.FSharp.Collections
#if CLI_AT_MOST_1_1
open Microsoft.FSharp.Compatibility
#else
open System.Collections.Generic
#endif

/// Basic operations on lists.  
[<CompilationRepresentation(CompilationRepresentationFlags.ModuleSuffix)>]
module List = 

    /// Return a new list that contains the elements of the first list
    /// followed by elements of the second
    val append: 'a list -> 'a list -> 'a list

    /// Return a new list that contains the elements of each the lists in order
    val concat: #seq<'a list> -> 'a list

    /// Return a new list that contains the elements of each the lists in order.
    /// Same as concat.
    val flatten: #seq<'a list> -> 'a list

    /// Return the length of the list
    val length: 'a list -> int

    /// Return the first element of the list.  Raise (Invalid_argument "hd") if undefined.
    val hd: 'a list -> 'a

    /// Return the tail of the list.  Raise (Invalid_argument "tl") if undefined.
    val tl: 'a list -> 'a list

    /// Index into the list.  The first element has index 0.
    val nth: 'a list -> int -> 'a

    /// Create a list by calling the given generator on each index
    val init: int -> (int -> 'a) -> 'a list

    /// Return true if the list is not empty.
    val nonempty: 'a list -> bool

    /// Apply a function to each element of the collection, threading an 'accumulator' argument
    /// through the computation. Take the second argument, and apply the function to it
    /// and the first element of the list.  Then feed this result into the function along
    /// with the second element and so on.  Return the final result.
    /// If the elements are "i0...iN" then computes "f (... (f s i0) i1 ...) iN"
    val fold_left: ('b -> 'a -> 'b) -> 'b -> 'a list -> 'b

    /// Apply a function to each element of the collection, threading an 'accumulator' argument
    /// through the computation. Apply the function to the first two elements of the list.
    /// Then feed this result into the function along with the third element and so on.  
    /// Return the final result. 
    /// If the elements are "i0...iN" then computes "f (... (f i0 i1) i2 ...) iN"
    /// Raises [[InvalidArgumentException]] if the list has no elements.      
    val reduce_left: ('a -> 'a -> 'a) -> 'a list -> 'a

    [<System.Obsolete("This function has been renamed to 'reduce_left'")>]
    val fold1_left: ('a -> 'a -> 'a) -> 'a list -> 'a

    /// Apply a function to corresponding elements of two collections, threading an 'accumulator' argument
    /// through the computation. The collections must have identical sizes.
    /// If the elements are "i0...iN" and "j0...jN"
    /// then computes "f (... (f s i0 j0)...) iN jN".
    val fold_left2: ('c -> 'a -> 'b -> 'c) -> 'c -> 'a list -> 'b list -> 'c

    /// Apply a function to each element of the collection, threading an 'accumulator' argument
    /// through the computation. If the elements are "i0...iN" then computes "f i0 (...(f iN s))".
    val fold_right: ('a -> 'b -> 'b) -> 'a list -> 'b -> 'b

    /// Apply a function to each element of the collection, threading an 'accumulator' argument
    /// through the computation. If the elements are "i0...iN" then computes "f i0 (...(f iN-1 iN))".
    /// Raises [[InvalidArgumentException]] if the list has no elements.
    val reduce_right: ('a -> 'a -> 'a) -> 'a list -> 'a

    [<System.Obsolete("This function has been renamed to 'reduce_right'")>]
    val fold1_right: ('a -> 'a -> 'a) -> 'a list -> 'a

    /// Apply a function to corresponding elements of two collections, threading an 'accumulator' argument
    /// through the computation. The collections must have identical sizes.
    /// If the elements are "i0...iN" and "j0...jN"
    /// then computes "f i0 j0 (...(f iN jN s))".
    val fold_right2: ('a -> 'b -> 'c -> 'c) -> 'a list -> 'b list -> 'c -> 'c

    /// Apply a function to each element of the collection, threading an 'accumulator' argument
    /// through the computation. Take the second argument, and apply the function to it
    /// and the first element of the list.  Then feed this result into the function along
    /// with the second element and so on.  Return the list of intermediate results and the final result.
    val scan_left : ('b -> 'a -> 'b) -> 'b -> 'a list -> 'b list

    /// Like fold_right, but return both the intermediary and final results
    val scan_right : ('a -> 'b -> 'b) -> 'a list -> 'b -> 'b list
    
    /// Like fold1_left, but return both the intermediary and final results
    val scan1_left : ('a -> 'a -> 'a) -> 'a list -> 'a list

    /// Like fold1_right, but return both the intermediary and final results
    val scan1_right : ('a -> 'a -> 'a) -> 'a list -> 'a list
    

    /// Test if any element of the collection satisfies the given predicate.
    /// If the elements are "i0...iN" 
    /// then computes "p i0 or ... or p iN".
    val exists: ('a -> bool) -> 'a list -> bool

    /// Test if any corresponding elements of the collection satisfy the given predicate pairwise.
    /// If the elements are "i0...iN" and "j0...jN"
    /// then computes "p i0 j0 or ... or p iN jN".
    /// Raises [[Invalid_argument]] exception if the lists do not zip (have different lengths).
    val exists2: ('a -> 'b -> bool) -> 'a list -> 'b list -> bool

    /// Test if all elements of the collection satisfy the given predicate.
    /// If the elements are "i0...iN" 
    /// then computes "p i0 &amp;&amp; ... &amp;&amp; p iN".
    val for_all: ('a -> bool) -> 'a list -> bool

    /// Test if all corresponding elements of the collection satisfy the given predicate pairwise.
    /// If the elements are "i0...iN" and "j0...jN"
    /// then computes "p i0 j0 &amp;&amp; ... &amp;&amp; p iN jN".
    /// Raises [[Invalid_argument]] if the lists do not zip (have different lengths).
    val for_all2: ('a -> 'b -> bool) -> 'a list -> 'b list -> bool

    /// Return a new collection containing only the elements of the collection
    /// for which the given predicate returns "true"
    val filter: ('a -> bool) -> 'a list -> 'a list

    /// Split the collection into two collections, containing the 
    /// elements for which the given predicate returns "true" and "false"
    /// respectively 
    val partition: ('a -> bool) -> 'a list -> ('a list * 'a list)

    
    /// Apply the given function to each element of the list. Return
    /// the list comprised of the results "x" for each element where
    /// the function returns Some(x)
    val choose: ('a -> 'b option) -> 'a list -> 'b list

    /// Apply the given function to successive elements, returning the first
    /// result where function returns "Some(x)" for some x.
    val first: ('a -> 'b option) -> 'a list -> 'b option

    /// Return the first element for which the given function returns "true".
    /// Raise Not_found if no such element exists.
    val find: ('a -> bool) -> 'a list -> 'a

    /// Return the index of the first element in the list
    /// that satisfies the given predicate.
    /// Raise Not_found if no such element exists.
    val find_index : ('a -> bool) -> 'a list -> int

    /// Return the index of the first element in the list
    /// that satisfies the given predicate.
    /// Raise Not_found if no such element exists.
    val find_indexi : (int -> 'a -> bool) -> 'a list -> int

    /// Return the first element for which the given function returns "true".
    /// Return None if no such element exists.
    val tryfind: ('a -> bool) -> 'a list -> 'a option

    /// Return the index of the first element in the list
    /// that satisfies the given predicate.
    /// Return None if no such element exists.
    val tryfind_index : ('a -> bool) -> 'a list -> int option

    /// Return the index of the first element in the list
    /// that satisfies the given predicate.
    /// Return None if no such element exists.
    val tryfind_indexi : (int -> 'a -> bool) -> 'a list -> int option

    /// Return a list containing all the elements for which the given function returns
    /// "true".  Same as "filter"
    val find_all: ('a -> bool) -> 'a list -> 'a list

    /// Apply the given function to each element of the collection. 
    val iter: ('a -> unit) -> 'a list -> unit

    /// Apply the given function to two collections simultaneously. The
    /// collections must have identical size.
    val iter2: ('a -> 'b -> unit) -> 'a list -> 'b list -> unit

    /// Apply the given function to each element of the collection. The integer passed to the
    /// function indicates the index of element.
    val iteri: (int -> 'a -> unit) -> 'a list -> unit

    /// Apply the given function to two collections simultaneously. The
    /// collections must have identical size. The integer passed to the
    /// function indicates the index of element.
    val iteri2: (int -> 'a -> 'b -> unit) -> 'a list -> 'b list -> unit

    /// Build a new collection whose elements are the results of applying the given function
    /// to each of the elements of the collection.
    val map: ('a -> 'b) -> 'a list -> 'b list

    /// Build a new collection whose elements are the results of applying the given function
    /// to the corresponding elements of the two collections pairwise.
    val map2: ('a -> 'b -> 'c) -> 'a list -> 'b list -> 'c list

    /// Build a new collection whose elements are the results of applying the given function
    /// to the corresponding elements of the three collections simultaneously.
    val map3: ('a -> 'b -> 'c -> 'd) -> 'a list -> 'b list -> 'c list ->'d list

    /// Build a new collection whose elements are the results of applying the given function
    /// to each of the elements of the collection. The integer index passed to the
    /// function indicates the index (from 0) of element being transformed.
    val mapi: (int -> 'a -> 'b) -> 'a list -> 'b list

    /// Like mapi, but mapping corresponding elements from two lists of equal length.
    val mapi2: (int -> 'a -> 'b -> 'c) -> 'a list -> 'b list -> 'c list
      
    /// "rev_map f l1" evaluates to "map f (rev l1)"
    val rev_map: ('a -> 'b) -> 'a list -> 'b list

    /// "rev_map2 f l1 l2" evaluates to "map2 f (rev l1) (rev l2)"
    val rev_map2: ('a -> 'b -> 'c) -> 'a list -> 'b list -> 'c list

    /// Return a new list with the elements in reverse order
    val rev: 'a list -> 'a list

    /// "rev_append l1 l2" evaluates to "append (rev l1) l2"
    val rev_append: 'a list -> 'a list -> 'a list

    /// Combine the two lists into a list of pairs. The two lists must have equal lengths.
    val zip: 'a list -> 'b list -> ('a * 'b) list

    /// Combine the three lists into a list of triples. The lists must have equal lengths.
    val zip3: 'a list -> 'b list -> 'c list -> ('a * 'b * 'c) list

    /// Split a list of pairs into two lists
    val unzip: ('a * 'b) list -> ('a list * 'b list)

    /// Split a list of triples into three lists
    val unzip3: ('a * 'b * 'c) list -> ('a list * 'b list * 'c list)

    /// A synonym for List.zip
    val combine: 'a list -> 'b list -> ('a * 'b) list

    /// A synonym for List.unzip
    val split: ('a * 'b) list -> ('a list * 'b list)

    /// Sort the given list using the given comparison function
    val sort: ('a -> 'a -> int) -> 'a list -> 'a list

    /// Sort the given list using the given comparison function, preserving order for equal elements.
    val stable_sort: ('a -> 'a -> int) -> 'a list -> 'a list

    /// Return the sum of the results generated by applying the function to each element of the array.
    val sumByInt     : ('a -> int)     -> 'a list -> int
    /// Return the sum of the results generated by applying the function to each element of the array.
    val sumByFloat   : ('a -> float)   -> 'a list -> float
    /// Return the sum of the results generated by applying the function to each element of the array.
    val sumByFloat32 : ('a -> float32) -> 'a list -> float32
    /// Return the sum of the results generated by applying the function to each element of the array.
    val sumByInt64   : ('a -> int64)   -> 'a list -> int64

    (*--------------------------------------------------------------------------
     * The following functions treat lists as sets, and lists-of-pairs as 
     * dictionaries.
     *
     * mem, assoc and friends use structural equality .
     * memq, assq and friends use object-identity equality (==).
     *--------------------------------------------------------------------------*)

    /// Is an element in the list, uses (=) equality.
    val mem: 'a -> 'a list -> bool

    /// Lookup key's data in association list, uses (=) equality.
    /// Raises [[Not_found]] exception if key not found, in which case you should typically use [[try_assoc]] instead.
    val assoc: 'a -> ('a * 'b) list -> 'b

    /// Lookup key's data in association list, uses (=) equality,
    /// returning "Some data" or "None".  
    val try_assoc: 'a -> ('a * 'b) list -> 'b option

    /// Does the key have pair in the association list?
    val mem_assoc: 'a -> ('a * 'b) list -> bool

    /// Remove pair for key from the association list (if it's there).
    val remove_assoc: 'a -> ('a * 'b) list -> ('a * 'b) list

    /// See [[assoc]], but uses the physical equality operator (==) for equality tests
    val assq: 'a -> ('a * 'b) list -> 'b
      
    /// See [[try_assoc]], but uses the physical equality operator (==) for equality tests.    
    val try_assq: 'a -> ('a * 'b) list -> 'b option

    /// See [[mem_assoc]], but uses the physical equality operator (==) for equality tests.      
    val mem_assq: 'a -> ('a * 'b) list -> bool

    /// See [[remove_assoc]], but uses the physical equality operator (==) for equality tests.        
    val remove_assq: 'a -> ('a * 'b) list -> ('a * 'b) list

    /// See [[mem]], but uses the physical equality operator (==) for equality tests.        
    val memq: 'a -> 'a list -> bool

    /// Build a collection from the given array
    val of_array: 'a array -> 'a list

    /// Build an array from the given collection
    val to_array: 'a list -> 'a array

    /// Return a view of the collection as an enumerable object
    [<System.Obsolete("Consider using to_seq instead")>]
    val to_IEnumerable: 'a list -> seq<'a>

    /// Build a new collection from the given enumerable object
    [<System.Obsolete("Consider using of_seq instead")>]
    val of_IEnumerable: #seq<'a> -> 'a list

    /// Return a view of the collection as an enumerable object
    val to_seq: 'a list -> seq<'a>

    /// Build a new collection from the given enumerable object
    val of_seq: #seq<'a> -> 'a list

    #if CLI_AT_MOST_1_1
    #else
    /// Return a view of the collection as a .NET collection. Results in a ReadOnly collection.
    [<System.Obsolete("This function will be removed in a future release of F#")>]
    val to_ICollection: 'a list -> ICollection<'a> 

    /// Build a new collection from any type that supports the .NET ICollection interface
    /// Unspecified behaviour if the underlying collection is mutated.
    [<System.Obsolete("Consider using of_seq instead")>]
    val of_ICollection: #ICollection<'a> -> 'a list

    /// Convert to the System.Collections.Generic.List type, which in F# is referred to as ResizeArray
    [<System.Obsolete("This function will be removed in a future release of F#. Use ResizeArray.of_seq instead")>]
    val to_ResizeArray: 'a list -> ResizeArray<'a>

    /// Convert from the System.Collections.Generic.List type, which in F# is referred to as ResizeArray
    [<System.Obsolete("This function will be removed in a future release of F#. Use List.of_seq instead")>]
    val of_ResizeArray: ResizeArray<'a> -> 'a list

    /// Convert to/from the System.Collections.Generic.List type 
    [<System.Obsolete("This function will be removed in a future release of F#. Use ResizeArray.of_seq instead")>]
    val to_List: 'a list -> ResizeArray<'a>

    [<System.Obsolete("Consider using of_seq instead")>]
    val of_List: ResizeArray<'a> -> 'a list
    #endif

    /// For each element of the list, apply the given function. Concatenate all the results and return the combined list.
    val map_concat : ('a -> 'b list) -> 'a list -> 'b list
